"""
Copyright (C) 2017-2021 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Nov 2, 2021

@author: maryam
"""
from __future__ import annotations
import re
from decimal import Decimal

from enum import Enum, auto, unique
from typing import Optional, List, Dict, TYPE_CHECKING

from util.singleton_logger import Logger
import util.gen_util as gen_util

from device.db import PeripheryDevice

from design.db_item import GenericParam, GenericParamGroup, PeriDesignGenPinItem, \
        PeriDesignRegistry, PeriDesignItem, GenericParamService

import common_device.ddr.ddr_design as ddr_des
from common_device.property import PropertyMetaData
from tx180_device.ddr.ddr_preset_service_adv import DDRPresetServiceAdv

if TYPE_CHECKING:
    from tx180_device.ddr.ddr_device_service_adv import DDRServiceAdvance


@gen_util.freeze_it
class DDRParamInfo(PropertyMetaData):
    class Id(Enum):
        """
        Parameter ID
        """
        clkin_sel = auto()
        axi_width_256 = auto()
        axi_enabled = auto()
        reg_inf_ena = auto()
        data_width = auto()
        physical_rank = auto()
        memory_type = auto()
        memory_density = auto()
        is_enable_pin_swap = auto()


@unique
class DDRPkgPinType(Enum):
    DQ = "DQ"
    DM = "DM"
    CA = "CA"


@gen_util.freeze_it
class DDRAdvConfigAdvance(ddr_des.DDRAdvConfig):
    """
    Container for advance configuration settings.

    Each category contains parameter of GenericParamGroup type. The advanced config does not
    track the dependency with basic setting (e.g. memory type), that needs to be managed by caller.
    Default setting will be set based on a unique setting id, managed by device db.
    The setting id itself can be a memory type or any other key parameter that device db choose.
    """

    def __init__(self):
        self._cs_adv_fpga = GenericParamGroup()  #: Parameter for advanced fpga setting
        self._cs_adv_memory = GenericParamGroup()  #: Parameter for advanced memory setting
        self._cs_adv_memory_timing = GenericParamGroup()  #: Parameter for advanced memory timing setting

        self._cs_adv_ctrl = None
        self._cs_adv_gate_delay = None

        self.logger = Logger

    def __str__(self, *args, **kwargs):

        fpga = "-- FPGA Settings\n{}\n".format(self._cs_adv_fpga)
        memory = "-- Memory Settings\n{}\n".format(self._cs_adv_memory)
        mem_timing = "-- Memory Timing Settings\n{}\n".format(self._cs_adv_memory_timing)
        info = "{}{}{}".format(fpga, memory, mem_timing)
        return info

    def set_default_setting(self, mem_type, setting_id=None, device_db=None, cat_type=None):
        """
        Set default configuration based on setting id. If None, use the
        generic defaults

        :param cat_type: Advanced setting category type
        :param mem_type: Memory type
        :param setting_id: Advanced setting id as in the preset/config sqlite db
        :param device_db: Device db
        """
        if cat_type is None:
            # set for all
            self.build_default_fpga_setting(mem_type, setting_id, device_db)
            self.build_default_memory_setting(mem_type, setting_id, device_db)
            self.build_default_memory_timing_setting(mem_type, setting_id, device_db)
            return

        if cat_type == ddr_des.DDRAdvConfig.CatType.fpga:
            self.build_default_fpga_setting(mem_type, setting_id, device_db)
            return

        if cat_type == ddr_des.DDRAdvConfig.CatType.mem:
            self.build_default_memory_setting(mem_type, setting_id, device_db)
            return

        if cat_type == ddr_des.DDRAdvConfig.CatType.mem_timing:
            self.build_default_memory_timing_setting(mem_type, setting_id, device_db)
            return

    def get_preset_service(self, device_db):
        return DDRPresetServiceAdv(device_db)

    def build_default_fpga_setting(self, mem_type, setting_id=None, device_db=None):
        '''
        The Titanium DDR memory settings are slightly different than Trion
        where it has parameters that depend on other parameters and they
        may be only applicable to a fixed list of parameters.

        :param mem_type:
        :param setting_id:
        :param device_db:
        :return:
        '''
        preset_service = self.get_preset_service(device_db)
        # Parameter count changes based on type
        self._cs_adv_fpga = GenericParamGroup()

        if device_db is None:  # mockup
            preset_service.enable_mockup()

        adv_dev_svc = preset_service.get_adv_service(self.CatType.fpga)
        param_name_list = adv_dev_svc.get_param(mem_type)

        for param_info in param_name_list:
            param_name, _ = param_info
            def_value = adv_dev_svc.get_default(param_name, mem_type, setting_id)
            if "VREF_RANGE0" in param_name or "VREF_RANGE1" in param_name:
                if def_value == "":
                    fp_value = 0.00
                else:
                    fp_value = float(def_value)

                self._cs_adv_fpga.add_param(param_name, fp_value, GenericParam.DataType.dflo)
                self.logger.debug("Default Memory {} to {}".format(param_name, fp_value))
            else:
                self._cs_adv_fpga.add_param(param_name, def_value)
                self.logger.debug("Default Memory {} to {}".format(param_name, def_value))

    def build_default_memory_setting(self, mem_type, setting_id=None, device_db=None):
        '''
        The Titanium DDR memory settings are slightly different than Trion
        where it has parameters that depend on other parameters and they
        may be only applicable to a fixed list of parameters.

        :param mem_type:
        :param setting_id:
        :param device_db:
        :return:
        '''
        preset_service = self.get_preset_service(device_db)
        # Parameter count changes based on type
        self._cs_adv_memory = GenericParamGroup()

        if device_db is None:  # mockup
            preset_service.enable_mockup()

        adv_dev_svc = preset_service.get_adv_service(self.CatType.mem)
        param_name_list = adv_dev_svc.get_param(mem_type)
        
        for param_info in param_name_list:
            param_name, _ = param_info
            def_value = adv_dev_svc.get_default(param_name, mem_type, setting_id)
            if "VREF_RANGE0" in param_name or "VREF_RANGE1" in param_name:
                if def_value == "":
                    fp_value = 0.0
                else:
                    fp_value = float(def_value)

                self._cs_adv_memory.add_param(param_name, fp_value, GenericParam.DataType.dflo)
                self.logger.debug("Default Memory {} to {}".format(param_name, fp_value))
            else:
                self._cs_adv_memory.add_param(param_name, def_value)
                self.logger.debug("Default Memory {} to {}".format(param_name, def_value))

    def build_default_memory_timing_setting(self, mem_type, setting_id=None, device_db=None):

        preset_service = self.get_preset_service(device_db)
        self._cs_adv_memory_timing.clear()

        if device_db is None:  # mockup
            preset_service.enable_mockup()

        adv_dev_svc = preset_service.get_adv_service(self.CatType.mem_timing)
        param_name_list = adv_dev_svc.get_param(mem_type)

        for param_info in param_name_list:
            param_name, _ = param_info
            def_value = adv_dev_svc.get_default(param_name, mem_type, setting_id)
            ptype = GenericParam.DataType.dflo

            if def_value == "":
                fp_value = 0.0
            else:
                if param_name in ["tCCD", "tCCDMW", "tPPD"]:
                    fp_value = int(def_value)
                    ptype = GenericParam.DataType.dint
                else:
                    fp_value = float(def_value)

            self._cs_adv_memory_timing.add_param(param_name, fp_value, ptype)
            self.logger.debug("Default Memory {} to {}".format(param_name, def_value))

    def build_default_ctrl_setting(self, mem_type, setting_id=None, device_db=None):
        raise NotImplementedError("Advance Controller Setting is not supported")

    def build_default_gate_delay_setting(self, mem_type, setting_id=None, device_db=None):
        raise NotImplementedError("Gate Delay Tuning Setting is not supported")

    def get_setting(self, cat_type=ddr_des.DDRAdvConfig.CatType.fpga):
        """
        Get parameters for target setting

        :param cat_type: Advanced setting category type
        :return: If valid, parameter group for the advanced setting, else, None
        """

        if cat_type == ddr_des.DDRAdvConfig.CatType.fpga:
            return self._cs_adv_fpga

        if cat_type == ddr_des.DDRAdvConfig.CatType.mem:
            return self._cs_adv_memory

        if cat_type == ddr_des.DDRAdvConfig.CatType.mem_timing:
            return self._cs_adv_memory_timing

        return None

    def is_gate_delay_override_enabled(self):
        raise NotImplementedError("Gate Delay Tuning Setting is not supported")

    def set_gate_delay_override_enabled(self, is_enable, mem_type, setting_id, device_db):
        raise NotImplementedError("Gate Delay Tuning Setting is not supported")

    def set_default_gate_delay_parameters(self, mem_type, setting_id=None, device_db=None):
        raise NotImplementedError("Gate Delay Tuning Setting is not supported")

    def clear_gate_delay_parameters(self, mem_type, setting_id, device_db):
        raise NotImplementedError("Gate Delay Tuning Setting is not supported")

    def get_advanced_setting_id_adv(self, device_db, memory_type,
                                data_width, memory_density, physical_rank):

        preset_service = self.get_preset_service(device_db)

        def_setting_id = preset_service.get_default_setting_id(
            memory_type, data_width, "", memory_density, "", physical_rank=physical_rank)

        return def_setting_id

    def set_read_latency_enabled(self, is_enable: bool, mem_type, device_db, setting_id: int):
        '''
        Set the Read latency RL_EN_RL_DBI_RD parameter
        :param is_enable:
        :param memory_type:
        :param device_db:
        :param setting_id: Integer indicating the advance setting id that goes
                together with the current basic memory setting
        :return:
        '''
        if is_enable:
            self.set_setting_param("RL_DBI_READ", "Yes", self.CatType.mem)
        else:
            self.set_setting_param("RL_DBI_READ", "No", self.CatType.mem)

        self.set_read_latency_parameters(mem_type, setting_id, is_enable, device_db)

    def set_read_latency_parameters(self, mem_type, setting_id, is_enable, device_db):

        if is_enable:
            param_name = "RL_DBI_READ_ENABLED"
        else:
            param_name = "RL_DBI_READ_DISABLED"

        if param_name != "":
            self.set_default_adv_memory_setting(device_db, mem_type, setting_id, self.CatType.mem, param_name)

    def set_write_latency(self, wl_set: str, mem_type, device_db, setting_id: int):
        '''
        Set the write latency WL_SET parameter
        :param wl_set: string
        :param mem_type:
        :param device_db:
        :param setting_id:
        :return:
        '''

        if wl_set == "Set A":
            self.set_setting_param("WL_SET", "Set A", self.CatType.mem)

        else:
            self.set_setting_param("WL_SET", "Set B", self.CatType.mem)

        self.set_write_latency_parameters(mem_type, setting_id, wl_set, device_db)

    def set_write_latency_parameters(self, mem_type, setting_id, wl_set, device_db):


        if wl_set == "Set A":
            param_name = "WL_SET_A"

        else:
            param_name = "WL_SET_B"

        if param_name != "":
            self.set_default_adv_memory_setting(device_db, mem_type, setting_id, self.CatType.mem, param_name)

    def set_fpga_vref_setting(self, range_type: str, mem_type, device_db, setting_id):

        self.set_setting_param("MEM_FPGA_VREF_RANGE", range_type, self.CatType.fpga)

        if range_type == "Range 1":
            param_name = "FPGA_VREF_RANGE1"
        else:
            param_name = "FPGA_VREF_RANGE0"

        self.set_default_adv_memory_setting(device_db, mem_type, setting_id, self.CatType.fpga, param_name)

    def set_vref_setting(self, range_type: str, mem_type, physical_rank, device_db, setting_id: int,
                         is_ca):
        '''
        Set the write latency WL_SET parameter
        :param wl_set: string
        :param mem_type:
        :param device_db:
        :param setting_id:
        :return:
        '''
        if is_ca:
            param_name = "MEM_CA_RANGE"
        else:
            param_name = "MEM_DQ_RANGE"

        if range_type == "RANGE[1]":
            self.set_setting_param(param_name, "RANGE[1]", self.CatType.mem)

            if is_ca:
                adv_param_name = "CA_VREF_RANGE1"
            else:
                adv_param_name = "DQ_VREF_RANGE1"

        else:
            self.set_setting_param(param_name, "RANGE[0]", self.CatType.mem)

            if is_ca:
                adv_param_name = "CA_VREF_RANGE0"
            else:
                adv_param_name = "DQ_VREF_RANGE0"

        self.set_default_adv_memory_setting(device_db, mem_type, setting_id, self.CatType.mem, adv_param_name)

    def set_default_adv_memory_setting(self, device_db, mem_type, setting_id, cat_type, param_name):
        preset_service = self.get_preset_service(device_db)
        adv_dev_svc = preset_service.get_adv_service(cat_type)
        param_group = self.get_setting(cat_type)

        if param_group is not None and adv_dev_svc is not None:

            def_value = adv_dev_svc.get_default(param_name, mem_type, setting_id)
            param_group.set_param_value(param_name, def_value)

    def set_adv_setting_parameters(self, mem_type, setting_id, param_names, device_db, cat_type):
        preset_service = self.get_preset_service(device_db)
        adv_dev_svc = preset_service.get_adv_service(cat_type)
        param_group = self.get_setting(cat_type)

        if param_group is not None and adv_dev_svc is not None:

            for param in param_names:

                def_value = adv_dev_svc.get_default(param, mem_type, setting_id)
                param_group.set_param_value(param, def_value)

    def get_fpga_vref_value(self) -> float:
        fpga_param_group = self.get_setting(self.CatType.fpga)

        if fpga_param_group is not None:
            vref_parent_param = self.get_fpga_vref_range_param()

            if vref_parent_param.value == "Range 0":
                return fpga_param_group.get_param_value("FPGA_VREF_RANGE0")

            else:
                return fpga_param_group.get_param_value("FPGA_VREF_RANGE1")

        return None

    def get_fpga_vref_range_param(self):
        fpga_param_group = self.get_setting(self.CatType.fpga)
        return fpga_param_group.get_param_by_name("MEM_FPGA_VREF_RANGE")

    def get_mem_ca_vref_value(self) -> float:
        mem_param_group = self.get_setting(self.CatType.mem)

        if mem_param_group is not None:
            vref_parent_param = mem_param_group.get_param_by_name("MEM_CA_RANGE")

            if vref_parent_param.value == "RANGE[0]":
                return mem_param_group.get_param_value("CA_VREF_RANGE0")

            else:
                return mem_param_group.get_param_value("CA_VREF_RANGE1")

        return None


    def get_mem_dq_vref_value(self) -> float:
        mem_param_group = self.get_setting(self.CatType.mem)

        if mem_param_group is not None:
            vref_parent_param = mem_param_group.get_param_by_name("MEM_DQ_RANGE")

            if vref_parent_param.value == "RANGE[0]":
                return mem_param_group.get_param_value("DQ_VREF_RANGE0")

            else:
                return mem_param_group.get_param_value("DQ_VREF_RANGE1")

        return None


@gen_util.freeze_it
class DDRPinSwap:
    """
    Saving pin swapping data
    """
    DQ_GROUP_COUNT = 8 # Number of dq in a group
    CA_GROUP_COUNT = 6 # Number of ca in a group
    is_read_only = False
    pin_swap_info: Optional[Dict[str, str]] = None

    def __init__(self) -> None:
        self._gen_param = GenericParamGroup()
        self.num_of_dq = 32 # Total number of DQ pins
        self.num_of_dm = self.num_of_dq // self.DQ_GROUP_COUNT # Every DQ/DM group should have 1 dm
        self.num_of_ca = 6 # Total number of CA pins

        self._param_info = self.build_param_info()
        self.build_param_group()

    @property
    def gen_param(self):
        return self._gen_param

    @gen_param.setter
    def gen_param(self, val: GenericParamGroup):
        self._gen_param = val

    @property
    def param_info(self):
        return self._param_info

    def build_param_info(self):
        param_info = DDRParamInfo()

        dq_gp_count = self.num_of_dm
        gp_num2options = {}

        # DM/DQ options
        for gp_num in range(dq_gp_count):
            gp_num2options[gp_num] = self.get_group_options(gp_num)

        # DQ
        for num in range(self.num_of_dq):
            gp_num = num // self.DQ_GROUP_COUNT
            prop_name = f"{DDRPkgPinType.DQ.value}[{num}]"
            prop_data = PropertyMetaData.PropData(
                id=prop_name,
                name=prop_name,
                data_type=GenericParam.DataType.dstr,
                default=prop_name,
                valid_setting=gp_num2options[gp_num],
                disp_name=prop_name,
                category=DDRPkgPinType.DQ.value
            )
            param_info.add_prop_by_data(prop_data.id, prop_data)

        # DM
        for num in range(self.num_of_dm):
            prop_name = f"{DDRPkgPinType.DM.value}[{num}]"
            prop_data = PropertyMetaData.PropData(
                id=prop_name,
                name=prop_name,
                data_type=GenericParam.DataType.dstr,
                default=prop_name,
                valid_setting=gp_num2options[num],
                disp_name=prop_name,
                category=DDRPkgPinType.DM.value
            )
            param_info.add_prop_by_data(prop_data.id, prop_data)

        # CA
        ca_options = self.get_ca_group_options()
        for num in range(self.num_of_ca):
            prop_name = f"{DDRPkgPinType.CA.value}[{num}]"
            prop_data = PropertyMetaData.PropData(
                id=prop_name,
                name=prop_name,
                data_type=GenericParam.DataType.dstr,
                default=prop_name,
                valid_setting=ca_options,
                disp_name=prop_name,
                category=DDRPkgPinType.CA.value
            )
            param_info.add_prop_by_data(prop_data.id, prop_data)

        # Add is_enable
        prop_data = PropertyMetaData.PropData(
                id=DDRParamInfo.Id.is_enable_pin_swap,
                name="ENABLE_PIN_SWAP",
                data_type=GenericParam.DataType.dbool,
                default=False,
                disp_name="Enable Package Pin Swapping",
                category="base"
            )
        param_info.add_prop_by_data(prop_data.id, prop_data)
        return param_info

    @property
    def is_enable_swap(self) -> bool:
        prop_name = self.param_info.get_prop_name(self.param_info.Id.is_enable_pin_swap)
        is_enable = self.gen_param.get_param_value(prop_name)
        return is_enable

    @is_enable_swap.setter
    def is_enable_swap(self, val: bool):
        prop_name = self.param_info.get_prop_name(self.param_info.Id.is_enable_pin_swap)
        self.gen_param.set_param_value(prop_name, val)

    def build_param_group(self):
        for param_info in self._param_info.get_all_prop():
            self._gen_param.add_param(param_info.name, param_info.default, param_info.data_type)

    def set_default_setting(self, device_db: Optional[PeripheryDevice] = None):
        if device_db is not None:
            if device_db.get_pin_swap() is not None:
                self.is_read_only = True
                self.pin_swap_info = device_db.get_pin_swap()
            else:
                self.is_read_only = False
                self.pin_swap_info = None

        pin_swap_info = self.pin_swap_info
        self.is_read_only = pin_swap_info is not None

        # Ensure fixed pin swap information will be used in later lpf writer
        if self.is_read_only:
            self.is_enable_swap = True

        for num in range(self.num_of_dq):
            param_name = f"DQ[{num}]"
            val = param_name

            if pin_swap_info is not None:
                val = pin_swap_info.get(param_name, val)

            self._gen_param.set_param_value(name=param_name, value=val)

        for num in range(self.num_of_dm):
            param_name = f"DM[{num}]"
            val = param_name

            if pin_swap_info is not None:
                val = pin_swap_info.get(param_name, val)

            self._gen_param.set_param_value(name=param_name, value=val)

        for num in range(self.num_of_ca):
            param_name = f"CA[{num}]"
            val = param_name

            if pin_swap_info is not None:
                val = pin_swap_info.get(param_name, val)

            self._gen_param.set_param_value(name=param_name, value=val)

    def get_dq_dm_pin_pos_by_pin_name(self, gp_num: int, pin_name: str):
        pos = ""
        for param in self.get_group_params(gp_num):
            if pin_name == param.value:
                pos = param.name
                break

        return pos

    def get_ca_pin_pos_by_pin_name(self, pin_name: str):
        pos = ""
        for param in self.get_ca_params():
            if pin_name == param.value:
                pos = param.name
                break

        return pos

    def get_group_params(self, group_num: int) -> List[GenericParam]:
        """
        Get a list of DQ/DM params by group number
        The order of group params will affect UI display and API ordering

        :param group_num: Group number
        :type group_num: int
        :return: List of DQ/DM params
        :rtype: List[GenericParam]
        """
        options: List[GenericParam] = []

        # DQ pins
        for num in range(self.DQ_GROUP_COUNT):
            dq_index = num + self.DQ_GROUP_COUNT * group_num
            param = self._gen_param.get_param_by_name(f"DQ[{dq_index}]")
            assert param is not None
            options.append(param)

        # DM pin
        param = self._gen_param.get_param_by_name(f"DM[{group_num}]")
        assert param is not None
        options.append(param)
        return options

    def get_mapping_by_pin_type(self, pin_type: DDRPkgPinType, ddr_width: int = 16):
        mapping = []
        reset_param_list = []

        if ddr_width == 16:
            match pin_type:
                case DDRPkgPinType.DQ:
                    reset_param_list = [f"DQ[{i}]" for i in range(16, 33)]
                case DDRPkgPinType.DM:
                    reset_param_list = [f"DM[{i}]" for i in range(2, 4)]

        for prop in self.param_info.get_prop_by_category(pin_type.value):
            param = self.gen_param.get_param_by_name(prop.name)
            assert param is not None

            if param.name in reset_param_list:
                value = self.param_info.get_default(param.name)
            else:
                value = param.value

            assert isinstance(value, str)
            mapping.append(value.lower())

        return mapping

    def get_ca_params(self) -> List[GenericParam]:
        """
        Get a list of CA params by group number
        The order of group params will affect UI display and API ordering

        :param group_num: Group number
        :type group_num: int
        :return: List of CA params
        :rtype: List[GenericParam]
        """
        options: List[GenericParam] = []

        for num in range(self.num_of_ca):
            param = self._gen_param.get_param_by_name(f"CA[{num}]")
            assert param is not None
            options.append(param)

        return options

    @staticmethod
    def get_group_options(group_num: int) -> List[str]:
        options: List[str] = []

        # DQ pins
        for num in range(DDRPinSwap.DQ_GROUP_COUNT):
            dq_index = num + DDRPinSwap.DQ_GROUP_COUNT * group_num
            options.append(f"DQ[{dq_index}]")

        # DM pin
        options.append(f"DM[{group_num}]")

        return options

    @staticmethod
    def get_ca_group_options() -> List[str]:
        options: List[str] = []

        for num in range(DDRPinSwap.CA_GROUP_COUNT):
            options.append(f"CA[{num}]")

        return options

    def __str__(self, *args, **kwargs):
        info = 'is_enable_swap:{} is_read_only:{} param_group:{}' \
            .format(self.is_enable_swap,
                    self.is_read_only, self.gen_param.to_string())

        return info


@gen_util.freeze_it
class DDRAxiTargetAdv(ddr_des.DDRAxiTarget):
    """
    Derived from the base DDRAxiTarget with the addition of new
    data member, ParamInfo
    """

    # Property meta data
    _param_info = None

    def __init__(self, target_id, apply_default=True):

        if self._param_info is None:
            self._param_info = DDRParamInfo()
            self.build_param_info(self._param_info)

        self.param_group = None
        self.build_param()

        super().__init__(target_id, apply_default)

    def __str__(self, *args, **kwargs):
        info = '{} target_id:{} is_enabled:{} param_group:{}' \
            .format(super(), self.target_id,
                    self.is_enabled, self.param_group.to_string())

        return info

    def build_param(self):
        """
        Build parameter storage
        """
        self.param_group = GenericParamGroup()
        for param_info in self._param_info.get_all_prop():
            self.param_group.add_param(param_info.name, param_info.default, param_info.data_type)

    def build_param_info(self, param_info: DDRParamInfo):
        """
        Build information about supported properties

        :param param_info: Property info
        """
        if param_info is not None:
            param_info.add_prop(DDRParamInfo.Id.axi_width_256, "AXI_WIDTH_256", GenericParam.DataType.dbool,
                                False, None, "Enable 256-bit AXI Width")
            param_info.add_prop(DDRParamInfo.Id.axi_enabled, "AXI_ENABLE", GenericParam.DataType.dbool,
                                True, None, "Enable AXI Target")

    def get_param_info(self) -> DDRParamInfo:
        return self._param_info

    def get_param_group(self) -> GenericParamGroup:
        return self.param_group

    @property
    def is_axi_width_256(self):
        return self.param_group.get_param_value("AXI_WIDTH_256")

    @is_axi_width_256.setter
    def is_axi_width_256(self, value):
        self.param_group.set_param_value("AXI_WIDTH_256", value)

    # We're overwriting the is_enabled flag in the base class with the
    # Generic Param usage
    @property
    def is_enabled(self):
        return self.param_group.get_param_value("AXI_ENABLE")

    @is_enabled.setter
    def is_enabled(self, value):
        self.param_group.set_param_value("AXI_ENABLE", value)

    def set_default_setting(self, device_db: PeripheryDevice = None):
        """
        Override
        """
        super().set_default_setting(device_db)

        param_service = GenericParamService(self.param_group, self._param_info)
        param_service.set_param_value(
            DDRParamInfo.Id.axi_width_256, self._param_info.get_default(DDRParamInfo.Id.axi_width_256))
        param_service.set_param_value(
            DDRParamInfo.Id.axi_enabled, self._param_info.get_default(DDRParamInfo.Id.axi_enabled))


@gen_util.freeze_it
class DDRAxiTarget0Adv(DDRAxiTargetAdv):
    """
    AXI Interface 0 design data for DDR Instance. Mostly generic simple pins.
    """

    def __init__(self, target_id, apply_default=True):
        super().__init__(target_id, apply_default)

    @staticmethod
    def build_port_info(device_db=None, is_mockup=False):
        """
        Override. Build port info for DDR block's AXI pins from definition in device db.

        :param device_db:
        :param is_mockup: True, build a mockup data, else build from device db
        """

        if is_mockup:
            PeriDesignGenPinItem.build_port_info(device_db, is_mockup)
        else:
            if device_db is None:
                return

            # Pull info from device db
            from device.db_interface import DeviceDBService
            from tx180_device.ddr.ddr_device_service_adv import DDRServiceAdvance

            dbi = DeviceDBService(device_db)
            dev_service = dbi.get_block_service(DeviceDBService.BlockType.DDR_ADV)
            DDRAxiTarget0Adv._device_port_map = dev_service.get_ports_by_class_group(
                DDRServiceAdvance.PortClassGroup.axi_target_0)

@gen_util.freeze_it
class DDRAxiTarget1Adv(DDRAxiTargetAdv):
    """
    AXI Interface 0 design data for DDR Instance. Mostly generic simple pins.
    """

    def __init__(self, target_id, apply_default=True):
        super().__init__(target_id, apply_default)

    @staticmethod
    def build_port_info(device_db=None, is_mockup=False):
        """
        Override. Build port info for DDR block's AXI pins from definition in device db.

        :param device_db:
        :param is_mockup: True, build a mockup data, else build from device db
        """

        if is_mockup:
            PeriDesignGenPinItem.build_port_info(device_db, is_mockup)
        else:
            if device_db is None:
                return

            # Pull info from device db
            from device.db_interface import DeviceDBService
            from tx180_device.ddr.ddr_device_service_adv import DDRServiceAdvance

            dbi = DeviceDBService(device_db)
            dev_service = dbi.get_block_service(DeviceDBService.BlockType.DDR_ADV)
            DDRAxiTarget1Adv._device_port_map = dev_service.get_ports_by_class_group(
                DDRServiceAdvance.PortClassGroup.axi_target_1)

@gen_util.freeze_it
class DDRConfigPinsBase(PeriDesignGenPinItem):
    """
    This is a class used for the 3 different config pages for DDR which only
    contains pins configuration.
    """
    def __init__(self, apply_default=True):
        super().__init__()

        if apply_default:
            self.set_default_setting()

    def create_chksum(self):
        pass

    def set_default_setting(self, device_db=None):
        """

        :param device_db:
        :return:
        """
        # Nothing to set
        pass

@gen_util.freeze_it
class DDRController(DDRConfigPinsBase):

    @staticmethod
    def build_port_info(device_db=None, is_mockup=False):
        """
        Override. Build port info for DDR block's AXI pins from definition in device db.

        :param device_db:
        :param is_mockup: True, build a mockup data, else build from device db
        """

        if is_mockup:
            PeriDesignGenPinItem.build_port_info(device_db, is_mockup)
        else:
            if device_db is None:
                return

            # Pull info from device db
            from device.db_interface import DeviceDBService
            from tx180_device.ddr.ddr_device_service_adv import DDRServiceAdvance

            dbi = DeviceDBService(device_db)
            dev_service = dbi.get_block_service(DeviceDBService.BlockType.DDR_ADV)
            DDRController._device_port_map = dev_service.get_ports_by_class_group(
                DDRServiceAdvance.PortClassGroup.controller)

    def generate_pin_name_from_inst(self, inst_name):
        """
        Override
        """
        super().generate_pin_name_from_inst(inst_name)

        # Clock from design does not have auto-gen name
        pin_name_list = ["CTRL_CLK"]
        for name in pin_name_list:
            pin = self.gen_pin.get_pin_by_type_name(name)
            if pin is not None:
                pin.name = ""

@gen_util.freeze_it
class DDRControllerRegisterInterface(DDRConfigPinsBase):

    _param_info = None

    def __init__(self, apply_default=True):

        if self._param_info is None:
            self._param_info = DDRParamInfo()
            self.build_param_info(self._param_info)

        self.param_group = None
        self.build_param()

        super().__init__(apply_default)

    def build_param(self):
        """
        Build parameter storage
        """
        self.param_group = GenericParamGroup()
        for param_info in self._param_info.get_all_prop():
            self.param_group.add_param(param_info.name, param_info.default, param_info.data_type)

    def build_param_info(self, param_info: DDRParamInfo):
        """
        Build information about supported properties

        :param param_info: Property info
        """
        if param_info is not None:
            param_info.add_prop(DDRParamInfo.Id.reg_inf_ena, "REG_ENA", GenericParam.DataType.dbool,
                                False, None, "Enable Configuration Register Interface")

    def get_param_info(self) -> DDRParamInfo:
        return self._param_info

    def get_param_group(self) -> GenericParamGroup:
        return self.param_group

    @property
    def is_reg_ena(self):
        return self.param_group.get_param_value("REG_ENA")

    @is_reg_ena.setter
    def is_reg_ena(self, value):
        self.param_group.set_param_value("REG_ENA", value)

    def set_default_setting(self, device_db: PeripheryDevice = None):
        """
        Override
        """
        super().set_default_setting(device_db)
        #self.is_reg_ena = self._param_info.get_default(DDRParamInfo.Id.reg_inf_ena)

        param_service = GenericParamService(self.param_group, self._param_info)
        param_service.set_param_value(
            DDRParamInfo.Id.reg_inf_ena, self._param_info.get_default(DDRParamInfo.Id.reg_inf_ena))

    def generate_pin_name_from_inst(self, inst_name):
        """
        Override
        """
        super().generate_pin_name_from_inst(inst_name)

        # Clock from design does not have auto-gen name
        pin_name_list = ["CR_ACLK"]
        for name in pin_name_list:
            pin = self.gen_pin.get_pin_by_type_name(name)
            if pin is not None:
                pin.name = ""

    @staticmethod
    def build_port_info(device_db=None, is_mockup=False):
        """
        Override. Build port info for DDR block's AXI pins from definition in device db.

        :param device_db:
        :param is_mockup: True, build a mockup data, else build from device db
        """

        if is_mockup:
            PeriDesignGenPinItem.build_port_info(device_db, is_mockup)
        else:
            if device_db is None:
                return

            # Pull info from device db
            from device.db_interface import DeviceDBService
            from tx180_device.ddr.ddr_device_service_adv import DDRServiceAdvance

            dbi = DeviceDBService(device_db)
            dev_service = dbi.get_block_service(DeviceDBService.BlockType.DDR_ADV)
            DDRControllerRegisterInterface._device_port_map = dev_service.get_ports_by_class_group(
                DDRServiceAdvance.PortClassGroup.ctrl_reg_inf)

@gen_util.freeze_it
class DDRConfigControl(DDRConfigPinsBase):

    @staticmethod
    def build_port_info(device_db=None, is_mockup=False):
        """
        Override. Build port info for DDR block's AXI pins from definition in device db.

        :param device_db:
        :param is_mockup: True, build a mockup data, else build from device db
        """

        if is_mockup:
            PeriDesignGenPinItem.build_port_info(device_db, is_mockup)
        else:
            if device_db is None:
                return

            # Pull info from device db
            from device.db_interface import DeviceDBService
            from tx180_device.ddr.ddr_device_service_adv import DDRServiceAdvance

            dbi = DeviceDBService(device_db)
            dev_service = dbi.get_block_service(DeviceDBService.BlockType.DDR_ADV)
            DDRConfigControl._device_port_map = dev_service.get_ports_by_class_group(
                DDRServiceAdvance.PortClassGroup.cfg_control)

@gen_util.freeze_it
class DDRAdvance(PeriDesignItem):
    """
    A ddr block instance.

    Either one or both of the AXI target can be used at a time.
    """

    _param_info = None

    # Adding this so that the parameter that has options dependent
    # on device is able to get access to it
    dev_service = None

    class ConfigClkDiv(Enum):
        div2 = auto()
        div4 = auto()
        div8 = auto()
        div16 = auto()

    clkdiv2str_map = {
        ConfigClkDiv.div2: "2",
        ConfigClkDiv.div4: "4",
        ConfigClkDiv.div8: "8",
        ConfigClkDiv.div16: "16"
    }

    str2clkdiv_map = {
        "2": ConfigClkDiv.div2,
        "4": ConfigClkDiv.div4,
        "8": ConfigClkDiv.div8,
        "16": ConfigClkDiv.div16
    }

    def __init__(self, name, ddr_def="", apply_default=True):
        super().__init__()

        self.name = name
        self.ddr_def = ddr_def
        self.axi_target0 = DDRAxiTarget0Adv(DDRAxiTargetAdv.TargetIdType.axi_0)
        self.axi_target1 = DDRAxiTarget1Adv(DDRAxiTargetAdv.TargetIdType.axi_1)

        self.controller = DDRController()
        self.ctrl_reg_inf = DDRControllerRegisterInterface()
        self.cfg_control = DDRConfigControl()

        if self._param_info is None:
            self._param_info = DDRParamInfo()
            self.build_param_info(self._param_info)

        self.param_group = None
        self.build_param()

        # NOT USED: Preset id
        # self.preset_id = ""
        # Advance configuration setting
        self.adv_config = DDRAdvConfigAdvance()
        self.pin_swap = DDRPinSwap()

        self.logger = Logger

        if apply_default:
            self.set_default_setting()

    def __str__(self, *args, **kwargs):
        if self.param_group is not None:
            info = "name:{} ddr_def:{} param: {}" \
                .format(self.name, self.ddr_def,
                        self.param_group.to_string())
        else:
            info = "name:{} ddr_def:{}" \
                .format(self.name, self.ddr_def)

        return info

    @staticmethod
    def build_device_info(device_db: Optional[PeripheryDevice]=None):

        if device_db is not None:
            from device.db_interface import DeviceDBService

            dbi = DeviceDBService(device_db)
            dev_service = dbi.get_block_service(dbi.BlockType.DDR_ADV) # type: Optional[DDRServiceAdvance]
            assert dev_service is not None
            DDRAdvance.dev_service = dev_service

    @staticmethod
    def get_data_width_options() -> List[int]:
        width_options: List[int] = []

        if DDRAdvance.dev_service is None:
            # Hard code since at this point, the device data is not ready
            return [16, 32]

        width_options = DDRAdvance.dev_service.get_data_width_options()
        width_options.sort()
        return width_options

    @staticmethod
    def get_default_data_width():
        width_options: List[int] = DDRAdvance.get_data_width_options()

        if 32 in width_options:
            return 32

        return 16

    @staticmethod
    def get_clkin_options() -> List[int]:
        width_options: List[int] = []

        if DDRAdvance.dev_service is None:
            # Hard code since at this point, the device data is not ready
            return [0, 1, 2]

        clkin_sel_options = DDRAdvance.dev_service.get_clkin_sel_list()
        clkin_sel_options.sort()
        return clkin_sel_options
    
    @staticmethod
    def get_memory_type_options() -> List[str]:
        if DDRAdvance.dev_service is None:
            return ['LPDDR4', 'LPDDR4x']
        memory_options: List[str] = DDRAdvance.dev_service.get_memory_type_options()
        return memory_options

    @staticmethod
    def get_default_memory_type():
        memory_options: List[str] = DDRAdvance.get_memory_type_options()
        if 'LPDDR4' in memory_options:
            return 'LPDDR4'
        return 'LPDDR4x'

    @staticmethod
    def get_default_memory_density():
        if DDRAdvance.dev_service is None:
            return '4G'
        memory_density: str = DDRAdvance.dev_service.get_memory_density_specific_default()
        if memory_density != "":
            return memory_density        
        return '4G'

    def build_param(self):
        """
        Build parameter storage
        """
        self.param_group = GenericParamGroup()
        for param_info in self._param_info.get_all_prop():
            self.param_group.add_param(param_info.name, param_info.default, param_info.data_type)

    def build_param_info(self, param_info: DDRParamInfo):
        """
        Build information about supported properties

        :param param_info: Property info
        """

        if param_info is not None:
            param_info.add_prop(DDRParamInfo.Id.clkin_sel, "CLKIN_SEL", GenericParam.DataType.dint,
                                2, self.get_clkin_options(), "DDR CLKIN Select")
            # This value is dependent also on the supported feature on a device. 16 is applicable to all
            # since 32 is only available on the device that supports it.
            param_info.add_prop(DDRParamInfo.Id.data_width, "DATA_WIDTH", GenericParam.DataType.dint,
                                self.get_default_data_width(), self.get_data_width_options(), "Data Width")
            param_info.add_prop(DDRParamInfo.Id.memory_type, "MEMORY_TYPE", GenericParam.DataType.dstr,
                                self.get_default_memory_type(), self.get_memory_type_options(), "Memory Type")
            param_info.add_prop(DDRParamInfo.Id.memory_density, "MEMORY_DENSITY", GenericParam.DataType.dstr,
                                self.get_default_memory_density(), ["1G", "2G", "3G", "4G", "6G", "8G", "12G", "16G"], "Memory Density (Per Channel)")
            param_info.add_prop(DDRParamInfo.Id.physical_rank, "PHYSICAL_RANK", GenericParam.DataType.dint,
                                1, [1, 2], "Physical Rank")

    def enumtype2str(self, prop_id, value):
        type2str_map = None

        if type2str_map is not None:
            return type2str_map.get(value, None)

        return None

    def str2enumtype(self, prop_id, value):
        str2enum_map = None

        if str2enum_map is not None:
            return str2enum_map.get(value, None)

        return None

    def get_auto_param_names(self):
        return ["DATA_WIDTH", "MEMORY_TYPE", "PHYSICAL_RANK", "MEMORY_DENSITY"]

    @property
    def clkin_sel(self):
        return self.param_group.get_param_value("CLKIN_SEL")

    @clkin_sel.setter
    def clkin_sel(self, value):
        self.param_group.set_param_value("CLKIN_SEL", value)

    def get_param_info(self) -> DDRParamInfo:
        """
        Get defined parameter info

        :return: Parameter info
        """
        return self._param_info

    def get_param_group(self) -> GenericParamGroup:
        """
        Get parameter data

        :return: Parameter data
        """
        return self.param_group

    def get_device(self):
        """
        Override
        """
        return self.ddr_def

    def set_device(self, device_name):
        """
        Override
        """
        self.ddr_def = device_name

    def set_default_data_width(self, param_service: GenericParamService, device_db: Optional[PeripheryDevice] = None):
        # Default for data width should be 16 for device that doesn't support 32. Else
        # it should be 32
        if device_db is not None:
            from device.db_interface import DeviceDBService

            dbi = DeviceDBService(device_db)
            dev_service = dbi.get_block_service(DeviceDBService.BlockType.DDR_ADV)
            valid_width_options = dev_service.get_data_width_options()

            if len(valid_width_options) == 1:
                param_service.set_param_value(
                    DDRParamInfo.Id.data_width, valid_width_options[0])
            else:
                # If there's more than one, then we just use the default value
                param_service.set_param_value(
                    DDRParamInfo.Id.data_width, self._param_info.get_default(DDRParamInfo.Id.data_width))

        else:
            param_service.set_param_value(
                DDRParamInfo.Id.data_width, self._param_info.get_default(DDRParamInfo.Id.data_width))

    def set_default_setting(self, device_db: Optional[PeripheryDevice] = None):
        """
        Override
        """
        param_service = GenericParamService(self.param_group, self._param_info)
        param_service.set_param_value(
            DDRParamInfo.Id.clkin_sel, self._param_info.get_default(DDRParamInfo.Id.clkin_sel))
        param_service.set_param_value(
            DDRParamInfo.Id.physical_rank, self._param_info.get_default(DDRParamInfo.Id.physical_rank))
        param_service.set_param_value(
            DDRParamInfo.Id.memory_type, self._param_info.get_default(DDRParamInfo.Id.memory_type))
        param_service.set_param_value(
            DDRParamInfo.Id.memory_density, self._param_info.get_default(DDRParamInfo.Id.memory_density))

        # Default for data width should be 16 for device that doesn't support 32. Else
        # it should be 32
        self.set_default_data_width(param_service, device_db)
        self.pin_swap.set_default_setting(device_db)

    def set_target_enabled(self, target_id, is_enabled):
        """
        Enable/disable AXI targets

        :param target_id: Target to enable/disable, TargetIdType
        :param is_enabled: True, enable else disable
        """
        target = None
        if target_id == DDRAxiTargetAdv.TargetIdType.axi_0:
            target = self.axi_target0
        elif target_id == DDRAxiTargetAdv.TargetIdType.axi_1:
            target = self.axi_target1

        if target is not None:
            target.set_enabled(is_enabled)

    def update_memory_type(self, mem_type, device_db, is_preset=False):
        self.set_param_value(DDRParamInfo.Id.memory_type, mem_type)

        if is_preset:
            def_setting_id = self.adv_config.get_advanced_setting_id_adv(
                device_db, self.get_memory_type(),
                self.get_param_value(DDRParamInfo.Id.data_width),
                self.get_param_value(DDRParamInfo.Id.memory_density),
                self.get_param_value(DDRParamInfo.Id.physical_rank))
        else:
            # If we are updating memory not because of preset change,
            # then just take the adv_comb_id = 0 which contains the
            # memory type default value (unassociated to preset).
            def_setting_id = "0"

        self.adv_config.set_default_setting(mem_type, def_setting_id, device_db)

    def build_default_configuration(self, device_db=None):
        """
        Build default structure and apply default setting.
        By default, it builds all except for advanced setting.
        To build advanced setting, specify a device db.

        Typucally this function is called right after
        a DDR instance is created.

        :param device_db: Device db

        .. note::
           This kind of function is useful for block type that
           requires complex setup instead of just setting default
           values.
        """

        # Build both AXI targets structure
        self.set_target_enabled(DDRAxiTargetAdv.TargetIdType.axi_0, True)
        self.set_target_enabled(DDRAxiTargetAdv.TargetIdType.axi_1, True)
        # Build pin info from device db
        self.build_generic_pin()
        # Generate all pin name
        self.generate_pin_name()
        # Default setting may clear pin name
        self.set_default_setting(device_db)

        # Setup the advance config setting
        if device_db is not None:
            self.update_memory_type(self.get_memory_type(), device_db=device_db)

    def create_chksum(self):
        pass

    def set_target_enabled(self, target_id, is_enabled):
        """
        Enable/disable AXI targets

        :param target_id: Target to enable/disable, TargetIdType
        :param is_enabled: True, enable else disable
        """
        target = None
        if target_id == DDRAxiTargetAdv.TargetIdType.axi_0:
            target = self.axi_target0
        elif target_id == DDRAxiTargetAdv.TargetIdType.axi_1:
            target = self.axi_target1

        if target is not None:
            target.set_enabled(is_enabled)

    def build_generic_pin(self):
        """
        Build simple pins design.

        For AXI target, it only build enabled target
        """
        if self.axi_target0.is_enabled:
            self.axi_target0.build_generic_pin()

        if self.axi_target1.is_enabled:
            self.axi_target1.build_generic_pin()

        self.controller.build_generic_pin()
        self.ctrl_reg_inf.build_generic_pin()
        self.cfg_control.build_generic_pin()

    def get_adv_setting_id(self, device_db: PeripheryDevice):
        setting_id = None
        param_service = GenericParamService(
            self.get_param_group(), self.get_param_info())

        memory_type = param_service.get_param_value(DDRParamInfo.Id.memory_type)
        data_width = param_service.get_param_value(DDRParamInfo.Id.data_width)
        memory_density = param_service.get_param_value(DDRParamInfo.Id.memory_density)
        physical_rank = param_service.get_param_value(DDRParamInfo.Id.physical_rank)

        setting_id = self.adv_config.get_advanced_setting_id_adv(
            device_db, memory_type, data_width, memory_density, physical_rank)

        return setting_id

    def generate_pin_name(self):
        """
        Override
        """
        if self.axi_target0.is_enabled:
            self.axi_target0.generate_pin_name_from_inst(self.name)

        if self.axi_target1.is_enabled:
            self.axi_target1.generate_pin_name_from_inst(self.name)

        self.controller.generate_pin_name_from_inst(self.name)
        self.ctrl_reg_inf.generate_pin_name_from_inst(self.name)
        self.cfg_control.generate_pin_name_from_inst(self.name)

    def refresh_pin_name(self, old_substr, new_substr):
        """
        Override
        """
        if self.axi_target0.is_enabled:
            self.axi_target0.refresh_pin_name(old_substr, new_substr)

        if self.axi_target1.is_enabled:
            self.axi_target1.refresh_pin_name(old_substr, new_substr)

        self.controller.refresh_pin_name(old_substr, new_substr)
        self.ctrl_reg_inf.refresh_pin_name(old_substr, new_substr)
        self.cfg_control.refresh_pin_name(old_substr, new_substr)

    def is_target_enabled(self, target_id):
        """
        Check if target 0 is enabled

        :return: True, if the target info is created, else False
        """
        target = None
        if target_id == DDRAxiTargetAdv.TargetIdType.axi_0:
            target = self.axi_target0
        elif target_id == DDRAxiTargetAdv.TargetIdType.axi_1:
            target = self.axi_target1

        if target is not None:
            return target.is_enabled

        return False

    def is_match_pin_name(self, text):
        """
        Override
        """
        is_match = False

        if self.controller is not None:
            is_match  = self.controller.is_match_pin_name(text)

        if not is_match:
            if self.ctrl_reg_inf is not None:
                is_match = self.ctrl_reg_inf.is_match_pin_name(text)

        if not is_match:
            if self.cfg_control is not None:
                is_match = self.cfg_control.is_match_pin_name(text)

        if not is_match:
            if self.axi_target0 is not None:
                is_match = self.axi_target0.is_match_pin_name(text)

        if not is_match:
            if self.axi_target1 is not None:
                is_match = self.axi_target1.is_match_pin_name(text)

        return is_match

    def find_phy_clock_info(self, ddr_dev_service: DDRServiceAdvance, pll_reg):
        """
        For the assigned DDR device resource, find the PLL resource name and its instance info.

        :param ddr_dev_service:
        :param pll_reg : PLL registry object
        :return: A tuple of resource, instance, clkout pin name, [(resource name, instance name, clkout pin name)]
        """
        if self.ddr_def == "" or ddr_dev_service is None or pll_reg is None:
            return None

        # Check which clkin was selected first
        info = None

        res_name_list, ref_pin_name = ddr_dev_service.get_all_resource_on_phy_clkin_pin(
            self.ddr_def, self.clkin_sel)

        # Figure out which PLL output clock pin based on the pin name
        if ref_pin_name is not None and ref_pin_name != "":

            if ref_pin_name.find("CLKOUT") != -1:
                out_clk_no = re.sub(
                    '.*?([0-9]*)$', r'\1', ref_pin_name)

                if out_clk_no != "":
                    outclk_idx = int(out_clk_no)

                    for res_name in res_name_list:
                        inst_name = ""
                        pin_name = ""

                        # find instance, if any
                        pll_inst = pll_reg.get_inst_by_device_name(res_name)
                        if pll_inst is not None:
                            inst_name = pll_inst.name

                            outclk = pll_inst.get_output_clock_by_number(outclk_idx)
                            if outclk is not None:
                                pin_name = outclk.name

                        # return the first one found, by right there is only one
                        return res_name, inst_name, pin_name

        return info

    def get_pll_outclk_num_for_phy_clk(self, ddr_dev_service: DDRServiceAdvance):
        outclk_idx = -1

        if self.ddr_def == "":
            return outclk_idx

        _, ref_pin_name = ddr_dev_service.get_all_resource_on_phy_clkin_pin(
            self.ddr_def, self.clkin_sel)

        # Figure out which PLL output clock pin based on the pin name
        if ref_pin_name in (None, "") or not isinstance(ref_pin_name, str) or \
            ref_pin_name.find("CLKOUT") == -1:
                return outclk_idx

        out_clk_no = re.sub(
            '.*?([0-9]*)$', r'\1', ref_pin_name)

        if out_clk_no != "":
            outclk_idx = int(out_clk_no)

        return outclk_idx

    def get_memory_type(self) -> str:
        return self.get_param_value(DDRParamInfo.Id.memory_type)

    def get_physical_rank(self) -> int:
        return self.get_param_value(DDRParamInfo.Id.physical_rank)

    def get_param_value(self, param_type: DDRParamInfo.Id):
        param_service = GenericParamService(self.param_group, self._param_info)

        return param_service.get_param_value(param_type)

    def set_param_value(self, param_type: DDRParamInfo.Id, param_value):
        param_service = GenericParamService(self.param_group, self._param_info)

        param_service.set_param_value(param_type, param_value)

    def is_valid_configuration(self, device_db):
        """
        Check if the config settings are valid combinations
        """

        from common_device.ddr.ddr_device_service import DDRService
        from device.db_interface import DeviceDBService

        if device_db is None:
            return False

        dbi = DeviceDBService(device_db)
        dev_service = dbi.get_block_service(DeviceDBService.BlockType.DDR_ADV)

        constraint_map = {
            DDRService.HEADER_MEMORY_TYPE_NAME: self.get_memory_type(),
            DDRService.HEADER_CTRL_WIDTH_NAME: self.get_param_value(DDRParamInfo.Id.data_width),
            DDRService.HEADER_DRAM_DENSITY_NAME: self.get_param_value(DDRParamInfo.Id.memory_density),
            DDRService.HEADER_PHYSICAL_RANK_NAME: self.get_param_value(DDRParamInfo.Id.physical_rank)}
        target_list = [DDRService.HEADER_COMB_ID_NAME]

        results = dev_service.get_config_data(constraint_map, target_list)

        if not results:
            return False

        return True

    def get_ddr_clock_speed(self, design, pll_inst_name, clk_name):
        ddr_freq = None

        if design is not None and design.device_db is not None:
            # Doing import at top level cause circular import problem
            import device.db_interface as devdb_int
            dbi = devdb_int.DeviceDBService(design.device_db)
            ddr_dev_service = dbi.get_block_service(devdb_int.DeviceDBService.BlockType.DDR_ADV)
            pll_output_freq = ddr_dev_service.get_pll_output_clock_freq(
                design.pll_reg, pll_inst_name, clk_name)

            if pll_output_freq is not None and isinstance(pll_output_freq, (float, Decimal)):
                ddr_freq = pll_output_freq * 2

        return ddr_freq

    def get_ddr_clock_speed_str(self, design, pll_inst_name, clk_name):
        """
        Get the DDR clock speed in string format for display purposes.
        :param design:
        :param pll_inst_name:
        :param clk_name:
        :return: speed in string
        """
        freq_str = ""
        output_freq = self.get_ddr_clock_speed(design, pll_inst_name, clk_name)

        if output_freq is not None and isinstance(output_freq, (float, Decimal)):
            # Used the same approach as PLL but with fixed Decimal point
            freq_str = "{}".format(round(Decimal(output_freq), 4))
        else:
            freq_str = ""

        return freq_str

    def copy_common(self, from_obj: DDRAdvance, exclude_attr: Optional[List[str]] = None):
        if exclude_attr is None:
            exclude_attr = ["pin_swap"]

        super().copy_common(from_obj, exclude_attr)

        # But then, we have to update data width that has different
        # parameter list depending on device
        if self._param_info is not None:
            self._param_info.set_valid_setting(DDRParamInfo.Id.data_width,
                                     self.get_data_width_options())
            self._param_info.set_default(DDRParamInfo.Id.data_width,
                                         self.get_default_data_width())
            self._param_info.set_valid_setting(DDRParamInfo.Id.memory_type, self.get_memory_type_options())
            self._param_info.set_default(DDRParamInfo.Id.memory_type, self.get_default_memory_type())

            self._param_info.set_valid_setting(DDRParamInfo.Id.clkin_sel,
                                               self.get_clkin_options())

        if self.param_group is not None:
            assert self._param_info is not None
            param_service = GenericParamService(self.param_group, self._param_info)

            cur_width = self.get_param_value(DDRParamInfo.Id.data_width)
            if cur_width not in self.get_data_width_options():
                self.set_default_data_width(param_service)

            cur_memtype = self.get_param_value(DDRParamInfo.Id.memory_type)
            if cur_memtype not in self.get_memory_type_options():
                self.set_param_value(DDRParamInfo.Id.memory_type, self._param_info.get_default(DDRParamInfo.Id.memory_type))

            cur_clkin = self.get_param_value(DDRParamInfo.Id.clkin_sel)
            if cur_clkin not in self.get_clkin_options():
                param_service.set_param_value(
                    DDRParamInfo.Id.clkin_sel, self._param_info.get_default(DDRParamInfo.Id.clkin_sel))

            # Pin swap
            if self.pin_swap.is_read_only:
                self.pin_swap.set_default_setting()
            else:
                self.pin_swap.gen_param = from_obj.pin_swap.gen_param


@gen_util.freeze_it
class DDRRegistryAdvance(PeriDesignRegistry):
    """
    A registry of all ddr instances
    """
    def create_instance(self, name, apply_default=True, auto_pin=False):
        """
        Create and register a ddr instance

        :param name: Instance name
        :param apply_default: True to apply default setting as per spec
        :param auto_pin:
        :return: A ddr instance
        :raise: db_item.CreateException
        """
        with self._write_lock:
            ddr = DDRAdvance(name)
            if apply_default:
                ddr.set_default_setting(self.device_db)

            if auto_pin:
                ddr.generate_pin_name()

            self._register_new_instance(ddr)

            return ddr

    def is_device_valid(self, device_def):
        """
        Override
        """

        # Doing import at top level cause circular import problem
        import device.db_interface as devdb_int
        dbi = devdb_int.DeviceDBService(self.device_db)
        svc = dbi.get_block_service(devdb_int.DeviceDBService.BlockType.DDR_ADV)

        dev_list = svc.get_usable_instance_names()
        if device_def not in dev_list:
            return False

        return True

    def apply_device_db(self, device_db: Optional[PeripheryDevice] = None):
        """
        Set device db and use it to extract device dependent data.
        This must be called before using the registry.

        :param device_db: Device db instance
        """

        if device_db is None:
            return

        self.device_db = device_db

        # Needed to initialize the param information
        DDRAdvance.build_device_info(self.device_db)

        # Extract generic pin info from device
        DDRAxiTarget0Adv.build_port_info(self.device_db, False)
        DDRAxiTarget1Adv.build_port_info(self.device_db, False)
        DDRController.build_port_info(self.device_db, False)
        DDRControllerRegisterInterface.build_port_info(self.device_db, False)
        DDRConfigControl.build_port_info(self.device_db, False)

        if self.device_db is not None and self.device_db.get_pin_swap() is not None:
            DDRPinSwap.is_read_only = True
            DDRPinSwap.pin_swap_info = device_db.get_pin_swap()

    def change_device(self, device_db):
        """
        Override. DDR uses dynamic pins, need to re-generate pin info from device db
        before migrating pin design data.
        """
        self.apply_device_db(device_db)

        # Enough to just check for target0 because by construction, target1 and config will be there too
        if DDRAxiTarget0Adv._device_port_map is not None:
            super().change_device(device_db)

        self.change_device_configuration(device_db)

    def change_device_configuration(self, device_db):
        """
        Check if the configuration of each instances are valid
        for the new device. If not, set to default.
        """
        if device_db is None:
            return

        for inst in self._name2inst_map:
            des_ins = self._name2inst_map[inst]

            # PT-1732: We should also update settings/params that may be device dependent
            # although they are the same DDR type (i.e. Data Width). So that the
            # UI shows only the valid options
            param_info = des_ins.get_param_info()
            param_info.set_valid_setting(DDRParamInfo.Id.data_width,
                                         des_ins.get_data_width_options())
            
            if des_ins is not None:
                if not des_ins.is_valid_configuration(device_db):
                    # Set default config settings.
                    des_ins.set_default_setting(device_db)

                    des_ins.update_memory_type(des_ins.get_memory_type(), device_db)

if __name__ == "__main__":
    pass
