"""
Copyright (C) 2017-2021 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Sept 29, 2021

@author: maryam
"""
from __future__ import annotations
from typing import List, TYPE_CHECKING, Optional

import util.gen_util
import device.db_interface as devdb_int

import common_device.rule as base_rule
import common_device.clock_mux.clkmux_rule as clkmux_rule
import tx60_device.clock_mux.clkmux_rule_adv as tx60_clkmux_rule
from tx180_device.clock_mux.clkmux_design_comp import ClockMuxComplex, ClockMuxRegistryComplex

if TYPE_CHECKING:
    from tx180_device.ddr.ddr_design_adv import DDRAdvance
    from tx180_device.clock_mux.clkmux_device_service_comp import ClockMuxServiceComplex
    from device.block_instance import PeripheryBlockInstance


@util.gen_util.freeze_it
class ClockMuxCheckerComp(tx60_clkmux_rule.ClockMuxCheckerAdv):

    def __init__(self, design):
        """
        Constructor
        """
        self.pll_dev_svc = None

        super().__init__(design)
        self.clkmuxdev_service: ClockMuxServiceComplex
        self.reg: ClockMuxRegistryComplex

        # Add the service
        if self.design.device_db is not None:
            dbi = devdb_int.DeviceDBService(self.design.device_db)
            self.pll_dev_svc = dbi.get_block_service(
                self.design.device_db.get_pll_type())

    def _get_pll_clkout_names_based_on_ins_pins(self, pll_reg, pll_pins_list):
        pll_out_clk = []
        visited_pll_ins_name = []

        for pll_pin in pll_pins_list:
            if pll_pin is not None:
                ins_name = pll_pin.get_connecting_instance_name(self.design.device_db)

                if ins_name != "" and ins_name not in visited_pll_ins_name:
                    # Find the instance in the pll registry
                    pll_design = pll_reg.get_inst_by_device_name(ins_name)
                    visited_pll_ins_name.append(ins_name)

                    if pll_design is not None:
                        # Get all the clkout names assigned to clkout0-clkout4
                        # Which is different than TX60 which has clkout4 reserved 
                        # for regional conn
                        for idx in range(5):
                            out_clk = pll_design.get_output_clock_by_number(idx)
                            if out_clk is not None:
                                pll_out_clk.append(out_clk.name)

        return pll_out_clk


    def _build_rules(self):
        """
        Build control bank rule database
        """
        self._add_rule(clkmux_rule.ClkMuxRuleClocksRouted())
        self._add_rule(tx60_clkmux_rule.ClkMuxRuleCoreClockPin())
        self._add_rule(tx60_clkmux_rule.ClkMuxRuleDynamicClockPin())
        self._add_rule(tx60_clkmux_rule.ClkMuxRuleDynamicSelectClockPin())
        self._add_rule(tx60_clkmux_rule.ClkMuxRuleGlobalRegionalPin())
        self._add_rule(tx60_clkmux_rule.ClkMuxRuleCoreClockStaticMux())
        self._add_rule(ClkMuxRuleGlobalRegionalResource())
        self._add_rule(ClkMuxRuleRegionalConnType())
        self._add_rule(tx60_clkmux_rule.ClkMuxRulePLLMultConnection())
        self._add_rule(ClkMuxRuleTypeConfig())
        self._add_rule(tx60_clkmux_rule.ClkMuxRuleDynamicMuxPLLInput())
        self._add_rule(ClkMuxDDRClockNotConnectedToCore())
        self._add_rule(tx60_clkmux_rule.ClkMuxRuleDiffPLLClockSameMux())

class ClkMuxRuleGlobalRegionalResource(tx60_clkmux_rule.ClkMuxRuleGlobalRegionalResource):
    """
    Check that regional connection with global has valid resource used
    """

    def _is_mipi_hard_dphy_allowed_to_connect_to_global_and_regional(self, checker, rbuf_obj, mipi_pins_list):
        is_allowed = False
        for mpin in mipi_pins_list:
            # Check that the same resource and pin on this rbuf exists in the mipi clk pin list
            pin_resource = mpin.get_connecting_instance_name(checker.design.device_db)
            ref_pin_name = mpin.get_connecting_ref_pin_name()

            if pin_resource != "" and pin_resource == rbuf_obj.get_resource() and\
                    ref_pin_name == rbuf_obj.get_connecting_pin():
                is_allowed = True
                break

        return is_allowed

    def _is_pll_allowed_to_connect_to_global_and_regional(self, checker, rbuf_obj):
        # Get the Device Service
        if checker.pll_dev_svc is not None:
            if checker.pll_dev_svc.is_pll_clkout_pin_support_mult_clkout_conn(
                rbuf_obj.get_resource(), rbuf_obj.get_connecting_pin()):
                return True

        return False

class ClkMuxRuleRegionalConnType(tx60_clkmux_rule.ClkMuxRuleRegionalConnType):
    """
    Check that regional connection has the right connection type. In ClkmuxCOMPLEX,
    the PLL can be connected to regional and/or global
    """

    def _check_rbuf_connection_type(self, checker, rbuf_obj, invalid_rbuf_type_user_ins):

        if rbuf_obj is not None:
            ins_obj, ins_type = rbuf_obj.get_input_design_instance(checker.design, True)

            if ins_obj is not None:

                # The instance has been configured. Now check it's connection type
                if not rbuf_obj.is_regional_conn_type(ins_obj, ins_type):
                    # But if it is global and the instance has a clkmux_buf_name, then it's invalid
                    if rbuf_obj.is_global:
                        invalid_rbuf_type_user_ins.append(ins_obj.name)


class ClkMuxRuleTypeConfig(tx60_clkmux_rule.ClkMuxRuleTypeConfig):
    """
    Check that the resource connected to dynamic mux is configured correctly.
    For example, if it was a GPIO connection, the resource can only be configured
    as LVDS/GPIO.
    """

    def _check_pll_regbuf_conn_pin(self, pin_name):
        is_valid = True
        outclk_idx_str = ""

        if pin_name != "CLKOUT4" and pin_name != "CLKOUT3":
            is_valid = False
            outclk_idx_str = "3,4"

        return is_valid, outclk_idx_str


class ClkMuxDDRClockNotConnectedToCore(base_rule.Rule):
    """
    PT-1639: The PLL outclk 3,4 that is driving the DDR should not be routed to core.
    Clkmux should handle it, but this is a safety net in case routing went wrong
    """

    def __init__(self):
        super().__init__()
        self.name = "clkmux_rule_ddr_clock"

    def check(self, checker: ClockMuxCheckerComp, design_block: ClockMuxComplex): # type: ignore

        clkmux_obj = design_block

        # We only need to check it for GCLKMUX_T. But to consider
        # all possible PLL for DDR, include GCLKMUX_L

        if clkmux_obj is not None and checker.design is not None and\
                checker.design.ddr_reg is not None and \
                checker.design.pll_reg is not None:

            ddr_reg = checker.design.ddr_reg
            pll_reg = checker.design.pll_reg

            ddr_all_inst: List[DDRAdvance] = ddr_reg.get_all_inst()

            is_ddr_config = True
            if len(ddr_all_inst) > 0:
                # Check if the ddr is configured correctly (all has res assigned)
                for ddr in ddr_all_inst:
                    if ddr is not None:
                        if ddr.ddr_def == "":
                            is_ddr_config = False
                    else:
                        is_ddr_config = False
            else:
                is_ddr_config = False

            # Only proceed with check if the DDR is configured with resource set
            if is_ddr_config:
                assert checker.design.device_db is not None
                dbi = devdb_int.DeviceDBService(checker.design.device_db)
                ddrdev_service = dbi.get_block_service(devdb_int.DeviceDBService.BlockType.DDR_ADV)

                clkmux_dev_ins = checker.design.device_db.find_instance(clkmux_obj.block_def)

                if clkmux_dev_ins is None:
                    return

                depend_name_list = clkmux_dev_ins.get_depended_instance_names()

                for ddr in ddr_all_inst:
                    result = None
                    if ddr is not None and ddr.ddr_def != "":
                        result = ddr.find_phy_clock_info(ddrdev_service, pll_reg)

                    if result is None:
                        continue

                    (pll_res, des_ins_name, _) = result

                    # Only ddr dependencies which is included in clockmux dependenies
                    if self.is_ti375_device(checker.design.device_db) and pll_res not in depend_name_list:
                        continue

                    # Get the design PLL to check the frequency
                    des_pll = pll_reg.get_inst_by_name(des_ins_name)

                    # We don't check if it is None since that is checked by
                    # DDRRuleClkin
                    if des_pll is None:
                        continue

                    clknum = ddr.get_pll_outclk_num_for_phy_clk(ddrdev_service)

                    if clknum == -1:
                        continue

                    clkout = des_pll.get_output_clock_by_number(clknum)

                    clk_name_list = []
                    if clkout is not None:
                        clk_name_list.append(clkout.name)

                    # CLKOUT3 cannot be routed only if on revA
                    if not ddrdev_service.is_ddr_rev_b():
                        clkout3 = des_pll.get_output_clock_by_number(3)

                        if clkout3 is not None:
                            clk_name_list.append(clkout3.name)

                    if clk_name_list:
                        # Check if the CLKMUX_T has dynamic mux input
                        # or static clock routed with the above clock name
                        used_clk_names = self.check_if_clkmux_has_clock_connected(
                            checker.design, checker.clkmuxdev_service,
                            clkmux_obj, clkmux_dev_ins, clk_name_list)

                        if len(used_clk_names) > 0:
                            self.error("Clock driving DDR cannot be routed to core: {}".format(
                                ",".join(sorted(used_clk_names))))

    def check_dyn_mux_input_pll_clocks(self, design, clkmux_obj, mux_id, clk_name_used):
        """
        Check if the input of the dynamic mux that is enabled is from
        any of the clock defined in clk_name_used

        :param design: Design DB
        :param clkmux_obj: The clockmux design instance
        :param mux_id: The dynamic mux id that we're checking (0/7)
        :param clk_name_used: A list of clock names to check if it is connected
                to the dynamic mux input
        :return: a set of clock names that is from the clk_name_used that is used
                to drive the dynamic mux input
        """
        connected_names = set()

        dyn_mux_inputs = clkmux_obj.get_user_dyn_mux_inputs(mux_id)

        if dyn_mux_inputs:
            for idx in sorted(dyn_mux_inputs):

                in_obj = dyn_mux_inputs[idx]
                if in_obj is not None:
                    _, in_pin_name = in_obj.get_user_resource_pin_name(
                        clkmux_obj, design, False, design.device_db)

                    if in_pin_name in clk_name_used:
                        connected_names.add(in_pin_name)

        return connected_names

    def check_if_clkmux_has_clock_connected(self, design, clkmuxdev_service: ClockMuxServiceComplex,
                                            clkmux_obj, clkmux_dev_ins, clk_name_list):
        """
        :param design: Design DB
        :param clkmuxdev_service: The clkmux device service
        :param clkmux_obj: The clkmux design instance
        :param clkmux_dev_ins: The clkmux device instance
        :param clk_name_list: A list of clock signal name to check
        :return: a list of clock name that is connected to
                the clkmux (dynamic or static input which got routed). Empty
                set if none of the clock in clk_name_list is connected to core
                through the clkmux
        """
        connected_names = set()

        clk_name_used = []
        for clk_name in clk_name_list:
            if clk_name != "":
                clk_name_used.append(clk_name)

        if design is not None and len(clk_name_used) > 0:
            # Check if the output of the routed clock mux has the signal
            # set to any of the passed clock name
            # Get the output map of the current clkmux
            output_map = clkmux_obj.get_all_io_setting()

            # Iterate through the clkmux output to see the pin that has been assigned
            for out_idx in output_map:
                io_setting = output_map[out_idx]

                if io_setting is not None:
                    user_in_name, _, _ = clkmuxdev_service.get_conn_input_user_name(
                        design, clkmux_dev_ins, io_setting.output_name,
                        io_setting.output_no, io_setting, clkmux_obj.gen_pin)

                    if user_in_name != "":
                        if user_in_name in clk_name_used:
                            connected_names.add(user_in_name)

            # Check if it is connected to any of the  dynamic mux
            if clkmux_obj.is_mux_bot0_dyn:
                mux0_conn_names = self.check_dyn_mux_input_pll_clocks(design, clkmux_obj, 0, clk_name_used)
                connected_names.update(mux0_conn_names)

            if clkmux_obj.is_mux_bot7_dyn:
                mux7_conn_names = self.check_dyn_mux_input_pll_clocks(design, clkmux_obj, 7, clk_name_used)
                connected_names.update(mux7_conn_names)

        return connected_names

    def is_ti375_device(self, device_db):
        device_engineer_name = "opx_466x964_b56_d28" # Ti375 device

        engine_name: str = device_db.get_device_mapped_name()

        if engine_name.startswith(device_engineer_name):
            return True

        return False
