"""

Copyright (C) 2017-2022 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Feb 7, 2022

@author: Shirley Chan
"""

import os
import sys
from typing import List, Optional, Tuple, Any

from PyQt5.QtCore import QPointF, QObject, QSize, Qt, QRectF, pyqtSignal, pyqtSlot
from PyQt5.QtGui import (
    QPainter,
    QMouseEvent,
    QIcon,
    QCursor,
    QPalette,
    QBrush,
    QPolygonF,
    QKeyEvent,
    QWheelEvent,
    QResizeEvent,
    QShowEvent,
    QImage,
    QPdfWriter,
    QPaintDevice,
    QImageWriter
)
from PyQt5.QtWidgets import QGraphicsView, QRubberBand, QWidget, QGraphicsItem, QApplication
from pkg_gui.gui.export_diagram import FileFormat

from pkg_gui.gui.model.model import PinoutViewerModel, LeadViwerModel, BallViewerModel
from pkg_gui.gui.presenter import ViewMode
from pkg_gui.gui.graphics_item.pin_icon import PinIcon
from design.package_pin import BallPackage, PackagePinService, PinFunction

from util.gen_util import freeze_it
from util.singleton_logger import Logger

sys.path.append(os.path.join(os.path.dirname(__file__), '..'))


@freeze_it
class PinoutViewer(QObject):
    """
    For each pin, it display the info related to pad.
    This include read-only info from device database.
    """

    top2bottom_mode = {
        ViewMode.UPPER_LEFT: ViewMode.UPPER_RIGHT,
        ViewMode.UPPER_RIGHT: ViewMode.UPPER_LEFT,
        ViewMode.LOWER_LEFT: ViewMode.LOWER_RIGHT,
        ViewMode.LOWER_RIGHT: ViewMode.LOWER_LEFT
    }
    scale_factor = 1.25

    def __init__(self, parent: Any, block_config=None):
        """
        Constructor

        :param parent: Pinout main window
        :block_config: pinout diagram view config page
        """
        QObject.__init__(self, parent)

        self.logger = Logger

        self.model: Optional[PinoutViewerModel] = None
        self.main_window = parent

        self.gw_pkg_view = PackageView(None)
        parent.verticalLayout.addWidget(self.gw_pkg_view)

        self.block_config = block_config
        self.block_name = ""
        self.design = None
        self.mode = ViewMode.UPPER_LEFT
        self.is_show_iobank = False
        self.is_show_mipi_rx_group = False
        self.is_show_assigned_pin = False

        self.world_view = WorldView(self.gw_pkg_view, parent)
        self.gw_pkg_view.horizontalScrollBar().valueChanged.connect(self.on_refresh_scale)
        self.gw_pkg_view.verticalScrollBar().valueChanged.connect(self.on_refresh_scale)
        self.world_view.sig_set_scene_center.connect(self.on_set_focus_view)
        self.pkg_inst: Optional[PackagePinService] = None
        self.is_monitor = False

    def build(self, design, pkg_inst: PackagePinService):
        """
        Override
        """
        # Ensure not connect more than 1 function for each Qt signal
        self.stop_monitor_changes()

        self.design = design
        self.pkg_inst = pkg_inst

        if isinstance(self.pkg_inst, BallPackage):
            self.model = BallViewerModel()
        else:
            self.model = LeadViwerModel()

        self.model.load(design, pkg_inst=self.pkg_inst)
        self.model.setSceneRect(self.model.itemsBoundingRect())

        self.gw_pkg_view.setRenderHints(QPainter.Antialiasing | QPainter.SmoothPixmapTransform)
        self.gw_pkg_view.setScene(self.model)
        tooltip_style = "QToolTip { color: rgb(87, 144, 189); " \
                        "border: 1px solid white;"\
                        "background-color: white; }"
        self.gw_pkg_view.setStyleSheet(tooltip_style)
        self.world_view.setScene(self.model)
        self.world_view.setStyleSheet(tooltip_style)
        self.world_view.hide_rubber_band()

        self.monitor_changes()

    def monitor_changes(self):
        if self.is_monitor:
            return

        if self.model is not None:
            self.model.sig_wheel_event.connect(self.zooming)
            self.model.sig_zoom_fit.connect(self.on_zoom_fit)
            self.model.sig_zoom_area.connect(self.on_zoom_area)
            self.model.sig_zooming.connect(self.on_click_zooming)

        self.is_monitor = True

    def stop_monitor_changes(self):
        if not self.is_monitor:
            return

        if self.model is not None:
            self.model.sig_wheel_event.disconnect()
            self.model.sig_zoom_fit.disconnect()
            self.model.sig_zoom_area.disconnect()
            self.model.sig_zooming.disconnect()

        self.is_monitor = False

    @pyqtSlot()
    def on_zoom_fit(self):
        '''
        Fit to window
        '''
        scene = self.gw_pkg_view.scene()
        rect = scene.sceneRect()
        self.on_zoom_area(rect)
        self.world_view.hide_rubber_band()

    @pyqtSlot(QRectF)
    def on_zoom_area(self, rect: QRectF):
        self.gw_pkg_view.fitInView(rect, Qt.KeepAspectRatio)

    @pyqtSlot(float, QPointF)
    def on_click_zooming(self, change_scale: float, focus_point: QPointF):
        assert self.model is not None
        if change_scale < 0:
            scale = 1 / (1 + change_scale * -1)
        else:
            scale = 1 + change_scale

        self.gw_pkg_view.scale(scale, scale)
        self.gw_pkg_view.centerOn(focus_point)

    def zooming(self, is_zoom_in: bool):
        """
        Enable wheelEvent zoom in/out
        """
        if is_zoom_in:
            self.zoom_in()
        else:
            self.zoom_out()

        focus_point = self.gw_pkg_view.mapToScene(QCursor.pos())
        self.gw_pkg_view.centerOn(focus_point)
        self.world_view.refresh_rubber_band()

    def zoom_in(self):
        scale = self.scale_factor
        self.gw_pkg_view.scale(scale, scale)
        self.world_view.refresh_rubber_band()

    def zoom_out(self):
        scale = 1 / self.scale_factor
        self.gw_pkg_view.scale(scale, scale)
        self.world_view.refresh_rubber_band()

    def reset_orientation(self):
        assert self.model is not None
        self.mode = ViewMode.UPPER_LEFT
        self.model.is_bottom = False
        self.model.rotate_pins(self.mode, self.is_show_iobank, self.is_show_mipi_rx_group)
        self.update_pkg_bottom_icon(self.main_window)

    def rotate(self, direction: str):
        assert self.mode is not None
        assert self.model is not None

        if direction == "left":
            if self.mode == ViewMode.UPPER_LEFT:
                self.mode = ViewMode.LOWER_LEFT
            else:
                mode_list = list(ViewMode)
                next_mode_index = mode_list.index(self.mode) - 1
                self.mode = mode_list[next_mode_index]
        else:
            if self.mode == ViewMode.LOWER_LEFT:
                self.mode = ViewMode.UPPER_LEFT
            else:
                mode_list = list(ViewMode)
                next_mode_index = mode_list.index(self.mode) + 1
                self.mode = mode_list[next_mode_index]

        self.model.rotate_pins(self.mode, self.is_show_iobank, self.is_show_mipi_rx_group)

        self.world_view.fitInView(self.model.sceneRect(), Qt.KeepAspectRatio)
        self.world_view.refresh_rubber_band()

    def view_pkg_bottom(self):
        assert self.model is not None

        mode = self.top2bottom_mode.get(self.mode)
        assert mode is not None
        self.mode = mode

        self.model.is_bottom = not self.model.is_bottom
        self.model.rotate_pins(self.mode, self.is_show_iobank, self.is_show_mipi_rx_group)

        # Change the tooltip and icon
        self.update_pkg_bottom_icon(self.main_window)

    def update_pkg_bottom_icon(self, window):
        '''
        Change the tooltip and icon for actionPackage_Bottom

        :param window: target window that contain the actionPackage_Bottom
        '''
        assert self.mode is not None
        assert self.model is not None

        if self.model.is_bottom:
            txt = 'Top'
            icon_file = 'package-top'
        else:
            txt = 'Bottom'
            icon_file = 'package-bottom'

        icon = QIcon()
        icon.addFile(":/icons/" + icon_file, QSize(16, 16), QIcon.Normal, QIcon.On)

        window.actionPackage_Bottom.setToolTip(f'Show {txt} View')
        window.actionPackage_Bottom.setIcon(icon)
        window.actionPackage_Bottom.setText(f'Show {txt} View')

    def show_all_iobank(self, is_show_iobank: bool):
        assert self.model is not None
        self.is_show_iobank = is_show_iobank

        if is_show_iobank:
            self.model.set_all_iobank()
        else:
            self.model.remove_all_iobank_from_screen()

    def show_all_mipi(self, is_show_group: bool):
        assert self.model is not None
        self.is_show_mipi_rx_group = is_show_group
        if is_show_group:
            self.model.set_all_mipi_group()
        else:
            self.model.remove_all_mipi_rx_group_from_screen()

    def show_func(self, func_name: PinFunction, is_show: bool):
        assert self.model is not None
        if is_show:
            self.model.set_all_func_by_func_name(func_name)
        else:
            self.model.remove_func_from_screen(func_name)

    def show_individual_iobank(self, iobank_name: str, is_show: bool):
        assert self.model is not None

        if is_show:
            self.model.set_iobank(iobank_name)
            if iobank_name in self.model.exclude_iobank:
                self.model.exclude_iobank.remove(iobank_name)
        else:
            self.model.remove_iobank_from_screen(iobank_name)

    def show_individual_mipi(self, group_name: str, is_show: bool):
        assert self.model is not None

        if is_show:
            self.model.set_mipi_rx_group(group_name)
            if group_name in self.model.exclude_mipi_rx_group:
                self.model.exclude_mipi_rx_group.remove(group_name)
        else:
            self.model.remove_mipi_rx_group_from_screen(group_name)

    def show_hide_world_view(self):
        assert self.model is not None

        self.world_view.setVisible(not self.world_view.isVisible())
        rect = self.model.sceneRect()
        self.world_view.fitInView(rect, Qt.KeepAspectRatio)

    def refresh_model(self):
        """
        Reload the diagram with updated information
        """
        assert self.model is not None

        self.model.clear()
        self.model.reload_diagram()
        self.model.setSceneRect(self.model.itemsBoundingRect())
        self.show_assigned_pins(self.is_show_assigned_pin)

    def highlight_pin(self, pin_name: str):
        assert self.model is not None
        pin_icon = self.model.pin2pin_icon.get(pin_name, None)

        if pin_icon is None:
            self.logger.error('Error for searching pin icon')
            return

        self.model.clearSelection()
        pin_icon.setSelected(True) # Auto call model.on_selected_pin()

        # Reset the screen
        self.zoom_in_pin()

        # Zoom the location of the screen to that pin
        focus_pos = self.model.get_focus_pos(pin_icon.pos())
        self.focus_pin(focus_pos)

    def clear_selection(self):
        assert self.model is not None
        self.model.clearSelection()

    def highlight_blk_res(self, blk_type: str, res_name: str):
        assert self.model is not None

        if blk_type != "gpio_bus":
            self.model.clearSelection()

        self.model.selectionChanged.disconnect(self.model.on_selected_pin)
        self.model.set_blk_res_selected(blk_type, res_name)
        self.model.selectionChanged.connect(self.model.on_selected_pin)

    def is_1_pin_highlighted(self) -> Tuple[bool, str]:
        if self.model is None:
            return False, ""

        selected_items = self.model.selectedItems()
        if len(selected_items) == 1:
            item = selected_items[0]
            assert isinstance(item, PinIcon) and item.pin_name is not None
            return True, item.pin_name

        return False, ""

    def show_assigned_pins(self, is_show: bool):
        self.is_show_assigned_pin = is_show
        all_items: List[QGraphicsItem] = self.gw_pkg_view.items()

        for item in all_items:
            if isinstance(item, PinIcon):
                item.show_assigned = is_show
                item.update()

    def zoom_in_pin(self):
        '''
        Reset the screen size to pin
        '''
        self.on_zoom_fit()

        screen = self.gw_pkg_view.scene()
        rect = screen.sceneRect()
        scale = rect.width()/150
        self.gw_pkg_view.scale(scale, scale)

    def focus_pin(self, pin_pos: QPointF):
        '''
        Zoom the screen to the pin location
        '''
        self.logger.debug(f"Pin coordinate: {pin_pos.x()}, {pin_pos.y()}")
        self.gw_pkg_view.centerOn(pin_pos)
        self.world_view.refresh_rubber_band()

    def update_pin_info(self, item_type: str, ori_res: str, res_name: str):
        assert self.model is not None

        self.logger.debug(f"Updating pin:  origin resource: {ori_res},  new resource: {res_name}")
        self.model.update_all_pins_by_res_name(item_type, res_name)
        self.model.update_all_pins_by_res_name(item_type, ori_res)
        self.model.update() # Ensure that QGraphicsView is updated
        self.highlight_blk_res(item_type, res_name)

    def gen_diagram_file(self, img_file: str, is_transparent: bool, file_type: FileFormat, qty: int) -> bool:
        '''
        Export pinout diagram into image file

        :param img_file: path of the file (with filename)
        :param is_transparent: enable/ disable transparent background (default: white)
        :param qty: quality of image file (range: 1 - 100)
        :return is_save: if image is save as an image file
        '''
        self.logger.debug("generating diagram file")
        assert self.model is not None
        assert isinstance(file_type.value, str)

        self.model.clearSelection()

        is_save: bool = False

        if file_type == FileFormat.PDF:
            is_save = self.gen_diagram_pdf(img_file)
        else:
            is_save = self.gen_diagram_image(img_file, file_type, is_transparent)

        return is_save

    def paint_diagram(self, paint_device: QPaintDevice):
        assert self.model is not None
        painter = QPainter(paint_device)
        painter.setRenderHints(QPainter.Antialiasing | QPainter.SmoothPixmapTransform)
        self.model.render(painter)
        painter.end() # Prevent crash

    def gen_diagram_pdf(self, img_file: str):
        pdf_writer = QPdfWriter(img_file)
        pdf_writer.setPageSize(QPdfWriter.A4)
        self.paint_diagram(pdf_writer)
        return True

    def gen_diagram_image(self, img_file: str, file_type: FileFormat, is_transparent: bool):
        assert self.model is not None

        # Enable pin and pad name to display clearly in image
        detail_rate = 2.5
        rect: QRectF = self.model.itemsBoundingRect()
        width, height = rect.width() * detail_rate, rect.height() * detail_rate

        # Ensure the size in QImage range
        # See https://stackoverflow.com/questions/7080052/qimage-qpixmap-size-limitations
        max_width = 7680
        max_height = 4320

        if width > max_width or height > max_height:
            if width > height:
                ratio = max_width/ width
            else:
                ratio = max_height/ height

            width = width * ratio
            height = height * ratio

        image = QImage(int(width), int(height), QImage.Format_RGBA64)

        if is_transparent:
            image.fill(Qt.transparent)
        else:
            image.fill(Qt.white)

        self.paint_diagram(image)

        if image.isNull():
            self.logger.error("Fail to paint diagram")
            return False

        # For checking image support formats
        # inst = QApplication.instance()
        # print(QImageWriter.supportedImageFormats())

        writer = QImageWriter(img_file, file_type.value.encode())
        is_save = writer.write(image)

        if not is_save:
            self.logger.error(writer.errorString())

        return is_save

    # noinspection PyArgumentList
    @pyqtSlot(QPointF)
    def on_set_focus_view(self, focus_point: QPointF):
        self.gw_pkg_view.centerOn(focus_point)

    @pyqtSlot()
    def on_refresh_scale(self):
        assert self.model is not None
        self.model.horizon_scale, self.model.vertical_scale = self.gw_pkg_view.get_current_scale_factor()
        self.world_view.refresh_rubber_band()


class PackageView(QGraphicsView):
    """
    QGraphicsView for adjusting view for pinout diagram
    """
    def __init__(self, parent: Optional[QWidget]):
        super().__init__(parent)
        self.setObjectName("gw_pkg_view")
        self.__init_h_factor = 0 # initial horizontal factor
        self.__init_v_factor = 0 # initial vertical factor

    def _set_init_factor(self):
        """
        Only called for first load
        """
        self.__init_h_factor, self.__init_v_factor = self.get_current_scale_factor()

    def get_current_scale_factor(self) -> Tuple[float, float]:
        """
        Get QGraphicsView scaling factor from QTransform

        m11() -> horizontal scaling factor
        m22() -> vertical scaling factor

        :return: horizontal scaling factor and vertical scaling factor
        :rtype: Tuple[float, float]
        """
        horizon_factor = self.transform().m11()
        vertical_factor = self.transform().m22()
        return horizon_factor, vertical_factor

    def scale(self, sx: float, sy: float) -> None:
        """
        Override Qt function.

        Check scale before zooming/ update scale
        """
        if self.is_validate_scale(sx, sy):
            return super().scale(sx, sy)
        return

    def is_validate_scale(self, scale_x: float, scale_y: float) -> bool:
        """
        Validate scaling to prevent scene display too small

        :param scale_x: horizontal scaling
        :type scale_x: float
        :param scale_y: vertical scaling
        :type scale_y: float
        :return: is valid scale provided
        :rtype: bool
        """
        is_valid = False

        horizon_factor, vertical_factor = self.get_current_scale_factor()

        valid_horizon_scale = self.__init_h_factor * (
            1/ PinoutViewer.scale_factor) ** 2
        valid_vertical_scale = self.__init_v_factor * (
            1/ PinoutViewer.scale_factor) ** 2

        if horizon_factor * scale_x >= valid_horizon_scale:
            is_valid = True
        elif vertical_factor * scale_y >= valid_vertical_scale:
            is_valid = True

        return is_valid

    # pylint: disable = invalid-name
    def showEvent(self, event: QShowEvent) -> None:
        """
        By default, if the view is first initialized, it will
        called ``resizeEvent()``, which will cause strange zoom fit problem.

        Overriding this Qt function can solve the zoom fit issue for the
        first load. This will affect the behavior of ``show()`` or
        ``setVisible(True)``

        :param event: Qt show event
        :type event: QShowEvent
        """
        if not event.spontaneous():
            scene = self.scene()
            assert isinstance(scene, PinoutViewerModel)
            rect = scene.sceneRect()
            self.fitInView(rect, Qt.KeepAspectRatio)
            self._set_init_factor()

            # Ensure the scales are updated
            scene.horizon_scale, scene.vertical_scale = self.get_current_scale_factor()

        return super().showEvent(event)


class WorldView(QGraphicsView):
    '''
    Provide a world view of the pinout diagram (QGraphicsView)
    '''
    sig_set_scene_center = pyqtSignal(QPointF)

    def __init__(self, main_view: QGraphicsView, parent) -> None:
        super().__init__(parent.centralwidget)
        self.setObjectName('gw_world_view')
        self.main_view = main_view

        # Refers to Qt designer MainWindow in engine
        self.setGeometry(9, 32, 200, 150)
        self.setVisible(False)
        self.setViewportUpdateMode(QGraphicsView.SmartViewportUpdate)
        self.setDragMode(QGraphicsView.NoDrag)
        self.setInteractive(True)
        self.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        self.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOff)

        # Built-in rubberband is for area selection. Create custom rubberband viewing rect.
        self._rect_rb = QRubberBand(QRubberBand.Rectangle, self)
        pal = QPalette()
        pal.setBrush(QPalette.Highlight, QBrush(Qt.black))
        self._rect_rb.setPalette(pal)

    def show_rubber_band(self, rect: QRectF):
        rect_point1 = self.mapFromScene(rect.topLeft())
        rect_point2 = self.mapFromScene(rect.bottomRight())

        self._rect_rb.setGeometry(QRectF(rect_point1, rect_point2).toRect())
        self._rect_rb.show()

    def hide_rubber_band(self):
        self._rect_rb.hide()

    def refresh_rubber_band(self):
        self._rect_rb.hide()
        scene_polygon = self.main_view.mapToScene(self.main_view.viewport().rect())
        assert isinstance(scene_polygon, QPolygonF)
        self.show_rubber_band(scene_polygon.boundingRect())

    def keyPressEvent(self, event: QKeyEvent) -> None:
        event.ignore()

    def keyReleaseEvent(self, event: QKeyEvent) -> None:
        event.ignore()

    def mouseDoubleClickEvent(self, event: QMouseEvent) -> None:
        event.ignore()

    def mousePressEvent(self, event: QMouseEvent) -> None:
        scene_pos = self.mapToScene(event.pos())
        self.sig_set_scene_center.emit(scene_pos)

    def mouseMoveEvent(self, event: QMouseEvent) -> None:
        event.ignore()

    def mouseReleaseEvent(self, event: QMouseEvent) -> None:
        event.ignore()

    def wheelEvent(self, event: QWheelEvent) -> None:
        event.ignore()

    def resizeEvent(self, event: QResizeEvent) -> None:
        return super().resizeEvent(event)


if __name__ == "__main__":
    pass
