"""

Copyright (C) 2017-2022 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Feb 7, 2022

@author: Shirley Chan
"""

from __future__ import annotations
from enum import Enum, auto, IntEnum
from typing import Optional, Union, Tuple

from PyQt5.QtCore import QPointF, Qt, QRectF, QLineF
from PyQt5.QtGui import QPolygonF, QBrush, QPen, QColor, QPainter
from PyQt5.QtWidgets import (
    QGraphicsPolygonItem,
    QStyleOptionGraphicsItem,
    QGraphicsObject,
    QWidget,
)

from design.package_pin import PinFunction, ALL_PIN_TYPE

from pkg_gui.gui.color_theme import (
    ColorTheme,
    DedicatedTheme,
    TrionTheme,
    TitaniumTheme,
    FuncTheme,
    IOBankTheme,
    MIPIRxGroupTheme
)
from pkg_gui.gui.graphics_item.pin_icon import (
    GNDVCCBallPin,
    GNDVCCLeadPin,
    PinRect,
    LeadPin,
    BallPinIcon,
    SemiCircle,
    SemiRect,
    SERDESBallPin,
)


class ViewMode(Enum):
    """
    Enum Class for Package Planner View direction
    """
    UPPER_LEFT = "upper_left"
    UPPER_RIGHT = "upper_right"
    LOWER_RIGHT = "lower_right"
    LOWER_LEFT = "lower_left"


class Direction(Enum):
    """
    Enum Class for Lead Package pin direction
    """
    LEFT = auto()
    RIGHT = auto()
    UP = auto()
    DOWN = auto()


class DiagramItem:
    """
    Class for generating all QGraphicsObject/ QGraphicsPolygonItem in the Package Planner.

    Expected all pin related items should have object name (i.e. called ``setObjectName()``)
    so that they can be used in squish test.
    """
    class ItemType(IntEnum):
        '''
        UI type for package planner
        '''
        # Ball-package specific
        BALL_CONTAINER = auto()
        BALL_PIN = auto()

        # Lead-package specific
        LEAD_CONTAINER = auto()
        LEAD_PIN = auto()

        IO_BANK = auto()
        LEAD_IO_BANK = auto()

        WORLD_VIEW = auto()

        FUNC = auto()
        LEAD_FUNC = auto()

        MIPI_RX_GROUP = auto()
        LEAD_MIPI_RX_GROUP = auto()

    def __init__(self, diagram_type, size: Optional[Union[Tuple[float, float], float]] = None, pin_name:Optional[str]= None,
                 pin_type = None, is_titanium=False, parent=None):
        self.item: Optional[Union[QGraphicsObject, QGraphicsPolygonItem]] = None
        self.polygon_svr = QGraphicsPolygonItem(parent)
        self.pin_name = pin_name

        self.diagram_type = diagram_type
        self.is_titanium = is_titanium
        self.color_theme:Optional[ColorTheme] = None

        if self.diagram_type == DiagramItem.ItemType.BALL_CONTAINER:
            assert isinstance(size, tuple)
            self.set_ball_container(size)

        elif self.diagram_type == DiagramItem.ItemType.LEAD_CONTAINER:
            assert isinstance(size, (float, int))
            self.set_lead_container(size)

        elif self.diagram_type == DiagramItem.ItemType.WORLD_VIEW:
            pass
            # set the diagram later

        if self.diagram_type == DiagramItem.ItemType.BALL_PIN:
            assert pin_type is not None
            self.set_ball_pin(pin_type)

        elif self.diagram_type == DiagramItem.ItemType.LEAD_PIN:
            assert pin_type is not None
            self.set_lead_pin(pin_type)

        elif self.diagram_type == DiagramItem.ItemType.IO_BANK:
            assert pin_type is not None
            self.set_io_bank(pin_type, False)

        elif self.diagram_type == DiagramItem.ItemType.LEAD_IO_BANK:
            assert pin_type is not None
            self.set_io_bank(pin_type, True)

        elif self.diagram_type == DiagramItem.ItemType.FUNC:
            assert pin_type is not None
            self.set_func(pin_type, False)

        elif self.diagram_type == DiagramItem.ItemType.LEAD_FUNC:
            assert pin_type is not None
            self.set_func(pin_type, True)

        elif self.diagram_type == DiagramItem.ItemType.MIPI_RX_GROUP:
            assert pin_type is not None
            self.set_mipi_rx_group(pin_type, False)

        elif self.diagram_type == DiagramItem.ItemType.LEAD_MIPI_RX_GROUP:
            assert pin_type is not None
            self.set_mipi_rx_group(pin_type, True)

    def get_item(self):
        """
        Get generated QGraphicsObject/ QGraphicsPolygonItem
        """
        return self.item

    def set_lead_container(self, size: float):
        """
        Set lead container by the given width

        :param size: width of the lead container
        :type size: float
        """
        width = size
        slop = width * 0.05
        lead_container = QPolygonF([
                QPointF(0, 0), QPointF(0, width),
                QPointF(slop, width + slop), QPointF(width + slop, width + slop),
                QPointF(width + slop*2, width), QPointF(width + slop*2, 0),
                QPointF(width + slop, -1*slop), QPointF(slop, -1*slop),
                QPointF(0, 0)])
        self.polygon_svr.setPolygon(lead_container)
        self.item = self.polygon_svr

    def set_lead_pin(self, pin_type: ALL_PIN_TYPE):
        """
        Set lead pin icon by the given pin type.

        To set or update color, refer to the constructor of
        ``TrionTheme``/ ``TitaniumTheme``

        :param pin_type: pin type of the lead pin
        :type pin_type: str
        """
        width = 55
        height = 15

        if self.is_titanium:
            self.color_theme = TitaniumTheme()
        else:
            self.color_theme = TrionTheme()

        lead_pin = LeadPin(self.pin_name, width=width, height=height)
        lead_pin.pin_type = pin_type

        if pin_type in DedicatedTheme.dedicated_pin2color_func or \
            pin_type in DedicatedTheme.semi_color:
            # Get the QColor function for setting specific color
            self.color_theme = DedicatedTheme()
            color, sec_color = self.color_theme.get_color(pin_type)

            if pin_type in DedicatedTheme.semi_color:
                if pin_type == "Dedicated Power or Ground":
                    lead_pin = GNDVCCLeadPin(self.pin_name, color, sec_color)
                    lead_pin.set_size(width, height)
            else:
                lead_pin.pen_color = lead_pin.brush_color = color

            self.item = lead_pin

        else:
            if self.color_theme.is_semi_color(pin_type):
                self.set_special_lead_pin(pin_type, width, height)
            else:
                lead_pin.pen_color, lead_pin.brush_color = self.color_theme.get_color(pin_type)
                self.item = lead_pin

    def set_special_lead_pin(self, pin_type: ALL_PIN_TYPE, width: int, height: int):
        """
        Generate rectangle with 2 color (left and right)

        To set or update color, refer to the constructor of
        ``TrionTheme``/ ``TitaniumTheme``

        :param pin_type: pin type of lead pin
        :param width: width of lead pin
        :param height: height of lead pin
        """
        assert self.color_theme is not None

        first_color, sec_color = self.color_theme.get_color(pin_type)

        spec_rec = SemiRect(self.pin_name, first_color, sec_color)
        spec_rec.set_size(width, height)
        spec_rec.pin_type = pin_type
        self.item = spec_rec

    def set_ball_container(self, size: Tuple[float, float]):
        """
        Set ball container by the given size

        :param size: width and height of the ball container
        :type size: Tuple[float, float]
        """
        width, height = size
        ball_container = QPolygonF([
            QPointF(0, 0), QPointF(0, height),
            QPointF(width, height), QPointF(width, 0),
            QPointF(0, 0)])

        self.polygon_svr.setPolygon(ball_container)
        self.item = self.polygon_svr

    def set_ball_pin(self, pin_type: ALL_PIN_TYPE):
        """
        Set ball pin icon by the given pin type.

        To set or update color, refer to the constructor of
        ``TrionTheme``/ ``TitaniumTheme``

        :param pin_type: pin type of the ball pin
        :type pin_type: str
        """
        width = 15

        # Dedicated ball pin
        if pin_type in DedicatedTheme.dedicated_pin2color_func or \
            pin_type in DedicatedTheme.semi_color:
            ball_pin = self.generate_dedicated_ball_pin_icon(pin_type, width)

        # Titanium/ Trion pin
        else:
            ball_pin = self.generate_other_ball_pin_icon(pin_type, width)

        self.item = ball_pin

    def generate_other_ball_pin_icon(self, pin_type: ALL_PIN_TYPE, radius: float) -> SemiCircle| BallPinIcon:
        """
        Generate Trion/ Titanium ball pin (Non-dedicated ball pin)

        :param pin_type: pin type of ball pin
        :param radius: radius of ball pin
        """
        if self.is_titanium:
            self.color_theme = TitaniumTheme()
        else:
            self.color_theme = TrionTheme()

        first_color, sec_color = self.color_theme.get_color(pin_type)

        ball_pin: Optional[BallPinIcon] = None

        if isinstance(self.color_theme, TrionTheme) and self.color_theme.is_semi_color(pin_type):
            ball_pin = SemiCircle(self.pin_name, first_color, sec_color)

        elif isinstance(self.color_theme, TitaniumTheme) and self.color_theme.is_semi_color(pin_type):
            ball_pin = SemiCircle(self.pin_name, first_color, sec_color)
            pen_color = self.color_theme.get_semi_pen_color(pin_type)
            ball_pin.set_pen(pen_color)
        elif pin_type == "Transceiver":
            ball_pin = SERDESBallPin(self.pin_name, first_color, sec_color)
        else:
            ball_pin = BallPinIcon(self.pin_name, first_color, sec_color)

        assert ball_pin is not None
        ball_pin.set_diameter(radius)
        ball_pin.pin_type = pin_type

        return ball_pin

    def generate_dedicated_ball_pin_icon(self, pin_type: ALL_PIN_TYPE, width: int):
        """
        Set dedicated ball pin icon by the given pin type.

        To set or update color, refer to the constructor of
        ``TrionTheme``/ ``TitaniumTheme``

        :param pin_type: pin type of the dedicated ball pin
        :type pin_type: str
        """
        color_theme = DedicatedTheme()
        color, sec_color = color_theme.get_color(pin_type)

        pin_icon = None
        if color_theme.is_semi_color(pin_type):
            if pin_type == "Dedicated Power or Ground":
                pin_icon = GNDVCCBallPin(self.pin_name, color, sec_color, width)
        else:
            pin_icon = PinRect(self.pin_name, color, width)

        if pin_icon is not None:
            pin_icon.pin_type = pin_type

        return pin_icon

    def set_io_bank(self, io_bank_name: str, is_lead: bool):
        """
        Set I/O bank rect by the given number of I/O bank and package type.

        To add/ change I/O bank color, refer to ``IOBankTheme``

        :param num: nth of I/O bank group
        :type num: int
        :param is_lead: is lead package
        :type is_lead: bool
        """
        self.color_theme = IOBankTheme()
        iobank_icon = self.create_rect(is_lead)
        self.polygon_svr.setPolygon(iobank_icon)

        color = self.color_theme.get_color(io_bank_name)
        self.polygon_svr.setBrush(QBrush(color))
        self.polygon_svr.setPen(QPen(color, 2, Qt.NoPen))

        self.item = self.polygon_svr

    def set_mipi_rx_group(self, num: int, is_lead: bool):
        """
        Set MIPI Rx rect by the given number of MIPI Rx and package type.

        To add/ change MIPI Rx color, refer to ``MIPIRxGroupTheme``

        :param num: nth of mipi rx group
        :type num: int
        :param is_lead: is lead package
        :type is_lead: bool
        """
        self.color_theme = MIPIRxGroupTheme()
        mipi_icon:QPolygonF = self.create_rect(is_lead)
        self.polygon_svr.setPolygon(mipi_icon)

        color:QColor = self.color_theme.get_color(num)
        self.polygon_svr.setBrush(QBrush(color))
        self.polygon_svr.setPen(QPen(color, 2, Qt.NoPen))

        self.item = self.polygon_svr

    def create_rect(self, is_lead: bool):
        """
        Generate highlight rectangle by the given package type

        :param is_lead: is lead package
        :type is_lead: bool
        :return: Rectangle
        :rtype: QPolygonF
        """
        width = 50
        height = 50

        if is_lead:
            width = (55 + 5) * -1
            height = 55
            highlight_icon = QPolygonF([
                        QPointF(0, 0), QPointF(width, 0),
                        QPointF(width, height), QPointF(0, height),
                        QPointF(0, 0)])
        else:
            highlight_icon = QPolygonF([
                        QPointF(0, 0), QPointF(width, 0),
                        QPointF(width, height), QPointF(0, height),
                        QPointF(0, 0)])
        return highlight_icon

    def set_func(self, func_name: PinFunction, is_lead: bool):
        """
        Generate function icon by the given function name and package type

        :param func_name: function type
        :type func_name: PinFunction
        :param is_lead: is lead package
        :type is_lead: bool
        """
        self.color_theme = FuncTheme()
        iobank_icon = self.create_rect(is_lead)
        self.polygon_svr.setPolygon(iobank_icon)

        color = self.color_theme.get_color(func_name)
        func_pattern = self.color_theme.get_pattern(func_name)

        brush = QBrush(color)
        brush.setStyle(func_pattern)
        self.polygon_svr.setBrush(brush)
        self.polygon_svr.setPen(QPen(Qt.darkGray, 2, Qt.SolidLine))

        self.item = self.polygon_svr


class LeadCorner(QGraphicsObject):
    """
    A mark for indicating lead package orientation
    """
    def __init__(self, slop: float, n_of_lead: int, is_bottom: bool = False, parent = None):
        super().__init__(parent)
        self.width = 100
        self.slop = slop
        self.set_size(n_of_lead)
        self.is_bottom = is_bottom
        self.pen = QPen(Qt.black)

    def set_size(self, n_of_lead: int):
        """
        Set the size of the icon by number of lead pins

        :param n_of_lead: number of lead pins
        :type n_of_lead: int
        """
        self.pin_width: int = n_of_lead // 4 // 9
        self.lead_size: int = 25 * self.pin_width

    # pylint: disable=invalid-name
    def boundingRect(self):
        width = hight = (50*self.pin_width)/8 * 6
        n_of_lead = self.pin_width
        lead_size = self.lead_size

        return QRectF((50 * n_of_lead-lead_size)/4 + self.slop, (50 * n_of_lead-lead_size)/4, width, hight)

    # pylint: disable = unused-argument
    def paint(self, painter:QPainter, option: QStyleOptionGraphicsItem, widget: Optional[QWidget] = None):
        """
        Qt built-in function: Create lead corner
        """
        n_of_lead = self.pin_width
        lead_size = self.lead_size

        color = Qt.black
        self.setToolTip("Top")

        if self.is_bottom:
            color = Qt.blue
            self.setToolTip("Bottom")

        painter.setPen(QPen(color, n_of_lead * 4 * 9 / 100, Qt.SolidLine))

        painter.drawEllipse(QRectF((50 * n_of_lead-lead_size)/2 + self.slop, (50 * n_of_lead-lead_size)//2, lead_size, lead_size))

        painter.drawLine(QLineF((50 * n_of_lead - lead_size/2)/2 + self.slop, (50 * n_of_lead)/2,
                                (50 * n_of_lead-lead_size)/4 + self.slop, (50 * n_of_lead)/2))
        painter.drawLine(QLineF((50 * n_of_lead - lead_size/2)/2 + self.slop + lead_size, (50 * n_of_lead)/2,
                                (50 * n_of_lead-lead_size)/4 + self.slop + lead_size, (50 * n_of_lead)/2))
        painter.drawLine(QLineF((50 * n_of_lead)/2 + self.slop, (50 * n_of_lead-lead_size)/4,
                                (50 * n_of_lead)/2 + self.slop, (50 * n_of_lead)/8 * 3))
        painter.drawLine(QLineF((50 * n_of_lead)/2 + self.slop, (50 * n_of_lead)/8 * 5,
                                (50*n_of_lead)/2 + self.slop, (50 * n_of_lead)/8 * 7))
        # Middle cross
        painter.drawLine(QLineF((50 * n_of_lead)/2 - lead_size/10 + self.slop, (50*n_of_lead)/2, (50*n_of_lead)/2 + lead_size/10 + self.slop, (50 * n_of_lead)/2))
        painter.drawLine(QLineF((50 * n_of_lead)/2 + self.slop, (50 * n_of_lead)/2 - lead_size/10, (50*n_of_lead)/2 + self.slop, (50 * n_of_lead)/2 + lead_size/10))

