'''
Copyright (C) 2017-2022 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.


Created on September 16, 2022

@author: Shirley Chan
'''
import os
import sys
from enum import Enum, unique
from pathlib import Path
from typing import List, Optional

from PyQt5 import QtCore
from PyQt5.QtWidgets import QDialog, QFileDialog, QMessageBox, QRadioButton, QWidget
from PyQt5.QtCore import Qt
import util.gen_util
from util.gui_util import WindowsFilenameValidator
from util.singleton_logger import Logger

sys.path.append(os.path.dirname(__file__))
from Ui_export_diagram import Ui_Dialog as EXPORT_DIAGRAM_UIFORM_CLASS

@unique
class FileFormat(Enum):
    PNG = "png"
    PDF = "pdf"


@util.gen_util.freeze_it
class ExportDiagram(QtCore.QObject):
    """
    Dialog to export package planner diagram
    """

    #: Signal when user select ok to export diagram
    sig_export_diagram_request = QtCore.pyqtSignal()

    def __init__(self, design_name: str, design_loc: str, parent: Optional[QWidget]=None):
        """
        Constructor
        """

        super().__init__(parent)

        self._last_directory = design_loc  #: Last directory in the open image dialog

        self.dialog = QDialog(parent)
        self.ui_dialog = EXPORT_DIAGRAM_UIFORM_CLASS()
        self.dialog.setWindowFlag(Qt.WindowContextHelpButtonHint, False)
        self.ui_dialog.setupUi(self.dialog)
        self._filename = ""

        self.default_file_name = f"{design_name}_pinout"

        self.file_type = FileFormat.PNG

        self.set_default_filename(self.file_type)
        self.monitor_changes()

        self.logger = Logger

    def monitor_changes(self):
        self.ui_dialog.le_name.editingFinished.connect(self.on_add_file_name)
        self.ui_dialog.pb_set_file_path.pressed.connect(self.on_add_location)

        self.ui_dialog.buttonBox.accepted.connect(self.on_ok)
        self.ui_dialog.buttonBox.rejected.connect(self.on_cancel)

        radio_buttons = [
            self.ui_dialog.rb_export_png,
            self.ui_dialog.rb_export_pdf
        ]
        for radio_button in radio_buttons:
            radio_button.pressed.connect(self.on_file_type_select)

    def show(self):
        # The Location will always shows last chosen location
        # if it was assigned in the same session previously.  However,
        # when user click on the push button, it will always show
        # to last chosen valid directory (either through manual typing
        # or through the push button)
        self.dialog.exec()

    def get_export_file(self) -> str:
        """
        Get the export file path
        """
        filepath: str = os.path.join(os.path.abspath(self._last_directory), self._filename)
        return filepath

    @QtCore.pyqtSlot()
    def on_file_type_select(self):
        sender = self.sender()

        if not isinstance(sender, QRadioButton):
            return

        file_type = sender.text().lower()
        self.set_file_type_by_str(file_type)

        filename_info = self._filename.split(".")

        if len(filename_info) > 1:
            filename_info[-1] = self.file_type.value

        self._filename = ".".join(filename_info)
        self.ui_dialog.le_name.setText(self._filename)

        # Show/Hide widget
        if self.file_type == FileFormat.PDF:
            self.ui_dialog.cb_transparent.setDisabled(True)
            self.ui_dialog.sb_quality.setDisabled(True)
        else:
            self.ui_dialog.cb_transparent.setDisabled(False)
            self.ui_dialog.sb_quality.setDisabled(False)

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_add_file_name(self) -> None:
        self.logger.debug("Reading filename: {}".format(self.ui_dialog.le_name.text()))
        self._filename = self.ui_dialog.le_name.text()

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_add_location(self):
        # Opens the Directory Dialog
        dialog = QFileDialog(self.parent(), "Export Design Location", self._last_directory)  # type: ignore

        dialog.setFileMode(QFileDialog.AnyFile)
        dialog.setViewMode(QFileDialog.Detail)
        dialog.setAcceptMode(QFileDialog.AcceptSave)
        # We did that when confirming the file exporting, so skip it in QFileDialog
        dialog.setOption(QFileDialog.DontConfirmOverwrite)
        dialog.setNameFilter(f"*.{self.file_type.value}")

        dialog.selectFile(self._filename)
        dirname_list = []

        if dialog.exec():
            dirname_list = dialog.selectedFiles()

        if len(dirname_list) != 1:
            return

        file_path = Path(dirname_list[0])
        if not str(file_path).endswith(f".{self.file_type.value}"):
            file_path_str = f'{file_path}.{self.file_type.value}'
        else:
            file_path_str = str(file_path)

        # Save the location to the line edit box
        self.ui_dialog.le_name.setText(file_path_str)
        self._last_directory = str(file_path.parent)

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_ok(self):
        # Check that filename is non-empty
        filename = self._filename
        validator = WindowsFilenameValidator()
        if self.ui_dialog.le_name.text() == "":
            QMessageBox.warning(
                self.parent(), "Invalid Export Diagram File Name", "File Name cannot be empty") # type: ignore
            return
        elif validator.check_windows_reserved_keyword(filename):
                QMessageBox.warning(
                        self.parent(), "Invalid Export Design File Path", f"Project path is not writable : {Path(filename)}")
                return
        elif filename.split(os.sep)[-1] == "":
                # Warning saying that export was not done
                QMessageBox.warning(
                    self.parent(), "Invalid Export Design File Path", f"Specified export file path is invalid: {filename}\nFile name not specified") # type: ignore
                return
        else:
            p = Path(filename)
            folder = p.parent
            if not folder.exists():
                # Warning saying that export was not done
                QMessageBox.warning(
                    self.parent(), "Invalid Export Design File Path", f"Specified export file path is invalid: {filename}\nOutput folder {folder} does not exist.") # type: ignore
                return

        # Check the extension of the filename
        for child in self.ui_dialog.gb_format.children():
            if not isinstance(child, QRadioButton) or \
                not child.isChecked():
                continue

            file_type = child.text().lower()
            self.set_file_type_by_str(file_type)

            if not filename.endswith(f".{file_type}"):
                QMessageBox.information(
                    self.parent(), "Export Diagram File Name Auto Correction", # type: ignore
                    f"Extension .{file_type} will be added to File Name {filename}")

                name_with_ext = f"{filename}.{file_type}"
                self.ui_dialog.le_name.setText(name_with_ext)
                filename = self.ui_dialog.le_name.text()
                self._filename = filename
            break

        fullpath = self._filename
        if os.path.exists(fullpath):
            messagebox = QMessageBox()
            messagebox.setWindowTitle("Export File Exists")

            msg = "The specified path file name {} already exists.".format(fullpath)
            messagebox.setText(msg)

            cancel_button = messagebox.addButton(
                "Cancel", QMessageBox.RejectRole)
            messagebox.addButton(
                "Overwrite", QMessageBox.ApplyRole)

            messagebox.exec()
            button = messagebox.clickedButton()

            if button == cancel_button:
                return self.on_cancel()

        # Send signal
        self.sig_export_diagram_request.emit()

    @property
    def quality(self) -> int:
        return self.ui_dialog.sb_quality.value()

    def is_transparent(self) -> bool:
        return self.ui_dialog.cb_transparent.isChecked()

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_cancel(self):
        pass

    def set_default_filename(self, file_format: FileFormat):
        assert isinstance(file_format.value, str)
        default_file_name: str = f"{self.default_file_name}.{file_format.value}"

        self._filename = str(Path(self._last_directory) / default_file_name)
        self.ui_dialog.le_name.setText(self._filename)
        self.file_type = file_format

    def set_file_type_by_str(self, str_file_type: str):
        file_type = None

        for file in FileFormat:
            if file.value == str_file_type.lower():
                file_type = file
                break

        assert file_type is not None
        self.file_type = file_type
