"""

Copyright (C) 2017-2022 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Jan 25, 2022

@author: Shirley Chan
"""

from __future__ import annotations
from enum import Enum
from typing import Optional, TYPE_CHECKING

from PyQt5 import QtCore, QtGui, QtWidgets

import util.gen_util
from util.singleton_logger import Logger
from util.widget_formatter import WidgetFormatter

from design.db import PeriDesign
from design.package_pin import PackagePinService, PkgItemType

from common_gui.base_control import BaseController

import pkg_gui.gui.main_window as pinout_window
import pkg_gui.gui.config as pinout_config
import pkg_gui.gui.viewer as pinout_viewer
import pkg_gui.gui.legend_view as legend_viewer
from pkg_gui.gui.color_theme import ColorTheme
from pkg_gui.gui.progress_bar import LoadingScreen
from pkg_gui.gui.export_diagram import ExportDiagram

if TYPE_CHECKING:
    from package_gui.package_window import MainWindow

@util.gen_util.freeze_it
class PinoutController(BaseController):
    """
    Controller for pinout diagram setting
    """

    def __init__(self):
        super().__init__()

        self.main_window: Optional[pinout_window.PinoutWindow]
        self.config: Optional[pinout_config.PinoutConfig]
        self.viewer: Optional[pinout_viewer.PinoutViewer]
        self.legend_window: Optional[legend_viewer.LegendViewer] = None
        self.export_diagram: Optional[ExportDiagram] = None

        self.color_theme = ColorTheme()
        self.logger = Logger

        self.progress_bar = None
        self.pkg_inst: Optional[PackagePinService] = None
        self.app_window: 'MainWindow'

    def build(self, design_db: PeriDesign, block_inst_name):
        """
        Build the main window to display design db content
        for the specified package. It then proceed to monitor
        user action in the Interface Designer main application window.

        :param design_db: Design db instance
        :param block_inst_name: Block instance (not in use, called by BasePlugin)
        """

        self.progress_bar = LoadingScreen(self.app_window)
        self.progress_bar.setup_ui()
        self.progress_bar.show()

        if self.main_window is None:
            self.setup_ui()

        assert self.main_window is not None
        self.main_window.update_design(design_db)

        pkg_svr = PackagePinService(design_db)
        self.pkg_inst = pkg_svr.build()
        assert self.pkg_inst is not None, "Error in building PackagePinService"

        # Prevent blocking of Qt event
        QtCore.QCoreApplication.processEvents()

        if self.viewer is not None:
            self.viewer.pkg_inst = self.pkg_inst
            self.viewer.build(design_db, self.pkg_inst)

        if self.config is not None:
            assert self.viewer is not None
            self.config.build(design_db, self.pkg_inst)
            self.monitor_changes()

            if self.viewer.model is not None:
                self.viewer.model.sig_pin_selected.connect(self.on_pin_selected)
                self.config.sig_show_all_iobank.connect(self.on_show_all_iobank)
                self.config.sig_show_individual_iobank.connect(self.on_show_individual_iobank)
                self.config.sig_show_all_mipi_group.connect(self.on_show_all_mipi_group)
                self.config.sig_show_individual_mipi_group.connect(self.on_show_individual_mipi_group)
                self.config.sig_show_func.connect(self.on_show_func)
                self.config.sig_show_assigned_pins.connect(self.on_show_assigned_pins)
            else:
                self.logger.error('Fail to set pinout diagram model')

            self.export_diagram = ExportDiagram(design_db.name, design_db.location, self.app_window)
            self.export_diagram.sig_export_diagram_request.connect(self.on_export_diagram)

        if self.legend_window is not None:
            assert self.viewer is not None
            self.legend_window.pkg_inst = self.viewer.pkg_inst
            self.legend_window.build(design_db, self.color_theme)

        self.progress_bar.close()

    def monitor_changes(self):
        assert self.main_window is not None
        self.main_window.actionFit_to_Window.triggered.connect(self.on_fit_to_window)
        self.main_window.actionZoom_in.triggered.connect(self.on_zoom_in)
        self.main_window.actionZoom_out.triggered.connect(self.on_zoom_out)
        self.main_window.actionLeft_Rotate.triggered.connect(self.on_left_rotate)
        self.main_window.actionRight_Rotate.triggered.connect(self.on_right_rotate)
        self.main_window.actionPackage_Bottom.triggered.connect(self.on_view_pkg_bottom)
        self.main_window.actionOpen_legend.triggered.connect(self.on_show_hide_legend)
        self.main_window.actionWorld_View.triggered.connect(self.on_show_hide_world_view)
        self.main_window.actionPin_Browser.triggered.connect(self.on_show_hide_pin_browser)
        self.main_window.actionReset_Orientation.triggered.connect(self.on_reset_orientation)
        self.main_window.actionExport_Diagram.triggered.connect(self.on_open_export_dialog)
        self.app_window.sig_iobank_std_change.connect(self.on_update_iobank_std)

    def setup_ui(self):
        self.main_window = pinout_window.PinoutWindow(None)
        self.config = pinout_config.PinoutConfig(self.main_window)
        self.viewer = pinout_viewer.PinoutViewer(self.main_window, self.config)
        self.legend_window = legend_viewer.LegendViewer(self.app_window)

        formatter = WidgetFormatter()
        formatter.format_plugin_window(self.main_window)

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_show_hide_config(self):
        assert self.main_window is not None
        self.main_window.show_config_window()

    # noinspection PyArgumentList
    @QtCore.pyqtSlot(str)
    def on_highlight_pin(self, pin_name):
        assert self.viewer is not None
        self.logger.debug(f"Highlighting the pin: {pin_name}")
        self.viewer.highlight_pin(pin_name)

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_show_hide_pin_browser(self):
        assert self.app_window is not None

        QtGui.QGuiApplication.setOverrideCursor(QtCore.Qt.WaitCursor)
        self.app_window.actionOpen_pin_browser.triggered.emit()
        QtGui.QGuiApplication.restoreOverrideCursor()

    # noinspection PyArgumentList
    @QtCore.pyqtSlot(bool)
    def on_show_all_iobank(self, is_show: bool):
        assert self.viewer is not None
        self.viewer.show_all_iobank(is_show)

    # noinspection PyArgumentList
    @QtCore.pyqtSlot(str, bool)
    def on_show_individual_iobank(self, iobank_name: str, is_show: bool):
        assert self.viewer is not None
        self.viewer.show_individual_iobank(iobank_name, is_show)

    # noinspection PyArgumentList
    @QtCore.pyqtSlot(bool)
    def on_show_all_mipi_group(self, is_show: bool):
        assert self.viewer is not None
        self.viewer.show_all_mipi(is_show)

    # noinspection PyArgumentList
    @QtCore.pyqtSlot(str, bool)
    def on_show_individual_mipi_group(self, group_name: str, is_show: bool):
        assert self.viewer is not None
        self.viewer.show_individual_mipi(group_name, is_show)

    # noinspection PyArgumentList
    @QtCore.pyqtSlot(str)
    def on_iobank_voltage_change(self, iobank_name: str):
        self.logger.debug(f'Updating voltage for I/O bank: {iobank_name}')
        # Do not need to update the diagram (no I/O bank voltage display)

        assert self.config is not None
        if self.config.pin_name != "":
            self.config.show_pin_info(self.config.pin_name)

    # noinspection PyArgumentList
    @QtCore.pyqtSlot(str)
    def on_iobank_std_change(self, res_name):
        self.logger.debug(f'Updating I/O standard for resource: {res_name}')
        # Do not need to update the diagram (no I/O standard display)

        assert self.config is not None
        if self.config.pin_name != "":
            self.config.show_pin_info(self.config.pin_name)

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_clear_selection(self):
        self.logger.debug('Cleaning selection')
        assert self.viewer is not None
        self.viewer.clear_selection()

    # noinspection PyArgumentList
    @QtCore.pyqtSlot(str, str)
    def on_blk_res_select(self, blk_type, res_name):
        self.logger.debug(f'block type {blk_type} is selected with resource name:{res_name}')
        assert self.viewer is not None
        assert self.config is not None
        self.viewer.highlight_blk_res(blk_type, res_name)

        is_select, pin_name =  self.viewer.is_1_pin_highlighted()

        # Show pin info if 1 pin selected
        if is_select:
            self.on_pin_selected(pin_name)

    # noinspection PyArgumentList
    @QtCore.pyqtSlot(Enum, bool)
    def on_show_func(self, func, is_show):
        assert self.viewer is not None
        self.viewer.show_func(func, is_show)

    # noinspection PyArgumentList
    @QtCore.pyqtSlot('QString')
    def on_pin_selected(self, pin_name):
        self.logger.debug(f"pin_name: {pin_name} in control")
        assert self.config is not None
        self.config.show_pin_info(pin_name)

    # noinspection PyArgumentList
    @QtCore.pyqtSlot(bool)
    def on_show_assigned_pins(self, is_show):
        assert self.viewer is not None
        self.viewer.show_assigned_pins(is_show)

    @QtCore.pyqtSlot()
    def on_refresh_screen(self):
        assert self.config is not None and self.viewer is not None

        QtGui.QGuiApplication.setOverrideCursor(QtCore.Qt.WaitCursor)
        self.viewer.refresh_model()
        if self.config.pin_name != "":
            self.config.show_pin_info(self.config.pin_name)
        QtGui.QGuiApplication.restoreOverrideCursor()

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_reset_orientation(self):
        assert self.viewer is not None

        QtGui.QGuiApplication.setOverrideCursor(QtCore.Qt.WaitCursor)
        self.viewer.reset_orientation()
        self.logger.debug("Orientation is reset")
        self.viewer.update_pkg_bottom_icon(self.app_window)
        QtGui.QGuiApplication.restoreOverrideCursor()

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_open_export_dialog(self):
        assert self.export_diagram is not None
        self.export_diagram.show()

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_export_diagram(self):
        assert self.viewer is not None
        assert self.export_diagram is not None

        img_file = self.export_diagram.get_export_file()

        self.logger.debug("Export diagram: {}".format(img_file))

        self.display_message("Export design...", is_timeout=False)

        QtGui.QGuiApplication.setOverrideCursor(QtCore.Qt.WaitCursor)
        is_success = self.viewer.gen_diagram_file(img_file, self.export_diagram.is_transparent(), self.export_diagram.file_type, self.export_diagram.quality)
        QtGui.QGuiApplication.restoreOverrideCursor()

        if is_success:
            self.display_message("Export diagram...done.")
        else:
            self.display_message("Fail to export diagram.")
            QtWidgets.QMessageBox.warning(self.main_window, "Fail to save file", "Fail to save file")

    # noinspection PyArgumentList
    @QtCore.pyqtSlot(str, str, str)
    def on_update_assign_resource(self, item_type, ori_res, res_name):
        assert self.viewer is not None
        assert self.config is not None

        self.viewer.update_pin_info(item_type, ori_res, res_name)

        if self.config.pin_name != "":
            self.config.show_pin_info(self.config.pin_name)

    @QtCore.pyqtSlot()
    def on_fit_to_window(self):
        assert self.viewer is not None
        self.viewer.on_zoom_fit()

    @QtCore.pyqtSlot()
    def on_zoom_in(self):
        assert self.viewer is not None
        self.viewer.zoom_in()

    @QtCore.pyqtSlot()
    def on_zoom_out(self):
        assert self.viewer is not None
        self.viewer.zoom_out()

    @QtCore.pyqtSlot()
    def on_left_rotate(self):
        assert self.viewer is not None

        QtGui.QGuiApplication.setOverrideCursor(QtCore.Qt.WaitCursor)
        self.viewer.rotate("left")
        QtGui.QGuiApplication.restoreOverrideCursor()

    @QtCore.pyqtSlot()
    def on_right_rotate(self):
        assert self.viewer is not None

        QtGui.QGuiApplication.setOverrideCursor(QtCore.Qt.WaitCursor)
        self.viewer.rotate("right")
        QtGui.QGuiApplication.restoreOverrideCursor()

    @QtCore.pyqtSlot()
    def on_view_pkg_bottom(self):
        assert self.viewer is not None

        self.viewer.view_pkg_bottom()
        self.viewer.update_pkg_bottom_icon(self.app_window)

    # noinspection PyArgumentList
    @QtCore.pyqtSlot(str)
    def on_update_iobank_std(self, res_name: str):
        assert self.viewer is not None and self.pkg_inst is not None
        if self.pkg_inst.is_titanium():
            # If gpio instance have differential STL I/O Standard,
            # both P/N pins are used (Display in tooltip)
            if "_P_" in res_name or "_N_" in res_name:
                p_pin = res_name.replace("_N_", "_P_")
                n_pin = res_name.replace("_P_", "_N_")
                self.viewer.update_pin_info(PkgItemType.GPIO.value, p_pin, n_pin)

    def on_show_hide_legend(self):
        assert self.legend_window is not None
        self.legend_window.show_hide_legend()

    def on_show_hide_world_view(self):
        assert self.viewer is not None
        self.viewer.show_hide_world_view()

    def display_message(self, msg: str, is_timeout: bool = True, is_status_bar: bool = True,
                        is_console: bool = True, is_log: bool = True) -> None:
        util.gen_util.mark_unused(is_console)
        assert self.app_window is not None

        if is_status_bar is True:
            if is_timeout is True:
                # Show message for 10 seconds, then auto clear
                self.app_window.status_bar.showMessage(msg, 10000)
            else:
                self.app_window.status_bar.showMessage(msg)

            QtWidgets.QApplication.processEvents()

        if is_log:
            self.logger.info(msg)


if __name__ == "__main__":
    pass
