"""

Copyright (C) 2017-2023 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Aug 2, 2023

@author: Shirley Chan
"""

from __future__ import annotations
from typing import Any, Optional, Callable, Tuple, Dict
import re
import string

from PyQt5.QtCore import Qt
from PyQt5.QtGui import QColor

from design.package_pin import PinFunction
from util.singleton_logger import Logger


class ColorTheme:
    """
    Base class for generating QColor
    """

    NC_color = (221, 221, 221)

    def __init__(self):
        self.color = QColor()
        self.sec_color = QColor()
        self.logger = Logger

    def get_color(self, color_type: Any):
        '''
        Give a default color (Grey)
        '''
        self.logger.debug(f"{self.__class__.__name__}.get_color: color_type={color_type}")
        self.color.setRgb(self.NC_color[0], self.NC_color[1], self.NC_color[2])
        return self.color, self.color


class DedicatedTheme(ColorTheme):
    dedicated_pin2color_func: Dict[str, Callable[[QColor], None]] = {
        "Dedicated JTAG":lambda color:color.setRgb(72, 95, 171),
        "Dedicated Ground":lambda color:color.setRgb(0, 0, 0),
        "Dedicated Configuration":lambda color:color.setRgb(27, 135, 201),
        "Dedicated Power":lambda color:color.setRgb(212, 32, 38),
        "Dedicated SPI Flash":lambda color:color.setRgb(27, 135, 201)
    }
    semi_color = {
        "Dedicated Power or Ground":((0, 0, 0), (212, 32, 38)),
    }
    def get_color(self, color_type: str):
        """
        Get 2 color for trion pin type

        :param color_type: Trion pin type
        """
        if not (color_type in self.dedicated_pin2color_func or color_type in self.semi_color):
            self.logger.debug(f"The type {color_type} have not define the color for dedicated pin. Please check the code")
            return self.color, self.color

        set_color = self.dedicated_pin2color_func.get(color_type, None)

        if set_color is not None:
            set_color(self.color)
            self.sec_color = self.color
        else:
            colors = self.semi_color.get(color_type)
            assert colors is not None
            first_color, second_color = colors

            self.color.setRgb(first_color[0], first_color[1], first_color[2])
            self.sec_color.setRgb(second_color[0], second_color[1], second_color[2])

        return self.color, self.sec_color

    def is_semi_color(self, color_type: str):
        return color_type in self.semi_color


class TrionTheme(ColorTheme):
    """
    Class for generating QColor for Trion pin type
    """
    def __init__(self):
        self.trion_color: Dict[str, Tuple[Tuple[int, int, int], Tuple[int, int, int]]] = {
            "GPIO":((71, 143, 204), (255, 255, 255)),
            "Configuration or GPIO": ((0, 183, 211), (0, 183, 211)),
            "Multi-Function or GPIO":((66, 169, 71), (66, 169, 71)),
            "MIPI":((247, 148, 29), (247, 148, 29)),
            "LVDS or GPIO":((134, 131, 191), (134, 131, 191)),
            "LVDS Reference Resistor Pin":((99, 97, 99), (134, 131, 191)),
            "DDR":((244, 226, 151), (244, 226, 151)),
            "TEST": ((72, 95, 171), (221, 221, 221)),
        }

        # Being used in SemiCircle/ SemiRect
        self.semi_color: Dict[str, Tuple[Tuple[int, int, int], Tuple[int, int, int]]] = {
            "Multi-Function, Configuration, or GPIO":((66, 160, 71), (0, 183, 211)),
            "Multi-Function, LVDS, or GPIO":((66, 160, 71), (134, 131, 191)),
            "Configuration, LVDS, or GPIO":((0, 183, 211), (134, 131, 191)),
        }
        super().__init__()

    def is_semi_color(self, color_type: str):
        """
        Check if pin type is semi-color type

        :param color_type: Trion pin type
        """
        return color_type in self.semi_color

    def get_color(self, color_type: str):
        """
        Get 2 color for trion pin type

        :param color_type: Trion pin type
        """
        if color_type == 'NC':
            return super().get_color(color_type)

        if (color_type in self.trion_color or self.is_semi_color(color_type)) is False:
            self.logger.debug(f"The type {color_type} have not define the color for trion. Please check the code")
            return self.color, self.color

        if self.is_semi_color(color_type):
            attribute = "semi_color"
        else:
            attribute = "trion_color"

        color_mapper: Optional[Dict] = getattr(self, attribute, None)
        assert color_mapper is not None

        pen_brush_color = color_mapper.get(color_type)
        assert pen_brush_color is not None
        pen_c, brush_c = pen_brush_color
        self.color.setRgb(brush_c[0], brush_c[1], brush_c[2])
        brush_color = self.color

        self.sec_color.setRgb(pen_c[0], pen_c[1], pen_c[2])
        pen_color = self.sec_color
        return pen_color, brush_color


class TitaniumTheme(ColorTheme):
    """
    Class for generating QColor for Titanium pin type
    """
    def __init__(self):
        self.titanium_color = {
            # Titanium
            "HVIO":((57, 139, 203), (255, 255, 255)),
            "HVIO or Multi-Function":((57, 139, 203), (174, 213, 128)),
            "HSIO":((65, 182, 85), (255, 255, 255)),
            "HSIO or Multi-Function":((65, 182, 85), (174, 213, 128)),
            "HSIO or Configuration":((65, 182, 85), (87, 192, 239)),
            "MIPI":((247, 148, 29), (247, 148, 29)),
            "DDR":((244, 226, 151), (244, 226, 151)),
            "Reference Resistor Pin":((50, 141, 66), (0,0,0)),
            "Transceiver": ((247, 148, 29), (87, 192, 239)),
            "TEST": ((72, 95, 171), (221, 221, 221)),
        }

        self.semi_color = {
            "HSIO, Configuration, or Multi-Function":((87, 192, 239), (174, 213, 128)),
        }

        self.semi_pen_color = {
            "HSIO, Configuration, or Multi-Function":(65, 182, 85),
        }
        super().__init__()

    def get_color(self, color_type: str):
        """
        Get 2 color for Titanium pin type

        :param color_type: Titanium pin type
        """
        if color_type == 'NC':
            return super().get_color(color_type)
        if (color_type in self.titanium_color or self.is_semi_color(color_type)) is False:
            self.logger.debug(f"The type {color_type} have not define the color for titanium. Please check the code")
            return self.color, self.color

        if self.is_semi_color(color_type):
            circle_color = self.semi_color.get(color_type)
            assert circle_color is not None
            upper_color, lower_color = circle_color

            self.color.setRgb(upper_color[0], upper_color[1], upper_color[2])
            upper_color = self.color

            self.sec_color.setRgb(lower_color[0], lower_color[1], lower_color[2])
            lower_color = self.sec_color
            return upper_color, lower_color

        else:
            pen_brush_color = self.titanium_color.get(color_type)
            assert pen_brush_color is not None
            pen_c, brush_c = pen_brush_color
            self.color.setRgb(brush_c[0], brush_c[1], brush_c[2])
            brush_color = self.color

            self.sec_color.setRgb(pen_c[0], pen_c[1], pen_c[2])
            pen_color = self.sec_color

            return pen_color, brush_color

    def is_semi_color(self, color_type: str):
        """
        Check if pin type is semi-color type

        :param color_type: Titanium pin type
        """
        return color_type in self.semi_color

    def get_semi_pen_color(self, color_type: str):
        '''
        Get pen for semi-circle color
        '''
        if color_type not in self.semi_pen_color:
            self.logger.debug(f"The type {color_type} have not define the pen color for titanium. Please check the code")
            return self.color

        rgb_color = self.semi_pen_color.get(color_type)
        assert rgb_color is not None
        red, green, blue = rgb_color
        color = QColor()
        color.setRgb(red, green, blue)
        return color


class IOBankTheme(ColorTheme):
    """
    Class for generating QColor for I/O bank
    """
    def __init__(self):
        # I/O bank name with pattern int + alpha
        self.iobank2color: Dict[str, Tuple[int, int, int]] = {
            "1": (255, 0, 0),
            "2": (0, 255, 193),
            "3": (255, 153, 0),
            "4": (0, 147, 255),
            "5": (175, 255, 0),
            "6": (74, 134, 255),
            "7": (0, 85, 255),
            "8": (139, 0, 247),
            "9": (0, 255, 183),
        }
        # Position/ Block related I/O Bank name
        self.pos_blk2color: Dict[str, Tuple[int, int, int]] = {
            "TL": (200, 0, 250),
            "TR": (210, 100, 0),
            "BL": (38, 252, 0),
            "BR": (182, 255, 3),
            "DDR": (5, 17, 250),
            "SERDES": (2, 137, 247),
            "SOC": (242, 75, 128),
            "CORE": (242, 175, 75),
        }
        super().__init__()

    def get_color(self, color_type: str):
        """
        Get QColor by I/O bank name

        :param color_type: I/O bank name
        :return color: QColor Object
        """
        iobank_pattern = r"^(\d+)([A-Z])$"
        pos_blk_pattern = r"([A-Z]+)(\d?)"

        # Some iobank are merged into 1, e.g. 3B_3C
        color_type = color_type.split("_")[0]

        iobank_result = re.match(iobank_pattern, color_type)
        blk_result = re.match(pos_blk_pattern, color_type)

        rgb = None
        darker_num = 0 # Times that make the color to darker

        if iobank_result is not None:
            index = iobank_result.group(1)
            alpha = iobank_result.group(2)
            rgb = self.iobank2color.get(index, None)
            darker_num = string.ascii_uppercase.index(alpha)

        elif blk_result is not None:
            position = blk_result.group(1)
            index = blk_result.group(2)
            rgb = self.pos_blk2color.get(position, None)
            if index != "":
                darker_num = int(index) + 1

        if rgb is None:
            self.logger.debug(f"The type {color_type} have not define the color for iobank. Please check the code")
            return self.color

        assert len(rgb) == 3
        r, g, b = rgb
        self.color.setRgb(r, g, b, 100)

        for _ in range(darker_num):
            self.color = self.color.darker(130)

        return self.color


class MIPIRxGroupTheme(ColorTheme):
    """
    Class for generating QColor for MIPI Rx Group
    """
    def __init__(self):
        # Assume the mipi color setting should not be limited by name,
        # it is set by the index of mipi (sorted)
        self.mipi_group2color: Dict[int, Callable[[QColor], None]] = {
            1: lambda color: color.setRgb(72, 95, 171, 100),
            2: lambda color: color.setRgb(27, 135, 201, 100),
            3: lambda color: color.setRgb(212, 32, 38, 100),
            4: lambda color: color.setRgb(128, 186, 229, 100),
            5: lambda color: color.setRgb(0, 183, 211, 100),
            6: lambda color: color.setRgb(134, 131, 191, 100),
            7: lambda color: color.setRgb(71, 143, 204, 100),
            8: lambda color: color.setRgb(66, 169, 71, 100),
            9: lambda color: color.setRgb(247, 148, 29, 100),
            10: lambda color: color.setRgb(135, 26, 37, 100),
            11: lambda color: color.setRgb(244, 226, 151, 100),
            12: lambda color: color.setRgb(233, 186, 167, 100),
            13: lambda color: color.setRgb(136, 204, 209, 100),
            14: lambda color: color.setRgb(156, 44, 124, 100),
            15: lambda color: color.setRgb(164, 212, 159, 100),
            16: lambda color: color.setRgb(243, 110, 33, 100),
            17: lambda color: color.setRgb(67, 162, 165, 100),
            18: lambda color: color.setRgb(61, 69, 156, 100),
            19: lambda color: color.setRgb(57, 145, 70, 100),
            20: lambda color: color.setRgb(193, 84, 39, 100),
            21: lambda color: color.setRgb(39, 89, 88, 100),
            22: lambda color: color.setRgb(170, 130, 187, 100),
        }
        super().__init__()

    def get_color(self, color_type: int):
        """
        Get QColor object by index of MIPIRxGroup

        :param color_type: nth of MIPI Rx group
        :return color: QColor Object
        """
        if color_type not in self.mipi_group2color:
            self.logger.debug(f"The type {color_type} have not define the color for mipi group. Please check the code")
            return self.color

        mipi_color = self.mipi_group2color.get(color_type)
        assert mipi_color is not None
        mipi_color(self.color)
        return self.color


class FuncTheme(ColorTheme):
    """
    Class for generating QColor for pin function
    """
    def __init__(self):
        # Hold color and patterns for each pin function
        self.func2color = {
            PinFunction.VREF:lambda color:color.setRgb(72, 95, 171, 100),
            PinFunction.PLL_EXTCLK:lambda color:color.setRgb(0, 19, 222, 100),
            PinFunction.PLL_REFCLK:lambda color:color.setRgb(222, 141, 0, 100),
            PinFunction.MIPI_CLK:lambda color:color.setRgb(0, 222, 222, 100),
            PinFunction.PCIE_PERSTN: lambda color: color.setRgb(235, 158, 52, 100),
            PinFunction.DDR_CLK:lambda color:color.setRgb(136, 204, 209, 100),
            PinFunction.GLOBAL_CLK:lambda color:color.setRgb(61, 235, 52, 100),
            PinFunction.GLOBAL_CTRL:lambda color: color.setRgb(235, 52, 52, 100),
            PinFunction.REGIONAL_CLK:lambda color:color.setRgb(128, 186, 229, 100),
            PinFunction.REGIONAL_CTRL:lambda color: color.setRgb(164, 212, 159, 100),
        }
        self.func2pattern = {
            PinFunction.VREF: Qt.Dense6Pattern,
            PinFunction.PLL_EXTCLK: Qt.FDiagPattern,
            PinFunction.PLL_REFCLK: Qt.BDiagPattern,
            PinFunction.MIPI_CLK: Qt.HorPattern,
            PinFunction.DDR_CLK: Qt.VerPattern,
            PinFunction.PCIE_PERSTN: Qt.Dense3Pattern,
            PinFunction.GLOBAL_CLK: Qt.DiagCrossPattern,
            PinFunction.GLOBAL_CTRL: Qt.DiagCrossPattern,
            PinFunction.REGIONAL_CLK: Qt.CrossPattern,
            PinFunction.REGIONAL_CTRL: Qt.CrossPattern,
        }
        super().__init__()

    def get_color(self, color_type: PinFunction):
        """
        Get QColor object by pin function (i.e. Class ``PinFunction``)

        :param color_type: Pin Function
        :return color: QColor object
        """
        if color_type not in self.func2color:
            self.logger.debug(f"The type {color_type} have not define the color for function. Please check the code")
            return self.color

        func2color = self.func2color.get(color_type)
        assert func2color is not None
        func2color(self.color)
        return self.color

    def get_pattern(self, func_name: PinFunction):
        """
        Get QBrush pattern by pin function

        :param color_type: Pin Function
        :return color: QColor object
        """
        if func_name not in self.func2pattern:
            self.logger.debug(f"The type {func_name} have not define the pattern for function. Please check the code")
            return Qt.SolidPattern

        func_pattern = self.func2pattern.get(func_name)
        assert func_pattern is not None
        return func_pattern

