'''
Copyright (C) 2017-2018 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.


Created on Jun 13, 2019

@author: maryam
'''
from pathlib import Path
import sys
import os

from PyQt5 import QtWidgets, QtCore
from PyQt5.QtCore import Qt
import util.gen_util
from util.gui_util import WindowsFilenameValidator
from util.singleton_logger import Logger

sys.path.append(os.path.dirname(__file__))
from Ui_export_gpio import Ui_Dialog as EXPORT_GPIO_UIFORM_CLASS

@util.gen_util.freeze_it
class ExportGPIO(QtCore.QObject):
    """
    Dialog to export GPIO assignments
    """

    #: Signal when user select ok to export gpio
    sig_export_gpio_request = QtCore.pyqtSignal()

    def __init__(self, design_name, design_loc, parent=None):
        """
        Constructor
        """

        super().__init__(parent)
        self.parent = parent

        self._last_directory = design_loc  #: Last directory in the open image dialog

        self.dialog = QtWidgets.QDialog(parent)
        self.ui_dialog = EXPORT_GPIO_UIFORM_CLASS()
        self.dialog.setWindowFlag(Qt.WindowContextHelpButtonHint, False)
        self.ui_dialog.setupUi(self.dialog)
        self._filename = ""
        self._is_isf_format_selected = True  # True = isf, False = csv. FIXME later when we have more format

        # Default filename to <project>_io.<extension>
        self.default_isf_filename = f"{design_name}_io.isf"
        self.default_csv_filename = f"{design_name}_io.csv"
        # Default is ISF
        self.set_default_filename(True)

        self.ui_dialog.le_name.editingFinished.connect(self.on_add_file_name)
        self.ui_dialog.pb_set_file_path.pressed.connect(self.on_add_location)

        self.ui_dialog.bb_export.accepted.connect(self.on_ok)
        self.ui_dialog.bb_export.rejected.connect(self.on_cancel)

        # Radiobuttons are in an exclusive group
        self.ui_dialog.rb_export_isf.toggled.connect(self.on_format_changed)

        self.logger = Logger

    def show(self):
        # The Location will always shows last chosen location
        # if it was assigned in the same session previously.  However,
        # when user click on the push button, it will always show
        # to last chosen valid directory (either through manual typing
        # or through the push button)
        self.dialog.exec()

    def disable_isf(self):
        """
        Disable isf export. Needed when new device support is not ready
        """
        self.set_default_filename(True)
        self.ui_dialog.rb_export_csv.setChecked(True)
        self.ui_dialog.gb_format.setEnabled(False)

    def get_export_file(self):
        """
        Get the export file path
        """
        return self._filename

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_add_file_name(self):
        self.logger.debug("Reading filename: {}".format(self.ui_dialog.le_name.text()))
        self._filename = self.ui_dialog.le_name.text()

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_add_location(self):
        # Do we expect use of env var?
        # TODO: check for '$' if we accept env var

        # Opens the Directory Dialog
        dialog = QtWidgets.QFileDialog(self.parent, "Export Design File", self._last_directory)

        dialog.setFileMode(QtWidgets.QFileDialog.AnyFile)
        dialog.setViewMode(QtWidgets.QFileDialog.Detail)
        dialog.setAcceptMode(QtWidgets.QFileDialog.AcceptSave)
        # We did that when confirming the file exporting, so skip it in QFileDialog
        dialog.setOption(QtWidgets.QFileDialog.DontConfirmOverwrite)
        isf_name_filter = "Interface Scripting File (*.isf)"
        csv_name_filter = "Comma Separated File (*.csv)"

        if self._is_isf_format_selected:
            dialog.setNameFilter(isf_name_filter)
        else:
            dialog.setNameFilter(csv_name_filter)

        dialog.selectFile(self._filename)
        dirname_list = []
        if dialog.exec():
            dirname_list = dialog.selectedFiles()

        if len(dirname_list) == 0 or len(dirname_list) > 1:
            return

        file_path = Path(dirname_list[0])
        file_path_str = str(file_path)
        selected_filter = dialog.selectedNameFilter()
        if selected_filter == isf_name_filter and \
                file_path.suffix != '.isf':
            file_path_str = str(file_path) + '.isf'
        elif selected_filter == csv_name_filter and \
                file_path.suffix != '.csv':
            file_path_str = str(file_path) + '.isf'

        self._filename = file_path_str
        self.ui_dialog.le_name.setText(file_path_str)
        self._last_directory = str(file_path.parent)

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_ok(self):
        invalid_input = False

        # Check that filename is non-empty
        filename = self._filename
        validator = WindowsFilenameValidator()
        if self.ui_dialog.le_name.text() == "":
            QtWidgets.QMessageBox.warning(
                self.parent, "Invalid Export Design File Name", "File Name cannot be empty")
            invalid_input = True

        else:
            # Check the path is valid
            if validator.check_windows_reserved_keyword(filename):
                QtWidgets.QMessageBox.warning(
                        self.parent, "Invalid Export Design File Path", f"Project path is not writable : {Path(filename)}")
                invalid_input = True
            elif filename.split(os.sep)[-1] == "":
                # Warning saying that export was not done
                QtWidgets.QMessageBox.warning(
                    self.parent, "Invalid Export Design File Path", f"Specified export file path is invalid: {filename}\nFile name not specified")
                invalid_input = True
            else:
                p = Path(filename)
                folder = p.parent
                if not folder.exists():
                    # Warning saying that export was not done
                    QtWidgets.QMessageBox.warning(
                        self.parent, "Invalid Export Design File Path", f"Specified export file path is invalid: {filename}\nOutput folder {folder} does not exist.")
                    invalid_input = True

            if not invalid_input:
                # Check the extension of the filename
                if self.ui_dialog.rb_export_isf.isChecked():
                    if not filename.endswith(".isf"):
                        QtWidgets.QMessageBox.information(
                            self.parent, "Export Design File Name Auto Correction",
                            "Extension .isf will be added to File Name {}".format(filename))

                        name_with_ext = filename + ".isf"
                        self.ui_dialog.le_name.setText(name_with_ext)
                        filename = self.ui_dialog.le_name.text()
                        self._filename = filename
                else:
                    if not filename.endswith(".csv"):
                        QtWidgets.QMessageBox.information(
                            self.parent, "Export Design File Name Auto Correction",
                            "Extension .csv will be added to File Name {}".format(filename))

                        name_with_ext = filename + ".csv"
                        self.ui_dialog.le_name.setText(name_with_ext)
                        filename = self.ui_dialog.le_name.text()
                        self._filename = filename

        # Check that the directory exists
        if not invalid_input:
            fullpath = self._filename

            # Check if there's already a file with the same
            # name and ask user to decide
            if os.path.exists(fullpath):
                messagebox = QtWidgets.QMessageBox()
                messagebox.setWindowTitle("Export File Exists")

                msg = "The specified path file name {} already exists.".format(fullpath)
                messagebox.setText(msg)

                cancel_button = messagebox.addButton(
                    "Cancel", QtWidgets.QMessageBox.RejectRole)
                messagebox.addButton(
                    "Overwrite", QtWidgets.QMessageBox.ApplyRole)

                messagebox.exec()
                button = messagebox.clickedButton()

                if button == cancel_button:
                    # Do nothing
                    invalid_input = True

        if not invalid_input:
            # Send signal
            self.sig_export_gpio_request.emit()

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_cancel(self):
        pass

    # noinspection PyArgumentList
    @QtCore.pyqtSlot(bool)
    def on_format_changed(self, is_checked):
        # True, ISF is selected, else CSV
        self.set_default_filename(is_checked)

    def set_default_filename(self, is_isf):
        self._is_isf_format_selected = is_isf
        if is_isf:
            default_file_name = self.default_isf_filename
        else:
            default_file_name = self.default_csv_filename

        self.ui_dialog.le_name.setText(str(Path(self._last_directory) / default_file_name))
        self._filename = str(Path(self._last_directory) / default_file_name)
