"""

Copyright (C) 2017-2018 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Apr 25, 2018

@author: yasmin
"""

from __future__ import annotations
from typing import TYPE_CHECKING, List, Dict, Optional, Tuple
import re
from enum import Enum, unique

from PyQt5 import QtWidgets, QtCore, QtGui

import util.gen_util

if TYPE_CHECKING:
    from main_gui.explorer import DesignExplorer


@util.gen_util.freeze_it
class DesignExpFinder(QtCore.QObject):
    """
    A finder that search for item in Design Explorer.
    User can specify search text and scoped down the search by specifying a filter.
    Search widgets are managed by Design Explorer except for filter options.
    Filter options control a custom QComboBox whose content/display is managed by finder.
    """

    @unique
    class FilterType(Enum):
        """
        Filter for search
        """
        display_text = "Display Text"  #: Search in item display text in explorer
        pin_name = "Pin Name"  #: Search in design item pin name
        gpio = "GPIO"  #: Search item under gpio folder
        pll = "PLL"  #: Search item under pll folder
        osc = "Oscillator"  #: Search item under osc folder
        lvds = "LVDS"  #: Search item under lvds folder
        jtag = "JTAG User Tap"  #: Search item under jtag folder
        mipi = "MIPI"  #: Search item under mipi folder
        h264 = "H.264 Encoder"  #: Search item under h264 folder
        ddr = "DDR"  #: Search item under ddr folder
        mipi_dphy = "MIPI LANE"  #: Search item under mipi dphy folder
        mipi_hard_dphy = "MIPI DPHY" #: Search item under mipi hard dphy folder
        pll_ssc = "PLL SSC" #: Search item under pll ssc folder
        spi_flash = "SPI Flash"  #: Search item under spi flash folder
        hyper_ram = "HyperRAM"  #: Search item under hyper ram folder
        quad_pcie = "PCI Express"  #: Search item under quad pcie folder
        soc = "Quad-Core RISC-V"  #: Search item under soc folder
        lane_10g = "Ethernet XGMII" #: Search item
        lane_1g = "Ethernet SGMII"  #: Search item under lane 1g folder
        raw_serdes = "PMA Direct" #: Search item under raw serdes folder

    def __init__(self, explorer: DesignExplorer):

        super().__init__()
        self.explorer = explorer  #: Design explorer
        self.filter_model = CheckableItemModel()  #: Filter view model
        self.display_text_filter_item = None
        self.pin_name_filter_item = None
        self.lbl_filter = explorer.explorer_filter.lbl_filter
        self.lw_filter = explorer.explorer_filter.lw_filter

        self.folder2filter_type: List[
            Tuple[Optional[QtWidgets.QTreeWidgetItem], DesignExpFinder.FilterType]
        ] = []

        # Filter set extracted from filter model

        # Basic blocks in all device families (Order will change search result order)
        self.current_filter = [DesignExpFinder.FilterType.display_text,
                               DesignExpFinder.FilterType.pin_name]

        for folder, filter_type in self.folder2filter_type:
            if folder is not None:
                self.current_filter.append(filter_type)

        self.proxy = QtCore.QSortFilterProxyModel()
        self.proxy.setSourceModel(self.filter_model)
        self.proxy.setFilterCaseSensitivity(QtCore.Qt.CaseInsensitive)

    def reset_filter(self):
        """
        Reset filter to its default ie all enable
        """
        if self.filter_model is not None:
            self.filter_model.set_all_check()

        if self.pin_name_filter_item is not None:
            self.pin_name_filter_item.setCheckState(QtCore.Qt.Unchecked)

        self.lbl_filter.setText("")

    def set_filter_option(self):
        """
        Set usable filter type for user selection.

        .. warning::
        It checks explorer to know what block type is available.
        Explorer in turn needs to have design loaded.
        """
        self.filter_model.clear()
        self.display_text_filter_item = self.filter_model.add_row_item("Display Text")
        self.pin_name_filter_item = self.filter_model.add_row_item("Pin Name", False)

        self.folder2filter_type: List[
            Tuple[Optional[QtWidgets.QTreeWidgetItem], DesignExpFinder.FilterType]
        ] = [
            (self.explorer.gpio_folder, DesignExpFinder.FilterType.gpio),
            (self.explorer.pll_folder, DesignExpFinder.FilterType.pll),
            (self.explorer.osc_folder, DesignExpFinder.FilterType.osc),
            (self.explorer.lvds_tx_folder, DesignExpFinder.FilterType.lvds),
            (self.explorer.mipi_tx_folder, DesignExpFinder.FilterType.mipi),
            (self.explorer.mipi_dphy_tx_folder, DesignExpFinder.FilterType.mipi_dphy),
            (self.explorer.mipi_hard_dphy_tx_folder, DesignExpFinder.FilterType.mipi_hard_dphy),
            (self.explorer.jtag_folder, DesignExpFinder.FilterType.jtag),
            (self.explorer.h264_folder, DesignExpFinder.FilterType.h264),
            (self.explorer.ddr_folder, DesignExpFinder.FilterType.ddr),
            (self.explorer.pll_ssc_folder, DesignExpFinder.FilterType.pll_ssc),
            (self.explorer.quad_pcie_folder, DesignExpFinder.FilterType.quad_pcie),
            (self.explorer.soc_folder, DesignExpFinder.FilterType.soc),
            (self.explorer.spi_flash_folder, DesignExpFinder.FilterType.spi_flash),
            (self.explorer.hyper_ram_folder, DesignExpFinder.FilterType.hyper_ram),
            (self.explorer.lane_10g_folder, DesignExpFinder.FilterType.lane_10g),
            (self.explorer.lane_1g_folder, DesignExpFinder.FilterType.lane_1g),
            (self.explorer.raw_serdes_folder, DesignExpFinder.FilterType.raw_serdes),
        ]

        # Filter set extracted from filter model
        for folder, filter_type in self.folder2filter_type:
            if folder is not None:
                self.current_filter.append(filter_type)

        for folder, filter_type in self.folder2filter_type:
            if folder is not None:
                self.filter_model.add_row_item(filter_type.value)

        self.lw_filter.setModel(self.proxy)

        # Make sure model will not update combobox display when clicking on checkbox
        self.filter_model.disconnect()
        self.lbl_filter.textChanged.connect(self.proxy.setFilterRegularExpression)

    def extract_filter(self):
        """
        Extract filter set from filter model
        """
        self.current_filter.clear()

        count = 0

        disp_text_filter = self.filter_model.item(count, 0)
        if disp_text_filter.checkState() == QtCore.Qt.Checked:
            self.current_filter.append(DesignExpFinder.FilterType.display_text)
        count += 1

        pin_name_filter = self.filter_model.item(count, 0)
        if pin_name_filter.checkState() == QtCore.Qt.Checked:
            self.current_filter.append(DesignExpFinder.FilterType.pin_name)
        count += 1

        for folder, filter_type in self.folder2filter_type:
            blk_filter = self.filter_model.item(count, 0)

            if folder is not None:
                # print(f"filter text: {blk_filter.text()}, filter_type: {filter_type}")
                if blk_filter.checkState() == QtCore.Qt.Checked:
                    self.current_filter.append(filter_type)

                count += 1

    def find(self, search_text: str):
        """
        Find an explorer item based on specified criteria. The search scoped can be controlled
        using one or more filter type. If filter is not specified, it will extract filter
        from the UI.

        :param search_text: Text to search on explorer item display text
        :return: An explorer item
        """

        target_folder_list: List[QtWidgets.QTreeWidgetItem] = []

        # Search criteria defines search scope, which top folders to search
        is_search_all = self.filter_model.is_all_checked()
        self.extract_filter()
        search_filter = self.current_filter

        if not is_search_all:

            for filter_type in search_filter:
                for folder, folder_type in self.folder2filter_type:
                    if not (filter_type == folder_type and folder is not None):
                        continue

                    match filter_type:
                        case DesignExpFinder.FilterType.mipi:
                            assert self.explorer.mipi_tx_folder is not None
                            assert self.explorer.mipi_rx_folder is not None
                            target_folder_list.append(self.explorer.mipi_tx_folder)
                            target_folder_list.append(self.explorer.mipi_rx_folder)

                        case DesignExpFinder.FilterType.mipi_dphy:
                            assert self.explorer.mipi_dphy_tx_folder is not None
                            assert self.explorer.mipi_dphy_rx_folder is not None
                            target_folder_list.append(self.explorer.mipi_dphy_tx_folder)
                            target_folder_list.append(self.explorer.mipi_dphy_rx_folder)

                        case DesignExpFinder.FilterType.mipi_hard_dphy:
                            assert self.explorer.mipi_hard_dphy_tx_folder is not None
                            assert self.explorer.mipi_hard_dphy_rx_folder is not None
                            target_folder_list.append(self.explorer.mipi_hard_dphy_tx_folder)
                            target_folder_list.append(self.explorer.mipi_hard_dphy_rx_folder)

                        case DesignExpFinder.FilterType.lvds:
                            assert self.explorer.lvds_tx_folder is not None
                            assert self.explorer.lvds_rx_folder is not None
                            target_folder_list.append(self.explorer.lvds_tx_folder)
                            target_folder_list.append(self.explorer.lvds_rx_folder)

                            if self.explorer.lvds_bidir_folder is not None:
                                target_folder_list.append(self.explorer.lvds_bidir_folder)

                        case _:
                            target_folder_list.append(folder)
                    break

        # Search for the first item that partially or fully match
        is_check_display_text = self.filter_model.is_item_checked(self.display_text_filter_item)
        is_check_pin_name = self.filter_model.is_item_checked(self.pin_name_filter_item)

        if is_search_all:
            assert self.explorer.design_folder is not None
            # Search all overwrites other filter
            iterator = QtWidgets.QTreeWidgetItemIterator(self.explorer.design_folder)
            while iterator.value():
                item = iterator.value()
                if item is not None:
                    if is_check_display_text:
                        if search_text in item.text(0):
                            return item

                    if is_check_pin_name:
                        if self.check_pin_name(item, search_text):
                            return item
                iterator += 1 # type: ignore

        else:
            for start_folder in target_folder_list:
                iterator = QtWidgets.QTreeWidgetItemIterator(start_folder)
                while iterator.value():
                    item = iterator.value()
                    if item is not None:
                        # Stop if load next block type
                        if re.match(r'(.+) \((.*)\)$', item.text(0)) and \
                            item.text(0) != start_folder.text(0):
                                break

                        if is_check_display_text:
                            if search_text in item.text(0):
                                return item

                        if is_check_pin_name:
                            if self.check_pin_name(item, search_text):
                                return item

                    iterator += 1 # type: ignore

        # Nothing found
        return None

    def check_pin_name(self, item, search_text):
        """
        Check if search text match any pin name in the target item

        :param item: Design explorer item to check
        :param search_text: Text to search in pin name
        :return: True, if match else False
        """

        db_item = item.get_db_item()
        if db_item is not None:
            return db_item.is_match_pin_name(search_text)

        return False


@util.gen_util.freeze_it
class CheckableItemModel(QtGui.QStandardItemModel):
    """
    Standard model with checkable item.
    Item is not selectable, it is meant to be checked and unchecked.
    """

    sig_check_state_changed = QtCore.pyqtSignal()

    def __init__(self, parent=None):
        super(CheckableItemModel, self).__init__(parent)

        self.count = 0  #: Row count in the model

    def add_row_item(self, item_text, is_checked=True):
        """
        Add an item row with specified text.

        :param item_text: Item display text
        :param is_checked: Check status, checked if True, else unchecked
        """
        item_list = []
        item = QtGui.QStandardItem(item_text)
        item.setFlags(QtCore.Qt.ItemIsUserCheckable | QtCore.Qt.ItemIsEnabled)

        if is_checked:
            item.setData(QtCore.Qt.Checked, QtCore.Qt.CheckStateRole)
        else:
            item.setData(QtCore.Qt.Unchecked, QtCore.Qt.CheckStateRole)

        # Disable selection since it is meant to be checked or unchecked
        item.setSelectable(False)
        item_list.append(item)

        self.insertRow(self.count, item_list)
        self.count = self.count + 1

        return item

    def is_item_checked(self, row_item):
        """
        Test if specified row item is checked

        :param row_item: Model row item
        :return: True, if checked, else False
        """

        if row_item is None:
            return False

        return row_item.checkState() == QtCore.Qt.Checked

    def is_all_checked(self):
        """
        Test if all rows are checked

        :return: True, if all are checked, else False
        """

        is_checked = True

        index = 0
        while index < self.count:
            item = self.item(index, 0)
            if item.checkState() != QtCore.Qt.Checked:
                is_checked = False
                break
            index = index + 1

        return is_checked

    def set_all_check(self):
        """
        Set all rows to checked
        """
        index = 0
        while index < self.count:
            item = self.item(index, 0)
            item.setCheckState(QtCore.Qt.Checked)
            index = index + 1

    def flags(self, index):
        """
        Override parent. Add check as part of item flags.

        :param index: item index
        :return: Item flag
        """
        return super().flags(index) | QtCore.Qt.ItemIsUserCheckable

    def data(self, index, role=QtCore.Qt.DisplayRole):
        """
        Override parent. For check state role, returns default if no data.

        :param index: row index
        :param role: model data role
        :return: Attached data or check role state
        """
        value = super().data(index, role)
        if index.isValid() and role == QtCore.Qt.CheckStateRole and value is None:
            value = QtCore.Qt.Unchecked
        return value

    # pylint: disable=C0103
    def setData(self, index, value, role=QtCore.Qt.EditRole):
        """
        Override parent. Set value as attached data. If role is check state, emit signal
        to show check state has changed.

        :param index: row index
        :param value: attached data
        :param role: model data role
        :return: True if successful, else False
        """
        is_pass = super().setData(index, value, role)
        if is_pass and role == QtCore.Qt.CheckStateRole:
            self.sig_check_state_changed.emit()

        return is_pass
