"""

Copyright (C) 2017-2018 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Sep 20, 2017

@author: yasmin
"""
from __future__ import annotations
from enum import IntEnum
import sys
import os
from typing import Optional, Union, List, TYPE_CHECKING, Dict, Tuple

from PyQt5 import QtWidgets, QtCore, QtGui
from common_device.mipi.mipi_design import MIPIRegistry

from util.singleton_logger import Logger
import util.gen_util
import util.excp as app_excp

import design.db as db
import design.db_item as db_item

import common_device.excp as blk_excp
from common_device.gpio.gpio_monitor import GPIOResourceObserver, GPIOBusResizeObserver

import an08_device.gpio.gui.bus_wizard as bus_wizard
from tx60_device.gpio.gui.bus_wizard import BusWizardComp
from tx60_device.clock_mux.clkmux_design_adv import ClockMuxRegistryAdvance
from tx180_device.clock_mux.clkmux_design_comp import ClockMuxRegistryComplex
from tx180_device.mipi_dphy.design import MIPIHardDPHYRx, MIPIHardDPHYTx
from tx375_device.clock_mux.clkmux_design_v4 import ClockMuxRegistryV4

import main_gui.explorer_item as expi
import main_gui.explorer_menu as exp_menu
import main_gui.explorer_finder as exp_finder
from main_gui.explorer_filter import ExplorerFilter
import main_gui.bus_dlg as bus_dlg

from api_service.common.object_db import APIObject

if TYPE_CHECKING:
    from common_device.ddr.ddr_design import DDR
    from common_device.gpio.gpio_design import GPIO, GPIOBus
    from common_device.hyper_ram.hyper_ram_design import HyperRAM
    from common_device.jtag.jtag_design import JTAG
    from common_device.lvds.lvds_design import LVDS
    from common_device.mipi.mipi_design import MIPI
    from common_device.osc.osc_design import OSC
    from common_device.pll.pll_design import PLL
    from common_device.spi_flash.spi_flash_design import SPIFlash
    from tx180_device.pll_ssc.design import PLLSSC
    from tx375_device.quad_pcie.design import QuadPCIE
    from tx375_device.lane10g.design import Lane10G
    from tx375_device.lane1g.design import Lane1G
    from tx375_device.raw_serdes.design import RawSerdes
    from tx375_device.soc.design import SOC
    from api_service.internal.int_design import IntDesignAPI
    from main_gui.main_window import MainWindow

sys.path.append(os.path.join(os.path.dirname(__file__), '..'))


@util.gen_util.freeze_it
class DesignExplorer(QtCore.QObject):
    """
    Design db tree view explorer
    """

    # signals
    #: Signal for when a design is selected
    sig_design_selected = QtCore.pyqtSignal('QString')

    #: Signal for when a gpio is selected
    sig_gpio_selected = QtCore.pyqtSignal('QString')
    #: Signal for when a gpio bus is selected
    sig_gpio_bus_selected = QtCore.pyqtSignal('QString')
    #: Signal for when a gpio folder or registry is selected
    sig_gpio_info_selected = QtCore.pyqtSignal()

    #: Signal for when a pll is selected
    sig_pll_selected = QtCore.pyqtSignal('QString')
    #: Signal for when a pll folder or registry is selected
    sig_pll_info_selected = QtCore.pyqtSignal()

    #: Signal for when a osc is selected
    sig_osc_selected = QtCore.pyqtSignal('QString')
    #: Signal for when an osc folder or registry is selected
    sig_osc_info_selected = QtCore.pyqtSignal()

    #: Signal for when a lvds bidir is selected
    sig_lvds_bidir_selected = QtCore.pyqtSignal('QString')
    #: Signal for when an lvds bidir folder or registry is selected
    sig_lvds_bidir_info_selected = QtCore.pyqtSignal()

    #: Signal for when a lvds is selected
    sig_lvds_tx_selected = QtCore.pyqtSignal('QString')
    #: Signal for when an lvds folder or registry is selected
    sig_lvds_tx_info_selected = QtCore.pyqtSignal()

    #: Signal for when a lvds is selected
    sig_lvds_rx_selected = QtCore.pyqtSignal('QString')
    #: Signal for when an lvds folder or registry is selected
    sig_lvds_rx_info_selected = QtCore.pyqtSignal()

    #: Signal for when iobank is selected
    sig_iobank_info_selected = QtCore.pyqtSignal()

    #: Signal for when control is selected
    sig_control_info_selected = QtCore.pyqtSignal()

    #: Signal for when clkmux is selected
    sig_clockmux_selected = QtCore.pyqtSignal('QString')
    #: Signal for when clkmux folder or registry is selected
    sig_clockmux_info_selected = QtCore.pyqtSignal()

    #: Signal for when a jtag is selected
    sig_jtag_selected = QtCore.pyqtSignal('QString')
    #: Signal for when a jtag folder or registry is selected
    sig_jtag_info_selected = QtCore.pyqtSignal()

    #: Signal for when a h264 is selected
    sig_h264_selected = QtCore.pyqtSignal('QString')
    #: Signal for when a h264 folder or registry is selected
    sig_h264_info_selected = QtCore.pyqtSignal()

    #: Signal for when a mipi is selected
    sig_mipi_selected = QtCore.pyqtSignal('QString')
    #: Signal for when a mipi folder or registry is selected
    sig_mipi_info_selected = QtCore.pyqtSignal()

    #: Signal for when a mipi is selected
    sig_mipi_tx_selected = QtCore.pyqtSignal('QString')
    #: Signal for when a mipi folder or registry is selected
    sig_mipi_tx_info_selected = QtCore.pyqtSignal()

    #: Signal for when a mipi is selected
    sig_mipi_rx_selected = QtCore.pyqtSignal('QString')
    #: Signal for when a mipi folder or registry is selected
    sig_mipi_rx_info_selected = QtCore.pyqtSignal()

    #: Signal for when a mipi dphy is selected
    sig_mipi_dphy_tx_selected = QtCore.pyqtSignal('QString')
    #: Signal for when a mipi folder or registry is selected
    sig_mipi_dphy_tx_info_selected = QtCore.pyqtSignal()

    #: Signal for when a mipi dphy is selected
    sig_mipi_dphy_rx_selected = QtCore.pyqtSignal('QString')
    #: Signal for when a mipi dphy folder or registry is selected
    sig_mipi_dphy_rx_info_selected = QtCore.pyqtSignal()

    #: Signal for when a mipi hard dphy is selected
    sig_mipi_hard_dphy_tx_selected = QtCore.pyqtSignal('QString')
    #: Signal for when a mipi folder or registry is selected
    sig_mipi_hard_dphy_tx_info_selected = QtCore.pyqtSignal()

    #: Signal for when a mipi hard dphy is selected
    sig_mipi_hard_dphy_rx_selected = QtCore.pyqtSignal('QString')
    #: Signal for when a mipi dphy folder or registry is selected
    sig_mipi_hard_dphy_rx_info_selected = QtCore.pyqtSignal()

    #: Signal for when a spi flash is selected
    sig_spi_flash_selected = QtCore.pyqtSignal('QString')
    #: Signal for when a spi flash folder or registry is selected
    sig_spi_flash_info_selected = QtCore.pyqtSignal()

    #: Signal for when a hyper ram is selected
    sig_hyper_ram_selected = QtCore.pyqtSignal('QString')
    #: Signal for when a hyper ram folder or registry is selected
    sig_hyper_ram_info_selected = QtCore.pyqtSignal()

    #: Signal for when a ddr is selected
    sig_ddr_selected = QtCore.pyqtSignal('QString')
    #: Signal for when a ddr folder or registry is selected
    sig_ddr_info_selected = QtCore.pyqtSignal()

    #: Signal for when a pll ssc is selected
    sig_pll_ssc_selected = QtCore.pyqtSignal('QString')
    #: Signal for when a pll ssc folder or registry is selected
    sig_pll_ssc_info_selected = QtCore.pyqtSignal()

    #: Signal for when a quad pcie is selected
    sig_quad_pcie_selected = QtCore.pyqtSignal('QString')
    #: Signal for when a quad pcie folder or registry is selected
    sig_quad_pcie_info_selected = QtCore.pyqtSignal()

    #: Signal for when a 10g is selected
    sig_lane_10g_selected = QtCore.pyqtSignal('QString')
    #: Signal for when a 10g folder or registry is selected
    sig_lane_10g_info_selected = QtCore.pyqtSignal()

    #: Signal for when a 1g is selected
    sig_lane_1g_selected = QtCore.pyqtSignal('QString')
    #: Signal for when a 1g folder or registry is selected
    sig_lane_1g_info_selected = QtCore.pyqtSignal()

    #: Signal for when a raw serdes is selected
    sig_raw_serdes_selected = QtCore.pyqtSignal('QString')
    #: Signal for when a raw serdes folder or registry is selected
    sig_raw_serdes_info_selected = QtCore.pyqtSignal()

    #: Signal for when a soc is selected
    sig_soc_selected = QtCore.pyqtSignal('QString')
    #: Signal for when a soc folder or registry is selected
    sig_soc_info_selected = QtCore.pyqtSignal()

    #: Signal for when a block instance is added
    sig_block_added = QtCore.pyqtSignal(int)

    #: Signal for when a block instance is deleted
    #: Args: ExploreItem type -> IntEnum, resource name -> str
    sig_block_deleted = QtCore.pyqtSignal(IntEnum, str)

    #: Signal for when a block instance resource changes, for now only gpio
    sig_gpio_res_changed = QtCore.pyqtSignal()

    #: Signal for when a block instance resource changes (For all block type)
    #: Args: ExploreItem type, original resource name, new resource name
    sig_block_res_changed =  QtCore.pyqtSignal(str, str, str)

    #: Signal for when an instance with resource is selected
    #: Args: ExploreItem type, resource name
    sig_res_highlighted = QtCore.pyqtSignal(str, str)

    #: Signal for package planner to clean selection
    sig_clear_selection = QtCore.pyqtSignal()

    #: Signal for when I/O bank voltage change
    #: Args: changed iobank name
    sig_iobank_volt_change = QtCore.pyqtSignal(str)

    #: Signal for when I/O standard change
    #: Args: Resource name of changes I/O standard
    sig_iobank_std_change = QtCore.pyqtSignal(str)

    foldertype2name_map = {
        expi.DesignExpItem.ItemType.gpio_info: "GPIO",
        expi.DesignExpItem.ItemType.pll_info: "PLL",
        expi.DesignExpItem.ItemType.osc_info: "Oscillator",
        expi.DesignExpItem.ItemType.jtag_info: "JTAG User Tap",
        expi.DesignExpItem.ItemType.h264_info: "H.264 Encoder",
        expi.DesignExpItem.ItemType.ddr_info: "DDR",
        expi.DesignExpItem.ItemType.lvds_bidir_info: "Bidirectional LVDS",
        expi.DesignExpItem.ItemType.lvds_tx_info: "LVDS TX",
        expi.DesignExpItem.ItemType.lvds_rx_info: "LVDS RX",
        expi.DesignExpItem.ItemType.mipi_tx_info: "MIPI TX",
        expi.DesignExpItem.ItemType.mipi_rx_info: "MIPI RX",
        expi.DesignExpItem.ItemType.mipi_dphy_tx_info: "MIPI TX Lane",
        expi.DesignExpItem.ItemType.mipi_dphy_rx_info: "MIPI RX Lane",
        expi.DesignExpItem.ItemType.mipi_hard_dphy_tx_info: "MIPI DPHY TX",
        expi.DesignExpItem.ItemType.mipi_hard_dphy_rx_info: "MIPI DPHY RX",
        expi.DesignExpItem.ItemType.spi_flash_info: "SPI Flash",
        expi.DesignExpItem.ItemType.hyper_ram_info: "HyperRAM",
        expi.DesignExpItem.ItemType.pll_ssc_info: "PLL SSC",
        expi.DesignExpItem.ItemType.quad_pcie_info: "PCI Express",
        expi.DesignExpItem.ItemType.soc_info: "Quad-Core RISC-V",
        expi.DesignExpItem.ItemType.lane_10g_info: "Ethernet XGMII",
        expi.DesignExpItem.ItemType.lane_1g_info: "Ethernet SGMII",
        expi.DesignExpItem.ItemType.raw_serdes_info: "PMA Direct",
    }

    def __init__(self, parent: MainWindow):
        """
        Constructor

        :param parent: App main window
        """
        QtCore.QObject.__init__(self, parent)

        self.logger = Logger

        self.parent = parent  #: App main window

        # pylint: disable=C0103
        self.actionCreate_Block = self.parent.actionCreate_Block
        # pylint: disable=C0103
        self.actionCreate_GPIO_Bus = self.parent.actionCreate_GPIO_Bus
        # pylint: disable=C0103
        self.actionDelete_Block = self.parent.actionDelete_Block

        self.actionExpand_All = self.parent.actionExpand_All
        self.actionCollapse_All = self.parent.actionCollapse_All

        self.dw_explorer = self.parent.dw_explorer
        self.tw_design_exp = self.parent.tw_design_exp
        self.tw_design_exp.setColumnCount(1)
        self.tw_design_exp.setSelectionMode(QtWidgets.QAbstractItemView.SingleSelection)

        self._menu = exp_menu.DesignExpMenu(self.tw_design_exp)

        self.le_search_text = self.parent.le_search_text
        self.pb_filter = self.parent.pb_filter

        self.explorer_filter = ExplorerFilter(self.pb_filter)
        self.pb_reset_filter = self.explorer_filter.pb_reset_filter
        self.actionShow_Hide_Search = self.parent.actionShow_Hide_Search

        self.foldertype2folder_map: Dict[expi.DesignExpItem.ItemType, Optional[expi.DesignExpItem]] = {}

        self.design: Optional[db.PeriDesign] = None  #: Design database whose items are shown in the design explorer tree
        self.design_api: Optional[IntDesignAPI] = None
        self.gpio_reg = None

        self.finder = exp_finder.DesignExpFinder(self)
        self._sel_item: Optional[expi.DesignExpItem] = None  #: Currently selected tree item
        self._dirty_item = []  #: Items marked as dirty

        self.plugin_map = None  #: Map of device to its block plugin

        self.move_to_bus_dlg = bus_dlg.MoveToBusDialog(self.parent)

        self.new_bus_wizard = None

        self.gpio_res_obs = None  #: Observer for resource change in GPIO Registry
        self.gpio_bus_obs = None  #: Observer for bus resize in
        expi.DesignExpItem.build_icons()

    @property
    def design_folder(self):
        return self.foldertype2folder_map.get(expi.DesignExpItem.ItemType.design_info, None)

    @property
    def device_folder(self):
        return self.foldertype2folder_map.get(expi.DesignExpItem.ItemType.device_info, None)

    @property
    def iobank_folder(self):
        return self.foldertype2folder_map.get(expi.DesignExpItem.ItemType.iobank_info, None)

    @property
    def control_folder(self):
        return self.foldertype2folder_map.get(expi.DesignExpItem.ItemType.control_info, None)

    @property
    def clockmux_folder(self):
        return self.foldertype2folder_map.get(expi.DesignExpItem.ItemType.clockmux_info, None)

    @property
    def gpio_folder(self):
        return self.foldertype2folder_map.get(expi.DesignExpItem.ItemType.gpio_info, None)

    @property
    def pll_folder(self):
        return self.foldertype2folder_map.get(expi.DesignExpItem.ItemType.pll_info, None)

    @property
    def osc_folder(self):
        return self.foldertype2folder_map.get(expi.DesignExpItem.ItemType.osc_info, None)

    @property
    def lvds_tx_folder(self):
        return self.foldertype2folder_map.get(expi.DesignExpItem.ItemType.lvds_tx_info, None)

    @property
    def lvds_rx_folder(self):
        return self.foldertype2folder_map.get(expi.DesignExpItem.ItemType.lvds_rx_info, None)

    @property
    def lvds_bidir_folder(self):
        return self.foldertype2folder_map.get(expi.DesignExpItem.ItemType.lvds_bidir_info, None)

    @property
    def mipi_tx_folder(self):
        return self.foldertype2folder_map.get(expi.DesignExpItem.ItemType.mipi_tx_info, None)

    @property
    def mipi_rx_folder(self):
        return self.foldertype2folder_map.get(expi.DesignExpItem.ItemType.mipi_rx_info, None)

    @property
    def mipi_dphy_tx_folder(self):
        return self.foldertype2folder_map.get(expi.DesignExpItem.ItemType.mipi_dphy_tx_info, None)

    @property
    def mipi_dphy_rx_folder(self):
        return self.foldertype2folder_map.get(expi.DesignExpItem.ItemType.mipi_dphy_rx_info, None)

    @property
    def mipi_hard_dphy_tx_folder(self):
        return self.foldertype2folder_map.get(expi.DesignExpItem.ItemType.mipi_hard_dphy_tx_info, None)

    @property
    def mipi_hard_dphy_rx_folder(self):
        return self.foldertype2folder_map.get(expi.DesignExpItem.ItemType.mipi_hard_dphy_rx_info, None)

    @property
    def spi_flash_folder(self):
        return self.foldertype2folder_map.get(expi.DesignExpItem.ItemType.spi_flash_info, None)

    @property
    def hyper_ram_folder(self):
        return self.foldertype2folder_map.get(expi.DesignExpItem.ItemType.hyper_ram_info, None)

    @property
    def jtag_folder(self):
        return self.foldertype2folder_map.get(expi.DesignExpItem.ItemType.jtag_info, None)

    @property
    def ddr_folder(self):
        return self.foldertype2folder_map.get(expi.DesignExpItem.ItemType.ddr_info, None)

    @property
    def h264_folder(self):
        return self.foldertype2folder_map.get(expi.DesignExpItem.ItemType.h264_info, None)

    @property
    def pll_ssc_folder(self):
        return self.foldertype2folder_map.get(expi.DesignExpItem.ItemType.pll_ssc_info, None)

    @property
    def quad_pcie_folder(self):
        return self.foldertype2folder_map.get(expi.DesignExpItem.ItemType.quad_pcie_info, None)

    @property
    def lane_10g_folder(self):
        return self.foldertype2folder_map.get(expi.DesignExpItem.ItemType.lane_10g_info, None)

    @property
    def lane_1g_folder(self):
        return self.foldertype2folder_map.get(expi.DesignExpItem.ItemType.lane_1g_info, None)

    @property
    def raw_serdes_folder(self):
        return self.foldertype2folder_map.get(expi.DesignExpItem.ItemType.raw_serdes_info, None)

    @property
    def soc_folder(self):
        return self.foldertype2folder_map.get(expi.DesignExpItem.ItemType.soc_info, None)


    def build(self, design, plugin_map, design_api):
        """
        Build design explorer tree based on given design

        :param plugin_map: Map of device name to its plugin
        :param design: Design db
        """

        self.plugin_map = plugin_map
        self.design = design
        self.design_api = design_api
        self.gpio_reg = self.design.gpio_reg

        if self.design.is_tesseract_design():
            self.new_bus_wizard = BusWizardComp(self.parent)
        else:
            self.new_bus_wizard = bus_wizard.BusWizard(self.parent)

        self.new_bus_wizard.sig_new_bus_request.connect(self.add_new_gpio_bus_wizard)
        self.new_bus_wizard.build(self.design)

        expi.DesignExpItem.design = self.design

        # Build design folder
        design_folder = expi.DesignExpItem(self.tw_design_exp, expi.DesignExpItem.ItemType.design_info)
        design_folder.setText(0, "Design : {}".format(design.device_def))
        var = design_folder.generate_data_var(self.design)
        design_folder.setData(0, QtCore.Qt.UserRole, var)
        self.foldertype2folder_map[expi.DesignExpItem.ItemType.design_info] = design_folder

        # Build device folder
        device_folder = expi.DesignExpItem(design_folder, expi.DesignExpItem.ItemType.device_info)
        device_folder.setText(0, "Device Setting")
        var = device_folder.generate_data_var(design.device_setting)
        device_folder.setData(0, QtCore.Qt.UserRole, var)
        self.foldertype2folder_map[expi.DesignExpItem.ItemType.device_info] = device_folder

        # supp_block = self.design.get_supported_block()
        # print(f"---------> Supported blocks: {supp_block}")

        self.build_iobank_folder()
        self.build_control_folder()
        if self.design.device_setting.clkmux_reg is not None and \
                (isinstance(self.design.device_setting.clkmux_reg, ClockMuxRegistryAdvance) or\
                        isinstance(self.design.device_setting.clkmux_reg, ClockMuxRegistryComplex) or \
                        isinstance(self.design.device_setting.clkmux_reg, ClockMuxRegistryV4)):
            self.build_clockmux_folder()

        # Build design block instance
        self.build_gpio_folder()
        self.build_pll_folder()

        if self.design.is_block_supported(db.PeriDesign.BlockType.osc) or \
                self.design.is_block_supported(db.PeriDesign.BlockType.adv_osc) or \
                self.design.is_block_supported(db.PeriDesign.BlockType.hposc):
            self.build_osc_folder()

        if self.design.is_block_supported(db.PeriDesign.BlockType.lvds):
            self.build_lvds_folder(expi.DesignExpItem.ItemType.lvds_tx_info)
            self.build_lvds_folder(expi.DesignExpItem.ItemType.lvds_rx_info)

        elif self.design.is_block_supported(db.PeriDesign.BlockType.adv_lvds):
            self.build_lvds_folder(expi.DesignExpItem.ItemType.lvds_tx_info)
            self.build_lvds_folder(expi.DesignExpItem.ItemType.lvds_rx_info)
            self.build_lvds_bidir_folder(expi.DesignExpItem.ItemType.lvds_bidir_info)

        if self.design.is_block_supported(db.PeriDesign.BlockType.mipi):
            self.build_mipi_folder(expi.DesignExpItem.ItemType.mipi_tx_info)
            self.build_mipi_folder(expi.DesignExpItem.ItemType.mipi_rx_info)

        if self.design.is_block_supported(db.PeriDesign.BlockType.mipi_dphy):
            self.build_directional_block_folder(db.PeriDesign.BlockType.mipi_dphy,
                                                expi.DesignExpItem.ItemType.mipi_dphy_tx_info,
                                                expi.DesignExpItem.ItemType.mipi_dphy_tx)
            self.build_directional_block_folder(db.PeriDesign.BlockType.mipi_dphy,
                                                expi.DesignExpItem.ItemType.mipi_dphy_rx_info,
                                                expi.DesignExpItem.ItemType.mipi_dphy_rx)

        if self.design.is_block_supported(db.PeriDesign.BlockType.mipi_hard_dphy):
            # DEVINFRA-1018: Some device does not have both Tx and Rx bonded out
            assert self.design.device_db is not None
            if self.design.is_device_block_tx_rx_supported(db.PeriDesign.BlockType.mipi_hard_dphy, True):
                self.build_mipi_hard_dphy_folder(expi.DesignExpItem.ItemType.mipi_hard_dphy_tx_info)
            if self.design.is_device_block_tx_rx_supported(db.PeriDesign.BlockType.mipi_hard_dphy, False):
                self.build_mipi_hard_dphy_folder(expi.DesignExpItem.ItemType.mipi_hard_dphy_rx_info)

        if self.design.is_block_supported(db.PeriDesign.BlockType.jtag):
            self.build_block_folder(db.PeriDesign.BlockType.jtag,
                                    expi.DesignExpItem.ItemType.jtag_info,
                                    expi.DesignExpItem.ItemType.jtag)

        if self.design.is_block_supported(db.PeriDesign.BlockType.h264):
            self.build_block_folder(db.PeriDesign.BlockType.h264,
                                    expi.DesignExpItem.ItemType.h264_info,
                                    expi.DesignExpItem.ItemType.h264)

        if self.design.is_block_supported(db.PeriDesign.BlockType.ddr) or \
                self.design.is_block_supported(db.PeriDesign.BlockType.adv_ddr):
            self.build_block_folder(db.PeriDesign.BlockType.ddr,
                                    expi.DesignExpItem.ItemType.ddr_info,
                                    expi.DesignExpItem.ItemType.ddr)

        if self.design.is_block_supported(db.PeriDesign.BlockType.spi_flash):
            self.build_block_folder(db.PeriDesign.BlockType.spi_flash,
                                    expi.DesignExpItem.ItemType.spi_flash_info,
                                    expi.DesignExpItem.ItemType.spi_flash)

        if self.design.is_block_supported(db.PeriDesign.BlockType.hyper_ram):
            self.build_block_folder(db.PeriDesign.BlockType.hyper_ram,
                                    expi.DesignExpItem.ItemType.hyper_ram_info,
                                    expi.DesignExpItem.ItemType.hyper_ram)

        if self.design.is_block_supported(db.PeriDesign.BlockType.pll_ssc):
            self.build_block_folder(db.PeriDesign.BlockType.pll_ssc,
                                    expi.DesignExpItem.ItemType.pll_ssc_info,
                                    expi.DesignExpItem.ItemType.pll_ssc)

        if self.design.is_block_supported(db.PeriDesign.BlockType.quad_pcie):
            self.build_block_folder(db.PeriDesign.BlockType.quad_pcie,
                                    expi.DesignExpItem.ItemType.quad_pcie_info,
                                    expi.DesignExpItem.ItemType.quad_pcie)

        if self.design.is_block_supported(db.PeriDesign.BlockType.lane_10g):
            self.build_block_folder(db.PeriDesign.BlockType.lane_10g,
                                    expi.DesignExpItem.ItemType.lane_10g_info,
                                    expi.DesignExpItem.ItemType.lane_10g)

        if self.design.is_block_supported(db.PeriDesign.BlockType.lane_1g):
            self.build_block_folder(db.PeriDesign.BlockType.lane_1g,
                                    expi.DesignExpItem.ItemType.lane_1g_info,
                                    expi.DesignExpItem.ItemType.lane_1g)

        if self.design.is_block_supported(db.PeriDesign.BlockType.raw_serdes):
            self.build_block_folder(db.PeriDesign.BlockType.raw_serdes,
                                    expi.DesignExpItem.ItemType.raw_serdes_info,
                                    expi.DesignExpItem.ItemType.raw_serdes)

        if self.design.is_block_supported(db.PeriDesign.BlockType.soc):
            self.build_block_folder(db.PeriDesign.BlockType.soc,
                                    expi.DesignExpItem.ItemType.soc_info,
                                    expi.DesignExpItem.ItemType.soc)

        design_folder.setExpanded(True)
        device_folder.setExpanded(True)

        self.finder.set_filter_option()
        self.update_action_status([])
        self.monitor_changes()

    def show_hide_explorer(self):
        is_show = self.dw_explorer.isVisible()
        self.dw_explorer.setVisible(not is_show)

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_refresh_import_folder(self):
        if self.design is None:
            return

        if self.gpio_folder is None and self.lvds_folder is None:
            return

        if self.plugin_map is None or not self.plugin_map.is_plugin_exist(self.design.device_def, "gpio"):
            return

        # Fixed to point to the GPIO info (top)
        item = self.design_folder
        self.tw_design_exp.setCurrentItem(item)
        self.on_item_selected(item, 0)

        if self.pll_folder is not None:
            pll_reg = self.design.get_block_reg(db.PeriDesign.BlockType.pll)
            if pll_reg is not None:
                count = pll_reg.get_inst_count()
                self.reset_block_folder(self.pll_folder, count, pll_reg)
                all_pll = pll_reg.get_all_inst()
                for pll in all_pll:
                    if pll is not None:
                        item = self.build_block_item(
                            self.pll_folder, expi.DesignExpItem.ItemType.pll, pll)
                        item.setText(0, "{} : {}".format(pll.name, pll.pll_def))

                self.pll_folder.sortChildren(0, QtCore.Qt.AscendingOrder)

                self.logger.info("Reload {} pll".format(count))

        # Due to API import support on OSC, we need to also refresh the oscillator folder
        if self.osc_folder is not None:
            osc_reg = self.design.get_block_reg(db.PeriDesign.BlockType.osc)
            if osc_reg is not None:
                count = osc_reg.get_osc_count()
                self.reset_block_folder(self.osc_folder,
                                        count,
                                        osc_reg)

                all_osc = osc_reg.get_all_inst()
                for osc in all_osc:
                    if osc is not None:
                        item = self.build_block_item(
                            self.osc_folder, expi.DesignExpItem.ItemType.osc, osc)
                        item.setText(0, "{} : {}".format(osc.name, osc.osc_def))

                self.osc_folder.sortChildren(0, QtCore.Qt.AscendingOrder)

                self.logger.info("Reload {} oscillator".format(count))

        if self.jtag_folder is not None:
            jtag_reg = self.design.get_block_reg(db.PeriDesign.BlockType.jtag)
            if jtag_reg is not None:
                count = jtag_reg.get_inst_count()
                self.reset_block_folder(self.jtag_folder,
                                        count,
                                        jtag_reg)

                all_jtag = jtag_reg.get_all_inst()
                for jtag in all_jtag:
                    if jtag is not None:
                        item = self.build_block_item(
                            self.jtag_folder, expi.DesignExpItem.ItemType.jtag, jtag)
                        item.setText(0, "{} : {}".format(jtag.name, jtag.jtag_def))

                self.jtag_folder.sortChildren(0, QtCore.Qt.AscendingOrder)

                self.logger.info("Reload {} jtag".format(count))

        if self.gpio_folder is not None:
            gpio_reg = self.design.get_block_reg(db.PeriDesign.BlockType.gpio)
            if gpio_reg is not None:
                count = gpio_reg.get_gpio_count()
                self.reset_block_folder(self.gpio_folder,
                                        count,
                                        gpio_reg)

                self.build_all_gpio_bus()

                all_gpio = gpio_reg.get_all_scalar_gpio()
                for gpio in all_gpio:
                    item = self.build_block_item(
                        self.gpio_folder, expi.DesignExpItem.ItemType.gpio, gpio)
                    item.setText(0, "{} : {}".format(gpio.name, gpio.gpio_def))

                self.gpio_folder.sortChildren(0, QtCore.Qt.AscendingOrder)

                self.logger.info("Reload {} scalar gpio".format(count))

        if self.lvds_tx_folder is not None:
            lvds_reg = self.design.get_block_reg(db.PeriDesign.BlockType.lvds)
            if lvds_reg is not None:
                count = lvds_reg.get_type_inst_count(True)
                self.reset_block_folder(self.lvds_tx_folder, count, lvds_reg)
                all_lvds = lvds_reg.get_all_inst()
                for lvds in all_lvds:
                    if lvds is not None and lvds.ops_type == lvds.OpsType.op_tx:
                        item = self.build_block_item(self.lvds_tx_folder, expi.DesignExpItem.ItemType.lvds_tx, lvds)
                        item.setText(0, "{} : {}".format(lvds.name, lvds.lvds_def))

                self.lvds_tx_folder.sortChildren(0, QtCore.Qt.AscendingOrder)

                self.logger.info("Reload {} lvds tx".format(count))

        if self.lvds_rx_folder is not None:
            lvds_reg = self.design.get_block_reg(db.PeriDesign.BlockType.lvds)
            if lvds_reg is not None:
                count = lvds_reg.get_type_inst_count(False)
                self.reset_block_folder(self.lvds_rx_folder, count, lvds_reg)

                all_lvds = lvds_reg.get_all_inst()
                for lvds in all_lvds:
                    if lvds is not None and lvds.ops_type == lvds.OpsType.op_rx:
                        item = self.build_block_item(
                            self.lvds_rx_folder, expi.DesignExpItem.ItemType.lvds_rx, lvds)
                        item.setText(0, "{} : {}".format(lvds.name, lvds.lvds_def))

                self.lvds_rx_folder.sortChildren(0, QtCore.Qt.AscendingOrder)

                self.logger.info("Reload {} lvds rx".format(count))

        if self.lvds_bidir_folder is not None:
            lvds_reg = self.design.get_block_reg(db.PeriDesign.BlockType.lvds)
            if lvds_reg is not None:
                count = lvds_reg.get_bidir_inst_count()
                self.reset_block_folder(self.lvds_bidir_folder, count, lvds_reg)

                all_lvds = lvds_reg.get_all_inst()
                for lvds in all_lvds:
                    if lvds is not None and lvds.ops_type == lvds.OpsType.op_bidir:
                        item = self.build_block_item(
                            self.lvds_bidir_folder, expi.DesignExpItem.ItemType.lvds_bidir, lvds)
                        item.setText(0, "{} : {}".format(lvds.name, lvds.lvds_def))

                self.lvds_bidir_folder.sortChildren(0, QtCore.Qt.AscendingOrder)

                self.logger.info("Reload {} lvds bidir".format(count))

        if self.mipi_tx_folder is not None:
            mipi_reg = self.design.get_block_reg(db.PeriDesign.BlockType.mipi)

            if mipi_reg is not None:
                mipi_reg: MIPIRegistry
                count = mipi_reg.get_type_inst_count(True)
                self.reset_block_folder(self.mipi_tx_folder, count, mipi_reg)
                all_mipi = mipi_reg.get_all_inst()

                for mipi in all_mipi:
                    mipi: MIPI
                    if mipi is None or mipi.ops_type != mipi.OpsType.op_tx:
                        continue
                    item = self.build_block_item(
                        self.mipi_tx_folder, expi.DesignExpItem.ItemType.mipi_tx, mipi)
                    item.setText(0, f"{mipi.name} : {mipi.get_device()}")
                self.mipi_tx_folder.sortChildren(0, QtCore.Qt.AscendingOrder)

            self.logger.info("Reload {} mipi tx".format(count))

        if self.mipi_rx_folder is not None:
            mipi_reg = self.design.get_block_reg(db.PeriDesign.BlockType.mipi)

            if mipi_reg is not None:
                mipi_reg: MIPIRegistry
                count = mipi_reg.get_type_inst_count(False)
                self.reset_block_folder(self.mipi_rx_folder, count, mipi_reg)
                all_mipi = mipi_reg.get_all_inst()

                for mipi in all_mipi:
                    mipi: MIPI
                    if mipi is None or mipi.ops_type != mipi.OpsType.op_rx:
                        continue
                    item = self.build_block_item(
                        self.mipi_rx_folder, expi.DesignExpItem.ItemType.mipi_rx, mipi)
                    item.setText(0, f"{mipi.name} : {mipi.get_device()}")

                self.mipi_rx_folder.sortChildren(0, QtCore.Qt.AscendingOrder)

            self.logger.info("Reload {} mipi tx".format(count))

        if self.mipi_dphy_tx_folder is not None:
            mipi_dphy_reg = self.design.get_block_reg(db.PeriDesign.BlockType.mipi_dphy)
            if mipi_dphy_reg is not None:
                count = mipi_dphy_reg.get_tx_inst_count()
                self.reset_block_folder(self.mipi_dphy_tx_folder, count, mipi_dphy_reg)
                self.build_directional_block_folder(db.PeriDesign.BlockType.mipi_dphy,
                                                    expi.DesignExpItem.ItemType.mipi_dphy_tx_info,
                                                    expi.DesignExpItem.ItemType.mipi_dphy_tx)
                self.logger.info("Reload {} mipi dphy tx".format(count))

        if self.mipi_dphy_rx_folder is not None:
            mipi_dphy_reg = self.design.get_block_reg(db.PeriDesign.BlockType.mipi_dphy)
            if mipi_dphy_reg is not None:
                count = mipi_dphy_reg.get_rx_inst_count()
                self.reset_block_folder(self.mipi_dphy_rx_folder, count, mipi_dphy_reg)
                self.build_directional_block_folder(db.PeriDesign.BlockType.mipi_dphy,
                                                    expi.DesignExpItem.ItemType.mipi_dphy_rx_info,
                                                    expi.DesignExpItem.ItemType.mipi_dphy_rx)
                self.logger.info("Reload {} mipi dphy rx".format(count))

        if self.mipi_hard_dphy_tx_folder is not None:
            reg = self.design.get_block_reg(db.PeriDesign.BlockType.mipi_hard_dphy)
            if reg is not None:
                count = reg.get_type_inst_count(is_tx=True)
                self.reset_block_folder(self.mipi_hard_dphy_tx_folder, count, reg)

                all_insts = reg.get_all_inst()
                for inst in all_insts:
                    if isinstance(inst, MIPIHardDPHYTx):
                        item = self.build_block_item(
                            self.mipi_hard_dphy_tx_folder, expi.DesignExpItem.ItemType.mipi_hard_dphy_tx, inst)
                        item.setText(0, "{} : {}".format(inst.name, inst.get_device()))

                self.mipi_hard_dphy_tx_folder.sortChildren(0, QtCore.Qt.AscendingOrder)

                self.logger.info("Reload {} mipi dphy tx".format(count))

        if self.mipi_hard_dphy_rx_folder is not None:
            reg = self.design.get_block_reg(db.PeriDesign.BlockType.mipi_hard_dphy)
            if reg is not None:
                count = reg.get_type_inst_count(is_tx=False)
                self.reset_block_folder(self.mipi_hard_dphy_rx_folder, count, reg)

                all_insts = reg.get_all_inst()
                for inst in all_insts:
                    if isinstance(inst, MIPIHardDPHYRx):
                        item = self.build_block_item(
                            self.mipi_hard_dphy_rx_folder, expi.DesignExpItem.ItemType.mipi_hard_dphy_rx, inst)
                        item.setText(0, "{} : {}".format(inst.name, inst.get_device()))

                self.mipi_hard_dphy_rx_folder.sortChildren(0, QtCore.Qt.AscendingOrder)

                self.logger.info("Reload {} mipi dphy rx".format(count))

        if self.hyper_ram_folder is not None:
            reg = self.design.get_block_reg(db.PeriDesign.BlockType.hyper_ram)
            if reg is not None:
                count = reg.get_inst_count()

                self.reset_block_folder(self.hyper_ram_folder,
                                        count,
                                        reg)

                all_hram = reg.get_all_inst()
                for hram in all_hram:
                    if hram is not None:
                        item = self.build_block_item(
                            self.hyper_ram_folder, expi.DesignExpItem.ItemType.hyper_ram, hram)
                        item.setText(0, "{} : {}".format(hram.name, hram.block_def))

                self.hyper_ram_folder.sortChildren(0, QtCore.Qt.AscendingOrder)

                self.logger.info("Reload {} hyper RAM".format(count))

        if self.spi_flash_folder is not None:
            reg = self.design.get_block_reg(db.PeriDesign.BlockType.spi_flash)
            if reg is not None:
                count = reg.get_inst_count()

                self.reset_block_folder(self.spi_flash_folder,
                                        count,
                                        reg)

                all_spi_flash = reg.get_all_inst()
                for spi_flash in all_spi_flash:
                    if spi_flash is not None:
                        item = self.build_block_item(
                            self.spi_flash_folder, expi.DesignExpItem.ItemType.spi_flash, spi_flash)
                        item.setText(0, "{} : {}".format(spi_flash.name, spi_flash.block_def))

                self.spi_flash_folder.sortChildren(0, QtCore.Qt.AscendingOrder)

                self.logger.info("Reload {} SPI Flash".format(count))

        if self.h264_folder is not None:
            reg = self.design.get_block_reg(db.PeriDesign.BlockType.h264)
            if reg is not None:
                count = reg.get_inst_count()
                self.reset_block_folder(self.h264_folder, count, reg)

                all_insts = reg.get_all_inst()
                for inst in all_insts:
                    item = self.build_block_item(
                        self.h264_folder, expi.DesignExpItem.ItemType.h264, inst)
                    item.setText(0, "{} : {}".format(inst.name, inst.get_device()))

                self.h264_folder.sortChildren(0, QtCore.Qt.AscendingOrder)

                self.logger.info("Reload {} H.264".format(count))


        if self.ddr_folder is not None:
            reg = self.design.get_block_reg(db.PeriDesign.BlockType.ddr)
            if reg is not None:
                count = reg.get_inst_count()

                self.reset_block_folder(self.ddr_folder,
                                        count,
                                        reg)

                all_ddr = reg.get_all_inst()
                for ddr in all_ddr:
                    if ddr is not None:
                        item = self.build_block_item(
                            self.ddr_folder, expi.DesignExpItem.ItemType.ddr, ddr)
                        item.setText(0, "{} : {}".format(ddr.name, ddr.ddr_def))

                self.ddr_folder.sortChildren(0, QtCore.Qt.AscendingOrder)

                self.logger.info("Reload {} DDR".format(count))

        # Because the clkmux has some dependencies with other block instances
        # (for RBUF related info), we need to also update the clkmux
        if self.clockmux_folder is not None and self.design.device_setting is not None:
            reg = self.design.device_setting.clkmux_reg
            if reg is not None:
                reg.update_regional_buffer_info(self.design)

        folder_info: List[Tuple[
            Optional[expi.DesignExpItem],
            db.PeriDesign.BlockType,
            expi.DesignExpItem.ItemType,
            expi.DesignExpItem.ItemType
        ]] = [
            (
                self.pll_ssc_folder,
                db.PeriDesign.BlockType.pll_ssc,
                expi.DesignExpItem.ItemType.pll_ssc,
                expi.DesignExpItem.ItemType.pll_ssc_info,
            ),
            (
                self.quad_pcie_folder,
                db.PeriDesign.BlockType.quad_pcie,
                expi.DesignExpItem.ItemType.quad_pcie,
                expi.DesignExpItem.ItemType.quad_pcie_info,
            ),
            (
                self.lane_10g_folder,
                db.PeriDesign.BlockType.lane_10g,
                expi.DesignExpItem.ItemType.lane_10g,
                expi.DesignExpItem.ItemType.lane_10g_info,
            ),
            (
                self.lane_1g_folder,
                db.PeriDesign.BlockType.lane_1g,
                expi.DesignExpItem.ItemType.lane_1g,
                expi.DesignExpItem.ItemType.lane_1g_info,
            ),
            (
                self.raw_serdes_folder,
                db.PeriDesign.BlockType.raw_serdes,
                expi.DesignExpItem.ItemType.raw_serdes,
                expi.DesignExpItem.ItemType.raw_serdes_info,
            ),
            (
                self.soc_folder,
                db.PeriDesign.BlockType.soc,
                expi.DesignExpItem.ItemType.soc,
                expi.DesignExpItem.ItemType.soc_info,
            ),
        ]
        for folder, block_type, item_type, info_type in folder_info:
            reg = self.design.get_block_reg(block_type)

            if folder is None or reg is None:
                continue

            # Reset block folder
            count = reg.get_inst_count()
            self.reset_block_folder(folder, count, reg)
            all_insts = reg.get_all_inst()

            # Build instance(s)
            for inst in all_insts:
                item = self.build_block_item(folder, item_type, inst)
                item.setText(0, "{} : {}".format(inst.name, inst.get_device()))

            folder.sortChildren(0, QtCore.Qt.AscendingOrder)

            # Logging
            type_name = self.foldertype2name_map.get(info_type, "")
            self.logger.info("Reload {} {}".format(count, type_name))


    def build_iobank_folder(self):
        """
        Build iobank grouping folder item
        """
        if self.design.device_setting is None:
            return

        iobank_reg = self.design.device_setting.iobank_reg
        if iobank_reg is not None:
            count = iobank_reg.get_iobank_count()
            device_folder = self.foldertype2folder_map.get(expi.DesignExpItem.ItemType.device_info, None)
            iobank_folder = expi.DesignExpItem(device_folder, expi.DesignExpItem.ItemType.iobank_info)
            iobank_folder.setText(0, "I/O Banks ({})".format(count))
            var = iobank_folder.generate_data_var(iobank_reg)
            iobank_folder.setData(0, QtCore.Qt.UserRole, var)
            iobank_folder.update_icon()
            self.foldertype2folder_map[expi.DesignExpItem.ItemType.iobank_info] = iobank_folder
            self.logger.info("Load {} iobank".format(count))

    def build_control_folder(self):
        """
        Build control grouping folder item
        """
        if self.design.device_setting is None:
            return

        ctrl_reg = self.design.device_setting.ctrl_reg
        if ctrl_reg is not None:
            count = ctrl_reg.get_inst_count()
            device_folder = self.foldertype2folder_map.get(expi.DesignExpItem.ItemType.device_info, None)
            control_folder = expi.DesignExpItem(device_folder, expi.DesignExpItem.ItemType.control_info)
            control_folder.setText(0, "Configuration")
            var = control_folder.generate_data_var(ctrl_reg)
            control_folder.setData(0, QtCore.Qt.UserRole, var)
            control_folder.update_icon()
            self.foldertype2folder_map[expi.DesignExpItem.ItemType.control_info] = control_folder
            self.logger.info("Load {} control".format(count))

    def build_clockmux_folder(self):
        """
        Build clockmux grouping folder item
        """
        if self.design.device_setting is None:
            return

        clkmux_reg = self.design.device_setting.clkmux_reg
        if clkmux_reg is not None:
            count = clkmux_reg.get_clock_mux_count()
            device_folder = self.foldertype2folder_map.get(expi.DesignExpItem.ItemType.device_info, None)
            clkmux_folder = expi.DesignExpItem(device_folder, expi.DesignExpItem.ItemType.clockmux_info)
            clkmux_folder.setText(0, "Clock/Control Configuration ({})".format(count))
            var = clkmux_folder.generate_data_var(clkmux_reg)
            clkmux_folder.setData(0, QtCore.Qt.UserRole, var)
            clkmux_folder.update_icon()

            self.foldertype2folder_map[expi.DesignExpItem.ItemType.clockmux_info] = clkmux_folder
            self.logger.info("Load {} clockmux".format(count))

            # Build the 4 clockmux
            # clkmux_ins_map = clkmux_reg.get_clock_mux_map()
            clkmux_list = clkmux_reg.get_all_inst()

            if clkmux_list:
                for clkmux_ins in clkmux_list:
                    # clkmux_ins = clkmux_ins_map[clkmux_name]

                    if clkmux_ins is not None:
                        clkmux_desc_name = clkmux_reg.get_clkmux_disp_name(clkmux_ins)
                        self.logger.info("Load clkmux instance {}".format(clkmux_desc_name))

                        item = expi.DesignExpItem(
                            clkmux_folder, expi.DesignExpItem.ItemType.clockmux)
                        item.setup(clkmux_ins)
                        item.setText(0, "{} : {}".format(clkmux_desc_name, clkmux_ins.block_def))

                        # self.register_dirty_item(item, True)

                # Sort the clkmux instance
                clkmux_folder.sortChildren(0, QtCore.Qt.AscendingOrder)

    def build_gpio_folder(self):
        """
        Build gpio grouping folder item
        """

        gpio_reg = self.design.get_block_reg(db.PeriDesign.BlockType.gpio)
        if gpio_reg is not None:
            count = gpio_reg.get_gpio_count()
            gpio_folder = self.build_block_folder_item(count, expi.DesignExpItem.ItemType.gpio_info, gpio_reg)
            self.foldertype2folder_map[expi.DesignExpItem.ItemType.gpio_info] = gpio_folder

            self.build_all_gpio_bus()

            all_gpio = gpio_reg.get_all_scalar_gpio()
            for gpio in all_gpio:
                item = self.build_block_item(gpio_folder, expi.DesignExpItem.ItemType.gpio, gpio)
                item.setText(0, "{} : {}".format(gpio.name, gpio.gpio_def))

            gpio_folder.sortChildren(0, QtCore.Qt.AscendingOrder)
            self.logger.info("Load {} scalar gpio".format(count))

    def build_all_gpio_bus(self):
        """
        Build all gpio bus
        """
        gpio_reg = self.design.get_block_reg(db.PeriDesign.BlockType.gpio)
        if gpio_reg is not None:
            all_bus = gpio_reg.get_all_bus()
            for bus in all_bus:
                self.build_gpio_bus(bus)

    def build_gpio_bus(self, bus_db_item):
        """
        Build gpio bus folder and its member
        """
        # For bus folder sorting to work, it must be added at the top
        bus_folder = expi.DesignExpItem.build_bus_item(self.gpio_folder, 0)
        var = bus_folder.generate_data_var(bus_db_item)
        bus_folder.setData(0, QtCore.Qt.UserRole, var)

        self.update_bus_folder(bus_folder)

        # load its gpio
        all_gpio = bus_db_item.get_member_list()
        for gpio in all_gpio:
            item = self.build_block_item(bus_folder, expi.DesignExpItem.ItemType.gpio, gpio)
            item.setText(0, "{} : {}".format(gpio.name, gpio.gpio_def))

        bus_folder.sortChildren(0, QtCore.Qt.AscendingOrder)

        return bus_folder

    def build_pll_folder(self):
        """
        Build pll grouping folder item
        """

        pll_reg = self.design.get_block_reg(db.PeriDesign.BlockType.pll)
        if pll_reg is not None:
            count = pll_reg.get_inst_count()
            pll_folder = self.build_block_folder_item(count,
                                                      expi.DesignExpItem.ItemType.pll_info,
                                                      pll_reg)

            all_pll = pll_reg.get_all_inst()
            for pll in all_pll:
                item = self.build_block_item(pll_folder, expi.DesignExpItem.ItemType.pll, pll)
                item.setText(0, "{} : {}".format(pll.name, pll.pll_def))

            pll_folder.sortChildren(0, QtCore.Qt.AscendingOrder)
            self.foldertype2folder_map[expi.DesignExpItem.ItemType.pll_info] = pll_folder

            self.logger.info("Load {} pll".format(count))

    def build_osc_folder(self):
        """
        Build osc grouping folder item
        """

        osc_reg = self.design.get_block_reg(db.PeriDesign.BlockType.osc)
        if osc_reg is not None:
            count = osc_reg.get_osc_count()
            osc_folder = self.build_block_folder_item(count, expi.DesignExpItem.ItemType.osc_info, osc_reg)

            all_osc = osc_reg.get_all_osc()
            for osc in all_osc:
                item = self.build_block_item(osc_folder, expi.DesignExpItem.ItemType.osc, osc)
                item.setText(0, "{} : {}".format(osc.name, osc.osc_def))

            osc_folder.sortChildren(0, QtCore.Qt.AscendingOrder)
            self.foldertype2folder_map[expi.DesignExpItem.ItemType.osc_info] = osc_folder

            self.logger.info("Load {} oscillator".format(count))

    def build_lvds_folder(self, folder_type: expi.DesignExpItem.ItemType):
        """
        Build lvds grouping folder item
        """

        if folder_type == expi.DesignExpItem.ItemType.lvds_tx_info:
            is_tx = True
            item_type = expi.DesignExpItem.ItemType.lvds_tx
        else:
            is_tx = False
            item_type = expi.DesignExpItem.ItemType.lvds_rx

        lvds_reg = self.design.get_block_reg(db.PeriDesign.BlockType.lvds)
        if lvds_reg is not None:
            count = lvds_reg.get_type_inst_count(is_tx)
            lvds_folder = self.build_block_folder_item(count, folder_type, lvds_reg)

            all_lvds = lvds_reg.get_all_inst()
            for lvds in all_lvds:

                if lvds is not None:
                    if is_tx and lvds.ops_type == lvds.OpsType.op_tx:
                        item = self.build_block_item(lvds_folder, item_type, lvds)
                        item.setText(0, "{} : {}".format(lvds.name, lvds.lvds_def))
                    elif not is_tx and lvds.ops_type == lvds.OpsType.op_rx:
                        item = self.build_block_item(lvds_folder, item_type, lvds)
                        item.setText(0, "{} : {}".format(lvds.name, lvds.lvds_def))

            lvds_folder.sortChildren(0, QtCore.Qt.AscendingOrder)
            self.foldertype2folder_map[folder_type] = lvds_folder

            if is_tx:
                self.logger.info(f"Load {count} lvds tx")
            else:
                self.logger.info(f"Load {count} lvds rx")

    def build_lvds_bidir_folder(self, folder_type: expi.DesignExpItem.ItemType):
        """
        Build lvds bidir grouping folder item
        """

        item_type = expi.DesignExpItem.ItemType.lvds_bidir
        lvds_reg = self.design.get_block_reg(db.PeriDesign.BlockType.lvds)
        if lvds_reg is not None:
            count = lvds_reg.get_bidir_inst_count()
            lvds_folder = self.build_block_folder_item(count, folder_type, lvds_reg)

            all_lvds = lvds_reg.get_all_bidir_inst()
            for lvds in all_lvds:
                if lvds is not None:
                    item = self.build_block_item(lvds_folder, item_type, lvds)
                    item.setText(0, "{} : {}".format(lvds.name, lvds.lvds_def))

            lvds_folder.sortChildren(0, QtCore.Qt.AscendingOrder)
            self.foldertype2folder_map[folder_type] = lvds_folder

            self.logger.info(f"Load {count} lvds bidir")

    def build_mipi_folder(self, folder_type: expi.DesignExpItem.ItemType):
        """
        Build mipi grouping folder item
        """

        if folder_type == expi.DesignExpItem.ItemType.mipi_tx_info:
            is_tx = True
            item_type = expi.DesignExpItem.ItemType.mipi_tx
        else:
            is_tx = False
            item_type = expi.DesignExpItem.ItemType.mipi_rx

        mipi_reg = self.design.get_block_reg(db.PeriDesign.BlockType.mipi)
        if mipi_reg is not None:
            count = mipi_reg.get_type_inst_count(is_tx)
            mipi_folder = self.build_block_folder_item(count, folder_type, mipi_reg)

            all_mipi = mipi_reg.get_all_inst()
            for mipi in all_mipi:
                if mipi is not None:
                    if is_tx and mipi.ops_type == mipi.OpsType.op_tx:
                        item = self.build_block_item(mipi_folder, item_type, mipi)
                    elif not is_tx and mipi.ops_type == mipi.OpsType.op_rx:
                        item = self.build_block_item(mipi_folder, item_type, mipi)
                    else:
                        item = None

                    if item is not None:
                        item.setText(0, "{} : {}".format(mipi.name, mipi.mipi_def))

            mipi_folder.sortChildren(0, QtCore.Qt.AscendingOrder)
            self.foldertype2folder_map[folder_type] = mipi_folder

            if is_tx:
                self.logger.info(f"Load {count} mipi tx lane")
            else:
                self.logger.info(f"Load {count} mipi rx lane")

    def build_mipi_hard_dphy_folder(self, folder_type: expi.DesignExpItem.ItemType):

        if folder_type == expi.DesignExpItem.ItemType.mipi_hard_dphy_tx_info:
            is_tx = True
            item_type = expi.DesignExpItem.ItemType.mipi_hard_dphy_tx
        else:
            is_tx = False
            item_type = expi.DesignExpItem.ItemType.mipi_hard_dphy_rx

        reg = self.design.get_block_reg(db.PeriDesign.BlockType.mipi_hard_dphy)
        if reg is not None:
            count = reg.get_type_inst_count(is_tx)
            folder = self.build_block_folder_item(count, folder_type, reg)

            all_mipi = reg.get_all_inst()
            for mipi in all_mipi:
                if mipi is not None:
                    if is_tx and isinstance(mipi, MIPIHardDPHYTx):
                        item = self.build_block_item(folder, item_type, mipi)
                    elif not is_tx and isinstance(mipi, MIPIHardDPHYRx):
                        item = self.build_block_item(folder, item_type, mipi)
                    else:
                        item = None

                    if item is not None:
                        item.setText(0, "{} : {}".format(mipi.name, mipi.mipi_def))

            folder.sortChildren(0, QtCore.Qt.AscendingOrder)
            self.foldertype2folder_map[folder_type] = folder

            if is_tx:
                self.logger.info(f"Load {count} mipi hard dphy tx")
            else:
                self.logger.info(f"Load {count} mipi hard dphy rx")

    def build_directional_block_folder(self, block_type: db.PeriDesign.BlockType,
                                       folder_type: expi.DesignExpItem.ItemType,
                                       item_type: expi.DesignExpItem.ItemType):
        """
        Build folder block that is directional, a tx or tx
        """
        if not self.design.is_block_supported(block_type):
            return

        if item_type == expi.DesignExpItem.ItemType.mipi_dphy_tx:
            ops_type = db_item.PeriDesignDirectItem.OpsType.op_tx
        elif item_type == expi.DesignExpItem.ItemType.mipi_dphy_rx:
            ops_type = db_item.PeriDesignDirectItem.OpsType.op_rx
        else:
            return

        block_reg = self.design.get_block_reg(block_type)
        if block_reg is not None:
            if ops_type == db_item.PeriDesignDirectItem.OpsType.op_tx:
                block_inst_list = block_reg.get_all_tx_inst()
            else:
                block_inst_list = block_reg.get_all_rx_inst()

            count = len(block_inst_list)

            block_folder = self.foldertype2folder_map.get(folder_type, None)
            if block_folder is None:
                block_folder = self.build_block_folder_item(count, folder_type, block_reg)

            for block_inst in block_inst_list:
                if block_inst is not None:
                    item = self.build_block_item(block_folder, item_type, block_inst)
                    item.setText(0, f"{block_inst.name} : {block_inst.get_device()}")

            block_folder.sortChildren(0, QtCore.Qt.AscendingOrder)
            self.foldertype2folder_map[folder_type] = block_folder

            self.logger.info(f"Load {count} {block_type}")

    def build_block_folder(self, block_type: db.PeriDesign.BlockType,
                           folder_type: expi.DesignExpItem.ItemType,
                           item_type: expi.DesignExpItem.ItemType):
        """
        Build default block folder that is based of PeriDesignRegistry, which is majority of the block.
        """

        block_reg = self.design.get_block_reg(block_type)
        if block_reg is not None:
            count = block_reg.get_inst_count()
            block_folder = self.build_block_folder_item(count, folder_type, block_reg)

            all_block = block_reg.get_all_inst()
            for block in all_block:
                item = self.build_block_item(block_folder, item_type, block)
                item.setText(0, "{} : {}".format(block.name, block.get_device()))

            block_folder.sortChildren(0, QtCore.Qt.AscendingOrder)
            self.foldertype2folder_map[folder_type] = block_folder

            self.logger.info(f"Load {count} {block_type}")

    def build_block_folder_item(self, item_count, item_type, design_reg):
        """
        Build block grouping folder

        :param item_count: Item count under the folder
        :param item_type: Block folder type
        :param design_reg: Block design db registry
        :return: New grouping folder
        """
        folder = expi.DesignExpItem(self.design_folder, item_type)
        var = folder.generate_data_var(design_reg)
        folder.setData(0, QtCore.Qt.UserRole, var)
        self.update_block_folder(folder, item_count)

        return folder

    def reset_block_folder(self, folder, item_count, design_reg):

        # Delete the folder contents
        folder.remove_children()

        var = folder.generate_data_var(design_reg)
        folder.setData(0, QtCore.Qt.UserRole, var)
        self.update_block_folder(folder, item_count)

    def build_block_item(self, parent_folder, item_type, design_item):
        """
        Build block item

        :param parent_folder: Grouping folder
        :param item_type: Item type
        :param design_item: Design db item
        :return: New explorer item
        """
        item = expi.DesignExpItem(parent_folder, item_type)
        item.setup(design_item)

        return item

    def add_block_item(self, folder_type, item_type, design_item, inst_count):
        """
        Add new block item for creating new block. Block item must have name and get_device().
        """
        folder = self.foldertype2folder_map.get(folder_type, None)
        item = expi.DesignExpItem(folder, item_type)
        item.setup(design_item)
        item.setText(0, "{} : {}".format(design_item.name, design_item.get_device()))
        self.register_dirty_item(item, True)
        if folder is not None:
            folder.sortChildren(0, QtCore.Qt.AscendingOrder)
            self.update_block_folder(folder, inst_count)

        return item

    def update_bus_folder(self, folder):
        folder_name = ""
        item_type = folder.get_type()

        if item_type == expi.DesignExpItem.ItemType.gpio_bus:
            bus_db_item = folder.get_db_item()
            if bus_db_item is not None:
                folder_name = bus_db_item.get_name()
                folder.update_icon()

                ileft = bus_db_item.get_left()
                iright = bus_db_item.get_right()
                folder.setText(0, "{} [{}:{}]".format(folder_name, ileft, iright))

    def update_block_folder(self, folder, item_count):
        """
        Update block folder info

        :param folder: Block grouping folder
        :param item_count: Item count in the folder
        """
        folder_name = ""
        item_type = folder.get_type()

        if item_type == expi.DesignExpItem.ItemType.gpio_bus:
            bus_db_item = folder.get_db_item()
            if bus_db_item is not None:
                folder_name = bus_db_item.get_name()
                folder.update_icon()
        else:
            folder_name = self.foldertype2name_map.get(item_type, "")

            # PT-1763: There's too much coupling between type so rather than creating a new
            # LVDS/SLVS type, we just update the description
            if (item_type == expi.DesignExpItem.ItemType.lvds_rx_info or \
                item_type == expi.DesignExpItem.ItemType.lvds_tx_info or \
                item_type == expi.DesignExpItem.ItemType.lvds_bidir_info) and \
                self.design.is_block_supported(db.PeriDesign.BlockType.adv_lvds):

                # Replace LVDS with LVDS/SLVS
                folder_name = folder_name.replace("LVDS", "LVDS / SLVS")

        folder.setText(0, "{} ({})".format(folder_name, item_count))

    def monitor_changes(self):
        """
        Monitor user interaction in design explorer and external changes
        """
        self.on_start_db_monitor()
        self.parent.sig_import_gpio_started.connect(self.on_stop_db_monitor)
        self.parent.sig_import_gpio_executed.connect(self.on_start_db_monitor)

        # Monitor if an item is selected
        self.tw_design_exp.itemActivated.connect(self.on_item_selected)
        # self.tw_design_exp.itemDoubleClicked.connect(self.on_item_selected)
        self.tw_design_exp.itemPressed.connect(self.on_item_selected) # PT-1971: Right-click item should trigger

        self.parent.sig_design_saved.connect(self.on_design_saved)
        self.parent.sig_sel_item_name_updated.connect(self.on_item_text_update)
        self.parent.sig_sel_item_resource_updated.connect(
            self.on_item_resource_update)
        self.parent.sig_sel_item_prop_updated.connect(self.on_icon_update)

        self.parent.sig_import_gpio_executed.connect(self.on_refresh_import_folder)

        self.tw_design_exp.setContextMenuPolicy(QtCore.Qt.CustomContextMenu)
        self.tw_design_exp.customContextMenuRequested.connect(
            self.show_side_menu)

        self._menu.sig_add_block_request.connect(self.on_block_added)
        self._menu.sig_delete_block_request.connect(self.on_block_deleted)
        self._menu.sig_expand_curr_block_request.connect(self.on_expand_selection)
        self._menu.sig_collapse_curr_block_request.connect(self.on_collapse_selection)
        self._menu.sig_add_bus_request.connect(self.on_gpio_bus_added)

        self.actionCreate_Block.triggered.connect(self.on_block_added)
        self.actionCreate_GPIO_Bus.triggered.connect(self.on_gpio_bus_added)
        self.actionDelete_Block.triggered.connect(self.on_block_deleted)

        self.actionExpand_All.triggered.connect(self.on_expand_all)
        self.actionCollapse_All.triggered.connect(self.on_collapse_all)

        self.le_search_text.editingFinished.connect(self.on_search_text_changed)
        self.pb_reset_filter.pressed.connect(self.on_reset_filter)
        self.pb_filter.pressed.connect(self.show_filter)

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def show_filter(self):
        self.explorer_filter.show()

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_start_db_monitor(self):
        if self.gpio_res_obs is None and self.gpio_reg is not None:
            self.logger.info("DsgExp - Start monitoring design db")
            gpio_res_subject = self.gpio_reg.get_resource_subject()
            self.gpio_res_obs = GPIOResourceObserver(gpio_res_subject, self)

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_stop_db_monitor(self):
        if self.gpio_res_obs is not None:
            self.logger.info("DsgExp - Stop monitoring design db")
            self.gpio_res_obs.disconnect()
            self.gpio_res_obs = None

        if self.gpio_bus_obs is not None:
            self.logger.info("DsgExp - Stop monitoring bus design db")
            self.gpio_bus_obs.disconnect()
            self.gpio_bus_obs = None

    def show_side_menu(self, pos: QtCore.QPoint):
        """
        Show context menu to operation on explorer item

        :param pos: Cursor position in design explorer
        """

        util.gen_util.mark_unused(pos)

        assert self.design is not None
        item = self.tw_design_exp.itemAt(pos)

        if item is None or not isinstance(item, expi.DesignExpItem):
            return

        self.logger.debug(f"show_side_menu: item text->{item.text(0)}")

        # Ensure only 1 item can be selected (used later in add/remove block)
        self.tw_design_exp.clearSelection()
        item.setSelected(True)
        self._sel_item = item

        is_show_menu = False
        item_type = item.get_type()

        if item_type == expi.DesignExpItem.ItemType.unknown:
            return

        self._menu.set_default_display()

        # When grouping folder is selected, user can only delete
        if item_type == expi.DesignExpItem.ItemType.gpio_info \
                or item_type == expi.DesignExpItem.ItemType.pll_info \
                or item_type == expi.DesignExpItem.ItemType.jtag_info \
                or item_type == expi.DesignExpItem.ItemType.spi_flash_info \
                or item_type == expi.DesignExpItem.ItemType.hyper_ram_info \
                or item_type == expi.DesignExpItem.ItemType.h264_info \
                or item_type == expi.DesignExpItem.ItemType.ddr_info \
                or item_type == expi.DesignExpItem.ItemType.osc_info \
                or item_type == expi.DesignExpItem.ItemType.lvds_tx_info \
                or item_type == expi.DesignExpItem.ItemType.lvds_rx_info \
                or item_type == expi.DesignExpItem.ItemType.lvds_bidir_info \
                or item_type == expi.DesignExpItem.ItemType.mipi_dphy_tx_info \
                or item_type == expi.DesignExpItem.ItemType.mipi_dphy_rx_info \
                or item_type == expi.DesignExpItem.ItemType.mipi_tx_info \
                or item_type == expi.DesignExpItem.ItemType.mipi_rx_info \
                or item_type == expi.DesignExpItem.ItemType.mipi_hard_dphy_tx_info \
                or item_type == expi.DesignExpItem.ItemType.mipi_hard_dphy_rx_info \
                or item_type == expi.DesignExpItem.ItemType.pll_ssc_info \
                or item_type == expi.DesignExpItem.ItemType.quad_pcie_info \
                or item_type == expi.DesignExpItem.ItemType.lane_10g_info \
                or item_type == expi.DesignExpItem.ItemType.lane_1g_info \
                or item_type == expi.DesignExpItem.ItemType.raw_serdes_info \
                or item_type == expi.DesignExpItem.ItemType.soc_info:

            is_show_menu = True

            if item_type == expi.DesignExpItem.ItemType.gpio_info:
                self._menu.set_display(self._menu.OpsType.add_bus_ops, True)
            else:
                self._menu.set_display(self._menu.OpsType.add_bus_ops, False)

            self._menu.set_display(self._menu.OpsType.del_ops, False)

        # When item is selected, user can delete
        elif item_type == expi.DesignExpItem.ItemType.gpio:
            is_show_menu = True
            self._menu.set_display(self._menu.OpsType.add_ops, False)
            self._menu.set_display(self._menu.OpsType.del_ops, True)

            self._menu.set_display(self._menu.OpsType.exp_curr_ops, False)
            self._menu.set_display(self._menu.OpsType.colp_curr_ops, False)

            parent = item.parent()
            assert isinstance(parent, expi.DesignExpItem)
            parent_type = parent.get_type()
            if parent_type == expi.DesignExpItem.ItemType.gpio_bus:
                self._menu.set_display(self._menu.OpsType.del_ops, False)

        elif item_type == expi.DesignExpItem.ItemType.gpio_bus:
            is_show_menu = True
            self._menu.set_display(self._menu.OpsType.add_ops, False)
            self._menu.set_display(self._menu.OpsType.add_bus_ops, False)

        # When item is selected, user can delete
        elif item_type == expi.DesignExpItem.ItemType.pll \
                or item_type == expi.DesignExpItem.ItemType.osc \
                or item_type == expi.DesignExpItem.ItemType.jtag \
                or item_type == expi.DesignExpItem.ItemType.spi_flash \
                or item_type == expi.DesignExpItem.ItemType.hyper_ram \
                or item_type == expi.DesignExpItem.ItemType.h264 \
                or item_type == expi.DesignExpItem.ItemType.ddr \
                or item_type == expi.DesignExpItem.ItemType.mipi \
                or item_type == expi.DesignExpItem.ItemType.lvds \
                or item_type == expi.DesignExpItem.ItemType.lvds_tx \
                or item_type == expi.DesignExpItem.ItemType.lvds_rx \
                or item_type == expi.DesignExpItem.ItemType.lvds_bidir \
                or item_type == expi.DesignExpItem.ItemType.mipi_dphy_tx \
                or item_type == expi.DesignExpItem.ItemType.mipi_dphy_rx \
                or item_type == expi.DesignExpItem.ItemType.mipi_tx \
                or item_type == expi.DesignExpItem.ItemType.mipi_rx \
                or item_type == expi.DesignExpItem.ItemType.mipi_hard_dphy_tx \
                or item_type == expi.DesignExpItem.ItemType.mipi_hard_dphy_rx \
                or item_type == expi.DesignExpItem.ItemType.pll_ssc \
                or item_type == expi.DesignExpItem.ItemType.quad_pcie \
                    or item_type == expi.DesignExpItem.ItemType.lane_10g \
                    or item_type == expi.DesignExpItem.ItemType.lane_1g \
                    or item_type == expi.DesignExpItem.ItemType.raw_serdes \
                    or item_type == expi.DesignExpItem.ItemType.soc:
            is_show_menu = True
            self._menu.set_display(self._menu.OpsType.add_ops, False)
            self._menu.set_display(self._menu.OpsType.add_bus_ops, False)
            self._menu.set_display(self._menu.OpsType.del_ops, True)

            self._menu.set_display(self._menu.OpsType.exp_curr_ops, False)
            self._menu.set_display(self._menu.OpsType.colp_curr_ops, False)

        elif item_type == expi.DesignExpItem.ItemType.design_info \
                or item_type == expi.DesignExpItem.ItemType.device_info \
                or item_type == expi.DesignExpItem.ItemType.clockmux_info:
            is_show_menu = True

            self._menu.set_display(self._menu.OpsType.add_ops, False)
            self._menu.set_display(self._menu.OpsType.add_bus_ops, False)
            self._menu.set_display(self._menu.OpsType.del_ops, False)
            self._menu.set_display(self._menu.OpsType.exp_curr_ops, True)
            self._menu.set_display(self._menu.OpsType.colp_curr_ops, True)

        if is_show_menu:
            self._menu.show_menu()

    def select_design(self):
        # This does not trigger selection signal since there can be a case
        # when the UI is not up yet.
        self.tw_design_exp.setCurrentItem(self.design_folder)

        # Check if it is already selecting the design folder. If not, then
        # we trigger the design folder selection.
        if self._sel_item is not None and self._sel_item.get_type() != expi.DesignExpItem.ItemType.design_info:
            # Trigger the selection only if it needs to (i.e. when UI is up)
            self.on_item_selected(self.design_folder, 0)

    # noinspection PyArgumentList
    @QtCore.pyqtSlot(QtWidgets.QTreeWidgetItem, int)
    def on_item_selected(self, item: Optional[QtWidgets.QTreeWidgetItem], column: int):
        """
        Handle tree item selection and emit appropriate signal.
        When item is selected it will enabled/disabled toolbar action based on what can be done
        on the selected item. Context menu accesibility is handled in show_side_menu()

        :param item: Selected tree item
        :param column: Tree view column
        """

        util.gen_util.mark_unused(column)

        if item is None:
            return

        # Check previous item
        self.check_dirty_state(self._sel_item)

        # Clear all unexpected dock widget before open other block editor
        self.parent.clear_all_dock_widget()

        assert isinstance(item, expi.DesignExpItem)
        self._sel_item = item
        data = self._sel_item.get_data_var()
        assert isinstance(data, dict)
        sel_db_item = data.get("item", None)
        assert sel_db_item is not None
        item_type: expi.DesignExpItem.ItemType = expi.DesignExpItem.ItemType(data.get("type", expi.DesignExpItem.ItemType.unknown))

        # Default toolbar action, Expand & Collapse should always be enabled
        action_list = [exp_menu.DesignExpMenu.OpsType.exp_curr_ops,
                       exp_menu.DesignExpMenu.OpsType.colp_curr_ops]

        if item_type == expi.DesignExpItem.ItemType.gpio:
            self.sig_gpio_selected.emit(sel_db_item.name)

            if item.parent().item_type == expi.DesignExpItem.ItemType.gpio_bus:  # select a gpio bus member
                # Bus member cannot be deleted, you can only remove
                action_list.extend([exp_menu.DesignExpMenu.OpsType.add_bus_ops])
            else:  # select a gpio
                action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops,
                                    exp_menu.DesignExpMenu.OpsType.add_bus_ops,
                                    exp_menu.DesignExpMenu.OpsType.del_ops])

            self.highlight_blk_res(item, item_type)

        elif item_type == expi.DesignExpItem.ItemType.gpio_bus:
            self.sig_gpio_bus_selected.emit(sel_db_item.get_name())
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops,
                                exp_menu.DesignExpMenu.OpsType.add_bus_ops,
                                exp_menu.DesignExpMenu.OpsType.del_ops])

            if self.gpio_bus_obs is not None:
                self.logger.debug("DsgExp - Stop monitoring bus resize")
                self.gpio_bus_obs.disconnect()
                self.gpio_bus_obs = None

            if self.gpio_bus_obs is None and self._sel_item is not None:
                gpio_bus = self._sel_item.get_db_item()
                if gpio_bus is not None:
                    self.logger.debug("DsgExp - Start monitoring bus selected type {}".format(gpio_bus.get_name()))
                    gpio_bus_subject = gpio_bus.get_resize_subject()
                    self.gpio_bus_obs = GPIOBusResizeObserver(gpio_bus_subject, self)

            self.highlight_blk_res(item, item_type)

        elif item_type == expi.DesignExpItem.ItemType.pll:
            self.sig_pll_selected.emit(sel_db_item.name)
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops,
                                exp_menu.DesignExpMenu.OpsType.del_ops])
            self.highlight_blk_res(item, item_type)

        elif item_type == expi.DesignExpItem.ItemType.osc:
            self.sig_osc_selected.emit(sel_db_item.name)
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops,
                                exp_menu.DesignExpMenu.OpsType.del_ops])

        elif item_type == expi.DesignExpItem.ItemType.spi_flash:
            self.sig_spi_flash_selected.emit(sel_db_item.name)
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops,
                                exp_menu.DesignExpMenu.OpsType.del_ops])
            self.highlight_blk_res(item, item_type)

        elif item_type == expi.DesignExpItem.ItemType.hyper_ram:
            self.sig_hyper_ram_selected.emit(sel_db_item.name)
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops,
                                exp_menu.DesignExpMenu.OpsType.del_ops])

        elif item_type == expi.DesignExpItem.ItemType.lvds_tx:
            self.sig_lvds_tx_selected.emit(sel_db_item.name)
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops,
                                exp_menu.DesignExpMenu.OpsType.del_ops])
            self.highlight_blk_res(item, item_type)

        elif item_type == expi.DesignExpItem.ItemType.lvds_rx:
            self.sig_lvds_rx_selected.emit(sel_db_item.name)
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops,
                                exp_menu.DesignExpMenu.OpsType.del_ops])
            self.highlight_blk_res(item, item_type)

        elif item_type == expi.DesignExpItem.ItemType.lvds_bidir:
            self.sig_lvds_bidir_selected.emit(sel_db_item.name)
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops,
                                exp_menu.DesignExpMenu.OpsType.del_ops])
            self.highlight_blk_res(item, item_type)

        elif item_type == expi.DesignExpItem.ItemType.mipi_tx:
            self.sig_mipi_tx_selected.emit(sel_db_item.name)
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops,
                                exp_menu.DesignExpMenu.OpsType.del_ops])
            self.highlight_blk_res(item, item_type)

        elif item_type == expi.DesignExpItem.ItemType.mipi_rx:
            self.sig_mipi_rx_selected.emit(sel_db_item.name)
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops,
                                exp_menu.DesignExpMenu.OpsType.del_ops])
            self.highlight_blk_res(item, item_type)

        elif item_type == expi.DesignExpItem.ItemType.mipi_dphy_tx:
            self.sig_mipi_dphy_tx_selected.emit(sel_db_item.name)
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops,
                                exp_menu.DesignExpMenu.OpsType.del_ops])
            self.highlight_blk_res(item, item_type)

        elif item_type == expi.DesignExpItem.ItemType.mipi_dphy_rx:
            self.sig_mipi_dphy_rx_selected.emit(sel_db_item.name)
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops,
                                exp_menu.DesignExpMenu.OpsType.del_ops])
            self.highlight_blk_res(item, item_type)

        elif item_type == expi.DesignExpItem.ItemType.mipi_hard_dphy_tx:
            self.sig_mipi_hard_dphy_tx_selected.emit(sel_db_item.name)
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops,
                                exp_menu.DesignExpMenu.OpsType.del_ops])
            self.highlight_blk_res(item, item_type)

        elif item_type == expi.DesignExpItem.ItemType.mipi_hard_dphy_rx:
            self.sig_mipi_hard_dphy_rx_selected.emit(sel_db_item.name)
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops,
                                exp_menu.DesignExpMenu.OpsType.del_ops])
            self.highlight_blk_res(item, item_type)

        elif item_type == expi.DesignExpItem.ItemType.jtag:
            self.sig_jtag_selected.emit(sel_db_item.name)
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops,
                                exp_menu.DesignExpMenu.OpsType.del_ops])
            self.highlight_blk_res(item, item_type)

        elif item_type == expi.DesignExpItem.ItemType.h264:
            self.sig_h264_selected.emit(sel_db_item.name)
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops,
                                exp_menu.DesignExpMenu.OpsType.del_ops])

        elif item_type == expi.DesignExpItem.ItemType.ddr:
            self.sig_ddr_selected.emit(sel_db_item.name)
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops,
                                exp_menu.DesignExpMenu.OpsType.del_ops])
            self.highlight_blk_res(item, item_type)

        elif item_type == expi.DesignExpItem.ItemType.pll_ssc:
            self.sig_pll_ssc_selected.emit(sel_db_item.name)
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops,
                                exp_menu.DesignExpMenu.OpsType.del_ops])
            self.highlight_blk_res(item, item_type)

        elif item_type == expi.DesignExpItem.ItemType.quad_pcie:
            self.sig_quad_pcie_selected.emit(sel_db_item.name)
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops,
                                exp_menu.DesignExpMenu.OpsType.del_ops])
            self.highlight_blk_res(item, item_type)

        elif item_type == expi.DesignExpItem.ItemType.lane_10g:
            self.sig_lane_10g_selected.emit(sel_db_item.name)
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops,
                                exp_menu.DesignExpMenu.OpsType.del_ops])
            self.highlight_blk_res(item, item_type)

        elif item_type == expi.DesignExpItem.ItemType.lane_1g:
            self.sig_lane_1g_selected.emit(sel_db_item.name)
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops,
                                exp_menu.DesignExpMenu.OpsType.del_ops])
            self.highlight_blk_res(item, item_type)

        elif item_type == expi.DesignExpItem.ItemType.raw_serdes:
            self.sig_raw_serdes_selected.emit(sel_db_item.name)
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops,
                                exp_menu.DesignExpMenu.OpsType.del_ops])
            self.highlight_blk_res(item, item_type)

        elif item_type == expi.DesignExpItem.ItemType.soc:
            self.sig_soc_selected.emit(sel_db_item.name)
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops,
                                exp_menu.DesignExpMenu.OpsType.del_ops])
            self.highlight_blk_res(item, item_type)

        elif item_type == expi.DesignExpItem.ItemType.design_info:
            self.sig_design_selected.emit(sel_db_item.name)

        elif item_type == expi.DesignExpItem.ItemType.gpio_info:
            self.sig_gpio_info_selected.emit()
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops,
                                exp_menu.DesignExpMenu.OpsType.add_bus_ops])

        elif item_type == expi.DesignExpItem.ItemType.pll_info:
            self.sig_pll_info_selected.emit()
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops])

        elif item_type == expi.DesignExpItem.ItemType.osc_info:
            self.sig_osc_info_selected.emit()
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops])

        elif item_type == expi.DesignExpItem.ItemType.jtag_info:
            self.sig_jtag_info_selected.emit()
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops])

        elif item_type == expi.DesignExpItem.ItemType.spi_flash_info:
            self.sig_spi_flash_info_selected.emit()
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops])

        elif item_type == expi.DesignExpItem.ItemType.hyper_ram_info:
            self.sig_hyper_ram_info_selected.emit()
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops])

        elif item_type == expi.DesignExpItem.ItemType.h264_info:
            self.sig_h264_info_selected.emit()
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops])

        elif item_type == expi.DesignExpItem.ItemType.ddr_info:
            self.sig_ddr_info_selected.emit()
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops])

        elif item_type == expi.DesignExpItem.ItemType.lvds_tx_info:
            self.sig_lvds_tx_info_selected.emit()
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops])

        elif item_type == expi.DesignExpItem.ItemType.lvds_rx_info:
            self.sig_lvds_rx_info_selected.emit()
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops])

        elif item_type == expi.DesignExpItem.ItemType.lvds_bidir_info:
            self.sig_lvds_bidir_info_selected.emit()
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops])

        elif item_type == expi.DesignExpItem.ItemType.mipi_tx_info:
            self.sig_mipi_tx_info_selected.emit()
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops])

        elif item_type == expi.DesignExpItem.ItemType.mipi_rx_info:
            self.sig_mipi_rx_info_selected.emit()
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops])

        elif item_type == expi.DesignExpItem.ItemType.mipi_dphy_tx_info:
            self.sig_mipi_dphy_tx_info_selected.emit()
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops])

        elif item_type == expi.DesignExpItem.ItemType.mipi_dphy_rx_info:
            self.sig_mipi_dphy_rx_info_selected.emit()
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops])

        elif item_type == expi.DesignExpItem.ItemType.mipi_hard_dphy_tx_info:
            self.sig_mipi_hard_dphy_tx_info_selected.emit()
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops])

        elif item_type == expi.DesignExpItem.ItemType.mipi_hard_dphy_rx_info:
            self.sig_mipi_hard_dphy_rx_info_selected.emit()
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops])

        elif item_type == expi.DesignExpItem.ItemType.pll_ssc_info:
            self.sig_pll_ssc_info_selected.emit()
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops])

        elif item_type == expi.DesignExpItem.ItemType.quad_pcie_info:
            self.sig_quad_pcie_info_selected.emit()
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops])

        elif item_type == expi.DesignExpItem.ItemType.lane_10g_info:
            self.sig_lane_10g_info_selected.emit()
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops])

        elif item_type == expi.DesignExpItem.ItemType.lane_1g_info:
            self.sig_lane_1g_info_selected.emit()
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops])

        elif item_type == expi.DesignExpItem.ItemType.raw_serdes_info:
            self.sig_raw_serdes_info_selected.emit()
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops])

        elif item_type == expi.DesignExpItem.ItemType.soc_info:
            self.sig_soc_info_selected.emit()
            action_list.extend([exp_menu.DesignExpMenu.OpsType.add_ops])

        elif item_type == expi.DesignExpItem.ItemType.iobank_info:
            self.sig_iobank_info_selected.emit()

        elif item_type == expi.DesignExpItem.ItemType.control_info:
            self.sig_control_info_selected.emit()

        elif item_type == expi.DesignExpItem.ItemType.clockmux_info:
            self.sig_clockmux_info_selected.emit()

        elif item_type == expi.DesignExpItem.ItemType.clockmux:
            # No menu item
            self.sig_clockmux_selected.emit(sel_db_item.name)

        self.update_action_status(action_list)

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_icon_update(self):
        """
        Handle icon updates when selected item design property is updated.
        """

        if self._sel_item is None:
            return

        # Check previous item
        # self.check_dirty_state(self._sel_item)

        self._sel_item.update_icon()
        item_type = self._sel_item.get_type()

        if item_type == expi.DesignExpItem.ItemType.gpio_bus:
            # refresh bus member icons
            bus_folder = self._sel_item
            count = bus_folder.childCount()
            for i in range(0, count):
                child_item = bus_folder.child(i)
                child_item.update_icon()

            # re-select bus item so its property info will be updated
            self.tw_design_exp.setCurrentItem(bus_folder)
            self.on_item_selected(bus_folder, 0)

    def highlight_blk_res(self, item: expi.DesignExpItem, item_type: expi.DesignExpItem.ItemType):
        if item_type == expi.DesignExpItem.ItemType.gpio_bus:
            self.sig_clear_selection.emit()
            for num in range(item.childCount()):
                bus_item = item.child(num)

                dev_name = ""
                dev_name = bus_item.get_current_dev_name()

                if dev_name != "":
                    self.sig_res_highlighted.emit(str(item_type), dev_name)
            return

        dev_name = ""
        dev_name = item.get_current_dev_name()

        if dev_name != "":
            self.sig_res_highlighted.emit(str(item_type), dev_name)

    def update_pkg_pin_highlight(self, item_type: expi.DesignExpItem.ItemType):
        assert self._sel_item is not None

        if item_type == expi.DesignExpItem.ItemType.iobank_info:
            iobank_reg = self.design.device_setting.iobank_reg
            if iobank_reg is not None:
                self.sig_iobank_volt_change.emit(iobank_reg.changed_iobank)
                iobank_reg.changed_iobank = None

        elif item_type == expi.DesignExpItem.ItemType.gpio:
            res_name: str = self._sel_item.get_current_dev_name()
            if res_name != '':
                self.sig_iobank_std_change.emit(res_name)

        elif item_type in (expi.DesignExpItem.ItemType.gpio,
                           expi.DesignExpItem.ItemType.pll,
                           expi.DesignExpItem.ItemType.mipi_hard_dphy_tx,
                           expi.DesignExpItem.ItemType.pll_ssc,
                           expi.DesignExpItem.ItemType.ddr,
                           expi.DesignExpItem.ItemType.quad_pcie,
                           expi.DesignExpItem.ItemType.lane_10g,
                           expi.DesignExpItem.ItemType.lane_1g,
                           expi.DesignExpItem.ItemType.raw_serdes):
            # Update resource for changing reference clock mode / data width (ddr)
            res_name = self._sel_item.get_current_dev_name()
            if res_name != '':
                self.sig_block_res_changed.emit(str(item_type), "", res_name)

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_item_text_update(self):
        """
        Handle design item display text change
        """

        if self._sel_item is None:
            return

        # Check previous item
        self.check_dirty_state(self._sel_item)

        item_type = self._sel_item.get_type()
        ori_dev = self._sel_item.get_current_dev_name()
        if item_type == expi.DesignExpItem.ItemType.gpio \
                or item_type == expi.DesignExpItem.ItemType.pll \
                or item_type == expi.DesignExpItem.ItemType.lvds \
                or item_type == expi.DesignExpItem.ItemType.mipi \
                or item_type == expi.DesignExpItem.ItemType.jtag \
                or item_type == expi.DesignExpItem.ItemType.spi_flash \
                or item_type == expi.DesignExpItem.ItemType.hyper_ram \
                or item_type == expi.DesignExpItem.ItemType.h264 \
                or item_type == expi.DesignExpItem.ItemType.ddr \
                or item_type == expi.DesignExpItem.ItemType.osc \
                or item_type == expi.DesignExpItem.ItemType.lvds_tx \
                or item_type == expi.DesignExpItem.ItemType.lvds_rx \
                or item_type == expi.DesignExpItem.ItemType.lvds_bidir \
                or item_type == expi.DesignExpItem.ItemType.mipi_dphy_tx \
                or item_type == expi.DesignExpItem.ItemType.mipi_dphy_rx \
                or item_type == expi.DesignExpItem.ItemType.mipi_tx \
                or item_type == expi.DesignExpItem.ItemType.mipi_rx \
                or item_type == expi.DesignExpItem.ItemType.mipi_hard_dphy_tx \
                or item_type == expi.DesignExpItem.ItemType.mipi_hard_dphy_rx \
                or item_type == expi.DesignExpItem.ItemType.pll_ssc \
                or item_type == expi.DesignExpItem.ItemType.quad_pcie \
                or item_type == expi.DesignExpItem.ItemType.lane_10g \
                or item_type == expi.DesignExpItem.ItemType.lane_1g \
                or item_type == expi.DesignExpItem.ItemType.raw_serdes \
                or item_type == expi.DesignExpItem.ItemType.soc:
            self._sel_item.update_text_display()
            new_device = self._sel_item.get_current_dev_name()
            self.sig_block_res_changed.emit(str(item_type), ori_dev, new_device)
            return

        if item_type == expi.DesignExpItem.ItemType.gpio_bus:
            self.update_bus_folder(self._sel_item)

            # refresh bus member icons
            bus_folder = self._sel_item
            count = bus_folder.childCount()
            for i in range(0, count):
                child_item = bus_folder.child(i)
                child_item.update_text_display()
                new_device = child_item.get_current_dev_name()
                self.sig_block_res_changed.emit(str(item_type), ori_dev, new_device)

            # TODO : Delete???
            # NOt turning the code below (copied from on_icon_update)
            # since it could lead to too many bus signal being thrown
            # unnecessarily
            # re-select bus item so its property info will be updated
            # self.tw_design_exp.setCurrentItem(bus_folder)
            # self.on_item_selected(bus_folder, 0)

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_item_resource_update(self):

        if self._sel_item is not None:
            design_db_item = self._sel_item.get_db_item()
            item_type = self._sel_item.get_type()
            ori_device = self._sel_item.get_current_dev_name()

            if design_db_item is not None:

                folder = None
                device_name = ""
                match item_type:
                    case expi.DesignExpItem.ItemType.gpio:
                        device_name = design_db_item.gpio_def
                        folder = self.gpio_folder
                        self._sel_item.update_icon()

                    case expi.DesignExpItem.ItemType.pll:
                        device_name = design_db_item.pll_def
                        folder = self.pll_folder
                        self._sel_item.update_icon()

                    case expi.DesignExpItem.ItemType.lvds_tx:
                        device_name = design_db_item.lvds_def
                        folder = self.lvds_tx_folder
                        self._sel_item.update_icon()

                    case expi.DesignExpItem.ItemType.lvds_rx:
                        device_name = design_db_item.lvds_def
                        folder = self.lvds_rx_folder
                        self._sel_item.update_icon()

                    case expi.DesignExpItem.ItemType.lvds_bidir:
                        device_name = design_db_item.lvds_def
                        folder = self.lvds_bidir_folder
                        self._sel_item.update_icon()

                    case expi.DesignExpItem.ItemType.mipi_tx:
                        device_name = design_db_item.mipi_def
                        folder = self.mipi_tx_folder
                        self._sel_item.update_icon()

                    case expi.DesignExpItem.ItemType.mipi_rx:
                        device_name = design_db_item.mipi_def
                        folder = self.mipi_rx_folder
                        self._sel_item.update_icon()

                    case expi.DesignExpItem.ItemType.mipi_dphy_tx:
                        device_name = design_db_item.block_def
                        folder = self.mipi_dphy_tx_folder
                        self._sel_item.update_icon()

                    case expi.DesignExpItem.ItemType.mipi_dphy_rx:
                        device_name = design_db_item.block_def
                        folder = self.mipi_dphy_rx_folder
                        self._sel_item.update_icon()

                    case expi.DesignExpItem.ItemType.mipi_hard_dphy_tx:
                        device_name = design_db_item.block_def
                        folder = self.mipi_hard_dphy_tx_folder
                        self._sel_item.update_icon()

                    case expi.DesignExpItem.ItemType.mipi_hard_dphy_rx:
                        device_name = design_db_item.block_def
                        folder = self.mipi_hard_dphy_rx_folder
                        self._sel_item.update_icon()

                    case expi.DesignExpItem.ItemType.jtag:
                        device_name = design_db_item.jtag_def
                        folder = self.jtag_folder
                        self._sel_item.update_icon()

                    case expi.DesignExpItem.ItemType.h264:
                        device_name = design_db_item.h264_def
                        folder = self.h264_folder
                        self._sel_item.update_icon()

                    case expi.DesignExpItem.ItemType.ddr:
                        device_name = design_db_item.ddr_def
                        folder = self.ddr_folder
                        self._sel_item.update_icon()

                    case expi.DesignExpItem.ItemType.osc:
                        device_name = design_db_item.osc_def
                        folder = self.osc_folder

                    case expi.DesignExpItem.ItemType.spi_flash:
                        # TODO : Does the block itself has a resource?
                        device_name = ""
                        folder = self.spi_flash_folder

                    case expi.DesignExpItem.ItemType.hyper_ram:
                        # TODO : Does the block itself has a resource?
                        device_name = ""
                        folder = self.hyper_ram_folder

                    case expi.DesignExpItem.ItemType.pll_ssc:
                        device_name = design_db_item.block_def
                        folder = self.pll_ssc_folder
                        self._sel_item.update_icon()

                    case expi.DesignExpItem.ItemType.quad_pcie:
                        device_name = design_db_item.get_device()
                        folder = self.quad_pcie_folder
                        self._sel_item.update_icon()

                    case expi.DesignExpItem.ItemType.lane_10g:
                        device_name = design_db_item.get_device()
                        folder = self.lane_10g_folder
                        self._sel_item.update_icon()

                    case expi.DesignExpItem.ItemType.lane_1g:
                        device_name = design_db_item.get_device()
                        folder = self.lane_1g_folder
                        self._sel_item.update_icon()

                    case expi.DesignExpItem.ItemType.raw_serdes:
                        device_name = design_db_item.get_device()
                        folder = self.raw_serdes_folder
                        self._sel_item.update_icon()

                    case expi.DesignExpItem.ItemType.soc:
                        device_name = design_db_item.block_def
                        folder = self.soc_folder
                        self._sel_item.update_icon()

                # Reset original item
                # findItems does not seem to work, search manually
                if folder is not None:
                    count = folder.childCount()
                    for i in range(0, count):
                        dep_item = folder.child(i)
                        dep_item_text = dep_item.text(0)
                        if device_name in dep_item_text:
                            dep_item.update_text_display()
                            self.check_dirty_state(dep_item)
                            dep_item.update_icon()

            # Apply to the new one
            self.on_item_text_update()
            self._sel_item.update_icon()

            # For package planner
            new_device = self._sel_item.get_current_dev_name()
            self.sig_block_res_changed.emit(str(item_type), ori_device, new_device)

    def obs_gpio_bus_resize(self, bus_name, is_members_different):
        """
        Design db observer handler. Update gpio bus folder when the bus
        has been resized.

        """

        # By right, the selected item should match with the bus
        # considering that the bus name might have been renamed
        if self._sel_item is not None:
            item = self._sel_item
            item_type = self._sel_item.get_type()

            # We are only refreshing the selected bus folder
            if item_type == expi.DesignExpItem.ItemType.gpio_bus:
                reg = self.design.gpio_reg
                if reg is not None:
                    gpio_bus = item.get_db_item()
                    if gpio_bus is not None:
                        # Update the bus folder
                        self.update_bus_folder(item)

                        # If the members are shuffled (exclude
                        # exchange of MSB and LSB)
                        if is_members_different:
                            # Update the gpio folder as well since the
                            # count can be changed
                            self.update_block_folder(
                                self.gpio_folder, reg.get_gpio_count())

                            # refresh bus member icons
                            item.remove_children()

                            # load its gpio
                            all_gpio = gpio_bus.get_member_list()
                            for gpio in all_gpio:
                                mem_item = self.build_block_item(
                                    item, expi.DesignExpItem.ItemType.gpio, gpio)
                                mem_item.setText(0, "{} : {}".format(gpio.name, gpio.gpio_def))

                            item.sortChildren(0, QtCore.Qt.AscendingOrder)

                        # re-select bus item so its property info will be updated
                        self.tw_design_exp.update()
                        self.tw_design_exp.setCurrentItem(item)

    def obs_gpio_resource_changed(self, res_name, inst_name):
        """
        Design db observer handler. Update gpio's resource name when it change in design db.

        :param res_name: Resource whose state changes
        :param inst_name: Instance that use this resource, if empty, resource is unused
        """

        folder = self.gpio_folder
        if inst_name != "":
            search_key = inst_name
        else:
            search_key = res_name

        bus_folder_list = []

        # findItems does not seem to work, search manually
        count = folder.childCount()
        for i in range(0, count):
            dep_item = folder.child(i)

            if dep_item.get_type() == expi.DesignExpItem.ItemType.gpio_bus:
                bus_folder_list.append(dep_item)
                continue

            dep_item_text = dep_item.text(0)
            if search_key in dep_item_text:
                ori_res_name:str = dep_item.get_current_dev_name()

                dep_item.update_text_display()
                self.check_dirty_state(dep_item)
                dep_item.update_icon()

                # Update for package planner
                self.sig_block_res_changed.emit(str(dep_item.get_type()), ori_res_name, res_name)

        for bus in bus_folder_list:
            count = bus.childCount()
            for i in range(0, count):
                dep_item = bus.child(i)
                dep_item_text = dep_item.text(0)
                if search_key in dep_item_text:
                    ori_res_name:str = dep_item.get_current_dev_name()

                    dep_item.update_text_display()
                    self.check_dirty_state(dep_item)
                    dep_item.update_icon()

                    # Update for package planner
                    self.sig_block_res_changed.emit(str(dep_item.get_type()), ori_res_name, res_name)

        self.sig_gpio_res_changed.emit()

    # noinspection PyArgumentList
    @QtCore.pyqtSlot('QString', 'QString')
    def on_external_select_request(self, inst_name: str, inst_type_name: str):

        folder = None

        match inst_type_name:
            case "gpio" | "clock" | "lvds gpio" | "bus gpio":
                folder = self.gpio_folder
            case "pll":
                folder = self.pll_folder
            case "lvds_tx":
                folder = self.lvds_tx_folder
            case "lvds_rx":
                folder = self.lvds_rx_folder
            case "lvds_bidir":
                folder = self.lvds_bidir_folder
            case "mipi_tx":
                folder = self.mipi_tx_folder
            case "mipi_rx":
                folder = self.mipi_rx_folder
            case "mipi_tx_lane":
                folder = self.mipi_dphy_tx_folder
            case "mipi_rx_lane":
                folder = self.mipi_dphy_rx_folder
            case "mipi_dphy_tx":
                folder = self.mipi_hard_dphy_tx_folder
            case "mipi_dphy_rx":
                folder = self.mipi_hard_dphy_rx_folder
            case "jtag":
                folder = self.jtag_folder
            case "h.264":
                folder = self.h264_folder
            case "ddr":
                folder = self.ddr_folder
            case "osc" | "hposc":
                folder = self.osc_folder
            case "spi_flash":
                folder = self.spi_flash_folder
            case "hyper_ram":
                folder = self.hyper_ram_folder
            case "iobank":
                # Special case for io bank which has no
                # instance name.
                self.tw_design_exp.setCurrentItem(self.iobank_folder)
                self.on_item_selected(self.iobank_folder, 0)
            case "control" | "configuration" | "seu" | "ext_flash":
                self.tw_design_exp.setCurrentItem(self.control_folder)
                self.on_item_selected(self.control_folder, 0)
            case "clkmux":
                folder = self.clockmux_folder
            case "pll_ssc":
                folder = self.pll_ssc_folder
            case "quad_pcie" | "pcie":
                folder = self.quad_pcie_folder
            case "10gbase_kr":
                folder = self.lane_10g_folder
            case "sgmii":
                folder = self.lane_1g_folder
            case "pma_direct":
                folder = self.raw_serdes_folder
            case "qcrv32":
                folder = self.soc_folder

        is_found = False
        bus_folder_list = []
        if folder is not None:
            count = folder.childCount()
            for i in range(0, count):
                dep_item = folder.child(i)
                if dep_item.get_type() == expi.DesignExpItem.ItemType.gpio_bus:
                    bus_folder_list.append(dep_item)


                # PT-2040: Exact search rather than substr
                # instance name : resource
                if inst_type_name == "clkmux":
                    dep_item_text = dep_item.text(0)
                    if inst_name in dep_item_text:
                        is_found = True
                        self.tw_design_exp.setCurrentItem(dep_item)
                        self.on_item_selected(dep_item, 0)
                        break


                elif dep_item.is_inst_name_match_folder_item(inst_name, False):

                    is_found = True
                    self.tw_design_exp.setCurrentItem(dep_item)
                    self.on_item_selected(dep_item, 0)
                    break

        if is_found is False:
            if inst_type_name == "gpio" or inst_type_name == "bus gpio" or inst_type_name == "clock":

                for folder in bus_folder_list:
                    # compare against bus name
                    dep_item = folder

                    if dep_item.is_inst_name_match_folder_item(inst_name, True):
                        is_found = True
                        self.tw_design_exp.setCurrentItem(dep_item)
                        self.on_item_selected(dep_item, 0)

                    else:
                        count = folder.childCount()
                        for i in range(0, count):
                            dep_item = folder.child(i)

                            if dep_item.is_inst_name_match_folder_item(inst_name, False):

                                is_found = True
                                self.tw_design_exp.setCurrentItem(dep_item)
                                self.on_item_selected(dep_item, 0)
                                break

                    if is_found:
                        break

    def check_dirty_state(self, item):

        if item is None:
            return

        if not item.is_dirty():
            design_item = item.get_db_item()
            if design_item is not None:
                is_dirty = design_item.is_changed()
                self.register_dirty_item(item, is_dirty)

    def register_dirty_item(self, item, is_dirty):
        """
        Register item as dirty. Add to dirty list.

        :param item: Explorer tree item
        :param is_dirty: True, set as dirty, else False
        """

        if item in self._dirty_item:
            return

        item.set_dirty(is_dirty)

        # Change colour to highlight dirty item
        if is_dirty:
            # item.setForeground(0, QtCore.Qt.darkBlue)
            self._dirty_item.append(item)

    def unregister_dirty_item(self, item):
        """
        Unregister dirty item. Remove from dirty item list.

        :param item: Explorer tree item
        """

        if item in self._dirty_item:
            item.set_dirty(False)
            # Restore colour
            # item.setForeground(0, QtCore.Qt.black)
            self._dirty_item.remove(item)

    def clear_all_dirty_item(self):
        """
        Clear dirty item list. Restore each item.
        """
        for item in self._dirty_item:
            item.set_dirty(False)
            item.setForeground(0, QtCore.Qt.black)

        self._dirty_item.clear()

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_design_saved(self):
        """
        When design is saved, restore all dirty item
        """
        self.clear_all_dirty_item()

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_block_added(self):
        """
        Add new block instance
        """
        sel_item_list = self.tw_design_exp.selectedItems()

        count = len(sel_item_list)
        if count > 0:
            item = sel_item_list[0]
            assert isinstance(item, expi.DesignExpItem)
            item_type = item.get_type()
            new_item = None

            if item_type == expi.DesignExpItem.ItemType.gpio_info \
                    or item_type == expi.DesignExpItem.ItemType.gpio \
                    or item_type == expi.DesignExpItem.ItemType.gpio_bus:
                new_item = self.add_new_gpio()
            elif item_type == expi.DesignExpItem.ItemType.pll_info \
                    or item_type == expi.DesignExpItem.ItemType.pll:
                new_item = self.add_new_pll()
            elif item_type == expi.DesignExpItem.ItemType.jtag_info \
                    or item_type == expi.DesignExpItem.ItemType.jtag:
                new_item = self.add_new_jtag()
            elif item_type == expi.DesignExpItem.ItemType.spi_flash_info \
                    or item_type == expi.DesignExpItem.ItemType.spi_flash:
                new_item = self.add_new_spi_flash()
            elif item_type == expi.DesignExpItem.ItemType.hyper_ram_info \
                    or item_type == expi.DesignExpItem.ItemType.hyper_ram:
                new_item = self.add_new_hyper_ram()
            elif item_type == expi.DesignExpItem.ItemType.h264_info \
                    or item_type == expi.DesignExpItem.ItemType.h264:
                new_item = self.add_new_h264()
            elif item_type == expi.DesignExpItem.ItemType.ddr_info \
                    or item_type == expi.DesignExpItem.ItemType.ddr:
                new_item = self.add_new_ddr()
            elif item_type == expi.DesignExpItem.ItemType.osc_info \
                    or item_type == expi.DesignExpItem.ItemType.osc:
                new_item = self.add_new_osc()
            elif item_type == expi.DesignExpItem.ItemType.lvds_tx_info \
                    or item_type == expi.DesignExpItem.ItemType.lvds_tx:
                new_item = self.add_new_lvds_tx()
            elif item_type == expi.DesignExpItem.ItemType.lvds_rx_info \
                    or item_type == expi.DesignExpItem.ItemType.lvds_rx:
                new_item = self.add_new_lvds_rx()
            elif item_type == expi.DesignExpItem.ItemType.lvds_bidir_info \
                    or item_type == expi.DesignExpItem.ItemType.lvds_bidir:
                new_item = self.add_new_advance_lvds(expi.DesignExpItem.ItemType.lvds_bidir_info,
                                                     expi.DesignExpItem.ItemType.lvds_bidir)
            elif item_type == expi.DesignExpItem.ItemType.mipi_tx_info \
                    or item_type == expi.DesignExpItem.ItemType.mipi_tx:
                new_item = self.add_new_mipi_tx()
            elif item_type == expi.DesignExpItem.ItemType.mipi_rx_info \
                    or item_type == expi.DesignExpItem.ItemType.mipi_rx:
                new_item = self.add_new_mipi_rx()
            elif item_type == expi.DesignExpItem.ItemType.mipi_dphy_tx_info \
                    or item_type == expi.DesignExpItem.ItemType.mipi_dphy_tx:
                new_item = self.add_new_mipi_dphy(expi.DesignExpItem.ItemType.mipi_dphy_tx_info,
                                                  expi.DesignExpItem.ItemType.mipi_dphy_tx)
            elif item_type == expi.DesignExpItem.ItemType.mipi_dphy_rx_info \
                    or item_type == expi.DesignExpItem.ItemType.mipi_dphy_rx:
                new_item = self.add_new_mipi_dphy(expi.DesignExpItem.ItemType.mipi_dphy_rx_info,
                                                  expi.DesignExpItem.ItemType.mipi_dphy_rx)

            elif item_type == expi.DesignExpItem.ItemType.mipi_hard_dphy_tx_info \
                    or item_type == expi.DesignExpItem.ItemType.mipi_hard_dphy_tx:
                new_item = self.add_new_mipi_hard_dphy_tx()
            elif item_type == expi.DesignExpItem.ItemType.mipi_hard_dphy_rx_info \
                    or item_type == expi.DesignExpItem.ItemType.mipi_hard_dphy_rx:
                new_item = self.add_new_mipi_hard_dphy_rx()

            elif item_type == expi.DesignExpItem.ItemType.pll_ssc_info \
                    or item_type == expi.DesignExpItem.ItemType.pll_ssc:
                new_item = self.add_new_pll_ssc()

            elif item_type == expi.DesignExpItem.ItemType.quad_pcie_info \
                    or item_type == expi.DesignExpItem.ItemType.quad_pcie:
                new_item = self.add_new_quad_pcie()

            elif item_type == expi.DesignExpItem.ItemType.lane_10g_info \
                    or item_type == expi.DesignExpItem.ItemType.lane_10g:
                new_item = self.add_new_10g()

            elif item_type == expi.DesignExpItem.ItemType.lane_1g_info \
                    or item_type == expi.DesignExpItem.ItemType.lane_1g:
                new_item = self.add_new_1g()

            elif item_type == expi.DesignExpItem.ItemType.raw_serdes_info \
                    or item_type == expi.DesignExpItem.ItemType.raw_serdes:
                new_item = self.add_new_raw_serdes()

            elif item_type == expi.DesignExpItem.ItemType.soc_info \
                    or item_type == expi.DesignExpItem.ItemType.soc:
                new_item = self.add_new_soc()

            if new_item is not None:
                self.tw_design_exp.update()
                self.tw_design_exp.setCurrentItem(new_item)
                # New selection signal is not triggered, need to manually call
                # this
                self.on_item_selected(new_item, 0)
                self.sig_block_added.emit(item_type)
                self.log_add_block_success(new_item.get_db_item().name, new_item.get_type().name)

    def add_new_gpio(self):
        """
        Add new gpio instance

        :return: GPIO instance tree item
        """
        reg = self.design.gpio_reg
        if reg is not None:

            name = reg.gen_unique_gpio_name()
            if name is None:
                self.logger.error("Fail to generate unique gpio name")
                return None

            try:
                gpio = reg.create_gpio(name, auto_pin=True)

                if gpio is not None:
                    # Setup the ddio support
                    gpio.enable_ddio_support(self.design.is_support_ddio())

                    item = expi.DesignExpItem(
                        self.gpio_folder, expi.DesignExpItem.ItemType.gpio)
                    item.setup(gpio)
                    item.setText(0, "{} : {}".format(gpio.name, ""))

                    self.register_dirty_item(item, True)
                    self.gpio_folder.sortChildren(0, QtCore.Qt.AscendingOrder)
                    self.update_block_folder(
                        self.gpio_folder, reg.get_gpio_count())

                    return item

            except blk_excp.CreateException as exc:
                QtWidgets.QMessageBox.warning(
                    self.parent, "Create gpio", exc.msg)
                self.logger.error("Fail to create new gpio")

            except app_excp.PTException:
                self.logger.error("Fail to create new gpio")

        return None

    def add_new_pll(self):
        """
        Add new pll instance

        :return: PLL instance tree item
        """

        reg = self.design.pll_reg
        if reg is not None:

            name = reg.gen_unique_inst_name(prefix="pll")
            if name is None:
                self.logger.error("Fail to generate unique pll name")
                return None

            try:

                pll = reg.create_pll(name, auto_pin=True)
                if pll is not None:
                    item = expi.DesignExpItem(
                        self.pll_folder, expi.DesignExpItem.ItemType.pll)
                    item.setup(pll)
                    item.setText(0, "{} : {}".format(pll.name, pll.pll_def))

                    self.register_dirty_item(item, True)
                    self.pll_folder.sortChildren(0, QtCore.Qt.AscendingOrder)
                    self.update_block_folder(
                        self.pll_folder, reg.get_inst_count())

                    if pll.pll_def != "":
                        self.sig_block_res_changed.emit(str(item.get_type()), '', pll.pll_def)

                    return item

            except blk_excp.CreateException as exc:
                QtWidgets.QMessageBox.warning(
                    self.parent, "Create pll", exc.msg)
                self.logger.error("Fail to create new pll")

            except app_excp.PTException:
                self.logger.error("Fail to create new pll")

        return None

    def add_new_osc(self):
        """
        Add new osc instance

        :return: osc instance tree item
        """

        reg = self.design.osc_reg
        if reg is not None:

            name = reg.gen_unique_osc_name()
            if name is None:
                self.logger.error("Fail to generate unique osc name")
                return None

            try:
                osc = reg.create_osc(name, auto_pin=True)
                if osc is not None:
                    item = expi.DesignExpItem(
                        self.osc_folder, expi.DesignExpItem.ItemType.osc)
                    item.setup(osc)
                    item.setText(0, "{} : {}".format(osc.name, osc.osc_def))

                    self.register_dirty_item(item, True)
                    self.osc_folder.sortChildren(0, QtCore.Qt.AscendingOrder)
                    self.update_block_folder(
                        self.osc_folder, reg.get_osc_count())
                    return item

            except blk_excp.CreateException as exc:
                QtWidgets.QMessageBox.warning(
                    self.parent, "Create oscillator", exc.msg)
                self.logger.error("Fail to create new oscillator")

            except app_excp.PTException:
                self.logger.error("Fail to create new osc")

        return None

    def add_new_lvds_tx(self):
        """
        Add new lvds tx instance

        :return: LVDS instance tree item
        """

        if self.design.is_block_supported(db.PeriDesign.BlockType.adv_lvds):
            return self.add_new_advance_lvds(expi.DesignExpItem.ItemType.lvds_tx_info,
                                             expi.DesignExpItem.ItemType.lvds_tx)

        reg = self.design.lvds_reg
        if reg is not None:

            name = reg.gen_unique_inst_name(prefix="lvds_tx")
            if name is None:
                self.logger.error("Fail to generate unique lvds tx name")
                return None

            try:

                lvds = reg.create_tx_instance(name, auto_pin=True)

                if lvds is not None:
                    item = expi.DesignExpItem(self.lvds_tx_folder, expi.DesignExpItem.ItemType.lvds_tx)
                    item.setup(lvds)
                    item.setText(0, "{} : {}".format(lvds.name, lvds.lvds_def))

                    self.register_dirty_item(item, True)
                    self.lvds_tx_folder.sortChildren(0, QtCore.Qt.AscendingOrder)
                    self.update_block_folder(self.lvds_tx_folder, reg.get_type_inst_count(True))

                    return item

            except blk_excp.CreateException as exc:
                QtWidgets.QMessageBox.warning(self.parent, "Create lvds tx", exc.msg)
                self.logger.error("Fail to create new lvds tx")

            except app_excp.PTException:
                self.logger.error("Fail to create new lvds tx")

        return None

    def add_new_lvds_rx(self):
        """
        Add new lvds rx instance

        :return: LVDS rx instance tree item
        """

        if self.design.is_block_supported(db.PeriDesign.BlockType.adv_lvds):
            return self.add_new_advance_lvds(expi.DesignExpItem.ItemType.lvds_rx_info,
                                             expi.DesignExpItem.ItemType.lvds_rx)

        reg = self.design.lvds_reg
        if reg is not None:

            name = reg.gen_unique_inst_name(prefix="lvds_rx")
            if name is None:
                self.logger.error("Fail to generate unique lvds rx name")
                return None

            try:

                lvds = reg.create_rx_instance(name, auto_pin=True)

                if lvds is not None:
                    item = expi.DesignExpItem(self.lvds_rx_folder, expi.DesignExpItem.ItemType.lvds_rx)
                    item.setup(lvds)
                    item.setText(0, "{} : {}".format(lvds.name, lvds.lvds_def))

                    self.register_dirty_item(item, True)
                    self.lvds_rx_folder.sortChildren(0, QtCore.Qt.AscendingOrder)
                    self.update_block_folder(self.lvds_rx_folder, reg.get_type_inst_count(False))

                    return item

            except blk_excp.CreateException as exc:
                QtWidgets.QMessageBox.warning(self.parent, "Create lvds rx", exc.msg)
                self.logger.error("Fail to create new lvds rx")

            except app_excp.PTException:
                self.logger.error("Fail to create new lvds rx")

        return None

    def add_new_advance_lvds(self, folder_type: expi.DesignExpItem.ItemType,
                             item_type: expi.DesignExpItem.ItemType) -> Union[expi.DesignExpItem, None]:
        """
        Add new advanced LVDS instance

        :return: lvds instance tree item
        """
        reg = self.design.lvds_reg
        if reg is not None:

            lvds = None
            inst_count = 0
            lvds_api = self.design_api.get_block_apimgr_by_db_type(db.PeriDesign.BlockType.adv_lvds)
            try:
                if item_type == expi.DesignExpItem.ItemType.lvds_tx:
                    lvds = lvds_api.create_tx_block_auto_name()
                    inst_count = reg.get_tx_inst_count()
                elif item_type == expi.DesignExpItem.ItemType.lvds_rx:
                    lvds = lvds_api.create_rx_block_auto_name()
                    inst_count = reg.get_rx_inst_count()
                elif item_type == expi.DesignExpItem.ItemType.lvds_bidir:
                    lvds = lvds_api.create_bidir_block_auto_name()
                    inst_count = reg.get_bidir_inst_count()

                if lvds is not None:
                    item = self.add_block_item(folder_type, item_type, lvds, inst_count)
                    return item

            except blk_excp.CreateException as exc:
                QtWidgets.QMessageBox.warning(self.parent, "Create lvds", exc.msg)
                self.logger.error("Fail to create new lvds")

            except app_excp.PTException:
                self.logger.error("Fail to create new lvds")

        return None

    def add_new_mipi_tx(self):
        """
        Add new mipi tx instance

        :return: MIPI instance tree item
        """

        reg = self.design.mipi_reg
        if reg is not None:

            name = reg.gen_unique_inst_name(prefix="mipi_tx")
            if name is None:
                self.logger.error("Fail to generate unique mipi tx name")
                return None

            try:
                mipi = reg.create_tx_instance(name, auto_pin=True)

                if mipi is not None:
                    item = expi.DesignExpItem(self.mipi_tx_folder, expi.DesignExpItem.ItemType.mipi_tx)
                    item.setup(mipi)
                    item.setText(0, "{} : {}".format(mipi.name, mipi.mipi_def))

                    self.register_dirty_item(item, True)
                    self.mipi_tx_folder.sortChildren(0, QtCore.Qt.AscendingOrder)
                    self.update_block_folder(self.mipi_tx_folder, reg.get_type_inst_count(True))

                    return item

            except blk_excp.CreateException as exc:
                QtWidgets.QMessageBox.warning(self.parent, "Create mipi tx", exc.msg)
                self.logger.error("Fail to create new mipi tx")

            except app_excp.PTException:
                self.logger.error("Fail to create new mipi tx")

        return None

    def add_new_mipi_rx(self):
        """
        Add new mipi rx instance

        :return: MIPI instance tree item
        """

        reg = self.design.mipi_reg
        if reg is not None:

            name = reg.gen_unique_inst_name(prefix="mipi_rx")
            if name is None:
                self.logger.error("Fail to generate unique mipi rx name")
                return None

            try:
                mipi = reg.create_rx_instance(name, auto_pin=True)

                if mipi is not None:
                    item = expi.DesignExpItem(self.mipi_rx_folder, expi.DesignExpItem.ItemType.mipi_rx)
                    item.setup(mipi)
                    item.setText(0, "{} : {}".format(mipi.name, mipi.mipi_def))

                    self.register_dirty_item(item, True)
                    self.mipi_rx_folder.sortChildren(0, QtCore.Qt.AscendingOrder)
                    self.update_block_folder(self.mipi_rx_folder, reg.get_type_inst_count(False))

                    return item

            except blk_excp.CreateException as exc:
                QtWidgets.QMessageBox.warning(self.parent, "Create mipi rx", exc.msg)
                self.logger.error("Fail to create new mipi rx")

            except app_excp.PTException:
                self.logger.error("Fail to create new mipi rx")

        return None

    def add_new_mipi_hard_dphy_tx(self) -> Union[None, expi.DesignExpItem]:
        """
        Add new mipi hard dphy tx instance
        """
        reg = self.design.mipi_hard_dphy_reg
        if reg is None:
            return None

        name = reg.gen_unique_direct_inst_name(direct_type='tx', prefix="mipi_dphy_tx")
        if name is None:
            self.logger.error("Failed to generate unique mipi hard dphy tx name")
            return None

        try:
            mipi = reg.create_tx_instance(name, auto_pin=True)
            if mipi is None:
                return None

            mipi.build_generic_pin()
            mipi.generate_pin_name()

            item = expi.DesignExpItem(self.mipi_hard_dphy_tx_folder, expi.DesignExpItem.ItemType.mipi_hard_dphy_tx)
            item.setup(mipi)
            item.setText(0, "{} : {}".format(mipi.name, mipi.mipi_def))

            self.register_dirty_item(item, True)
            self.mipi_hard_dphy_tx_folder.sortChildren(0, QtCore.Qt.AscendingOrder)
            self.update_block_folder(self.mipi_hard_dphy_tx_folder, reg.get_type_inst_count(True))
            return item

        except (blk_excp.CreateException, app_excp.PTException) as exc:
            self.logger.error("Failed to create new mipi hard dphy tx")
            return None

        return None

    def add_new_mipi_hard_dphy_rx(self):
        """
        Add new mipi hard dphy rx instance
        """
        reg = self.design.mipi_hard_dphy_reg
        if reg is None:
            return None

        name = reg.gen_unique_direct_inst_name(direct_type='rx', prefix="mipi_dphy_rx")
        if name is None:
            self.logger.error("Failed to generate unique mipi hard dphy rx name")
            return None

        try:
            mipi = reg.create_rx_instance(name, auto_pin=True)
            if mipi is None:
                return None

            mipi.build_generic_pin()
            mipi.generate_pin_name()

            item = expi.DesignExpItem(self.mipi_hard_dphy_rx_folder, expi.DesignExpItem.ItemType.mipi_hard_dphy_rx)
            item.setup(mipi)
            item.setText(0, "{} : {}".format(mipi.name, mipi.mipi_def))

            self.register_dirty_item(item, True)
            self.mipi_hard_dphy_rx_folder.sortChildren(0, QtCore.Qt.AscendingOrder)
            self.update_block_folder(self.mipi_hard_dphy_rx_folder, reg.get_type_inst_count(True))
            return item

        except (blk_excp.CreateException, app_excp.PTException) as exc:
            self.logger.error("Failed to create new mipi advance rx")
            return None

    def add_new_mipi_dphy(self, folder_type: expi.DesignExpItem.ItemType,
                          item_type: expi.DesignExpItem.ItemType) -> Union[expi.DesignExpItem, None]:
        """
        Add new mipi dphy instance

        :return: mipi dphy instance tree item
        """

        reg = self.design.mipi_dphy_reg
        if reg is not None:

            mipi = None
            inst_count = 0
            mipi_dphy_api = self.design_api.get_block_apimgr_by_db_type(db.PeriDesign.BlockType.mipi_dphy)
            try:
                if item_type == expi.DesignExpItem.ItemType.mipi_dphy_tx:
                    mipi = mipi_dphy_api.create_tx_block_auto_name()
                    inst_count = reg.get_tx_inst_count()
                elif item_type == expi.DesignExpItem.ItemType.mipi_dphy_rx:
                    mipi = mipi_dphy_api.create_rx_block_auto_name()
                    inst_count = reg.get_rx_inst_count()

                if mipi is not None:
                    item = self.add_block_item(folder_type, item_type, mipi, inst_count)
                    return item

            except blk_excp.CreateException as exc:
                QtWidgets.QMessageBox.warning(self.parent, "Create mipi lane", exc.msg)
                self.logger.error("Fail to create new mipi lane")

            except app_excp.PTException:
                self.logger.error("Fail to create new mipi lane")

        return None

    def add_new_jtag(self):
        """
        Add new jtag instance

        :return: JTAG instance tree item
        """

        reg = self.design.jtag_reg
        if reg is not None:

            name = reg.gen_unique_inst_name(prefix="jtag")
            if name is None:
                self.logger.error("Fail to generate unique jtag name")
                return None

            try:
                jtag = reg.create_instance(name)
                if jtag is not None:
                    jtag.build_generic_pin()
                    # Need to have the pins first before we can automate the pin name
                    jtag.generate_pin_name()

                    item = expi.DesignExpItem(self.jtag_folder, expi.DesignExpItem.ItemType.jtag)
                    item.setup(jtag)
                    item.setText(0, "{} : {}".format(jtag.name, jtag.jtag_def))

                    self.register_dirty_item(item, True)
                    self.jtag_folder.sortChildren(0, QtCore.Qt.AscendingOrder)
                    self.update_block_folder(self.jtag_folder, reg.get_inst_count())

                    return item

            except blk_excp.CreateException as exc:
                QtWidgets.QMessageBox.warning(self.parent, "Create jtag", exc.msg)
                self.logger.error("Fail to create new jtag")

            except app_excp.PTException:
                self.logger.error("Fail to create new jtag")

        return None

    def add_new_h264(self):
        """
        Add new h264 instance

        :return: H264 instance tree item
        """

        reg = self.design.h264_reg
        if reg is not None:

            name = reg.gen_unique_inst_name(prefix="h264")
            if name is None:
                self.logger.error("Fail to generate unique h264 name")
                return None

            try:
                h264 = reg.create_instance(name)
                if h264 is not None:
                    h264.build_generic_pin()

                    # Need to have the pins first before we can automate the pin name
                    h264.generate_pin_name()

                    item = expi.DesignExpItem(self.h264_folder, expi.DesignExpItem.ItemType.h264)
                    item.setup(h264)
                    item.setText(0, "{} : {}".format(h264.name, h264.h264_def))

                    self.register_dirty_item(item, True)
                    self.h264_folder.sortChildren(0, QtCore.Qt.AscendingOrder)
                    self.update_block_folder(self.h264_folder, reg.get_inst_count())

                    return item

            except blk_excp.CreateException as exc:
                QtWidgets.QMessageBox.warning(self.parent, "Create h264", exc.msg)
                self.logger.error("Fail to create new h264")

            except app_excp.PTException:
                self.logger.error("Fail to create new h264")

        return None

    def add_new_ddr(self):
        """
        Add new ddr instance

        :return: DDR instance tree item
        """

        reg = self.design.ddr_reg
        if reg is not None:

            name = reg.gen_unique_inst_name(prefix="ddr")
            if name is None:
                self.logger.error("Fail to generate unique ddr name")
                return None

            try:
                ddr = reg.create_instance(name)
                if ddr is not None:
                    ddr.build_default_configuration(self.design.device_db)
                    item = expi.DesignExpItem(self.ddr_folder, expi.DesignExpItem.ItemType.ddr)
                    item.setup(ddr)
                    item.setText(0, "{} : {}".format(ddr.name, ddr.ddr_def))

                    self.register_dirty_item(item, True)
                    self.ddr_folder.sortChildren(0, QtCore.Qt.AscendingOrder)
                    self.update_block_folder(self.ddr_folder, reg.get_inst_count())

                    return item

            except blk_excp.CreateException as exc:
                QtWidgets.QMessageBox.warning(self.parent, "Create ddr", exc.msg)
                self.logger.error("Fail to create new ddr")

            except app_excp.PTException:
                self.logger.error("Fail to create new ddr")

        return None

    def add_new_spi_flash(self):
        """
        Add new spi flash instance

        :return: SPI Flash instance tree item
        """

        reg = self.design.spi_flash_reg
        if reg is not None:
            spi_flash_api = self.design_api.get_block_apimgr_by_db_type(db.PeriDesign.BlockType.spi_flash)
            try:
                # Set empty name to auto generate
                spf = spi_flash_api.create_block(name="", is_register=True)
                inst_count = reg.get_inst_count()
                if spf is not None:
                    item = self.add_block_item(expi.DesignExpItem.ItemType.spi_flash_info,
                                               expi.DesignExpItem.ItemType.spi_flash, spf, inst_count)
                    return item

            except blk_excp.CreateException as exc:
                QtWidgets.QMessageBox.warning(self.parent, "Create spi flash", exc.msg)
                self.logger.error("Fail to create new spi flash")

            except app_excp.PTException:
                self.logger.error("Fail to create new spi flash")

        return None

    def add_new_hyper_ram(self):
        """
        Add new hyper ram instance

        :return: HyperRAM instance tree item
        """

        reg = self.design.hyper_ram_reg
        if reg is not None:
            hyper_ram_api = self.design_api.get_block_apimgr_by_db_type(db.PeriDesign.BlockType.hyper_ram)
            try:
                # Set empty name to auto generate
                hpram = hyper_ram_api.create_block(name="", is_register=True)
                inst_count = reg.get_inst_count()
                if hpram is not None:
                    item = self.add_block_item(expi.DesignExpItem.ItemType.hyper_ram_info,
                                               expi.DesignExpItem.ItemType.hyper_ram, hpram, inst_count)
                    return item

            except blk_excp.CreateException as exc:
                QtWidgets.QMessageBox.warning(self.parent, "Create hyper ram", exc.msg)
                self.logger.error("Fail to create new hyper ram")

            except app_excp.PTException:
                self.logger.error("Fail to create new hyper ram")

        return None

    def add_new_pll_ssc(self):
        """
        Add new pll ssc instance

        :return: PLL SSC instance tree item
        """
        assert self.design is not None
        reg = self.design.pll_ssc_reg

        if reg is None or self.pll_ssc_folder is None:
            return None

        name = reg.gen_unique_inst_name(prefix="pll_ssc")
        if name is None:
            self.logger.error("Failed to generate unique pll ssc name")
            return None

        try:
            inst = reg.create_instance(name, auto_pin=True)
            if inst is None:
                return None

            inst.build_generic_pin()
            inst.generate_pin_name()

            item = expi.DesignExpItem(self.pll_ssc_folder, expi.DesignExpItem.ItemType.pll_ssc)
            item.setup(inst)
            item.setText(0, "{} : {}".format(inst.name, inst.mipi_def))

            self.register_dirty_item(item, True)
            self.pll_ssc_folder.sortChildren(0, QtCore.Qt.AscendingOrder)
            self.update_block_folder(self.pll_ssc_folder, reg.get_inst_count())
            return item

        except (blk_excp.CreateException, app_excp.PTException) as exc:
            self.logger.error("Failed to create new pll ssc")
            return None

    def add_new_quad_pcie(self):
        """
        Add new quad pcie instance

        :return: QuadPCIE instance tree item
        """
        assert self.design is not None
        reg = self.design.quad_pcie_reg

        if reg is None or self.quad_pcie_folder is None:
            return None

        name = reg.gen_unique_inst_name(prefix="pcie")
        if name is None:
            self.logger.error("Failed to generate unique quad pcie name")
            return None

        try:
            inst = reg.create_instance(name, auto_pin=True)
            if inst is None:
                return None

            inst.build_generic_pin()
            inst.generate_pin_name()

            item = expi.DesignExpItem(self.quad_pcie_folder, expi.DesignExpItem.ItemType.quad_pcie)
            item.setup(inst)
            item.setText(0, "{} : {}".format(inst.name, inst.get_device()))

            self.register_dirty_item(item, True)
            self.quad_pcie_folder.sortChildren(0, QtCore.Qt.AscendingOrder)
            self.update_block_folder(self.quad_pcie_folder, reg.get_inst_count())
            return item

        except (blk_excp.CreateException, app_excp.PTException) as exc:
            self.logger.error("Failed to create new quad pcie")
            return None

    def add_new_10g(self):
        """
        Add new 10g instance

        :return: Lane10G instance tree item
        """
        assert self.design is not None
        reg = self.design.lane_10g_reg

        if reg is None or self.lane_10g_folder is None:
            return None

        name = reg.gen_unique_inst_name(prefix="xgmii")
        if name is None:
            self.logger.error("Failed to generate unique Ethernet XGMII name")
            return None

        try:
            inst = reg.create_instance(name, auto_pin=True)
            if inst is None:
                return None

            inst.build_generic_pin()
            inst.generate_pin_name()

            item = expi.DesignExpItem(self.lane_10g_folder, expi.DesignExpItem.ItemType.lane_10g)
            item.setup(inst)
            item.setText(0, "{} : {}".format(inst.name, inst.get_device()))

            self.register_dirty_item(item, True)
            self.lane_10g_folder.sortChildren(0, QtCore.Qt.AscendingOrder)
            self.update_block_folder(self.lane_10g_folder, reg.get_inst_count())
            return item

        except (blk_excp.CreateException, app_excp.PTException) as exc:
            self.logger.error("Failed to create new Ethernet XGMII")
            return None

    def add_new_1g(self):
        """
        Add new 1g instance

        :return: Lane1G instance tree item
        """
        assert self.design is not None
        reg = self.design.lane_1g_reg

        if reg is None or self.lane_1g_folder is None:
            return None

        name = reg.gen_unique_inst_name(prefix="sgmii")
        if name is None:
            self.logger.error("Failed to generate unique Ethernet SGMII name")
            return None

        try:
            inst = reg.create_instance(name, auto_pin=True)
            if inst is None:
                return None

            inst.build_generic_pin()
            inst.generate_pin_name()

            item = expi.DesignExpItem(self.lane_1g_folder,
                                      expi.DesignExpItem.ItemType.lane_1g)
            item.setup(inst)
            item.setText(0, "{} : {}".format(inst.name, inst.get_device()))

            self.register_dirty_item(item, True)
            self.lane_1g_folder.sortChildren(0, QtCore.Qt.AscendingOrder)
            self.update_block_folder(self.lane_1g_folder, reg.get_inst_count())
            return item

        except (blk_excp.CreateException, app_excp.PTException) as exc:
            self.logger.error("Failed to create new Ethernet XGMII")
            return None

    def add_new_raw_serdes(self):
        """
        Add new raw serdes instance

        :return: RawSerdes instance tree item
        """
        assert self.design is not None
        reg = self.design.raw_serdes_reg

        if reg is None or self.raw_serdes_folder is None:
            return None

        name = reg.gen_unique_inst_name(prefix="pma_direct")
        if name is None:
            self.logger.error("Failed to generate unique PMA Direct name")
            return None

        try:
            inst = reg.create_instance(name, auto_pin=True)
            if inst is None:
                return None

            inst.build_generic_pin()
            inst.generate_pin_name()

            item = expi.DesignExpItem(self.raw_serdes_folder, expi.DesignExpItem.ItemType.raw_serdes)
            item.setup(inst)
            item.setText(0, "{} : {}".format(inst.name, inst.get_device()))

            self.register_dirty_item(item, True)
            self.raw_serdes_folder.sortChildren(0, QtCore.Qt.AscendingOrder)
            self.update_block_folder(self.raw_serdes_folder, reg.get_inst_count())
            return item

        except (blk_excp.CreateException, app_excp.PTException) as exc:
            self.logger.error("Failed to create new PMA Direct")
            return None

    def add_new_soc(self):
        """
        Add new soc instance
        """
        assert self.design is not None
        reg = self.design.soc_reg

        if reg is None or self.soc_folder is None:
            return None

        name = reg.gen_unique_inst_name(prefix="qcrv32")
        if name is None:
            self.logger.error("Failed to generate unique soc name")
            return None

        try:
            inst = reg.create_instance(name, auto_pin=True)
            if inst is None:
                return None

            inst.build_generic_pin()
            inst.generate_pin_name()

            item = expi.DesignExpItem(self.soc_folder, expi.DesignExpItem.ItemType.soc)
            item.setup(inst)
            item.setText(0, "{} : {}".format(inst.name, inst.get_device()))

            self.register_dirty_item(item, True)
            self.soc_folder.sortChildren(0, QtCore.Qt.AscendingOrder)
            self.update_block_folder(self.soc_folder, reg.get_inst_count())
            return item

        except (blk_excp.CreateException, app_excp.PTException) as exc:
            self.logger.error("Failed to create new soc")
            return None

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_gpio_bus_added(self):
        """
        Add new gpio_bus instance
        """
        # Get bus config from user
        self.new_bus_wizard.show()

    def add_new_gpio_bus_wizard(self):
        """
        Add new gpio bus instance

        :return: GPIO bus instance tree item
        """
        if self.design is None:
            return

        name, left, right, gpio_template = self.new_bus_wizard.get_config()
        reg = self.design.gpio_reg
        if reg is not None:
            bus_inst = reg.create_bus(name, left, right, gpio_template.mode,
                                      gpio_template, gpio_template.io_standard)
            if bus_inst is not None:
                QtGui.QGuiApplication.setOverrideCursor(QtCore.Qt.WaitCursor)
                bus_folder = self.build_gpio_bus(bus_inst)
                self.tw_design_exp.update()
                self.tw_design_exp.setCurrentItem(bus_folder)
                # New selection signal is not triggered, need to manually call
                # this
                self.on_item_selected(bus_folder, 0)
                self.gpio_folder.sortChildren(0, QtCore.Qt.AscendingOrder)
                self.update_block_folder(self.gpio_folder, reg.get_gpio_count())

                QtGui.QGuiApplication.restoreOverrideCursor()

                self.sig_block_added.emit(expi.DesignExpItem.ItemType.gpio_bus)

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_block_deleted(self):
        """
        Remove selected block instance
        """
        sel_item_list = self.tw_design_exp.selectedItems()

        is_clean_up = False

        count = len(sel_item_list)
        if count > 0:
            item = sel_item_list[0]
            assert isinstance(item, expi.DesignExpItem)
            item_type = item.get_type()
            parent_folder = None
            ori_dev = ""
            assert self.design is not None

            if item_type == expi.DesignExpItem.ItemType.gpio:
                reg = self.design.gpio_reg
                if reg is not None:
                    gpio:Optional[GPIO] = item.get_db_item()
                    if gpio is not None:
                        parent_folder = self.gpio_folder
                        is_clean_up = True
                        ori_dev = gpio.get_device()
                        if self.design.is_tesseract_design():
                            reg.delete_gpio(gpio.name)
                        else:
                            if gpio.is_lvds_gpio():
                                self.design.delete_lvds_gpio(gpio)
                            else:
                                reg.delete_gpio(gpio.name)

                        self.update_block_folder(self.gpio_folder, reg.get_gpio_count())

            elif item_type == expi.DesignExpItem.ItemType.gpio_bus:
                reg = self.design.gpio_reg
                if reg is not None:
                    gpio_bus:Optional[GPIOBus] = item.get_db_item()
                    if gpio_bus is not None:
                        ori_dev = []
                        parent_folder = self.gpio_folder
                        is_clean_up = True

                        count = item.childCount()
                        for i in range(0, count):
                            child_item = item.child(i)
                            res_name:str = child_item.get_current_dev_name()
                            if res_name != '':
                                ori_dev.append(res_name)

                        if self.design.is_tesseract_design():
                            reg.delete_bus(gpio_bus.get_name())
                        else:
                            # PT-1028: Change to calling delete bus at the design
                            self.design.delete_gpio_bus(gpio_bus.get_name())

                        self.update_block_folder(self.gpio_folder, reg.get_gpio_count())

            elif item_type == expi.DesignExpItem.ItemType.pll:
                reg = self.design.pll_reg
                if reg is not None:
                    pll:Optional[PLL] = item.get_db_item()
                    if pll is not None:
                        parent_folder = self.pll_folder
                        is_clean_up = True
                        ori_dev = pll.get_device()
                        reg.delete_inst(pll.name)
                        self.update_block_folder(
                            self.pll_folder, reg.get_inst_count())

            # TODO : lvds, jtag, mipi, ddr and jtag use the common base design registry class, so
            # we should not need to replicate the code for each block type
            elif item_type == expi.DesignExpItem.ItemType.jtag:
                reg = self.design.jtag_reg
                if reg is not None:
                    jtag:Optional[JTAG] = item.get_db_item()
                    if jtag is not None:
                        parent_folder = self.jtag_folder
                        is_clean_up = True
                        ori_dev = jtag.get_device()
                        reg.delete_inst(jtag.name)
                        self.update_block_folder(self.jtag_folder, reg.get_inst_count())

            elif item_type == expi.DesignExpItem.ItemType.h264:
                reg = self.design.h264_reg
                if reg is not None:
                    h264 = item.get_db_item()
                    if h264 is not None:
                        parent_folder = self.h264_folder
                        is_clean_up = True
                        reg.delete_inst(h264.name)
                        self.update_block_folder(self.h264_folder, reg.get_inst_count())

            elif item_type == expi.DesignExpItem.ItemType.ddr:
                reg = self.design.ddr_reg
                if reg is not None:
                    ddr:Optional[DDR] = item.get_db_item()
                    if ddr is not None:
                        parent_folder = self.ddr_folder
                        is_clean_up = True
                        ori_dev = ddr.get_device()
                        reg.delete_inst(ddr.name)
                        self.update_block_folder(self.ddr_folder, reg.get_inst_count())

            elif item_type == expi.DesignExpItem.ItemType.osc:
                reg = self.design.osc_reg
                if reg is not None:
                    osc:Optional[OSC] = item.get_db_item()
                    if osc is not None:
                        parent_folder = self.osc_folder
                        is_clean_up = True
                        reg.delete_inst(osc.name)
                        self.update_block_folder(
                            self.osc_folder, reg.get_osc_count())

            elif item_type == expi.DesignExpItem.ItemType.lvds_tx:
                reg = self.design.lvds_reg
                if reg is not None:
                    lvds:Optional[LVDS] = item.get_db_item()
                    if lvds is not None:
                        parent_folder = self.lvds_tx_folder
                        is_clean_up = True
                        ori_dev = lvds.get_device()
                        reg.reset_inst_device(lvds)
                        reg.delete_inst(lvds.name)
                        self.update_block_folder(self.lvds_tx_folder, reg.get_type_inst_count(True))

            elif item_type == expi.DesignExpItem.ItemType.lvds_rx:
                reg = self.design.lvds_reg
                if reg is not None:
                    lvds:Optional[LVDS] = item.get_db_item()
                    if lvds is not None:
                        parent_folder = self.lvds_rx_folder
                        is_clean_up = True
                        ori_dev = lvds.get_device()
                        reg.reset_inst_device(lvds)
                        reg.delete_inst(lvds.name)
                        self.update_block_folder(self.lvds_rx_folder, reg.get_type_inst_count(False))

            elif item_type == expi.DesignExpItem.ItemType.lvds_bidir:
                reg = self.design.lvds_reg
                if reg is not None:
                    lvds:Optional[LVDS] = item.get_db_item()
                    if lvds is not None:
                        parent_folder = self.lvds_bidir_folder
                        is_clean_up = True
                        ori_dev = lvds.get_device()
                        reg.reset_inst_device(lvds)
                        reg.delete_inst(lvds.name)
                        self.update_block_folder(self.lvds_bidir_folder, reg.get_bidir_inst_count())

            elif item_type == expi.DesignExpItem.ItemType.mipi_tx:
                reg = self.design.mipi_reg
                if reg is not None:
                    mipi:Optional[MIPI] = item.get_db_item()
                    if mipi is not None:
                        parent_folder = self.mipi_tx_folder
                        is_clean_up = True
                        ori_dev = mipi.get_device()
                        reg.delete_inst(mipi.name)
                        self.update_block_folder(self.mipi_tx_folder, reg.get_type_inst_count(True))

            elif item_type == expi.DesignExpItem.ItemType.mipi_rx:
                reg = self.design.mipi_reg
                if reg is not None:
                    mipi:Optional[MIPI] = item.get_db_item()
                    if mipi is not None:
                        parent_folder = self.mipi_rx_folder
                        is_clean_up = True
                        ori_dev = mipi.get_device()
                        reg.delete_inst(mipi.name)
                        self.update_block_folder(self.mipi_rx_folder, reg.get_type_inst_count(False))

            elif item_type == expi.DesignExpItem.ItemType.mipi_dphy_tx:
                reg = self.design.mipi_dphy_reg
                if reg is not None:
                    mipi:Optional[MIPI] = item.get_db_item()
                    if mipi is not None:
                        parent_folder = self.mipi_dphy_tx_folder
                        is_clean_up = True
                        ori_dev = mipi.get_device()
                        reg.delete_inst(mipi.name)
                        self.update_block_folder(self.mipi_dphy_tx_folder, reg.get_type_inst_count(True))

            elif item_type == expi.DesignExpItem.ItemType.mipi_dphy_rx:
                reg = self.design.mipi_dphy_reg
                if reg is not None:
                    mipi:Optional[MIPI] = item.get_db_item()
                    if mipi is not None:
                        parent_folder = self.mipi_dphy_rx_folder
                        is_clean_up = True
                        ori_dev = mipi.get_device()
                        reg.delete_inst(mipi.name)
                        self.update_block_folder(self.mipi_dphy_rx_folder, reg.get_type_inst_count(False))

            elif item_type == expi.DesignExpItem.ItemType.mipi_hard_dphy_tx:
                reg = self.design.mipi_hard_dphy_reg
                if reg is not None:
                    mipi:Optional[MIPI] = item.get_db_item()
                    if mipi is not None:
                        parent_folder = self.mipi_hard_dphy_tx_folder
                        is_clean_up = True
                        ori_dev = mipi.get_device()
                        reg.delete_inst(mipi.name)
                        self.update_block_folder(self.mipi_hard_dphy_tx_folder, reg.get_type_inst_count(True))

            elif item_type == expi.DesignExpItem.ItemType.mipi_hard_dphy_rx:
                reg = self.design.mipi_hard_dphy_reg
                if reg is not None:
                    mipi:Optional[MIPI] = item.get_db_item()
                    if mipi is not None:
                        parent_folder = self.mipi_hard_dphy_rx_folder
                        is_clean_up = True
                        ori_dev = mipi.get_device()
                        reg.delete_inst(mipi.name)
                        self.update_block_folder(self.mipi_hard_dphy_rx_folder, reg.get_type_inst_count(False))

            elif item_type == expi.DesignExpItem.ItemType.spi_flash:
                reg = self.design.spi_flash_reg
                if reg is not None:
                    spi_flash:Optional[SPIFlash] = item.get_db_item()
                    if spi_flash is not None:
                        parent_folder = self.spi_flash_folder
                        is_clean_up = True
                        ori_dev = spi_flash.get_device()
                        reg.delete_inst(spi_flash.name)
                        self.update_block_folder(self.spi_flash_folder, reg.get_inst_count())

            elif item_type == expi.DesignExpItem.ItemType.hyper_ram:
                reg = self.design.hyper_ram_reg
                if reg is not None:
                    hyper_ram:Optional[HyperRAM] = item.get_db_item()
                    if hyper_ram is not None:
                        parent_folder = self.hyper_ram_folder
                        is_clean_up = True
                        ori_dev = hyper_ram.get_device()
                        reg.delete_inst(hyper_ram.name)
                        self.update_block_folder(self.hyper_ram_folder, reg.get_inst_count())

            elif item_type == expi.DesignExpItem.ItemType.pll_ssc:
                reg = self.design.pll_ssc_reg
                if reg is not None:
                    pll_ssc: Optional[PLLSSC] = item.get_db_item()
                    if pll_ssc is not None:
                        parent_folder = self.pll_ssc_folder
                        is_clean_up = True
                        ori_dev = pll_ssc.get_device()
                        reg.delete_inst(pll_ssc.name)
                        self.update_block_folder(self.pll_ssc_folder, reg.get_inst_count())

            elif item_type == expi.DesignExpItem.ItemType.quad_pcie:
                reg = self.design.quad_pcie_reg
                if reg is not None:
                    quad_pcie = item.get_db_item() # type: Optional[QuadPCIE]
                    if quad_pcie is not None:
                        parent_folder = self.quad_pcie_folder
                        is_clean_up = True
                        ori_dev = quad_pcie.get_device()
                        reg.delete_inst(quad_pcie.name)
                        self.update_block_folder(self.quad_pcie_folder, reg.get_inst_count())

            elif item_type == expi.DesignExpItem.ItemType.lane_10g:
                reg = self.design.lane_10g_reg
                if reg is not None:
                    lane_10g_inst: Optional[Lane10G]= item.get_db_item()
                    if lane_10g_inst is not None:
                        parent_folder = self.lane_10g_folder
                        is_clean_up = True
                        ori_dev = lane_10g_inst.get_device()
                        reg.delete_inst(lane_10g_inst.name)
                        self.update_block_folder(self.lane_10g_folder, reg.get_inst_count())

            elif item_type == expi.DesignExpItem.ItemType.lane_1g:
                reg = self.design.lane_1g_reg
                if reg is not None:
                    lane_1g_inst: Optional[Lane1G]= item.get_db_item()
                    if lane_1g_inst is not None:
                        parent_folder = self.lane_1g_folder
                        is_clean_up = True
                        ori_dev = lane_1g_inst.get_device()
                        reg.delete_inst(lane_1g_inst.name)
                        self.update_block_folder(self.lane_1g_folder, reg.get_inst_count())

            elif item_type == expi.DesignExpItem.ItemType.raw_serdes:
                reg = self.design.raw_serdes_reg
                if reg is not None:
                    raw_serdes_inst: Optional[RawSerdes] = item.get_db_item()
                    if raw_serdes_inst is not None:
                        parent_folder = self.raw_serdes_folder
                        is_clean_up = True
                        ori_dev = raw_serdes_inst.get_device()
                        reg.delete_inst(raw_serdes_inst.name)
                        self.update_block_folder(self.raw_serdes_folder, reg.get_inst_count())

            elif item_type == expi.DesignExpItem.ItemType.soc:
                reg = self.design.soc_reg
                if reg is not None:
                    soc: Optional[SOC] = item.get_db_item()
                    if soc is not None:
                        parent_folder = self.soc_folder
                        is_clean_up = True
                        ori_dev = soc.get_device()
                        reg.delete_inst(soc.name)
                        self.update_block_folder(self.soc_folder, reg.get_inst_count())

            if is_clean_up:
                self.log_delete_block_success(item.get_db_item().name, item_type.name)
                self.unregister_dirty_item(item)
                assert parent_folder is not None
                parent_folder.removeChild(item)
                self.tw_design_exp.update()

                # Need to manually trigger select signal
                sel_item_list = self.tw_design_exp.selectedItems()
                count = len(sel_item_list)
                if count > 0:
                    item = sel_item_list[0]
                    self.on_item_selected(item, 0)

                if isinstance(ori_dev, list):
                    for res_name in ori_dev:
                        self.sig_block_deleted.emit(item_type, res_name)
                else:
                    self.sig_block_deleted.emit(item_type, ori_dev)

    def reset_device_settings(self, setting_type: str):
        """
        Reset device settings to default one

        :param setting_type: Device setting type which match APIObject.str2otype_map
        :type setting_type: str
        """
        assert self.design is not None and self.design.device_setting is not None

        if setting_type not in APIObject.str2otype_map:
            return

        device_db = self.design.device_db
        assert device_db is not None
        self.design.device_setting.reset_setting(setting_type)

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_expand_all(self):
        """
        Expand the whole design tree
        """

        QtGui.QGuiApplication.setOverrideCursor(QtCore.Qt.WaitCursor)
        self.tw_design_exp.expandAll()
        QtGui.QGuiApplication.restoreOverrideCursor()

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_expand_selection(self):
        """
        Expand selected item
        """

        if self._sel_item is None:
            return

        QtGui.QGuiApplication.setOverrideCursor(QtCore.Qt.WaitCursor)
        self.set_expand_item(self._sel_item, True)
        QtGui.QGuiApplication.restoreOverrideCursor()

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_collapse_all(self):
        """
        Collapse the whole design tree
        """

        QtGui.QGuiApplication.setOverrideCursor(QtCore.Qt.WaitCursor)
        self.tw_design_exp.collapseAll()
        QtGui.QGuiApplication.restoreOverrideCursor()

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_collapse_selection(self):
        """
        Collapse selected item
        """

        if self._sel_item is None:
            return

        QtGui.QGuiApplication.setOverrideCursor(QtCore.Qt.WaitCursor)
        self.set_expand_item(self._sel_item, False)
        QtGui.QGuiApplication.restoreOverrideCursor()

    def set_expand_item(self, parent_item: QtWidgets.QTreeWidgetItem, is_expand: bool):
        parent_item.setExpanded(is_expand)

        for count in range(parent_item.childCount()):
            child = parent_item.child(count)
            self.set_expand_item(child, is_expand)

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_search_text_changed(self):
        """
        Start search when user key in text and press Enter in search box
        """

        search_text = self.le_search_text.text()
        self.logger.debug("DsgExp - Searched item: {}".format(search_text))
        if search_text == "":
            return

        QtGui.QGuiApplication.setOverrideCursor(QtCore.Qt.WaitCursor)

        item = self.finder.find(search_text)

        # If search fails, select design
        if item is None:
            item = self.design_folder

        self.tw_design_exp.setCurrentItem(item)
        self.on_item_selected(item, 0)

        QtGui.QGuiApplication.restoreOverrideCursor()

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_reset_filter(self):
        """
        Reset filter view to default
        """
        self.finder.reset_filter()

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_move_gpio_to_bus_request(self):
        """
        Move selected gpio to a bus
        """
        if self._sel_item is None:
            return

        gpio_reg = self.design.gpio_reg
        if gpio_reg.get_bus_count() > 0:
            self.move_to_bus_dlg.show(self.design)
        else:
            msg = "Your design does not have any bus. Please create one first."
            QtWidgets.QMessageBox.warning(None, "Move gpio instance to bus", msg)

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_move_gpio_to_bus(self):

        if self._sel_item is None or self._sel_item.get_type() != expi.DesignExpItem.ItemType.gpio:
            return

        gpio_item = self._sel_item

        gpio_inst = gpio_item.get_db_item()
        gpio_reg = self.design.gpio_reg
        if gpio_inst is not None and gpio_reg is not None:

            # Update design db
            bus_name = self.move_to_bus_dlg.get_target_bus()
            bus = gpio_reg.get_bus_by_name(bus_name)
            if bus is not None:
                bus.add(gpio_inst)

                # Update ui, remove from gpio folder
                self.gpio_folder.removeChild(gpio_item)

                # Update ui, move to bus folder
                bus_folder = None
                iterator = QtWidgets.QTreeWidgetItemIterator(self.gpio_folder)
                while iterator.value():
                    item = iterator.value()
                    if item.get_type() == expi.DesignExpItem.ItemType.gpio_bus:
                        if bus_name in item.text(0):
                            bus_folder = item
                            break
                    iterator += 1

                if bus_folder is not None:
                    gpio_item.update_icon()
                    gpio_item.update_text_display()
                    bus_folder.addChild(gpio_item)
                    bus_folder.sortChildren(0, QtCore.Qt.AscendingOrder)
                    self.update_block_folder(bus_folder, bus.get_width())

                self.sig_block_added.emit(expi.DesignExpItem.ItemType.gpio)
                self.tw_design_exp.update()

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_move_gpio_from_bus_request(self):
        """
        Move gpio instance from bus to gpio folder.
        Changing gpio instance from multi-bit to single bit.
        """

        if self._sel_item is None:
            return

        gpio_item = self._sel_item
        if gpio_item.get_type() != expi.DesignExpItem.ItemType.gpio:
            return

        # The operation is only enable on gpio instance which has a parent item
        bus_item = gpio_item.parent()
        if bus_item.get_type() != expi.DesignExpItem.ItemType.gpio_bus:
            return

        gpio_inst = gpio_item.get_db_item()
        gpio_reg = self.design.gpio_reg
        if gpio_inst is not None and gpio_reg is not None:

            # Update design db
            bus = gpio_reg.get_bus_by_name(gpio_inst.bus_name)
            if bus is not None:
                bus.remove(gpio_inst)

                # Update ui, remove gpio from bus
                bus_folder = gpio_item.parent()

                if bus_folder is not None:
                    bus_folder.removeChild(gpio_item)
                    self.update_block_folder(bus_folder, bus.get_width())

                # Update ui, add gpio to top gpio folder
                gpio_item.update_icon()
                gpio_item.update_text_display()
                self.gpio_folder.addChild(gpio_item)
                self.gpio_folder.sortChildren(0, QtCore.Qt.AscendingOrder)
                self.update_block_folder(self.gpio_folder, self.gpio_folder.childCount())

                self.sig_block_added.emit(expi.DesignExpItem.ItemType.gpio)
                self.tw_design_exp.update()

    def update_action_status(self, enable_ops_list):

        self.actionCreate_Block.setEnabled(False)
        self.actionCreate_GPIO_Bus.setEnabled(False)
        self.actionDelete_Block.setEnabled(False)
        self.actionExpand_All.setEnabled(False)
        self.actionCollapse_All.setEnabled(False)

        for ops in enable_ops_list:
            if ops == exp_menu.DesignExpMenu.OpsType.add_ops:
                self.actionCreate_Block.setEnabled(True)

            elif ops == exp_menu.DesignExpMenu.OpsType.del_ops:
                self.actionDelete_Block.setEnabled(True)

            elif ops == exp_menu.DesignExpMenu.OpsType.add_bus_ops:
                self.actionCreate_GPIO_Bus.setEnabled(True)

            elif ops == exp_menu.DesignExpMenu.OpsType.exp_curr_ops:
                self.actionExpand_All.setEnabled(True)

            elif ops == exp_menu.DesignExpMenu.OpsType.colp_curr_ops:
                self.actionCollapse_All.setEnabled(True)

        self.tw_design_exp.update()

    def log_delete_block_success(self, name: str, item_type: str) -> bool:
        """
        Show the success of deleting block

        :param name: the name of the item
        :param item_type: the ype of the ite,
        :return: The success of print the log (True)
        """
        self.logger.debug("Success to delete block, name: {}, type: {}".format(name, item_type))
        return True

    def log_add_block_success(self, name: str, item_type: str) -> bool:
        """
        Show the success of creating block

        :param name: the name of the item
        :param item_type: the ype of the ite,
        :return: The success of print the log (True)
        """
        self.logger.debug("Success to create block, name: {}, type: {}".format(name, item_type))
        return True


if __name__ == "__main__":
    pass
