#!/usr/bin/env python3

"""
Python runscript for running interface design flow

Support both unified flow / legacy flow
"""

from __future__ import annotations
import argparse
from itertools import chain
from pathlib import Path
import sys
import os
import subprocess
import platform
import traceback

from typing import Optional, TypedDict, NotRequired, List, TYPE_CHECKING
from enum import IntEnum

from prettytable import PrettyTable
from projectdb.db import EfxProjectDatabase

PT_HOME = os.environ['EFXPT_HOME']
sys.path.append(PT_HOME + "/bin")

from device.service import DeviceService
from api_service.design import DesignAPI
from api_service.excp.design_excp import PTDsgGenConstException, PTDsgGenReportException, PTDsgImportException
from api_service.unified_netlist.peri_netlist_builder import PeriNetlistDBLoader

if TYPE_CHECKING:
    from design.db import DesignIssue


class PTFlowArgument(TypedDict):
    design: str
    family: str
    device: str
    timing_model: str

    project_xml: Path
    work_dir: Path
    output_dir: Path

    create: bool
    peri_netlist: NotRequired[os.PathLike]
    peri_scripts: NotRequired[List[os.PathLike]]


class PTFlowRunnerStatusCode(IntEnum):
    OK = 0
    ERROR = 1
    # Skip by env / sandbbox test that doesn't require interface
    # Remarks: set to 0 to avoid breaking the GUI flow
    WARNING_SKIP = 0


def check_bitstream_enabled(device: str, family: str) -> bool:
    executable = 'efx_pgm.exe' if (
        platform.system() == 'Windows') else 'efx_pgm'
    pgm_cmd = os.path.join(os.environ.get(
        'EFINITY_HOME'), 'bin', executable)

    cmd_args = [pgm_cmd, '--query_bitstreams',
                '--family=' + family, '--device=' + device]
    result = True
    try:
        subprocess.check_call(
            cmd_args, stdout=sys.stdout, stderr=sys.stderr)
    except subprocess.CalledProcessError:
        result = False
    return result


def sync_design_file_efxproject(design_file: os.PathLike, project_file: os.PathLike):
    # To update the design name to match with the name in efinity project
    # Need this because of tech debt where rename project doesn't update PT
    # See: DesignService.sync_name_design_efxproject
    proj_db = EfxProjectDatabase()
    proj_db.load(project_file)

    from lxml import etree
    tree = etree.parse(design_file)
    design_root: etree.Element = tree.getroot()
    design_name = design_root.attrib['name']

    # design_db = design_api._get_design_db()
    if proj_db.project_name != design_name:
        print(f"Design name changed from {design_name} to {proj_db.project_name}")
        design_root.attrib['name'] = proj_db.project_name

        proj_path = Path(project_file).parent
        if proj_path.exists() and proj_path.is_dir():
            old_design_file = Path(design_file)
            print(f"Old design file: {old_design_file}")

            new_design_file = proj_path / f"{proj_db.project_name}.peri.xml"
            print(f"New design file: {new_design_file}")

            with open(new_design_file, 'wb') as fp:
                fp.write(etree.tostring(tree, encoding='utf-8', pretty_print=True))

            if old_design_file.exists() and old_design_file.resolve() != new_design_file.resolve():
                print(f"Remove old design file: {old_design_file}")
                old_design_file.unlink()


def pretty_print_design_issues(issues: List[DesignIssue]):
    table = PrettyTable()
    table.field_names = ["Instance Name", "Instance Type", "Serverity",
                            "Rule", "Description"]
    for i in issues:
        table.add_row([i.instance_name,
                        i.instance_type,
                        str(i.severity.value),
                        i.rule_name,
                        i.msg])
    print(table)

class PTFlowRunner():

    def __init__(self, args: PTFlowArgument):
        self.args = args

    def __call__(self) -> PTFlowRunnerStatusCode:
        # Check ENV which skip the flow
        if os.getenv('EFX_SKIP_PT', '0') != '0':
            print("Skipping Interface Designer flow due to EFX_SKIP_PT environment variable")
            return PTFlowRunnerStatusCode.WARNING_SKIP

        # Check device and family
        family = self.args.get('family')
        if not family:
            print("No device family specified to Interface Designer")
            return PTFlowRunnerStatusCode.ERROR

        # Legacy behaviour
        if family.lower() not in ('trion', 'titanium', 'topaz'):
            print(f"Skipping non applicable {family}")
            return PTFlowRunnerStatusCode.WARNING_SKIP

        device = self.args.get('device')
        if not device:
            print("No device name specified to Interface Designer")
            return PTFlowRunnerStatusCode.ERROR

        # Legacy behaviour
        if device.startswith(('QX', 'Q')):
            print(f"Skipping internal test device: {device}")
            return PTFlowRunnerStatusCode.WARNING_SKIP

        # TODO: Should we expose the device is supported API in DesignAPI / DeviceAPI?
        # Or should we use the devicedb in efinity?
        dev_service = DeviceService()
        if not dev_service.is_device_exists(device):
            print(f"ERROR, unusupported device {device} in Interface Designer")
            return PTFlowRunnerStatusCode.ERROR

        timing_model = self.args.get('timing_model')
        if not timing_model:
            # TODO: Valid Timing Model checks
            pass

        # Main routine:
        proj_file = self.args.get('project_xml')
        proj_db: Optional[EfxProjectDatabase] = None
        if proj_file is not None:
            proj_db = EfxProjectDatabase()
            loaded = proj_db.load(proj_file)
            if not loaded:
                print(f"Fail to load Efinity project file {proj_file}")
                return PTFlowRunnerStatusCode.ERROR

        design_api = DesignAPI()

        # Load design or create an empty one
        design_file = Path(self.args.get('design') + '.peri.xml')
        if not design_file.exists():
            if self.args.get('create', False):
                print(f"Design file {design_file} doesn't exist, create a new one")
                design_api.create(self.args.get('design'),
                                  self.args.get('device'),
                                  str(Path(self.args.get('design_dir'))))
                design_api.save()
                print(f"Design file created {design_file}")
            else:
                print(f"Design file {design_file} doesn't exist")
                return PTFlowRunnerStatusCode.ERROR
        else:
            print(f"Load design file {design_file}")

            # Sync project name
            if proj_db is not None:
                sync_design_file_efxproject(design_file, proj_file)

            design_api.load(str(design_file))

        design_name = design_api._get_design_db().name
        output_dir = self.args.get('output_dir', design_file.parent / 'outflow')
        Path(output_dir).mkdir(parents=True, exist_ok=True)

        # Check device consistency if project file provided
        proj_file = self.args.get('project_xml')
        proj_db: Optional[EfxProjectDatabase] = None
        if proj_file is not None:
            proj_db = EfxProjectDatabase()
            loaded = proj_db.load(proj_file)
            if not loaded:
                print(f"Fail to load Efinity project file {proj_file}")
                return PTFlowRunnerStatusCode.ERROR
            else:
                if proj_db.device_family != family:
                    print("ERROR: Device family mismatch between Efinity project file and Periphery design file")
                    return PTFlowRunnerStatusCode.ERROR
                if proj_db.device_model != device:
                    print("ERROR: Device name mismatch between Efinity project file and Periphery design file")
                    return PTFlowRunnerStatusCode.ERROR
                if proj_db.device_timing_model != timing_model:
                    print("ERROR: Timing model mismatch between Efinity project file and Periphery design file")
                    return PTFlowRunnerStatusCode.ERROR

        # Unified Netlist / ISF Flow to update the design
        if self.args.get('un_flow') is True:
            peri_netlist = self.args.get('peri_netlist')
            peri_scripts = self.args.get('peri_scripts', [])
            if peri_netlist:
                # UN: Convert peri_netlist to ISF
                peri_netlist = Path(peri_netlist)
                if not peri_netlist.exists():
                    print(f"ERROR: Periphery netlist specificed but file {peri_netlist} not found")
                    return PTFlowRunnerStatusCode.ERROR

                peri_netlist_isf = Path(output_dir, f'{design_name}.unified.isf')
                design_api.convert_periphery_netlist_to_isf_file(peri_netlist, peri_netlist_isf)
                print(f"Periphery netlistdb converted to ISF: {peri_netlist_isf}")
                # Always insert to the start and execute script from periphery netlist first
                peri_scripts.insert(0, peri_netlist_isf)

            # Load ISF files
            load_success = True
            for script in peri_scripts:
                print(f"Loading {script}......")
                try:
                    load_success, import_issues = design_api.import_design(script)
                    if not load_success:
                        print(f"Failed to load {script} with the following message:")
                        pretty_print_design_issues(import_issues)
                        break
                except PTDsgImportException:
                    print(f"Failed to load {script} with the following exception:")
                    traceback.print_exc()
                    break
                print(f"Loaded {script}")

            if not load_success:
                return PTFlowRunnerStatusCode.ERROR

            # Write the merged peri.xml (For Debugging Only now)
            # merged_design_file = Path(output_dir, f"{design_name}.merged.peri.xml")
            # design_api.save_as(str(merged_design_file))
            # print(f"Intermediate peri xml saved to {merged_design_file}")

            # Auto resource assignment
            auto_assign_stmts = design_api.auto_assign_resource()
            if len(auto_assign_stmts) > 0:
                print("The design has some design instances with resource auto assigned")
                auto_assign_isf = Path(output_dir, f"{design_name}.auto_asg.isf")
                design_api.export_isf_commands_file(auto_assign_stmts, auto_assign_isf)
                print(f"Auto resource assignments ISF saved to {auto_assign_isf}")

            design_api.save()

        # Run design check
        is_design_pass = design_api.check_design()
        if not is_design_pass:
            print("Design check fails with the following message:")
            design_issues: List[DesignIssue] = design_api.get_design_check_issue(True)
            pretty_print_design_issues(design_issues)
            return PTFlowRunnerStatusCode.ERROR

        bitstream_status = check_bitstream_enabled(device, family)
        try:
            design_api.generate(bitstream_status, str(output_dir))

            # Unified Simulation: Generate wrapper sim netlist
            # TODO:  Make it part of generate() once the feature is finalized
            if self.args.get('un_flow') is True and design_api.is_periphery_sim_supported_for_all_blocks():
                print("Unified Simulation supported")
                top_module_name = design_name
                if proj_db:
                    top_module_name = proj_db.design_top_module_name

                if top_module_name == "":
                    top_module_name = design_name

                ndb = None
                if peri_netlist and peri_netlist.exists():
                    loader = PeriNetlistDBLoader()
                    ndb = loader.load_netlistdb(peri_netlist)
                    def get_top_name(ndb) -> str:
                        with ndb.get_session() as session:
                            top = ndb.get_top_module()
                            assert top is not None
                            return top.cell_name
                    top_module_name = get_top_name(ndb)

                rtl_sim = Path(output_dir, f'{design_name}.peri.rtl.v')
                pt_sim = Path(output_dir, f'{design_name}.peri.pt.v')
                design_api.export_periphery_sim_netlist(top_module_name, rtl_sim, pt_sim, ndb, 'verilog')
                print(f"Unified RTL sim verilog generated at {rtl_sim}")
                print(f"Unifed Full Chip sim verilog generate at {pt_sim}")

        except (PTDsgGenConstException, PTDsgGenReportException):
            traceback.print_exc()
            print("Fail to generate outputs")
            return PTFlowRunnerStatusCode.ERROR

        return PTFlowRunnerStatusCode.OK


def parse_cmdline_options(argv: List[str]) -> PTFlowArgument:
    parser = argparse.ArgumentParser(description='Efinity Interface Designer Flow Runner')
    # Legacy: All are positional arguments
    parser.add_argument('design', type=str, help='Design name')
    parser.add_argument('family', type=str, help='Device family name')
    parser.add_argument('device', type=str, help='Device model name')

    parser.add_argument('--timing_model', default=None, type=str, help='Timing model name')
    parser.add_argument('--project_xml', default=None, type=str, help='Project file')
    parser.add_argument('--output_dir', default='outflow', help='Output directory')
    parser.add_argument('--work_dir', default='work_pt', help='Work directory')
    parser.add_argument('--design_dir', default=str(os.getcwd()), help='Project / Design directory')
    parser.add_argument('--create', default=True, help='Create periphery design')

    parser.add_argument('--peri_netlist', default=None, help='Periphery netlist in netlistdb format')
    parser.add_argument('--peri_scripts', default=None, action='append', nargs='+',  help='Interface script files')
    parser.add_argument('--un_flow', action='store_true', help='Toggle the unified flow')

    args = parser.parse_args(argv)

    d = vars(args)

    if args.peri_scripts is None:
        d['peri_scripts'] = []
    else:
        # Parser return a nested list when multiple --peri_scripts given, flatten them
        d['peri_scripts'] = list(chain.from_iterable(args.peri_scripts))
    return d


if __name__ == "__main__":
    try:
        args = parse_cmdline_options(sys.argv[1:])
        runner = PTFlowRunner(args)
        ret = runner()
    except Exception:
        ret = PTFlowRunnerStatusCode.ERROR
        traceback.print_exc()
    sys.exit(ret)
