'''
Copyright (C) 2017-2018 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Nov 21, 2017

@author: maryam
'''
import os
from pathlib import Path
import re
import csv
import sys
import pickle
import hashlib
import queue
from concurrent.futures import ThreadPoolExecutor, as_completed, wait
from typing import List, Optional, Tuple

import xml.etree.ElementInclude as ei
import xml.etree.ElementTree as et
import xmlschema

from util.singleton_logger import Logger
import util.app_setting as aps

import device.db as db
import device.service_interface as dbi
import device.parser.timing_model_service as tms
import device.parser.package_service as pkgs
import device.parser.die_service as dies

import device.resources as resources
import device.config_model as cmodel
import device.device_config as blkcfg
import device.res_config as rcfg
from device.pcr_device import PCRMap
import gzip


# Flag to use to print out the clock region to resource map
PRINT_CLK_REGION_TO_RESOURCE_MAP = False


class DeviceMap:
    def __init__(self):
        self._dev2file = {}  #: A map of device name to its devicefile

    @staticmethod
    def build(device_map_file):

        dev_map = DeviceMap()
        if not os.path.exists(device_map_file):
            return dev_map

        with open(device_map_file, encoding='UTF-8') as csvfile:
            reader = csv.reader(csvfile)
            for row in reader:
                # Strip out any training and leading space for each entry
                device_name = row[0]
                device_filename = row[2]

                # build() is meant to be friend function to the class
                # but friend is not supported in python
                # pylint: disable=protected-access
                dev_map._dev2file[device_name] = device_filename

        return dev_map

    @staticmethod
    def build_from_default_file():
        efxpt_home = os.environ.get('EFXPT_HOME', "")
        devicemap_file = os.path.normpath(efxpt_home + "/db/devicemap.csv")
        return DeviceMap.build(devicemap_file)

    def get_device_count(self):
        """
        Get total device count in map

        :return: Count
        """
        return len(self._dev2file)

    def get_device_filename(self, device_name):
        """
        Given a device name, it returns a list of applicable plugin name

        :param device_name: Device name
        :return: Device filename. Empty string if not found
        """
        dev_filename = self._dev2file.get(device_name, "")
        return dev_filename

    def is_device_exist(self, device_name):
        """
        Check if device exists

        :param device_name: Device name
        :return: True, if exist else False
        """
        return device_name in self._dev2file

    def is_device_virtual(self, dev_name1, dev_name2):
        """
        Check if the passed devices are virtual devices in which
        they both are of different name and mapped to the same device file
        :param dev_name1: Device name 1
        :param dev_name2: Device name 2
        :return True if they are virtual
                False if they are not virtual or the same name
        """

        if dev_name1 != dev_name2:
            dev_file1 = self.get_device_filename(dev_name1)
            dev_file2 = self.get_device_filename(dev_name2)

            if dev_file1 == dev_file2:
                return True

        return False

    def get_device_names(self):
        return list(self._dev2file.keys())


class DevicePickled:
    def __init__(self, device_name):
        self.__device_name = device_name
        self.__binary_file = str(self.__device_name) + '.db'
        self.__checksum = None  # validate if the db file are different
        self.__mode = "body"

        efxpt_home = os.environ["EFXPT_HOME"]
        self.__resource_path = efxpt_home + '/db/resource'

        # Set path for reading resource and checksum
        self.obj_path = self.__resource_path + '/' + self.__binary_file
        self.device_info_file = "devices_info.py"
        self.devices_path = efxpt_home + '/db/' + self.device_info_file

        self.logger = Logger

    def pickle_file(self, device_db:db.PeripheryDevice):
        try:
            if self.mode in ("head", "end"):
                return

            if not os.path.exists(self.obj_path):
                os.makedirs(self.__resource_path, exist_ok=True)
            with gzip.open(self.obj_path,'wb') as b_file:
                pickle.dump(device_db, b_file)
                b_file.close()
        except OSError as excp:
            self.logger.error("Error pickle file {}".format(excp))

        except Exception as excp:
            import traceback
            traceback.print_stack()
            self.logger.error("Exception: {}".format(excp))

    def hashing_file(self):
        '''
        Hash the file by using SHA256 and save the result to checksum
        '''
        try:
            if self.mode in ("head", "end"):
                return

            with open(self.obj_path,'rb') as f:
                sha256_hash = hashlib.sha256()
                for byte_block in iter(lambda: f.read(4096),b""):
                    sha256_hash.update(byte_block)

                checksum = sha256_hash.hexdigest()
                self.checksum = checksum

                f.close()
        except OSError as excp:
            self.logger.error("Error hashing file {}".format(excp))

        except Exception as excp:
            import traceback
            traceback.print_stack()
            self.logger.error("Exception: {}".format(excp))

    def get_device_db(self):
        '''
        Getting device_db by loading db file from resource

        :return pickle.load(f) device_db (PeripheryDevice)
        '''
        try:
            with gzip.open(self.obj_path,'rb') as f:
                return pickle.load(f)

        except OSError as excp:
            self.logger.error("Error get pickled device {}".format(excp))

        except Exception as excp:
            import traceback
            traceback.print_stack()
            self.logger.error("Exception: {}".format(excp))

    @property
    def device_name(self):
        return self.__device_name

    @property
    def checksum(self):
        return self.__checksum

    @checksum.setter
    def checksum(self, checksum):
        self.__checksum = checksum

    @property
    def binary_file(self):
        return self.__binary_file

    @binary_file.setter
    def binary_file(self, bin_file):
        self.__binary_file = bin_file

    @property
    def resource_path(self):
        return self.__resource_path

    @resource_path.setter
    def resource_path(self, pwd):
        self.__resource_path = pwd
        self.obj_path = self.__resource_path + '/' + self.__binary_file

    @property
    def mode(self):
        return self.__mode

    @mode.setter
    def mode(self, value):
        self.__mode = value

    def set_devices_info_path(self, pwd):
        self.devices_path = pwd + f'/{self.device_info_file}'

    def save_device_db(self, pwd=None):
        '''
        Save the information of device db for normal user flow
        '''
        device_inf = {"checksum":self.__checksum, "binary_file":self.__binary_file}
        info = "\"{}\":{}".format(self.__device_name,device_inf)

        try:
            if self.devices_path is None and pwd is None:
                return self.logger.error("Please set the devices_path")

            if pwd is not None:
                self.set_devices_info_path(pwd)

            if self.mode == "head":
                with open(self.devices_path,'w+') as f:
                    data =  "devices = { \n"
                    f.write(data)
                    f.close()

            elif self.mode == "body":
                is_removed = self.remove_device_info(self.device_name)

                if is_removed:
                    with open(self.devices_path,'r',encoding='utf-8') as f:
                        devices = f.readlines()

                    devices.insert(-2, info + ",\n")
                    with open(self.devices_path,'w',encoding='utf-8') as f:
                        f.writelines(devices)
                        f.close()
                else:
                    with open(self.devices_path,'a',encoding='utf-8') as f:
                        f.write(info + ",\n")

            elif self.mode == "end":
                with open(self.devices_path,'r',encoding='utf-8') as f:
                    data = f.readlines()
                    f.close()

                    # Remove the last ",\n"
                    data[-1] = data[-1][:-2]
                    data.append("\n}\n")

                data = "".join(data)

                with open(self.devices_path,'w',encoding='utf-8') as f:
                    f.write(data)
                    f.close()

        except OSError as excp:
            self.logger.error("Error save pickled device {}".format(excp))

        except Exception as excp:
            self.logger.error("Exception: {}".format(excp))

    @staticmethod
    def gen_all_device_info(devicemap_file: str, output_path: Optional[str] = None):
        """
        Enable to generate all pickled data information (devices_info.py)

        :param devicemap_file (str): path to devicemap.csv file
        """
        if output_path is None:
            output_path = Path(os.environ["EFXPT_HOME"]) /'db'
        else:
            output_path = Path(output_path)

        output_file = output_path / 'devices_info.py'

        with open(devicemap_file, encoding='UTF-8', mode='r') as csvfile:
            reader = csv.reader(csvfile, delimiter=',')
            device_records: List[Tuple[str, str, str]] = []
            for row in reader:
                if len(row) == 3:
                    device_records.append(tuple(row))

            devinfos = queue.Queue()

            def load_and_save_checksum(device_name: str) :
                dev_pickle = DevicePickled(device_name)
                dev_pickle.hashing_file()
                devinfos.put((device_name, dev_pickle.checksum, dev_pickle.binary_file))

            with ThreadPoolExecutor() as executor:
                futures = {executor.submit(load_and_save_checksum, device_name): device_name for device_name, _, _ in device_records}
                wait(futures)

            with open(output_file, encoding='UTF-8', mode='w') as file:
                file.write('devices = { \n')
                while not devinfos.empty():
                    item = devinfos.get()
                    device_name, checksum, binary_file = item
                    inf = {"checksum":checksum, "binary_file": binary_file}
                    info = "\"{}\":{},\n".format(device_name, inf)
                    file.write(info)
                file.write('}\n')

    def remove_device_info(self, device_name):
        '''
        Removing the device if the device information is in the devices_info.py
        '''
        device_checker = "\"{}\":".format(device_name)
        devices = ""
        with open(self.devices_path,'r',encoding='utf-8') as f:
            devices = f.readlines()

        is_removed = False
        for i in range(len(devices)):
            if device_checker in devices[i]:
                devices[i] = ""
                is_removed = True
                break

        if is_removed:
            with open(self.devices_path,'w',encoding='utf-8') as f:
                f.writelines(devices)
                f.close()

        return is_removed

    def __str__(self):
        return f"device_name: {self.__device_name}, binary_file: {self.binary_file}, " \
                f"checksum: {self.__checksum}"


class DeviceService:
    '''
    Provide device related operations
    '''

    # Map of parsed string to its enum types
    def __init__(self):
        '''
        constructor
        '''
        self.logger = Logger
        self.pwd = ""
        self.__is_validate = True # For checking device
        self.__is_save = False
        self.saved_device = None
        self.efx_home = os.environ['EFXPT_HOME']

    def get_device_map_file(self, map_file=""):
        '''
        Get the map file. If map_file is empty then we get it
        from the pre-defined file.

        :param map_file: Path to map file
        :return full path of the map file. Empty string if it
            cannot find the file.
        '''
        device_csv = ""

        if map_file != "":
            if os.path.exists(map_file):
                device_csv = map_file
            else:
                self.logger.debug(
                    "Unable to find map file {}".format(
                        map_file))
        else:
            efxpt_home = os.environ["EFXPT_HOME"]
            tmp_csv = efxpt_home + "/db/devicemap.csv"
            if os.path.exists(tmp_csv):
                device_csv = tmp_csv

        return device_csv

    def is_device_exists(self, name, map_file=""):
        '''
        Check if the passed device marketing name is defined.

        :param name: device alias/marketing name
        :return True if it is found, False otherwise.
        '''
        is_found = False

        device_csv = self.get_device_map_file(map_file)
        if device_csv != "":

            pickled_db, filename, _ = self.get_file_and_mapped_name(
                name, device_csv)

            if pickled_db != "" or filename != "":
                is_found = True

        return is_found

    def get_file_and_mapped_name(self, search_name, device_csv):
        '''
        Get the device file path and the mapped name based
        on parsing the device map file.
        :param search_name: device alias/marketing name
        :param device_csv: Full path to the device map file to
                parse.
        :param parent_dir: parent directory of the device_csv file

        :return the tuple of device filename, device mapped name
        '''

        mapped_dev_name = ""
        filename = ""
        pickled_db= ""
        found = False

        try:
            with open(device_csv, encoding='UTF-8') as csvfile:
                read_csv = csv.reader(csvfile, delimiter=',')
                for row in read_csv:
                    if len(row) == 3:
                        marketing_name = row[0]
                        device_name = row[1]
                        device_filename = row[2]

                        if search_name == marketing_name:
                            mapped_dev_name = device_name
                            filename = device_filename
                            found = True
                            break

        # Return if error reading the file
        except Exception as excp:
            self.logger.error("Error parsing {}: {}".format(
                device_csv, excp))
            return "", ""

        parent_dir = os.path.dirname(os.path.abspath(device_csv))

        if found:
            pickled_db = os.path.join(parent_dir, "resource", f"{search_name}.db")
            filename = parent_dir + "/" + filename

        return pickled_db, filename, mapped_dev_name

    def parse_device_map_file(self, search_name, map_file=""):
        '''
        Get the device filename that corresponds to the
        passed device name.  This is done by reading the
        devicemap.csv file.

        :param search_name: device alias/marketing name
        :param map_file: If there's a specific device map file
                that we want to use instead of in db/devicemap.csv
        :return the tuple of device filename, device mapped name
        '''

        device_csv = self.get_device_map_file(map_file)
        if device_csv == "":
            return "", "", ""

        pickled_db, filename, mapped_dev_name = self.get_file_and_mapped_name(
            search_name, device_csv)

        return pickled_db, filename, mapped_dev_name

    def get_project_mode_width(self, prog_info):
        '''
        Get the config mode setting from the efinity project
        file.

        :return mode_settting in string and width_setting in string
        '''
        mode_setting = ""
        width_setting = 0

        if prog_info is not None:
            mode_setting = prog_info.get("mode", "")
            width_str = prog_info.get("width", "")
            if width_str != "":
                width_setting = int(width_str)

        return mode_setting, width_setting

    def load_device(self, name, prog_info=None, tmodel_name="", file="", map_file=""):
        '''
        Creates the device DB from the device file of a given device

        :param file: Full path to the device filename
        :param name: device name

        :returns A device. None if an error occurred.
        '''

        dev_filename = ""
        mapped_name = name
        pickled_db = ""

        # Priority is given to filename specification. Then
        # Only we parse the map file if it was provided.
        if file == "":
            pickled_db, dev_filename, mapped_name = self.parse_device_map_file(
                name, map_file)
            if dev_filename == "":
                self.logger.error('Device {} not defined'.format(name))
                return None
        else:
            dev_filename = file

        self.logger.debug("Loading device name: {} file: {}".format(
            name, dev_filename))

        # CHeck that the file exists
        if not os.path.exists(dev_filename) and not os.path.exists(pickled_db):
            self.logger.error(
                'Unable to find device file: {}'.format(
                    dev_filename))
            return None

        self.pwd = os.getcwd()
        device_db_dir = os.path.dirname(dev_filename)

        os.chdir(device_db_dir)

        builder = DeviceBuilderXmlEventBased(mapped_name, name)

        if self.is_check and not builder.validate(dev_filename):
            self.logger.error('Device file {} contains invalid syntax'.format(
                dev_filename))

            os.chdir(self.pwd)
            return None

        self.logger.debug("Load Device file: {}".format(dev_filename))
        self.logger.debug("Load Device dir: {}".format(device_db_dir))

        if self.is_device_pickled(name):
            device = self.load_device_from_db(name)
        else:
            self.is_check = True
            device = builder.build(dev_filename)

        # Check that the passed name matched with the device name
        if device is None:
            self.logger.error('Error while parsing device {}'.format(name))
            os.chdir(self.pwd)
            return None

        else:
            if self.is_check:
                # Save the device marketing name before we identify block type
                device.set_device_name(name)

            # Build the necessary block services
            device.identify_supported_block_types()

            if not self.is_check:
                self.reset_file_path(device)

            # Checks that the parsed device is valid
            if not self.is_check or builder.check():
                if self.__is_save:
                    # Save the device for pickle, only used in make check
                    self.saved_device = device

                # Get the mode and width configuration from
                # project programming info
                mode, width = self.get_project_mode_width(prog_info)

                if not device.assign_timing_model(tmodel_name):
                    device = None

                elif device.get_config_model() is not None and \
                        not device.assign_config_model(mode, width):
                    device = None

                else:
                    if not self.is_device_pickled(name):
                        # Create the secondary function map
                        device.create_secondary_conn_functions()

                        # Verify block service equivalent only when
                        # it is parsing the device file and not duirng
                        # pickle load
                        if not device.verify_common_device_block_equivalence():
                            device = None

                    if device is not None:
                        if PRINT_CLK_REGION_TO_RESOURCE_MAP:
                            device.print_clock_region_instance_map()

            else:
                self.logger.error("Device {} is invalid".format(
                    name))
                device = None

        os.chdir(self.pwd)

        return device

    def reset_file_path(self, device):
        efx_home = os.environ['EFXPT_HOME']
        self.logger.debug("Resetting path, efx_home = {}".format(efx_home))

        self.reset_periphery_definition(device)

        self.reset_timimg_model(device)

    def reset_periphery_definition(self, device):
        efx_home = self.efx_home
        periphery_definition = device.get_periphery_definition()

        for peri_block in periphery_definition.values():

            # BlockPCR
            pcr_block = peri_block.get_block_pcr()
            if pcr_block is not None:
                # BlockPCRLookup
                pcr_lookup = pcr_block.get_pcr_lookup()
                if pcr_lookup is not None:
                    lookup_path = pcr_lookup.get_permutation_file()
                    abs_path = os.path.join(efx_home, lookup_path)
                    pcr_lookup.set_permutation_file(abs_path)
                    pcr_lookup.set_db_connection()

                    setting_files = {}
                    id2file_map = {}

                    for k, settings in pcr_lookup.get_all_settings_file().items():
                        settings = os.path.join(efx_home, settings)
                        setting_files[k] = settings
                    pcr_lookup.set_all_settings_file(setting_files)

                    for k, file in pcr_lookup.get_all_id2file_map().items():
                        file = os.path.join(efx_home, file)
                        id2file_map[k] = file
                    pcr_lookup.set_all_id2file_map(id2file_map)

                # PCRMap
                for pcr_map in pcr_block.get_pcr_list().values():
                    filepath = pcr_map.get_filename()
                    if filepath is not None:
                        abs_path = os.path.join(efx_home, filepath)
                        pcr_map.set_filename(abs_path)

            # BlockMuxPattern
            for mux_pattern in peri_block.get_all_mux_pattern().values():
                pattern_path = mux_pattern.get_mux_pattern_file()
                abs_path = os.path.join(efx_home, pattern_path)
                mux_pattern.set_mux_pattern_file(abs_path)

            # BlockInternalConnectivity
            internal_conn = peri_block.get_internal_conn()
            if internal_conn is not None:
                conn_path = internal_conn.get_conn_filename()
                abs_path = os.path.join(efx_home, conn_path)
                internal_conn.set_conn_filename(abs_path)

            # Config file Mapping
            config_file_map = {}
            for k, config_file in peri_block.get_config_file_map().items():
                config_file = os.path.join(efx_home, config_file)
                config_file_map[k] = config_file
            peri_block.set_config_file_map(config_file_map)

    def reset_timimg_model(self, device):
        efx_home = self.efx_home
        timimg_model = device.get_timing_model()

        if timimg_model is not None:
            time_combinations = timimg_model.get_all_timing_model()
            for time_combine in time_combinations.values():
                # TimingCombination.Model
                for model in time_combine.get_models():
                    model_path = model.get_filename()
                    abs_path = os.path.join(efx_home, model_path)
                    model.set_filename(abs_path)

    def load_device_from_db(self, name):
        import device.devices_info as dev_info
        devices = dev_info.devices

        dev_pickle = DevicePickled(name)
        dev_pickle.hashing_file()

        if devices[name]['checksum']==dev_pickle.checksum:
            return dev_pickle.get_device_db()
        else:
            self.logger.error(f'Device {name} is changed (Different checksum)')
            return None

    def save_device(self, name:str, device_db:db.PeripheryDevice, device_path: Optional[str]=None, resource_path: Optional[str] = None, mode: Optional[str] = "body"):
        pickle_device = DevicePickled(name)

        if resource_path is not None:
            pickle_device.resource_path = resource_path

        pickle_device.mode = mode

        if mode in ('body', 'all'):
            pickle_device.pickle_file(device_db)
            pickle_device.hashing_file()

        if device_path is not None:
            pickle_device.save_device_db(pwd=device_path)

    def check_device(self, name, file="", map_file=""):
        '''
        Validate the device prior to parsing it by
        validating the xml file against the schema.

        :param file: Full path to the device filename

        :returns True if good, else False
        '''
        dev_filename = ""

        if file == "":
            _, dev_filename, _ = self.parse_device_map_file(name, map_file)
            if dev_filename == "":
                self.logger.error("Device {} not defined".format(name))
                return False
        else:
            dev_filename = file

        # CHeck that the file exists
        if not os.path.exists(dev_filename):
            self.logger.error(
                'Unable to find device file: {}'.format(
                    dev_filename))
            return False

        self.pwd = os.getcwd()
        device_db_dir = os.path.dirname(dev_filename)

        os.chdir(device_db_dir)

        self.logger.debug("Device file: {}".format(dev_filename))
        self.logger.debug("Device dir: {}".format(device_db_dir))

        # We don't care about the mapped name for validation
        builder = DeviceBuilderXmlEventBased("")
        result_validate = builder.validate(dev_filename)
        os.chdir(self.pwd)

        return result_validate

    @property
    def is_check(self):
        return self.__is_validate

    @is_check.setter
    def is_check(self, is_check: bool):
        self.__is_validate = is_check

    @property
    def is_save(self):
        return self.__is_save

    @is_save.setter
    def is_save(self, is_save: bool):
        self.__is_save = is_save

    def is_device_pickled(self, name):
        cur_path = os.path.dirname( os.path.abspath(__file__) )
        if not os.path.exists(cur_path + "/devices_info.py") or self.is_check:
            return False

        import device.devices_info as dev_info
        if name in dev_info.devices:
            return True
        else:
            return False


class DeviceVerticalMigration:
    '''
    Identify whether a device migration is a Vertical
    Migration type.
    '''

    def __init__(self):
        '''
        Constructor
        '''
        #self._old_device_name = old_device
        #self._new_device_name = new_device

        # A map of the device name mapped to a list of
        # other device that is considered as vertical migration
        self._dev2verticalmig = {}

        self.logger = Logger

    @staticmethod
    def build_mockup():
        vmap = DeviceVerticalMigration()

        an120_row = ["T35F324", "T20F324", "T20F324VC"]
        an35_row = ["T120F324", "T85F324", "T55F324", "T70F324VC"]

        # pylint: disable=protected-access
        vmap._dev2verticalmig["T120F324"] = an120_row
        vmap._dev2verticalmig["T85F324"] = an120_row
        vmap._dev2verticalmig["T55F324"] = an120_row
        vmap._dev2verticalmig["T70F324VC"] = an120_row

        vmap._dev2verticalmig["T35F324"] = an35_row
        vmap._dev2verticalmig["T20F324"] = an35_row
        vmap._dev2verticalmig["T20F324VC"] = an35_row

        return vmap

    @staticmethod
    def build(migration_file):
        '''
        Parse the migration file and returns the object
        that contains the vertical migration map
        '''
        if migration_file != "":
            if not os.path.exists(migration_file):
                return None
        else:
            return None

        dev_vm = DeviceVerticalMigration()

        with open(migration_file, encoding='UTF-8') as csvfile:
            reader = csv.reader(csvfile)
            for row in reader:
                if row:
                    old_device_name = row[0]

                    new_device_list = []
                    is_found_device = False
                    for col in row:

                        # Skip the first entry (old device)
                        if is_found_device:
                            new_device_list.append(col)

                        if is_found_device is False:
                            is_found_device = True

                    # build() is meant to be friend function to the class
                    # but friend is not supported in python
                    # pylint: disable=protected-access
                    dev_vm._dev2verticalmig[old_device_name] = new_device_list

        return dev_vm

    @staticmethod
    def build_from_default_file():
        '''
        Use the db migration file to determine the
        vertical migration map
        '''

        efxpt_home = os.environ["EFXPT_HOME"]
        migration_file = efxpt_home + "/db/migrationmap.csv"
        return DeviceVerticalMigration.build(migration_file)

    @staticmethod
    def get_db_migration_file():
        efxpt_home = os.environ["EFXPT_HOME"]
        migration_file = efxpt_home + "/db/migrationmap.csv"
        return migration_file

    def is_device_exist(self, device_name):
        """
        Check if device exists

        :param device_name: Device name
        :return: True, if exist else False
        """
        return device_name in self._dev2verticalmig

    def get_migrated_device_list(self, device_name):
        """
        Given a device name, it returns a list of devices
        that if migrated to it is considered a vertical
        migration

        :param device_name: Device name
        :return: vertical migration device name list
        """
        migrated_list = self._dev2verticalmig.get(device_name, [])
        return migrated_list

    def is_device_vertical_migration(self, old_device, new_device):
        '''
        Check if it is considered as vertical migration if
        changing from the old_device to new_device.  This is
        when the key is old_device and if new_device appears
        in the list of values associated to the key.

        :param old_device: The source device name
        :param new_device: The device that is going to be migrated to
        :return True if it is a Virtual Migration (exists in the map).
                Otherwise, False.
        '''
        if old_device in self._dev2verticalmig:
            dev_list = self._dev2verticalmig[old_device]

            if new_device in dev_list:
                return True

        return False

    def get_device_names(self):
        """
        :return the list of devices (keys)
        """
        return list(self._dev2verticalmig.keys())


class DeviceBuilderXmlEventBased(dbi.DeviceBuilder):
    '''
    Build device from xml device file using event based parsing
    '''

    _ns = "{http://www.efinixinc.com/peri_device_db}"
    '''
    XML Namespace for peri design db
    '''

    _util_ns = "{http://www.efinixinc.com/peri_tool}"
    '''
    XML Namespace for peri tool util
    '''

    _schema_file = ""
    '''
    If device schema is empty, it will be detected automatically using
     $EFXPT_HOME.
    This is mainly for unit testing to set schema from different location.
    '''

    def __init__(self, mapped_name, marketing_name=""):
        '''
        Constructor
        '''

        super().__init__()
        self.device = None
        self.package = None
        self.device_dir = ""
        self.device_file = ""
        self.mapped_name = mapped_name
        self.logger = Logger
        self.marketing_name = marketing_name

    def build(self, file):
        '''
        Build device from an XML file

        :param device_file: Full path to the device filename

        :return the device_db
        '''

        device_file = file

        # Get the directory of the device file
        self.device_dir = os.path.dirname(os.path.abspath(device_file))

        # get an iterable
        context = et.iterparse(device_file, events=(
            "start", "end", "start-ns", "end-ns"))

        # turn it into an iterator
        context = iter(context)

        # Top-level tag
        device_tag = DeviceBuilderXmlEventBased._ns + "device_db"

        # Resource related tag
        resource_map_tag = DeviceBuilderXmlEventBased._ns + "resource_map"
        resource_tag = DeviceBuilderXmlEventBased._ns + "resource"

        # Configuration mode related tag
        config_modes_tag = DeviceBuilderXmlEventBased._ns + "configuration_modes"
        config_tag = DeviceBuilderXmlEventBased._ns + "config_type"

        # Block specific configurations
        block_config_tag = DeviceBuilderXmlEventBased._ns + "block_configurations"
        block_tag = DeviceBuilderXmlEventBased._ns + "block"
        block_property_tag = DeviceBuilderXmlEventBased._ns + "property"

        # Internal resource specific configuration
        internal_config_tag = DeviceBuilderXmlEventBased._ns + \
            "internal_resource_configurations"
        res_config_tag = DeviceBuilderXmlEventBased._ns + "resource_config"

        # Pin Swapping
        ddr_pin_swap_tag = DeviceBuilderXmlEventBased._ns + "ddr_pin_swap"
        pin_swap_tag = DeviceBuilderXmlEventBased._ns + "pin_swap"

        include_tag = "{http://www.w3.org/2001/XInclude}include"
        incl_list = []

        error = 0
        resource = None
        config_modes = None
        block_config = None
        res_config = None
        pin_swap_mapping = None

        # Build top level design
        for event, elem in context:
            if event == "start":
                if elem.tag == device_tag:

                    if self.device is None:
                        self._build_device(
                            self.mapped_name,
                            os.path.abspath(device_file))
                    else:
                        self.logger.error(
                            "Found multiple device_db")
                        error += 1

                elif elem.tag == resource_map_tag:
                    resource = self._build_resource_map()

                # Resource related map.
                # We parse the instance supported before
                # parsing the list of instances defined
                elif elem.tag == resource_tag:
                    self._build_resource(resource, elem)

                # Configuration related. In this case,
                # we parse the list of device supported config mode
                # before we read the config model
                elif elem.tag == config_modes_tag:
                    config_modes = self._build_config_modes()

                elif elem.tag == config_tag:
                    if not self._build_config(config_modes, elem):
                        error += 1

                elif elem.tag == block_config_tag:
                    dev_block_config = self._build_device_block_config()

                elif elem.tag == block_tag:
                    block_config = self._build_block_config(
                        dev_block_config, elem)
                    if block_config is None:
                        error += 1

                elif elem.tag == block_property_tag:
                    if not self._build_block_property(block_config, elem):
                        error += 1

                elif elem.tag == internal_config_tag:
                    res_config = self._build_internal_res_cfg()
                    if res_config is None:
                        error += 1

                elif elem.tag == res_config_tag:
                    if not self._build_res_config(res_config, elem):
                        error += 1

                elif elem.tag == ddr_pin_swap_tag:
                    pin_swap_mapping = {}

                elif elem.tag == pin_swap_tag and pin_swap_mapping is not None:
                    swap_info = elem.attrib
                    fpga = swap_info['fpga']
                    dram = swap_info['dram']
                    pin_swap_mapping[fpga] = dram

                elif elem.tag == include_tag:
                    incl_attr = elem.attrib
                    incl_file = incl_attr.get("href", None)
                    if incl_file is not None:
                        incl_list.append(incl_file)

                # Free up process element
                elem.clear()

            # When we reach the end, reset the handle
            elif event == "end":
                if elem.tag == block_tag:
                    block_config = None

        if self.device is not None:
            self.device.set_pin_swap(pin_swap_mapping)
            # continue parsing the included files

            self.logger.debug(
                "Device name : {}".format(self.device.get_device_name()))

            # Build package
            error += self._build_package_file(incl_list)

            # Build die
            error += self._build_die_file(incl_list)

            # Build timing model
            error += self._build_timing_model_file(incl_list)

        if error > 0:
            del self.device
            self.device = None

        return self.device

    def _build_package_file(self, incl_list):
        '''
        Build the device using the information in the
        package device file.

        :param incl_list: The list of included files in
                the top-level device file
        :return error count
        '''
        package_service = pkgs.PackageService(self.device)
        error = 0

        # This is for the case where the design file does not have include
        if not incl_list:
            tmp_error, _ = package_service.load_package(
                self.device.get_device_file())
            error += tmp_error

        else:
            for incl_file in incl_list:
                # Construct the full path to the die file
                abs_incl_file = self.device_dir + "/" + incl_file

                # This is rather loose. If the filename
                match = re.search(r'^package\/\w+\.xml', incl_file)
                if match:
                    #print("Reading pkg include file: {}".format(abs_incl_file))

                    tmp_error, _ = package_service.load_package(
                        abs_incl_file)
                    error += tmp_error

        return error

    def _build_die_file(self, incl_list):
        '''
        Build the device using the information in the
        die device file.

        :param incl_list: The list of included files in
                the top-level device file
        :return error count
        '''
        die_service = dies.DieService(self.device)
        error = 0

        # This is for the case where the design file does not have include
        if not incl_list:
            tmp_error, _ = die_service.load_die(
                self.device.get_device_file(), self.marketing_name)
            error += tmp_error

        else:
            for incl_file in incl_list:
                # Construct the full path to the die file
                abs_incl_file = self.device_dir + "/" + incl_file

                # This is rather loose. If the filename
                match = re.search(r'^die\/\w+\.xml', incl_file)
                if match:

                    #print("Reading die include file: {}".format(abs_incl_file))

                    tmp_error, _ = die_service.load_die(abs_incl_file, self.marketing_name)

                    error += tmp_error

        return error

    def _build_timing_model_file(self, incl_list):
        '''
         Build the device using the information in the
        timing_model device file.

        :param incl_list: The list of included files in
                the top-level device file
        :return error count
       '''
        tmodel_service = tms.TimingModelService(self.device)
        error = 0

        # This is for the case where the design file does not have include
        if not incl_list:
            tmp_error, _ = tmodel_service.load_timing_model(
                self.device.get_device_file())
            error += tmp_error

        else:
            for incl_file in incl_list:
                # Construct the full path to the die file
                abs_incl_file = self.device_dir + "/" + incl_file

                match = re.search(r'^timing_models\/\w+\.xml', incl_file)
                if match:

                    #print("Reading timing include file: {}".format(abs_incl_file))

                    tmp_error, _ = tmodel_service.load_timing_model(
                        abs_incl_file)
                    error += tmp_error

        return error

    def _build_device(self, device_mapped_name,
                      device_file):
        '''
        Reads the device_db tag and create a device db

        :param device_mapped_name: The device mapped name from
                mapping file
        :param device_file: Fullpath of the device filename

        '''

        # Create the device db
        self.device = db.PeripheryDevice(
            device_mapped_name)
        # Save the filename
        self.device.set_device_file(device_file)

    def _build_resource_map(self):
        '''
        Builds the resource map

        :param xml_elem: resource_map tag

        :return UseResources object that has been added to device
        '''
        resource = resources.UseResources()
        self.device.set_resources(resource)

        return resource

    def _build_resource(self, resource, xml_elem):
        '''
        Add the passed resource info the resource map

        :param resource: UseResources object
        :param xml_elem: resource tag
        '''
        resource_attr = xml_elem.attrib

        if resource is not None:
            ins_name = resource_attr.get("instance", "")
            sub_ins_name = resource_attr.get("sub_ins", "")
            resource.add_instance(ins_name, sub_ins_name)

    def _build_config_modes(self):
        '''
        '''
        config_modes = self.device.get_config_model()

        if config_modes is None:
            config_modes = cmodel.ConfigModel()
            self.device.set_config_model(config_modes)

        return config_modes

    def _build_config(self, config_modes, xml_elem):
        '''
        Add the supported config mode to the
        config model
        '''

        config_attr = xml_elem.attrib

        default_mode = False
        is_valid = False

        # Return if top-level object hasn't been created
        if config_modes is None:
            return is_valid

        if "default" in config_attr:
            default_attr = config_attr["default"]
            if default_attr == "true":
                default_mode = True

        width = 0
        if "width" in config_attr:
            width = int(config_attr["width"])

        cmode = config_modes.add_supported_mode(
            config_attr["mode"], width)

        if cmode is not None:
            is_valid = True

        if default_mode and cmode is not None:
            if config_modes.get_default() is None:
                if not config_modes.set_default(cmode):
                    is_valid = False
            else:
                # Error since there has already been a default
                self.logger.error('Found more than one mode set'
                                  ' as default mode')
                is_valid = False

        return is_valid

    def _build_device_block_config(self):
        dev_blk_config = self.device.get_block_configuration()

        if dev_blk_config is None:
            dev_blk_config = blkcfg.DeviceBlockConfig()

            self.device.set_block_configuration(dev_blk_config)

        return dev_blk_config

    def _build_block_config(self, dev_block_config, xml_elem):

        config_attr = xml_elem.attrib

        bname = config_attr["name"]

        blk_config = dev_block_config.get_block_property(bname)

        if blk_config is None:
            blk_config = blkcfg.BlockProperty(bname)

            if not dev_block_config.set_block_property(bname, blk_config):
                self.logger.error(
                    "Error loading block config for {}".format(bname))
                return None

        return blk_config

    def _build_block_property(self, block_config, xml_elem):
        prop_attr = xml_elem.attrib

        is_valid = False

        prop_name = prop_attr["name"]
        prop_value = prop_attr["value"]

        if block_config is not None:

            # Check if the property name is expected
            if prop_name not in blkcfg.DeviceBlockConfig.str2bprop_map:
                self.logger.error("Unexpected {} property name {}".format(
                    block_config.get_block_name(), prop_name))

            else:
                is_added = block_config.add_property(prop_name, prop_value)
                if not is_added:
                    self.logger.error("Error adding {} property name: {} value: {}".format(
                        block_config.get_block_name(), prop_name, prop_value))
                else:
                    is_valid = True

        else:
            self.logger.error("Error adding property name: {} value: {}".format(
                prop_name, prop_value))

        return is_valid

    def _build_internal_res_cfg(self):
        dev_res_config = self.device.get_internal_res_config()

        if dev_res_config is None:
            dev_res_config = rcfg.InternalResConfig()

            self.device.set_internal_res_config(dev_res_config)

        return dev_res_config

    def _build_res_config(self, res_cfg, xml_elem):
        is_valid = False

        if res_cfg is not None:
            prop_attr = xml_elem.attrib

            prop_ins = prop_attr.get("instance", "")
            prop_config = prop_attr.get("config", "")

            if res_cfg.create_config(prop_ins, prop_config) is not None:
                is_valid = True

        return is_valid

    def check(self):
        '''
        Checks that the parsed device is valid in terms
        of contents

        List of checks to execute:
        1) Package:
            - Nothing to Check
        2) Periphery Definition:
            - For each block, check that the timing parameter
            exists in the timing model
            - Check that the timing arc from and to name
            exists in the block port list
            - Check that the block PCR sequence only involves
            PCR name that has been defined
        3) Instance Check:
            - Check that the pin name in the connection matches
            with the block definition
            - Check that the function name is defined in the
            global clocks
        4) Global Clocks:
            - Check that the instance and pin names exist
            in the device
        5) IO:
            - Check that the bank name in the IO Pad map
             matches with the list of banks available in
            the device
            - Check that the pad name exists in the package
            definition
            - Check that the instance name exists in the device
        6) Device PCR Sequence:
            - Check that the instance name specified
              is defined in the device
        7) Timing Model:
            - If there are more than one operating combination,
            check that all the combination has the same parameter
            list
        8) Resource Map:
            - Check that the specified instance name exists
              in the device instance list
            - store the hierarchical instance names for instance
              with modes
        9) Config Model:
            - Check that the supported mode list matches with
              the list of available modes in config model
            - Check that all the pads listed exist in the device
              die io pad list


        :return True if the device is valid. Otherwise, False
        '''
        return self.device.check_after_parse()

    def validate(self, file):
        '''
        Validate that the device file does not violate the
        schema.

        :param file: Full path to the device filename
        '''

        device_file = file

        is_pass = True

        # Validate the device
        if DeviceBuilderXmlEventBased._schema_file == "":
            app_setting = aps.AppSetting()
            DeviceBuilderXmlEventBased._schema_file = os.path.join(app_setting.app_path[
                aps.AppSetting.PathType.schema], "peri_device_db.xsd")

        self.logger.debug("SCHEMA FILE For Validating {}".format(
            DeviceBuilderXmlEventBased._schema_file))

        try:
            schema = xmlschema.XMLSchema(
                DeviceBuilderXmlEventBased._schema_file)
        except Exception as excp:
            self.logger.error("Exception at device: {}".format(excp))
            return False

        self.logger.info("Validating {} using schema {}".
                         format(device_file,
                                DeviceBuilderXmlEventBased._schema_file))

        # Schema validation will expand top level schema to include
        # children schema. So it will validate againsts all.
        try:

            # To be able to validation on children nodes from included
            # xml, we need to use ElementTree instead of directly from
            # reading the file. With reading a file, it will always failed
            # when it hits the include
            # Base dir is the parent dir of device file
            tree = et.ElementTree(file=device_file)
            expanded_tree = None

            # Expand all includes until the leaf, not just at the top
            root = tree.getroot()
            try:
                # et.dump(root)
                # With the latest 3.9 ElementInclude, it uses base_url to
                # find embedded include file
                ei.include(root, base_url=device_file)

            except ei.FatalIncludeError as excp:
                self.logger.error("Include error: {}".format(excp))
                is_pass = False

            except OSError as excp:
                self.logger.error("File not found error: {}".format(excp))
                is_pass = False

            except Exception as excp:
                self.logger.error(
                    "Exception device validation: {}".format(excp))
                is_pass = False

            if is_pass:
                schema.validate(tree)

        except xmlschema.XMLSchemaValidationError as excp:
            self.logger.debug("Device XML contains error: {}".format(excp))
            self.logger.error("XML Element : {}".format(excp.elem))
            self.logger.error("Reason : {}".format(excp.reason))
            is_pass = False

        return is_pass

# Command line opts


def parse_cmdline_options():
    from argparse import ArgumentParser

    parser = ArgumentParser(
        description='EFXPT Device DB Parser and Validator')

    parser.add_argument('-d', '--device_file',
                        required=False,
                        help='Device file in XML format')
    parser.add_argument('-c', '--check_syntax',
                        action='store_true',
                        help='Check that the device is syntatically correct')
    parser.add_argument('-f', '--parse_device',
                        action='store_true',
                        help='Parses the device file and exit')
    parser.add_argument('-n', '--device_name',
                        required=False,
                        help='Device name to be parsed')
    parser.add_argument('-m', '--devicemap_file',
                        required=False,
                        help='Device map file where the device is also defined. '
                        'If not specified, then it uses the one in EFXPT_HOME')
    parser.add_argument('-s', '--save_device',
                        help='Save the device information into devices_info.py')
    parser.add_argument('-p', '--devices_path',
                        help='Set the path for devices_info.py')


    # run parser
    cmd_args = parser.parse_args()

    # error if file not found
    if cmd_args.device_file and not os.path.isfile(cmd_args.device_file):
        raise ValueError('could not find device file %s' %
                         cmd_args.device_file)

    return cmd_args


def parse_device():
    # parse command line options
    args = parse_cmdline_options()

    # We change directory to where the xml file is since there's
    # include statements that may need to be taken care of
    if args.device_file:
        abs_device_file = os.path.abspath(args.device_file)
        device_dir = os.path.dirname(os.path.abspath(args.device_file))
        print("device_dir: {}".format(device_dir))
        print("abs device_file: {}".format(abs_device_file))

    else:
        abs_device_file = ""

    if args.devicemap_file:
        devicemap_file = os.path.abspath(args.devicemap_file)
        print("abs devicemap file: {}".format(devicemap_file))
    else:
        devicemap_file = ""

    service = DeviceService()

    if args.save_device:
        if args.save_device == "head" or args.save_device == "end":
            pickle_file = DevicePickled(args.device_name)
            pickle_file.mode = args.save_device
            pickle_file.save_device_db()
            sys.exit(0)
        elif args.save_device == "all":
            if not args.devices_path:
                args.devices_path = None
                print("Generate devices_info.py file to default path")

            DevicePickled.gen_all_device_info(devicemap_file, args.devices_path)
            sys.exit(0)


    if args.check_syntax:
        print("Check device {} syntax".format(args.device_name))
        if not service.check_device(
                args.device_name, file=abs_device_file, map_file=devicemap_file):
            print('Device has invalid syntax!')
            sys.exit(1)
    elif args.parse_device:
        print("Check and parse device {}".format(args.device_name))
        if not service.check_device(
                args.device_name, file=abs_device_file, map_file=devicemap_file):
            print('Device has invalid syntax!')
            sys.exit(1)
        else:
            # Set the parse setting pcr file to True so
            # we can do a verification at compile time
            # pylint: disable=import-outside-toplevel
            import device.pcr_device as dev_pcr
            dev_pcr.PARSE_PCR_SETTINGS_FILE = True

            if args.save_device:
                service.is_save = True

            device_db = service.load_device(
                args.device_name, None, "", abs_device_file, devicemap_file)

            if args.save_device:
                if args.devices_path:
                    service.save_device(args.device_name, service.saved_device, args.devices_path)
                else:
                    service.save_device(args.device_name, service.saved_device)

            if device_db is None:
                print('Failed parsing device')
                sys.exit(1)

            dev_pcr.PARSE_PCR_SETTINGS_FILE = False

    sys.exit(0)


if __name__ == "__main__":
    parse_device()
