'''
Copyright (C) 2017-2018 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Dec 20, 2017

@author: maryam
'''

import sys
import os
import re

import xml.etree.ElementInclude as ei
import xml.etree.ElementTree as et
import xmlschema
# Use the one below to do hierarchical validation
#import util.ElementInclude as ei


import util.app_setting as aps
from util.singleton_logger import Logger

import device.service_interface as dbi
import device.parser.block_service as blks
import device.parser.config_model_service as cms
import device.timing_model as dev_timing

import device.block_instance as peri_ins
import device.clocks as clocks
import device.io_info as dev_io
import device.device_config as blkcfg

from device.mipi_group import MIPIDPHYRxGroupRegistry, MIPIDPHYRxGroup

sys.path.append(os.path.join(os.path.dirname(__file__), '..'))


class DieService(object):
    '''
    Hanldes parsing of the die block in the
    device file
    '''

    inf_pin_map = {
        "input": peri_ins.PinConnection.InterfaceType.input,
        "output": peri_ins.PinConnection.InterfaceType.output,
        "clkout": peri_ins.PinConnection.InterfaceType.clkout,
        "pad": peri_ins.PinConnection.InterfaceType.pad,
        "internal": peri_ins.PinConnection.InterfaceType.internal
    }

    secondary_conn_map = {
        "internal": peri_ins.SecondaryConn.SecondaryConnType.internal,
        "external": peri_ins.SecondaryConn.SecondaryConnType.external
    }

    def __init__(self, device_db, skip_check=True):
        """
        Constructor
        """
        self.device = device_db
        # We skip check if it was parse from the top device file (default)
        self.skip_check = skip_check

    def load_die(self, die_file, device_name=""):
        """
        Create the die model for a specific device

        :param die_file: path to die file name

        :return: Error count and The device db
        """
        if self.device is not None:
            builder = DieBuilderXmlEventBased(self.device, device_name)
            error, device_db = builder.build(die_file)

            # Check if there was no error and that skip check is not set
            if error == 0 and not self.skip_check:
                if not builder.check():
                    error = 1
        else:
            error = 1
            device_db = None

        return error, device_db

    def check_die(self, die_file):
        """
        Validate the die file

        :param die_file: die device file

        :return: True if good, else False
        """
        builder = DieBuilderXmlEventBased(self.device)
        return builder.validate(die_file)


class DieBuilderXmlEventBased(dbi.DeviceBuilder):
    '''
    Build die info from xml timing file using event based parsing
    '''

    _ns = "{http://www.efinixinc.com/peri_device_db}"
    '''
    XML Namespace for peri design db
    '''

    _util_ns = "{http://www.efinixinc.com/peri_tool}"
    '''
    XML Namespace for peri tool util
    '''

    _schema_file = ""
    '''
    If die schema is empty, it will be detected automatically using
     $EFXPT_HOME.
    This is mainly for unit testing to set schema from different location.
    '''

    def __init__(self, device_db, device_name=""):
        '''
        Constructor
        '''
        super().__init__()
        self.device = device_db
        self.device_dir = os.path.dirname(device_db.get_device_file())
        self.logger = Logger
        self.device_name = device_name

    def build(self, file):
        '''
        Build die info from an XML file

        :param file: Full path to the die filename

        :return error count and DeviceDB
        '''

        # We need to parse the block definition first before
        # Reading the instance
        error, _ = self._parse_included_files(file)

        # get an iterable
        context = et.iterparse(file, events=(
            "start", "end", "start-ns", "end-ns"))

        # turn it into an iterator
        context = iter(context)

        die_tag = DieBuilderXmlEventBased._ns + "die_definition"

        # layout
        layout_tag = DieBuilderXmlEventBased._ns + "layout"

        # Periphery Instance
        peri_instance_tag = DieBuilderXmlEventBased._ns + \
            "periphery_instance"
        single_conn_tag = DieBuilderXmlEventBased._ns + "single_conn"
        mult_conn_tag = DieBuilderXmlEventBased._ns + "mult_conn"
        conn_type_tag = DieBuilderXmlEventBased._ns + "conn_type"
        dependency_tag = DieBuilderXmlEventBased._ns + "dependency"
        ins_dep_tag = DieBuilderXmlEventBased._ns + "instance_dep"
        shared_ins_tag = DieBuilderXmlEventBased._ns + "shared_instance"

        mux_pattern_tag = DieBuilderXmlEventBased._ns + "ref_mux_pattern"
        swappable_pattern_tag = DieBuilderXmlEventBased._ns + "swappable_input_pattern"
        swap_connection_tag = DieBuilderXmlEventBased._ns + "swap_connection"

        # Global clocks
        global_clk_tag = DieBuilderXmlEventBased._ns + "global_clocks"
        clk_func_tag = DieBuilderXmlEventBased._ns + "function"
        clk_instance_tag = DieBuilderXmlEventBased._ns + "clk_instance"

        # IO Pad Definition
        io_pad_defn_tag = DieBuilderXmlEventBased._ns + "io_pad_definition"
        io_pad_map_tag = DieBuilderXmlEventBased._ns + "io_pad_map"

        # IO Bank list
        io_banks_tag = DieBuilderXmlEventBased._ns + "io_banks"
        bank_info_tag = DieBuilderXmlEventBased._ns + "bank_info"
        voltage_tag = DieBuilderXmlEventBased._ns + "valid_voltage"

        # Top-level PCR sequence
        top_pcr_seq_tag = DieBuilderXmlEventBased._ns + "pcr_sequence"
        pcr_seq_ins_tag = DieBuilderXmlEventBased._ns + "pcr_instance"

        # Timing parameter for die
        top_die_timing_tag = DieBuilderXmlEventBased._ns + "die_timing"
        timing_parameter_tag = DieBuilderXmlEventBased._ns + "timing_parameter"

        # Block specific configurations
        block_config_tag = DieBuilderXmlEventBased._ns + "die_block_configurations"
        block_tag = DieBuilderXmlEventBased._ns + "die_block"
        block_property_tag = DieBuilderXmlEventBased._ns + "die_property"

        # Instance specific timing info
        timing_tag = DieBuilderXmlEventBased._ns + "instance_timing"
        timing_delay_tag = DieBuilderXmlEventBased._ns + "timing_delay"

        # MIPI Grouping
        mipi_dphy_rx_group_tag = DieBuilderXmlEventBased._ns + "mipi_dphy_rx_group"
        mipi_group_tag = DieBuilderXmlEventBased._ns + "mipi_group"
        mipi_member_tag = DieBuilderXmlEventBased._ns + "mipi_member"

        # Shared instance
        shared_tag = DieBuilderXmlEventBased._ns + "shared"

        peri_instance = None

        mult_conn = None
        global_clk = None
        clk_function = ""
        bank_info = None
        device_io = None
        bank_default_io_std = ""
        die_timing = None
        block_config = None
        ins_timing_info = None
        mipi_group_obj = None
        mipi_group_top = None

        # Build top level design
        for event, elem in context:
            if event == "start":
                if elem.tag == die_tag:
                    pass

                elif elem.tag == layout_tag:
                    self._add_die_layout_size(elem)

                elif elem.tag == peri_instance_tag:
                    peri_instance = self._build_peri_instance(elem)
                    if peri_instance is None:
                        error += 1
                elif elem.tag == single_conn_tag:
                    if not self._build_ins_single_conn(peri_instance, elem):
                        error += 1
                elif elem.tag == mult_conn_tag:
                    conn_added, mult_conn = self._build_ins_mult_conn(
                        peri_instance, elem)
                    if not conn_added:
                        error += 1
                elif elem.tag == conn_type_tag:
                    if not self._build_ins_mult_conn_type(
                            mult_conn, elem):
                        error += 1
                elif elem.tag == dependency_tag:
                    pass
                elif elem.tag == ins_dep_tag:
                    self._build_ins_dep(peri_instance, elem)

                elif elem.tag == shared_ins_tag:
                    pass

                elif elem.tag == shared_tag:
                    if not self._build_shared_instance(peri_instance, elem):
                        error += 1

                # Mux pattern name
                elif elem.tag == mux_pattern_tag:
                    self._set_mux_pattern_name(peri_instance, elem)

                # Mux input connectivity
                elif elem.tag == swappable_pattern_tag:
                    pass
                elif elem.tag == swap_connection_tag:
                    if not self._build_swap_connection(peri_instance, elem):
                        error += 1

                elif elem.tag == global_clk_tag:
                    global_clk = self._build_global_clocks()
                elif elem.tag == clk_func_tag:
                    clk_function = self._build_clk_function(elem)
                elif elem.tag == clk_instance_tag:
                    self._build_clk_instance(global_clk,
                                             clk_function, elem)

                elif elem.tag == io_pad_defn_tag:
                    device_io = self._build_io_pad_defn()
                elif elem.tag == io_pad_map_tag:
                    self._build_io_pad_map(device_io, elem)

                elif elem.tag == io_banks_tag:
                    pass

                elif elem.tag == bank_info_tag:
                    bank_info, bank_default_io_std = \
                        self._build_bank_info(device_io, elem)
                elif elem.tag == voltage_tag:
                    self._build_bank_voltage(bank_info, elem)

                elif elem.tag == top_pcr_seq_tag:
                    self._build_top_pcr()
                elif elem.tag == pcr_seq_ins_tag:
                    if not self._build_top_pcr_sequence(elem):
                        error += 1

                elif elem.tag == top_die_timing_tag:
                    die_timing = self._build_die_timing()
                elif elem.tag == timing_parameter_tag:
                    if not self._add_timing_parameter(elem, die_timing):
                        error += 1

                elif elem.tag == block_config_tag:
                    dev_block_config = self._build_device_block_config()

                elif elem.tag == block_tag:
                    block_config = self._build_block_config(
                        dev_block_config, elem)
                    if block_config is None:
                        error += 1

                elif elem.tag == block_property_tag:
                    if not self._build_block_property(block_config, elem):
                        error += 1

                elif elem.tag == timing_tag:
                    ins_timing_info = self._build_instance_timing(
                        peri_instance)
                    if ins_timing_info is None:
                        error += 1

                elif elem.tag == timing_delay_tag:
                    if ins_timing_info is not None:
                        if not self._build_timing_delay(ins_timing_info, elem):
                            error += 1

                elif elem.tag == mipi_dphy_rx_group_tag:
                    # Create the MIPIDPHYRxGroupRegistry
                    mipi_group_top = self._build_device_mipi_group(elem)

                elif elem.tag == mipi_group_tag:
                    mipi_group_obj = self._build_mipi_group(
                        mipi_group_top, elem)
                    if mipi_group_obj is None:
                        error += 1

                elif elem.tag == mipi_member_tag:
                    if not self._build_mipi_group_member(mipi_group_obj, elem):
                        error += 1

                # Free up process element
                elem.clear()

            # When we reach the end, reset the handle
            elif event == "end":
                if elem.tag == bank_info_tag:
                    if not bank_info.set_default_io_standard(bank_default_io_std):
                        error += 1
                    bank_default_io_std = ""
                    bank_info = None
                elif elem.tag == peri_instance_tag:
                    peri_instance = None
                elif elem.tag == mult_conn_tag:
                    mult_conn = None
                elif elem.tag == clk_func_tag:
                    clk_function = ""
                elif elem.tag == block_tag:
                    block_config = None
                elif elem.tag == timing_tag:
                    ins_timing_info = None
                elif elem.tag == mipi_group_tag:
                    mipi_group_obj = None

                # Free up process element
                elem.clear()

        return error, self.device

    def _parse_included_files(self, die_file):
        '''
        This parses the die file but to read the block
        definition part ONLY.
        :param die_file: Full path to the die filename

        :return error count and DeviceDB
        '''

        # get an iterable
        context = et.iterparse(die_file, events=(
            "start", "end", "start-ns", "end-ns"))

        # turn it into an iterator
        context = iter(context)

        die_tag = DieBuilderXmlEventBased._ns + "die_definition"

        include_tag = "{http://www.w3.org/2001/XInclude}include"
        incl_list = []

        error = 0
        parse_die = False

        # Build top level design
        for event, elem in context:
            if event == "start":
                if elem.tag == die_tag:
                    parse_die = True

                elif elem.tag == include_tag and parse_die:
                    incl_attr = elem.attrib
                    incl_file = incl_attr.get("href", None)
                    if incl_file is not None:
                        incl_list.append(incl_file)

                # Free up process element
                elem.clear()

            # When we reach the end, reset the handle
            elif event == "end":
                if elem.tag == die_tag:
                    parse_die = False

                # Free up process element
                elem.clear()

        if self.device is not None:
            # continue parsing the included files

            # Build periphery block
            error += self._build_block(die_file, incl_list)

            # Build config model
            error += self._build_config_model(die_file, incl_list)

        return error, self.device

    def _build_block(self, die_file, incl_list):
        '''
        Build the periphery block. It is expected that
        all the included file is a block definition file.

        :param incl_list: The list of included files in
                    the die file
        :return error count
        '''
        blk_service = blks.BlockService(self.device)
        error = 0

        if not incl_list:
            tmp_error, _ = blk_service.load_block(
                self.device.get_device_file(), self.device_name)
            error += tmp_error

        else:
            for incl_file in incl_list:
                # Get the parent directory of the die file
                die_dir = os.path.dirname(os.path.abspath(die_file))
                abs_incl_file = os.path.join(die_dir, incl_file)

                match = re.search(r'block_models\/\w+\.xml', incl_file)
                if match:

                    tmp_error, _ = blk_service.load_block(abs_incl_file, self.device_name)
                    error += tmp_error

        return error

    def _build_config_model(self, die_file, incl_list):
        '''
        Build the config model using the information in the
        config model xml.

        :param incl_list: The list of included files in the die
                    file
        :return error count
        '''
        cmodel_service = cms.ConfigModelService(self.device)
        error = 0

        if not incl_list:
            tmp_error, _ = cmodel_service.load_config_model(
                self.device.get_device_file())
            error += tmp_error

        else:
            for incl_file in incl_list:

                die_dir = os.path.dirname(os.path.abspath(die_file))
                abs_incl_file = os.path.join(die_dir, incl_file)

                match = re.search(r'^config_models\/\w+\.xml', incl_file)
                if match:

                    tmp_error, _ = cmodel_service.load_config_model(
                        abs_incl_file)
                    error += tmp_error

        return error

    def _add_die_layout_size(self, xml_elem):
        '''
        Save the die layout size into the device db
        '''
        layout_attr = xml_elem.attrib
        width = layout_attr.get("width", 0)
        height = layout_attr.get("height", 0)

        self.device.set_die_layout_size(int(width), int(height))

    def _build_peri_instance(self, xml_elem):
        '''
        Create the PeripheryBlockInstance with the pass
        name and type.

        :param xml_elem: periphery_instance tag

        :return The created PeripheryBlockInstance
        '''

        ins_attr = xml_elem.attrib

        cur_inst = self.device.create_peri_instance(
            ins_attr["block"],
            ins_attr["name"])

        if cur_inst is None:
            self.logger.error('Unable to create periphery instance'
                              ' {} of type {}'.format(
                                  ins_attr["name"],
                                  ins_attr["block"]))

        return cur_inst

    def _build_ins_single_conn(self, peri_instance, xml_elem):
        '''
        Create the single connection of an instance.
        Single connection means that the instance pin
        connects to a single pin.

        :param peri_instance: PeripheryBlockInstance
        :param xml_elem: singel_conn tag

        :return True if the connection was added to the instance
        '''
        single_attr = xml_elem.attrib
        added = False

        # Get the translated enum type
        conn_type = DieService.inf_pin_map.get(
            single_attr["type"], "")

        # primary attribute is optional
        if "primary" in single_attr:
            primary = peri_ins.ConnCoordinate(single_attr["primary"])
        else:
            primary = None

        index = single_attr.get("index", None)
        if index is not None:
            index = int(index)

        # Applicable if internal type connection
        instance = single_attr.get("instance", None)
        pin_name = single_attr.get("pin", None)
        pin_index = single_attr.get("pin_index", None)

        # If the pin has constraint to other pin within the block instance
        out_constraint = single_attr.get("out_constraint", None)

        # We check this since there might be a case where
        # there was an error and the object not created.
        if peri_instance is not None:
            pin_conn = peri_ins.PinConnection(single_attr["name"],
                                              conn_type,
                                              primary, instance, pin_name,
                                              index, pin_index, out_constraint)

            if not peri_instance.add_inf_pin_conn(pin_conn):
                self.logger.error('Unable to add instance connection on'
                                  ' {}:{}'.format(peri_instance.get_name(),
                                                  single_attr["name"]))
            else:
                added = True

        return added

    def _build_ins_mult_conn(self, peri_instance, xml_elem):
        '''
        Create the multiple connection of an instance.
        Multiple connection means that the instance pin
        connects to multiple pin.

        :param peri_instance: PeripheryBlockInstance
        :param xml_elem: mult_conn tag

        :return added: Indicating if connection was added to instance pin
                mult_conn: The created PinConnection

        '''
        mult_attr = xml_elem.attrib
        mult_conn = None
        added = False

        # Get the translated enum type
        conn_type = DieService.inf_pin_map.get(
            mult_attr["type"], "")

        # primary attribute is optional
        if "primary" in mult_attr:
            primary = peri_ins.ConnCoordinate(mult_attr["primary"])
        else:
            primary = None

        index = mult_attr.get("index", None)
        if index is not None:
            index = int(index)

        if peri_instance is not None:
            mult_conn = peri_ins.PinConnection(mult_attr["name"],
                                               conn_type,
                                               primary, None, None,
                                               index)

            if not peri_instance.add_inf_pin_conn(mult_conn):
                self.logger.error('Unable to add instance connection on'
                                  ' {}:{}'.format(peri_instance.get_name(),
                                                  mult_attr["name"]))
            else:
                added = True

        return added, mult_conn

    def _build_ins_mult_conn_type(self, mult_conn, xml_elem):
        '''
        Add connection type (SecondaryConn) that indicates
        the remaining connectivity on the pin.

        :param mult_conn: The primary PinConnection that this secondary
                    connection belongs to
        :param xml_elem: conn_type tag
        '''
        valid = True

        mult_attr = xml_elem.attrib

        # Get the translated enum type
        conn_type = DieService.secondary_conn_map.get(
            mult_attr["type"], "")

        sec_conn = None
        secondary = None

        if conn_type == peri_ins.SecondaryConn.SecondaryConnType.external:
            # Get the coordinate
            # primary attribute is optional
            if "secondary" in mult_attr:
                secondary = peri_ins.ConnCoordinate(mult_attr["secondary"])

        function_name = mult_attr.get("function", "")
        sec_conn = peri_ins.SecondaryConn(function_name,
                                          conn_type, secondary)

        # Check that the function type is expected
        if not sec_conn.is_function_name_valid():
            self.logger.error(
                "Invalid instance secondary pin function name: {}".format(
                    function_name))
            valid = False

        mult_conn.add_secondary_connection(sec_conn)

        return valid

    def _build_ins_dep(self, peri_instance, xml_elem):
        '''
        Create instance dependency, if there is one.

        :param peri_instance: PeripheryBlockInstance
        :param xml_elem:  dependency tag
        '''
        dep_attr = xml_elem.attrib

        if peri_instance is None:
            self.logger.error("Unexpected None instance")

        pin_name = dep_attr["pin"]
        if pin_name.find(":") != -1:
            # If it is multiple pins associated to the same instance,
            # then we break it here
            pin_list = pin_name.split(":")
            for pin in pin_list:
                peri_instance.add_dependency(dep_attr["name"], pin)
        else:
            peri_instance.add_dependency(dep_attr["name"], pin_name)

    def _build_shared_instance(self, peri_instance, xml_elem):
        '''
        Create list of shared instances between this instance and 
        other instance, if there is one.

        :param peri_instance: PeripheryBlockInstance
        :param xml_elem:  dependency tag
        '''
        dep_attr = xml_elem.attrib

        if peri_instance is None:
            self.logger.error("Unexpected None instance")

        shared_instance = dep_attr.get("name", "")

        # some attributes are optional: ins_type, sub_ins
        if not peri_instance.add_shared_instance(
                shared_instance, dep_attr.get("block", ""),
                dep_attr.get("ins_type", ""), dep_attr.get("sub_ins", "")):

            self.logger.debug(
                "Unable to add shared instance {} to instance {}".format(
                    shared_instance, peri_instance.get_name()))

            return False

        return True

    def _set_mux_pattern_name(self, instance, xml_elem):
        '''
        Set the instance mux_pattern name to the parsed
        name.
        :param instance: instance that this belongs to
        :param xml_elem: swap_connection tag
        '''
        mux_attr = xml_elem.attrib
        pattern_name = mux_attr["name"]

        instance.set_mux_pattern_name(pattern_name)

    def _build_swap_connection(self, instance, xml_elem):
        '''
        Add a swap connection pattern to the group
        :param instance: instance that this belongs to
        :param xml_elem: swap_connection tag

        :return True if it was added successfully.
        '''
        valid = True

        swap_attr = xml_elem.attrib
        input_pin = swap_attr["input"]
        swap_input_str = swap_attr["output"]

        # The output should be a string with comma delimer
        if swap_input_str.find(',') != -1:
            output_list = swap_input_str.split(',')

            valid = instance.add_input_swap_connection(input_pin, output_list)

        else:
            self.logger.warning(
                "Expecting a list for output in swap_connection ({}) for input {}".format(
                    swap_input_str, input))

        return valid

    def _build_global_clocks(self):
        '''
        Creates an empty GlobalClocks object and return it

        :return GlobalClocks object
        '''

        global_clock = clocks.GlobalClocks(self.device.get_device_name())

        self.device.set_global_clocks(global_clock)

        return global_clock

    def _build_clk_function(self, xml_elem):
        '''
        Return the clock function name

        :param xml_elem: function tag in global_clocks

        :return The function name
        '''
        clk_attr = xml_elem.attrib

        return clk_attr["name"]

    def _build_clk_instance(self, global_clock,
                            func_name, xml_elem):
        '''
        Save the information to the global_clock

        :param global_clock: GlobalClocks object
        :param func_name: The function name of the clock
        :param xml_elem: clk_instance tag
        '''
        clk_attr = xml_elem.attrib

        global_clock.create_global_clock(func_name,
                                         clk_attr["name"],
                                         clk_attr["pin"])

    def _build_io_pad_defn(self):
        '''
        Build the device IO definition (DeviceIO) and
        save it to the device.

        :return The created DeviceIO
        '''
        device_io = dev_io.DeviceIO(self.device.get_device_name())

        self.device.set_io_pad(device_io)

        return device_io

    def _build_io_pad_map(self, device_io, xml_elem):
        '''
        Create the io pad mapping info and save it to the
        DeviceIO

        :param device_io: DeviceIO object that is already saved
                        to the device
        :param xml_elem: io_pad_map tag
        '''

        io_attr = xml_elem.attrib

        # x_coord = int(io_attr["x"])
        # y_coord = int(io_attr["y"])
        instances = []
        instance = ""
        if "instance" in io_attr:
            instance = io_attr["instance"]

            # Check if the instance is a string delimited with ':'
            # TODO: how to model multiple instance per pad (shared pad)
            instances = instance.split(':')

        bank_name = ""
        if "bank" in io_attr:
            bank_name = io_attr["bank"]

        direction = dev_io.IOPadInfo.PadDirection.none
        function = ""

        if "direction" in io_attr:
            dir_str = io_attr["direction"]

            if dir_str in dev_io.IOPadInfo.str2dir_map:
                direction = dev_io.IOPadInfo.str2dir_map[dir_str]

        if "function" in io_attr:
            function = io_attr["function"]

        bscan_seq_nums = None
        if "bscan_seq_nums" in io_attr:
            bscan_seq_nums = (int(v) for v in str(io_attr['bscan_seq_nums']).split(","))

        pin_name = io_attr.get("pin", None)

        device_io.create_pad(bank_name, io_attr["pad_name"],
                             instances, direction, function, pin_name, bscan_seq_nums)

    def _build_bank_info(self, device_io, xml_elem):
        '''
        Create IO bank info and save it to the device IO. At this
        point DeviceIO should have already been created.

        :param device_io: DeviceIO of a device
        :param xml_elem: io_bank_info tag

        :return IOBankInfo, default io standard name:
                IOBankInfo: The created IOBankInfo
                default: IO standard name save as default
        '''

        bank_attr = xml_elem.attrib
        default_io_std = ""

        if "default" in bank_attr:
            default_io_std = bank_attr["default"]

        return device_io.create_bank_info(bank_attr["name"]), default_io_std

    def _build_bank_voltage(self, bank_info, xml_elem):
        '''
        Save the voltage to the bank_info

        :param bank_info: IOBankInfo that has been saved to the
                    DeviceIO
        :param xml_elem: valid_voltage tag
        '''
        volt_attr = xml_elem.attrib
        bank_info.add_io_standard(volt_attr["voltage"])

    def _build_top_pcr(self):
        '''
        Clears the device pcr_top
        '''
        self.device.reset_pcr_bits_sequence()

    def _build_top_pcr_sequence(self, xml_elem):
        '''
        Add a pcr sequence into the device pcr_sequence_list.
        At this point we don't check that the instance name
        specified exists in the die.

        :param xml_elem: pcr_instance tag
        '''

        pcr_attr = xml_elem.attrib
        int_index = int(pcr_attr["index"])

        added = self.device.add_pcr_bits_sequence(pcr_attr["name"],
                                                  int_index)
        return added

    def _build_die_timing(self):
        '''
        Create the device common_timing object
        '''
        tparam = dev_timing.CommonTiming()
        self.device.set_common_timing(tparam)

        return tparam

    def _add_timing_parameter(self, xml_elem, die_timing):
        '''
        Add the timing_parameter to the device
        :param xml_elem: timing_parameter tag
        '''
        param_attr = xml_elem.attrib

        param_obj = dev_timing.TimingParameter(
            param_attr.get("name", None),
            param_attr.get("value", None))

        return die_timing.add_parameter(param_obj)

    # The device block config is the same as device.  This is
    # to allow the information be saved either at die level
    # (common across device) or at device level (device-specific)
    def _build_device_block_config(self):
        dev_blk_config = self.device.get_block_configuration()

        if dev_blk_config is None:
            dev_blk_config = blkcfg.DeviceBlockConfig()

            self.device.set_block_configuration(dev_blk_config)

        return dev_blk_config

    def _build_block_config(self, dev_block_config, xml_elem):

        config_attr = xml_elem.attrib

        bname = config_attr["name"]

        blk_config = dev_block_config.get_block_property(bname)

        if blk_config is None:
            blk_config = blkcfg.BlockProperty(bname)

            if not dev_block_config.set_block_property(bname, blk_config):
                self.logger.error(
                    "Error loading block config for {}".format(bname))
                return None

        return blk_config

    def _build_block_property(self, block_config, xml_elem):
        prop_attr = xml_elem.attrib

        is_valid = False

        prop_name = prop_attr["name"]
        tmp_value = prop_attr["value"]

        # Remove extra whitespace at beginning and end of each entry if it was
        # a list
        if tmp_value.find(",") != -1:
            value_list = tmp_value.split(",")
            prop_value = ""
            for value_str in value_list:
                new_value = value_str.strip()
                if prop_value == "":
                    prop_value = new_value
                else:
                    prop_value = prop_value + "," + new_value
        else:
            prop_value = tmp_value

        if block_config is not None:

            # Check if the property name is expected
            if prop_name not in blkcfg.DeviceBlockConfig.str2bprop_map:
                self.logger.error("Unexpected {} property name {}".format(
                    block_config.get_block_name(), prop_name))

            else:
                is_added = block_config.add_property(prop_name, prop_value)
                if not is_added:
                    self.logger.error("Error adding {} property name: {} value: {}".format(
                        block_config.get_block_name(), prop_name, prop_value))
                else:
                    is_valid = True

        else:
            self.logger.error("Error adding property name: {} value: {}".format(
                prop_name, prop_value))

        return is_valid

    def _build_instance_timing(self, peri_instance):
        timing_obj = None

        if peri_instance is not None:
            timing_obj = peri_instance.create_instance_timing()

        if timing_obj is None:
            self.logger.error(
                "Error creating instance timing info for {}".format(
                    peri_instance.get_name()))

        return timing_obj

    def _build_timing_delay(self, ins_timing_info, xml_elem):
        is_added = False

        delay_attr = xml_elem.attrib

        delay_type = delay_attr["type"]
        delay_value = int(delay_attr["value"])

        if ins_timing_info is not None:
            is_added = ins_timing_info.add_delay(delay_type, delay_value)

        if not is_added:
            self.logger.error(
                "Error adding the instance timing type: {}, value: {} for {}".format(
                    delay_type, delay_value, ins_timing_info.get_ins_name()))

        return is_added

    def _build_device_mipi_group(self, xml_elem):
        mipi_attr = xml_elem.attrib

        blk_name = mipi_attr.get("block", "")
        mode_name = mipi_attr.get("mode", "")

        mipi_device_obj = MIPIDPHYRxGroupRegistry(blk_name, mode_name)

        self.device.set_mipi_dphy_rx_group(mipi_device_obj)

        return mipi_device_obj

    def _build_mipi_group(self, mipi_group_top, xml_elem):
        mipi_attr = xml_elem.attrib
        grp_name = mipi_attr.get("name", "")

        mipi_group_obj = None
        if mipi_group_top is not None and grp_name != "":
            mipi_group_obj = mipi_group_top.create_group(grp_name)
        else:
            self.logger.error(
                "Error creating the MIPI DPHY RX group {}".format(grp_name))

        return mipi_group_obj

    def _build_mipi_group_member(self, mipi_group_obj, xml_elem):
        is_valid = True

        if mipi_group_obj is not None:
            mipi_attr = xml_elem.attrib

            ins_name = mipi_attr.get("instance", "")
            pad_name = mipi_attr.get("pad_name", "")
            index = mipi_attr.get("index", "")
            func_name = mipi_attr.get("function", "")
            type_str = mipi_attr.get("type", "")

            # Convert the type
            if type_str != "" and type_str in MIPIDPHYRxGroup.str2type_map:
                type_enum = MIPIDPHYRxGroup.str2type_map[type_str]

                if not mipi_group_obj.add_member(ins_name, pad_name, index, func_name, type_enum):
                    is_valid = False
                    self.logger.error("Error in adding MIPI DPHY RX member {} to group {}".format(
                        pad_name, mipi_group_obj.get_group_name()))

            else:
                is_valid = False
                self.logger.error("Invalid empty type string for mipi group {} with pad name {}".format(
                    mipi_group_obj.get_group_name(), pad_name))

        return is_valid

    def check(self):
        """
        Apply device check. The check is implemented at the
        Top device

        :returns True if valid, else False
        """
        if self.device is not None:
            shared_ins = []
            return self.device.check_die_post_parse(shared_ins)

        return True

    def validate(self, file):
        '''
        Validate that the die file does not violate the
        schema.

        :param file: Full path to the die filename
        '''

        is_pass = True

        # Validate the timing model
        if DieBuilderXmlEventBased._schema_file == "":
            app_setting = aps.AppSetting()
            DieBuilderXmlEventBased._schema_file = app_setting.app_path[
                aps.AppSetting.PathType.schema] + "/die_definition.xsd"

        try:
            schema = xmlschema.XMLSchema(
                DieBuilderXmlEventBased._schema_file)
        except Exception as excp:
            self.logger.error("Exception at die: {}".format(excp))
            return False

        self.logger.info("Validating {} using schema {}".
                         format(file,
                                DieBuilderXmlEventBased._schema_file))

        # Schema validation will expand top level schema to include
        # children schema. So it will validate againsts all.
        # However it does not process include files.

        if is_pass:
            # Only check the device against schema if schema is good
            try:
                # To be able to validation on children nodes from included
                # xml, we need to use ElementTree instead of directly from
                # reading the file. With reading a file, it will always failed
                # when it hits the include
                tree = et.ElementTree(file=file)

                # Expand all includes until the leaf, not just at the top
                root = tree.getroot()
                try:
                    ei.include(root)

                except ei.FatalIncludeError as excp:
                    self.logger.error("Include error: {}".format(excp))
                    is_pass = False

                except OSError as excp:
                    self.logger.error("File not found error: {}".format(excp))
                    is_pass = False

                schema.validate(tree)

            except xmlschema.XMLSchemaValidationError as excp:
                self.logger.error("XML Element : {}".format(excp.elem))
                self.logger.error("Reason : {}".format(excp.reason))
                is_pass = False

        return is_pass
