'''
Copyright (C) 2017-2018 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Dec 20, 2017

@author: maryam
'''
import sys
import os
import xml.etree.ElementTree as et
import xmlschema

import util.app_setting as aps
from util.singleton_logger import Logger
import util.gen_util as gen_util

import device.service_interface as dbi
import device.block_definition as peri_block
import device.pcr_device as blk_pcr


sys.path.append(os.path.join(os.path.dirname(__file__), '..'))


class BlockService(object):
    '''
    Handles parsing of the periphery block
    in the device file.
    '''
    port_dir_map = {
        "input": peri_block.PortDir.input,
        "output": peri_block.PortDir.output,
        "inout": peri_block.PortDir.inout
    }

    timing_arc_map = {
        "delay": peri_block.TimingArc.TimingArcType.delay,
        "setup": peri_block.TimingArc.TimingArcType.setup,
        "hold": peri_block.TimingArc.TimingArcType.hold,
        "clk_to_q": peri_block.TimingArc.TimingArcType.clk_to_q
    }

    beh_param_map = {
        "int": peri_block.BehavioralParameterDefn.BehParamType.int,
        "boolean": peri_block.BehavioralParameterDefn.BehParamType.boolean,
        "string": peri_block.BehavioralParameterDefn.BehParamType.string,
        "float": peri_block.BehavioralParameterDefn.BehParamType.float
    }

    def __init__(self, device_db, skip_check=True):
        """
        Constructor
        """
        self.device = device_db
        # We skip check if it was parse from the top device file (default)
        self.skip_check = skip_check
        self.logger = Logger

    def load_block(self, block_file, device_name=""):
        """
        Create the PeripheryBlock for a specific device

        :param block_file: path to block file name

        :return: Error count and The device db
        """
        if self.device is not None:
            builder = BlockBuilderXmlEventBased(self.device, device_name)
            error, device_db = builder.build(block_file)
            self.logger.debug("Load device {} block {} with error: {}".format(
                device_name, block_file, error))

            # Check if there was no error and that skip check is not set
            if error == 0 and not self.skip_check:
                if not builder.check():
                    error = 1
        else:
            error = 1
            device_db = None

        return error, device_db

    def check_block(self, block_file):
        """
        Validate the block file

        :param block_file: periphery block file

        :return: True if good, else False
        """
        builder = BlockBuilderXmlEventBased(self.device)
        return builder.validate(block_file)


class BlockBuilderXmlEventBased(dbi.DeviceBuilder):
    '''
    Build timing model from xml timing file using event based parsing
    '''

    _ns = "{http://www.efinixinc.com/peri_device_db}"
    '''
    XML Namespace for peri design db
    '''

    _util_ns = "{http://www.efinixinc.com/peri_tool}"
    '''
    XML Namespace for peri tool util
    '''

    _schema_file = ""
    '''
    If periphery block schema is empty, it will be detected
    automatically using $EFXPT_HOME.
    This is mainly for unit testing to set schema from different location.
    '''

    def __init__(self, device_db, device_name=""):
        '''
        Constructor
        '''
        super().__init__()
        self.device = device_db
        self.block = None
        self.logger = Logger
        self.device_name = device_name

    def build(self, file):
        '''
        Build timing model from an XML file

        :param file: Full path to the block filename

        :return error count, the device_db
        '''
        # get an iterable
        context = et.iterparse(file, events=(
            "start", "end", "start-ns", "end-ns"))

        # turn it into an iterator
        context = iter(context)

        # Periphery Definition
        periphery_defn_tag = BlockBuilderXmlEventBased._ns + \
            "periphery_definition"
        port_tag = BlockBuilderXmlEventBased._ns + "port"

        mode_tag = BlockBuilderXmlEventBased._ns + "mode"
        port_mode_tag = BlockBuilderXmlEventBased._ns + "port_mode"
        timing_arc_mode_tag = BlockBuilderXmlEventBased._ns + "timing_arc_mode"

        timing_arc_tag = BlockBuilderXmlEventBased._ns + "timing_arc"
        beh_param_tag = BlockBuilderXmlEventBased._ns + "beh_parameter"

        # mux related to block with mux connections
        mux_pattern_tag = BlockBuilderXmlEventBased._ns + "mux_pattern"

        # Mux connectivity, alternative to mux_patter_tag. It has to be either
        # or
        internal_conn_tag = BlockBuilderXmlEventBased._ns + "internal_connectivity"
        internal_file_conn_tag = BlockBuilderXmlEventBased._ns + "internal_conn"
        buf_conn_tag = BlockBuilderXmlEventBased._ns + "buf_conn"

        #
        pcr_config_tag = BlockBuilderXmlEventBased._ns + "pcr_config"
        pcr_defn_tag = BlockBuilderXmlEventBased._ns + "pcr_defn"
        pcr_int_defn_tag = BlockBuilderXmlEventBased._ns + "pcr_int_defn"
        pcr_bitvec_defn_tag = BlockBuilderXmlEventBased._ns + "pcr_bitvec_defn"
        pcr_mode_tag = BlockBuilderXmlEventBased._ns + "pcr_mode"
        pcr_int_mode_tag = BlockBuilderXmlEventBased._ns + "pcr_int_mode"
        pcr_bitvec_mode_tag = BlockBuilderXmlEventBased._ns + "pcr_bitvec_mode"
        pcr_seq_tag = BlockBuilderXmlEventBased._ns + "sequence"
        pcr_defn_seq_tag = BlockBuilderXmlEventBased._ns + "pcr_defn_seq"
        pcr_file_map_tag = BlockBuilderXmlEventBased._ns + "pcr_file_map"

        pcr_lookup_tag = BlockBuilderXmlEventBased._ns + "pcr_lookup"
        pcr_permutations_tag = BlockBuilderXmlEventBased._ns + "pcr_permutations"
        pcr_fixed_settings_tag = BlockBuilderXmlEventBased._ns + "pcr_fixed_settings"

        embed_res_tag = BlockBuilderXmlEventBased._ns + "embedded_resource_map"
        resource_tag = BlockBuilderXmlEventBased._ns + "embed_resource"

        ins_config_tag = BlockBuilderXmlEventBased._ns + "ins_config"

        block = None
        pcr_block = None
        pcr_defn = None
        pcr_int_defn = None
        pcr_bitvec_defn = None
        mode_obj = None
        pcr_lookup = None
        error = 0
        pcr_map_default = ""
        seq_group_name = ""
        seq_group_obj = None

        mux_pattern_rep = 0
        mux_conn_rep = 0
        block_name = ""
        blk_conn = None
        blk_embed_res = None

        common_pcr_permutation_file = ""

        for event, elem in context:
            if event == "start":
                # Block related
                if elem.tag == periphery_defn_tag:
                    block = self._build_peri_block(elem)

                    if block is None:
                        error += 1
                        # No point of creating the block if it already
                        # exists
                        break
                    else:
                        self.block = block
                        block_name = block.get_name()

                elif elem.tag == port_tag:
                    if not self._build_block_port(block, elem):
                        error += 1
                elif elem.tag == mode_tag:
                    mode_obj = self._build_mode(block, elem)

                    if mode_obj is None:
                        error += 1

                elif elem.tag == port_mode_tag:
                    if not self._build_mode_port(mode_obj, elem):
                        error += 1

                elif elem.tag == timing_arc_tag:
                    self._build_timing_arc(block, elem)

                elif elem.tag == timing_arc_mode_tag:
                    self._build_timing_arc_mode(mode_obj, elem)

                elif elem.tag == beh_param_tag:
                    self._build_beh_param(block, elem)

                # Mux related elements
                elif elem.tag == mux_pattern_tag:
                    mux_pattern_rep += 1
                    if not self._build_mux_pattern(block, elem, file):
                        error += 1

                elif elem.tag == internal_conn_tag:
                    mux_conn_rep += 1
                    blk_conn = self._build_internal_connectivity(block, elem)
                    if blk_conn is None:
                        error += 1
                elif elem.tag == internal_file_conn_tag:
                    if not self._save_internal_file_conn(elem, block, blk_conn, file):
                        error += 1
                elif elem.tag == buf_conn_tag:
                    self._save_buf_conn(elem, blk_conn)

                elif elem.tag == pcr_config_tag:
                    pcr_block = self._build_pcr_block(block, elem)
                elif elem.tag == pcr_defn_tag:
                    pcr_defn, pcr_map_default = self._build_pcr_defn(
                        pcr_block, elem)
                elif elem.tag == pcr_int_defn_tag:
                    pcr_int_defn, pcr_map_default = self._build_pcr_int_defn(
                        pcr_block, elem)
                elif elem.tag == pcr_bitvec_defn_tag:
                    pcr_bitvec_defn, pcr_map_default = self._build_pcr_bitvec_defn(
                        pcr_block, elem)
                elif elem.tag == pcr_mode_tag:
                    if not self._build_pcr_mode(pcr_defn, elem):
                        error += 1
                elif elem.tag == pcr_int_mode_tag:
                    if not self._build_pcr_int_mode(pcr_int_defn, elem):
                        error += 1
                elif elem.tag == pcr_bitvec_mode_tag:
                    if not self._build_pcr_bitvec_mode(pcr_bitvec_defn, elem):
                        error += 1
                elif elem.tag == pcr_file_map_tag:
                    if not self._build_pcr_map_from_file(pcr_int_defn, elem, file):
                        error += 1
                elif elem.tag == pcr_seq_tag:
                    seq_group_name, seq_group_obj, is_added = self._build_sequence(
                        pcr_block, elem, block_name)

                    if not is_added:
                        error += 1

                elif elem.tag == pcr_defn_seq_tag:
                    if not self._build_block_pcr_seq(pcr_block, elem, seq_group_name):
                        error += 1

                elif elem.tag == pcr_lookup_tag:
                    pcr_lookup = self._build_pcr_lookup(pcr_block)
                    if pcr_lookup is None:
                        error += 1

                elif elem.tag == pcr_permutations_tag:
                    if pcr_lookup is not None:
                        is_pfile_valid, cur_common_pfile, is_cur_dev_pfile = self._build_pcr_permutations(pcr_lookup, elem, file)
                        if not is_pfile_valid:
                            error += 1
                        elif is_cur_dev_pfile:
                            # Once we found at least a file used for device specific
                            # then, it means we never need the common file
                            common_pcr_permutation_file = ""
                        elif cur_common_pfile != "":
                            # If there's at least one dev_specific, we don't need to save
                            # the common file
                            # If there's more than one common permutation file f ound,
                            # then, it's an error
                            if common_pcr_permutation_file != "":
                                self.logger.error(f"Found multiple common pcr permutation file: {cur_common_pfile}, {common_pcr_permutation_file}")
                                error += 1

                            if pcr_lookup.get_permutation_file() == "":
                                common_pcr_permutation_file = cur_common_pfile
                            

                elif elem.tag == pcr_fixed_settings_tag:
                    if pcr_lookup is not None and not self._build_pcr_fixed_settings(pcr_lookup, elem, file):
                        error += 1

                elif elem.tag == ins_config_tag:
                    if not self._build_ins_config(block, elem, file):
                        error += 1

                elif elem.tag == embed_res_tag:
                    blk_embed_res = self._build_block_embed_res(block)

                    if blk_embed_res is None:
                        error += 1

                elif elem.tag == resource_tag:
                    if blk_embed_res is not None:
                        self._build_embed_res(blk_embed_res, elem)

                # Free up process element
                elem.clear()

            # When we reach the end, reset the handle
            elif event == "end":
                if elem.tag == periphery_defn_tag:
                    block = None

                elif elem.tag == internal_conn_tag:
                    top_mux_conn = None

                elif elem.tag == pcr_config_tag:
                    pcr_block = None
                elif elem.tag == pcr_seq_tag:
                    seq_group_name = ""
                    seq_group_obj = None
                elif elem.tag == pcr_defn_tag:
                    # Set the default
                    if not self._set_default_pcr_defn(pcr_defn, pcr_map_default):
                        error += 1
                    pcr_defn = None
                    pcr_map_default = ""
                elif elem.tag == pcr_int_defn_tag:
                    # Set the default
                    if not self._set_default_pcr_defn(pcr_int_defn, pcr_map_default):
                        error += 1
                    pcr_int_defn = None
                    pcr_map_default = ""
                elif elem.tag == pcr_bitvec_defn_tag:
                    # Set the default
                    if not self._set_default_pcr_defn(pcr_bitvec_defn, pcr_map_default):
                        error += 1
                    pcr_bitvec_defn = None
                    pcr_map_default = ""
                elif elem.tag == mode_tag:
                    mode_obj = None
                elif elem.tag == pcr_lookup_tag:
                    if pcr_lookup is not None and common_pcr_permutation_file != "":
                        pcr_lookup.set_permutation_file(common_pcr_permutation_file)

                    common_pcr_permutation_file = ""
                    pcr_lookup = None

                elif elem.tag == embed_res_tag:
                    blk_embed_res = None

                elem.clear()

        # Check if there were more than 1 mux representation
        if mux_pattern_rep > 0 and mux_conn_rep > 0:
            if block_name != "":
                self.logger.error(
                    "Block {} has multiple mux representation".format(block_name))
            else:
                self.logger.error(
                    "Found a block with multiple mux representation")
            error += 1

        return error, self.device

    def _build_peri_block(self, xml_elem):
        '''
        Create a periphery block definition (PeripheryBlock) and
        save it to the device db.

        :param xml_elem: periphery_definition tag

        :return The created PeripheryBlock
        '''
        block_attr = xml_elem.attrib

        block = self.device.create_peri_definition(
            block_attr["name"])

        if block is None:
            self.logger.error('Unable to create periphery '
                              'definition {}'.format(
                                  block_attr["name"]))

        return block

    def _build_block_port(self, block, xml_elem):
        '''
        Create the port for the current periphery definition.

        :param block: The PeripheryBlock that this port
                            belongs to
        :param xml_elem: port tag
        '''
        valid = True

        port_attr = xml_elem.attrib
        dir_val = BlockService.port_dir_map.get(
            port_attr["direction"], "")
        type_val = port_attr.get("type", None)
        description_val = port_attr.get("description", None)
        class_val = port_attr.get("class", None)

        # Check if it is a bus port or not
        if ("left" in port_attr.keys() and
                "right" not in port_attr.keys()) or \
                ("left" not in port_attr.keys() and
                 "right" in port_attr.keys()):
            # Error when both are required to exist
            self.logger.error("Bus port requires both left and right index")
            valid = False
        elif "left" not in port_attr.keys():
            # not a bus
            valid = block.create_port(port_attr["name"],
                                      dir_val, type_val,
                                      description_val,
                                      class_val)

        else:
            # For now we assume that it is always MSB:LSB
            valid = block.create_bus_port(
                port_attr["name"], dir_val,
                int(port_attr["left"]), int(port_attr["right"]),
                type_val, description_val,
                class_val)

        return valid

    def _build_mode(self, block, xml_elem):
        '''
        Create the block mode

        :param block: The PeripheryBlock that this timing_arc
                            belongs to
        :param xml_elem: mode tag
        '''
        mode_attr = xml_elem.attrib

        mode_obj = peri_block.BlockMode(block, mode_attr["name"])
        block.add_mode(mode_obj)

        return mode_obj

    def _build_mode_port(self, mode_obj, xml_elem):
        '''
        Add the port to the mode obj
        :param mode_obj: BlockMode object
        :param xml_elem: mode_port tag
        '''
        valid = True

        pmode_attr = xml_elem.attrib

        port_name = pmode_attr.get("name", "")
        inf_name = pmode_attr.get("interface_name", "")
        description = pmode_attr.get("description", None)
        port_index = pmode_attr.get("index", "")
        inf_index = pmode_attr.get("interface_index", "")
        class_name = pmode_attr.get("class", None)
        type_str = pmode_attr.get("type", None)

        # Check the validity of the data being parsed
        # 1) index cannot exists without any name associated to it
        # 2) port_name cannot be empty (which is checked by the schema)
        if port_index != "" and port_name == "":
            self.logger.error("Mode {} port index {} must have port name specified".format(
                mode_obj.get_name(), port_index))
            valid = False

        elif inf_index != "" and inf_name == "":
            self.logger.error("Mode {} interface index {} must have port name specified".format(
                mode_obj.get_name(), inf_index))
            valid = False

        elif port_name == "":
            self.logger.error("Mode {} has empty port name".format(
                mode_obj.get_name()))
            valid = False

        if valid:
            # Creates an interface
            if not mode_obj.add_interface(inf_name, inf_index, port_name,
                                          port_index, description, class_name, type_str):
                valid = False

        return valid

    def _build_mux_pattern(self, block, xml_elem, block_file):
        '''
        :param block: The PeripheryBlock that this mux belongs
                    to
        :param xml_elem: mux_distribution tag
        :param block_file: The fullpath to the block file.
                The pattern file should be in the same directory
        '''
        from device.block_instance import PeripheryBlockInstance

        mux_attr = xml_elem.attrib
        mux_file = mux_attr["filename"]

        if "name" not in mux_attr:
            mux_name = PeripheryBlockInstance.DEFAULT_MUX_PATTERN
        else:
            mux_name = mux_attr["name"]

        # The filename given is a relative path
        # Therefore, expand and get the fullpath based on the
        # device db files
        block_dir = os.path.dirname(os.path.abspath(block_file))
        fullpath = block_dir + "/" + mux_file

        # Check if the file exists
        if os.path.exists(fullpath):
            if not block.create_block_mux_pattern(mux_name, fullpath):
                self.logger.error(
                    "Unable to parse {} mux pattern successfully".format(block.get_name()))
                return False

            return True

        self.logger.error("Block {} mux_pattern file {} not found".format(
            block.get_name(), fullpath))

        return False

    def _build_swappable_group(self, mux_block, xml_elem):
        '''
        create a swap group for the mux
        :param mux_block: Block's MuxBlock
        :param xml_elem: mux_swappable tag
        '''
        if mux_block is not None:
            mux_attr = xml_elem.attrib

            swap_group = mux_block.create_mux_group(mux_attr["weight"])
            return swap_group

        return None

    def _add_swappable_input(self, swap_group, xml_elem):
        '''
        Add the input pin info to the mux swap group
        :param swap_group: Block's MuxGroup
        :param xml_elem: mux_swappable tag
        '''
        if swap_group is not None:
            mux_attr = xml_elem.attrib

            swap_group.add_input(mux_attr["index"])

    def _build_timing_arc(self, block, xml_elem):
        '''
        Create the timing arc and set it to the cur_defn.

        :param block: The PeripheryBlock that this timing_arc
                            belongs to
        :param xml_elem: timing_arc tag
        '''
        timing_attr = xml_elem.attrib

        arc_type = BlockService.timing_arc_map.get(
            timing_attr["type"], "")

        arc = peri_block.TimingArc(timing_attr["from"],
                                   timing_attr["to"],
                                   arc_type,
                                   timing_attr["value"])

        if "pair" in timing_attr:
            arc.add_pair(timing_attr["pair"])

        if "edge" in timing_attr:
            arc.set_edge(peri_block.TimingArc.TimingArcEdge(timing_attr["edge"]))

        block.add_timing_arc(arc)

    def _build_timing_arc_mode(self, mode_obj, xml_elem):
        '''
        Create the timing arc and set it to the cur_defn.

        :param block: The PeripheryBlock that this timing_arc
                            belongs to
        :param xml_elem: timing_arc tag
        '''
        timing_attr = xml_elem.attrib

        arc_type = BlockService.timing_arc_map.get(
            timing_attr["type"], "")

        arc = peri_block.TimingArc(timing_attr["from"],
                                   timing_attr["to"],
                                   arc_type,
                                   timing_attr["value"])

        if "pair" in timing_attr:
            arc.add_pair(timing_attr["pair"])

        mode_obj.add_timing_arc(arc)

    def _build_beh_param(self, block, xml_elem):
        '''
        Create the behavioral parameter and set it to the cur_defn.

        :param block: The PeripheryBlock that this behavioral
                            parameter belongs to
        :param xml_elem: beh_parameter
        '''
        param_attr = xml_elem.attrib

        param_type = BlockService.beh_param_map.get(
            param_attr["type"], "")

        block.create_beh_parameter(param_attr["name"],
                                   param_type,
                                   param_attr["value"])

    def _build_ins_config(self, block, xml_elem, block_file):
        param_attr = xml_elem.attrib

        config_type_str = param_attr.get("config", "")
        config_file = param_attr.get("filename", "")

        # Convert the file to the fullpath:
        block_dir = os.path.dirname(os.path.abspath(block_file))
        fullpath = block_dir + "/" + config_file

        # Check if the file exists
        if os.path.exists(fullpath):
            if not block.add_ins_config(config_type_str, fullpath):
                self.logger.error(
                    "Unable to add {} config file {}".format(
                        config_type_str, fullpath))
                return False

            return True

        self.logger.error("Block {} config file {} not found".format(
            block.get_name(), fullpath))

        return False

    def _build_block_embed_res(self, block):
        return block.create_embed_res()

    def _build_embed_res(self, blk_embed_res, xml_elem):
        shared_attr = xml_elem.attrib

        blk_embed_res.add_resource(shared_attr.get("name", ""), shared_attr.get("sub_ins", ""))

    def _build_pcr_block(self, block, xml_elem):
        '''
        Create the block pcr (BlockPCR) and save to the
        periperhy block. At this point, it is an empty object.

        :param block: The PeripheryBlock that this behavioral
                            parameter belongs to
        '''

        block_pcr = blk_pcr.BlockPCR(block.get_name())

        # Save this block PCR to the block definition
        block.set_block_pcr(block_pcr)

        return block_pcr

    def _build_pcr_defn(self, pcr_block, xml_elem):
        '''
        Create the pcr map for a periphery block of type MODE.

        :param pcr_block: The BlockPCR which this pcr definition
                    belongs to
        :param xml_elem: pcr_defn tag
        '''

        pcr_attr = xml_elem.attrib

        pcr_defn = blk_pcr.PCRMap(pcr_attr["name"],
                                  int(pcr_attr["length"]),
                                  blk_pcr.PCRMap.PCRMapType.mode)

        # We save the default name first since
        pcr_map_default = pcr_attr["default"]
        pcr_block.add_pcr_map(pcr_defn)

        return (pcr_defn, pcr_map_default)

    def _build_pcr_int_defn(self, pcr_block, xml_elem):
        '''
        Create the pcr map for a periphery block of type INT.

        :param pcr_block: The BlockPCR which this pcr definition
                    belongs to
        :param xml_elem: pcr_defn tag
        '''

        pcr_attr = xml_elem.attrib

        pcr_defn = blk_pcr.PCRMap(pcr_attr["name"],
                                  int(pcr_attr["length"]),
                                  blk_pcr.PCRMap.PCRMapType.int)

        # We save the default name first since
        pcr_map_default = pcr_attr["default"]
        pcr_block.add_pcr_map(pcr_defn)

        return (pcr_defn, pcr_map_default)

    def _build_pcr_bitvec_defn(self, pcr_block, xml_elem):
        '''
        Create the pcr map for a periphery block of type bitvec.

        :param pcr_block: The BlockPCR which this pcr definition
                    belongs to
        :param xml_elem: pcr_defn tag
        '''
        pcr_attr = xml_elem.attrib

        pcr_defn = None
        if "is_lookup" in pcr_attr:
            pcr_defn = blk_pcr.PCRMap(pcr_attr["name"],
                                      int(pcr_attr["length"]),
                                      blk_pcr.PCRMap.PCRMapType.bitvec,
                                      gen_util.xmlstr2bool(pcr_attr.get("is_lookup", "false")))

        else:
            pcr_defn = blk_pcr.PCRMap(pcr_attr["name"],
                                      int(pcr_attr["length"]),
                                      blk_pcr.PCRMap.PCRMapType.bitvec)

        # We save the default name first since
        pcr_map_default = pcr_attr["default"]
        pcr_block.add_pcr_map(pcr_defn)

        return (pcr_defn, pcr_map_default)

    def _build_pcr_mode(self, pcr_defn, xml_elem):
        '''
        Create the pcr mode for a given pcr definition.

        :param pcr_defn: The PCRMap in which the mode should be
                        saved to
        :param xml_elem: pcr_mode tag
        '''

        pcr_attr = xml_elem.attrib

        # We assume that the map has been created
        added = pcr_defn.add_mode(
            pcr_attr["name"],
            pcr_attr["value"])

        # If it wasn't added, then we flag
        if not added:
            self.logger.error("PCR Mode {} with value {} was not added.".
                              format(pcr_attr["name"],
                                     pcr_attr["value"]))
        return added

    def _build_pcr_int_mode(self, pcr_defn, xml_elem):
        '''
        Create the pcr mode for a given pcr definition.

        :param pcr_defn: The PCRMap in which the mode should be
                        saved to
        :param xml_elem: pcr_int_mode tag
        '''

        pcr_attr = xml_elem.attrib
        int_min = int(pcr_attr["min"])
        int_max = int(pcr_attr["max"])

        endianness = pcr_attr["endianness"]
        # We assume that the map has been created
        added = pcr_defn.add_generic_mode(
            int_min, int_max,
            blk_pcr.PCRMap.str2endianness_map[endianness])

        # If it wasn't added, then we flag
        if not added:
            self.logger.error('PCR definition {} with min {}, max {}'
                              ' was not added.'.format(
                                  pcr_defn.get_name(),
                                  int_min, int_max))
        return added

    def _build_pcr_bitvec_mode(self, pcr_defn, xml_elem):
        '''
        Create the pcr mode for a given pcr definition.

        :param pcr_defn: The PCRMap in which the mode should be
                        saved to
        :param xml_elem: pcr_bitvec_mode tag
        '''

        pcr_attr = xml_elem.attrib

        # This is only called if it has the efxpt:pcr_bitvec_mode
        # attribute, which is used to override the default endianness
        endianness = pcr_attr["endianness"]

        format_type = pcr_attr.get("format", "binary")

        added = pcr_defn.add_bitvec_mode(
            blk_pcr.PCRMap.str2endianness_map[endianness],
            blk_pcr.PCRMap.str2format_map[format_type])

        # If it wasn't added, then we flag
        if not added:
            self.logger.error('PCR definition {} was not added.'.format(
                pcr_defn.get_name()))

        return added

    def _build_pcr_map_from_file(self, pcr_defn, xml_elem,
                                 block_file):
        '''
        Read the file that stores the mapping of the PCR value.

        :param pcr_defn: The PCRMap in which the mode should be
                        saved to
        :param xml_elem: pcr_int_mode tag
        '''
        pcr_attr = xml_elem.attrib
        pcr_file = pcr_attr["name"]

        # Check that the filename exists
        block_dir = os.path.dirname(os.path.abspath(block_file))
        fullpath = block_dir + "/" + pcr_file

        valid = False

        if os.path.exists(fullpath):
            valid = pcr_defn.set_pcr_map_file(fullpath)

        return valid

    def _set_default_pcr_defn(self, pcr_defn,
                              pcr_map_default):
        '''
        After reading the mode belonging to a pcr definition,
        we then set its default. This is done later since
        we want to check that the default value is valid
        (is part of the available modes).

        :param pcr_defn: The PCRMap that this default belongs to
        :param pcr_map_default: The default value that was read
                        when parsing the pcr_defn tag
        '''

        default_set = False
        if pcr_defn is not None:
            default_set = pcr_defn.set_default(pcr_map_default)

            if not default_set:
                self.logger.error(
                    'Unable to set default for pcr {}'.format(pcr_defn.get_name()))
        else:
            self.logger.error('PCR defn is None')

        return default_set

    def _build_sequence(self, pcr_block, xml_elem, block_name):
        seq_group_name = ""
        seq_group_obj = None
        is_added = True

        pcr_attr = xml_elem.attrib
        if "seq_group" in pcr_attr:
            seq_group_name = pcr_attr["seq_group"]

            seq_group_obj = blk_pcr.BlockPCRSequenceGroup(
                block_name, pcr_block, seq_group_name)
            is_added = pcr_block.add_group_sequence(seq_group_name, seq_group_obj)

        return seq_group_name, seq_group_obj, is_added

    def _build_block_pcr_seq(self, pcr_block, xml_elem, seq_group_name):
        '''
        Save the current map name into the PCRMap __pcr_list.
        The information has to be added in the correct order
        as the element in the list matters.

        :param pcr_block: The BlockPCR which the sequence b
                    belongs to
        :param xml_elem: pcr_defn_seq tag
        '''
        pcr_attr = xml_elem.attrib
        int_index = int(pcr_attr["index"])
        is_added = pcr_block.add_bits_sequence(
            pcr_attr["name"], int_index, seq_group_name)

        if not is_added:
            if seq_group_name == "":
                self.logger.error('Unable to add PCR definition {}:{}'
                                  ' int the sequence list'.format(
                                      int_index, pcr_attr["name"]))
            else:
                self.logger.error('Unable to add PCR definition {}:{}'
                                  ' int the sequence list for group {}'.format(
                                      int_index, pcr_attr["name"], seq_group_name))

        return is_added

    def _build_pcr_lookup(self, pcr_block):
        '''
        Create the BlockPCRLookup in the block.
        :param pcr_block: The BlockPCR which the sequence
                    belongs to
        :return the BlockPCRLookup object created or None if
                there was issue.
        '''
        lookup_obj = None

        if pcr_block is not None:
            lookup_obj = pcr_block.create_pcr_lookup()

        return lookup_obj

    def _build_pcr_permutations(self, pcr_lookup, xml_elem,
                                block_file):
        '''
        Save the permutation file that contains the
        table of the parameters that influences the PCR
        :param pcr_lookup: BlockPCRLookup object
        :param xml_elem: pcr_permutations tag
        :return True if data valid. Else, False
                The pcr permutation file if this is not associated to any
                device (devices attribute doesn't exists)
        '''
        valid = False

        pcr_attr = xml_elem.attrib

        perm_filename = pcr_attr.get("filename", None)
        device_names = pcr_attr.get("devices", None)

        common_perm_file = ""
        is_device_perm_file = False
        
        #print(f"build_pcr_perm: {self.device_name}-{perm_filename}-{device_names}")
        if perm_filename is not None:
            block_dir = os.path.dirname(os.path.abspath(block_file))
            fullpath = block_dir + "/" + perm_filename

            if os.path.exists(fullpath):
                valid = True

                if device_names is None:
                    # Save it if it's common. It will be
                    # populated to the object at the caller
                    common_perm_file = fullpath
                    
                elif device_names != "":
                    # We found a specific permutation file to save
                    device_list = device_names.split(",")
                    if self.device_name in device_list:
                        # Save this instead
                        pcr_lookup.set_permutation_file(fullpath)
                        is_device_perm_file = True

            else:
                self.logger.error("File {} does not exists".format(fullpath))
        else:
            self.logger.error("pcr_permutations filename empty")

        return valid, common_perm_file, is_device_perm_file

    def _build_pcr_fixed_settings(self, pcr_lookup, xml_elem,
                                  block_file):
        '''
        Save the file that contains the pcr settings
        :param pcr_lookup: BlockPCRLookup object
        :param xml_elem: pcr_permutations tag
        :return True if data valid. Else, False
        '''
        valid = False

        pcr_attr = xml_elem.attrib

        perm_filename = pcr_attr.get("filename", None)
        if perm_filename is not None:
            block_dir = os.path.dirname(os.path.abspath(block_file))
            fullpath = block_dir + "/" + perm_filename

            if os.path.exists(fullpath):
                if pcr_lookup.add_settings_file(fullpath):
                    valid = True
            else:
                self.logger.error("File {} does not exists".format(fullpath))
        else:
            self.logger.error("pcr_fixed_settings filename empty")

        return valid

    def _build_internal_connectivity(self, block_obj, xml_elem):
        if block_obj is not None:
            return block_obj.create_internal_connection()

        return None

    def _save_internal_file_conn(self, xml_elem, block, block_conn, block_file):
        # We already check in another function that block conn has to be
        # created (not None)
        conn_file = xml_elem.attrib

        conn_filename = conn_file.get("filename", "")

        # The filename given is a relative path
        # Therefore, expand and get the fullpath based on the
        # device db files
        block_dir = os.path.dirname(os.path.abspath(block_file))
        fullpath = block_dir + "/" + conn_filename

        # Check if the file exists
        if os.path.exists(fullpath):
            if block_conn is not None:
                # schema ensures that they are non-empty string
                block_conn.set_conn_filename(fullpath)

                # We build the connectivity
                if not block_conn.create_connectivity_graph():
                    self.logger.error(
                        "Unable to create the block {} internal connectivity successfully".format(
                            block.get_name()))
                    return False

                return True

        else:
            self.logger.error("Block {} internal connectivity file {} not found".format(
                block.get_name(), fullpath))

        return False

    def _save_buf_conn(self, xml_elem, block_conn):
        # We already check in another function that block conn has to be
        # created (not None)
        if block_conn is not None:
            buf_conn = xml_elem.attrib

            # schema ensures that they are non-empty string
            block_conn.add_buf_connection(buf_conn.get(
                "input", ""), buf_conn.get("output", ""))

    def check(self):
        """
        Apply periphery block specific check. If the creation
        is at the top-level device, that should already
        take care of checking the block definition.
        If this was reading a list of block, then the check is
        applicable to the last block parsed.

        :returns True if valid, else False
        """
        valid = True
        if self.block is not None:

            valid = self.block.check_pcr_config(self.device)

        return valid

    def validate(self, file):
        '''
        Validate that the periphery block file does not violate the
        schema.

        :param file: Full path to the timing filename
        '''

        is_pass = True

        # Validate the periphery definition
        if BlockBuilderXmlEventBased._schema_file == "":
            app_setting = aps.AppSetting()
            BlockBuilderXmlEventBased._schema_file = app_setting.app_path[
                aps.AppSetting.PathType.schema] + "/periphery_definition.xsd"

        try:
            schema = xmlschema.XMLSchema(
                BlockBuilderXmlEventBased._schema_file)
        except Exception as excp:
            self.logger.error("Exception at block: {}".format(excp))
            return False

        self.logger.info("Validating {} using schema {}".
                         format(file,
                                BlockBuilderXmlEventBased._schema_file))

        if is_pass:
            # Only check the device against schema if schema is good
            try:
                schema.validate(file)

            except xmlschema.XMLSchemaValidationError as excp:
                print("Periphery Block XMLDevice Timing Mode contains error")
                self.logger.error("XML Element : {}".format(excp.elem))
                self.logger.error("Reason : {}".format(excp.reason))
                is_pass = False

        return is_pass
