'''
Copyright (C) 2017-2019 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Jul 25, 2019

@author: maryam
'''

import enum

import common_device.ddr.ddr_device_service as ddr_svc


class BlockProperty(object):
    '''
    Represents the block property
    '''

    def __init__(self, block_name):

        self.__block_name = block_name

        self.__property_maps = {}

    def get_block_name(self):
        return self.__block_name

    def get_prop_value(self, prop_name):
        if prop_name in self.__property_maps:
            return self.__property_maps[prop_name]

        return None

    def add_property(self, name, value):
        is_added = False

        # Don't allow to have multiple values for a given property
        if name not in self.__property_maps:
            self.__property_maps[name] = value
            is_added = True

        return is_added

    def is_prop_value_list(self, prop_name):
        '''
        Check if the property value is a string delimited by comma (indicating
        a list).
        :param prop_name: Property name
        :return True if it is a string delimited by comma (i.e. "3,4,5")
        '''
        if prop_name in self.__property_maps:
            value = self.__property_maps[prop_name]

            if ',' in value:
                return True

        return False

    def get_prop_value_list(self, prop_name):
        '''
        Returns the list of prop values in string format
        :param prop_name:
        :return: List of options in the string format
        '''
        val_list = []

        if self.is_prop_value_list(prop_name):
            value = self.get_prop_value(prop_name)

            if value is not None and value != "":
                val_list = value.split(',')

        return val_list

    def __eq__(self, other):
        if isinstance(other, BlockProperty):
            return self.__block_name == other.__block_name and \
            self.__property_maps == other.__property_maps
        return False

    def __hash__(self):
        return super().__hash__()

class DeviceBlockConfig(object):
    '''
    Captures the block configuration that
    is assocated to specific device.
    '''

    # The list of acceptable block property name
    class BlockPropertyType(enum.Enum):
        '''
        Enum that represents the acceptable defined block
        property names
        '''
        bprop_ddr_config_width = 1
        bprop_lvds_gpio_out_max_bank_usage = 2
        bprop_lvds_tx_ds_default = 3
        bprop_lvds_tx_ds_list = 4
        bprop_gpio_io_std_default = 5
        bprop_gpio_io_std_list = 6
        bprop_hsio_gpio_io_std_default = 7
        bprop_hsio_gpio_io_std_list = 8
        bprop_hsio_lvds_io_std_default = 9
        bprop_hsio_lvds_io_std_list = 10
        bprop_hsio_mipi_dphy_io_std_default = 11
        bprop_hsio_mipi_dphy_io_std_list = 12
        bprop_pll_outclk_interface_inverted = 13
        bprop_seu_automatic_mode_enabled = 14
        bprop_hsio_mipi_lane_static_delay_default_085v_data = 15
        bprop_hsio_mipi_lane_static_delay_default_095v_data = 16
        bprop_hsio_lvds_static_delay_default_085v_data = 17
        bprop_hsio_lvds_static_delay_default_095v_data = 18
        bprop_ddr_pll_clkin_list = 19
        bprop_ddr_memory_type = 20
        bprop_lvds_delay_setting_default = 21
        bprop_hvio_poc_unbonded_bank_mode = 22
        bprop_hsio_sstl_hstl_input_unsupported = 23
        bprop_ddr_memory_density_default = 24

    bprop2str_map = \
        {
            BlockPropertyType.bprop_ddr_config_width: ddr_svc.DDRService.HEADER_CTRL_WIDTH_NAME,
            BlockPropertyType.bprop_lvds_gpio_out_max_bank_usage: "Max_bank_usage",
            BlockPropertyType.bprop_lvds_tx_ds_default: "TX_DS_Default",
            BlockPropertyType.bprop_lvds_tx_ds_list: "TX_DS_List",
            BlockPropertyType.bprop_lvds_delay_setting_default: "Delay_setting_default",
            BlockPropertyType.bprop_gpio_io_std_default: "IO_standard_default",
            BlockPropertyType.bprop_gpio_io_std_list: "IO_standard_list",
            BlockPropertyType.bprop_hsio_gpio_io_std_default: "hsio_gpio_IO_standard_default",
            BlockPropertyType.bprop_hsio_gpio_io_std_list: "hsio_gpio_IO_standard_list",
            BlockPropertyType.bprop_hsio_lvds_io_std_default: "hsio_lvds_IO_standard_default",
            BlockPropertyType.bprop_hsio_lvds_io_std_list: "hsio_lvds_IO_standard_list",
            BlockPropertyType.bprop_hsio_mipi_dphy_io_std_default: "hsio_mipi_dphy_IO_standard_default",
            BlockPropertyType.bprop_hsio_mipi_dphy_io_std_list: "hsio_mipi_dphy_IO_standard_list",
            BlockPropertyType.bprop_pll_outclk_interface_inverted: "pll_outclk_interface_inverted",
            BlockPropertyType.bprop_seu_automatic_mode_enabled: "seu_automatic_mode_enabled",
            BlockPropertyType.bprop_hsio_mipi_lane_static_delay_default_085v_data: "hsio_mipi_lane_static_delay_default_085v_data",
            BlockPropertyType.bprop_hsio_mipi_lane_static_delay_default_095v_data: "hsio_mipi_lane_static_delay_default_095v_data",
            BlockPropertyType.bprop_hsio_lvds_static_delay_default_085v_data: "hsio_lvds_static_delay_default_085v_data",
            BlockPropertyType.bprop_hsio_lvds_static_delay_default_095v_data: "hsio_lvds_static_delay_default_095v_data",
            BlockPropertyType.bprop_ddr_pll_clkin_list: "ddr_pll_clkin_list",
            BlockPropertyType.bprop_ddr_memory_type: ddr_svc.DDRService.HEADER_MEMORY_TYPE_NAME,
            BlockPropertyType.bprop_hvio_poc_unbonded_bank_mode: "Unbonded_Bank_PCR_MODE_18V",
            BlockPropertyType.bprop_hsio_sstl_hstl_input_unsupported: "hsio_sstl_hstl_input_unsupported",
            BlockPropertyType.bprop_ddr_memory_density_default: "ddr_memory_density_default"
        }

    str2bprop_map = \
        {
            ddr_svc.DDRService.HEADER_CTRL_WIDTH_NAME: BlockPropertyType.bprop_ddr_config_width,
            "Max_bank_usage": BlockPropertyType.bprop_lvds_gpio_out_max_bank_usage,
            "TX_DS_Default": BlockPropertyType.bprop_lvds_tx_ds_default,
            "Delay_setting_default": BlockPropertyType.bprop_lvds_delay_setting_default,
            "TX_DS_List": BlockPropertyType.bprop_lvds_tx_ds_list,
            "IO_standard_default": BlockPropertyType.bprop_gpio_io_std_default,
            "IO_standard_list": BlockPropertyType.bprop_gpio_io_std_list,
            "hsio_gpio_IO_standard_default" : BlockPropertyType.bprop_hsio_gpio_io_std_default,
            "hsio_gpio_IO_standard_list": BlockPropertyType.bprop_hsio_gpio_io_std_list,
            "hsio_lvds_IO_standard_default": BlockPropertyType.bprop_hsio_lvds_io_std_default,
            "hsio_lvds_IO_standard_list": BlockPropertyType.bprop_hsio_lvds_io_std_list,
            "hsio_mipi_dphy_IO_standard_default": BlockPropertyType.bprop_hsio_mipi_dphy_io_std_default,
            "hsio_mipi_dphy_IO_standard_list": BlockPropertyType.bprop_hsio_mipi_dphy_io_std_list,
            "pll_outclk_interface_inverted": BlockPropertyType.bprop_pll_outclk_interface_inverted,
            "seu_automatic_mode_enabled": BlockPropertyType.bprop_seu_automatic_mode_enabled,
            "hsio_mipi_lane_static_delay_default_085v_data": BlockPropertyType.bprop_hsio_mipi_lane_static_delay_default_085v_data,
            "hsio_mipi_lane_static_delay_default_095v_data": BlockPropertyType.bprop_hsio_mipi_lane_static_delay_default_095v_data,
            "hsio_lvds_static_delay_default_085v_data": BlockPropertyType.bprop_hsio_lvds_static_delay_default_085v_data,
            "hsio_lvds_static_delay_default_095v_data": BlockPropertyType.bprop_hsio_lvds_static_delay_default_095v_data,
            "ddr_pll_clkin_list": BlockPropertyType.bprop_ddr_pll_clkin_list,
            ddr_svc.DDRService.HEADER_MEMORY_TYPE_NAME: BlockPropertyType.bprop_ddr_memory_type,
            "Unbonded_Bank_PCR_MODE_18V": BlockPropertyType.bprop_hvio_poc_unbonded_bank_mode,
            "hsio_sstl_hstl_input_unsupported": BlockPropertyType.bprop_hsio_sstl_hstl_input_unsupported,
            "ddr_memory_density_default": BlockPropertyType.bprop_ddr_memory_density_default
        }

    def __init__(self):
        '''
        Constructor
        '''

        # A map of block name to the properties object
        self.__block_to_properties = {}

    def get_block_property(self, block_name):

        if block_name in self.__block_to_properties:
            return self.__block_to_properties[block_name]

        return None

    def set_block_property(self, block_name, bprop_obj):
        is_added = False

        if bprop_obj is not None:
            if block_name not in self.__block_to_properties:
                self.__block_to_properties[block_name] = bprop_obj
                is_added = True

        return is_added

    def __eq__(self, other):
        if isinstance(other, DeviceBlockConfig):
            return self.__block_to_properties == other.__block_to_properties
        return False

    def __hash__(self):
        return super().__hash__()