'''
Copyright (C) 2013-2017 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Stores the information on the periphery so that it could be used
with the periphery tool.

@author: Maryam
Date created:  9/19/2017

'''
from __future__ import annotations
import os
import re
import traceback
import xml.etree.ElementTree as et
from typing import TYPE_CHECKING, Optional, Dict

# 3rd party library to display pretty table
from prettytable import PrettyTable

import util.gen_util as pt_util
from util.singleton_logger import Logger
import util.excp as app_excp
import util.plugin_map as plmap

import device.excp as dev_excp
import device.package as dev_pkg
import device.block_definition as peri_block
import device.block_instance as peri_instance
from device.core_clock_delay import DeviceCoreClockDelay

if TYPE_CHECKING:
    from device.timing_model import TimingModel

class PeripheryDevice(object):
    '''
    A periphery device DB that represents the selected device.
    There are a few main information found in a device:

    (A) Die
        1) Device name: This should match with Efinity
        2) The list of blocks definition supported for a device
        3) List of instances in a device
        4) Resource map of used/unused instances
    (B) Packaging
        1) pad-pin map
    (C) Timing Model
        1) Delay values for the specific device (speedgrade dependent
           too)
    (D) Resource map
        Indicates the list of instances that are enabled in a device

    '''
    # do nothing in the initializer for now

    def __init__(self, name_in_file):
        '''
        Constructor
        '''
        self.__logger = Logger

        # The device db file that was read
        self.__device_db_file = None

        # The device that is being used. This is the name
        # passed with load_device
        self.__device_name = ""

        # The device that is saved in the device filename
        self.__device_mapped_name = name_in_file

        # We need to point to the package information of DevicePackage type
        self.__device_package = None

        # Store the periphery block definition of each type. (key: type/name)
        self.__periphery_definition: Dict[str, peri_block.PeripheryBlock] = {}

        # This container should be efficient since it will outgrow when
        # there are many periphery for a device. The order of the periphery
        # matters? since the top-level bitstream is constructed by the order
        # of the periphery (key is the instance name)
        self.__periphery_instance = {}

        # Store the bits sequence of the entire periphery (not the value but
        # the pattern). This is a map of index to the intance name.  The index
        # is allowed to not be in consecutive order. A warning will show up
        # when checking.
        self.__pcr_bits_sequence = {}

        # Timing model supported
        self.__timing_model:Optional[TimingModel] = None

        # Resource map
        self.__resources = None

        # IO pads (DeviceIO)
        self.__io_pad = None

        # Global Clocks (GlobalClocks)
        self.__global_clocks = None

        # Secondary connection functions. which is a
        # map of function names to the list of instance names
        self.__secfunc_to_insnames = {}
        self.__insname_to_secfunc = {}

        # Config model
        self.__config_model = None

        # Die common timing parameters
        self.__timing_parameters = None

        # The width and height are optional in schema.
        # However, for Titanium, it has to be compulsory
        # in order to write out proper timing sdc constraint/timing report

        # Die size: width (max x)
        self.__width = 0

        # Die size: height (max y)
        self.__height = 0

        # List of clock region names that the
        # device has based on available resources
        self.__clock_region_names = []

        self.__hier_to_flat_ins_map = {}
        self.__flat_to_hier_ins_map = {}

        # Device specific block configurations
        self.__block_config = None

        # Store the MIPI DPHY Rx grouping
        self.__mipi_dphy_rx_group = None

        # Store the internal resource configuration for
        # instances that are not in resource map but it
        # has very specific configuration settings
        self.__internal_config = None

        # Store DDR pin swap information
        self.__pin_swap: Optional[Dict[str, str]] = None

        # Store the table of core clock network delay
        # which is based on the timing model. Since PT
        # has fixed timing model, we are keeping only the
        # table associated to the choosen timing model
        self.__core_clk_delay: DeviceCoreClockDelay = None

        # Supported block service types (for those that have variations across
        # devices)
        self.pll_type = None
        self.gpio_type = None
        self.osc_type = None
        self.clkmux_type = None
        self.ddr_type = None
        self.mipi_rx_type = None
        self.mipi_tx_type = None
        self.rclkmux_types = []

        # Keep a list of resources that are shared but of different block type
        # so that we don't consider HSIO instances (Tesseract) in the list or
        # LVDS instances (Trion) in the list.  The info saved is one way but
        # the implication is both ways. For example:
        # SPI_FLASH0 -> GPIOL_PN_XX, GPIOL_P_X, GPIOL_N_X
        # but it also means that GPIOL_PN_XX is shared with SPI_FLASH0 and
        # vice versa
        self._shared_instances = {}

    def get_clkmux_type(self):
        return self.clkmux_type

    def get_pll_type(self):
        return self.pll_type

    def get_pin_swap(self):
        return self.__pin_swap

    def set_pin_swap(self, pin_swap_mapping: Dict[str, str]):
        self.__pin_swap = pin_swap_mapping

    def is_sample_device(self):
        '''
        Depending on the device name, if it ends with "ES", it's considered
        a sample device
        :return:
        '''
        # DEVINFRA-1012: Exclude Ti135N676ES since it's exactly the same
        # characteristic as real device in periphery support although its named "ES"
        if self.__device_name == "Ti135N676ES":
            return False
        # Special case for Titanium 60 fullbond. TX60 is based on ES and Ti60 is MP
        elif self.__device_name.endswith("ES") or self.__device_name == "TX60":
            return True

        return False

    def set_mipi_dphy_rx_group(self, mipi_rx_group):
        self.__mipi_dphy_rx_group = mipi_rx_group

    def get_mipi_dphy_rx_group(self):
        return self.__mipi_dphy_rx_group

    def set_internal_res_config(self, res_config):
        self.__internal_config = res_config

    def get_internal_res_config(self):
        return self.__internal_config

    def identify_supported_block_types(self, plugin_map=None):
        """
        Save the list of DeviceDBService.BlockType that
        are supported in this device into sel.__supported_blocks.
        This will be used to later dictate where to get the services
        from for block that may varies between devices (i.e. GPIO, PLL).

        :param plugin_map: The plugin map object
        """
        if plugin_map is None:
            tmp_plugin_map = plmap.PluginMap.build_from_default_file()
        else:
            tmp_plugin_map = plugin_map

        # Identify the special block types that may vary depending on device
        self.__identify_special_block_types(tmp_plugin_map)

    def __identify_special_block_types(self, plugin_map):
        """
        Only used for identifying the blocks that can vary depending on device.
        This would align with the device block services associated to it.
        :param plugin_map: The PluginMap that has been built
        """
        #from device.db_interface import DeviceDBService

        if plugin_map is not None:
            # GPIO
            self._determine_gpio_type(plugin_map)

            # PLL
            # CLKMUX: TEMP, we use the PLL to determine
            self._determine_pll_clkmux_type(plugin_map)

            # OSC
            self._determine_osc_type(plugin_map)

            # DDR
            self._determine_ddr_type(plugin_map)

            # MIPI TX/RX
            self._determine_mipi_type(plugin_map)

            # RCLKMUX
            self._determine_rclkmux_type(plugin_map)

            self.__logger.debug(
                'Device {} gpio_type: {} pll_type: {} osc_type: {} clkmux_type: {}'\
                'ddr_type: {} mipi_rx_type: {} mipi_tx_type: {}'.format(
                    self.__device_name, self.gpio_type, self.pll_type,
                    self.osc_type, self.clkmux_type, self.ddr_type,
                    self.mipi_tx_type, self.mipi_rx_type))

        else:
            from device.db_interface import DeviceDBService

            # Assume that it's all the block type (with default gpio and pll)
            self.gpio_type = DeviceDBService.BlockType.GPIO
            self.pll_type = DeviceDBService.BlockType.PLL
            self.osc_type = DeviceDBService.BlockType.OSC
            self.clkmux_type = DeviceDBService.BlockType.CLKMUX
            self.ddr_type = DeviceDBService.BlockType.DDR
            self.mipi_rx_type = DeviceDBService.BlockType.MIPI_RX
            self.mipi_tx_type = DeviceDBService.BlockType.MIPI_TX
            self.rclkmux_types = [DeviceDBService.BlockType.RCLKMUX]

    def _determine_gpio_type(self, plugin_map):
        from device.db_interface import DeviceDBService

        group_type = plugin_map.get_group_by_type_name(
            self.__device_name, "gpio")
        if group_type == plmap.PluginMap.GroupType.tx60:
            self.gpio_type = DeviceDBService.BlockType.HVIO
        else:
            self.gpio_type = DeviceDBService.BlockType.GPIO

    def _determine_pll_clkmux_type(self, plugin_map):
        from device.db_interface import DeviceDBService

        # TODO: Determine a proper way to tell what kind of clkmux
        group_type = plugin_map.get_group_by_type_name(
            self.__device_name, "pll")
        if group_type == plmap.PluginMap.GroupType.tx60 or \
            group_type == plmap.PluginMap.GroupType.tx180 or \
                group_type == plmap.PluginMap.GroupType.tx375:
            self.pll_type = DeviceDBService.BlockType.PLL_COMPLEX
            if group_type == plmap.PluginMap.GroupType.tx375:
                self.pll_type = DeviceDBService.BlockType.EFX_FPLL_V1

            clkmux_group_type = plugin_map.get_group_by_type_name(
                self.__device_name, "clkmux")

            if clkmux_group_type == plmap.PluginMap.GroupType.tx375:
                self.clkmux_type = DeviceDBService.BlockType.CLKMUX_V4
            elif clkmux_group_type == plmap.PluginMap.GroupType.tx180:
                self.clkmux_type = DeviceDBService.BlockType.CLKMUX_COMPLEX
            else:
                self.clkmux_type = DeviceDBService.BlockType.CLKMUX_ADV
        else:
            self.pll_type = DeviceDBService.BlockType.PLL
            self.clkmux_type = DeviceDBService.BlockType.CLKMUX

    def _determine_osc_type(self, plugin_map):
        from device.db_interface import DeviceDBService

        group_type = plugin_map.get_group_by_type_name(
            self.__device_name, "osc")
        if group_type == plmap.PluginMap.GroupType.tx60:
            self.osc_type = DeviceDBService.BlockType.OSC_ADV

        elif group_type != plmap.PluginMap.GroupType.unknown:
            self.osc_type = DeviceDBService.BlockType.OSC
        else:
            # If it was unknown, check if it is possibly hposc
            group_type = plugin_map.get_group_by_type_name(
                self.__device_name, "hposc")

            if group_type == plmap.PluginMap.GroupType.tx60:
                self.osc_type = DeviceDBService.BlockType.HPOSC
            else:
                self.osc_type = DeviceDBService.BlockType.OSC

    def _determine_ddr_type(self, plugin_map):
        from device.db_interface import DeviceDBService

        group_type = plugin_map.get_group_by_type_name(
            self.__device_name, "ddr")
        if group_type == plmap.PluginMap.GroupType.tx180:
            self.ddr_type = DeviceDBService.BlockType.DDR_ADV
        elif group_type == plmap.PluginMap.GroupType.an35:
            self.ddr_type = DeviceDBService.BlockType.DDR
        else:
            # There is a case where the device is using DDR TX60 information
            # but pluginmap.csv excluded ddr. So, we need to still detect
            # DDR type as DDR_ADV using gpio as a reference
            gpio_group_type = plugin_map.get_group_by_type_name(
                self.__device_name, "gpio")

            if gpio_group_type == plmap.PluginMap.GroupType.tx60:
                self.ddr_type = DeviceDBService.BlockType.DDR_ADV
            else:
                self.ddr_type = DeviceDBService.BlockType.DDR

    def _determine_mipi_type(self, plugin_map):
        from device.db_interface import DeviceDBService

        group_type = plugin_map.get_group_by_type_name(
            self.__device_name, "mipi")
        if group_type == plmap.PluginMap.GroupType.tx180:
            self.mipi_rx_type = DeviceDBService.BlockType.MIPI_RX_ADV
            self.mipi_tx_type = DeviceDBService.BlockType.MIPI_TX_ADV
        else:
            # There is a case where the device is using MIPI TX180 but it does
            # not have any of it bonded out. So, we need to still detect
            # the type as Tx180 (i.e. Ti180F529) using the GPIO type as
            # reference
            gpio_group_type = plugin_map.get_group_by_type_name(
                self.__device_name, "gpio")
            if gpio_group_type == plmap.PluginMap.GroupType.tx60:
                self.mipi_rx_type = DeviceDBService.BlockType.MIPI_RX_ADV
                self.mipi_tx_type = DeviceDBService.BlockType.MIPI_TX_ADV
            else:
                self.mipi_rx_type = DeviceDBService.BlockType.MIPI_RX
                self.mipi_tx_type = DeviceDBService.BlockType.MIPI_TX

    def _determine_rclkmux_type(self, plugin_map):
        from device.db_interface import DeviceDBService

        # HACK: Use clkmux to determine it is Ti375 since
        # not all Ti375 has quad available
        group_type = plugin_map.get_group_by_type_name(
            self.__device_name, "clkmux")

        if group_type == plmap.PluginMap.GroupType.tx375:
            self.rclkmux_types = [DeviceDBService.BlockType.RCLKMUX_V2_21, DeviceDBService.BlockType.RCLKMUX_V2_41]
        else:
            self.rclkmux_types = [DeviceDBService.BlockType.RCLKMUX]

    def set_block_configuration(self, dev_block_config):
        '''
        Save the device block configuration
        '''
        self.__block_config = dev_block_config

    def get_block_configuration(self):
        return self.__block_config

    def set_common_timing(self, timing_param):
        '''
        Create the timing parameter that's common and not
        block specific.
        :param timing_param: CommonTiming object
        '''
        # We don't check if it already exists because by
        # right schema will validate
        self.__timing_parameters = timing_param

    def get_common_timing(self):
        return self.__timing_parameters

    def set_config_model(self, cmodel):
        '''
        Save the config model to the device db. If there is
        already a config model, it will then be
        replaced.

        :param cmodel: ConfigModel object
        '''
        if self.__config_model is not None:
            self.__logger.warning("Overwriting existing configuration model")

        self.__config_model = cmodel

    def get_config_model(self):
        '''
        :return the config model
        '''
        return self.__config_model

    def get_block_names(self):
        '''
        Return a list of periphery block names defined in
        the database.

        :return a list of periphery block names
        '''

        return sorted(self.__periphery_definition.keys())

    # Create the secondary connection functions mapping
    def create_secondary_conn_functions(self):
        '''
        This is used to create the secondary connection function
        map. It should be called once after a successful parsing.
        '''
        used_instances = self.__resources.get_instances()

        for ins_name in self.__periphery_instance:

            ins_obj = self.__periphery_instance[ins_name]
            sec_con_list = ins_obj.get_secondary_connection(ins_name)

            # Only iterate on instances that are part of the
            # resource map
            if ins_name not in used_instances:
                # self.__logger.debug(
                #    "SKIP non-used instance {}".format(ins_name))
                continue

            # TODO: Do we need to filter only external function?
            # We don't filter out the function whether internal or external
            if sec_con_list:
                for function in sec_con_list:
                    #self.__logger.debug("Instance {}: Sec Func: {}".format(
                    #    ins_name, function))

                    # Add this to the __secfunc_to_insnames
                    if function in self.__secfunc_to_insnames:
                        ins_list = list(
                            self.__secfunc_to_insnames[function])
                        if ins_name not in ins_list:
                            ins_list.append(ins_name)
                            self.__secfunc_to_insnames[function] = ins_list
                    else:
                        ins_list = [ins_name]
                        self.__secfunc_to_insnames[function] = ins_list

                    # Add this to the __insname_to_secfunc
                    if ins_name not in self.__insname_to_secfunc:
                        func_list = [function]
                        self.__insname_to_secfunc[ins_name] = func_list
                    else:
                        # For now an instance does not have more than one
                        # secondary connections
                        func_list = self.__insname_to_secfunc[ins_name]
                        if function not in func_list:
                            func_list.append(function)
                            self.__insname_to_secfunc[ins_name] = func_list

    def has_secondary_connection(self, ins_name, mode_name=""):
        '''
        Check if the instance has any secondary connection.
        In the case that the instance is of a block with multiple mode,
        the secondary connection may be associated to only specific modes.
        Hence, if we want to know exactly based on a mode, the mode
        name has to be specified.  If the specified mode name is invalid
        or we cannot find the referenced block, the secondary connection
        on the instance is assume to be nullified (False).

        :param ins_name: The device instance name
        :param mode_name: The instance mode name queried
        :return: True if there is secondary connection associated to the instance
                or if the mode has a secondary connection (when mode_name is
                specified).
        '''
        found = False
        ins_obj = None

        if ins_name != "" and ins_name in self.__insname_to_secfunc:
            # If it was very specific based on modes
            ins_obj = self.find_instance(ins_name)

            if mode_name == "":
                found = True
            else:

                if ins_obj is not None:
                    # Get the reference block and check if the mode name exists
                    ref_block = ins_obj.get_block_definition()

                    if ref_block is not None:
                        if ref_block.is_mode_exists(mode_name):
                            # Get the pins associated to the mode
                            pin_name_to_conn_map = ins_obj.get_inf_pins_by_mode(
                                mode_name)

                            if pin_name_to_conn_map:
                                # Check there is any secondary connection
                                # associated
                                found_sec_conn = False
                                for pin_obj in pin_name_to_conn_map.values():
                                    if pin_obj.has_secondary_connection():
                                        found_sec_conn = True
                                        break

                                if found_sec_conn:
                                    found = True
                                else:
                                    # Reset it if not found
                                    ins_obj = None

        return found, ins_obj

    def get_secondary_connection(self, ins_name, mode_name=""):
        sec_conn = []

        if ins_name != "" and ins_name in self.__insname_to_secfunc:
            if mode_name == "":
                sec_conn = self.__insname_to_secfunc[ins_name]

            else:
                ins_obj = self.find_instance(ins_name)

                if ins_obj is not None:
                    # Get the reference block and check if the mode name exists
                    ref_block = ins_obj.get_block_definition()

                    if ref_block is not None:
                        if ref_block.is_mode_exists(mode_name):
                            # Get the pins associated to the mode
                            pin_name_to_conn_map = ins_obj.get_inf_pins_by_mode(
                                mode_name)

                            if pin_name_to_conn_map:
                                # Check there is any secondary connection
                                # associated
                                found_sec_conn = False
                                for pin_obj in pin_name_to_conn_map.values():
                                    if pin_obj.has_secondary_connection():
                                        tmp_functions = pin_obj.get_secondary_functions()

                                        if tmp_functions:
                                            # Combine and remove duplicates
                                            set_top = set(sec_conn)
                                            set_cur = set(tmp_functions)
                                            in_cur_not_in_top = set_cur - set_top

                                            sec_conn = sec_conn + \
                                                list(in_cur_not_in_top)

                                            self.__logger.debug("secondary conn for {} with mode {}: {}".format(
                                                ins_name, mode_name, ",".join(sec_conn)))

        # Sort the list
        if sec_conn:
            sec_conn = sorted(sec_conn)

        return sec_conn

    def get_secondary_function_names(self):
        '''
        Return the list of secondary function names.
        It could be empty if none of the instances have any secondary
        connections.

        :return List of secondary connection function names
        '''
        return list(self.__secfunc_to_insnames.keys())

    def get_secondary_func_to_instance(self):
        '''
        Return a map of the function to the list of instances

        :return Map of function name to list of instance names
        '''
        return self.__secfunc_to_insnames

    def get_secondary_function_instance_names(self, function):
        '''
        Return the list of instances associated to the function specified.

        :param function: The secondary function to be searched
        :return a list of instance names associated to it
        '''
        ins_list = []

        if self.__secfunc_to_insnames:
            if function in self.__secfunc_to_insnames:
                return self.__secfunc_to_insnames[function]

        return ins_list

    def create_package(self, package_name):
        '''
        Create a package with the passed name and save it to
        the device.

        :param package_name: Package name to be created

        :return The created DevicePackage
        '''

        if self.__device_package is not None:
            self.__logger.warning("Overwriting existing package")

        self.__device_package = dev_pkg.DevicePackage(package_name)

        return self.__device_package

    def set_device_file(self, file):
        '''
        Save the device filename

        :param file: The device filename
        '''

        self.__device_db_file = file

    def get_device_file(self):
        '''
        Return the filename
        :return the device filename
        '''
        return self.__device_db_file

    def get_device_dir(self):
        '''
        Return the directory of where the device file resides
        :return the absolute path to the device directory.
        '''

        dir_name = ""
        if self.__device_db_file != "":
            if not os.path.isdir(self.__device_db_file):
                dir_name = os.path.dirname(self.__device_db_file)
            else:
                dir_name = self.__device_db_file

        return dir_name

    def set_device_name(self, device_name):
        '''
        Set the device name
        '''

        self.__device_name = device_name

    def get_device_name(self):
        '''
        Get the device name

        :return device name
        '''

        return self.__device_name

    def get_device_mapped_name(self):
        '''
        Get the device mapped name (saved in the device file)

        :return device mapped name
        '''

        return self.__device_mapped_name

    def get_io_pad(self):
        '''
        Return the io pad info object
        :return DeviceIO object
        '''

        return self.__io_pad

    def set_timing_model(self, dev_tmodel):
        '''
        Set the device timing model to the passed object

        :param dev_tmodel: TimingModel object
        '''

        # We don't check if the passed tmodel is not None
        if self.__timing_model is not None:
            self.__logger.warning("Overwriting existing timing model")

        self.__timing_model = dev_tmodel

    def get_timing_model(self):
        '''
        Get the device timing model.

        :return tmode: TimingModel object
        '''

        return self.__timing_model

    def get_current_timing_model_name(self) -> str:
        '''

        :return: name of the currently assigned timing model model. An empty
            string is returned if it's not set
        '''
        if self.__timing_model is not None:
            return self.__timing_model.get_selected_model_name()

        return ""

    def assign_timing_model(self, tmodel_name=""):
        '''
        Set the timing model active to the selected
        name. If name is empty, then we choose the
        first one.

        :param tmodel_name: Timing model name (empty if
                    auto set to the first one)
        :return True if was successfully set.
        '''
        is_successful = False

        # In this case, it means that A device will always have
        # timing model (compulsory)
        if self.__timing_model is not None:
            is_successful = \
                self.__timing_model.set_selected_model(tmodel_name)
        else:
            # Let it pass if timing model isn't available
            is_successful = True

        return is_successful

    def assign_config_model(self, cmodel_name="", cmodel_width=0):
        '''
        Set the config model active to the selected
        name. If name is empty, then we choose the
        default one.

        :param cmodel_name: Config model name (empty if
                    auto set to the first one)
        :param cmodel_width: Config model width in integer
        :return True if was successfully set.
        '''
        is_successful = False

        if self.__config_model is not None:
            is_successful = \
                self.__config_model.set_selected_mode(
                    cmodel_name, cmodel_width)

        return is_successful

    def populate_core_clock_network_delay_table(self):
        '''
        Populate the core clock network delay table to be used
        later for writing timing data. This is called by demand once
        we call timing writer. The table will stays unless a different
        timing model is requested.
        '''
        if self.__core_clk_delay is None:
            self.__core_clk_delay = DeviceCoreClockDelay(self)

        self.__core_clk_delay.populate_clock_network_delay_table()
        
        return self.__core_clk_delay
                
    #def get_core_clock_network_delay(self):
    #    return self.__core_clk_delay
    
    def set_resources(self, dev_resource):
        '''
        Set the device resources to the passed object

        :param resource: UseResources object
        '''

        # We dont' check if the passed resource is not None
        if self.__resources is not None:
            self.__logger.warning("Overwriting existing resource map")

        self.__resources = dev_resource

    def create_peri_definition(self, name):
        '''
        Create the PeripheryBlock definition (empty) based on name.
        It is then saved to the db, provided that there is no
        existing block with the same name

        :param name: Name of the periphery block

        :return The created PeripheryBlock
        '''

        if name not in self.__periphery_definition:
            # self.__logger.debug('Creating peri block {}'.format(
            #    name))
            block = peri_block.PeripheryBlock(name)
            self.__periphery_definition[name] = block
            return block

        return None

    def add_definition(self, block):
        '''
        Add the passed PeripheryBlock object to the db. This is
        an alternative to create_peri_definition

        :param block: PeripheryBlock object

        :return True, if the block was added
        '''

        if block is not None:
            if block.get_name() not in self.__periphery_definition:
                self.__periphery_definition[block.get_name()] = block
                return True
            else:
                self.__logger.debug(
                    "Block {} already exists in DB".format(
                        block.get_name()))
        return False

    def create_peri_instance(self, ref, name):
        '''
        Create a periphery instance based on the passed info
        and saved it to the db

        :param ref: The reference block name
        :param name: The name of the instance

        :return The created PeripheryBlockInstance or None (if
                it wasn't created)
        '''

        # Check if the ref block has already been defined
        if ref in self.__periphery_definition:
            # Check that there hasn't been an instance with that name
            if name not in self.__periphery_instance:
                # self.__logger.debug('Create peri ins {} of {}'.format(
                #    name, ref))
                instance = peri_instance.PeripheryBlockInstance(name, ref)
                self.__periphery_instance[name] = instance

                return instance
            else:
                self.__logger.debug("Instance {} already exists in DB".format(
                    name))
        else:
            self.__logger.debug("Referenced block {} has not been saved in DB".
                                format(ref))

        return None

    def add_instance(self, instance):
        '''
        Add the instance PeripheryBlockInstance object to the db.

        :param instance: The PeripheryBlockInstance to be added to db

        :return True if it was added. False otherwise
        '''

        if instance is not None:
            if instance.get_name() not in self.__periphery_instance:
                self.__periphery_instance[instance.get_name()] = instance
                return True
            else:
                self.__logger.debug("Instance {} already exists in DB".format(
                    instance.get_name()))

        return False

    def set_io_pad(self, device_io):
        '''
        Save the device_io (DeviceIO) to the device db.

        :param device_io: DeviceIO object

        '''

        if self.__io_pad is not None:
            self.__logger.warning("Overwriting existing io_pad")

        self.__io_pad = device_io

    def set_global_clocks(self, global_clk):
        '''
        Save the global_clk (GlobalClocks) to the device db.

        :param global_clk: GlobalClocks object
        '''

        if self.__global_clocks is not None:
            self.__logger.warning("Overwriting existing global clocks")

        self.__global_clocks = global_clk

    def get_global_clocks(self):
        '''
        Get the global clocks obj.

        :return global clocks of type GlobalClocks if exists.
        '''

        return self.__global_clocks

    def add_pcr_bits_sequence(self, ins_name, index):
        '''
        Append the specified ins_name to the pcr_bits_sequence list.
        Don't check if the name already exists or not. Just allow it.

        :param ins_name: Instance name
        :param index: The index of the sequence in integer

        :return added that indicates if the sequence was added (True)
        '''

        added = False

        # Check that there is no sequence assigned to the
        # specified index
        if index not in self.__pcr_bits_sequence:
            self.__pcr_bits_sequence[index] = ins_name
            added = True
        else:
            self.__logger.error('Cannot add pcr instance {} because '
                                'index {} has already been assigned to '
                                'instance {}'.format(
                                    ins_name, index,
                                    self.__pcr_bits_sequence[index]))

        return added

    def reset_pcr_bits_sequence(self):
        '''
        Clear the PCR bits sequence list.
        '''

        self.__pcr_bits_sequence = {}

    def is_pcr_sequence_exists(self):
        if self.__pcr_bits_sequence and\
                len(self.__pcr_bits_sequence) > 0:
            return True

        return False

    def generate_pcr_sequence(self, root):
        '''
        Get the top-level device PCR sequence and add it to the
        ElementTree root. This is used later when generating
        the LPF.

        :param root: The root ElementTree where the attribute is to be
                    added to

        :return block_names: a list of PCR blocks name available
                    in the current device
                ins_names: a list of instance names that are part of
                    the sequence
        '''
        seq_size = str(len(self.__pcr_bits_sequence))
        sequence_elem = et.SubElement(root, "efxpt:sequence", size=seq_size)

        block_names = []
        ins_names = []

        for key, value in sorted(self.__pcr_bits_sequence.items()):
            ins_obj = self.find_instance(value)

            # The instance printed does not have to be filtered
            # from resource map. Since the PCR is die dependent.
            # Even if the IO is not bonded out, the PCR setting
            # (default) will still need to be provided.
            ref_name = ''
            seq_ref_name = ""

            if ins_obj is not None:
                ref_name = ins_obj.get_ref_name()
                seq_ref_name = ref_name
            else:
                # If the value is a block instance with the
                # sequence group: <Inst>:<seq_group>
                # such as DDR_0:DDR_PCR, DDR_0:DDR_CFG
                if value.find(":") != -1:
                    pcr_blk_ins_name, pcr_group_name, ins_obj = \
                        self._get_instance_with_pcr_seq_group(value)

                    if ins_obj is not None and pcr_group_name != "":
                        # We use the hierarchical name in the file
                        # And we use that as the instance name as well
                        # id (value) = DDR_0:DDR_CFG, type = ddr:DDR_CFG
                        ref_name = ins_obj.get_ref_name()
                        seq_ref_name = "{}:{}".format(ref_name, pcr_group_name)

                if ins_obj is None:
                    msg = 'Unable to find pcr instance {}'.format(value)
                    raise dev_excp.InstanceDoesNotExistException(
                        msg, app_excp.MsgLevel.warning)

            et.SubElement(sequence_elem, "efxpt:instance",
                          id=value, index=str(key), type=seq_ref_name)

            if ref_name not in block_names:
                block_names.append(ref_name)

            ins_names.append(value)

        return block_names, ins_names

    def get_package(self):
        '''
        Returns the device package object

        :return self.__device_package
        '''

        return self.__device_package

    def check_after_parse(self):
        '''
        This function is used for checking the validity of the
        device data after it is parsed.

        :return True if the data is valid. Otherwise, False
        '''

        error = 0
        # A list of instance name with shared resource.
        # This will be checked later if no error is found
        instance_with_shared_resource = []

        # Check die related info
        if not self.check_die_post_parse(instance_with_shared_resource):
            self.__logger.error("Post-parse check failed: die check")
            error += 1

        # Iterate through the periphery definition
        for block in self.__periphery_definition.values():
            if not block.post_parse_check(self,
                                          self.__timing_model):
                self.__logger.error("Post-parse check failed: block check")
                error += 1

        # Check timing model
        if self.__timing_model is None:
            # Allow device with no timing model defined
            self.__logger.warning("Device is missing timing model")

        # We check consistencies of parameter names across all combination files
        # while the parameter check in block is to make sure all parameter name used
        # is defined in the timing model
        elif not self.__timing_model.post_parse_check():
            self.__logger.error("Post-parse check failed: timing_model check")
            error += 1

        # Get the list of instance names
        instances = list(self.__periphery_instance.keys())

        # Get the map of instances that are being depended (key)
        # and who depends on it (value)
        dep_instances = self.get_depended_instances()

        # Check config_model
        if self.__config_model is not None:
            if not self.__config_model.post_parse_check():
                self.__logger.error(
                    "Post-parse check failed: config_model check")
                error += 1
            # Check that the pad_names all exist
            if not self.__config_model.is_pad_names_valid(
                    self.__device_package, self.__io_pad,
                    self.__resources, self.__internal_config):

                self.__logger.error(
                    "Post-parse check failed: pad name in config_model invalid")
                error += 1

        # Update the instance names saved such that it
        # contains the mode name to make it unique and matches
        # with design db
        self.uniquify_instances()

        # Check resource map after uniquify because there are resource
        # that might not have all it's pins bonded out (i.e. HSIO/LVDS
        # where only P or N are bonded
        if not self.__resources.post_parse_check(self.__periphery_instance.keys(), dep_instances):
            self.__logger.error("Post-parse check failed: resource map check")
            error += 1

        # Assign the package to io pad
        if self.__device_package is not None and self.__io_pad is not None:
            self.__io_pad.set_package(self.__device_package)

        # Check LVDS GPIO compatibility between RX and TX
        if not self._check_lvds_gpio_compatibility():
            error += 1

        # Check the mipi rx group resource are valid
        if self.__mipi_dphy_rx_group is not None:
            if not self.__mipi_dphy_rx_group.check_post_parse(instances, self.__io_pad):
                error += 1

        # For block that has embedded resources, check that all those resource
        # exists in the device
        for block in self.__periphery_definition.values():
            embed_res = block.get_embedded_resource()

            if embed_res is not None:
                if not embed_res.post_parse_check(self):
                    self.__logger.error("Post-parse check failed: block embeded resource check")
                    error += 1

        # Check that all instances in the shared instance list
        # are valid
        if instance_with_shared_resource:
            if not self.check_shared_instance(instance_with_shared_resource):
                error += 1

        is_pass = True

        if error > 0:
            is_pass = False

        return is_pass

    def check_shared_instance(self, instance_with_shared_resource):
        error = 0

        for iname in instance_with_shared_resource:

            # Get the instance object
            ins_obj = self.__periphery_instance[iname]

            # Get the shared instance map in the instance
            ins_shared_map = ins_obj.get_shared_ins_map()

            saved_shared_ins_names = []

            for ins_name in ins_shared_map:
                # Get the instance shared object and check
                # that the sub_ins name exists in device
                shared_obj = ins_shared_map[ins_name]

                if shared_obj is not None:
                    # Check that the instance name exists in device
                    other_ins = self.find_instance(ins_name)
                    if other_ins is None:
                        self.__logger.debug("Instance with shared instance {} not found in device".format(
                            ins_name))
                        error += 1

                    else:
                        # check that the block name matches
                        if other_ins.get_ref_name() != shared_obj.get_block_name():
                            error += 1
                            self.__logger.debug("Mismatch block type on shared instance {}: {} vs {}".format(
                                ins_name, other_ins.get_ref_name(), shared_obj.get_block_name()))

                        else:
                            saved_shared_ins_names.append(ins_name)

                    sub_ins_list = shared_obj.get_sub_ins_names()
                    if sub_ins_list:
                        for sub_ins_name in sub_ins_list:
                            if sub_ins_name != "" and self.find_instance(sub_ins_name) is None:
                                error += 1
                                self.__logger.debug("Shared sub instance {} of {} not found in device".format(
                                    sub_ins_name, ins_name))
                            else:
                                saved_shared_ins_names.append(sub_ins_name)

            if saved_shared_ins_names:
                if iname not in self._shared_instances:
                    self._shared_instances[iname] = saved_shared_ins_names
                else:
                    cur_ins_list = self._shared_instances[iname]
                    new_list = cur_ins_list + saved_shared_ins_names
                    self._shared_instances[iname] = new_list

        is_valid = True
        if error > 0:
            is_valid = False

        return is_valid

    def get_shared_instance_map(self):
        return self._shared_instances

    def get_shared_instances(self, ins_name):
        '''
        Return a list of device instances that this instance is shared with.
        It could be of a few possibilities:
        1) the ins_name is in the items mapped by the key in the _shared_instance:
           So, the shared instance is the key name
        2) the ins_name is the key in the _shared_instance:
           So, the shared instance is the list mapped by the key

        :param ins_name: The device instance name that we're trying to find the other
                instances that is shared with it
        :return: A list of device instance names that is shared with the passed instance.
                This can also includes HSIO GPIO instance names
        '''

        shared_ins_list = []

        if self._shared_instances:
            for iname in self._shared_instances:
                if ins_name == iname:
                    shared_ins_list = shared_ins_list + self._shared_instances[iname]
                elif ins_name in self._shared_instances[iname]:
                    # If it matches in the items list, then we check for other key as well
                    shared_ins_list.append(iname)

        return shared_ins_list

    def _check_lvds_gpio_compatibility(self):
        '''
        Check that both LVDS Tx and Rx has the same LVDS GPIO feature that
        the UI will use (see LVDSGPIOPropertyDeviceCheck)
        :return True if both are compatible and False otherwise
        '''
        from device.db_interface import DeviceDBService

        # This check is only applicable if LVDS block exists
        rx_name = DeviceDBService.block_type2str_map[DeviceDBService.BlockType.LVDS_RX]
        tx_name = DeviceDBService.block_type2str_map[DeviceDBService.BlockType.LVDS_TX]

        # Only check if both exists. If either one exists, nothing to check
        if rx_name in self.__periphery_definition and tx_name in self.__periphery_definition:
            dbi = DeviceDBService(self)
            rx_svc = dbi.get_block_service(DeviceDBService.BlockType.LVDS_RX)
            tx_svc = dbi.get_block_service(DeviceDBService.BlockType.LVDS_TX)

            if rx_svc is not None and tx_svc is not None:
                error = 0

                if rx_svc.is_support_gpio_slew_rate() != \
                        tx_svc.is_support_gpio_slew_rate():
                    error += 1

                if rx_svc.is_support_gpio_drive_strength() != \
                        tx_svc.is_support_gpio_drive_strength():
                    error += 1

                if rx_svc.is_support_gpio_pull_down() != \
                        tx_svc.is_support_gpio_pull_down():
                    error += 1

                if rx_svc.is_support_gpio_schmitt_trigger() != \
                        tx_svc.is_support_gpio_schmitt_trigger():
                    error += 1

                if error > 0:
                    self.__logger.error(
                        "Incompatible GPIO feature between LVDS Rx and Tx")
                    return False
            else:
                self.__logger.error("Missing LVDS Rx/Tx service interface")
                return False

        return True

    def get_instance_clock_region(self, dev_ins_name):
        '''
        :param des_ins: The design instance object
        :return a string that displays the instance's clock region
        '''

        clk_region = ""

        dev_ins = self.find_instance(dev_ins_name)
        if dev_ins is not None:
            clk_region = dev_ins.get_clock_region_names()

        return clk_region

    def print_clock_region_instance_map(self):
        '''
        Print the map of clock region to instances
        '''
        # A map of clock region name to a set of instance object
        clk2ins_map = {}

        for ins_obj in self.__periphery_instance.values():
            # Only iterate through instances that
            # are available in the resource map if printing to file
            # if not self.__resources.is_used(ins_name):
            #    continue

            if ins_obj is not None:
                region_list = ins_obj.get_clock_region_list()

                for region_name in region_list:
                    ins_set = set()

                    if region_name in clk2ins_map:
                        ins_set = clk2ins_map[region_name]

                    ins_set.add(ins_obj)
                    clk2ins_map[region_name] = ins_set

        efxpt_home = os.environ.get('EFXPT_HOME', "")
        self.__logger.debug("Reading in EFXPT_HOME: {}".format(efxpt_home))

        if efxpt_home != "":
            filename = "{}/{}_clock_region.txt".format(
                efxpt_home, self.__device_name)
        else:
            filename = "./{}_clock_region.txt".format(
                self.__device_name)

        rptfile = open(filename, 'w')

        title_list = ["Clock Region", "Resource", "Pin Name"]

        table = PrettyTable(title_list)

        def get_ins_name(ins_obj):
            return ins_obj.get_name()

        self.__logger.info(
            "The following are the list of clock region to instance:")
        for clk_region in sorted(clk2ins_map.keys()):

            # sorted_ins = sorted(clk2ins_map[clk_region],
            #                    key=pt_util.natural_sort_key_for_list)
            sorted_ins = sorted(clk2ins_map[clk_region],
                                key=lambda x: [pt_util.natural_sort_key_str(get_ins_name(x))])
            # self.__logger.info("Region: {} - Instances: {}".format(
            #    clk_region, ','.join(sorted_ins)))
            index = 0
            for ins in sorted_ins:
                # Iterate through instance pins that are of
                # type clkout
                clkregion2pin_map = ins.get_clock_region_to_pins_map()
                for pin_cregion in clkregion2pin_map:
                    if pin_cregion == clk_region:
                        pin_count = 0

                        pin_obj_list = clkregion2pin_map[pin_cregion]
                        sorted_pins = sorted(pin_obj_list,
                                             key=lambda x: [pt_util.natural_sort_key_str(get_ins_name(x))])

                        for pin_conn in sorted_pins:
                            row_list = []

                            if index == 0:
                                row_list.append(clk_region)
                            else:
                                row_list.append("")

                            if pin_count == 0:
                                row_list.append(ins.get_name())
                            else:
                                row_list.append("")

                            pin_name = pin_conn.get_name()
                            pin_index = pin_conn.get_index()

                            if pin_index is not None:
                                pin_name = "{}[{}]".format(pin_name, pin_index)

                            row_list.append(pin_name)
                            index += 1
                            pin_count += 1

                            table.add_row(row_list)

        rptfile.write("{}".format(table.get_string()))
        rptfile.close()

    def uniquify_lvds_instances(self, ins_name, mode_name, ins_to_add_map, ins_obj,
                                ins_list, hier_name):
        '''
        Uniquify LVDS_TX/RX instances which has multiple mode support.

        GPIOL_RX00 -> lvttl1: GPIOL_RXP00
                   -> lvttl2: GPIOL_RXN00
                   -> lvds: GPIOL_RX00
        :return:
        '''

        match = re.search(r'\d', ins_name)

        if match:
            digit_pos = match.start()
            new_name = ""

            if mode_name == "lvttl1":
                new_name = ins_name[0:digit_pos] + \
                    "P" + ins_name[digit_pos:]
                # self.__logger.debug("Adding lvttl1 instance: {}".format(new_name))

                ins_to_add_map[new_name] = ins_obj
                ins_list.append(new_name)
                self.__hier_to_flat_ins_map[hier_name] = new_name
                self.__flat_to_hier_ins_map[new_name] = hier_name
                ins_obj.add_mode_to_resource(
                    mode_name, new_name)

            elif mode_name == "lvttl2":
                new_name = ins_name[0:digit_pos] + \
                    "N" + ins_name[digit_pos:]
                # self.__logger.debug("Adding lvttl2 instance: {}".format(new_name))

                ins_to_add_map[new_name] = ins_obj
                ins_list.append(new_name)
                self.__hier_to_flat_ins_map[hier_name] = new_name
                self.__flat_to_hier_ins_map[new_name] = hier_name
                ins_obj.add_mode_to_resource(
                    mode_name, new_name)

            elif mode_name == "lvds":
                self.__hier_to_flat_ins_map[hier_name] = ins_name
                self.__flat_to_hier_ins_map[ins_name] = hier_name
                ins_obj.add_mode_to_resource(
                    mode_name, ins_name)

    def uniquify_hsio_instances(self, ins_name, mode_name, ins_to_add_map, ins_obj,
                                ins_list, hier_name):
        '''
        Uniquify HSIO block instances which has multiple mode support.

        GPIOL_PN_00 -> lvttl1/gpiox4p: GPIOL_P_00
                    -> lvttl2/gpiox4n: GPIOL_N_00
                    -> lvds_rx/lvds_tx/lvds_bidir/mipi_dphy: GPIOL_PN_00

        In this case, there are a few of the new instance name being shared
        across the different mode. This means that the flat name can be the same but
        the hierarchical name can be different.  This impacts the dictionary _flat_to_hier_ins_map
        in the key can map to several value (i.e. list of string instead of a single string)

        :return:
        '''

        if ins_name.find("_PN_") != -1:
            new_name = ins_name

            if mode_name == "lvttl1" or mode_name == "gpiox4p":
                # Remove the N from PN
                new_name = new_name.replace("_PN_", "_P_")

                # self.__logger.debug("Adding lvttl1 instance: {}".format(new_name))

                ins_to_add_map[new_name] = ins_obj
                ins_list.append(new_name)

            elif mode_name == "lvttl2" or mode_name == "gpiox4n":
                # Remove the P from PN
                new_name = new_name.replace("_PN_", "_N_")

                # self.__logger.debug("Adding lvttl2 instance: {}".format(new_name))

                ins_to_add_map[new_name] = ins_obj
                ins_list.append(new_name)

            self.uniquify_mode_instance(new_name, mode_name, ins_obj,
                                hier_name)

    def uniquify_quad_instances(self, ins_name, mode_name, ins_to_add_map, ins_obj,
                                ins_list, hier_name):
        '''
        Uniquify QUAD (serdes) block instances which has multiple mode support.

        QUAD_0 -> pcie: QUAD_0
                -> lane based protocol (ie 10g): Q0_LN0,Q0_LN1,Q0_LN2,Q0_LN3

        In this case, there are a few of the new instance name being shared
        across the different mode. This means that the flat name can be the same but
        the hierarchical name can be different.  This impacts the dictionary _flat_to_hier_ins_map
        in the key can map to several value (i.e. list of string instead of a single string)

        :return:
        '''
        if ins_name.find("QUAD_") != -1:
            new_name = ins_name
            index_pos = new_name.find("_")
            assert index_pos != -1

            index_str = new_name[index_pos+1:]

            # The individual lanes can be mapped to different
            # protocol and each protocol of the same l ane usage
            # can have different pins list
            # 10g_CH0 vs 10g_CH1 vs 1g_CH0, etc.
            # if mode_name.startswith("10g"):
            if mode_name != "pcie":
                protocol_name = ""
                if mode_name.endswith("_10G"):
                    protocol_name = "10G"
                elif mode_name.endswith("_1G"):
                    protocol_name = "1G"              
                elif mode_name.endswith("_RAW_SERDES"):
                    protocol_name = "RAW_SERDES"                    

                assert protocol_name != ""

                # At most there are 4 possible lanes
                # Append the quad instance name (Q#_LN#)
                # to the mode name with the protocol postfix dropped (_CH#)
                #new_name = new_name.replace(protocol_name, "Q"+index_str)
                end_pos = mode_name.find(f'_{protocol_name}')
                assert end_pos != -1
                new_name = f'Q{index_str}_{mode_name[:end_pos]}'

                # self.__logger.debug("Adding quad lane instance: {}".format(new_name))

                ins_to_add_map[new_name] = ins_obj
                ins_list.append(new_name)

            # pcie mode retains the resource name since it's taking
            # entire quad_pcie.
            # TODO: quad without pcie (quad) will not have a pcie mode but it
            # will still have the entire quad as a resource although not used
            # in any mode
            self.uniquify_mode_instance(new_name, mode_name, ins_obj,
                                        hier_name)

    def uniquify_mode_instance(self, ins_name, mode_name, ins_obj,
                                hier_name):

        # For the rest of the modes, we retain the original instance name
        self.__hier_to_flat_ins_map[hier_name] = ins_name

        # We add the name to the flat_to_hier_ins_map outside since there can be more than
        # one value associated to the same key
        if ins_name not in self.__flat_to_hier_ins_map:
            self.__flat_to_hier_ins_map[ins_name] = hier_name
        else:
            cur_hier_name = self.__flat_to_hier_ins_map[ins_name] + \
                            "," + hier_name
            self.__flat_to_hier_ins_map[ins_name] = cur_hier_name

        ins_obj.add_mode_to_resource(
            mode_name, ins_name)

    def uniquify_instances(self):
        '''
        Update the instance names stored in device db and
        the resource map.
        '''

        # Find the list of blocks that have multiple modes
        # Map of name to list of modes
        blocks_with_modes = {}

        for bname, bobj in self.__periphery_definition.items():
            if bobj.has_mode():
                # Get the list of mode names
                modes = bobj.get_mode_names()

                blocks_with_modes[bname] = modes

        if blocks_with_modes:
            from device.db_interface import DeviceDBService

            # Store a map of the base instance name to the list
            # of instances hierarchical names
            ins2hierarchical = {}
            ins_to_add_map = {}

            # Iterate through the list of instances and add the new
            # hierarchical instance names
            for ins_name, ins_obj in self.__periphery_instance.items():
                ref_name = ins_obj.get_ref_name()

                if ref_name in blocks_with_modes:
                    modes = blocks_with_modes[ref_name]
                    ins_list = []

                    if ins_name not in ins2hierarchical:

                        for mode_name in modes:
                            hier_name = ins_name + "." + mode_name
                            # DEBUG
                            #self.__logger.debug("Uniquify: instance {} with mode {} - {}".format(
                            #    ins_name, mode_name, hier_name))

                            # Add the names to the periperhy_instance map as well
                            # but all of them mapped to the same ins_obj

                            if hier_name not in self.__periphery_instance:
                                ins_to_add_map[hier_name] = ins_obj

                                # Also add the new name
                                # TODO: Proper implementation with storing the info
                                # in each instances in the die
                                if ref_name.startswith("lvds"):
                                    self.uniquify_lvds_instances(ins_name, mode_name, ins_to_add_map, ins_obj,
                                                                 ins_list, hier_name)

                                elif ref_name == "hsio" or ref_name == "hsio_max":
                                    self.uniquify_hsio_instances(ins_name, mode_name, ins_to_add_map, ins_obj,
                                                                 ins_list, hier_name)

                                # TODO: Add quad once we have other protocols supported
                                elif ref_name == "quad_pcie" or ref_name == "quad":
                                    self.uniquify_quad_instances(ins_name, mode_name, ins_to_add_map, ins_obj,
                                                                 ins_list, hier_name)

                                # Only applicable to MIPI Hard DPHY Tx Adv
                                elif ref_name == "mipi_tx" and \
                                        self.mipi_tx_type == DeviceDBService.BlockType.MIPI_TX_ADV:
                                    self.uniquify_mode_instance(ins_name, mode_name, ins_obj,
                                                        hier_name)

                            else:
                                raise ValueError(
                                    "Duplicated hierarchical instance name: {}".format(hier_name))

                        ins2hierarchical[ins_name] = ins_list

                    else:
                        raise ValueError(
                            "Duplicated instance names found: {}".format(ins_name))

            # Add the instance to the db instance map
            for ins_name, ins_obj in ins_to_add_map.items():
                self.__periphery_instance[ins_name] = ins_obj

            # Iterate through the list of resource instances
            if self.__resources is not None:
                self.__resources.uniquify_instances(ins2hierarchical)

    def get_base_ins_name(self, ins_mode_name):
        if ins_mode_name in self.__flat_to_hier_ins_map:
            dev_ins = self.__periphery_instance.get(ins_mode_name, None)
            if dev_ins is not None:
                return dev_ins.get_name()

        return ins_mode_name

    def get_flat_name(self, hier_name):
        if hier_name in self.__hier_to_flat_ins_map:
            ins_name = self.__hier_to_flat_ins_map[hier_name]
            dev_ins = self.find_instance(ins_name)
            if dev_ins is not None:
                return ins_name

        return ""

    def get_flat_to_hier_ins_map(self):
        return self.__flat_to_hier_ins_map

    def get_depended_instances(self):
        '''
        Iterate through the instance list and gather the
        instance that it may depends on.

        :return a map of the depended instance name (key) to the
                instances that depends on it (value)
        '''
        dep_instances = {}

        for ins_name, ins_obj in self.__periphery_instance.items():
            dependencies = ins_obj.get_depended_instance_names()

            if dependencies:
                for dep_name in dependencies:
                    if dep_name not in dep_instances:
                        cur_names = []
                    else:
                        cur_names = dep_instances[dep_name]

                    cur_names.append(ins_name)
                    dep_instances[dep_name] = cur_names

        return dep_instances

    def check_die_post_parse(self, instance_with_shared_resource):
        '''
        Check die related data after parsing

        :return True if it is valid. Otherwise, False
        '''
        is_pass = True
        error = 0

        # TODO: Make this data driven
        is_titanium = False
        if self.find_block("hsio"):
            is_titanium = True

        # Iterate through the instances
        for instance in self.__periphery_instance.values():

            dependencies = instance.get_dependencies()

            if dependencies:
                # Check that the intance pin name is valid in dependencies list
                for ins_name in dependencies:
                    if ins_name not in self.__periphery_instance:
                        self.__logger.debug(
                            "Post-parse check failed: ins dep name {} not-exists".format(
                                ins_name))
                        error += 1
                    else:
                        ins_obj = self.__periphery_instance[ins_name]
                        if not ins_obj.is_pins_valid(dependencies[ins_name]):
                            self.__logger.debug(
                                "Post-parse check failed: ins dep pin on {} invalid".format(
                                    ins_name))
                            error += 1

            if instance.has_shared_instance():
                instance_with_shared_resource.append(instance.get_name())

            ref_block = self.find_block(instance.get_ref_name())

            if ref_block is not None:
                # indicate if it is a tesseract in order to
                # determine the right clock region
                clkout_offset = self.get_clkout_region_start_offset()

                if not instance.post_parse_check(
                        ref_block, self.__global_clocks,
                        self.__periphery_instance, self.__width,
                        self.__height, self.__periphery_definition,
                        clkout_offset, is_titanium):
                    self.__logger.debug(
                        "Post-parse check failed: instance {} invalid".format(instance.get_name()))
                    error += 1

                # Save the clock region name to device database
                # only if the instance is in the resource map
                clk_region_list = instance.get_clock_region_list()
                for clk_region in clk_region_list:
                    if clk_region not in self.__clock_region_names:
                        if self.__resources is not None and \
                                self.__resources.is_used(instance.get_name()):
                            self.__clock_region_names.append(clk_region)

            else:
                self.__logger.error('Cannot find definition of block '
                                    '{} referenced by instance {}'.format(
                                        instance.get_ref_name(),
                                        instance.get_name()))
                error += 1

        # Check global clocks
        if not self.check_global_clocks():
            self.__logger.debug("Post-parse check failed: global clocks check")
            error += 1

        # Get the list of instance names
        instances = list(self.__periphery_instance.keys())

        # Check the IO
        if self.__device_package is not None:
            if not self.__io_pad.post_parse_check(
                    instances, self.__device_package,
                    self.__resources):
                self.__logger.debug("Post-parse check failed: io_pad check")
                error += 1

        # Check the PCR sequence
        if not self.check_pcr_sequence():
            self.__logger.debug("Post-parse check failed: pcr sequence check")
            error += 1

        # Check that the titanium device has the width and height
        if is_titanium and self.__timing_model is not None:
            if self.__width == 0 or self.__height == 0:
                self.__logger.debug(
                    "Post-parse check failed. Titanium device requires width and height to be set")
                error += 1

        if error > 0:
            is_pass = False

        return is_pass

    def check_global_clocks(self):
        '''
        Check that the instance name and pin exist in
        the device. Also check that the instance has the
        pin with the secondary function specified.

        :return True if it is valid. Otherwise, False
        '''

        valid = True

        # The is a map of instance to the list of pin names
        instance_pin_map = self.__global_clocks.get_clock_instance_pin()

        for ins in instance_pin_map.keys():
            pins = instance_pin_map[ins]

            # Find the instance
            if ins in self.__periphery_instance:
                ins_obj = self.__periphery_instance[ins]
            else:
                self.__logger.error('Cannot find global_clocks '
                                    'instance {} defined in device'.format(
                                        ins))
                valid = False
                continue

            if ins_obj is not None:
                for pin_name in pins:
                    if ins_obj.get_connection_pin(pin_name) is None:
                        self.__logger.error('Clock instance {} pin name '
                                            '{} is not defined'.format(
                                                ins, pin_name))
                        valid = False
            else:
                self.__logger.error('Clock instance {} does not'
                                    'exists in device instance list'.format(
                                        ins))
                valid = False

        return valid

    def check_pcr_sequence(self):
        '''
        Check that the instance name specified
        is defined in the device

        :param instances: list of instance names in this device

        :return True if it is valid. Otherwise, False
        '''
        valid = True
        count = 0

        for index in sorted(self.__pcr_bits_sequence.keys()):
            ins_name = self.__pcr_bits_sequence[index]

            if count != index:
                # Warn if the index does not match the total size which
                # indicates that it is not consecutive
                self.__logger.warning(
                    'PCR sequence is not indexed consecutively.'
                    ' Expected index {}. Read {}'.format(
                        count, index))
                count = index

            # Check if the instance exists
            if ins_name not in self.__periphery_instance:

                # Check if it is an instance with pcr group name
                if not self._is_instance_pcr_with_sequence_group(ins_name):
                    self.__logger.error('Instance name in PCR sequence {}'
                                        ' does not exists in device'.format(
                                            ins_name))
                    valid = False

            # We don't need to check if the instance is part of resource
            # map. We can have a sequence of instance that is not
            # part of map because sequence is based on die (fixed)
            count += 1

        return valid

    def _is_instance_pcr_with_sequence_group(self, ins_name_in_seq):
        is_valid = False

        if ins_name_in_seq.find(":") != -1:
            main_ins_name, pcr_group_name, ins_obj = self._get_instance_with_pcr_seq_group(ins_name_in_seq)
            if ins_obj is not None and pcr_group_name != "":
                blk_def = self.find_block(ins_obj.get_ref_name())

                if blk_def is not None and \
                    blk_def.is_pcr_seq_group_exists(pcr_group_name):
                    is_valid = True

        return is_valid

    def _get_instance_with_pcr_seq_group(self, ins_name_in_seq):
        ins_obj = None
        main_ins_name = ""
        pcr_group_name = ""

        match_grp = re.match(r'^([A-Za-z0-9_]+):([A-Za-z0-9_]+)$', ins_name_in_seq)
        if match_grp:
            main_ins_name = match_grp.group(1)
            pcr_group_name = match_grp.group(2)

            if main_ins_name in self.__periphery_instance and pcr_group_name != "":
                # Check that the group name exists in the block
                ins_obj = self.find_instance(main_ins_name)

        return main_ins_name, pcr_group_name, ins_obj

    def get_resources(self):
        '''
        Return the device resource object

        :return UseResource object
        '''
        return self.__resources

    def get_all_resources(self):
        '''
        Returns a list of instances that are enabled
        in the resource map, regardless of the type.

        :return list of instances name
        '''
        enabled_ins = self.__resources.get_instances()

        return list(enabled_ins.keys())

    def get_resources_by_type(self, name):
        '''
        Return a list of instances that are enabled in the resource
        map of the specified block name

        :param name: Periphery block name
        :return list of instances enabled in the device
        '''

        enabled_ins = self.__resources.get_instances()

        used_ins_name = []
        ins_obj_set = set()

        for ins_name in enabled_ins:
            # The periphery instance contains the instance name
            # without the extended mode name and also the instance
            # name with the extended mode name due to uniquify instance
            if ins_name in self.__periphery_instance:
                ins_obj = self.__periphery_instance[ins_name]

                if name == ins_obj.get_ref_name():
                    used_ins_name.append(ins_name)
                    ins_obj_set.add(ins_obj)

            else:
                raise ValueError('Unable to find resource instance {} '
                                 'in the device'.format(ins_name))

        used_ins_obj = list(ins_obj_set)

        return used_ins_name, used_ins_obj

    def find_pad_instances(self, pad_name):
        '''
        Find the instance associated to the passed pad_name.

        :param pad_name: Pad name

        :return a list of instance names associated to the pad
        '''

        instances = []

        if self.__io_pad.find_pad(pad_name) is not None:
            instances = self.__io_pad.find_design_instance(pad_name, self)

        return instances

    def is_io_pad_shareable(self, pad_name):
        '''
        Given the pad_name, check if the pad is shareable
        among instances.

        :param pad_name: IO Pad name

        :return True if the pad is shared, False otherwise
        '''

        return self.__io_pad.is_pad_shareable(pad_name)

    def find_instance(self, name: str) -> peri_instance.PeripheryBlockInstance | None:
        '''
        Find an instance with the specified name.
        Return the instance if found, else None.

        :param name: instance name

        :return PeripheryBlockInstance if found. Else, None
        '''
        return self.__periphery_instance.get(name, None)

    def find_block(self, name):
        '''
        Find Periphery block with the passed name

        :param name: Block name

        :return PeripheryBlock if found. Else, None
        '''
        return self.__periphery_definition.get(name, None)

    def find_instance_dependencies(self, ref_type):
        '''
        Iterate through the list of instances to find any
        instance that depends on other instances.

        NOTE: This does not check against whether the instance
                is in the resource map

        :param type: Block name. This is used to filter only
                finding instances of this block type without
                differentiating the other instances type that
                it depends on. If type is "", then no filter is
                applied

        :return a map of instance and the list of instance
                that it depends on
        '''

        ins_dep = {}

        for ins_name in self.__periphery_instance:
            ins_obj = self.__periphery_instance[ins_name]

            # If type is not specified or type is specified
            # and it matches with the instance ref type
            if ref_type == "" or ref_type is None or ins_obj.get_ref_name() == ref_type:
                # This returns a map of instance and pin that this
                # depends on. We only want the instance name (key)
                cur_dep = ins_obj.get_dependencies()

                if cur_dep:
                    ins_dep_names = list(cur_dep.keys())

                    ins_dep[ins_name] = ins_dep_names

        return ins_dep

    def find_pad(self, pad_name):
        '''
        Return the pad object based on pad_name specified

        :param pad_name: Pad name to search

        :return
        '''

        return self.__io_pad.find_pad(pad_name)

    def get_num_block_instances(self, block_name):
        '''
        Find the total number of available instances (in resource)
        of type block_name defined in this device.

        :param block_name: Block name

        :return Number of available instances of that type (int)
        '''

        # Iterate through the resources instances list and
        # find instance of the specified type name
        available_instances = self.__resources.get_instances()
        count = 0

        visited_instances = set()
        for ins_name in available_instances.keys():
            ins_obj = self.__periphery_instance[ins_name]

            if ins_obj in visited_instances:
                continue

            # We don't want to count the flatten instance (with mode)
            if ins_obj.get_ref_name() == block_name and\
                    ins_obj.get_name().find(".") == -1:
                count += 1

            visited_instances.add(ins_obj)

        return count

    def find_instances_of_block(self, block_name):
        '''
        Find the instances in the device db that are referenced
        of the passed block_name. Doesn't care if the instance
        is in the resource map or not.

        :param block_name: Block name
        :return a list of PeripheryBlockInstance
        '''
        ins_found = []

        for ins_obj in self.__periphery_instance.values():
            if ins_obj.get_ref_name() == block_name:
                ins_found.append(ins_obj)

        return ins_found

    def print_(self):
        '''
        Print the db information into logger.
        '''

        # Print the device name first
        self.__logger.debug("Device file: {}".format(self.__device_db_file))
        self.__logger.debug("Device name: {}".format(self.__device_name))

        # Print package information
        if self.__device_package is not None:
            self.__device_package.print_()

        # Print the periphery definition
        self.__logger.debug("Found {} periphery block types".format(
            len(self.__periphery_definition)))

        index = 0
        for obj in self.__periphery_definition.values():
            self.__logger.debug("Periphery definition: {}".format(index))
            obj.print_()
            index += 1

        # Print the periphery instances
        self.__logger.debug("Found {} periphery instances".format(
            len(self.__periphery_instance)))

        index = 0
        for obj in self.__periphery_instance.values():
            self.__logger.debug("Periphery instance: {}".format(index))
            obj.print_()
            index += 1

    def get_resource_migration_map(self):
        '''
        This is used for getting all the instance resource name
        and creates a map of old name to new name if there
        is a change.refer to PT-424. Since resource names
        are unique we don't need to know whether the name is
        set on specific block type.
        Pre-requisite: This has to be called after the device
                instance names are uniquified.

        :return a map of old resource name to the new
                name
        '''
        from device.db_interface import DeviceDBService

        old2new_instances_map = {}

        # Return if the device does not support any PLL and LVDS
        pll_found = self.find_block(
            DeviceDBService.block_type2str_map[DeviceDBService.BlockType.PLL])
        lvds_rx_found = self.find_block(
            DeviceDBService.block_type2str_map[DeviceDBService.BlockType.LVDS_RX])
        lvds_tx_found = self.find_block(
            DeviceDBService.block_type2str_map[DeviceDBService.BlockType.LVDS_TX])

        if pll_found is None and lvds_rx_found is None and lvds_tx_found is None:
            return old2new_instances_map

        if self.__resources is not None:
            resource_map = self.__resources.get_instances()
            resource_ins_names = list(resource_map.keys())

            for ins_name in resource_ins_names:

                if ins_name.find("PLL_") != -1:
                    # We want to create a map of old PLL instance name
                    # to the new name in device. This will be used
                    # as a reference to tell if older design needs migration
                    # of its resource name

                    # Get the type str (ie. "TR") without any integer
                    pll_index_str = re.sub('.*?([0-9]*)$', r'\1', ins_name)
                    if pll_index_str != "":
                        start_pos = ins_name.find("_")
                        end_pos = ins_name.find(pll_index_str)
                        pll_type_str = ins_name[start_pos + 1:end_pos]

                        # Swap PLL and type str
                        concat_type_str = "_{}".format(pll_type_str)
                        old_name = ins_name.replace("PLL", pll_type_str, 1)
                        old_name = old_name.replace(concat_type_str, "_PLL", 1)
                        self.__logger.debug("Replacing pll resource {} to {}".format(
                            old_name, ins_name))

                        # Only do migration if the new name exists
                        # in the instance list
                        if old_name not in old2new_instances_map:
                            old2new_instances_map[old_name] = ins_name

                # Find the LVDS instances and rename them
                # GPIOB_RX00.lvds -> GPIOB_RX00
                # GPIOB_RX00.lvttl1 -> GPIOB_RXP00
                # GPIOB_RX00.lvttl2 -> GPIOB_RXN00
                # if ins_name.find(".") != -1:
                elif ins_name in self.__flat_to_hier_ins_map:
                    # Get the old name
                    old_name = self.__flat_to_hier_ins_map[ins_name]
                    old2new_instances_map[old_name] = ins_name

        return old2new_instances_map

    def set_die_layout_size(self, width, height):
        self.__width = width
        self.__height = height

    def get_width(self):
        return self.__width

    def get_height(self):
        return self.__height

    def is_instance_in_resource_map(self, ins_name):
        '''
        Check if the passed instance name exists in
        the resource map (bonded out)
        :param ins_name: The instance name to check
        :return True if found in the device resource.
                False, otherwise.
        '''
        if self.__resources is not None:
            return self.__resources.is_used(ins_name)

        return False

    def is_block_in_resource_map(self, blk_name: str):
        '''
        Check if the device has any instance of blk_name
        bonded out
        :param blk_name: Periphery block name
        :return True if at least one instance of blk_name is found
                in the resource map. Otherwise, False
        '''
        if self.get_num_block_instances(blk_name) > 0:
            return True

        return False

    def get_all_clock_region_names(self):
        '''
        :return the list of clock region names found in
                the device.
        '''
        return self.__clock_region_names

    def get_configuration_mode_name(self):
        config_mode_str = ""

        config_model = self.get_config_model()
        if config_model is not None:
            selected_mode = config_model.get_selected()
            if selected_mode is not None:
                if selected_mode.get_width != 0:
                    config_mode_str = "{} (x{})".format(
                        selected_mode.get_name(),
                        selected_mode.get_width())
                else:
                    config_mode_str = selected_mode.get_name()

        return config_mode_str

    def get_periphery_definition(self):
        return self.__periphery_definition

    def get_periphery_instance(self):
        return self.__periphery_instance

    def __getstate__(self):
        state = self.__dict__.copy()

        # Remove the unpicklable entries.
        del state['_PeripheryDevice__logger']
        return state

    def __setstate__(self, state):
        self.__dict__.update(state)
        self.__logger = Logger

    def __eq__(self, other):
        if isinstance(other, PeripheryDevice):
            is_equal = self.__periphery_definition == other.__periphery_definition and \
            self.__device_db_file == other.__device_db_file and \
            self.__device_name == other.__device_name and \
            self.__device_mapped_name == other.__device_mapped_name and \
            self.__device_package == other.__device_package and \
            self.__periphery_instance == other.__periphery_instance and \
            self.__pcr_bits_sequence == other.__pcr_bits_sequence and \
            self.__timing_model == other.__timing_model and \
            self.__resources == other.__resources and \
            self.__io_pad == other.__io_pad and \
            self.__global_clocks == other.__global_clocks and \
            self.__secfunc_to_insnames == other.__secfunc_to_insnames and \
            self.__insname_to_secfunc == other.__insname_to_secfunc and \
            self.__config_model == other.__config_model and \
            self.__timing_parameters == other.__timing_parameters and \
            self.__width == other.__width and \
            self.__height == other.__height and \
            self.__clock_region_names == other.__clock_region_names and \
            self.__hier_to_flat_ins_map == other.__hier_to_flat_ins_map and \
            self.__flat_to_hier_ins_map == other.__flat_to_hier_ins_map and \
            self.__mipi_dphy_rx_group == other.__mipi_dphy_rx_group and \
            self.__internal_config == other.__internal_config and \
            self.pll_type == other.pll_type and \
            self.gpio_type == other.gpio_type and \
            self.osc_type == other.osc_type and \
            self.clkmux_type == other.clkmux_type and \
            self.ddr_type == other.ddr_type and \
            self.mipi_rx_type == other.mipi_rx_type and \
            self.mipi_tx_type == other.mipi_tx_type and \
            self._shared_instances == other._shared_instances and \
            self.__block_config == other.__block_config

            if not is_equal:
                traceback.print_stack()
            return is_equal
        return False

    def __hash__(self):
        return super().__hash__()

    def verify_common_device_block_equivalence(self):
        is_valid = True

        if self.find_block("hsio_max") is not None:
            from tx375_device.hsio_max.device_service import HSIOUnionDeviceService

            hsio_svc = HSIOUnionDeviceService(self, "hsio")
            if not hsio_svc.verify_block_similarities():
                self.__logger.error(
                    "Differences found in HSIO block definition: hsio,hsio_max {}".format(excp))
                is_valid = False

        return is_valid

    def get_device_family_die_name(self, is_exact=False):
        '''
        Determine which device die type it is
        :param is_exact: Indicates if we want to know exactly the
                    device name associated to the family instead
                    of its reference
        :return a string name indicating the family device die type name.
                    Empty string if not applicable.
        '''
        
        engineering_name_to_device_map = {
            "oph_77x162_b3_d1": ("Trion", "T8"),
            "oph_77x162v_b3_d1": ("Trion", "T8"),
            "oph_159x242_b17_d3": ("Trion", "T20"),
            "oph_337x642_b33_d10": ("Trion", "T120"),
            "oph_129x482_b12_d5": ("Trion", "T35"),
            "opx_218x322_b16_d10": ("Titanium", "Ti60"),
            "opx_333x642_b40_d20": ("Titanium", "Ti180"),
            "opx_466x964_b56_d28": ("Titanium", "Ti375"),
            "opx_334x484_b40_d20": ("Titanium", "Ti135"),             
        }

        dev_tup_found = None

        for eng_base_name in engineering_name_to_device_map:
            if eng_base_name in self.__device_mapped_name:
                dev_tup_found = engineering_name_to_device_map[eng_base_name]
                break

        dev_die_name = ""
        family_name = ""
        if dev_tup_found is not None:
            family_name, dev_die_name = dev_tup_found

            if is_exact and self.__device_mapped_name.endswith("_tz") and\
                family_name == "Titanium":

                family_name, dev_die_name = self.get_topaz_die_name(family_name, dev_die_name)
                assert dev_die_name != "", f'Unexpected Topaz device base {dev_die_name}'

        return family_name, dev_die_name

    def get_topaz_die_name(self, base_family: str , base_die: str):
        family_name = ""
        dev_die_name = ""

        if base_family in ["Titanium", "Topaz"] and base_die != "":

            topaz_map = {
                "Ti60": "Tz50",
                "Ti180": "Tz170",
                "Ti375": "Tz325",
                "Ti135": "Tz100"
            }

            if base_die in topaz_map:
                family_name = "Topaz"
                dev_die_name = topaz_map[base_die]

        return family_name, dev_die_name
    
    def get_clkout_region_start_offset(self):
        # Default to 2 since both Trion and earlier Titanium offset by 2
        offset = 2

        family_name, device_base_name = self.get_device_family_die_name()

        # Don't need to differentiate Topaz
        if family_name == "Titanium":
            if device_base_name == "Ti375" or device_base_name == "Ti135":
                offset = 3
        
        return offset
            
