"""
 Copyright (C) 2017-2020 Efinix Inc. All rights reserved.

 No portion of this code may be reused, modified or
 distributed in any way without the expressed written
 consent of Efinix Inc.

 Created on November 3, 2020

 @author: maryam
"""
from enum import Enum
import csv
import os
import pathlib
from typing import Optional, Union, Dict, List, Set, Tuple

import util.gen_util
from util.singleton_logger import Logger
from device.pcr_device import PCRMap

logger = Logger
@util.gen_util.freeze_it
class ConnVertex:

    class VertexType(Enum):
        input = 0
        output = 1
        mux_in = 2
        mux_out = 3


    def __init__(self, name: str, vtype: Optional[VertexType], mux_name: str = "", in_index: Union[str, int] = -1):
        self.name = name
        # List of vertex name (str) that are neighbor of this vertex
        self.neighbors: List[str] = list()
        self.mux = mux_name
        # Index can either be an integer or string:
        # string - output and int for input (mux input index)
        self.index = in_index
        self.vtype = vtype

        # Map of neighbor name to the tuple of (mux, select) that it connects
        # to. If it doesn't go to a mux then it will be mapped to None
        #self.neighbor2muxsel_map = {}
        self.set_default_setting()

    def set_default_setting(self):
        self.distance = 9999
        # black if it hasn't been visited and red if it has
        self.color = 'black'
        # Indicates if the node has been used
        self.cost = 0

    def reset(self):
        self.set_default_setting()

    def copy(self, gname, from_vertex):
        '''
        This is copying the vertex with modifying the name
        :param gname:
        :param from_vertex:
        :return:
        '''
        self.name = "{}.{}".format(gname, from_vertex.name)

        self.distance = from_vertex.distance
        self.color = from_vertex.color
        self.cost = from_vertex.cost

        for nnode in from_vertex.neighbors:
            new_name = "{}.{}".format(gname, nnode)
            self.neighbors.append(new_name)

    def duplicate(self, from_vertex):
        '''
        This is exact duplicate of the vertex
        :param from_vertex:
        :return:
        '''
        self.distance = from_vertex.distance
        self.color = from_vertex.color
        self.cost = from_vertex.cost

        for nnode in from_vertex.neighbors:
            self.neighbors.append(nnode)

    def add_neighbor(self, v:str):
        '''
        Add the vertex name (string) to the current vertex neighbors list
        :param v: neighbor vertex NAME (string)
        '''
        # Add it only if it's not already in the list of neighbors
        if v not in self.neighbors:
            self.neighbors.append(v)
            self.neighbors.sort()

    def is_neighbor(self, v:str):
        '''
        Check if the vertex name, v is a neighbor of this vertex
        :param v: vertex name in string
        :return: True if it is found in the neighbor list. Otherwise, False
        '''
        if v in self.neighbors:
            return True

        return False

    def remove_neighbor(self, v:str):
        '''
        Remove the vertex name, v from the neighbor list
        :param v: vertex name in string
        '''
        #logger.debug("Before remove neighbor: {}".format(",".join(self.neighbors)))
        if v in self.neighbors:
            self.neighbors.remove(v)
            self.neighbors.sort()
            #logger.debug("After remove neighbor: {}".format(",".join(self.neighbors)))

    def __eq__(self, other):
        if isinstance(other, ConnVertex):
            return self.name == other.name and \
            self.neighbors == other.neighbors and \
            self.mux == other.mux and \
            self.index == other.index and \
            self.vtype == other.vtype
        return False

    def __hash__(self):
        return super().__hash__()

class TraverseNode:

    def __init__(self, node):
        self.node = node # str
        self.func_f = 0
        self.func_h = 0
        self.func_p = 0

    def calc_f(self):
        # f = (g + h)*p
        # g = 1
        # h = HIST_COST(n)
        # p = 1 + pres_mult * overuse
        self.func_f = (1 + self.func_h ) * self.func_p

    def calc_h(self, hist_cost_n):
        self.func_h = hist_cost_n

    def calc_p(self, overuse, pres_mult):
        self.func_p = 1 + overuse * pres_mult

@util.gen_util.freeze_it
class MuxConnGraph:
    MAX_ROUTER_ITERATION = 6

    def __init__(self):
        # This is a map of ConnVertex name to the object itself
        self.vertices: Dict[str, ConnVertex] = {}
        
        self.traverse_nodes: Dict[str, TraverseNode] = {}

        self.input_node_dependencies: Dict[str, Tuple[List[str], List[str]]] = {}
    

    def is_vertex_exists(self, vname):
        if vname not in self.vertices:
            return False

        return True

    def add_vertex(self, vertex, is_input=False):

        if isinstance(vertex, ConnVertex) and vertex.name not in self.vertices:
            #logger.debug("INFO: Adding vertex {}".format(vertex.name))
            self.vertices[vertex.name] = vertex

            return True
        else:
            return False

    def add_edge(self, u:str, v:str):
        '''
        u and v are name of the vertex (string format)
        :param u: source vertex name (string)
        :param v: destination vertex name (string
        :return: True if it was added to graph
        '''
        # u = src and v = destination
        # This is a directed graph.
        if u in self.vertices and v in self.vertices:
            src_vertex = self.vertices[u]
            src_vertex.add_neighbor(v)

            #logger.debug("Add edge {} from {}".format(v, u))
            return True
        else:
            if u not in self.vertices:
                logger.debug("ERROR: Cannot find node {}".format(u))
            if v not in self.vertices:
                logger.debug("ERROR: Cannot find node {}".format(v))

            return False

    def remove_edge(self, u:str, v:str):
        '''
        Remove the edge between ConnVertex u and v
        :param u: src vertex name
        :param v: destination vertex
        :return:
        '''
        logger.debug("Remove edge {} from {}".format(v, u))

        if u in self.vertices and v in self.vertices:
            src_vertex = self.vertices[u]
            if src_vertex is not None and src_vertex.is_neighbor(v):
                # Remove v from neighbor
                src_vertex.remove_neighbor(v)

    def get_vertex(self, vname:str):
        '''

        :param vname: vertex name
        :return: ConnVertex with the specified name
        '''
        if vname in self.vertices:
            return self.vertices[vname]

        return None

    def reset_graph(self):

        for vobj in self.vertices.values():
            vobj.reset()

    def print_graph(self):
        for key in sorted(list(self.vertices.keys())):
            logger.debug(key + str(self.vertices[key].neighbors) + "  " + str(self.vertices[key].distance))

    def remove_dangling_primary_input_vertices(self):
        '''
        Remove nodes that are of type input if it has no neighbor
        :return: list of vertex names that were removed
        '''
        vname_to_remove = []
        for vname in self.vertices:
            vertex = self.vertices[vname]
            if len(vertex.neighbors) == 0 and \
                vertex.vtype == vertex.VertexType.input:

                vname_to_remove.append(vname)

        if vname_to_remove:
            for vname in vname_to_remove:
                vertex = self.vertices[vname]

                if vertex is not None:
                    self.vertices[vname] = None
                    # Delete from the map
                    del self.vertices[vname]

            logger.debug("Removed vertices: {} from graph".format(",".join(vname_to_remove)))

        return vname_to_remove

    def backtrace(self, parent, start, end_node, route_results):
        '''

        :param parent:
        :param start:
        :param end_node:
        :param route_results: Map of the input to the list of vertex hops
        :return:
        '''
        valid = True

        path = [end_node]
        #logger.debug("\tBacktrace: parent - {} start - {} end - {}\n".format(
        #    parent, start, path))

        # Search for node with cost > 0
        while path[-1] != start:
            logger.debug("-{}".format(path[-1]))
            # Increase the cost
            if path[-1] in self.vertices:
                node = self.vertices[path[-1]]
                # Found a node that is already part of a separate path
                if node.cost > 0:
                    # Skip if the node cost is positive but it was
                    # part of the current input being routed
                    is_cur_path = False
                    if start in route_results:
                        vertex_hops = route_results[start]
                        if node.name in vertex_hops:
                            is_cur_path = True

                    if not is_cur_path:
                        valid = False
                        break

            path.append(parent[path[-1]])

        if valid:
            for node_name in path:
                node = self.vertices[node_name]
                node.cost = node.cost + 1

            path.reverse()
            #logger.debug("\n\tbacktrace: {}\n".format(",".join(path)))

        else:
            # Clear the path
            path = []
            logger.debug("Dismiss backtrace due to high cost\n")

        return path, valid

    def clear_nodes_visits(self):
        for vertex in self.vertices.values():
            vertex.color = 'black'
            vertex.distance = 0

    def bfs(self, vert, route_results):
        logger.debug("Routing node {}".format(vert.name))

        # Clear all nodes visited flag
        self.clear_nodes_visits()

        route_map = {}
        route_path = []

        q = list()
        vert.distance = 0
        vert.color = 'red'
        start_name = vert.name

        q.append(vert.name)

        while len(q) > 0:
            # print("q: {}".format(",".join(q)))
            u = q.pop(0)

            node_u = self.vertices[u]
            node_u.color = 'red'

            if node_u.neighbors:
                # Sort the neighbors by the cost
                def get_cost(node_name):
                    node = self.vertices[node_name]
                    return node.cost

                adj_list = node_u.neighbors

                # iterate through the lowest cost first
                for v in sorted(adj_list, key=get_cost):

                    node_v = self.vertices[v]
                    #logger.debug("Reading node: {} cost: {} neighbor: {} cost{}".format(
                    #    u, node_u.cost, v, node_v.cost))

                    if node_v.color == 'black':
                        q.append(v)
                        route_map[v] = u
                        if node_v.distance > node_u.distance + 1:
                            node_v.distance = node_u.distance + 1
            else:

                #print("Reach end: {}".format(",".join(output_traversal)))
                # output_traversal.pop()
                # Found a complete route. But don't take it unless the cost of all nodes
                # are 0. So, we continue to the next neighbor
                if node_u.cost > 0:
                    continue

                tmp_route, is_valid = self.backtrace(route_map, start_name, u, route_results)
                if is_valid:
                    route_path = tmp_route
                    break

        # print("Output traversal: {}".format(",".join(output_traversal)))
        return route_path

    def is_overuse_found(self, route_results):
        '''

        :return:
        '''
        is_overuse = False

        if route_results:
            dup_nodes = set()
            visited_nodes = []

            for route_list in route_results.values():
                for node_name in route_list:
                    if node_name not in visited_nodes:
                        visited_nodes.append(node_name)
                    else:
                        dup_nodes.add(node_name)

            if dup_nodes:
                is_overuse = True

        return is_overuse

    def route_input_bfs(self, user_inputs: List[ConnVertex]):
        '''

        :param user_inputs: List of ConnVertex representing inputs
        :return:
            route_result: Map of input name to the list of nodes it goes through
            unrouted_input: List of input name that were not routed succesfully
        '''
        route_results = {}
        unrouted_input = []

        is_successful = False

        # Attempt to route within MAX_ROUTER_ITERATION tries
        for i in range(self.MAX_ROUTER_ITERATION):
            logger.debug("\nRouting iteration {}\n".format(i))

            for input in user_inputs:

                # Input is a node object
                output_route = self.bfs(input, route_results)

                if output_route:
                    # If it has previously been routed and now the result is
                    # different, we change the cost of the earlier visited nodes
                    # of the old path
                    if input.name in route_results:
                        #logger.debug("\nInput {} has been rerouted\n".format(input.name))
                        old_route = route_results[input.name]
                        for vertex_name in old_route:
                            vertex_obj = self.get_vertex(vertex_name)
                            if vertex_obj is not None:
                                vertex_obj.cost = vertex_obj.cost - 1

                    route_results[input.name] = output_route

                else:
                    logger.debug("\nFAILED to route input {}\n".format(input.name))

            #logger.debug("results on current iteration:\n")
            #for inp in route_results:
            #    logger.debug("*** Routing {}: {}\n".format(inp, ",".join(route_results[inp])))

            # Check if there is overuse
            if route_results and not self.is_overuse_found(route_results) and len(user_inputs) == len(route_results):
                is_successful = True
                break

        if not is_successful:
            #logger.debug("\nRouting did not complete successfully but the following are those that were routed\n")
            #for inp in route_results:
            #    logger.debug("Completed Routing {}: {}\n".format(inp, ",".join(route_results[inp])))

            for input in user_inputs:
                if input.name not in route_results:
                    unrouted_input.append(input.name)

            logger.debug("Unrouted inputs: {}".format(",".join(unrouted_input)))

        return route_results, unrouted_input


    def copy_graph_with_prefix(self, gname, from_graph):
        '''
        This is duplicating the graph but with creating additional prefix names
        :param gname:
        :param from_graph:
        :return:
        '''
        # When we copy the graph, we need to add prefix to the vertices so that
        # they become unique vertices (unless it is the top-level input node)
        for vertex in from_graph.vertices.values():
            new_vname = "{}.{}".format(gname, vertex.name)
            new_vertex = ConnVertex(new_vname, vertex.vtype, vertex.mux, vertex.index)
            new_vertex.copy(gname, vertex)

            self.vertices[new_vertex.name] = new_vertex

    def duplicate_graph(self, from_graph):
        '''
        This is duplicating the graph to create another exact copy of contents
        :param from_graph:
        :return:
        '''
        # When we copy the graph, we need to add prefix to the vertices so that
        # they become unique vertices (unless it is the top-level input node)
        for vertex in from_graph.vertices.values():
            new_vertex = ConnVertex(vertex.name, vertex.vtype, vertex.mux, vertex.index)
            new_vertex.duplicate(vertex)

            self.vertices[new_vertex.name] = new_vertex

    def is_mux_check_valid(self, refblk):
        '''
        Check that vertices of type mux_in and mux_out has valid mux names.
        :param refblk: The PeripheryBlock object that this block conn belongs to
        :return: True if everything is valid. Otherwise, False
        '''
        is_valid = True

        if refblk is not None:
            pcr_block = refblk.get_block_pcr()

            if pcr_block is not None:
                # Check that there's a pcr entry with the mux name
                for vertex in self.vertices.values():
                    if vertex.vtype == ConnVertex.VertexType.mux_in:
                        logger.debug("Check mux_in vertex: {} in {}".format(
                            vertex.name, refblk.get_name()))

                        if not pcr_block.is_pcr_map_name_exists(vertex.mux):
                            logger.debug("Unable to find pcr {} in {}".format(
                                vertex.mux, refblk.get_name()))
                            is_valid = False
                        else:
                            # Check that the index is a valid value in
                            # the pcr mode
                            pcr_map = pcr_block.get_pcr_map(vertex.mux)
                            if pcr_map is not None:
                                # The index is an integer
                                if pcr_map.get_type() == PCRMap.PCRMapType.mode:
                                    if not pcr_map.is_mode_exists(str(vertex.index)):
                                        logger.debug("Invalid input mux {} index {}".format(
                                            vertex.mux, vertex.index))
                                        is_valid = False
                                else:
                                    logger.debug("PCRMap of {} is not a mode type".format(vertex.mux))
                                    is_valid = False
                            else:
                                logger.debug("Cannot find pcr {} in {}".format(
                                    vertex.mux, refblk.get_name()))
                                is_valid = False

                    elif vertex.vtype == ConnVertex.VertexType.mux_out:
                        logger.debug("Check mux_out vertex: {} in {}".format(
                            vertex.name, refblk.get_name()))

                        # Check that the mux name matches
                        if not pcr_block.is_pcr_map_name_exists(vertex.mux):
                            logger.debug("Unable to find pcr {} in {}".format(
                                vertex.mux, refblk.get_name()))
                            is_valid = False

            else:
                logger.debug("No PCR Block found for {}".format(refblk.get_name()))
                is_valid = False

        else:
            logger.debug("No clockmux block found")
            is_valid = False

        return is_valid

    def __eq__(self, other):
        if isinstance(other, MuxConnGraph):
            return self.vertices == other.vertices
        return False

    def __hash__(self):
        return super().__hash__()

    def get_node_occupancy(self, trv_node, route_results, exclude_input):
        '''

        :param trv_node: TraverseNode object
        :param route_results: The map of input to its routed path
        :param exclude_input: The input node name to exclude from being considered
        :return: The occupancy value of the TraverseNode
        '''
        occupancy = 0

        if trv_node is not None:
            node_name = trv_node.node

            for input, path in route_results.items():
                if input != exclude_input:
                    # Look at the routing result for all except itself (exclude)
                    if path:
                        if node_name in path:
                            occupancy += 1

        # We have the occupancy of that node
        return occupancy

    def calculate_overuse(self, occupancy):
        '''
        Calculate the overuse based on the passed occupancy count
        :param occupancy: the occupancy value of a node
        :return: calculated overuse of a node
        '''
        capacity = 1
        return max(0, occupancy - capacity)

    def check_for_pll_dependencies_routed(self, cur_node: str,
                                          cur_node_dep: List[str], route_results):
        
        # cur_node has its own list of dependencies: other_dep_input
        # For example PLL.C2 - [PLL.C3]
        # At the same time there may be linked dependencies:
        # C1 - [C3]
        # C2 - {C3]
        # C3 - [C1, C2]
        # So, when looking at C2, we should also check if C1 has already been routed
        # since C1 is indirectly linked to C2 through C3

        # Contains the list of all the node input names that the
        # current node input name is dependent on (indirect and direct)
        all_dep_set = set(cur_node_dep)
        dep_clkmux_name = ""

        for in_name in sorted(self.input_node_dependencies):
            # skip same node input
            if in_name == cur_node:
                continue

            tup_info = self.input_node_dependencies[in_name]
            _, other_dep_input = tup_info

            # Check if the cur node dependencies is in the other_dep_input
            #print(f'before combined: {all_dep_set}')
            
            if in_name in cur_node_dep:
                # combined the two sets
                #print(f'\nchecking overlap between {cur_node} - {cur_node_dep} and {in_name} - {other_dep_input}')            
                #print(f'before combined: {all_dep_set}')            
                all_dep_set = all_dep_set | set(other_dep_input)

            #print(f'after combined: {all_dep_set}')

        #print(f"\nCheck pll dependencies on {cur_node}")
        #self.print_current_route_result(route_results)

        #print(f'\nall_dep_set: {all_dep_set}')
        # Check if the dependent clock is already routed
        
        # Indicate how many of the depended clocks that
        # have been routed to the clkmux with the count of clock found
        # key: clkmux name, value: the count of how many depended pll clock
        # connected to that clkmux side
        routed_dep_clks: Dict[str, int] = {}
        for other_clk in sorted(all_dep_set, reverse=True):

            if other_clk == cur_node:
                continue

            if other_clk in route_results:
                # This has been routed
                path = route_results[other_clk]
                #print(f'route_results for {other_clk}:  {len(path)} - {path}')
                
                # First element in path is the input
                assert len(path) > 1
                blk_in_node = path[1]
                if blk_in_node.find(".") != -1:
                    # Stop at the first one found. Assumption is the rest should
                    # have been routed to the same clkmux sides
                    dep_clkmux = blk_in_node[:blk_in_node.find(".")]
                    #print(f'input {cur_node} depends on {other_clk}: {dep_clkmux}')

                    if dep_clkmux in routed_dep_clks:
                        routed_dep_clks[dep_clkmux] = routed_dep_clks[dep_clkmux] + 1
                    else:
                        routed_dep_clks[dep_clkmux] = 1   
        
        # Find the key with the maximum value
        if routed_dep_clks:
            dep_clkmux_name = max(routed_dep_clks, key=routed_dep_clks.get)
        
        #print(f'\ndebug_list {dep_clkmux_name} with all_dep_order: {debug_list}')
       
        return dep_clkmux_name
    
    def astar_pathfinder(self, trv_node: TraverseNode, pres_mult: float,
                         node2overuse_map: Dict[str, int], route_results):
        """
        Modified for version 2

        :param trv_node: The starting TraverseNode
        :param pres_mult: The current value of pres_mult
        :param node2overuse_map: A map of node to the overuse count
        :param route_results: A map of input name to the path that has been routed
        :return: A list of string which indicate the routing for the specific input
        """
        open_set: List[TraverseNode] = []
        closed_set: List[TraverseNode] = []
        # A map of the next node name to the current node object
        # (backward). used for backtrace
        came_from = {}

        node_occupancy = self.get_node_occupancy(trv_node, route_results, trv_node.node)
        overuse_val = self.calculate_overuse(node_occupancy + 1)
        node_overuse = self.get_node_overuse(node2overuse_map, trv_node.node)

        # This is the starting input node
        trv_node.calc_h(node_overuse)
        trv_node.calc_p(overuse_val, pres_mult)
        trv_node.calc_f()

        dep_clkmux_name = ""

        if trv_node.node in self.input_node_dependencies:
            logger.debug(f"Check {trv_node.node} in input_node_dependencies")
            
            # This node has dependencies
            tup_info = self.input_node_dependencies[trv_node.node]
            _, other_dep_input = tup_info
            #print(f'\nclkmux_in_list: {clkmux_in_list}')
            #print(f'other_dep_input: {other_dep_input}\n')

            # Check if the dependent clock is already routed
            dep_clkmux_name = self.check_for_pll_dependencies_routed(
                trv_node.node, other_dep_input, route_results)
            
        open_set.append(trv_node)
    
        while len(open_set) > 0:
            # Get the node with lowest f score
            cur_node = self.lowest_f_score(open_set)
            # move it from the open to close set
            open_set.remove(cur_node)
            closed_set.append(cur_node)

            # Get the current node to see if it has any neighbor.
            # It's the sink node if there is no neighbors.
            node_obj = self.get_vertex(cur_node.node)
            if node_obj is None:
                raise ValueError("ERROR: Unexpected graph node {} does not exists".format(cur_node.node))

            if not node_obj.neighbors:
                # Backtrace to construct the full path since we reach sink node
                return self.reconstruct_path(came_from, cur_node)

            # Iterate through the neighbors of current node
            for neighbor in node_obj.neighbors:
                # Skip neighbor that has been visited and has a lower F(n)
                neighbor_trv = self.get_node_in_trv_list(closed_set, neighbor)
                if neighbor_trv is not None:
                    if neighbor_trv.func_f < cur_node.func_f:
                        continue

                # If this has depdencies
                congest_cost = 0
                if dep_clkmux_name != "":
                    if neighbor.find(".") != -1:
                        neighbor_mux_name = neighbor[:neighbor.find(".")]
                        #print(f'Check node {cur_node.node}-{dep_clkmux_name} neighbor: {neighbor}-{neighbor_mux_name}')

                        if neighbor_mux_name != dep_clkmux_name:
                            # Put a high cost to p
                            congest_cost += 1
                            logger.debug(f'Increase cong_cost for {cur_node.node} neighbor: {congest_cost}')
                            
                # Add Neighbor into open set if not included yet
                neighbor_trv = self.get_node_in_trv_list(open_set, neighbor)
                if neighbor_trv is None:
                    # Find the TraverseNode with the name matching neighbor
                    assert neighbor in self.traverse_nodes
                    neighbor_trv = self.traverse_nodes[neighbor]
                    open_set.append(neighbor_trv)

                came_from[neighbor] = cur_node
                node_occupancy = self.get_node_occupancy(neighbor_trv, route_results, trv_node.node)
                overuse_val = self.calculate_overuse(node_occupancy + congest_cost + 1)
                node_overuse = self.get_node_overuse(node2overuse_map, neighbor_trv.node)

                neighbor_trv.calc_h(node_overuse)
                neighbor_trv.calc_p(overuse_val, pres_mult)
                neighbor_trv.calc_f()

        return []

    def get_node_overuse(self, node2overuse_map, nnode_name: str):
        """
        Get the overuse value of a node
        :param node2overuse_map: a dict of node name to the overruse value
        :param nnode_name: Node name to find
        :return: Node overuse value (int)
        """
        if nnode_name in node2overuse_map:
            return node2overuse_map[nnode_name]

        return 0

    def get_node_in_trv_list(self, trv_list: List[TraverseNode], nname: str):
        """
        Get the TraverseNode with the matching name from the list
        :param trv_list: List of TraverseNode to search from
        :param nname: Node name to find
        :return: TraverseNode with the matching nname in the list. None if
                not found.
        """
        for trv_node in trv_list:
            if trv_node.node == nname:
                return trv_node

        return None

    def reconstruct_path(self, came_from, current: TraverseNode):
        '''
        Reconstruct the routing path that we found
        :param came_from:
        :param current:
        :return:
        '''
        # Traverse backward from end to start to
        # see the full path (when done)
        path = [current]
        current_key = current.node

        while current_key in came_from:
            # traverse backward as the came_from map is a map
            # of the node name to its parent node object
            current = came_from[current_key]
            current_key = current.node
            path.insert(0, current)

        return path

    def lowest_f_score(self, node_list: List[TraverseNode]):
        '''
        Find the TraverseNode with the smallest F(n) value.

        :param node_list: A list of Traverse node
        :return: TraverseNode that has the smallest F(n) value in
                the node_list
        '''
        final_node = None
        for node in node_list:
            if not final_node or node.func_f < final_node.func_f:
                final_node = node
        return final_node

    def print_current_route_result(self, route_results):
        '''
        Debug function to print all the routed path in the dict
        :param route_results:
        :return:
        '''
        if route_results:
            logger.debug("\n** Routing results:")
            for input_name, path in route_results.items():
                logger.debug("Input: {} Path: {}".format(input_name, ",".join(path)))

    def check_overuse_nodes_in_result(self, node2overuse_map, route_results):
        '''
        Check if we have any overuse found in the routed results
        :param node2overuse_map:
        :param route_results:
        :return: True if we have found overuse (one node has more than one
                routing path that goes through it)
        '''
        is_overuse = False

        if route_results:
            dup_nodes = set()
            visited_nodes = []

            for route_list in route_results.values():
                for node_name in route_list:
                    if node_name not in visited_nodes:
                        visited_nodes.append(node_name)
                    else:
                        # Same nodes visited more than once
                        dup_nodes.add(node_name)

            if dup_nodes:
                is_overuse = True

                for node_name in dup_nodes:
                    if node_name in node2overuse_map:
                        cur_overuse_count = node2overuse_map[node_name]
                        node2overuse_map[node_name] = cur_overuse_count + 1

                    else:
                        raise ValueError("ERROR: Node {} not found in node2overuse_map".format(node_name))

        return is_overuse

    def route_input(self, user_inputs: List[ConnVertex]):
        """
        Version 2 after found some flaw from v1 (PT-1754).

        :param user_inputs: List of vertices representing the input nodes.
                The caller  has already sort the list based on the input names.
                So we don't need to sort it.
        :return:
                route_results: A map of the input name to the list of node names
                        that make up the path
                unrouted_input: A list of input name that are not routeable
        """
        route_results = {}
        unrouted_input = []

        is_overuse = True
        pres_mult = 1.0

        # Keep a map of the overuse count for each node. Key is the node name
        # This is H(n): Historical overuse per iteration
        # overuse = max(0, occupancy - capacity) where capacity is always 1
        node2overuse_map: Dict[str, int] = {}

        # Create the TraverseNode for all nodes in the graph after reset/clearing it
        self.traverse_nodes.clear()

        for node_name in self.vertices:
            trv_node = TraverseNode(node_name)
            self.traverse_nodes[node_name] = trv_node

            # Set the overuse of all nodes to 0:
            node2overuse_map[node_name] = 0

        #logger.debug(f"input with dependencies: {self.input_node_dependencies}")

        i = 0
        while is_overuse and i < 10:
            logger.debug("\nRouting iteration {}\n".format(i))

            for input in user_inputs:
                # Get the traverse node
                if input.name in self.traverse_nodes:
                    trv_node = self.traverse_nodes[input.name]

                else:
                    raise ValueError("ERROR: Unable to find input traverse node {}".format(input.name))

                logger.debug(f"\n*** Routing input {input.name}")

                output_path = self.astar_pathfinder(trv_node, pres_mult, node2overuse_map, route_results)

                # If length is 0 or 1, it means that input was not routed
                if len(output_path) > 1:
                    path_node_names = []
                    for tnode in output_path:
                        path_node_names.append(tnode.node)

                    route_results[input.name] = path_node_names

            #logger.debug(f"\n---routing result at {i}")
            self.print_current_route_result(route_results)

            # Check if within the routed path, any overuse was found
            is_overuse = self.check_overuse_nodes_in_result(node2overuse_map, route_results)

            #if not is_overuse and self.input_node_dependencies:
                # check if the PLL clock with dependencies are on different sides
            #    if self.is_pll_routed_to_different_sides(route_results):
            #        is_overuse = True

            # increment variables for next iteration
            pres_mult *= 1.6
            i += 1

        if is_overuse:
            # It means we exit without a successful route:
            unrouted_input, in2path_removed_map = self.determine_unrouted_input(
                user_inputs, route_results)

            # Debug
            #if in2path_removed_map:
            #    logger.debug("\n*** Unrouted inputs that are removed")
            #    self.print_current_route_result(in2path_removed_map)
            #    logger.debug("\n### Final routed results")
            #    self.print_current_route_result(route_results)

        return route_results, unrouted_input

    def is_pll_routed_to_different_sides(self, route_results):
        found_diff_mux = False

        for pll_clock in route_results:
            path = route_results[pll_clock]
            #print(f'route_results for {other_clk}:  {len(path)} - {path}')
            is_diff_mux = False

            # Check which mux it went to
            assert len(path) > 1
            blk_in_node = path[1]
            if blk_in_node.find(".") != -1:
                # Stop at the first one found. Assumption is the rest should
                # have been routed to the same clkmux sides
                cur_clkmux_name = blk_in_node[:blk_in_node.find(".")]
            
            if pll_clock in self.input_node_dependencies:
                tup_info = self.input_node_dependencies[pll_clock]
                _, other_dep_input = tup_info

                for other_clk in other_dep_input:
                    if other_clk in route_results:
                        other_path = route_results[other_clk]

                        assert len(other_path) > 1
                        blk_in_node = other_path[1]
                        if blk_in_node.find(".") != -1:
                            # Stop at the first one found. Assumption is the rest should
                            # have been routed to the same clkmux sides
                            other_clkmux_name = blk_in_node[:blk_in_node.find(".")]
                            if cur_clkmux_name != other_clkmux_name:
                                is_diff_mux = True
                                break

            if is_diff_mux:
                found_diff_mux = True
                break

        return found_diff_mux

    def check_for_overuse_resource(self, mux_resources_visited, current_path):
        '''
        Check if the same resource is repeated in both current_path and
        mux_resources_visited. If yes, it means there's overuse.

        :param mux_resources_visited: Set of node names that have been visited
        :param current_path: A set of resource/node names
        :return: True if found a resource/node used in multiple places
        '''
        is_overuse = False

        if current_path:
            for conn_info in current_path:
                # Skip the first one which is input
                assert conn_info != ""

                if conn_info in mux_resources_visited:
                    is_overuse = True
                    break

        return is_overuse

    def get_routed_result_resource(self, in_name: str, current_path: List[str]):
        '''
        Extract only the resource information (without the specific mux selection)
        from the routed path result.

        :param in_name: The top-level input name to route
        :param current_path: The routing path result of that input
        :return: A set of all the resource names associated to the routed result
        '''
        resources = set()

        if current_path:
            for conn_info in current_path:
                if conn_info != in_name:
                    assert conn_info != ""
                    if conn_info.find(":") != -1:
                        split_conn = conn_info.split(":")
                        assert len(split_conn) == 2
                        # Get the mux name, ignoring which input it takes
                        res_name = split_conn[0]
                    else:
                        # This must be the output
                        res_name = conn_info

                    resources.add(res_name)

        return resources

    def determine_unrouted_input(self, user_inputs, route_results):
        """
        From the list of user_inputs and route_results, check if there is
        an input that is considered not routed.  This is in case we ended up
        with overuse when we exit the search loop.

        :param user_inputs: list of ConnVertex
        :param route_results: a map of the input name to the list of string
                    that makes up the path
        :return:
            unrouted_input:  The list of input name that is not routed
            in2path_removed_map: A map of input name to the path that is removed
                        from the route_results (now considered unrouted)
        """
        unrouted_input = []

        in2path_removed_map = {}
        # A list of mux resources visited. This is used to keep
        # track of input that may have used same resources (overuse)
        mux_visited = set()

        # Follow the order of the user_inputs
        for input_v in user_inputs:
            assert input_v is not None
            in_name = input_v.name

            if in_name in route_results:
                resource_names = self.get_routed_result_resource(in_name, route_results[in_name])

                # Check if there are overuse (shared resource)
                if self.check_for_overuse_resource(mux_visited, resource_names):
                    unrouted_input.append(in_name)

                mux_visited.update(resource_names)

        if unrouted_input:
            # Remove the path from the route_results map
            for in_name in unrouted_input:

                if in_name in route_results:
                    removed_path = route_results[in_name]
                    in2path_removed_map[in_name] = removed_path
                    del route_results[in_name]

        return unrouted_input, in2path_removed_map

@util.gen_util.freeze_it
class BlockInternalConnectivity:
    '''
    Class that captures the internal block mux connectivity. This is an
    approach that is slightly different that the BlockMuxPattern.  Used
    for capturing the Tesseract internal clock mux connectivity.
    '''

    def __init__(self, name, ref_blk):

        # The block name that this mux belongs to
        self.__name = name

        # The reference block itself (PeripheryBlock)
        self.__refblock = ref_blk

        # Filename that stores the connectivity inside the block
        self.__conn_filename = ""

        # Buffer connection input name (key) -> list of output names
        self.__buf_connection = {}

        # The connectivity graph (MuxConnGraph)
        self.__mux_graph = None

    def get_mux_graph(self):
        return self.__mux_graph

    def get_block_name(self):
        return self.__name

    def get_conn_filename(self):
        return self.__conn_filename

    def set_conn_filename(self, filename):
        self.__conn_filename = filename

    def add_buf_connection(self, in_name, out_name):
        out_list = []
        if in_name in self.__buf_connection:
            out_list = self.__buf_connection[in_name]

        out_list.append(out_name)
        self.__buf_connection[in_name] = out_list

    def create_connectivity_graph(self):
        '''
        Create the connectivity graph based on the file and the buffer connection
        :return: False if error is found
        '''
        valid = False

        # Open up the file and parse while creating the edges and vertices

        if self.__mux_graph is None and self.__conn_filename != "" and self.__refblock is not None:

            # Create the graph
            new_graph = MuxConnGraph()

            in_size = 0
            out_size = 0

            with open(self.__conn_filename, encoding='UTF-8') as csvfile:
                read_csv = csv.reader(csvfile, delimiter=',')
                start = True
                valid = True
                lineno = 0

                for row in read_csv:
                    lineno += 1

                    # Skip comments and empty lines
                    if row and not row[0].startswith('#'):

                        if start:
                            out_size = len(row)
                            start = False

                        elif len(row) != out_size:
                            logger.error(
                                "Row size inconsistent: {} vs {}".format(out_size, len(row)))
                            valid = False
                            break
                        elif len(row) != 2 or out_size != 2:
                            # We expect that each line is of 2 names which
                            # represents the edge connection between 2 nodes
                            logger.error(
                                "Expected row to contains 2 elements instead of {}: {}".format(len(row), ",".join(row)))
                            valid = False
                            break

                        # We should be reading 2 entries: from_node,to_node of type string
                        src_node = row[0]
                        sink_node = row[1]

                        if src_node == "":
                            valid = False
                            logger.error("Empty source node found in line {}".format(lineno))
                            continue

                        if src_node == "":
                            valid = False
                            logger.error("Empty sink node found in line {}".format(lineno))
                            continue

                        # If the node doesn't exists we create it provided that they are legal
                        # (exists in the design if they are ports). Mux name should match
                        # the block PCR names which we can verify later
                        if not new_graph.is_vertex_exists(src_node):
                            node_obj = None

                            if src_node.find(":") == -1:
                                valid, vtype = self._determine_port_node_attributes(src_node)
                                if not valid:
                                    continue

                                node_obj = ConnVertex(src_node, vtype)
                            else:
                                valid, vtype, mname, index = self._determine_mux_pin_node_attributes(src_node)
                                if not valid:
                                    continue

                                node_obj = ConnVertex(src_node, vtype, mname, index)

                            new_graph.add_vertex(node_obj)

                        if not new_graph.is_vertex_exists(sink_node):
                            node_obj = None

                            if sink_node.find(":") == -1:
                                valid, vtype = self._determine_port_node_attributes(sink_node)
                                if not valid:
                                    continue

                                node_obj = ConnVertex(sink_node, vtype)
                            else:
                                valid, vtype, mname, index = self._determine_mux_pin_node_attributes(sink_node)
                                if not valid:
                                    continue

                                node_obj = ConnVertex(sink_node, vtype, mname, index)

                            new_graph.add_vertex(node_obj)

                        # Add the edge
                        if not new_graph.add_edge(src_node, sink_node):
                            logger.error("Failed to add edges from {} to {}".format(src_node, sink_node))
                            valid = False

            # Save the graph to the object
            if valid:
                self.__mux_graph = new_graph


        return valid

    def _determine_port_node_attributes(self, node_name):
        valid = True
        vtype = None
        from device.block_definition import PortDir

        # This is either input or output. Search for the port name
        port_obj = self.__refblock.find_port(node_name)
        if port_obj is not None:
            if port_obj.get_direction() == PortDir.input:
                vtype = ConnVertex.VertexType.input
            elif port_obj.get_direction() == PortDir.output:
                vtype = ConnVertex.VertexType.output
            elif port_obj.get_direction() == PortDir.inout:
                logger.error("Unsupported inout port on node {}".format(node_name))
                valid = False
            else:
                logger.error("Unsupport undefined port direction on node {}".format(node_name))
                valid = False
        else:
            valid = False
            logger.error("Cannot find the source port node {}".format(node_name))

        return valid, vtype

    def _determine_mux_pin_node_attributes(self, node_name):
        valid = True
        vtype = None
        mux_name = ""
        index = None

        conn_list = node_name.split(":")

        if len(conn_list) == 2:
            mux_name = conn_list[0]
            # Index is a string
            tmp_index = conn_list[1]

            # If the index is an integer, then it's a mux input
            if util.gen_util.is_string_int(tmp_index):
                vtype = ConnVertex.VertexType.mux_in
                index = int(tmp_index)

            # Else it's a mux output
            else:
                vtype = ConnVertex.VertexType.mux_out
                index = tmp_index
        else:
            logger.error("Unexpected mux connectivity: {}".format(node_name))
            valid = False
        return valid, vtype, mux_name, index

    def post_parse_check(self):
        '''
        Check the connection graph after the periphery block has
        been fully parsed.  Things to check for:
        1) The mux names matches with the PCR modes
        2) The mux indexes matches with the PCR mode values

        :return: False if any of failure is found.
        '''
        is_valid = True

        # Check that all mux_in and mux_out nodes are valid
        #if self.__mux_graph is not None:
            # Skip this first as the name doesn't match exactly with the PCR name
            #if not self.__mux_graph.is_mux_check_valid(self.__refblock):
            #    logger.error("Failed checking the muxes in Internal Connectivity Graph for {}".format(self.__name))
            #    is_valid = False

        if self.__mux_graph is None:
            logger.error("Empty Internal Connectivity Graph in block {}".format(self.__name))
            is_valid = False

        return is_valid

    def __getstate__(self):
        state = self.__dict__.copy()

        db_dir = os.getcwd()
        efx_home = os.path.dirname(db_dir)

        home = pathlib.PureWindowsPath(efx_home).as_posix()
        home_len = len(home.split("/"))
        abs_file = pathlib.PureWindowsPath(state['_BlockInternalConnectivity__conn_filename']).as_posix()

        if home in abs_file:
            state['_BlockInternalConnectivity__conn_filename'] = "/".join(abs_file.split('/')[home_len:])

        return state

    def __setstate__(self, state):
        self.__dict__.update(state)

    def __eq__(self, other):
        if isinstance(other, BlockInternalConnectivity):
            return self.__name == other.__name and \
            self.__conn_filename == other.__conn_filename and \
            self.__buf_connection == other.__buf_connection and \
            self.__mux_graph == other.__mux_graph
        return False

    def __hash__(self):
        return super().__hash__()
