"""
Copyright (C) 2017-2018 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Oct 19, 2017

@author: yasmin
"""

from __future__ import annotations
import os
import pathlib
from typing import List, Optional, TYPE_CHECKING
import xml.etree.ElementTree as et
import time
import xmlschema
from unittest.mock import patch
from contextlib import contextmanager
import inspect
import PyQt5.QtCore as qt
from common_device.hyper_ram.hyper_ram_design_patch import HyperRAMPatch
from common_device.lvds.lvds_design import LVDSRegistry

from tx60_device.gpio.gpio_design_comp import GPIORegistryComplex
from tx60_device.lvds.lvds_design_adv import LVDSRegistryAdvance
from tx180_device.mipi_dphy.design import MIPIHardDPHYRegistry
from tx180_device.pll_ssc.design import PLLSSCRegistry
from tx180_device.pll.pll_design_service import PLLDesignServiceV3Complex
from tx375_device.pma_clock_mux.design import PMAClockMuxRegistry
from tx375_device.quad.design import QuadRegistry
from tx375_device.quad_pcie.design import QuadPCIERegistry
from tx375_device.lane10g.design import Lane10GRegistry
from tx375_device.raw_serdes.design import RawSerdesRegistry
from tx375_device.lane1g.design import Lane1GRegistry
from tx375_device.soc.design import SOCRegistry
from tx375_device.soc.design_service import SOCDesignService
from common_device.spi_flash.spi_flash_design import SPIFlashRegistry

from util.singleton_logger import Logger
import util.app_setting as aps
import util.excp as app_excp
import util.gen_util as gen_util
import util.efxproj_setting as efxproj_set
import util.plugin_map as plmap

from _version import __version__

import device.service as dev_srv
import device.db_interface as dev_dbi

import common_device.gpio.gpio_design_service as gds
from common_device.gpio.gpio_design_patch import GPIOPatch

import common_device.pll.pll_design_service as pllds
from common_device.pll.pll_design_patch import PLLPatch

import common_device.osc.osc_design_service as oscds
from common_device.osc.osc_design_patch import OSCPatch
import common_device.jtag.jtag_design_service as jtagds

import common_device.lvds.lvds_design_service as lvdsds
from common_device.lvds.lvds_design_patch import LVDSPatch

import common_device.mipi.mipi_design_service as mipids
from common_device.mipi.mipi_design_patch import MIPIPatch
import common_device.mipi_dphy.mipi_dphy_design_service as mdphyds
from common_device.mipi_dphy.mipi_dphy_design_patch import MIPIDPhyPatch

import common_device.spi_flash.spi_flash_design_service as spfds
import common_device.hyper_ram.hyper_ram_design_service as hpramds

import common_device.ctrl.ctrl_design_service as ctrlds
import common_device.clock_mux.clkmux_design_service as clkmuxds
import common_device.seu.seu_design_service as seuds
from common_device.seu.seu_design_patch import SEUPatch

import common_device.iobank.iobank_design_service as iobds
import common_device.iobank.iobank_design as iobd
from common_device.rclock_mux.rclkmux_design_service import RClockMuxDesignService
from common_device.ext_flash.ext_flash_design_service import ExtFlashControllerService

import common_device.h264.h264_design_service as h264ds
import common_device.ddr.ddr_design_service as ddrds
from common_device.ddr.ddr_design_patch import DDRPatch
from common_device.spi_flash.spi_flash_design_patch import SPIFlashPatch

import an20_device.pll.pll_design_service_adv as adv_pllds

from tx60_device.iobank.iobank_design_adv import IOBankRegistryAdvance
from tx60_device.iobank.iobank_design_patch import IOBankAdvPatch
import tx60_device.pll.pll_design_service_comp as comp_pllds
import tx60_device.gpio.gpio_design_service_comp as comp_gds
import tx60_device.lvds.lvds_design_service_adv as adv_lvdsds
import tx60_device.osc.osc_design_service_adv as adv_oscds
import tx60_device.hposc.hposc_design_service as hposcds
from tx60_device.clock_mux.clkmux_design_service_adv import ClockMuxDesignServiceAdvance
from tx60_device.clock_mux.clkmux_design_patch import ClockMuxPatch
from tx60_device.iobank.iobank_design_service_adv import IOBankDesignServiceAdvance
import tx60_device.iobank.iobank_design_adv as adv_iobd
from tx180_device.clock_mux.clkmux_design_service_comp import ClockMuxDesignServiceComplex
import tx180_device.ddr.ddr_design_service_adv as adv_ddrds
from tx180_device.mipi_dphy.design_service import MIPIHardDPHYDesignService
from tx180_device.mipi_dphy.mipi_dphy_patch import MIPIHardDPHYPatch
from tx180_device.clock_mux.clkmux_design_comp import ClockMuxRegistryComplex
from tx180_device.ddr.ddr_design_adv import DDRAdvance, DDRRegistryAdvance

from tx375_device.clock_mux.clkmux_design_service_v4 import ClockMuxDesignServiceV4
from tx180_device.pll_ssc.design_service import PLLSSCDesignService

from tx375_device.rclock_mux.design_service import RClockMuxV2DesignService
from tx375_device.common_quad.design_service import QuadLaneCommonDesignService
from tx375_device.quad_pcie.design_service import QuadPCIEDesignService
from tx375_device.lane10g.design_service import Lane10GDesignService
from tx375_device.raw_serdes.design_service import RawSerdesDesignService
from tx375_device.lane1g.design_service import Lane1GDesignService
from tx375_device.fpll.design_service import EfxFpllV1DesignService
from tx375_device.lane10g.design_patch import Lane10GPatch
from tx375_device.quad_pcie.design_patch import QuadPCIEPatch
from tx375_device.lane1g.design_patch import Lane1GPatch
from tx375_device.raw_serdes.design_patch import RawSerdesPatch
from tx375_device.common_quad.design_patch import QuadLaneCommonPatch

import design.db as db
import design.service_interface as dbi
import design.excp as db_excp
import design.device_setting as dev_set
import design.design_migrator as dsg_mig

from design.rule_service import RuleCheckService

if TYPE_CHECKING:
    from design.db_item import PeriDesignRegistry
    from device.db import PeripheryDevice
    from common_device.quad.lane_design import LaneBaseRegistry



@gen_util.freeze_it
class DesignIntegrityCheck:
    """
    Provides a quick check to determine the integrity of the loaded design.
    This was created based on PT-841 where there could be a possibility that the
    device_info can be corrupted (empty)
    """

    def __init__(self, design, backup_file):
        self._is_setting_reset = False
        self.design = design
        self.backup_file = backup_file
        # We save everything to this internal device_setting and only
        # override it to the existing design when restore is successful
        self.device_setting = None

        self.logger = Logger

    def copy_device_setting(self):
        if self.design is not None and self.design.device_setting is not None:
            # using copy.deepcopy results in error (recursive)
            # TypeError: cannot pickle '_thread.RLock' object
            self.device_setting = self.design.device_setting.copy_setting()

    def check_design_integrity(self):
        """
        Check that the design integrity is intact.
        :return: True if design valid. False, if it fails the check
        """
        if self.design is None:
            return False

        is_valid = True

        # Check that the device settings is not empty
        if self.design.device_setting is None:
            is_valid = False
        else:
            # Check that the io registry exists.
            if self.design.device_setting.iobank_reg is None:
                is_valid = False

            if self.design.plugin_map is not None:

                if self.design.plugin_map.is_plugin_exist(self.design.device_def, "an20_ctrl"):

                    if self.design.device_setting.ctrl_reg is None or\
                            self.design.device_setting.clkmux_reg is None:
                        is_valid = False

                if self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_ctrl"):

                    if self.design.device_setting.seu_reg is None or \
                            self.design.device_setting.ctrl_reg is None:
                        is_valid = False

                    if self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_spi_flash"):
                        if self.design.device_setting.ext_flash_reg is None:
                            is_valid = False

                if self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_clkmux") or\
                        self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx180_clkmux") or \
                        self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx375_clkmux"):

                    if self.design.device_setting.clkmux_reg is None:
                        is_valid = False

                    if self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx375_clkmux") and \
                            self.design.device_setting.pma_clkmux_reg is None:
                        is_valid = False

        return is_valid

    def _restore_iobank_reg_from_file(self, device_setting):
        is_valid = False

        self.logger.debug("restore IOBank from file")

        iobank_service = iobds.IOBankDesignService()

        # Check if iobank_reg needs to be restored
        iobank_reg = iobank_service.load_design(
            self.backup_file)

        if iobank_reg is not None:
            device_setting.iobank_reg = iobank_reg
            device_setting.iobank_reg.check_loaded_design(
                self.design.device_db)
            is_valid = True

        return is_valid

    def _restore_ctrl_reg_from_file(self, device_setting):
        is_valid = False

        self.logger.debug("restore ctrl reg from file")

        ctrl_service = ctrlds.ControlDesignService()

        ctrl_reg = ctrl_service.load_design(self.backup_file)
        if ctrl_reg is not None:
            ctrl_reg.device_db = self.design.device_db
            device_setting.ctrl_reg = ctrl_reg
            is_valid = True

        return is_valid

    def _restore_seu_reg_from_file(self, device_setting):
        is_valid = False

        self.logger.debug("restore seu reg from file")

        seu_service = seuds.SEUDesignService()

        seu_reg = seu_service.load_design(self.backup_file)
        if seu_reg is not None:
            seu_reg.device_db = self.design.device_db
            device_setting.seu_reg = seu_reg
            is_valid = True

        return is_valid

    def _restore_ext_flash_reg_from_file(self, device_setting):
        is_valid = False

        self.logger.debug("restore ext_flash reg from file")

        ext_flash_service = ExtFlashControllerService()

        ext_flash_reg = ext_flash_service.load_design(self.backup_file)
        if ext_flash_reg is not None:
            device_setting.ext_flash_reg = ext_flash_reg
            is_valid = True

        return is_valid

    def _restore_clkmux_reg_from_file(self, device_setting):
        is_valid = False

        self.logger.debug("restore clkmux reg from file")
        clkmux_svc = None

        if self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx375_clkmux"):
            clkmux_svc = ClockMuxDesignServiceV4()

        elif self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx180_clkmux"):
            clkmux_svc = ClockMuxDesignServiceComplex()

        elif self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_clkmux"):
            clkmux_svc = ClockMuxDesignServiceAdvance()

        if clkmux_svc is not None:
            clkmux_reg = clkmux_svc.load_design(self.backup_file)
            if clkmux_reg is not None:
                device_setting.clkmux_reg = clkmux_reg
                is_valid = True

        return is_valid

    def _restore_backup(self):
        """
        Restore the information that is missing or corrupted
        :return:
        """
        backup_fully_restored = True

        # Check if the restore file exists
        if self.backup_file != "" or self.backup_file is not None:
            if os.path.exists(self.backup_file):
                # Read the device_setting section from the file assuming
                # that the file is valid
                self.logger.debug("Restoring device setting using backup file: {}".format(
                    self.backup_file))

                new_dev_set = False
                if self.device_setting is None:
                    self.device_setting = dev_set.DeviceSetting()
                    new_dev_set = True

                if self.device_setting.iobank_reg is None:
                    if not self._restore_iobank_reg_from_file(self.device_setting):
                        backup_fully_restored = False

                # Control block is not part of instantiable design block.
                # Need to check manually since is_block_supported() cannot be
                # used.
                if self.design.plugin_map.is_plugin_exist(self.design.device_def, "an20_ctrl") or\
                        self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_ctrl"):
                    if new_dev_set or self.device_setting.ctrl_reg is None:
                        # Load ctrl reg
                        if not self._restore_ctrl_reg_from_file(self.device_setting):
                            backup_fully_restored = False

                # Restore the SEU
                if self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_ctrl"):
                    if new_dev_set or self.device_setting.seu_reg is None:
                        # Load seu reg
                        if not self._restore_seu_reg_from_file(self.device_setting):
                            backup_fully_restored = False

                # Restore the external flash setting
                if self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_ctrl") and\
                        self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_spi_flash"):
                    if new_dev_set or self.device_setting.ext_flash_reg is None:
                        # Load ext flash reg
                        if not self._restore_ext_flash_reg_from_file(self.device_setting):
                            backup_fully_restored = False

                # Restore the Advance and complex clockmux only since that
                # is when it has info in the peri.xml. For Trion, we will build default
                # which is handled in _restore_default function since the clkmux info does not get
                # saved to the peri.xml file.
                if self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_clkmux") or \
                        self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx180_clkmux") or \
                        self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx375_clkmux"):
                    if new_dev_set or self.device_setting.clkmux_reg is None:
                        # Load clkmux
                        if not self._restore_clkmux_reg_from_file(self.device_setting):
                            backup_fully_restored = False

            else:
                backup_fully_restored = False
        else:
            backup_fully_restored = False

        return backup_fully_restored

    def _restore_default(self):
        """
        Restore the default setting for the corrupted data
        :return:
        """

        if self.design.device_db is not None:
            self.logger.debug("Restoring using default setting")

            if self.device_setting is None:
                devset_svc = DesignSettingService()
                device_setting = devset_svc.create_setting(self.design)
                if device_setting is not None:
                    self.device_setting = device_setting
                    self._is_setting_reset = True

            else:
                if self.device_setting.iobank_reg is None:
                    devset_svc = DesignSettingService(self.design)
                    iobank_reg = devset_svc.build_iobank(self.design.device_db)
                    if iobank_reg is not None:
                        self.device_setting.iobank_reg = iobank_reg
                        self._is_setting_reset = True

                if self.design.plugin_map.is_plugin_exist(self.design.device_def, "an20_ctrl"):
                    if self.device_setting.ctrl_reg is None:
                        devset_svc = DesignSettingService(self.design)
                        ctrl_reg = devset_svc.build_ctrl(self.design.device_db)
                        if ctrl_reg is not None:
                            self.device_setting.ctrl_reg = ctrl_reg
                            self._is_setting_reset = True

                if self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_ctrl"):
                    devset_svc = DesignSettingService(self.design)

                    if self.device_setting.ctrl_reg is None:
                        ctrl_reg = devset_svc.build_ctrl(self.design.device_db)
                        if ctrl_reg is not None:
                            self.device_setting.ctrl_reg = ctrl_reg
                            self._is_setting_reset = True

                    if self.device_setting.seu_reg is None:
                        seu_reg = devset_svc.build_seu(self.design.device_db)
                        if seu_reg is not None:
                            self.device_setting.seu_reg = seu_reg
                            self._is_setting_reset = True

                    if self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_spi_flash"):
                        if self.device_setting.ext_flash_reg is None:
                            ext_flash_reg = devset_svc.build_ext_flash_controller()
                            if ext_flash_reg is not None:
                                self.device_setting.ext_flash_reg = ext_flash_reg
                                self._is_setting_reset = True

                if self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_clkmux") or \
                        self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx180_clkmux") or \
                        self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx375_clkmux") or \
                        self.design.plugin_map.is_plugin_exist(self.design.device_def, "an20_pll"):
                    # Rebuild the clkmux where it has no entry in design file but it gets
                    # built each time design is created or loaded when T20 and later

                    if self.device_setting.clkmux_reg is None:
                        devset_svc = DesignSettingService(self.design)

                        # The build_<devset> will again filter accordingly
                        # (whether to create or not a clkmux reg)
                        clkmux_reg = devset_svc.build_clkmux(self.design.device_db)
                        if clkmux_reg is not None:
                            self.device_setting.clkmux_reg = clkmux_reg
                            self._is_setting_reset = True

                    if self.device_setting.rclkmux_reg is None and\
                            self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx180_clkmux"):
                        devset_svc = DesignSettingService(self.design)

                        # The build_<devset> will again filter accordingly
                        # (whether to create or not a clkmux reg)
                        rclkmux_reg = devset_svc.build_regional_clkmux(self.design.device_db)
                        if rclkmux_reg is not None:
                            self.device_setting.rclkmux_reg = rclkmux_reg
                            self._is_setting_reset = True

                        # TODO: Build rclkmux for Ti375 which is of diff architecture

                    if self.device_setting.pma_clkmux_reg is None and\
                            self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx375_clkmux"):
                        devset_svc = DesignSettingService(self.design)

                        # The build_<devset> will again filter accordingly
                        # (whether to create or not a clkmux reg)
                        pma_clkmux_reg = devset_svc.build_pma_clkmux(self.design.device_db)
                        if pma_clkmux_reg is not None:
                            self.device_setting.pma_clkmux_reg = pma_clkmux_reg
                            self._is_setting_reset = True

            is_restored = True

        else:
            # If device_db is None, we don't restore anything because
            # it means that there was really a serious problem
            is_restored = False

        return is_restored

    def restore_design(self):
        """
        Restore the missing information
        :return:
        """
        restore_successful = False

        try:
            self.logger.debug("Restoring design device setting")

            # Read the information from the backup file only if design device has been loaded
            # successfully (not None)
            if os.path.exists(self.backup_file) and self.design.device_db is not None:
                if not self._restore_backup():
                    if self._restore_default():
                        restore_successful = True

                else:
                    restore_successful = True

            else:
                if self._restore_default():
                    restore_successful = True

        except Exception:
            self.logger.debug("Error restoring design")

        return restore_successful

    def is_device_setting_reset(self):
        if self.device_setting is not None and self._is_setting_reset:
            return True

        return False


@gen_util.freeze_it
class DesignService:
    """
    Provides design related operation
    """

    _is_plugin_mockup = True  #: By default, plugin is loaded from mockup for unit test

    _is_backup_design = False  #: Do not backup when testing

    def __init__(self):
        """
        Constructor
        """
        self.logger = Logger
        #: Indicator if any of the device resource name was migrated
        self.is_resource_name_migration = False
        self.is_bus_member_pin_name_regen = False

        # : Indicator that the device setting was reset to default
        #   due to unexpected empty device_info field when loading
        self.is_reset_device_setting_to_default = False

        self._change_msg = []  #: Change messages for backward compatibility updates
        self.is_check_device = True # For skipping the xml device validation

    def is_design_upgraded(self):
        """
        Check if the design has been upgraded for backward compatibility purpose

        :return: True, yes else False
        """
        return len(self._change_msg) > 0

    def get_change_msg(self):
        """
        Get change messages generated by block's builder

        :return: A list of messages
        """
        return self._change_msg

    def create_design(self, name, device_name, design_path, is_save=True):
        """
        Create in-memory design.

        :param name: Design name, if use with Efinity, this is Efinity project name
        :param device_name: Device use for this periphery design
        :param design_path: Design file location
        :param is_save: True, write design to disk else does not write.
        :return: A design
        """

        # Friend class
        builder = DesignBuilderEmpty()
        # pylint: disable=protected-access
        builder._is_plugin_mockup = self._is_plugin_mockup
        builder.is_check_device = self.is_check_device
        design = builder.build(name, device_name, design_path)
        design_file = self.get_design_filename(design_path, name)

        if not os.path.exists(design_path):
            pathlib.Path(design_path).mkdir(parents=True, exist_ok=True)

        if is_save:
            self.save_design(design, design_file)

        return design

    def backup_design(self, design_file):
        """
        Save the design file as a backup in the <design_path>/work_pt directory

        :return: The backup file path if exists
        """
        backup_file = ""
        if os.path.exists(design_file):
            design_dir = os.path.dirname(design_file)

            work_dir = os.path.join(os.path.abspath(design_dir), "work_pt")

            if not os.path.exists(work_dir):
                os.makedirs(work_dir)

            # save the backup file named as peri_load.bak
            backup_file = os.path.join(work_dir, "peri_load.bak")

            self.logger.debug("Backup {} to {}".format(
                design_file, backup_file))

            # Delete first if it already exists
            if os.path.exists(backup_file):
                os.remove(backup_file)

            from shutil import copyfile
            copyfile(design_file, backup_file)

        return backup_file

    def check_design_integrity(self, design, design_backup_file, design_file):

        integrity_chk = DesignIntegrityCheck(design, design_backup_file)

        if not integrity_chk.check_design_integrity():
            self.logger.debug("Failed design integrity check")
            integrity_chk.copy_device_setting()

            if integrity_chk.restore_design():
                design.device_setting = integrity_chk.device_setting

                # Save if it was restored
                self.save_design(design, design_file)

                if integrity_chk.is_device_setting_reset():
                    self.logger.debug("Reset the device setting to default")
                    self.is_reset_device_setting_to_default = True

    def load_design(self, design_file, ignore_device_proj_setting=False) -> db.PeriDesign:
        """
        Create design from a design file

        :param design_file: Full path to design file name
        :param ignore_device_proj_setting: Flag to indicate if we can
                    ignore the project setting when loading device
                    (such as timing model and prog mode) so that it
                    stick to default.
        :return: A design
        :raise: design.LoadDesignException
        """

        # Skip backing up if we're doing test
        design_backup_file = ""
        if self._is_backup_design:
            design_backup_file = self.backup_design(design_file)

        # Friend class
        # pylint: disable=protected-access
        builder = DesignBuilderXml(ignore_device_proj_setting)
        builder._is_plugin_mockup = self._is_plugin_mockup
        try:
            builder.is_check_device = self.is_check_device
            design = builder.build(design_file)

            # Get the design upgrade change message, if any
            self._change_msg = builder.change_msg

            # Save the design if there was any migration
            # of device instance resource name change
            if design is not None:
                if builder.is_resource_name_changed():
                    self.save_design(design, design_file)
                    self.is_resource_name_migration = True
                elif len(self._change_msg) > 0:
                    self.save_design(design, design_file)

                # TODO: Should we save the design as well when regen member pin
                self.is_bus_member_pin_name_regen = \
                    builder.is_bus_member_pin_name_regen()

                # Check design integrity only if we enable backup
                if self._is_backup_design and design is not None:
                    self.check_design_integrity(
                        design, design_backup_file, design_file)

        except db_excp.DesignFileException as excp:
            msg = "Fail to load design from {}".format(design_file)
            raise db_excp.LoadDesignException(
                msg, app_excp.MsgLevel.error, excp)

        if design is None:
            msg = "Fail to load design from {}".format(design_file)
            raise db_excp.LoadDesignException(msg, app_excp.MsgLevel.error)

        return design

    def save_design(self, design, design_file=None):
        """
        Write in-memory design to a file.

        :param design: Design to write
        :param design_file: Target file to write to. If None, the file name in design will be used.
        """
        saver = DesignWriterXml()

        if design_file is not None:
            saver.write(design, design_file)
        else:
            saver.write(design, design.design_file)

    def check_device_status(self, design, is_wait_load=False):
        """
        Check if device in design is in sync with Efinity project device

        :param design : Design database
        :param is_wait_load : Set to true if it needs to delay proj file reading.
        :return: None if device match, a tuple of new device name and family, if device has changed
        """
        error_msg = ""
        try:
            setting = efxproj_set.EfxProjectSetting.build(design.name, design.location,
                                                          [efxproj_set.EfxProjectSetting.LoadFilterType.device],
                                                          is_wait_load)
            if setting is not None:
                device_info = setting.get_device_info()
                if device_info is not None:
                    project_device = device_info.get("device", "")
                    if project_device == design.device_def:
                        return None
                    else:
                        return project_device, device_info.get("family", "")
                else:
                    error_msg = "Device info is None"
            else:
                error_msg = "Efinity project setting is None"

        except app_excp.PTFileException as exc:
            error_msg = "File error : {}".format(exc.msg)

        if error_msg != "":
            raise db_excp.CheckDeviceException(
                error_msg, app_excp.MsgLevel.error)

    def check_design_name_status(self, design, is_wait_load=False):
        """
        Check if design name is in sync with Efinity project name

        :param design : Design database
        :param is_wait_load : Set to true if it needs to delay proj file reading.
        :return: None if name match else return new Efinity project name
        """
        error_msg = ""
        try:
            setting = efxproj_set.EfxProjectSetting.build(design.name, design.location,
                                                          [efxproj_set.EfxProjectSetting.LoadFilterType.proj],
                                                          is_wait_load)
            if setting is not None:
                proj_info = setting.get_proj_info()
                if proj_info is not None:
                    project_name = proj_info.get("name", "")
                    if project_name == design.name:
                        return None
                    else:
                        return project_name
                else:
                    error_msg = "Project info is None"
            else:
                error_msg = "Efinity project setting is None"

        except app_excp.PTFileException as exc:
            error_msg = "File error : {}".format(exc.msg)

        if error_msg != "":
            raise db_excp.CheckDeviceException(
                error_msg, app_excp.MsgLevel.error)

    def check_design_file(self, design_file):
        """
        Validate the design file

        :param design_file: design file
        :returns True if good, else False
        :raises design.CheckDesignException
        """
        builder = DesignBuilderXml()
        return builder.validate(design_file)

    def check_design(self, design, result_file=""):
        """
        Check each block instance against its design rule.

        :param design: Design
        :param result_file: File where the result will be written to
        """
        # TODO : result_file will be deprecated. Use DesignIssueRegistry to
        # write result
        gen_util.mark_unused(result_file)

        checker_service = RuleCheckService(design)
        return checker_service.run(RuleCheckService.RunModeType.STD)

    def get_design_filename(self, design_path, name=""):
        """
        Get design filename.

        :param design_path: Full path to design file directory
        :param name: Project name aka design name. If not specified, it will be auto-detected.
        :return: Design file full path name
        """

        full_path = os.path.abspath(design_path)

        if name != "":
            design_file = os.path.normpath(
                '{}/{}.peri.xml'.format(full_path, name))
        else:
            design_file = self.find_design_file(full_path)

        return design_file

    def find_design_file(self, full_design_path):
        """
        Find periphery design file in specified directory. If there is multiple
        design files, when it shouldn't, it will just take the first one.

        :param full_design_path: Full path to design directory
        :return: Design file name, if found, else emptry string
        """
        design_file = ""
        try:
            if os.path.exists(full_design_path):
                for file in os.listdir(full_design_path):
                    if file.endswith(".peri.xml"):
                        design_file = os.path.normpath(
                            '{}/{}'.format(full_design_path, file))
                        break
        except os.error as excp:
            msg = "Exception when searching for design file at {}".format(
                full_design_path)
            self.logger.error(msg)
            self.logger.error("{}".format(excp))

        return design_file

    def get_efxproj_file(self, design):
        """
        Get Efinity project file name. The periphery design name is the same as Efinity project name.
        Both Efinity and periphery design are stored at the same location.

        :param design: A design instance
        :return: A Efinity project filename
        """

        efxproj_filename = "{}/{}.xml".format(design.location, design.name)
        try:
            checked_file = efxproj_set.EfxProjectSetting.check_efx_project_file(
                efxproj_filename)
            efxproj_filename = checked_file
        except app_excp.PTFileException:
            self.logger.warning(
                "File does not exist, {}".format(efxproj_filename))

        self.logger.debug("Efinity project file {}".format(efxproj_filename))

        return efxproj_filename

    def sync_name_design_efxproject(self, design, design_file, design_path, is_save=True):
        """
        Synchronize design info againsts Efinity project info. Info to sync
        - design name and Efinity project name
        - design name and design file name, if design name change

        :param is_save: True, save to design file, else no
        :param design: A design object
        :param design_file: Design file that match the design object
        :param design_path: Full path to design file
        :return : A tuple of a status flag and design file name.
                  The status flag is true if there is a change.
                  The design file name is the updated file name,
                  if there is a change, else existing name.

        """

        is_change = False
        # Check design name
        new_design_name = self.check_design_name_status(design)
        if new_design_name is not None:  # There has been a change

            is_change = True
            self.logger.warning("Design name changed from {} to {}".format(
                design.name, new_design_name))

            # Update design name
            design.name = new_design_name

            old_design_file = design_file
            self.logger.info("Old design file : " + old_design_file)

            # Update design file name in design and physical file
            design_file = self.get_design_filename(design_path, design.name)
            design.design_file = design_file
            self.logger.info("New design file : " + design_file)

            if is_save:
                self.logger.info("Save new design file : " + design_file)
                self.save_design(design, design_file)

            # PT-848: Only delete the file if it is not the same name
            # as the new design file, regardless if we saved it or not
            if os.path.exists(old_design_file) and \
                    os.path.abspath(old_design_file) != os.path.abspath(design_file):
                self.logger.info("Remove old design file : " + old_design_file)
                os.remove(old_design_file)

        return is_change, design_file

    def apply_new_device(self, design, new_device, is_update):
        """
        Apply new device update. The design maybe recreated or updated.

        :param design: Design instance
        :param new_device: New device name
        :param is_update: True, update the design. False, delete and create a new one
        :return: A design instance
        """

        mig_file_found = True

        if is_update:
            # Update device
            mig = dsg_mig.DesignMigrator()
            # Write migration log to outflow directory
            mig.enable_migration_log(True)
            mig_file_found = mig.migrate(design, new_device)

            # Save
            self.save_design(design, design.design_file)

            return design, mig_file_found
        else:
            # Create new design
            new_design = self.create_design(
                design.name, new_device, design.location)
            return new_design, mig_file_found

    @staticmethod
    def get_proj_device_name(design_name, design_loc, is_wait_load=False):
        """
        Check if device in design is in sync with Efinity project device

        :param design_name: The design name found in he Interface Designer.
                    Need to make sure that the name here matches with the
                    Efinity project name.
        :param design_file: The Interface Designer filename
        :param is_wait_load : Set to true if it needs to delay proj file reading.
        :return: the project device name
        """
        error_msg = ""
        project_device = ""

        try:

            setting = efxproj_set.EfxProjectSetting.build(design_name, design_loc,
                                                          [efxproj_set.EfxProjectSetting.LoadFilterType.device],
                                                          is_wait_load)
            if setting is not None:
                device_info = setting.get_device_info()
                if device_info is not None:
                    project_device = device_info.get("device", "")

                else:
                    error_msg = "Device info is None"
            else:
                error_msg = "Efinity project setting is None"

        except app_excp.PTFileException as exc:
            error_msg = "File error : {}".format(exc.msg)

        if error_msg != "":
            raise db_excp.CheckDeviceException(
                error_msg, app_excp.MsgLevel.error)

        return project_device

    @staticmethod
    def check_device_migration(design_file):
        """
        Check if there is a device mismatch between the name
        in the design file and efx project file.  This will
        be an indicator if migration is required.

        :param design_file: The design file
        :return True if migration is required. False, otherwise.
                Followed by the device name in peri.xml and
                device name in efx project file
        """

        peri_dev_name = ""
        efx_dev_name = ""
        is_migration = False

        try:
            logger = Logger

            # Get the device name from design file
            builder = DesignBuilderXml()
            design_name, file_dev_name = builder.get_device_project_in_design_file(
                design_file)
            logger.debug("Read proj {} device {} from {}".format(
                design_name, file_dev_name, design_file))

            # Get the device name from efx project file
            design_loc = os.path.abspath(os.path.dirname(design_file))
            proj_dev_name = DesignService.get_proj_device_name(
                design_name, design_loc, is_wait_load=True)

            logger.debug(
                "Read device {} from project file".format(proj_dev_name))

            peri_dev_name = file_dev_name
            efx_dev_name = proj_dev_name

            if proj_dev_name != file_dev_name:
                is_migration = True

        except db_excp.DesignFileException as excp:
            msg = "Fail to get device info from {}".format(design_file)
            raise db_excp.LoadDesignException(
                msg, app_excp.MsgLevel.error, excp)

        except db_excp.CheckDeviceException as excp:
            raise excp

        return is_migration, peri_dev_name, efx_dev_name


@gen_util.freeze_it
class DesignServiceMockupPlugin():
    """
    Context manager for DesignService used for unit test where
    mockup plugin map is used.

    The context manager make sure static data is reset properly
    when the DesignService object is disposed. This is important
    when multiple design db are created in memory.
    """

    def __enter__(self):
        DesignService._is_plugin_mockup = True
        return DesignService()

    def __exit__(self, *args):
        DesignService._is_plugin_mockup = False

def DesignServiceConcretePlugin():
    curframe = inspect.currentframe()

    @contextmanager
    def patch_design_service():
        def mock_save_func(*args, **kwargs):
            curframe = kwargs.get('curframe')
            calframe = inspect.getouterframes(curframe, 2)
            caller_info = calframe[1][1] + " function_name: " + calframe[1][3]
            print(caller_info)
            # For checking called test
            # outfile = pathlib.Path(__file__).parent / "test" / "called_saved_design_test.txt"
            # with open(str(outfile), 'a', encoding='utf-8') as f:
            #     f.write(caller_info + "\n")

        patcher = None
        ori_is_plugin_mockup = DesignService._is_plugin_mockup

        try:
            patcher = patch.object(DesignService, 'save_design')
            mock_func = patcher.start()
            mock_func.side_effect = lambda *args, **kwargs: mock_save_func(*args, **kwargs, curframe=curframe)
            DesignService._is_plugin_mockup = False
            service = DesignService()
            service.is_check_device = False
            yield service

        finally:
            if patcher:
                patcher.stop()
            DesignService._is_plugin_mockup = ori_is_plugin_mockup

    return patch_design_service()


@gen_util.freeze_it
class DesignBuilderEmpty:
    """
    Build an empty design
    """
    _is_plugin_mockup = True  #: By default, plugin is loaded from mockup for unit test

    def __init__(self):
        self.design = None
        self.prog_info = None
        self.logger = Logger
        self.is_check_device = True

    def build(self, name, device_name, design_path):

        self.design = db.PeriDesign(name, device_name)

        # TODO : Having both design file and location is redundant
        self.design.design_file = os.path.normpath(
            '{}/{}.peri.xml'.format(design_path, name))
        self.design.location = os.path.normpath(design_path)
        self.design.version = "{}".format(__version__)

        self.logger.info('Create design : {}'.format(self.design.design_file))

        # Build device-plugins map
        if self._is_plugin_mockup:
            plugin_map = plmap.PluginMap.build_mockup()
            self.design.setup_plugin_map(plugin_map)
        else:
            self.design.setup_plugin_map()

        self.logger.info("Supported blocks : {}".format(
            self.design.get_supported_block()))

        # Save the prog setting so that can be passed to load_device
        self._get_prog_info()

        self._build_device()
        if self.design.device_db is not None:
            devset_svc = DesignSettingService()
            self.design.device_setting = devset_svc.create_setting(self.design)
            if self.design.device_setting is None:
                msg = "Fail to build device setting for {}".format(
                    self.design.device_def)
                raise db_excp.DeviceSettingException(
                    msg, app_excp.MsgLevel.error)
        else:
            # This should happen only in unit test
            self.logger.warning(
                "Device db is None. Skipped building device setting.")

        self._build_block_registry()
        self.design.setup_tesseract_hsio_service()
        self.design.setup_serdes_res_service()

        # Populate clkmux info (need to be done after the registry of all
        # blocks are built)
        if self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_clkmux") or \
                self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx180_clkmux") or\
                self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx375_clkmux"):
            if self.design.device_setting is not None and self.design.device_setting.clkmux_reg is not None:
                self.design.device_setting.clkmux_reg.load_device_clkmux_ins(
                    self.design.device_db, self.design)


        return self.design

    def _build_block_registry(self):
        assert self.design is not None

        supp_block = self.design.get_supported_block()
        for block_type in supp_block:

            # L2 gpio still use the same gpio design class since its property
            # does not change
            if block_type == db.PeriDesign.BlockType.gpio or block_type == db.PeriDesign.BlockType.adv_l2_gpio:
                gpio_service = gds.GPIODesignService()
                self.design.gpio_reg = gpio_service.create_registry()
                self.design.gpio_reg.device_name = self.design.device_def
                self.design.gpio_reg.device_db = self.design.device_db

                # Special case for gpio because the resource
                # allocator needs to list the bank name associated
                # to each gpio. This is how it's done during load. So
                # during creating a new design, the same thing is needed.
                self.design.gpio_reg.create_gpio_to_bank_map(
                    self.design.device_db)

            elif block_type == db.PeriDesign.BlockType.comp_gpio:
                gpio_service = comp_gds.GPIODesignServiceComplex()
                self.design.gpio_reg = gpio_service.create_registry()
                self.design.gpio_reg.device_name = self.design.device_def
                self.design.gpio_reg.apply_device_db(self.design.device_db)

                # Special case for gpio because the resource
                # allocator needs to list the bank name associated
                # to each gpio. This is how it's done during load. So
                # during creating a new design, the same thing is needed.
                self.design.gpio_reg.create_gpio_to_bank_map(
                    self.design.device_db)

            elif block_type == db.PeriDesign.BlockType.pll:
                pll_service = pllds.PLLDesignService()
                self.design.pll_reg = pll_service.create_registry()
                self.design.pll_reg.device_db = self.design.device_db

            elif block_type == db.PeriDesign.BlockType.adv_pll:
                pll_service = adv_pllds.PLLDesignServiceAdvance()
                self.design.pll_reg = pll_service.create_registry()
                self.design.pll_reg.device_db = self.design.device_db

            elif block_type == db.PeriDesign.BlockType.comp_pll:
                pll_service = comp_pllds.PLLDesignServiceComplex()
                self.design.pll_reg = pll_service.create_registry()
                self.design.pll_reg.apply_device_db(self.design.device_db)
            elif block_type == db.PeriDesign.BlockType.efx_pll_v3_comp:
                pll_service = PLLDesignServiceV3Complex()
                self.design.pll_reg = pll_service.create_registry()
                self.design.pll_reg.apply_device_db(self.design.device_db)
            elif block_type == db.PeriDesign.BlockType.efx_fpll_v1:
                pll_service = EfxFpllV1DesignService()
                self.design.pll_reg = pll_service.create_registry()
                self.design.pll_reg.apply_device_db(self.design.device_db)

            elif block_type == db.PeriDesign.BlockType.osc:
                osc_service = oscds.OSCDesignService()
                self.design.osc_reg = osc_service.create_registry()
                self.design.osc_reg.device_db = self.design.device_db

            elif block_type == db.PeriDesign.BlockType.adv_osc:
                osc_service = adv_oscds.OSCDesignServiceAdvance()
                self.design.osc_reg = osc_service.create_registry()
                self.design.osc_reg.apply_device_db(self.design.device_db)

            elif block_type == db.PeriDesign.BlockType.hposc:
                osc_service = hposcds.HPOSCDesignService()
                self.design.osc_reg = osc_service.create_registry()
                self.design.osc_reg.apply_device_db(self.design.device_db)

            elif block_type == db.PeriDesign.BlockType.lvds:
                lvds_service = lvdsds.LVDSDesignService()
                self.design.lvds_reg = lvds_service.create_registry()
                self.design.lvds_reg.device_db = self.design.device_db
                self.design.lvds_reg.apply_device_db(self.design.device_db)

            elif block_type == db.PeriDesign.BlockType.adv_lvds:
                lvds_service = adv_lvdsds.LVDSDesignServiceAdvance()
                self.design.lvds_reg = lvds_service.create_registry()
                self.design.lvds_reg.device_db = self.design.device_db
                self.design.lvds_reg.apply_device_db(self.design.device_db)

            elif block_type == db.PeriDesign.BlockType.mipi:
                mipi_service = mipids.MIPIDesignService()
                self.design.mipi_reg = mipi_service.create_registry()
                self.design.mipi_reg.apply_device_db(self.design.device_db)

            elif block_type == db.PeriDesign.BlockType.mipi_hard_dphy:
                service = MIPIHardDPHYDesignService()
                self.design.mipi_hard_dphy_reg = service.create_registry()
                self.design.mipi_hard_dphy_reg.apply_device_db(self.design.device_db)

            elif block_type == db.PeriDesign.BlockType.jtag:
                jtag_service = jtagds.JTAGDesignService()
                self.design.jtag_reg = jtag_service.create_registry()
                self.design.jtag_reg.apply_device_db(self.design.device_db)

            elif block_type == db.PeriDesign.BlockType.h264:
                h264_service = h264ds.H264DesignService()
                self.design.h264_reg = h264_service.create_registry()
                self.design.h264_reg.apply_device_db(self.design.device_db)

            elif block_type == db.PeriDesign.BlockType.ddr:
                ddr_service = ddrds.DDRDesignService()
                self.design.ddr_reg = ddr_service.create_registry()
                self.design.ddr_reg.apply_device_db(self.design.device_db)

            elif block_type == db.PeriDesign.BlockType.adv_ddr:
                ddr_service = adv_ddrds.DDRDesignServiceAdvance()
                self.design.ddr_reg = ddr_service.create_registry()
                self.design.ddr_reg.apply_device_db(self.design.device_db)

            elif block_type == db.PeriDesign.BlockType.mipi_dphy:
                mdphy_service = mdphyds.MIPIDPhyDesignService()
                self.design.mipi_dphy_reg = mdphy_service.create_registry()
                self.design.mipi_dphy_reg.apply_device_db(
                    self.design.device_db)

            elif block_type == db.PeriDesign.BlockType.spi_flash:
                spf_service = spfds.SPIFlashDesignService()
                self.design.spi_flash_reg = spf_service.create_registry()
                self.design.spi_flash_reg.apply_device_db(
                    self.design.device_db)

            elif block_type == db.PeriDesign.BlockType.hyper_ram:
                hpram_service = hpramds.HyperRAMDesignService()
                self.design.hyper_ram_reg = hpram_service.create_registry()
                self.design.hyper_ram_reg.apply_device_db(
                    self.design.device_db)

            elif block_type == db.PeriDesign.BlockType.pll_ssc:
                service = PLLSSCDesignService()
                self.design.pll_ssc_reg = service.create_registry()
                self.design.pll_ssc_reg.apply_device_db(self.design.device_db)

            elif block_type == db.PeriDesign.BlockType.soc:
                self.design.soc_reg = SOCRegistry()
                self.design.soc_reg.apply_device_db(self.design.device_db)

            elif block_type == db.PeriDesign.BlockType.quad:
                self.design.quad_reg = QuadRegistry()
                self.design.quad_reg.apply_device_db(self.design.device_db)
                service = QuadLaneCommonDesignService(self.design.device_db)
                self.design.common_quad_lane_reg = service.create_registry()

            elif block_type == db.PeriDesign.BlockType.quad_pcie:
                service = QuadPCIEDesignService()
                self.design.quad_pcie_reg = service.create_registry()
                self.design.quad_pcie_reg.apply_device_db(self.design.device_db)

            elif block_type == db.PeriDesign.BlockType.lane_10g:
                service = Lane10GDesignService()
                self.design.lane_10g_reg = service.create_registry()
                self._build_lane_based_registry(self.design.lane_10g_reg)
                self.design.lane_10g_reg.load_settings_file(self.design.location)

            elif block_type == db.PeriDesign.BlockType.raw_serdes:
                service = RawSerdesDesignService()
                self.design.raw_serdes_reg = service.create_registry()
                self._build_lane_based_registry(self.design.raw_serdes_reg)
                #self.design.raw_serdes_reg.load_settings_file(self.design.location)

            elif block_type == db.PeriDesign.BlockType.lane_1g:
                service = Lane1GDesignService()
                self.design.lane_1g_reg = service.create_registry()
                self._build_lane_based_registry(self.design.lane_1g_reg)

    def _build_lane_based_registry(self, reg: LaneBaseRegistry):
        assert self.design is not None and self.design.device_db is not None
        # Make sure common quad lane reg is not None since it will be used later for
        # lane based registry
        assert self.design.common_quad_lane_reg is not None

        reg.common_quad_reg = self.design.common_quad_lane_reg
        assert reg.common_quad_reg is not None
        reg.apply_device_db(self.design.device_db)

    def _get_prog_info(self):
        """
        Save the prog_info from the efxproj setting file
        """
        self.prog_info = None

        try:
            setting = efxproj_set.EfxProjectSetting.build(
                self.design.name, self.design.location,
                [efxproj_set.EfxProjectSetting.LoadFilterType.prog])

            if setting is not None:
                self.prog_info = setting.get_prog_info()

        except app_excp.PTFileException as exc:
            self.logger.warning("File error : {}".format(exc.msg))

    def _get_timing_model_name(self):
        """
        Return timing model info retrieved from project file

        :return timing model name
        """
        timing_model_name = ""

        try:
            setting = efxproj_set.EfxProjectSetting.build(
                self.design.name, self.design.location,
                [efxproj_set.EfxProjectSetting.LoadFilterType.device])

            if setting is not None:
                device_info = setting.get_device_info()
                if device_info is not None:
                    timing_model_name = device_info.get("timing_model", "")
                else:
                    self.logger.warning(
                        'Unable to retrieve timing_model name from project file')

        except app_excp.PTFileException as exc:
            self.logger.warning("File error : {}".format(exc.msg))

        return timing_model_name

    def _build_device(self):

        timing_model_name = self._get_timing_model_name()

        dev_service = dev_srv.DeviceService()
        dev_service.is_check = self.is_check_device
        self.design.device_db = dev_service.load_device(
            self.design.device_def, self.prog_info,
            timing_model_name)

        if self.design.device_db is None:
            self.logger.error(
                "Fail to load device database for {}".format(self.design.device_def))
        else:
            self.logger.debug(
                "Load device database for {}".format(self.design.device_def))

        return self.design


@gen_util.freeze_it
class DesignBuilderXml(dbi.DesignBuilder):
    """
    Build design from xml design file
    """

    _ns = "{http://www.efinixinc.com/peri_design_db}"
    '''
    XML Namespace for peri design db
    '''

    _schema_file = ""
    '''
    If design schema is empty, it will be detected automatically using $EFXPT_HOME.
    This is mainly for unit testing to set schema from different location.
    '''

    _is_build_device = True

    _is_plugin_mockup = True  #: By default, plugin is loaded from mockup for unit test

    def __init__(self, ignore_device_proj_setting=False):
        super().__init__()
        self.design = None
        self.prog_info = None
        self.logger = Logger
        self._is_design_save_needed = False

        # : Indicator if the resource name has to be changed (PT-424)
        self._resource_name_changed = False

        # : Indicator that a bus pin name has been regenerated to
        #   follow the standardized name
        self._is_bus_member_pin_name_regen = False

        # A map of older instance name mapped to new name
        self.old2new_resource_map = {}

        self.change_msg = []
        self.new_db_version = None

        self._ignore_device_proj_setting = ignore_device_proj_setting

        self.device_setting_last_patch_no = None
        self.is_check_device = True

    @staticmethod
    def is_unittest_design():
        return DesignBuilderXml._is_plugin_mockup

    def is_resource_name_changed(self):
        return self._resource_name_changed

    def is_bus_member_pin_name_regen(self):
        return self._is_bus_member_pin_name_regen

    def build(self, design_file):

        self.design = self._build_event_based(design_file)
        self.design.setup_tesseract_hsio_service()
        self.design.setup_serdes_res_service()
        
        if self._is_design_save_needed is True:
            dsg_svc = DesignService()
            dsg_svc.save_design(self.design)

        return self.design

    def _get_prog_info(self):
        """
        Save the prog_info from the efxproj setting file
        """
        self.prog_info = None

        try:
            setting = efxproj_set.EfxProjectSetting.build(
                self.design.name, self.design.location,
                [efxproj_set.EfxProjectSetting.LoadFilterType.prog])

            if setting is not None:
                self.prog_info = setting.get_prog_info()

        except app_excp.PTFileException as exc:
            self.logger.warning("File error : {}".format(exc.msg))

    def _build_event_based(self, design_file):

        self.logger.info("Load design from {}".format(design_file))

        with open(design_file, 'r', encoding='UTF-8') as xml_file:

            # get an iterable
            context = et.iterparse(xml_file, events=(
                "start", "end", "start-ns", "end-ns"))

            # turn it into an iterator
            context = iter(context)

            db_tag = self._ns + "design_db"
            device_info_tag = self._ns + "device_info"
            pd_tag = self._ns + "partial_design"

            include_tag = "{http://www.w3.org/2001/XInclude}include"
            incl_list = []

            # Build top level design
            for event, elem in context:
                if event == "start":
                    if elem.tag == db_tag:
                        design_attr = elem.attrib
                        self.design = db.PeriDesign(design_attr.get(
                            "name", None), design_attr.get("device_def", None))
                        self.design.design_file = design_file
                        self.design.location = os.path.abspath(
                            os.path.dirname(design_file))
                        self.design.last_change_date = design_attr.get(
                            "last_change_date", "")
                        self.design.version = design_attr.get("version", "")

                        db_version_name = design_attr.get("db_version", None)
                        if db_version_name is None or db_version_name == "":
                            #: Old design, no db version
                            self.design.set_default_design_version()
                        else:
                            self.design.set_design_version_name(
                                db_version_name)

                    elif elem.tag == device_info_tag:
                        if self.design is not None and self.design.device_setting is None:
                            self.design.device_setting = dev_set.DeviceSetting()

                    elif elem.tag == include_tag:
                        incl_attr = elem.attrib
                        incl_file = incl_attr.get("href", None)
                        if incl_file is not None:
                            incl_list.append(incl_file)

                    elif elem.tag == pd_tag:
                        self.design.add_partial_design(elem.attrib['source'], {})

                    elem.clear()

        if self.design is not None:

            if self.design.name is None or self.design.device_def is None:
                raise db_excp.DesignFileException(
                    "Fail to extract design or device name", app_excp.MsgLevel.error)

            # Build device-plugins map
            if self.design.get_supported_block_count() == 0:
                if self._is_plugin_mockup:
                    plugin_map = plmap.PluginMap.build_mockup()
                    self.design.setup_plugin_map(plugin_map)
                else:
                    self.design.setup_plugin_map()

            self.logger.info("Supported blocks : {}".format(
                self.design.get_supported_block()))

            # Load the prog setting
            self._get_prog_info()

            self.logger.info("Device name : {}".format(self.design.device_def))
            self.logger.info("Include: {}".format(incl_list))

            # Build device
            if self._is_build_device is True:
                self._build_device()

            if self.design.device_db is not None:
                self.old2new_resource_map = self.design.device_db.get_resource_migration_map()

                if self.design.device_setting is not None:
                    # Exist in design file, process it
                    self._build_device_setting(incl_list)
                else:
                    # For legacy file without device setting, create it
                    devset_svc = DesignSettingService()
                    self.design.device_setting = devset_svc.create_setting(
                        self.design)
                    if self.design.device_setting is None:
                        msg = "Fail to build device setting for {}".format(
                            self.design.device_def)
                        raise db_excp.DeviceSettingException(
                            msg, app_excp.MsgLevel.error)
                    else:
                        self._is_design_save_needed = True
            else:
                # This should happen only in unit test
                self.logger.warning(
                    "Device db is None. Skipped building device setting.")

            # Handle legacy design with device setting
            self._handle_legacy_device_settings()

            self._build_block_registry(incl_list)

            # Populate clkmux info
            if self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_clkmux") or \
                    self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx180_clkmux") or\
                    self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx375_clkmux"):
                if self.design.device_setting is not None and self.design.device_setting.clkmux_reg is not None:
                    self.design.device_setting.clkmux_reg.load_device_clkmux_ins(
                        self.design.device_db, self.design)

            self.design.update_chksum()

        return self.design

    def _handle_legacy_device_settings(self):

        # Handle legacy design with device setting but without control
        # block
        if self.design.device_setting is None:
            return

        dev_setting = self.design.device_setting
        devset_svc = DesignSettingService(self.design)

        if self.design.plugin_map.is_plugin_exist(self.design.device_def, "an20_ctrl"):

            if dev_setting.ctrl_reg is None:

                self.logger.debug(
                    "Building control assuming legacy design")
                dev_setting = self.design.device_setting

                # Build control block
                dev_setting.ctrl_reg = devset_svc.build_ctrl(
                    self.design.device_db)
                # If not unit test, save the auto-created control
                # setting
                if not self._is_plugin_mockup:
                    # Make sure it will be save to file
                    self._is_design_save_needed = True

        # Handle legacy design which supports ext flash control but wasn't
        # there yet
        if self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_ctrl") and \
                self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_spi_flash"):

            if dev_setting.ext_flash_reg is None:
                self.logger.debug(
                    "Building ext_flash assuming legacy design")
                dev_setting = self.design.device_setting

                dev_setting.ext_flash_reg = devset_svc.build_ext_flash_controller()

                if not self._is_plugin_mockup:
                    # Make sure it will be save to file
                    self._is_design_save_needed = True


    def _register_patch(self, last_patch_no, patch_no):
        if patch_no is not None and last_patch_no is not None:
            if patch_no > last_patch_no:
                return patch_no
        elif patch_no is not None:
            return patch_no
        else:
            return last_patch_no

    def _build_block_registry(self, incl_list: List[str]):

        last_patch_no = self.device_setting_last_patch_no
        supp_block = self.design.get_supported_block()
        patch_no: Optional[int] = None

        for block_type in supp_block:

            if block_type == db.PeriDesign.BlockType.gpio \
                    or block_type == db.PeriDesign.BlockType.adv_l2_gpio:
                patch_no = self._build_gpio_reg(incl_list)
                last_patch_no = self._register_patch(last_patch_no, patch_no)

            elif block_type == db.PeriDesign.BlockType.comp_gpio:
                self._build_comp_gpio_reg(incl_list)

            elif block_type == db.PeriDesign.BlockType.pll:
                self._build_pll_reg(incl_list)

            elif block_type == db.PeriDesign.BlockType.adv_pll:
                self._build_adv_pll_reg(incl_list)

            elif block_type == db.PeriDesign.BlockType.comp_pll:
                patch_no = self._build_comp_pll_reg(incl_list)
                last_patch_no = self._register_patch(last_patch_no, patch_no)

            elif block_type == db.PeriDesign.BlockType.efx_pll_v3_comp:
                patch_no = self._build_pll_v4_reg(incl_list)
                last_patch_no = self._register_patch(last_patch_no, patch_no)

            elif block_type == db.PeriDesign.BlockType.efx_fpll_v1:
                patch_no = self._build_fpll_v1_reg(incl_list)
                last_patch_no = self._register_patch(last_patch_no, patch_no)

            elif block_type == db.PeriDesign.BlockType.osc:
                self._build_osc_reg(incl_list)

            elif block_type == db.PeriDesign.BlockType.adv_osc:
                self._build_adv_osc_reg(incl_list)

            elif block_type == db.PeriDesign.BlockType.hposc:
                patch_no = self._build_hposc_reg(incl_list)
                last_patch_no = self._register_patch(last_patch_no, patch_no)

            elif block_type == db.PeriDesign.BlockType.lvds:
                patch_no = self._build_lvds_reg(incl_list)
                last_patch_no = self._register_patch(last_patch_no, patch_no)

            elif block_type == db.PeriDesign.BlockType.adv_lvds:
                patch_no = self._build_adv_lvds_reg(incl_list)
                last_patch_no = self._register_patch(last_patch_no, patch_no)

            elif block_type == db.PeriDesign.BlockType.mipi:
                patch_no = self._build_mipi_reg(incl_list)
                last_patch_no = self._register_patch(last_patch_no, patch_no)

            elif block_type == db.PeriDesign.BlockType.mipi_hard_dphy:
                patch_no = self._build_mipi_hard_dphy_reg(incl_list)
                last_patch_no = self._register_patch(last_patch_no, patch_no)

            elif block_type == db.PeriDesign.BlockType.jtag:
                self._build_jtag_reg(incl_list)

            elif block_type == db.PeriDesign.BlockType.h264:
                self._build_h264_reg(incl_list)

            elif block_type == db.PeriDesign.BlockType.ddr:
                patch_no = self._build_ddr_reg(incl_list)
                last_patch_no = self._register_patch(last_patch_no, patch_no)

            elif block_type == db.PeriDesign.BlockType.adv_ddr:
                patch_no = self._build_adv_ddr_reg(incl_list)
                last_patch_no = self._register_patch(last_patch_no, patch_no)

            elif block_type == db.PeriDesign.BlockType.mipi_dphy:
                patch_no = self._build_mipi_dphy_reg(incl_list)
                last_patch_no = self._register_patch(last_patch_no, patch_no)

            elif block_type == db.PeriDesign.BlockType.spi_flash:
                patch_no = self._build_spi_flash_reg(incl_list)
                last_patch_no = self._register_patch(last_patch_no, patch_no)

            elif block_type == db.PeriDesign.BlockType.hyper_ram:
                patch_no = self._build_hyper_ram_reg(incl_list)
                last_patch_no = self._register_patch(last_patch_no, patch_no)

            elif block_type == db.PeriDesign.BlockType.pll_ssc:
                self._build_pll_ssc_reg(incl_list)

            elif block_type == db.PeriDesign.BlockType.soc:
                self._build_soc_reg(incl_list)

            elif block_type == db.PeriDesign.BlockType.quad:
                patch_no = self._build_quad_reg(incl_list)
                last_patch_no = self._register_patch(last_patch_no, patch_no)

            elif block_type == db.PeriDesign.BlockType.quad_pcie:
                patch_no = self._build_quad_pcie_reg(incl_list)
                last_patch_no = self._register_patch(last_patch_no, patch_no)

            elif block_type == db.PeriDesign.BlockType.lane_10g:
                patch_no = self._build_lane_10g_reg(incl_list)
                last_patch_no = self._register_patch(last_patch_no, patch_no)

            elif block_type == db.PeriDesign.BlockType.raw_serdes:
                patch_no = self._build_raw_serdes_reg(incl_list)
                last_patch_no = self._register_patch(last_patch_no, patch_no)

            elif block_type == db.PeriDesign.BlockType.lane_1g:
                patch_no = self._build_lane_1g_reg(incl_list)
                last_patch_no = self._register_patch(last_patch_no, patch_no)

            if last_patch_no is not None:
                db_version = self.design.db_version
                db_version.set_applied_patch_no(last_patch_no)

    def build_from_xmltree(self, xml_tree):
        # Not supported
        pass

    def check(self, design):
        gen_util.mark_unused(design)

    def validate(self, design_file):

        is_pass = True

        # Validate design top level
        if DesignBuilderXml._schema_file == "":
            app_setting = aps.AppSetting()
            app_path = app_setting.app_path
            DesignBuilderXml._schema_file = app_path[app_setting.PathType.schema] + \
                "/peri_design_db.xsd"

        schema = xmlschema.XMLSchema(DesignBuilderXml._schema_file)

        # Schema validation will expand top level schema to include children schema.
        # So it will validate against all.
        # However it does not process include xml files. Need to expand
        # manually.
        try:

            include_tag = "{http://www.w3.org/2001/XInclude}include"
            xml_doc = et.ElementTree(file=design_file)

            is_hier = False
            for elem in xml_doc.iter(tag=include_tag):
                gen_util.mark_unused(elem)

                is_hier = True
                break

            if not is_hier:
                schema.validate(design_file)
            else:
                is_pass = self._validate_hier_xml(design_file, schema)

        except et.ParseError as excp:
            self.logger.error("Design parsing error : {}".format(excp))
            is_pass = False

        except xmlschema.XMLSchemaValidationError as excp:
            self.logger.error("Design validation error, {}".format(excp))
            self.logger.error("XML Element : {}".format(excp.elem))
            self.logger.error("Reason : {}".format(excp.reason))
            is_pass = False

        return is_pass

    def _validate_hier_xml(self, design_file, schema):
        import xml.etree.ElementInclude as ei
        import xml.etree.ElementTree as etree

        is_pass = True

        original_dir = os.getcwd()
        file_dir = os.path.dirname(design_file)

        # to make relative include work
        os.chdir(file_dir)

        tree = etree.ElementTree(file=design_file)

        # Expand all includes until the leaf, not just at the top
        root = tree.getroot()
        try:
            ei.include(root)
            schema.validate(tree)

        except ei.FatalIncludeError as excp:
            self.logger.error("Include error: {}".format(excp))
            is_pass = False
            os.chdir(original_dir)

        except OSError as excp:
            self.logger.error("File not found error: {}".format(excp))
            is_pass = False
            os.chdir(original_dir)

        except Exception as excp:
            self.logger.error("Exception: {}".format(excp))
            is_pass = False
            os.chdir(original_dir)

        os.chdir(original_dir)
        return is_pass

    def _build_gpio_reg(self, incl_list: List[str]):
        self.logger.info("Load gpio")

        gpio_service = gds.GPIODesignService()

        # This is for the case where the design file does not have include
        count = len(incl_list)
        if count == 0:
            gpio_reg = gpio_service.load_design(self.design.design_file)
            self.design.gpio_reg = gpio_reg
        else:
            for incl_file in incl_list:
                if 'gpio' in incl_file:
                    gpio_reg = gpio_service.load_design(incl_file)
                    self.design.gpio_reg = gpio_reg

        if self.design.gpio_reg is not None:
            # Note : The device def in schema is optional for backward
            # compatibility. Not parse.
            self.design.gpio_reg.device_name = self.design.device_def
            self.design.gpio_reg.device_db = self.design.device_db

            if self.design.is_support_ddio():
                for gpio in self.design.gpio_reg.get_all_gpio():
                    gpio.enable_ddio_support()

            self.design.gpio_reg.create_gpio_to_bank_map(self.design.device_db)

            # update any of the lvds GPIO device instance name
            count = self.design.gpio_reg.update_dev_def_migration(
                self.old2new_resource_map)
            if count > 0:
                self._resource_name_changed = True

            self._is_bus_member_pin_name_regen = gpio_service.is_bus_member_pin_name_regen()

            # If not unit test, don't run patch as it will create new design file
            # if it uses alternate conn_type
            if not self._is_plugin_mockup:

                # Get the iobank reg
                iobank_reg = None
                if self.design.device_setting is not None:
                    iobank_reg = self.design.device_setting.iobank_reg

                patcher = GPIOPatch(self.design.gpio_reg,
                                    iobank_reg, self.design.device_db)
                db_version = self.design.db_version
                patch_no = patcher.patch(db_version.get_design_version())
                if patcher.is_patch():
                    self.change_msg.extend(patcher.get_patch_message())

                return patch_no

        return None

    def _build_comp_gpio_reg(self, incl_list: List[str]):
        assert self.design is not None
        self.logger.info("Load gpio")

        gpio_service = comp_gds.GPIODesignServiceComplex()
        gpio_reg = GPIORegistryComplex("")
        gpio_reg.apply_device_db(self.design.device_db)

        # This is for the case where the design file does not have include
        count = len(incl_list)
        if count == 0:
            gpio_reg = gpio_service.load_design(self.design.design_file)
            self.design.gpio_reg = gpio_reg
        else:
            for incl_file in incl_list:
                if 'gpio' in incl_file:
                    gpio_reg = gpio_service.load_design(incl_file)
                    self.design.gpio_reg = gpio_reg

        if self.design.gpio_reg is not None:
            self.design.gpio_reg.device_db = self.design.device_db

            if self.design.is_support_ddio():
                for gpio in self.design.gpio_reg.get_all_gpio():
                    gpio.enable_ddio_support()

            self.design.gpio_reg.create_gpio_to_bank_map(self.design.device_db)

            # update any of the lvds GPIO device instance name
            count = self.design.gpio_reg.update_dev_def_migration(
                self.old2new_resource_map)
            if count > 0:
                self._resource_name_changed = True

            self._is_bus_member_pin_name_regen = gpio_service.is_bus_member_pin_name_regen()

            # If not unit test, don't run patch as it will create new design file
            # if it uses alternate conn_type
            if not self._is_plugin_mockup:

                # Get the iobank reg
                iobank_reg = None
                if self.design.device_setting is not None:
                    iobank_reg = self.design.device_setting.iobank_reg

                patcher = GPIOPatch(self.design.gpio_reg,
                                    iobank_reg, self.design.device_db)
                db_version = self.design.db_version
                patch_no = patcher.patch(db_version.get_design_version())
                if patcher.is_patch():
                    self.change_msg.extend(patcher.get_patch_message())

                return patch_no

        return None

    def _build_pll_reg(self, incl_list: List[str]):
        self.logger.info("Load pll")

        pll_service = pllds.PLLDesignService()

        # This is for the case where the design file does not have include
        count = len(incl_list)
        if count == 0:
            pll_reg = pll_service.load_design(self.design.design_file)
            self.design.pll_reg = pll_reg
        else:
            for incl_file in incl_list:
                if 'pll' in incl_file:
                    pll_reg = pll_service.load_design(incl_file)
                    self.design.pll_reg = pll_reg

        if self.design.pll_reg is not None:
            self.design.pll_reg.device_db = self.design.device_db

    def _build_adv_pll_reg(self, incl_list: List[str]):
        self.logger.info("Load advance pll")
        pll_service = adv_pllds.PLLDesignServiceAdvance()
        self.design.pll_reg = self._build_block_reg(
            incl_list, "pll", pll_service)
        if self.design.pll_reg is not None:
            self.design.pll_reg.device_db = self.design.device_db
            # update any of the PLL instance name
            count = self.design.pll_reg.update_dev_def_migration(
                self.old2new_resource_map)
            if count > 0:
                self._resource_name_changed = True

    def _build_comp_pll_reg(self, incl_list: List[str]):
        self.logger.info("Load complex pll")
        pll_service = comp_pllds.PLLDesignServiceComplex()
        self.design.pll_reg = self._build_block_reg(
            incl_list, "pll", pll_service)
        if self.design.pll_reg is not None:
            # If not design level unit test, apply device to set it and
            # generate pins
            if not self._is_plugin_mockup:
                self.design.pll_reg.apply_device_db(self.design.device_db)

            # TODO : MIgration
            # # update any of the PLL instance name
            # count = self.design.pll_reg.update_dev_def_migration(self.old2new_resource_map)
            # if count > 0:
            #     self._resource_name_changed = True

            # If not unit test, don't run patch as it will create new design
            # file
            if not self._is_plugin_mockup:
                patcher = PLLPatch(self.design.pll_reg, self.design.device_db)
                db_version = self.design.db_version
                patch_no = patcher.patch(db_version.get_design_version())
                if patcher.is_patch():
                    self.change_msg.extend(patcher.get_patch_message())

                return patch_no

        return None

    def _build_pll_v4_reg(self, incl_list: List[str]):
        pll_service = PLLDesignServiceV3Complex()
        self.design.pll_reg = self._build_block_reg(
            incl_list, "pll", pll_service)
        if self.design.pll_reg is not None:
            # If not design level unit test, apply device to set it and
            # generate pins
            if not self._is_plugin_mockup:
                self.design.pll_reg.apply_device_db(self.design.device_db)

            # TODO : MIgration
            # # update any of the PLL instance name
            # count = self.design.pll_reg.update_dev_def_migration(self.old2new_resource_map)
            # if count > 0:
            #     self._resource_name_changed = True

            # If not unit test, don't run patch as it will create new design
            # file
            if not self._is_plugin_mockup:
                patcher = PLLPatch(self.design.pll_reg, self.design.device_db)
                db_version = self.design.db_version
                patch_no = patcher.patch(db_version.get_design_version())
                if patcher.is_patch():
                    self.change_msg.extend(patcher.get_patch_message())

                return patch_no

        return None

    def _build_fpll_v1_reg(self, incl_list: List[str]):
        pll_service = EfxFpllV1DesignService()
        self.design.pll_reg = self._build_block_reg(
            incl_list, "pll", pll_service)
        if self.design.pll_reg is not None:
            # If not design level unit test, apply device to set it and
            # generate pins
            if not self._is_plugin_mockup:
                self.design.pll_reg.apply_device_db(self.design.device_db)

            # TODO : MIgration
            # # update any of the PLL instance name
            # count = self.design.pll_reg.update_dev_def_migration(self.old2new_resource_map)
            # if count > 0:
            #     self._resource_name_changed = True

            # If not unit test, don't run patch as it will create new design
            # file
            if not self._is_plugin_mockup:
                patcher = PLLPatch(self.design.pll_reg, self.design.device_db)
                db_version = self.design.db_version
                patch_no = patcher.patch(db_version.get_design_version())
                if patcher.is_patch():
                    self.change_msg.extend(patcher.get_patch_message())

                return patch_no

        return None

    def _build_osc_reg(self, incl_list: List[str]):
        self.logger.info("Load oscillator")
        osc_service = oscds.OSCDesignService()
        # This is for the case where the design file does not have include
        count = len(incl_list)
        if count == 0:
            osc_reg = osc_service.load_design(self.design.design_file)
            self.design.osc_reg = osc_reg
        else:
            for incl_file in incl_list:
                if 'osc' in incl_file:
                    osc_reg = osc_service.load_design(incl_file)
                    self.design.osc_reg = osc_reg

        if self.design.osc_reg is not None:
            self.design.osc_reg.device_db = self.design.device_db

    def _build_adv_osc_reg(self, incl_list: List[str]):
        self.logger.info("Load oscillator")
        osc_service = adv_oscds.OSCDesignServiceAdvance()
        # This is for the case where the design file does not have include
        count = len(incl_list)
        if count == 0:
            osc_reg = osc_service.load_design(self.design.design_file)
            self.design.osc_reg = osc_reg
        else:
            for incl_file in incl_list:
                if 'osc' in incl_file:
                    osc_reg = osc_service.load_design(incl_file)
                    self.design.osc_reg = osc_reg

        if self.design.osc_reg is not None:
            self.design.osc_reg.device_db = self.design.device_db

            if not self._is_plugin_mockup:
                self.design.osc_reg.apply_device_db(self.design.device_db)

    def _build_hposc_reg(self, incl_list: List[str]):
        self.logger.info("Load high precision oscillator")
        osc_service = hposcds.HPOSCDesignService()
        # This is for the case where the design file does not have include
        count = len(incl_list)
        if count == 0:
            osc_reg = osc_service.load_design(self.design.design_file)
            self.design.osc_reg = osc_reg
        else:
            for incl_file in incl_list:
                if 'osc' in incl_file:
                    osc_reg = osc_service.load_design(incl_file)
                    self.design.osc_reg = osc_reg

        if self.design.osc_reg is not None:

            if not self._is_plugin_mockup:
                self.design.osc_reg.apply_device_db(self.design.device_db)

                # If not unit test, don't run patch, may break existing test
                patcher = OSCPatch(self.design.osc_reg, self.design.device_db)
                db_version = self.design.db_version
                patch_no = patcher.patch(db_version.get_design_version())
                if patcher.is_patch():
                    self.change_msg.extend(patcher.get_patch_message())

                return patch_no

        return None

    def _build_lvds_reg(self, incl_list: List[str]) -> Optional[int]:
        self.logger.info("Load lvds")
        lvds_reg = LVDSRegistry()
        lvds_reg.apply_device_db(device_db=self.design.device_db)

        lvds_service = lvdsds.LVDSDesignService()
        lvds_reg = self._build_block_reg(incl_list, "lvds", lvds_service)

        if lvds_reg is None:
            lvds_reg = lvds_service.create_registry()
            # If not unit test, save the auto-created registry
            if not self._is_plugin_mockup:
                # Make sure it will be save to file
                self._is_design_save_needed = True

        self.design.lvds_reg = lvds_reg

        if not self._is_plugin_mockup:
            # If not design level unit test, apply device to set it and
            # generate pins
            self.design.lvds_reg.apply_device_db(device_db=self.design.device_db)

        # update any of the lvds GPIO device instance name
        count = self.design.lvds_reg.update_dev_def_migration(
            self.old2new_resource_map)
        if count > 0:
            self._resource_name_changed = True

        if self.design.gpio_reg is not None:
            # At this point, gpio registry has been build and if there is
            # any lvds gpio, it has been parsed. Ready to track lvds gpio
            # device usage.
            lvds_gpio_list = self.design.gpio_reg.get_all_lvds_gpio()
            count = len(lvds_gpio_list)
            if count > 0:
                for gpio_inst in lvds_gpio_list:
                    self.design.lvds_reg.register_lvds_gpio(gpio_inst)

        # If not unit test, don't run patch as it will create new design file
        # if it uses alternate conn_type
        if not self._is_plugin_mockup:
            patcher = LVDSPatch(self.design.lvds_reg,
                                self.design.device_db)
            db_version = self.design.db_version
            patch_no: int = patcher.patch(db_version.get_design_version())
            if patcher.is_patch():
                self.change_msg.extend(patcher.get_patch_message())

            return patch_no

        return None

    def _build_adv_lvds_reg(self, incl_list: List[str]) -> Optional[int]:

        self.logger.info("Load lvds")
        lvds_reg = LVDSRegistryAdvance()
        lvds_reg.apply_device_db(device_db=self.design.device_db)

        lvds_service = adv_lvdsds.LVDSDesignServiceAdvance()
        lvds_reg = self._build_block_reg(
            incl_list, "lvds", lvds_service)

        if lvds_reg is None:
            lvds_reg = lvds_service.create_registry()
            # If not unit test, save the auto-created registry
            if not self._is_plugin_mockup:
                # Make sure it will be save to file
                self._is_design_save_needed = True

        self.design.lvds_reg = lvds_reg

        # If not design level unit test, apply device to set it and
        # generate pins
        if not self._is_plugin_mockup:
            self.design.lvds_reg.apply_device_db(self.design.device_db)

        # If not unit test, don't run patch, may break existing test
        if not self._is_plugin_mockup:
            patcher = LVDSPatch(self.design.lvds_reg,
                                self.design.device_db)
            db_version = self.design.db_version
            patch_no: int = patcher.patch(db_version.get_design_version())
            if patcher.is_patch():
                self.change_msg.extend(patcher.get_patch_message())

            return patch_no

        return None

    def _build_mipi_reg(self, incl_list: List[str]):
        self.logger.info("Load mipi")
        mipi_service = mipids.MIPIDesignService()
        mipi_reg = self._build_block_reg(incl_list, "mipi", mipi_service)

        # MIPI is introduced after an20 is released but it is available for an20 onwards.
        # Upgrade legacy design if mipi info does not exist in design file.
        # We do legacy design update here because this function will only be called
        # if the device supports mipi.
        if mipi_reg is None:
            mipi_reg = mipi_service.create_registry()
            # If not unit test, save the auto-created registry
            if not self._is_plugin_mockup:
                # Make sure it will be save to file
                self._is_design_save_needed = True

        self.design.mipi_reg = mipi_reg

        # If not design level unit test, apply device to set it and generate
        # pins
        if not self._is_plugin_mockup:
            self.design.mipi_reg.apply_device_db(self.design.device_db)
            patcher = MIPIPatch(mipi_reg, self.design.device_db)
            db_version = self.design.db_version
            patch_no = patcher.patch(db_version.get_design_version())
            if patcher.is_patch():
                self.change_msg.extend(patcher.get_patch_message())

            return patch_no

    def _build_mipi_hard_dphy_reg(self, incl_list: List[str]):
        # Ugly code to initialize the port information
        mipi_reg = MIPIHardDPHYRegistry()
        mipi_reg.apply_device_db(device_db=self.design.device_db)

        mipi_service = MIPIHardDPHYDesignService()
        mipi_reg = self._build_block_reg(incl_list, "mipi", mipi_service)

        # MIPI is introduced after an20 is released but it is available for an20 onwards.
        # Upgrade legacy design if mipi info does not exist in design file.
        # We do legacy design update here because this function will only be called
        # if the device supports mipi.
        if mipi_reg is None:
            mipi_reg = mipi_service.create_registry()
            # If not unit test, save the auto-created registry
            if not self._is_plugin_mockup:
                # Make sure it will be save to file
                self._is_design_save_needed = True

        self.design.mipi_hard_dphy_reg = mipi_reg

        # If not design level unit test, apply device to set it and generate
        # pins
        if not self._is_plugin_mockup:
            self.design.mipi_hard_dphy_reg.apply_device_db(self.design.device_db)
            patcher = MIPIHardDPHYPatch(mipi_reg, self.design.device_db)
            db_version = self.design.db_version
            patch_no = patcher.patch(db_version.get_design_version())
            if patcher.is_patch():
                self.change_msg.extend(patcher.get_patch_message())

            return patch_no

        return None

    def _build_mipi_dphy_reg(self, incl_list: List[str]):
        self.logger.info("Load mipi dphy")
        mdphy_service = mdphyds.MIPIDPhyDesignService()
        mdphy_reg = self._build_block_reg(
            incl_list, "mipi_dphy", mdphy_service)
        self.design.mipi_dphy_reg = mdphy_reg

        # If not design level unit test, apply device to set it and generate
        # pins
        if not self._is_plugin_mockup:
            self.design.mipi_dphy_reg.apply_device_db(self.design.device_db)

        # If not unit test, don't run patch as it will create new design file
        if not self._is_plugin_mockup:
            patcher = MIPIDPhyPatch(mdphy_reg, self.design.device_db)
            db_version = self.design.db_version
            patch_no = patcher.patch(db_version.get_design_version())
            if patcher.is_patch():
                self.change_msg.extend(patcher.get_patch_message())

            return patch_no

        return None

    def _build_spi_flash_reg(self, incl_list: List[str]):
        self.logger.info("Load spi flash")
        spf_reg = SPIFlashRegistry()
        spf_reg.apply_device_db(device_db=self.design.device_db)

        spf_service = spfds.SPIFlashDesignService()
        spf_reg = self._build_block_reg(incl_list, "spi_flash", spf_service)

        if spf_reg is None:
            spf_reg = spf_service.create_registry()
            # If not unit test, save the auto-created registry
            if not self._is_plugin_mockup:
                # Make sure it will be save to file
                self._is_design_save_needed = True

        self.design.spi_flash_reg = spf_reg

        # SPI Flash block for T20/T13 was introducted after T20 released
        # Old T20 design file may not have spi_flash_info hence no registry loaded
        if self.design.spi_flash_reg is None:
            self.design.spi_flash_reg = spf_service.create_registry()
            self._is_design_save_needed = True

        # If not design level unit test, apply device to set it and generate
        # pins
        if not self._is_plugin_mockup:
            self.design.spi_flash_reg.apply_device_db(self.design.device_db)

            patcher = SPIFlashPatch(self.design.spi_flash_reg, self.design.device_db)
            db_version = self.design.db_version
            patch_no = patcher.patch(db_version.get_design_version())
            if patcher.is_patch():
                self.change_msg.extend(patcher.get_patch_message())

            return patch_no

        return None

    def _build_hyper_ram_reg(self, incl_list: List[str]):
        self.logger.info("Load hyper ram")
        hpram_service = hpramds.HyperRAMDesignService()
        hpram_reg = self._build_block_reg(
            incl_list, "hyper_ram", hpram_service)
        self.design.hyper_ram_reg = hpram_reg

        # If not design level unit test, apply device to set it and generate
        # pins
        if not self._is_plugin_mockup:
            self.design.hyper_ram_reg.apply_device_db(self.design.device_db)

            patcher = HyperRAMPatch(self.design.hyper_ram_reg, self.design.pll_reg, self.design.device_db)
            db_version = self.design.db_version
            patch_no = patcher.patch(db_version.get_design_version())
            if patcher.is_patch():
                self.change_msg.extend(patcher.get_patch_message())

            return patch_no

        return None

    def _build_h264_reg(self, incl_list: List[str]):
        self.logger.info("Load h264")
        h264_service = h264ds.H264DesignService()
        h264_reg = self._build_block_reg(incl_list, "h264", h264_service)
        self.design.h264_reg = h264_reg

        # If not design level unit test, apply device to set it and generate
        # pins
        if not self._is_plugin_mockup:
            self.design.h264_reg.apply_device_db(self.design.device_db)

    def _build_ddr_reg(self, incl_list: List[str]):
        self.logger.info("Load ddr")
        ddr_service = ddrds.DDRDesignService()
        ddr_reg = self._build_block_reg(incl_list, "ddr", ddr_service)
        self.design.ddr_reg = ddr_reg

        # If not design level unit test, apply device to set it and generate
        # pins
        if not self._is_plugin_mockup:
            self.design.ddr_reg.apply_device_db(self.design.device_db)

        # For backward compatibility
        # If the device has restrictions, we need to take it into
        # account and reset to default
        self.design.ddr_reg.resolve_backward_compatibility_config()

        # If not unit test, don't run patch as it will create new design file
        # if it uses alternate conn_type
        if not self._is_plugin_mockup:
            patcher = DDRPatch(ddr_reg, self.design.device_db)
            db_version = self.design.db_version
            patch_no = patcher.patch(db_version.get_design_version())
            if patcher.is_patch():
                self.change_msg.extend(patcher.get_patch_message())

            return patch_no

        return None

    def _build_adv_ddr_reg(self, incl_list: List[str]):
        self.logger.info("Load ddr")
        # Needed to initialize the param information
        ddr_reg = DDRRegistryAdvance()
        ddr_reg.apply_device_db(device_db=self.design.device_db)

        ddr_service = adv_ddrds.DDRDesignServiceAdvance()
        ddr_reg = self._build_block_reg(incl_list, "ddr", ddr_service)

        # DDR plugin was defined before we really have the service
        if ddr_reg is None:
            ddr_reg = ddr_service.create_registry()
            # If not unit test, save the auto-created registry
            if not self._is_plugin_mockup:
                # Make sure it will be save to file
                self._is_design_save_needed = True

        self.design.ddr_reg = ddr_reg

        # If not design level unit test, apply device to set it and generate
        # pins
        if not self._is_plugin_mockup:
            self.design.ddr_reg.apply_device_db(self.design.device_db)

        # If not unit test, don't run patch as it will create new design file
        # if it uses alternate conn_type
        if not self._is_plugin_mockup:
            patcher = DDRPatch(ddr_reg, self.design.device_db)
            db_version = self.design.db_version
            patch_no = patcher.patch(db_version.get_design_version())
            if patcher.is_patch():
                self.change_msg.extend(patcher.get_patch_message())

            return patch_no

        return None

    def _build_jtag_reg(self, incl_list: List[str]):
        self.logger.info("Load jtag")
        jtag_service = jtagds.JTAGDesignService()
        jtag_reg = self._build_block_reg(incl_list, "jtag", jtag_service)

        # JTAG is introduced after an20 is released but it is available for all family
        # starting from an08. Upgrade legacy design if jtag info does not exist in design file.
        # We do legacy design update here because this function will only be called
        # if the device supports jtag.
        if jtag_reg is None:
            jtag_reg = jtag_service.create_registry()
            # If not unit test, save the auto-created registry
            if not self._is_plugin_mockup:
                # Make sure it will be save to file
                self._is_design_save_needed = True

        self.design.jtag_reg = jtag_reg

        # If not design level unit test, apply device to set it and generate
        # pins
        if not self._is_plugin_mockup:
            self.design.jtag_reg.apply_device_db(self.design.device_db)

    def _build_pll_ssc_reg(self, incl_list: List[str]):
        assert self.design is not None and self.design.device_db is not None
        # Ugly code to initialize the port information
        pll_ssc_reg = PLLSSCRegistry()
        pll_ssc_reg.apply_device_db(device_db=self.design.device_db)

        pll_ssc_service = PLLSSCDesignService()
        pll_ssc_reg = self._build_block_reg(incl_list, "pll_ssc", pll_ssc_service)

        # PLL SSC is introduced after tx180 is released but it is available for tx180 onwards.
        # Upgrade legacy design if PLL SSC info does not exist in design file.
        # We do legacy design update here because this function will only be called
        # if the device supports PLL SSC.
        if pll_ssc_reg is None:
            pll_ssc_reg = pll_ssc_service.create_registry()
            # If not unit test, save the auto-created registry
            if not self._is_plugin_mockup:
                # Make sure it will be save to file
                self._is_design_save_needed = True

        pll_ssc_reg.device_db = self.design.device_db
        self.design.pll_ssc_reg = pll_ssc_reg

        return None

    def _build_soc_reg(self, incl_list: List[str]):
        assert self.design is not None and self.design.device_db is not None
        # Ugly code to initialize the port information
        soc_reg = SOCRegistry()
        soc_reg.apply_device_db(device_db=self.design.device_db)

        soc_service = SOCDesignService()
        soc_reg = self._build_block_reg(incl_list, "soc", soc_service)

        # Upgrade legacy design if SOC info does not exist in design file.
        # We do legacy design update here because this function will only be called
        # if the device supports SOC.
        if soc_reg is None:
            soc_reg = soc_service.create_registry()
            # If not unit test, save the auto-created registry
            if not self._is_plugin_mockup:
                # Make sure it will be save to file
                self._is_design_save_needed = True

        soc_reg.device_db = self.design.device_db
        self.design.soc_reg = soc_reg

    def _build_quad_reg(self, incl_list: List[str]):
        assert self.design is not None and self.design.device_db is not None

        # TODO: DesignService and load reg from xml
        self.design.quad_reg = QuadRegistry()
        self.design.quad_reg.apply_device_db(self.design.device_db)

        # Common quad reg
        service = QuadLaneCommonDesignService(self.design.device_db)
        reg = self._build_block_reg(incl_list, "quad_lane", service)
        if reg is None:
            reg = service.create_registry()
            # If not unit test, save the auto-created registry
            if not self._is_plugin_mockup:
                # Make sure it will be save to file
                self._is_design_save_needed = True

        self.design.common_quad_lane_reg = reg

        if not self._is_plugin_mockup:
            patcher = QuadLaneCommonPatch(self.design.common_quad_lane_reg, self.design.device_db)
            db_version = self.design.db_version
            patch_no = patcher.patch(db_version.get_design_version())
            if patcher.is_patch():
                self.change_msg.extend(patcher.get_patch_message())

            return patch_no

        return None

    def _build_quad_pcie_reg(self, incl_list: List[str]):
        assert self.design is not None and self.design.device_db is not None
        # Ugly code to initialize the port information
        quad_pcie_reg = QuadPCIERegistry()
        quad_pcie_reg.apply_device_db(device_db=self.design.device_db)

        quad_pcie_service = QuadPCIEDesignService()
        quad_pcie_reg = self._build_block_reg(incl_list, "quad_pcie", quad_pcie_service)

        if quad_pcie_reg is None:
            quad_pcie_reg = quad_pcie_service.create_registry()
            # If not unit test, save the auto-created registry
            if not self._is_plugin_mockup:
                # Make sure it will be save to file
                self._is_design_save_needed = True

        assert isinstance(quad_pcie_reg, QuadPCIERegistry)
        quad_pcie_reg.apply_device_db(self.design.device_db)
        self.design.quad_pcie_reg = quad_pcie_reg
        assert self.design is not None and self.design.device_db is not None

        # If not design level unit test, apply device to set it and generate
        # pins
        if not self._is_plugin_mockup:
            patcher = QuadPCIEPatch(quad_pcie_reg, self.design.device_db)
            db_version = self.design.db_version
            patch_no = patcher.patch(db_version.get_design_version())
            if patcher.is_patch():
                self.change_msg.extend(patcher.get_patch_message())

            return patch_no

    def _build_lane_10g_reg(self, incl_list: List[str]):
        assert self.design is not None and self.design.device_db is not None
        # Ugly code to initialize the port information
        reg = Lane10GRegistry()
        reg.apply_device_db(device_db=self.design.device_db)

        design_service = Lane10GDesignService()
        reg = self._build_block_reg(incl_list, "lane_10g", design_service)

        if reg is None:
            reg = design_service.create_registry()
            # If not unit test, save the auto-created registry
            if not self._is_plugin_mockup:
                # Make sure it will be save to file
                self._is_design_save_needed = True

        assert isinstance(reg, Lane10GRegistry)
        reg.apply_device_db(self.design.device_db)
        # Make sure common quad lane reg is not None since it will be used later for
        # lane based registry
        reg.common_quad_reg = self.design.common_quad_lane_reg
        assert reg.common_quad_reg is not None
        reg.load_settings_file(self.design.location)

        is_valid, unlinked_none_cmn_inst = design_service.check_common_quad_lane_reg(reg)
        assert is_valid is True, f"Fail to check common quad lane reg"
        self.design.lane_10g_reg = reg        

        # If not design level unit test, apply device to set it and generate
        # pins
        if not self._is_plugin_mockup:
            patcher = Lane10GPatch(self.design.lane_10g_reg, self.design.device_db, unlinked_none_cmn_inst)
            db_version = self.design.db_version
            patch_no = patcher.patch(db_version.get_design_version())
            if patcher.is_patch():
                self.change_msg.extend(patcher.get_patch_message())

            return patch_no

        return None

    def _build_lane_1g_reg(self, incl_list: List[str]):
        assert self.design is not None and self.design.device_db is not None
        # Ugly code to initialize the port information
        reg = Lane1GRegistry()
        reg.apply_device_db(device_db=self.design.device_db)

        design_service = Lane1GDesignService()
        reg = self._build_block_reg(incl_list, "lane_1g", design_service)

        if reg is None:
            reg = design_service.create_registry()
            # If not unit test, save the auto-created registry
            if not self._is_plugin_mockup:
                # Make sure it will be save to file
                self._is_design_save_needed = True

        assert isinstance(reg, Lane1GRegistry)
        reg.apply_device_db(self.design.device_db)
        # Make sure common quad lane reg is not None since it will be used later for
        # lane based registry
        reg.common_quad_reg = self.design.common_quad_lane_reg
        assert reg.common_quad_reg is not None

        is_valid, _ = design_service.check_common_quad_lane_reg(reg)
        assert is_valid is True, f"Fail to check common quad lane reg"
        self.design.lane_1g_reg = reg

        if not self._is_plugin_mockup:
            patcher = Lane1GPatch(self.design.lane_1g_reg, self.design.device_db)
            db_version = self.design.db_version
            patch_no = patcher.patch(db_version.get_design_version())
            if patcher.is_patch():
                self.change_msg.extend(patcher.get_patch_message())

            return patch_no

        return None

    def _build_raw_serdes_reg(self, incl_list: List[str]):
        assert self.design is not None and self.design.device_db is not None
        # Ugly code to initialize the port information
        reg = RawSerdesRegistry()
        reg.apply_device_db(device_db=self.design.device_db)

        design_service = RawSerdesDesignService()
        reg = self._build_block_reg(incl_list, "raw_serdes", design_service)

        if reg is None:
            reg = design_service.create_registry()
            # If not unit test, save the auto-created registry
            if not self._is_plugin_mockup:
                # Make sure it will be save to file
                self._is_design_save_needed = True

        assert isinstance(reg, RawSerdesRegistry)
        reg.apply_device_db(self.design.device_db)
        # Make sure common quad lane reg is not None since it will be used later for
        # lane based registry
        reg.common_quad_reg = self.design.common_quad_lane_reg
        assert reg.common_quad_reg is not None
        #reg.load_settings_file(self.design.location)

        is_valid, _ = design_service.check_common_quad_lane_reg(reg)
        assert is_valid is True, f"Fail to check common quad lane reg"
        self.design.raw_serdes_reg = reg

        if not self._is_plugin_mockup:
            patcher = RawSerdesPatch(self.design.raw_serdes_reg, self.design.device_db)
            db_version = self.design.db_version
            patch_no = patcher.patch(db_version.get_design_version())
            if patcher.is_patch():
                self.change_msg.extend(patcher.get_patch_message())

            return patch_no

        return None

    def _build_block_reg(self, incl_list: List[str], incl_text: str,
                         design_service: dbi.DesignBlockService) -> Optional[PeriDesignRegistry]:
        # This is for the case where the design file does not have include
        assert self.design is not None
        count = len(incl_list)
        block_reg = None
        if count == 0:
            block_reg = design_service.load_design(self.design.design_file)
        else:
            for incl_file in incl_list:
                if incl_text in incl_file:
                    block_reg = design_service.load_design(incl_file)

        return block_reg

    def _get_timing_model_name(self):
        """
        Return timing model info retrieved from project file

        :return timing model name
        """
        timing_model_name = ""

        try:
            setting = efxproj_set.EfxProjectSetting.build(
                self.design.name, self.design.location,
                [efxproj_set.EfxProjectSetting.LoadFilterType.device])

            if setting is not None:
                device_info = setting.get_device_info()
                if device_info is not None:
                    timing_model_name = device_info.get("timing_model", "")
                else:
                    self.logger.warning(
                        'Unable to retrieve timing_model name from project file')

        except app_excp.PTFileException as exc:
            self.logger.warning("File error : {}".format(exc.msg))

        return timing_model_name

    def _build_device(self):

        # We are allowing the device to be built without
        # looking at the timing model and programming mode
        # because it could be a migration where the information
        # is for the new device, while in this case we are loading
        # design that points to older device that does not have
        # the same options as the new device setting.
        if not self._ignore_device_proj_setting:
            timing_model_name = self._get_timing_model_name()

            dev_service = dev_srv.DeviceService()
            dev_service.is_check = self.is_check_device
            self.design.device_db = dev_service.load_device(
                self.design.device_def, self.prog_info,
                timing_model_name)

        else:
            dev_service = dev_srv.DeviceService()
            dev_service.is_check = self.is_check_device
            self.design.device_db = dev_service.load_device(
                self.design.device_def)

        if self.design.device_db is None:
            self.logger.error(
                "Fail to load device database for {}".format(self.design.device_def))
        else:
            self.logger.debug(
                "Load device database for {}".format(self.design.device_def))

    def _build_device_setting(self, incl_list: List[str]):
        """
        Build device setting for hierarchical design file

        :param incl_list: Design file includes
        """
        assert self.design is not None

        device_setting = self.design.device_setting

        if device_setting is not None:
            device_setting.device_db = self.design.device_db

            if self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_iob"):
                iobank_service = IOBankDesignServiceAdvance()
            else:
                iobank_service = iobds.IOBankDesignService()

            # Control block is not part of instantiable design block.
            # Need to check manually since is_block_supported() cannot be used.
            is_support_ctrl = False
            is_support_seu = False
            if self.design.plugin_map.is_plugin_exist(self.design.device_def, "an20_ctrl"):
                is_support_ctrl = True
                ctrl_service = ctrlds.ControlDesignService()
            elif self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_ctrl"):
                is_support_ctrl = True
                ctrl_service = ctrlds.ControlDesignService()
                is_support_seu = True
                seu_service = seuds.SEUDesignService()

            # Indicate if it has a clkmux plugin
            is_support_clkmux_plugin = False
            clkmux_service = None
            if self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_clkmux"):
                is_support_clkmux_plugin = True
                clkmux_service = ClockMuxDesignServiceAdvance()
            elif self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx180_clkmux"):
                is_support_clkmux_plugin = True
                clkmux_service = ClockMuxDesignServiceComplex()
            elif self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx375_clkmux"):
                is_support_clkmux_plugin = True
                clkmux_service = ClockMuxDesignServiceV4()

            # External Flash controller is only applicable when the
            # device supports spi_flash and tx60_ctrl (use seu flag)
            is_support_ext_flash_ctrl = False
            if is_support_seu and self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_spi_flash"):
                is_support_ext_flash_ctrl = True
                ext_flash_svc = ExtFlashControllerService()

            is_support_pma_clkmux = False
            if self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx375_clkmux"):
                is_support_pma_clkmux = True

            # This is for the case where the design file does not have include
            count = len(incl_list)
            if count == 0:
                # Build iobank
                iobank_reg = iobank_service.load_design(
                    self.design.design_file)
                if iobank_reg is not None:
                    device_setting.iobank_reg = iobank_reg
                    device_setting.iobank_reg.check_loaded_design(
                        self.design.device_db)

                if is_support_ctrl:
                    # Build control block
                    ctrl_reg = ctrl_service.load_design(
                        self.design.design_file)
                    if ctrl_reg is not None:
                        ctrl_reg.device_db = self.design.device_db
                        device_setting.ctrl_reg = ctrl_reg

                if is_support_seu:
                    # Build seu block
                    seu_reg = seu_service.load_design(self.design.design_file)
                    if seu_reg is not None:
                        # If not design level unit test, apply device to set it
                        # and generate pins
                        if not self._is_plugin_mockup:
                            seu_reg.apply_device_db(self.design.device_db)
                        device_setting.seu_reg = seu_reg

                if is_support_clkmux_plugin:
                    # There's a chance that the clkmux_info hasn't exists yet
                    assert clkmux_service is not None
                    clkmux_reg = clkmux_service.load_design(
                        self.design.design_file)
                    if clkmux_reg is None:
                        # Create the clkmux registry
                        clkmux_reg = self._build_clkmux_reg(
                            self.design.device_db)

                    # If not design level unit test, apply device to set it
                    # and generate pins
                    if not self._is_plugin_mockup:
                        clkmux_reg.apply_device_db(self.design.device_db)
                    device_setting.clkmux_reg = clkmux_reg

                if is_support_ext_flash_ctrl:
                    ext_flash_reg = ext_flash_svc.load_design(
                        self.design.design_file)

                    if ext_flash_reg is not None:
                        device_setting.ext_flash_reg = ext_flash_reg

                if is_support_pma_clkmux:
                    pma_clkmux_reg = self._build_pma_clkmux_reg(self.design.device_db)
                    device_setting.pma_clkmux_reg = pma_clkmux_reg

            else:
                for incl_file in incl_list:
                    if 'iobank' in incl_file:
                        iobank_reg = iobank_service.load_design(incl_file)

                        if iobank_reg is not None:
                            device_setting.iobank_reg = iobank_reg
                            device_setting.iobank_reg.check_loaded_design(
                                self.design.device_db)

                    if is_support_ctrl:
                        if 'ctrl_info' in incl_file:
                            ctrl_reg = ctrl_service.load_design(
                                self.design.design_file)
                            if ctrl_reg is not None:
                                ctrl_reg.device_db = self.design.device_db
                                device_setting.ctrl_reg = ctrl_reg

                    if is_support_seu:
                        if 'seu_info' in incl_file:
                            seu_reg = seu_service.load_design(
                                self.design.design_file)
                            if seu_reg is not None:
                                # If not design level unit test, apply device
                                # to set it and generate pins
                                if not self._is_plugin_mockup:
                                    seu_reg.apply_device_db(
                                        self.design.device_db)
                                device_setting.seu_reg = seu_reg

                    if is_support_clkmux_plugin:
                        if 'clkmux_info' in incl_file:
                            assert clkmux_service is not None
                            clkmux_reg = clkmux_service.load_design(
                                self.design.design_file)
                            if clkmux_reg is None:
                                # Create the clkmux registry
                                clkmux_reg = self._build_clkmux_reg(
                                    self.design.device_db)

                            if not self._is_plugin_mockup:
                                clkmux_reg.apply_device_db(
                                    self.design.device_db)
                            device_setting.clkmux_reg = clkmux_reg

                    if is_support_ext_flash_ctrl:
                        if 'ext_flash_info' in incl_file:
                            ext_flash_reg = ext_flash_svc.load_design(
                                self.design.design_file)

                            if ext_flash_reg is not None:
                                device_setting.ext_flash_reg = ext_flash_reg

            if device_setting.ctrl_reg is not None:
                if device_setting.ctrl_reg.check_resource():
                    # If not unit test, save the auto-created control setting
                    if not self._is_plugin_mockup:
                        # Make sure it will be save to file
                        self._is_design_save_needed = True

            if device_setting.seu_reg is not None:
                if device_setting.seu_reg.check_resource():
                    # If not unit test, save the auto-created seu setting
                    if not self._is_plugin_mockup:
                        # Make sure it will be save to file
                        self._is_design_save_needed = True

            # Build clock mux, it is always dynamically created, not stored in
            # file if no clkmux plugin exists
            if not is_support_clkmux_plugin:
                # Don't need to call apply_device_db since this version does not have
                # any gen pin
                device_setting.clkmux_reg = self._build_clkmux_reg(
                    self.design.device_db)

            # Apply the SEU and clockmux complex patch
            if not self._is_plugin_mockup:
                last_patch_no = None

                if device_setting.iobank_reg is not None and \
                    isinstance(device_setting.iobank_reg, IOBankRegistryAdvance):
                        patcher = IOBankAdvPatch(
                            device_setting.iobank_reg,
                            self.design.device_db
                        )
                        db_version = self.design.db_version
                        patch_no = patcher.patch(db_version.get_design_version())
                        if patcher.is_patch():
                            self.change_msg.extend(patcher.get_patch_message())

                        bank_last_patch_no = self._register_patch(None, patch_no)
                        if bank_last_patch_no is not None:
                            last_patch_no = bank_last_patch_no

                if device_setting.seu_reg is not None:
                    # If not unit test, don't run patch as it will create new
                    # design file
                    patcher = SEUPatch(device_setting.seu_reg,
                                       self.design.device_db)
                    db_version = self.design.db_version
                    patch_no = patcher.patch(db_version.get_design_version())
                    if patcher.is_patch():
                        self.change_msg.extend(patcher.get_patch_message())

                    seu_last_patch_no = self._register_patch(None, patch_no)
                    if seu_last_patch_no is not None:
                        last_patch_no = seu_last_patch_no

                if device_setting.clkmux_reg is not None and\
                        isinstance(device_setting.clkmux_reg, ClockMuxRegistryComplex):

                    # This would be impacting ClockMuxRegistryV4 too since it's inheriting
                    # from ClockMuxRegistryComplex
                    patcher = ClockMuxPatch(device_setting.clkmux_reg,
                                       self.design.device_db)
                    db_version = self.design.db_version
                    patch_no = patcher.patch(db_version.get_design_version())
                    if patcher.is_patch():
                        self.change_msg.extend(patcher.get_patch_message())

                    clkmux_last_patch_no = self._register_patch(None, patch_no)
                    if clkmux_last_patch_no is not None:
                        last_patch_no = clkmux_last_patch_no

                if last_patch_no is not None:
                    db_version = self.design.db_version
                    db_version.set_applied_patch_no(last_patch_no)
                    self.device_setting_last_patch_no = last_patch_no


            # Call the builder for regional clock mux and it will be built
            # accordingly.
            device_setting.rclkmux_reg = self._build_regional_clkmux_reg(
                self.design.device_db)


    def _build_clkmux_reg(self, device_db):
        # Build the clkmux registry from scratch
        devset_svc = DesignSettingService(self.design)
        clkmux_reg = devset_svc.build_clkmux(device_db)

        return clkmux_reg

    def _build_regional_clkmux_reg(self, device_db):
        # Build the clkmux registry from scratch
        devset_svc = DesignSettingService(self.design)
        rclkmux_reg = devset_svc.build_regional_clkmux(device_db)

        return rclkmux_reg

    def _build_pma_clkmux_reg(self, device_db: PeripheryDevice) -> PMAClockMuxRegistry:
        # Build the clkmux registry from scratch
        devset_svc = DesignSettingService(self.design)
        pma_clkmux_reg = devset_svc.build_pma_clkmux(device_db)

        return pma_clkmux_reg

    @staticmethod
    def rebuild_device_setting(design):
        """
        Build device setting for given design. To be used when updating design due to device change.

        :param design: Design instance
        """
        if design is None:
            raise app_excp.PTArgException("Design is None")

        if design.device_db is None:
            raise app_excp.PTArgException("Device db is None")

        design.device_setting = None
        devset_svc = DesignSettingService()
        device_setting = devset_svc.create_setting(design)
        if device_setting is None:
            msg = "Fail to build device setting for {}".format(
                design.device_def)
            raise db_excp.DeviceSettingException(msg, app_excp.MsgLevel.error)

        else:
            device_setting.device_db = design.device_db

            # Build iobank
            if design.plugin_map is not None and \
                    design.plugin_map.is_plugin_exist(design.device_def, "tx60_iob"):
                iobank_service = IOBankDesignServiceAdvance()
            else:
                iobank_service = iobds.IOBankDesignService()

            iobank_reg = iobank_service.load_design(design.design_file)
            device_setting.iobank_reg = iobank_reg

            device_setting.iobank_reg.check_loaded_design(
                device_setting.device_db)

            # Control block is not part of instantiable design block.
            # Need to check manually since is_block_supported() cannot be used.
            if design.plugin_map is not None and \
                    design.plugin_map.is_plugin_exist(design.device_def, "an20_ctrl"):
                # Build control block
                ctrl_service = ctrlds.ControlDesignService()
                ctrl_reg = ctrl_service.load_design(design.design_file)
                ctrl_reg.device_db = design.device_db
                device_setting.ctrl_reg = ctrl_reg

            # Control block is not part of instantiable design block.
            # Need to check manually since is_block_supported() cannot be used.
            if design.plugin_map is not None and \
                    design.plugin_map.is_plugin_exist(design.device_def, "tx60_ctrl"):
                # Build control block
                ctrl_service = ctrlds.ControlDesignService()
                ctrl_reg = ctrl_service.load_design(design.design_file)
                ctrl_reg.device_db = design.device_db
                device_setting.ctrl_reg = ctrl_reg

                # Build seu block
                seu_service = seuds.SEUDesignService()
                seu_reg = seu_service.load_design(design.design_file)
                seu_reg.device_db = design.device_db
                device_setting.seu_reg = seu_reg

                if design.plugin_map.is_plugin_exist(design.device_def, "tx60_spi_flash"):
                    # Build the ext flash controller
                    ext_flash_service = ExtFlashControllerService()
                    ext_flash_reg = ext_flash_service.load_design(design.design_file)
                    device_setting.ext_flash_reg = ext_flash_reg

        design.device_setting = device_setting

        if device_setting.ctrl_reg is not None:
            device_setting.ctrl_reg.check_resource()

        if device_setting.seu_reg is not None:
            device_setting.seu_reg.check_resource()

    def get_device_project_in_design_file(self, design_file):

        self.logger.info("Get device from {}".format(design_file))

        with open(design_file, 'r', encoding='UTF-8') as xml_file:
            # get an iterable
            context = et.iterparse(xml_file, events=(
                "start", "end", "start-ns", "end-ns"))

            # turn it into an iterator
            context = iter(context)

            db_tag = self._ns + "design_db"

            device_name = ""
            proj_name = ""

            # Build top level design
            for event, elem in context:
                if event == "start":
                    if elem.tag == db_tag:
                        design_attr = elem.attrib
                        device_name = design_attr.get("device_def", "")
                        proj_name = design_attr.get("name", "")
                        break

            return proj_name, device_name


@gen_util.freeze_it
class DesignWriterXml:

    def __init__(self):
        self.design = None
        self.logger = Logger
        self.design = None # type: Optional[db.PeriDesign]
        self.design_file = ""

    def write(self, design: db.PeriDesign, design_file: str):
        self.design = design
        self.design_file = design_file

        dirname = os.path.dirname(design_file)
        if dirname != "" and not os.path.exists(dirname):
            os.mkdir(dirname)

        writer = qt.QXmlStreamWriter()
        writer.setCodec(qt.QTextCodec.codecForName("UTF-8"))
        xml_file = qt.QSaveFile(design_file)

        if xml_file.open(qt.QFile.WriteOnly) is True:
            writer.setDevice(xml_file)
        else:
            self.logger.warning(
                "Fail to open design file for writing, " + design_file)
            return

        writer.setAutoFormatting(True)
        writer.writeStartDocument()

        efx_ns = "efxpt"
        writer.writeStartElement(efx_ns + ":design_db")
        writer.writeAttribute("name", self.design.name)
        writer.writeAttribute("device_def", self.design.device_def)
        self.design.version = "{}".format(__version__)
        writer.writeAttribute("version", self.design.version)

        self.design.update_design_version()
        db_ver_name = self.design.get_design_version_name()
        self.logger.warning("Save db version: {}".format(db_ver_name))
        writer.writeAttribute("db_version", db_ver_name)

        self.design.last_change_date = time.asctime(
            time.localtime(time.time()))
        writer.writeAttribute("last_change_date", self.design.last_change_date)

        writer.writeAttribute(
            "xmlns:efxpt", "http://www.efinixinc.com/peri_design_db")
        writer.writeAttribute(
            "xmlns:xsi", "http://www.w3.org/2001/XMLSchema-instance")
        writer.writeAttribute(
            "xsi:schemaLocation", "http://www.efinixinc.com/peri_design_db peri_design_db.xsd ")

        device_setting = self.design.device_setting
        if device_setting is not None:
            writer.writeStartElement(efx_ns + ":device_info")

            if device_setting.iobank_reg is not None:
                if self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_iob"):
                    iobank_service = IOBankDesignServiceAdvance()
                else:
                    iobank_service = iobds.IOBankDesignService()

                iobank_service.save_design(self.design, design_file, writer)

            # Control block is not part of instantiable design block.
            # Need to check manually since is_block_supported() cannot be used.
            if self.design.plugin_map.is_plugin_exist(self.design.device_def, "an20_ctrl"):
                if device_setting.ctrl_reg is not None:
                    ctrl_service = ctrlds.ControlDesignService()
                    ctrl_service.save_design(self.design, design_file, writer)

            # Control and seu block is not part of instantiable design block.
            # Need to check manually since is_block_supported() cannot be used.
            elif self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_ctrl"):
                if device_setting.ctrl_reg is not None:
                    ctrl_service = ctrlds.ControlDesignService()
                    ctrl_service.save_design(self.design, design_file, writer)

                if device_setting.seu_reg is not None:
                    seu_service = seuds.SEUDesignService()
                    seu_service.save_design(self.design, design_file, writer)

            if self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_clkmux"):
                if device_setting.clkmux_reg is not None:
                    clkmux_service = ClockMuxDesignServiceAdvance()
                    clkmux_service.save_design(
                        self.design, design_file, writer)

            elif self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx180_clkmux"):
                if device_setting.clkmux_reg is not None:
                    clkmux_service = ClockMuxDesignServiceComplex()
                    clkmux_service.save_design(
                        self.design, design_file, writer)

            elif self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx375_clkmux"):
                if device_setting.clkmux_reg is not None:
                    clkmux_service = ClockMuxDesignServiceV4()
                    clkmux_service.save_design(
                        self.design, design_file, writer)

            # Write out the ext flash info
            if self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_spi_flash") and\
                    self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_ctrl"):
                if device_setting.ext_flash_reg is not None:
                    ext_flash_service = ExtFlashControllerService()
                    ext_flash_service.save_design(self.design, design_file, writer)

            # device info
            writer.writeEndElement()

        # Blocks must be written in correct order
        blk_service_list: List[dbi.DesignBlockService] = []

        # GPIO
        if self.design.is_block_supported(db.PeriDesign.BlockType.gpio) or \
                self.design.is_block_supported(db.PeriDesign.BlockType.adv_l2_gpio):
            blk_service_list.append(gds.GPIODesignService())

        elif self.design.is_block_supported(db.PeriDesign.BlockType.comp_gpio):
            blk_service_list.append(comp_gds.GPIODesignServiceComplex())

        # PLL
        if self.design.is_block_supported(db.PeriDesign.BlockType.pll):
            blk_service_list.append(pllds.PLLDesignService())

        elif self.design.is_block_supported(db.PeriDesign.BlockType.adv_pll):
            blk_service_list.append(adv_pllds.PLLDesignServiceAdvance())

        elif self.design.is_block_supported(db.PeriDesign.BlockType.comp_pll):
            blk_service_list.append(comp_pllds.PLLDesignServiceComplex())

        elif self.design.is_block_supported(db.PeriDesign.BlockType.efx_pll_v3_comp):
            blk_service_list.append(PLLDesignServiceV3Complex())

        elif self.design.is_block_supported(db.PeriDesign.BlockType.efx_fpll_v1):
            blk_service_list.append(EfxFpllV1DesignService())

        if self.design.is_block_supported(db.PeriDesign.BlockType.osc):
            blk_service_list.append(oscds.OSCDesignService())
        elif self.design.is_block_supported(db.PeriDesign.BlockType.adv_osc):
            blk_service_list.append(adv_oscds.OSCDesignServiceAdvance())
        elif self.design.is_block_supported(db.PeriDesign.BlockType.hposc):
            blk_service_list.append(hposcds.HPOSCDesignService())

        # LVDS
        if self.design.is_block_supported(db.PeriDesign.BlockType.lvds):
            blk_service_list.append(lvdsds.LVDSDesignService())

        elif self.design.is_block_supported(db.PeriDesign.BlockType.adv_lvds):
            blk_service_list.append(adv_lvdsds.LVDSDesignServiceAdvance())

        # MIPI
        if self.design.is_block_supported(db.PeriDesign.BlockType.mipi):
            blk_service_list.append(mipids.MIPIDesignService())
        elif self.design.is_block_supported(db.PeriDesign.BlockType.mipi_hard_dphy):
            blk_service_list.append(MIPIHardDPHYDesignService())

        if self.design.is_block_supported(db.PeriDesign.BlockType.jtag):
            blk_service_list.append(jtagds.JTAGDesignService())

        # DDR
        if self.design.is_block_supported(db.PeriDesign.BlockType.ddr):
            blk_service_list.append(ddrds.DDRDesignService())
        elif self.design.is_block_supported(db.PeriDesign.BlockType.adv_ddr):
            blk_service_list.append(adv_ddrds.DDRDesignServiceAdvance())

        if self.design.is_block_supported(db.PeriDesign.BlockType.h264):
            blk_service_list.append(h264ds.H264DesignService())

        if self.design.is_block_supported(db.PeriDesign.BlockType.mipi_dphy):
            blk_service_list.append(mdphyds.MIPIDPhyDesignService())

        if self.design.is_block_supported(db.PeriDesign.BlockType.spi_flash):
            blk_service_list.append(spfds.SPIFlashDesignService())

        if self.design.is_block_supported(db.PeriDesign.BlockType.hyper_ram):
            blk_service_list.append(hpramds.HyperRAMDesignService())

        if self.design.is_block_supported(db.PeriDesign.BlockType.pll_ssc):
            blk_service_list.append(PLLSSCDesignService())

        if self.design.is_block_supported(db.PeriDesign.BlockType.common_quad):
            blk_service_list.append(QuadLaneCommonDesignService(design.device_db))

        if self.design.is_block_supported(db.PeriDesign.BlockType.quad_pcie):
            blk_service_list.append(QuadPCIEDesignService())

        if self.design.is_block_supported(db.PeriDesign.BlockType.lane_10g):
            blk_service_list.append(Lane10GDesignService())

        if self.design.is_block_supported(db.PeriDesign.BlockType.lane_1g):
            blk_service_list.append(Lane1GDesignService())

        if self.design.is_block_supported(db.PeriDesign.BlockType.raw_serdes):
            blk_service_list.append(RawSerdesDesignService())

        if self.design.is_block_supported(db.PeriDesign.BlockType.soc):
            blk_service_list.append(SOCDesignService())

        for blk_service in blk_service_list:
            blk_service.save_design(self.design, design_file, writer)

        # Partial design
        for pd in design.get_all_partial_designs():
            writer.writeStartElement(f"{efx_ns}:partial_design")
            writer.writeAttribute("source", pd)
            writer.writeEndElement()

        # design
        writer.writeEndElement()

        writer.writeEndDocument()

        # Save changes to harddisk
        if xml_file.commit() is True:
            self.logger.info("Write design to {}".format(design_file))
        else:
            self.logger.info("Fail to write design to {}".format(design_file))


@gen_util.freeze_it
class DesignSettingService:
    """
    Provides device setting related operation

    TODO : This class needs refactoring to properly support different cases of creating device setting blocks
    a) For new design
    b) Loading design
    c) Legacy design
    """

    def __init__(self, design: Optional[db.PeriDesign]=None):
        self.design = design
        self.logger = Logger

    def create_setting(self, design):
        """
        Build device setting based on chosen device. This is to be used for new design
        or upgrading legacy design.
        :return: Device setting instance
        """

        if design is None or design.device_db is None:
            return

        self.design = design
        dev_setting = dev_set.DeviceSetting()
        dev_setting.device_db = design.device_db

        # Build IOBank setting
        dev_setting.iobank_reg = self.build_iobank(dev_setting.device_db)

        # Build Clock Mux setting
        dev_setting.clkmux_reg = self.build_clkmux(dev_setting.device_db)

        # Build control block
        dev_setting.ctrl_reg = self.build_ctrl(dev_setting.device_db)

        # Build seu block
        dev_setting.seu_reg = self.build_seu(dev_setting.device_db)

        # Build the regional clock mux
        dev_setting.rclkmux_reg = self.build_regional_clkmux(dev_setting.device_db)

        # Build external flash controller
        dev_setting.ext_flash_reg = self.build_ext_flash_controller()

        # Build PMA Clock Mux
        dev_setting.pma_clkmux_reg = self.build_pma_clkmux(dev_setting.device_db)

        return dev_setting

    def migrate_setting(self, design):
        """
        Build device setting based on chosen device. This is to be used
        for migrating a design since we want to retain control setting
        but not the rest.

        :return: Device setting instance
        """

        if design is None or design.device_db is None:
            return

        self.design = design
        dev_setting = self.design.device_setting

        # Update the device db saved in device setting
        dev_setting.device_db = design.device_db

        # Build IOBank setting
        dev_setting.iobank_reg = self._migrate_iobank(dev_setting.device_db)

        # Build Clock Mux setting
        dev_setting.clkmux_reg = self.build_clkmux(dev_setting.device_db)

        # Migrate control block
        dev_setting.ctrl_reg = self.migrate_ctrl(dev_setting.device_db)

        # Migrate the seu
        dev_setting.seu_reg = self.migrate_seu(dev_setting.device_db)

        # Build the regional clkmux
        dev_setting.rclkmux_reg = self.build_regional_clkmux(dev_setting.device_db)

        # Build external flash controller
        dev_setting.ext_flash_reg = self.migrate_ext_flash_controller()

        # Build PMA Clock Mux
        dev_setting.pma_clkmux_reg = self.build_pma_clkmux(dev_setting.device_db)

    def build_iobank(self, device_db):
        """
        Build IOBank instances as per device

        :param device_db: Device db instance
        :return: IO bank registry instance
        """
        if device_db is None:
            return None

        if self.design.plugin_map is None:
            return None

        # Tesseract has a different iobank
        is_adv_iobank = False
        if self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_iob"):
            iobank_reg = adv_iobd.IOBankRegistryAdvance()
            is_adv_iobank = True
        else:
            iobank_reg = iobd.IOBankRegistry()

        iobank_map = {}
        # pylint: disable=W0212
        iobank_reg._iobank = iobank_map

        io_info = device_db.get_io_pad()
        if io_info is not None:

            # Get the io banks based on the package and not die
            dev_svc = dev_dbi.DeviceDBService(device_db)
            dev_io_banks = dev_svc.get_resource_io_bank_names()
            bonded_bank_map = dev_svc.get_resource_bonded_bank_map()

            for bank_name in dev_io_banks:
                def_iostd = ""

                # Assign the bank to the default io std
                io_bank_obj = io_info.get_bank_info(bank_name)

                if io_bank_obj is not None:
                    def_iostd = io_bank_obj.get_default_io_standard()

                if is_adv_iobank:
                    iob_obj = adv_iobd.IOBankAdvance(bank_name, def_iostd)
                    iob_obj.setup_mode_sel_name(bonded_bank_map.get(bank_name, []))
                    iob_obj.generate_pin_name()
                else:
                    iob_obj = iobd.IOBank(bank_name, def_iostd)

                iobank_map[bank_name] = iob_obj

        return iobank_reg

    def build_clkmux(self, device_db):
        """
        Build Clock Mux instances as per device

        :param device_db: Device db instance
        :return: Clock mux registry instance
        """
        if device_db is None:
            return None

        # Initially we used the lvds an indicator if clkmux needs to be supported
        # where this is clkmux with autorouting (no clkmux plugin required).
        # However, there were cases where a device has clkmux but no lvds was available.
        # With newer devices, there are clkmux plugin to support user configuration on
        # the clkmux.
        if not self.design.is_block_supported(db.PeriDesign.BlockType.adv_pll) and\
                not self.design.is_block_supported(db.PeriDesign.BlockType.comp_pll) and\
                not self.design.is_block_supported(db.PeriDesign.BlockType.efx_pll_v3_comp) and\
                not self.design.is_block_supported(db.PeriDesign.BlockType.efx_fpll_v1):
            return None

        if self.design.plugin_map is None:
            return None

        # Tesseract has a different clkmux
        if self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx375_clkmux"):
            clkmux_service = ClockMuxDesignServiceV4()
        elif self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx180_clkmux"):
            clkmux_service = ClockMuxDesignServiceComplex()
        elif self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_clkmux"):
            clkmux_service = ClockMuxDesignServiceAdvance()
        else:
            clkmux_service = clkmuxds.ClockMuxDesignService()

        # In this case, we recreate the clockmuxes (ignoring what is stored in
        # file, if any)
        clkmux_reg = clkmux_service.create_registry()
        clkmux_reg.apply_device_db(device_db)

        device_inf = dev_dbi.DeviceDBService(device_db)
        clkmux_dev_service = device_inf.get_block_service(
            device_db.clkmux_type)

        if clkmux_dev_service is not None:
            clkmux_dev_instances = clkmux_dev_service.get_usable_instance_names()

            for ins_name in clkmux_dev_instances:
                clkmux = clkmux_reg.create_clock_mux(ins_name)
                self.logger.info("Build clock mux : {}".format(clkmux.name))

        return clkmux_reg

    def build_regional_clkmux(self, device_db):
        """
        Build Clock Mux instances as per device

        :param device_db: Device db instance
        :return: Clock mux registry instance
        """
        if device_db is None:
            return None

        if self.design.plugin_map is None:
            return None

        skip = True
        # There is no plugin for regional clockmux.  This is exactly
        # like Trion clkmux.  We associate the regional clockmux with the Ti180 clkmux
        # and Ti180 MIPI since the connection to the regional is all mipi signals
        if self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx375_clkmux") or \
                self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx180_clkmux"):
            skip = False

        if skip:
            return None

        if self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx375_clkmux"):
            rclkmux_service = RClockMuxV2DesignService()
        elif self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx180_clkmux"):
            rclkmux_service = RClockMuxDesignService()

        # In this case, we recreate the clockmuxes (ignoring what is stored in
        # file, if any)
        rclkmux_reg = rclkmux_service.create_registry()
        rclkmux_reg.apply_device_db(device_db)

        device_inf = dev_dbi.DeviceDBService(device_db)

        for rclkmux_type in self.design.device_db.rclkmux_types:
            rclkmux_dev_service = device_inf.get_block_service(rclkmux_type)

            if rclkmux_dev_service is not None:
                clkmux_dev_instances = rclkmux_dev_service.get_usable_instance_names()

                for ins_name in clkmux_dev_instances:
                    dev_inst = device_db.find_instance(ins_name)
                    # PT-2446: Don't create the clock mux instance if none of its
                    # inputs are connected to another instance (unusable)
                    if dev_inst and dev_inst.check_pins_connected_to_instance(device_db, "IN"):
                        rclkmux_reg.create_clock_mux_from_dev_inst(dev_inst)

        return rclkmux_reg

    def build_pma_clkmux(self, device_db: PeripheryDevice) -> PMAClockMuxRegistry:
        """
        Build Clock Mux instances as per device

        :param device_db: Device db instance
        :return: Clock mux registry instance
        """
        if device_db is None:
            return None

        if self.design.plugin_map is None:
            return None

        if not self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx375_clkmux"):
            return None

        # In this case, we recreate the clockmuxes (ignoring what is stored in
        # file, if any)
        pma_clkmux_reg = PMAClockMuxRegistry()
        pma_clkmux_reg.apply_device_db(device_db)

        device_inf = dev_dbi.DeviceDBService(device_db)
        dev_service = device_inf.get_block_service(
            dev_dbi.DeviceDBService.BlockType.PMA_CLKMUX)

        if dev_service is not None:
            clkmux_dev_instances = dev_service.get_usable_instance_names()

            for ins_name in clkmux_dev_instances:
                clkmux = pma_clkmux_reg.create_clock_mux(ins_name)
                self.logger.info("Build PMA clock mux : {}".format(clkmux.name))

        return pma_clkmux_reg

    def build_ctrl(self, device_db):
        """
        Build control instance as per device

        :param device_db: Device db instance
        :return: control registry instance
        """

        if self.design is None:
            return None

        if device_db is None:
            return None

        if self.design.plugin_map is None:
            return None

        # Control block is not part of instantiable design block.
        # Need to check manually since is_block_supported() cannot be used.
        if not self.design.plugin_map.is_plugin_exist(self.design.device_def, "an20_ctrl") and \
                not self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_ctrl"):
            return None

        ctrl_service = ctrlds.ControlDesignService()
        ctrl_reg = ctrl_service.create_registry()
        ctrl_reg.device_db = device_db

        # One control per device
        ctrl = ctrl_reg.create_instance("cfg", auto_pin=True)
        self.logger.debug("Recreating default ctrl setting")

        # Default resource will be assigned
        ctrl_reg.assign_inst_device(ctrl, "")

        return ctrl_reg

    def build_seu(self, device_db):
        """
        Build seu instance as per device

        :param device_db: Device db instance
        :return: seu registry instance
        """

        if self.design is None:
            return None

        if device_db is None:
            return None

        if self.design.plugin_map is None:
            return None

        # SEU block is not part of instantiable design block.
        # Need to check manually since is_block_supported() cannot be used.
        # It is part of control plugin for tesseract.
        if not self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_ctrl"):
            return None

        seu_service = seuds.SEUDesignService()
        seu_reg = seu_service.create_registry()
        seu_reg.apply_device_db(self.design.device_db)

        # One seu per device
        seu = seu_reg.create_instance("seu", auto_pin=True)
        self.logger.debug("Recreating default seu setting")

        # Default resource will be assigned
        seu_reg.assign_inst_device(seu, "")

        return seu_reg

    def build_ext_flash_controller(self):
        """
        Build External Flash Controller setting

        :return: ext flash controller registry instance
        """

        if self.design is None:
            return None

        if self.design.plugin_map is None:
            return None

        # Ext Flash Controller only supported if tx60 control and spi flash exists
        if not self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_ctrl") or\
                not self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_spi_flash"):
            return None

        ext_flash_svc = ExtFlashControllerService()
        ext_flash_reg = ext_flash_svc.create_registry()

        # Create instance with default setting
        ext_flash = ext_flash_reg.create_instance()
        self.logger.debug("Recreating default external flash controller setting")

        return ext_flash_reg

    def migrate_ctrl(self, device_db: Optional[PeripheryDevice]):
        """
        Migrate the control instance as per device,

        Old design need to be migrated in this function,
        while design.device_def is updated with new device name

        :param device_db: Device db instance (new device)
        :return: control registry instance if support, else None
        """

        if self.design is None or self.design.device_setting is None:
            return None

        if device_db is None:
            return None

        if self.design.plugin_map is None:
            return None

        # Control block is not part of instantiable design block.
        # Need to check manually since is_block_supported() cannot be used.
        if not self.design.plugin_map.is_plugin_exist(self.design.device_def, "an20_ctrl") and \
                not self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_ctrl"):
            return None

        # Check if the design already has the control registry.
        ctrl_reg = self.design.device_setting.ctrl_reg

        if ctrl_reg is None:
            # Create it if it did not exists when it supports it
            # Example: Migrate from T8 to T120
            ctrl_service = ctrlds.ControlDesignService()
            ctrl_reg = ctrl_service.create_registry()
            ctrl_reg.device_db = device_db

            # One control per device
            ctrl = ctrl_reg.create_instance("cfg", auto_pin=True)
            self.logger.debug("Recreating default sctrl setting")

            # Default resource will be assigned
            ctrl_reg.assign_inst_device(ctrl, "")

        else:
            # The resource has to be reassigned

            # Reset
            ctrl_reg.change_device(device_db)

            # Reassign
            ctrl = ctrl_reg.get_ctrl()
            ctrl_reg.assign_inst_device(ctrl, "")

            # PT-2162 Keep the value if having PCIe plugin, else always disable it
            if not self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx375_quad_pcie"):
                ctrl.is_user_mode_enable = False

        return ctrl_reg

    def migrate_seu(self, device_db):
        """
        Migrate the seu instance as per device

        :param device_db: Device db instance
        :return: control registry instance
        """

        if self.design is None:
            return None

        if device_db is None:
            return None

        if self.design.plugin_map is None:
            return None

        # Control block is not part of instantiable design block.
        # Need to check manually since is_block_supported() cannot be used.
        if not self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_ctrl"):
            return None

        # Check if the design already has the control registry.
        seu_reg = self.design.device_setting.seu_reg

        if seu_reg is None:
            # Create it if it did not exists when it supports it
            # Example: Migrate from T8 to Ti60
            seu_service = seuds.SEUDesignService()
            seu_reg = seu_service.create_registry()
            seu_reg.apply_device_db(device_db)

            # One seu per device
            seu = seu_reg.create_instance("seu", auto_pin=True)
            self.logger.debug("Recreating default seu setting")

            # Default resource will be assigned
            seu_reg.assign_inst_device(seu, "")
        else:
            # The resource has to be reassigned

            # Reset
            seu_reg.change_device(device_db)

            # Reassign
            all_seu = seu_reg.get_all_inst()
            for seu in all_seu:
                seu_reg.assign_inst_device(seu, "")

        return seu_reg

    def migrate_ext_flash_controller(self):
        """
        Migrate the external flash controller setting

        :param device_db:
        :return:
        """

        if self.design is None:
            return None

        if self.design.plugin_map is None:
            return None

        # Control block is not part of instantiable design block.
        # Need to check manually since is_block_supported() cannot be used.
        if not self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_ctrl") or\
                not self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_spi_flash"):
            return None

        # Check if the design already has the control registry.
        ext_flash_reg = self.design.device_setting.ext_flash_reg

        if ext_flash_reg is None:

            ext_flash_svc = ExtFlashControllerService()
            ext_flash_reg = ext_flash_svc.create_registry()

            # Create instance with default setting
            ext_flash_reg.create_instance()
            self.logger.debug("Recreating default external flash controller setting")

        return ext_flash_reg

    def _migrate_iobank(self, device_db):
        """
        Build IOBank instances as per device

        :param device_db: Device db instance
        :return: IO bank registry instance
        """
        if device_db is None:
            return None

        # PT-369: We always create a new registry but the
        # map contents are created looking at the old map
        # in retaining the iostandard assignment if they
        # are compatible
        is_adv_iob = False
        if self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_iob"):
            new_iobank_reg = adv_iobd.IOBankRegistryAdvance()
            is_adv_iob = True
        else:
            new_iobank_reg = iobd.IOBankRegistry()

        iobank_map = {}

        # Current IOBank Registry
        old_iobank_reg = self.design.device_setting.iobank_reg
        old_iobank_list = []

        migrate_iostd = True
        if old_iobank_reg is None:
            # If the old design has no iostd set, then just
            # set all to default
            migrate_iostd = False
        else:
            old_iobank_list = old_iobank_reg.get_all_iobank()

        # The new device ios
        io_info = device_db.get_io_pad()

        if io_info is not None:

            # Get the io banks based on the package and not die
            dev_svc = dev_dbi.DeviceDBService(device_db)
            dev_io_banks = dev_svc.get_resource_io_bank_names()
            bonded_bank_map = dev_svc.get_resource_bonded_bank_map()
            hvio_poc_svc = dev_svc.get_block_service(
                dev_svc.BlockType.HVIO_POC)

            # This is iterating through the new device bank list
            for bank_name in dev_io_banks:
                set_iostd = ""
                is_dyn_voltage = False
                mode_sel_name = ""

                # Assign the bank to the default io std
                io_bank_obj = io_info.get_bank_info(bank_name)

                if io_bank_obj is not None:

                    def_iostd = io_bank_obj.get_default_io_standard()

                    if migrate_iostd:
                        old_iob_obj = old_iobank_reg.get_iobank_by_name(
                            bank_name)

                        if old_iob_obj is not None:
                            old_iostd = old_iob_obj.get_raw_io_standard()

                            # Check if the old io std has the information
                            # on the dynamic voltage and mode pin sel
                            if isinstance(old_iob_obj, adv_iobd.IOBankAdvance):
                                # We only retain it if the new device supports
                                # dynamic voltage. If migrate to ES device, we reset
                                # the flag since feature isn't supported even if UI
                                # still exposes it.
                                if hvio_poc_svc is not None and hvio_poc_svc.is_hvio_bank_with_dyn_voltage(bank_name):
                                    is_dyn_voltage = old_iob_obj.is_dyn_voltage
                                    mode_sel_name = old_iob_obj.bank2mode_sel_name.get(bank_name, "")

                            if io_bank_obj.is_io_standard_in_bank(old_iostd):
                                set_iostd = old_iostd
                                self.logger.debug("Migrate io bank {} with io std {}".format(
                                    bank_name, set_iostd))
                            else:
                                # PT-369: Warning if the setting is not
                                # preserved. Reset the io standard since the iostd in old
                                # device does not exist in the new device
                                set_iostd = def_iostd
                                self.logger.warning(
                                    'I/O Voltage {} for I/O Bank {} does not exists and '
                                    'was migrated to {}'.format(
                                        old_iostd, bank_name, set_iostd))

                        else:
                            # Set to default since nothing to migrate
                            set_iostd = def_iostd
                            self.logger.debug("Old device does not have bank {} to migrate".format(
                                bank_name))
                    else:
                        set_iostd = def_iostd
                        self.logger.debug("Setting io std {} to bank {}".format(
                            set_iostd, bank_name))

                if is_adv_iob:
                    bonded_bank_list = bonded_bank_map.get(bank_name, [])
                    new_iob = adv_iobd.IOBankAdvance(bank_name, set_iostd, is_dyn_voltage)
                    new_iob.setup_mode_sel_name(bonded_bank_list)
                    new_iob.generate_pin_name()

                    if mode_sel_name != "" and bank_name in bonded_bank_list:
                        new_iob.bank2mode_sel_name[bank_name] = mode_sel_name

                else:
                    new_iob = iobd.IOBank(bank_name, set_iostd)

                iobank_map[bank_name] = new_iob

            # Check if there were iobank in the old device which
            # no longer exists in the new device
            non_exist_banks = ""
            for iob in old_iobank_list:
                if iob.name not in dev_io_banks:
                    if non_exist_banks != "":
                        non_exist_banks = non_exist_banks + ", " + iob.name
                    else:
                        non_exist_banks = iob.name

            if non_exist_banks != "":
                self.logger.warning(
                    "The following bank no longer exists in the migrated device: {}".format(
                        non_exist_banks))

        # pylint: disable=W0212
        new_iobank_reg._iobank = iobank_map

        return new_iobank_reg


if __name__ == "__main__":
    pass
