"""
Copyright (C) 2017-2020 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Mar 04, 2020

@author: yasmin
"""

from __future__ import annotations
import os
import sys
import enum
from typing import Optional


try:
    from enum import auto
    import util.gen_util as gen_util
except ImportError:
    import util.gen_util as gen_util

    auto = gen_util.auto_builder()

from util.singleton_logger import Logger

import device.db_interface as device_dbi

from design.db import PeriDesign, DesignIssueRegistry
from common_device.gpio.gpio_rule import GPIOChecker, GPIOCompChecker
from common_device.gpio.lvds_gpio_rule import LVDSGPIOChecker, LVDSGPIOCompChecker
from common_device.gpio.bus_rule import BusChecker
from common_device.pll.pll_rule import PLLChecker
from common_device.osc.osc_rule import OSCChecker
from common_device.lvds.lvds_rule import LVDSChecker
from common_device.h264.h264_rule import H264Checker
from common_device.mipi.mipi_rule import MIPIChecker
from common_device.clock.clock_rule import ClockChecker
from common_device.ctrl.ctrl_rule import ControlChecker
from common_device.ddr.ddr_rule import DDRChecker
from common_device.iobank.iobank_rule import IOBankChecker
from common_device.jtag.jtag_rule import JTAGChecker
from common_device.clock_mux.clkmux_rule import ClockMuxChecker
from common_device.seu.seu_rule import SEUChecker
from common_device.mipi_dphy.mipi_dphy_rule import MIPIDPhyChecker
from common_device.spi_flash.spi_flash_rule import SPIFlashChecker
from common_device.hyper_ram.hyper_ram_design_param_info import HyperRAMParamId
from common_device.hyper_ram.hyper_ram_rule import HyperRAMChecker
from common_device.rclock_mux.rclkmux_rule import RClockMuxChecker
from common_device.ext_flash.ext_flash_rule import ExtFlashControllerChecker

from an20_device.pll.pll_rule_adv import PLLCheckerAdv

from tx60_device.osc.osc_rule_adv import OSCCheckerAdv
from tx60_device.pll.pll_rule_comp import PLLCheckerComp
from tx60_device.gpio.gpio_rule_comp import GPIOCheckerComp, GPIOCompCheckerComp
from tx60_device.gpio.hsio_gpio_rule import HSIOGPIOChecker, HSIOGPIOCompChecker
from tx60_device.lvds.lvds_rule_adv import LVDSCheckerAdv
from tx60_device.clock_mux.clkmux_rule_adv import ClockMuxCheckerAdv
from tx60_device.clock.clock_rule_adv import ClockCheckerAdv
from tx60_device.gpio.bus_rule_comp import BusCheckerComp
from tx60_device.iobank.iobank_rule_adv import IOBankCheckerAdv

from tx180_device.clock_mux.clkmux_rule_comp import ClockMuxCheckerComp
from tx180_device.mipi_dphy.mipi_rule_adv import MIPICheckerAdv
from tx180_device.pll.pll_rule import PLLCheckerV3Complex
from tx180_device.ddr.ddr_rule_adv import DDRCheckerAdv
from tx180_device.pll_ssc.pll_ssc_rule import PLLSSCChecker

from tx375_device.osc.rule import OSCCheckerV3
from tx375_device.soc.rule import SOCChecker
from tx375_device.quad_pcie.rule import QuadPCIEChecker
from tx375_device.fpll.rule import EfxFpllV1Checker
from tx375_device.pma_clock_mux.rule import PMAClockMuxChecker
from tx375_device.lane10g.rule import Lane10GChecker
from tx375_device.common_quad.rule import QuadLaneCommonChecker
from tx375_device.raw_serdes.rule import RawSerdesChecker
from tx375_device.lane1g.rule import Lane1GChecker

sys.path.append(os.path.join(os.path.dirname(__file__), '..'))


@gen_util.freeze_it
class RuleCheckService:
    """
    Provides rule-based design checks operation.

    Two type of run mode supported
    - STD : Run standard rule checks, optimized for Interface Designer application
    - COMP : Run comprehensive rule checks, when design db is edited outside from Interface Designer app

    It supports rule filtering through pt_rule.ini stored at design file location.

    Running a check will
    - generate issue into :class:`DesignIssueRegistry` owned by :class:`PeriDesign`
    - run clock mux routing on top of checking it
    """

    class RunModeType(enum.Enum):
        """
        Run mode with specific rule set
        """
        STD = auto()  #: Optimized checks
        COMP = auto()  #: Comprehensive checks

    class RuleSetType(enum.Enum):
        """
        Enum to indicate rule set type
        """
        GPIO = auto()
        PLL = auto()
        OSC = auto()
        IO_BANK = auto()
        CLOCK = auto()
        LVDS = auto()
        MIPI = auto()
        CTRL = auto()
        CLKMUX = auto()
        JTAG = auto()
        DDR = auto()
        H264 = auto()
        SEU = auto()
        MIPI_DPHY = auto()
        SPI_FLASH = auto()
        HYPER_RAM = auto()
        RCLKMUX = auto()
        MIPI_HARD_DPHY = auto()
        EXT_FLASH_CTRL = auto()
        PLL_SSC = auto()
        SOC = auto()
        QUAD_PCIE = auto()
        PMA_CLKMUX = auto()
        LANE_10G = auto()
        COMMON_QUAD_LANE = auto()
        LANE_1G = auto()
        RAW_SERDES = auto()


    # NOTE: Please make sure that the
    # rule name starts with the specified string here in order
    # for the rule filtering (exclude) to work properly.
    # Also, for now, no special character as part of the rule
    # prefix. For example, we name it as h264_rule instead of
    # h.264_rule. Update this note if it has to change.

    rule_type2str_map = {
        RuleSetType.GPIO: "gpio_rule",
        RuleSetType.PLL: "pll_rule",
        RuleSetType.OSC: "osc_rule",
        RuleSetType.IO_BANK: "io_bank_rule",
        RuleSetType.CLOCK: "clock_rule",
        RuleSetType.LVDS: "lvds_rule",
        RuleSetType.MIPI: "mipi_rule",
        RuleSetType.CTRL: "configuration_rule",
        RuleSetType.CLKMUX: "clkmux_rule",
        RuleSetType.JTAG: "jtag_rule",
        RuleSetType.DDR: "ddr_rule",
        RuleSetType.H264: "h264_rule",
        RuleSetType.SEU: "seu_rule",
        RuleSetType.MIPI_DPHY: "mipi_ln_rule",
        RuleSetType.SPI_FLASH: "spi_flash_rule",
        RuleSetType.HYPER_RAM: "hyper_ram_rule",
        RuleSetType.RCLKMUX: "rclkmux_rule",
        RuleSetType.MIPI_HARD_DPHY: "mipi_dphy_rule",
        RuleSetType.EXT_FLASH_CTRL: "ext_flash_rule",
        RuleSetType.PLL_SSC: "pll_ssc_rule",
        RuleSetType.SOC: "qcrv32_rule",
        RuleSetType.QUAD_PCIE: "pcie_rule",
        RuleSetType.PMA_CLKMUX: "pma_clkmux_rule",
        RuleSetType.LANE_10G: "10gbase_kr_rule",
        RuleSetType.COMMON_QUAD_LANE: "common_quad_lane_rule",
        RuleSetType.LANE_1G: "sgmii_rule",
        RuleSetType.RAW_SERDES: "pma_direct_rule"
    }  # Type to string map

    rule_str2type_map = {
        "gpio_rule": RuleSetType.GPIO,
        "pll_rule": RuleSetType.PLL,
        "osc_rule": RuleSetType.OSC,
        "io_bank_rule": RuleSetType.IO_BANK,
        "clock_rule": RuleSetType.CLOCK,
        "lvds_rule": RuleSetType.LVDS,
        "mipi_rule": RuleSetType.MIPI,
        "configuration_rule": RuleSetType.CTRL,
        "clkmux_rule": RuleSetType.CLKMUX,
        "jtag_rule": RuleSetType.JTAG,
        "ddr_rule": RuleSetType.DDR,
        "h264_rule": RuleSetType.H264,
        "seu_rule": RuleSetType.SEU,
        "mipi_ln_rule": RuleSetType.MIPI_DPHY,
        "spi_flash_rule": RuleSetType.SPI_FLASH,
        "hyper_ram_rule": RuleSetType.HYPER_RAM,
        "rclkmux_rule": RuleSetType.RCLKMUX,
        "mipi_dphy_rule": RuleSetType.MIPI_HARD_DPHY,
        "ext_flash_rule": RuleSetType.EXT_FLASH_CTRL,
        "pll_ssc_rule": RuleSetType.PLL_SSC,
        "qcrv32_rule": RuleSetType.SOC,
        "pcie_rule": RuleSetType.QUAD_PCIE,
        "pma_clkmux_rule": RuleSetType.PMA_CLKMUX,
        "10gbase_kr_rule": RuleSetType.LANE_10G,
        "common_quad_lane_rule": RuleSetType.COMMON_QUAD_LANE,
        "sgmii_rule": RuleSetType.LANE_1G,
        "pma_direct_rule": RuleSetType.RAW_SERDES
    }  # string to type map

    blk_type_map = {
        PeriDesign.BlockType.gpio: RuleSetType.GPIO,
        PeriDesign.BlockType.adv_l2_gpio: RuleSetType.GPIO,
        PeriDesign.BlockType.pll: RuleSetType.PLL,
        PeriDesign.BlockType.adv_pll: RuleSetType.PLL,
        PeriDesign.BlockType.osc: RuleSetType.OSC,
        PeriDesign.BlockType.adv_osc: RuleSetType.OSC,
        PeriDesign.BlockType.lvds: RuleSetType.LVDS,
        PeriDesign.BlockType.mipi: RuleSetType.MIPI,
        PeriDesign.BlockType.jtag: RuleSetType.JTAG,
        PeriDesign.BlockType.ddr: RuleSetType.DDR,
        PeriDesign.BlockType.h264: RuleSetType.H264,
        PeriDesign.BlockType.comp_pll: RuleSetType.PLL,
        PeriDesign.BlockType.comp_gpio: RuleSetType.GPIO,
        PeriDesign.BlockType.adv_lvds: RuleSetType.LVDS,
        PeriDesign.BlockType.mipi_dphy: RuleSetType.MIPI_DPHY,
        PeriDesign.BlockType.spi_flash: RuleSetType.SPI_FLASH,
        PeriDesign.BlockType.hyper_ram: RuleSetType.HYPER_RAM,
        PeriDesign.BlockType.hposc: RuleSetType.OSC,
        PeriDesign.BlockType.efx_pll_v3_comp: RuleSetType.PLL,
        PeriDesign.BlockType.adv_ddr: RuleSetType.DDR,
        PeriDesign.BlockType.mipi_hard_dphy: RuleSetType.MIPI_HARD_DPHY,
        PeriDesign.BlockType.pll_ssc: RuleSetType.PLL_SSC,
        PeriDesign.BlockType.soc: RuleSetType.SOC,
        PeriDesign.BlockType.quad_pcie: RuleSetType.QUAD_PCIE,
        PeriDesign.BlockType.efx_fpll_v1: RuleSetType.PLL,
        PeriDesign.BlockType.lane_10g: RuleSetType.LANE_10G,
        PeriDesign.BlockType.common_quad: RuleSetType.COMMON_QUAD_LANE,
        PeriDesign.BlockType.lane_1g: RuleSetType.LANE_1G,
        PeriDesign.BlockType.raw_serdes: RuleSetType.RAW_SERDES
    }  #: A map of periphery design block type to its rule set type

    def __init__(self, design=None):
        """
        Constructor
        """
        self.logger = Logger

        self.gpio_checker = None
        self.bus_checker = None
        self.lvds_gpio_checker = None
        self.jtag_checker = None
        self.iobank_checker = None
        self.clock_checker = None
        self.clkmux_checker = None
        self.ctrl_checker = None
        self.pll_checker = None
        self.h264_checker = None
        self.mipi_checker = None
        self.lvds_checker = None
        self.ddr_checker = None
        self.osc_checker = None
        self.seu_checker = None
        self.mipi_dphy_checker = None
        self.spi_flash_checker = None
        self.hyper_ram_checker = None
        self.rclkmux_checker = None
        self.mipi_hard_dphy_checker = None
        self.ext_flash_checker = None
        self.pll_ssc_checker = None
        self.soc_checker = None
        self.quad_pcie_checker = None
        self.pma_clkmux_chcker = None
        self.lane_10g_checker = None
        self.common_quad_lane_checker = None
        self.lane_1g_checker = None
        self.lane_raw_serdes_checker = None

        self.run_mode = self.RunModeType.STD
        #: A map of rule set type and its run status, True pass, else something fails
        self.run_status = {}
        self.exc_rules = {}  #: A map of rule set type and its excluded rules

        self.design: Optional[PeriDesign] = None
        self.setup_design(design)

        self.runner_map = {
            self.RuleSetType.GPIO: self.check_gpio,
            self.RuleSetType.PLL: self.check_pll,
            self.RuleSetType.OSC: self.check_osc,
            self.RuleSetType.IO_BANK: self.check_iobank,
            self.RuleSetType.CLOCK: self.check_clock,
            self.RuleSetType.LVDS: self.check_lvds,
            self.RuleSetType.MIPI: self.check_mipi,
            self.RuleSetType.CTRL: self.check_control,
            self.RuleSetType.CLKMUX: self.check_clockmux,
            self.RuleSetType.JTAG: self.check_jtag,
            self.RuleSetType.DDR: self.check_ddr,
            self.RuleSetType.H264: self.check_h264,
            self.RuleSetType.SEU: self.check_seu,
            self.RuleSetType.MIPI_DPHY: self.check_mipi_dphy,
            self.RuleSetType.SPI_FLASH: self.check_spi_flash,
            self.RuleSetType.HYPER_RAM: self.check_hyper_ram,
            self.RuleSetType.RCLKMUX: self.check_rclockmux,
            self.RuleSetType.MIPI_HARD_DPHY: self.check_mipi_hard_dphy,
            self.RuleSetType.EXT_FLASH_CTRL: self.check_ext_flash_ctrl,
            self.RuleSetType.PLL_SSC: self.check_pll_ssc,
            self.RuleSetType.SOC: self.check_soc,
            self.RuleSetType.QUAD_PCIE: self.check_quad_pcie,
            self.RuleSetType.PMA_CLKMUX: self.check_pma_clock_mux,
            self.RuleSetType.LANE_10G: self.check_lane_10g,
            self.RuleSetType.COMMON_QUAD_LANE: self.check_common_quad_lane,
            self.RuleSetType.LANE_1G: self.check_lane_1g,
            self.RuleSetType.RAW_SERDES: self.check_lane_raw_serdes,
        }  #: A map of rule set type and its check function. It does not store checker since a block can have many checker.

    def setup_design(self, design):
        """
        Set a new design db to analyze

        :param design: design db object
        """
        self.design = design

        self._reset()

        # Force checker rebuild for those that have different checkers for
        # different device
        self.pll_checker = None

    def _reset(self):
        """
        Reset run setting/result to default
        """

        self.run_mode = self.RunModeType.STD
        self.exc_rules.clear()

        rule_blk_type_list = [member for member in self.RuleSetType]
        for blk_type in rule_blk_type_list:
            self.run_status.setdefault(blk_type, True)

        if self.design is not None:
            if self.design.issue_reg is None:
                self.design.issue_reg = DesignIssueRegistry.build()
            else:
                self.design.issue_reg.clear()

    def run(self, run_mode):
        """
        Run design check

        :param run_mode:
        :return: True, all checks passed, else one or more failure detected
        """

        if self.design is None:
            return False

        self._reset()

        if run_mode is not None:
            self.run_mode = run_mode

        is_pass = True
        self.build_rule_filter()

        check_pipeline = self.build_run_pipeline()
        for check_func in check_pipeline:
            if is_pass:
                is_pass = check_func()
            else:  # Retain fail status
                check_func()

        return is_pass

    def build_run_pipeline(self):
        """
        Build run pipeline. Each check function will create appropriate checker
        based on run mode.

        :return: A list of check functions
        """

        pipeline = []
        assert self.design is not None

        # Design block check
        design_blk_type_list = self.design.get_supported_block()
        for design_blk_type in design_blk_type_list:
            rule_type = self.blk_type_map.get(design_blk_type, None)
            if rule_type is not None:
                block_runner = self.runner_map.get(rule_type, None)
                if block_runner is not None:
                    pipeline.append(block_runner)

        # Device setting check
        pipeline.append(self.check_device_setting)

        # Global check
        pipeline.append(self.check_clock)

        return pipeline

    def build_rule_filter(self):
        """
        Build rule set to exclude from check run

        It reads from pt_rule.ini stored at the same place as design file
        """
        assert self.design is not None

        # Check for any pt_rule.ini for parsing excluded rule list.
        common_exclude_rules = None
        if self.design.location != "":
            rule_file = self.design.location + "/pt_rule.ini"
            if os.path.exists(rule_file):
                common_exclude_rules = self.parse_rule_exclude(rule_file)

        if common_exclude_rules:
            self.exc_rules = self._categorize_exclude_rules(
                common_exclude_rules)

    def is_check_pass(self, rule_set_type):
        """
        Check if target rule set pass its check

        :param rule_set_type: Target rule set
        :return: True, pass, else False
        """
        return self.run_status.get(rule_set_type, False)

    def _categorize_exclude_rules(self, exclude_rules):
        """
        The excluded_rules is a list of collective rule name
        that is not categorized per block type. When we run the
        block specific check, it needs to be separated since
        the stats of each block simply do a total rule minus exclude
        rule count.

        :param exclude_rules: List of rule names to exclude from check
        :return: a dictionary of type to the list of exclude rules.

        ..note::
          If rule name of each block is no longer standardized by
          a prefix of <block>_rule, then this method of categorizing
          rule per block needs to be revisited.
        """

        # Map of block type to the list of rules to exclude
        block_exclude_rules = {}

        for rule_name in exclude_rules:
            # Get the prefix <block>_rule
            exc_rule = rule_name
            end = exc_rule.find('rule')
            rule_prefix = exc_rule[0:end] + "rule"

            if rule_prefix in self.rule_str2type_map:
                # if no prefix, then we ignore the exclude
                rule_type = self.rule_str2type_map[rule_prefix]

                rule_list = []
                if rule_type in block_exclude_rules:
                    rule_list = block_exclude_rules[rule_type]

                rule_list.append(rule_name)
                block_exclude_rules[rule_type] = rule_list

        return block_exclude_rules

    def parse_rule_exclude(self, rule_file):
        """
        Open up the file which contains a list of excluded
        rules to check and save the rule names into the returned
        list.

        :param rule_file: The path to the excluded rule file (pt_rule.ini)
        :return: list of excluded rule names
        """
        exclude_rules = []

        try:
            with open(rule_file, encoding='UTF-8') as file:
                # Read line by line instead of all at once
                for line in file:
                    if line != "":
                        exclude_rules.append(line.rstrip())

        except IOError:
            self.logger.warning("Cannot open rule file: {}".format(rule_file))

        # Print out the list of rules to exclude to logger
        if exclude_rules:
            rule_names = ""
            for rname in exclude_rules:
                if rule_names != "":
                    rule_names = rule_names + "," + rname
                else:
                    rule_names = rname

            self.logger.info(
                "Found {} rules to exclude from check".format(len(exclude_rules)))
            self.logger.info("Excluded rule names: {}".format(rule_names))

        return exclude_rules

    def get_exclude_rule(self, rule_set_type):
        """
        Given a rule set type, get its excluded rules

        :param rule_set_type:
        :return: A list of rule or empty list if no exclusion
        """
        if rule_set_type in self.exc_rules:
            return self.exc_rules[rule_set_type]
        else:
            return []

    def check_gpio(self):
        """
        Check I/O design

        :return: True pass, else False
        """
        assert self.design is not None

        if self.design.gpio_reg is None:
            return True

        is_lvds_pass = True
        is_bus_pass = True

        if self.gpio_checker is None:
            if self.design.is_block_supported(PeriDesign.BlockType.comp_gpio):
                if self.run_mode == self.RunModeType.STD:
                    self.gpio_checker = GPIOCheckerComp(self.design)
                else:
                    self.gpio_checker = GPIOCompCheckerComp(self.design)

            else:
                if self.run_mode == self.RunModeType.STD:
                    self.gpio_checker = GPIOChecker(self.design)
                else:
                    self.gpio_checker = GPIOCompChecker(self.design)

        exclude_rules = self.get_exclude_rule(self.RuleSetType.GPIO)
        is_gpio_pass = self.gpio_checker.run_all_check(exclude_rules)

        # If the device has LVDS GPIO, we check those too
        if self.design.is_block_supported(PeriDesign.BlockType.adv_lvds):

            if self.lvds_gpio_checker is None:
                if self.run_mode == self.RunModeType.STD:
                    self.lvds_gpio_checker = HSIOGPIOChecker(self.design)
                else:
                    self.lvds_gpio_checker = HSIOGPIOCompChecker(self.design)

            is_lvds_pass = self.lvds_gpio_checker.run_all_check(exclude_rules)

        elif self.design.is_block_supported(PeriDesign.BlockType.lvds):

            if self.lvds_gpio_checker is None:
                if self.run_mode == self.RunModeType.STD:
                    self.lvds_gpio_checker = LVDSGPIOChecker(self.design)
                else:
                    self.lvds_gpio_checker = LVDSGPIOCompChecker(self.design)

            is_lvds_pass = self.lvds_gpio_checker.run_all_check(exclude_rules)

        # Run checks on bus
        if self.design.gpio_reg.get_bus_count() > 0:
            if self.bus_checker is None:
                if self.design.is_block_supported(PeriDesign.BlockType.comp_gpio):
                    self.bus_checker = BusCheckerComp(self.design)
                else:
                    self.bus_checker = BusChecker(self.design)

            is_bus_pass = self.bus_checker.run_all_check(exclude_rules)

        is_pass = is_gpio_pass and is_lvds_pass and is_bus_pass
        self.run_status[self.RuleSetType.GPIO] = is_pass

        return is_pass

    def check_pll(self):
        """
        Check pll block design

        :return: True pass, else False
        """
        assert self.design is not None

        if self.design.pll_reg is None:
            return True

        if self.design.is_block_supported(PeriDesign.BlockType.adv_pll):
            if self.pll_checker is None:
                self.pll_checker = PLLCheckerAdv(self.design)

        elif self.design.is_block_supported(PeriDesign.BlockType.comp_pll):
            if self.pll_checker is None:
                self.pll_checker = PLLCheckerComp(self.design)

        elif self.design.is_block_supported(PeriDesign.BlockType.efx_pll_v3_comp):
            if self.pll_checker is None:
                self.pll_checker = PLLCheckerV3Complex(self.design)

        elif self.design.is_block_supported(PeriDesign.BlockType.efx_fpll_v1):
            assert self.pll_checker is None
            self.pll_checker = EfxFpllV1Checker(self.design)

        else:
            if self.pll_checker is None:
                self.pll_checker = PLLChecker(self.design)

        exclude_rules = self.get_exclude_rule(self.RuleSetType.PLL)
        is_pass = self.pll_checker.run_all_check(exclude_rules)
        self.run_status[self.RuleSetType.PLL] = is_pass

        return is_pass

    def check_jtag(self):
        """
        Check jtag block design

        :return: True pass, else False
        """
        assert self.design is not None

        if self.design.jtag_reg is None:
            return True

        if self.jtag_checker is None:
            self.jtag_checker = JTAGChecker(self.design)

        exclude_rules = self.get_exclude_rule(self.RuleSetType.JTAG)
        is_pass = self.jtag_checker.run_all_check(exclude_rules)
        self.run_status[self.RuleSetType.JTAG] = is_pass

        return is_pass

    def check_osc(self):
        """
        Check oscillator block design

        :return: True pass, else False
        """
        assert self.design is not None

        if self.design.osc_reg is None:
            return True

        if self.design.is_block_supported(PeriDesign.BlockType.adv_osc) or \
                self.design.is_block_supported(PeriDesign.BlockType.hposc):
            if self.osc_checker is None:
                # Ti375 will need additional rule check
                if self.design.is_serdes_reg_supported():
                    self.osc_checker = OSCCheckerV3(self.design)
                else:
                    self.osc_checker = OSCCheckerAdv(self.design)

        else:

            if self.osc_checker is None:
                self.osc_checker = OSCChecker(self.design)

        exclude_rules = self.get_exclude_rule(self.RuleSetType.OSC)
        is_pass = self.osc_checker.run_all_check(exclude_rules)
        self.run_status[self.RuleSetType.OSC] = is_pass

        return is_pass

    def check_lvds(self):
        """
        Check lvds block design, both tx and rx

        :return: True pass, else False
        """
        assert self.design is not None

        if self.design.lvds_reg is None:
            return True

        if self.design.is_block_supported(PeriDesign.BlockType.adv_lvds):
            if self.lvds_checker is None:
                self.lvds_checker = LVDSCheckerAdv(self.design)
        else:
            if self.lvds_checker is None:
                self.lvds_checker = LVDSChecker(self.design)

        exclude_rules = self.get_exclude_rule(self.RuleSetType.LVDS)
        is_pass = self.lvds_checker.run_all_check(exclude_rules)
        self.run_status[self.RuleSetType.LVDS] = is_pass

        return is_pass

    def check_mipi_dphy(self):
        """
        Check mipi dphy block design, both tx and rx

        :return: True pass, else False
        """
        assert self.design is not None

        if self.design.mipi_dphy_reg is None:
            return True

        if self.mipi_dphy_checker is None:
            self.mipi_dphy_checker = MIPIDPhyChecker(self.design)

        exclude_rules = self.get_exclude_rule(self.RuleSetType.MIPI_DPHY)
        is_pass = self.mipi_dphy_checker.run_all_check(exclude_rules)
        self.run_status[self.RuleSetType.MIPI_DPHY] = is_pass

        return is_pass

    def check_mipi(self):
        """
        Check mipi block design, both tx and rx

        :return: True pass, else False
        """
        assert self.design is not None

        if self.design.mipi_reg is None:
            return True

        if self.mipi_checker is None:
            self.mipi_checker = MIPIChecker(self.design)

        exclude_rules = self.get_exclude_rule(self.RuleSetType.MIPI)
        is_pass = self.mipi_checker.run_all_check(exclude_rules)
        self.run_status[self.RuleSetType.MIPI] = is_pass

        return is_pass

    def check_mipi_hard_dphy(self):
        """
        Check mipi block design, both tx and rx

        :return: True pass, else False
        """
        assert self.design is not None

        if self.design.mipi_hard_dphy_reg is None:
            return True

        if self.mipi_hard_dphy_checker is None:
            self.mipi_hard_dphy_checker = MIPICheckerAdv(self.design)

        exclude_rules = self.get_exclude_rule(self.RuleSetType.MIPI_HARD_DPHY)
        is_pass = self.mipi_hard_dphy_checker.run_all_check(exclude_rules)
        self.run_status[self.RuleSetType.MIPI_HARD_DPHY] = is_pass

        return is_pass

    def check_pll_ssc(self):
        assert self.design is not None
        if self.design.pll_ssc_reg is None:
            return True
        if self.pll_ssc_checker is None:
            self.pll_ssc_checker = PLLSSCChecker(self.design)

        exclude_rules = self.get_exclude_rule(self.RuleSetType.PLL_SSC)
        is_pass = self.pll_ssc_checker.run_all_check(exclude_rules)
        self.run_status[self.RuleSetType.PLL_SSC] = is_pass

        return is_pass

    def check_h264(self):
        """
        Check H.264 design

        :return: True pass, else False
        """
        assert self.design is not None

        if self.design.h264_reg is None:
            return True

        if self.h264_checker is None:
            self.h264_checker = H264Checker(self.design)

        exclude_rules = self.get_exclude_rule(self.RuleSetType.H264)
        is_pass = self.h264_checker.run_all_check(exclude_rules)
        self.run_status[self.RuleSetType.H264] = is_pass

        return is_pass

    def check_ddr(self):
        """
        Check ddr block design

        :return: True pass, else False
        """
        assert self.design is not None

        if self.design.ddr_reg is None:
            return True

        if self.design.is_block_supported(PeriDesign.BlockType.adv_ddr):
            if self.ddr_checker is None:
                self.ddr_checker = DDRCheckerAdv(self.design)
        else:
            if self.ddr_checker is None:
                self.ddr_checker = DDRChecker(self.design)

        exclude_rules = self.get_exclude_rule(self.RuleSetType.DDR)
        is_pass = self.ddr_checker.run_all_check(exclude_rules)
        self.run_status[self.RuleSetType.DDR] = is_pass
        return is_pass

    def check_device_setting(self):
        """
        Check full device setting

        :return: True pass, else False
        """

        is_pass_io = self.check_iobank()
        is_pass_ctrl = self.check_control()
        # PMA Clockmux need to be checked before global clock mux
        # Because it forwards clock to the global clkmux
        is_pass_pma_clkmux = self.check_pma_clock_mux()
        is_pass_clkmux = self.check_clockmux()
        is_pass_seu = self.check_seu()
        is_pass_ext_flash = self.check_ext_flash_ctrl()

        # We don't stop if fail rclockmux since rclkmux routing failure
        # goes into the summary report just like the Trion clkmux routing
        self.check_rclockmux()

        return is_pass_io and is_pass_ctrl and is_pass_clkmux and is_pass_seu and \
            is_pass_ext_flash and is_pass_pma_clkmux

    def check_iobank(self):
        """
        Check iobank design

        :return: True pass, else False
        """
        assert self.design is not None

        if self.design.device_setting is None or self.design.device_setting.iobank_reg is None:
            return True

        if self.iobank_checker is None:
            from tx60_device.iobank.iobank_design_adv import IOBankRegistryAdvance
            if isinstance(self.design.device_setting.iobank_reg, IOBankRegistryAdvance):
                if self.iobank_checker is None:
                    self.iobank_checker = IOBankCheckerAdv(self.design)
            else:
                if self.iobank_checker is None:
                    self.iobank_checker = IOBankChecker(self.design)

        exclude_rules = self.get_exclude_rule(self.RuleSetType.IO_BANK)
        is_pass = self.iobank_checker.run_all_check(exclude_rules)
        self.run_status[self.RuleSetType.IO_BANK] = is_pass
        return is_pass

    def check_control(self):
        """
        Check control block design, aka configuration

        :return: True pass, else False
        """
        assert self.design is not None

        if self.design.device_setting is None or self.design.device_setting.ctrl_reg is None:
            return True

        if self.ctrl_checker is None:
            self.ctrl_checker = ControlChecker(self.design)

        exclude_rules = self.get_exclude_rule(self.RuleSetType.CTRL)
        is_pass = self.ctrl_checker.run_all_check(exclude_rules)
        self.run_status[self.RuleSetType.CTRL] = is_pass

        return is_pass

    def check_seu(self):
        """
        Check seu block design, aka error detection

        :return: True pass, else False
        """
        assert self.design is not None

        if self.design.device_setting is None or self.design.device_setting.seu_reg is None:
            return True

        if self.seu_checker is None:
            self.seu_checker = SEUChecker(self.design)

        exclude_rules = self.get_exclude_rule(self.RuleSetType.SEU)
        is_pass = self.seu_checker.run_all_check(exclude_rules)
        self.run_status[self.RuleSetType.SEU] = is_pass

        return is_pass

    def check_rclockmux(self):
        """
        Route and check regional clock mux design

        :return: True pass, else False
        """
        assert self.design is not None

        if self.design.device_setting is None or self.design.device_setting.rclkmux_reg is None:
            return True

        # Run the check on clock mux routing after running routing
        # However, if it fails routing, we still want to generate
        # the summary report that prints out the clkmux routing info
        self.design.device_setting.rclkmux_reg.route_all(
            self.design, self.design.device_db)

        if self.rclkmux_checker is None:
            self.rclkmux_checker = RClockMuxChecker(self.design)

        exclude_rules = self.get_exclude_rule(self.RuleSetType.RCLKMUX)
        is_pass = self.rclkmux_checker.run_all_check(exclude_rules)
        self.run_status[self.RuleSetType.RCLKMUX] = is_pass

        return True

    def check_clockmux(self):
        """
        Route and check clock mux design

        :return: True pass, else False
        """
        assert self.design is not None

        if self.design.device_setting is None or self.design.device_setting.clkmux_reg is None:
            return True

        assert self.design.device_db is not None

        exclude_rules = self.get_exclude_rule(self.RuleSetType.CLKMUX)

        # Run the check on clock mux routing after running routing
        # However, if it fails routing, we still want to generate
        # the summary report that prints out the clkmux routing info

        clkmux_reg = self.design.device_setting.clkmux_reg
        if hasattr(clkmux_reg, 'skip_pll_same_side_restriction'):
            if 'clkmux_rule_pll_serial_parallel_clocks' in exclude_rules:
                clkmux_reg.skip_pll_same_side_restriction = True
            else:
                clkmux_reg.skip_pll_same_side_restriction = False

        self.design.device_setting.clkmux_reg.route_all(
            self.design, self.design.device_db)

        
        stop_if_fail = False
        if self.design.device_db.clkmux_type == device_dbi.DeviceDBService.BlockType.CLKMUX_COMPLEX or \
                self.design.device_db.clkmux_type == device_dbi.DeviceDBService.BlockType.CLKMUX_V4:
            if self.clkmux_checker is None:
                self.clkmux_checker = ClockMuxCheckerComp(self.design)
                stop_if_fail = True
        elif self.design.device_db.clkmux_type == device_dbi.DeviceDBService.BlockType.CLKMUX_ADV:
            if self.clkmux_checker is None:
                self.clkmux_checker = ClockMuxCheckerAdv(self.design)
                stop_if_fail = True
        else:
            if self.clkmux_checker is None:
                self.clkmux_checker = ClockMuxChecker(self.design)

        is_pass = self.clkmux_checker.run_all_check(exclude_rules)
        self.run_status[self.RuleSetType.CLKMUX] = is_pass

        # We let check pass through if it fails clkmux because
        # later we want to generate the summary which will gate
        # the other constraints generation.
        # TODO: For adv clockmux, we only want it to pass through if it failed
        # the router check only
        if stop_if_fail and not is_pass:
            return False

        return True

    def check_clock(self):
        """"
        Check clock design

        :return: True pass, else False
        """

        if self.design is None:
            return True

        # Some unit test using design don't have device_db. However, at run time
        # device_db should always exists when successful loading/creating
        # design
        if self.design.device_db is not None and self.is_titanium_clock_mux():
            if self.clock_checker is None:
                self.clock_checker = ClockCheckerAdv(self.design)
        else:
            if self.clock_checker is None:
                self.clock_checker = ClockChecker(self.design)

        exclude_rules = self.get_exclude_rule(self.RuleSetType.CLOCK)
        is_pass = self.clock_checker.run_all_check(exclude_rules)
        self.run_status[self.RuleSetType.CLOCK] = is_pass
        return is_pass

    def is_titanium_clock_mux(self):
        assert self.design is not None and self.design.device_db is not None
        if self.design.device_db.clkmux_type == device_dbi.DeviceDBService.BlockType.CLKMUX_ADV or\
            self.design.device_db.clkmux_type == device_dbi.DeviceDBService.BlockType.CLKMUX_COMPLEX or \
                self.design.device_db.clkmux_type == device_dbi.DeviceDBService.BlockType.CLKMUX_V4:
            return True

        return False

    def check_spi_flash(self):
        """
        Check spi flash design

        :return: True pass, else False
        """
        assert self.design is not None

        if self.design.spi_flash_reg is None:
            return True

        if self.spi_flash_checker is None:
            self.spi_flash_checker = SPIFlashChecker(self.design)

        exclude_rules = self.get_exclude_rule(self.RuleSetType.SPI_FLASH)
        is_pass = self.spi_flash_checker.run_all_check(exclude_rules)
        self.run_status[self.RuleSetType.SPI_FLASH] = is_pass

        return is_pass

    def check_hyper_ram(self):
        """
        Check HyperRAM design

        :return: True pass, else False
        """
        assert self.design is not None

        if self.design.hyper_ram_reg is None:
            return True

        if self.hyper_ram_checker is None:
            self.hyper_ram_checker = HyperRAMChecker(self.design)

        exclude_rules = self.get_exclude_rule(self.RuleSetType.HYPER_RAM)
        is_pass = self.hyper_ram_checker.run_all_check(exclude_rules)
        self.run_status[self.RuleSetType.HYPER_RAM] = is_pass

        # Post Check action:
        # PT-2015: assign drive strength based on freq
        if is_pass:
            # Suppose clk / clk90 / clkcal all have the same frequency (guaranteed by rule)
            # Just use the frequency of calibration clk
            def determine_drive_strength(hpram, pll_reg) -> int:
                """
                Determine drive strength needed based on the clk freq
                """
                result = 4  # Default 4mA
                clk_name = hpram.gen_pin.get_pin_name_by_type('CLKCAL')
                if clk_name == "":
                    return result
                pll_data = pll_reg.find_output_clock(clk_name)
                if pll_data is None:
                    return result
                pll_inst, pll_clk_idx = pll_data
                if pll_inst is None or pll_clk_idx == -1:
                    return result
                pll_clk = pll_inst.get_output_clock_by_number(pll_clk_idx)
                if pll_clk is None:
                    return result
                freq = pll_clk.out_clock_freq
                if freq is None:
                    return result

                if freq > 200:
                    result = 8
                return result

            pll_reg = self.design.pll_reg
            for hpram in self.design.hyper_ram_reg.get_all_inst():
                drive_strength = determine_drive_strength(hpram, pll_reg)
                hpram.param_group.set_param_value(
                    HyperRAMParamId.DRIVE_STRENGTH.value, drive_strength)

        return is_pass

    def check_ext_flash_ctrl(self):
        """
        Check external flash controller setting

        :return: True pass, else False
        """
        assert self.design is not None

        if self.design.device_setting is None or self.design.device_setting.ext_flash_reg is None:
            return True

        if self.ext_flash_checker is None:
            self.ext_flash_checker = ExtFlashControllerChecker(self.design)

        exclude_rules = self.get_exclude_rule(self.RuleSetType.EXT_FLASH_CTRL)
        is_pass = self.ext_flash_checker.run_all_check(exclude_rules)
        self.run_status[self.RuleSetType.EXT_FLASH_CTRL] = is_pass
        return is_pass

    def check_pma_clock_mux(self) -> bool:
        assert self.design is not None
        if self.design.device_setting is None or self.design.device_setting.pma_clkmux_reg is None:
            return True

        # Run the check on clock mux routing after running routing
        # However, if it fails routing, we still want to generate
        # the summary report that prints out the clkmux routing info
        self.design.device_setting.pma_clkmux_reg.route_all(
            self.design, self.design.device_db)

        if self.pma_clkmux_chcker is None:
            self.pma_clkmux_chcker = PMAClockMuxChecker(self.design)

        exclude_rules = self.get_exclude_rule(self.RuleSetType.PMA_CLKMUX)
        is_pass = self.pma_clkmux_chcker.run_all_check(exclude_rules)
        self.run_status[self.RuleSetType.PMA_CLKMUX] = is_pass

        return True

    def check_soc(self) -> bool:
        assert self.design is not None
        if self.design.soc_reg is None:
            return True

        if self.soc_checker is None:
            self.soc_checker = SOCChecker(self.design)

        exclude_rules = self.get_exclude_rule(self.RuleSetType.SOC)
        is_pass = self.soc_checker.run_all_check(exclude_rules)
        self.run_status[self.RuleSetType.SOC] = is_pass

        return is_pass

    def check_quad_pcie(self):
        assert self.design is not None
        if self.design.quad_pcie_reg is None:
            return True
        if self.quad_pcie_checker is None:
            self.quad_pcie_checker = QuadPCIEChecker(self.design)

        exclude_rules = self.get_exclude_rule(self.RuleSetType.QUAD_PCIE)
        is_pass = self.quad_pcie_checker.run_all_check(exclude_rules)
        self.run_status[self.RuleSetType.QUAD_PCIE] = is_pass

        return is_pass

    def check_lane_10g(self):
        assert self.design is not None
        if self.design.lane_10g_reg is None:
            return True
        if self.lane_10g_checker is None:
            self.lane_10g_checker = Lane10GChecker(self.design)

        exclude_rules = self.get_exclude_rule(self.RuleSetType.LANE_10G)
        is_pass = self.lane_10g_checker.run_all_check(exclude_rules)
        self.run_status[self.RuleSetType.LANE_10G] = is_pass

        return is_pass

    def check_common_quad_lane(self):
        assert self.design is not None
        if self.design.common_quad_lane_reg is None:
            return True
        if self.common_quad_lane_checker is None:
            self.common_quad_lane_checker = QuadLaneCommonChecker(self.design)

        exclude_rules = self.get_exclude_rule(self.RuleSetType.COMMON_QUAD_LANE)
        is_pass = self.common_quad_lane_checker.run_all_check(exclude_rules)
        self.run_status[self.RuleSetType.COMMON_QUAD_LANE] = is_pass

        return is_pass

    def check_lane_1g(self):
        assert self.design is not None
        if self.design.lane_1g_reg is None:
            return True
        if self.lane_1g_checker is None:
            self.lane_1g_checker = Lane1GChecker(self.design)

        exclude_rules = self.get_exclude_rule(self.RuleSetType.LANE_1G)
        is_pass = self.lane_1g_checker.run_all_check(exclude_rules)
        self.run_status[self.RuleSetType.LANE_1G] = is_pass

        return is_pass

    def check_lane_raw_serdes(self):
        assert self.design is not None
        if self.design.raw_serdes_reg is None:
            return True
        if self.lane_raw_serdes_checker is None:
            self.lane_raw_serdes_checker = RawSerdesChecker(self.design)

        exclude_rules = self.get_exclude_rule(self.RuleSetType.RAW_SERDES)
        is_pass = self.lane_raw_serdes_checker.run_all_check(exclude_rules)
        self.run_status[self.RuleSetType.RAW_SERDES] = is_pass

        return is_pass

if __name__ == "__main__":
    pass
