'''
Copyright (C) 2017-2022 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Feb 7, 2022

@author: Shirley Chan
'''

from __future__ import annotations
import re
import gc
from pathlib import Path
from enum import Enum, auto, unique
from dataclasses import dataclass
from typing import Dict, List, Optional, Set, Tuple, Union, Literal, TYPE_CHECKING

from util.singleton_logger import Logger
import util.efxproj_setting as efxproj_set
import util.excp as app_excp
import util.gen_util as pt_util
from util.app_setting import AppSetting

from design.db import PeriDesign
from design.db_item import PeriDesignRegistry, GenericParamService

from device.io_info import IOPadInfo
from device.block_instance import PeripheryBlockInstance, SecondaryConn
from device.db_interface import DeviceDBService
from device.mipi_group import MIPIDPHYRxGroup, MIPIDPHYRxGroupRegistry

from common_device.gpio.gpio_design import GPIO, GPIOUnusedConfig
from common_device.gpio.gpio_design_service import GPIODesignService
from common_device.hsio.hsio_device_service import HSIOService
from common_device.pll.pll_design import PLL, PLLRegistry
from common_device.mipi.mipi_design import MIPI, MIPIRegistry, MIPITx
import common_device.lvds.lvds_utility as lvds_util

from an20_device.pll.pll_design_adv import PLLAdvance, PLLRegistryAdvance

from tx60_device.hio_res_service import HIOResService
from tx60_device.gpio.gpio_design_comp import GPIOComplex
from tx60_device.gpio.gpio_design_service_comp import GPIODesignServiceComplex
from tx60_device.pll.pll_design_comp import PLLComplex, PLLRegistryComplex

from tx180_device.mipi_dphy.design import MIPIHardDPHYRegistry, MIPIHardDPHYTx, MIPIHardDPHYRx
from tx180_device.mipi_dphy.design_param_info import MIPIHardDPHYTxDesignParamInfo
from tx180_device.ddr.ddr_design_adv import DDRAdvance, DDRPinSwap
from tx180_device.pll_ssc.design import PLLSSC, PLLSSCRegistry
from tx180_device.pll_ssc.design_param_info import PLLSSCDesignParamInfo

from common_device.quad.res_service import QuadType, QuadResService
from tx375_device.common_quad.quad_param_info import build_param_info as build_quad_param_info
from tx375_device.common_quad.quad_prop_id import CommonQuadConfigParamInfo as QuadConfigParamInfo

from tx375_device.quad_pcie.design import QuadPCIE
from tx375_device.quad_pcie.design_param_info import QuadPCIEDesignParamInfo as PCIEParamInfo
from tx375_device.quad_pcie.quad_pcie_prop_id import QuadPCIEConfigParamInfo as PCIEConfigParamInfo
from tx375_device.quad_pcie.quad_pcie_param_info import build_param_info as build_pcie_param_info

if TYPE_CHECKING:
    from device.db import PeripheryDevice
    from device.io_info import DeviceIO, IOBankInfo
    from device.package import DevicePackage, PadPackagePinMap

    from common_device.iobank.iobank_design import IOBank
    from common_device.mipi.mipi_tx_device_service import MIPITxService
    from common_device.mipi.mipi_rx_device_service import MIPIRxService
    from common_device.ddr.ddr_device_service import DDRService
    from common_device.ddr.ddr_design import DDR
    from common_device.spi_flash.spi_flash_design import SPIFlash

    from tx180_device.mipi_dphy.mipi_tx_device_service_adv import MIPITxServiceAdvance
    from tx180_device.mipi_dphy.mipi_rx_device_service_adv import MIPIRxServiceAdvance

    from tx375_device.common_quad.design import QuadLaneCommon
    from tx375_device.lane10g.device_service import QuadLane10GDeviceService
    from tx375_device.quad_pcie.device_service import QuadPCIEDeviceService


DEDICATED_PIN_TYPE = Literal[
    "Dedicated Ground",
    "Dedicated Power",
    "Dedicated Configuration",
    "Dedicated JTAG",
    "Dedicated SPI Flash",
    "Dedicated Power or Ground"
]

TITANIUM_PIN_TYPE = Literal[
    DEDICATED_PIN_TYPE,
    "HVIO",
    "HVIO or Multi-Function",
    "HSIO",
    "HSIO or Multi-Function",
    "HSIO or Configuration",
    "MIPI",
    "DDR",
    "Reference Resistor Pin",
    "Transceiver",
    "TEST",
    "NC"
]
TRION_PIN_TYPE = Literal[
    DEDICATED_PIN_TYPE,
    "GPIO",
    "Configuration or GPIO",
    "Multi-Function or GPIO",
    "Multi-Function, Configuration, or GPIO",
    "Multi-Function, LVDS, or GPIO",
    "Configuration, LVDS, or GPIO",
    "MIPI",
    "LVDS or GPIO",
    "LVDS Reference Resistor Pin",
    "DDR",
    "TEST",
    "NC"
]

ALL_PIN_TYPE = Literal[
    TITANIUM_PIN_TYPE,
    TRION_PIN_TYPE
]

class PinType(Enum):
    HSIO = "HSIO"
    HVIO = "HVIO"
    GPIO = "GPIO"
    LVDS = "LVDS"
    MULTI_FUNCTION = "Multi-Function"
    CONFIG =  "Configuration"


@unique
class PinInformation(Enum):
    PIN_NUMBER = "Pin Number"
    PIN_NAME = "Pin Name"
    PIN_TYPE = "Pin Type"
    SIGNAL_NAME = "Signal Name"
    DIRECTION = "Direction"
    IO_BANK = "I/O Bank"
    IO_BANK_VOLTAGE = "I/O Bank Voltage"
    IO_STANDARD = "I/O Standard"
    PULL_TYPE = "Pull Type"
    BLOCK_TYPE = "Block Type"
    BLOCK_INSTANCE = "Block Instance"
    RESOURCE_NAME = "Resource Name"
    MIPI_RX_GROUP = "MIPI Rx Group"
    FUNCTION = "Function"


@unique
class PinFunction(Enum):
    VREF = SecondaryConn.SEC_TYPE_VREF
    PLL_REFCLK = SecondaryConn.SEC_TYPE_PLL_CLKIN
    PLL_EXTCLK = SecondaryConn.SEC_TYPE_PLL_EXTFB
    GLOBAL_CLK = SecondaryConn.SEC_TYPE_GCLK
    GLOBAL_CTRL = SecondaryConn.SEC_TYPE_GCTRL
    REGIONAL_CLK = SecondaryConn.SEC_TYPE_RCLK
    REGIONAL_CTRL = SecondaryConn.SEC_TYPE_RCTRL
    DDR_CLK = SecondaryConn.SEC_TYPE_DDR_CLKIN
    MIPI_CLK = SecondaryConn.SEC_TYPE_MIPI_CLKIN
    PCIE_PERSTN = SecondaryConn.SEC_TYPE_PCIE_PERSTN

    def __str__(self) -> str:
        return self.value


class PkgItemType(Enum):
    GPIO = 'gpio'
    GPIO_BUS = 'gpio_bus'
    PLL = 'pll'
    SPI_FLASH = 'spi'
    MIPI_TX = 'mipi_tx'
    MIPI_HARD_DPHY_TX = 'mipi_hard_dphy_tx'
    DDR = 'ddr'
    PLL_SSC = 'pll_ssc'
    QUAD_PCIE = 'quad_pcie'
    LANE_10G = '10g'
    LANE_1G = '1g'
    RAW_SERDES = 'raw_serdes'


class PackagePinService(object):
    '''
    Class for getting package related information
    '''
    def __init__(self, design: Optional[PeriDesign], device: Optional[PeripheryDevice] = None,
                 is_coldboot: Optional[bool] = None):
        '''
        Assume that the name of the package have the standard format
        E.g. 144-pin QFP, 324-ball FBGA, 80-ball WLCSP
        '''
        assert (design is not None and design.device_db is not None) or device is not None, \
            f"Design/ device information is required"

        self.logger = Logger
        self.design = design

        if design is not None and design.device_db is not None:
            self._device = design.device_db
        elif device is not None:
            self._device = device

        assert self._device is not None

        self._is_load_all = False # For increasing performance

        # Check if coldboot was selected
        self.is_coldboot = is_coldboot

        package = self._device.get_package() # type: Optional[DevicePackage]
        assert package is not None
        pkg_name: str = package.get_package_name()
        self.invisible_bank_names = package.get_package_invisible_bank_names()

        name_list = pkg_name.split(" ")

        self._package_type: str = ""

        if len(name_list) == 2:
            self._package_type = name_list[-1]
        else:
            # Do not support package with strange name
            self._package_type = pkg_name

        self.dbi = DeviceDBService(self._device)

        self.func2pin_name: Dict[PinFunction, List[str]] = {}
        self.pin_name2pin_type: Dict[str, str] = {}
        self.pin2inst_map: dict = {}
        self.res2pin_map: Dict[str, List[str]] = {}
        self.pll_res2res_name: Dict[str, List[str] | Dict[str|PLLAdvance.RefClockIDType, List[str]]] = {}
        self.spi2res_name: Dict[str, list] = {}
        self.iobank2pin_map: Dict[str, list] = {}
        self.ddr_pad2ori_pin_name: Dict[str, str] = {} # PT-1939 For DDR pin swap (Ti180)
        self.local_gpio_reg = None

    def build(self):
        """
        Build the package information by checking the package definition
        (e.g. FBGA, WLCSP, QFP).
        For some package, the package type cannot identify directly,
        then check one of the pin name to identify type
        """
        assert self.design is not None
        is_coldboot = self._is_coldboot_selected()

        if self._package_type == "FBGA":
            return FBGABallPackage(self.design, is_coldboot=is_coldboot)

        elif self._package_type == "WLCSP":
            return WLCSPBallPackage(self.design, is_coldboot=is_coldboot)

        elif self._package_type == "QFP":
            return QFPLeadPackage(self.design, is_coldboot=is_coldboot)

        else:
            pin_name = tuple(self.get_all_pin2pad_map().keys())[0]

            # Cannot identify FBGA/ WLCSP
            if pin_name.isnumeric():
                return QFPLeadPackage(self.design, is_coldboot=is_coldboot)

            return FBGABallPackage(self.design, is_coldboot=is_coldboot)

    def get_package_type(self) -> str:
        return self._package_type

    def get_all_pin2pad_map(self) -> Dict[str, PadPackagePinMap]:
        pkg = self._device.get_package()
        assert pkg is not None
        return pkg.get_package_pad_map()

    def get_mipi_rx_group2obj_map(self) -> Dict[str, MIPIDPHYRxGroup]:
        '''
        Get dictionary of mipi rx group name to MIPIDPHYRxGroup obj
        '''
        mipi_dphy_rx_group:Optional[MIPIDPHYRxGroupRegistry] = self._device.get_mipi_dphy_rx_group()

        if mipi_dphy_rx_group is None:
            return {}

        return mipi_dphy_rx_group.get_group2obj_map()

    def get_mipi_rx_group2pin_map(self) -> Dict[str, List[str]]:
        mipi_group2pin_name = {}

        group2obj_map = self.get_mipi_rx_group2obj_map()

        for group, group_obj in group2obj_map.items():

            if group_obj is None:
                continue

            res_name_list = group_obj.get_data_member_instances()
            clk_inst_list = group_obj.get_clock_member_instances()

            # PT-1992 Clock resource in the group isn’t bonded out
            clk_pin_list = self.translate_res_list2pin_list(clk_inst_list)

            if len(clk_pin_list) <= 0:
                pin_list = []
            else:
                res_name_list += clk_inst_list
                pin_list = self.translate_res_list2pin_list(res_name_list)
            mipi_group2pin_name[group] = pin_list

        return mipi_group2pin_name

    def get_pin2_mipi_rx_group_map(self) -> Dict[str, List[str]]:
        pin2group_name: Dict[str, List[str]]  = {}
        mipi_group2pin_name = self.get_mipi_rx_group2pin_map()

        for group_name, pin_list in mipi_group2pin_name.items():
            for pin_name in pin_list:
                group_list = pin2group_name.get(pin_name, [])

                if group_name not in group_list:
                    group_list.append(group_name)

                pin2group_name[pin_name] = group_list

        return pin2group_name

    def get_mipi_rx_group_by_pin(self, pin_name: str) -> List[str]:
        return self.get_pin2_mipi_rx_group_map().get(pin_name, [])

    def translate_res_list2pin_list(self, res_list: List[str]) -> List[str]:
        pin_list: List[str] = []
        res2pin_map:Dict[str, List[str]] = self.get_res2pin_name()

        for res_name in res_list:
            pin_name = res2pin_map.get(res_name, None)

            if pin_name is None:
                continue

            pin_list += pin_name

        pin_set = set(pin_list)
        return list(pin_set)

    def get_pin2padmap_by_pin_name(self, pin_name: str) -> Optional[PadPackagePinMap]:
        return self.get_all_pin2pad_map().get(pin_name, None)

    def get_all_pin_type(self):
        if len(self.pin_name2pin_type) > 0:
            pin_type_set = set(self.pin_name2pin_type.values())

        else:
            pin_type_list = []
            pin_map = self.get_all_pin2pad_map()
            for pin in pin_map.values():
                pin_name = pin.get_package_pin_name()
                pin_type = self.get_pin_type_by_pin_name(pin_name)
                pin_type_list.append(pin_type)

                if self.is_load_all:
                    pad_name = pin.get_pad_name()
                    io_bank = self.get_iobank_by_pad_name(pad_name)
                    self.add_iobank2pin_map(io_bank, pin_name)

            pin_type_set = set(pin_type_list)

        return tuple(pin_type_set)

    def is_titanium(self) -> bool:
        return self._device.find_block("hsio") is not None

    def get_pin_type_by_pin_name(self, pin_name: str) -> str:
        pin_type = self.pin_name2pin_type.get(pin_name, "")

        if pin_type != "":
            return pin_type

        pad_name = self.get_pad_name_by_pin(pin_name)
        pin_type = self.get_pin_type_by_pad_name(pad_name)

        if self.is_load_all:
            self.pin_name2pin_type[pin_name] = pin_type

        return pin_type

    def get_pin_type_by_pad_name(self, pad_name:str) -> str:
        """
        The pin type should be identified by the pin function
        If it failed, use pin name to identify
        """
        if pad_name == 'NC':
            pin_type = 'NC'

        elif self._device.find_block("hsio"):
            # Check pin type by function first
            pin_type = self.get_tx_pin_type_by_function(pad_name)

            if pin_type == "":
                pin_type = self.get_titanium_pin_type_by_pad(pad_name)

        else:
            pin_type = self.get_trion_pin_type_by_function(pad_name)
            if pin_type == "":
                pin_type = self.get_trion_pin_type_by_pad_name(pad_name)

        return pin_type

    def get_trion_pin_type_by_pad_name(self, pad_name:str) -> Literal[TRION_PIN_TYPE, ""]:
        pin_type: Literal[TRION_PIN_TYPE, ""]

        pin_type = self.check_dedicated_setting(pad_name)

        if pin_type != "":
            return pin_type

        pin_type_list: List[PinType] = []
        if pad_name.startswith("GPIO"):
            if self.is_multifunction(pad_name):
                pin_type_list.append(PinType.MULTI_FUNCTION)

            if self.is_configuration(pad_name):
                pin_type_list.append(PinType.CONFIG)

            if self.is_lvds(pad_name):
                pin_type_list.append(PinType.LVDS)

            pin_type_list.append(PinType.GPIO)
            pin_type = self.combine_pin_type(pin_type_list) # type: ignore

        elif pad_name.startswith("MIPI"):
            pin_type = "MIPI"
        elif pad_name.startswith("DDR"):
            pin_type = "DDR"
        elif pad_name.startswith("REF_RES"):
            pin_type = "LVDS Reference Resistor Pin"
        # PT-2330 Add a new pin type "TEST"
        elif pad_name.startswith("TEST_PROBE"):
            pin_type = "TEST"
        else:
            pin_type = "NC"
            self.logger.error(f"{pad_name}: Pin type is not defined")

        return pin_type

    def get_titanium_pin_type_by_pad(self, pad_name:str) -> str:
        '''
        Get Titanium related pin type
        '''
        pin_type: Literal[TITANIUM_PIN_TYPE, ""] = ""
        pad_name_list = pad_name.split("_")

        pin_type = self.check_dedicated_setting(pad_name)

        if pin_type != "":
            return pin_type

        pin_type_list = []
        if pad_name_list[0].startswith("GPIO"):
            if self.is_hsio(pad_name):
                pin_type_list.append(PinType.HSIO)
            else:
                pin_type_list.append(PinType.HVIO)

            if self.is_configuration(pad_name):
                pin_type_list.append(PinType.CONFIG)
            if self.is_multifunction(pad_name):
                pin_type_list.append(PinType.MULTI_FUNCTION)
            pin_type = self.combine_pin_type(pin_type_list) # type: ignore

        elif "MIPI" in pad_name:
            pin_type = "MIPI"

        elif "DDR" in pad_name:
            pin_type = "DDR"

        elif pad_name.startswith("REF_RES") or pad_name == "ZQ":
            pin_type = "Reference Resistor Pin"

        elif re.match(r"^Q([0-9]+)_", pad_name) is not None:
            pin_type = "Transceiver"

        # PT-2330 Add a new pin type "TEST"
        elif pad_name.startswith("TEST_PROBE"):
            pin_type = "TEST"

        else:
            pin_type = "NC"
            self.logger.error(f"{pad_name}: Pin type is not defined")

        return pin_type

    def get_trion_pin_type_by_function(self, pad_name: str) -> Literal[TRION_PIN_TYPE, ""]:
        pin_type: Literal[TRION_PIN_TYPE, ""] = ""

        if pad_name == "":
            return pin_type

        pin_function, dev_blk_type = self.get_function_info_by_pad_name(pad_name)
        pin_type = self.check_dedicate_pin_type_by_func(pin_function, pad_name)

        if pin_type != "":
            return pin_type

        if "User IO" in pin_function:
            pin_type_list = []
            if self.is_multi_pin_function(pin_function):
                pin_type_list.append(PinType.MULTI_FUNCTION)

            if pin_function and "Configuration" in pin_function:
                pin_type_list.append(PinType.CONFIG)

            if dev_blk_type == DeviceDBService.BlockType.LVDS_RX or\
                dev_blk_type == DeviceDBService.BlockType.LVDS_TX:
                pin_type_list.append(PinType.LVDS)

            if len(pin_type_list) == 0:
                pin_type = PinType.GPIO.value
            else:
                pin_type_list.append(PinType.GPIO)

                if len(pin_type_list) >= 2:
                    pin_type = self.combine_pin_type(pin_type_list) # type: ignore

                    # Cannot identify pin type "Configuration, LVDS, or GPIO"
                    # by using function, use name to identify it
                    if pin_type == "LVDS or GPIO":
                        return ""

        elif pad_name.startswith("REF_RES"):
            pin_type = "LVDS Reference Resistor Pin"

        elif dev_blk_type in [DeviceDBService.BlockType.MIPI_TX,
                            DeviceDBService.BlockType.MIPI_RX,
                            DeviceDBService.BlockType.MIPI_TX_ADV,
                            DeviceDBService.BlockType.MIPI_RX_ADV]:
            pin_type = "MIPI"

        elif dev_blk_type in [DeviceDBService.BlockType.DDR, DeviceDBService.BlockType.DDR_ADV]:
            pin_type = "DDR"

        if pin_type == "":
            self.logger.debug(f"Cannot identify pin type by function: {pin_function}")

        return pin_type

    def get_tx_pin_type_by_function(self, pad_name: str) -> str:
        pin_type: Literal[TITANIUM_PIN_TYPE, ""] = ""

        if pad_name == "":
            return pin_type

        pin_function, dev_blk_type = self.get_function_info_by_pad_name(pad_name)
        pin_type = self.check_dedicate_pin_type_by_func(pin_function, pad_name)

        if pin_type != "":
            return pin_type

        pin_type_list = []
        if dev_blk_type == DeviceDBService.BlockType.GPIO or\
            dev_blk_type == DeviceDBService.BlockType.HSIO or\
                dev_blk_type == DeviceDBService.BlockType.HSIO_MAX:

            if dev_blk_type == DeviceDBService.BlockType.GPIO:
                pin_type_list.append(PinType.HVIO)
            else:
                pin_type_list.append(PinType.HSIO)

            if "Configuration" in pin_function:
                pin_type_list.append(PinType.CONFIG)

            if self.is_multi_pin_function(pin_function):
                pin_type_list.append(PinType.MULTI_FUNCTION)

            pin_type = self.combine_pin_type(pin_type_list) # type: ignore

        elif dev_blk_type in [DeviceDBService.BlockType.MIPI_TX,
                            DeviceDBService.BlockType.MIPI_RX,
                            DeviceDBService.BlockType.MIPI_TX_ADV,
                            DeviceDBService.BlockType.MIPI_RX_ADV]:
            pin_type = "MIPI"

        elif dev_blk_type in [DeviceDBService.BlockType.DDR, DeviceDBService.BlockType.DDR_ADV]:
            pin_type = "DDR"

        elif pad_name.startswith("REF_RES") or pad_name == "ZQ":
            pin_type = "Reference Resistor Pin"

        elif pin_function in [DeviceDBService.BlockType.QUAD| DeviceDBService.BlockType.QUAD_PCIE]:
            pin_type = "Transceiver"

        if pin_type == "":
            self.logger.debug(f"Cannot find pin type by the function:{pin_function}")
        return pin_type

    def combine_pin_type(self, pin_type_list: List[PinType]) -> str:
        pin_type = pin_type_list[0].value

        if len(pin_type_list) == 2:
            pin_type += " or " + pin_type_list[1].value

        elif len(pin_type_list) >= 2:
            for i in pin_type_list[1:]:
                pin_type += ", "

                if i == pin_type_list[-1]:
                    pin_type += "or "

                pin_type += i.value

        return pin_type

    def is_multi_pin_function(self, pin_function: str) -> bool:
        multi_key_list = [
            "GCLK", "GCTRL", "RCLK", "RCTRL", "PLL_EXTFB", "PLL_CLKIN"
        ]

        return any(pin_func in pin_function for pin_func in multi_key_list)

    def check_dedicate_pin_type_by_func(self, pin_func: str, pad_name: str) -> Literal[DEDICATED_PIN_TYPE, ""]:
        pin_type: Literal[DEDICATED_PIN_TYPE, ""] = ""

        if pin_func == "":
            rpt_sig_name = self.get_rpt_sig_name_by_pad_name(pad_name)

            if rpt_sig_name == "GND":
                pin_func = "Ground"
            elif rpt_sig_name == "VCC":
                pin_func = "Power"

        if pad_name == "VQPS_GND":
            pin_type = "Dedicated Power or Ground"

        elif "Ground" in pin_func:
            pin_type = "Dedicated Ground"

        elif "Power" in pin_func:
            pin_type = "Dedicated Power"

        elif self.is_jtag(pad_name):
            pin_type = "Dedicated JTAG"

        elif pin_func == "Configuration":
            pin_type = "Dedicated Configuration"

        elif pin_func == "Chip Select":
            pin_type = "Dedicated SPI Flash"

        return pin_type

    def is_lvds(self, pad_name:str) -> bool:
        pad_name_list = pad_name.split("_")
        return pad_name.startswith("GPIO") and \
                (pad_name_list[1].startswith(("TX", "RX")))

    def is_configuration(self, pad_name:str) -> bool:
        config_key_list = [
            "CDI", "CSI", "CSO", "CBSEL",
            "CBUS", "CCK", "NSTATUS", "SS_N",
            "TEST_N", "SSL", "SSU_N", "CDO"
        ]

        return any(pin_func in pad_name for pin_func in config_key_list)

    def is_multifunction(self, pad_name:str) -> bool:
        multi_func_key_list = [
            "CLK", "CTRL", "PLLIN", "EXTFB"
        ]

        return any(pin_func in pad_name for pin_func in multi_func_key_list)

    def is_hsio(self, pad_name) -> bool:
        pad_name_list = pad_name.split("_")
        return pad_name_list[1] in ("N", "P")

    def check_dedicated_setting(self, pad_name:str) -> Literal[DEDICATED_PIN_TYPE, ""]:
        pin_type: Literal[DEDICATED_PIN_TYPE, ""] = ""

        if pad_name == "VQPS_GND":
            pin_type = "Dedicated Power or Ground"

        elif pad_name.startswith("GND"):
            pin_type = "Dedicated Ground"

        elif self.is_power(pad_name):
            pin_type = "Dedicated Power"

        elif pad_name == "CDONE" or pad_name == "CRESET_N":
            pin_type = "Dedicated Configuration"

        elif self.is_jtag(pad_name):
            pin_type = "Dedicated JTAG"

        return pin_type

    def is_power(self, pad_name:str) -> bool:
        return pad_name.startswith(("VCC", "VQPS", "VDD")) and pad_name != "VQPS_GND"

    def is_jtag(self, pad_name:str) -> bool:
        return pad_name in ("TCK", "TMS", "TDI", "TDO")

    def get_function_by_pad_name(self, pad_name: str) -> str:
        function_str, dev_blk_type = self.get_function_info_by_pad_name(pad_name)
        return function_str

    def get_function_info_by_pad_name(self, pad_name: str) -> Tuple[str, Optional[DeviceDBService.BlockType]]:
        io_pad = self._device.get_io_pad() # type: Optional[DeviceIO]

        if io_pad is None:
            return "", None

        instances = io_pad.find_instance(pad_name)
        pad_info_obj = io_pad.find_pad(pad_name) # type: Optional[IOPadInfo]
        assert self.is_coldboot is not None

        function_str, dev_blk_type = self._get_pin_function(
            pad_info_obj, instances)

        if self.is_load_all and function_str != "":
            self.save_pin_func(pad_name, function_str)

        return function_str, dev_blk_type

    def save_pin_func(self, pad_name: str, function_str: str):
        if any(pin_func.value in function_str for pin_func in PinFunction):
            for i in self.get_all_pin2pad_map().values():
                if pad_name == i.get_pad_name():
                    pin_name = i.get_package_pin_name()
                    self.set_all_func_map(pin_name, function_str)

    def get_function_by_pin_name(self, pin_name: str) -> str:
        """
        for getting function of that pad name
        At the same time, do the func2pin mapping
        """
        pin = self.get_pin2padmap_by_pin_name(pin_name)

        if pin is None:
            return ""

        pad_name = pin.get_pad_name()

        function_str, dev_blk_type = self.get_function_info_by_pad_name(pad_name)

        return function_str

    def set_all_func_map(self, pin_name: str, function: str):
        """
        Check all pin function and create mapping
        """
        for pin_func in tuple(PinFunction):
            if pin_func.value in function:
                self._set_func2pin_name(pin_func, pin_name)

    def _set_func2pin_name(self, func: PinFunction, pin_name: str):
        pad_list = self.func2pin_name.get(func, [])

        if pin_name in pad_list:
            return

        pad_list.append(pin_name)
        self.func2pin_name[func] = pad_list

    def get_func2pin_name(self):
        return self.func2pin_name

    def get_pad_name_by_pin(self, pin_name: str):
        pin = self.get_pin2padmap_by_pin_name(pin_name)

        if pin is None:
            return ""

        return pin.get_pad_name()

    def _get_pin_function(self, pad_info_obj: Optional[IOPadInfo],
                          instances: List[str]) -> Tuple[str, Optional[DeviceDBService.BlockType]]:
        '''
        Refer to PinoutReport

        Get the function description of the specified pin name.

        :param pad_info_obj: The IOPadInfo object
        :param instances: List of instances associated to the pad
        :param is_coldboot: Flag indicating if coldboot feature was set

        :return Function string
        '''
        function_name = ""
        gpio_pin_type = ""
        dev_blk_type = None
        config_model = self._device.get_config_model()

        if pad_info_obj is None:
            return function_name, dev_blk_type

        # Check if it is a GPIO pad
        if instances and len(instances) == 1:
            found_gpio = False
            ins_obj = None
            sec_function_set: Set[str] = set()

            for ins_name in instances:
                tmp_sec_func_list = []

                # Get the reference
                ins_obj = self._device.find_instance(ins_name)

                if ins_obj is None:
                    continue

                assert isinstance(ins_obj, PeripheryBlockInstance)
                ref_name = ins_obj.get_ref_name()

                if ref_name not in DeviceDBService.block_str2type_map:
                    continue

                dev_blk_type = DeviceDBService.block_str2type_map[ref_name]

                # TODO: If the assumption that it only involves GPIO and LVDS
                # is longer true, then the following needs to be
                # updated
                match dev_blk_type:
                    case DeviceDBService.BlockType.GPIO | DeviceDBService.BlockType.LVDS_RX |\
                        DeviceDBService.BlockType.LVDS_TX | DeviceDBService.BlockType.HSIO |\
                        DeviceDBService.BlockType.HSIO_MAX:
                        found_gpio = True

                        # Check if it has an instance and if any of the
                        # instance pin has a secondary function
                        tmp_sec_func_list = ins_obj.get_secondary_connection(
                            ins_obj.get_name())

                        # PT-1766 N pins should not have VREF
                        if PinFunction.VREF.value in tmp_sec_func_list and \
                            pad_info_obj.get_pin_name() == "N":
                                tmp_sec_func_list.remove(PinFunction.VREF.value)

                        sec_function_set = sec_function_set.union(
                            set(tmp_sec_func_list))

            if found_gpio:
                function_name += "User IO"

            if sec_function_set:
                # PT-1814: Don't highlight MIPI resources as GCLK/RCLK
                # We need to check if the function is on HSIO with the type
                # associated to "RCLK" pin
                mipi_clk_only = False
                if ins_obj is not None and self.dbi.check_if_sec_func_mipi_clock_only(ins_obj):
                    mipi_clk_only = True

                for func_name in sorted(sec_function_set):
                    if (func_name == "RCLK" or func_name == "GCLK") and\
                        mipi_clk_only:
                        continue

                    if function_name == "":
                        function_name += func_name
                    else:
                        function_name += "/" + func_name

            # Check in configuration
            if config_model is not None:

                # Get the configuration function
                if config_model.is_pad_used_in_selected_mode(
                        pad_info_obj, self.is_coldboot):

                    if function_name == "":
                        function_name += "Configuration"
                    else:
                        function_name += "/Configuration"
            if gpio_pin_type != "":
                function_name += gpio_pin_type
        else:
            function_name = pad_info_obj.get_function()
            function_name = pad_info_obj.update_pad_func_voltage(self._get_timing_model_voltage())

        return function_name, dev_blk_type

    def get_rpt_sig_name_by_pad_name(self, pad_name: str):
        rpt_signal_name = ""
        io_pad = self._device.get_io_pad() # type: Optional[DeviceIO]

        if io_pad is None:
            return rpt_signal_name

        pad_info_obj = io_pad.find_pad(pad_name)
        config_model = self._device.get_config_model()
        if pad_info_obj is None or config_model is None:
            return rpt_signal_name

        # Check if the pin has other function
        # Get the specific function name regardless
        # if it is used or not
        rpt_signal_name = config_model.get_pad_function(
            pad_info_obj.get_pad_name())

        if rpt_signal_name is None or rpt_signal_name == "":
            rpt_signal_name = ""
            pad_name = pad_info_obj.get_pad_name()
            if pad_info_obj.get_direction() == IOPadInfo.PadDirection.power:
                if pad_name == "VQPS_GND":
                    rpt_signal_name = "VCC/GND"
                elif pad_name.find("GND") != -1:
                    rpt_signal_name = "GND"
                else:
                    rpt_signal_name = "VCC"
            elif pad_name == "SPI_CS_N":
                rpt_signal_name = pad_name

        return rpt_signal_name

    def get_quad_res_name_by_pad_name(self, pad_name: str) -> str:
        """
        Check the resource name for that pad.

        :param pad_name: Pad name
        :type pad_name: str
        :return: QUAD_n/ Qn_LNn if the pattern match, else empty string
        :rtype: str
        """
        pattern = r"Q([0-9])_([A-Z_]+)([0-9]*)[A-Z]*"
        result = re.match(pattern, pad_name)

        if result:
            quad_num = result.group(1)
            lane_num = result.group(3)

            if result.group(2) in ["REFCLK", "CAL_RES"]:
                return "QUAD_" + quad_num

            return "Q" + quad_num + "_LN" + lane_num

        return ""

    def set_res2pin_name(self, resource_list: List[str], pin_name: str, pad_name: str):
        # Special case for handling lane resource
        if re.match(r"Q[0-9]_", pad_name):
            quad_res = self.get_quad_res_name_by_pad_name(pad_name)
            if quad_res != "" and quad_res not in resource_list:
                resource_list.append(quad_res)

        for res_name in resource_list:
            pin_list = self.res2pin_map.get(res_name, [])

            if pin_name in pin_list:
                continue

            pin_list.append(pin_name)
            self.res2pin_map[res_name] = pin_list

    def get_res2pin_name(self)-> Dict[str, List[str]]:
        if len(self.res2pin_map) == 0:
            self.build_res2_pin_name()

        return self.res2pin_map

    def build_res2_pin_name(self):
        for pin in self.get_all_pin2pad_map().values():
            pin_name = pin.get_package_pin_name()
            pad_name = pin.get_pad_name()
            resource_list = self.get_resource_name_by_pad_name(pad_name)
            self.set_res2pin_name(resource_list, pin_name, pad_name)

    def get_resource_name_by_pad_name(self, pad_name: str) -> List[str]:
        instances = ""
        io_pad = self._device.get_io_pad() # type: Optional[DeviceIO]

        if io_pad is None:
            return [instances]

        return io_pad.find_design_instance(pad_name, self._device)

    def get_block_type_by_pad_name(self, pad_name: str) -> str:
        block_type: str = ""
        ins_name_list = self.get_resource_name_by_pad_name(pad_name)
        if len(ins_name_list) >= 1:
            ins_obj = self._device.find_instance(ins_name_list[0])

            if ins_obj is not None:
                ref_name: str = ins_obj.get_ref_name()
                if ref_name in DeviceDBService.block_str2type_map:
                    block_type = ref_name

        return block_type.upper()

    def get_block_type_by_pin_name(self, pin_name: str) -> str:
        pad_name = self.get_pad_name_by_pin(pin_name)
        return self.get_block_type_by_pad_name(pad_name)

    def get_display_block_type(self, raw_blk_type: str):
        '''
        Using this function to filter out any block type that should be
        replaced by another string for displaying to user purpose.

        Example usage: HSIO_MAX block type displayed as HSIO
        '''
        if raw_blk_type.lower() == "hsio_max":
            return "HSIO"

        return raw_blk_type

    def get_block_instance_by_pin_name(self, pin_name: str) -> List[str]:
        pad_name = self.get_pad_name_by_pin(pin_name)

        if pad_name == "":
            return []

        block_inst_list = self.get_block_instance_by_pad_name(pad_name)

        return block_inst_list

    def get_signal_name_by_pin_name(self, pin_name: str) -> str:
        pad_name = self.get_pad_name_by_pin(pin_name)

        if pad_name == "":
            return ""

        return self.get_signal_name_by_pad_name(pad_name)

    def get_pull_type_by_pin_name(self, pin_name: str) -> str:
        pad_name = self.get_pad_name_by_pin(pin_name)

        if pad_name == "":
            return ""

        return self.get_pull_type_by_pad_name(pad_name)

    def get_block_instance_by_pad_name(self, pad_name: str) -> List[str]:
        """
        Get the instance by pad name
        """
        block_inst_list: List[str] = []
        block_type = self.get_block_type_by_pad_name(pad_name).lower()
        resource_name_list = self.get_resource_name_by_pad_name(pad_name)
        if block_type != "" and block_type in DeviceDBService.block_str2type_map:
            dev_blk_type = DeviceDBService.block_str2type_map[block_type]
            if dev_blk_type == DeviceDBService.BlockType.GPIO or \
                dev_blk_type == DeviceDBService.BlockType.HSIO or \
                dev_blk_type == DeviceDBService.BlockType.HVIO or \
                dev_blk_type == DeviceDBService.BlockType.LVDS_TX or\
                dev_blk_type == DeviceDBService.BlockType.LVDS_RX or \
                dev_blk_type == DeviceDBService.BlockType.MIPI_TX or\
                dev_blk_type == DeviceDBService.BlockType.MIPI_RX or\
                dev_blk_type == DeviceDBService.BlockType.MIPI_TX_ADV or\
                dev_blk_type == DeviceDBService.BlockType.MIPI_RX_ADV or\
                dev_blk_type == DeviceDBService.BlockType.QUAD or\
                dev_blk_type == DeviceDBService.BlockType.QUAD_PCIE or\
                dev_blk_type == DeviceDBService.BlockType.QUAD_LANE_10G or\
                    dev_blk_type == DeviceDBService.BlockType.HSIO_MAX:

                for res_name in resource_name_list:
                    block_inst_list = self.get_inst_list_by_pll(res_name, resource_name_list[0])

                    block_name = self.get_gpio_res_block_name(res_name, pad_name)

                    mipi_inst_name: str = self.get_mipi_instance(res_name, block_type)

                    if mipi_inst_name != "":
                        block_inst_list.append(mipi_inst_name)

                    for quad_type in QuadType.get_lane_based_type_list():
                        lane_name = self.get_lane_based_instance(pad_name, res_name, quad_type)
                        if lane_name != "":
                            block_inst_list.append(lane_name)

                    quad_pcie_name = self.get_quad_pcie_instance(pad_name, res_name)
                    if quad_pcie_name != "":
                        block_inst_list.append(quad_pcie_name)

                    if block_name != "":
                        block_inst_list.append(block_name)
                        break

            elif dev_blk_type == DeviceDBService.BlockType.JTAG:
                block_inst_list = self.get_jtag_instance_list()
            elif dev_blk_type == DeviceDBService.BlockType.DDR:
                ddr_inst_name = self.get_ddr_instance(pad_name, resource_name_list[0])
                if ddr_inst_name != "":
                    block_inst_list.append(ddr_inst_name)

        block_inst_list = list(set(block_inst_list))
        block_inst_list.sort()
        return block_inst_list

    @property
    def is_load_all(self):
        return self._is_load_all

    @is_load_all.setter
    def is_load_all(self, is_load):
        self._is_load_all = is_load

    def get_inst_list_by_pll(self, target_res: str, ori_res_name: str) -> List[str]:
        assert self.design is not None, f"design information is not set"
        blk_name_list: List[str] = []
        pll_reg:Optional[PLLRegistry] = self.design.pll_reg

        if pll_reg is None:
            return blk_name_list

        for blk_inst in pll_reg.get_all_inst():
            res_name = blk_inst.get_device()

            if res_name == "":
                continue

            res_list = self.get_pll_res_list_by_pll_inst(blk_inst)

            if target_res in res_list or ori_res_name in res_list:
                blk_name_list.append(blk_inst.name)

        return blk_name_list

    def is_mipi_ref_clk(self, blk_inst: Union[MIPI, MIPITx, MIPIHardDPHYTx, PLLSSC], target_res: str) -> bool:
        """
        Check if resource is used as reference clock in MIPI/ PLL SSC instance

        :param blk_inst: MIPI/ PLL SSC instance
        :type blk_inst: Union[MIPI, MIPITx, MIPIHardDPHYTx, PLLSSC]
        :param target_res: Resource name
        :type target_res: str
        :return: ``True`` if used as reference clock, otherwise ``False``
        :rtype: bool
        """
        is_ref_clk = False

        if target_res == "":
            return is_ref_clk

        mipi_res = blk_inst.get_device()

        if mipi_res == "" or mipi_res is None:
            return is_ref_clk

        res_name_list: List[str] = []
        if isinstance(blk_inst, (MIPI, MIPITx)):
            if isinstance(blk_inst, MIPI) and blk_inst.tx_info is None:
                return is_ref_clk

            res_name_list = self.get_mipi_ref_clk_by_mipi_res(mipi_res, PkgItemType.MIPI_TX.value)

        elif isinstance(blk_inst, (MIPIHardDPHYTx, PLLSSC)):
            param_service = GenericParamService(blk_inst.get_param_group(), blk_inst.get_param_info())
            param_id: Optional[Enum] = None
            item_type: Optional[PkgItemType] = None

            if isinstance(blk_inst, MIPIHardDPHYTx):
                param_id = MIPIHardDPHYTxDesignParamInfo.Id.ref_clk_source
                item_type = PkgItemType.MIPI_HARD_DPHY_TX
            elif isinstance(blk_inst, PLLSSC):
                param_id = PLLSSCDesignParamInfo.Id.ref_clk_source
                item_type = PkgItemType.PLL_SSC

            assert param_id is not None
            assert item_type is not None

            ref_clk_src = param_service.get_param_value(param_id)
            clk_src_index = {
                "core": 0,
                "pll": 1,
                "gpio": 2
            }.get(ref_clk_src, 0)

            res_name_list = self.get_mipi_ref_clk_by_mipi_res(mipi_res, item_type.value, clk_src_index)

        if target_res in res_name_list:
            is_ref_clk = True

        return is_ref_clk

    def is_quad_pcie_ref_clk(self, blk_inst: QuadPCIE, pad_name: str, target_res: str):
        is_ref_clk = False

        if target_res == "":
            return is_ref_clk

        quad_pcie_res = blk_inst.get_device()

        if quad_pcie_res == "" or quad_pcie_res is None:
            return is_ref_clk

        param_service = GenericParamService(blk_inst.get_param_group(), blk_inst.get_param_info())

        ref_clk_src = param_service.get_param_value(PCIEConfigParamInfo.Id.PIPE_CONFIG_CMN__config_reg_2__pma_cmn_refclk_sel)

        res_name_list: List[str] = []
        blk_type = DeviceDBService.BlockType.QUAD_PCIE
        dev_service_adv = self.dbi.get_block_service(blk_type) # type: QuadPCIEDeviceService

        match ref_clk_src:
            case "External":
                ext_ref_clk = param_service.get_param_value(PCIEConfigParamInfo.Id.PMA_CMN__cmn_plllc_gen_preg__cmn_plllc_pfdclk1_sel_preg)
                assert isinstance(ext_ref_clk, str)
                res_name_list = dev_service_adv.get_external_refclk_pins(
                            quad_pcie_res, f"REFCLK{ext_ref_clk[-1]}")

            case "PLL":
                pll_res_list: List[str] = []
                ref_clk0 = param_service.get_param_value(PCIEParamInfo.Id.pll_ref_clk0)
                ref_clk1 = param_service.get_param_value(PCIEParamInfo.Id.pll_ref_clk1)
                for pll_refclk in [ref_clk0, ref_clk1]:
                    ref_clk_list, _ = dev_service_adv.get_all_resource_on_ins_pin(
                                quad_pcie_res, pll_refclk, None)
                    pll_res_list += ref_clk_list

                for res_name in set(pll_res_list):
                    if res_name.startswith("PLL"):
                        res_name_list += self.get_res_list_by_pll_res_name(res_name)

        if target_res in res_name_list or pad_name in res_name_list:
            is_ref_clk = True

        return is_ref_clk

    def is_lane_based_ref_clk(self, blk_inst: QuadLaneCommon, pad_name: str, target_res: str,
                              blk_type: DeviceDBService.BlockType):
        is_ref_clk = False

        if target_res == "":
            return is_ref_clk

        quad_res = blk_inst.get_device()

        if quad_res == "" or quad_res is None:
            return is_ref_clk

        param_service = GenericParamService(blk_inst.get_param_group(), blk_inst.get_param_info())

        param_id = QuadConfigParamInfo.Id.PIPE_CONFIG_CMN__config_reg_2__pma_cmn_refclk_sel
        ref_clk_src = param_service.get_param_value(param_id)

        res_name_list: List[str] = []
        dev_service_adv = self.dbi.get_block_service(blk_type) # type: QuadLane10GDeviceService

        match ref_clk_src:
            case "External":
                param_id = QuadConfigParamInfo.Id.PMA_CMN__cmn_plllc_gen_preg__cmn_plllc_pfdclk1_sel_preg
                val = param_service.get_param_value(param_id)
                assert isinstance(val, str)
                ref_clk_str = val.replace(' ', "").upper()
                res_name_list = dev_service_adv.get_external_refclk_pins(quad_res, ref_clk_str)

        if target_res in res_name_list or pad_name in res_name_list:
            is_ref_clk = True

        return is_ref_clk

    def is_quad_pcie_perstn(self, blk_inst: QuadPCIE, target_res: str):
        is_perstn = False

        if target_res == "":
            return is_perstn

        quad_pcie_res = blk_inst.get_device()

        if quad_pcie_res == "" or quad_pcie_res is None:
            return is_perstn

        blk_type = DeviceDBService.BlockType.QUAD_PCIE
        dev_service_adv = self.dbi.get_block_service(blk_type) # type: QuadPCIEDeviceService
        _, res_name_list = blk_inst.get_perstn_info(dev_service_adv, None)

        if target_res in res_name_list:
            is_perstn = True

        return is_perstn

    def get_mipi_ref_clk_by_mipi_res(self, mipi_res: str, mipi_type: str,
                                     clk_src_index: Optional[int] = None) -> List[str]:
        """
        Get a list of reference clock resource name by MIPI resource name

        MIPI_HARD_DPHY_TX/ PLL_SSC:
        - Given ``clk_src_index`` -> Load reference clock resource for that source type
        - Without ``clk_src_index`` -> Load reference clock resource for all source type
        - For PLL resource, they will be converted to GPIO resource name (e.g. PLL_TL0 -> GPIOL_PN_01, GPIOL_P_01)

        :param mipi_res: MIPI Resource name
        :type mipi_res: str
        :param mipi_type: MIPI type
        :type mipi_type: str
        :param clk_src_index: Reference clock source index, defaults to None
        :type clk_src_index: Optional[int], optional
        :return: List of reference clock resource name
        :rtype: List[str]
        """
        if mipi_type == PkgItemType.MIPI_TX.value:
            blk_type = DeviceDBService.BlockType.MIPI_TX
            mipi_tx_dev_service = self.dbi.get_block_service(blk_type) # type: MIPITxService
            ref_clk: Optional[str] = mipi_tx_dev_service.get_resource_on_ref_clk_pin(mipi_res)
            if ref_clk is None:
                return []
            return [ref_clk]

        elif mipi_type in (PkgItemType.MIPI_HARD_DPHY_TX.value, PkgItemType.PLL_SSC.value):
            blk_type = DeviceDBService.BlockType.MIPI_TX_ADV
            mipi_tx_dev_service_adv = self.dbi.get_block_service(blk_type) # type: MIPITxServiceAdvance

            if clk_src_index is None:
                clk_src_index_list = (0,1,2)
                ref_clk_list: List[str] = []

                for index in clk_src_index_list:
                    name_list, _ = mipi_tx_dev_service_adv.get_all_resource_on_refclk_pin(mipi_res, index)
                    ref_clk_list += name_list
            else:
                ref_clk_list, _ = mipi_tx_dev_service_adv.get_all_resource_on_refclk_pin(mipi_res, clk_src_index)

            res_name_list: List[str] = []
            for res_name in ref_clk_list:
                if res_name.startswith("PLL"):
                    res_name_list += self.get_res_list_by_pll_res_name(res_name)
                else:
                    res_name_list.append(res_name)
            res_name_list.sort()

            return res_name_list
        return []

    def get_quad_pcie_ref_clk_by_res(self, res_name: str, refclk_src: Optional[str] = None):
        """
        Get a list of pad name that can be used as Quad PCIe Reference Clock

        :param res_name: Quad PCIe resource name (e.g. QUAD_0)
        :type res_name: str
        :param refclk_src: Quad PCIe reference clock source (if given)
        :type refclk_src: Optional[str]
        :return pad_name_list: List of pad names in package
        :type pad_name_list: List[str]
        """
        blk_type = DeviceDBService.BlockType.QUAD_PCIE
        dev_service_adv = self.dbi.get_block_service(blk_type) # type: QuadPCIEDeviceService

        ref_clk_list: List[str] = []

        if refclk_src is None:
            ref_clk_list = dev_service_adv.get_instance_dependencies_name(res_name)
        else:
            match refclk_src:
                case "PLL":
                    for pll_ref_clk in QuadPCIE.get_all_ref_clk0_pll_options(res_name):
                        new_ref_clk_list, _ = dev_service_adv.get_all_resource_on_ins_pin(
                            res_name, pll_ref_clk, None)
                        ref_clk_list += new_ref_clk_list

                case "External":
                    param_info = build_pcie_param_info()
                    param_id = PCIEConfigParamInfo.Id.PMA_CMN__cmn_plllc_gen_preg__cmn_plllc_pfdclk1_sel_preg
                    options = param_info.get_valid_setting(param_id)
                    assert isinstance(options, list)
                    for option in options:
                        assert isinstance(option, str)
                        ref_clk_str = option.replace(' ', "").upper()
                        ref_clk_list = dev_service_adv.get_external_refclk_pins(
                            res_name, ref_clk_str)

        return sorted(list(set(ref_clk_list)))

    def get_lane_based_ref_clk_by_res(self, res_name: str, quad_type: QuadType,
                                      refclk_src: Optional[str] = None):
        """
        Get a list of pad name that can be used as Quad Reference Clock

        :param res_name: lane resource name (e.g. Q0_LN1)
        :type res_name: str
        :param refclk_src: Quad reference clock source (if given)
        :type refclk_src: Optional[str]
        :return pad_name_list: List of pad names in package
        :type pad_name_list: List[str]
        """
        ref_clk_list: List[str] = []
        blk_type = self.get_quad_device_blk_type(quad_type)
        quad_num, _ = QuadResService.break_res_name(res_name)

        if quad_num != -1 or blk_type is None:
            return ref_clk_list

        quad_res = f"QUAD_{quad_num}"

        dev_service_adv = self.dbi.get_block_service(blk_type) # type: QuadLane10GDeviceService

        if refclk_src is None:
            ref_clk_list = dev_service_adv.get_instance_dependencies_name(quad_res)
        else:
            match refclk_src:
                case "External":
                    # Skip for external ref clock (handled by get_lane_based_ref_clk_by_design)
                    param_info = build_quad_param_info()
                    param_id = QuadConfigParamInfo.Id.PMA_CMN__cmn_plllc_gen_preg__cmn_plllc_pfdclk1_sel_preg
                    options = param_info.get_valid_setting(param_id)
                    assert isinstance(options, list)
                    for option in options:
                        assert isinstance(option, str)
                        ref_clk_str = option.replace(' ', "").upper()
                        ref_clk_list = dev_service_adv.get_external_refclk_pins(
                            quad_res, ref_clk_str)

        return sorted(list(set(ref_clk_list)))

    def get_pll_res_list_by_pll_inst(self, blk_inst:PLL) -> List[str]:
        '''
        Get pll resource list by checking pll reference clock mode.

        :param  : blk_inst pll block instance
        :param  : is_get_all skip the reference clock mode checking
        :return : List of resource name (e.g. [GPIOL_12, GPIOL_N_01] )
        '''
        res_list: List[str] = []

        # Run set_pll_res2res_name before running this function
        assert len(self.pll_res2res_name) != 0

        res_name = blk_inst.get_device()

        if res_name == "":
            return res_list

        if isinstance(blk_inst, (PLLComplex, PLLAdvance)):
            pll_res: Union[List[str], Dict[Union[str, PLLAdvance.RefClockIDType], List[str]]] = self.pll_res2res_name.get(res_name, {})
            assert isinstance(pll_res, dict)

            if blk_inst.ref_clock_mode == PLLAdvance.RefClockModeType.dynamic:
                res_list = [resource_name for v in pll_res.values() for resource_name in v ]

            elif blk_inst.ref_clock_mode == PLLAdvance.RefClockModeType.core:
                return res_list

            elif blk_inst.ref_clock_mode == PLLAdvance.RefClockModeType.external:
                assert blk_inst.ext_ref_clock_id is not None
                res_list = pll_res.get(blk_inst.ext_ref_clock_id, [])

        else:
            pll_res_list = self.pll_res2res_name.get(res_name, [])
            assert isinstance(pll_res_list, list)
            res_list = pll_res_list

        return res_list

    def get_res_list_by_pll_res_name(self, res_name: str) -> List[str]:
        '''
        This function will get resource list,
        ignore the ref clock mode and get res list

        :param  : res_name pll resource name (e.g. PLL_BL0)
        :return : List of resource name (e.g. [GPIOL_12, GPIOL_N_01] )
        '''
        assert self.design is not None, f"design information is not set"
        res_list: List[str] = []

        # Run set_pll_res2res_name before running this function
        assert len(self.pll_res2res_name) != 0

        pll_reg: Optional[PLLRegistry] = self.design.pll_reg

        if pll_reg is None:
            return res_list

        if isinstance(pll_reg, (PLLRegistryComplex, PLLRegistryAdvance)):
            pll_res = self.pll_res2res_name.get(res_name, {})
            assert isinstance(pll_res, dict)
            res_list = [resource_name for v in pll_res.values() for resource_name in v ]
            res_list = list(set(res_list))

        else:
            pll_res_list = self.pll_res2res_name.get(res_name, [])
            assert isinstance(pll_res_list, list)
            res_list = pll_res_list

        return res_list

    def set_pll_res2res_name(self):
        '''
        Set the mapping for pll res name to res name
        (e.g. PLL_TL0 -> [GPIOL_00, GPIOB_11])

        - PLLRegistryAdvance/ PLLRegistryComplex:
            pll res name may change by ref clock mode/ clock id.
            Have an extra mapping for different mode

        - PLLRegistry
            directly set the res_name list
        '''
        assert self.design is not None, f"design information is not set"
        pll_reg = self.design.pll_reg

        if pll_reg is None:
            return

        plldev_service = self.dbi.get_block_service(
                        DeviceDBService.BlockType.PLL)
        pll_dev_list = plldev_service.get_usable_instance_names()

        if isinstance(pll_reg, PLLRegistryComplex) or isinstance(pll_reg, PLLRegistryAdvance):
            for dev_name in pll_dev_list:
                if isinstance(pll_reg, PLLRegistryComplex):
                    ref_clk_id_type = PLLComplex.RefClockIDType
                else:
                    ref_clk_id_type = PLLAdvance.RefClockIDType

                clk2res_name = {}
                for ref_clk in ref_clk_id_type:
                    if ref_clk.is_core_clock():
                        continue

                    if isinstance(pll_reg, PLLRegistryComplex):
                        clk_index = self.get_external_clkin_index(plldev_service, ref_clk, dev_name)
                    else:
                        clk_index = self.get_pll_adv_external_clkin_index(plldev_service, ref_clk)
                    clk2res_name[ref_clk] = plldev_service.get_all_resource_on_ref_clk_pin(
                                        dev_name, clk_index)
                self.pll_res2res_name[dev_name] = clk2res_name

        else:
            for dev_name in pll_dev_list:
                res_name_list = plldev_service.get_instance_dependencies_name(
                                dev_name)
                self.pll_res2res_name[dev_name] = res_name_list

    def set_spi_res2res_name(self):
        assert self.design is not None, f"design information is not set"
        spi_flash_reg = self.design.spi_flash_reg

        if spi_flash_reg is None:
            return

        spf_dev_service = self.dbi.get_block_service(DeviceDBService.BlockType.SPI_FLASH)
        dev_list = spf_dev_service.get_usable_instance_names()

        for spi_res in dev_list:
            resv_res = spf_dev_service.get_shared_instance_names(spi_res)
            self.spi2res_name[spi_res] = resv_res

    def set_ddr_pad2ori_pin_name(self):
        """
        Save ddr pad position for later pin swapping (Used in Ti180 DDR only)
        """
        assert self.design is not None, f"design information is not set"
        assert len(self.res2pin_map) > 0, "Please call build_res2_pin_name first"
        ddr_reg = self.design.ddr_reg

        if ddr_reg is None or not self.is_titanium():
            return

        ddr_dev_service = self.dbi.get_block_service(DeviceDBService.BlockType.DDR_ADV)
        ddr_dev_list = ddr_dev_service.get_usable_instance_names()

        for res_name in ddr_dev_list:
            for pin_name in self.res2pin_map.get(res_name, ()):
                pad_name = self.get_pad_name_by_pin(pin_name)
                self.ddr_pad2ori_pin_name[pad_name] = pin_name

    def get_ddr_pin_cur_pos_by_pad_name(self, pad_name: str) -> str:
        """
        Get DDR current pad position (Current pin name)
        For tx180_ddr, Some pad will be swapped according to the design

        :param pad_name: DDR pad name
        :type pad_name: str
        :return: current pad position
        :rtype: str
        """
        assert self.design is not None, f"design information is not set"
        ori_pos = ""
        ddr_reg = self.design.ddr_reg

        if ddr_reg is None or not self.design.is_block_supported(PeriDesign.BlockType.adv_ddr):
            return ori_pos

        assert len(self.ddr_pad2ori_pin_name) > 0, "Please call self.set_ddr_pad2ori_pin_name first"

        ori_pos = self.ddr_pad2ori_pin_name.get(pad_name, "")
        pad_name_pattern = r"(DDR([0-9]+)?)_((DQ|DM|A)\[([0-9]+)\])"

        # Check Pin swap for DQ/DM, CA pin
        result = re.match(pad_name_pattern, pad_name)
        if result is None:
            return ori_pos

        # Get instance by resource name
        res_name_list = self.get_resource_name_by_pad_name(pad_name)
        if len(res_name_list) <= 0:
            return ori_pos

        inst = ddr_reg.get_inst_by_device_name(res_name_list[0])

        if inst is None:
            return ori_pos

        # Check DDR design information
        pin_swap = inst.pin_swap # type: DDRPinSwap

        if not pin_swap.is_enable_swap:
            return ori_pos

        res_name = result.group(1)
        pin_name = result.group(3).replace("A", "CA")
        pin_type = result.group(4)
        bit = int(result.group(5))

        gp_num = -1

        match pin_type:
            case "DM":
                gp_num = bit

            case "DQ":
                gp_num = bit // DDRPinSwap.DQ_GROUP_COUNT

            case "CA":
                gp_num = -1

        cur_pos = ""

        if gp_num >= 0:
            cur_pos = pin_swap.get_dq_dm_pin_pos_by_pin_name(gp_num, pin_name)
        else:
            cur_pos = pin_swap.get_ca_pin_pos_by_pin_name(pin_name)
            cur_pos = cur_pos.replace("CA", "A")

        cur_pos = res_name + "_" + cur_pos

        return self.ddr_pad2ori_pin_name.get(cur_pos, ori_pos)

    def get_ddr_pad_name_by_type(self, ddr_pad_type: str, ddr_def: str):
        src_name = "DDR" if self.is_ti180_device() else ddr_def.replace("_", "")
        ddr_pad_type = ddr_pad_type.replace("CA", "A")
        return src_name + "_" + ddr_pad_type

    def get_ddr_swap_pins(self, inst: DDRAdvance) -> Dict[str, str]:
        """
        Get current pad position to DDR pad name (DQ/DM/CA)

        :param inst: DDR Instance
        :type inst: DDRAdvance
        """
        pin_swap = inst.pin_swap
        pad_pos2pad_name: Dict[str, str] = {}

        # DQ/DM pin
        for num in range(pin_swap.num_of_dm):
            for param in pin_swap.get_group_params(num):
                pad_pos = self.get_ddr_pad_name_by_type(param.name, inst.ddr_def)
                cur_pad = self.get_ddr_pad_name_by_type(param.value, inst.ddr_def)
                pad_pos2pad_name[pad_pos] = cur_pad

        # CA pin
        for param in pin_swap.get_ca_params():
            assert isinstance(param.value, str)
            pad_pos = self.get_ddr_pad_name_by_type(param.name, inst.ddr_def)
            cur_pad = self.get_ddr_pad_name_by_type(param.value, inst.ddr_def)
            pad_pos2pad_name[pad_pos] = cur_pad

        return pad_pos2pad_name

    def is_ti180_device(self):
        assert self.design is not None and self.design.device_db is not None
        device_db = self.design.device_db
        mapped_name: str = device_db.get_device_mapped_name()

        if mapped_name.startswith("opx_333x642_b40_d20"):
            return True

        return False

    def is_ddr_swap_pins(self, pad_name: str):
        pad_name_pattern = r"(DDR([0-9]+)?)_((DQ|DM|A)\[([0-9]+)\])"
        return re.match(pad_name_pattern, pad_name) is not None

    def get_pll_adv_external_clkin_index(self, pll_dev_service, ref_clock_id: "PLLAdvance.RefClockIDType") -> Union[int, None]:
        """
        Refers to PLLAdvance.get_external_clkin_index function
        """
        # It just direct mapping for ref_clock_id to hw_index for Trion
        return int(ref_clock_id)

    def get_external_clkin_index(self, pll_dev_service, ref_clock_id: PLLComplex.RefClockIDType, pll_def: str):
        '''
        Refers to PLLComplex.get_external_clkin_index function
        '''
        index = None

        # Only if the ref id is of external type, then we return a meaningful value
        if ref_clock_id.is_core_clock():
            return index

        core_pin_cnt, ext_pin_cnt = pll_dev_service.get_ref_clk_type_count(pll_def)

        refclk_id_2_hw_clock_idx = {}
        if core_pin_cnt == 2 and ext_pin_cnt == 2:
            refclk_id_2_hw_clock_idx = {
                PLLComplex.RefClockIDType.ext_clock_0: 0,
                PLLComplex.RefClockIDType.ext_clock_1: 1,
            }
        elif core_pin_cnt == 1 and ext_pin_cnt == 3:
            refclk_id_2_hw_clock_idx = {
                PLLComplex.RefClockIDType.ext_clock_0: 0,
                PLLComplex.RefClockIDType.ext_clock_1: 1,
                PLLComplex.RefClockIDType.ext_clock_2: 2
            }

        if refclk_id_2_hw_clock_idx:
            index = refclk_id_2_hw_clock_idx.get(ref_clock_id, None)

        return index

    def get_mipi_instance(self, target_res: str, block_type: str) -> str:
        """
        Get MIPI instance name by checking if:

        - Target resource is used by MIPI/ PLL SSC, or
        - Resource is used as reference clock resource by MIPI/ PLL SSC

        :param target_res: Resource name
        :type target_res: str
        :param block_type: device block type
        :type block_type: str
        :return: Block instance name if found, otherwise empty string
        :rtype: str
        """
        assert self.design is not None, f"design information is not set"
        inst = ""

        mipi_reg = self.get_mipi_reg()

        if mipi_reg is None:
            return inst

        if "mipi" in block_type:
            inst = self.get_block_inst_by_resource(mipi_reg, target_res)

            # try another mipi_reg (tx180 only)
            if inst == "" and self.is_titanium():
                if mipi_reg != self.design.mipi_hard_dphy_reg and self.design.mipi_hard_dphy_reg is not None:
                    mipi_reg = self.design.mipi_hard_dphy_reg
                    inst = self.get_block_inst_by_resource(mipi_reg, target_res)

                # Can be used by PLLSSC (MIPI Resource)
                if inst == "" and self.design.pll_ssc_reg is not None:
                    inst = self.get_block_inst_by_resource(self.design.pll_ssc_reg, target_res)

        else:
            # Check reference clock resource
            mipi_reg = None

            if self.is_titanium() and self.design.mipi_hard_dphy_reg is not None:
                mipi_reg = self.design.mipi_hard_dphy_reg
            elif self.design.mipi_reg is not None:
                mipi_reg = self.design.mipi_reg

            if mipi_reg is None:
                return inst

            inst = self.get_mipi_res_clk_by_resource(mipi_reg, target_res)

            # Also check PLL SSC Reference Clock
            if inst == "" and self.is_titanium() and self.design.pll_ssc_reg is not None:
                inst = self.get_mipi_res_clk_by_resource(self.design.pll_ssc_reg, target_res)

        return inst

    def get_mipi_res_clk_by_resource(self, blk_reg: PeriDesignRegistry, target_res: str) -> str:
        """
        Get block instance name by checking ref clock

        :param blk_reg: MIPI/ PLL SSC Registry
        :type blk_reg: PeriDesignRegistry
        :param target_res: Resource name
        :type target_res: str
        :return: Block instance name if found, otherwise empty string
        :rtype: str
        """
        inst = ""
        if not isinstance(blk_reg, (MIPIHardDPHYRegistry, MIPIRegistry, PLLSSCRegistry)):
            return inst

        for blk_inst in blk_reg.get_all_inst():
            if self.is_mipi_ref_clk(blk_inst, target_res):
                inst = blk_inst.name
                break

        return inst

    def get_mipi_reg(self):
        assert self.design is not None, f"design information is not set"
        mipi_reg = None

        if self.design.mipi_reg is not None:
            mipi_reg = self.design.mipi_reg

        elif self.design.mipi_dphy_reg is not None:
            mipi_reg = self.design.mipi_dphy_reg

        elif self.design.mipi_hard_dphy_reg is not None:
            mipi_reg = self.design.mipi_hard_dphy_reg

        return mipi_reg

    def get_ddr_instance(self, pad_name: str, target_device: str):
        """
        Get ddr instance by pad name and device name
        """
        assert self.design is not None, f"design information is not set"
        inst_name = ""
        ddr_reg = self.design.ddr_reg

        if ddr_reg is None:
            return inst_name

        io_pad = self._device.get_io_pad() # type: Optional[DeviceIO]

        if io_pad is None:
            return ""

        inst = ddr_reg.get_inst_by_device_name(target_device) # type: Optional[DDR| DDRAdvance]

        instances: List = io_pad.find_instance(pad_name)

        if len(instances) == 1 and inst is not None:
            pad_info_obj = io_pad.find_pad(pad_name) # type: Optional[IOPadInfo]
            assert pad_info_obj is not None

            if self.design.is_block_supported(PeriDesign.BlockType.adv_ddr):
                blk_type = DeviceDBService.BlockType.DDR_ADV
            else:
                blk_type = DeviceDBService.BlockType.DDR

            dbi = DeviceDBService(self._device)
            dev_service = dbi.get_block_service(blk_type) # type: Optional[DDRService]
            assert dev_service is not None

            ext_name = dev_service.get_ddr_pad_name_extension(pad_info_obj, inst)  # type: ignore

            if ext_name != "":
                inst_name = inst.name

        return inst_name

    @staticmethod
    def get_block_inst_by_resource(block_reg:PeriDesignRegistry, res: str) -> str:
        inst = ""

        for blk_inst in block_reg.get_all_inst():
            res_name = blk_inst.get_device()
            if res == res_name:
                inst = blk_inst.name
                break

        return inst

    def get_jtag_instance_list(self) -> List[str]:
        """
        Get a list of jtag that has assigned resource
        """
        assert self.design is not None, f"design information is not set"
        inst_list: List[str] = []
        jtag_reg = self.design.jtag_reg

        if jtag_reg is None:
            return inst_list

        for inst in jtag_reg.get_all_inst():
            res_name = inst.get_device()
            if res_name != "":
                inst_list.append(inst.name)

        return inst_list

    def get_lane_based_ref_clk_by_design(self, blk_reg: PeriDesignRegistry, pad_name: str, target_res: str,
                                         blk_type: DeviceDBService.BlockType) -> str:
        """
        Get block instance name if the target resource is used as reference clock in Quad PCIe

        :param blk_reg: Lane based Registry
        :type blk_reg: PeriDesignRegistry
        :param target_res: Resource name
        :type target_res: str
        :return: Block instance name if found, otherwise empty string
        :rtype: str
        """
        inst = ""

        for blk_inst in blk_reg.get_all_inst():
            if self.is_lane_based_ref_clk(blk_inst, pad_name, target_res, blk_type):
                inst = blk_inst.name
                break

        return inst

    def get_quad_pcie_ref_clk_by_design(self, blk_reg: PeriDesignRegistry, pad_name: str, target_res: str) -> str:
        """
        Get block instance name if the target resource is used as reference clock in Quad PCIe

        :param blk_reg: Quad PCIe Registry
        :type blk_reg: PeriDesignRegistry
        :param target_res: Resource name
        :type target_res: str
        :return: Block instance name if found, otherwise empty string
        :rtype: str
        """
        inst = ""

        for blk_inst in blk_reg.get_all_inst():
            if self.is_quad_pcie_ref_clk(blk_inst, pad_name, target_res):
                inst = blk_inst.name
                break

        return inst

    def get_quad_pcie_perstn_by_design(self, blk_reg: PeriDesignRegistry, target_res: str) -> str:
        """
        Get block instance name if the target resource is used as PERSTn in Quad PCIe

        :param blk_reg: Quad PCIe Registry
        :type blk_reg: PeriDesignRegistry
        :param target_res: Resource name
        :type target_res: str
        :return: Block instance name if found, otherwise empty string
        :rtype: str
        """
        inst = ""

        for blk_inst in blk_reg.get_all_inst():
            if self.is_quad_pcie_perstn(blk_inst, target_res):
                inst = blk_inst.name
                break

        return inst

    def get_quad_device_blk_type(self, quad_type: QuadType):
        blk_type = None

        match quad_type:
            case QuadType.lane_10g:
                blk_type = DeviceDBService.BlockType.QUAD_LANE_10G
            case QuadType.lane_1g:
                blk_type = DeviceDBService.BlockType.QUAD_LANE_1G
            case QuadType.raw_serdes:
                blk_type = DeviceDBService.BlockType.QUAD_LANE_RAW_SERDES
            case QuadType.quad_pcie:
                blk_type = DeviceDBService.BlockType.QUAD_PCIE

        return blk_type

    def get_lane_based_instance(self, pad_name: str, target_device: str, protocal: QuadType) -> str:
        """
        Get a list of Quad PCIe that has assigned resource
        """
        assert self.design is not None
        inst_name = ""
        reg = self.design.common_quad_lane_reg

        if reg is None:
            return inst_name

        inst: Optional[QuadLaneCommon] = reg.get_inst_by_device_name(target_device)

        blk_type = None
        lane_reg = None
        is_ref_clk = False

        if protocal.is_quad_based():
            return inst_name

        match protocal:
            case QuadType.lane_10g:
                blk_type = DeviceDBService.BlockType.QUAD_LANE_10G
                lane_reg = self.design.lane_10g_reg
            case QuadType.lane_1g:
                blk_type = DeviceDBService.BlockType.QUAD_LANE_1G
                lane_reg = self.design.lane_1g_reg
            case QuadType.raw_serdes:
                blk_type = DeviceDBService.BlockType.QUAD_LANE_RAW_SERDES
                lane_reg = self.design.raw_serdes_reg
            case _:
                return inst_name

        if lane_reg is None:
            return inst_name

        if inst is not None:
            dev_service_adv = self.dbi.get_block_service(blk_type) # type: QuadLane10GDeviceService

            # Skip for external ref clock (handled by get_lane_based_ref_clk_by_design)
            param_info = inst.get_param_info()
            param_id = QuadConfigParamInfo.Id.PMA_CMN__cmn_plllc_gen_preg__cmn_plllc_pfdclk1_sel_preg
            options = param_info.get_valid_setting(param_id)
            assert isinstance(options, list)
            for option in options:
                assert isinstance(option, str)
                ref_clk_str = option.replace(' ', "").upper()
                ref_clk_pins = dev_service_adv.get_external_refclk_pins(target_device, ref_clk_str)
                if pad_name in ref_clk_pins:
                    inst_name = ""
                    is_ref_clk = True
                    break

            if 'CAL_RES' in pad_name:
                inst_name = inst.name

            if not is_ref_clk:
                quad_res = self.get_quad_res_name_by_pad_name(pad_name)
                quad_num, lane_num = QuadResService.break_res_name(quad_res)

                # Get instance if the quad_res is Qn_LNn
                if quad_num != -1 and lane_num != -1:
                    inst = lane_reg.get_inst_by_device_name(quad_res)
                    inst_name = "" if inst is None else inst.name

        # Check ref clock
        if is_ref_clk:
            inst_name = self.get_lane_based_ref_clk_by_design(reg, pad_name, target_device,
                                                              blk_type)

        return inst_name

    def get_quad_pcie_instance(self, pad_name: str, target_device: str) -> str:
        """
        Get a list of Quad PCIe that has assigned resource
        """
        assert self.design is not None
        inst_name = ""
        quad_pcie_reg = self.design.quad_pcie_reg

        if quad_pcie_reg is None:
            return inst_name

        inst: Optional[QuadPCIE] = quad_pcie_reg.get_inst_by_device_name(target_device)

        if inst is not None:
            assert isinstance(inst, QuadPCIE)
            is_ref_clk = False

            blk_type = DeviceDBService.BlockType.QUAD_PCIE
            dev_service_adv = self.dbi.get_block_service(blk_type) # type: QuadPCIEDeviceService
            param_id = PCIEConfigParamInfo.Id.PMA_CMN__cmn_plllc_gen_preg__cmn_plllc_pfdclk1_sel_preg
            options = inst.param_info.get_valid_setting(param_id)
            assert isinstance(options, list)

            # Skip for external ref clock (handled by get_quad_pcie_ref_clk_by_design)
            for option in options:
                ref_clk_str = option.replace(' ', "").upper()
                ref_clk_pins = dev_service_adv.get_external_refclk_pins(
                    target_device, ref_clk_str)
                if pad_name in ref_clk_pins:
                    inst_name = ""
                    is_ref_clk = True
                    break

            if not is_ref_clk:
                inst_name = inst.name

        # Check ref clock and perstn resource
        if inst_name == "":
            inst_name = self.get_quad_pcie_ref_clk_by_design(quad_pcie_reg, pad_name, target_device)
            if inst_name == "":
                inst_name = self.get_quad_pcie_perstn_by_design(quad_pcie_reg, target_device)

        return inst_name

    def get_gpio_res_block_name(self, res_name: str, pad_name: str) -> str:
        """
        Get GPIO resource related block name and its pin name list
        """
        assert self.design is not None, f"design information is not set"
        if self.is_titanium():
            res_svr = HIOResService()
            res_svr.build(self.design)
            block_obj_list = res_svr.find_resource_user(res_name)

            if len(block_obj_list) > 0:
                for user_inst in block_obj_list:
                    if isinstance(user_inst, GPIOComplex):
                        is_diff_tl = user_inst.is_gpio_differential_stl_io_std()
                        if user_inst.gpio_def == res_name or is_diff_tl:
                            return user_inst.name

                    elif user_inst is not None and hasattr(user_inst, "get_device"):
                        device = getattr(user_inst, "get_device")()
                        if device != "":
                            return user_inst.name

        else:
            gpio_reg = self.design.gpio_reg
            assert gpio_reg is not None
            block_obj = gpio_reg.get_gpio_by_device_name(res_name)

            if block_obj is None:
                block_obj = gpio_reg.get_gpio_by_device_name(pad_name)

            if block_obj is None:
                if self.design.lvds_reg is not None and self.design.lvds_reg.is_lvds_mode_used(res_name):
                    lvds_mode_base_res_name = self._device.get_base_ins_name(res_name)
                    block_obj = self.design.lvds_reg.get_inst_by_device_name(lvds_mode_base_res_name)
                    return block_obj.name if block_obj is not None else "" # type: ignore
            else:
                return block_obj.name

        return ""

    def _get_prog_info(self):
        '''
        Refers to PinoutReport

        Read the project setting since there's a possibility that
        the file changes between loading the device and generating
        the pinout report.

        :return the Programming section of the project file
        '''
        assert self.design is not None, f"design information is not set"

        prog_info = None

        try:
            setting = efxproj_set.EfxProjectSetting.build(
                self.design.name, self.design.location,
                [efxproj_set.EfxProjectSetting.LoadFilterType.prog])

            if setting is not None:
                prog_info = setting.get_prog_info()

        except app_excp.PTFileException as exc:
            self.logger.debug(f"File error : {exc.msg}")

        return prog_info

    def _is_coldboot_selected(self):
        '''
        Refer to PinoutReport
        Read in the project setting (from prog_info) and
        get the settings of coldboot.

        :return True if coldboot feature was set to on
        '''
        is_coldboot = False

        try:
            prog_info = self._get_prog_info()

            if prog_info is not None:
                cold_boot_setting = prog_info.get("cold_boot", "")

                if cold_boot_setting == "on":
                    is_coldboot = True

        except app_excp.PTFileException as exc:
            self.logger.debug(f"File error : {exc.msg}")

        return is_coldboot

    def get_all_iobank2pin_map(self) -> Dict[str, List[str]]:
        if self.iobank2pin_map == {}:
            pin_map = self.get_all_pin2pad_map()
            for pin in pin_map.values():
                pad_name: str = pin.get_pad_name()
                pin_name: str = pin.get_package_pin_name()
                io_bank: str = self.get_iobank_by_pad_name(pad_name)
                self.add_iobank2pin_map(io_bank, pin_name)

        return self.iobank2pin_map

    def add_iobank2pin_map(self, io_bank: str, pin_name: str):
        if io_bank == "":
            return

        pin_list = self.iobank2pin_map.get(io_bank, [])

        if pin_name in pin_list:
            return

        pin_list.append(pin_name)
        self.iobank2pin_map[io_bank] = pin_list

    def get_iobank_by_pad_name(self, pad_name: str) -> str:
        io_bank = ""
        io_pad = self._device.get_io_pad() # type: Optional[DeviceIO]

        if io_pad is None:
            return io_bank

        pad_info_obj = io_pad.find_pad(pad_name) # type: Optional[IOPadInfo]

        if pad_info_obj is not None:
            io_bank = pad_info_obj.get_bank_name()
            if io_bank in self.invisible_bank_names:
                return ""

        return io_bank

    def get_iobank_by_pin(self, pin_name: str) -> str:
        pin_map = self.get_all_pin2pad_map()
        pin = pin_map.get(pin_name, None)

        if pin is None:
            return ""

        pad_name = pin.get_pad_name()
        return self.get_iobank_by_pad_name(pad_name)

    def get_iobank_voltage_by_pad_name(self, pad_name: str) -> str:
        io_bank_voltage = ""
        io_pad = self._device.get_io_pad() # type: Optional[DeviceIO]

        if io_pad is None:
            return io_bank_voltage

        pad_info_obj = io_pad.find_pad(pad_name) # type: Optional[IOPadInfo]

        if pad_info_obj is not None:
            io_bank_voltage = self._get_io_bank_voltage(io_pad, pad_info_obj)

        return io_bank_voltage

    def get_iobank_info_by_pad_name(self, pad_name: str) -> Tuple[str, str, str]:
        """
        Get I/O bank information

        :param pad_name: Pad Name
        :return tuple of I/O bank information
        """
        io_bank = ""
        io_bank_voltage = ""
        io_std = ""

        io_pad = self._device.get_io_pad() # type: Optional[DeviceIO]

        if io_pad is None:
            return io_bank, io_std, io_bank_voltage

        pad_info_obj = io_pad.find_pad(pad_name) # type: Optional[IOPadInfo]
        instances = io_pad.find_instance(pad_name)

        if pad_info_obj is not None:
            # I/O Bank Name
            io_bank = pad_info_obj.get_bank_name()

            # I/O Bank Voltage
            io_bank_voltage = self._get_io_bank_voltage(io_pad, pad_info_obj)

            # I/O Bank Standard
            ins2io_std = {}
            ins2io_std = self._get_io_std(
                pad_info_obj, instances)

            if len(ins2io_std) > 1:
                for ins_name in sorted(instances,  key=pt_util.natural_sort_key_for_list):
                    io_std = ins2io_std[ins_name]
            else:
                for tmp_io_std in ins2io_std.values():
                    io_std = tmp_io_std

        return io_bank, io_std, io_bank_voltage

    def get_io_standard_by_pad_name(self, pad_name: str) -> str:
        io_std = ""
        io_pad = self._device.get_io_pad() # type: Optional[DeviceIO]

        if io_pad is None:
            return io_std

        pad_info_obj = io_pad.find_pad(pad_name) # type: Optional[IOPadInfo]
        instances = io_pad.find_instance(pad_name)

        if pad_info_obj is not None:
            ins2io_std = self._get_io_std(pad_info_obj, instances)

            if len(ins2io_std) > 1:
                for ins_name in sorted(instances,  key=pt_util.natural_sort_key_for_list):
                    io_std = ins2io_std[ins_name]
            else:
                for tmp_io_std in ins2io_std.values():
                    io_std = tmp_io_std
        return io_std

    def get_signal_name_by_pad_name(self, pad_name: str) -> str:
        signal_name = ""
        ins2signal_names = self.get_ins2signal_names_by_pad_name(pad_name)

        if len(ins2signal_names) == 1:
            signal_name = list(ins2signal_names.values())[0]

        return signal_name

    def get_ins2signal_names_by_pad_name(self, pad_name: str):
        """
        JTAG has pad with multiple instances but
        we don't care about the instances when printing
        out the signal name on the JTAG pins.
        """
        signal_names = {}

        pad_info_obj = self._device.find_pad(pad_name)  # type: Optional[IOPadInfo]
        if pad_info_obj is None:
            return signal_names

        io_pad = self._device.get_io_pad() # type: Optional[DeviceIO]
        assert io_pad is not None

        for ins_name in io_pad.find_instance(pad_name):
            ins_obj = self._device.find_instance(ins_name)
            if ins_obj is not None:
                user_ins_name = self._get_user_instance_name(pad_info_obj, ins_obj)
                # self.logger.debug("Saving signal name ins: {} to {}".format(
                #    ins_obj.get_name(), user_ins_name))

                # Map of device instance name to design instance name
                signal_names[ins_obj.get_name()] = user_ins_name

        return signal_names

    def get_direction_by_pad_name(self, pad_name: str):
        dir_str = ""
        ins2dir_types = self.get_ins2dir_types_by_pad_name(pad_name)

        if len(ins2dir_types) == 1:
            dir_str = list(ins2dir_types.values())[0]

        return dir_str

    def get_direction_by_pin_name(self, pin_name: str) -> str:
        pad_name = self.get_pad_name_by_pin(pin_name)

        if pad_name == "":
            return ""

        return self.get_direction_by_pad_name(pad_name)

    def get_ins2dir_types_by_pad_name(self, pad_name: str):
        ins2dir_types = {}
        pad_info_obj = self._device.find_pad(pad_name)  # type: Optional[IOPadInfo]
        if pad_info_obj is None:
            return ins2dir_types

        pad_dir = pad_info_obj.get_direction()

        if pad_dir == IOPadInfo.PadDirection.configurable:
            # If it is a configurable pad, then there's a chance that
            # there's multiple dirction for each instance
            ins2dir_types = self._get_pin_direction(
                pad_info_obj, pad_name)

        return ins2dir_types

    def get_pull_type_by_pad_name(self, pad_name: str):
        signal_name = ""
        ins2pull_type = self.get_ins2pull_type_by_pad_name(pad_name)

        if len(ins2pull_type) == 1:
            signal_name = list(ins2pull_type.values())[0]

        return signal_name

    def get_ins2pull_type_by_pad_name(self, pad_name: str):
        pull_types = {}

        pad_info_obj = self._device.find_pad(pad_name)  # type: Optional[IOPadInfo]
        if pad_info_obj is None:
            return pull_types

        io_pad = self._device.get_io_pad() # type: Optional[DeviceIO]
        assert io_pad is not None


        for ins_name in io_pad.find_instance(pad_name):
            ins_obj = self._device.find_instance(ins_name)
            if ins_obj is not None:
                pull_type = self._get_ins_pulltype(pad_info_obj, ins_obj)
                # self.logger.debug("Saving signal name ins: {} to {}".format(
                #    ins_obj.get_name(), user_ins_name))

                # Map of device instance name to design instance name
                pull_types[ins_obj.get_name()] = pull_type

        return pull_types

    def replace_spi_flash_embedded_ins_names(self, spi_obj: SPIFlash, gpio_ins: GPIO):
        '''
        Modify the passed gpio_ins pin name to use the user pin name
        saved in the spi_obj.gen_pin. This does not modify the gpio instance
        name.

        :param spi_obj: SPIFlash design instance
        :param gpio_ins: The gpio instance associated to the used config
        '''

        ins_gen_pin = spi_obj.gen_pin

        if ins_gen_pin is None:
            return

        template_pin_name = gpio_ins.name

        # Find the gen pin with the same name as the instance name
        gpin = ins_gen_pin.get_pin_by_type_name(template_pin_name)

        # This is very specific to SPI where we expect its very simple.
        # We also don't change the instance name, just the pin name.
        if gpin is not None and gpin.name != "":
            # Only change the instance name but not the pin names since
            # we don't care
            gpio_ins.name = gpin.name

    def _get_io_bank_voltage(self, io_pad: DeviceIO, pad_info: IOPadInfo) -> str:
        '''
        Refers to the function in Pinout Report

        Get the selected io standard for the specified
        bank name.

        :param io_pad: DeviceIO object
        :param pad_info: IOPadInfo object

        :return io voltage name configured to the specified
                batx60_spi_flashnk name
        '''
        assert self.design is not None, f"design information is not set"
        io_volt_name = ""

        # We only print out the io standard on configurable
        # i/o bank. If the i/o bank has no selection (fixed), then
        # an empty string is returned
        if pad_info is None:
            return io_volt_name

        pad_name = pad_info.get_pad_name()

        # PT-1622 VCCA should have fixed voltage according to the timing model
        if pad_name.startswith("VCCA_") or pad_name == "VCC":
            return io_volt_name

        bank_name = pad_info.get_bank_name()

        bank_info = io_pad.get_bank_info(bank_name) # type: Optional[IOBankInfo]

        if bank_info is None:
            return io_volt_name

        io_std_list = bank_info.get_io_standards()

        if len(io_std_list) > 1 and self.design.device_setting is not None:
            iob_reg = self.design.device_setting.iobank_reg

            if iob_reg is None:
                return io_volt_name

            iob_obj = iob_reg.get_iobank_by_name(bank_name) # type: Optional[IOBank]

            if iob_obj is None:
                return io_volt_name

            raw_name = iob_obj.get_raw_io_standard()
            io_volt_name = iob_obj.get_user_io_voltage()

            # print("REading pad {} bank: {} io_std {}".format(
            # pad_info.get_pad_name(), bank_name,
            # io_volt_name))

            # Unexpected error when the chosen
            # io standard name is not listed in
            # the list of io standard for the bank
            if raw_name not in io_std_list:
                raise ValueError(
                    'Configured io voltage {} not'
                    ' defined in bank {}'.format(
                        io_volt_name, bank_name))

        return io_volt_name

    def _get_timing_model_voltage(self):
        io_volt_name = ""

        timing_model = self._device.get_timing_model()
        if timing_model is None:
            return io_volt_name

        sel_model = timing_model.get_selected_model()
        assert sel_model is not None

        io_volt_name = sel_model.get_voltage()
        return io_volt_name

    def _get_io_std(self, pad_info_obj: IOPadInfo, instances: List[str]):
        '''
        Refers to PinoutReport._get_pull_type_and_io_std()
        Get the pull type string for a specific pin number
        (pin_name) and pin name (pad_name).

        :param instances: The list of instance names

        :return map of device instance name to io std string
        '''
        ins2iostd_map = {}

        # Iterate through the list of device instance name
        for ins_name in instances:
            # Get the reference
            ins_obj = self._device.find_instance(ins_name)

            if ins_obj is not None:
                io_std = self._get_ins_io_std(
                    pad_info_obj, ins_obj)

                ins2iostd_map[ins_obj.get_name()] = io_std

        return ins2iostd_map

    def _get_ins_io_std(self, pad_info_obj: IOPadInfo, ins_obj: PeripheryBlockInstance):
        '''
        Refers to PinoutReport._get_ins_pull_type_and_io_std()

        :param ins_obj: Instance object

        :return io std
        '''
        assert self.design is not None, f"design information is not set"
        io_std_str = ""

        ref_name = ins_obj.get_ref_name()

        if ref_name in DeviceDBService.block_str2type_map:
            dev_blk_type = DeviceDBService.block_str2type_map[ref_name]
            match dev_blk_type:
                case DeviceDBService.BlockType.GPIO:
                    gpio_reg = self.design.get_block_reg(PeriDesign.BlockType.gpio)
                    assert gpio_reg is not None

                    gpio_obj = gpio_reg.get_gpio_by_device_name(ins_obj.get_name()) # type: ignore
                    if gpio_obj is not None:
                        io_std_str = gpio_obj.io_standard

                case DeviceDBService.BlockType.LVDS_RX | DeviceDBService.BlockType.LVDS_TX:
                    io_std_str = self._get_lvds_iostd(pad_info_obj, ins_obj)

                case DeviceDBService.BlockType.HSIO | DeviceDBService.BlockType.HSIO_MAX:
                    io_std_str = self._get_hsio_iostd(pad_info_obj, ins_obj)

        return io_std_str

    def _is_lvds_pad_type_p(self, pad_info_obj: IOPadInfo):
        '''
        :return None if the pad has no pin name associated
                True: If the pin is of type P (RXP/TXP)
                False: If the pin is NOT of type P
        '''

        is_p = None

        if pad_info_obj is not None:

            pin_name = pad_info_obj.get_pin_name()
            if pin_name is not None:
                is_p = False

                # If is_p is True, it indicates that we are looking
                # at the P entry for the pad
                if pin_name == "TXP" or pin_name == "RXP":
                    is_p = True

        return is_p

    def _is_hsio_pad_type_p(self, pad_info_obj: IOPadInfo):
        '''
        :return None if the pad has no pin name associated
                True: If the pin is of type P (RXP/TXP)
                False: If the pin is NOT of type P
        '''

        is_p = None

        if pad_info_obj is not None:

            pin_name = pad_info_obj.get_pin_name()
            if pin_name is not None:
                is_p = False

                # If is_p is True, it indicates that we are looking
                # at the P entry for the pad
                if pin_name == "P":
                    is_p = True

        return is_p

    def _get_lvds_iostd(self, pad_info_obj: IOPadInfo, dev_ins: PeripheryBlockInstance):
        '''
        :param dev_ins: Device instance object
        :return string: pull type, io standard - associated to the lvds
                gpio instance
        '''
        io_std_str = ""
        inst = self._get_lvds_design_ins(pad_info_obj, dev_ins)

        if isinstance(inst, GPIO):
            io_std_str = inst.io_standard

        return io_std_str

    def _get_hsio_iostd(self, pad_info_obj: IOPadInfo, dev_ins: PeripheryBlockInstance):
        '''
        :param dev_ins: Device instance object
        :return string: pull type, io standard - associated to the lvds
                gpio instance
        '''
        assert self.design is not None
        io_std_str = ""

        # Get the pad pin name
        is_p = self._is_hsio_pad_type_p(pad_info_obj)

        # Get the HSIO Device service
        dev_svc = DeviceDBService(self._device)
        hsio_dev_svc = dev_svc.get_block_service(
            DeviceDBService.BlockType.HSIO)
        if hsio_dev_svc is None:
            return io_std_str


        reg = self.design.get_block_reg(self.design.gpio_type)
        assert reg is not None

        for inst in reg.get_all_lvds_gpio(): # type: ignore
            inst: GPIOComplex
            # This is the name with extension
            hsio_dev_ins = inst.gpio_def

            # This is the device base name without extension
            lvds_dev_base_name = self._device.get_base_ins_name(
                hsio_dev_ins)

            if dev_ins.get_name() != lvds_dev_base_name or is_p is None:
                continue

            hsio_type = hsio_dev_svc.get_instance_type_by_pad_usage(
                hsio_dev_ins, self._device)

            lvds_dev_base_name = self._device.get_base_ins_name(
                hsio_dev_ins)

            is_diff_tl: bool = inst.is_gpio_differential_stl_io_std()
            if (is_p and (hsio_type == HSIOService.HSIOBlockModeType.lvttl1 or
                            hsio_type == HSIOService.HSIOBlockModeType.gpiox4p)) or \
                    (not is_p and (hsio_type == HSIOService.HSIOBlockModeType.lvttl2 or
                                    hsio_type == HSIOService.HSIOBlockModeType.gpiox4n)) or\
                    (not is_p and is_diff_tl):

                io_std_str = inst.io_standard
                break

        return io_std_str

    def _get_ins_pulltype(self, pad_info_obj: IOPadInfo, ins_obj: PeripheryBlockInstance):
        assert self.design is not None
        pull_type_str = ""

        ref_name = ins_obj.get_ref_name()

        if ref_name not in DeviceDBService.block_str2type_map:
            return pull_type_str

        dev_blk_type = DeviceDBService.block_str2type_map[ref_name]

        match dev_blk_type:
            case DeviceDBService.BlockType.GPIO:
                pull_type_str = self._get_gpio_pulltype(ins_obj)
            case DeviceDBService.BlockType.LVDS_RX| DeviceDBService.BlockType.LVDS_TX:
                pull_type_str = self._get_lvds_pulltype(pad_info_obj, ins_obj)
            case DeviceDBService.BlockType.HSIO| DeviceDBService.BlockType.HSIO_MAX:
                pull_type_str = self._get_hsio_pulltype(pad_info_obj, ins_obj)

        return pull_type_str

    def _get_gpio_pulltype(self, ins_obj: PeripheryBlockInstance):
        assert self.design is not None
        pull_type_str = ""
        reg = self.design.get_block_reg(PeriDesign.BlockType.gpio)
        assert reg is not None

        inst = reg.get_inst_by_device_name(ins_obj.get_name()) # type: ignore

        if inst is None:
            # global Non-configured instance setting
            pull_type_str = self._get_gpio_unused_config_pull_type(reg.global_unused_config) # type: ignore

        else:
            inst: GPIO
            io_std_str = inst.io_standard
            if io_std_str == "":
                io_std_str = inst.get_default_io_standard(self._device)

            match inst.mode:
                case inst.PadModeType.input| inst.PadModeType.inout:
                    input_config = inst.input

                    if input_config is not None:
                        pull_option = input_config.pull_option

                        if pull_option == input_config.PullOptType.weak_pullup or \
                                pull_option == input_config.PullOptType.weak_pulldown:

                            pull_type_str = input_config.pullopt2str_map[pull_option]

                case inst.PadModeType.none:
                    if inst.unused_config is not None:
                        # This is an instance set by user to mode none
                        pull_type_str = self._get_gpio_unused_config_pull_type(
                            inst.unused_config)

        return pull_type_str

    def _get_lvds_pulltype(self, pad_info_obj: IOPadInfo, ins_obj: PeripheryBlockInstance):
        assert self.design is not None
        pull_type_str = ""

        inst = self._get_lvds_design_ins(pad_info_obj, ins_obj)
        if isinstance(inst, GPIO):
            if inst.mode == inst.PadModeType.input or inst.mode == inst.PadModeType.inout:
                input_config = inst.input

                if input_config is None:
                    return pull_type_str

                pull_option = input_config.pull_option

                if pull_option == input_config.PullOptType.weak_pullup:
                    # TODO: check for pulldown when it is updated such that it is
                    # supported for LVDS
                    pull_type_str = input_config.pullopt2str_map[pull_option]

        return pull_type_str

    def _get_hsio_pulltype(self, pad_info_obj: IOPadInfo, ins_obj: PeripheryBlockInstance):
        assert self.design is not None
        '''
        :param dev_ins: Device instance object
        :return string: pull type, io standard - associated to the lvds
                gpio instance
        '''
        pull_type_str = ""

        # Get the pad pin name
        is_p = self._is_hsio_pad_type_p(pad_info_obj)

        # Get the HSIO Device service
        hsio_dev_svc = self.dbi.get_block_service(DeviceDBService.BlockType.HSIO)

        if hsio_dev_svc is None or is_p is None:
            return pull_type_str

        reg = self.design.get_block_reg(PeriDesign.BlockType.gpio)
        assert reg is not None

        for inst in reg.get_all_lvds_gpio(): # type: ignore
            inst: GPIOComplex
            # This is the name with extension
            lvds_dev_ins = inst.gpio_def

            # This is the device base name without extension
            lvds_dev_base_name = self._device.get_base_ins_name(
                lvds_dev_ins)

            if ins_obj.get_name() != lvds_dev_base_name or is_p is None:
                continue

            hsio_dev_ins = inst.gpio_def

            hsio_type = hsio_dev_svc.get_instance_type_by_pad_usage(
                hsio_dev_ins, self._device)

            lvds_dev_base_name = self._device.get_base_ins_name(
                hsio_dev_ins)

            is_diff_tl = False
            if lvds_dev_base_name == ins_obj.get_name() and is_p is not None:

                # In this case, we need to align the mode to the pad that
                # we're looking at since:
                # LVTTL1 - Connects to 'P'
                # LVTTL2 - Connects to 'N'

                # Because we just want to tell whether it's gpio or not, we don't
                # need to be very specific to determine mode based on the design instance
                # If we need to really distinguish between mipi/gpio/lvds, then use the
                # call below
                #mode_str = hsio_dev_svc.get_design_instance_mode(hsio_dev_ins, self.__device, inst)
                #hsio_type = HSIOService.str2mode_map[mode_str]

                is_diff_tl: bool = inst.is_gpio_differential_stl_io_std()
                if (is_p and (hsio_type == HSIOService.HSIOBlockModeType.lvttl1 or
                                hsio_type == HSIOService.HSIOBlockModeType.gpiox4p)) or \
                        (not is_p and (hsio_type == HSIOService.HSIOBlockModeType.lvttl2 or
                                        hsio_type == HSIOService.HSIOBlockModeType.gpiox4n)) or \
                        (not is_p and is_diff_tl):

                    if inst.mode == inst.PadModeType.input or \
                            inst.mode == inst.PadModeType.inout:

                        input_config = inst.input

                        if input_config is not None:
                            pull_option = input_config.pull_option

                            if pull_option == input_config.PullOptType.weak_pullup:
                                # TODO: check for pulldown when it is updated such that it is
                                # supported for HSIO
                                pull_type_str = input_config.pullopt2str_map[pull_option]
                                break

        return pull_type_str

    def _get_pin_direction(self, pad_info_obj: IOPadInfo, pad_name: str) -> Dict[str,str]:
        io_pad = self._device.get_io_pad() # type: Optional[DeviceIO]
        assert io_pad is not None

        direction_types = {}

        for ins_name in io_pad.find_instance(pad_name):
            ins_dir = self._get_ins_direction_by_design(pad_info_obj, ins_name)
            direction_types[ins_name] = ins_dir

        return direction_types

    def _get_ins_direction_by_design(self, pad_info_obj: IOPadInfo, ins_name: str):
        assert self.design is not None
        ins_dir = ""

        ins_obj = self._device.find_instance(ins_name)
        if ins_obj is None:
            return ins_dir

        ref_name: str = ins_obj.get_ref_name()
        dev_blk_type = DeviceDBService.block_str2type_map.get(ref_name, None)

        match dev_blk_type:
            case DeviceDBService.BlockType.GPIO:
                reg = self.design.get_block_reg(self.design.gpio_type)
                assert reg is not None
                inst = reg.get_inst_by_device_name(ins_obj.get_name())
                ins_dir = "Input" if inst is None else inst.get_direction_str()

            case DeviceDBService.BlockType.LVDS_TX | DeviceDBService.BlockType.LVDS_RX:
                ins_dir = self._get_lvds_ins_direction(pad_info_obj, ins_obj)

            case DeviceDBService.BlockType.HSIO | DeviceDBService.BlockType.HSIO_MAX:
                ins_dir = self._get_hsio_ins_direction(pad_info_obj, ins_obj)

        return ins_dir

    def _get_lvds_design_ins(self, pad_info_obj: IOPadInfo, ins_obj: PeripheryBlockInstance):
        assert self.design is not None
        ins = None

        is_p = self._is_lvds_pad_type_p(pad_info_obj)
        reg = self.design.get_block_reg(self.design.lvds_type)

        if reg is None:
            return ins

        res_name_list = self.get_resource_name_by_pad_name(pad_info_obj.get_pad_name())

        is_found = False

        for res_name in res_name_list:
            inst = reg.get_inst_by_device_name(res_name) # type: ignore
            if inst is None:
                continue
            is_found = True
            ins = inst
            break

        if not is_found and is_p is not None:
            reg = self.design.get_block_reg(PeriDesign.BlockType.gpio)
            assert reg is not None

            for inst in reg.get_all_lvds_gpio(): # type: ignore
                inst: GPIOComplex
                # This is the name with extension
                res_name = inst.gpio_def

                # This is the device base name without extension
                lvds_dev_base_name = self._device.get_base_ins_name(
                    res_name)

                if ins_obj.get_name() != lvds_dev_base_name or is_p is None:
                    continue

                lvds_type = lvds_util.get_instance_lvds_type(res_name, self._device)

                if (lvds_type == lvds_util.BlockModeType.lvttl1 and is_p) or\
                    (lvds_type == lvds_util.BlockModeType.lvttl2 and not is_p):

                    ins = inst
                    is_found = True

                    self.logger.debug("Setting lvds gpio ins pin: {} dev_ins: {} type: {}".format(
                        pad_info_obj.get_pin_name(),
                        res_name, lvds_util.mode2str_map[lvds_type]))
                    break

        return ins

    def _get_lvds_ins_direction(self, pad_info_obj: IOPadInfo, ins_obj: PeripheryBlockInstance):
        ins_dir = ""
        inst = self._get_lvds_design_ins(pad_info_obj, ins_obj)

        if inst is not None:
            ins_dir = inst.get_direction_str()

        return ins_dir

    def _get_mipi_design_ins(self, ins_obj: PeripheryBlockInstance):
        assert self.design is not None

        if self.design.is_block_supported(PeriDesign.BlockType.mipi_hard_dphy):
            blk_type = PeriDesign.BlockType.mipi_hard_dphy
        else:
            blk_type = PeriDesign.BlockType.mipi

        reg = self.design.get_block_reg(blk_type)
        if reg is None:
            return
        inst = reg.get_inst_by_device_name(ins_obj.get_name())
        return inst

    def _get_mipi_user_ins(self, pad_info_obj: IOPadInfo, ins_obj: PeripheryBlockInstance):
        user_ins_name = ""

        inst = self._get_mipi_design_ins(ins_obj)
        if inst is None:
            return user_ins_name

        dev_blk_type = None
        if isinstance(inst, MIPI):
            if inst.is_tx():
                dev_blk_type = DeviceDBService.BlockType.MIPI_TX
            else:
                dev_blk_type = DeviceDBService.BlockType.MIPI_RX

        elif isinstance(inst, MIPIHardDPHYTx):
            dev_blk_type = DeviceDBService.BlockType.MIPI_TX_ADV

        elif isinstance(inst, MIPIHardDPHYRx):
            dev_blk_type = DeviceDBService.BlockType.MIPI_RX_ADV

        if dev_blk_type is None:
            return user_ins_name

        dev_service_adv = self.dbi.get_block_service(dev_blk_type) # type: MIPITxService| MIPIRxService| MIPITxServiceAdvance| MIPIRxServiceAdvance
        ext_name = dev_service_adv.get_pad_name_extension(pad_info_obj, inst) # type: ignore

        if ext_name != "":
            user_ins_name = "{}.{}".format(inst.name, ext_name)
            self.logger.debug("Setting mipi ins name: {}".format(user_ins_name))

        return user_ins_name

    def _get_ddr_design_ins(self, ins_obj: PeripheryBlockInstance):
        assert self.design is not None
        ddr_reg = self.design.get_block_reg(PeriDesign.BlockType.ddr)
        assert ddr_reg is not None
        return ddr_reg.get_inst_by_device_name(ins_obj.get_name())

    def _get_ddr_user_ins(self, pad_info_obj: IOPadInfo, ins_obj: PeripheryBlockInstance):
        '''
        :param dev_ins: Device instance object
        :return the user instance name that's associated to the device instance
                    with the pad name suffix
        '''
        assert self.design is not None
        user_ins_name = ""
        ext_name = ""

        inst = self._get_ddr_design_ins(ins_obj)

        if inst is None:
            return user_ins_name

        if self.design.is_block_supported(PeriDesign.BlockType.adv_ddr):
            blk_type = DeviceDBService.BlockType.DDR_ADV
        else:
            blk_type = DeviceDBService.BlockType.DDR

        dev_service = self.dbi.get_block_service(blk_type) # type: Optional[DDRService]
        assert dev_service is not None

        ext_name = dev_service.get_ddr_pad_name_extension(pad_info_obj, inst)

        if ext_name != "":
            user_ins_name = "{}.{}".format(inst.name, ext_name)

            self.logger.debug(
                "Setting ddr ins name: {}".format(user_ins_name))

        return user_ins_name

    def _get_hsio_user_ins(self, pad_info_obj: IOPadInfo, ins_obj: PeripheryBlockInstance):
        assert self.design is not None

        user_ins_name = ""
        is_found = False
        pin_name = ""

        # Get the pad pin name
        is_p = self._is_hsio_pad_type_p(pad_info_obj)
        res_name_list = self.get_resource_name_by_pad_name(pad_info_obj.get_pad_name())

        if is_p is not None:
            pin_name = pad_info_obj.get_pin_name()
            if pin_name is None:
                pin_name = ""

        # There's 3 main possible design for hsio (GPIO/LVDS/MIPI)
        reg_list: List[Optional[PeriDesignRegistry]] = [
            self.design.mipi_dphy_reg,
            self.design.lvds_reg
        ]

        for reg in reg_list:
            if reg is None:
                continue
            for res_name in res_name_list:
                inst = reg.get_inst_by_device_name(res_name) # type: ignore
                if inst is None:
                    continue
                is_found = True
                user_ins_name = "{}.{}".format(inst.name, pin_name)
                break
            if is_found:
                break

        # GPIO
        # dev_ins is the HSIO instance (GPIOY_PN_XX). Whereas, the name stored
        # in the shared_res2username_map has key with the GPIO-typed resource name
        # (GPIOY_P_XX / GPIOY_N_XX). Therefore, we use the io pad instance name
        # and pin to get the correct gpio resource name.
        if not is_found and is_p is not None:
            reg = self.design.get_block_reg(PeriDesign.BlockType.gpio)
            assert reg is not None

            for inst in reg.get_all_lvds_gpio(): # type: ignore
                inst: GPIOComplex
                # This is the name with extension
                res_name = inst.gpio_def

                # This is the device base name without extension
                lvds_dev_base_name = self._device.get_base_ins_name(
                    res_name)

                if ins_obj.get_name() != lvds_dev_base_name or is_p is None:
                    continue

                lvds_type = lvds_util.get_instance_lvds_type(res_name, self._device)
                is_diff_tl = inst.is_gpio_differential_stl_io_std()

                # In this case, we need to align the mode to the pad that
                # we're looking at since:
                # LVTTL1 - Connects to 'P'
                # LVTTL2 - Connects to 'N'

                if (lvds_type == lvds_util.BlockModeType.lvttl1 and is_p) or\
                        (lvds_type == lvds_util.BlockModeType.lvttl2 and not is_p) or\
                            (not is_p and is_diff_tl):
                    user_ins_name = "{}.{}".format(inst.name, pin_name)
                    is_found = True

        if not is_found and is_p is not None:
            # dev_ins is the HSIO instance (GPIOY_PN_XX). Whereas, the name stored
            # in the shared_res2username_map has key with the GPIO-typed resource name
            # (GPIOY_P_XX / GPIOY_N_XX). Therefore, we use the io pad instance name
            # and pin to get the correct gpio resource name.
            pin_name = pad_info_obj.get_pin_name()

            if pin_name != "" and pin_name is not None:
                # P - _P_, N - _N_
                hio_svc = HIOResService()
                # Not building the service because only need to use this function
                prefix, tid = hio_svc.break_res_name(ins_obj.get_name())

                gpio_ins_name = "{}_{}_{}".format(prefix, pin_name, tid)
                gpio_inst, spi_inst = self._get_shared_resource_inst(gpio_ins_name)
                if gpio_inst is not None and spi_inst is not None:
                    user_ins_name = "{}.{}.{}".format(spi_inst.name, gpio_inst.name, pin_name)

        return user_ins_name

    def _get_hsio_ins_direction(self, pad_info_obj: IOPadInfo, ins_obj: PeripheryBlockInstance):
        assert self.design is not None

        ins_dir = ""
        is_found = False

        # Get the pad pin name
        is_p = self._is_hsio_pad_type_p(pad_info_obj)
        res_name_list = self.get_resource_name_by_pad_name(pad_info_obj.get_pad_name())

        # There's 3 main possible design for hsio (GPIO/LVDS/MIPI)
        reg_list: List[Optional[PeriDesignRegistry]] = [
            self.design.mipi_dphy_reg,
            self.design.lvds_reg
        ]

        for reg in reg_list:
            if reg is None:
                continue

            for res_name in res_name_list:
                inst = reg.get_inst_by_device_name(res_name) # type: ignore
                if inst is None:
                    continue
                is_found = True
                ins_dir = inst.get_direction_str()
                break
            if is_found:
                break

        # GPIO
        # dev_ins is the HSIO instance (GPIOY_PN_XX). Whereas, the name stored
        # in the shared_res2username_map has key with the GPIO-typed resource name
        # (GPIOY_P_XX / GPIOY_N_XX). Therefore, we use the io pad instance name
        # and pin to get the correct gpio resource name.
        if not is_found and is_p is not None:
            reg = self.design.get_block_reg(PeriDesign.BlockType.gpio)
            assert reg is not None

            for inst in reg.get_all_lvds_gpio(): # type: ignore
                inst: GPIOComplex
                # This is the name with extension
                res_name = inst.gpio_def

                # This is the device base name without extension
                lvds_dev_base_name = self._device.get_base_ins_name(
                    res_name)

                if ins_obj.get_name() != lvds_dev_base_name or is_p is None:
                    continue

                lvds_type = lvds_util.get_instance_lvds_type(res_name, self._device)

                is_diff_tl = inst.is_gpio_differential_stl_io_std()

                # In this case, we need to align the mode to the pad that
                # we're looking at since:
                # LVTTL1 - Connects to 'P'
                # LVTTL2 - Connects to 'N'

                if (lvds_type == lvds_util.BlockModeType.lvttl1 and is_p) or\
                        (lvds_type == lvds_util.BlockModeType.lvttl2 and not is_p) or\
                            (not is_p and is_diff_tl):
                    ins_dir = inst.get_direction_str()
                    is_found = True

        if not is_found and is_p is not None:
            # dev_ins is the HSIO instance (GPIOY_PN_XX). Whereas, the name stored
            # in the shared_res2username_map has key with the GPIO-typed resource name
            # (GPIOY_P_XX / GPIOY_N_XX). Therefore, we use the io pad instance name
            # and pin to get the correct gpio resource name.
            pin_name = pad_info_obj.get_pin_name()

            if pin_name != "" and pin_name is not None:
                # P - _P_, N - _N_
                hio_svc = HIOResService()
                # Not building the service because only need to use this function
                prefix, tid = hio_svc.break_res_name(ins_obj.get_name())

                gpio_ins_name = "{}_{}_{}".format(prefix, pin_name, tid)
                ins_dir = self._get_shared_resource_direction(gpio_ins_name)

        return ins_dir

    def _get_shared_resource_inst(self, res_name: str):
        assert self.design is not None
        spi_inst = None
        gpio_inst = None

        if self.design.spi_flash_reg is None or self.design.gpio_reg is None:
            return gpio_inst, spi_inst

        dev_shared_ins_map = self._device.get_shared_instance_map()

        if dev_shared_ins_map is None:
            return gpio_inst, spi_inst

        local_gpio_reg = self._get_local_gpio_reg()
        if local_gpio_reg is None:
            return gpio_inst, spi_inst

        for spi_res_name, shared_ins_list in dev_shared_ins_map.items():
            spi_inst = self.design.spi_flash_reg.get_inst_by_device_name(spi_res_name)
            if spi_inst is None or res_name not in shared_ins_list:
                continue

            gpio_inst = local_gpio_reg.get_gpio_by_device_name(res_name)
            if gpio_inst is not None:
                break

        if spi_inst is not None and gpio_inst is not None:
            self.replace_spi_flash_embedded_ins_names(spi_inst, gpio_inst)
            return gpio_inst, spi_inst

        return None, None

    def _get_local_gpio_reg(self):
        assert self.design is not None

        if self.local_gpio_reg is not None:
            return self.local_gpio_reg

        local_gpio_reg = None

        sflash_blk = self._device.find_block("spi_flash")
        spiflash_reg = self.design.get_block_reg(self.design.BlockType.spi_flash)

        # Get the configuration instance, specifically
        if sflash_blk is None or not sflash_blk.has_config_files() or spiflash_reg is None:
            return local_gpio_reg

        sf_ins = spiflash_reg.get_inst_by_device_name("SPI_FLASH0")

        if sf_ins is None:
            return local_gpio_reg

        file_attr_name = sf_ins.get_configured_instance_filename()
        used_cfg_file = sflash_blk.get_config_file(file_attr_name)

        if used_cfg_file == "":
            return local_gpio_reg

        if self.design.is_tesseract_design():
            gpio_service = GPIODesignServiceComplex()
        else:
            gpio_service = GPIODesignService()

        app_setting = AppSetting()
        local_gpio_reg = gpio_service.load_design(
            Path(app_setting.app_path.get(AppSetting.PathType.install, "")) / used_cfg_file)

        self.local_gpio_reg = local_gpio_reg

        gc.collect()

        return local_gpio_reg

    def _get_shared_resource_direction(self, res_name: str):
        assert self.design is not None
        direction = ""

        gpio_inst, spi_inst = self._get_shared_resource_inst(res_name)

        if gpio_inst is not None:
            direction = gpio_inst.get_direction_str()

        return direction

    def _get_user_instance_name(self, pad_info_obj: IOPadInfo, ins_obj: PeripheryBlockInstance):
        assert self.design is not None
        user_ins_name = ""

        ref_name = ins_obj.get_ref_name()

        ref_name: str = ins_obj.get_ref_name()
        dev_blk_type = DeviceDBService.block_str2type_map.get(ref_name, None)

        match dev_blk_type:
            case DeviceDBService.BlockType.GPIO| DeviceDBService.BlockType.PLL| \
                DeviceDBService.BlockType.OSC:

                blk_type2design_type = {
                    DeviceDBService.BlockType.GPIO: self.design.gpio_type,
                    DeviceDBService.BlockType.PLL: PeriDesign.BlockType.pll,
                    DeviceDBService.BlockType.OSC: PeriDesign.BlockType.osc,
                }

                design_blk_type = blk_type2design_type.get(dev_blk_type, None)
                assert design_blk_type is not None

                reg = self.design.get_block_reg(design_blk_type)
                assert reg is not None

                inst = reg.get_inst_by_device_name(ins_obj.get_name())
                if inst is not None:
                    user_ins_name = inst.name

            case DeviceDBService.BlockType.LVDS_RX | DeviceDBService.BlockType.LVDS_TX:
                inst = self._get_lvds_design_ins(pad_info_obj, ins_obj)

                if inst is not None:
                    pin_name = pad_info_obj.get_pin_name()
                    user_ins_name = f"{inst.name}.{pin_name}"

            case DeviceDBService.BlockType.MIPI_RX | DeviceDBService.BlockType.MIPI_TX:
                user_ins_name = self._get_mipi_user_ins(pad_info_obj, ins_obj)

            case DeviceDBService.BlockType.DDR:
                user_ins_name = self._get_ddr_user_ins(pad_info_obj, ins_obj)

            case DeviceDBService.BlockType.HSIO | DeviceDBService.BlockType.HSIO_MAX:
                user_ins_name = self._get_hsio_user_ins(pad_info_obj, ins_obj)

            case DeviceDBService.BlockType.QUAD | DeviceDBService.BlockType.QUAD_PCIE:
                blk_ins_name = self.get_block_instance_by_pad_name(pad_info_obj.get_pad_name())
                ins_name_str = ''.join(map(str, blk_ins_name))
                user_ins_name = f"{ins_name_str}.{re.sub(r'Q[0-3]_', '', pad_info_obj.get_pad_name())}" if ins_name_str else ''

        return user_ins_name

    def _get_gpio_unused_config_pull_type(self, unused_config: GPIOUnusedConfig):
        '''
        Get the pull type of a GPIO Unused config.

        :param unused_config: GPIOUnusedConfig object

        :return string that indicates the pull type
        '''
        pull_type_str = ""

        # We cannot use the class type2string map since
        # it has additional string ("input with") which we don't
        # want to show in the report file
        if unused_config.state == unused_config.StateType.input_weak_pullup:
            pull_type_str = "weak pullup"
        elif unused_config.state == unused_config.StateType.input_weak_pulldown:
            pull_type_str = "weak pulldown"

        return pull_type_str


class BallPackage(PackagePinService):
    '''
    Getting the ball-type package information
    '''
    class CoordinateType(Enum):
        ALPHA = auto()
        NUMBER = auto()

    def __init__(self, design: Optional[PeriDesign], device: Optional[PeripheryDevice] = None, is_coldboot=None):
        self._ball_count: int = 0
        self._pkg_dimension = PkgDimension()
        self.coordinate2type_map: Dict[str, BallPackage.CoordinateType] = {}
        super().__init__(design, device, is_coldboot)

        if self._package_type is not None:
            str_ball = self._package_type.split("-")[0]
            if str_ball.isnumeric():
                self._ball_count=int(str_ball)
            else:
                pkg = self._device.get_package()
                assert pkg is not None
                self._ball_count = len(pkg.get_package_pad_map())

        self.set_pll_res2res_name()
        self.set_spi_res2res_name()

    def get_ball_count(self) -> int:
        return self._ball_count

    def set_ball_count(self, count:int):
        self._ball_count = count

    def get_pkg_dimension(self):
        return self._pkg_dimension

    def set_pkg_dimension(self, max_x, max_y, min_x, min_y):
        self._pkg_dimension.max_x = max_x
        self._pkg_dimension.max_y = max_y
        self._pkg_dimension.min_x = min_x
        self._pkg_dimension.min_y = min_y

    def set_coordinate(self):
        pin_map = self.get_all_pin2pad_map()
        max_alpha = min_alpha = ""
        num_list = []
        for coord in pin_map.keys():
            alpha, number = self.str2alpha_num(coord)

            if alpha == "" or number == "":
                self.logger.error(f"Invalid pin name {coord} while setting coordinate")
                continue

            num_list.append(number)

            if max_alpha == "" and min_alpha== "":
                max_alpha = min_alpha = alpha

            elif len(max_alpha) < len(alpha) or \
                (len(max_alpha) == len(alpha) and max_alpha < alpha):
                max_alpha = alpha

            elif len(min_alpha) > len(alpha) or \
                (len(min_alpha) == len(alpha) and min_alpha > alpha):
                min_alpha = alpha

        min_num = min(num_list)
        max_num = max(num_list)

        if self.coordinate2type_map.get("x") == BallPackage.CoordinateType.NUMBER:
            self.set_pkg_dimension(max_num, max_alpha, min_num, min_alpha)
        else:
            self.set_pkg_dimension(max_alpha, max_num, min_alpha, min_num)

    def str2alpha_num(self, target:str):
        """
        for alpha + number only
        """
        if target.isalnum():
            split_list = re.split(r'(\d+)',target)
            if len(split_list) == 3:
                alpha, str_number, sub_alpha  = split_list
                if sub_alpha != "":
                    return "", ""
                num = int(str_number)
                return alpha, num
        return "", ""


class FBGABallPackage(BallPackage):
    def __init__(self, design: Optional[PeriDesign], device: Optional[PeripheryDevice] = None,
                 is_coldboot: Optional[bool] = None):
        super().__init__(design, device, is_coldboot)
        self.coordinate2type_map = {
            "x":BallPackage.CoordinateType.NUMBER,
            "y":BallPackage.CoordinateType.ALPHA
        }
        self.set_coordinate()


class WLCSPBallPackage(BallPackage):
    def __init__(self, design: Optional[PeriDesign], device: Optional[PeripheryDevice] = None,
                 is_coldboot: Optional[bool] = None):
        super().__init__(design, device, is_coldboot)
        self.coordinate2type_map = {
            "x":BallPackage.CoordinateType.ALPHA,
            "y":BallPackage.CoordinateType.NUMBER
        }
        self.set_coordinate()


class LeadPackage(PackagePinService):
    def __init__(self, design: Optional[PeriDesign], device: Optional[PeripheryDevice] = None,
                 is_coldboot: Optional[bool] = None):
        super().__init__(design, device, is_coldboot=is_coldboot)
        self.lead_count = 0
        self.min_pin:Optional[int] = None
        self.max_pin:Optional[int] = None

        if self._package_type is not None:
            str_ball = self._package_type.split("-")[0]

            if str_ball.isnumeric():
                self.lead_count=int(str_ball)
            else:
                pkg = self._device.get_package()
                assert pkg is not None
                self.lead_count = len(pkg.get_package_pad_map())
            self.set_coordinate()

        self.set_pll_res2res_name()
        self.set_spi_res2res_name()

    def get_lead_count(self):
        return self.lead_count

    def set_coordinate(self):
        pkg_pad_map = self.get_all_pin2pad_map()
        for pin_name in pkg_pad_map.keys():
            if not pin_name.isnumeric():
                self.logger.error(f"Invalid pin name set for lead package: {pin_name}")
                continue
            pin = int(pin_name)
            if self.min_pin is None or self.min_pin > pin:
                self.min_pin = pin
            elif self.max_pin is None or self.max_pin < pin:
                self.max_pin = pin


class QFPLeadPackage(LeadPackage):
    def __init__(self, design: Optional[PeriDesign], device: Optional[PeripheryDevice] = None,
                 is_coldboot: Optional[bool] = None):
        super().__init__(design, device, is_coldboot)
        self.name = "QFP"


@dataclass
class PkgDimension:
    max_x: int | str = 0
    max_y: int | str = 0
    min_x: int | str = 0
    min_y: int | str = 0
