"""
Copyright (C) 2017-2018 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Sept 6, 2018

@author: yasmin
"""
import os
from enum import Enum, auto
from dataclasses import dataclass
from typing import List, Dict, Union, TypeAlias, Tuple, Optional
from abc import ABC, abstractmethod
from datetime import datetime

from util.singleton_logger import Logger
import util.excp as app_excp
import util.efxproj_setting as efxproj_set
import util.gen_util as pt_util
import util.plugin_map as plmap
from util.app_setting import AppSetting

import device.service as dev_srv
import design.db as db
import design.excp as db_excp
from design.db import PeriDesign
from design.db_item import PeriDesignItem

import common_device.excp as dsg_excp
import common_device.osc.osc_design as oscd
import common_device.osc.osc_design_service as oscds
import common_device.gpio.gpio_design_service as gpiods
import common_device.gpio.gpio_design as gpiod
import common_device.pll.pll_design_service as pllds
import common_device.pll.pll_design as plld
import common_device.lvds.lvds_design_service as lvdsds
import common_device.lvds.lvds_design as lvdsd
import common_device.mipi.mipi_design as mipid

import common_device.mipi.mipi_design_service as mipids
import common_device.jtag.jtag_design_service as jtagds
import common_device.h264.h264_design_service as h264ds
import common_device.ddr.ddr_design_service as ddrds
import common_device.ddr.ddr_design as ddrd
import common_device.mipi_dphy.mipi_dphy_design_service as mdphyds
import common_device.spi_flash.spi_flash_design_service as spids
import common_device.hyper_ram.hyper_ram_design_service as hyperds

import an20_device.pll.pll_design_adv as adv_plld
import tx60_device.pll.pll_design_comp as comp_plld

import tx60_device.osc.osc_design_adv as adv_oscd
import tx60_device.hposc.hposc_design as hposcd
import tx60_device.gpio.gpio_design_comp as comp_gpiod
import tx60_device.lvds.lvds_design_adv as adv_lvdsd

import tx180_device.pll.pll_design as v3comp_plld
import tx180_device.ddr.ddr_design_adv as adv_ddrd
import tx180_device.mipi_dphy.design_service as mhdphyds
import tx180_device.mipi_dphy.design as mhdphyd
import tx180_device.pll_ssc.design_service as pllsscds
import tx375_device.common_quad.design_service as common_quads
import tx375_device.quad_pcie.design_service as quad_pcieds
import tx375_device.lane10g.design_service as lane10gds
import tx375_device.lane1g.design_service as lane1gds
import tx375_device.raw_serdes.design_service as raw_serdesds
import tx375_device.fpll.design as v1_fplld


def update_migrated_new_param(migration_log, new_obj, old_obj):
    '''
    General function to update the newly created instance/object parameters
    from the older (before migration) parameters. We only want to update
    the parameters that only existed in the old (carry over).  If the
    parameters exists but they no longer are common (i.e. different values), then
    we don't updated them and it will be default setting
    (handled by GenericParamGroup.update()). We do this only if both
    new_obj and old_obj has param_group

    :param new_obj: Object that was newly created based on the migrated
                design
    :param old_obj: design object in the old design (pre-migration)

    '''
    if new_obj is not None and old_obj is not None and \
            hasattr(new_obj, "param_group") and hasattr(old_obj, "param_group"):

        # Update the param value based on the old if they existed in old
        new_obj.param_group.update(old_obj.param_group)

        # Generic Fallback
        param_info = new_obj.get_param_info()
        for p in new_obj.param_group.get_all_param():
            prop_data = param_info.get_prop_info_by_name(p.name)
            is_valid, _ = new_obj.param_group.check_param_value(p.name, param_info)
            if not is_valid and prop_data is not None:
                if hasattr(old_obj, "name"):
                    migration_log.reg_param_change(
                        old_obj.name,
                        f'Existing {prop_data.disp_name} is unsupported, reset to default')

class MigrationLog:
    """
    A repository of migration message generated by migrator.
    Migrator can generate its own log message or use register function
    which will format the full message. It also provides function to write to a log file.
    """

    # In future, will have user facing block version for each type.
    # Does not have to match with design db type map.
    db_blktype_2_user_blktype_map = \
        {
            PeriDesign.BlockType.gpio: "GPIO V1",
            PeriDesign.BlockType.pll: "PLL V1",
            PeriDesign.BlockType.adv_pll: "PLL V2",
            PeriDesign.BlockType.osc: "OSC V1",
            PeriDesign.BlockType.lvds: "LVDS V1",
            PeriDesign.BlockType.mipi: "MIPI V1",
            PeriDesign.BlockType.adv_l2_gpio: "GPIO V2",
            PeriDesign.BlockType.ddr: "DDR V1",
            PeriDesign.BlockType.h264: "H264 V1",
            PeriDesign.BlockType.jtag: "JTAG V1",
            PeriDesign.BlockType.comp_gpio: "GPIO V3",
            PeriDesign.BlockType.comp_pll: "PLL V3",
            PeriDesign.BlockType.adv_lvds: "LVDS V2",
            PeriDesign.BlockType.mipi_dphy: "MIPI_LANE V1",
            PeriDesign.BlockType.adv_osc: "OSC V2",
            PeriDesign.BlockType.hposc: "OSC V3",
            PeriDesign.BlockType.spi_flash: "SPI_FLASH V1",
            PeriDesign.BlockType.hyper_ram: "HYPER_RAM V1",
            PeriDesign.BlockType.efx_pll_v3_comp: "PLL V3 EXT",
            PeriDesign.BlockType.adv_ddr: "DDR V2",
            PeriDesign.BlockType.mipi_hard_dphy: "MIPI DPHY V1",
            PeriDesign.BlockType.pll_ssc: "PLL SSC V1",
            PeriDesign.BlockType.quad_pcie: "QUAD PCIE V1",
            PeriDesign.BlockType.lane_10g: "LANE 10G V1",
            PeriDesign.BlockType.lane_1g: "LANE 1G V1",
            PeriDesign.BlockType.raw_serdes: "RAW SERDES V1",
            PeriDesign.BlockType.efx_fpll_v1: "FPLL V1",
            PeriDesign.BlockType.common_quad: "COMMON QUAD V1",
        }  #: Mapping of block type in enum to string type

    @dataclass
    class LogData:
        """
        Each log data must have a message.
        """
        functionality: str = ""  #: Category or affected functionality being logged
        instance: Union[str, None] = ""  #: Affected instance, if any
        msg: str = ""  #: Log message, migrator is reponsible for formatting its content

    def __init__(self):
        self.old_block_type: Union[PeriDesign.BlockType, None] = None  #: Old block type
        self.old_block_type_name: str = ""
        self.new_block_type: Union[PeriDesign.BlockType, None] = None  #: New block type
        self.new_block_type_name: str = ""
        self.log_data: List[MigrationLog.LogData] = []  #: A list of log data

    def set_block_type(self, old_block_type: PeriDesign.BlockType, new_block_type: PeriDesign.BlockType):
        """
        Setup new block type info

        :param new_block_type: New block type
        """
        self.old_block_type = old_block_type
        self.old_block_type_name = self.db_blktype_2_user_blktype_map.get(self.old_block_type, "unknown")
        self.new_block_type = new_block_type
        self.new_block_type_name = self.db_blktype_2_user_blktype_map.get(self.new_block_type, "unknown")

    def add_log(self, functionality: str, msg: str):
        """
        Add log without instance into the repository

        :param functionality: Affected functionality
        :param msg: Log message
        """
        log_data = self.LogData(functionality, None, msg)
        self.log_data.append(log_data)

    def add_log_data(self, log_data: LogData):
        """
        Add log data into repository

        :param log_data: Log data object
        """
        if log_data is not None:
            self.log_data.append(log_data)

    def reg_device_change(self, old_device: str, new_device: str):
        """
        Register device change message

        :param old_device: Old device name
        :param new_device: New device name
        """
        log_data = self.LogData("Device Change", None, f"Device change from {old_device} to {new_device}")
        self.add_log_data(log_data)

    def reg_block_change(self, old_block_type: Union[PeriDesign.BlockType, None],
                         new_block_type: Union[PeriDesign.BlockType, None],
                         migration_type: str = None):
        """
        Register block change message

        :param old_block_type: Old block type
        :param new_block_type: New block type
        :param migration_type: Upgrade or downgrade
        """
        new_block_name = self.db_blktype_2_user_blktype_map.get(new_block_type, "")
        old_block_name = self.db_blktype_2_user_blktype_map.get(old_block_type, "")

        log_data = None
        msg = ""
        if migration_type is not None:
            msg = f"{old_block_name} {migration_type} to {new_block_name}"
        else:
            if new_block_type is None:
                msg = f"{old_block_name} is not supported"
            elif old_block_type is None:
                msg = f"{new_block_name} is now supported"

        if msg != "":
            log_data = self.LogData("Block Type Change", None, msg)

        self.add_log_data(log_data)

    def reg_param_change(self, inst_name: str, msg: str):
        """
        Register parameter change

        :param inst_name: Affected instance name
        :param msg: Log message
        """
        log_data = self.LogData("Parameter Change", inst_name, msg)
        self.add_log_data(log_data)

    def write_header(self, file_obj):
        """
        Write header for migration log file. Follow similar info like report etc.

        :param file_obj: Opened file handle
        """
        from _version import __version__
        timestamp = datetime.now()
        file_obj.write("Efinity Interface Designer Device Migration Log\n")
        file_obj.write("Version: {}\n".format(__version__))
        file_obj.write("Date: {}\n".format(timestamp.strftime("%Y-%m-%d %H:%M")))

    def write_log(self, file_obj):
        """
        Traverse the log repository and write into file

        :param file_obj: Opened file handle
        """

        if len(self.log_data) > 0 and self.old_block_type_name != "":
            file_obj.write(
                f"== Block Type Version Change : {self.old_block_type_name} to {self.new_block_type_name} ==\n")

        for data in self.log_data:
            if data.functionality is None or data.functionality == "":
                file_obj.write(f"{data.msg}\n")
            else:
                if data.instance is None or data.instance == "":
                    file_obj.write(f"{data.functionality}: {data.msg}\n")
                else:
                    file_obj.write(f"{data.functionality}: {data.instance}, {data.msg}\n")


class BlockMigrator(ABC):
    """
    Main interface for block migrator.
    Perform migration from one specific block type to another within the same general block type.
    For example, general block type is gpio and specific block types are gpio and complex gpio.
    """

    class MigrationType(Enum):
        """
        Block migration type
        """
        upgrade = auto()  #: Moving to device with more functionality
        downgrade = auto()  #: Moving to device with less functionality
        unknown = auto()  #: Undetermine or no change

    @abstractmethod
    def __init__(self, new_design: db.PeriDesign, new_block_type: db.PeriDesign.BlockType,
                 old_block_type: db.PeriDesign.BlockType):
        self.design = new_design  #: Design db
        self.new_block_type = new_block_type  #: New block type to migrate to
        self.old_block_type = old_block_type  #: Old block type to migrate from
        self.migration_type: Optional[BlockMigrator.MigrationType] = None  #: Migration type
        self.deleted_instance = []  #: Instance deleted after migration
        self.migration_log = MigrationLog()  #: Migration log repository

    @abstractmethod
    def migrate(self):
        """
        Migrate design blocks from old type to new type
        """
        pass

    @abstractmethod
    def migrate_registry(self):
        """
        Migrate registry data.
        It will create new empty registry if currently unavailable else it does current data migration.
        With data migration, it will create new empty registry and then migrate individual block instance.
        """
        pass

    def migrate_block(self, new_inst: PeriDesignItem, old_inst: PeriDesignItem):
        """
        Migrate block instance data. It does data member shallow copy and copy
        only what is needed in the new instance. For multi-type block, override
        this to handle different type and possibly its semantic.

        :param new_inst: New block instance
        :param old_inst: Old block instance
        """
        if new_inst is not None and old_inst is not None:
            new_inst.copy_common(old_inst)


class GPIOMigrator(BlockMigrator):
    """
    Migrate GPIO design
    """

    # Alias type for easy type hint, for multi-types block
    GPIOInstTypeAlias: TypeAlias = gpiod.GPIO| comp_gpiod.GPIOComplex

    def __init__(self, new_design: db.PeriDesign, new_block_type: db.PeriDesign.BlockType,
                 old_block_type: db.PeriDesign.BlockType):
        super().__init__(new_design, new_block_type, old_block_type)

        self.mig_direct_map = {
            (PeriDesign.BlockType.gpio, PeriDesign.BlockType.adv_l2_gpio): BlockMigrator.MigrationType.upgrade,
            (PeriDesign.BlockType.gpio, PeriDesign.BlockType.comp_gpio): BlockMigrator.MigrationType.upgrade,
            (PeriDesign.BlockType.adv_l2_gpio, PeriDesign.BlockType.comp_gpio): BlockMigrator.MigrationType.upgrade,
            (PeriDesign.BlockType.comp_gpio, PeriDesign.BlockType.adv_l2_gpio): BlockMigrator.MigrationType.downgrade,
            (PeriDesign.BlockType.comp_gpio, PeriDesign.BlockType.gpio): BlockMigrator.MigrationType.downgrade,
            (PeriDesign.BlockType.adv_l2_gpio, PeriDesign.BlockType.gpio): BlockMigrator.MigrationType.downgrade
        }  #: A map of migration path defined as tuple of (old instance type, new instance type) to its migration type

        self.migration_type = self.mig_direct_map.get((self.old_block_type, self.new_block_type),
                                                      BlockMigrator.MigrationType.unknown)
        self.migration_log.set_block_type(old_block_type, new_block_type)
        self.conntype_change_msg = "Existing Connection Type is unsupported, reset to default"
        self.ddiotype_change_msg = "Existing DDIO Type is unsupported, disable"
        self.pullopt_change_msg = "Existing Pull Option is unsupported, disable"
        self.drvstrength_change_msg = "Existing Drive Strength is unsupported, reset to default"
        self.iostd_change_msg = "Existing IO Standard is unsupported, reset to default"

    def migrate(self):
        """
        Override
        """
        if self.new_block_type is None:
            self.design.gpio_reg = None
        else:
            self.migration_log.reg_block_change(self.old_block_type, self.new_block_type)
            if self.design.gpio_reg is None:
                svc = gpiods.GPIODesignService.build_service(self.new_block_type)
                self.design.gpio_reg = svc.create_registry()
                self.design.gpio_reg.change_device(self.design.device_db, self.design.is_support_ddio())
            else:
                self.migrate_registry()

    def migrate_registry(self):
        """
        Override. Migrate block instance and bus.
        """
        old_reg = self.design.gpio_reg
        svc = gpiods.GPIODesignService.build_service(self.new_block_type)
        new_reg = svc.create_registry()

        # Exclude
        exclude_attr = ["_name2gpio_map", "_name2bus_map", "_dev2gpio_map", "_write_lock", "gpio2bank_map",
                        "_res_subject", "_inst_subject"]
        new_reg.copy_common(old_reg, exclude_attr)
        new_reg.change_device(self.design.device_db, self.design.is_support_ddio())

        # AN08 from design db perspective
        is_an08 = self.new_block_type == PeriDesign.BlockType.gpio and self.design.is_support_ddio() is False

        # Rebuild and migrate gpio instance
        old_name2inst_map = old_reg.get_name2inst_map()
        for name, inst in old_name2inst_map.items():
            if inst is not None:

                # AN08 doesn't support lvds io
                if inst.is_lvds_gpio() and is_an08:
                    inst.gpio_def = ""
                    inst.mark_lvds_gpio(False)

                if inst.get_device() == "":
                    new_inst = new_reg.create_instance(name, auto_pin=True)
                else:
                    new_inst = new_reg.create_instance_with_device(name, inst.get_device(), auto_pin=True)

                self.migrate_block(new_inst, inst)

        # Rebuild and migrate bus instance
        old_name2inst_map = old_reg.get_busname2inst_map()
        for name, inst in old_name2inst_map.items():
            if inst is not None:
                new_bus = new_reg.create_empty_bus(inst._name, inst._mode, inst._left, inst._right)
                self.migrate_bus(new_bus, inst, new_reg)

        self.design.gpio_reg = new_reg

    def migrate_bus(self, new_bus: gpiod.GPIOBus, old_bus: gpiod.GPIOBus, new_reg: gpiod.GPIORegistry):
        """
        Migrate bus instance

        :param new_bus: New bus instance
        :param old_bus: Old bus instance
        :param new_reg: New gpio registry
        """

        exclude_attr = ["_member"]
        new_bus.copy_common(old_bus, exclude_attr)

        # Member must point to instance in the new registry
        old_member = old_bus._member
        new_member = {}
        for bus_index, old_gpio_inst in old_member.items():
            new_gpio_inst = new_reg.get_gpio_by_name(old_gpio_inst.name)
            if new_gpio_inst is not None:
                new_member[bus_index] = new_gpio_inst

        new_bus._member = new_member

    def migrate_block(self, new_inst: GPIOInstTypeAlias, old_inst: GPIOInstTypeAlias):
        """
        Override
        """

        if self.new_block_type != self.old_block_type:
            # Detail migration needed of block types are different
            is_input = False
            is_output = False

            if old_inst.mode == old_inst.PadModeType.input:
                is_input = True
            elif old_inst.mode == old_inst.PadModeType.output:
                is_output = True
            elif old_inst.mode == old_inst.PadModeType.inout:
                is_input = True
                is_output = True
            elif old_inst.mode == old_inst.PadModeType.clkout:
                is_output = True

            new_inst.update_mode_config(old_inst.mode)
            if self.migration_type == BlockMigrator.MigrationType.downgrade:

                self.downgrade_gpio(new_inst, old_inst)
                if is_input:
                    self.downgrade_input(new_inst, old_inst)

                if is_output:
                    self.downgrade_output(new_inst, old_inst)

            elif self.migration_type == BlockMigrator.MigrationType.upgrade:

                self.upgrade_gpio(new_inst, old_inst)
                if is_input:
                    self.upgrade_input(new_inst, old_inst)

                if is_output:
                    self.upgrade_output(new_inst, old_inst)

        else:
            # Same type, just copy all
            new_inst.copy_common(old_inst)

        # The new instance may copy those feature flags from the old instance, Update them back according the the design
        new_inst.enable_ddio_support(self.design.is_support_ddio())

    def upgrade_gpio(self, new_inst: GPIOInstTypeAlias, old_inst: GPIOInstTypeAlias):
        """
        Migrate basic gpio to an advanced version

        :param new_inst: New gpio instance
        :param old_inst: Old gpio instance
        """

        exclude_attr = ["input", "output", "output_enable", "_mode_subject", "beh_param", "_param_info", "param_group"]
        new_inst.copy_common(old_inst, exclude_attr)
        # New Instance's param_group is updated with old_inst's param_group's value, So it requires check and fallback
        new_inst.param_group.update(old_inst.param_group)

        # Special handling for io standard
        valid_iostd = self.design.device_setting.iobank_reg.get_all_raw_io_standard()
        if new_inst.io_standard not in valid_iostd:
            new_inst.io_standard = new_inst.get_default_io_standard(self.design.device_db)
            self.migration_log.reg_param_change(old_inst.name, self.iostd_change_msg)

        # Generic Fallback
        param_info = new_inst.get_param_info()
        for p in new_inst.param_group.get_all_param():
            prop_data = param_info.get_prop_info_by_name(p.name)
            is_valid, _ = new_inst.param_group.check_param_value(p.name, param_info)
            if not is_valid and prop_data is not None:
                self.migration_log.reg_param_change(old_inst.name, f'Existing {prop_data.disp_name} is unsupported, reset to default')

    def upgrade_input(self, new_inst: GPIOInstTypeAlias, old_inst: GPIOInstTypeAlias):
        """
        Migrate basic gpio input to an advanced version

        :param new_inst: New gpio instance
        :param old_inst: Old gpio instance
        """

        old_input = old_inst.input
        new_input = new_inst.input
        assert old_input is not None and new_input is not None

        exclude_attr = ["param_group"]
        new_input.copy_common(old_input, exclude_attr=exclude_attr)
        # Get the new reference from the Owner Instance
        new_input.param_group = new_inst.param_group

        if self.new_block_type == PeriDesign.BlockType.comp_gpio:
            # Custom migration
            conn_type_str = old_input.conn2str_map.get(old_input.conn_type, None)
            if conn_type_str is None:
                self.migration_log.reg_param_change(old_inst.name, self.conntype_change_msg)

            if conn_type_str is not None and conn_type_str.upper() in new_inst.conn_type_list \
                and conn_type_str in new_input.str2conn_map:
                    new_input.conn_type = new_input.str2conn_map.get(conn_type_str, new_input.ConnType.normal_conn)
            else:
                new_input.conn_type = new_input.ConnType.normal_conn

            new_input.conn_type = new_input.str2conn_map.get(conn_type_str, new_input.ConnType.normal_conn)

            ddio_opt_str = old_input.ddio2str_map.get(old_input.ddio_type, None)
            if ddio_opt_str is None:
                self.migration_log.reg_param_change(old_inst.name, self.ddiotype_change_msg)
            new_input.ddio_type = new_input.str2ddio_map.get(ddio_opt_str, new_input.DDIOType.none)

            pullopt_str = old_input.pullopt2str_map.get(old_input.pull_option, None)
            if pullopt_str is None:
                self.migration_log.reg_param_change(old_inst.name, self.pullopt_change_msg)
            new_input.pull_option = new_input.str2pullopt_map.get(pullopt_str, new_input.PullOptType.none)

    def upgrade_output(self, new_inst: GPIOInstTypeAlias, old_inst: GPIOInstTypeAlias):
        """
        Migrate basic gpio output and output enable to an advanced version.

        :param new_inst: New gpio instance
        :param old_inst: Old gpio instance
        """

        old_output = old_inst.output
        new_output = new_inst.output

        if self.new_block_type == PeriDesign.BlockType.comp_gpio:
            # Drive strength remains as default since it is not a simple migration
            exclude_attr = ["drive_strength", "param_group"]
            new_output.copy_common(old_output, exclude_attr)
            self.migration_log.reg_param_change(old_inst.name, self.drvstrength_change_msg)

            ddio_opt_str = old_output.ddio2str_map.get(old_output.ddio_type, None)
            if ddio_opt_str is None:
                self.migration_log.reg_param_change(old_inst.name, self.ddiotype_change_msg)
            new_output.ddio_type = new_output.str2ddio_map.get(ddio_opt_str, new_output.DDIOType.none)
        else:
            exclude_attr = ["param_group"]
            new_output.copy_common(old_output, exclude_attr)

        # Get the new reference from the Owner Instance
        new_output.param_group = new_inst.param_group

        if new_inst.output_enable is not None:
            exclude_attr = ["param_group"]
            new_inst.output_enable.copy_common(old_inst.output_enable, exclude_attr)
            # Get the new reference from the Owner Instance
            new_inst.output_enable.param_group = new_inst.param_group

    def downgrade_gpio(self, new_inst: GPIOInstTypeAlias, old_inst: GPIOInstTypeAlias):
        """
        Migrate advanced gpio version to basic version

        :param new_inst: New gpio instance
        :param old_inst: Old gpio instance
        """

        exclude_attr = ["input", "output", "output_enable", "_mode_subject", "beh_param", "_param_info", "param_group"]
        new_inst.copy_common(old_inst, exclude_attr)

        # New Instance's param_group is updated with old_inst's param_group's value, So it requires check and fallback
        new_inst.param_group.update(old_inst.param_group)

        # Special handling for io standard
        valid_iostd = self.design.device_setting.iobank_reg.get_all_raw_io_standard()
        if new_inst.io_standard not in valid_iostd:
            new_inst.io_standard = new_inst.get_default_io_standard(self.design.device_db)
            self.migration_log.reg_param_change(old_inst.name, self.iostd_change_msg)

        # Generic Fallback
        param_info = new_inst.get_param_info()
        for p in new_inst.param_group.get_all_param():
            prop_data = param_info.get_prop_info_by_name(p.name)
            is_valid, _ = new_inst.param_group.check_param_value(p.name, param_info)
            if not is_valid and prop_data is not None:
                self.migration_log.reg_param_change(old_inst.name, f'Existing {prop_data.disp_name} is unsupported, reset to default')

    def downgrade_input(self, new_inst: GPIOInstTypeAlias, old_inst: GPIOInstTypeAlias):
        """
        Migrate advanced gpio input version to basic version

        :param new_inst: New gpio instance
        :param old_inst: Old gpio instance
        """

        old_input = old_inst.input
        new_input = new_inst.input
        assert old_input is not None and new_input is not None

        # Get the new reference from the Owner Instance
        new_input.param_group = new_inst.param_group

        # an08
        if self.new_block_type == PeriDesign.BlockType.gpio and self.design.is_support_ddio() is False:
            exclude_attr = ["name_ddio_lo", "ddio_type", "param_group"]
            new_input.copy_common(old_input, exclude_attr)
        else:
            exclude_attr = ["param_group"]
            new_input.copy_common(old_input, exclude_attr)

        if self.old_block_type == PeriDesign.BlockType.comp_gpio:
            conn_type_str = old_input.conn2str_map.get(old_input.conn_type, None)
            if conn_type_str is None:
                self.migration_log.reg_param_change(old_inst.name, self.conntype_change_msg)

            if conn_type_str is not None and conn_type_str.upper() in new_inst.conn_type_list \
                and conn_type_str in new_input.str2conn_map:
                    new_input.conn_type = new_input.str2conn_map.get(conn_type_str, new_input.ConnType.normal_conn)
            else:
                new_input.conn_type = new_input.ConnType.normal_conn

            ddio_opt_str = old_input.ddio2str_map.get(old_input.ddio_type, None)
            if ddio_opt_str is None:
                self.migration_log.reg_param_change(old_inst.name, self.ddiotype_change_msg)
            new_input.ddio_type = new_input.str2ddio_map.get(ddio_opt_str, new_input.DDIOType.none)

            pullopt_str = old_input.pullopt2str_map.get(old_input.pull_option, None)
            if pullopt_str is None:
                self.migration_log.reg_param_change(old_inst.name, self.pullopt_change_msg)
            new_input.pull_option = new_input.str2pullopt_map.get(pullopt_str, new_input.PullOptType.none)

    def downgrade_output(self, new_inst: GPIOInstTypeAlias, old_inst: GPIOInstTypeAlias):
        """
        Migrate advanced gpio output and output enable version to basic version

        :param new_inst: New gpio instance
        :param old_inst: Old gpio instance
        """

        old_output = old_inst.output
        new_output = new_inst.output

        # an08
        exclude_attr = ['param_group']
        if self.new_block_type == PeriDesign.BlockType.gpio and self.design.is_support_ddio() is False:
            exclude_attr += ['name_ddio_lo', 'ddio_type']

        if self.old_block_type == PeriDesign.BlockType.comp_gpio:
            # Drive strength remains as default since it is not a simple migration
            exclude_attr += ["drive_strength"]
            self.migration_log.reg_param_change(old_inst.name, self.drvstrength_change_msg)

        new_output.copy_common(old_output, exclude_attr)

        if self.old_block_type == PeriDesign.BlockType.comp_gpio:
            ddio_opt_str = old_output.ddio2str_map.get(old_output.ddio_type, None)
            if ddio_opt_str is None:
                self.migration_log.reg_param_change(old_inst.name, self.ddiotype_change_msg)
            new_output.ddio_type = new_output.str2ddio_map.get(ddio_opt_str, new_output.DDIOType.none)

        # Get the new reference from the Owner Instance
        new_output.param_group = new_inst.param_group

        if new_inst.output_enable is not None:
            exclude_attr = ["param_group"]
            new_inst.output_enable.copy_common(old_inst.output_enable, exclude_attr)
            # Get the new reference from the Owner Instance
            new_inst.output_enable.param_group = new_inst.param_group


class PLLMigrator(BlockMigrator):
    """
    Migrate PLL design
    """

    # Alias type for easy type hint, for multi-types block
    PLLInstTypeAlias: TypeAlias = Union[plld.PLL, adv_plld.PLLAdvance, comp_plld.PLLComplex, v3comp_plld.PLLDesignV3Complex, v1_fplld.EfxFpllV1]

    def __init__(self, new_design: db.PeriDesign, new_block_type: db.PeriDesign.BlockType,
                 old_block_type: db.PeriDesign.BlockType):
        super().__init__(new_design, new_block_type, old_block_type)

        self.mig_direct_map = {
            (PeriDesign.BlockType.pll, PeriDesign.BlockType.adv_pll): BlockMigrator.MigrationType.upgrade,
            (PeriDesign.BlockType.adv_pll, PeriDesign.BlockType.comp_pll): BlockMigrator.MigrationType.upgrade,
            (PeriDesign.BlockType.adv_pll, PeriDesign.BlockType.efx_pll_v3_comp): BlockMigrator.MigrationType.upgrade,
            (PeriDesign.BlockType.adv_pll, PeriDesign.BlockType.efx_fpll_v1): BlockMigrator.MigrationType.upgrade,
            (PeriDesign.BlockType.pll, PeriDesign.BlockType.comp_pll): BlockMigrator.MigrationType.upgrade,
            (PeriDesign.BlockType.pll, PeriDesign.BlockType.efx_pll_v3_comp): BlockMigrator.MigrationType.upgrade,
            (PeriDesign.BlockType.pll, PeriDesign.BlockType.efx_fpll_v1): BlockMigrator.MigrationType.upgrade,
            (PeriDesign.BlockType.comp_pll, PeriDesign.BlockType.efx_fpll_v1): BlockMigrator.MigrationType.upgrade,
            (PeriDesign.BlockType.efx_pll_v3_comp, PeriDesign.BlockType.efx_fpll_v1): BlockMigrator.MigrationType.upgrade,
            (PeriDesign.BlockType.adv_pll, PeriDesign.BlockType.pll): BlockMigrator.MigrationType.downgrade,
            (PeriDesign.BlockType.comp_pll, PeriDesign.BlockType.adv_pll): BlockMigrator.MigrationType.downgrade,
            (PeriDesign.BlockType.efx_pll_v3_comp, PeriDesign.BlockType.adv_pll): BlockMigrator.MigrationType.downgrade,
            (PeriDesign.BlockType.comp_pll, PeriDesign.BlockType.pll): BlockMigrator.MigrationType.downgrade,
            (PeriDesign.BlockType.efx_pll_v3_comp, PeriDesign.BlockType.pll): BlockMigrator.MigrationType.downgrade,
            (PeriDesign.BlockType.comp_pll, PeriDesign.BlockType.efx_pll_v3_comp): BlockMigrator.MigrationType.upgrade,
            (PeriDesign.BlockType.efx_pll_v3_comp, PeriDesign.BlockType.comp_pll): BlockMigrator.MigrationType.downgrade,
            (PeriDesign.BlockType.efx_fpll_v1, PeriDesign.BlockType.comp_pll): BlockMigrator.MigrationType.downgrade,
            (PeriDesign.BlockType.efx_fpll_v1, PeriDesign.BlockType.adv_pll): BlockMigrator.MigrationType.downgrade,
            (PeriDesign.BlockType.efx_fpll_v1, PeriDesign.BlockType.pll): BlockMigrator.MigrationType.downgrade,
            (PeriDesign.BlockType.efx_fpll_v1, PeriDesign.BlockType.efx_pll_v3_comp): BlockMigrator.MigrationType.downgrade,
        }  #: A map of migration path defined as tuple of (old instance type, new instance type) to its migration type

        self.migration_type = self.mig_direct_map.get((self.old_block_type, self.new_block_type),
                                                      BlockMigrator.MigrationType.unknown)
        self.migration_log.set_block_type(old_block_type, new_block_type)
        self.intfb_change_msg = "Internal feedback mode is not supported, reset to default"
        self.extfb_change_msg = "External feedback mode is not supported, reset to default"
        self.extclksrc_change_msg = "Clock source of external clock 2 is not supported, reset to default"
        self.prediv_oor_msg = "Pre-divider(N) is out of range, reset to default"
        self.postdiv_oor_msg = "Post-divider(O) is out of range, reset to default"
        self.mult_oor_msg = "Multipler(M) is out of range, reset to default"
        self.clkdiv_oor_msg = "Output clock divider is out of range, reset to default"
        self.ssc_unsupport_msg = "SSC feature is not supported, reset to default"
        self.fractional_unsupport_msg = "Fractional mode is not supported, reset to default"
        self.pdc_unsupport_msg = "Programable Duty Cycle is not supported, reset to default"

    def migrate(self):
        """
        Override
        """

        if self.new_block_type is None:
            self.design.pll_reg = None
        else:
            if self.design.pll_reg is None:
                svc = pllds.PLLDesignService.build_service(self.new_block_type)
                self.design.pll_reg = svc.create_registry()
            else:
                self.migrate_registry()

    def migrate_registry(self):
        """
        Override
        """
        old_reg = self.design.pll_reg
        svc = pllds.PLLDesignService.build_service(self.new_block_type)
        new_reg = svc.create_registry()
        new_reg.change_device(self.design.device_db)

        old_name2inst_map = old_reg.get_name2inst_map()
        for name, old_inst in old_name2inst_map.items():
            if old_inst is not None:
                if old_inst.get_device() == "":
                    new_inst = new_reg.create_instance(name, auto_pin=True)
                else:
                    new_inst = new_reg.create_instance_with_device(name, old_inst.get_device(), auto_pin=True)

                self.migrate_block(new_inst, old_inst)

        self.design.pll_reg = new_reg

    def migrate_block(self, new_inst: PLLInstTypeAlias, old_inst: PLLInstTypeAlias):
        """
        Override
        """

        # name and device resource have been set in creation
        if self.new_block_type != self.old_block_type:
            if self.migration_type == BlockMigrator.MigrationType.upgrade:
                self.upgrade_pll(new_inst, old_inst)
                self.upgrade_outclock(new_inst, old_inst)
            elif self.migration_type == BlockMigrator.MigrationType.downgrade:
                self.downgrade_pll(new_inst, old_inst)
                self.downgrade_outclock(new_inst, old_inst)
        else:
            new_inst.copy_common(old_inst)

    def upgrade_pll(self, new_inst: PLLInstTypeAlias, old_inst: PLLInstTypeAlias):
        """
        Migrate pll from simpler version to a more advance version

        :param new_inst: New pll instance
        :param old_inst: Old pll instance
        """

        if self.new_block_type == db.PeriDesign.BlockType.comp_pll or \
                self.new_block_type == db.PeriDesign.BlockType.efx_pll_v3_comp or \
                self.new_block_type == db.PeriDesign.BlockType.efx_fpll_v1:
            # We don't want to override the param info list with the old copy. So, skip _param_info
            exclude_attr = ["_out_clock_list", "vco_freq", "pll_freq", "freq_calc",
                            "_calc_output_no", "_param_info", "param_group"]
            new_inst.copy_common(old_inst, exclude_attr)
            param_info = new_inst.get_param_info()
            update_migrated_new_param(self.migration_log, new_inst, old_inst)

            if param_info.is_valid_setting(param_info.Id.prediv, new_inst.pre_divider) is False:
                self.migration_log.reg_param_change(old_inst.name, self.prediv_oor_msg)
                new_inst.pre_divider = param_info.get_default(param_info.Id.prediv)

            if param_info.is_valid_setting(param_info.Id.postdiv, new_inst.post_divider) is False:
                self.migration_log.reg_param_change(old_inst.name, self.postdiv_oor_msg)
                new_inst.post_divider = param_info.get_default(param_info.Id.postdiv)

            if param_info.is_valid_setting(param_info.Id.mult, new_inst.multiplier) is False:
                self.migration_log.reg_param_change(old_inst.name, self.mult_oor_msg)
                new_inst.multiplier = param_info.get_default(param_info.Id.mult)

            if self.old_block_type == db.PeriDesign.BlockType.adv_pll:
                if new_inst.fb_mode == old_inst.FeedbackModeType.internal:
                    self.migration_log.reg_param_change(old_inst.name, self.intfb_change_msg)
                    new_inst.fb_mode = param_info.get_default(param_info.Id.fb_mode)

        elif self.new_block_type == db.PeriDesign.BlockType.adv_pll:
            exclude_attr = ["_out_clock_list", "vco_freq", "pll_freq", "freq_calc", "_calc_output_no"]
            new_inst.copy_common(old_inst, exclude_attr)

    def downgrade_pll(self, new_inst: PLLInstTypeAlias, old_inst: PLLInstTypeAlias):
        """
        Migrate pll from more advance version to a simpler version

        :param new_inst: New pll instance
        :param old_inst: Old pll instance
        """
        if self.old_block_type == db.PeriDesign.BlockType.efx_fpll_v1:
            if old_inst.fractional_mode is True and old_inst.fractional_coefficient != 0:
                self.migration_log.reg_param_change(old_inst.name, self.fractional_unsupport_msg)
            if old_inst.ssc_mode != 'DISABLE':
                self.migration_log.reg_param_change(old_inst.name, self.ssc_unsupport_msg)
            if old_inst.pdc_enable is True:
                self.migration_log.reg_param_change(old_inst.name, self.pdc_unsupport_msg)

        if self.new_block_type == db.PeriDesign.BlockType.comp_pll or \
                self.new_block_type == db.PeriDesign.BlockType.efx_pll_v3_comp or \
                self.new_block_type == db.PeriDesign.BlockType.efx_fpll_v1:
            exclude_attr = ["_out_clock_list", "vco_freq", "pll_freq", "freq_calc", "_calc_output_no",
                            "_param_info", "param_group"]
            new_inst.copy_common(old_inst, exclude_attr)

            update_migrated_new_param(self.migration_log, new_inst, old_inst)

        elif self.new_block_type == db.PeriDesign.BlockType.adv_pll:
            exclude_attr = ["_out_clock_list", "vco_freq", "pll_freq", "freq_calc", "_calc_output_no",
                            "_param_info", "param_group"]
            new_inst.copy_common(old_inst, exclude_attr)

            update_migrated_new_param(self.migration_log, new_inst, old_inst)

            if self.old_block_type == db.PeriDesign.BlockType.comp_pll or\
                    self.old_block_type == db.PeriDesign.BlockType.efx_pll_v3_comp:
                fbk_clk_no = new_inst.get_feedback_clock_no()
                if fbk_clk_no == 3 or fbk_clk_no == 4:
                    self.migration_log.reg_param_change(
                        old_inst.name,
                        f"Feedback output clock {fbk_clk_no} is not supported, reset to default")
                    new_inst.set_feedback_clock(0)

                if new_inst.fb_mode == old_inst.FeedbackModeType.external:
                    self.migration_log.reg_param_change(old_inst.name, self.extfb_change_msg)
                    new_inst.fb_mode = new_inst.FeedbackModeType.local

                if new_inst.ext_ref_clock_id == old_inst.RefClockIDType.ext_clock_2:
                    self.migration_log.reg_param_change(old_inst.name, self.extclksrc_change_msg)
                    new_inst.ext_ref_clock_id = new_inst.RefClockIDType.ext_clock_0

        elif self.new_block_type == db.PeriDesign.BlockType.pll:
            exclude_attr = ["_out_clock_list", "vco_freq", "pll_freq"]
            new_inst.copy_common(old_inst, exclude_attr)

    def upgrade_outclock(self, new_inst: PLLInstTypeAlias, old_inst: PLLInstTypeAlias):
        """
        Migrate output clock from simpler version to a more advance version.

        :param new_inst: New pll instance
        :param old_inst: Old pll instance
        """
        if self.new_block_type == db.PeriDesign.BlockType.comp_pll or\
                self.new_block_type == db.PeriDesign.BlockType.efx_pll_v3_comp or\
                self.new_block_type == db.PeriDesign.BlockType.efx_fpll_v1:
            param_info = new_inst.get_param_info()

        else:
            param_info = None

        old_clock_list = old_inst.get_output_clock_list()
        for old_clk in old_clock_list:
            new_clk = new_inst.create_output_clock(old_clk.name, old_clk.number)

            exclude_attr = ["_param_info", "param_group"]

            new_clk.copy_common(old_clk, exclude_attr)

            # New Instance's param_group is updated with old_inst's param_group's value, So it requires check and fallback
            if param_info is not None and self.old_block_type != db.PeriDesign.BlockType.adv_pll and\
                    self.old_block_type != db.PeriDesign.BlockType.pll:
                update_migrated_new_param(self.migration_log, new_clk, old_clk)

            if param_info is not None and not param_info.is_valid_setting(param_info.Id.clkout_div,
                                                                          new_clk.out_divider):
                self.migration_log.reg_param_change(old_inst.name, f"{new_clk.number}: {self.clkdiv_oor_msg}")
                new_clk.out_divider = param_info.get_default(param_info.Id.clkout_div)

            if (self.new_block_type == PeriDesign.BlockType.comp_pll or\
                    self.new_block_type == PeriDesign.BlockType.efx_pll_v3_comp) and \
                    self.old_block_type == PeriDesign.BlockType.adv_pll:
                try:
                    new_clk.phase_setting = new_inst.phase_degree2setting_by_outclk(new_clk.number)
                    if new_clk.phase_setting is None:
                        self.migration_log.reg_param_change(old_inst.name,
                                                            f"Fail to convert output clock {new_clk.number}'s phase shift in degree to phase setting. Reset to default")
                        new_clk.phase_setting = param_info.get_default(param_info.Id.phase_setting)
                except dsg_excp.CalcException:
                    self.migration_log.reg_param_change(old_inst.name,
                                                        f"Fail to convert output clock {new_clk.number}'s phase shift in degree to phase setting. Reset to default")
                    new_clk.phase_setting = param_info.get_default(param_info.Id.phase_setting)

    def downgrade_outclock(self, new_inst: PLLInstTypeAlias, old_inst: PLLInstTypeAlias):
        """
        Migrate output clock from advance version to a simpler version.

        :param new_inst: New pll instance
        :param old_inst: Old pll instance
        """

        if self.new_block_type == db.PeriDesign.BlockType.comp_pll or\
                self.new_block_type == db.PeriDesign.BlockType.efx_pll_v3_comp:
            param_info = new_inst.get_param_info()

        else:
            param_info = None

        old_clock_list = old_inst.get_output_clock_list()
        for old_clk in old_clock_list:

            # Both basic and adv PLL has 3 output clocks. If downgrade to V3 PLL, don't need to
            # delete the output clk
            if self.migration_type == BlockMigrator.MigrationType.downgrade and old_clk.number > 2\
                    and (self.new_block_type == PeriDesign.BlockType.adv_pll or\
                      self.new_block_type == PeriDesign.BlockType.pll):
                self.migration_log.reg_param_change(
                    old_inst.name, f"Output clock {old_clk.number} is not supported, delete")
                continue

            new_clk = new_inst.create_output_clock(old_clk.name, old_clk.number)

            exclude_attr = ["_param_info", "param_group"]
            new_clk.copy_common(old_clk, exclude_attr)

            # New Instance's param_group is updated with old_inst's param_group's value, So it requires check and fallback
            if param_info is not None and self.old_block_type != db.PeriDesign.BlockType.adv_pll and\
                    self.old_block_type != db.PeriDesign.BlockType.pll:
                update_migrated_new_param(self.migration_log, new_clk, old_clk)

            if self.new_block_type == PeriDesign.BlockType.adv_pll and \
                    (self.old_block_type == PeriDesign.BlockType.comp_pll or\
                            self.old_block_type == PeriDesign.BlockType.efx_pll_v3_comp or
                            self.old_block_type == PeriDesign.BlockType.efx_fpll_v1):
                try:
                    # The return value is a float, but the requirement on phase_shift on AdvPLL
                    # output clock and schema is expecting integer
                    new_clk.phase_shift = int(old_inst.phase_setting2degree_by_outclk(old_clk.number))
                    if new_clk.phase_shift is None:
                        self.migration_log.reg_param_change(old_inst.name,
                                                            f"Fail to convert output clock {new_clk.number}'s phase shift in setting to phase degree. Reset to default")
                        new_clk.phase_shift = 0
                except dsg_excp.CalcException:
                    self.migration_log.reg_param_change(old_inst.name,
                                                        f"Fail to convert output clock {new_clk.number}'s phase shift in setting to phase degree. Reset to default")
                    new_clk.phase_shift = 0


@pt_util.freeze_it
class OSCMigrator(BlockMigrator):
    """
    Migrate OSC design
    """

    # Alias type for easy type hint, for multi-types block
    OSCInstTypeAlias: TypeAlias = Union[oscd.OSC, adv_oscd.OSCAdvance, hposcd.HPOSC]

    def __init__(self, new_design: db.PeriDesign, new_block_type: db.PeriDesign.BlockType,
                 old_block_type: db.PeriDesign.BlockType):
        super(OSCMigrator, self).__init__(new_design, new_block_type, old_block_type)
        self.migration_log.set_block_type(old_block_type, new_block_type)

        self.mig_direct_map = {
            (PeriDesign.BlockType.osc, PeriDesign.BlockType.adv_osc): BlockMigrator.MigrationType.upgrade,
            (PeriDesign.BlockType.osc, PeriDesign.BlockType.hposc): BlockMigrator.MigrationType.upgrade,
            (PeriDesign.BlockType.adv_osc, PeriDesign.BlockType.hposc): BlockMigrator.MigrationType.upgrade,

            (PeriDesign.BlockType.hposc, PeriDesign.BlockType.adv_osc): BlockMigrator.MigrationType.downgrade,
            (PeriDesign.BlockType.hposc, PeriDesign.BlockType.osc): BlockMigrator.MigrationType.downgrade,
            (PeriDesign.BlockType.adv_osc, PeriDesign.BlockType.osc): BlockMigrator.MigrationType.downgrade
        }  #: A map of a tuple of (old instance type, new instance type) to its migration direction

        self.migration_type = self.mig_direct_map.get((self.old_block_type, self.new_block_type),
                                                      BlockMigrator.MigrationType.unknown)

    def migrate(self):
        """
        Override
        """
        if self.new_block_type is None:
            self.design.osc_reg = None
        else:
            if self.design.osc_reg is None:
                svc = oscds.OSCDesignService.build_service(self.new_block_type)
                self.design.osc_reg = svc.create_registry()
            else:
                self.migrate_registry()

    def migrate_registry(self):
        """
        Override
        """
        old_reg = self.design.osc_reg
        svc = oscds.OSCDesignService.build_service(self.new_block_type)
        new_reg = svc.create_registry()
        new_reg.change_device(self.design.device_db)

        old_name2inst_map = old_reg.get_name2inst_map()
        for name, inst in old_name2inst_map.items():
            if inst is not None:
                if inst.osc_def == "":
                    new_inst = new_reg.create_instance(name, auto_pin=True)
                else:
                    new_inst = new_reg.create_instance_with_device(name, inst.osc_def, auto_pin=True)

                self.migrate_block(new_inst, inst)

        self.design.osc_reg = new_reg

    def migrate_block(self, new_inst: OSCInstTypeAlias, old_inst: OSCInstTypeAlias):
        """
        Override
        """

        # name and device resource have been set in creation
        if self.new_block_type != self.old_block_type:
            if self.migration_type == BlockMigrator.MigrationType.upgrade:
                self.upgrade_osc(new_inst, old_inst)
            elif self.migration_type == BlockMigrator.MigrationType.downgrade:
                self.downgrade_osc(new_inst, old_inst)
        else:
            new_inst.copy_common(old_inst)

    def upgrade_osc(self, new_inst: OSCInstTypeAlias, old_inst: OSCInstTypeAlias):
        """
        Migrate osc from simpler version to a more advance version

        :param new_inst: New osc instance
        :param old_inst: Old osc instance
        """
        new_inst.copy_common(old_inst)
        if self.old_block_type == db.PeriDesign.BlockType.osc:
            # Clockout pin in osc is an attribute, newer version is genpin
            clk_pin = new_inst.gen_pin.get_pin_by_type_name("CLKOUT")
            if clk_pin is not None:
                clk_pin.name = old_inst.clock_name

    def downgrade_osc(self, new_inst: OSCInstTypeAlias, old_inst: OSCInstTypeAlias):
        """
        Migrate osc from more advance version to a simpler version

        :param new_inst: New osc instance
        :param old_inst: Old osc instance
        """
        new_inst.copy_common(old_inst)
        if self.new_block_type == db.PeriDesign.BlockType.osc:
            clk_pin = old_inst.gen_pin.get_pin_by_type_name("CLKOUT")
            if clk_pin is not None:
                new_inst.clock_name = clk_pin.name

@pt_util.freeze_it
class DDRMigrator(BlockMigrator):
    """
    Migrate DDR design
    """

    # Alias type for easy type hint, for multi-types block
    DDRInstTypeAlias: TypeAlias = Union[ddrd.DDR, adv_ddrd.DDRAdvance]

    def __init__(self, new_design: db.PeriDesign, new_block_type: db.PeriDesign.BlockType,
                 old_block_type: db.PeriDesign.BlockType):
        super(DDRMigrator, self).__init__(new_design, new_block_type, old_block_type)
        self.migration_log.set_block_type(old_block_type, new_block_type)

        self.mig_direct_map = {
            (PeriDesign.BlockType.ddr, PeriDesign.BlockType.adv_ddr): BlockMigrator.MigrationType.upgrade,
            (PeriDesign.BlockType.adv_ddr, PeriDesign.BlockType.ddr): BlockMigrator.MigrationType.downgrade,

        }  #: A map of a tuple of (old instance type, new instance type) to its migration direction

        self.migration_type = self.mig_direct_map.get((self.old_block_type, self.new_block_type),
                                                      BlockMigrator.MigrationType.unknown)

        self.ddr_reset_msg = "Reset configuration to default except for AXI Enable status"

    def migrate(self):
        """
        Override
        """
        if self.new_block_type is None:
            self.design.ddr_reg = None
        else:
            if self.design.ddr_reg is None:
                svc = ddrds.DDRDesignService.build_service(self.new_block_type)
                self.design.ddr_reg = svc.create_registry()
            else:
                self.migrate_registry()

    def migrate_registry(self):
        """
        Override
        """
        old_reg = self.design.ddr_reg
        svc = ddrds.DDRDesignService.build_service(self.new_block_type)
        new_reg = svc.create_registry()
        new_reg.change_device(self.design.device_db)

        old_name2inst_map = old_reg.get_name2inst_map()
        for name, inst in old_name2inst_map.items():
            if inst is not None:
                if inst.ddr_def == "":
                    new_inst = new_reg.create_instance(name, auto_pin=True)
                else:
                    new_inst = new_reg.create_instance_with_device(name, inst.ddr_def, auto_pin=True)

                self.migrate_block(new_inst, inst)

        self.design.ddr_reg = new_reg

    def migrate_block(self, new_inst: DDRInstTypeAlias, old_inst: DDRInstTypeAlias):
        """
        Override
        """
        # name and device resource have been set in creation
        if self.new_block_type != self.old_block_type:
            if self.migration_type == BlockMigrator.MigrationType.upgrade:
                self.upgrade_ddr(new_inst, old_inst)
            elif self.migration_type == BlockMigrator.MigrationType.downgrade:
                self.downgrade_ddr(new_inst, old_inst)
        else:
            new_inst.copy_common(old_inst)

    def upgrade_ddr(self, new_inst: DDRInstTypeAlias, old_inst: DDRInstTypeAlias):
        """
        Migrate ddr from simpler version to a more advance version

        :param new_inst: New ddr instance
        :param old_inst: Old ddr instance
        """
        # Reset everything except specified later
        new_inst.build_default_configuration(self.design.device_db)

        # Retain the AXI enable
        new_inst.axi_target0.is_enabled = old_inst.axi_target0.is_enabled
        new_inst.axi_target1.is_enabled = old_inst.axi_target1.is_enabled

        # Copy the AXI clock pin
        self.copy_axi_clock_pin_name(old_inst, new_inst, 0)
        self.copy_axi_clock_pin_name(old_inst, new_inst, 1)

        self.migration_log.reg_param_change(old_inst.name, self.ddr_reset_msg)

    def copy_axi_clock_pin_name(self, old_ddr, new_ddr, target_id):
        if target_id == 0:
            old_axi = old_ddr.axi_target0
            new_axi = new_ddr.axi_target0
            gpin_name = "ACLK_0"
        else:
            old_axi = old_ddr.axi_target1
            new_axi = new_ddr.axi_target1
            gpin_name = "ACLK_1"

        ogpin_clk = None
        if old_axi is not None and old_axi.gen_pin is not None:
            ogpin_clk = old_axi.gen_pin.get_pin_by_type_name(gpin_name)

        ngpin_clk = None
        if new_axi is not None and new_axi.gen_pin is not None:
            ngpin_clk = new_axi.gen_pin.get_pin_by_type_name(gpin_name)

        if ogpin_clk is not None and ngpin_clk is not None:
            ngpin_clk.name = ogpin_clk.name

    def downgrade_ddr(self, new_inst: DDRInstTypeAlias, old_inst: DDRInstTypeAlias):
        """
        Migrate ddr from more advance version to a simpler version

        :param new_inst: New ddr instance
        :param old_inst: Old ddr instance
        """
        # Reset everything except specified later
        new_inst.build_default_configuration(self.design.device_db)

        # Retain the AXI enable
        new_inst.axi_target0.is_enabled = old_inst.axi_target0.is_enabled
        new_inst.axi_target1.is_enabled = old_inst.axi_target1.is_enabled

        # Copy the AXI clock pin
        self.copy_axi_clock_pin_name(old_inst, new_inst, 0)
        self.copy_axi_clock_pin_name(old_inst, new_inst, 1)

        self.migration_log.reg_param_change(old_inst.name, self.ddr_reset_msg)

@pt_util.freeze_it
class LVDSMigrator(BlockMigrator):
    """
    Migrate LVDS design
    """

    # Alias type for easy type hint, for multi-types block

    def __init__(self, new_design: db.PeriDesign, new_block_type: db.PeriDesign.BlockType,
                 old_block_type: db.PeriDesign.BlockType):
        super(LVDSMigrator, self).__init__(new_design, new_block_type, old_block_type)
        self.migration_log.set_block_type(old_block_type, new_block_type)
        self.mig_direct_map = {
            (PeriDesign.BlockType.lvds, PeriDesign.BlockType.adv_lvds): BlockMigrator.MigrationType.upgrade,
            (PeriDesign.BlockType.adv_lvds, PeriDesign.BlockType.lvds): BlockMigrator.MigrationType.downgrade
        }  #: A map of a tuple of (old instance type, new instance type) to its migration direction

        self.migration_type = self.mig_direct_map.get((self.old_block_type, self.new_block_type),
                                                      BlockMigrator.MigrationType.unknown)
        self.serwidth_change_msg = "Serialization width is not supported, reset to default"
        self.conntype_change_msg = "Existing Connection Type is unsupported, reset to default"
        self.deserwidth_change_msg = "Deserialization width is not supported, reset to default"
        self.termtype_change_msg = "Termination type is not supported, reset to default"

    def migrate(self):
        """
        Override
        """
        if self.new_block_type is None:
            self.design.lvds_reg = None
        else:
            if self.design.lvds_reg is None:
                svc = lvdsds.LVDSDesignService.build_service(self.new_block_type)
                self.design.lvds_reg = svc.create_registry()
            else:
                self.migrate_registry()

    def migrate_registry(self):
        """
        Override
        """
        old_reg = self.design.lvds_reg
        svc = lvdsds.LVDSDesignService.build_service(self.new_block_type)
        new_reg = svc.create_registry()
        new_reg.change_device(self.design.device_db)

        old_name2inst_map = old_reg.get_name2inst_map()
        for name, inst in old_name2inst_map.items():
            if inst is not None:
                # Bidir instance not migrated unless it's the same block type
                if inst.ops_type == adv_lvdsd.LVDSAdvance.OpsType.op_bidir:
                    if self.new_block_type != self.old_block_type:
                        continue

                if inst.get_device() == "":
                    new_inst = new_reg.create_instance(name, auto_pin=True)
                else:
                    new_inst = new_reg.create_instance_with_device(name, inst.get_device(), auto_pin=True)

                self.migrate_block(new_inst, inst)

        self.design.lvds_reg = new_reg

    def migrate_block(self, new_inst: Union[lvdsd.LVDS, adv_lvdsd.LVDSAdvance], old_inst: Union[lvdsd.LVDS, adv_lvdsd.LVDSAdvance]):
        """
        Override.
        """

        # name and device resource have been set in creation
        if self.new_block_type != self.old_block_type:

            is_tx = False
            is_rx = False
            if old_inst.tx_info is not None:
                is_tx = True

            if old_inst.rx_info is not None:
                is_rx = True

            # Bidir is not migrated, by right it shouldn't reach this
            if is_tx and is_rx:
                return

            # For member with different type or semantic, do manual migration
            # e.g. serial width vs serialization enum

            # Migrate from newer object to LVDS object
            if self.migration_type == self.MigrationType.downgrade:

                if is_tx:
                    self.downgrade_tx(new_inst, old_inst)

                elif is_rx:
                    self.downgrade_rx(new_inst, old_inst)

            # Migrate from LVDS to newer object
            elif self.migration_type == self.MigrationType.upgrade:

                if is_tx:
                    self.upgrade_tx(new_inst, old_inst)

                elif is_rx:
                    self.upgrade_rx(new_inst, old_inst)
        else:
            new_inst.copy_common(old_inst)

    def downgrade_tx(self, new_inst: Union[lvdsd.LVDS, adv_lvdsd.LVDSAdvance], old_inst: Union[lvdsd.LVDS, adv_lvdsd.LVDSAdvance]):
        """
        Migrate advanced lvds tx and to basic lvds tx

        :param new_inst: New lvds instance
        :param old_inst: Old lvds instance
        """

        new_inst.update_type(new_inst.OpsType.op_tx, self.design.device_db, True)
        otx_info = old_inst.tx_info
        ntx_info = new_inst.tx_info

        exclude_attr = ["is_reduced_swing", "output_load", "clock_div"]
        ntx_info.copy_common(otx_info, exclude_attr)
        # Serialization status takes precendence over actual width
        if otx_info.is_serial:
            param_info = otx_info.get_param_info()
            if param_info.is_valid_setting(otx_info.param_id.serial_width, otx_info.serial_width):
                serial_type_list = list(ntx_info.SerialType)
                serial_type = serial_type_list[otx_info.serial_width - 1]
                ntx_info.serialization = serial_type
            else:
                self.migration_log.reg_param_change(old_inst.name, self.serwidth_change_msg)
                ntx_info.serialization = ntx_info.SerialType.st8
        else:
            ntx_info.serialization = ntx_info.SerialType.st1

    def downgrade_rx(self, new_inst: Union[lvdsd.LVDS, adv_lvdsd.LVDSAdvance], old_inst: Union[lvdsd.LVDS, adv_lvdsd.LVDSAdvance]):
        """
        Migrate advanced lvds rx and to basic lvds tx

        :param new_inst: New lvds instance
        :param old_inst: Old lvds instance
        """

        new_inst.update_type(new_inst.OpsType.op_rx, self.design.device_db, True)
        orx_info = old_inst.rx_info
        nrx_info = new_inst.rx_info

        nrx_info.copy_common(orx_info)

        param_info = orx_info.get_param_info()

        conn_type_str = orx_info.conn2str_map.get(orx_info.conn_type, None)
        if conn_type_str is None:
            self.migration_log.reg_param_change(old_inst.name, self.conntype_change_msg)
        nrx_info.conn_type = nrx_info.str2conn_map.get(conn_type_str, nrx_info.ConnType.normal_conn)

        # Serialization status takes precendence over actual width
        if orx_info.is_serial:
            if param_info.is_valid_setting(orx_info.param_id.serial_width, orx_info.serial_width):
                serial_type_list = list(nrx_info.SerialType)
                serial_type = serial_type_list[orx_info.serial_width - 1]
                nrx_info.serialization = serial_type
            else:
                self.migration_log.reg_param_change(old_inst.name, self.deserwidth_change_msg)
                nrx_info.serialization = nrx_info.SerialType.st8
        else:
            nrx_info.serialization = nrx_info.SerialType.st1

        if orx_info.termination == orx_info.TerminationType.dynamic:
            nrx_info.is_terminated = True
            self.migration_log.reg_param_change(old_inst.name, self.termtype_change_msg)
        elif orx_info.termination == orx_info.TerminationType.off:
            nrx_info.is_terminated = False
        else:
            nrx_info.is_terminated = True

        if orx_info.delay_mode == orx_info.DelayModeType.static:
            nrx_info.is_delay_ena = True
        else:
            nrx_info.is_delay_ena = False

    def upgrade_tx(self, new_inst: Union[lvdsd.LVDS, adv_lvdsd.LVDSAdvance], old_inst: Union[lvdsd.LVDS, adv_lvdsd.LVDSAdvance]):
        """
        Migrate basic lvds tx and to advanced lvds tx

        :param new_inst: New lvds instance
        :param old_inst: Old lvds instance
        """

        new_inst.update_type(new_inst.OpsType.op_tx, self.design.device_db, True)
        otx_info = old_inst.tx_info
        ntx_info = new_inst.tx_info
        adv_param_info = ntx_info.get_param_info()

        ntx_info.copy_common(otx_info)
        if otx_info.serialization is not None and otx_info.serialization.is_enable_serial():
            ntx_info.is_serial = True
            ntx_info.serial_width = otx_info.serialization.value
        else:
            ntx_info.is_serial = False
            ntx_info.serial_width = adv_param_info.get_default(ntx_info.param_id.serial_width)

    def upgrade_rx(self, new_inst: Union[lvdsd.LVDS, adv_lvdsd.LVDSAdvance], old_inst: Union[lvdsd.LVDS, adv_lvdsd.LVDSAdvance]):
        """
        Migrate basic lvds rx and to advanced lvds rx

        :param new_inst: New lvds instance
        :param old_inst: Old lvds instance
        """
        new_inst.update_type(new_inst.OpsType.op_rx, self.design.device_db, True)
        orx_info = old_inst.rx_info
        nrx_info = new_inst.rx_info
        assert orx_info is not None
        assert nrx_info is not None
        adv_param_info = nrx_info.get_param_info()

        nrx_info.copy_common(orx_info)

        conn_type_str = orx_info.conn2str_map.get(orx_info.conn_type, None)
        if conn_type_str is None:
            self.migration_log.reg_param_change(old_inst.name, self.conntype_change_msg)
        nrx_info.conn_type = nrx_info.str2conn_map.get(conn_type_str, nrx_info.ConnType.normal_conn)

        if orx_info.serialization is not None and orx_info.serialization.is_enable_serial():
            nrx_info.is_serial = True
            nrx_info.serial_width = orx_info.serialization.value
        else:
            nrx_info.is_serial = False
            nrx_info.serial_width = adv_param_info.get_default(nrx_info.param_id.serial_width)

        if orx_info.is_terminated:
            nrx_info.termination = nrx_info.TerminationType.on
        else:
            nrx_info.termination = nrx_info.TerminationType.off

        if orx_info.is_delay_ena:
            nrx_info.delay_mode = nrx_info.DelayModeType.static
        else:
            nrx_info.delay_mode = adv_param_info.get_default(nrx_info.param_id.delay_mode)

@pt_util.freeze_it
class MIPIHardDPHYMigrator(BlockMigrator):
    """
    Migrate MIPI Hard DPHY design
    """

    # Alias type for easy type hint, for multi-types block
    MIPIHardDphyInstTypeAlias: TypeAlias = mhdphyd.MIPIHardDPHYTx | mhdphyd.MIPIHardDPHYRx

    def __init__(self, new_design: db.PeriDesign, new_block_type: db.PeriDesign.BlockType,
                 old_block_type: db.PeriDesign.BlockType):
        super().__init__(new_design, new_block_type, old_block_type)
        self.migration_log.set_block_type(old_block_type, new_block_type)
        self.mig_direct_map: Dict[Tuple[PeriDesign.BlockType, PeriDesign.BlockType], BlockMigrator.MigrationType] = {}
        self.migration_type = BlockMigrator.MigrationType.unknown

    def migrate(self):
        """
        Override
        """
        if self.new_block_type is None:
            self.design.mipi_hard_dphy_reg = None
        else:
            if self.design.mipi_hard_dphy_reg is None:
                svc = mhdphyds.MIPIHardDPHYDesignService()
                self.design.mipi_hard_dphy_reg = svc.create_registry()
            else:
                self.migrate_registry()

    def migrate_registry(self):
        """
        Override
        """
        old_reg = self.design.mipi_hard_dphy_reg
        svc = mhdphyds.MIPIHardDPHYDesignService()
        new_reg = svc.create_registry()
        new_reg.change_device(self.design.device_db)

        assert old_reg is not None
        old_name2inst_map = old_reg.get_name2inst_map()
        for name, inst in old_name2inst_map.items():
            if inst is not None:
                # PT-2661:
                if (not self.design.is_device_block_tx_rx_supported(PeriDesign.BlockType.mipi_hard_dphy, True) and\
                   inst.ops_type == mipid.MIPI.OpsType.op_tx) or \
                   (not self.design.is_device_block_tx_rx_supported(PeriDesign.BlockType.mipi_hard_dphy, False) and\
                   inst.ops_type == mipid.MIPI.OpsType.op_rx):
                    # Skip creating the instance in new device if it doesn't supports it
                    continue

                if inst.ops_type == mipid.MIPI.OpsType.op_tx:
                    new_inst = new_reg.create_tx_instance(name, auto_pin=True)
                else:
                    new_inst = new_reg.create_rx_instance(name, auto_pin=True)

                if inst.block_def != "":
                    new_reg.assign_inst_device(new_inst, inst.block_def)

                self.migrate_block(new_inst, inst) # type: ignore

        self.design.mipi_hard_dphy_reg = new_reg

    def migrate_block(self, new_inst: MIPIHardDphyInstTypeAlias, old_inst: MIPIHardDphyInstTypeAlias):
        """
        Override
        """
        new_inst.copy_common(old_inst)


@pt_util.freeze_it
class DesignMigrator:
    """
    Migrate design from one device to another.

    The migration goal is to ensure that the design can be loaded, can be checked for design error
    and any issue can be handled through manual edits.

    The migration process does not hardcode based on family but instead relies on supported block types
    for a particular device.

    The expected migration scenarios
    a) moving from small device to big device
    - can be within the same family or across family

    b) moving from big device to small device
    - can be within the same family or across family

    c) moving from simple to advance block type ie PLL
    - moving across family

    d) moving from simple to advance device type ie an08 to an20
    - moving across family

    Migration outcomes must ensure correct design db structure matching currect device and all invalid resources
    are reset to prevent downstream error.

    Design db structure affected by migration are
    - block registry existance
    - block registry python class type
    - multi version block instance python class type e.g pll
    - cross registry block instance e.g. lvds gpio
    - default value when migrating between "family"

    """

    @dataclass
    class BlockMigrationData:
        old_block_type: db.PeriDesign.BlockType
        new_block_type: db.PeriDesign.BlockType

    class MigrationType(Enum):
        """
        Migration Type defined in PT-773:
        Default: Covers the Package and No Migration type. We don't need
                to differentiate them as it is not useful:
            Package Migration: Same die, different package - This include VC
                                devices with VC migration although it is same
                                die and same package. Because the resource
                                is slightly different (H264).
            No migration: Different die, different package
        Virtual Migration: Same die, same package migration between real
                            and virtual or two virtual parts. An exception
                            is for VC devices where it's not considered
                            Virtual since there's a difference in resource
                            although die and package are the same
                            (T70F324VC -> T120F324: Package Migration).        
        Vertical Migration: Different die, same package        
        """
        default = 1
        virtual = 2
        vertical = 3

    def __init__(self):
        """
        Constructor
        """
        self.logger = Logger
        self.design = None
        self.is_old_design_tesseract = False
        self.is_new_design_tesseract = False

        # Check valid new device, by default it should be done in build
        self.is_check_device = False
        # We don't need to store the new device since during migrate(),
        # the design will have the new device name saved
        self.old_device = ""
        self.old_device_supp_block_list = None
        self.old_device_db = None

        self.mtype = self.MigrationType.default

        self.migration_log = MigrationLog()
        self.migration_log_path = ""
        self.is_log_enable = False  #: Set to False so that by default unit test doesn't generate log

        self.log_file_handle = None

        self.gpio_migrator = None
        self.pll_migrator = None
        self.osc_migrator = None
        self.lvds_migrator = None
        self.ddr_migrator = None
        self.mdphy_migrator = None

    @staticmethod
    def determine_migration_type(old_device_name, new_device_name):
        """
        Determine the migration type depending on the current
        device and the new device name. It is done in two stages:
        1) Determine if it is a Virtual migration
        2) If not, determine if it is a Vertical migration
        3) If none of the above, then it is the default migration process

        :return True if the migration file is found or it is virtual migration. Else, False.
        """
        file_found = True
        logger = Logger

        # Get the device filenames
        dev_service = dev_srv.DeviceService()
        _, old_dev_file, _ = dev_service.parse_device_map_file(search_name=old_device_name)
        _, new_dev_file, _ = dev_service.parse_device_map_file(search_name=new_device_name)

        # Default migration type
        mtype = DesignMigrator.MigrationType.default

        # If the filenames are exactly the same, then it's a Virtual migration
        if old_dev_file == new_dev_file:
            mtype = DesignMigrator.MigrationType.virtual
        else:
            # Check if it could be a Vertical Migration
            dvm_obj = dev_srv.DeviceVerticalMigration.build_from_default_file()

            if dvm_obj is not None:

                if dvm_obj.is_device_vertical_migration(old_device_name, new_device_name):
                    mtype = DesignMigrator.MigrationType.vertical

            # Yasmin suggested that if it cannot find the file, we
            # warn the user that it's going to use default (resource
            # migration)
            else:
                file_found = False
                logger.warning("Defaulting to resource based migration due to file {} not found".format(
                    dev_srv.DeviceVerticalMigration.get_db_migration_file()))

        return mtype, file_found

    def enable_migration_log(self, is_enable):
        """
        Enable/disable writing of migration log

        :param is_enable: True, enable writing, else no
        """
        self.is_log_enable = is_enable

    def setup_log_file(self, location=""):

        if not self.is_log_enable:
            return ""

        app_setting = AppSetting()
        if location == "":
            self.migration_log_path = f"{self.design.location}/{app_setting.output_dir}"
        else:
            self.migration_log_path = location

        log_file = f"{self.migration_log_path}/{app_setting.migration_log_file}"
        if os.path.exists(self.migration_log_path) is False:
            os.mkdir(self.migration_log_path)

        return log_file

    def write_log(self):

        if not self.is_log_enable:
            return

        self.migration_log.write_header(self.log_file_handle)
        self.migration_log.write_log(self.log_file_handle)
        # Order of writing
        migrator_list = [self.gpio_migrator, self.pll_migrator, self.lvds_migrator, self.osc_migrator]
        for migrator in migrator_list:
            if migrator is not None:
                mig_log = migrator.migration_log
                mig_log.write_log(self.log_file_handle)

    def migrate(self, design: PeriDesign, new_device: str, log_location: str=""):
        """
        Execute design migration.
        
        :param design: The design database
        :param new_device: The new device to migrate the design to
        :return a flag that indicates whether the migration file was found (True) or not
        """

        if design is None:
            return

        if design.device_def == new_device:
            return

        if new_device == "":
            return

        self.design = design
        self.old_device_supp_block_list = self.design.get_supported_block()

        # Once device supports family, we need to update this
        if self.design.is_tesseract_design():
            self.is_old_design_tesseract = True

        # Update the device name stored in the design
        self.old_device = design.device_def
        design.device_def = new_device

        def run_migration():
            self.logger.debug("Device migration from {} to {}".format(self.old_device, new_device))
            self.migration_log.add_log("Device Change", "{} to {}".format(self.old_device, new_device))

            # TODO : The design is fully loaded old design, why can't we get type directly from design db?
            # Determine the migration type
            self.mtype, mig_file_found = DesignMigrator.determine_migration_type(self.old_device,
                                                                                 self.design.device_def)

            msg = "Migration type identified: {}".format(self.mtype)
            self.logger.debug(msg)
            self.migration_log.add_log("Migration Type", f"{self.mtype.name}")

            if self.mtype == self.MigrationType.virtual:
                if design.is_tesseract_design():
                    self.is_new_design_tesseract = True

                # Should we check that the plugin is the same
                if self.design.plugin_map is not None and \
                        not self.design.plugin_map.is_identical_plugin(self.old_device, new_device):
                    # Device is the same but plugin different (This should not happen)
                    msg = "Error migrating from {} to {}".format(
                        self.old_device, new_device)
                    raise db_excp.DevicePluginException(msg, app_excp.MsgLevel.error)

                # We don't need to recreate the device and design since
                # it is exactly the same. Just make sure the device
                # name is updated
                device_db = design.device_db
                if device_db is not None:
                    device_db.set_device_name(design.device_def)

                    self.logger.debug("Virtual device migration from {} to {}".format(
                        self.old_device, new_device))

            else:
                self.design.identify_supported_block()
                if design.is_tesseract_design():
                    self.is_new_design_tesseract = True

                preserve_pin = False
                if self.mtype == self.MigrationType.vertical:
                    preserve_pin = True
                    # Save the old device so we can get the
                    # map of resource change
                    self.old_device_db = design.device_db

                self.rebuild_device(design)
                self._rebuild_device_setting(design, self.design.plugin_map, new_device)
                self._rebuild_design(preserve_pin, self.design.plugin_map)

                # Populate clkmux info (need to be done after the registry of all
                # blocks are built) unless if the old and new clkmux is the same (retain
                # the old clkmux)
                if self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx60_clkmux") or \
                        self.design.plugin_map.is_plugin_exist(self.design.device_def, "tx180_clkmux"):
                    if self.design.device_setting is not None and self.design.device_setting.clkmux_reg is not None:
                        self.design.device_setting.clkmux_reg.load_device_clkmux_ins(
                            self.design.device_db, self.design)

            return mig_file_found

        if self.is_log_enable:
            mig_log_file = self.setup_log_file(log_location)
            with open(mig_log_file, "w", encoding='UTF-8') as file_handle:
                self.log_file_handle = file_handle
                mig_file_found = run_migration()
                self.write_log()
        else:
            mig_file_found = run_migration()

        return mig_file_found

    def _determine_old_multi_block_type_reg(self, plugin_map):
        """
        Determine old design registry that has multiple versions of the same block type

        :param plugin_map: Device plugin map
        """

        old_pll_block_type = None
        if self.design.pll_reg is not None:
            if isinstance(self.design.pll_reg, v3comp_plld.PLLRegistryV3Complex):
                old_pll_block_type = db.PeriDesign.BlockType.efx_pll_v3_comp
            elif isinstance(self.design.pll_reg, comp_plld.PLLRegistryComplex):
                old_pll_block_type = db.PeriDesign.BlockType.comp_pll
            elif isinstance(self.design.pll_reg, adv_plld.PLLRegistryAdvance):
                old_pll_block_type = db.PeriDesign.BlockType.adv_pll
            else:
                old_pll_block_type = db.PeriDesign.BlockType.pll

        old_osc_block_type = None
        if self.design.osc_reg is not None:
            if isinstance(self.design.osc_reg, hposcd.HPOSCRegistry):
                old_osc_block_type = db.PeriDesign.BlockType.hposc
            elif isinstance(self.design.osc_reg, adv_oscd.OSCRegistryAdvance):
                old_osc_block_type = db.PeriDesign.BlockType.adv_osc
            else:
                old_osc_block_type = db.PeriDesign.BlockType.osc

        # GPIO has 3 types but only 2 real registry type. Because the adv type is just
        # to distinguish design feature for the an120 gpio.
        old_gpio_block_type = None
        if self.design.gpio_reg is not None:
            if isinstance(self.design.gpio_reg, comp_gpiod.GPIORegistryComplex):
                old_gpio_block_type = db.PeriDesign.BlockType.comp_gpio
            else:
                # Get the gpio plugin on the old device
                if plugin_map is not None:
                    gpio_old_grp_type = plugin_map.get_group_by_type_name(self.old_device, "gpio")
                    if gpio_old_grp_type == plmap.PluginMap.GroupType.tx60:
                        old_gpio_block_type = db.PeriDesign.BlockType.comp_gpio
                    elif gpio_old_grp_type == plmap.PluginMap.GroupType.an120:
                        old_gpio_block_type = self.design.BlockType.adv_l2_gpio
                    else:
                        old_gpio_block_type = self.design.BlockType.gpio
                else:
                    # We just set to the basic one
                    old_gpio_block_type = self.design.BlockType.gpio

        old_lvds_block_type = None
        if self.design.lvds_reg is not None:
            if isinstance(self.design.lvds_reg, adv_lvdsd.LVDSRegistryAdvance):
                old_lvds_block_type = db.PeriDesign.BlockType.adv_lvds
            else:
                old_lvds_block_type = db.PeriDesign.BlockType.lvds

        old_ddr_block_type = None
        if self.design.ddr_reg is not None:
            if isinstance(self.design.ddr_reg, adv_ddrd.DDRRegistryAdvance):
                old_ddr_block_type = db.PeriDesign.BlockType.adv_ddr
            else:
                old_ddr_block_type = db.PeriDesign.BlockType.ddr

        old_mdphy_block_type = None
        if self.design.mipi_hard_dphy_reg is not None:
            if isinstance(self.design.mipi_hard_dphy_reg, mhdphyd.MIPIHardDPHYRegistry):
                old_mdphy_block_type = db.PeriDesign.BlockType.mipi_hard_dphy

        return old_gpio_block_type, old_pll_block_type, old_osc_block_type, \
               old_lvds_block_type, old_ddr_block_type, old_mdphy_block_type

    def _rebuild_multi_block_type_reg(self, all_supported_reg: List, block_mig_data: Dict):
        """
        Rebuild registry for block with multiple types

        :param all_supported_reg: A list of block registry
        :param block_mig_data: Migration data for each affected block
        """
        assert self.design is not None

        mig_data = block_mig_data.get("gpio", None)
        if mig_data is not None:
            self.gpio_migrator = GPIOMigrator(self.design, mig_data.new_block_type, mig_data.old_block_type)
            self.gpio_migrator.migrate()
            all_supported_reg.append(self.design.gpio_reg)
        else:
            if self.design.gpio_reg is not None:
                self.design.gpio_reg = None

        mig_data = block_mig_data.get("pll", None)
        if mig_data is not None:
            self.pll_migrator = PLLMigrator(self.design, mig_data.new_block_type, mig_data.old_block_type)
            self.pll_migrator.migrate()
            all_supported_reg.append(self.design.pll_reg)
        else:
            # Not supported, clear it
            if self.design.pll_reg is not None:
                self.design.pll_reg = None

        mig_data = block_mig_data.get("osc", None)
        if mig_data is not None:
            self.osc_migrator = OSCMigrator(self.design, mig_data.new_block_type, mig_data.old_block_type)
            self.osc_migrator.migrate()
            all_supported_reg.append(self.design.osc_reg)
        else:
            if self.design.osc_reg is not None:
                self.design.osc_reg = None

        mig_data = block_mig_data.get("lvds", None)
        if mig_data is not None:
            self.lvds_migrator = LVDSMigrator(self.design, mig_data.new_block_type, mig_data.old_block_type)
            self.lvds_migrator.migrate()
            all_supported_reg.append(self.design.lvds_reg)
        else:
            if self.design.lvds_reg is not None:
                self.design.lvds_reg = None

        mig_data = block_mig_data.get("ddr", None)
        if mig_data is not None:
            self.ddr_migrator = DDRMigrator(self.design, mig_data.new_block_type, mig_data.old_block_type)
            self.ddr_migrator.migrate()
            all_supported_reg.append(self.design.ddr_reg)
        else:
            if self.design.ddr_reg is not None:
                self.design.ddr_reg = None

        mig_data = block_mig_data.get("mipi_hard_dphy", None)
        if mig_data is not None:
            self.mdphy_migrator = MIPIHardDPHYMigrator(self.design, mig_data.new_block_type, mig_data.old_block_type)
            self.mdphy_migrator.migrate()
            all_supported_reg.append(self.design.mipi_hard_dphy_reg)
        else:
            if self.design.mipi_hard_dphy_reg is not None:
                self.design.mipi_hard_dphy_reg = None


    def _rebuild_design(self, preserve_pin, plugin_map):
        """
        Make sure only supported block registry exist.
        If the registry contains invalid device resource, clears it.
        This ensure there's no donwstream error that assumes all devices are valid.
        """

        if self.design is None:
            return

        self.design.clear_block_map()

        count = self.design.get_supported_block_count()
        if count == 0:
            return

        all_supported_reg = []

        # A map of block name to its migration data
        block_mig_data = {}

        # At this point, the design is still the old (loaded) design but the list
        # of supported blocks is based on the new device.  So, the block_type in design
        # are all referring to the new design when migrated
        blk_types = self._determine_old_multi_block_type_reg(plugin_map)
        old_gpio_block_type, old_pll_block_type, old_osc_block_type, \
            old_lvds_block_type, old_ddr_block_type, old_mdphy_block_type = blk_types

        # Check each registry and delete/create based on supported block type
        for block_type_item in db.PeriDesign.BlockType:

            block_type = db.PeriDesign.BlockType(block_type_item.value)
            if block_type == db.PeriDesign.BlockType.other:
                continue

            # Check if the block is supported on the new device
            # If single block type, create a new registry
            # Else handled in multi block type registry builder
            if self.design.is_block_supported(block_type):

                if block_type == db.PeriDesign.BlockType.gpio or \
                        block_type == db.PeriDesign.BlockType.adv_l2_gpio or \
                        block_type == db.PeriDesign.BlockType.comp_gpio:
                    block_mig_data["gpio"] = DesignMigrator.BlockMigrationData(old_gpio_block_type, block_type)

                elif block_type == db.PeriDesign.BlockType.pll or \
                        block_type == db.PeriDesign.BlockType.adv_pll or \
                        block_type == db.PeriDesign.BlockType.comp_pll or \
                        block_type == db.PeriDesign.BlockType.efx_pll_v3_comp or \
                        block_type == db.PeriDesign.BlockType.efx_fpll_v1:
                    block_mig_data["pll"] = DesignMigrator.BlockMigrationData(old_pll_block_type, block_type)

                elif block_type == db.PeriDesign.BlockType.osc \
                        or block_type == db.PeriDesign.BlockType.adv_osc \
                        or block_type == db.PeriDesign.BlockType.hposc:
                    block_mig_data["osc"] = DesignMigrator.BlockMigrationData(old_osc_block_type, block_type)

                elif block_type == db.PeriDesign.BlockType.lvds or \
                        block_type == db.PeriDesign.BlockType.adv_lvds:
                    block_mig_data["lvds"] = DesignMigrator.BlockMigrationData(old_lvds_block_type, block_type)

                elif block_type == db.PeriDesign.BlockType.mipi:
                    svc = mipids.MIPIDesignService()
                    if self.design.mipi_reg is None:
                        self.design.mipi_reg = svc.create_registry()
                        self.migration_log.reg_block_change(None, block_type)

                    all_supported_reg.append(self.design.mipi_reg)

                elif block_type == db.PeriDesign.BlockType.jtag:
                    svc = jtagds.JTAGDesignService()
                    if self.design.jtag_reg is None:
                        self.design.jtag_reg = svc.create_registry()
                        self.migration_log.reg_block_change(None, block_type)

                    all_supported_reg.append(self.design.jtag_reg)

                elif block_type == db.PeriDesign.BlockType.h264:
                    svc = h264ds.H264DesignService()
                    if self.design.h264_reg is None:
                        self.design.h264_reg = svc.create_registry()
                        self.migration_log.reg_block_change(None, block_type)

                    all_supported_reg.append(self.design.h264_reg)

                elif block_type == db.PeriDesign.BlockType.ddr or \
                    block_type == db.PeriDesign.BlockType.adv_ddr:
                    block_mig_data["ddr"] = DesignMigrator.BlockMigrationData(old_ddr_block_type, block_type)

                elif block_type == db.PeriDesign.BlockType.mipi_dphy:
                    svc = mdphyds.MIPIDPhyDesignService()
                    if self.design.mipi_dphy_reg is None:
                        self.design.mipi_dphy_reg = svc.create_registry()
                        self.migration_log.reg_block_change(None, block_type)

                    all_supported_reg.append(self.design.mipi_dphy_reg)

                elif block_type == db.PeriDesign.BlockType.spi_flash:
                    svc = spids.SPIFlashDesignService()
                    if self.design.spi_flash_reg is None:
                        self.design.spi_flash_reg = svc.create_registry()
                        self.migration_log.reg_block_change(None, block_type)

                    all_supported_reg.append(self.design.spi_flash_reg)

                elif block_type == db.PeriDesign.BlockType.hyper_ram:
                    svc = hyperds.HyperRAMDesignService()
                    if self.design.hyper_ram_reg is None:
                        self.design.hyper_ram_reg = svc.create_registry()
                        self.migration_log.reg_block_change(None, block_type)

                    all_supported_reg.append(self.design.hyper_ram_reg)

                elif block_type == db.PeriDesign.BlockType.mipi_hard_dphy:
                    block_mig_data["mipi_hard_dphy"] = DesignMigrator.BlockMigrationData(
                        old_mdphy_block_type, block_type)

                elif block_type == db.PeriDesign.BlockType.pll_ssc:
                    svc = pllsscds.PLLSSCDesignService()
                    if self.design.pll_ssc_reg is None:
                        self.design.pll_ssc_reg = svc.create_registry()
                        self.migration_log.reg_block_change(None, block_type)

                    all_supported_reg.append(self.design.pll_ssc_reg)

                elif block_type == db.PeriDesign.BlockType.common_quad:
                    svc = common_quads.QuadLaneCommonDesignService(self.design.device_db)
                    if self.design.common_quad_lane_reg is None:
                        self.design.common_quad_lane_reg = svc.create_registry()
                        self.migration_log.reg_block_change(None, block_type)

                    all_supported_reg.append(self.design.common_quad_lane_reg)

                elif block_type == db.PeriDesign.BlockType.quad_pcie:
                    svc = quad_pcieds.QuadPCIEDesignService()
                    if self.design.quad_pcie_reg is None:
                        self.design.quad_pcie_reg = svc.create_registry()
                        self.migration_log.reg_block_change(None, block_type)

                    all_supported_reg.append(self.design.quad_pcie_reg)

                elif block_type == db.PeriDesign.BlockType.lane_10g:
                    svc = lane10gds.Lane10GDesignService()
                    if self.design.lane_10g_reg is None:
                        self.design.lane_10g_reg = svc.create_registry()
                        self.design.lane_10g_reg.common_quad_reg = self.design.common_quad_lane_reg
                        self.design.lane_10g_reg.load_settings_file(self.design.location)
                        self.migration_log.reg_block_change(None, block_type)

                    all_supported_reg.append(self.design.lane_10g_reg)

                elif block_type == db.PeriDesign.BlockType.lane_1g:
                    svc = lane1gds.Lane1GDesignService()
                    if self.design.lane_1g_reg is None:
                        self.design.lane_1g_reg = svc.create_registry()
                        self.design.lane_1g_reg.common_quad_reg = self.design.common_quad_lane_reg
                        self.migration_log.reg_block_change(None, block_type)

                    all_supported_reg.append(self.design.lane_1g_reg)

                elif block_type == db.PeriDesign.BlockType.raw_serdes:
                    svc = raw_serdesds.RawSerdesDesignService()
                    if self.design.raw_serdes_reg is None:
                        self.design.raw_serdes_reg = svc.create_registry()
                        self.design.raw_serdes_reg.common_quad_reg = self.design.common_quad_lane_reg
                        #self.design.raw_serdes_reg.load_settings_file(self.design.location)
                        self.migration_log.reg_block_change(None, block_type)

                    all_supported_reg.append(self.design.raw_serdes_reg)

            else:
                # Clear unsupported block registry for single block type
                if block_type == db.PeriDesign.BlockType.mipi:
                    if self.design.mipi_reg is not None:
                        self.design.mipi_reg = None
                        self.migration_log.reg_block_change(block_type, None)

                elif block_type == db.PeriDesign.BlockType.jtag:
                    if self.design.jtag_reg is not None:
                        self.design.jtag_reg = None
                        self.migration_log.reg_block_change(block_type, None)

                elif block_type == db.PeriDesign.BlockType.h264:
                    if self.design.h264_reg is not None:
                        self.design.h264_reg = None
                        self.migration_log.reg_block_change(block_type, None)

                elif block_type == db.PeriDesign.BlockType.mipi_dphy:
                    if self.design.mipi_dphy_reg is not None:
                        self.design.mipi_dphy_reg = None
                        self.migration_log.reg_block_change(block_type, None)

                elif block_type == db.PeriDesign.BlockType.spi_flash:
                    if self.design.spi_flash_reg is not None:
                        self.design.spi_flash_reg = None
                        self.migration_log.reg_block_change(block_type, None)

                elif block_type == db.PeriDesign.BlockType.hyper_ram:
                    if self.design.hyper_ram_reg is not None:
                        self.design.hyper_ram_reg = None
                        self.migration_log.reg_block_change(block_type, None)

                elif block_type == db.PeriDesign.BlockType.pll_ssc:
                    if self.design.pll_ssc_reg is not None:
                        self.design.pll_ssc_reg = None
                        self.migration_log.reg_block_change(block_type, None)

                elif block_type == db.PeriDesign.BlockType.common_quad:
                    if self.design.common_quad_lane_reg is not None:
                        self.design.common_quad_lane_reg = None
                        self.migration_log.reg_block_change(block_type, None)

                elif block_type == db.PeriDesign.BlockType.quad_pcie:
                    if self.design.quad_pcie_reg is not None:
                        self.design.quad_pcie_reg = None
                        self.migration_log.reg_block_change(block_type, None)

                elif block_type == db.PeriDesign.BlockType.lane_10g:
                    if self.design.lane_10g_reg is not None:
                        self.design.lane_10g_reg = None
                        self.migration_log.reg_block_change(block_type, None)

                elif block_type == db.PeriDesign.BlockType.lane_1g:
                    if self.design.lane_1g_reg is not None:
                        self.design.lane_1g_reg = None
                        self.migration_log.reg_block_change(block_type, None)

                elif block_type == db.PeriDesign.BlockType.raw_serdes:
                    if self.design.raw_serdes_reg is not None:
                        self.design.raw_serdes_reg = None
                        self.migration_log.reg_block_change(block_type, None)

        self._rebuild_multi_block_type_reg(all_supported_reg, block_mig_data)

        if self.design.is_tesseract_design():
            self.design.setup_tesseract_hsio_service()
            self.design.setup_serdes_res_service()

        if self.is_old_design_tesseract or self.is_new_design_tesseract:
            if self.is_old_design_tesseract and self.is_new_design_tesseract:
                # Preserve resource within Tesseract
                # Check if it is a vertical migration
                if not preserve_pin:
                    self.migration_log.add_log("Resource/Pin Assignment", "Retained available pin/resource assignments")
                    self.update_tesseract_resource_assignment(all_supported_reg, False)
                else:
                    # Vertical migration between Tessaract devices
                    self.migration_log.add_log("Resource/Pin Assignment", "Retained available pin assignments")
                    self._migrate_design_preserve_pin(all_supported_reg)
            else:
                # Reset resource for cross family migration as well as
                # within Tesseract for now, virtual/vertical migration not supported
                self.migration_log.add_log("Resource/Pin Assignment", "All assignments have been reset")
                self.update_tesseract_resource_assignment(all_supported_reg, True)
        else:
            if not preserve_pin:
                self._migrate_design_preserve_resource(all_supported_reg)
            else:
                self._migrate_design_preserve_pin(all_supported_reg)

    def update_tesseract_resource_assignment(self, all_supported_reg, is_clear):
        """
        Reset all resource assignment

        :param all_supported_reg: The list of registry available in design
        """
        for reg in all_supported_reg:

            # Apply new device
            if not isinstance(reg, gpiod.GPIORegistry):
                reg.change_device(self.design.device_db)
            else:
                reg.change_device(self.design.device_db, self.design.is_support_ddio())

            # Clear all assignment
            if is_clear:
                reg.clear_all_resource()

    def _migrate_design_preserve_resource(self, all_supported_reg):
        """
        Migrate the instances based on the resource availability in the
        new device.

        :param all_supported_reg: The list of registry available in design        
        """
        # For all instances, clear invalid resource at registry level
        for reg in all_supported_reg:
            # gpio and osc do not inherit from PeriDesignRegistry yet
            # but python does not care

            if not isinstance(reg, gpiod.GPIORegistry):
                reg.change_device(self.design.device_db)
            else:
                reg.change_device(self.design.device_db, self.design.is_support_ddio())

                # PT-806: Preserve LVDS GPIO if the resource exists
                # Retain the instance (undefined resource) if the new device
                # does not support LVDS.
                lvds_gpio_list = self.design.gpio_reg.get_all_lvds_gpio()
                if lvds_gpio_list:
                    self._reassign_lvds_gpio_instances(lvds_gpio_list)

    def _migrate_design_preserve_pin(self, all_supported_reg):
        """
        Migrate the instances based on the pin in the new device.  We
        map the instance to resource that shares the same pin.

        :param all_supported_reg: The list of registry available in design
        """

        # Gather a map of old resource name mapped to the new resource
        # name based on retaining the package pin name
        old2new_resource_map = self._generate_resource_migration_map()

        for reg in all_supported_reg:
            # gpio and osc do not inherit from PeriDesignRegistry yet
            # but python does not care

            if not isinstance(reg, gpiod.GPIORegistry):
                reg.vertical_device_migration(
                    self.design.device_db, old2new_resource_map)
            else:
                reg.vertical_device_migration(
                    self.design.device_db, self.design.is_support_ddio(),
                    old2new_resource_map)

                # PT-806: Preserve LVDS GPIO if the resource exists
                # Call the function again because by right if the LVDS registry.
                # This is only applicable to Trion device since the information
                # is stored in design
                if not self.design.is_tesseract_design():
                    lvds_gpio_list = self.design.gpio_reg.get_all_lvds_gpio()

                    if lvds_gpio_list:
                        self._vertical_migrate_lvds_gpio_instances(
                            lvds_gpio_list, old2new_resource_map)

    def _is_instances_valid_resource(self, resource_names, instance_names):
        """
        Check that the name in the instance list are all part of the
        resource names list.

        :param resource_names: The list of device resource names
        :param instances: A list of device instance name (resource)

        :return True if all instances in the list are in the resource_names.
                Otherwise, False.
        """

        for ins in instance_names:
            if ins not in resource_names:
                return False

        return True

    def _generate_resource_migration_map(self):
        """
        Create a mapping of the resources in the old device to the new
        device by retaining the same pin

        :return a map of old resource to new resource name. Resource
                here only cover instances that are represented
                as block (i.e. device instance), including LVDS
                GPIO (GPIOB_RXP00, GPIOB_RXN00) and its LVDS
                instances (GPIOB_RX00).
        """
        old2new_resource_map = {}

        # Get the DevicePackage
        new_pkg = self.design.device_db.get_package()
        old_pkg = self.old_device_db.get_package()

        # Get the DeviceIO
        new_io_pad = self.design.device_db.get_io_pad()
        old_io_pad = self.old_device_db.get_io_pad()

        # A pad name may have multiple pin name associated (i.e. GND).
        # However, for now, we don't have one pin mapped to
        # multiple pad name. What we want to do is to retain the pin used
        # despite the resource assigned to it is no longer the same.
        # Example: pin P1 maps to pad GPIOB_RXP00 in old device.
        #        In the new device P1 now maps to GPIOB_RXP04.  Hence,
        #        the return map will contain "GPIOB_RXP00" as key with the
        #        value "GPIOB_RXP04".
        # A block may have more than one pin associated (i.e. DDR, MIPI).
        # But, we are concerned with the resource in this case. So, for this
        # scenario, we look further at the instance associated to the pad
        # which is saved in the pad's DeviceIO.
        # Example: Each MIPI resource has 10 pins (RXDP[4:0] and RXDN[4:0]).
        #         The 10 pad names will map to the same instance and that
        #         shall be the resource to migrate to if all the 10 pin names
        #         match.  ICD should already guarantee that the entire group
        #         of pins are mapped accordingly to the same instance.

        # Exception:  
        #       1) There are some instances that are not associated to
        #       pin in the pads such as PLL, H264, control, oscillator, LVDS bandgap.
        #       2) Also JTAG instance has pads in the pad/pin list but it is maped to
        #       multiple instances.
        #
        #       Because of #1, we still need to migrate based on looking at
        #       the resource map on the resources that are not from the pin
        #       just to double check that the instance still
        #       exists (although ICD should already guarantee it).

        # package pin map to PadPackagePinMap
        old_pkg_pad_map = old_pkg.get_package_pad_map()
        new_pkg_pad_map = new_pkg.get_package_pad_map()

        is_error = False
        old_resource_names = self.old_device_db.get_all_resources()
        new_resource_names = self.design.device_db.get_all_resources()

        old_resource_with_pins = []

        for pin_name in sorted(old_pkg_pad_map.keys(),
                               key=pt_util.natural_sort_key_for_list):

            if pin_name in new_pkg_pad_map:
                old_pin_pad_obj = old_pkg_pad_map[pin_name]
                old_pad_name = old_pin_pad_obj.get_pad_name()

                new_pin_pad_obj = new_pkg_pad_map[pin_name]
                new_pad_name = new_pin_pad_obj.get_pad_name()

                # Find the instance associated to the pad resource
                old_pad_obj = old_io_pad.find_pad(old_pad_name)
                new_pad_obj = new_io_pad.find_pad(new_pad_name)

                if old_pad_obj is not None and new_pad_obj is not None:
                    old_instances = old_pad_obj.get_instances()
                    new_instances = new_pad_obj.get_instances()

                    # Check if this instance exists in device resource
                    if self._is_instances_valid_resource(old_resource_names, old_instances) and \
                            self._is_instances_valid_resource(new_resource_names, new_instances):

                        # If there are multiple instance, how do we decide which maps to which?
                        if len(old_instances) == 1 and len(new_instances) == 1:
                            # Fill in the old2new_resource_map map
                            old_ins_name = old_instances[0]
                            new_ins_name = new_instances[0]

                            if old_ins_name not in old2new_resource_map:
                                old2new_resource_map[old_ins_name] = new_ins_name
                                old_resource_with_pins.append(old_ins_name)

                            elif new_ins_name != old2new_resource_map[old_ins_name]:
                                # If it's multiple (i.e. it could be LVDS GPIO or Block
                                # with multiple pin), we check that they mapped to the same
                                # new
                                is_error = True
                                self.logger.debug(
                                    "Old resource {} mapped to multiple different resource: {} vs {}".format(
                                        old_ins_name, old2new_resource_map[old_ins_name],
                                        new_ins_name))

                            # If this is LVDS, then we also add the LVDS GPIO
                            # resource by using the pad name
                            old_ins_obj = self.old_device_db.find_instance(old_ins_name)
                            if old_ins_obj is not None:
                                ins_refname = old_ins_obj.get_ref_name()

                                if ins_refname.startswith("lvds") or ins_refname.startswith("hsio"):
                                    # Do not simply use the pad name as it can contain
                                    # other names that indicates the pad function but
                                    # not used as resource name. So an LVDS will have 3
                                    # resources associated: LVDS (GPIOL_RXYY),
                                    # LVDS GPIO P (GPIOL_RXPYY), LVDS GPIO N (GPIOL_RXNYY)
                                    # For HSIO: HSIO GPIO P (GPIOL_P_XX), HSIO GPIO N (GPIOL_N_XX),
                                    # LVDS, MIPI DPHY (both - GPIO_PN_XX)
                                    if ins_refname.startswith("lvds"):
                                        old_lvds_gpio_name = old_pad_name
                                        if old_pad_name.count('_') > 1:
                                            # Get the name before the second _
                                            tmp_names = old_pad_name.split('_')
                                            old_lvds_gpio_name = tmp_names[0] + '_' + tmp_names[1]

                                        new_lvds_gpio_name = new_pad_name
                                        if new_pad_name.count('_') > 1:
                                            tmp_names = new_pad_name.split('_')
                                            new_lvds_gpio_name = tmp_names[0] + '_' + tmp_names[1]

                                    else:
                                        old_lvds_gpio_name = old_pad_name
                                        if old_pad_name.count('_') > 2:
                                            # Get the name before the second _
                                            tmp_names = old_pad_name.split('_')
                                            old_lvds_gpio_name = tmp_names[0] + '_' + tmp_names[1] + '_' + tmp_names[2]

                                        new_lvds_gpio_name = new_pad_name
                                        if new_pad_name.count('_') > 2:
                                            tmp_names = new_pad_name.split('_')
                                            new_lvds_gpio_name = tmp_names[0] + '_' + tmp_names[1] + '_' + tmp_names[2]

                                    if old_lvds_gpio_name not in old2new_resource_map:
                                        old2new_resource_map[old_lvds_gpio_name] = new_lvds_gpio_name
                                        old_resource_with_pins.append(old_lvds_gpio_name)

                                    elif new_lvds_gpio_name != old2new_resource_map[old_lvds_gpio_name]:
                                        is_error = True
                                        self.logger.debug(
                                            "Old resource {} mapped to multiple different resource: {} vs {}".format(
                                                old_lvds_gpio_name, old2new_resource_map[old_lvds_gpio_name],
                                                new_lvds_gpio_name))

                        else:
                            # This is expected for JTAG but not other blocks yet,
                            # which in this case they should map to the same name
                            if len(old_instances) == len(new_instances):
                                old_instances.sort()
                                new_instances.sort()

                                if old_instances == new_instances:
                                    for ins_name in old_instances:
                                        if ins_name not in old2new_resource_map:
                                            old2new_resource_map[ins_name] = ins_name

                                            old_resource_with_pins.append(ins_name)

                                        elif ins_name != old2new_resource_map[ins_name]:
                                            is_error = True
                                            self.logger.debug(
                                                "Old resource {} mapped to multiple different resource: {} vs {}".format(
                                                    old_ins_name, old2new_resource_map[old_ins_name],
                                                    new_ins_name))

                                else:
                                    is_error = True
                                    self.logger.debug(
                                        "Pin to migrate with multiple resources {} has mismatch list in {} vs {}".format(
                                            pin_name, self.old_device, self.design.device_def))
                            else:
                                is_error = True
                                self.logger.debug(
                                    "Pin to migrate {} has mismatch resources in {} vs {}".format(
                                        pin_name, self.old_device, self.design.device_def))

                else:

                    if old_pad_obj is None:
                        # Exception is if the pad name is an NC pin
                        if old_pad_name != "NC":
                            is_error = True
                            # This should not happen with vertical migration
                            self.logger.debug("Pad object name {} not found in {}".format(
                                old_pad_name, self.old_device))
                    else:
                        # Exception is if the pad name is an NC pin
                        if new_pad_name != "NC":
                            is_error = True
                            self.logger.debug("Pad object name {} not found in {}".format(
                                new_pad_name, self.design.device_def))

            else:
                # This should not happen with vertical migration
                self.logger.debug("Pin name {} in {} not found in {}".format(
                    pin_name, self.old_device, self.design.device_def))
                is_error = True

        if is_error:
            msg = "Error in finding compatible pins when migrating from {} to {}".format(
                self.old_device, self.design.device_def)
            raise db_excp.DeviceMigrationException(msg, app_excp.MsgLevel.error)

        # There are some resources that are not part of the pad package which
        # will simply retained as is (i.e. PLL, OSC, H264, LVDS_BG - not in design)
        for old_res in old_resource_names:
            if old_res not in old_resource_with_pins and \
                    old_res in new_resource_names:
                # Ignore it if there's a pad for it and map this to itself
                if old_res not in old2new_resource_map:
                    old2new_resource_map[old_res] = old_res

        # Comment out debug code: Dump the map for testing
        # for old in sorted(old2new_resource_map.keys()):
        #    self.logger.debug("Vertical migration from {} to {}".format(
        #        old, old2new_resource_map[old]))

        return old2new_resource_map

    def _vertical_migrate_lvds_gpio_instances(self, lvds_gpio_list,
                                              old2new_resource_map):
        """
        Reassign LVDS GPIO resources accordingly.  This is done outside the
        GPIO Registry itself because it requires the interaction of the LVDS
        registry as well.  This can be seen in how LVDS GPIO is handled in the
        design DB.
        
        :param lvds_gpio_list: The list of design instances that are of
                        type LVDS GPIO
        :param old2new_resource_map: A map of the old resource name to the
                        new name from the new device
        """
        if self.design.is_block_supported(db.PeriDesign.BlockType.lvds) or \
                self.design.is_block_supported(db.PeriDesign.BlockType.adv_lvds):
            device_db = self.design.device_db

            for inst in lvds_gpio_list:
                old_ins = inst.gpio_def

                if old_ins in old2new_resource_map:
                    new_ins = old2new_resource_map[old_ins]

                    self.logger.debug("Vertical migration of LVDS GPIO {} from {} to {}".format(
                        inst.name, old_ins, new_ins))

                    # De register the resource                        
                    self.design.reset_lvds_gpio_device(inst)

                    if not device_db.is_instance_in_resource_map(new_ins):
                        inst.mark_lvds_gpio(False)
                        # This should never happen for virtual migration
                        self.logger.debug("LVDS GPIO {} is not an LVDS GPIO {}".format(
                            inst.name, new_ins))
                    else:
                        # This will register with the related registry and
                        # no harm if it is called again.                        
                        self.design.assign_lvds_gpio_device(inst, new_ins)
                        inst.mark_lvds_gpio()
                        self.logger.debug("LVDS GPIO {} reassigned to {}".format(inst.name, new_ins))
                else:
                    # This should not happen
                    msg = "Invalid GPIO resource {} to migrate".format(old_ins)
                    raise db_excp.DeviceMigrationException(msg, app_excp.MsgLevel.error)
        else:
            # If the new device does not spport LVDS, then we
            # retain the instances but unassigned the resources
            for inst in lvds_gpio_list:
                prev_res = inst.gpio_def
                self.design.reset_lvds_gpio_device(inst)
                inst.mark_lvds_gpio(False)
                self.logger.debug("LVDS Not supported: Reset LVDS GPIO {} from {} to {}".format(
                    inst.name, prev_res, inst.gpio_def))

    def _reassign_lvds_gpio_instances(self, lvds_gpio_list):
        """
        Reassign LVDS GPIO resources accordingly.  This is done outside the
        GPIO Registry itself because it requires the interaction of the LVDS
        registry as well.  This can be seen in how LVDS GPIO is handled in the
        design DB.
        
        :param lvds_gpio_list: The list of design instances that are of
                        type LVDS GPIO
        """
        if self.design.is_block_supported(db.PeriDesign.BlockType.lvds):

            device_db = self.design.device_db

            for inst in lvds_gpio_list:
                dev_ins = inst.gpio_def

                # De register the resource
                self.design.reset_lvds_gpio_device(inst)

                if not device_db.is_instance_in_resource_map(dev_ins):
                    inst.mark_lvds_gpio(False)
                    # self.logger.debug("LVDS GPIO {} is not an LVDS GPIO {}".format(
                    #    inst.name, dev_ins))
                else:
                    # This will register with the related registry and
                    # no harm if it is called again.
                    self.design.assign_lvds_gpio_device(inst, dev_ins)
                    # self.logger.debug("LVDS GPIO {} retained {}".format(inst.name,dev_ins))

        else:
            # If the new device does not spport LVDS, then we
            # retain the instances but unassigned the resources
            for inst in lvds_gpio_list:
                self.design.reset_lvds_gpio_device(inst)
                inst.mark_lvds_gpio(False)
                # self.logger.debug("LVDS Not supported: LVDS GPIO {} is not an LVDS GPIO {}".format(
                #        inst.name, inst.gpio_def))

    def rebuild_device(self, design):
        """
        Build device database for given design. To be used when updating design due to device change.

        :param design: Design instance
        """

        # TODO : Static for now since writer test is using it

        if design is None:
            raise app_excp.PTArgException("Design is None")

        # Get the bitstream_generation setting
        prog_info = None
        timing_model_name = ""
        logger = Logger

        try:

            setting = efxproj_set.EfxProjectSetting.build(
                design.name, design.location,
                [efxproj_set.EfxProjectSetting.LoadFilterType.prog])

            if setting is not None:
                prog_info = setting.get_prog_info()

            # Get the device timing model setting
            setting = efxproj_set.EfxProjectSetting.build(
                design.name, design.location,
                [efxproj_set.EfxProjectSetting.LoadFilterType.device])

            if setting is not None:
                device_info = setting.get_device_info()
                if device_info is not None:
                    timing_model_name = device_info.get("timing_model", "")
                else:
                    logger.warning(
                        'Unable to retrieve timing_model name from project file')

        except app_excp.PTFileException as exc:
            logger.warning("File error : {}".format(exc.msg))

        # If there's any issue with parsing the project file,
        # we don't raise exception. But continue with the flow,
        # using defaults.
        design.device_db = None

        dev_service = dev_srv.DeviceService()
        dev_service.is_check = self.is_check_device
        design.device_db = dev_service.load_device(design.device_def, prog_info, timing_model_name)

        if design.device_db is None:
            msg = "Fail to load device database for {}".format(design.device_def)
            raise db_excp.DeviceException(msg, app_excp.MsgLevel.error)
        else:
            logger.debug("Load device database for {}".format(design.device_def))

    def _rebuild_device_setting(self, design, plugin_map, new_device_name):
        """
        Build device setting for given design. To be used when updating design due to device change.

        :param design: Design instance
        """

        if design is None:
            raise app_excp.PTArgException("Design is None")

        if design.device_db is None:
            raise app_excp.PTArgException("Device db is None")

        # Debug
        # if design.device_setting is not None:
        #    iob = design.device_setting.iobank_reg
        #    if iob is not None:
        #        self.logger.debug("Reading old design iobank setting")

        #        all_iobank = iob.get_all_iobank()
        #        for io in all_iobank:
        #            self.logger.debug("Bank: {} iostd: {}".format(
        #                io.name, io.iostd))

        # PT-369: Migrate the control setting instead of overwrite with default
        import design.service as dsg_svc

        old_clk_mux = design.device_setting.clkmux_reg
        devset_svc = dsg_svc.DesignSettingService()
        devset_svc.migrate_setting(design)

        if design.device_setting is None:
            msg = "Fail to migrate device setting for {}".format(
                design.device_def)
            raise db_excp.DeviceSettingException(msg, app_excp.MsgLevel.error)

        if design.device_setting.ctrl_reg is not None:
            design.device_setting.ctrl_reg.check_resource()

        if design.device_setting.seu_reg is not None:
            design.device_setting.seu_reg.check_resource()

        # Just retain for now.
        # Refactor to its own migrator class when more seamless migration is needed

        if self.is_old_design_tesseract and self.is_new_design_tesseract:
            # We need to clear it if moving between Ti60 and Ti180 although
            # both are tesseract since their clkmux architecture is different
            is_retain_clkmux = False

            # Get the gpio plugin on the old device
            if plugin_map is not None:
                old_clkmux_block_type = plugin_map.get_group_by_type_name(self.old_device, "clkmux")
                new_clkmux_block_type = plugin_map.get_group_by_type_name(new_device_name, "clkmux")
                if old_clkmux_block_type == new_clkmux_block_type:
                    # We only retain if the old and new device are tesseract
                    # and within the same group (Ti60<->Ti60, Ti180<->Ti180).
                    # Otherwise, just clear it.
                    is_retain_clkmux = True

            if is_retain_clkmux:
                design.device_setting.clkmux_reg = old_clk_mux

if __name__ == "__main__":
    pass
