"""
Copyright (C) 2017-2019 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Aug 30, 2019

@author: yasmin
"""

import os
import sys
from collections import OrderedDict
from configparser import ConfigParser
from typing import List

import util.gen_util
from util.singleton_logger import Logger

sys.path.append(os.path.join(os.path.dirname(__file__), '..'))


@util.gen_util.freeze_it
class DesignVersion:
    """
    Design database version

    .. note::
       See peri_design/readme_db_version_and_patch.txt for more info
    """

    INITIAL_DESIGN_VERSION = 20192999  #: Initial version number
    RELEASE_DESIGN_VERSION_20193 = 20193999  #: 2019.3 release db version no
    RELEASE_DESIGN_VERSION_20201 = 20201999  #: 2020.1 release db version no
    RELEASE_DESIGN_VERSION_20202 = 20202999  #: 2020.2 release db version no, latest release
    RELEASE_DESIGN_VERSION_20211500 = 20211500  #: 2021.1EA release db version no, latest release
    RELEASE_DESIGN_VERSION_20211 = 20211999 # 2021.1 production release db version no
    RELEASE_DESIGN_VERSION_20212 = 20212999 # 2021.2 production release db version no
    RELEASE_DESIGN_VERSION_20221 = 20221999 # 2022.1 production release db version no
    RELEASE_DESIGN_VERSION_20222 = 20222999  #: 2022.2 production release version
    RELEASE_DESIGN_VERSION_20231 = 20231999  #: 2023.1 production release version
    RELEASE_DESIGN_VERSION_20232 = 20232999 #: 2023.2 production release db version
    RELEASE_DESIGN_VERSION_20241 = 20241999  #: 2024.1 production release version

    # Released design version, update this in master and release when design
    # db changed is finalized
    RELEASE_DESIGN_VERSION = 20242999  #: 2024.2 production release version

    #################################################################################

    # Patch list added prior to 2019.3 customer release
    PATCH_VERSION_20193001 = 20193001  #: First patch version no for 2019.3 release
    PATCH_VERSION_20193002 = 20193002  #: Second patch version no for 2019.3 release
    PATCH_VERSION_20193003 = 20193003  #: Third patch version no for 2019.3 release
    PATCH_VERSION_20193004 = 20193004  #: Fourth patch version no for 2019.3 release

    # Patch list added prior to 2020.1 customer release
    PATCH_VERSION_20201001 = 20201001  #: First patch version no for 2020.1 release
    #: Second patch version for 2020.1 (before official release)
    PATCH_VERSION_20201002 = 20201002

    # Patch list added prior to 2020.2 customer release
    PATCH_VERSION_20202001 = 20202001  #: First patch version no for 2020.2 release

    # Patch list added prior to 2021.1 EA customer release, aka 2021.1.Ti
    PATCH_VERSION_20211001 = 20211001  #: First patch version no for 2021.1 release, deprecated see PT-1184
    PATCH_VERSION_20211002 = 20211002  #: Second patch version no for 2021.1 release, deprecated see PT-1184
    PATCH_VERSION_20211003 = 20211003  #: Third patch version no for 2021.1 release, deprecated see PT-1184
    PATCH_VERSION_20211004 = 20211004  #: Fourth patch version no for 2021.1 release
    PATCH_VERSION_20211005 = 20211005  #: Fifth patch version no for 2021.1 release
    PATCH_VERSION_20211006 = 20211006  #: Sixth patch version no for 2021.1 release

    # Patch list added prior to 2021.1 customer release
    PATCH_VERSION_20211501 = 20211501  #: First patch version no for 2021.1 release
    PATCH_VERSION_20211502 = 20211502  #: Second patch version no for 2021.1 release
    PATCH_VERSION_20211503 = 20211503  #: Third patch version no for 2021.1 release

    # Patch list added prior to 2021.2 customer release
    PATCH_VERSION_20212001 = 20212001  #: First patch version no for 2021.2 release

    # Patch list added prior to 2022.1 customer release
    PATCH_VERSION_20221001 = 20221001  #: First patch version no for 2022.1 release
    PATCH_VERSION_20221002 = 20221002
    PATCH_VERSION_20221003 = 20221003
    PATCH_VERSION_20221004 = 20221004  #: This is not needed for 2022.1 but added
                                       #: since the 2021.2 patch needed this for Trion
                                       #: DDR changes
    PATCH_VERSION_20221005 = 20221005
    PATCH_VERSION_20221006 = 20221006

    # Patch list added prior to 2022.2 customer release
    PATCH_VERSION_20222001 = 20222001  #: First patch version no for 2022.2 release
    PATCH_VERSION_20222002 = 20222002

    # Patch list added prior to 2023.1 customer release
    PATCH_VERSION_20231001 = 20231001
    PATCH_VERSION_20231002 = 20231002
    PATCH_VERSION_20231003 = 20231003
    PATCH_VERSION_20231004 = 20231004  # PT-1679 Support per output clock inversion in Titanium PLL

    # Patch list added prior to 2023.2 customer release
    PATCH_VERSION_20232001 = 20232001  # PT-1901 Support drive strength for HyperRAM
    PATCH_VERSION_20232002 = 20232002  # PT-1603 Disable dynamic voltage for HVIO bank with only JTAG pins bonded out
    PATCH_VERSION_20232003 = 20232003  # PT-2007 Disable Swap P&N Pin when Phy Lane is unused

    # Patch list added prior to 2024.1 customer release
    PATCH_VERSION_20241001 = 20241001 # PT-2037 Add Mode Select Pin name for I/O bank
    PATCH_VERSION_20241002 = 20241002 # PT-2209 PCIE ss_sim_enable default setting changed
    PATCH_VERSION_20241003 = 20241003 # PT-2211 PCIE only supports External Clock REFCLK0
    PATCH_VERSION_20241004 = 20241004 # PT-2220 Move common parameters and pins to common quad lane instances
    PATCH_VERSION_20241005 = 20241005 # PT-2235 Remove duplicate parameter
    PATCH_VERSION_20241006 = 20241006 # PT-2074 Update CFG_CLK for EFX_FPLL_V1
    PATCH_VERSION_20241007 = 20241007 # PT-2321 Rename pin type with LN_ prefix for 10G
    PATCH_VERSION_20241008 = 20241008 # PT-2325 Allow 10G instance without resource to have own common quad design
    PATCH_VERSION_20241009 = 20241009 # PT-2325 Remove ss_raw_data_rate in PMA Direct

    # Patch list added prior to 2024.2 customer release
    PATCH_VERSION_20242001 = 20242001 # PT-2151 Remove parameter for PCIe
    PATCH_VERSION_20242002 = 20242002 # PT-2453 Move some params between common quad and raw serdes
    PATCH_VERSION_20242003 = 20242003 # PT-2358 Update ss_raw_mode_lane_NID value for raw serdes
    PATCH_VERSION_20242004 = 20242004 # PT-2469 Reassign new raw serdes pll config parameter based on 3 base params
    PATCH_VERSION_20242005 = 20242005 # PT-2471/PT-2393: pma direct Clk output is autoconnected
    PATCH_VERSION_20242006 = 20242006 # PT-2433: Rename CONN_TYPE for RX_FWD_CLK 1G
    PATCH_VERSION_20242007 = 20242007 # PT-2522: Update PCIe params for 2024.2 release
    PATCH_VERSION_20242008 = 20242008 # Rebumping PT-2151 (from 20242001 so user from 2024.1 will also get the change in 2024.2))
    PATCH_VERSION_20242009 = 20242009 # PT-2543: Change APB interface to compulsory for PCIe
    PATCH_VERSION_20242010 = 20242010 # PT-2558: Add new param CLK_RESOURCE_EN for selecting PMA Direct clock resource

    # Patch list prior to 2025.1 customer release
    PATCH_VERSION_20251001 = 20251001 # PT-2559: Rename pin type USER_PHY_RESET_N to PHY_CMN_RESET_N (Common Quad Lane)
    PATCH_VERSION_20251002 = 20251002 # PT-2595: Rename param fo raw_serdes
    PATCH_VERSION_20251003 = 20251003 # PT-2598: Rename param option for ss_1gbe_refclk_freq by removing unit
    PATCH_VERSION_20251004 = 20251004 # PT-2639: 1G param rename from ICD
    PATCH_VERSION_20251005 = 20251005 # PT-2157 Status enable correction

    def __init__(self):

        self._design_ver_no = 0  #: User design db version number
        self._applied_patch_ver_no = 0  #: Maximum patch number, applied
        self.patch_list = [
            DesignVersion.PATCH_VERSION_20193001,
            DesignVersion.PATCH_VERSION_20193002,
            DesignVersion.PATCH_VERSION_20193003,
            DesignVersion.PATCH_VERSION_20193004,
            DesignVersion.PATCH_VERSION_20201001,
            DesignVersion.PATCH_VERSION_20201002,
            DesignVersion.PATCH_VERSION_20202001,
            DesignVersion.PATCH_VERSION_20211001,
            DesignVersion.PATCH_VERSION_20211002,
            DesignVersion.PATCH_VERSION_20211003,
            DesignVersion.PATCH_VERSION_20211004,
            DesignVersion.PATCH_VERSION_20211005,
            DesignVersion.PATCH_VERSION_20211006,
            DesignVersion.PATCH_VERSION_20211501,
            DesignVersion.PATCH_VERSION_20211502,
            DesignVersion.PATCH_VERSION_20211503,
            DesignVersion.PATCH_VERSION_20212001,
            DesignVersion.PATCH_VERSION_20221001,
            DesignVersion.PATCH_VERSION_20221002,
            DesignVersion.PATCH_VERSION_20221003,
            DesignVersion.PATCH_VERSION_20221004,
            DesignVersion.PATCH_VERSION_20221005,
            DesignVersion.PATCH_VERSION_20221006,
            DesignVersion.PATCH_VERSION_20222001,
            DesignVersion.PATCH_VERSION_20222002,
            DesignVersion.PATCH_VERSION_20231001,
            DesignVersion.PATCH_VERSION_20231002,
            DesignVersion.PATCH_VERSION_20231003,
            DesignVersion.PATCH_VERSION_20231004,
            DesignVersion.PATCH_VERSION_20232001,
            DesignVersion.PATCH_VERSION_20232002,
            DesignVersion.PATCH_VERSION_20232003,
            DesignVersion.PATCH_VERSION_20241001,
            DesignVersion.PATCH_VERSION_20241002,
            DesignVersion.PATCH_VERSION_20241003,
            DesignVersion.PATCH_VERSION_20241004,
            DesignVersion.PATCH_VERSION_20241005,
            DesignVersion.PATCH_VERSION_20241006,
            DesignVersion.PATCH_VERSION_20241007,
            DesignVersion.PATCH_VERSION_20241008,
            DesignVersion.PATCH_VERSION_20241009,
            DesignVersion.PATCH_VERSION_20242001,
            DesignVersion.PATCH_VERSION_20242002,
            DesignVersion.PATCH_VERSION_20242003,
            DesignVersion.PATCH_VERSION_20242004,
            DesignVersion.PATCH_VERSION_20242005,
            DesignVersion.PATCH_VERSION_20242006,
            DesignVersion.PATCH_VERSION_20242007,
            DesignVersion.PATCH_VERSION_20242008,
            DesignVersion.PATCH_VERSION_20242009,
            DesignVersion.PATCH_VERSION_20242010,
            DesignVersion.PATCH_VERSION_20251001,
            DesignVersion.PATCH_VERSION_20251002,
            DesignVersion.PATCH_VERSION_20251003,
            DesignVersion.PATCH_VERSION_20251004,
            DesignVersion.PATCH_VERSION_20251005,
        ]  #: List of all patch version number

    def __str__(self):
        info = "Design Version {}".format(self._design_ver_no)
        return info

    def set_design_version(self, design_ver_no):
        """
        Set design version number

        :param design_ver_no: Integer number
        """
        self._design_ver_no = design_ver_no

    def get_design_version(self):
        """
        Get design version number

        :return: A version number
        """
        return self._design_ver_no

    def get_applied_patch_version(self):
        return self._applied_patch_ver_no

    def set_default_design_version(self):
        """
        Set default version number in case where design file does not have version number.

        .. note::
           Design version implementation started from 2019.3, design without version just set it to 2019.2.
           This way all the patch will be applied.
        """
        self._design_ver_no = self.INITIAL_DESIGN_VERSION

    @staticmethod
    def get_release_design_version_name():
        """
        Get release design version in string format

        :return: Release version number
        """
        return DesignVersion.convert_version_no2name(DesignVersion.RELEASE_DESIGN_VERSION)

    @staticmethod
    def get_release_design_version_no():
        """
        Get release design version in integer format

        :return: Release version number
        """
        return DesignVersion.RELEASE_DESIGN_VERSION

    @staticmethod
    def convert_version_name2no(version_name):
        """
        Convert version name to version number.

        :param version_name: Version name
        :return: Version number
        """
        return int(version_name)

    @staticmethod
    def convert_version_no2name(version_no):
        """
        Convert version number to version number

        :param version_no: Version number
        :return: Version name
        """
        return "{}".format(version_no)

    def get_required_patch_version(self):
        """
        Get required patch number for current user design version.

        :return: A list of patch number
        """
        req_patch = []
        for patch_no in self.patch_list:
            if patch_no > self.get_design_version():
                req_patch.append(patch_no)

        return req_patch

    def set_applied_patch_no(self, patch_no):
        if patch_no > self._applied_patch_ver_no:
            self._applied_patch_ver_no = patch_no


@util.gen_util.freeze_it
class DesignPatch:
    """
    A patcher to apply backward compatibility code to design db.

    This class will be inherited by each block registry when backward compatibility code is needed.
    """

    def __init__(self, block_reg, device_db=None):
        self.logger = Logger

        #: A map of patch version to its (patch_func, patch_msg) tuple list
        self.verpatch_map = OrderedDict()
        self.verpatch_map[DesignVersion.PATCH_VERSION_20193001] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20193002] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20193003] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20193004] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20201001] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20201002] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20202001] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20211001] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20211002] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20211003] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20211004] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20211005] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20211006] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20211501] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20211502] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20211503] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20212001] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20221001] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20221002] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20221003] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20221004] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20221005] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20221006] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20222001] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20222002] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20231001] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20231002] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20231003] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20231004] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20232001] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20232002] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20232003] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20241001] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20241002] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20241003] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20241004] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20241005] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20241006] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20241007] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20241008] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20241009] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20242001] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20242002] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20242003] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20242004] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20242005] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20242006] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20242007] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20242008] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20242009] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20242010] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20251001] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20251002] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20251003] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20251004] = []
        self.verpatch_map[DesignVersion.PATCH_VERSION_20251005] = []

        self.block_reg = block_reg
        self.device_db = device_db
        self.applied_patch: List[int] = []  #: A list of applied patch version/s

    def is_patch(self):
        """
        Check if patch is applied.

        :return: True, if applied, else False

        .. note::
           Each patch function must generate a message
        """
        return len(self.applied_patch) > 0

    def get_patch_message(self):
        """
        Get all applied patch message

        :return: A list of messages
        """
        msg = []
        for patch_ver in self.applied_patch:
            patch_info_list = self.verpatch_map.get(patch_ver, None)
            if patch_info_list is not None:
                for patch_info in patch_info_list:
                    _, patch_msg = patch_info
                    msg.append(patch_msg)

        return msg

    def find_patch(self, version):
        """
        For a given version number, find applicable patch function

        :param version: A version number, integer
        :return: A tuple of a list of patch info tuple list and version, (info list, version list)

        .. note::
           This function must preserve the order of patches based on its number. There can be dependency
           between patches where the older patch need to be applied before the newer one can be applied.
        """

        all_patch_info = []
        all_patch_version = []
        sorted_verpatch = list(self.verpatch_map.keys())
        sorted_verpatch.sort()
        for patch_no in sorted_verpatch:
            if version is not None and patch_no > version:
                patch_info_list = self.verpatch_map.get(patch_no, None)
                if patch_info_list is not None and len(patch_info_list) > 0:
                    all_patch_info.extend(patch_info_list)
                    all_patch_version.append(patch_no)

        return all_patch_info, all_patch_version

    def is_force_patch(self):
        """
        Check if all patches need to be applied regardless of current design version.
        Whether the patch will actually be applied depends on each patch further test.

        :return: True, force all patch, else no
        """

        config_parser = ConfigParser()
        try:
            if "EFXPT_HOME" in os.environ:
                efxpt_home = os.getenv("EFXPT_HOME")
                if efxpt_home is not None and efxpt_home != "":
                    config_parser.read("{}/config.ini".format(efxpt_home))
                    check_version = config_parser.get(
                        "db_patch", "check_version")
                    if check_version == "off":
                        return True
        except:
            # Just trap and fall through
            pass

        return False

    def patch(self, user_version):
        """
        For a given user design db version, check what patch version is applicable
        and invoke each function in its patch function list.

        :param user_version: User design db version number
        """

        if self.block_reg is None:
            return None

        if user_version == 0:
            version = DesignVersion.INITIAL_DESIGN_VERSION
        else:
            version = user_version

        patch_info_list, patch_ver_list = self.find_patch(version)
        if len(patch_info_list) == 0:
            return None

        self.applied_patch = patch_ver_list
        func_name = util.gen_util.get_function_name()
        self.logger.debug("{}: Applicable patch/s to run:\n{}".
                          format(func_name, "\n".join(str(x) for x, y in patch_info_list)))
        self.logger.debug("{}: Design db version {}, applicable patch version/s:\n{}"
                          .format(func_name, version, ",".join(str(x) for x in patch_ver_list)))

        inst_list = []
        if hasattr(self.block_reg, "get_all_inst"):  # Generic registry
            inst_list = self.block_reg.get_all_inst()
        elif hasattr(self.block_reg, "get_all_gpio"):
            inst_list = self.block_reg.get_all_gpio()
        elif hasattr(self.block_reg, "get_all_osc"):
            inst_list = self.block_reg.get_all_osc()

        for inst in inst_list:
            for patch_func, _ in patch_info_list:
                patch_func.__call__(self, inst)

        # Return the biggest patch number
        return max(self.applied_patch)

    def add_patch(self, ver_no, func, msg=""):
        """
        Register patch function to a specific patch version

        :param ver_no: Target patch version number
        :param func: Target patch function name
        :param msg: Target patch message
        """
        patch_list = self.verpatch_map.get(ver_no, None)
        if patch_list is not None:
            patch_list.append((func, msg))


if __name__ == "__main__":
    pass
