"""
Copyright (C) 2017-2018 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Dec 21, 2017

@author: yasmin
"""

from __future__ import annotations
import abc
import re
import threading
import warnings
from enum import Enum, IntFlag, auto
from typing import Any, List, Optional, Protocol, Tuple, Union, Dict, TYPE_CHECKING


import util.excp as app_excp
from util.singleton_logger import Logger
import util.gen_util as gen_util

from design.db_monitor import RegResourceSubject, RegInstanceSubject
from common_device.property import PropValidator, PropertyMetaData

if TYPE_CHECKING:
    from common_gui.base_dep_graph import DependencyGraph


class CreateException(app_excp.PTException):
    """
    Exception related to instance/object creation
    """
    pass


class InstanceDoesNotExistException(app_excp.PTException):
    """
    Exception related to instance/object existance
    """
    pass

class AbstractParamObserver:

    @abc.abstractmethod
    def on_param_changed(self, param_group: GenericParamGroup, param_name: str):
        pass


class ParamGroupMonitor(AbstractParamObserver):
    """
    This class act as a bridge between ParamGroup and Depenedency Graph
    """

    def __init__(self, graph: DependencyGraph) -> None:
        super().__init__()
        self._graph = graph

    def on_param_changed(self, param_group: GenericParamGroup, param_name: str):
        self._graph.propgrate_param(param_group=param_group, param_name=param_name)

class GenericItemFlags(IntFlag):
    NONE = 0
    READONLY = 1
    AUTO_COMPUTED = 2


@gen_util.freeze_it
class GenericParam:
    """
    A generic parameter that requires minimal properties.
    Can be used for any block type when appropriate.
    """

    class DataType(Enum):
        """
        Value data type
        """
        dstr = 0  #: String type
        dint = 1  #: Integer type
        dflo = 2  #: Float type
        dbool = 3  #: Boolean type
        denum = 4  #: Enum type
        dhex = 5    #: Hex  type

    dtype2str_map = \
        {
            DataType.dstr: "str",
            DataType.dint: "int",
            DataType.dflo: "float",
            DataType.dbool: "bool",
            DataType.denum: "enum",
            DataType.dhex: "hex"
        }  #: Mapping of data type in enum to string type

    str2dtype_map = \
        {
            "str": DataType.dstr,
            "int": DataType.dint,
            "float": DataType.dflo,
            "bool": DataType.dbool,
            "enum": DataType.denum,
            "hex": DataType.dhex
        }  #: Mapping of data type in string to enum type

    def __init__(self, name: str, value="", value_type=DataType.dstr):
        self._name = name  #: Parameter type name
        self._value = value  #: Parameter value, always a Python string
        self._value_type = value_type  #: Value's interpreted data type indicator, not enforced
        self._flags = GenericItemFlags.NONE

    def __str__(self, *args, **kwargs):
        info = 'type_name:{} value:{} val_type:{}'.format(self.name, self.value, self.value_type)
        return info

    def update(self, another: 'GenericParam'):
        """
        Copy data from another GenericParam and update self
        """
        self._value = another._value
        self._value_type = another._value_type
        self._flags = another._flags

    @property
    def name(self) -> str:
        return self._name

    @property
    def value_type(self) -> GenericParam.DataType:
        return self._value_type

    @property
    def flags(self) -> GenericItemFlags:
        return self._flags

    @flags.setter
    def flags(self, new_flags: GenericItemFlags):
        self._flags = new_flags

    @property
    def value(self) -> Any:
        return self._value

    @value.setter
    def value(self, new_value: Any):
        if self._flags & GenericItemFlags.READONLY:
            raise Exception(f"Parameter {self._name} is readonly")
        self._value = new_value

    @property
    def raw_value(self):
        if self.value_type == self.DataType.dint:
            if self.value == "":
                return 0
            else:
                return int(self.value)

        elif self.value_type == self.DataType.dflo:
            if self.value == "":
                return 0.0
            else:
                return float(self.value)

        if self.value_type == self.DataType.dhex:
            if self.value == "":
                return "0"
            else:
                # Treat it as a string with hex value
                return self.value

        elif self.value_type == self.DataType.dbool:
            if self.value == "":
                return False
            else:
                return bool(self.value)

        return self.value


@gen_util.freeze_it
class GenericParamGroup:
    """
    Container for generic parameters.

    ..notes:
      Thread safe control was removed since it prevents deepcopy. Caller has to manage.
    """

    def __init__(self):
        self._param_map = {}  #: A map of param type name to its parameter info.
        self._param_observers: List[AbstractParamObserver] = []

    def __str__(self, *args, **kwargs):
        info = 'param_count:{}'.format(self.get_param_count())
        return info

    def to_string(self):
        param_info = []
        all_param = self.get_all_param()
        for param in all_param:
            param_info.append(f"{param.name}:{param.value}")

        info = " ".join(param_info)
        return info

    def update(self, another: 'GenericParamGroup'):
        """
        Update the GenericParamGroup with the GenericParam from another GenericParamGroup object

        :param another: GenericParamGroup
        :type another: GenericParamGroup
        """
        for from_p in another.get_all_param():
            to_p = self.get_param_by_name(from_p.name)
            if to_p is not None:
                to_p.update(from_p)

    def get_param_count(self):
        """
        Get parameter count

        :return: Count
        """
        return len(self._param_map)

    def get_all_param(self) -> List[GenericParam]:
        """
        Get a list of all parameter info

        :return: Parameter info list
        """
        return list(self._param_map.values())

    def get_param_by_name(self, name) -> Optional[GenericParam]:
        """
        Get parameter info from parameter type name

        :param name: Target type name
        :return: If found, param info else None
        """
        return self._param_map.get(name, None)

    def add_param(self, name, value: Any ="", value_type: GenericParam.DataType=GenericParam.DataType.dstr):
        """
        Add new parameter

        :param name: Parameter type name
        :param value: User configured value for the parameter
        :param value_type: Interpreted data type for the value, not python data type
        :return: New parameter or existing parameter
        """

        # with self._write_lock:
        param = self.get_param_by_name(name)
        if param is None:
            param = GenericParam(name, value, value_type)
            self._param_map[name] = param

        return param

    def remove_param(self, name):
        """
        Remove parameter

        :param name: Parameter type name
        """
        param = self.get_param_by_name(name)
        if param is not None:
            del self._param_map[name]

    def clear(self):
        """
        Delete all parameters
        """
        self._param_map.clear()

    def set_param_value(self, name, value) -> bool:
        """
        Set the parameters value

        Exception:
        If the parameter has READONLY flags set.

        Return:
        bool: the value is updated successfully
        """
        param = self.get_param_by_name(name)
        if param is None:
            return False

        param.value = value
        self.notify_param_changed(name)
        return True

    def get_param_value(self, name) -> Any:
        param = self.get_param_by_name(name)
        if param is not None:
            return param.value

        return ""

    def check_param_value_by_id(self, p_id, param_info):
        is_valid = False
        msg = ""

        # Get the parameter object name
        if param_info is not None:
            prop_name = param_info.get_prop_name(p_id)

            if prop_name != "":
                param = self.get_param_by_name(prop_name)

                if param is not None:
                    pvalid_setting = param_info.get_valid_setting(p_id)
                    is_valid, msg = self._check_param_valid(param, pvalid_setting)

        return is_valid, msg

    def _check_param_valid(self, param, pvalid_setting):
        is_valid = False
        msg = ""

        if param is not None:
            param_name = param.name
            param_value = param.value

            # Check the param type
            if param.value_type == GenericParam.DataType.dint or \
                    param.value_type == GenericParam.DataType.dflo:

                # Check what is the valid setting type
                if param.value_type == GenericParam.DataType.dint:
                    value_to_check = int(param_value)
                else:
                    value_to_check = float(param_value)

                if pvalid_setting is None:
                    is_valid = True
                elif isinstance(pvalid_setting, list):
                    if value_to_check in pvalid_setting:
                        is_valid = True
                    else:
                        msg = "Value, {} for {} not in the supported list".format(
                                value_to_check, param_name)

                elif isinstance(pvalid_setting, tuple):
                    # Check that it is within range if there is one
                    min_val, max_val = pvalid_setting

                    if value_to_check >= min_val and value_to_check <= max_val:
                        is_valid = True
                    else:
                        msg = "Value, {} out of range for {} (min={},max={})".format(
                            value_to_check, param_name, min_val, max_val)

                elif isinstance(pvalid_setting, PropValidator):
                    is_valid, msg = pvalid_setting.is_valid(value_to_check)

            elif param.value_type == GenericParam.DataType.denum:
                if pvalid_setting is not None and param_value in pvalid_setting:
                    is_valid = True

                else:
                    msg = "Invalid option, {} for {}".format(param_value, param_name)

            elif param.value_type == GenericParam.DataType.dbool:
                if isinstance(param_value, bool):
                    is_valid = True
                else:
                    msg = "Invalid bool value, {} for {}".format(param_value, param_name)

            elif param.value_type == GenericParam.DataType.dhex:
                try:
                    value_to_check = int(param_value, 16)

                    if pvalid_setting is None:
                        is_valid = True

                    elif isinstance(pvalid_setting, tuple):
                        # Check that it is within range if there is one
                        min_val, max_val = pvalid_setting
                        int_min_val = int(min_val, 16)
                        int_max_val = int(max_val, 16)

                        if value_to_check >= int_min_val and value_to_check <= int_max_val:
                            is_valid = True
                        else:
                            msg = "Value ({}) out of range (min={},max={})".format(
                                value_to_check, int_min_val, int_max_val)

                    else:
                        msg = "Hex parameter type for {} only supports tuple type".format(param_name)

                except Exception:
                    is_valid = False
                    msg = f"The value ({param_value}) is not hexadecimal."

            elif param.value_type == GenericParam.DataType.dstr:
                # string type: Just check that it's not empty
                if param_value != "" and param_value is not None and isinstance(param_value, str):
                    if isinstance(pvalid_setting, PropValidator):
                        is_valid, msg = pvalid_setting.is_valid(param_value)
                    else:
                        is_valid = True
                else:
                    is_valid = False
                    msg = "Invalid string value, {} for {}".format(param_value, param_name)
            else:
                # Invalid type
                msg = "Invalid type for {}".format(param_name)

        return is_valid, msg

    def check_param_value(self, name, param_info):
        is_valid = False
        msg = ""

        param = self.get_param_by_name(name)
        if param is not None and param_info is not None:
            # param_name = param.name
            # param_value = param.value

            p_id = param_info.get_prop_id(name)
            pvalid_setting = None
            if p_id is not None:
                pvalid_setting = param_info.get_valid_setting(p_id)

            is_valid, msg = self._check_param_valid(param, pvalid_setting)

        return is_valid, msg

    def register_param_observer(self, observer: AbstractParamObserver):
        if observer is not None and observer not in self._param_observers:
            self._param_observers.append(observer)

    def unregister_param_observer(self, observer: AbstractParamObserver):
        if observer in self._param_observers:
            self._param_observers.remove(observer)

    def notify_param_changed(self, name: str):
        for observer in self._param_observers:
            observer.on_param_changed(self, name)

    def clear_observers(self):
        self._param_observers.clear()

    @property
    def observers(self):
        return self._param_observers

    @observers.setter
    def observers(self, observers):
        self._param_observers = observers

@gen_util.freeze_it
class GenericParamService:
    """
    Provide quick parameter operation based on parameter name or property id
    """

    def __init__(self, param_group: GenericParamGroup, param_info: PropertyMetaData):
        self.param_group = param_group
        self.param_info = param_info

    def get_param_info(self):
        return self.param_info

    def get_param_disp_name(self, param_id) -> str:
        return self.param_info.get_display_name(param_id)

    def get_param_value(self, param_id):
        prop_name = self.param_info.get_prop_name(param_id)
        return self.param_group.get_param_value(prop_name)

    def set_param_value(self, param_id, value):
        prop_name = self.param_info.get_prop_name(param_id)
        return self.param_group.set_param_value(prop_name, value)

    def get_param_prop_info(self, param_id):
        return self.param_info.get_prop_info(param_id)

    def check_param_valid(self, param_id) -> Tuple[bool, str]:
        return self.param_group.check_param_value_by_id(param_id, self.param_info)


@gen_util.freeze_it
class GenericPin:
    """
    A generic pin that requires minimal properties.
    Can be used for any block type when appropriate.
    """

    def __init__(self, type_name, name="", is_bus=False):
        self._type_name = type_name  #: Pin type name, usually match port name in device
        self._name = name  #: Pin name
        self._is_bus = is_bus  #: True, a bus, else a pin
        self._flags = GenericItemFlags.NONE

    @property
    def name(self) -> str:
        return self._name

    @name.setter
    def name(self, new_name: str):
        if self._flags & GenericItemFlags.READONLY:
            raise Exception(f"Pin {self._type_name} is readonly")
        self._name = new_name

    @property
    def type_name(self) -> str:
        return self._type_name

    @type_name.setter
    def type_name(self, new_name: str):
        self._type_name = new_name

    @property
    def is_bus(self):
        return self._is_bus

    @is_bus.setter
    def is_bus(self, value: bool):
        self._is_bus = value

    @property
    def flags(self) -> GenericItemFlags:
        return self._flags

    @flags.setter
    def flags(self, new_flags: GenericItemFlags):
        self._flags = new_flags

    def __str__(self, *args, **kwargs):
        info = 'type_name:{} name:{} is_bus:{}'.format(self.type_name, self.name, self.is_bus)
        return info

    def copy_common(self, from_obj, exclude_attr=None):
        """
        Copy all class attribute value from from_obj to this object.
        This is different from copy() and deepcopy() since it only
        copies what is common and doesn't do deepcopy of non-primitive
        attributes.

        :param from_obj: Object to copy from
        :param exclude_attr: A list of attribute names to skip ie the data member

        .. note::
        Python does not support static_cast so this can be use to emulate it.
        """

        if exclude_attr is None:
            exclude_attr_list = []
        else:
            exclude_attr_list = exclude_attr

        local_attr_list = list(vars(self).keys())
        for attr in local_attr_list:

            if attr in exclude_attr_list:
                continue

            if hasattr(from_obj, attr):
                value = getattr(from_obj, attr)
                setattr(self, attr, value)

    def generate_pin_name(self, prefix):
        """
        Generate pin name by applying prefix to the type name
        :param prefix: Prefix to prepend
        """
        self.name = prefix + "_" + self.type_name


@gen_util.freeze_it
class GenericClockPin(GenericPin):
    """
    A clock pin.

    .. note::
       Separate class to optimize memory footprint and to isolate
       future clock specific functionality. To identify if a db item is
       a simple pin or a clock pin, caller can query device db based on type
       name or check for object class type.
    """

    def __init__(self, type_name, name="", is_bus=False, is_invert=False):
        self.is_inverted = is_invert  #: True, clock pin is inverted, else False
        super().__init__(type_name, name, is_bus)

    def __str__(self, *args, **kwargs):
        info = '{} is_inverted:{}'.format(super(), self.is_inverted)
        return info

class GenericInputPin(GenericPin):
    pass

class GenericOutputPin(GenericPin):
    pass

@gen_util.freeze_it
class GenericPinGroup:
    """
    Container for generic pins.

    .. note::
       In general, it consists of one class type ie GenericPin. However if clock pin
       is supported, it will have a mixture of GenericPin and GenericClockPin. Python does not
       care about type so this is perfectly valid. Unlike C++, dynamic casting is not supported.
       If caller wanted to access clock pin property but not sure if it is there, caller can check
       if it is a GenericClockPin. See note in :class:`GenericClockPin`. Just remember that
       python does duck typing.
    """

    def __init__(self):
        self._pin_map: Dict[str, GenericPin] = {}  #: A map of pin type name to its pin info.
        self._write_lock = threading.RLock()  #: To control edit operation which is not reentrant.

    def __str__(self, *args, **kwargs):
        info = 'pin_count:{}'.format(self.get_pin_count())
        return info

    def to_string(self):
        pin_info = []
        all_pin = self.get_all_pin()
        for pin in all_pin:
            pin_info.append(f"{pin.type_name}:{pin.name}")

        info = " ".join(pin_info)
        return info

    def copy_common(self, from_obj, exclude_pin_type=None):
        """
        Copy pin data of matching type
        """
        if exclude_pin_type is None:
            exclude_pin_type_list = []
        else:
            exclude_pin_type_list = exclude_pin_type

        for pin in self.get_all_pin():
            if pin.type_name in exclude_pin_type_list:
                continue

            from_pin = from_obj.get_pin_by_type_name(pin.type_name)
            if from_pin is not None:
                pin.copy_common(from_pin)

    def get_pin_by_type_name(self, type_name):
        """
        Get pin info from pin type name

        :param type_name: Target type name
        :return: If found, pin info else None
        """
        return self._pin_map.get(type_name, None)

    def get_pin_by_name(self, pin_name):
        """
        Get pin info from pin name. Returns the first pin that match the name.

        :param pin_name: Pin name
        :return: If found, pin info else None
        """
        pin = None
        for pin in self._pin_map.values():
            if pin.name == pin_name:
                return pin

        return pin

    def get_all_pin(self):
        """
        Get a list of all pin info

        :return: Pin info list
        """
        return list(self._pin_map.values())

    def get_all_pin_type_name(self):
        return list(self._pin_map.keys())

    def get_pin_count(self):
        """
        Get pin count

        :return: Count
        """
        return len(self._pin_map)

    def add_pin_obj(self, pin: Union[GenericPin, GenericClockPin, GenericInputPin, GenericOutputPin]):
        """
        Add a GenericPin to the group without having to create a new object
        :param pin: the GenericPin to add
        :return: True if it was added succesfully. Otherwise, False
        """
        is_added = False

        with self._write_lock:
            if pin is not None:
                dup_pin = self.get_pin_by_type_name(pin.type_name)
                if dup_pin is None:
                    self._pin_map[pin.type_name] = pin
                    is_added = True

                else:
                    logger = Logger
                    logger.debug("Pin {} cannot be added since it already exists".format(pin.type_name)) # type: ignore

        return is_added

    def add_pin(self, type_name, name="", is_bus=False):
        """
        Add new pin type

        :param type_name: Pin type name
        :param name: User configured pin name
        :param is_bus: True, a bus else a pin
        :return: New pin or existing pin
        """

        with self._write_lock:
            pin = self.get_pin_by_type_name(type_name)
            if pin is None:
                pin = GenericPin(type_name, name, is_bus)
                self._pin_map[type_name] = pin
            else:
                pin.name = name
                pin.is_bus = is_bus

            return pin

    def add_clock_pin(self, type_name, name="", is_bus=False, is_invert=False):
        """
        Add new clock pin type

        :param type_name: Pin type name
        :param name: User configured pin name
        :param is_bus: True, a bus else a pin
        :param is_invert: True, clock is inverted
        :return: New pin or existing pin
        """

        with self._write_lock:
            pin = self.get_pin_by_type_name(type_name)
            if pin is None:
                pin = GenericClockPin(type_name, name, is_bus, is_invert)
                self._pin_map[type_name] = pin
            else:
                if isinstance(pin, GenericClockPin):
                    pin.name = name
                    pin.is_bus = is_bus
                    pin.is_inverted = is_invert
                else:
                    self.delete_pin_by_type(type_name)
                    pin = GenericClockPin(type_name, name, is_bus, is_invert)
                    self._pin_map[type_name] = pin

            return pin

    def add_input_pin(self, type_name: str, name: str = "", is_bus: bool = False) -> GenericInputPin:
        with self._write_lock:
            pin = self.get_pin_by_type_name(type_name)
            if pin is None:
                pin = GenericInputPin(type_name, name, is_bus)
                self._pin_map[type_name] = pin
            else:
                if isinstance(pin, GenericInputPin):
                    pin.name = name
                    pin.is_bus = is_bus
                else:
                    self.delete_pin_by_type(type_name)
                    pin = GenericInputPin(type_name, name, is_bus)
                    self._pin_map[type_name] = pin

            return pin

    def add_output_pin(self, type_name: str, name: str = "", is_bus: bool = False) -> GenericOutputPin:
        with self._write_lock:
            pin = self.get_pin_by_type_name(type_name)
            if pin is None:
                pin = GenericOutputPin(type_name, name, is_bus)
                self._pin_map[type_name] = pin
            else:
                if isinstance(pin, GenericOutputPin):
                    pin.name = name
                    pin.is_bus = is_bus
                else:
                    self.delete_pin_by_type(type_name)
                    pin = GenericOutputPin(type_name, name, is_bus)
                    self._pin_map[type_name] = pin

            return pin

    def get_pin_name_by_type(self, type_name):
        """
        Given a type name, find the pin and set a new name

        :param type_name: Target type name
        :param pin_name: New pin name
        """
        with self._write_lock:
            pin = self.get_pin_by_type_name(type_name)
            if pin is not None:
                return pin.name

            return ""

    def set_pin_name_by_type(self, type_name, pin_name):
        """
        Given a type name, find the pin and set a new name

        :param type_name: Target type name
        :param pin_name: New pin name
        """
        with self._write_lock:
            pin = self.get_pin_by_type_name(type_name)
            if pin is not None:
                pin.name = pin_name

    def generate_pin_name(self, prefix):
        """
        Generate and replace all pin name

        :param prefix: Prefix to apply
        """
        with self._write_lock:
            for pin in self._pin_map.values():
                pin.generate_pin_name(prefix)

    def generate_pin_name_by_type(self, prefix, pin_type_list):
        """
        Generate and replace all pin name for the specified pin type

        :param prefix: Prefix to apply
        :param pin_type_list: A list of target pin type name
        """
        with self._write_lock:
            for pin in self._pin_map.values():
                if pin.type_name in pin_type_list:
                    pin.generate_pin_name(prefix)

    def refresh_pin_name(self, old_substr, new_substr):
        """
        Replace the old substring in pin name to the new substring. If exists.

        :param old_substr: Old substring in pin name to replace
        :param new_substr: The new string to replace the old substring with
        """
        with self._write_lock:
            for pin in self._pin_map.values():
                if old_substr in pin.name:
                    name = pin.name
                    pin.name = name.replace(old_substr, new_substr)

    def refresh_pin_name_by_type(self, old_substr, new_substr, pin_type_list):
        """
        Replace the old substring in pin name to the new substring. If exists.

        :param old_substr: Old substring in pin name to replace
        :param new_substr: The new string to replace the old substring with
        :param pin_type_list: A list of target pin type name
        """
        with self._write_lock:
            for pin in self._pin_map.values():
                if pin.type_name in pin_type_list:
                    if old_substr in pin.name:
                        name = pin.name
                        pin.name = name.replace(old_substr, new_substr)

    def clear(self):
        """
        Delete all pins
        """
        with self._write_lock:
            self._pin_map.clear()

    def clear_all_pin_name(self):
        """
        Clear all pin name
        """
        with self._write_lock:
            for pin in self._pin_map.values():
                pin.name = ""

    def clear_clock_pin_name(self):
        """
        Clear clock pin name
        """
        with self._write_lock:
            for pin in self._pin_map.values():
                if isinstance(pin, GenericClockPin):
                    pin.name = ""

    def is_match_pin_name(self, text):
        """
        Check if the text fully or partially match any pin name

        :param text: Text to find in pin name
        :return: True, if match, else False
        """
        for pin in self._pin_map.values():
            if text in pin.name:
                return True

        return False

    def delete_pin_by_type(self, type_name):
        """
        Delete first pin found that match with the type name.

        :param type_name: Pin device type name

        .. note::
           By right the type name will be unique, the first pin should be the only pin.
        """

        with self._write_lock:
            if type_name in self._pin_map:
                # Delete from map
                del self._pin_map[type_name]

    def replace_pin_type_name(self, cur_type_name, new_type_name):
        """
        The existing pin type name has to be renamed to the new_type_name.
        Therefore the map with the key being the type name has to be updated
        and also the gen pin type name

        :param new_type_name:
        :return: the modified pin
        """
        mod_pin = None

        with self._write_lock:
            if cur_type_name in self._pin_map:
                mod_pin = self._pin_map[cur_type_name]

                # Delete from the map
                del self._pin_map[cur_type_name]

                # Add back the same object to the map with the new name
                assert isinstance(mod_pin, GenericPin)
                assert hasattr(mod_pin, 'type_name')
                mod_pin.type_name = new_type_name
                self._pin_map[new_type_name] = mod_pin

        return mod_pin

class PeriDesignItemNetlistConfig:

    def __init__(self, netlist_checksum: str):
        self.__checksum = netlist_checksum
        self.__netlist_parameters: Dict[str, str] = {}

    @property
    def netlist_checksum(self) -> str:
        return self.__checksum

    def add_parameter(self, name: str, value: str):
        self.__netlist_parameters[name] = value

    @property
    def parameters(self) -> List[Tuple[str, str]]:
        return list(self.__netlist_parameters.items())

class PeriDesignItem(metaclass=abc.ABCMeta):
    """
    Base class for design database item. This include blocks and also its
    member sub-class.
    """

    def __init__(self):
        """
        Constructor
        """
        super().__init__()
        self._save_chksum = None
        self._netlist_config: Optional[PeriDesignItemNetlistConfig] = None

    @property
    def netlist_checksum(self) -> Optional[str]:
        return None if self._netlist_config is None else self._netlist_config.netlist_checksum

    @property
    def netlist_config(self) -> Optional[PeriDesignItemNetlistConfig]:
        return self._netlist_config

    @netlist_config.setter
    def netlist_config(self, new: PeriDesignItemNetlistConfig):
        self._netlist_config = new

    @property
    def readonly(self) -> bool:
        if self._netlist_config is None:
            return False
        return True

    @abc.abstractmethod
    def create_chksum(self):
        """
        Create object checksum. Checksum can be used to check if object has been modified.

        Object checksum has to be based on object member hash.
        Object member is different from class member so vars() won't work.
        Each object must calculate its checksum based on its member.

        :return: Object hash
        """
        pass

    def copy_common(self, from_obj, exclude_attr=None):
        """
        Copy all class attribute value from from_obj to this object.
        This is different from copy() and deepcopy() since it only
        copies what is common and doesn't do deepcopy of non-primitive
        attributes.

        :param from_obj: Object to copy from
        :param exclude_attr: A list of attribute names to skip ie the data member

        .. note::
        Python does not support static_cast so this can be use to emulate it.
        """

        if exclude_attr is None:
            exclude_attr_list = []
        else:
            exclude_attr_list = exclude_attr

        exclude_attr_list += [
            '_save_chksum',
        ]

        local_attr_list = list(vars(self).keys())
        for attr in local_attr_list:

            if attr in exclude_attr_list:
                continue

            if hasattr(from_obj, attr):
                value = getattr(from_obj, attr)
                setattr(self, attr, value)

    def update_chksum(self):
        self._save_chksum = self.create_chksum()
        return self._save_chksum

    def is_changed(self):
        chksum = self.create_chksum()
        if chksum != self._save_chksum:
            return True

        return False

    @abc.abstractmethod
    def set_default_setting(self, device_db=None):
        pass

    def is_match_pin_name(self, text):
        """
        Check if the text fully or partially match this db item one or more pin names

        :param text: Text to find in pin name
        :return: True, if match, else False
        """
        gen_util.mark_unused(text)
        return False

    def generate_pin_name(self):
        """
        Auto generate pin name for applicable db item.
        It will override existing name.
        """
        pass

    def refresh_pin_name(self, old_substr, new_substr):
        """
        Refresh pin name for applicable db item.

        This does not full override the pin name. It will refresh certain part of the name.
        Which part depends on per block implementation.

        :param old_substr: Existing substring to be replaced
        :param new_substr: New substring
        """
        pass

    def get_device(self):
        """
        Get device def for block instance item. Device refers to resource name.

        Child class should inherit this to allow caller to get access to device def
        in a generic way.

        :return: Device def name, if override
        """
        return None

    def set_device(self, device_name):
        """
        Set device def for block instance item

        Child class should inherit this to allow caller to set device def
        in a generic way.
        """
        pass


class PeriDesignGenPinItem(PeriDesignItem):
    """
    A design item that has container of generic pin.

    A generic pin info is specified in device database and dynamically created
    as design element at runtime when the item is first created.

    .. note::
       Since this inherits from metaclass, the class variable has unique variable for
       each inherited class.
    """

    # This is a map of the port name to the Port object from device (block_definition)
    _device_port_map: Dict = {}  #: A map of device port type to its info, linked to device db

    class DevicePortService:

        def __init__(self):
            self.device_port = None

        def get_type(self):
            assert self.device_port is not None
            return self.device_port.get_type()

        def get_name(self):
            assert self.device_port is not None
            return self.device_port.get_name()

        def get_class(self):
            assert self.device_port is not None
            return self.device_port.get_class()

        def is_bus_port(self):
            assert self.device_port is not None
            return self.device_port.is_bus_port()

        def get_desc(self):
            assert self.device_port is not None
            return self.device_port.get_description()

    class MultiModeDevicePortService(DevicePortService):

        def get_type(self):
            assert self.device_port is not None
            return self.device_port.get_type_name()

        def get_class(self):
            assert self.device_port is not None
            return self.device_port.get_class_name()

    @abc.abstractmethod
    def __init__(self):
        super().__init__()

        self.gen_pin = GenericPinGroup()
        self.dp_service = PeriDesignGenPinItem.DevicePortService()

        # Child class need to call set_default setting in implementation
        # If created through builder, caller need to call it.

    def __str__(self, *args, **kwargs):
        info = 'pin_count:{}'.format(self.gen_pin.get_pin_count())
        return info

    def copy_common(self, from_obj, exclude_attr=None):
        """
        Override
        """

        if exclude_attr is None:
            exclude_attr_list = ["gen_pin"]
        else:
            exclude_attr_list = exclude_attr
            exclude_attr_list.append("gen_pin")

        super(PeriDesignGenPinItem, self).copy_common(from_obj, exclude_attr_list)

        if self.gen_pin is not None:
            if hasattr(from_obj, "gen_pin") and from_obj.gen_pin is not None:
                self.gen_pin.copy_common(from_obj.gen_pin, exclude_attr)

        else:
            # If pin is none, it means the from_obj does not have pin.
            # However this instance needs one, so create default pin.
            self.gen_pin = GenericPinGroup()

        if self.gen_pin.get_pin_count() == 0:
            self.build_generic_pin()

    def generate_pin_name_from_inst(self, inst_name):
        self.gen_pin.generate_pin_name(inst_name)

    def generate_pin_name_from_inst_by_class(self, inst_name, device_class):
        pin_type_name_list = self.get_pin_type_by_class(device_class)
        self.gen_pin.generate_pin_name_by_type(inst_name, pin_type_name_list)

    def refresh_pin_name(self, old_substr, new_substr):
        """
        Override
        """
        self.gen_pin.refresh_pin_name(old_substr, new_substr)

    def refresh_pin_name_by_class(self, old_substr, new_substr, device_class):
        pin_type_name_list = self.get_pin_type_by_class(device_class)
        self.gen_pin.refresh_pin_name_by_type(old_substr, new_substr, pin_type_name_list)

    @staticmethod
    def build_port_info(device_db=None, is_mockup=False):
        """
        Build port info for generic pin from definition in device db.
        This function need to be call only once since it is static shared
        between class.

        :param device_db:
        :param is_mockup: True, build a mockup data, else build from device db
        """
        gen_util.mark_unused(device_db)

        if is_mockup:
            raise NotImplementedError

        else:
            msg = "PeriDesignGenPinItem::build_port_info - Using in non-mockup mode. Need to override to pull data from device db."
            warnings.warn(msg, Warning)

    def build_generic_pin(self):
        """
        Dynamically generate generic pins based on device info
        """
        assert self._device_port_map is not None
        self.gen_pin.clear()

        from device.block_definition import Port as DevicePort

        for device_port in self._device_port_map.values():
            self.dp_service.device_port = device_port
            if self.dp_service.get_type() == DevicePort.TYPE_CLOCK:
                self.gen_pin.add_clock_pin(self.dp_service.get_name(), "", self.dp_service.is_bus_port())
            else:
                self.gen_pin.add_pin(self.dp_service.get_name(), "", self.dp_service.is_bus_port())

    def get_pin_property_name(self, pin_type_name):
        """
        Based on device pin type, get its pin name.

        Pin type is the port name defined in device db.

        :param pin_type_name: Device pin type name
        :return: Pin name if found, else empty string
        """
        assert self._device_port_map is not None

        device_port = self._device_port_map.get(pin_type_name, None)
        if device_port is not None:
            self.dp_service.device_port = device_port
            desc = self.dp_service.get_desc()
            if desc != None:
                return desc

        return ""

    def get_pin_class(self, pin_type_name):
        """
        Based on device pin type, get its pin class.

        Pin type is the port name defined in device db.
        Pin class is the port class name defined in device db.

        :param pin_type_name: Device pin type name
        :return: Device pin class if found, else None
        """
        assert self._device_port_map is not None

        device_port = self._device_port_map.get(pin_type_name, None)
        if device_port is not None:
            self.dp_service.device_port = device_port
            return self.dp_service.get_class()

        return None

    def get_pin_by_class(self, device_class) -> List[GenericPin]:
        """
        Get a list of pin for target class.

        Pin class is the port class name defined in device db.

        :param device_class: Pin class name
        :return: A list of pin, if any
        """
        pin_list = []
        for pin in self.gen_pin.get_all_pin():
            if self.get_pin_class(pin.type_name) == device_class:
                pin_list.append(pin)

        return pin_list

    def get_pin_type_by_class(self, device_class):
        """
        Get a list of pin type name for target class.

        Pin class is the port class name defined in device db.

        :param device_class: Pin class name
        :return: A list of pin type name, if any
        """
        pin_type_list = []
        for pin in self.gen_pin.get_all_pin():
            if self.get_pin_class(pin.type_name) == device_class:
                pin_type_list.append(pin.type_name)

        return pin_type_list

    def get_pins_by_class_pattern(self, class_pattern: re.Pattern[str]) -> List[GenericPin]:
        """
        Get a list of pin that match the target class pattern

        :param class_pattern: Regex pattern
        :type class_pattern: re.Pattern[str]
        :return: List of pin
        :rtype: List[GenericPin]
        """
        pin_list = []
        for pin in self.gen_pin.get_all_pin():
            pin_class = self.get_pin_class(pin.type_name)
            if class_pattern.match(pin_class if pin_class is not None else ""):
                pin_list.append(pin)
        return pin_list

    def get_all_pin_type_name(self)->List[str]:
        """
        Get a list of supported pin type name

        :return: A list of pin type name, if any
        """
        pin_type_list = []
        for pin in self.gen_pin.get_all_pin():
            pin_type_list.append(pin.type_name)

        return pin_type_list

    def is_pin_type_clock(self, pin_type_name):
        """
        Based device pin type, check if it is a clock pin

        :param pin_type_name: Device pin type name
        :return: True, if clock, else False
        """
        from device.block_definition import Port as DevicePort

        device_port = self._device_port_map.get(pin_type_name, None)
        if device_port is not None:
            self.dp_service.device_port = device_port
            dev_port_type = self.dp_service.get_type()
            return dev_port_type == DevicePort.TYPE_CLOCK

        return False

    def is_pin_type_reset(self, pin_type_name):
        """
        Based device pin type, check if it is a reset pin

        :param pin_type_name: Device pin type name
        :return: True, if clock, else False
        """
        from device.block_definition import Port as DevicePort

        device_port = self._device_port_map.get(pin_type_name, None)
        if device_port is not None:
            self.dp_service.device_port = device_port
            dev_port_type = self.dp_service.get_type()
            return dev_port_type == DevicePort.TYPE_RESET

        return False

    def is_match_pin_name(self, text):
        """
        Override
        """
        return self.gen_pin.is_match_pin_name(text)

    def clear_all_pin_name(self):
        """
        Reset all pin name.
        """
        self.gen_pin.clear_all_pin_name()

    def clear_all_pin_name_by_class(self, device_class):
        """
        Reset all pin name.

        """
        if device_class is None or device_class == "":
            return

        pin_list = self.get_pin_by_class(device_class)
        for pin in pin_list:
            pin.name = ""

    def is_pin_configured(self, pin_name):
        """
        Check if the specified generic pin type name is non-empty in
        design.

        :param pin_name: The generic pin type name to search
        :return True if the specified pin is non-empty. Otherwise, False.
        """

        gen_pin_obj = self.gen_pin.get_pin_by_type_name(pin_name)
        if gen_pin_obj is not None and gen_pin_obj.name != "":
            return True

        return False

# TODO : Review if needed, gen_pin_item cannot be constructed as is since it has abstract constructor
# class PeriDesignGenPinItemBuilder:
#
#     def __init__(self):
#         self.gen_pin_item = PeriDesignGenPinItem()
#         # Pin Type Name, UI Property Name, Is Bus flag, Pin class type
#         # This map is cleared if port info is extracted from device db.
#         self.port_info_mockup_data = {}
#         # This is a map of the port name to the Port object from device (block_definition)
#         self.device_port_map = None  #: A map of device port type to its info, linked to device db
#
#     def build_mockup_pin(self, mockup_port)->PeriDesignGenPinItem:
#
#         gen_pin = self.gen_pin_item.gen_pin
#         gen_pin.clear()
#         self.port_info_mockup_data = mockup_port
#         for port_info in self.port_info_mockup_data.values():
#             gen_pin.add_pin(port_info[0], "", port_info[2])
#
#         return self.gen_pin_item
#
#     def build_device_pin(self, device_port)->PeriDesignGenPinItem:
#
#         gen_pin = self.gen_pin_item.gen_pin
#         gen_pin.clear()
#         self.device_port_map = device_port
#         from device.block_definition import Port as DevicePort
#         for device_port in self.device_port_map.values():
#             if device_port.get_type() == DevicePort.TYPE_CLOCK:
#                 gen_pin.add_clock_pin(device_port.get_name(), "", device_port.is_bus_port())
#             else:
#                 gen_pin.add_pin(device_port.get_name(), "", device_port.is_bus_port())
#
#         return self.gen_pin_item

class PeriDesignDirectItem(PeriDesignItem):
    """
    A directional block instance. It is either tx or rx.
    """

    class OpsType(Enum):
        """
        Operation type. Use as tx or rx.
        """
        op_tx = auto()
        op_rx = auto()
        op_bidir = auto()
        unknown = auto()

    opstype2str_map = \
        {
            OpsType.op_tx: "tx",
            OpsType.op_rx: "rx",
            OpsType.op_bidir: "bidir"

        }  #: Mapping of direction type in enum to string type

    str2opstype_map = \
        {
            "tx": OpsType.op_tx,
            "rx": OpsType.op_rx,
            "bidir": OpsType.op_bidir
        }  #: Mapping of direction type in string to enum type

    def __init__(self):
        super().__init__()
        self.name = ""
        self.ops_type = None
        self.tx_info = None  #: Block used in tx mode
        self.rx_info = None  #: Block used in rx mode

    def __str__(self, *args, **kwargs):

        info = ""
        if self.tx_info is not None:
            info = "{}".format(self.tx_info)

        if self.rx_info is not None:
            info = "{}".format(self.rx_info)

        info = f"op_type:{self.ops_type} info:({info})"

        return info

    @abc.abstractmethod
    def build_tx(self):
        pass

    @abc.abstractmethod
    def build_rx(self):
        pass

    def set_default_setting(self, device_db=None):

        if self.ops_type == PeriDesignDirectItem.OpsType.op_tx:
            self.tx_info = self.build_tx()
            assert self.tx_info is not None
            self.tx_info.set_default_setting(device_db)
        elif self.ops_type == PeriDesignDirectItem.OpsType.op_rx:
            self.rx_info = self.build_rx()
            assert self.rx_info is not None
            self.rx_info.set_default_setting(device_db)
        elif self.ops_type == PeriDesignDirectItem.OpsType.op_bidir:
            self.tx_info = self.build_tx()
            assert self.tx_info is not None
            self.tx_info.set_default_setting(device_db)
            self.rx_info = self.build_rx()
            assert self.rx_info is not None
            self.rx_info.set_default_setting(device_db)
        else:
            # If not configured, it means right after creation. Should be changed
            # after user input is processed.
            self.tx_info = self.build_tx()
            assert self.tx_info is not None
            self.tx_info.set_default_setting(device_db)
            self.ops_type = PeriDesignDirectItem.OpsType.op_tx

    def update_type(self, direct_type, device_db=None, auto_pin=False):
        self.ops_type = direct_type
        if self.ops_type == self.OpsType.op_tx:
            self._set_tx_type(device_db, auto_pin)
        elif self.ops_type == self.OpsType.op_rx:
            self._set_rx_type(device_db, auto_pin)
        elif self.ops_type == self.OpsType.op_bidir:
            self._set_bidir_type(device_db, auto_pin)

    def _set_tx_type(self, device_db=None, auto_pin=False):
        """
        Set to TX type.
        Clear existing tx/rx data and reset to defaults.
        """
        self.rx_info = None
        self.tx_info = self.build_tx()
        self.ops_type = PeriDesignDirectItem.OpsType.op_tx
        assert self.tx_info is not None
        self.tx_info.set_default_setting(device_db)

        if auto_pin:
            self.generate_pin_name()

    def _set_rx_type(self, device_db=None, auto_pin=False):
        """
        Set to RX type.
        Clear existing tx/rx data and reset to defaults.
        """
        self.tx_info = None
        self.rx_info = self.build_rx()
        self.ops_type = PeriDesignDirectItem.OpsType.op_rx
        assert self.rx_info is not None
        self.rx_info.set_default_setting(device_db)

        # Auto pin name not supported for internal pins
        if auto_pin:
            self.generate_pin_name()

    def _set_bidir_type(self, device_db=None, auto_pin=False):
        """
        Set to bidirectional type.
        Clear existing tx/rx data and reset to defaults.
        """
        self.tx_info = self.build_tx()
        self.rx_info = self.build_rx()
        self.ops_type = PeriDesignDirectItem.OpsType.op_bidir
        assert self.tx_info is not None
        assert self.rx_info is not None
        self.rx_info.set_default_setting(device_db)
        self.tx_info.set_default_setting(device_db)

        if auto_pin:
            self.generate_pin_name()

    def build_generic_pin(self):
        """
        Build simple pins design
        """
        if self.ops_type == PeriDesignDirectItem.OpsType.op_tx:
            assert self.tx_info is not None
            self.tx_info.build_generic_pin()
        elif self.ops_type == PeriDesignDirectItem.OpsType.op_rx:
            assert self.rx_info is not None
            self.rx_info.build_generic_pin()
        elif self.ops_type == PeriDesignDirectItem.OpsType.op_bidir:
            assert self.tx_info is not None
            assert self.rx_info is not None
            self.rx_info.build_generic_pin()
            self.tx_info.build_generic_pin()

    def generate_pin_name(self):
        is_tx_gen = self.ops_type == PeriDesignDirectItem.OpsType.op_tx \
                    or self.ops_type == PeriDesignDirectItem.OpsType.op_bidir

        is_rx_gen = self.ops_type == PeriDesignDirectItem.OpsType.op_rx \
                    or self.ops_type == PeriDesignDirectItem.OpsType.op_bidir

        if is_rx_gen and self.rx_info is not None:
            self.rx_info.generate_pin_name_from_inst(self.name)

        if is_tx_gen and self.tx_info is not None:
            self.tx_info.generate_pin_name_from_inst(self.name)

    def refresh_pin_name(self, old_substr, new_substr):
        is_tx_gen = self.ops_type == PeriDesignDirectItem.OpsType.op_tx \
                    or self.ops_type == PeriDesignDirectItem.OpsType.op_bidir

        is_rx_gen = self.ops_type == PeriDesignDirectItem.OpsType.op_rx \
                    or self.ops_type == PeriDesignDirectItem.OpsType.op_bidir

        if is_rx_gen and self.rx_info is not None:
            self.rx_info.refresh_pin_name(old_substr, new_substr)

        if is_tx_gen and self.tx_info is not None:
            self.tx_info.refresh_pin_name(old_substr, new_substr)

    def is_match_pin_name(self, text):

        # Check tx
        target = self.tx_info
        if target is not None and target.is_match_pin_name(text):
            return True

        # Check rx
        target = self.rx_info
        if target is not None and target.is_match_pin_name(text):
            return True

        return False


class PeriDesignModeGenPinItem(PeriDesignGenPinItem):
    """
    TODO : This class is not used in design db. It is currently used in device service testing.
    It needs to be refactored to device db.

    A design item that has container of generic pin.

    A generic pin info is specified in device database and dynamically created
    as design element at runtime when the item is first created.

    The difference between this and the PeriDesignGenPinItem is that
    this is pin that are associated to a mode and not the top-level
    port.

    This has to be captured differently because some of the details of the interface
    pin have different values in various modes.

    .. note::
       Since this inherits from metaclass, the class variable has unique variable for
       each inherited class.
    """

    # Store the mode (block_definition.BlockMode type) that is associated to this generic pin
    # _device_block_mode = None

    # The _device_port_map in this class is actually of a different type:
    # inf name(str): ModeInterface


    @abc.abstractmethod
    def __init__(self):
        super().__init__()

        self.gen_pin = GenericPinGroup()

        # Child class need to call set_default setting in implementation

    def __str__(self, *args, **kwargs):
        info = '{} pin_count:{}' \
            .format(self,
                    self.gen_pin.get_pin_count())
        return info

    def build_generic_pin(self):
        """
        Dynamically generate generic pins based on device info
        """
        assert self._device_port_map is not None
        self.gen_pin.clear()

        from device.block_definition import Port as DevicePort

        for device_inf in self._device_port_map.values():

            if device_inf.get_type_name() == DevicePort.TYPE_CLOCK:
                self.gen_pin.add_clock_pin(device_inf.get_name(), "", device_inf.is_bus_port())
            else:
                self.gen_pin.add_pin(device_inf.get_name(), "", device_inf.is_bus_port())

    def get_pin_class(self, pin_type_name):
        """
        Based on device pin type, get its pin class.

        Pin type is the port name defined in device db.
        Pin class is the port class name defined in device db.

        :param pin_type_name: Device pin type name
        :return: Device pin class if found, else None
        """
        assert self._device_port_map is not None

        device_inf = self._device_port_map.get(pin_type_name, None)
        if device_inf is not None:
            return device_inf.get_class_name()

        return None

    def is_pin_type_clock(self, pin_type_name):
        """
        Based device pin type, check if it is a clock pin

        :param pin_type_name: Device pin type name
        :return: True, if clock, else False
        """
        from device.block_definition import Port as DevicePort

        device_inf = self._device_port_map.get(pin_type_name, None)
        if device_inf is not None:
            device_inf_type = device_inf.get_type_name()
            return device_inf_type == DevicePort.TYPE_CLOCK

        return False


@gen_util.freeze_it
class PeriDesignRegistry(PeriDesignItem):
    """
    A generic design registry for any block instances.

    .. note::
    Created starting from AN20 support. AN08 blocks are not ported/refactored(yet).
    """

    def __init__(self):
        super().__init__()

        self._dev2inst_map = {}  #: A map of used block def (ie dev inst) to its block instance
        self._name2inst_map = {}  #: A map of instance name and its instance
        self._write_lock = threading.RLock()  #: Sync lock for multi-threaded write
        self.device_db = None  #: Device db
        self.is_dirty = False  #: Identify change state, if it has been changed but not save, is_dirty is True

        # Database monitors
        self._res_subject = RegResourceSubject()  #: Monitor changes on device resource usage
        self._inst_subject = RegInstanceSubject()  #: Monitor changes on device instance usage

        self.logger = Logger

    def __str__(self, *args, **kwargs):

        info = 'inst_count:{} dev_count:{}' \
            .format(len(self._name2inst_map), len(self._dev2inst_map))

        inst_list = []
        for inst in self._name2inst_map.values():
            inst_list.append("{}\n".format(inst))

        dev_list = []
        for dev in self._dev2inst_map.values():
            dev_list.append("{}\n".format(dev))

        info = "{}\nname2inst{}\ndev2inst:{}\n".format(info, "".join(inst_list), "".join(dev_list))

        return info

    def copy_common(self, from_obj, exclude_attr=None):
        """
        Override. Copy data state. Operation states remain as caller states.
        """

        # Same class access cause a warning, ridiculous
        # pylint: disable=protected-access
        # noinspection PyProtectedMember
        self._dev2inst_map = from_obj._dev2inst_map

        # pylint: disable=protected-access
        # noinspection PyProtectedMember
        self._name2inst_map = from_obj._name2inst_map

        self.device_db = from_obj.device_db

    def get_name_map_iterator(self):
        return iter(self._name2inst_map)

    def get_dev_map_iterator(self):
        return iter(self._dev2inst_map)

    def get_resource_subject(self):
        return self._res_subject

    def get_instance_subject(self):
        return self._inst_subject

    def create_chksum(self):
        return hash(PeriDesignRegistry)

    def set_default_setting(self, device_db=None):
        # To comply with parent interface
        gen_util.mark_unused(device_db)

    @abc.abstractmethod
    def create_instance(self, name: str, apply_default=True, auto_pin=False):
        """
        Create and register a block instance

        :param name: Instance name
        :param apply_default: True to apply default setting as per spec
        :param auto_pin: True, auto generate the pin name
        :return: A block instance
        :raise: db_item.excp.CreateException
        """
        # For sample implementation, see unit test
        pass

    def create_unregister_instance(self, name, apply_default=True, auto_pin=False):
        """
        Create a block instance but does not register it.
        This does not check for collision.

        :param name: Instance name
        :param apply_default: True to apply default setting as per spec
        :param auto_pin: True, auto generate the pin name
        :return: A block instance
        :raise: db_item.excp.CreateException

        ..notes:
          Useful for migration or when a standalone instance is needed
        """
        gen_util.mark_unused(name)
        gen_util.mark_unused(apply_default)
        gen_util.mark_unused(auto_pin)
        warnings.warn("create_unregister_instance(), using unimplemented function", RuntimeWarning)
        return None

    def _register_new_instance(self, inst):
        with self._write_lock:
            if self._name2inst_map.get(inst.name, None) is None:
                self._name2inst_map[inst.name] = inst

                # Broadcast instance creation
                self._inst_subject.set_state(inst.get_device(), inst.name, "", RegInstanceSubject.ChangeType.add)

            else:
                msg = "Fail to create. Instance exists. : {}".format(inst.name)
                raise CreateException(msg, app_excp.MsgLevel.warning)

    def create_instance_with_device(self, name, device_def, apply_default=True, auto_pin=False) -> Optional[Any]:
        """
        Create and register an instance, then apply the instance device resource.
        This function make sure that the device is registered when the instance is created.

        :param auto_pin: True, enable auto-pin generation else no
        :param name: Instance name
        :param device_def: Device resource for new instance
        :param apply_default: True to apply default setting as per spec
        :return: An instance
        :raise: db_item.CreateException
        """
        if device_def is None or device_def == "":
            msg = "Fail to create. Device resource is not specified."
            raise CreateException(msg, app_excp.MsgLevel.warning)

        inst: Optional[Any] = self.create_instance(name, apply_default, auto_pin)
        if inst is not None:
            self.assign_inst_device(inst, device_def)

        return inst

    def rename_inst(self, current_name, new_name, auto_pin=False):
        """
        Set new name to block instance object and update the registry mapping

        :param current_name: Current instance name
        :param new_name: New instance name
        :param auto_pin: True, auto generate the pin name
        :return: The updated instance object
        """
        with self._write_lock:
            if current_name == new_name:
                return self.get_inst_by_name(current_name)

            elif self.get_inst_by_name(new_name) is not None:
                msg = "Instance name {} is already used".format(new_name)
                raise InstanceDoesNotExistException(msg, app_excp.MsgLevel.warning)

            inst = self.get_inst_by_name(current_name)
            if inst is not None:
                self._name2inst_map.pop(current_name, None)
                inst.name = new_name
                self._name2inst_map[new_name] = inst

                if auto_pin:
                    inst.refresh_pin_name(current_name, new_name)

                # Broadcast instance rename
                self._inst_subject.set_state(inst.get_device(), inst.name, current_name,
                                                RegInstanceSubject.ChangeType.rename)

            else:
                msg = "Fail to find instance : {}".format(current_name)
                raise InstanceDoesNotExistException(msg, app_excp.MsgLevel.warning)

            return inst

    def delete_inst(self, name):

        with self._write_lock:
            if name in self._name2inst_map:
                inst = self._name2inst_map.get(name, None)
                dev_name = ""
                if inst is not None:

                    dev_name = inst.get_device()
                    if dev_name in self._dev2inst_map:
                        self._dev2inst_map[dev_name] = None
                        # Delete from device map
                        del self._dev2inst_map[dev_name]

                    # Broadcast resource clear
                    self._res_subject.set_state(dev_name, "")

                # Delete from name map
                del self._name2inst_map[name]

                # Broadcast instance delete
                self._inst_subject.set_state(dev_name, name, "", RegInstanceSubject.ChangeType.delete)

    def delete_all_insts(self):
        name_list = list(self._name2inst_map.keys())
        for inst_name in name_list:
            self.delete_inst(inst_name)

    def assign_inst_device(self, inst, new_dev):
        """
        Assign new block device to block instance.

        :param inst: Current block instance
        :param new_dev: New block device instance name
        """
        if inst is None:
            return

        if inst.get_device() == new_dev:
            return

        with self._write_lock:
            if inst.get_device() != "":
                self._dev2inst_map.pop(inst.get_device(), None)
                # Broadcast resource clear
                self._res_subject.set_state(inst.get_device(), "")

            inst.set_device(new_dev)
            if new_dev != "":
                self._dev2inst_map[new_dev] = inst

            # Broadcast resource use
            self._res_subject.set_state(new_dev, inst.name)

    def reset_inst_device(self, inst):
        """
        Reset device to empty. Remove from device map.

        :param inst: Current block instance
        """
        if inst is None:
            return

        with self._write_lock:
            res_name = inst.get_device()
            if res_name != "":
                self._dev2inst_map.pop(res_name, None)

            inst.set_device("")

            # Broadcast resource clear
            self._res_subject.set_state(res_name, "")

    def gen_unique_inst_name(self, prefix=""):

        for i in range(1, 10):

            if prefix == "":
                name = "inst{}".format(self.get_inst_count() + i)
            else:
                name = "{}_inst{}".format(prefix, self.get_inst_count() + i)

            inst = self._name2inst_map.get(name, None)
            if inst is None:
                return name
            else:
                name = "{}_".format(name)
                if self._name2inst_map.get(name) is None:
                    return name

        # Fail to generate after several attempts
        return None

    def get_inst_count(self) -> int:
        """
        Get the count of registered instance

        :returns All instance count
        """
        return len(self._name2inst_map)

    def get_inst_by_name(self, name: str) -> PeriDesignItem | None:
        """
        Find instance that match given name

        :returns If found returns the instance, else None
        """
        return self._name2inst_map.get(name, None)

    def get_all_inst(self):
        """
        Get a list of all instances

        :return: Instance list
        """
        return list(self._name2inst_map.values())

    def is_inst_name_used(self, name):
        """
        Check if the specific instance name has been used in registry

        :param name: Instance name
        :return: True, name is used, else False
        """
        return name in self._name2inst_map

    def is_device_used(self, device_def):
        """
        Check if a specific device name has been used in registry

        :param device_def: Block instance device name
        :return: True, device is used, else False
        """
        return device_def in self._dev2inst_map

    @abc.abstractmethod
    def is_device_valid(self, device_def):
        """
        Check if the device is a valid resource in device db

        :param device_def: Device resource name to check
        :return: True, if valid, else False
        """
        pass

    def get_all_device(self):
        """
        Get a list of devices used by block instance

        :return: A list of device name
        """
        return list(self._dev2inst_map.keys())

    def get_device_count(self):
        """
        Get count of device used by block instance

        :return: A count
        """
        return len(self._dev2inst_map)

    def get_dev2inst_map(self):
        """
        Get a map of all device name map to instance

        :return: Block device used map
        """
        return self._dev2inst_map

    def get_name2inst_map(self):
        """
        Get a map of all name map to instance

        :return: Name used map
        """
        return self._name2inst_map

    def overwrite_internal_map(self, name_map, dev_map):
        """
        Overwrite internal name and device tracking map. For migration support.

        :param name_map: Name usage map
        :param dev_map: Device usage map
        """
        with self._write_lock:
            self._name2inst_map = name_map
            self._dev2inst_map = dev_map

    def get_instance_class(self):
        """
        Get class of the first instance. This should be the class of all instances
        for homegeneous registry item.

        :return: Instance's Class
        """
        try:
            item_key = next(iter(self._name2inst_map))
            if item_key is not None:
                inst = self._name2inst_map.get(item_key, None)
                if inst is not None:
                    return type(inst)
        except StopIteration:
            pass

        return None

    def get_inst_by_device_name(self, device_def):
        """
        Find instance that match given device name

        :return: If found returns the instance, else None
        """
        return self._dev2inst_map.get(device_def, None)

    def clear_invalid_resource(self):
        """
        Check all instance if the device is valid.
        If not reset the assignment.
        """
        with self._write_lock:
            inst_list = self._name2inst_map.values()
            for inst in inst_list:
                if not self.is_device_valid(inst.get_device()):
                    self.reset_inst_device(inst)

    def clear_all_resource(self):
        """
        Clear all assignment. Does not broadcast changes.
        """
        with self._write_lock:
            for inst in self._name2inst_map.values():
                inst.set_device("")
            self._dev2inst_map.clear()

    def change_device(self, device_db):
        """
        Update registry and its instances to the new device.
        If instance uses invalid resource, clear the assigment.

        :param device_db: Device db object matching the device name
        """
        with self._write_lock:
            self.device_db = device_db
            self.clear_invalid_resource()

    def vertical_device_migration(self, device_db, old2new_resource_map):
        """
        Update registry by reassigning the resource from
        the current to the new one by referring to the map

        :param device_db: Device db object matching the device
        :param old2new_resource_map: A dictionary of old device instance
                name mapped to device instance in the new device
        """
        logger = Logger

        # List of instances that have been visited for reassignment
        visited_ins_names = []
        # map of instance name to the original device resource name
        # if they had to be cleared due to conflicting resource after
        # reassignment
        reassigned_resource = {}

        with self._write_lock:
            self.device_db = device_db

            inst_list = self._name2inst_map.values()
            for inst in inst_list:
                old_resource = inst.get_device()
                if old_resource == "" or old_resource is None:
                    # Check if it was reset due to conflict from
                    # earlier reassignment
                    if inst.name not in visited_ins_names and \
                            inst.name in reassigned_resource:
                        old_resource = reassigned_resource[inst.name]

                if old_resource in old2new_resource_map:
                    new_resource = old2new_resource_map[old_resource]

                    logger.debug("Vertical migration of {} from {} to {}".format( # type: ignore
                        inst.name, old_resource, new_resource))

                    if self.is_device_valid(new_resource):
                        # Check if the new resource is currently being assigned
                        # to some other instance from old assignment
                        if self.is_device_used(new_resource):
                            conflicted_ins = self.get_inst_by_device_name(new_resource)
                            if conflicted_ins is not None and conflicted_ins != inst:
                                reassigned_resource[conflicted_ins.name] = new_resource
                                # Remove the assignment from the instance
                                self.assign_inst_device(conflicted_ins, "")

                        self.assign_inst_device(inst, new_resource)
                    else:
                        self.reset_inst_device(inst)

                else:
                    # Clear the resource
                    self.reset_inst_device(inst)

                visited_ins_names.append(inst.name)


class PeriDesignRegistryDirectMixin:
    """
    Provides operation for registry with directional block instance.
    Thix mixin works on PeriDesignRegistry and PeriDesignDirectItem or
    any class that provides those reuired registry/item data, type and function.

    ..notes::
      Checkout Python books for more info about mixin. It is not enforced by
      Python intepreter like other language. More like a design pattern. It must not
      have data member. Just operation and cannot be instantiated. It is used
      to separate data from operation.
    """

    @abc.abstractmethod
    def __init__(self):
        self._name2inst_map = {}
        self.device_db = None

    def create_instance(self, name, apply_default=True, auto_pin=False):
        pass

    def create_tx_instance(self, name, apply_default=True, auto_pin=False):
        """
        Create and register a tx block instance

        :param auto_pin:
        :param name: Instance name
        :param apply_default: True to apply default setting as per spec
        :return: A block instance
        :raise: db_item.CreateException
        """
        return self.create_instance(name, apply_default, auto_pin)

    def create_rx_instance(self, name, apply_default=True, auto_pin=False):
        """
        Create and register a rx block instance

        :param auto_pin:
        :param name: Instance name
        :param apply_default: True to apply default setting as per spec
        :return: A block instance
        :raise: db_item.CreateException
        """
        block_inst = self.create_instance(name, apply_default, auto_pin)
        assert block_inst is not None
        block_inst.update_type(type(block_inst).OpsType.op_rx, self.device_db, auto_pin)
        return block_inst

    def create_bidir_instance(self, name, apply_default=True, auto_pin=False):
        """
        Create and register a bidir lvds instance

        :param auto_pin:
        :param name: Instance name
        :param apply_default: True to apply default setting as per spec
        :return: A lvds instance
        :raise: db_item.CreateException
        """
        block_inst = self.create_instance(name, apply_default, auto_pin)
        assert block_inst is not None
        block_inst.update_type(type(block_inst).OpsType.op_bidir, self.device_db, auto_pin)
        return block_inst

    def get_type_inst_count(self, is_tx) -> int:
        if is_tx:
            return self.get_tx_inst_count()
        else:
            return self.get_rx_inst_count()

    def get_instance_class(self) -> Any:
        pass

    def get_name_map_iterator(self) -> Any:
        pass

    def get_inst_by_name(self, name: str):
        pass

    def get_tx_inst_count(self) -> int:
        count = 0
        block_inst_class = self.get_instance_class()
        inst_iter = self.get_name_map_iterator()
        try:
            item_key = next(inst_iter)
            while item_key is not None:
                inst = self.get_inst_by_name(item_key)
                assert inst is not None
                if inst.ops_type == block_inst_class.OpsType.op_tx:
                    count += 1
                item_key = next(inst_iter)
        except StopIteration:
            pass

        return count

    def get_rx_inst_count(self) -> int:
        count = 0
        block_inst_class = self.get_instance_class()
        inst_iter = self.get_name_map_iterator()
        try:
            item_key = next(inst_iter)
            while item_key is not None:
                inst = self.get_inst_by_name(item_key)
                assert inst is not None
                if inst.ops_type == block_inst_class.OpsType.op_rx:
                    count += 1
                item_key = next(inst_iter)
        except StopIteration:
            pass

        return count

    def get_bidir_inst_count(self) -> int:
        count = 0
        block_inst_class = self.get_instance_class()
        inst_iter = self.get_name_map_iterator()
        try:
            item_key = next(inst_iter)
            while item_key is not None:
                inst = self.get_inst_by_name(item_key)
                assert inst is not None
                if inst.ops_type == block_inst_class.OpsType.op_bidir:
                    count += 1
                item_key = next(inst_iter)
        except StopIteration:
            pass

        return count

    def get_all_inst(self) -> List:
        return []

    def get_all_tx_inst(self) -> List:

        block_inst_class = self.get_instance_class()

        def is_tx(inst):
            return inst.ops_type == block_inst_class.OpsType.op_tx

        return list(filter(is_tx, self.get_all_inst()))

    def get_all_rx_inst(self) -> List:

        block_inst_class = self.get_instance_class()

        def is_rx(inst):
            return inst.ops_type == block_inst_class.OpsType.op_rx

        return list(filter(is_rx, self.get_all_inst()))

    def get_all_bidir_inst(self) -> List:

        block_inst_class = self.get_instance_class()

        def is_bidir(inst):
            return inst.ops_type == block_inst_class.OpsType.op_bidir

        return list(filter(is_bidir, self.get_all_inst()))

    def gen_unique_direct_inst_name(self, direct_type: PeriDesignDirectItem.OpsType, prefix="") -> Union[str, None]:

        if direct_type == PeriDesignDirectItem.OpsType.op_tx:
            count = self.get_tx_inst_count()
        elif direct_type == PeriDesignDirectItem.OpsType.op_rx:
            count = self.get_rx_inst_count()
        else:
            count = self.get_bidir_inst_count()

        if prefix == "":
            name = "inst{}".format(count + 1)
        else:
            name = "{}_inst{}".format(prefix, count + 1)

        inst = self._name2inst_map.get(name, None)
        if inst is None:
            return name
        else:
            max_limit = 5
            try_count = 0
            while True:
                inst = self._name2inst_map.get(name, None)
                if inst is None:
                    return name
                else:
                    name = "{}_".format(name)
                    try_count += 1
                    if try_count > max_limit:
                        return None


class HasGenPin(Protocol):
    def build_generic_pin(self) -> None: ...

    def generate_pin_name(self) -> None: ...

    @property
    def gen_pin(self) -> GenericPinGroup: ...


class HasParamInfoGroup(Protocol):

    def get_param_group(self) -> GenericParamGroup: ...

    @property
    def param_group(self) -> GenericParamGroup: ...

    def get_param_info(self) -> PropertyMetaData: ...

    @property
    def param_info(self) -> PropertyMetaData: ...


if __name__ == "__main__":
    pass
