"""
Copyright (C) 2017-2018 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Oct 11, 2017

@author: yasmin
"""

from __future__ import annotations
import os
import csv
from enum import Enum
from typing import Dict, Optional, Union, TYPE_CHECKING, List, Tuple
import xml.etree.ElementTree as et
# 3rd party library to display pretty table
from prettytable import PrettyTable

import util.gen_util
from util.singleton_logger import Logger
import util.excp as app_excp
import util.plugin_map as plmap

import common_device.rule as common_rule

import design.db_item as db_item
from design.db_patch import DesignVersion

if TYPE_CHECKING:
    from design.device_setting import DeviceSetting
    from device.db import PeripheryDevice
    from common_device.gpio.gpio_design import GPIORegistry
    from tx60_device.gpio.gpio_design_comp import GPIORegistryComplex
    from common_device.pll.pll_design import PLLRegistry
    from common_device.osc.osc_design import OSCRegistry
    from common_device.lvds.lvds_design import LVDSRegistry
    from tx60_device.lvds.lvds_design_adv import LVDSRegistryAdvance
    from common_device.mipi.mipi_design import MIPIRegistry
    from common_device.mipi_dphy.mipi_dphy_design import MIPIDPhyRegistry
    from tx180_device.mipi_dphy.design import MIPIHardDPHYRegistry
    from common_device.jtag.jtag_design import JTAGRegistry
    from common_device.ddr.ddr_design import DDRRegistry
    from common_device.spi_flash.spi_flash_design import SPIFlashRegistry
    from common_device.h264.h264_design import H264Registry
    from common_device.hyper_ram.hyper_ram_design import HyperRAMRegistry
    from common_device.clock.clock_design import ClockRegistry
    from common_device.quad.lane_design import LaneBaseRegistry
    from tx180_device.pll_ssc.design import PLLSSCRegistry
    from tx375_device.soc.design import SOCRegistry
    from tx375_device.quad.design import QuadRegistry
    from tx375_device.common_quad.design import QuadLaneCommonRegistry
    from tx375_device.quad_pcie.design import QuadPCIERegistry
    from tx375_device.lane10g.design import Lane10GRegistry
    from tx375_device.raw_serdes.design import RawSerdesRegistry
    from tx375_device.lane1g.design import Lane1GRegistry


@util.gen_util.freeze_it
class PeriDesign(db_item.PeriDesignItem):
    """
    Periphery design

    This is a database that holds all designed periphery blocks stored in block specific container or registry.
    """

    class BlockType(Enum):
        """
        Block type supported in design, in general. It is not based on family.

        Block type naming should falls under basic, advance, complex and more category as needed.
        When block type only has minor differences compare to previous family, it should be differentiated by adding
        level. Minor as in extension rather than property redesign. This difference relates to device design.

        For example, gpio properties differs between an08, an20 and an120. So the block type names can be

        basic : gpio
        advance : adv_gpio           <---- This type scheme is not implemented yet
        more advance : adv_l2_gpio   <---- Level 2 advance

        basic : pll
        advance : adv_pll

        A device block that does not need configuration is not considered as a block type for design
        """
        other = 1
        gpio = 2
        pll = 3
        adv_pll = 4  #: PLL with feedback and multiplex clocks, AN20 pll
        osc = 5
        clock = 6
        lvds = 7
        mipi = 8
        adv_l2_gpio = 9  #: AN120 gpio
        jtag = 10
        ddr = 11
        h264 = 12
        comp_gpio = 13
        comp_pll = 14     #: Tesseract-based pll
        adv_lvds = 15     #: Tesseract-based lvds
        mipi_dphy = 16     #: Tesseract-based mipi
        adv_osc = 17        #: Tesseract-based osc
        adv_clock = 18      #: Tesseract-based clock
        spi_flash = 19  #: Tesseract-based spi flash interface
        hyper_ram = 20  #: Tesseract-based hyper_ram interface
        hposc = 21  #: Tesseract-based high precision oscillator
        comp_clock = 22 # Ti180 clockmux
        rclkmux = 23  # Ti180 regional clockmux
        efx_pll_v3_comp = 24 # Ti180 PLL
        mipi_hard_dphy = 25 # Ti180 Harden MIPI dphy
        adv_ddr = 26 # Ti180 ddr
        pll_ssc = 27 # Ti180 pll ssc
        soc = 28
        quad = 29
        quad_pcie = 30
        pma_clkmux = 31
        efx_fpll_v1 = 32 # Ti375 FPLL
        lane_10g = 33
        common_quad = 34 # Common quad
        raw_serdes = 35 # Raw serdes
        lane_1g = 36

    blktype2str_map = \
        {
            BlockType.gpio: "GPIO",
            BlockType.pll: "PLL",
            BlockType.adv_pll: "PLL",
            BlockType.osc: "OSC",
            BlockType.lvds: "LVDS",
            BlockType.mipi: "MIPI",
            BlockType.adv_l2_gpio: "GPIO",
            BlockType.ddr: "DDR",
            BlockType.h264: "H264",
            BlockType.jtag: "JTAG",
            BlockType.comp_gpio: "GPIO",
            BlockType.comp_pll: "PLL",
            BlockType.adv_lvds: "LVDS",
            BlockType.mipi_dphy: "MIPI_DPHY",
            BlockType.adv_osc: "OSC",
            BlockType.hposc: "OSC",
            BlockType.spi_flash: "SPI_FLASH",
            BlockType.hyper_ram: "HYPER_RAM",
            BlockType.efx_pll_v3_comp: "PLL",
            BlockType.mipi_hard_dphy: "MIPI",
            BlockType.adv_ddr: "DDR",
            BlockType.pll_ssc: "PLL_SSC",
            BlockType.soc: "SOC",
            BlockType.quad: "QUAD",
            BlockType.quad_pcie: "QUAD_PCIE",
            BlockType.efx_fpll_v1: "PLL",
            BlockType.lane_10g: "10GBASE_KR",
            BlockType.common_quad: "COMMON_QUAD",
            BlockType.raw_serdes: "PMA_DIRECT",
            BlockType.lane_1g: "SGMII",
        }  #: Mapping of block type in enum to string type

    str2blktype_map = \
        {
            "GPIO": [BlockType.gpio, BlockType.adv_l2_gpio, BlockType.comp_gpio],
            "PLL": [BlockType.pll, BlockType.adv_pll, BlockType.comp_pll, BlockType.efx_pll_v3_comp, BlockType.efx_fpll_v1],
            "OSC": [BlockType.osc, BlockType.adv_osc, BlockType.hposc],
            "LVDS": [BlockType.lvds, BlockType.adv_lvds],
            "MIPI": [BlockType.mipi, BlockType.mipi_hard_dphy],
            "MIPI_DPHY": BlockType.mipi_dphy,
            "DDR": [BlockType.ddr, BlockType.adv_ddr],
            "H264": BlockType.h264,
            "JTAG": BlockType.jtag,
            "SPI_FLASH": BlockType.spi_flash,
            "HYPER_RAM": BlockType.hyper_ram,
            "PLL_SSC": BlockType.pll_ssc,
            "SOC": BlockType.soc,
            "QUAD": BlockType.quad,
            "QUAD_PCIE": BlockType.quad_pcie,
            "10GBASE_KR": BlockType.lane_10g,
            "COMMON_QUAD": BlockType.common_quad,
            "RAW_SERDES": BlockType.raw_serdes,
            "SGMII": BlockType.lane_1g,
        }  #: Mapping of block type in string to enum type

    def __init__(self, name, device_name):
        super().__init__()

        self.name = name  #: Design name
        self.device_def = device_name  #: A valid device name as per device database
        self.timing_model = ""  #: A timing model that match with the device
        self.last_change_date = ""  #: Timestamp when the design file is saved to file
        self.version = ""  #: Last tool version used to update the db
        self.location = ""  #: Location of the db file

        self.plugin_map = None  #: A map of device name to its supported plugin
        self.gpio_reg: Optional[Union[GPIORegistry, GPIORegistryComplex]] = None  #: GPIO registry
        self.pll_reg: Optional[PLLRegistry] = None  #: PLL registry
        self.osc_reg: Optional[OSCRegistry] = None  #: Oscillator registry, if advance osc and hposc co-exist, need to manage the builder
        self.clock_reg: Optional[ClockRegistry] = None  #: Clock registry
        self.lvds_reg: Optional[Union[LVDSRegistry, LVDSRegistryAdvance]] = None  #: LVDS registry
        self.mipi_reg: Optional[MIPIRegistry] = None  #: MIPI registry
        self.mipi_dphy_reg: Optional[MIPIDPhyRegistry] = None  #: MIPI DPHY registry
        self.mipi_hard_dphy_reg: Optional[MIPIHardDPHYRegistry] = None # MIPI Harden DPHY registry
        self.jtag_reg: Optional[JTAGRegistry] = None  #: JTAG registry
        self.ddr_reg: Optional[DDRRegistry] = None  #: DDR registry
        self.h264_reg: Optional[H264Registry] = None  #: H264 registry
        self.spi_flash_reg: Optional[SPIFlashRegistry] = None  #: SPI Flash registry
        self.hyper_ram_reg: Optional[HyperRAMRegistry] = None  #: Hyper RAM registry
        self.pll_ssc_reg: Optional[PLLSSCRegistry] = None #: PLL SSC registry
        self.soc_reg: Optional[SOCRegistry] = None
        self.quad_reg: Optional[QuadRegistry] = None
        self.common_quad_lane_reg: Optional[QuadLaneCommonRegistry] = None
        self.quad_pcie_reg: Optional[QuadPCIERegistry] = None
        self.lane_10g_reg: Optional[Lane10GRegistry] = None
        self.raw_serdes_reg: Optional[RawSerdesRegistry] = None
        self.lane_1g_reg: Optional[Lane1GRegistry] = None
        self.blktype2reg_map = None  #: Map of block type to its registry

        # Specific type used for multi-type block, only one type can be used at
        # a time
        self.gpio_type = self.BlockType.gpio  #: Runtime gpio used type
        self.pll_type = self.BlockType.pll  #: Runtime pll used type
        self.lvds_type = self.BlockType.lvds  #: Runtime lvds used type
        self.osc_type = self.BlockType.osc  #: Runtime osc used type
        self.mipi_type = self.BlockType.mipi #: Runtime mipi used type
        self.ddr_type = self.BlockType.ddr #: Runtime ddr used type

        self.device_setting: Optional[DeviceSetting] = None
        self.design_file = ""  #: Design file name
        self.device_db: Optional[PeripheryDevice] = None  #: Device database, generated from device name in design file
        self._supported_block = []

        #: Identify change state, if it has been changed but not save, is_dirty is True
        self.is_dirty = False

        self.issue_reg: Optional[DesignIssueRegistry] = None  #: Design issue registry
        self.export_import_issue_reg = None  # GPIO import/export issue registry

        self.db_version = DesignVersion()
        self.__partial_designs = {}

    def __str__(self, *args, **kwargs):

        gpio_count = 0
        if self.gpio_reg is not None:
            gpio_count = self.gpio_reg.get_gpio_count()

        pll_count = 0
        if self.pll_reg is not None:
            pll_count = self.pll_reg.get_inst_count()

        osc_count = 0
        if self.osc_reg is not None:
            osc_count = self.osc_reg.get_osc_count()

        lvds_count = 0
        if self.lvds_reg is not None:
            lvds_count = self.lvds_reg.get_inst_count()

        mipi_count = 0
        if self.mipi_reg is not None:
            mipi_count = self.mipi_reg.get_inst_count()

        mipi_dphy_count = 0
        if self.mipi_dphy_reg is not None:
            mipi_dphy_count = self.mipi_dphy_reg.get_inst_count()

        jtag_count = 0
        if self.jtag_reg is not None:
            jtag_count = self.jtag_reg.get_inst_count()

        ddr_count = 0
        if self.ddr_reg is not None:
            ddr_count = self.ddr_reg.get_inst_count()

        h264_count = 0
        if self.h264_reg is not None:
            h264_count = self.h264_reg.get_inst_count()

        spi_flash_count = 0
        if self.spi_flash_reg is not None:
            spi_flash_count = self.spi_flash_reg.get_inst_count()

        hyper_ram_count = 0
        if self.hyper_ram_reg is not None:
            hyper_ram_count = self.hyper_ram_reg.get_inst_count()

        info = 'name:{} device:{} date:{} version:{}' \
            .format(self.name,
                    self.device_def,
                    self.last_change_date,
                    self.version)

        info = '{} gpio:{} pll:{} osc:{} lvds:{} mipi:{} mipi_dphy:{} jtag:{} ddr:{} h264:{} spi_flash:{} hyper_ram:{}' \
            .format(info,
                    gpio_count,
                    pll_count,
                    osc_count,
                    lvds_count,
                    mipi_count,
                    mipi_dphy_count,
                    jtag_count,
                    ddr_count,
                    h264_count,
                    spi_flash_count,
                    hyper_ram_count)

        return info

    def get_db_version(self):
        """
        Get design version object

        :return: Design db version
        """
        return self.db_version

    def set_design_version_name(self, version_name):
        """
        Set design db version name

        :param version_name: A version name
        """
        version_no = DesignVersion.convert_version_name2no(version_name)
        self.db_version.set_design_version(version_no)

    def get_design_version_name(self):
        """
        Get design version name

        :return: A version name
        """
        version_name = DesignVersion.convert_version_no2name(
            self.db_version.get_design_version())
        return version_name

    def get_design_version_no(self):
        """
        Get design version number

        :return: A version number
        """
        return self.db_version.get_design_version()

    def set_default_design_version(self):
        """
        Set a default design version when no version exists
        """

        if self.version == "":
            # For unit test, where version in file maybe empty
            default_version = DesignVersion.RELEASE_DESIGN_VERSION
            self.db_version.set_design_version(default_version)
        else:
            # Take the last version used to save as defaults
            default_version = self.version

            # Any master build, just assumes it is the first quarter release
            default_version = default_version.replace('M', '1')

            # So it can be converted to integer
            default_version = default_version.replace('.', '')

            self.set_design_version_name(default_version)

    def update_design_version(self):
        """
        Update design version to the latest release version
        """
        rel_design_ver = DesignVersion.get_release_design_version_no()
        curr_design_ver = self.db_version.get_design_version()
        app_patch_ver = self.db_version.get_applied_patch_version()

        if app_patch_ver > curr_design_ver:
            new_version = app_patch_ver
        else:
            if curr_design_ver > rel_design_ver:
                new_version = curr_design_ver
            else:
                new_version = rel_design_ver

        self.db_version.set_design_version(new_version)

    def setup_plugin_map(self, plugin_map=None):

        self.plugin_map = plugin_map
        if self.plugin_map is None:
            self.plugin_map = plmap.PluginMap.build_from_default_file()

        self.identify_supported_block()

    def is_tesseract_design(self):
        # Use complex gpio to identify Tesseract family
        return self.is_block_supported(self.BlockType.comp_gpio)

    def setup_tesseract_hsio_service(self):
        """
        Setup service to manage HSIO and HVIO resources.
        This function must called after the design is fully setup
        so that all relevant registry is available.
        """

        if self.is_tesseract_design():
            from tx60_device.hio_res_service import HIOResService
            hio_svc = HIOResService()
            hio_svc.build(self)
            if self.gpio_reg is not None:
                from tx60_device.gpio.gpio_design_comp import GPIORegistryComplex
                assert isinstance(self.gpio_reg, GPIORegistryComplex)
                self.gpio_reg.set_resource_service(hio_svc)

            if self.lvds_reg is not None:
                from tx60_device.lvds.lvds_design_adv import LVDSRegistryAdvance
                assert isinstance(self.lvds_reg, LVDSRegistryAdvance)
                self.lvds_reg.set_resource_service(hio_svc)

            if self.mipi_dphy_reg is not None:
                self.mipi_dphy_reg.set_resource_service(hio_svc)

            # TODO : SPI Flash

    def setup_serdes_res_service(self):
        """
        Setup service to manage Serdes resources that spans across
        different block (quad, quad_pcie).
        This function must called after the design is fully setup
        so that all relevant registry is available.
        """
        if self.is_block_supported(self.BlockType.quad_pcie) or\
            self.is_block_supported(self.BlockType.quad):

            from common_device.quad.res_service import QuadResService

            serdes_svc = QuadResService()
            serdes_svc.build(self)

            if self.quad_pcie_reg is not None:
                self.quad_pcie_reg.set_resource_service(serdes_svc)

            for reg in self.get_lane_based_reg():
                if reg is None:
                    continue
                reg.set_resource_service(serdes_svc)

    def clear_block_map(self):
        self.blktype2reg_map = None

    def setup_block_map(self):
        self.blktype2reg_map = \
            {
                self.BlockType.gpio: self.gpio_reg,
                self.BlockType.adv_l2_gpio: self.gpio_reg,
                self.BlockType.comp_gpio: self.gpio_reg,
                self.BlockType.pll: self.pll_reg,
                self.BlockType.adv_pll: self.pll_reg,
                self.BlockType.comp_pll: self.pll_reg,
                self.BlockType.efx_pll_v3_comp: self.pll_reg,
                self.BlockType.lvds: self.lvds_reg,
                self.BlockType.adv_lvds: self.lvds_reg,
                self.BlockType.mipi: self.mipi_reg,
                self.BlockType.mipi_dphy: self.mipi_dphy_reg,
                self.BlockType.ddr: self.ddr_reg,
                self.BlockType.h264: self.h264_reg,
                self.BlockType.osc: self.osc_reg,
                self.BlockType.adv_osc: self.osc_reg,
                self.BlockType.hposc: self.osc_reg,
                self.BlockType.clock: self.clock_reg,
                self.BlockType.jtag: self.jtag_reg,
                self.BlockType.spi_flash: self.spi_flash_reg,
                self.BlockType.hyper_ram: self.hyper_ram_reg,
                self.BlockType.mipi_hard_dphy: self.mipi_hard_dphy_reg,
                self.BlockType.adv_ddr: self.ddr_reg,
                self.BlockType.pll_ssc: self.pll_ssc_reg,
                self.BlockType.soc: self.soc_reg,
                self.BlockType.quad: self.quad_reg,
                self.BlockType.quad_pcie: self.quad_pcie_reg,
                self.BlockType.lane_10g: self.lane_10g_reg,
                self.BlockType.lane_1g: self.lane_1g_reg,
                self.BlockType.efx_fpll_v1: self.pll_reg,
                self.BlockType.common_quad: self.common_quad_lane_reg,
                self.BlockType.raw_serdes: self.raw_serdes_reg,
            }

    def get_all_common_block_reg(self):
        """
        Get all registry that inherits from PeriDesignRegistry

        :return: A list of registry objects, any of its member can be None
        """

        reg_list = [self.pll_reg, self.lvds_reg, self.mipi_reg, self.jtag_reg, self.ddr_reg,
                    self.h264_reg, self.mipi_dphy_reg, self.spi_flash_reg, self.hyper_ram_reg,
                    self.mipi_hard_dphy_reg]

        # once osc and gpio registry are migrated, it can be added to this list

        return reg_list

    def get_all_custom_block_reg(self):
        """
        Get all registry that has custom registry

        :return: A list of registry objects, any of its member can be None
        """

        reg_list = [self.gpio_reg, self.osc_reg]

        return reg_list

    def get_all_block_reg(self):
        """
        Get all registry

        :return: A list of registry objects, any of its member can be None
        """

        reg_list = self.get_all_common_block_reg()
        reg_list.extend(self.get_all_custom_block_reg())

        return reg_list

    def get_block_reg(self, block_type) -> Optional[db_item.PeriDesignRegistry]:
        """
        Get block registry for given type, which has all its blocks.

        For block with multiple variation, to get the registry it is ok to
        the basic type since a design can only support one type.

        For example, for pll. Using the basic type BlockType.pll will
        return the correct registry regardless of family. Using its specific type
        works too.

        :param block_type: Periphery block type
        :return: Target block registry
        """
        if self.blktype2reg_map is None:
            self.setup_block_map()
        assert self.blktype2reg_map is not None

        block_reg = self.blktype2reg_map.get(block_type, None)
        return block_reg

    def identify_supported_block(self):
        """
        Based on plugin information, identify design specific block type
        """

        if self.device_def == "":
            return

        self._supported_block.clear()
        assert self.plugin_map is not None

        if self.plugin_map.is_plugin_exist(self.device_def, "an08_gpio"):
            self._supported_block.append(PeriDesign.BlockType.gpio)
        elif self.plugin_map.is_plugin_exist(self.device_def, "an120_gpio"):
            self._supported_block.append(PeriDesign.BlockType.adv_l2_gpio)
            self.gpio_type = PeriDesign.BlockType.adv_l2_gpio
        elif self.plugin_map.is_plugin_exist(self.device_def, "tx60_gpio"):
            self._supported_block.append(PeriDesign.BlockType.comp_gpio)
            self.gpio_type = PeriDesign.BlockType.comp_gpio

        if self.plugin_map.is_plugin_exist(self.device_def, "an08_pll"):
            self._supported_block.append(PeriDesign.BlockType.pll)
        elif self.plugin_map.is_plugin_exist(self.device_def, "an20_pll"):
            self._supported_block.append(PeriDesign.BlockType.adv_pll)
            self.pll_type = PeriDesign.BlockType.adv_pll
        elif self.plugin_map.is_plugin_exist(self.device_def, "tx60_pll"):
            self._supported_block.append(PeriDesign.BlockType.comp_pll)
            self.pll_type = PeriDesign.BlockType.comp_pll
        elif self.plugin_map.is_plugin_exist(self.device_def, "tx180_pll"):
            self._supported_block.append(PeriDesign.BlockType.efx_pll_v3_comp)
            self.pll_type = PeriDesign.BlockType.efx_pll_v3_comp
        elif self.plugin_map.is_plugin_exist(self.device_def, "tx375_pll"):
            self._supported_block.append(PeriDesign.BlockType.efx_fpll_v1)
            self.pll_type = PeriDesign.BlockType.efx_fpll_v1

        if self.plugin_map.is_plugin_exist(self.device_def, "an08_osc"):
            self._supported_block.append(PeriDesign.BlockType.osc)
        elif self.plugin_map.is_plugin_exist(self.device_def, "tx60_osc"):
            self._supported_block.append(PeriDesign.BlockType.adv_osc)
            self.osc_type = PeriDesign.BlockType.adv_osc
        elif self.plugin_map.is_plugin_exist(self.device_def, "tx60_hposc"):
            self._supported_block.append(PeriDesign.BlockType.hposc)
            self.osc_type = PeriDesign.BlockType.hposc

        if self.plugin_map.is_plugin_exist(self.device_def, "an20_lvds"):
            self._supported_block.append(PeriDesign.BlockType.lvds)
        elif self.plugin_map.is_plugin_exist(self.device_def, "tx60_lvds"):
            self._supported_block.append(PeriDesign.BlockType.adv_lvds)
            self.lvds_type = PeriDesign.BlockType.adv_lvds

        if self.plugin_map.is_plugin_exist(self.device_def, "an20_mipi"):
            self._supported_block.append(PeriDesign.BlockType.mipi)
        elif self.plugin_map.is_plugin_exist(self.device_def, "tx180_mipi"):
            self._supported_block.append(PeriDesign.BlockType.mipi_hard_dphy)
            self.mipi_type = PeriDesign.BlockType.mipi_hard_dphy

        if self.plugin_map.is_plugin_exist(self.device_def, "tx60_mipi_dphy"):
            self._supported_block.append(PeriDesign.BlockType.mipi_dphy)

        if self.plugin_map.is_plugin_exist(self.device_def, "an08_jtag"):
            self._supported_block.append(PeriDesign.BlockType.jtag)

        if self.plugin_map.is_plugin_exist(self.device_def, "an35_ddr"):
            self._supported_block.append(PeriDesign.BlockType.ddr)
        elif self.plugin_map.is_plugin_exist(self.device_def, "tx180_ddr"):
            self._supported_block.append(PeriDesign.BlockType.adv_ddr)
            self.ddr_type = PeriDesign.BlockType.adv_ddr

        if self.plugin_map.is_plugin_exist(self.device_def, "an35_h264"):
            self._supported_block.append(PeriDesign.BlockType.h264)

        if self.plugin_map.is_plugin_exist(self.device_def, "tx60_spi_flash"):
            self._supported_block.append(PeriDesign.BlockType.spi_flash)

        if self.plugin_map.is_plugin_exist(self.device_def, "tx60_hyper_ram"):
            self._supported_block.append(PeriDesign.BlockType.hyper_ram)

        if self.plugin_map.is_plugin_exist(self.device_def, "tx180_pll_ssc"):
            self._supported_block.append(PeriDesign.BlockType.pll_ssc)

        if self.plugin_map.is_plugin_exist(self.device_def, "tx375_soc"):
            self._supported_block.append(PeriDesign.BlockType.soc)

        if self.plugin_map.is_plugin_exist(self.device_def, "tx375_quad"):
            self._supported_block.append(PeriDesign.BlockType.quad)
            self._supported_block.append(PeriDesign.BlockType.common_quad)

        if self.plugin_map.is_plugin_exist(self.device_def, "tx375_quad_pcie"):
            self._supported_block.append(PeriDesign.BlockType.quad_pcie)

        if self.plugin_map.is_plugin_exist(self.device_def, "tx375_lane_10g"):
            self._supported_block.append(PeriDesign.BlockType.lane_10g)

        if self.plugin_map.is_plugin_exist(self.device_def, "tx375_raw_serdes"):
            self._supported_block.append(PeriDesign.BlockType.raw_serdes)

        if self.plugin_map.is_plugin_exist(self.device_def, "tx375_lane_1g"):
            self._supported_block.append(PeriDesign.BlockType.lane_1g)

    def is_block_supported(self, block_type: BlockType)->bool:
        """
        Check if a block type supported in the device

        :param block_type: A block type
        :return: True, if supported, else False
        """
        if self.device_def == "":
            return False

        return block_type in self._supported_block

    def is_device_block_tx_rx_supported(self, block_type: BlockType, is_tx: bool):
        """
        Check if the specific type of block which has tx,rx variant is supported
        based on device resource map (bonded out). We can't use is_block_supported
        as the type is based on the registry 'type'. This instead check if the device
        has both the tx and rx block definition. This is applicable when the device
        block definition is separated (ie LVDS TX and RX in Trion / MIPI). But if it
        is based on same block definition (ie HSIO for LVDS and MIPI Lane), then
        both types are always supported.

        :param block_type: A block type
        :return: True, if supported, else False
        """
        is_supported = False

        if self.is_block_supported(block_type):
            if block_type in [PeriDesign.BlockType.adv_lvds, PeriDesign.BlockType.mipi_dphy]:
                # These are types using  HSIO, so it always support both
                is_supported = True
            elif block_type in [PeriDesign.BlockType.lvds,
                                PeriDesign.BlockType.mipi,
                                PeriDesign.BlockType.mipi_hard_dphy] \
                                    and self.device_db is not None:
                block_ref_name = ""
                match block_type:
                    case PeriDesign.BlockType.lvds:
                        block_ref_name = "lvds_tx" if is_tx else "lvds_rx"
                    case PeriDesign.BlockType.mipi | PeriDesign.BlockType.mipi_hard_dphy:
                        block_ref_name = "mipi_tx" if is_tx else "mipi_rx"

                if block_ref_name != "":
                    if self.device_db.is_block_in_resource_map(block_ref_name):
                        is_supported = True

            else:
                # Assume the rest as supported since they're not tx/rx type or device db is None
                is_supported = True
        
        return is_supported

    def get_supported_block(self) -> List[PeriDesign.BlockType]:
        """
        Get a list of supported block type based on device used

        :return: A list of block type
        """
        return list(self._supported_block)

    def get_supported_block_name_list(self) -> List[str]:
        """
        Get a list of supported block type name based on device used

        :return: A list of block type name
        """

        block_list: List[str] = []
        for blk_type in self._supported_block:
            blk_name = self.blktype2str_map.get(blk_type, None)
            if blk_name is not None:
                block_list.append(blk_name)

        return block_list

    def get_supported_block_count(self):
        """
        Get supported block count

        :return: Count
        """
        return len(self._supported_block)

    def is_support_ddio(self):
        """
        Check if device supports ddio.
        For quick access, use the flag in gpio instance.

        :return: True, if yes, else False
        """
        # If lvds is supported, it means it is AN20 and gpio supports ddio
        # Also, there are some devices that don't have LVDS but support ddio
        return self.is_block_supported(PeriDesign.BlockType.lvds) or \
            self.is_block_supported(PeriDesign.BlockType.adv_lvds) or \
            self.device_def == "T20W80"

    def create_chksum(self):
        return hash(PeriDesign)

    def set_default_setting(self):
        # Not applicable
        pass

    def clear_design(self, block_types: List[str]):
        """
        Clear design by provided string, which should match with
        keys in str_blk_type.

        **Note**: block type can be from `DesignExpItem.ItemType` or API block type

        :param block_types: A list of block types
        :type block_types: List[str]
        """
        for str_blk_type in block_types:

            match str_blk_type:
                case "GPIO":
                    blk_type = self.gpio_type
                case "PLL":
                    blk_type = self.pll_type
                case "LVDS_TX" | "LVDS_RX" | "LVDS_BIDIR":
                    blk_type = self.lvds_type
                case "OSC":
                    blk_type = self.osc_type
                case "MIPI_TX" | "MIPI_RX" | "MIPI_HARD_DPHY_TX" | "MIPI_HARD_DPHY_RX":
                    blk_type = self.mipi_type
                case "MIPI_DPHY_TX" | "MIPI_DPHY_RX":
                    blk_type = self.BlockType.mipi_dphy
                case "DDR":
                    blk_type = self.ddr_type
                case "LANE_10G" | "10GBASE_KR":
                    blk_type = self.BlockType.lane_10g
                case "LANE_1G" | "SGMII":
                    blk_type = self.BlockType.lane_1g
                case "PMA_DIRECT" | "RAW_SERDES":
                    blk_type = self.BlockType.raw_serdes
                case _:
                    blk_type = self.str2blktype_map.get(str_blk_type, None)
                    assert isinstance(blk_type, self.BlockType), f"{str_blk_type} is not found"

            reg = self.get_block_reg(blk_type)

            if reg is None:
                continue

            if str_blk_type.endswith("TX"):
                reg.delete_all_tx_inst()
            elif str_blk_type.endswith("RX"):
                reg.delete_all_rx_inst()
            elif str_blk_type.endswith("BIDIR"):
                reg.delete_all_bidir_inst()
            else:
                reg.delete_all_insts()

    def create_lvds_gpio(self, name, apply_default=True, auto_pin=False):
        """
        Create lvds gpio instance.

        lvds gpio instance is stored in gpio registry with special marking.
        Caller can query created gpio instance to know whether it is
        a lvds gpio.

        :param name: Instance name
        :param apply_default: If True, apply default pad configuration
        :param auto_pin:

        :return: A new registered lvds gpio instance
        """
        if self.is_tesseract_design():
            msg = "PeriDesign::create_lvds_gpio - Calling Trion flow on Tesseract design"
            raise app_excp.PTException(msg, app_excp.MsgLevel.error)

        if self.gpio_reg is not None:
            gpio = self.gpio_reg.create_gpio(name, "", apply_default, auto_pin)
            if gpio is not None:
                gpio.mark_lvds_gpio()

            return gpio

        return None

    def delete_lvds_gpio(self, gpio_inst):
        """
        Delete lvds gpio instance.

        Clear lvds device usage in both lvds and gpio registry.

        :param gpio_inst: Instance name
        """

        if self.is_tesseract_design():
            msg = "PeriDesign::delete_lvds_gpio - Calling Trion flow on Tesseract design"
            raise app_excp.PTException(msg, app_excp.MsgLevel.error)

        if gpio_inst is None:
            return

        if self.lvds_reg is not None:
            from common_device.lvds.lvds_design import LVDSRegistry
            assert isinstance(self.lvds_reg, LVDSRegistry)

            self.lvds_reg.deregister_lvds_gpio(gpio_inst)

        if self.gpio_reg is not None:
            self.gpio_reg.delete_gpio(gpio_inst.name)

    def delete_any_gpio(self, gpio_inst):
        """
        Delete any type of gpio. The gpio can be a dedicated gpio or lvds gpio. A bus member or not.

        :param gpio_inst: Instance name

        ..notes:
          Refactored from gpio, bus_main_window.py
        """

        if self.is_tesseract_design():
            msg = "PeriDesign::delete_any_gpio - Calling Trion flow on Tesseract design"
            raise app_excp.PTException(msg, app_excp.MsgLevel.error)

        if gpio_inst is not None:
            assert self.gpio_reg is not None

            # Remove from the bus first if the name is part of bus
            if gpio_inst.bus_name != "":
                self.gpio_reg.remove_bus_member_by_name(gpio_inst.name)

            # Delete the instance
            if gpio_inst.is_lvds_gpio():
                self.delete_lvds_gpio(gpio_inst)

            else:
                self.gpio_reg.delete_gpio(gpio_inst.name)

    def assign_lvds_gpio_device(self, gpio_inst, lvds_device):
        """
        Assigned lvds_device to gpio instance.

        lvds device usage is registered in both lvds and gpio registry.

        :param gpio_inst: Instance name
        :param lvds_device: Device name
        """

        if self.is_tesseract_design():
            msg = "PeriDesign::assign_lvds_gpio_device - Calling Trion flow on Tesseract design"
            raise app_excp.PTException(msg, app_excp.MsgLevel.error)

        if gpio_inst is None:
            return

        if self.gpio_reg is not None:
            self.gpio_reg.assign_gpio_device(gpio_inst, lvds_device)

        if self.lvds_reg is not None:
            from common_device.lvds.lvds_design import LVDSRegistry
            assert isinstance(self.lvds_reg, LVDSRegistry)

            self.lvds_reg.register_lvds_gpio(gpio_inst)

    def reset_lvds_gpio_device(self, gpio_inst):
        """
        Reset lvds_device assigned to gpio instance.

        lvds device usage is registered in both lvds and gpio registry.

        :param gpio_inst: Instance name
        """
        if self.is_tesseract_design():
            msg = "PeriDesign::reset_lvds_gpio_device - Calling Trion flow on Tesseract design"
            raise app_excp.PTException(msg, app_excp.MsgLevel.error)

        if gpio_inst is None:
            return

        if self.lvds_reg is not None:
            from common_device.lvds.lvds_design import LVDSRegistry
            assert isinstance(self.lvds_reg, LVDSRegistry)

            self.lvds_reg.deregister_lvds_gpio(gpio_inst)

        if self.gpio_reg is not None:
            self.gpio_reg.assign_gpio_device(gpio_inst, "")

    def delete_gpio_bus(self, bus_name):
        '''
        PT-1028: Delete bus needs to have design when working
        on Trion devices. This is because design needs to manage
        the resource of LVDS GPIO.
        :param bus_name: The GPIO bus name to be deleted
        '''
        if self.gpio_reg is not None:
            gpio_bus = self.gpio_reg.get_bus_by_name(bus_name)

            if gpio_bus is not None:
                bus_member_list = gpio_bus.get_member_list()
                for member in bus_member_list:
                    if member.is_lvds_gpio():
                        self.delete_lvds_gpio(member)
                    else:
                        self.gpio_reg.delete_gpio(member.name)

                gpio_bus.clear()

                # Delete the entry from the gpio registry
                self.gpio_reg.delete_bus_in_map(bus_name)

    def get_instance_and_type(self, dev_ins):
        """
        Given the device instance name, check in the design database
        if it is configured by user.

        :param dev_ins: Device instance name to query
        :return a tuple of:
            1) the design instance object found or None (if not found)
            2) the BlockType
        """
        des_ins = None
        btype = self.BlockType.other

        # Iterate through all registry to find instance with
        # the matching device instance name
        if self.gpio_reg is not None:
            des_ins = self.gpio_reg.get_gpio_by_device_name(dev_ins)
            btype = self.gpio_type

        if des_ins is None:
            reg2blk_type: List[Tuple[Optional[db_item.PeriDesignRegistry], PeriDesign.BlockType]] = [
                (self.pll_reg, self.pll_type),
                (self.osc_reg, self.osc_type),
                (self.lvds_reg, self.lvds_type),
                (self.mipi_reg, self.BlockType.mipi),
                (self.mipi_dphy_reg, self.BlockType.mipi_dphy),
                (self.mipi_hard_dphy_reg, self.BlockType.mipi_hard_dphy),
                (self.pll_ssc_reg, self.BlockType.pll_ssc),
                (self.jtag_reg, self.BlockType.jtag),
                (self.ddr_reg, self.BlockType.ddr),
                (self.h264_reg, self.BlockType.h264),
                (self.spi_flash_reg, self.BlockType.spi_flash),
                (self.quad_pcie_reg, self.BlockType.quad_pcie),
                (self.lane_10g_reg, self.BlockType.lane_10g),
                (self.quad_reg, self.BlockType.quad),
                (self.raw_serdes_reg, self.BlockType.raw_serdes),
                (self.lane_1g_reg, self.BlockType.lane_1g),
            ]
            for reg, blk_type in reg2blk_type:
                if reg is not None:
                    des_ins = reg.get_inst_by_device_name(dev_ins)
                    btype = blk_type

                if des_ins is not None:
                    break

        if des_ins is None:
            btype = self.BlockType.other

        return des_ins, btype

    def get_lane_based_reg(self) -> List[Optional[LaneBaseRegistry]]:
        return [
            self.lane_10g_reg,
            self.raw_serdes_reg,
            self.lane_1g_reg,
        ]

    def is_serdes_reg_supported(self):
        """
        Check if any of the serdes registers are supported
        """
        serdes_reg_list = [
            self.quad_pcie_reg,
        ] + self.get_lane_based_reg()

        return any(serdes_reg_list)

    def find_partial_design(self, name: str):
        pd = self.__partial_designs.get(name)
        return pd

    def add_partial_design(self, name: str, pd: Dict):
        self.__partial_designs[name] = pd

    def delete_partial_design(self, name: str):
        assert name in self.__partial_designs
        # TODO: Remove instances assoicated
        del self.__partial_designs[name]

    def get_all_partial_designs(self) -> List[str]:
        return list(self.__partial_designs.keys())


class DesignIssue:
    """
    Store a single design issue
    """

    def __init__(self):
        self.issue_id = ""  #: Unique issue id
        self.instance_name = ""
        self.instance_type = "other"
        self.severity = common_rule.Rule.SeverityType.unknown
        #: Issue generated outside from checker can be identified through its rule name
        self.rule_name = "general"
        self.msg = ""

    def __str__(self, *args, **kwargs):
        info = 'id:{} name:{} type:{} sev:{} rule:{} msg:{}'.format(
            self.issue_id, self.instance_name, self.instance_type, self.severity, self.rule_name, self.msg)
        return info

    def get_all(self):
        return self.issue_id, self.instance_name, self.instance_type, self.severity, self.rule_name, self.msg


class DesignIssueRegistry:
    """
    Repository for design issues detected mostly by checker.
    """
    logger = Logger

    def __init__(self):
        self._issue = {}  #: A map of issue id and its details
        self._last_issue_id = 0  #: Running number to be used only in memory
        self._exclude_rules = []

    def clear(self):
        self._issue.clear()
        self._last_issue_id = 0
        self._exclude_rules = []

    @staticmethod
    def build():
        """
        Build empty repository

        :return:
        """
        registry = DesignIssueRegistry()
        return registry

    @staticmethod
    def build_from_file(result_file):
        """
        Build repository and load data from result file

        :param result_file:
        :return: Issue registry object
        """
        registry = DesignIssueRegistry.build()
        if os.path.exists(result_file) is False:
            msg = "Result file does not exists : " + result_file
            DesignIssueRegistry.logger.warning(msg)
        else:
            # TODO : Check for valid xml
            # app_setting = aps.AppSetting()
            # # pylint: disable=W0212
            # GPIODesignBuilderXml._schema_file = app_setting.app_path[
            #                                         aps.AppSetting.PathType.schema] + "/gpio_design.xsd"
            #
            # schema = xmlschema.XMLSchema(
            #     GPIODesignBuilderXmlEventBased._schema_file)
            #
            # try:
            #     schema.validate(design_file)
            # except xmlschema.XMLSchemaValidationError as excp:
            #     print("GPIO design contains error")
            #     self.logger.error("XML Element : {}".format(excp.elem))
            #     self.logger.error("Reason : {}".format(excp.reason))
            #     return False

            tree = et.parse(result_file)
            issue_info = tree.getroot()

            nspace = "{http://www.efinixinc.com/peri_design_db}"
            issue_tag = nspace + "issue"

            for child in issue_info:
                if child.tag == issue_tag:
                    attr = child.attrib
                    reg_issue = DesignIssue()
                    reg_issue.instance_name = attr.get("instance_name", "")
                    reg_issue.instance_type = attr.get("instance_type", "")
                    sev_name = attr.get("severity", "")
                    reg_issue.severity = common_rule.Rule.str2sev_map.get(
                        sev_name, common_rule.Rule.SeverityType.unknown)
                    reg_issue.rule_name = attr.get("rule_name", "")
                    reg_issue.msg = attr.get("message", "")
                    registry.append_issue(reg_issue)

        return registry

    def save_to_file(self, result_file):
        """
        Write repository data to file in xml format. This is the tool storage format.

        :param result_file: Target file to write to
        """
        root = et.Element("efxpt:issue_info", {
            "xmlns:efxpt": "http://www.efinixinc.com/peri_design_db"})
        for issue in self._issue.values():
            et.SubElement(root, "efxpt:issue",
                          instance_name="{}".format(issue.instance_name),
                          instance_type="{}".format(issue.instance_type),
                          severity="{}".format(
                              common_rule.Rule.sev2str_map.get(issue.severity, "")),
                          rule_name="{}".format(issue.rule_name),
                          message="{}".format(issue.msg))

        et.ElementTree(root).write(result_file, method="xml",
                                   encoding="UTF-8", xml_declaration=True)

    def write_to_csv_file(self, csv_file):
        """
        Write repository data to file in csv format. This is meant for user use.

        :param csv_file: Target file to write to
        """
        with open(csv_file, 'w', encoding='UTF-8') as csvfile:
            csv_writer = csv.writer(csvfile, delimiter=',')

            csv_writer.writerow(
                ["Instance Name", "Instance Type", "Severity", "Rule", "Description"])

            for issue in self._issue.values():
                csv_writer.writerow([issue.instance_name,
                                     issue.instance_type,
                                     common_rule.Rule.sev2str_map.get(
                                         issue.severity, ""),
                                     issue.rule_name,
                                     issue.msg
                                     ])

    def write_pretty_table_to_file(self, outfile):
        """
        Write issues to a file in a table format. This is meant to be used
        as part of the summary report.

        :param outfile: The file handle that has already been opened to
                        write to.
        """

        if self._issue:
            # Write the header
            issue_table = PrettyTable(
                ["Instance Name", "Instance Type", "Severity", "Rule", "Description"])

            def get_inst_name(issue_obj):
                return issue_obj.instance_name

            # Write the list of issues sorted based on instance name
            for issue in sorted(self._issue.values(), key=get_inst_name):
                row_list = []

                row_list.append(issue.instance_name)
                row_list.append(issue.instance_type)
                row_list.append(common_rule.Rule.sev2str_map.get(
                    issue.severity, ""))
                row_list.append(issue.rule_name)
                row_list.append(issue.msg)

                issue_table.add_row(row_list)

            outfile.write("{}\n".format(issue_table.get_string()))

    def _gen_id(self):
        """
        Generate unique issue id. It uses simple method for now.

        :return: New issue id
        """
        self._last_issue_id = self._last_issue_id + 1
        return self._last_issue_id

    def append_issue(self, design_issue):
        """
        Append design issue to this repostory

        :param design_issue: Design issue object
        """
        self._gen_id()
        design_issue.issue_id = self._last_issue_id
        self._issue[self._last_issue_id] = design_issue

    def append_issue_by_tuple(self, design_issue):
        issue = DesignIssue()
        issue.instance_name = design_issue[0]
        issue.instance_type = design_issue[1]
        issue.severity = design_issue[2]
        issue.rule_name = design_issue[3]
        issue.msg = design_issue[4]
        self.append_issue(issue)

    def append_exclude_rules(self, exclude_rules):
        # Remove redundancy if there are rule duplicates
        if self._exclude_rules:
            for rule_name in exclude_rules:
                if rule_name not in self._exclude_rules:
                    self._exclude_rules.append(rule_name)

        else:
            self._exclude_rules = exclude_rules

    def get_exclude_count(self):
        return len(self._exclude_rules)

    def get_all_issue(self):
        return list(self._issue.values())

    def get_issue_by_id(self, issue_id):
        """
        Find design issue by id

        :param issue_id: Unique issue id
        :return: Design issue, if any, else None
        """
        if issue_id in self._issue:
            return self._issue[issue_id]

        return None

    def get_issue_count(self):
        """
        Get total issue count

        :return: A count
        """
        return len(self._issue)

    def get_issue_count_by_sev(self, sev_type):
        """
        Count issue by severity.

        :param sev_type: Severity type from common_device.Rule
        :return: A count
        """
        count = 0
        for issue in self._issue.values():
            if issue.severity == sev_type:
                count = count + 1

        return count

    def get_all_sev_count(self):
        """
        Get count breakdown for each severity

        :return: A tuple of count (critical, error, warning, info, unknown)
        """
        crit_count = 0
        err_count = 0
        warn_count = 0
        info_count = 0
        unknown_count = 0

        for issue in self._issue.values():
            if issue.severity == common_rule.Rule.SeverityType.critical:
                crit_count = crit_count + 1
            elif issue.severity == common_rule.Rule.SeverityType.error:
                err_count = err_count + 1
            elif issue.severity == common_rule.Rule.SeverityType.warning:
                warn_count = warn_count + 1
            elif issue.severity == common_rule.Rule.SeverityType.info:
                info_count = info_count + 1
            elif issue.severity == common_rule.Rule.SeverityType.unknown:
                unknown_count = unknown_count + 1

        return crit_count, err_count, warn_count, info_count, unknown_count

    def get_issue_by_sev(self, sev_type):
        """
        Count issue by severity.

        :param sev_type: Severity type from common_device.Rule
        :return: A count
        """
        issue_list = []
        for issue in self._issue.values():
            if issue.severity == sev_type:
                issue_list.append(issue)

        return issue_list


if __name__ == "__main__":
    pass
