from __future__ import annotations

from typing import TYPE_CHECKING, Union, Optional, Tuple
from PyQt5.QtWidgets import QSpacerItem, QSizePolicy, QVBoxLayout

from common_gui.builder.base import PTWidget, PinEditorWidget, LabelWidget, \
                                    ChoiceParamWidget, TextParamWidget, \
                                    ExtraControlWidget, ParamWidget
from common_gui.builder.primitive_widget import PTWidgetBuilder
from common_gui.builder.smart_widget import GenericPinWidgetBuilderV2
from common_gui.spec_builder import SpecBuilder

if TYPE_CHECKING:
    from typing import Callable, List, Dict

    from PyQt5.QtWidgets import QHBoxLayout, QWidget, QComboBox, QLineEdit

    from design.db_item import GenericPin, HasGenPin, HasParamInfoGroup
    from common_device.property import PropertyMetaData

    FILTER_PROP_FUNC = Callable[[PropertyMetaData], List[PropertyMetaData.PropData]]
    FILTER_PIN_FUNC = Callable[[HasGenPin], List[GenericPin]]


def build_param_widgets(container: Union[QHBoxLayout, QHBoxLayout], param_defs: List[PropertyMetaData.PropData], is_vertical: bool = True) -> Dict[str, PTWidget]:
    """
    Create widgets based on parameter information

    created widgets will be added to the container,
    the order of widgets is based on the order of the list passed

    :returns: A dictionary for the parameter name to widget mapping
    :rtype: Dict[str, WidgetGroup]
    """
    spec_builder = SpecBuilder()
    widget_builder = PTWidgetBuilder()
    for idx, param_def in enumerate(param_defs):
        widget_spec = spec_builder.build_single_param_widget_spec_by_def(param_def)
        assert widget_spec is not None, param_def.name
        widget_spec.seqno = idx
        widget_spec.layout_type = widget_spec.LayoutType.vertical if is_vertical else widget_spec.LayoutType.horizontal
        widget_builder.add_widget_spec(widget_spec)
    return widget_builder.build(container, is_sort=True, is_sort_seqno=True, is_last_spacer=False)


def build_param_widgets_by_category(block_inst: HasParamInfoGroup, container: Union[QVBoxLayout, QHBoxLayout], category: str) -> Dict[str, PTWidget]:
    """
    Create widgets based on parameter information in the block for parameter belong to specific category

    created widgets will be added to the container,
    the order of widgets is based on the parameter name

    :returns: A dictionary for the parameter name to widget mapping
    :rtype: Dict[str, WidgetGroup]
    """
    spec_builder = SpecBuilder()
    widget_builder = PTWidgetBuilder()
    param_info = block_inst.param_info
    param_group = block_inst.param_group

    filter_func = lambda p,c =category: p.get_prop_by_category(c) # type: ignore
    assert filter_func is not None

    for idx, prop_info in enumerate(filter_func(param_info)):
        param = param_group.get_param_by_name(prop_info.name)
        assert param is not None
        widget_spec = spec_builder.build_single_param_widget_spec(param, param_info)
        assert widget_spec is not None
        widget_builder.add_widget_spec(widget_spec)

    return widget_builder.build(container, is_sort=True, is_sort_seqno=False, is_last_spacer=False)


def build_pin_widgets_by_category(block_inst: HasGenPin, container: Union[QHBoxLayout, QVBoxLayout], category: str) -> Dict[str, PinEditorWidget]:
    """
    Create widgets based on pin information in the block for pins belong to specific category

    created widgets will be added to the container
    the order of widgets is based on the pin type name

    :returns: A dictionary for the pin type name to pin widget mapping
    :rtype: Dict[str, PinWidgetGroup]
    """
    assert hasattr(block_inst, 'gen_pin')

    pin_builder = GenericPinWidgetBuilderV2()
    spec_builder = SpecBuilder()

    filter_func = lambda b, c=category: b.get_pin_by_class(c) # type: ignore

    assert filter_func is not None
    pins = filter_func(block_inst)
    pin_name_list = [p.type_name for p in pins]
    spec_builder.add_pin_widget_spec(block_inst, pin_name_list, pin_builder)

    return pin_builder.build(container, is_sort=True, is_last_spacer=False)

def build_pin_widgets(block_inst: HasGenPin, container: QHBoxLayout | QVBoxLayout, pin_type_names: List[str]) -> Dict[str, PTWidget]:
    """
    Create widgets based on pin information in the block for pins

    :param block_inst: _description_
    :type block_inst: HasGenPin
    :param container: _description_
    :type container: QHBoxLayout | QVBoxLayout
    :param pin_type_names: _description_
    :type pin_type_names: List[str]
    :return: A dictionary for the pin type name to pin widget mapping
    :rtype: Dict[str, PTWidget]
    """
    assert hasattr(block_inst, 'gen_pin')
    pin_builder = GenericPinWidgetBuilderV2()
    spec_builder = SpecBuilder()
    spec_builder.add_pin_widget_spec(block_inst, pin_type_names, pin_builder)
    return pin_builder.build(container, is_sort=False, is_last_spacer=False)


def remove_widgets_from_container(widget: QWidget):
    parent = widget.parent()
    assert parent is not None
    container = parent.layout()
    assert container is not None
    container.removeWidget(widget)
    widget.setParent(None)


def add_widget_to_container(widget: QWidget, container: QWidget, with_spacer: bool = True):
    layout = container.layout()
    assert layout is not None
    layout.addWidget(widget)
    if with_spacer:
        spacer = QSpacerItem(20, 5, QSizePolicy.Minimum, QSizePolicy.Fixed)
        layout.addItem(spacer)


def add_last_expanding_spacer(container: QWidget):
    layout = container.layout()
    assert layout is not None
    spacer = QSpacerItem(20, 40, QSizePolicy.Expanding, QSizePolicy.Expanding)
    layout.addItem(spacer)


def build_choice_pt_widget(parent: QWidget, data_widget: QComboBox, label_widget: Optional[LabelWidget] = None, add_spacer: bool = True) -> PTWidget:
    """
    Convert a existing QComboBox and LabelWidget into a ChoiceParamWidget
    and added to the parent widget's layout
    """
    container = QVBoxLayout()
    container.setContentsMargins(0, 0, 0, 0)
    editor_widget = ChoiceParamWidget(container=container, label_widget=label_widget, parent=parent, editor_widget=data_widget)
    add_widget_to_container(editor_widget, parent, add_spacer)
    return editor_widget


def build_text_pt_widget(parent: QWidget, data_widget: QLineEdit, label_widget: Optional[LabelWidget] = None, add_spacer: bool = True) -> PTWidget:
    """
    Convert a existing QLineEdit and LabelWidget into a TextParamWidget
    and added to the parent widget's layout
    """
    container = QVBoxLayout()
    container.setContentsMargins(0, 0, 0, 0)
    editor_widget = TextParamWidget(container=container, label_widget=label_widget, parent=parent, editor_widget=data_widget)
    add_widget_to_container(editor_widget, parent, add_spacer)
    return editor_widget

def build_pin_wgt_grp_from_qwidgets(pin_name: str, disp_name: str, parent: QWidget, data_widget: QLineEdit | QComboBox, label_widget = None, add_spacer: bool = True) -> Tuple[str, PTWidget]:
    """
    Convert an existing QLineEdit and LabelWidget into a PinEditorWidget
    and add to the parent widget's layout
    """
    container = parent.layout()
    pin_editor_widget = PinEditorWidget(parent, label_widget=label_widget, editor_widget=data_widget, ex_ctrl_widget=ExtraControlWidget())
    pin_editor_widget.set_label_text(disp_name)
    assert pin_editor_widget.get_editor_widget() == data_widget
    container.addWidget(pin_editor_widget)
    if add_spacer:
        spacer = QSpacerItem(20, 5, QSizePolicy.Minimum, QSizePolicy.Fixed)
        container.addItem(spacer)
    return pin_name, pin_editor_widget
