from __future__ import annotations

import pprint
import operator
from util.singleton_logger import Logger

import re
from dataclasses import dataclass
from abc import abstractmethod
from networkx import DiGraph
from networkx import to_dict_of_dicts
from networkx.algorithms import dfs_successors as find_graph_successors

from typing import TYPE_CHECKING, Callable, List, Set, Any, Optional, TypedDict, Tuple

from design.db_item import GenericParamGroup, GenericPin, GenericPinGroup, GenericParam
from common_device.property import PropertyMetaData, RangeValidator

if TYPE_CHECKING:
    from common_gui.base_graph_observer import AbstractGraphObserver

OnParamChangedFunc = Callable[['DependencyGraph', GenericParamGroup, str, str], bool]
OnPinChangedFunc = Callable[['DependencyGraph', GenericParamGroup, str, GenericPinGroup, str], bool]

class PinAttributeSet(TypedDict):
    is_available: bool
    is_editable: bool
    was_available: bool
    was_editable: bool
    data: GenericPin

class ParamAttributeSet(TypedDict):
    is_available: bool
    is_editable: bool
    was_available: bool
    was_editable: bool
    data: PropertyMetaData.PropData

@dataclass
class Condition:
    param_name: str
    operator_str: str
    val: int| float | str


class GraphPublisher:
    """
    Base Implementation to publish event to the GraphObserver
    """

    def __init__(self) -> None:
        self._graph_observers: List[AbstractGraphObserver] = []

    def register_observer(self, observer: AbstractGraphObserver):
        self._graph_observers.append(observer)

    def unregister_observer(self, observer: AbstractGraphObserver):
        self._graph_observers.remove(observer)

    def notify(self, graph: DependencyGraph, updated_nodes: List[str]):
        for observer in self._graph_observers:
            observer.on_graph_updated(graph=graph, updated_nodes=updated_nodes)

def get_condition_operators(str_op: str) -> Callable[[Any, Any], bool]:
    """
    Get condition related operators

    :param str_op: string operators
    :type str_op: str
    :return: operator function
    :rtype: Callable[[Any, Any], bool]
    """
    op_map = {
        "==": operator.eq,
        "!=": operator.ne,
        ">=": operator.ge,
        ">": operator.gt,
        "<=": operator.le,
        "<": operator.lt,
    }
    func = op_map.get(str_op, None)
    assert func is not None, f"{str_op} is not supported"
    return func

def convert_condition_str2bool_str(param_group: GenericParamGroup, condition_str: str, graph: DependencyGraph) -> str:
    """
    Convert condition str to bool str by checking param value

    If the param name ends with '_en' and value is not given,
    the condition value will be set to `True` by default

    E.g.
    condition_str:  MODE != "Endpoint" || APB_EN
    param group:    MODE = "Endpoint", APB_EN = True

    Return
    bool_condition: "False or True"
    """
    bool_conditions = condition_str

    name_pattern = r'([0-9a-zA-Z][0-9a-zA-Z_]+)'
    operator_pattern = r'(==|>=|<=|!=|<|>|=)'
    text_or_numbers = r'?:"([^"]+)"|\b(\d+(\.\d+)?)'
    enable_pattern = r'[0-9a-zA-Z_]+[0-9]en'

    condition_pattern = fr'({name_pattern}\s*({operator_pattern}\s*({text_or_numbers}))?)'

    results: List = re.findall(condition_pattern, condition_str)

    def _get_condition_by_result(result: List[str]):
        condition_str = result[0]
        param_name = result[1]
        operator_str = result[3]
        text = result[4]
        num = result[5]

        # Numbers
        if num.isdigit():
            val = int(num)
        elif num.isnumeric():
            val = float(num)

        # Enable related, set to True by default
        elif operator_str == "" and (
            param_name.lower().endswith(("_en", "_enable")) or \
            re.match(enable_pattern, param_name.lower())):
                operator_str = "=="
                val = True
        else:
            val = text.strip()

        if operator_str == "=":
            operator_str = "=="

        assert operator_str != "", f"Condition {condition_str} do not have operator for param_name: {param_name} "

        condition_obj = Condition(
                        param_name=param_name,
                        operator_str=operator_str,
                        val=val
                    )
        return condition_str, condition_obj

    if results:
        for result in results:
            condition_str, condition_obj = _get_condition_by_result(result)
            is_match = graph.is_match_condition(param_group, condition_obj)
            bool_conditions = bool_conditions.replace(condition_str, "True" if is_match else "False")

    # Convert operator to python one
    bool_conditions = bool_conditions.replace("&&", " and ")
    bool_conditions = bool_conditions.replace("||", " or ")

    return bool_conditions


class DependencyGraph(GraphPublisher):
    """
    Storing the parameters and pin dependency in a graph
    """
    is_mock_port = False # Used in unit test
    def __init__(self, param_info: PropertyMetaData, port_info: GenericPinGroup, is_build_dep=True) -> None:
        super().__init__()
        self._graph = DiGraph()
        self._graph = self.build_graph(param_info, port_info)
        self._param_info = param_info
        self.is_test = False

        if is_build_dep:
            self.build_dependency(param_info, port_info)
        self.logger = Logger

    def add_param_to_port_dependency(self, param_name: str, port_name: str, change_func: OnPinChangedFunc):
        graph = self._graph
        assert graph is not None
        node_name_parent = f'DEP_PARAM_{param_name}'
        node_name_child = f'DEP_PORT_{port_name}'
        assert node_name_parent in graph.nodes

        if self.is_mock_port and node_name_child not in graph.nodes:
            print(f"{port_name} not built")
            return
        assert node_name_child in graph.nodes, f"{port_name} not built"

        # Port depends on Param
        graph.add_edge(node_name_parent, node_name_child, change_func=change_func)

    def add_param_dependency(self, param_a_name: str, param_b_name: str, change_func: OnParamChangedFunc):
        graph = self._graph
        assert graph is not None
        node_name_parent = f'DEP_PARAM_{param_a_name}'
        node_name_child = f'DEP_PARAM_{param_b_name}'
        assert node_name_parent in graph.nodes, node_name_parent
        assert node_name_child in graph.nodes, node_name_child
        # B depend on A
        graph.add_edge(node_name_parent, node_name_child, change_func=change_func)

    def add_param(self, prop_info: PropertyMetaData.PropData, **attr):
        graph = self._graph
        assert graph is not None
        assert prop_info is not None
        assert isinstance(prop_info, PropertyMetaData.PropData)
        node_name = f'DEP_PARAM_{prop_info.name}'
        graph.add_node(node_name, attr) # type: ignore

    def add_port(self, pin: GenericPin, **attr):
        graph = self._graph
        assert graph is not None
        assert pin is not None
        assert isinstance(pin, GenericPin)
        node_name = f'DEP_PORT_{pin.type_name}'
        graph.add_node(node_name, attr) # type: ignore

    def get_pin_attributes(self, pin_type_name: str) -> PinAttributeSet:
        graph = self._graph
        assert graph is not None
        node_name = f'DEP_PORT_{pin_type_name}'
        assert node_name in graph.nodes, f"{node_name} is not in graph nodes"
        return graph.nodes[node_name]

    def get_param_attributes(self, param_name: str) -> ParamAttributeSet:
        graph = self._graph
        assert graph is not None
        node_name = f'DEP_PARAM_{param_name}'
        assert node_name in graph.nodes, f"{node_name} is not in graph nodes"
        return graph.nodes[node_name]

    def is_param_available(self, param_name: str) -> bool:
        is_available = self.get_param_attributes(param_name)['is_available']
        return is_available

    def set_pin_attributes(self, pin_type_name: str, is_available: bool, is_editable: bool):
        graph = self._graph
        assert graph is not None
        node_name = f'DEP_PORT_{pin_type_name}'
        assert node_name in graph.nodes
        graph.nodes[node_name]['was_available'] = graph.nodes[node_name].get('is_available', False)
        graph.nodes[node_name]['was_editable'] = graph.nodes[node_name].get('is_editable', False)
        graph.nodes[node_name]['is_available'] = is_available
        graph.nodes[node_name]['is_editable'] = is_editable

    def set_param_attributes(self, param_name: str, is_available: bool, is_editable: bool):
        graph = self._graph
        assert graph is not None
        node_name = f'DEP_PARAM_{param_name}'
        assert node_name in graph.nodes
        graph.nodes[node_name]['is_available'] = is_available
        graph.nodes[node_name]['is_editable'] = is_editable

    @staticmethod
    def get_id_from_node_name(node_name: str) -> str:
        result = node_name.replace('DEP_PARAM_', '')
        result = result.replace('DEP_PORT_', '')
        return result

    def propgrate_param(self, param_group: GenericParamGroup, param_name: str) -> List[str]:
        # Logger.debug(f"IN propgrate_param, param_name = {param_name}, value = {param_group.get_param_value(param_name)}")
        graph = self._graph
        assert graph is not None

        updated_nodes: Set[str] = set()
        node_name = f'DEP_PARAM_{param_name}'
        assert node_name in graph.nodes
        updated_nodes.add(param_name)
        dfs_successors = find_graph_successors(graph, node_name)
        for node, edges in dfs_successors.items():
            for edge in edges:
                change_func = graph[node][edge]['change_func']
                # Invoke the change func to update the node's attribute
                parent_id = self.get_id_from_node_name(node)
                child_id = self.get_id_from_node_name(edge)
                change_func(self, param_group, parent_id, child_id)
                updated_nodes.add(child_id)

        self.notify(self, list(updated_nodes))
        # Return the updated parameter name / pin type name
        # Logger.debug(f"OUT propgrate_param, param_name = {param_name}, updated_nodes = {updated_nodes}")
        return list(updated_nodes)

    def debug(self):
        print("-----------------------------------------------------------------")
        pprint.pprint(to_dict_of_dicts(self._graph))
        print("-----------------------------------------------------------------")

    @abstractmethod
    def build_graph(self, param_info: PropertyMetaData, port_info: GenericPinGroup) -> DiGraph:
        raise NotImplementedError

    @abstractmethod
    def build_dependency(self, param_info: PropertyMetaData, port_info: GenericPinGroup):
        raise NotImplementedError

    def get_all_pin_nodes(self) -> List[str]:
        graph = self._graph
        pin_nodes = list(filter(lambda name: name.startswith('DEP_PORT_'), graph.nodes))
        return [DependencyGraph.get_id_from_node_name(n) for n in pin_nodes]

    def get_all_param_nodes(self) -> List[str]:
        graph = self._graph
        param_nodes = list(filter(lambda name: name.startswith('DEP_PARAM_'), graph.nodes))
        return [DependencyGraph.get_id_from_node_name(n) for n in param_nodes]

    def on_parent_param_changed(self, graph: DependencyGraph, param_group: GenericParamGroup, parent_param_name: str, my_param_name: str,
                                condition_str: str):
        """
        When match condition(s), enable child param.

        E.g. condition_str: MODE != "Endpoint" || APB_EN
            - Check if param 'MODE' is not equal to "Endpoint", or
            - Check if APB_EN is set to `True`
        """
        bool_conditions = convert_condition_str2bool_str(param_group, condition_str, graph)
        try:
            is_enable = eval(bool_conditions)
            self.set_param_attributes(my_param_name, is_available=is_enable, is_editable=is_enable)
            return True

        except SyntaxError as exp:
            self.logger.error(f"Error when convert condition string {condition_str}"\
                f" to bool_condition ({bool_conditions})")
            raise exp

    def on_enable_changed_pin(self, graph: DependencyGraph, param_group: GenericParamGroup, parent_param_name: str, pin_type_name: str,
                            condition_str: Optional[str] = None) -> bool:
        """
        Once enabled parent param, enable pin
        """
        if condition_str is None:
            param: Optional[GenericParam] = param_group.get_param_by_name(parent_param_name)
            assert param is not None
            assert param.value_type == GenericParam.DataType.dbool
            is_enable = param.value
        else:
            bool_conditions = convert_condition_str2bool_str(param_group, condition_str, graph)
            is_enable = eval(bool_conditions)

        self.set_pin_attributes(pin_type_name, is_available=is_enable, is_editable=is_enable)
        return True

    def is_match_condition(self, param_group: GenericParamGroup, condition: Condition):
        # Validate condition value
        if self.is_test:
            assert self.is_condition_with_valid_value(self._param_info, condition) is True

        cur_parent_val = param_group.get_param_value(condition.param_name)
        func = get_condition_operators(condition.operator_str)
        return func(cur_parent_val, condition.val)

    def is_condition_with_valid_value(self, param_info: PropertyMetaData, condition: Condition):
        param = param_info.get_prop_info_by_name(condition.param_name)
        assert param is not None, f"Error when getting parameter info {condition.param_name}"

        data_type = param.data_type
        valid_setting = param.valid_setting
        val = condition.val

        # Check data type of condition
        match data_type:
            case GenericParam.DataType.dstr | GenericParam.DataType.dhex:
                assert isinstance(val, str), \
                    f"Invalid type in condition value (param: {param.name} val: {val})"
            case GenericParam.DataType.dint:
                assert isinstance(val, int), \
                    f"Invalid type in condition value (param: {param.name} val: {val})"
            case GenericParam.DataType.dflo:
                assert isinstance(val, (int, float)), \
                    f"Invalid type in condition value (param: {param.name} val: {val})"
            case GenericParam.DataType.dbool:
                assert isinstance(val, bool) or val in (0, 1), \
                    f"Invalid type in condition value (param: {param.name} val: {val})"
            case _:
                raise ValueError(f"Unsupported data type {data_type} for condition")

        # Check options
        if valid_setting is None:
            return True

        if isinstance(valid_setting, RangeValidator):
            is_valid, msg = valid_setting.is_valid(val)
            assert is_valid, msg
        elif isinstance(valid_setting, list):
            assert val in valid_setting, \
                f"Invalid value is set in condition {val}, valid setting: {valid_setting}"

        return True
