'''
Copyright (C) 2017-2018 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Aug 6, 2018

@author: maryam
'''
from __future__ import annotations
import abc
import enum
from functools import partial
import xml.etree.ElementTree as et
import re
from typing import Any, List, Dict, Protocol, Set, TYPE_CHECKING, Literal, Optional, TextIO, runtime_checkable
import logging
from dataclasses import dataclass
from decimal import Decimal, ROUND_HALF_UP

import util.excp as app_excp
from util.singleton_logger import Logger
import util.gen_util as gutil

import device.excp as dev_excp

import writer.arc_delay_table as writer_arc
import writer.used_core_port as ucp

if TYPE_CHECKING:
    from design.db import PeriDesign
    from design.db_item import PeriDesignItem
    from device.db import PeripheryDevice
    from device.clocks import GlobalClocks
    from prettytable import PrettyTable
    from device.timing_model import Model
    from device.block_definition import TimingArc


class Writer(metaclass=abc.ABCMeta):
    '''
    Base class for all writer types
    '''

    def __init__(self, device_db: PeripheryDevice, name):
        '''
        constructor
        '''

        self._name = name

        self._device_db = device_db

        self.logger = Logger

    def get_name(self):
        '''
        :return the block name
        '''
        return self._name

    def get_device(self):
        '''
        Return the device

        :return device_db
        '''
        return self._device_db


@runtime_checkable
class DesignToPCRValueConverter(Protocol):
    """
    Convert value from design space to PCR space
    """

    def __call__(self, design_obj: PeriDesignItem, design_db: PeriDesign) -> str:
        ...


class LogicalPeriphery(Writer):
    '''
    Base class for LPF writer. Some of the functions
    have the same implementation regardless of the block
    type
    '''

    def __init__(self, device_db, name):
        '''
        Constructor
        '''

        super().__init__(device_db, name)

        # Indicate the pcr sequence group name that we will print,
        # if relevant
        self._seq_group_name_to_print = ""

    def set_seq_group_name(self, seq_group_name):
        self._seq_group_name_to_print = seq_group_name

    def generate_pcr_definition(self, root):
        '''
        Generate the PCR definition of an oscillator as a
        subelement of the passed root.

        :param root: ElementTree root that should contain the
                    Block PCR definition
        '''
        peri_block = self.get_device().find_block(self._name)
        if peri_block is None:
            msg = 'Unable to find block {}'.format(
                self._name)
            raise dev_excp.BlockDoesNotExistException(
                msg, app_excp.MsgLevel.error)

        pcr_block = peri_block.get_block_pcr()
        if pcr_block is None:
            msg = 'Block {} does not have PCR defined'.format(
                self._name)
            raise dev_excp.BlockPCRDoesNotExistException(
                msg, app_excp.MsgLevel.error)

        # If the block has multiple sequence group, then we print all of
        # them per each sequence group
        seq_group_names = pcr_block.get_sequence_group_names()

        if not seq_group_names:
            self._generate_pcr_blk_defn(root, pcr_block)

        else:
            self._generate_pcr_group_seq_defn(root, pcr_block, seq_group_names)

    def _generate_pcr_blk_defn(self, root, pcr_block):
        '''
        Generates the PCR block sequence + definition if the block
        does not have sequence group
        :param root:
        :param pcr_block:
        :return:
        '''
        bits_len = str(pcr_block.get_bits_len())
        param_len = str(pcr_block.get_param_size())

        block_def = et.SubElement(root, "efxpt:block_def",
                                  bitsize=bits_len,
                                  name=self._name,
                                  paramsize=param_len)

        # Prints out the block pcr sequence
        pcr_block.generate_sequence(block_def)

        # Prints out the parameter definition
        pcr_block.generate_parameters(block_def)

    def _generate_pcr_group_seq_defn(self, root, pcr_block, seq_group_names):

        for grp_name in sorted(seq_group_names):
            bits_len = str(pcr_block.get_bits_len(grp_name))
            param_len = str(pcr_block.get_param_size(grp_name))

            blk_def_name = "{}:{}".format(self._name, grp_name)

            block_def = et.SubElement(root, "efxpt:block_def",
                                      bitsize=bits_len,
                                      name=blk_def_name,
                                      paramsize=param_len)

            # Prints out the block pcr sequence
            pcr_block.generate_sequence(block_def, grp_name)

            # Prints out the parameter definition
            pcr_block.generate_parameters(block_def, grp_name)

    def generate_pcr_user_parent_instance(self, parent_ins, parent,
                                          design, block_def,
                                          child_ins):
        '''
        Generate the PCR setting of a user configured instance.

        :param parent_ins: The device parent instance name without any
                        extension.
        :param parent: ElementTree that should contain the
                        parameter setting
        :param design: Design database
        :param block_def: The reference BlockPCR definition of the
                    instance
        :param child_ins: List of child configured user design instance
        '''
        # Only selective block will have a real implementation. If
        # the block doesn't support it, it should follow
        # generate_pcr_user_instance. So those who does will override
        # with their own implementation
        if child_ins and len(child_ins) == 1:
            self.generate_pcr_user_instance(parent_ins, parent,
                                            design, block_def, child_ins[0])
        else:
            raise ValueError(
                "Unexpected error in LPF generation. Expected {} with single mode support".format(
                    block_def.get_name()))

    @abc.abstractmethod
    def generate_pcr_user_instance(self, blk_ins, parent,
                                   design, block_def,
                                   user_ins):
        '''
        Generate the PCR setting of a user configured instance.

        :param blk_ins: The device instance name. This should be
                        the same as the osc_obj.osc_def. It could also be
                        a combination of the device instance name and the
                        pcr sequence group name: DDR_0:DDR_CFG
        :param parent: ElementTree that should contain the
                        parameter setting
        :param design: Design database
        :param block_def: The reference BlockPCR definition of the
                    instance
        :param user_ins: Configured user design instance name
        '''
        pass

    @abc.abstractmethod
    def generate_default_pcr(self, blk_ins, parent, design,
                             block_def):
        '''
        Generate the PCR setting for a device instance
        that is not configured by user (default settings).

        :param blk_ins: The device instance name. This should be
                        the same as the osc_obj.osc_def
        :param parent: ElementTree that should contain the
                        parameter setting
        :param design: Design database
        :param block_def: The reference BlockPCR definition of the
                    instance
        '''
        pass


class InterfaceConfig(Writer):
    '''
    Base class for interface config writer. Some of the functions
    have the same implementation regardless of the block
    type
    '''

    @abc.abstractmethod
    def get_configured_interface(self, design, active_instances):
        '''
        Generate the interface configuration data of
        used gpio.

        :param design: design db
        :param active_instances: List of device available
                            resource instance names
        :return list of the UsedCorePort object
        '''
        pass

    def _save_inf_pin(self, ins_name, block_def, inf_config, pin_conn, inf_name,
                      clk_name, is_clk_inverted, is_required=False, tied_option=None):

        if pin_conn is not None:
            primary_conn = pin_conn.get_coord()

            if primary_conn is not None:
                x_loc = primary_conn.get_x()
                y_loc = primary_conn.get_y()
                z_loc = primary_conn.get_z()

                # If there is no specified user pin name, then
                # it is ok
                if inf_name != '' or clk_name != '':
                    # TODO: If the clock supports inversion, then
                    # this has to be updated
                    port_config = ucp.UsedCorePort(
                        pin_conn.get_type(), x_loc, y_loc,
                        z_loc, clk_name, is_clk_inverted, inf_name, block_def,
                        ins_name, self.get_name(),
                        is_required, tied_option)

                    inf_config.append(port_config)


class Summary(Writer):
    '''
    Base class for summary report writer. Some of the functions
    have the same implementation regardless of the block
    type
    '''

    def generate_global_summary(self, design: PeriDesign, db_global: GlobalClocks,
                                ins2func_map: Dict[str, Set[str]], ins_to_block_map: Dict[str, str],
                                table: PrettyTable):
        '''
        Writes out the details on the global signals.

        :param db_global: global signal
        :param ins2func_map: Map of instance name to function name
        :param ins_to_block_map: A map of instance name to the
                        referenced block name
        :param table: PrettyTable object
        '''
        # Nothing to print since the clocks are tied to the periphery
        # instances
        pass

    @abc.abstractmethod
    def generate_summary(self, design, outfile,
                         ins_to_blk_map, blk_service):
        '''
        Writes out osc related summary

        :param outfile: file handle that has already been opened
        :param ins_to_blk_map: A map of instance to reference block name
        '''
        pass


@dataclass
class SetDelayCommand:
    command_name: str
    clock_name: str
    pin_list: List[str]
    delay: Decimal
    delay_type: Literal['max', 'min', 'typical'] = 'typical'
    edge_type: Literal['rising', 'falling'] = 'rising'
    reference_pin_name: Optional[str] = None

    def build(self) -> str:
        """
        Build the SDC command in string
        """
        tokens = [
            self.command_name
        ]
        tokens += [
            '-clock', self.clock_name
        ]
        if self.edge_type == 'falling':
            tokens.append('-clock_fall')

        if self.reference_pin_name is not None:
            tokens += [
                '-reference_pin',
                f'[get_ports {{{self.reference_pin_name}}}]'
            ]

        match self.delay_type:
            case 'max':
                tokens.append('-max')
            case 'min':
                tokens.append('-min')

        tokens.append(
            str(self.delay.quantize(Decimal('0.000'), ROUND_HALF_UP))
        )

        tokens.append(
            f"[get_ports {{{' '.join(self.pin_list)}}}]"
        )
        return " ".join(tokens)


@dataclass(init=False)
class SetOutputDelayCommand(SetDelayCommand):
    """
    Store values used to construct a set_output_delay SDC command
    """

    def __init__(self,
                 clock_name: str,
                 pin_list: List[str],
                 delay: Decimal,
                 delay_type: Literal['max', 'min', 'typical'] = 'typical',
                 edge_type: Literal['rising', 'falling'] = 'rising',
                 reference_pin_name: str | None = None):
        super().__init__(command_name='set_output_delay',
                         clock_name=clock_name,
                         pin_list=pin_list,
                         delay=delay,
                         delay_type=delay_type,
                         edge_type=edge_type,
                         reference_pin_name=reference_pin_name)


@dataclass(init=False)
class SetInputDelayCommand(SetDelayCommand):
    """
    Store values used to construct a set_input_delay SDC command
    """

    def __init__(self,
                 clock_name: str,
                 pin_list: List[str],
                 delay: Decimal,
                 delay_type: Literal['max', 'min', 'typical'] = 'typical',
                 edge_type: Literal['rising', 'falling'] = 'rising',
                 reference_pin_name: str | None = None):
        super().__init__(command_name='set_input_delay',
                         clock_name=clock_name,
                         pin_list=pin_list,
                         delay=delay,
                         delay_type=delay_type,
                         edge_type=edge_type,
                         reference_pin_name=reference_pin_name)


@dataclass(init=False)
class SetCUCommand:
    """
    Store values used to construct a set_clock_uncertainty SDC command
    """
    command_name: str
    clock_name: str
    mode: str
    value: float

    def __init__(self,
                 clock_name: str,
                 mode: str,
                 value: float):
        self.command_name = "set_clock_uncertainty"
        self.clock_name = clock_name
        self.mode = mode
        self.value = value

    def build(self) -> str:
        """
        Build the SDC command in string
        """
        tokens = [
            self.command_name
        ]
        tokens += [
            '-to', self.clock_name
        ]

        if self.mode != "":
            if self.mode == "setup" or self.mode == "hold":
                tokens.append(f'-{self.mode}')
        tokens.append(str(self.value))

        return " ".join(tokens)


def write_cmds(stream: TextIO, cmds: List[SetDelayCommand| SetCUCommand]):
    """
    Helper function to write sdc cmds to io stream (file)
    """
    stream.writelines([c.build() + "\n" for c in cmds])


class BlockTiming(metaclass=abc.ABCMeta):
    """
    Writes the timing information for specific block
    """
    class DelayType(enum.Enum):
        '''
        Indicates if it was a min or max value
        '''
        max = 0
        min = 1

    def __init__(self, bname: str, device: PeripheryDevice, design: PeriDesign, report, sdc):
        '''
        Constructor
        '''
        self.logger = Logger

        self._name = bname
        self._device = device
        self._design = design
        self._report_file = report
        self._sdc_file = sdc

    def get_clkout_info(self, ins_name, dev_ins_obj,
                        clk_pin_name, user_clk):
        '''

        :param ins_name: The design object instance name
        :param dev_ins_obj: The device instance object
        :param clk_pin_name: The device instance pin name that were looking for
        :param user_clk: The user specific pin name associated to the clock pin
        :return:
            ref_pin_name: The expected pin name to write out to the argument
            clkout_str: The string to write out to sdc file
        '''
        clkout_str = ""
        ref_pin_name = ""

        if dev_ins_obj is not None:

            ref_pin_name, _ = self._get_instance_ref_pin_clkout_pin_name(
                dev_ins_obj, clk_pin_name, user_clk)

            if ref_pin_name != "":
                clkout_str = "{} -clock {} -reference_pin [get_ports {{{}}}]".format(
                    ins_name, user_clk, ref_pin_name)

        return ref_pin_name, clkout_str

    def get_clkout_info_with_coord(self, ins_name, dev_ins_obj,
                        clk_pin_name, user_clk):
        '''

        :param ins_name: The design object instance name
        :param dev_ins_obj: The device instance object
        :param clk_pin_name: The device instance pin name that were looking for
        :param user_clk: The user specific pin name associated to the clock pin
        :return:
            ref_pin_name: The expected pin name to write out to the argument
            clkout_str: The string to write out to sdc file
            clkout pin coordinate: Tuple (x,y,z)
        '''
        clkout_str = ""
        ref_pin_name = ""
        clkout_coord_tup = None

        if dev_ins_obj is not None:

            ref_pin_name, clkout_coord_tup = self._get_instance_ref_pin_clkout_pin_name(
                dev_ins_obj, clk_pin_name, user_clk)

            if ref_pin_name != "":
                clkout_str = "{} -clock {} -reference_pin [get_ports {{{}}}]".format(
                    ins_name, user_clk, ref_pin_name)

        return ref_pin_name, clkout_str, clkout_coord_tup

    def _get_device_instance_clkout_pin(self, ins_obj, clkout_pin_name):
        '''
        Placing this as standalone function so that it can be customized for specific block type
        that does not have direct pins but embedded (i.e.  HyperRAM, SPIFlash)
        :param ins_obj: Device instance object
        :param clkout_pin_name: The clkout pin name we're looking for
        :return: the Instance Pin Object associated to the clkout pin
        '''
        return ins_obj.get_connection_pin(clkout_pin_name)

    def _get_instance_ref_pin_clkout_pin_name(self, ins_obj, clkout_pin_name, user_pin_name):
        '''
        Given the device instance name, get the instance object. Then find
        the pin object based on the specified pin name.

        Then based on the specified pin, return the expected core pin name

        :param ins_obj: Device instance object
        :param clkout_pin_name: device instance pin name
        :param user_pin_name: Specified user pin name connected to the clkout_pin_name
        :return: 
            1) The reference pin name
            2) clkout coordinate (Tuple of x,y,z)
        '''
        ref_pin_name = ""
        clkout_pin_coord = None

        pin_obj = self._get_device_instance_clkout_pin(
            ins_obj, clkout_pin_name)

        if pin_obj is not None:

            if pin_obj.get_type() == pin_obj.InterfaceType.clkout:
                pin_coord = pin_obj.get_coord()

                if pin_coord is not None:
                    x = pin_coord.get_x()
                    y = pin_coord.get_y()
                    z = pin_coord.get_z()

                    clkout_pin_coord = (x,y,z)

                    # Device width and height
                    width = self._device.get_width()
                    height = self._device.get_height()

                    # print("pin: w,h: {},{} = {},{},{}".format(
                    #    width, height, x, y, z))
                    # Do this only if the device has valid height and width
                    if width > 0 and height > 0:

                        # PT-1518: This function is used for determining the generated
                        # core generated clkout pin name which is depending on
                        # the core EFTIO architecture.
                        # The correct way to do it can be observed in vpr/CoreInterface.cpp
                        # function GetInterfaceLocation/ProcessConstraint
                        # For now, we hardcode it since this is only for clkout which will either be:
                        # -1 or 6: offset by 1
                        # 14: offset by 2
                        ref_x = -1
                        ref_y = -1

                        if z != 14:
                            offset = 1
                        else:
                            offset = 2

                        # Based on the x,y and also the device width and height,
                        # the we determine the actual value to use in the string
                        if x == 0:
                            # This is at the left
                            ref_y = y
                            ref_x = x + offset
                        elif y == 0:
                            # This is at the bottom
                            ref_y = y + offset
                            ref_x = x
                        elif x == width:
                            # This is at the right
                            ref_y = y
                            ref_x = x - offset
                        elif y == height:
                            # This is at the top
                            ref_y = y - offset
                            ref_x = x
                        else:
                            # This is not at the edge and shouldn't happen
                            self.logger.debug('Unexpected instance {} with clkout pin not '
                                              'on the edge {} with coord {},{}'.format(
                                                  ins_obj.get_name(), clkout_pin_name, x, y))

                        if ref_x != -1 and ref_y != -1:
                            ref_pin_name = "{}~CLKOUT~{}~{}".format(
                                user_pin_name, ref_x, ref_y)

        return ref_pin_name, clkout_pin_coord

    @abc.abstractmethod
    def write(self, index, ins_to_block_map, clk2delay_map):
        """
        Write the timing report file and the sdc constraint
        for the specific block.

        :param index: The index of this section. Used in printing to report
        :param ins_to_block_map: Map of device instance name to the block name

        :return True if there was any valid instance found
        """
        return False

    # Special write where it takes in additional argument
    # to resolve any other delay calculation. May not be
    # applicable to all blocks
    def write_with_delay(self, index, ins_to_block_map, 
                         core_delay_obj, design_input_to_loc_map,
                         blktype2alt_delay_map, blktype2clkout_delay_map):
        '''
        :return a tuple of 2 items:
            1) True if there was any valid instance found
            2) None or a list of tuple that contains latency info
                clock_latency_sdc_list: List[Tuple[str, float, str]]
                (applicable only to PLL)
            3) A list of warning messages to be printed out
        '''
        return False, None, []

class BlockTimingWithArcs(BlockTiming):
    """
    Derived class for writing timing data of blocks
    that have timing arcs
    """

    xml_ns = "{http://www.efinixinc.com/peri_device_db}"

    # Define empty map. Only specific block needs it
    str2var_map: Dict = {}

    def __init__(self, bname, device, design, report, sdc,
                 max_model: Model, min_model: Model):
        '''
        Constructor
        '''
        super().__init__(bname, device, design, report, sdc)

        # Max and min model of the selected combination
        self._max_model = max_model
        self._min_model = min_model

        # Map of the GPIO mode to the list of arcs
        self._mode2arcs: Dict[enum.Enum, List[TimingArc]] = {}

        # We cache the map of parameter name to the delay table
        # each time it is read. The key is following the convention:
        # <block>.<parameter_name>
        # Hence, it means that there can be duplicates of same parameter
        # name of different block
        self._max_parameter_map: Dict = {}
        self._min_parameter_map: Dict = {}

        # Stores the variable map found in this block.
        # The map is of type
        # VariableTYpe->list of strings
        self._variable_map: Dict = {}

        # Save a list of clkout pins list:
        # <clkout/block instance name> -clock <clkout pin name in Interface Designer > -reference_pin [get_ports {<clkout pad name in core>}]
        # Where clkout pad name in core follows the following format:
        # <Clock pin>~CLKOUT~X~Y
        # Example of the full string for each clkout interface pin:
        # rx_fclk_output -clock rx_fclk -reference_pin [get_ports {rx_fclk~CLKOUT~1~15}]
        # This is a set of string with the format mentioned above (without the comment)
        # so that no duplicate exists
        #self.clkout_str_names = set()

        # Map of alternate gpio instance to the tuple of:
        # clock name, max delay, min delay (GPIO_CLK_IN/GPIO_IN), is_clk
        # The last is a flag indicating if it is clock or not.
        # We don't want to write out 
        self.ins2_alt_delay_map = {}

        # Map of clkout gpio mode instance to the tuple of:
        # clock_name, clkout ref pin, max delay value, 
        # min delay value (GPIO_CLK_OUT), clkout_coord
        self.ins2_clkout_delay_map = {}

    def _save_source_latency_and_clkout(self, ins_to_block_map):
        # Unused unless needed by specific block
        pass

    @abc.abstractmethod
    def build_arc(self):
        """
        Build Timing Arcs
        """
        pass

    def _load_arc_parameter(self, elem_block, param_name, is_max,
                            is_optional=False):
        '''
        Find the arc parameter name in the block and upload
        it to the parameter_map.  However, if it is already
        loaded in map, then just return it.

        :param gpio_block: The xml element that points to block
        :param param_name: The arc parameter name

        :return The ArcDelayTable object
        '''

        arc_table = None

        if elem_block is not None:
            # Get the block name
            block_attrib = elem_block.attrib
            block_str = block_attrib["type"]

            table_key = block_str + "." + param_name

            if is_max:
                if table_key in self._max_parameter_map:
                    arc_table = self._max_parameter_map[table_key]
                else:
                    arc_table = self._create_delay_table(
                        block_str, elem_block, param_name)
                    if arc_table is not None:
                        self._max_parameter_map[table_key] = arc_table
                    elif not is_optional:
                        raise ValueError(
                            "Empty arc table for {}".format(table_key))
            else:
                if table_key in self._min_parameter_map:
                    arc_table = self._min_parameter_map[table_key]
                else:
                    arc_table = self._create_delay_table(
                        block_str, elem_block, param_name)
                    if arc_table is not None:
                        self._min_parameter_map[table_key] = arc_table
                    elif not is_optional:
                        raise ValueError(
                            "Empty arc table for {}".format(table_key))

        return arc_table

    def _create_delay_table(self, block_name, elem_block, param_name):
        '''
        '''
        arc_table = None

        # arc_param_tag = ".//" + self.xml_ns + "arc_parameter[@name=\"" + param_name + "\""
        arc_param_tag = ".//" + self.xml_ns + "arc_parameter"

        arc_block = None
        for elem in elem_block.iterfind(arc_param_tag):
            arc_attrib = elem.attrib

            if arc_attrib["name"] == param_name:
                arc_block = elem
                break

        if arc_block is not None:
            pcount = 0
            if "pcount" in arc_block.attrib:
                pcount = int(arc_block.attrib["pcount"])

            arc_table = writer_arc.ArcDelayTable(
                block_name, param_name, pcount)

            use_combined_param = False
            use_variable_option = False

            param_tag = ".//" + self.xml_ns + "parameter"
            for elem in arc_block.iterfind(param_tag):
                # print("{}: {}".format(elem.tag, elem.attrib))
                use_combined_param = True

                param_attrib = elem.attrib

                pname = ""
                if "name" in param_attrib:
                    pname = param_attrib["name"]

                pdelay = float(param_attrib["delay"])

                if not arc_table.add_delay(pname, pdelay):
                    raise ValueError("Unexpected delay table for {}: {}".format(
                        block_name, param_name))

            var_opt_param_tag = ".//" + self.xml_ns + "variable_option"
            for elem in arc_block.iterfind(var_opt_param_tag):
                # print("{}: {}".format(elem.tag, elem.attrib))
                use_variable_option = True

                param_attrib = elem.attrib

                var_name = ""
                if "name" in param_attrib:
                    var_name = param_attrib["name"]
                    overlap_var_str = param_attrib.get("overlap_var", "")

                    var_opt_block = elem

                    var_table = arc_table.create_variable_table(var_name)
                    if overlap_var_str != "":
                        var_table.set_overlap_variable(overlap_var_str)

                    self._parse_arc_variable_option(
                        block_name, param_name, var_table, var_name, var_opt_block)

                else:
                    raise ValueError("Missing name in the variable_option of {} and arc {}".format(
                        block_name, param_name))

        return arc_table

    def _parse_arc_variable_option(self, block_name, param_name, var_table,
                                   var_name, var_opt_block):

        var_opt_block_tag = ".//" + self.xml_ns + "var_value"

        gpio_block = None
        for elem in var_opt_block.iterfind(var_opt_block_tag):
            # print("{}: {}".format(elem.tag, elem.attrib))

            param_attrib = elem.attrib

            var_value = ""
            if "name" in param_attrib:
                var_value = param_attrib["name"]

            pdelay = float(param_attrib["delay"])

            if not var_table.add_delay(var_value, pdelay):
                raise ValueError("Unexpected delay table for block {} - {}: {}".format(
                    block_name, param_name, var_name))

    def _get_delay_variable(self, block_name, block_elem):
        '''
        Find the variable attribute in the block section.
        :param block_name: The name of the block (type string)
        :param block_elem: The xml element that represents the
                        entire block section

        :return a map of variable name to the list of
                acceptable values
        '''
        gutil.mark_unused(block_name)

        var_map = {}

        # First, check if it already exists
        if self._variable_map:
            var_map = self._variable_map

        else:
            var_tag = ".//" + self.xml_ns + "variable"

            for elem in block_elem.iterfind(var_tag):
                # print("{}: {}".format(elem.tag, elem.attrib))

                var_attrib = elem.attrib

                vname = var_attrib["name"]

                # Translate the name to enum
                if vname not in self.str2var_map:
                    raise ValueError(
                        "Unexpected timing variable name: {}".format(vname))

                vtype = self.str2var_map[vname]

                values = var_attrib["value"]
                # If it matches a pattern of start,stop,step, then we just enumerate
                # numbers from that pattern
                val_list = []
                if values:
                    if values.find(":") == -1:
                        val_list = values.split(",")
                    else:
                        self.logger.debug(
                            "Reading variable {}: {}".format(vname, values))

                        range_regex = re.compile(r'(\d+):(\d+):(\d+)')
                        re_result = range_regex.search(values)
                        if re_result:
                            start, stop, step = re_result.groups()

                            # Convert the range to a list
                            var_list = [
                                *range(int(start), int(stop), int(step))]
                            print("Match regex: start,stop,step: {},{},{} -> {}: {}".format(
                                start, stop, step, type(var_list), var_list))

                        else:
                            val_list = values.split(",")
                else:
                    val_list = values.split(",")
                var_map[vtype] = val_list

            self._variable_map = var_map

        return var_map

    def _add_clk_name(self, clk2ins_map, clk_name, gpio_obj):
        '''
        Add the clk name into the map.

        :param clk2ins_map: A map of clock name to the
                        list of gpio instance object
        :param clk_name: Clock name
        :param gpio_obj: GPIO design object
        '''

        ins_list = []

        if clk_name not in clk2ins_map:
            ins_list.append(gpio_obj)
            clk2ins_map[clk_name] = ins_list
        else:
            ins_list = clk2ins_map[clk_name]
            if gpio_obj not in ins_list:
                ins_list.append(gpio_obj)
                clk2ins_map[clk_name] = ins_list

    def _get_delay(self, delay_map, tscale, arc_type):
        '''
        :param delay_map: A map of arc to the delay value

        :return the delay that represents the arc
        '''

        delay = 0
        if delay_map:
            count = 0
            for item in delay_map:
                if item.get_type() == arc_type:
                    # Delay are in ps. Therefore we must scale to ns (divide 1000)
                    raw_delay = (delay_map[item] * tscale) / 1000
                    delay = raw_delay
                    count += 1

            if count > 1:
                raise ValueError("Found multiple arc associated")

        return delay

    def _get_delay_by_arc(self, delay_map, tscale, arc_obj):
        '''
        :param delay_map: A map of arc object to the delay value

        :return the delay that represents the arc
        '''
        delay = 0
        if delay_map:

            if arc_obj in delay_map:
                # Delay are in ps. Therefore we must scale to ns (divide 1000)
                raw_delay = (delay_map[arc_obj] * tscale) / 1000
                delay = raw_delay

            else:
                raise ValueError("Unable to find delay for {}".format(
                    arc_obj.get_delay()))

        return delay

    def _parse_simple_timing_table(self, table_model, blk_name):
        '''
        Parse the timing model to get the map of
        parameter name to the delay value for a specific block.
        This should be used on a simple block where it has no
        timing parameter variations.
        :param table_model: The timing model (i.e. Max/Min Model)
        :param blk_name: The block section in the model to parse
        :return A map of parameter name to the delay value after
                it is scaled
        '''

        param2delay_map = {}

        try:
            tree = et.ElementTree(file=table_model.get_filename())
            # print("REading file: {}".format(table_model.get_filename()))

            block_tag = ".//" + self.xml_ns + "block"

            blocksec = None
            for elem in tree.iterfind(block_tag):
                # print("{}: {}".format(elem.tag, elem.attrib))

                block_attrib = elem.attrib
                if block_attrib["type"] == blk_name:
                    blocksec = elem
                    break

            if blocksec is not None:
                # This will be the list of names without any
                # variables
                arc_param_tag = ".//" + self.xml_ns + "arc_parameter"

                for elem in blocksec.iterfind(arc_param_tag):
                    arc_attrib = elem.attrib
                    param_count = 0
                    delay_value = None

                    if "name" in arc_attrib and arc_attrib["name"] != "":
                        pname = arc_attrib["name"]
                        # Get the delay with the expectation that there is
                        # no variable parameter
                        if arc_attrib["pcount"] is not None and\
                                int(arc_attrib["pcount"]) > 0:
                            raise ValueError(
                                '{} variable pcount is expected '
                                'to be 0 instead of {}'.format(
                                    self._name, arc_attrib["pcount"]))

                        # Get the delay (children of arc_parameter)
                        param_tag = ".//" + self.xml_ns + "parameter"

                        for pelem in elem.iterfind(param_tag):
                            param_attrib = pelem.attrib

                            delay_value = float(param_attrib["delay"])
                            param_count += 1

                        if param_count == 1 and pname not in param2delay_map:

                            # We need to scale it
                            scaled_delay = (
                                delay_value * table_model.get_tscale()) / 1000

                            param2delay_map[pname] = scaled_delay
                            # self.logger.debug("Saving {} parameter {} from {} scaled {} to {}".format(
                            #    self._name, pname, delay_value, table_model.get_tscale(), scaled_delay))

                        elif param_count > 1:
                            raise ValueError(
                                '{} arc_parameter should only have one'
                                ' delay value. Instead found {}'.format(
                                    self._name, param_count))

                        elif pname in param2delay_map:
                            raise ValueError(
                                'Found duplicated arc_parameter {} in {} block'.format(
                                    pname, self._name))

                        else:
                            raise ValueError(
                                'No delay value stated for arc_parameter {}'.format(
                                    pname))

        except Exception as excp:
            # self.logger.error("Error with reading the timing common_models: {}".format(excp))
            raise excp

        return param2delay_map


    def _get_block_timing_tag_section(self, is_max):
        tree = None

        if is_max:
            if self._max_model is not None:
                tree = et.ElementTree(file=self._max_model.get_filename())
        else:
            if self._min_model is not None:
                tree = et.ElementTree(file=self._min_model.get_filename())

        blk_elem_tag = None
        if tree is not None:
            block_tag = ".//" + self.xml_ns + "block"
            blk_elem_tag = None
            for elem in tree.iterfind(block_tag):

                block_attrib = elem.attrib
                if block_attrib["type"] == self._name:
                    blk_elem_tag = elem
                    break
            assert blk_elem_tag is not None

        return blk_elem_tag
    
    def _get_timing_delay(self, ins_obj, path_type):
        '''
        Based on the object, it gets the required delay.

        :param gpio_obj: The gpio object

        :return the delay value represented in tuple max,min
        '''
        # Get the max model
        max_delay = self._parse_timing_table(
            self._max_model, ins_obj, path_type, True)

        # Get the min model
        min_delay = self._parse_timing_table(
            self._min_model, ins_obj, path_type, False)

        return max_delay, min_delay

    def _parse_timing_table(self, table_model,
                            ins_obj, path_type, is_max):
        '''
        :param table_model: a Model object
        :param sflash_obj: SPIFLash design object
        :param path_type: SPIFlashPathType
        :param is_max: Boolean that indicates if it is trying
                to read max or min model
        :return a map of arc to the delay value
        '''

        arc2delay_map = {}

        try:
            tree = et.ElementTree(file=table_model.get_filename())
            # print("REading file: {}".format(table_model.get_filename()))

            block_tag = ".//" + self.xml_ns + "block"

            blk_elem = None
            for elem in tree.iterfind(block_tag):
                # print("{}: {}".format(elem.tag, elem.attrib))

                block_attrib = elem.attrib
                if block_attrib["type"] == self._name:
                    blk_elem = elem
                    break

            if blk_elem is not None:
                # Get the delay variable
                delay_var_map = self._get_delay_variable(self._name, blk_elem)

                # Get the timing arc associated to this mode
                if path_type in self._mode2arcs:
                    arcs_list = self._mode2arcs[path_type]

                    if arcs_list:
                        for arc in arcs_list:

                            param_name = arc.get_delay()

                            # Get the block parameter table. If it has not
                            # been created, then do so (cache)
                            arc_table = self._load_arc_parameter(
                                blk_elem, param_name, is_max)

                            delay_val = self._get_delay_value(
                                ins_obj, delay_var_map, arc_table, is_max)

                            # print("arc: {} delay: {}".format(arc, delay_val))
                            arc2delay_map[arc] = delay_val
                else:
                    self.logger.error("Unable to get the arcs")

        except Exception as excp:
            # self.logger.error("Error with reading the timing common_models: {}".format(excp))
            raise excp

        return arc2delay_map

    def _get_delay_value(self, ins_obj, delay_var_map, arc_table, is_max=True):
        # This will be overidden by block that needs it
        return 0

    def determine_clock_sdc_constraint(self, gen_pin_grp, clk_port_name, ins_obj, dev_ins):
        clk_str = ""
        ref_arg_str = ""
        sdc_comment = ""

        # Check if the clock name was specified by user since some of
        # the signals if empty results in warning (allowed) rather than error
        clk_pin = gen_pin_grp.get_pin_by_type_name(clk_port_name)

        if clk_pin is not None:
            clk_str = self.get_user_clock_constraint(
                gen_pin_grp, clk_port_name)

            if clk_str != "":
                if clk_pin.name == "":
                    # Create a template instead
                    clk_str += "<CLOCK>"
                    sdc_comment = "# "

                else:
                    # Only get the reference_pin if user has defined the clk name
                    ref_arg_str = self.set_clkout_ref_pin(
                        ins_obj, clk_port_name, clk_pin.name, dev_ins)

        return clk_str, ref_arg_str, sdc_comment

    def set_clkout_ref_pin(self, ins_obj, clk_type_name, clk_name, dev_ins, is_ref_pin_name_only=False):
        # Get the clkout string
        dev_ins_obj = self._device.find_instance(dev_ins)
        ref_pin_name = ""
        #clkout_str = ""
        clkout_coord = None

        if dev_ins_obj is not None:
            # self.logger.debug("set_clkout_ref_pin find pin {} in instance {}".format(
            #    clk_type_name, ins_obj.name))

            ref_pin_name, _, _ = self.get_clkout_info_with_coord(
                ins_obj.name, dev_ins_obj,
                clk_type_name, clk_name)

        if not is_ref_pin_name_only:
            ref_arg_str = ""
            if ref_pin_name != "":
                ref_arg_str = " -reference_pin [get_ports {{{}}}]".format(
                    ref_pin_name)
        else:
            ref_arg_str = ref_pin_name

        return ref_arg_str

    def get_gen_pin_group(self, ins_obj):
        if ins_obj is not None and hasattr(ins_obj, "gen_pin"):
            return ins_obj.gen_pin
        
        return None
    
    def get_user_gen_pin_bus_member_names(self, gen_pin_grp, port_name, blk_port):
        '''
        Based on the passed port name, we want to find
        the user specified name. There's no other way than to compare
        the port name with specific string and get the necessary
        member in design object.

        :param ins_obj: User instance
        :param port_name: The port name
        :return the list of user name that has been expanded
                with the individual member bits
        '''
        pin_name = ""
        
        # Find the pin with the name
        if gen_pin_grp is not None:
            gen_pin = gen_pin_grp.get_pin_by_type_name(port_name)

            if gen_pin is not None and gen_pin.name != "":
                user_pname = gen_pin.name

                # Iterate through the index
                if blk_port is not None:
                    bit_list = blk_port.get_members_index()

                    # For readability: If the members are too many
                    # then we do wildcard (bus_name*)
                    if len(bit_list) > 5:
                        pin_name = "{}[*]".format(user_pname)

                    elif bit_list:

                        count = 0
                        for bit_index in bit_list:

                            pin_name += "{}[{}]".format(user_pname, bit_index)

                            if count + 1 < len(bit_list):
                                pin_name += " "

                            count += 1

                    else:
                        pin_name = user_pname

                else:
                    raise ValueError(
                        "Unable to find bus port {}".format(port_name))

        return pin_name


@runtime_checkable
class DesignParameterToPCRConverter(Protocol):
    """
    Convert Design Parameter value to PCR Value (Value in LPF)
    """
    def __call__(self, value: Any) -> str:
        ...


def build_lookup_based_pcr_converter(lookup_map: Dict[Any, str], default_val: str | None = None) -> DesignParameterToPCRConverter:
    """
    Build a PCR converter function for sceario like these:

    +-------+-----+
    | Value | PCR |
    +-------+-----+
    | 0     | A   |
    | 1     | B   |
    | 6     | C   |
    | 8     | D   |
    +-------+-----+
    """
    def __func(value: Any, lookup_map: Dict[Any, str], default_val: str | None) -> str:
        converted = lookup_map.get(value)
        if converted is None:
            if default_val is None:
                raise KeyError
            else:
                converted = default_val
        return converted
    return partial(__func,
                   lookup_map=lookup_map,
                   default_val=default_val)


def build_offset_based_pcr_converter(offset: int) -> DesignParameterToPCRConverter:
    """
    Build a PCR converter for sceario like these:

    +-------+-----+-----------------+
    | Value | PCR |                 |
    +-------+-----+-----------------+
    | 16    | 15  | PCR = Value - 1 |
    +-------+-----+-----------------+
    """
    def __func(value: Any, offset: int) -> str:
        assert isinstance(value, int)
        converted = value + offset
        return str(converted)

    return partial(__func,
                   offset=offset)
