'''
Copyright (C) 2017-2021 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Jun 17, 2021

@author: maryam
'''
import os
import sys
import enum
import gc

from typing import IO, List

import xml.etree.ElementTree as et

# 3rd party library to display pretty table
from prettytable import PrettyTable

import util.gen_util as pt_util
import util.excp as app_excp

import device.excp as dev_excp
from device.block_definition import TimingArc
import device.db_interface as device_dbi
import design.db as des_db

import common_device.writer as tbi
from common_device.gpio.writer.timing import GPIOTiming
from common_device.hsio.writer.gpio_timing import HSIOGPIOTiming
from common_device.spi_flash.writer.interface_config import SPIFlashInterfaceConfig
from common_device.spi_flash.spi_flash_design import SPIFlashParamInfo, SPIFlash

from tx60_device.gpio.gpio_design_service_comp import GPIODesignServiceComplex

sys.path.append(os.path.join(os.path.dirname(__file__), '..'))


class SPIFlashTiming(tbi.BlockTimingWithArcs):
    '''
    Builds the timing data for SPI Flash used for printing out
    timing report and sdc file.
    '''

    class SPIFlashPathType(enum.Enum):
        output = 0
        input = 1
        comb_input = 2
        comb_output = 3

    def __init__(self, bname, device, design, report, sdc,
                 max_model, min_model, ins2iostd_map):
        '''
        Constructor
        '''
        super().__init__(bname, device, design, report, sdc,
                         max_model, min_model)

        # We're building it a bit different since there's no
        # really a mode for this block. The mode is actually
        # to differentiate between input and outupt arcs
        self._mode2arcs = {}

        # Map of instance to io standard
        self._ins2iostd_map = ins2iostd_map

    def build_arc(self):
        """
        Build Timing Arcs
        """
        self._mode2arcs = {}

        # Get the SPI Flash device block
        sf_block = self._device.find_block(self._name)
        if sf_block is None:
            msg = "Unable to find {} block definition to build timing arcs".format(self._name)
            raise ValueError(msg)

        input_arcs = sf_block.get_all_input_arcs(True)

        output_arcs = sf_block.get_all_output_arcs(True)

        # Arcs with input port direction
        comb_in_arcs = sf_block.get_all_input_arcs(False)

        # Arcs with output port direction
        comb_out_arcs = sf_block.get_all_output_arcs(False)

        self._mode2arcs[self.SPIFlashPathType.input] = input_arcs
        self._mode2arcs[self.SPIFlashPathType.output] = output_arcs
        self._mode2arcs[self.SPIFlashPathType.comb_input] = comb_in_arcs
        self._mode2arcs[self.SPIFlashPathType.comb_output] = comb_out_arcs

    def write(self, index, ins_to_block_map, clk2delay_map):
        '''
        Write out the report and sdc constraint for this block.
        Due to the fact that there is a specific delay for this block, we
        cannot use the GPIO timing data. Hence, we read directly from
        the spi_flash timing section.

        :param index: The index of this section. Used in printing to report
        :param ins_to_block_map: Map of device instance name to the block name

        :return True if there was any valid instance found

        '''
        #pt_util.mark_unused(clk2delay_map)

        ins_exists = False

        try:
            spiflash_reg = self._design.get_block_reg(
                des_db.PeriDesign.BlockType.spi_flash)

            if spiflash_reg is not None:

                ins_name2obj_map = spiflash_reg.get_name2inst_map()

                # Build the info if the instance exists
                if ins_name2obj_map:

                    # Build the arc
                    self.build_arc()

                    # Write out the output data
                    ins_exists = self.write_output(
                        index, ins_to_block_map,
                        ins_name2obj_map, clk2delay_map)

        except Exception as excp:
            raise excp

        return ins_exists

    def write_output(self, index, ins_to_block_map,
                     ins_name2obj_map, clk2delay_map):
        '''
        :param index: The index used to print into report
        :param ins_to_block_map: A map of device instance name to the block name
        :param ins_name2obj_map: A map of SPI Flash design instance name to the design obj
        :param clk2delay_map: A map of the die parameter to the delay

        :return True if there was any jtag instance
                that got printed
        '''

        ins_written = False
        write_successful = None

        rptfile = None
        sdcfile = None

        try:
            write_successful = False

            if ins_name2obj_map:
                # Open the file
                rptfile = open(self._report_file, 'a')
                sdcfile = open(self._sdc_file, 'a')

                valid_ins = 0
                print_report = False
                comb_table = PrettyTable(
                    ["Instance Name", "Pin Name", "Parameter", "Max (ns)", "Min (ns)"])

                # There should only be one in the map
                # TODO: Update if we support more than one SPI Flash instance
                for ins_name in sorted(ins_name2obj_map.keys(),
                                       key=pt_util.natural_sort_key_for_list):
                    ins_obj = ins_name2obj_map[ins_name]

                    # Work on instance that are part of device
                    if ins_obj.block_def in ins_to_block_map:

                        # Write the section header (independent of SPI Flash param)
                        if valid_ins == 0:
                            sdcfile.write("\n# SPI Flash Constraints\n")
                            sdcfile.write("########################\n")

                            valid_ins += 1

                        # If it is not registered, then we have to print
                        # data into report file
                        if not ins_obj.get_param_value(SPIFlashParamInfo.Id.reg_en):
                            print_report = True

                            rptfile.write(
                                "\n---------- {}. SPI Flash Timing Report (begin) ----------\n".format(index))

                        # Write the timing data
                        self.write_instance(sdcfile, rptfile, ins_obj, comb_table)

                if valid_ins > 0:
                    ins_written = True

                if print_report:
                    assert comb_table is not None
                    rptfile.write("\n{}\n".format(comb_table.get_string()))
                    rptfile.write(
                        "\n---------- SPI Flash Timing Report (end) ----------\n")

                rptfile.close()
                sdcfile.close()

            write_successful = True

        except Exception as excp:
            if write_successful is not None and\
                    not write_successful:
                if rptfile is not None:
                    rptfile.close()

                if sdcfile is not None:
                    sdcfile.close()
            raise excp

        return ins_written

    def write_instance(self, sdcfile: IO, rptfile: IO, ins_obj: SPIFlash,
                       comb_table: PrettyTable):

        # Get the list of interfaces that is relevant
        pin_type_names = ins_obj.get_pin_names_based_on_param()

        if ins_obj.gen_pin is not None:
            sf_gpin = ins_obj.gen_pin

            # If it is registered, then we don't need to care about the side
            # that goes to the spi flash device
            if ins_obj.get_param_value(SPIFlashParamInfo.Id.reg_en):
                # They should be related to the clock name that users provides
                sclk_name = ins_obj.get_param_value(SPIFlashParamInfo.Id.clk_name)
                self._write_registered_output(sdcfile, ins_obj, sclk_name, pin_type_names)
                self._write_registered_input(sdcfile, ins_obj, sclk_name, pin_type_names)

            else:
                # Write out the clock constraint first
                sclk_name = sf_gpin.get_pin_name_by_type("SCLK_OUT")
                if sclk_name != "" and "SCLK_OUT" in pin_type_names:
                    sdcfile.write(
                        '# create_generated_clock -source <SOURCE_CLK> <CLOCK RELATIONSHIP>' \
                        ' [get_ports {{{}}}]\n'.format(
                            sclk_name))

                # Write out the comb paths
                self._write_all_combination_arcs(sdcfile, comb_table, ins_obj, sclk_name, pin_type_names)

    def _write_combination_arcs(self, sdcfile: IO, comb_table: PrettyTable, ins_obj: SPIFlash,
                                sclk_name: str, path_type: SPIFlashPathType, pin_type_names_used: List[str]):

        max_arc2delay_map, min_arc2delay_map = self._get_timing_delay(
            ins_obj, path_type)

        comb_arcs = self._mode2arcs[path_type]

        gen_pin = ins_obj.gen_pin
        # Caller should already check
        assert gen_pin is not None

        for arc in comb_arcs:
            row_list = []

            # Sink and Source is the same name for comb arc on SPIFlash
            sink_name = arc.get_sink()

            # Find the pin from the list
            if sink_name in pin_type_names_used:
                user_pin_name = gen_pin.get_pin_name_by_type(sink_name)

                if user_pin_name != "":
                    row_list.append(ins_obj.name)
                    row_list.append(user_pin_name)

                    if path_type == self.SPIFlashPathType.comb_input:
                        row_list.append("SPI_FLASH_OUT")
                    else:
                        row_list.append("SPI_FLASH_IN")

                    max_delay = self._get_delay_by_arc(
                        max_arc2delay_map, self._max_model.get_tscale(), arc)

                    row_list.append("{0:.3f}".format(max_delay))

                    min_delay = self._get_delay_by_arc(
                        min_arc2delay_map, self._min_model.get_tscale(), arc)

                    row_list.append("{0:.3f}".format(min_delay))

                    comb_table.add_row(row_list)

                    if path_type == self.SPIFlashPathType.comb_input:
                        sdcfile.write("# set_output_delay -clock {} -max <MAX CALCULATION> [get_ports {{{}}}]\n".format(
                                sclk_name, user_pin_name))
                        sdcfile.write("# set_output_delay -clock {} -min <MIN CALCULATION> [get_ports {{{}}}]\n".format(
                                sclk_name, user_pin_name))

                    else:
                        sdcfile.write("# set_input_delay -clock {} -max <MAX CALCULATION> [get_ports {{{}}}]\n".format(
                                sclk_name, user_pin_name))
                        sdcfile.write("# set_input_delay -clock {} -min <MIN CALCULATION> [get_ports {{{}}}]\n".format(
                                sclk_name, user_pin_name))

    def _write_all_combination_arcs(self, sdcfile: IO, comb_table: PrettyTable, ins_obj: SPIFlash,
                                sclk_name: str, pin_type_names_used: List[str]):

        # Write out the output combination arc
        self._write_combination_arcs(sdcfile, comb_table, ins_obj, sclk_name, self.SPIFlashPathType.comb_output,
                                     pin_type_names_used)

        # Write out the input combination arc
        self._write_combination_arcs(sdcfile, comb_table, ins_obj, sclk_name, self.SPIFlashPathType.comb_input,
                                     pin_type_names_used)

    def _write_registered_input(self, sdcfile: IO, ins_obj: SPIFlash, clk_name: str,
                                pin_type_names_used: List[str]):

        max_arc2delay_map, min_arc2delay_map = self._get_timing_delay(
            ins_obj, self.SPIFlashPathType.input)

        in_arcs = self._mode2arcs[self.SPIFlashPathType.input]
        gen_pin = ins_obj.gen_pin
        # Caller should already check
        assert gen_pin is not None

        for arc in in_arcs:
            sink_name = arc.get_sink()

            # Find the pin from the list
            if sink_name in pin_type_names_used:
                user_pin_name = gen_pin.get_pin_name_by_type(sink_name)

                if user_pin_name != "":
                    # TODO: We're not printing out the reference_pin.  If we
                    # need to, then we find the config file and load it into a
                    # local gpio registry and then find the INCLK/OUTCLK pin to
                    # get the value by also calling the function set_clkout_ref_pin
                    if arc.get_type() == TimingArc.TimingArcType.setup:
                        if arc in max_arc2delay_map:
                            # Get the calculated delay
                            delay = self._get_delay_by_arc(
                                max_arc2delay_map, self._max_model.get_tscale(), arc)

                            max_delay = "{0:.3f}".format(delay)

                            sdcfile.write(
                                "set_output_delay -clock {} -max {} [get_ports {{{}}}]\n".format(
                                    clk_name, max_delay, user_pin_name))

                    elif arc.get_type() == TimingArc.TimingArcType.hold:
                        if arc in min_arc2delay_map:
                            delay = self._get_delay_by_arc(
                                min_arc2delay_map, self._min_model.get_tscale(), arc)

                            min_delay = "{0:.3f}".format(-1 * delay)

                            sdcfile.write(
                                "set_output_delay -clock {} -min {} [get_ports {{{}}}]\n".format(
                                    clk_name, min_delay, user_pin_name))

    def _write_registered_output(self, sdcfile: IO, ins_obj: SPIFlash, clk_name: str,
                                pin_type_names_used: List[str]):

        max_arc2delay_map, min_arc2delay_map = self._get_timing_delay(
            ins_obj, self.SPIFlashPathType.output)

        in_arcs = self._mode2arcs[self.SPIFlashPathType.output]
        gen_pin = ins_obj.gen_pin
        # Caller should already check
        assert gen_pin is not None

        # There should not be any OE arcs in the list
        for arc in in_arcs:
            sink_name = arc.get_sink()

            # Find the pin from the list
            if sink_name in pin_type_names_used:
                user_pin_name = gen_pin.get_pin_name_by_type(sink_name)

                if user_pin_name != "":
                    # TODO: We're not printing out the reference_pin.  If we
                    # need to, then we find the config file and load it into a
                    # local gpio registry and then find the INCLK/OUTCLK pin to
                    # get the value by also calling the function set_clkout_ref_pin
                    if arc in max_arc2delay_map:
                        # Get the calculated delay
                        delay = self._get_delay_by_arc(
                            max_arc2delay_map, self._max_model.get_tscale(), arc)

                        max_delay = "{0:.3f}".format(delay)

                        sdcfile.write(
                            "set_input_delay -clock {} -max {} [get_ports {{{}}}]\n".format(
                                clk_name, max_delay, user_pin_name))

                    if arc in min_arc2delay_map:
                        delay = self._get_delay_by_arc(
                            min_arc2delay_map, self._min_model.get_tscale(), arc)

                        min_delay = "{0:.3f}".format(delay)

                        sdcfile.write(
                            "set_input_delay -clock {} -min {} [get_ports {{{}}}]\n".format(
                                clk_name, min_delay, user_pin_name))

    # Older codes using approach where printing it out based on HSIO related
    # timing parameters
    def write_old(self, index, ins_to_block_map, clk2delay_map):
        '''
        Write out the report and sdc constraint for this block.

        :param index: The index of this section. Used in printing to report
        :param ins_to_block_map: Map of device instance name to the block name

        :return True if there was any valid instance found

        '''
        #pt_util.mark_unused(clk2delay_map)

        ins_exists = False

        try:
            spiflash_reg = self._design.get_block_reg(
                des_db.PeriDesign.BlockType.spi_flash)

            if spiflash_reg is not None:

                ins_name2obj_map = spiflash_reg.get_name2inst_map()

                # Build the info if the instance exists
                if ins_name2obj_map:

                    ins_exists = self.write_output_old(
                        index, ins_to_block_map,
                        ins_name2obj_map, clk2delay_map)

        except Exception as excp:
            raise excp

        return ins_exists

    def write_output_old(self, index, ins_to_block_map,
                     ins_name2obj_map, clk2delay_map):
        '''
        :param index: The index used to print into report
        :param ins_to_block_map: A map of device instance name to the block name
        :param ins_name2obj_map: A map of SPI Flash design instance name to the design obj
        :param clk2delay_map: A map of the die parameter to the delay

        :return True if there was any jtag instance
                that got printed
        '''

        ins_written = False
        write_successful = None

        rptfile = None
        sdcfile = None

        try:
            write_successful = False

            # Get the device block definition
            ref_blk = self._device.find_block(self._name)
            if ref_blk is None:
                raise ValueError("Unable to find block definition for {}".format(self._name))

            gpio_dev_blk, gpio_writer = self.get_embedded_block_info()

            # Disable printing the report table header and the instance name column
            # along with other embedded instance reporting
            gpio_writer.print_header = False

            local_gpio_reg = self.get_embedded_instance_registry(ins_name2obj_map)

            sflash_icfg = SPIFlashInterfaceConfig(
                self._device, self._name)

            if ins_name2obj_map:
                # Open the file
                rptfile = open(self._report_file, 'a')
                sdcfile = open(self._sdc_file, 'a')

                sdcfile.write("\n# SPI Flash Constraints\n")
                sdcfile.write("########################\n")

                rptfile.write(
                    "\n---------- {}. SPI Flash Timing Report (begin) ----------\n".format(index))

                # Close it so that the GPIO will use it
                sdcfile.close()
                sdcfile = None

                rptfile.close()
                rptfile = None

            # There should only be one in the map
            # TODO: Update if we support more than one SPI Flash instance
            for ins_name in ins_name2obj_map.keys():
                ins_obj = ins_name2obj_map[ins_name]

                # Work on instance that are part of device
                if ins_obj.block_def in ins_to_block_map:

                    if local_gpio_reg is not None:
                        all_gpio = local_gpio_reg.get_all_gpio()

                        for gpio_ins in all_gpio:
                            # Check that the resource is in the device
                            # okay if the are not in active list (not in
                            # resource map)
                            if gpio_ins is not None:
                                if gpio_ins.gpio_def == "":
                                    msg = 'Instance {} of {} has invalid resource for {}'.format(
                                        ins_obj.name, self._name, gpio_ins.name)
                                    raise dev_excp.ConfigurationInvalidException(
                                        msg, app_excp.MsgLevel.error)

                                if self._device.find_instance(gpio_ins.gpio_def) is not None:
                                    # Before writing it, we need to replace the pin name in the
                                    # file to the user pin name using the gen_pin.type_name and
                                    # do a string replace
                                    sflash_icfg.replace_instance_used_pin_names(
                                        ins_obj, gpio_ins)

                        self.generate_timing(ins_obj, index, gpio_dev_blk, gpio_writer, all_gpio, clk2delay_map)

                        ins_written = True

            if ins_written:
                rptfile = open(self._report_file, 'a')
                rptfile.write(
                    "\n---------- SPI Flash Timing Report (end) ----------\n")

                rptfile.close()

            write_successful = True

        except Exception as excp:
            if write_successful is not None and\
                    not write_successful:
                if rptfile is not None:
                    rptfile.close()
                if sdcfile is not None:
                    sdcfile.close()

            raise excp

        return ins_written

    def get_embedded_block_info(self):
        if not self._design.is_tesseract_design():
            # Get the block definition (hsio/gpio depending on device)
            gpio_dev_block = self._device.find_block("gpio")

            gpio_writer = GPIOTiming(
                "gpio", self._device, self._design, self._report_file, self._sdc_file,
                self._max_model, self._min_model, self._ins2iostd_map)

        else:
            # Get the block definition (hsio/gpio depending on device)
            gpio_dev_block = self._device.find_block("hsio")

            # TODO: We're assuming that routing delay is not on device supporting HyperRAM
            # Hence, we'll pass empty container      
            gpio_writer = HSIOGPIOTiming(
                "hsio", self._device, self._design, self._report_file, self._sdc_file,
                self._max_model, self._min_model, self._ins2iostd_map, {}, {}, {})

        if gpio_dev_block is None:
            msg = "Unable to find embedded block for {}".format(self._name)

            raise dev_excp.BlockDoesNotExistException(
                msg, app_excp.MsgLevel.error)

        return gpio_dev_block, gpio_writer

    def generate_timing(self, spi_des_ins, index, gpio_dev_blk, gpio_writer, all_gpio, clk2delay_map):

        # If it is registered, then we don't need to care about the side
        # that goes to the
        is_register = spi_des_ins.get_param_value(SPIFlashParamInfo.Id.reg_en)

        if not self._design.is_tesseract_design():
            gpio_writer.write(index, )

        else:
            # Use the HSIO GPIO Timing writer to write out the instance.
            # However, some of the parameter has additional delay.
            gpio_writer.write(index, 0, gpio_dev_blk, all_gpio, {})
            #self.clkout_str_names.update(gpio_writer.clkout_str_names)

    def _load_used_config_design(self, used_cfg_file):
        '''
        :param used_cfg_file: The config file that contain the used configuration
        :return a gpio registry
        '''

        gpio_service = GPIODesignServiceComplex()
        local_gpio_reg = gpio_service.load_design(used_cfg_file)
        # Turn off the check since the file is internally controlled
        # if not gpio_service.check_design_file(used_cfg_file):
        #    raise ValueError(
        #        "Failure parsing SPIFlash config file: {}".format(used_cfg_file))

        return local_gpio_reg

    def get_embedded_instance_registry(self, ins_name2obj_map):
        '''
        Figure out which configu file to read based on the parameters.
        Load the design into a local design registry.

        :param ins_name2obj_map: A map of SPI Flash design instance name to the design obj
        :return: Local gpio registry containing the instances that are
            embedded
        '''

        # Get the block definition
        sflash_blk = self._device.find_block(self._name)
        if sflash_blk is None:
            msg = "No device block defined for {}".format(self._name)
            raise dev_excp.PortsNotFoundException(
                msg, app_excp.MsgLevel.error)

        # Find the block config file
        if not sflash_blk.has_config_files():
            msg = "No config files for {} defined".format(self._name)
            raise dev_excp.PortsNotFoundException(
                msg, app_excp.MsgLevel.error)

        used_cfg_file = ""
        local_gpio_reg = None

        # TODO: SPI Flash needs changes if more than one instance is allowed
        for ins_name in ins_name2obj_map.keys():
            ins_obj = ins_name2obj_map[ins_name]

            if ins_obj is not None:
                cfg_file_name = ins_obj.get_configured_instance_filename()

                used_cfg_file = sflash_blk.get_config_file(cfg_file_name)
                if used_cfg_file == "":
                    msg = "Config file {} for {} not defined".format(cfg_file_name, self._name)
                    raise dev_excp.ConfigurationInvalidException(
                        msg, app_excp.MsgLevel.error)
                else:
                    break

        if used_cfg_file != "":
            # Read the contents of the file by calling the GPIODesignService
            local_gpio_reg = self._load_used_config_design(used_cfg_file)

            # call garbage collector since we create a registry
            gc.collect()

        return local_gpio_reg

    # No longer used
    def get_spi_flash_oe_status(self, ins_name2obj_map):
        is_oe_ena = False

        if ins_name2obj_map:
            # Just read the first design instance
            # TODO: SPI Flash needs changes if more than one instance is allowed
            for ins_name in ins_name2obj_map.keys():
                ins_obj = ins_name2obj_map[ins_name]

                if ins_obj.get_param_value(SPIFlashParamInfo.Id.ena_oe):
                    is_oe_ena = True
                    break

        return is_oe_ena

    # No longer used
    def get_embedded_instance_reg(self, is_oe_ena):
        '''
        Generate the interface configuration data of
        used gpio.

        :param design: design db
        :param active_instances: List of device available
                            resource instance names

        :return list of the UsedCorePort object
        '''
        # Get the block definition
        sflash_blk = self._device.find_block(self._name)
        if sflash_blk is None:
            msg = "No device block defined for {}".format(self._name)
            raise dev_excp.PortsNotFoundException(
                msg, app_excp.MsgLevel.error)

        # Find the block config file
        if not sflash_blk.has_config_files():
            msg = "No config files for {} defined".format(self._name)
            raise dev_excp.PortsNotFoundException(
                msg, app_excp.MsgLevel.error)

        if is_oe_ena:
            from device.block_definition import PeripheryBlock
            cfg_file_name = PeripheryBlock.SPI_FLASH_OE_USED_CFG_FILE

        else:
            cfg_file_name = "used"

        used_cfg_file = sflash_blk.get_config_file(cfg_file_name)
        if used_cfg_file == "":
            msg = "Config file {} for {} defined".format(cfg_file_name, self._name)
            raise dev_excp.ConfigurationInvalidException(
                msg, app_excp.MsgLevel.error)

        # Read the contents of the file by calling the GPIODesignService
        local_gpio_reg = self._load_used_config_design(used_cfg_file)

        # call garbage collector since we create a registry
        gc.collect()

        return local_gpio_reg

    def _get_delay_value(self, ins_obj, delay_var_map, arc_table, is_max=True):
        '''
        :param ins_obj: The user spi_flash design object
        :param delay_var_map: The table of variables mapped to acceptable
                        values
        :param arc_table: The arc_parameter table with the name
                        representing the variables concatenated (as
                        in the xml file). An ArcDelayTable object

        :return the delay value of that parameter based on the required
                lvds configuration
        '''
        pt_util.mark_unused(ins_obj)
        pt_util.mark_unused(delay_var_map)

        delay_val = 0

        # We don't expect any of the SPI Flash mode arcs to have
        # be dependent on variables
        # TODO: Modify this when the arcs supports it

        # Only if the arc parameter has variable associated, then
        # we figure out, else take that one value
        if arc_table.get_pcount() == 0:
            delay_val = arc_table.get_variable_delay("")

        return delay_val
