"""
Copyright (C) 2017-2020 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Oct 31, 2020

@author: maryam
"""

from __future__ import annotations
import abc
from typing import Mapping, Dict

import util.gen_util
import device.db_interface as devdb_int

import common_device.rule as base_rule
from common_device.seu.seu_design import SEU, SEUParamInfo



@util.gen_util.freeze_it
class SEUChecker(base_rule.Checker):
    """
    Provide checkers to validate user configuration of osc design.

    These checks mostly focused on logical check but also include redundant schema checks.
    This is because the checker is meant to check in memory database which may not be written
    to file yet.
    """

    def __init__(self, design):
        """
        Constructor
        """
        super().__init__(design)
        self.reg = None
        self.seudev_service = None
        self.block_tag = "seu"

        if self.design is not None:
            if self.design.device_setting is not None:
                self.reg = self.design.device_setting.seu_reg
                if self.design.device_db is not None:
                    dbi = devdb_int.DeviceDBService(self.design.device_db)
                    self.seudev_service = dbi.get_block_service(
                    devdb_int.DeviceDBService.BlockType.SEU)

    def _build_rules(self):
        """
        Build control bank rule database
        """
        self._add_rule(SEURuleStart())
        self._add_rule(SEURuleInterval())
        self._add_rule(SEURuleError())
        self._add_rule(SEURuleConfig())
        self._add_rule(SEURulePinEmptyName())
        self._add_rule(SEURulePinInvalidName())
        self._add_rule(SEUSampleDeviceFeature())


class SEURuleStart(base_rule.Rule):
    """
    Check that the start pin is non-empty when used in manual mode
    """

    def __init__(self):
        super().__init__()
        self.name = "seu_rule_start"

    def check(self, checker, design_block: SEU):
        util.gen_util.mark_unused(checker)

        # Get the mode
        seu = design_block

        # Skip if it is not enabled
        if not seu.is_ena_detect:
            return

        if seu.mode == SEU.ModeType.manual:
            seu_pin = seu.gen_pin

            if seu_pin is not None:
                start_pin = seu_pin.get_pin_by_type_name("START")
                if start_pin is not None:

                    if self.check_name_empty_or_invalid(start_pin.name,
                        "Empty SEU Start Detection pin name found",
                        "Invalid SEU Start Detection pin name found"
                    ):
                        return


class SEURuleInterval(base_rule.Rule):
    """
    Check the start interval is valid in terms of value and usage (automatic mode)
    """

    def __init__(self):
        super().__init__()
        self.name = "seu_rule_interval"

    def check(self, checker, design_block: SEU):
        util.gen_util.mark_unused(checker)

        # Get the mode
        seu = design_block

        # Skip if it is not enabled
        if not seu.is_ena_detect:
            return

        if seu.mode == SEU.ModeType.auto:
            if seu.param_group is not None:
                param_info = seu.get_param_info()
                assert param_info is not None

                #is_valid, msg = seu.param_group.check_param_value("WAIT_INTERVAL", seu.wait_interval)
                is_valid, msg = seu.param_group.check_param_value_by_id(
                    SEUParamInfo.Id.wait_interval, param_info)
                if not is_valid:
                    time_interval = param_info.wi_setting2time(
                        seu.wait_interval)
                    self.error("Invalid wait interval {} microsec".format(
                        time_interval))
                    return


class SEURuleError(base_rule.Rule):
    """
    Check that the Error pin is non-empty when SEU detection is enabled
    """

    def __init__(self):
        super().__init__()
        self.name = "seu_rule_error"

    def check(self, checker, design_block: SEU):
        util.gen_util.mark_unused(checker)

        # Get the mode
        seu = design_block

        # Skip if it is not enabled
        if not seu.is_ena_detect:
            return

        if seu.mode == SEU.ModeType.manual:
            seu_pin = seu.gen_pin

            if seu_pin is not None:
                start_pin = seu_pin.get_pin_by_type_name("ERROR")
                if start_pin is not None:

                    if self.check_name_empty_or_invalid(start_pin.name,
                        "Empty Error Status pin name found",
                        "Invalid Error Status pin name found"
                    ):
                        return


class SEURuleConfig(base_rule.Rule):
    """
    Check that all the params are valid
    """

    def __init__(self):
        super().__init__()
        self.name = "seu_rule_param"

    def check(self, checker, design_block: SEU):
        util.gen_util.mark_unused(checker)

        # Get the mode
        seu = design_block

        # Skip if it is not enabled
        if not seu.is_ena_detect:
            return

        param_info = seu.get_param_info()
        if seu.param_group is not None and param_info is not None:
            all_param = seu.param_group.get_all_param()

            def get_param_name(param_obj):
                return param_obj.name

            invalid_param_names = []
            for param_obj in sorted(all_param, key=get_param_name):
                if param_obj is not None:
                    is_valid, _ = seu.param_group.check_param_value(
                        param_obj.name, param_info)
                    if not is_valid:
                        invalid_param_names.append(param_obj.name)

            if invalid_param_names:
                self.error("Invalid parameters configuration: {}".format(
                    ",".join(invalid_param_names)))
                return

class SEURulePin(base_rule.Rule, metaclass=abc.ABCMeta):
    """
    Helper class for checking empty/invalid pin names
    Implmented class needs to inherit either RuleEmptyName or RuleInvalidName
    """
    @abc.abstractmethod
    def validate(self, pins : Mapping[str, base_rule.PinData]):
        pass

    def check(self, checker, design_block: SEU):
        util.gen_util.mark_unused(checker)

        seu = design_block

        all_pins = seu.gen_pin.get_all_pin()
        pins: Dict[str, base_rule.PinData] = {}

        for pin in all_pins:
            # Skip pins that are checked by other rules
            if pin.type_name in ["START", "ERROR"]:
                continue

            desc_name = seu.get_pin_property_name(pin.type_name)
            pins[desc_name] = base_rule.PinData(pin.name)

        self.validate(pins)

class SEURulePinEmptyName(base_rule.RuleEmptyName, SEURulePin):
    """
    Check that pin names are filled
    """
    def __init__(self):
        super().__init__()
        self.name = "seu_rule_empty_pins"
        # PT-1341: raise Warning for now
        self.severity_mode = self.SeverityType.info

class SEURulePinInvalidName(base_rule.RuleInvalidName, SEURulePin):
    """
    Check that pin names are valid
    """
    def __init__(self):
        super().__init__()
        self.name = "seu_rule_invalid_pins"


# DEVINFRA-1012: Leaving the rule here but instead is_sample_device will
# filter it out based on which sample devices requires special handling
class SEUSampleDeviceFeature(base_rule.Rule):
    """
    Check that SEU is not used in ES device
    """

    def __init__(self):
        super().__init__()
        self.name = "seu_rule_es_device"

    def check(self, checker: SEUChecker, design_block: SEU): # type: ignore

        # Get the mode
        seu = design_block

        if checker.design.device_db is not None and \
            checker.design.device_db.is_sample_device() and\
                seu.is_ena_detect:
            self.error("SEU Detection is not supported in ES device")


# PT-1549: Disable the check. Check isn't called but we'll just
# leave this for a while.
class SEUAutomaticMode(base_rule.Rule):
    """
    PT-1428: Flag if automatic mode is used but not supported
    """

    def __init__(self):
        super().__init__()
        self.name = "seu_rule_auto_mode"

    def check(self, checker: SEUChecker, design_block: SEU): # type: ignore

        # Get the mode
        seu = design_block

        # Skip if it is not enabled
        if not seu.is_ena_detect:
            return

        if checker.seudev_service is not None:
            if seu.mode == SEU.ModeType.auto and \
                not checker.seudev_service.is_seu_automatic_mode_supported():
                self.error("SEU detection: automatic mode is currently not supported")

if __name__ == "__main__":
    pass
