"""
Copyright (C) 2017-2018 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Dec 1, 2017

@author: yasmin
"""

from __future__ import annotations
import abc
from dataclasses import dataclass
import enum
import re
from typing import Mapping, TYPE_CHECKING, Optional, Callable, List, Dict, Any

from util.singleton_logger import Logger

from design.db_item import PeriDesignItem, PeriDesignRegistry

if TYPE_CHECKING:
    from design.db import PeriDesign

class Rule(metaclass=abc.ABCMeta):
    """
    Block rule interface
    """

    @enum.unique
    class SeverityType(enum.Enum):
        """
        Rule result severity type
        """
        critical = "critical"
        error = "error"
        warning = "warning"
        info = "info"
        unknown = "unknown"  # For case when rule is not run

    class Category(enum.Enum):
        """
        Type of rule category

        Typically good configuration is a pre-requisite for design check
        """
        design = 1  #: Constraint related rule check, focus on extended hardware constraint violation
        config = 2  #: Configuration and basic constraint rule check, focus on traps invalid configuration like illegal data etc

    sev2str_map = \
        {
            SeverityType.critical: "critical",
            SeverityType.error: "error",
            SeverityType.warning: "warning",
            SeverityType.info: "info",
            SeverityType.unknown: "unknown"
        }  #: Mapping of severity type in enum to string type

    str2sev_map = \
        {
            "critical": SeverityType.critical,
            "error": SeverityType.error,
            "warning": SeverityType.warning,
            "info": SeverityType.info,
            "unknown": SeverityType.unknown
        }  #: Mapping of severity type in string to enum type

    # Valid names:
    # Must be started with character
    # Can have number / characters / dash / underscore / tilde / square bucket
    valid_name_pattern = re.compile(r"[a-zA-Z][\/\~0-9a-zA-Z\_\[\]\.]*$")

    def __init__(self):
        self.name = ""
        self.msg = ""
        self.pass_status = True
        self.severity = self.SeverityType.info
        self.category = Rule.Category.design

    def __str__(self, *args, **kwargs):
        info = 'name:{} status:{} sev:{} msg:{}'.format(
            self.name,
            self.pass_status,
            self.severity,
            self.msg)

        return info

    def clear(self):
        self.severity = self.SeverityType.info
        self.pass_status = True
        self.msg = ""

    def warning(self, msg: str = ""):
        """
        Set self parameters for a warning.
        If msg is not given, use current self.msg (set outside the function)
        """
        self.severity = self.SeverityType.warning
        self.pass_status = True
        if msg:
            self.msg = msg

    def error(self, msg: str = ""):
        """
        Set self parameters for an error.
        If msg is not given, use current self.msg (set outside the function)
        """
        self.severity = self.SeverityType.error
        self.pass_status = False
        if msg:
            self.msg = msg

    def info(self, msg: str = ""):
        """
        Set self parameters for an info.
        If msg is not given, use current self.msg (set outside the function)
        """
        self.severity = self.SeverityType.info
        self.pass_status = True
        if msg:
            self.msg = msg

    def bypass(self):
        """
        Called when the rule is bypassed.
        """
        self.clear()

    @abc.abstractmethod
    def check(self, checker: Checker, design_block: PeriDesignItem):
        """
        Check block setting against this rule.
        :param checker: Checker object
        :param design_block: Specific block instance object
        """
        pass

    def is_check_pass(self):
        return self.pass_status

    def is_name_empty(self, name):
        """
        Returns True if name is empty.
        """
        return (name == "") or (name is None)

    def is_name_pattern_invalid(self, name):
        """
        Returns True if name does not pass the regex check.

        Note: will also return True if name is empty.
        """
        if name == '__gnd__' or name == '__vcc__':
            return False
        return self.valid_name_pattern.match(name) is None

    def is_pin_name_pattern_invalid(self, name):
        """
        Returns True if name does not pass the regex check.
        If name is in bus-generated format "name[x]", check base name only.

        Note: will also return True if name is empty.
        """
        match_obj = re.match(r'(.*)(\[\d+\])$', name)
        if match_obj:
            # [x] matched, this is a bus generated pin name, ignore the [x]
            base_name = match_obj.group(1)
        else:
            # validate this as normal
            base_name = name

        return self.is_name_pattern_invalid(base_name)

    def check_name_empty_or_invalid(self, name, msg_empty="", msg_invalid=""):
        """
        If name is empty or invalid, set error parameters and returns True.
        This assumes empty/invalid names are Error by default.
        """
        if self.is_name_empty(name):
            self.error(msg_empty if msg_empty else "Name is empty.")
            return True
        if self.is_name_pattern_invalid(name):
            self.error(msg_invalid if msg_invalid else "Valid characters are alphanumeric characters with dash and underscore only.")
            return True
        return False

@dataclass
class PinData:
    """
    Helper class for creating set of pins for empty/invalid name checking.

    Set allow_empty to True to skip checking emptiness for this pin.
    Set allow_invalid to True to skip checking valid naming for this pin (not recommended).
    """
    name: str
    allow_empty: bool = False
    allow_invalid: bool = False

class RuleEmptyName(Rule, metaclass=abc.ABCMeta):
    """
    Common rule for checking if pin names are filled.

    Pass all pin names of block to check.

    Set self.severity_mode to "warning" if rule should raise warning for empty names,
    or to "info" if ignoring empty names

    Returns True if check passes, False if error is found
    """

    def __init__(self):
        super().__init__()
        self.severity_mode = Rule.SeverityType.error

    def validate(self, pins: Mapping[str, PinData]):
        """
        Generic function for validating multiple pin names
        """
        empty_pins = []

        for desc_name, pin in pins.items():
            if self.is_name_empty(pin.name) and not pin.allow_empty:
                if desc_name not in empty_pins:
                    empty_pins.append(desc_name)

        if empty_pins:
            if self.severity_mode == Rule.SeverityType.warning:
                self.warning("Empty pin names found: {}".format(",".join(empty_pins)))
            elif self.severity_mode == Rule.SeverityType.error:
                self.error("Empty pin names found: {}".format(",".join(empty_pins)))
                return False

        return True


class RuleInvalidName(Rule, metaclass=abc.ABCMeta):
    """
    Common rule for checking if pin names are valid.

    Pass all pin names of block to check.
    Allow bus_name[x] format.
    Ignore empty names.

    Returns True if check passes, False if error is found
    """

    def validate(self, pins: Mapping[str, PinData]):
        """
        Generic function for validating multiple pin names
        """
        invalid_pins = []

        for desc_name, pin in pins.items():
            if self.is_pin_name_pattern_invalid(pin.name) and not self.is_name_empty(pin.name) and not pin.allow_invalid:
                if desc_name not in invalid_pins:
                    invalid_pins.append(desc_name)

        if invalid_pins:
            self.error("Invalid pin names found: {}".format(",".join(invalid_pins)))
            return False

        return True


class Checker(metaclass=abc.ABCMeta):
    """
    Rule checker. It provides a rule set and check operations.
    """

    class Type(enum.Enum):
        """
        Type of check scope
        """
        instance_chk = 1  #: Run check on single instance
        global_chk = 2  #: Run check across the design database

    type2str_map = \
        {
            Type.instance_chk: "Instance",
            Type.global_chk: "global_check"
        }  #: Mapping of check type in enum to string type

    def __init__(self, design: PeriDesign):
        """
        Constructor
        """
        self._rules: Dict[str, Rule] = {}  #: Maps of rule name to its rule object
        self._reg_rules: Dict[str, Rule] = {} # Maps of rule name to its rule object (Reg level)
        self.fail_count = 0  # Total rules that fails
        self.logger = Logger
        self.exclude_rule: List = []  #: List of excluded rule name
        self.fail_result_data = {}   # type: ignore #: Map of instance name to a list of fail result tuple

        self.design = design
        self.reg: Optional[Any] = None
        self.block_tag = "unknown_block"  #: Tag for customizing message
        self.block_tag_gen = None  #: Block tag generator function, if None, will use self.block_tag as the name
        self.inst_gen: Optional[Callable[[], List]] = None  #: Instance generator function, if None will use reg.get_all_inst()

        self._build_rules()

    def __str__(self, *args, **kwargs):
        info = 'rule_count:{} fail_count:{}'.format(len(self._rules), self.fail_count)
        return info

    def clear_run_data(self):
        """
        Clear output from running checks
        """
        self.fail_count = 0
        self.fail_result_data.clear()
        self.exclude_rule.clear()
        for rule in self._rules.values():
            rule.clear()
        for rule in self._reg_rules.values():
            rule.clear()

    def generate_statistic(self):
        """
        Generate rule statistic
        """
        rule_info = []
        rule_list = self._rules.values()
        total_count = len(rule_list)
        config_count = 0
        design_count = 0

        for rule in rule_list:
            rule_info.append("Rule: {}, Category: {}".format(rule.name, rule.category))
            if rule.category == Rule.Category.design:
                design_count += 1
            elif rule.category == Rule.Category.config:
                config_count += 1

        rule_info.append("Total Rule Count: {}".format(total_count))
        rule_info.append("Total Config Rule Count: {}".format(config_count))
        rule_info.append("Total Design Rule Count: {}".format(design_count))

        return total_count, rule_info

    @abc.abstractmethod
    def _build_rules(self):
        """
        Build rule database
        """
        pass

    def _add_rule(self, rule):
        self._rules[rule.name] = rule

    def _add_reg_rule(self, rule):
        self._reg_rules[rule.name] = rule

    def run_check(self, db_item: PeriDesignItem| PeriDesignRegistry,
                  exclude_rule=None, rule_category=Rule.Category.design):
        """
        Run all checks for target category except for the exclude rules, if any.

        :param db_item: A design object, typically block instance
        :param exclude_rule: A list of excluded rule names
        :param rule_category: Rule category to run
        :return True, if all rule check pass or the rule category doesn't exists, else False
        """

        self.logger.info("Running rule with {} category".format(rule_category))

        if exclude_rule is not None:
            self.exclude_rule = exclude_rule
        else:
            self.exclude_rule.clear()

        is_any_run = False
        is_pass = True
        is_reg = False

        if isinstance(db_item, PeriDesignRegistry):
            rules = self._reg_rules
            is_reg = True
        else:
            rules = self._rules

        for rule in rules.values():

            # Need to clear here since it is run repeatedly
            rule.clear()

            # Skip if it is in exclude
            if rule.name in self.exclude_rule:
                self.logger.info("Skipping rule {}".format(rule.name))
                continue

            # Run only the target category
            if rule.category != rule_category:
                continue

            is_any_run = True
            self.logger.info("Run {}".format(rule.name))
            rule.check(self, db_item)  # type: ignore

        total = self.get_total_run_count(is_reg)
        self.fail_count = len(self.get_fail_rule(is_reg))
        if self.fail_count > 0:
            is_pass = False

        if is_any_run:
            if rule_category == Rule.Category.design:
                category_name = "design_check"
            else:
                category_name = "config_check"

            self.logger.info("Run {} :{} Pass:{} Fail:{} Exclude:{}"
                             .format(category_name, total, (total - self.fail_count), self.fail_count,
                                     len(self.exclude_rule)))

        return is_pass

    def run_all_check(self, exclude_rule=None):
        """
        Run all checks on all instances in registry.
        Runs config checks first and then runs design checks if that pass.

        :param exclude_rule: A list of excluded rule name
        :return: True, if all pass, else False if one or more fail
        """

        self.clear_run_data()

        self.logger.info("Check all {} starts..".format(self.block_tag))

        is_register = False
        issue_reg = None
        if self.design is not None and self.design.issue_reg is not None:
            issue_reg = self.design.issue_reg
            is_register = True

        if self.inst_gen is None:
            assert self.reg is not None
            inst_list = self.reg.get_all_inst()
        else:
            # Get custom instance list from generator function
            # https://github.com/PyCQA/pylint/issues/1493
            inst_list = self.inst_gen()  # pylint: disable=not-callable


        for block_inst in inst_list:

            if self.block_tag_gen is not None:
                # https://github.com/PyCQA/pylint/issues/1493
                self.block_tag = self.block_tag_gen(block_inst)  # pylint: disable=not-callable

            self.logger.info("Check {} : {}".format(self.block_tag, block_inst.name))

            # First run all config check
            # If any of the config check is error and above, it will stop checking
            # If the config check is either warning or below, it will proceed to do design check
            is_pass = self.run_check(block_inst, exclude_rule, Rule.Category.config)
            self.__process_result(block_inst, is_register)

            if is_pass:
                self.run_check(block_inst, exclude_rule, Rule.Category.design)
                self.__process_result(block_inst, is_register)
            else:
                sev_rule_map = self.extract_rule_by_run_severity()
                error_rule_list = sev_rule_map.get(Rule.SeverityType.error, [])
                # By right, critical severity is fatal, just check
                critical_rule_list = sev_rule_map.get(Rule.SeverityType.critical, [])

                if len(error_rule_list) == 0 and len(critical_rule_list) == 0:
                    self.run_check(block_inst, exclude_rule, Rule.Category.design)
                    self.__process_result(block_inst, is_register)

        # Check reg
        if self.reg is not None and isinstance(self.reg, PeriDesignRegistry):
            self.run_check(self.reg, exclude_rule, Rule.Category.design)
            self.__process_reg_result(is_register)

        # Save the exclude rule count
        if is_register and exclude_rule is not None:
            assert issue_reg is not None
            issue_reg.append_exclude_rules(exclude_rule)

        self.logger.info("Check all {} done".format(self.block_tag))

        if len(self.fail_result_data) > 0:
            return False

        return True

    def __process_result(self, block_inst, is_register):
        """
        Process rule result for each instance

        :param block_inst: Block instance
        :param is_register: True, register in design issue registry, else no
        """

        # Need to keep somewhere
        result_list = []
        fail_rules = self.get_fail_rule()

        # Get warning rules
        warning_rules = self.get_warning_rules(fail_rules)
        info_rules = self.get_info_rules()

        combined_rules = fail_rules + warning_rules + info_rules

        for rule in combined_rules:
            data = (rule.severity, rule.name, rule.msg)

            if rule in fail_rules:
                result_list.append(data)

            if is_register:
                assert self.design.issue_reg is not None
                self.design.issue_reg.append_issue_by_tuple(
                    (block_inst.name, self.block_tag, rule.severity, rule.name, rule.msg)
                )

        count = len(result_list)
        if count > 0:
            self.fail_result_data[block_inst.name] = result_list

    def __process_reg_result(self, is_register: bool):
        """
        Process rule result for registry
        """
        # Need to keep somewhere
        result_list = []
        fail_rules = self.get_fail_rule(True)

        # Get warning rules
        warning_rules = self.get_warning_rules(fail_rules, True)

        combined_rules = fail_rules + warning_rules

        for rule in combined_rules:
            data = (rule.severity, rule.name, rule.msg)

            if rule in fail_rules:
                result_list.append(data)

            if is_register:
                assert self.design.issue_reg is not None
                glob_name = Checker.type2str_map[Checker.Type.global_chk]
                self.design.issue_reg.append_issue_by_tuple(
                    (glob_name, self.block_tag, rule.severity, rule.name, rule.msg)
                )

        count = len(result_list)
        if count > 0:
            self.fail_result_data[""] = result_list

    def get_total_run_count(self, is_register=False):
        """
        Get total rules executed, does not include excluded rules.

        :return: Run count
        """
        if is_register:
            return len(self._reg_rules)

        return len(self._rules) - len(self.exclude_rule)

    def get_result(self, is_register=False):
        """
        Get full result for last run

        :return: A list of tuple (name, status)
        """

        result = []
        if is_register:
            rules = self._reg_rules
        else:
            rules = self._rules

        for rule in rules.values():
            result.append((rule.name, rule.pass_status))

        return result

    def get_fail_result(self, is_register=False):
        """
        Get fail result

        :return: A list of tuple of name, msg
        """
        result = []
        if is_register:
            rules = self._reg_rules
        else:
            rules = self._rules

        for rule in rules.values():
            if rule.pass_status is False:
                result.append((rule.name, rule.msg))

        return result

    def get_fail_rule(self, is_register=False):
        """
        Get fail rules

        :return: A list of failed rule
        """
        result = []
        if is_register:
            rules = self._reg_rules
        else:
            rules = self._rules

        for rule in rules.values():
            if rule.pass_status is False:
                result.append(rule)

        return result

    def get_warning_rules(self, fail_rules, is_register=False):
        """
        Get the rules that are warning and not from the list of fail rules

        :param fail_rules: A list of failed rules
        :return a list of warning fules that are not part of fail rules
        """

        rules_map = self._rules

        warning_rules = []
        if is_register:
            rules = self._reg_rules
        else:
            rules = self._rules

        for rule in rules.values():
            if rule.severity == Rule.SeverityType.warning and rule not in fail_rules:
                warning_rules.append(rule)

        return warning_rules

    def get_info_rules(self, is_register=False):
        info_rules = []
        if is_register:
            rules = self._reg_rules
        else:
            rules = self._rules

        for rule in rules.values():
            if rule.severity == Rule.SeverityType.info and rule.msg != '':
                info_rules.append(rule)

        return info_rules

    def extract_rule_by_run_severity(self):
        """
        Classify run rules into its severity. It doesn't process excluded rules.

        :return: A map of run severity and its list of rule.
        """
        sev_rule_map = {
            Rule.SeverityType.critical: [],
            Rule.SeverityType.error: [],
            Rule.SeverityType.warning: [],
            Rule.SeverityType.info: []
        }

        for rule in self._rules.values():
            if rule in self.exclude_rule:
                continue

            sev_rule_map[rule.severity].append(rule)

        return sev_rule_map

    def get_rules(self):
        """
        Get all rules

        :return: A list of rules
        """
        return self._rules

    def get_total_rules(self):
        """
        Get total rule count

        :return: A count
        """
        return len(self._rules)

    def get_rule_by_name(self, rule_name):
        return self._rules.get(rule_name, None)

    def get_skip_test_count(self):
        """
        Get the number of tests that were skipped because it does not satisfy its pre-requisite

        :return: count
        """
        count = 0
        for rule in self._rules.values():
            if rule.severity == rule.SeverityType.unknown:
                count = count + 1

        return count

    def get_exclude_rule(self):
        return self.exclude_rule


if __name__ == '__main__':
    pass
