from __future__ import annotations
import os
import abc

from typing import TYPE_CHECKING
from util.excp import pt_assert
import util.excp as app_excp

from device.block_instance import PinConnection
from device.excp import InstanceDoesNotExistException, ConfigurationInvalidException, InstanceInvalidResourceException, InstancePinConnectionNotFoundException
from tx375_device.common_quad.design import QuadLaneCommon

from common_device.writer import InterfaceConfig
from common_device.quad.res_service import QuadResService
from common_device.gpio.gpio_design import GPIOOutput

from design.db_item import GenericClockPin

if TYPE_CHECKING:
    from typing import List, Dict
    from design.db import PeriDesign
    from writer.used_core_port import UsedCorePort
    from common_device.quad.lane_design import LaneBasedItem
    from common_device.device_service_interface import BlockService

class LaneInterfaceConfig(InterfaceConfig):

    def __init__(self, device_db, name):
        '''
        constructor
        '''
        super().__init__(device_db, name)

        self._is_mockup = False

    def get_all_configured_interface(self, design: PeriDesign,
                                     active_instances: List[str],
                                     des_ins_skip_pin_names: Dict[str,str]) -> List[UsedCorePort]:
        """

        Get the interface configuration of all pin connectiosn on this
        device instance.  This is for INTERNAL USE ONLY where we want
        to dump all interface pin with using pin name like default gen pin names.
        It will write out all the interfaces of used instance in design only.

        :param design: design db
        :type design: PeriDesign
        :param active_instances: List of device available resource instance names
        :type active_instances: List[str]
        :param des_ins_skip_pin_names: Map of design instance name
                    to list of pin names (bitblasted) to be skipped. In case
                    we don't want to generate the one in pcie usage (or any)
        :type des_ins_skip_pin_names: Dict[str,str]
        :return: _description_
        :rtype: List[UsedCorePort]
        """

        assert self._device_db is not None
        assert self._name is not None

        dev_service = self.get_device_service()

        inf_config: List[UsedCorePort] = []

        all_instances = dev_service.get_all_instances(design)

        for design_obj in all_instances:
            self.verify_instance_type(design_obj)   # type: ignore

            pt_assert(design_obj.get_device() in active_instances,
                      f'Resource {design_obj.get_device()} assigned to {design_obj.name} does not exists in device', InstanceInvalidResourceException)

            device_inst = self.get_device().find_instance(design_obj.get_device())

            ins_gen_pin = design_obj.gen_pin
            pt_assert(ins_gen_pin is not None,
                      f'Instance {design_obj.name} of {self._name} has invalid generated pins data', ConfigurationInvalidException)

            # This would be for the entire device instance
            interface_pins : Dict[str, PinConnection] = \
                device_inst.get_inf_pins()

            # This would mean that it will still print out the pcie
            # pins which are not used but part of pcie mode since
            # the skip list is the list of pins used in pcie mode only
            # within the design context. So, if debug is disabled,
            # the debug pins are not in skip list and therefore
            # it will be written out with default name
            skip_pin_names = des_ins_skip_pin_names.get(design_obj.name, [])

            for pin_name, pin_conn in sorted(interface_pins.items()):
                pt_assert(pin_conn is not None,
                          f'Instance {design_obj.name} of {self._name} with undefined {pin_name}', InstancePinConnectionNotFoundException)

                if pin_conn.get_type() == \
                            PinConnection.InterfaceType.internal or\
                    pin_name in skip_pin_names:
                    # Skip internal pin
                    continue

                # Generate a default pin name
                inf_name = f'{design_obj.name}_{pin_name}'
                clk_name = ""

                if pin_conn.get_type() == PinConnection.InterfaceType.clkout:
                    clk_name = inf_name
                    inf_name = ""

                # Disable clock inversion by always setting it to False
                self._save_inf_pin(ins_name=design_obj.name,
                                   block_def=design_obj.get_device(),
                                   inf_config=inf_config,
                                   pin_conn=pin_conn,
                                   inf_name=inf_name,
                                   clk_name=clk_name,
                                   is_clk_inverted=False,
                                   is_required=False,
                                   tied_option=None)

        return inf_config

    def get_quad_block_definition(self, des_res_name: str):
        quad_idx, lane_idx = QuadResService.break_res_name(des_res_name)

        dev_ins_obj = None
        if quad_idx != -1 and lane_idx != -1:
            dev_ins_name = f'QUAD_{quad_idx}'
            dev_ins_obj = self.get_device().find_instance(dev_ins_name)
        
        return dev_ins_obj

    def get_common_pins(self, blk_ins):
        common_pin_class = self.get_common_pin_class()
        common_pin_type_name = []

        for pin_class in common_pin_class:
            # There can be a case that only some in that category is common
            pin_names = [pin.type_name for pin in blk_ins.get_pin_by_class(f"{pin_class}:SW_COMMON")]

            if pin_names:
                common_pin_type_name += pin_names

        return common_pin_type_name
        
    def get_lane_available_pins_for_connection(self, inst: LaneBasedItem):
        '''
        Depending on the lane instance, there might be some pins that is considered
        needed/available even if it's not enabled. This is to give way for the clock
        loopback to work where only the input clock is a gen pin but not the corresponding
        pcs clkout (ie RAW_SERDES_TX_CLK -> PCS_CLK_TX)
        '''
        return inst.get_available_pins()

    def get_configured_interface(self, design: PeriDesign, active_instances: List[str]) -> List[UsedCorePort]:
        """
        Get the interface configuration of used lane instance

        :param design: design db
        :type design: PeriDesign
        :param active_instances: List of device available resource instance names
        :type active_instances: List[str]
        :return: _description_
        :rtype: List[UsedCorePort]
        """
        assert self._device_db is not None
        assert self._name is not None

        dev_service = self.get_device_service()

        inf_config: List[UsedCorePort] = []

        # List of lane 10g design instance
        all_instances = dev_service.get_all_instances(design)
        
        # Map of the quad to the design instance
        first_quad_ins_map = {}

        common_gpin_type_name_list = []

        if len(all_instances) <= 0:
            return inf_config

        assert design.common_quad_lane_reg is not None
        cmn_inst = design.common_quad_lane_reg.get_inst_by_device_name(all_instances[0].get_device())
        assert isinstance(cmn_inst, QuadLaneCommon)

        common_gpin_type_name_list = self.get_common_pins(cmn_inst)
        assert len(common_gpin_type_name_list) > 0

        for design_obj in all_instances:
            self.verify_instance_type(design_obj)   # type: ignore

            pt_assert(design_obj.get_device() in active_instances,
                      f'Resource {design_obj.get_device()} assigned to {design_obj.name} does not exists in device', InstanceInvalidResourceException)

            # Device instance
            device_inst = self.get_quad_block_definition(design_obj.get_device())
            if device_inst is None:
                msg = 'Unable to find device instance {}'.format(
                    design_obj.get_device())
                raise InstanceDoesNotExistException(
                    msg, app_excp.MsgLevel.warning)

            # Device block definition
            ref_block = device_inst.get_block_definition()

            skip_common = False
            # TODO: If we support multiple blk type in quad, then
            # we need to filter across block types and not just
            # look at instance in the same registry only
            if device_inst.get_name() in first_quad_ins_map:
                skip_common = True


            # Determine which mode this is b ased on the res name
            mode_name = self.get_mode_name(design_obj.get_device())
            pt_assert(mode_name != "",
                      f'Resource {design_obj.get_device()} does not have valid mode name', InstanceInvalidResourceException)

            # A map of the mode inf name to the block port name
            # with all the names bit-blasted and not taken in context
            # of usage
            inf_group_pins_to_bitblast_port_names_map: Dict[str, str] = \
                            device_inst.get_block_inf_to_port_name_mapping_by_mode(mode_name, is_group_inf=False)

            # Get the map of block port name to the PinConnection of
            # ports that's only associated to the mode
            inf_pins_map: Dict[str, PinConnection] = device_inst.get_inf_pins_by_mode(mode_name)

            ins_gen_pin = design_obj.gen_pin
            cmn_inst = design.common_quad_lane_reg.get_inst_by_device_name(design_obj.get_device())
            assert isinstance(cmn_inst, QuadLaneCommon)
            cmn_gen_pin = cmn_inst.gen_pin

            pt_assert(ins_gen_pin is not None,
                      f'Instance {design_obj.name} of {self._name} has invalid generated pins data', ConfigurationInvalidException)
            pt_assert(cmn_gen_pin is not None,
                      f"Instance {design_obj.name} of {self._name} has invalid common generated pins data", ConfigurationInvalidException)

            # The list of generic pin type name (block interface) that is
            # valid in the current design context
            used_gpin_list = self.get_lane_available_pins_for_connection(design_obj) + \
                cmn_inst.get_available_pins_by_quad_type(design_obj.quad_type, design_obj.name)

            #print("\nUsed gpin list {}: {}".format(design_obj.name, used_gpin_list))
            
            used_bitblasted_inf_names: List[str] = device_inst.get_mode_inf_names_based_on_used_parent(mode_name, used_gpin_list)
            used_pin_names = []

            if used_bitblasted_inf_names:

                # Iterate through the used generic pin only. The list
                # only covers the generic pin
                for inf_name in used_bitblasted_inf_names:
                    assert inf_name in inf_group_pins_to_bitblast_port_names_map

                    # Check that the interface is to be used within current context
                    base_inf_name = inf_name
                    left_br_pos = base_inf_name.find("[")
                    if left_br_pos != -1:
                        # Get the base name before the left bracket
                        base_inf_name = base_inf_name[:left_br_pos]

                    port_name = inf_group_pins_to_bitblast_port_names_map[inf_name]

                    # Find the port in the inf map
                    pin_conn = None
                    if port_name in inf_pins_map:
                        pin_conn = inf_pins_map[port_name]

                    if pin_conn is None:
                        port_ref = ref_block.find_port(port_name)
                        self.logger.debug(
                            "Skip reading internal or pad {}".format(port_name))
                        if port_ref is not None and (port_ref.get_type() == port_ref.TYPE_PAD
                                                        or port_ref.get_type() == port_ref.TYPE_INTERNAL):
                            continue

                        else:
                            msg = 'Unable to find {} connection' \
                            ' in {} {}'.format(
                                port_name, self.get_user_blk_name(),
                                design_obj.name)
                            raise InstancePinConnectionNotFoundException(
                                msg, app_excp.MsgLevel.warning)

                    used_pin_names.append(pin_conn.get_expanded_name())

                    clkout2pin_map = {}

                    # Skip internal interface (primary)
                    if pin_conn.get_type() == \
                            PinConnection.InterfaceType.internal:

                        is_skip, clkout2pin_map = self.resolve_internal_pin(
                            design_obj, inf_name, inf_group_pins_to_bitblast_port_names_map,
                            inf_pins_map, design)   # type: ignore

                        if is_skip:
                            continue

                    # Find the corresponding generic pin
                    cur_gpin = ins_gen_pin.get_pin_by_type_name(base_inf_name)
                    if cur_gpin is None:
                        cur_gpin = cmn_gen_pin.get_pin_by_type_name(base_inf_name)
                    assert cur_gpin is not None

                    # Skip if this is common and we already
                    # wrote the common on the first lane in quad found
                    if skip_common and cur_gpin.type_name in common_gpin_type_name_list:
                        continue

                    # This  could be the generic pin user pin name or
                    # for PMA direct input clock, it could be the user clock input
                    # pin name coming from different lane instance (if bundled)
                    user_pin_name = self.get_user_pin_name(design, design_obj, cur_gpin)

                    # Need to append the index to user pin name
                    if user_pin_name != "":
                        # Check if the name is bit member. We replace the inf name
                        # with the user name and retain the index: HS_IN[3]
                        # ->user_name[3]
                        if inf_name.find("[") != -1:
                            tmp_name = inf_name.replace(
                                cur_gpin.type_name, user_pin_name)
                        else:
                            tmp_name = user_pin_name

                        cur_inf_name = ""
                        clk_name = ""
                        is_clk_inverted = False

                        if isinstance(cur_gpin, GenericClockPin):
                            clk_name = tmp_name
                            is_clk_inverted = cur_gpin.is_inverted
                            assert len(clkout2pin_map) == 0

                            self._save_inf_pin(ins_name=design_obj.name,
                                        block_def=design_obj.get_device(),
                                        inf_config=inf_config,
                                        pin_conn=pin_conn,
                                        inf_name=cur_inf_name,
                                        clk_name=clk_name,
                                        is_clk_inverted=is_clk_inverted,
                                        is_required=False,
                                        tied_option=None)

                        else:
                            # This is also applicable to the clkout
                            # that connects to the same clock input
                            # where there is no dedicated generic pin for the clkout

                            if clkout2pin_map:
                                for clkout_pin, pin_conn in sorted(clkout2pin_map.items()):

                                    if pin_conn.get_type() == PinConnection.InterfaceType.clkout:
                                        clk_name = tmp_name

                                        # if clkout_pin.find(":") != -1:
                                        #     start_loc = clkout_pin.find(":")
                                        #     # Use the pin name from the additional string
                                        #     # where that is the user clock pin name (which could
                                        #     # be from different lane instance - with bundle mode usage
                                        #     # in pma_direct))
                                        #     clk_name = clkout_pin[start_loc + 1:]

                                        self._save_inf_pin(ins_name=design_obj.name,
                                                    block_def=design_obj.get_device(),
                                                    inf_config=inf_config,
                                                    pin_conn=pin_conn,
                                                    inf_name=cur_inf_name,
                                                    clk_name=clk_name,
                                                    is_clk_inverted=is_clk_inverted,
                                                    is_required=False,
                                                    tied_option=None)

                            else:
                                if pin_conn.get_type() == PinConnection.InterfaceType.clkout:
                                    clk_name = tmp_name

                                else:
                                    cur_inf_name = tmp_name

                                self._save_inf_pin(ins_name=design_obj.name,
                                            block_def=design_obj.get_device(),
                                            inf_config=inf_config,
                                            pin_conn=pin_conn,
                                            inf_name=cur_inf_name,
                                            clk_name=clk_name,
                                            is_clk_inverted=is_clk_inverted,
                                            is_required=False,
                                            tied_option=None)

            if not skip_common:
                first_quad_ins_map[device_inst.get_name()] = design_obj

            self.set_tied_high_interface(
                design, design_obj, used_bitblasted_inf_names, inf_group_pins_to_bitblast_port_names_map,
                inf_pins_map, inf_config)

            used_pin_names = used_pin_names + self.get_lane_tie_high_ports()

        return inf_config

    # Override for clock input pins on pma direct with bundle mode
    def get_user_pin_name(self, design, inst, gen_pin):
        return gen_pin.name

    def set_tied_high_interface(self, design: PeriDesign, design_obj: LaneBasedItem,
                                 used_bitblasted_inf_names: List[str],
                                 inf2port_names_map: Dict[str, str],
                                 inf2pin_map: Dict[str, PinConnection],
                                 inf_config: List[UsedCorePort]):

        # The tied pins are both raw port and inf name
        tied_pins: List[str] = self.get_lane_tie_high_ports()

        common_reg = design.common_quad_lane_reg
        assert common_reg is not None
        common_inst = common_reg.get_inst_by_lane_name(design_obj.quad_type, design_obj.name)
        assert common_inst is not None

        for inf_name in tied_pins:

            if design_obj.gen_pin.get_pin_name_by_type(inf_name) == "":
                # It could be a common pin too
                if design_obj.gen_pin.get_pin_by_type_name(inf_name) is None:
                    # Use the common_inst gen pin to check
                    pin = common_inst.gen_pin.get_pin_by_type_name(inf_name)
                    assert pin is not None, f"Fail to find pin type {inf_name}"
                    if self.is_pin_used(cmn_inst=common_inst, inst=design_obj, pin=pin):
                        continue

                # Get the instance pin
                port_name = inf2port_names_map.get(inf_name, "")
                assert port_name != ""

                # They are actually inf_name == port_name unless the
                # list change
                if port_name != "":
                    pin_conn = inf2pin_map.get(port_name, None)

                    pt_assert(pin_conn is not None,
                            f'Instance {design_obj.name} of {self._name} does not have pin connected to {port_name}', ConfigurationInvalidException)

                    tied_option = GPIOOutput.TiedOptType.vcc
                    inf_name = GPIOOutput.tiedopt2str_map[tied_option]

                    self._save_inf_pin(ins_name=design_obj.name,
                                    block_def=design_obj.get_device(),
                                    inf_config=inf_config,
                                    pin_conn=pin_conn,
                                    inf_name=inf_name,
                                    clk_name="",
                                    is_clk_inverted=False,
                                    is_required=False,
                                    tied_option=tied_option)

    def is_pin_used(self, cmn_inst: QuadLaneCommon, inst: LaneBasedItem, pin):
        if pin.type_name == "PHY_CMN_RESET_N":
            return cmn_inst.is_pin_available(inst.quad_type, inst.name, pin.type_name) and\
                pin.name != ""
        else:
            return pin.name != ""

    def get_clkout_pin_names(self, inst:LaneBasedItem, inf_name:str, design_db: PeriDesign) -> List[str]:
        """
        :param inf_name: The clock input name that we want
                to find the corresponding clkout name to use
        :return a list of clkout inf name that is associated to
                the pass input clock name that has auto-connection
                (loopback of clk input back to clkout interface)
        """
        return ["PCS_CLK_RX", "PCS_CLK_TX"]

    def resolve_internal_pin(self, inst, inf_name, inf_group_pins_to_bitblast_port_names_map,
                             inf_pins_map, design_db: PeriDesign):
        '''
         -> bool, Dict[str, PinConnection]

        : return the following items:
            bool: indicate if the pin is to be skipped
            Dict[str, PinConnection]: A map of the pin name to the pin object
        '''
        is_skip = False
        clkout2pin_map = {}

        if self.is_input_clock_inf_name(inf_name):
            # Get the pin connected to its clkout pin instead
            # PCS_CLK_RX and PCS_CLK_TX connects to the same clock
            clkout_pin_list = self.get_clkout_pin_names(inst, inf_name, design_db)

            for clkout_pin in clkout_pin_list:

                inf_name = clkout_pin
                if clkout_pin.find(":") != -1:
                    stop_idx = clkout_pin.find(":")
                    inf_name = clkout_pin[:stop_idx]

                assert inf_name in inf_group_pins_to_bitblast_port_names_map

                port_name = inf_group_pins_to_bitblast_port_names_map[inf_name]

                pin_conn = inf_pins_map[port_name]
                if pin_conn is None or \
                    (pin_conn is not None and pin_conn.get_type() == PinConnection.InterfaceType.internal):
                    continue
                else:
                    clkout2pin_map[clkout_pin] = pin_conn

            if not clkout2pin_map:
                is_skip = True

        else:
            is_skip = True

        return is_skip, clkout2pin_map

    @abc.abstractmethod
    def get_device_service(self) -> BlockService:
        pass

    @abc.abstractmethod
    def verify_instance_type(self, ins: LaneBasedItem):
        pass

    @abc.abstractmethod
    def get_mode_name(self, res_name) -> str:
        pass

    @abc.abstractmethod
    def get_common_pin_class(self) -> List[str]:
        pass

    @abc.abstractmethod
    def get_user_blk_name(self) -> str:
        pass

    @abc.abstractmethod
    def get_lane_tie_high_ports(self) -> List[str]:
        pass

    @abc.abstractmethod
    def is_input_clock_inf_name(self, inf_name) -> bool:
        pass
