from __future__ import annotations
from abc import abstractmethod
from typing import List, Tuple, Optional
from pathlib import Path
import xmlschema
from xml.etree.ElementTree import Element, ParseError, iterparse as et_iterparse

from PyQt5.QtCore import QFile, QSaveFile, QXmlStreamWriter

import util.gen_util as gutil
from util.singleton_logger import Logger
from util.app_setting import SchemaService
import util.app_setting as aps

from design.db import PeriDesign
from design.db_item import GenericParamGroup, GenericPinGroup, GenericParam
from design.persistent_service import BlockBuilderXml, GenPinWriterXml
import design.service_interface as dbi

from common_device.quad.lane_design import LaneBaseRegistry, LaneBasedItem
from tx375_device.common_quad.design import QuadLaneCommonRegistry, QuadLaneCommon

class LaneBaseDesignService(dbi.DesignBlockService):
    """
    Provide operation for lane_based design
    """

    def __init__(self):
        """
        Constructor
        """
        super().__init__()
        self.logger = Logger
        self.checker = None

    # Overload
    def load_design(self, design_file: str):
        builder = self.get_design_builder()
        reg = builder.build(design_file)
        return reg

    def load_design_from_xmltree(self, xml_tree):
        """
        Not implemented
        """

    @abstractmethod
    def create_registry(self) -> LaneBaseRegistry:
        pass

    # Overload
    def save_design(self, design, design_file, xml_writer=None):
        saver = self.get_design_writer()

        if xml_writer is not None:
            saver.write(design, xml_writer)
        else:
            writer = QXmlStreamWriter()
            xml_file = QSaveFile(design_file)

            if xml_file.open(QFile.WriteOnly) is True:
                writer.setDevice(xml_file)
            else:
                self.logger.warning(
                    "Fail to open design file for writing, " + design_file)
                return

            writer.setAutoFormatting(True)
            writer.writeStartDocument()
            saver.write(design, writer)
            writer.writeEndDocument()

            # Save changes to harddisk
            if xml_file.commit() is True:
                self.logger.info("Write design to {}".format(design_file))
            else:
                self.logger.info(
                    "Fail to write design to {}".format(design_file))

    @abstractmethod
    def get_design_builder(self) -> LaneBaseDesignBuilderXml:
        pass

    @abstractmethod
    def get_design_writer(self) -> LaneBaseDesignWriterXml:
        pass

    # Overload
    def check_design_file(self, design_file):
        builder = self.get_design_builder()
        return builder.validate(design_file)

    # Overload
    @abstractmethod
    def check_design(self, design, exclude_rules=None, result_file=None, writer=None):
        pass

    def check_common_quad_lane_reg(self, reg: LaneBaseRegistry) -> Tuple[bool, Optional[QuadLaneCommon]]:
        """
        Check if common quad lane information is missing from design file
        lane based instance will use common instance information for display

        :param reg: Lane Based Registry
        :type reg: LaneBaseRegistry
        :return: return True if the common instance information is valid, otherwise False
        :rtype: bool
        """
        is_valid = False
        unlinked_none_cmn_inst = None
        common_reg = reg.common_quad_reg

        if common_reg is None:
            return is_valid, unlinked_none_cmn_inst

        for inst, cmn_inst_name in reg.inst2cmn_inst.items():
            device = inst.get_device()
            if cmn_inst_name != "":
                cmn_inst = common_reg.get_inst_by_name(cmn_inst_name)
            elif device == "" and unlinked_none_cmn_inst is None:
                cmn_inst = self.get_cmn_inst_without_device(common_reg)
                unlinked_none_cmn_inst = cmn_inst
            else:
                cmn_inst = common_reg.get_inst_by_device_name(device)

            if cmn_inst is None:
                cmn_inst = reg.create_new_common_quad_lane_inst(device)

            assert cmn_inst is not None

            # Build dependency graph
            inst.build_dep(cmn_inst, common_reg)

        reg.clear_inst2cmn_inst_connection()

        is_valid = True

        return is_valid, unlinked_none_cmn_inst

    def get_cmn_inst_without_device(self, reg: QuadLaneCommonRegistry):
        inst_list = [inst for inst in reg.get_all_inst() if inst.get_device() == ""]
        if len(inst_list) == 1:
            return inst_list[0]
        return None


class LaneBaseDesignBuilderXml(dbi.DesignBuilder):

    _ns = "{http://www.efinixinc.com/peri_design_db}"

    '''
    XML Namespace for peri design db
    '''

    _schema_file = ""
    '''
    If design schema is empty, it will be detected automatically using $EFXPT_HOME.
    This is mainly for unit testing to set schema from different location.
    '''
    # Tag name will be used in xml
    lane_tag = ""

    def __init__(self, service: LaneBaseDesignService):
        super().__init__()
        self.reg: Optional[LaneBaseRegistry] = None
        self._service = service
        self._param_builder = BlockBuilderXml({})
        self.logger = Logger


        # Will be used in error messages
        self.block_display_name = ""

    # Overload
    def build(self, design_file: str):
        with open(design_file, 'r', encoding='UTF-8') as xml_file:

            # get an iterable
            context = et_iterparse(xml_file, events=(
                "start", "end", "start-ns", "end-ns"))

            # turn it into an iterator
            context = iter(context)

            lane_info_tag = f"{self._ns}{self.lane_tag}_info"
            lane_tag = self._ns + self.lane_tag
            param_tag = self._ns + "param"
            pin_tag = f"{self._ns}pin"

            curr_inst: Optional[LaneBasedItem] = None
            curr_pin_group: Optional[GenericPinGroup] = None
            curr_def: str = ""
            curr_inst_name: str = ""

            for event, elem in context:
                if event == "start":
                    if elem.tag == lane_info_tag:
                        self._build_reg()

                    elif elem.tag == lane_tag:
                        curr_inst_name, curr_def, common_inst = self._get_inst_info(
                            elem)

                        assert self.reg is not None
                        curr_inst = self.reg.create_instance( # type: ignore
                            name=curr_inst_name)
                        assert curr_inst is not None
                        if curr_def != "":
                            self.reg.assign_inst_device(
                                curr_inst, curr_def)

                        self._build_inst(curr_inst, elem)
                        self.reg.add_inst2cmn_inst_connection(curr_inst, common_inst)
                        curr_pin_group = curr_inst.gen_pin

                    elif curr_inst is not None:
                        if elem.tag == param_tag:
                            self._build_generic_param(
                                curr_inst.param_group, elem)
                        elif elem.tag == pin_tag:
                            self._build_generic_pin(curr_pin_group, elem)

                    # Free up process element
                    elem.clear()

                elif event == "end":
                    if elem.tag == lane_tag:
                        if curr_inst is not None:
                            curr_inst.update_chksum()
                            curr_inst = None

                    elif elem.tag == lane_info_tag:
                        # Stop parsing once target registry content is done
                        break

        if self.reg is not None:
            self.reg.update_chksum()

        return self.reg

    def build_from_xmltree(self, xml_tree):
        """
        Not implemented
        """
        pass

    def _build_generic_param(self, param_group: Optional[GenericParamGroup], xml_elem):

        if param_group is not None:
            attrs = xml_elem.attrib
            name = attrs.get("name", "")
            value_type = attrs.get("value_type", "str")
            value_type_id = GenericParam.str2dtype_map.get(
                value_type, GenericParam.DataType.dstr)

            # We treat hex as string
            value = attrs.get("value", "")
            if value_type_id == GenericParam.DataType.dflo:
                value = float(value)
            elif value_type_id == GenericParam.DataType.dint:
                value = int(value)
            elif value_type_id == GenericParam.DataType.dbool:
                value = gutil.xmlstr2bool(value)

            if param_group.get_param_by_name(name) is None:
                param_group.add_param(name, value, value_type_id)
            else:
                param_group.set_param_value(name, value)

    def _build_generic_pin(self, pin_group, xml_elem):

        if pin_group is not None:
            attrs = xml_elem.attrib
            type_name = attrs.get("type_name", "")
            name = attrs.get("name", "")
            is_bus = gutil.xmlstr2bool(attrs.get("is_bus", "false"))

            is_clk = attrs.get("is_clk", None)
            if is_clk is not None:
                is_clk = gutil.xmlstr2bool(is_clk)
                if is_clk:
                    is_clk_invert = gutil.xmlstr2bool(
                        attrs.get("is_clk_invert", "false"))
                    pin_group.add_clock_pin(
                        type_name, name, is_bus, is_clk_invert)
            else:
                pin_group.add_pin(type_name, name, is_bus)

    def _build_reg(self):
        assert self._service is not None
        self.reg = self._service.create_registry()

    def _get_inst_info(self, xml_elem: Element) -> Tuple[str, str, str]:
        assert self.reg is not None
        attrs = xml_elem.attrib
        device: str = ""
        name: str = ""
        common_inst: str = ""
        try:
            device = attrs.get(f"{self.lane_tag}_def", "")
            name = attrs.get("name", "")
            common_inst = attrs.get("common_quad", "")
        except:
            self.logger.warning(
                "Fail to get instance information ignore: {}".format(xml_elem))

        return (name, device, common_inst)

    def _build_inst(self, inst: Optional[LaneBasedItem], xml_elem: Element):
        assert inst is not None
        assert isinstance(inst, LaneBasedItem)
        self._param_builder.build_param(inst, xml_elem)

    def check(self, design):
        gutil.mark_unused(design)

    def validate(self, design_file: str):
        app_setting = aps.AppSetting()
        schema_path: str = app_setting.app_path[aps.AppSetting.PathType.schema]

        # Lane Based schema
        if LaneBaseDesignBuilderXml._schema_file == "":
            LaneBaseDesignBuilderXml._schema_file = str(
                Path(schema_path) / f"{self.lane_tag}_design.xsd")

        schema = xmlschema.XMLSchema(
            LaneBaseDesignBuilderXml._schema_file)

        try:
            # validate Lane Based
            schema.validate(design_file)

        except ParseError as excp:
            self.logger.error(f"{self.block_display_name} design parsing error : {excp}")
            return False

        except xmlschema.XMLSchemaValidationError as excp:
            self.logger.error(f"{self.block_display_name} design validation error")
            self.logger.error(f"XML Element : {excp.elem}")
            self.logger.error(f"Reason : {excp.reason}")

            return False

        return True


class LaneBaseDesignWriterXml:

    def __init__(self):
        self.design: Optional[LaneBaseRegistry | PeriDesign] = None
        self.efx_ns = "efxpt"
        self.writer: Optional[QXmlStreamWriter] = None
        self.logger = Logger

    @abstractmethod
    def get_builder_class(self) -> LaneBaseDesignBuilderXml:
        pass

    @abstractmethod
    def get_reg_by_design(self, design: PeriDesign) -> Optional[LaneBaseRegistry]:
        pass

    # Overload
    def write(self, design: LaneBaseRegistry | PeriDesign, xml_writer: QXmlStreamWriter):
        """
        Write design to an xml file.

        If the writer is not None, it will be used to write. Otherwise new writer will be constructed.
        :param design: Design can be a PeriDesign or LaneBaseRegistry
        :param xml_writer:
        """
        self.design = design
        self.writer = xml_writer
        is_full_design = False

        reg = None
        if isinstance(design, LaneBaseRegistry):
            reg = design
        elif isinstance(design, PeriDesign):
            reg = self.get_reg_by_design(design)
            is_full_design = True

        assert isinstance(reg, LaneBaseRegistry)
        builder_class = self.get_builder_class()

        self.writer.writeStartElement(self.efx_ns + f":{builder_class.lane_tag}_info")

        if not is_full_design:
            self.writer.writeAttribute(
                "xmlns:efxpt", "http://www.efinixinc.com/peri_design_db")
            self.writer.writeAttribute(
                "xmlns:xsi", "http://www.w3.org/2001/XMLSchema-instance")

        all_inst_list: List[LaneBasedItem] = reg.get_all_inst()

        for inst in all_inst_list:
            self.writer.writeStartElement(self.efx_ns + f":{builder_class.lane_tag}")
            self.writer.writeAttribute("name", inst.name)
            self.writer.writeAttribute(f"{builder_class.lane_tag}_def", inst.get_device())
            cmn_inst = reg.get_cmn_inst_by_lane_name(inst.name)
            assert cmn_inst is not None
            self.writer.writeAttribute("common_quad", cmn_inst.name)

            # Pre-write cleanup: Update unavailable pin names to empty
            available_pin_type_names = set(inst.get_available_pins())
            for pin in inst.gen_pin.get_all_pin():
                if pin.type_name not in available_pin_type_names:
                    pin.name = ""

            assert isinstance(inst, LaneBasedItem)
            self.write_inst(inst)

            # lane_xx
            self.writer.writeEndElement()

        # lane_xx_info
        self.writer.writeEndElement()

    def write_inst(self, inst: LaneBasedItem):
        assert inst is not None
        assert self.writer is not None

        self.write_generic_param_group(inst.param_group)
        self.write_generic_pin(inst.gen_pin)

    def write_generic_pin(self, pin_group):
        assert self.writer is not None
        schema_svc = SchemaService()
        genpin_writer = GenPinWriterXml(schema_svc.setting, self.writer)
        # OMG: 'write_' instead of 'write'
        genpin_writer.write_(pin_group)

    def write_generic_param_group(self, param_grp_inst: Optional[GenericParamGroup]):
        assert self.writer is not None

        if param_grp_inst is None:
            return

        self.writer.writeStartElement(self.efx_ns + ":gen_param")
        param_list = param_grp_inst.get_all_param()

        def get_param_name(param: GenericParam):
            return param.name

        for param in sorted(param_list, key=get_param_name):
            self.write_generic_param(param)

        # param group
        self.writer.writeEndElement()

    def write_generic_param(self, param_inst: GenericParam):
        assert self.writer is not None

        self.writer.writeStartElement(self.efx_ns + ":param")
        self.writer.writeAttribute("name", param_inst.name)

        match param_inst.value_type:
            case param_inst.DataType.dstr | param_inst.DataType.dhex:
                self.writer.writeAttribute("value", param_inst.value)
            case param_inst.DataType.dbool:
                self.writer.writeAttribute("value", "{}".format(
                    gutil.bool2xmlstr(param_inst.value)))
            case param_inst.DataType.dflo:
                self.writer.writeAttribute(
                    "value", "{:.4f}".format(param_inst.value))
            case param_inst.DataType.dint:
                self.writer.writeAttribute(
                    "value", "{}".format(param_inst.value))

        self.writer.writeAttribute("value_type",
                                   GenericParam.dtype2str_map.get(param_inst.value_type, "str"))

        # param
        self.writer.writeEndElement()
