from __future__ import annotations
from typing import List, Tuple, Optional, TYPE_CHECKING, Type, Any, Dict, Set
from abc import abstractmethod

from common_device.device_service_interface import BlockService
from common_device.property import PropertyMetaData
from common_device.quad.res_service import QuadResService, QuadType

from tx375_device.common_quad.design_param_info import QuadDesignParamInfo
from tx375_device.common_quad.quad_prop_id import CommonQuadConfigParamInfo as CommonQuadParamInfo
from tx375_device.common_quad.design import QuadLaneCommon

from design.db_item import (
    PeriDesignRegistry,
    PeriDesignGenPinItem,
    GenericParamGroup,
    GenericParam,
    GenericPin,
    GenericParamService,
    ParamGroupMonitor
)


if TYPE_CHECKING:
    from enum import Enum
    from design.db import PeriDesign
    from device.db import PeripheryDevice

    from common_device.pll.pll_design import PLL
    from common_device.quad.res_service import QuadResService
    from common_device.quad.dep_graph import LaneBaseDependencyGraph
    from common_device.quad.lane_device_service import QuadLaneBasedDeviceService

    from tx375_device.common_quad.design import QuadLaneCommonRegistry


class LaneBasedItem(PeriDesignGenPinItem):

    def __init__(self, name: str, block_def: str = ""):
        super().__init__()
        self.__name = name
        self.__block_def = block_def
        self._categories: Set = set()
        self.quad_type = QuadType.lane_10g

        self._param_info = self.build_param_info()
        self._param_group = self.build_param()
        self.build_generic_pin()

        self._dep_graph: Optional[LaneBaseDependencyGraph] = None
        self.gp_monitor: Optional[ParamGroupMonitor] = None

    @property
    def name(self):
        return self.__name

    @name.setter
    def name(self, value: str):
        self.__name = value

    @property
    def dependency_graph(self):
        assert self._dep_graph is not None
        return self._dep_graph

    def get_device(self):
        return self.__block_def

    def set_device(self, device_name: str):
        self.__block_def = device_name

    def get_param_info(self):
        return self._param_info

    def get_param_group(self):
        return self._param_group

    @property
    def param_group(self):
        return self._param_group

    @property
    def param_info(self):
        return self._param_info

    @staticmethod
    def get_all_precision() -> Dict[Enum, int]:
        return {
        }

    @staticmethod
    def get_precision(param_id: Enum):
        return None

    def create_chksum(self):
        pass

    @abstractmethod
    def build_param_info(self) -> PropertyMetaData:
        pass

    @abstractmethod
    def get_input_clock_type_names(self) -> List[str]:
        pass

    @abstractmethod
    def get_used_pcs_clk_names(self, reg) -> Dict[str, Tuple[str, LaneBasedItem]]:
        pass

    def get_used_pcs_clkout_pin_names(self, reg) -> Dict[str, Tuple[str, LaneBasedItem]]:
        '''
        :return a map of the pcs clkout port name (ie PCS_CLK_RX0) mapped
                to tuple of:
                user clock name, lane instance
        '''
        clk2user_map = {}

        if self.get_device() != "":
            # We need the resource assigned if we want to get the port
            # since it is tied to lane index
            clk2user_map = self.get_used_pcs_clk_names(reg)

        return clk2user_map

    def get_user_clock_input_names_used(self):
        '''
        :return the user clock input names
        '''
        user_clock_input = []

        if self.gen_pin is None:
            return  user_clock_input
        
        type_names = self.get_input_clock_type_names()

        for inf_name in type_names:
            clk_name = self.gen_pin.get_pin_name_by_type(inf_name)
            if clk_name != "":
                user_clock_input.append(clk_name)

        return user_clock_input

    def build_param(self):
        assert self._param_info is not None
        param_group = GenericParamGroup()
        for param_info in self._param_info.get_all_prop():
            param_group.add_param(param_info.name, param_info.default, param_info.data_type)
            self._categories.add(param_info.category)

        return param_group

    def build_generic_pin(self):
        from device.block_definition import Port as DevicePort, PortDir
        self.gen_pin.clear()
        for device_port in self._device_port_map.values():
            self.dp_service.device_port = device_port
            if self.dp_service.get_type() == DevicePort.TYPE_CLOCK:
                self.gen_pin.add_clock_pin(self.dp_service.get_name(), "", self.dp_service.is_bus_port())
            else:
                direction = BlockService.get_port_direction_from_core(device_port)
                if direction == PortDir.input:
                    self.gen_pin.add_input_pin(self.dp_service.get_name(), "", self.dp_service.is_bus_port())
                elif direction == PortDir.output:
                    self.gen_pin.add_output_pin(self.dp_service.get_name(), "", self.dp_service.is_bus_port())
                else:
                    self.gen_pin.add_pin(self.dp_service.get_name(), "", self.dp_service.is_bus_port())

            self._categories.add(self.dp_service.get_class())

    def generate_pin_name(self):
        self.generate_pin_name_from_inst(self.__name)

    @property
    def categories(self):
        return self._categories

    # Overload
    def generate_pin_name_from_inst(self, inst_name: str):
        assert self.gen_pin is not None and self._dep_graph is not None

        for pin in self.gen_pin.get_all_pin():
            is_available = self._dep_graph.get_pin_attributes(pin.type_name)['is_available']
            if is_available:
                pin.generate_pin_name(inst_name)

        # Clockout interface and clock input need user to key in name
        self.update_default_pin_name()

    def update_default_pin_name(self):
        pin_list: List[GenericPin] = self.gen_pin.get_all_pin()
        for pin in pin_list:
            # No auto-generate pin name for CLKOUT and input clock pins
            if self._is_clkout_pin(pin.type_name) or \
                self._is_input_clk_pin(pin.type_name) or \
                self._is_hidden_pin(pin.type_name):
                pin.name = ""

    def _is_clkout_pin(self, pin_type_name: str) -> bool:
        return pin_type_name in {}

    def _is_input_clk_pin(self, pin_type_name: str) -> bool:
        return pin_type_name in {}

    def _is_hidden_pin(self, pin_type_name: str) -> bool:
        return pin_type_name in self.get_hidden_ports()

    def get_hidden_ports(self):
        return []

    @staticmethod
    def get_supported_common_parameters() -> List[str]:
        return []

    def generate_specific_pin_name_from_inst(self, pin_name, is_only_empty_name: bool = False):
        '''

        :param pin_name: The pin type_name to have its pin name generated
        :param is_empty_name: IF true, only regenerate if the pin name is empty.
        :return:
        '''
        assert self.gen_pin is not None

        if self._is_clkout_pin(pin_name) or self._is_input_clk_pin(pin_name) or\
            self._is_hidden_pin(pin_name):
            return

        pin = self.gen_pin.get_pin_by_type_name(pin_name)
        if pin is not None and (not is_only_empty_name or \
                                    (pin.name == "" and is_only_empty_name)):
            pin.generate_pin_name(self.name)

    def get_available_params(self) -> List[str]:
        assert self._dep_graph is not None
        param_name_list: List[str] = []
        for param in self._param_group.get_all_param():
            is_available = self.is_param_available(param.name)
            if is_available:
                if CommonQuadParamInfo.Id.has_member(param.name) and \
                    param.name not in self.get_supported_common_parameters():
                        continue
                param_name_list.append(param.name)

        return param_name_list

    def get_available_pins(self) -> List[str]:
        assert self._dep_graph is not None
        pin_type_name_list: List[str] = []
        for pin in self.gen_pin.get_all_pin():
            if self.is_pin_available(pin.type_name):
                pin_type_name_list.append(pin.type_name)
        return pin_type_name_list

    def get_ref_clk_info(self, dev_service: QuadLaneBasedDeviceService,
                         design: PeriDesign, param_id: CommonQuadParamInfo.Id| QuadDesignParamInfo.Id) -> Tuple[str, List[str]]:
        """
        Get ref clock information, return ref clk instance name and list of resource name

        :param dev_service: Lane device service
        :type dev_service: LaneDeviceService
        :param design: PeriDesign
        :type design: PeriDesign
        :return: ref clk instance name and list of resource name
        :rtype: Tuple[str, List[str]]
        """
        ref_clk_inst = ""
        res_name_list: List[str] = []

        # Get common instance
        common_reg = design.common_quad_lane_reg
        assert common_reg is not None
        lane_res = self.get_device()
        common_inst = common_reg.get_inst_by_lane_name(self.quad_type, self.name)
        assert isinstance(common_inst, QuadLaneCommon)

        # Get ref clk resource
        param_service = GenericParamService(common_inst.get_param_group(), common_inst.get_param_info())
        ref_clk_src = param_service.get_param_value(CommonQuadParamInfo.Id.PIPE_CONFIG_CMN__config_reg_2__pma_cmn_refclk_sel)

        match ref_clk_src:
            case "External":
                if lane_res == "":
                    return ref_clk_inst, res_name_list

                val = param_service.get_param_value(param_id)
                assert isinstance(val, str)
                val = val.replace(" ", "").upper()
                res_name_list = dev_service.get_external_refclk_pins(common_inst.get_device(), val)
                ref_clk_inst = ""

            case "Internal":
                assert param_id == QuadDesignParamInfo.Id.ref_clk_pin_name
                ref_clk_inst = param_service.get_param_value(param_id)
                assert isinstance(ref_clk_inst, str)

            case "PLL":
                ref_clk_inst, res_name_list = self.get_pll_ref_clk_info(
                    dev_service, design, param_id)

        return ref_clk_inst, res_name_list

    def get_pll_ref_clk_info(self, dev_service: QuadLaneBasedDeviceService,
                         design: PeriDesign, param_id: CommonQuadParamInfo.Id| QuadDesignParamInfo.Id) -> Tuple[str, List[str]]:
        ref_clk_inst = ""
        res_name_list: List[str] = []

        param_service = GenericParamService(self.get_param_group(), self.get_param_info())
        ref_clk_src = param_service.get_param_value(CommonQuadParamInfo.Id.PIPE_CONFIG_CMN__config_reg_2__pma_cmn_refclk_sel)
        pll_ref_clk = param_service.get_param_value(param_id)

        if ref_clk_src !=  "PLL":
            return ref_clk_inst, res_name_list

        inst_res_name = self.get_device()

        if inst_res_name == "":
            return ref_clk_inst, res_name_list

        res_name_list, ref_pin = dev_service.get_all_resource_on_ins_pin(
            inst_res_name, pll_ref_clk, None)

        pin = ""

        if design.pll_reg is None or ref_pin in ("", None):
            return ref_clk_inst, res_name_list

        assert isinstance(ref_pin, str)
        ref_pll: Optional[PLL] = None
        for res_name in res_name_list:
            pll_inst = design.pll_reg.get_inst_by_device_name(res_name)
            if pll_inst is not None:
                ref_pll = pll_inst
                break

        if ref_pll is not None:
            assert isinstance(ref_pin, str)
            pin_clk_no = {
                "CLKOUT0": 0,
                "CLKOUT1": 1,
                "CLKOUT2": 2,
                "CLKOUT3": 3,
                "CLKOUT4": 4,
            }.get(ref_pin, None)
            assert pin_clk_no is not None

            pll_out = ref_pll.get_output_clock_by_number(pin_clk_no)
            if pll_out is not None:
                pin = pll_out.name

        ref_clk_inst = pin

        return ref_clk_inst, res_name_list

    def get_cmn_parameters_from_cmn_reg(self, common_reg: QuadLaneCommonRegistry):
        param_map = {}
        common_inst = common_reg.get_inst_by_lane_name(self.quad_type, self.name)
        assert isinstance(common_inst, QuadLaneCommon)
        param_info = common_inst.param_info
        param_service = GenericParamService(common_inst.param_group, param_info)

        for param_id in CommonQuadParamInfo.Id:
            # The following parameter is for SW purpose use only
            if param_id.name.startswith('sw_'):
                continue

            pval = param_service.get_param_value(param_id)
            pname = param_info.get_prop_name(param_id)
            assert pname not in param_map

            prop_data = param_info.get_prop_info_by_name(pname)
            assert prop_data is not None
            pval = self.translate_hw_param_value(pval, prop_data.data_type)

            param_map[pname] = str(pval)

        return param_map

    def translate_hw_param_value(self, value: Any, data_type: GenericParam.DataType) -> str:
        match data_type:
            case GenericParam.DataType.dbool:
                if value:
                    value = "1"
                else:
                    value = "0"

            case GenericParam.DataType.dhex:
                assert isinstance(value, str)
                if not value.startswith("0x"):
                    value = "0x" + value
        return value

    def translate_hw_param_name(self, param_name: str, lane_idx: int):
        """
        Convert the SW param name to HW expected param name based
        on the resource that has been assigned
        """
        new_pname = param_name

        if new_pname.find("_NID") != -1:
            assert lane_idx is not None

            new_pname = new_pname.replace("_NID", str(lane_idx))

        else:
            # No translation needed
            new_pname = param_name

        return new_pname

    def get_param_in_category(self, category:str):
        param_info = self.get_param_info()
        param_group = self.get_param_group()

        filter_func = lambda p,c =category: p.get_prop_by_category(c) # type: ignore
        assert filter_func is not None

        cat_param = []

        for prop_info in filter_func(param_info):
            param = param_group.get_param_by_name(prop_info.name)
            assert param is not None

            cat_param.append(param)

        return cat_param

    def build_dep(self, cmn_inst: QuadLaneCommon, reg: QuadLaneCommonRegistry):
        assert self._dep_graph is not None

        # Build dependency graph with common parameters/pins
        self._dep_graph.cmn_reg = reg
        self._dep_graph.build_graph(cmn_inst.param_info, cmn_inst.gen_pin)
        self._dep_graph.build_dependency(self.param_info, self.gen_pin)
        cmn_inst.dependency_graph.add_lane_inst(self.quad_type, self.name)

        # Add observer
        self.gp_monitor = ParamGroupMonitor(self._dep_graph)
        self._param_group.register_param_observer(self.gp_monitor)
        cmn_inst.param_group.register_param_observer(self.gp_monitor)

        # Add connection in common registers
        reg.connect_lane_inst_2cmn_inst(self.quad_type, self.name, cmn_inst)

        # Synchronize the param / pin availablity
        self.update_all_availabilty(cmn_inst)

    def update_all_availabilty(self, cmn_inst: QuadLaneCommon):
        for prop in cmn_inst.param_info.get_all_prop():
            self.param_group.notify_param_changed(prop.name)

        for prop in self.param_info.get_all_prop():
            self.param_group.notify_param_changed(prop.name)

    def unregister_param_observer(self, cmn_inst: QuadLaneCommon):
        assert self.gp_monitor is not None
        cmn_inst.dependency_graph.delete_lane_inst(self.quad_type, self.name)
        self._param_group.unregister_param_observer(self.gp_monitor)
        cmn_inst.param_group.unregister_param_observer(self.gp_monitor)

    def is_pin_available(self, pin_type_name: str) -> bool:
        if self.gen_pin.get_pin_by_type_name(pin_type_name) is None:
            return False

        assert self._dep_graph is not None

        is_available = self._dep_graph.get_pin_attributes(pin_type_name)['is_available']
        return is_available

    def is_disconnect_clock_with_clkout_loopback(self, design_db, inf_name: str) -> bool:
        # By default, we assume that the clock is always connected to
        # the clock mux unless lane based specific on specific clkmux where
        # it cannot connect due to clkout loopback with rclk region.
        return False
       
    def is_param_available(self, param_name: str) -> bool:
        assert self._dep_graph is not None
        is_available = self._dep_graph.get_param_attributes(param_name)['is_available']
        return is_available

    def is_skip_clk_port_name(self, type_name: str) -> bool:
        return False


class LaneBaseRegistry(PeriDesignRegistry):
    def __init__(self, common_quad_reg: Optional[QuadLaneCommonRegistry]=None):
        self.res_svc: Optional[QuadResService] = None  #: Service for shared HSIO resource
        self.common_quad_reg = common_quad_reg

        # Used only when load design
        self.__inst2cmn_inst: Dict[LaneBasedItem, str] = {}
        super().__init__()

    def set_resource_service(self, quad_svc: QuadResService):
        self.res_svc = quad_svc

    @abstractmethod
    def get_instance_class(self) -> Type[LaneBasedItem]:
        pass

    def create_instance(self, name, apply_default=True, auto_pin=False):
        with self._write_lock:
            inst = self.get_instance_class()(name)
            if self.common_quad_reg is not None:
                # Create common instance
                cmn_inst = self.create_new_common_quad_lane_inst("")
                assert isinstance(cmn_inst, QuadLaneCommon)
                self.common_quad_reg.connect_lane_inst_2cmn_inst(inst.quad_type, inst.name, cmn_inst)

                # Load dep graph for inst
                inst.build_dep(cmn_inst, self.common_quad_reg)

            if apply_default:
                inst.set_default_setting(self.device_db)
            if auto_pin:
                inst.generate_pin_name()

            self._register_new_instance(inst)
            return inst

    @property
    def inst2cmn_inst(self) -> Dict[LaneBasedItem, str]:
        return self.__inst2cmn_inst

    def add_inst2cmn_inst_connection(self, inst: LaneBasedItem, cmn_inst_name:str):
        self.__inst2cmn_inst[inst] = cmn_inst_name

    def clear_inst2cmn_inst_connection(self):
        self.__inst2cmn_inst.clear()

    def is_device_valid(self, device_def: str) -> bool:
        """
        Override
        """
        assert self.res_svc is not None
        return self.res_svc.is_device_valid(device_def)

    def assign_inst_device(self, inst: LaneBasedItem, new_dev: str):
        cur_device = inst.get_device()

        # if same device, do nothing
        if cur_device == new_dev:
            return

        if self.common_quad_reg is None or inst.gp_monitor is None:
            return super().assign_inst_device(inst, new_dev)

        common_inst = self.get_cmn_inst_by_lane_name(inst.name)
        assert isinstance(common_inst, QuadLaneCommon)

        super().assign_inst_device(inst, new_dev)

        # If old/new devices are in the same quad, do normal flow
        if self.is_same_quad(cur_device, new_dev):
            return

        assert self.res_svc is not None

        new_cmn_inst = self.common_quad_reg.get_inst_by_device_name(new_dev)

        if new_cmn_inst is not None:
            cur_quad_num, _ = QuadResService.break_res_name(cur_device)

            self.disconnect_cmn_inst(inst, common_inst)

            # If no one using this QUAD/None, remove it
            if not (cur_device != "" and self.is_quad_used(cur_quad_num)):
                self.common_quad_reg.delete_inst(common_inst.name)

        else:
            # The new QUAD design not exist, create a new one
            cur_quad_num, _ = QuadResService.break_res_name(cur_device)

            if cur_device != "" and self.is_quad_used(cur_quad_num):
                new_cmn_inst = self.create_new_common_quad_lane_inst(new_dev)
                self.disconnect_cmn_inst(inst, common_inst)
            else:
                self.common_quad_reg.assign_inst_device(common_inst, new_dev)
                new_cmn_inst = common_inst

            assert isinstance(new_cmn_inst, QuadLaneCommon)

        self.connect_cmn_inst(inst, new_cmn_inst)

    def connect_cmn_inst(self, inst: LaneBasedItem, cmn_inst: QuadLaneCommon):
        """
        Override
        """
        assert self.common_quad_reg is not None
        if inst.gp_monitor is not None:
            cmn_inst.register_param_observer(inst.quad_type, inst.name, inst.gp_monitor)
        self.common_quad_reg.connect_lane_inst_2cmn_inst(inst.quad_type, inst.name, cmn_inst)
        inst.update_all_availabilty(cmn_inst)

    def disconnect_cmn_inst(self, inst: LaneBasedItem, cmn_inst: QuadLaneCommon):
        assert self.common_quad_reg is not None
        cmn_inst.unregister_param_observer(inst.quad_type, inst.name, inst.gp_monitor)
        self.common_quad_reg.disconnect_lane_inst_2cmn_inst(inst.quad_type, inst.name)

    def get_cmn_inst_by_lane_name(self, name: str) -> Optional[QuadLaneCommon]:
        """
        Get common instance by lane instance name

        :param name: Instance name of `LaneBasedItem`
        :type name: str
        """
        cmn_inst = None
        assert self.common_quad_reg is not None
        lane_inst = self.get_inst_by_name(name)

        if lane_inst is None:
            return cmn_inst

        assert isinstance(lane_inst, LaneBasedItem)
        device = lane_inst.get_device()

        if device != "":
            cmn_inst = self.common_quad_reg.get_inst_by_device_name(device)
        else:
            cmn_inst = self.common_quad_reg.get_inst_by_lane_name(lane_inst.quad_type, lane_inst.name)

        return cmn_inst

    def reset_inst_device(self, inst: LaneBasedItem):
        current_device = inst.get_device()
        super().reset_inst_device(inst)

        assert self.common_quad_reg is not None
        if current_device != "":
            quad_num, _ = QuadResService.break_res_name(current_device)
            old_cmn_inst = self.common_quad_reg.get_inst_by_device_name(current_device)
            assert isinstance(old_cmn_inst, QuadLaneCommon)

            # Update common instance
            if self.is_quad_used(quad_num):
                new_cmn_inst = self.create_new_common_quad_lane_inst("")
                assert new_cmn_inst is not None
                self.disconnect_cmn_inst(inst, old_cmn_inst)
                self.connect_cmn_inst(inst, new_cmn_inst)

                # Restore previous settings
                new_cmn_inst.update_settings_by_others(old_cmn_inst)
            else:
                self.common_quad_reg.reset_inst_device(old_cmn_inst)

    def create_new_common_quad_lane_inst(self, device_name: str) -> Optional[QuadLaneCommon]:
        """
        Create common instance by device name

        :param device_name: Quad/ Lane resource name
        :type device_name: str
        :return: Common instance
        :rtype: Optional[QuadLaneCommon]
        """
        if self.common_quad_reg is None:
            return None

        if self.common_quad_reg.is_device_used(device_name):
            return None

        new_inst_name = self.common_quad_reg.gen_unique_inst_name("cmn")

        if device_name == "":
            inst = self.common_quad_reg.create_instance(
                new_inst_name, apply_default=True, auto_pin=True)
        else:
            inst = self.common_quad_reg.create_instance_with_device(
                new_inst_name, device_name, auto_pin=True)

        assert isinstance(inst, QuadLaneCommon)
        return inst

    def is_quad_used(self, quad_num: int) -> bool:
        """
        Check if the given quad is used by any instance.
        E.g. Q0_LN1 -> used by inst1
            quad_num = 1, return False
            quad_num = 0, return True

        :param quad_num: quad number
        :type quad_num: int
        :return: True if the quad is used, otherwise False
        :rtype: bool
        """
        if self.res_svc is None:
            return False

        return self.res_svc.is_quad_used(quad_num)

    def delete_inst(self, name: str):
        inst = self.get_inst_by_name(name)
        if inst is None or self.common_quad_reg is None or self.res_svc is None:
            return

        assert isinstance(inst, LaneBasedItem)
        device = inst.get_device()
        cmn_inst = self.get_cmn_inst_by_lane_name(name)
        assert cmn_inst is not None
        inst.unregister_param_observer(cmn_inst)
        self.disconnect_cmn_inst(inst, cmn_inst)

        super().delete_inst(name)

        # Check if the device being used by other lane instance
        is_device_used = False

        if device != "":
            quad_num, _ = QuadResService.break_res_name(device)
            if self.is_quad_used(quad_num):
                is_device_used = True

        if not is_device_used:
            self.common_quad_reg.delete_inst(cmn_inst.name)

    def is_same_quad(self, lane_res_name: str, lane_res_name2) -> bool:
        if self.common_quad_reg is not None:
            return self.common_quad_reg.is_lane_same_quad(lane_res_name, lane_res_name2)
        return lane_res_name[:2] == lane_res_name2[:2]

    def get_insts_by_quad_res(self, quad_res_name: str):
        insts = []

        if quad_res_name == "":
            res_name_prefix = ""
        else:
            quad_num, _ = QuadResService.break_res_name(quad_res_name)
            assert quad_num != -1, f"Invalid quad resource name"
            res_name_prefix = f"Q{quad_num}_"

        for inst in self.get_all_inst():
            if (res_name_prefix != "" and inst.get_device().startswith(res_name_prefix)) or\
                (res_name_prefix == "" and inst.get_device() == ""):
                insts.append(inst)

        return insts

    def rename_inst(self, current_name, new_name, auto_pin=False):
        inst = super().rename_inst(current_name, new_name, auto_pin)

        if self.common_quad_reg is not None:
            assert isinstance(inst, LaneBasedItem)
            self.common_quad_reg.rename_lane_inst(inst.quad_type, current_name, new_name)

        return inst

    def get_instance_associated_to_common_inst(self, cmn_ins_name: str) -> List[LaneBasedItem]:
        inst_found: List[LaneBasedItem] = []

        # Find lane instance that points to the common instance name
        all_inst = self.get_all_inst()
        for inst in all_inst:
            cmn_inst = self.get_cmn_inst_by_lane_name(inst.name)
            if cmn_inst is not None and cmn_inst.name == cmn_ins_name:
                inst_found.append(inst)

        return inst_found

    def apply_device_db(self, device: PeripheryDevice):
        pass

    def get_ins_same_quad_clkout_input_names(self, quad_res: str) -> Dict[str, Tuple[str, str]]:
        '''
        This is used to get the clock input user name connected to
        the clkout pin of lane instance of this blk type found 
        associated to this quad.

        For some protocol, it's a direct loopback within same lane while
        some it's a clock input found other lanes if same/diff quad loopback
        to the clkout.  Also, for some case, the clkout has not connection
        when there's no loopback (ie raw serdes reg mode)

        :param quad_res: The quad resource name (ie "QUAD_0")
        :return A map of the clockout port name in the quad (not clkout lane inf name) 
            to a tuple of (user clock name, insance obj)
        '''
        quad_clk2user_map =  {}

        quad_inst_list =  self.get_insts_by_quad_res(quad_res)        

        # This is resource based and we are only looking at lane
        # instance associated to the quad. The clkout port name
        # return is the unique port name (ie PCS_CLK_TX0) instead
        # of the inf name
        for inst in quad_inst_list:
            if inst is not None:

                # The usage of the PCS_CLK is context-quad_type dependent
                # Get the map of used CLKOUT pins based on the port name
                # ie. PCS_CLK_TX0:
                # <clkout pin and NOT inf name>: <user clk pin name>
                clockout_port2user_name_map = inst.get_used_pcs_clkout_pin_names(self)

                if clockout_port2user_name_map:
                    quad_clk2user_map.update(clockout_port2user_name_map)

        return quad_clk2user_map