"""

Copyright (C) 2017-2024 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on May 13, 2024

@author: Shirley Chan
"""
from __future__ import annotations
from typing import TYPE_CHECKING, Dict, Optional, List, Any
from abc import abstractmethod
from PyQt5.QtWidgets import QComboBox, QLabel, QLineEdit, QWidget, QDoubleSpinBox

import util.gui_util as gui_util

from device.db_interface import DeviceDBService

from design.db_item import GenericParamService, PeriDesignGenPinItem

from common_device.quad.lane_design import LaneBasedItem
from common_device.quad.gui.builder import RefClkGpBoxBuilder

from tx375_device.common_quad.design import QuadLaneCommon
from tx375_device.common_quad.design_param_info import QuadDesignParamInfo
from tx375_device.common_quad.quad_prop_id import CommonQuadConfigParamInfo as QuadConfigParamInfo

from common_gui.base_presenter import BasePresenter
from common_gui.builder.base import ParamWidget, PinEditorWidget

if TYPE_CHECKING:
    from design.db import PeriDesign

    from common_device.quad.lane_device_service import QuadLaneBasedDeviceService


class LaneBasedConfigUIPresenter(BasePresenter):

    def __init__(self, parent=None):
        super().__init__(parent=parent)
        self._dev_service: Optional[QuadLaneBasedDeviceService] = None

    def setup_design(self, design: PeriDesign):
        self.design = design
        dev_db_service = DeviceDBService(design.device_db)
        self._dev_service = dev_db_service.get_block_service(DeviceDBService.BlockType.QUAD_LANE_10G)
        assert self._dev_service is not None

    def get_resource_names(self) -> List[str]:
        assert self._dev_service is not None
        return self._dev_service.get_usable_instance_names()

    def setup_ext_ref_clk_option(self, block: QuadLaneCommon, cb_ext_clk: QComboBox, is_show_hidden: bool):
        cb_ext_clk.clear()
        options = self.get_displayed_ext_clk_option(block, is_show_hidden)
        cb_ext_clk.addItems(options)

    def setup_pll_ref_clk_option(self, block: QuadLaneCommon, pll_ref_clk_widget_map: Dict[str, QWidget]):
        pll_clk_options = block.get_all_ref_clk_pll_options()

        for widget in pll_ref_clk_widget_map.values():
            if not isinstance(widget, QComboBox):
                continue

            widget.clear()
            widget.addItems(pll_clk_options)

    def load_common_name(self, block: QuadLaneCommon, le_name: QLineEdit):
        gui_util.set_config_widget(le_name, block.name, block.name)

    def get_displayed_ext_clk_option(self, block: QuadLaneCommon, is_show_hidden: bool):
        display_options = []
        param_id = QuadConfigParamInfo.Id.PMA_CMN__cmn_plllc_gen_preg__cmn_plllc_pfdclk1_sel_preg
        options = block.param_info.get_valid_setting(param_id)
        hidden_options = self.get_hidden_param_values().get(param_id.value, [])
        assert isinstance(options, list)

        for option in options:
            assert isinstance(option, str)
            if not is_show_hidden and option in hidden_options:
                continue
            display_options.append(self.get_displayed_ext_clk_name(option))

        return display_options

    def get_displayed_ext_clk_name(self, ref_clk: str):
        return ref_clk.replace("Refclk ", "CMN_REFCLK")

    def get_ext_clk_name(self, ref_clk: str):
        return ref_clk.replace("CMN_REFCLK", "Refclk ")

    def load_ext_ref_clk_res(self, block: QuadLaneCommon, cb_ext_clk: QComboBox, is_show_hidden: bool):
        assert block is not None
        self.setup_ext_ref_clk_option(block, cb_ext_clk, is_show_hidden)

        param_service = GenericParamService(block.get_param_group(), block.get_param_info())
        ext_ref_clk: str = param_service.get_param_value(QuadConfigParamInfo.Id.PMA_CMN__cmn_plllc_gen_preg__cmn_plllc_pfdclk1_sel_preg)
        ext_ref_clk = self.get_displayed_ext_clk_name(ext_ref_clk)
        gui_util.set_config_widget(widget=cb_ext_clk, value=ext_ref_clk, tooltip=ext_ref_clk)

    def load_pll_ref_clk_res(self, block: QuadLaneCommon, pll_ref_clk_widget_map: Dict[str, QWidget],
                             is_show_hidden: bool):
        self.setup_pll_ref_clk_option(block, pll_ref_clk_widget_map)

        param_service = GenericParamService(block.get_param_group(), block.get_param_info())
        param_list = [
            QuadDesignParamInfo.Id.pll_ref_clk0,
            QuadDesignParamInfo.Id.pll_ref_clk1,
        ]
        for idx, param_id in enumerate(param_list):
            val: str = param_service.get_param_value(param_id)
            widget = pll_ref_clk_widget_map.get(RefClkGpBoxBuilder.CB_PLL_CLK_OBJ_NAME + str(idx))
            assert widget is not None
            gui_util.set_config_widget(widget=widget, value=val, tooltip=val)

    def updated_param_option(self, block: LaneBasedItem| QuadLaneCommon, param_id , cb_param: QComboBox):
        cb_param.clear()
        settings = block.param_info.get_valid_setting(param_id)
        assert isinstance(settings, list)

        # Get hidden options
        hidden_vals = []
        if param_id.value in self.get_hidden_param_values():
            hidden_vals = self.get_hidden_param_values().get(param_id.value, [])

        display_settings = set(settings) - set(hidden_vals)
        cb_param.addItems(list(display_settings))

        self.update_option_display(cb_param, param_id.value)

    def load_property(self, block: LaneBasedItem, le_name: QLineEdit, cb_device: QComboBox):
        assert block is not None
        assert le_name is not None
        assert cb_device is not None

        gui_util.set_config_widget(widget=le_name, value=block.name, tooltip=block.name)

        self.setup_device_option(cb_device=cb_device)
        gui_util.set_config_widget(widget=cb_device, value=block.get_device(), tooltip=block.get_device())

    def load_generic_pin_property(self, db_item: PeriDesignGenPinItem, widget_map: Dict):
        """
        Load generic pin data from design to widget

        :param db_item: Design db item with generic pin
        :param widget_map: A map of widget name to its auto generated widget container/group
        """

        if db_item is None or widget_map is None:
            return

        gen_pin = db_item.gen_pin
        for widget_name, widget_grp in widget_map.items():
            # widget name is pin type name
            pin = gen_pin.get_pin_by_type_name(widget_name)
            if pin is not None:
                # control accessibility/applicability in each block editor as needed
                widget_grp.set_enable(True)
                if isinstance(widget_grp, PinEditorWidget):
                    widget_grp.set_current_data(pin)
                else:
                    widget_grp.set_current_data(pin.name)

    def load_generic_param_property(self, db_item, widget_map: Dict, enumtype2str_func=None, exclude_type=..., user_data: bool = False):
        super().load_generic_param_property(db_item, widget_map, enumtype2str_func, exclude_type, user_data)

        assert isinstance(db_item, (LaneBasedItem, QuadLaneCommon))
        self.set_widget_precision(db_item, widget_map)

    def set_widget_precision(self, db_item: LaneBasedItem| QuadLaneCommon, widget_map: Dict[str, ParamWidget]):
        param_group = db_item.param_group
        for param_id, precision in db_item.get_all_precision().items():
            widget: QWidget = widget_map[param_id.value].editor_widget
            param_val = float(param_group.get_param_value(param_id.value))

            if isinstance(widget, QDoubleSpinBox):
                widget.setDecimals(precision)
                widget.setValue(param_val)

            elif isinstance(widget, QComboBox):
                set_text = lambda text, precision=precision: "{:.{p}f}".format(float(text), p=precision)
                for num in range(widget.count()):
                    text = widget.itemText(num)
                    new_text = set_text(text)
                    widget.removeItem(num)
                    widget.insertItem(num, new_text)
                widget.setCurrentText(set_text(param_val))

    def load_ref_clk_property(self, db_item: LaneBasedItem, ref_clk_inst: Optional[QLabel],
                                  ref_clk_res: QLabel, param_id: QuadConfigParamInfo.Id| QuadDesignParamInfo.Id):
        assert db_item is not None
        assert ref_clk_res is not None
        assert self._dev_service is not None
        assert self.design is not None
        assert self.design.device_db is not None

        res_name = db_item.get_device()
        if res_name == "":
            if ref_clk_inst is not None:
                ref_clk_inst.setText("Unknown")

            ref_clk_res.setText("Unknown")
            return

        pin, res_name_list = db_item.get_ref_clk_info(self._dev_service, self.design, param_id)

        if ref_clk_inst is not None:
            ref_clk_inst.setText(pin)

        ref_clk_res.setText(", ".join(res_name_list))

    def update_all_option_display(self, param_widget_map: Dict[str, ParamWidget], is_show_hidden: bool):
        for param_name in self.get_param2db2ui_mapping().keys():
            widget_group = param_widget_map.get(param_name, None)
            assert widget_group is not None, f"cannot find parameter {param_name}"
            widget = widget_group.editor_widget
            assert isinstance(widget, QComboBox)
            self.update_option_display(widget, param_name)

    def update_option_display(self, widget: QComboBox, param_name: str):
        db_options = [widget.itemText(idx) for idx in range(widget.count())]

        widget.clear()

        for idx, db_option in enumerate(db_options):
            ui_value = self.convert_db2display_value(param_name, db_option)
            assert isinstance(ui_value, str)
        
            widget.addItem(ui_value)
            widget.setItemData(idx, db_option)

    def convert_db2display_value(self, param_name: str, db_value: Any) -> Any:
        ui_value = db_value

        if param_name in self.get_param2db2ui_mapping():
            display_mapping = self.get_param2db2ui_mapping().get(param_name)
            assert display_mapping is not None and isinstance(ui_value, str)

            for db_str, display_str in display_mapping.items():
                ui_value = ui_value.replace(db_str, display_str)

        return ui_value

    def convert_display_value2db_value(self, param_name: str, display_value: str):
        db_value = display_value

        if param_name in self.get_param2db2ui_mapping():
            display_mapping = self.get_param2db2ui_mapping().get(param_name)
            assert display_mapping is not None and isinstance(db_value, str)

            for db_str, display_str in display_mapping.items():
                db_value = db_value.replace(display_str, db_str)

        return db_value

    def get_display_param_value(self, param_grp, param_name):
        ui_value = param_grp.get_param_value(param_name)
        ui_value = self.convert_db2display_value(param_name, ui_value)
        return ui_value

    @abstractmethod
    def get_param2db2ui_mapping(self) -> Dict[str, Dict[str, str]]:
        raise NotImplemented

    @abstractmethod
    def get_hidden_param_values(self) -> Dict[str, List[str| float| int]]:
        raise NotImplemented

    def get_param_info(self, param_widget: QWidget, db_item: LaneBasedItem, str2enumtype_funcs, user_data: bool = False):
        param_name = None
        param_value = None
        widget_name = param_widget.objectName()

        if isinstance(param_widget, QComboBox):
            param_name = widget_name[3:]

            if param_name in self.get_param2db2ui_mapping():
                param_value = param_widget.itemData(param_widget.currentIndex())

        if param_name is None or param_value is None:
            return super().get_param_info(param_widget, db_item, str2enumtype_funcs, user_data)

        return param_name, param_value
