"""

Copyright (C) 2017-2024 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Dec 06, 2024

@author: Shirley Chan
"""

from __future__ import annotations
from typing import Callable, Dict, List, Tuple, Optional
from PyQt5.QtCore import pyqtSlot, Qt
from abc import abstractmethod
from functools import partial

from PyQt5.QtWidgets import (QHBoxLayout, QSizePolicy,
                             QVBoxLayout, QWidget,
                             QComboBox, QLineEdit,
                             QMessageBox, QTabWidget,
                             QLabel, QStackedWidget)

import pt_version

import util.gui_util as gui_util
import util.excp as app_excp

from design.db import PeriDesign
from design.db_item import PeriDesignGenPinItem, GenericPin, GenericParamService

from common_gui.spec_builder import SpecBuilder, WidgetSpec
from common_gui.builder.primitive_widget import PTWidgetBuilder
from common_gui.builder.base import ParamWidget, PinEditorWidget
from common_gui.base_config import BaseConfig
from common_gui.builder.smart_widget import GenericPinWidgetBuilderV2

from common_device.property import PropertyMetaData
from common_device.quad.lane_design import LaneBasedItem, LaneBaseRegistry
from common_device.quad.res_service import QuadResService
from common_device.quad.gui.builder import RefClkGpBoxBuilder, LaneTabBuilder
from common_device.quad.gui.lane_presenter import LaneBasedConfigUIPresenter
from common_device.quad.gui.graph_observer import LaneBasedGraphObserver, BaseCommonQuadGraphObserver

from tx375_device.common_quad.design_param_info import QuadDesignParamInfo, get_supported_params_by_quad_type as get_sw_supported_cmn_parm
from tx375_device.common_quad.quad_prop_id import CommonQuadConfigParamInfo as CommonQuadParamInfo
from tx375_device.common_quad.design import QuadLaneCommonRegistry, QuadLaneCommon


FILTER_PROP_FUNC = Callable[[PropertyMetaData], List[PropertyMetaData.PropData]]
FILTER_PIN_FUNC = Callable[[PeriDesignGenPinItem], List[GenericPin]]

IS_SHOW_HIDDEN = True if pt_version.PT_DEBUG_VERSION == True else False



class BaseLaneConfig(BaseConfig):
    """
    Controller for specific instance configuration
    """

    def __init__(self, parent):
        super().__init__(parent)
        self.parent: QWidget

        self.res_svc = QuadResService()

        # Qt widget
        self.tabw_config: QTabWidget = parent.tabw_config
        self.tabw_config.tabBar().setUsesScrollButtons(False)
        self.tab_base: QWidget = parent.tab_base

        self.vl_base: QVBoxLayout = parent.vl_base

        # Ref clock related
        self.cb_ext_clk = QComboBox()
        self.lbl_refclk_res = QLabel()

        self.le_name: QLineEdit = parent.le_name
        self.cb_device: QComboBox = parent.cb_device

        self.param_widget_map: Dict[str, ParamWidget] = {}
        self.common_param_widget_map: Dict[str, ParamWidget] = {}
        self.pin_widget_map: Dict[str, PinEditorWidget] = {}
        self.common_pin_widget_map: Dict[str, PinEditorWidget] = {}

        self.graph_observer = self.build_blk_graph_observer()
        self.common_graph_observer = self.build_cmn_graph_observer()
        self.presenter: Optional[LaneBasedConfigUIPresenter]
        self.block_inst: Optional[LaneBasedItem]
        self.block_reg: LaneBaseRegistry
        self.common_inst: Optional[QuadLaneCommon] = None
        self.is_monitor_dep_graph = False

        # Widget builder related
        self.is_built_widget = False
        self.category2tab_page: Dict[str, QWidget] = {}
        self.category2vl: Dict[str, QVBoxLayout] = {}
        self.parent_category2sub_tabw: Dict[str, QTabWidget] = {}
        self.common_quad_reg: Optional[QuadLaneCommonRegistry] = None
        self.lbl_common_inst_name: Optional[QLabel] = None
        self.le_common_name: Optional[QLineEdit] = None
        self.refclk_stack_widget: Optional[QStackedWidget] = None
        self.refclk2vl: Dict[str, QVBoxLayout] = {}
        self.refclk2pg: Dict[str, QWidget] = {}
        self.pll_clk_widget_map: Dict[str, QWidget] = {}
        self.vl_ext_clk = QVBoxLayout()
        self.category2filter_func: Dict[str, Tuple[
            Optional[FILTER_PROP_FUNC], Optional[FILTER_PIN_FUNC]]] = {}
        self.category2keep_filter_order: Dict[str, Tuple[bool, bool]] = {}

    @abstractmethod
    def get_design_block_type(self) -> PeriDesign.BlockType:
        raise NotImplementedError

    @abstractmethod
    def get_ref_clk_cmn_params(self) -> List[CommonQuadParamInfo.Id| QuadDesignParamInfo.Id]:
        raise NotImplementedError

    @abstractmethod
    def get_hidden_parameters(self) -> List[str]:
        raise NotImplementedError

    def get_skip_parameters(self) -> List[str]:
        return ['ss_topology_lane_NID']

    @abstractmethod
    def build_blk_graph_observer(self) -> LaneBasedGraphObserver:
        raise NotImplementedError

    @abstractmethod
    def build_cmn_graph_observer(self) -> BaseCommonQuadGraphObserver:
        raise NotImplementedError

    @abstractmethod
    def build_presenter(self) -> LaneBasedConfigUIPresenter:
        raise NotImplementedError

    @abstractmethod
    def build_tab_builder(self) -> LaneTabBuilder:
        raise NotImplementedError

    def build(self, block_name: str, design: PeriDesign):
        """
        Update editor content based on currently selected block instance
        """
        self.design = design
        self.common_quad_reg = self.design.common_quad_lane_reg
        assert self.common_quad_reg is not None

        self.res_svc.build(design)

        reg = self.design.get_block_reg(self.get_design_block_type())
        assert reg is not None
        assert isinstance(reg, LaneBaseRegistry)

        inst: Optional[LaneBasedItem] = reg.get_inst_by_name(block_name) # type: ignore

        if inst is None:
            return

        ### Qt widget related ###
        if self.is_monitor:
            self.stop_monitor_changes()

        self.block_reg = reg
        self.block_inst = inst
        self.common_inst = self.block_reg.get_cmn_inst_by_lane_name(self.block_inst.name)
        assert self.common_inst is not None

        # Rebuilt
        self.disable_mouse_wheel_scroll()

        if self.presenter is None:
            self.presenter = self.build_presenter()
            self.presenter.setup_design(self.design)

        if not self.is_built_widget:
            self.clear_all_layout_widget()
            self.build_all_widgets()
            self.is_built_widget = True

        self.load_all_widgets_info()
        
        self.tabw_config.setCurrentIndex(0)
        self.monitor_changes()

    def build_all_widgets(self):
        self.build_tabs()

        self.build_category2filter_func()
        self.build_mix_widgets()
        self.build_param_widgets()
        self.build_pin_widgets()

        self.update_option_display()

    def build_tabs(self):
        # blk_inst = self.block_inst
        # print("List: ", list(blk_inst.categories))

        # Build all tabs
        builder = self.build_tab_builder()
        builder.build(IS_SHOW_HIDDEN)

        self.category2tab_page.update(builder.category2tabs)
        self.category2vl.update(builder.category2vl)
        self.parent_category2sub_tabw = builder.parent_category2sub_tabw

        # Add param tab
        for category in builder.param_categories:
            tab = builder.category2tabs.get(category)
            if tab is None or tab.parent() is not None:
                continue

            builder.add_tab(tab, self.tabw_config, category)

    def build_mix_widgets(self) -> None:
        pass

    def build_category2filter_func(self):
        pass

    def build_param_widgets(self) -> None:
        inst = self.block_inst
        assert inst is not None and self.presenter is not None

        cmn_inst = self.common_inst
        assert cmn_inst is not None

        param_only_categories = [
            "Per Lane Parameters - Control Register "
        ]
        tab_builder = self.build_tab_builder()

        for category, container in self.category2vl.items():
            is_last_spacer = False
            filter_func = None
            blk = inst
            widget_map = self.param_widget_map

            if category in tab_builder.skip_categories:
                continue

            if category in param_only_categories:
                is_last_spacer = True
            elif category in tab_builder.parent_category2children.keys():
                is_last_spacer = False
            elif category in tab_builder.common_categories2display_name.keys():
                blk = cmn_inst
                widget_map = self.common_param_widget_map

            # Special filter
            if category in self.category2filter_func:
                filter_func = self.category2filter_func[category][0]

            keep_filter_order = self.category2keep_filter_order.get(category, (False, False))[0]

            widget_map.update(
                self.gen_param_widgets_by_builder(blk,
                                    container,
                                    category=category,
                                    filter_func=filter_func, # type: ignore
                                    keep_filter_order=keep_filter_order,
                                    is_last_spacer=is_last_spacer
                                    )
            )

        # Common params
        self.build_cmn_param_widgets()

    def build_cmn_param_widgets(self):
        cmn_inst = self.common_inst
        assert cmn_inst is not None

        # Common instance Name
        category = "Common Parameters"
        cmn_container = self.category2vl.get(category)
        assert cmn_container is not None
        self.lbl_common_inst_name, self.le_common_name = self.setup_ln_editor("common_pin_name", "Common Instance Name")
        assert self.lbl_common_inst_name is not None and self.le_common_name is not None

        cmn_container.addWidget(self.lbl_common_inst_name)
        cmn_container.addWidget(self.le_common_name)

        self.common_param_widget_map.update(
            self.gen_param_widgets_by_builder(cmn_inst,
                                cmn_container,
                                category=category,
                                filter_func=self.get_cmn_param_info,# type: ignore
                                keep_filter_order=True,
                                is_last_spacer=False
                                )
        )

        # Reference clock
        ref_clk_gpbox = self.build_ref_clk_widget()
        cmn_container.addWidget(ref_clk_gpbox)

        widget_builder = PTWidgetBuilder()
        widget_builder.add_spacer(cmn_container)

    def build_ref_clk_widget(self):
        assert self.block_inst is not None and self.common_inst is not None

        # Build all reference clock widgets
        gpbox_builder = RefClkGpBoxBuilder()
        gpbox_builder.build()

        self.setup_ref_clk_widget(gpbox_builder)

        # Param from Common instance
        def common_ref_clk_filter(param_info: PropertyMetaData):
            return param_info.get_prop_info_by_id_list(self.get_ref_clk_cmn_params())

        # Load reference clock param in reference clock group box
        category = "ref_clk"
        container = gpbox_builder.parentcategory2layout.get(category)
        assert container is not None
        self.category2vl[category] = container

        inst2filter = [
            (self.block_inst, self.get_blk_ref_freq_param_info, self.param_widget_map),
            (self.common_inst, common_ref_clk_filter, self.common_param_widget_map),
        ]

        for inst, filter_func, widget_map in inst2filter:
            widget_map.update(
                self.gen_param_widgets_by_builder(inst,
                                    container,
                                    category=category,
                                    filter_func=filter_func, # type: ignore
                                    keep_filter_order=True,
                                    is_last_spacer=False
                                    )
            )

        def common_ext_ref_clk_filter_after(param_info: PropertyMetaData):
            return [
                param_info.get_prop_info(
                    CommonQuadParamInfo.Id.PIPE_CONFIG_CMN__config_reg_2__pma_cmn_refclk_termen),
                param_info.get_prop_info(
                    CommonQuadParamInfo.Id.PIPE_CONFIG_CMN__config_reg_2__pma_cmn_refclk1_termen),
            ]

        ref_clk_gpbox, _ = gpbox_builder.category2gp_box_info.get(category, (None, None))
        assert ref_clk_gpbox is not None

        self.vl_ext_clk = gpbox_builder.category2stack_layout.get("External")
        assert self.vl_ext_clk is not None

        # Load external reference clock parameters
        container = self.vl_ext_clk
        category = "External Clock"
        self.common_param_widget_map.update(
            self.gen_param_widgets_by_builder(self.common_inst,
                                container,
                                category=category,
                                filter_func=common_ext_ref_clk_filter_after, # type: ignore
                                keep_filter_order=True,
                                is_last_spacer=True
                                )
        )

        return ref_clk_gpbox

    def build_pin_widgets(self):
        assert self.block_inst is not None and self.common_quad_reg is not None
        assert self.res_svc is not None
        assert self.common_inst is not None

        common_postfix = self.res_svc.get_common_class_postfix()
        apb_top_pin = [
            "USER_APB_CLK"
        ]
        pin_class = f"APB{common_postfix}"
        container = self.category2vl.get(pin_class, None)
        assert self.res_svc is not None

        device = self.block_inst.get_device()
        usable_pin_types = []

        if device != "":
            usable_pin_types = self.res_svc.get_usable_pin_types_by_res_name(device, self.block_inst.quad_type)

        def _is_pin_type_usable(pin_type: str):
            return pin_type in usable_pin_types if len(usable_pin_types) > 0 else True

        def apb_filter_pin(blk: PeriDesignGenPinItem):
            pins = [pin for pin in blk.get_pin_by_class(f"APB{common_postfix}") \
                if pin.type_name not in apb_top_pin and _is_pin_type_usable(pin.type_name)]
            return pins

        def common_filter_pin(blk: PeriDesignGenPinItem, pin_class: str):
            pins = [pin for pin in blk.get_pin_by_class(pin_class) if _is_pin_type_usable(pin.type_name)]
            return pins

        # APB top pins
        if container is not None:
            is_last_spacer = False
            filter_func = lambda b: [b.gen_pin.get_pin_by_type_name(pin) for pin in apb_top_pin]

            self.common_pin_widget_map.update(
                self.gen_pin_widgets_by_builder(self.common_inst,
                                    container,
                                    usable_pin_types,
                                    pin_class,
                                    is_laster_spacer=is_last_spacer,
                                    filter_func=filter_func,
            ))


        tab_builder = self.build_tab_builder()
        # Other pin class
        for pin_class, container in self.category2vl.items():
            is_last_spacer = True
            filter_func = None

            if pin_class in tab_builder.parent_category2children.keys():
                is_last_spacer = False
            if pin_class in tab_builder.common_categories2display_name:
                continue

            if pin_class in self.category2filter_func:
                filter_func = self.category2filter_func[pin_class][1]

            self.pin_widget_map.update(
                self.gen_pin_widgets_by_builder(self.block_inst,
                                  container,
                                  usable_pin_types,
                                  pin_class,
                                  is_laster_spacer=is_last_spacer,
                                  filter_func=filter_func
            ))

        # Common pin class
        for pin_class in tab_builder.common_categories2display_name.keys():
            is_last_spacer = True
            filter_func = None

            if not IS_SHOW_HIDDEN and pin_class in tab_builder.hidden_categories:
                continue

            container = self.category2vl.get(pin_class, None)
            assert container is not None, f"Can't find category {pin_class}"

            if pin_class in tab_builder.parent_category2children.keys():
                is_last_spacer = False
            if pin_class.startswith("APB"):
                filter_func = apb_filter_pin
            else:
                filter_func = lambda inst, pin_class=pin_class: common_filter_pin(inst, pin_class)

            self.common_pin_widget_map.update(
                self.gen_pin_widgets_by_builder(
                    self.common_inst,
                    container,
                    usable_pin_types,
                    pin_class,
                    is_laster_spacer=is_last_spacer,
                    filter_func=filter_func
            ))

    def clear_all_layout_widget(self):
        vl_list = list(self.category2vl.values())
        for vl in vl_list:
            gui_util.remove_layout_children(vl)

        self.param_widget_map.clear()
        self.common_param_widget_map.clear()
        self.pin_widget_map.clear()
        self.common_pin_widget_map.clear()

    def load_all_widgets_info(self):
        assert self.presenter is not None and self.block_inst is not None
        assert self.common_inst is not None

        if not IS_SHOW_HIDDEN:
            self.hide_unsupported_param()

        self.presenter.load_base_property(self.block_inst, self.le_name, self.cb_device)
        assert self.le_common_name is not None
        self.presenter.load_common_name(self.common_inst, self.le_common_name)
        self.presenter.load_ext_ref_clk_res(self.common_inst, self.cb_ext_clk, IS_SHOW_HIDDEN)
        self.presenter.load_pll_ref_clk_res(self.common_inst, self.pll_clk_widget_map, IS_SHOW_HIDDEN)

        # Trigger the graph observer to synchronize the param / pin availablity
        self.graph_observer.on_graph_updated(self.block_inst.dependency_graph,
                                                updated_nodes=list(self.param_widget_map.keys()) + \
                                                    list(self.pin_widget_map.keys()))
        self.common_graph_observer.on_graph_updated(self.common_inst.dependency_graph,
                                                updated_nodes=list(self.common_param_widget_map.keys()) + \
                                                    list(self.common_pin_widget_map.keys()))
        self.presenter.load_ref_clk_property(self.block_inst, None, self.lbl_refclk_res,
                CommonQuadParamInfo.Id.PMA_CMN__cmn_plllc_gen_preg__cmn_plllc_pfdclk1_sel_preg)

        gui_util.set_child_combo_spinbox_filter(self)

    def setup_ln_editor(self, name: str, display_name: str):
        lbl_widget = QLabel(display_name)
        lbl_widget.setObjectName(f"lbl_{name}")

        le_widget = QLineEdit()
        le_widget.setObjectName(f"le_{name}")
        le_widget.setClearButtonEnabled(True)
        le_widget.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Fixed)
        return lbl_widget, le_widget

    def get_cmn_param_info(self, param_info: PropertyMetaData):
        return [param for param in param_info.get_prop_by_category("Common Parameters")\
            if not param.id.name.startswith('pcr') and "Reference Clock" not in param.disp_name]

    def get_blk_ref_freq_param_info(self, param_info: PropertyMetaData):
        return []

    def get_label_display_name(self, widget_spec: WidgetSpec, prop_info: PropertyMetaData.PropData):
        return widget_spec.display_name

    def get_inst_resource(self):
        """
        Get resource name of instance
        """
        assert self.block_inst is not None
        return self.block_inst.get_device()

    def setup_ref_clk_widget(self, gpbox_builder: RefClkGpBoxBuilder):
        """
        Load widget from the builder
        """
        self.refclk2pg = gpbox_builder.category2pg

        # Stack widget
        refclk_stack_widget = gpbox_builder.category2stack_widget.get('ref_clk')
        assert refclk_stack_widget is not None

        self.refclk_stack_widget = refclk_stack_widget

        # Load External Clock editable widget
        lbl_res = gpbox_builder.ref_clk_widget_list.get(gpbox_builder.LBL_RES_NAME)
        assert isinstance(lbl_res, QLabel)
        self.lbl_refclk_res = lbl_res

        cb_ext_clk = gpbox_builder.ref_clk_widget_list.get(gpbox_builder.CB_EXT_CLK_OBJ_NAME)
        assert isinstance(cb_ext_clk, QComboBox)
        self.cb_ext_clk = cb_ext_clk

        # Load PLL Reference Clock Widget
        widget_list = [
            gpbox_builder.CB_PLL_CLK_OBJ_NAME,
            gpbox_builder.LBL_PLL_CLK_RES,
            gpbox_builder.LBL_PLL_CLK_INST,
        ]
        for idx in range(RefClkGpBoxBuilder.PLL_CLK_NUM):
            for widget_name in widget_list:
                cb_widget = gpbox_builder.ref_clk_widget_list.get(widget_name + str(idx))
                assert isinstance(cb_widget, QWidget), f"Fail to find {widget_name + str(idx)}"
                self.pll_clk_widget_map[cb_widget.objectName()] = cb_widget

    def hide_unsupported_param(self):
        assert self.presenter is not None and self.block_inst is not None
        assert self.common_inst is not None

        for param_name in self.presenter.get_hidden_param_values().keys():
            if self.graph_observer.get_hw_param_id_class().has_member(param_name): # type: ignore
                param_id = self.graph_observer.get_hw_param_id_class()(param_name)
                widget_group = self.param_widget_map.get(param_id.value, None)
                blk = self.block_inst
            else:
                assert CommonQuadParamInfo.Id.has_member(param_name)
                param_id = CommonQuadParamInfo.Id(param_name)
                widget_group = self.common_param_widget_map.get(param_id.value, None)
                blk = self.common_inst

            if widget_group is None:
                self.logger.debug(f"widget_group is not built, param_name= {param_name}")
                continue

            widget = widget_group.editor_widget
            assert isinstance(widget, QComboBox)
            self.presenter.updated_param_option(blk, param_id, widget)

    def update_option_display(self):
        assert self.presenter is not None
        self.presenter.update_all_option_display(self.param_widget_map, IS_SHOW_HIDDEN)

    @pyqtSlot()
    def on_param_changed(self):
        assert self.block_inst is not None
        assert self.presenter is not None

        widget = self.sender()
        assert isinstance(widget, QWidget)

        if self.is_monitor:
            # Keep dependency graph monitor changes
            self.is_monitor_dep_graph = False
            self.stop_monitor_changes()
            self.is_monitor_dep_graph = True

        if self.presenter.save_generic_param_property(self.block_inst, widget):
            self.sig_prop_updated.emit()

        self.monitor_changes()

    @pyqtSlot()
    def on_common_param_changed(self):
        assert self.common_inst is not None
        assert self.presenter is not None
        assert self.block_inst is not None

        widget = self.sender()
        assert isinstance(widget, QWidget)

        if self.is_monitor:
            # Keep dependency graph monitor changes
            self.is_monitor_dep_graph = False
            self.stop_monitor_changes()
            self.is_monitor_dep_graph = True

        if self.presenter.save_generic_param_property(self.common_inst, widget):
            self.sig_prop_updated.emit()

        self.monitor_changes()

    @pyqtSlot(int)
    def on_ext_clk_changed(self, index: int):
        assert self.common_inst is not None
        assert self.block_inst is not None
        assert self.presenter is not None

        sender = self.sender()
        assert isinstance(sender, QComboBox)

        if self.is_monitor:
            # Keep dependency graph monitor changes
            self.is_monitor_dep_graph = False
            self.stop_monitor_changes()
            self.is_monitor_dep_graph = True

        ext_clk = sender.itemText(index)
        ext_clk_value = self.presenter.get_ext_clk_name(ext_clk)
        param_service = GenericParamService(self.common_inst.get_param_group(),
                                            self.common_inst.get_param_info())
        param_id = CommonQuadParamInfo.Id.PMA_CMN__cmn_plllc_gen_preg__cmn_plllc_pfdclk1_sel_preg
        param_service.set_param_value(param_id, ext_clk_value)
        self.presenter.load_ref_clk_property(self.block_inst, None, self.lbl_refclk_res, param_id)
        self.sig_prop_updated.emit()

        self.monitor_changes()

    @pyqtSlot()
    def on_pin_changed(self):
        assert self.block_inst is not None
        assert self.presenter is not None

        widget = self.sender()
        if self.presenter.save_generic_pin_property(self.block_inst, widget):
            self.sig_prop_updated.emit()

    @pyqtSlot(int)
    def on_clkpin_invert_changed(self, state):
        assert self.block_inst is not None
        assert self.presenter is not None

        widget = self.sender()

        if self.presenter.save_generic_pin_clk_invert_property(self.block_inst, widget):
            self.sig_prop_updated.emit()

    @pyqtSlot()
    def on_common_pin_changed(self):
        assert self.common_inst is not None
        assert self.presenter is not None

        widget = self.sender()
        if self.presenter.save_generic_pin_property(self.common_inst, widget):
            self.sig_prop_updated.emit()

    @pyqtSlot(int)
    def on_common_clkpin_invert_changed(self, state):
        assert self.common_inst is not None
        assert self.presenter is not None

        widget = self.sender()

        if self.presenter.save_generic_pin_clk_invert_property(self.common_inst, widget):
            self.sig_prop_updated.emit()

    @pyqtSlot()
    def on_device_changed(self):
        if self.block_inst is None:
            return

        assert self.presenter is not None and self.res_svc is not None
        cb_device = self.sender()

        assert isinstance(cb_device, QComboBox)
        self.stop_monitor_changes()
        new_device = cb_device.currentText()
        cur_device = self.block_inst.get_device()
        if new_device != "":
            if new_device == cur_device:
                self.monitor_changes()
                return

            # If the quad_pcie device name is not in the valid list, then retain
            # current assignment (No change)
            res_label = self.lbl_refclk_res

            if not self.block_reg.is_same_quad(new_device, cur_device):
                msg = "Any changes you make in the Common Properties tab "\
                    "are applied to all lanes in the same quad"
                result = QMessageBox.warning(self.parent, "Properties Update", msg,
                        QMessageBox.StandardButton.Yes|QMessageBox.StandardButton.Cancel)

                if result == QMessageBox.StandardButton.Cancel:
                    cb_device.setCurrentText(cur_device if cur_device != "" else "None")
                    self.monitor_changes()
                    return

            if new_device == "None":
                self.block_reg.reset_inst_device(self.block_inst)
                res_label.setText("Unknown")
            else:
                quad_res_name = self.res_svc.translate_lane_res2quad_res_name(new_device)

                # Check both quad/lane based registers
                if self.res_svc.is_resource_used(new_device, is_exact=True) or\
                    self.res_svc.is_resource_used(quad_res_name, is_exact=True):
                    msg = "Resource {} is already used. Please choose another.".format(new_device)
                    QMessageBox.warning(self.parent, "Update resource", msg) # type: ignore
                    self.logger.error("Fail to assign resource, it has been used")

                    # Revert to original name before edit
                    cb_device.setCurrentText(cur_device if cur_device != "" else "None")
                    self.monitor_changes()
                    return

                self.block_reg.assign_inst_device(self.block_inst, new_device)

            assert self.common_quad_reg is not None
            self.common_inst = self.block_reg.get_cmn_inst_by_lane_name(self.block_inst.name)

            # Reload information
            self.disable_mouse_wheel_scroll()

            self.load_all_widgets_info()

            self.sig_prop_updated.emit()
            self.sig_resource_updated.emit(new_device)
            self.monitor_changes()

    def update_ref_clk_property(self):
        if self.block_inst is None:
            return

        assert isinstance(self.block_inst, LaneBasedItem)
        res_label = self.lbl_refclk_res
        assert self.presenter is not None

        param_id = CommonQuadParamInfo.Id.PMA_CMN__cmn_plllc_gen_preg__cmn_plllc_pfdclk1_sel_preg
        self.presenter.load_ref_clk_property(self.block_inst, None, res_label, param_id)
        self.sig_prop_updated.emit()

    def monitor_dep_graph(self):
        assert self.block_inst is not None
        assert self.common_inst is not None

        if not self.is_monitor_dep_graph:
            self.is_monitor_dep_graph = True
            self.block_inst.dependency_graph.register_observer(self.graph_observer)
            self.common_inst.dependency_graph.register_observer(self.common_graph_observer)

    def monitor_changes(self):
        self.monitor_dep_graph()
        self.le_name.editingFinished.connect(self.on_name_changed)
        assert self.le_common_name is not None
        self.le_common_name.editingFinished.connect(self.on_common_name_changed)
        self.cb_device.currentIndexChanged.connect(self.on_device_changed)

        self.monitor_param_widget(self.param_widget_map, True, self.on_param_changed)
        self.monitor_param_widget(self.common_param_widget_map, True, self.on_common_param_changed)
        self.monitor_pin_widget(self.pin_widget_map, True, self.on_pin_changed)
        self.monitor_pin_widget(self.common_pin_widget_map, True, self.on_common_pin_changed)
        self.monitor_clk_pin_inversion_widget(self.pin_widget_map, True, self.on_clkpin_invert_changed)
        self.monitor_clk_pin_inversion_widget(self.common_pin_widget_map, True, self.on_common_clkpin_invert_changed)
        self.monitor_ref_clk_widget(True)

        self.is_monitor = True

    @pyqtSlot()
    def on_common_name_changed(self):
        le_name = self.sender()
        assert isinstance(le_name, QLineEdit)
        le_name.blockSignals(True)

        assert self.common_inst is not None
        name = le_name.text()
        if name == self.common_inst.name:
            le_name.blockSignals(False)
            return

        is_valid = self.validator.is_name_valid(name)
        if is_valid:
            assert self.common_quad_reg is not None
            assert self.block_inst is not None
            if self.common_quad_reg.is_inst_name_used(name):
                msg = "Common instance name {} is already used. Please choose another name.".format(name)
                QMessageBox.warning(self.parent, "Update instance name", msg) # type: ignore
                self.logger.error("Fail to rename common instance, name has been used")

                # Revert to original name before edit
                le_name.setText(self.common_inst.name)
                le_name.blockSignals(False)
                return

            try:
                current_name = self.common_inst.name
                self.common_inst = self.common_quad_reg.rename_inst(current_name, name, auto_pin=True) # type: ignore
                self.sync_pin_display()

                self.sig_name_updated.emit(self.block_inst.name)
                le_name.setToolTip(name)
                gui_util.change_widget_bg_colour(
                    le_name, gui_util.get_valid_config_colour())

            except app_excp.PTException:
                self.logger.warning(
                    "Fail to update common name change to {}".format(name))
        else:
            le_name.setToolTip(self.validator.msg)
            gui_util.change_widget_bg_colour(
                le_name, gui_util.get_invalid_config_colour())

        le_name.blockSignals(False)

    def monitor_ref_clk_widget(self, is_monitor: bool):
        if is_monitor:
            self.cb_ext_clk.currentIndexChanged.connect(self.on_ext_clk_changed)
        else:
            self.cb_ext_clk.currentIndexChanged.disconnect(self.on_ext_clk_changed)

    def stop_monitor_dep_graph(self):
        assert self.block_inst is not None
        assert self.common_inst is not None

        if self.is_monitor_dep_graph:
            self.is_monitor_dep_graph = False
            self.block_inst.dependency_graph.unregister_observer(self.graph_observer)
            self.common_inst.dependency_graph.unregister_observer(self.common_graph_observer)

    def stop_monitor_changes(self):
        self.stop_monitor_dep_graph()

        self.le_name.editingFinished.disconnect(self.on_name_changed)
        assert self.le_common_name is not None
        self.le_common_name.editingFinished.disconnect(self.on_common_name_changed)
        self.cb_device.currentIndexChanged.disconnect(self.on_device_changed)

        self.monitor_param_widget(self.param_widget_map, False, self.on_param_changed)
        self.monitor_param_widget(self.common_param_widget_map, False, self.on_common_param_changed)
        self.monitor_pin_widget(self.pin_widget_map, False, self.on_pin_changed)
        self.monitor_pin_widget(self.common_pin_widget_map, False, self.on_common_pin_changed)
        self.monitor_clk_pin_inversion_widget(self.pin_widget_map, False, self.on_clkpin_invert_changed)
        self.monitor_clk_pin_inversion_widget(self.common_pin_widget_map, False, self.on_common_clkpin_invert_changed)
        self.monitor_ref_clk_widget(False)

        self.is_monitor = False

    def sync_pin_display(self):
        """
        Refresh any widget that identifies pin name with design data
        """
        if self.block_inst is not None:

            if isinstance(self.block_inst, LaneBasedItem):
                pin_widget_map = self.pin_widget_map
            else:
                self.logger.error(f"Unknown block instance: {type(self.block_inst)}")
                return

            gen_pin = self.block_inst.gen_pin

            all_pin_list = gen_pin.get_all_pin()
            for pin in all_pin_list:
                if isinstance(pin, GenericPin):
                    pin_widget = pin_widget_map.get(pin.type_name, None)
                    if pin_widget is not None:
                        gui_util.set_config_widget(pin_widget.editor_widget, pin.name, pin.name)

            if self.common_quad_reg is not None:
                # Common pin widget
                pin_widget_map = self.common_pin_widget_map
                common_inst = self.common_inst
                assert common_inst is not None

                gen_pin = common_inst.gen_pin

                all_pin_list = gen_pin.get_all_pin()
                for pin in all_pin_list:
                    if isinstance(pin, GenericPin):
                        pin_widget = pin_widget_map.get(pin.type_name, None)
                        if pin_widget is not None:
                            gui_util.set_config_widget(pin_widget.editor_widget, pin.name, pin.name)

    def gen_param_widgets_by_builder(self,
            block_inst: LaneBasedItem| QuadLaneCommon, container, category: Optional[str] = None,
            filter_func: Optional[FILTER_PROP_FUNC] = None, keep_filter_order: bool = False,
            is_last_spacer=False) -> Dict[str, ParamWidget]:
        assert (category is None and filter_func is not None) or (category is not None)

        spec_builder = SpecBuilder()
        widget_builder = PTWidgetBuilder()
        param_info = block_inst.get_param_info()
        param_group = block_inst.get_param_group()

        if category is not None and filter_func is None:
            def get_prop(p: PropertyMetaData, c: str):
                return p.get_prop_by_category(c)
            filter_func = partial(get_prop, c=category)

        assert filter_func is not None
        is_common = isinstance(block_inst, QuadLaneCommon)
        assert self.block_inst is not None

        for idx, prop_info in enumerate(filter_func(param_info)):
            # Skip building hidden param
            if not IS_SHOW_HIDDEN and prop_info.name in self.get_hidden_parameters():
                continue
            elif prop_info.name in self.get_skip_parameters():
                continue

            if is_common and \
                not (prop_info.name in self.block_inst.get_supported_common_parameters() or \
                    prop_info.id in get_sw_supported_cmn_parm(self.block_inst.quad_type)):
                continue

            param = param_group.get_param_by_name(prop_info.name)
            assert param is not None
            widget_spec = spec_builder.build_single_param_widget_spec(param, param_info)
            assert widget_spec is not None, f"{prop_info.name} is None"
            widget_spec.display_name = self.get_label_display_name(widget_spec, prop_info)

            if keep_filter_order:
                widget_spec.seqno = idx
            widget_builder.add_widget_spec(widget_spec)

        return widget_builder.build(container, is_sort=True, is_sort_seqno=keep_filter_order, is_last_spacer=is_last_spacer)

    def gen_pin_widgets_by_builder(self, block_inst: PeriDesignGenPinItem, container: QVBoxLayout| QHBoxLayout, usable_pin_types: List[str],
                        category: Optional[str] = None, filter_func: Optional[FILTER_PIN_FUNC] = None,
                        is_laster_spacer: bool = False) -> Dict[str, PinEditorWidget]:
        assert hasattr(block_inst, 'gen_pin')
        assert (category is None and filter_func is not None) or (category is not None)

        pin_builder = GenericPinWidgetBuilderV2()
        spec_builder = SpecBuilder()

        if category is not None and filter_func is None:
            def get_pin(b: PeriDesignGenPinItem, c: str):
                return b.get_pin_by_class(c)
            filter_func = partial(get_pin, c=category)

        assert filter_func is not None
        pins = filter_func(block_inst)

        hidden_ports = []
        assert self.block_inst is not None
        if not IS_SHOW_HIDDEN:
            hidden_ports = self.block_inst.get_hidden_ports()

        if isinstance(block_inst, QuadLaneCommon):
            hidden_ports += block_inst.get_filter_pins_by_quad_type(self.block_inst.quad_type)

        pin_name_list = [
            p.type_name for p in pins if p.type_name not in hidden_ports and \
                (len(usable_pin_types) <= 0 or p.type_name in usable_pin_types)]
        spec_builder.add_pin_widget_spec(block_inst, pin_name_list, pin_builder)

        return pin_builder.build(container, is_sort=True, is_last_spacer=is_laster_spacer)

    def gen_mixed_widget_by_builder(self, block_inst: LaneBasedItem,
                            container: QVBoxLayout| QHBoxLayout,
                            pins: List[GenericPin],
                            params: List[PropertyMetaData.PropData],
                            usable_pin_types: List[str],
                            is_last_spacer=False) -> Tuple[
                                Dict[str, PinEditorWidget],
                                Dict[str, ParamWidget]
                            ] :
        """
        Build A Grouped Widget where pin and parameters widget put together
        """
        assert len(pins) > 0
        assert len(params) > 0

        pin_widgets_map = self.gen_pin_widgets_by_builder(block_inst,
                                            container,
                                            filter_func=lambda block_inst: pins,
                                            usable_pin_types=usable_pin_types
                                            )

        params_widgets_map = self.gen_param_widgets_by_builder(block_inst,
                                                container,
                                                filter_func=lambda param_info: params
                                                )

        if is_last_spacer:
            builder = PTWidgetBuilder()
            builder.add_spacer(container)

        return pin_widgets_map, params_widgets_map
