from __future__ import annotations
from typing import TYPE_CHECKING, List, Type, Set
from abc import abstractmethod

from PyQt5.QtWidgets import QCheckBox, QLabel, QComboBox, QTabWidget
import pt_version

from tx375_device.common_quad.design import QuadLaneCommon
from tx375_device.common_quad.quad_param_info import CommonQuadParamInfo as QuadConfigInfo
from tx375_device.common_quad.design_param_info import QuadDesignParamInfo as QuadParamInfo

from common_device.quad.gui.builder import RefClkGpBoxBuilder
from common_device.quad.lane_design import LaneBasedItem

from common_gui.base_graph_observer import AbstractGraphObserver

from design.db_item import GenericParamService

if TYPE_CHECKING:
    from enum import Enum
    from common_gui.base_dep_graph import DependencyGraph

    from common_device.quad.gui.lane_config import BaseLaneConfig
    from tx375_device.common_quad.dep_graph import QuadCommonDependencyGraph

IS_SHOW_HIDDEN = True if pt_version.PT_DEBUG_VERSION == True else False

class LaneBasedGraphObserver(AbstractGraphObserver):

    def __init__(self, config: BaseLaneConfig) -> None:
        super().__init__()
        self._config = config

    def on_graph_updated(self, graph: DependencyGraph, updated_nodes: List[str]):
        assert self._config.presenter is not None
        block_inst = self._config.block_inst
        assert block_inst is not None
        updated_params = set()
        updated_pins = set()
        for node in updated_nodes:
            if node in self._config.param_widget_map:
                updated_params.add(node)
                widget = self._config.param_widget_map[node]
                is_available = graph.get_param_attributes(node)["is_available"]
                widget.set_visible(is_available)

            if node in self._config.pin_widget_map:
                updated_pins.add(node)
                widget = self._config.pin_widget_map[node]
                pin_attr = graph.get_pin_attributes(node)
                was_available = pin_attr["was_available"]
                is_available = pin_attr["is_available"]

                if was_available is False and is_available is True:
                    # Regenerate the pin names that are empty when enabled.  When the design is
                    # loaded, the pin names can be empty when the feature is disabled.
                    if is_available and block_inst.gen_pin is not None:
                        block_inst.generate_specific_pin_name_from_inst(
                            node, is_only_empty_name=True
                        )

                widget.set_visible(is_available)

        self.update_specific_widget(block_inst, updated_params, updated_pins)

        for param_name in updated_params:
            self.update_param_widget_highlighting(param_name)

        excluded_params = set(self._config.param_widget_map.keys()).intersection(updated_params)
        self._config.presenter.load_generic_param_property(block_inst, self._config.param_widget_map, exclude_type=excluded_params)
        self._config.presenter.load_generic_pin_property(block_inst, self._config.pin_widget_map)

    def update_specific_widget(self, block_inst: LaneBasedItem, updated_params: Set[str], updated_pins: Set[str]):
        pass

    @abstractmethod
    def get_sw_param_id_class(self) -> Type[Enum]:
        raise NotImplemented

    @abstractmethod
    def get_hw_param_id_class(self) -> Type[Enum]:
        raise NotImplemented

    def update_param_widget_highlighting(self, param_name: str):
        # Block info
        block_inst = self._config.block_inst
        assert isinstance(block_inst, LaneBasedItem)
        param_service = GenericParamService(block_inst.get_param_group(), block_inst.get_param_info())

        if self.get_sw_param_id_class().has_member(param_name): # type: ignore
            param_id = self.get_sw_param_id_class()(param_name)
        else:
            param_id = self.get_hw_param_id_class()(param_name)

        # Get widget
        widget_group = self._config.param_widget_map.get(param_name, None)

        if widget_group is None:
            return

        widget = widget_group.editor_widget

        if widget is None or isinstance(widget, (QCheckBox, QComboBox)):
            return

        # Check valid value
        is_valid, msg = param_service.check_param_valid(param_id)

        widget_class = widget.__class__.__name__
        if not is_valid:
            widget.setStyleSheet(f"{widget_class} {{ background-color: rgba(255, 0, 0, 10); }}")
        else:
            widget.setStyleSheet(f"{widget_class} {{ background-color: white; }}")

        widget.update()

    def is_any_pin_available(self, category):
        block_inst = self._config.block_inst
        assert block_inst is not None

        is_pin_available = False
        hidden_ports = [] if IS_SHOW_HIDDEN else block_inst.get_hidden_ports()

        for pin_type in block_inst.get_pin_type_by_class(category):
            if block_inst.is_pin_available(pin_type) and pin_type not in hidden_ports:
                is_pin_available = True
                break

        return is_pin_available

    def set_tab_visible(self, category: str, tab_widget: QTabWidget, is_visible: bool):
        """
        Show/ Hide tab page by given category and tab widget that contains that tab page widget.

        **Note**:
        This function will get display error when hide the last tab page in the tab widget.
        The temporary solution is to use set_tab_enable() to disable the tab page.

        :param category: category for params/ pins
        :type category: str
        :param tab_widget: Tab Widget that contain that category
        :type tab_widget: QTabWidget
        :param is_visible: True if display tab page else False
        :type is_visible: bool
        """
        tab_page = self._config.category2tab_page.get(category)

        if tab_page is None:
            return

        tab_idx = tab_widget.indexOf(tab_page)

        if tab_idx != -1:
            tab_widget.setTabEnabled(tab_idx, is_visible)
            tab_widget.setTabVisible(tab_idx, is_visible)

    def set_tab_enable(self, category: str, tab_widget: QTabWidget, is_enable: bool):
        """
        Enable tab page by given category and tab widget that contains that tab page widget.

        :param category: Quad PCIe category for params/ pins
        :type category: str
        :param tab_widget: Tab Widget that contain that category
        :type tab_widget: QTabWidget
        :param is_enable: True if enable tab page else False
        :type is_enable: bool
        """
        tab_page = self._config.category2tab_page.get(category)

        if tab_page is None:
            return

        tab_idx = tab_widget.indexOf(tab_page)

        if tab_idx != -1:
            tab_widget.setTabEnabled(tab_idx, is_enable)


class BaseCommonQuadGraphObserver(AbstractGraphObserver):

    def __init__(self, config: BaseLaneConfig) -> None:
        super().__init__()
        self._config = config

    def on_graph_updated(self, graph: QuadCommonDependencyGraph, updated_nodes: List[str]):
        assert self._config.presenter is not None
        block_inst = self._config.common_inst
        assert isinstance(block_inst, QuadLaneCommon)
        assert self._config.block_inst is not None
        quad_type = self._config.block_inst.quad_type
        updated_params = set()
        updated_pins = set()
        for prop_name in updated_nodes:
            if prop_name in self._config.common_param_widget_map:
                updated_params.add(prop_name)
                widget = self._config.common_param_widget_map[prop_name]
                node = graph.get_param_attributes(prop_name)
                is_available = node["is_available"][quad_type][self._config.block_inst.name]
                widget.set_visible(is_available)

            if prop_name in self._config.common_pin_widget_map:
                updated_pins.add(prop_name)
                widget = self._config.common_pin_widget_map[prop_name]
                pin_attr = graph.get_pin_attributes(prop_name)
                was_available = pin_attr["was_available"][quad_type]
                is_available = pin_attr["is_available"][quad_type][self._config.block_inst.name]

                if was_available is False and is_available is True:
                    # Regenerate the pin names that are empty when enabled.  When the design is
                    # loaded, the pin names can be empty when the feature is disabled.
                    if is_available and block_inst.gen_pin is not None:
                        block_inst.generate_specific_pin_name_from_inst(
                            prop_name, is_only_empty_name=True
                        )

                widget.set_visible(is_available)

        self.update_specific_widget(block_inst, updated_params, updated_pins)

        for param_name in updated_params:
            self.update_param_widget_highlighting(param_name)

        excluded_params = set(self._config.param_widget_map.keys()).intersection(updated_params)
        self._config.presenter.load_generic_param_property(block_inst, self._config.common_param_widget_map, exclude_type=excluded_params)
        self._config.presenter.load_generic_pin_property(block_inst, self._config.common_pin_widget_map)

    def update_specific_widget(self, block_inst: QuadLaneCommon,
                               updated_params: Set[str], updated_pins: Set[str]):
        if QuadConfigInfo.Id.PIPE_CONFIG_CMN__config_reg_2__pma_cmn_refclk_sel.value in \
            updated_params:
            self.update_ref_clk_widget()

    def update_param_widget_highlighting(self, param_name: str):
        # Block info
        block_inst = self._config.common_inst
        assert isinstance(block_inst, QuadLaneCommon)
        param_service = GenericParamService(block_inst.get_param_group(), block_inst.get_param_info())

        if QuadConfigInfo.Id.has_member(param_name):
            param_id = QuadConfigInfo.Id(param_name)
        else:
            param_id = QuadParamInfo.Id(param_name)

        # Get widget
        widget_group = self._config.common_param_widget_map.get(param_name, None)

        if widget_group is None:
            return

        widget = widget_group.editor_widget

        if widget is None or isinstance(widget, (QCheckBox, QComboBox)):
            return

        # Check valid value
        is_valid, msg = param_service.check_param_valid(param_id)

        widget_class = widget.__class__.__name__
        if not is_valid:
            widget.setStyleSheet(f"{widget_class} {{ background-color: rgba(255, 0, 0, 10); }}")
        else:
            widget.setStyleSheet(f"{widget_class} {{ background-color: white; }}")

        widget.update()

    def update_ref_clk_widget(self):
        block_inst = self._config.common_inst
        assert isinstance(block_inst, QuadLaneCommon)

        param_service = GenericParamService(block_inst.get_param_group(), block_inst.get_param_info())
        value = param_service.get_param_value(
            QuadConfigInfo.Id.PIPE_CONFIG_CMN__config_reg_2__pma_cmn_refclk_sel)
        assert self._config.refclk_stack_widget is not None
        page = self._config.refclk2pg.get(value, None)

        self._config.common_param_widget_map[QuadParamInfo.Id.ref_clk_pin_name.value].set_visible(
            value=="Internal")

        match value:
            case "Internal":
                self._config.refclk_stack_widget.setVisible(False)

            case "PLL":
                assert page is not None
                self._config.refclk_stack_widget.setVisible(True)
                self._config.refclk_stack_widget.setCurrentWidget(page)
                self.load_pll_ref_clk()

            case "External":
                assert page is not None
                self._config.refclk_stack_widget.setVisible(True)
                self._config.refclk_stack_widget.setCurrentWidget(page)
                self.load_ext_ref_clk()

            case _:
                self._config.refclk_stack_widget.setVisible(False)

    def load_pll_ref_clk(self):
        assert self._config.presenter is not None
        assert self._config.block_inst is not None

        param_id_list = [
            QuadParamInfo.Id.pll_ref_clk0,
            QuadParamInfo.Id.pll_ref_clk1,
        ]
        for idx, param_id in enumerate(param_id_list):
            lbl_inst = self._config.pll_clk_widget_map[RefClkGpBoxBuilder.LBL_PLL_CLK_INST + str(idx)]
            lbl_res = self._config.pll_clk_widget_map[RefClkGpBoxBuilder.LBL_PLL_CLK_RES + str(idx)]
            assert isinstance(lbl_inst, QLabel) and isinstance(lbl_res, QLabel)
            self._config.presenter.load_ref_clk_property(self._config.block_inst,
                                                        lbl_inst,
                                                        lbl_res,
                                                        param_id)

    def load_ext_ref_clk(self):
        assert self._config.presenter is not None
        assert self._config.block_inst is not None

        self._config.presenter.load_ref_clk_property(
            self._config.block_inst,
            None,
            self._config.lbl_refclk_res,
            QuadConfigInfo.Id.PMA_CMN__cmn_plllc_gen_preg__cmn_plllc_pfdclk1_sel_preg
        )

    def is_any_pin_available(self, category):
        block_inst = self._config.block_inst
        assert block_inst is not None
        common_inst = self._config.common_inst
        assert common_inst is not None

        is_pin_available = False

        hidden_ports = [] if IS_SHOW_HIDDEN else block_inst.get_hidden_ports()
        hidden_ports += common_inst.get_filter_pins_by_quad_type(block_inst.quad_type)
        hidden_ports = set(hidden_ports)

        for pin_type in common_inst.get_pin_type_by_class(category):
            if common_inst.is_pin_available(block_inst.quad_type, block_inst.name, pin_type) and \
                pin_type not in hidden_ports:
                is_pin_available = True
                break

        return is_pin_available

    def set_tab_visible(self, category: str, tab_widget: QTabWidget, is_visible: bool):
        """
        Show/ Hide tab page by given category and tab widget that contains that tab page widget.

        **Note**:
        This function will get display error when hide the last tab page in the tab widget.
        The temporary solution is to use set_tab_enable() to disable the tab page.

        :param category: category for params/ pins
        :type category: str
        :param tab_widget: Tab Widget that contain that category
        :type tab_widget: QTabWidget
        :param is_visible: True if display tab page else False
        :type is_visible: bool
        """
        tab_page = self._config.category2tab_page.get(category)

        if tab_page is None:
            return

        tab_idx = tab_widget.indexOf(tab_page)

        if tab_idx != -1:
            tab_widget.setTabEnabled(tab_idx, is_visible)
            tab_widget.setTabVisible(tab_idx, is_visible)

    def set_tab_enable(self, category: str, tab_widget: QTabWidget, is_enable: bool):
        """
        Enable tab page by given category and tab widget that contains that tab page widget.

        :param category: Quad PCIe category for params/ pins
        :type category: str
        :param tab_widget: Tab Widget that contain that category
        :type tab_widget: QTabWidget
        :param is_enable: True if enable tab page else False
        :type is_enable: bool
        """
        tab_page = self._config.category2tab_page.get(category)

        if tab_page is None:
            return

        tab_idx = tab_widget.indexOf(tab_page)

        if tab_idx != -1:
            tab_widget.setTabEnabled(tab_idx, is_enable)

