from PyQt5.QtWidgets import(
    QGroupBox, QVBoxLayout, QFormLayout, QStackedWidget, QWidget, QLayout,
    QLabel, QComboBox, QGridLayout)
from typing import Dict, List, Tuple, Type
from common_gui.builder.base import TabBuilder


class LaneTabBuilder(TabBuilder):
    param_categories: List[str] = []
    pin_categories: List[str] = []

    # Special categories
    skip_categories: List[str] = []
    common_categories2display_name: Dict[str, str] = {}

    def build(self, is_show_hidden: bool):
        self.is_show_hidden = is_show_hidden
        param_categories = self.param_categories
        pin_categories = self.pin_categories

        sub_categories = self.get_all_sub_categories()

        all_categories = param_categories + pin_categories + sub_categories
        all_categories = list(dict.fromkeys(all_categories))

        self.build_tabs_page_by_list(all_categories)

    def get_all_sub_categories(self):
        sub_categories = []

        for str_list in self.parent_category2children.values():
            sub_categories += str_list

        return sub_categories


class RefClkGpBoxBuilder:
    """
    Class for building group box and stack widget
    """
    group_box_categories: List[str] = [
        "ref_clk",
        "External",
        "PLL",
    ]
    # Children list order will change stack widget order
    category2children: Dict[str, List[str]] = {
        "ref_clk": [
            "External",
            "Internal",
            "PLL",
        ]
    }
    category2display_name: Dict[str, str] = {
        "ref_clk": "Reference Clock",
        "PLL": "PLL Clock",
        "External": "External Clock",
    }
    category2layout: Dict[str, Type[QLayout]] = {
        "External": QFormLayout,
    }
    CB_EXT_CLK_OBJ_NAME = 'cb_ext_clk'
    LBL_RES_NAME = 'lbl_res_name_list'

    # PLL
    PLL_CLK_NUM = 2
    LBL_PLL_CLK_RES = 'lbl_pll_clk_res'
    LBL_PLL_CLK_INST = 'lbl_pll_clk_inst'
    CB_PLL_CLK_OBJ_NAME = 'cb_pll_clk'

    GP_BOX_PREFIX = 'gb_'
    STACKWIDGET_PREFIX = 'sw_'
    PAGE_WIDGET_PREFIX = 'pg_'

    def __init__(self):
        self.category2gp_box_info: Dict[str, Tuple[QGroupBox, QLayout]] = dict()
        self.category2stack_layout: Dict[str, QLayout] = dict()
        self.parentcategory2layout: Dict[str, QVBoxLayout] = dict()
        self.category2pg: Dict[str, QWidget] = dict()
        self.category2stack_widget: Dict[str, QStackedWidget] = dict()
        self.ref_clk_widget_list: Dict[str, QWidget] = dict()

    def build(self):
        for category in self.group_box_categories:
            group_box, layout = self.build_group_box(category)
            self.category2gp_box_info[category] = (group_box, layout)

            if category not in self.category2children:
                continue

            # Create layout (Used for adding param to parent category)
            parent_layout = QVBoxLayout()
            if isinstance(layout, QVBoxLayout):
                layout.addLayout(parent_layout)
            else:
                layout.addChildLayout(parent_layout)
            self.parentcategory2layout[category] = parent_layout

            # Build stack widget (For child category)
            stack_widget = self.build_stack_widget(category)
            stack_widget.setParent(group_box)
            self.category2stack_widget[category] = stack_widget

            # Add pages to stack
            for child in self.category2children.get(category, []):
                page_widget, vl_pg = self.build_page_widget(child)
                stack_widget.addWidget(page_widget)
                self.category2pg[child] = page_widget
                self.category2stack_layout[child] = vl_pg

            layout.addWidget(stack_widget)
            stack_widget.setCurrentIndex(0)

        self.build_ext_ref_clk_widget()
        self.build_pll_ref_clk_widget()

    def gen_obj_name_postfix(self, category: str) -> str:
        obj_name = category.replace(" ", "_")
        return obj_name

    def get_display_name(self, category: str) -> str:
        return self.category2display_name.get(category, category)

    def build_group_box(self, category: str):
        obj_name = self.gen_obj_name_postfix(category)

        group_box = QGroupBox()
        group_box.setObjectName(f"{self.GP_BOX_PREFIX}_{obj_name}")
        group_box.setTitle(self.get_display_name(category))
        layout = self.category2layout.get(category, QVBoxLayout)()
        group_box.setLayout(layout)
        layout.setObjectName(f"vl_{obj_name}")
        return group_box, layout

    def build_stack_widget(self, category: str):
        obj_name = self.gen_obj_name_postfix(category)

        stack_widget = QStackedWidget()
        stack_widget.setObjectName(f"{self.STACKWIDGET_PREFIX}_{obj_name}")
        return stack_widget

    def build_page_widget(self, category: str):
        obj_name = self.gen_obj_name_postfix(category)

        page_widget = QWidget()
        page_widget.setObjectName(f"{self.PAGE_WIDGET_PREFIX}_{obj_name}")
        vl_pg = QVBoxLayout(page_widget)
        vl_pg.setObjectName(f"vl_{obj_name}")
        return page_widget, vl_pg

    def build_ext_ref_clk_widget(self):
        category = "External"

        self.add_ref_clk_gp_box_to_pg(category)

        _, vl_gpbox = self.category2gp_box_info.get(category, (None, None))
        assert isinstance(vl_gpbox, QFormLayout)

        rows_label_list = [
            ("clock", "Clock"),
            ("res_name", "Resource :"),
        ]
        for idx, (label, display_name) in enumerate(rows_label_list):
            label = self.gen_label_widget(label, display_name)
            vl_gpbox.setWidget(idx, QFormLayout.LabelRole, label)

        cb_ext_clk = QComboBox()
        cb_ext_clk.setObjectName(self.CB_EXT_CLK_OBJ_NAME)
        vl_gpbox.setWidget(0, QFormLayout.FieldRole, cb_ext_clk)

        res_label = self.gen_label_widget(self.LBL_RES_NAME[3:], "Unknown")
        vl_gpbox.setWidget(1, QFormLayout.FieldRole, res_label)
        self.ref_clk_widget_list[self.CB_EXT_CLK_OBJ_NAME] = cb_ext_clk
        self.ref_clk_widget_list[self.LBL_RES_NAME] = res_label

    def build_pll_ref_clk_widget(self):
        category = "PLL"

        self.add_ref_clk_gp_box_to_pg(category)

        _, vl_gpbox = self.category2gp_box_info.get(category, (None, None))
        assert isinstance(vl_gpbox, QVBoxLayout)
        vl_pll = QFormLayout()
        vl_pll.setObjectName('vl_pll_clk')
        vl_gpbox.addLayout(vl_pll)

        # Clock
        for idx in range(self.PLL_CLK_NUM):
            label = self.gen_label_widget(f'pll_clk{idx}', f"Clock {idx}")
            vl_pll.setWidget(idx, QFormLayout.LabelRole, label)

            cb_pll_clk = QComboBox()
            cb_pll_clk.setObjectName(self.CB_PLL_CLK_OBJ_NAME + str(idx))
            vl_pll.setWidget(idx, QFormLayout.FieldRole, cb_pll_clk)
            self.ref_clk_widget_list[cb_pll_clk.objectName()] = cb_pll_clk

        # Resource to Instance
        vl_pll_res2_inst = QGridLayout()
        vl_pll_res2_inst.setObjectName('vl_pll_res2_inst')
        vl_gpbox.addLayout(vl_pll_res2_inst)
        container = vl_pll_res2_inst

        pll_table_header = [
            "Resource",
            "Instance",
        ]
        for idx, header in enumerate(pll_table_header, 1):
            lbl = self.gen_label_widget(header.lower(), header)
            container.addWidget(lbl, 0, idx, 1, 1)
            self.ref_clk_widget_list[lbl.objectName()] = lbl

        # PLL, Resource, Instance
        for idx in range(self.PLL_CLK_NUM):
            lbl = self.gen_label_widget(f'pll_clk{idx}',
                                        f"[{idx}] PLL Clock {idx}")
            container.addWidget(lbl, idx + 1, 0, 1, 1)

            res_lbl = self.gen_label_widget(self.LBL_PLL_CLK_RES[4:] + str(idx), "Unknown")
            container.addWidget(res_lbl, idx + 1, 1, 1, 1)

            inst_lbl = self.gen_label_widget(self.LBL_PLL_CLK_INST[4:] + str(idx), "Unknown")
            container.addWidget(inst_lbl, idx + 1, 2, 1, 1)

            self.ref_clk_widget_list[res_lbl.objectName()] = res_lbl
            self.ref_clk_widget_list[inst_lbl.objectName()] = inst_lbl

    def gen_label_widget(self, name: str, display_name: str):
        label = QLabel(display_name)
        label.setObjectName(f"lbl_{name}")
        return label

    def add_ref_clk_gp_box_to_pg(self, category: str):
        #: Group box
        refclk_gpbox, _ = self.category2gp_box_info.get(category, (None, None))
        assert refclk_gpbox is not None

        stack_vl = self.category2stack_layout.get(category)
        assert stack_vl is not None

        # Add group box into stack widget
        stack_vl.addWidget(refclk_gpbox)
