"""
 Copyright (C) 2017-2020 Efinix Inc. All rights reserved.

 No portion of this code may be reused, modified or
 distributed in any way without the expressed written
 consent of Efinix Inc.

 Created on September 03, 2020

 @author: yasmin
"""

import sys
import os
from abc import abstractmethod
from typing import Optional, Set, Union, Any, List, Tuple, Type
from dataclasses import dataclass
from enum import Enum, auto

import util.gen_util as gen_util

sys.path.append(os.path.join(os.path.dirname(__file__), '..'))

# Alias type for legal value types
ValidSettingTypeAlias = Union['PropValidator', Tuple, List]

class PropValidator:
    """
    Abstract class for controlling what property value is valid
    """

    @abstractmethod
    def is_valid(self, value: Any) -> Tuple[bool, str]:
        """
        Check the value is valid or not,

        Return:
        bool -> valid or not
        str -> hint message (only used when invalid)
        """
        raise NotImplementedError

class RangeValidator(PropValidator):

    def __init__(self, min_val: Union[float, int], max_val: Union[float, int]):
        self._min_val = min_val
        self._max_val = max_val

    def get_range(self) -> Tuple[Union[float, int], Union[float, int]]:
        return (self._min_val, self._max_val)

    def is_valid(self, value: Any) -> Tuple[bool, str]:
        if not isinstance(value, (int, float)):
            return False, f"Value, {value} is not number, it is {type(value)}"
        if value < self._min_val or value > self._max_val:
            return False, f"Value, {value} out of range (min={self._min_val}, max={self._max_val})"
        return True, ""

class OptionValidator(PropValidator):

    def __init__(self, options: List[Any]):
        # Check all options are having same type
        t = type(options[0])
        assert all(isinstance(i, t) for i in options)

        self._options = options

    def get_options(self) -> Tuple:
        return tuple(self._options)

    def is_valid(self, value: Any) -> Tuple[bool, str]:
        if value not in self._options:
            return False, f"Value, {value} not in the supported list"
        return True, ""

class PropUsageType(Enum):
    """
    Property Usage type
    """
    BITGEN = auto()
    PNR = auto()
    SYN = auto()
    GUI = auto()

@gen_util.freeze_it
class PropertyMetaData:
    """
    Metadata for property, typically block parameters
    """

    @dataclass
    class PropData:
        id: Any = None  #: Property id, Any type
        name: str = ""  #: Property name
        disp_name: str = ""  #: UI display name
        data_type: Any = None  #: Typically, db_item.GenericParam.DataType
        default: Any = None  #: Default value, Any type
        valid_setting: Optional[ValidSettingTypeAlias] = None # None mean no validation check
        category: str = "general"  #: Property category
        usage_type: PropUsageType = PropUsageType.BITGEN

    def __init__(self):
        self.prop = {}  #: A map of prop id and its data

    def __str__(self, *args, **kwargs):
        info = []
        for prop_info in self.prop.values():
            info.append(prop_info.__str__())

        return "\n".join(info)

    def is_prop_exists(self, prop_id: Any) -> bool:
        return prop_id in self.prop

    def get_prop_name(self, prop_id: Any) -> str:
        prop_info = self.prop.get(prop_id, None)
        if prop_info is not None:
            return prop_info.name

        return ""

    def get_display_name(self, prop_id: Any) -> str:
        """
        Get user facing name

        :param prop_id: Property id
        :return: Name if any, else empty string
        """
        prop_info = self.prop.get(prop_id, None)
        if prop_info is not None:
            return prop_info.disp_name

        return ""

    def get_prop_id(self, prop_name: str) -> Union[Any, None]:
        """
        Given property name find its id

        :param prop_name: Property name
        :return: Name if any, else None
        """
        for prop in self.prop.values():
            if prop is not None and prop_name == prop.name:
                return prop.id

        return None

    def get_all_prop(self) -> List[PropData]:
        """
        Get all property data

        :return: A list of prop data
        """
        return list(self.prop.values())

    def get_prop_info_by_id_list(self, prop_id_list: List[Any]) -> List[PropData]:
        """
        Get a list of prop info by a list of prop id

        :param prop_id_list: Prop Id list
        :type prop_id_list: List[Any]
        :return: List of Prop Data
        :rtype: List[PropData]
        """
        prop_info_list: List[PropertyMetaData.PropData] = []

        for param_id in prop_id_list:
            prop_info = self.get_prop_info(param_id)
            if prop_info is None:
                continue
            prop_info_list.append(prop_info)

        return prop_info_list

    def add_prop(self, prop_id: Any, prop_name: str, prop_data_type: Any, prop_default: Any,
                 valid_setting: Union[ValidSettingTypeAlias, None] = None, disp: str = "",
                 usage_type: PropUsageType = PropUsageType.BITGEN):
        """
        Insert new property data

        :param prop_id: Property unique id
        :param prop_name: Property name
        :param prop_data_type: Property data type
        :param prop_default: Property default value
        :param valid_setting: Legal value, a range limit or distinct value items
        :param disp: Property user facing name
        """
        self.prop[prop_id] = self.PropData(prop_id, prop_name, disp, prop_data_type, prop_default, valid_setting, usage_type)

    def get_prop_id_by_category(self, category) -> List:
        prop_map = dict(filter(lambda elem: elem[1].category == category, self.prop.items()))
        return list(prop_map.keys())

    def get_prop_by_category(self, category) -> List[PropData]:
        return list(filter(lambda value: value.category == category, self.prop.values()))

    def add_prop_by_data(self, prop_id: Any, prop_data: PropData):
        """
        Insert new property data given prop data object

        :param prop_id: Property id
        :param prop_data: Property data
        """
        self.prop[prop_id] = prop_data

    def get_prop_info(self, prop_id: Any) -> Union[PropData, None]:
        """
        Get property data given a property id

        :param prop_id: Property id
        :return: Property data is any
        """
        return self.prop.get(prop_id, None)

    def get_prop_info_by_name(self, prop_name: str) -> Union[PropData, None]:
        """
        Get property data given a property id

        :param prop_id: Property id
        :return: Property data if name exists
        """
        for prop_info in self.prop.values():
            if prop_info.name == prop_name:
                return prop_info

        return None

    def get_default(self, prop_id: Any) -> Union[Any, None]:
        """
        Get default value given a property id

        :param prop_id: Property id
        :return: Default value if property exists
        """
        prop = self.get_prop_info(prop_id)
        if prop is not None:
            return prop.default
        return None

    def set_default(self, prop_id: Any, prop_value: Any):
        prop = self.get_prop_info(prop_id)
        if prop is not None:
            prop.default = prop_value

    def get_valid_setting(self, prop_id: Any) -> Union[ValidSettingTypeAlias, None]:
        prop = self.get_prop_info(prop_id)
        if prop is not None:
            return prop.valid_setting
        return None

    def set_valid_setting(self, prop_id: Any, prop_valid_setting: Union[ValidSettingTypeAlias, None]):
        prop = self.get_prop_info(prop_id)
        if prop is not None:
            prop.valid_setting = prop_valid_setting

    def is_valid_setting(self, prop_id: Any, value: Any) -> bool:
        """
        For a given property and its value, checks if it is legal.
        Only checks for range limit or distinct legal values.

        :param prop_id: Property id
        :param value: Value to check
        :return: True if valid, else False
        """
        prop = self.get_prop_info(prop_id)
        if prop is not None:
            valid_setting = prop.valid_setting
            if isinstance(valid_setting, list):
                if not value in valid_setting:
                    return False
            elif isinstance(valid_setting, tuple):
                min_val, max_val = valid_setting
                if value < min_val or value > max_val:
                    return False
            elif isinstance(valid_setting, PropValidator):
                return valid_setting.is_valid(value)[0]
            else:
                return True

        return True

    def concat_param_info(self, param_info):
        assert param_info is not None
        all_prop = param_info.get_all_prop()

        for prop_data in all_prop:
            if prop_data.id in self.prop:
                raise ValueError(f'Property {prop_data.id} already exists')
            self.add_prop_by_data(prop_data.id, prop_data)


class PropertyMetaDataSharedView:
    """
    Sharing some properties from PropertyMetaData, accessing / writing are delegate to the original one
    """
    def __init__(self, source: PropertyMetaData, shared_props: List[Tuple[Any, str]]) -> None:
        self.__source = source

        if len(shared_props) == 0:
            raise ValueError("Must specify non-empty list for properties to be shared")

        for pid, pname in shared_props:
            if not self.__source.is_prop_exists(pid):
                raise KeyError(f"Property ID: {pid} not found in the source")
            prop_info = self.__source.get_prop_info(pid)
            if not prop_info:
                raise KeyError(f"Property ID: {pid} has empty PropData in the source")
            if pname != prop_info.name:
                raise KeyError(f"Property ID: {pid} have unexpected name, expected {pname} got {prop_info.name}")

        self.__shared_prop_ids = set([p[0] for p in shared_props])
        self.__shared_prop_names = set([p[1] for p in shared_props])

    def __is_prop_id_shared(self, prop_id: Any) -> bool:
        return prop_id in self.__shared_prop_ids

    def __is_prop_name_shared(self, prop_name: str) -> bool:
        return prop_name in self.__shared_prop_names

    def is_prop_exists(self, prop_id: Any) -> bool:
        return self.__is_prop_id_shared(prop_id) and \
                self.__source.is_prop_exists(prop_id)

    def get_prop_name(self, prop_id: Any) -> str:
        if not self.__is_prop_id_shared(prop_id):
            return ""
        return self.__source.get_prop_name(prop_id)

    def get_display_name(self, prop_id: Any) -> str:
        if not self.__is_prop_id_shared(prop_id):
            return ""
        return self.__source.get_display_name(prop_id)

    def get_prop_id(self, prop_name: str) -> Union[Any, None]:
        if not self.__is_prop_name_shared(prop_name):
            return None
        return self.__source.get_prop_id(prop_name)

    def get_all_prop(self) -> List[PropertyMetaData.PropData]:
        def is_shared(p: PropertyMetaData.PropData) -> bool:
            return self.__is_prop_id_shared(p.id)
        return list(filter(is_shared, self.__source.get_all_prop()))

    def add_prop(self, prop_id: Any, prop_name: str, prop_data_type: Any, prop_default: Any,
                 valid_setting: Union[ValidSettingTypeAlias, None] = None, disp: str = "",
                 usage_type: PropUsageType = PropUsageType.BITGEN):
        self.__shared_prop_ids.add(prop_id)
        self.__shared_prop_names.add(prop_name)
        self.__source.add_prop(prop_id=prop_id, prop_name=prop_name, prop_data_type=prop_data_type,
                prop_default=prop_default, valid_setting=valid_setting, disp=disp,
                usage_type=usage_type)

    def get_prop_by_category(self, category: str) -> List[PropertyMetaData.PropData]:
        result = self.__source.get_prop_by_category(category)
        return list(filter(lambda p: self.__is_prop_id_shared(p.id), result))

    def get_prop_id_by_category(self, category: str) -> List[Any]:
        result = self.get_prop_by_category(category)
        return [p.id for p in result]

    def add_prop_by_data(self, prop_id: Any, prop_data: PropertyMetaData.PropData):
        self.__shared_prop_ids.add(prop_id)
        self.__shared_prop_names.add(prop_data.name)
        return self.__source.add_prop_by_data(prop_id=prop_id, prop_data=prop_data)

    def get_prop_info(self, prop_id: Any) -> Union[PropertyMetaData.PropData, None]:
        if not self.__is_prop_id_shared(prop_id):
            return None
        return self.__source.get_prop_info(prop_id)

    def get_prop_info_by_name(self, prop_name: str) -> Union[PropertyMetaData.PropData, None]:
        if not self.__is_prop_name_shared(prop_name):
            return None
        return self.__source.get_prop_info_by_name(prop_name)

    def get_default(self, prop_id: Any) -> Union[Any, None]:
        if not self.__is_prop_id_shared(prop_id):
            raise KeyError(f'Property ID: {prop_id} not found")')
        return self.__source.get_default(prop_id)

    def set_default(self, prop_id: Any, prop_value: Any):
        if not self.__is_prop_id_shared(prop_id):
            raise KeyError(f'Property ID: {prop_id} not found")')
        return self.__source.set_default(prop_id, prop_value)

    def get_valid_setting(self, prop_id: Any) -> Union[ValidSettingTypeAlias, None]:
        if not self.__is_prop_id_shared(prop_id):
            raise KeyError(f'Property ID: {prop_id} not found")')
        return self.__source.get_valid_setting(prop_id)

    def set_valid_setting(self, prop_id: Any, prop_valid_setting: Union[ValidSettingTypeAlias, None]):
        if not self.__is_prop_id_shared(prop_id):
            raise KeyError(f'Property ID: {prop_id} not found")')
        return self.__source.set_valid_setting(prop_id, prop_valid_setting)

    def is_valid_setting(self, prop_id: Any, value: Any) -> bool:
        if not self.__is_prop_id_shared(prop_id):
            raise KeyError(f'Property ID: {prop_id} not found")')
        return self.__source.is_valid_setting(prop_id, value)


if __name__ == "__main__":
    pass
