"""
Copyright (C) 2017-2018 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Jun 27, 2018

@author: yasmin
"""

import sys
import os
from typing import Optional
import xml.etree.ElementTree as et
import xmlschema
import PyQt5.QtCore as qt

import design.db as design_db
import design.db_item as design_db_item

import design.service_interface as dbi
import util.app_setting as aps
from util.singleton_logger import Logger
import util.gen_util as gutil
from util.app_setting import AppSetting

import common_device.mipi.mipi_design as mipid
import common_device.mipi.mipi_rule as mipir
from common_device.mipi.timing_calculator import MIPITimingCalculator

import common_device.excp as dsg_excp

sys.path.append(os.path.join(os.path.dirname(__file__), '..'))


class MIPIDesignService(dbi.DesignBlockService):
    """
    Provide operation for MIPI design
    """

    def __init__(self):
        """
        Constructor
        """
        super().__init__()
        self.logger = Logger
        self.checker = None

    def create_registry(self):
        """
        Create empty in memory MIPI registry
        :return: MIPI registry
        """
        mipi_reg = mipid.MIPIRegistry()
        MIPIDesignBuilderXml.build_timing_calc(mipi_reg)
        mipi_reg.update_chksum()

        return mipi_reg

    def load_design(self, design_file):
        """
        Create lvds design registry from a design file

        :param design_file: Full path to design file name

        :returns A mipi design registry
        """
        builder = MIPIDesignBuilderXml()
        mipi_reg = builder.build(design_file)
        return mipi_reg

    def load_design_from_xmltree(self, xml_tree):
        """
        Not implemented
        """

    def save_design(self, design, design_file, xml_writer=None):
        """
        Write in-memory design to a file.

        :param design: Design to write. This can be full design or lvds registry
        :param design_file: Target file to write to
        :param xml_writer: XML writer object of QXmlStreamWriter type
        """
        saver = MIPIDesignWriterXml()
        if xml_writer is not None:
            saver.write(design, xml_writer)
        else:
            writer = qt.QXmlStreamWriter()
            xml_file = qt.QSaveFile(design_file)

            if xml_file.open(qt.QFile.WriteOnly) is True:
                writer.setDevice(xml_file)
            else:
                self.logger.warning(
                    "Fail to open design file for writing, " + design_file)
                return

            writer.setAutoFormatting(True)
            writer.writeStartDocument()
            saver.write(design, writer)
            writer.writeEndDocument()

            # Save changes to harddisk
            if xml_file.commit() is True:
                self.logger.info("Write design to {}".format(design_file))
            else:
                self.logger.info(
                    "Fail to write design to {}".format(design_file))

    def check_design_file(self, design_file):
        """
        Validate the design file

        :param design_file: mipi design file

        :returns True if good, else False
        """
        builder = MIPIDesignBuilderXml()
        return builder.validate(design_file)

    def check_design(self, design, exclude_rules=None,
                     result_file=None, writer=None):

        if self.checker is None:
            self.checker = mipir.MIPIChecker(design)

        self.checker.run_all_check(exclude_rules)


class MIPIDesignBuilderXml(dbi.DesignBuilder):
    """
    Build mipi design using event based parsing
    """

    _ns = "{http://www.efinixinc.com/peri_design_db}"

    '''
    XML Namespace for peri design db
    '''

    _schema_file = ""
    '''
    If design schema is empty, it will be detected automatically using $EFXPT_HOME.
    This is mainly for unit testing to set schema from different location.
    '''

    def __init__(self):
        super().__init__()
        self.mipi_reg = None
        self.logger = Logger

    def build(self, design_file):
        """
        Build mipi registry from mipi design

        :param design_file: mipi design file

        :return mipi registry
        """

        with open(design_file, 'r', encoding='UTF-8') as xml_file:

            # get an iterable
            context = et.iterparse(xml_file, events=("start", "end", "start-ns", "end-ns"))

            # turn it into an iterator
            context = iter(context)

            mipi_info_tag = MIPIDesignBuilderXml._ns + "mipi_info"
            mipi_tag = MIPIDesignBuilderXml._ns + "mipi"
            rx_tag = MIPIDesignBuilderXml._ns + "mrx_info"
            tx_tag = MIPIDesignBuilderXml._ns + "mtx_info"
            pin_tag = MIPIDesignBuilderXml._ns + "pin"
            phy_lane_tag = MIPIDesignBuilderXml._ns + "phy_lane"
            rx_timing_tag = MIPIDesignBuilderXml._ns + "rx_timing"
            tx_timing_tag = MIPIDesignBuilderXml._ns + "tx_timing"

            curr_mipi = None
            curr_pin_group = None
            curr_phy_map = None

            for event, elem in context:
                if event == "start":
                    if elem.tag == mipi_info_tag:
                        self._build_mipi_reg()

                    elif elem.tag == mipi_tag:
                        curr_mipi = self._build_mipi(elem)

                    elif elem.tag == tx_tag:
                        self._build_tx(curr_mipi, elem)
                        tx_info = curr_mipi.tx_info
                        if tx_info is not None:
                            curr_pin_group = tx_info.gen_pin
                            curr_phy_map = tx_info.phy_lane_map

                    elif elem.tag == rx_tag:
                        self._build_rx(curr_mipi, elem)
                        rx_info = curr_mipi.rx_info
                        if rx_info is not None:
                            curr_pin_group = rx_info.gen_pin
                            curr_phy_map = rx_info.phy_lane_map

                    elif elem.tag == pin_tag:
                        self._build_generic_pin(curr_pin_group, elem)

                    elif elem.tag == phy_lane_tag:
                        self._build_phy_lane(curr_phy_map, elem)

                    elif elem.tag == rx_timing_tag:
                        self._build_rx_timing(curr_mipi.rx_info, elem)

                    elif elem.tag == tx_timing_tag:
                        self._build_tx_timing(curr_mipi.tx_info, elem)

                    # Free up process element
                    elem.clear()

                elif event == "end":
                    if elem.tag == mipi_tag:
                        if curr_mipi is not None:
                            curr_mipi.update_chksum()

                    elif elem.tag == mipi_info_tag:
                        # Stop parsing once target registry content is done
                        break

        if self.mipi_reg is not None:
            MIPIDesignBuilderXml.build_timing_calc(self.mipi_reg)
            self.mipi_reg.update_chksum()

        return self.mipi_reg

    def build_from_xmltree(self, xml_tree):
        """
        Not implemented
        """
        pass

    @staticmethod
    def build_timing_calc(mipi_reg):

        if mipi_reg is None:
            return

        setting = AppSetting()
        efxpt_home = setting.app_path[AppSetting.PathType.install]
        csvfile = os.path.normpath(efxpt_home + "/db/die/block_models/mipi_functional_timing.csv")
        calculator = MIPITimingCalculator(csvfile)
        mipi_reg.timing_calc = calculator

    def _build_mipi_reg(self):
        self.mipi_reg = mipid.MIPIRegistry()

    def _build_mipi(self, xml_elem):

        inst = None
        if self.mipi_reg:
            attrs = xml_elem.attrib
            try:
                mipi_def = attrs.get("mipi_def", "")
                if mipi_def == "":
                    inst = self.mipi_reg.create_instance(attrs["name"])
                else:
                    inst = self.mipi_reg.create_instance_with_device(attrs["name"], mipi_def)

                ops_type = mipid.MIPI.str2opstype_map.get(attrs.get("ops_type"), mipid.MIPI.OpsType.unknown)
                inst.update_type(ops_type)

            except dsg_excp.CreateException:
                self.logger.warning(
                    "Fail to create mipi instance, ignore: {}".format(inst.name))

        return inst

    def _build_tx(self, parent_mipi, xml_elem):

        if parent_mipi:
            parent_mipi.update_type(mipid.MIPI.OpsType.op_tx)
            tx_info = parent_mipi.tx_info
            if tx_info is not None:
                attrs = xml_elem.attrib
                tx_info.ref_clock_freq = float(attrs.get("ref_clock_freq", "0.0"))
                tx_info.phy_tx_freq_code = int(attrs.get("phy_tx_freq_code", tx_info.get_default_phy_freq_code()))

                tx_info.is_cont_phy_clocking = gutil.xmlstr2bool(attrs.get("is_cont_phy_clocking", "false"))

                esc_clk_freq = attrs.get("esc_clock_freq", None)
                def_val = tx_info.get_default_escclk_freq()
                if esc_clk_freq is None:
                    # backward compatibility code
                    tx_info.esc_clock_freq = def_val
                else:
                    # backward compatibility code
                    if esc_clk_freq == "0" or esc_clk_freq == "0.0":
                        tx_info.esc_clock_freq = def_val
                    else:
                        tx_info.esc_clock_freq = float(esc_clk_freq)

    def _build_tx_timing(self, tx_info, xml_elem):

        if tx_info is not None:
            attrs = xml_elem.attrib

            defval_str = "{}".format(tx_info.get_timer_default_value(mipid.MIPITx.TimerType.clk_post))
            tx_info.t_clk_post = int(attrs.get("t_clk_post", defval_str))

            defval_str = "{}".format(tx_info.get_timer_default_value(mipid.MIPITx.TimerType.clk_trail))
            tx_info.t_clk_trail = int(attrs.get("t_clk_trail", defval_str))

            defval_str = "{}".format(tx_info.get_timer_default_value(mipid.MIPITx.TimerType.clk_prepare))
            tx_info.t_clk_prepare = int(attrs.get("t_clk_prepare", defval_str))

            defval_str = "{}".format(tx_info.get_timer_default_value(mipid.MIPITx.TimerType.clk_zero))
            tx_info.t_clk_zero = int(attrs.get("t_clk_zero", defval_str))

            defval_str = "{}".format(tx_info.get_timer_default_value(mipid.MIPITx.TimerType.clk_pre))
            tx_info.t_clk_pre = int(attrs.get("t_clk_pre", defval_str))

            defval_str = "{}".format(tx_info.get_timer_default_value(mipid.MIPITx.TimerType.hs_prepare))
            tx_info.t_hs_prepare = int(attrs.get("t_hs_prepare", defval_str))

            defval_str = "{}".format(tx_info.get_timer_default_value(mipid.MIPITx.TimerType.hs_zero))
            tx_info.t_hs_zero = int(attrs.get("t_hs_zero", defval_str))

            defval_str = "{}".format(tx_info.get_timer_default_value(mipid.MIPITx.TimerType.hs_trail))
            tx_info.t_hs_trail = int(attrs.get("t_hs_trail", defval_str))

    def _build_rx(self, parent_mipi, xml_elem):

        gutil.mark_unused(xml_elem)

        if parent_mipi:
            parent_mipi.update_type(mipid.MIPI.OpsType.op_rx)

        rx_info: Optional[mipid.MIPIRx] = parent_mipi.rx_info
        if rx_info is not None:
            attrs = xml_elem.attrib

            calib_clk_freq = attrs.get("dphy_calib_clock_freq", None)
            def_val = rx_info.get_param_info().get_default(mipid.MIPIParamInfo.Id.CAL_CLK_FREQ)
            if calib_clk_freq is None:
                # backward compatibility code
                rx_info.dphy_calib_clock_freq = def_val
            else:
                # backward compatibility code
                if calib_clk_freq == "0" or calib_clk_freq == "0.0":
                    rx_info.dphy_calib_clock_freq = def_val
                else:
                    rx_info.dphy_calib_clock_freq = float(calib_clk_freq)

            # Status enable
            def_val = rx_info.get_param_info().get_default(mipid.MIPIParamInfo.Id.STATUS_EN)
            is_status_en = attrs.get("status_en", None)

            if is_status_en is None:
                is_status_en = def_val
            else:
                is_status_en = gutil.xmlstr2bool(is_status_en)
            assert isinstance(is_status_en, bool)

            rx_info.is_status_en = is_status_en

    def _build_rx_timing(self, rx_info, xml_elem):

        if rx_info is not None:
            attrs = xml_elem.attrib

            defval_str = "{}".format(rx_info.get_timer_default_value(mipid.MIPIRx.TimerType.clk_settle))
            rx_info.t_clk_settle = int(attrs.get("t_clk_settle", defval_str))

            defval_str = "{}".format(rx_info.get_timer_default_value(mipid.MIPIRx.TimerType.hs_settle))
            rx_info.t_hs_settle = int(attrs.get("t_hs_settle", defval_str))

    def _build_generic_pin(self, pin_group, xml_elem):

        if pin_group is not None:
            attrs = xml_elem.attrib
            type_name = attrs.get("type_name", "")
            name = attrs.get("name", "")
            is_bus = gutil.xmlstr2bool(attrs.get("is_bus", "false"))

            is_clk = attrs.get("is_clk", None)
            if is_clk is not None:
                is_clk = gutil.xmlstr2bool(is_clk)
                if is_clk:
                    is_clk_invert = gutil.xmlstr2bool(attrs.get("is_clk_invert", "false"))
                    pin_group.add_clock_pin(type_name, name, is_bus, is_clk_invert)
            else:
                # For backward compatibility
                # At this point, device port is not available so cannot check if it is a clock type
                # Have to hard code!!
                if type_name == "PIXEL_CLK" or type_name == "ESC_CLK" or type_name == "CAL_CLK":
                    pin_group.add_clock_pin(type_name, name, is_bus, False)
                else:
                    pin_group.add_pin(type_name, name, is_bus)

    def _build_phy_lane(self, phy_lane_map, xml_elem):

        if phy_lane_map is not None:
            attrs = xml_elem.attrib
            lane_id = attrs.get("lane_id", None)
            if lane_id is not None:
                lane_id = mipid.PhyLane.PhyLaneIdType(int(lane_id))
                phy_lane = phy_lane_map.get_phy_lane(lane_id)
                if phy_lane is not None:
                    log_lane_id = attrs.get("logical_lane_id", None)
                    if log_lane_id is not None:
                        phy_lane.logical_lane_id = mipid.PhyLane.LogLaneIdType(int(log_lane_id))
                    else:
                        phy_lane.logical_lane_id = mipid.PhyLane.LogLaneIdType.unused

                    phy_lane.is_pn_swap = gutil.xmlstr2bool(attrs.get("is_pn_swap", "false"))

    def check(self, design):
        gutil.mark_unused(design)

    def validate(self, design_file):

        app_setting = aps.AppSetting()
        schema_path = app_setting.app_path[aps.AppSetting.PathType.schema]

        # mipi schema
        if MIPIDesignBuilderXml._schema_file == "":
            MIPIDesignBuilderXml._schema_file = schema_path + "/mipi_design.xsd"

        schema = xmlschema.XMLSchema(
            MIPIDesignBuilderXml._schema_file)

        try:

            # validate mipi
            schema.validate(design_file)

        except et.ParseError as excp:
            self.logger.error("MIPI design parsing error : {}".format(excp))
            return False

        except xmlschema.XMLSchemaValidationError as excp:
            self.logger.error("MIPI design validation error")
            self.logger.error("XML Element : {}".format(excp.elem))
            self.logger.error("Reason : {}".format(excp.reason))
            return False

        return True


class MIPIDesignWriterXml:

    def __init__(self):
        self.design = None
        self.efx_ns = "efxpt"
        self.writer = None
        self.logger = Logger

    def write(self, design, xml_writer):
        """
        Write design to an xml file.

        If the writer is not None, it will be used to write. Otherwise new writer will be constructed.
        :param design: Design can be a PeriDesign or MIPIRegistry
        :param xml_writer:
        """
        self.design = design
        self.writer = xml_writer
        is_full_design = False

        mipi_reg = None
        if isinstance(design, mipid.MIPIRegistry) is True:
            mipi_reg = design
        elif isinstance(design, design_db.PeriDesign) is True:
            mipi_reg = self.design.mipi_reg
            is_full_design = True

        if mipi_reg is not None:
            self.writer.writeStartElement(self.efx_ns + ":mipi_info")

            if is_full_design is False:
                self.writer.writeAttribute(
                    "xmlns:efxpt", "http://www.efinixinc.com/peri_design_db")
                self.writer.writeAttribute(
                    "xmlns:xsi", "http://www.w3.org/2001/XMLSchema-instance")

            all_mipi = mipi_reg.get_all_inst()
            for mipi in all_mipi:
                self.writer.writeStartElement(self.efx_ns + ":mipi")
                self.writer.writeAttribute("name", mipi.name)
                self.writer.writeAttribute("mipi_def", mipi.mipi_def)
                self.writer.writeAttribute("ops_type", mipid.MIPI.opstype2str_map.get(mipi.ops_type, ""))

                if mipi.ops_type == mipid.MIPI.OpsType.op_tx:
                    self.write_tx(mipi.tx_info)
                elif mipi.ops_type == mipid.MIPI.OpsType.op_rx:
                    self.write_rx(mipi.rx_info)

                # mipi
                self.writer.writeEndElement()

            # mipi_info
            self.writer.writeEndElement()

    def write_tx(self, tx_info):
        if tx_info is not None:
            self.writer.writeStartElement(self.efx_ns + ":mtx_info")

            self.writer.writeAttribute("ref_clock_freq", "{}".format(tx_info.ref_clock_freq))
            self.writer.writeAttribute("phy_tx_freq_code", "{}".format(tx_info.phy_tx_freq_code))
            self.writer.writeAttribute("is_cont_phy_clocking", gutil.bool2xmlstr(tx_info.is_cont_phy_clocking))
            self.writer.writeAttribute("esc_clock_freq", "{}".format(tx_info.esc_clock_freq))
            self.write_generic_pin(tx_info.gen_pin)
            self.write_phy_lane(tx_info.phy_lane_map)
            self.write_tx_timing(tx_info)
            self.writer.writeEndElement()

    def write_tx_timing(self, tx_info):
        self.writer.writeStartElement(self.efx_ns + ":tx_timing")
        self.writer.writeAttribute("t_clk_post", "{}".format(int(tx_info.t_clk_post)))
        self.writer.writeAttribute("t_clk_trail", "{}".format(int(tx_info.t_clk_trail)))
        self.writer.writeAttribute("t_clk_prepare", "{}".format(int(tx_info.t_clk_prepare)))
        self.writer.writeAttribute("t_clk_zero", "{}".format(int(tx_info.t_clk_zero)))
        self.writer.writeAttribute("t_clk_pre", "{}".format(int(tx_info.t_clk_pre)))
        self.writer.writeAttribute("t_hs_prepare", "{}".format(int(tx_info.t_hs_prepare)))
        self.writer.writeAttribute("t_hs_zero", "{}".format(int(tx_info.t_hs_zero)))
        self.writer.writeAttribute("t_hs_trail", "{}".format(int(tx_info.t_hs_trail)))
        self.writer.writeEndElement()

    def write_rx(self, rx_info):
        if rx_info is not None:
            self.writer.writeStartElement(self.efx_ns + ":mrx_info")
            self.writer.writeAttribute("status_en", "{}".format(gutil.bool2xmlstr(rx_info.is_status_en)))
            self.writer.writeAttribute("dphy_calib_clock_freq", "{}".format(rx_info.dphy_calib_clock_freq))
            self.write_generic_pin(rx_info.gen_pin)
            self.write_phy_lane(rx_info.phy_lane_map)
            self.write_rx_timing(rx_info)
            self.writer.writeEndElement()

    def write_rx_timing(self, rx_info):
        self.writer.writeStartElement(self.efx_ns + ":rx_timing")
        self.writer.writeAttribute("t_clk_settle", "{}".format(int(rx_info.t_clk_settle)))
        self.writer.writeAttribute("t_hs_settle", "{}".format(int(rx_info.t_hs_settle)))
        self.writer.writeEndElement()

    def write_generic_pin(self, pin_group):

        if pin_group is None:
            return

        self.writer.writeStartElement(self.efx_ns + ":gen_pin")

        all_pin = pin_group.get_all_pin()
        for pin in all_pin:
            self.writer.writeStartElement(self.efx_ns + ":pin")
            self.writer.writeAttribute("name", pin.name)
            self.writer.writeAttribute("type_name", pin.type_name)
            self.writer.writeAttribute("is_bus", gutil.bool2xmlstr(pin.is_bus))

            if isinstance(pin, design_db_item.GenericClockPin):
                self.writer.writeAttribute("is_clk", gutil.bool2xmlstr(True))
                self.writer.writeAttribute("is_clk_invert", gutil.bool2xmlstr(pin.is_inverted))

            # pin
            self.writer.writeEndElement()

        # other pin
        self.writer.writeEndElement()

    def write_phy_lane(self, phy_lane_map):

        if phy_lane_map is None:
            return

        all_lane = phy_lane_map.get_phy_lane_list()
        for lane in all_lane:
            self.writer.writeStartElement(self.efx_ns + ":phy_lane")
            self.writer.writeAttribute("lane_id", "{}".format(int(lane.lane_id)))
            self.writer.writeAttribute("logical_lane_id", "{}".format(int(lane.logical_lane_id)))
            self.writer.writeAttribute("is_pn_swap", gutil.bool2xmlstr(lane.is_pn_swap))
            self.writer.writeEndElement()


if __name__ == "__main__":
    pass
