"""
Copyright (C) 2017-2024 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on January 16, 2024

@author: Shirley Chan
"""

import sys
import os

import util.gen_util

from design.db_patch import DesignVersion, DesignPatch


sys.path.append(os.path.join(os.path.dirname(__file__), '..'))


class MIPIPatch(DesignPatch):
    """
    MIPI design backward compatibility patcher
    """

    def __init__(self, mipi_dphy_reg, device_db=None):
        super().__init__(mipi_dphy_reg, device_db)

        #
        # Patches sequence can imply dependency between patches. When the patch is applied,
        # it will be applied from the lowest version number to the highest.
        # This means it is safe for patch function to assume certain db states.
        #

        # PT-2157
        msg = "MIPI: Status enable correction"
        self.add_patch(DesignVersion.PATCH_VERSION_20251005,
                       MIPIPatch.patch_status_enable_definition, msg)

    def patch_status_enable_definition(self, inst):
        """
        Fix the status enable changed by status pin name

        :param inst: MIPI instance
        :return: True, if design is patched, else False
        """
        is_patch = False
        func_name = util.gen_util.get_function_name()

        if inst is not None and inst.ops_type == inst.OpsType.op_rx:
            info = inst.rx_info
            gen_pin = info.gen_pin
            if gen_pin is not None:
                is_status_enable = False
                pin_type_list = [
                    "ERROR",
                    "CLEAR",
                    "ULPS_CLK",
                    "ULPS"
                ]
                for pin_type in pin_type_list:
                    pin = gen_pin.get_pin_by_type_name(pin_type)
                    if pin.name != "":
                        is_status_enable = True
                        break

                info.is_status_en = is_status_enable
                is_patch = True

        if is_patch:
            self.logger.warning(
                "{}: Patch {} applied to {}".format(
                    func_name, DesignVersion.PATCH_VERSION_20251005, inst.name))


        return is_patch
