"""
Copyright (C) 2017-2018 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Jun 27, 2018

@author: yasmin
"""

from __future__ import annotations
import abc
from enum import Enum, IntEnum
from typing import Dict, Optional, Tuple, List
from common_device.mipi.mipi_tx_device_service import MIPITxService
from common_device.mipi.timing_calculator import MIPITimingCalculator
from common_device.property import PropertyMetaData
from device.db import PeripheryDevice

import util.gen_util as gen_util

import design.db_item as db_item


@gen_util.freeze_it
class PhyLane:
    """
    MIPI physical lane instance.
    """

    class PhyLaneIdType(IntEnum):
        """
        Physical lane id type
        """
        lane0 = 0
        lane1 = 1
        lane2 = 2
        lane3 = 3
        lane4 = 4

    tx_phylane2pad_map = \
        {
            PhyLaneIdType.lane0: "TXD0",
            PhyLaneIdType.lane1: "TXD1",
            PhyLaneIdType.lane2: "TXD2",
            PhyLaneIdType.lane3: "TXD3",
            PhyLaneIdType.lane4: "TXD4"
        }  #: Mapping of physical lane type in enum to pad name for mipi tx

    rx_phylane2pad_map = \
        {
            PhyLaneIdType.lane0: "RXD0",
            PhyLaneIdType.lane1: "RXD1",
            PhyLaneIdType.lane2: "RXD2",
            PhyLaneIdType.lane3: "RXD3",
            PhyLaneIdType.lane4: "RXD4"
        }  #: Mapping of physical lane type in enum to pad name for mipi rx

    class LogLaneIdType(IntEnum):
        """
        Logical lane id type
        """
        data0_lane = 0
        data1_lane = 1
        data2_lane = 2
        data3_lane = 3
        clk_lane = 4
        unused = 5

    loglane2str_map = \
        {
            LogLaneIdType.data0_lane: "data0",
            LogLaneIdType.data1_lane: "data1",
            LogLaneIdType.data2_lane: "data2",
            LogLaneIdType.data3_lane: "data3",
            LogLaneIdType.clk_lane: "clk",
            LogLaneIdType.unused: "unused"
        }  #: Mapping of logical lane type in enum to string type

    str2loglane_map = \
        {
            "data0": LogLaneIdType.data0_lane,
            "data1": LogLaneIdType.data1_lane,
            "data2": LogLaneIdType.data2_lane,
            "data3": LogLaneIdType.data3_lane,
            "clk": LogLaneIdType.clk_lane,
            "unused": LogLaneIdType.unused
        }  #: Mapping of logical type in string to enum type

    def __init__(self, lane_id, logical_lane_id=LogLaneIdType.unused, is_pn_swap=False):
        self.lane_id = lane_id  #: This physical lane id
        self.logical_lane_id = logical_lane_id  #: The assigned logical lane id
        self.is_pn_swap = is_pn_swap  #: Indicate whether the P&N pin on physical lane is swapped

    def __str__(self, *args, **kwargs):
        info = 'phylane_id:{} loglane_id:{} is_pn_swap:{}' \
            .format(self.lane_id,
                    self.logical_lane_id,
                    self.is_pn_swap)
        return info


@gen_util.freeze_it
class PhyLaneMap:
    """
    Container for a map of mipi physical lane id to its data, PhyLane.

    .. note::
       Due to PT-530, the terminology and concepts of lane mapping differs from UI and code.

       In trying to understand the code and to reconcile the changes, think of it this way
       RXD0 match with Phy0
       RXD1 match with Phy1
       RXD2 match with Phy2
       RXD3 match with Phy3
       RXD4 match with Phy4

       The same applies to tx.
    """

    def __init__(self):
        self.default_map = {
            PhyLane.PhyLaneIdType.lane0: PhyLane.LogLaneIdType.data0_lane,
            PhyLane.PhyLaneIdType.lane1: PhyLane.LogLaneIdType.data1_lane,
            PhyLane.PhyLaneIdType.lane2: PhyLane.LogLaneIdType.clk_lane,
            PhyLane.PhyLaneIdType.lane3: PhyLane.LogLaneIdType.data2_lane,
            PhyLane.PhyLaneIdType.lane4: PhyLane.LogLaneIdType.data3_lane,
        }

        # Map created with default mapping
        self.phy_lane_map: Dict[PhyLane.PhyLaneIdType, PhyLane] = {}

        for phy_lane, log_laneid in self.default_map.items():
            self.phy_lane_map[phy_lane] = PhyLane(phy_lane, log_laneid)

    def __str__(self):
        info_list = []
        for lane in self.phy_lane_map:
            info = "{}".format(lane)
            info_list.append(info)

        return ','.join(info_list)

    def get_phy_lane_list(self):
        """
        Get a list of physical lane instance

        :return: A list
        """
        return list(self.phy_lane_map.values())

    def get_phy_lane(self, phy_lane_id):
        """
        Get physical lane instance given a physical lane id

        :param phy_lane_id: Physical lane id
        :return: If found, lane instance else None
        """
        return self.phy_lane_map.get(phy_lane_id, None)

    def get_phy_lane_by_log_lane(self, log_lane_id):
        """
        Get physical lane instance that is mapped with the given logical lane id

        :param log_lane_id: Target logical lane id
        :return: If mapping found, return the physical lane instance else None
        """
        for lane in self.phy_lane_map.values():
            if lane.logical_lane_id == log_lane_id:
                return lane

        return None

    def is_logical_lane_used(self, log_lane_id):
        """
        Check if the logical lane id has been mapped to any physical lane

        :param log_lane_id: Target logical lane id
        :return: True, if found map, else False
        """

        phy_lane = self.get_phy_lane_by_log_lane(log_lane_id)
        if phy_lane is not None:
            return True

        return False

    def swap_lane_mapping(self, phy_lane_id1, phy_lane_id2):
        """
        Swap lane mapping between two physical lanes

        :param phy_lane_id1: Physical lane instance 1
        :param phy_lane_id2: Physical lane instance 2
        """

        phy_lane1 = self.get_phy_lane(phy_lane_id1)
        phy_lane2 = self.get_phy_lane(phy_lane_id2)

        if phy_lane1 is not None and phy_lane2 is not None:
            tmp_log_lane = phy_lane1.logical_lane_id
            phy_lane1.logical_lane_id = phy_lane2.logical_lane_id
            phy_lane2.logical_lane_id = tmp_log_lane


@gen_util.freeze_it
class MIPIParamInfo(PropertyMetaData):
    class Id(Enum):
        """
        Parameter ID
        """
        PHY_FREQ = "PHY_FREQ"
        REFCLK_FREQ = "REFCLK_FREQ"
        REFCLK_RES = "REFCLK_RES"
        REFCLK_INST = "REFCLK_INST"

        # Timing related
        TCLK_POST = "TCLK_POST"
        TCLK_TRAIL = "TCLK_TRAIL"
        TCLK_PREPARE = "TCLK_PREPARE"
        TCLK_ZERO = "TCLK_ZERO"
        TCLK_PRE = "TCLK_PRE"
        THS_PREPARE = "THS_PREPARE"
        THS_ZERO = "THS_ZERO"
        THS_TRAIL = "THS_TRAIL"

        TCLK_SETTLE = "TCLK_SETTLE"
        THS_SETTLE = "THS_SETTLE"

        # Common
        ESC_CLK_INVERTED_EN = "ESC_CLK_INVERTED_EN"

        # Lane Mapping
        TXD0_LANE = "TXD0_LANE"
        TXD1_LANE = "TXD1_LANE"
        TXD2_LANE = "TXD2_LANE"
        TXD3_LANE = "TXD3_LANE"
        TXD4_LANE = "TXD4_LANE"

        RXD0_LANE = "RXD0_LANE"
        RXD1_LANE = "RXD1_LANE"
        RXD2_LANE = "RXD2_LANE"
        RXD3_LANE = "RXD3_LANE"
        RXD4_LANE = "RXD4_LANE"

        RXD0_PN_SWAP = "RXD0_PN_SWAP"
        RXD1_PN_SWAP = "RXD1_PN_SWAP"
        RXD2_PN_SWAP = "RXD2_PN_SWAP"
        RXD3_PN_SWAP = "RXD3_PN_SWAP"
        RXD4_PN_SWAP = "RXD4_PN_SWAP"

        # Tx
        CONT_PHY_CLK_EN = "CONT_PHY_CLK_EN"
        PIXEL_CLK_INVERTED_EN = "PIXEL_CLK_INVERTED_EN"
        ESC_CLK_FREQ = "ESC_CLK_FREQ"

        # Rx
        CAL_CLK_INVERTED_EN = "CAL_CLK_INVERTED_EN"
        CAL_CLK_FREQ = "CAL_CLK_FREQ"
        STATUS_EN = "STATUS_EN"


# @gen_util.freeze_it
class MIPIBase(db_item.PeriDesignGenPinItem):
    """
    Base class for common implementation for TX and RX
    """

    # Property meta data
    _param_info: Optional[MIPIParamInfo] = None

    @abc.abstractmethod
    def __init__(self):
        self.device: Optional[PeripheryDevice] = None

        if self._param_info is None:
            self._param_info = MIPIParamInfo()
            self.build_param_info(self._param_info)

        self.param_group = db_item.GenericParamGroup()
        self.build_param()

        super().__init__()
        self.build_generic_pin()

    def __str__(self, *args, **kwargs):
        info = self.param_group.to_string()
        info = f"{info}"
        return info

    def build_param(self):
        """
        Build parameter storage
        """
        assert self._param_info is not None
        for param_info in self._param_info.get_all_prop():
            self.param_group.add_param(param_info.name, param_info.default, param_info.data_type)

    def build_param_info(self, param_info: MIPIParamInfo):
        """
        Build information about supported properties

        :param param_info: Property info
        """
        if param_info is not None:
            param_info.add_prop(MIPIParamInfo.Id.PIXEL_CLK_INVERTED_EN, MIPIParamInfo.Id.PIXEL_CLK_INVERTED_EN.value, db_item.GenericParam.DataType.dbool,
                                False)

    def get_param_info(self) -> Optional[MIPIParamInfo]:
        """
        Get defined parameter info

        :return: Parameter info
        """
        return self._param_info

    def get_param_group(self) -> db_item.GenericParamGroup:
        """
        Get parameter data

        :return: Parameter data
        """
        return self.param_group

    def set_default_setting(self, device_db: Optional[PeripheryDevice] = None):
        """
        Override
        """
        super().set_default_setting(device_db)

    def create_chksum(self):
        pass


@gen_util.freeze_it
class MIPITx(MIPIBase):
    """
    Specific mipi instance property in tx mode
    """

    class TimerType(IntEnum):
        clk_post = 1
        clk_trail = 2
        clk_prepare = 3
        clk_zero = 4
        clk_pre = 5
        hs_prepare = 6
        hs_zero = 7
        hs_trail = 8

    _device_port_map = {}  #: A map of device port type to its info, linked to device db
    _phy_freq_map = None  #: A map of freq code to its frequency value string, linked to device db
    _phy_freq2code: Dict[float, int] = {} #: A map of frequency value string to its freq code , linked to device db

    _max_freq_code = None
    _min_freq_code = None

    def __init__(self, apply_default=True):
        self.timer_def_value = {
            MIPITx.TimerType.clk_post: 8,
            MIPITx.TimerType.clk_trail: 7,
            MIPITx.TimerType.clk_prepare: 2,
            MIPITx.TimerType.clk_zero: 3,
            MIPITx.TimerType.clk_pre: 2,
            MIPITx.TimerType.hs_prepare: 2,
            MIPITx.TimerType.hs_zero: 2,
            MIPITx.TimerType.hs_trail: 24
        }

        super().__init__()

        self.ref_clock_freq = 0.0  #: Reference clock frequency, MHz
        self.phy_tx_freq_code = 0.0  #: PHY Transmitting frequency code, see mipi_tx_pll_clk_sel.csv

        self.is_cont_phy_clocking = False  #: Continous PHY clocking
        self.phy_lane_map = PhyLaneMap()  #: Physical lane data, pin swap is not applicable for mipi tx

        self.esc_clock_freq = 0.0  #: Escape clock frequency, MHz
        self.t_clk_post = 0
        self.t_clk_trail = 0
        self.t_clk_prepare = 0
        self.t_clk_zero = 0
        self.t_clk_pre = 0
        self.t_hs_prepare = 0
        self.t_hs_zero = 0
        self.t_hs_trail = 0

        if apply_default:
            self.set_default_setting()

    def __str__(self, *args, **kwargs):
        info = 'ref_clk_freq:{} phy_tx_freq_code:{} is_cont_phy_clk:{} pin_count:{}' \
            .format(self.ref_clock_freq,
                    self.phy_tx_freq_code,
                    self.is_cont_phy_clocking,
                    self.gen_pin.get_pin_count())
        return info

    def get_default_phy_freq_code(self):
        """
        Get default PHY Bandwidth value's code

        :return: A freq code, int
        """
        assert self._param_info is not None
        return self._param_info.get_default(MIPIParamInfo.Id.PHY_FREQ)

    def get_default_refclk_freq(self):
        """
        Get default reference clock frequency value

        :return: A freq value, float
        """
        assert self._param_info is not None
        return self._param_info.get_default(MIPIParamInfo.Id.REFCLK_FREQ)

    def get_default_escclk_freq(self):
        """
        Get default escape clock frequency value

        :return: A freq value, float
        """
        assert self._param_info is not None
        return self._param_info.get_default(MIPIParamInfo.Id.ESC_CLK_FREQ)

    def get_min_max_phy_freq_str(self):
        """
        Get minimum and maximum PHY Bandwidth value in string

        :return: A tuple of min, max frequencies
        """
        min_freq_str = self.get_phy_freq_str(self._min_freq_code)
        max_freq_str = self.get_phy_freq_str(self._max_freq_code)

        return min_freq_str, max_freq_str

    def get_timer_default_value(self, timer_type):
        return self.timer_def_value.get(timer_type, 1)

    def set_default_setting(self, device_db=None):
        assert self._param_info is not None
        self.ref_clock_freq = self._param_info.get_default(MIPIParamInfo.Id.REFCLK_FREQ)
        self.phy_tx_freq_code = self._param_info.get_default(MIPIParamInfo.Id.PHY_FREQ)
        self.is_cont_phy_clocking = self._param_info.get_default(MIPIParamInfo.Id.CONT_PHY_CLK_EN)

        # TODO: TIMING Default setting for timing stuff and confirms its data type
        self.esc_clock_freq = self._param_info.get_default(MIPIParamInfo.Id.ESC_CLK_FREQ)

        self.t_clk_post = self._param_info.get_default(MIPIParamInfo.Id.TCLK_POST)
        self.t_clk_trail = self._param_info.get_default(MIPIParamInfo.Id.TCLK_TRAIL)
        self.t_clk_prepare = self._param_info.get_default(MIPIParamInfo.Id.TCLK_PREPARE)
        self.t_clk_zero = self._param_info.get_default(MIPIParamInfo.Id.TCLK_ZERO)
        self.t_clk_pre = self._param_info.get_default(MIPIParamInfo.Id.TCLK_PRE)
        self.t_hs_prepare = self._param_info.get_default(MIPIParamInfo.Id.THS_PREPARE)
        self.t_hs_zero = self._param_info.get_default(MIPIParamInfo.Id.THS_ZERO)
        self.t_hs_trail = self._param_info.get_default(MIPIParamInfo.Id.THS_TRAIL)

    @property
    def ref_clock_freq(self):
        return self.param_group.get_param_value(MIPIParamInfo.Id.REFCLK_FREQ.value)

    @ref_clock_freq.setter
    def ref_clock_freq(self, value):
        self.param_group.set_param_value(MIPIParamInfo.Id.REFCLK_FREQ.value, value)

    @property
    def phy_tx_freq_code(self):
        return self.param_group.get_param_value(MIPIParamInfo.Id.PHY_FREQ.value)

    @phy_tx_freq_code.setter
    def phy_tx_freq_code(self, value):
        self.param_group.set_param_value(MIPIParamInfo.Id.PHY_FREQ.value, value)

    @property
    def is_cont_phy_clocking(self):
        return self.param_group.get_param_value(MIPIParamInfo.Id.CONT_PHY_CLK_EN.value)

    @is_cont_phy_clocking.setter
    def is_cont_phy_clocking(self, value):
        self.param_group.set_param_value(MIPIParamInfo.Id.CONT_PHY_CLK_EN.value, value)

    @property
    def esc_clock_freq(self):
        return self.param_group.get_param_value(MIPIParamInfo.Id.ESC_CLK_FREQ.value)

    @esc_clock_freq.setter
    def esc_clock_freq(self, value):
        self.param_group.set_param_value(MIPIParamInfo.Id.ESC_CLK_FREQ.value, value)

    @property
    def t_clk_post(self):
        return self.param_group.get_param_value(MIPIParamInfo.Id.TCLK_POST.value)

    @t_clk_post.setter
    def t_clk_post(self, value):
        self.param_group.set_param_value(MIPIParamInfo.Id.TCLK_POST.value, value)

    @property
    def t_clk_trail(self):
        return self.param_group.get_param_value(MIPIParamInfo.Id.TCLK_TRAIL.value)

    @t_clk_trail.setter
    def t_clk_trail(self, value):
        self.param_group.set_param_value(MIPIParamInfo.Id.TCLK_TRAIL.value, value)

    @property
    def t_clk_prepare(self):
        return self.param_group.get_param_value(MIPIParamInfo.Id.TCLK_PREPARE.value)

    @t_clk_prepare.setter
    def t_clk_prepare(self, value):
        self.param_group.set_param_value(MIPIParamInfo.Id.TCLK_PREPARE.value, value)

    @property
    def t_clk_zero(self):
        return self.param_group.get_param_value(MIPIParamInfo.Id.TCLK_ZERO.value)

    @t_clk_zero.setter
    def t_clk_zero(self, value):
        self.param_group.set_param_value("TCLK_ZERO", value)

    @property
    def t_clk_pre(self):
        return self.param_group.get_param_value(MIPIParamInfo.Id.TCLK_PRE.value)

    @t_clk_pre.setter
    def t_clk_pre(self, value):
        self.param_group.set_param_value(MIPIParamInfo.Id.TCLK_PRE.value, value)

    @property
    def t_hs_prepare(self):
        return self.param_group.get_param_value(MIPIParamInfo.Id.THS_PREPARE.value)

    @t_hs_prepare.setter
    def t_hs_prepare(self, value):
        self.param_group.set_param_value(MIPIParamInfo.Id.THS_PREPARE.value, value)

    @property
    def t_hs_zero(self):
        return self.param_group.get_param_value(MIPIParamInfo.Id.THS_ZERO.value)

    @t_hs_zero.setter
    def t_hs_zero(self, value):
        self.param_group.set_param_value(MIPIParamInfo.Id.THS_ZERO.value, value)

    @property
    def t_hs_trail(self):
        return self.param_group.get_param_value(MIPIParamInfo.Id.THS_TRAIL.value)

    @t_hs_trail.setter
    def t_hs_trail(self, value):
        self.param_group.set_param_value(MIPIParamInfo.Id.THS_TRAIL.value, value)

    def create_chksum(self):
        pass

    def build_param_info(self, param_info: MIPIParamInfo):
        """
        Build information about additional RX supported properties

        :param param_info: Property info
        """

        if param_info is not None:
            super().build_param_info(param_info)
            param_info.add_prop(MIPIParamInfo.Id.REFCLK_FREQ, MIPIParamInfo.Id.REFCLK_FREQ.value, db_item.GenericParam.DataType.dflo, 6.0, self.get_ref_clk_ops())
            param_info.add_prop(MIPIParamInfo.Id.REFCLK_RES, MIPIParamInfo.Id.REFCLK_RES.value, db_item.GenericParam.DataType.dstr, "")
            param_info.add_prop(MIPIParamInfo.Id.REFCLK_INST, MIPIParamInfo.Id.REFCLK_INST.value, db_item.GenericParam.DataType.dstr, "")
            param_info.add_prop(MIPIParamInfo.Id.TCLK_POST, MIPIParamInfo.Id.TCLK_POST.value, db_item.GenericParam.DataType.dint, self.get_timer_default_value(MIPITx.TimerType.clk_post))
            param_info.add_prop(MIPIParamInfo.Id.TCLK_TRAIL, MIPIParamInfo.Id.TCLK_TRAIL.value, db_item.GenericParam.DataType.dint, self.get_timer_default_value(MIPITx.TimerType.clk_trail))
            param_info.add_prop(MIPIParamInfo.Id.TCLK_PREPARE, MIPIParamInfo.Id.TCLK_PREPARE.value, db_item.GenericParam.DataType.dint, self.get_timer_default_value(MIPITx.TimerType.clk_prepare))
            param_info.add_prop(MIPIParamInfo.Id.TCLK_ZERO, MIPIParamInfo.Id.TCLK_ZERO.value, db_item.GenericParam.DataType.dint, self.get_timer_default_value(MIPITx.TimerType.clk_zero))
            param_info.add_prop(MIPIParamInfo.Id.TCLK_PRE, MIPIParamInfo.Id.TCLK_PRE.value, db_item.GenericParam.DataType.dint, self.get_timer_default_value(MIPITx.TimerType.clk_pre))
            param_info.add_prop(MIPIParamInfo.Id.THS_PREPARE, MIPIParamInfo.Id.THS_PREPARE.value, db_item.GenericParam.DataType.dint, self.get_timer_default_value(MIPITx.TimerType.hs_prepare))
            param_info.add_prop(MIPIParamInfo.Id.THS_ZERO, MIPIParamInfo.Id.THS_ZERO.value, db_item.GenericParam.DataType.dint, self.get_timer_default_value(MIPITx.TimerType.hs_zero))
            param_info.add_prop(MIPIParamInfo.Id.THS_TRAIL, MIPIParamInfo.Id.THS_TRAIL.value, db_item.GenericParam.DataType.dint, self.get_timer_default_value(MIPITx.TimerType.hs_trail))

            phy_freq_map = self.get_phy_freq_map()
            freq_options = []
            if phy_freq_map is not None:
                freq_options = [phy_freq for phy_freq in phy_freq_map.values()]

            # Default is 1.5GHz
            param_info.add_prop(MIPIParamInfo.Id.PHY_FREQ, MIPIParamInfo.Id.PHY_FREQ.value, db_item.GenericParam.DataType.dint, 364, freq_options)
            param_info.add_prop(MIPIParamInfo.Id.CONT_PHY_CLK_EN, MIPIParamInfo.Id.CONT_PHY_CLK_EN.value, db_item.GenericParam.DataType.dbool,
                                False)
            param_info.add_prop(MIPIParamInfo.Id.ESC_CLK_INVERTED_EN, MIPIParamInfo.Id.ESC_CLK_INVERTED_EN.value, db_item.GenericParam.DataType.dbool,
                                False)

            esc_range = self.get_esc_clk_freq_range()
            param_info.add_prop(MIPIParamInfo.Id.ESC_CLK_FREQ, MIPIParamInfo.Id.ESC_CLK_FREQ.value, db_item.GenericParam.DataType.dflo, 20.00,
                                esc_range)
            lane_map_param = [
                (MIPIParamInfo.Id.TXD0_LANE, PhyLane.LogLaneIdType.data0_lane),
                (MIPIParamInfo.Id.TXD1_LANE, PhyLane.LogLaneIdType.data1_lane),
                (MIPIParamInfo.Id.TXD2_LANE, PhyLane.LogLaneIdType.clk_lane),
                (MIPIParamInfo.Id.TXD3_LANE, PhyLane.LogLaneIdType.data2_lane),
                (MIPIParamInfo.Id.TXD4_LANE, PhyLane.LogLaneIdType.data3_lane),
            ]
            lane_map_options = list(PhyLane.LogLaneIdType)
            for lane_info in lane_map_param:
                param_id, default_val = lane_info
                param_info.add_prop(param_id, param_id.value, db_item.GenericParam.DataType.denum, default_val,
                                lane_map_options)

    def get_esc_clk_freq_range(self) -> Tuple[float, float]:
        from device.db_interface import DeviceDBService

        esc_range: Tuple[float, float] = (0.0, 0.0)
        if self.device is not None:
            dbi = DeviceDBService(self.device)
            dev_service = dbi.get_block_service(DeviceDBService.BlockType.MIPI_TX)
            assert isinstance(dev_service, MIPITxService)
            min_freq, max_freq = dev_service.get_min_max_esc_freq()
            esc_range = (min_freq, max_freq)

        return esc_range

    def get_ref_clk_ops(self):
        return [6.0, 12.0, 19.2, 25.0, 26.0, 27.0, 38.4, 52.0]

    def generate_pin_name_from_inst(self, inst_name):
        self.gen_pin.generate_pin_name(inst_name)

        pin = self.gen_pin.get_pin_by_type_name("ESC_CLK")
        if pin is not None:
            pin.name = ""

        pin = self.gen_pin.get_pin_by_type_name("PIXEL_CLK")
        if pin is not None:
            pin.name = ""

    def refresh_pin_name(self, old_substr, new_substr):
        """
        Override
        """
        self.gen_pin.refresh_pin_name(old_substr, new_substr)

    @staticmethod
    def build_port_info(device_db=None, is_mockup=False):
        """
        Build port info for MIPI TX block from definition in device db.
        This function need to be call only once since it is static shared
        between class.

        :param device_db:
        :param is_mockup: True, build a mockup data, else build from device db
        """

        if is_mockup:
            raise NotImplementedError

        else:
            if device_db is None:
                return

            # Pull info from device db
            from device.db_interface import DeviceDBService

            dbi = DeviceDBService(device_db)
            dev_service = dbi.get_block_service(DeviceDBService.BlockType.MIPI_TX)

            MIPITx._device_port_map = dev_service.get_ports()

            freqstr_map = dev_service.get_code_to_phy_frequencies_map()
            MIPITx._phy_freq_map = {}
            MIPITx._phy_freq2code = {}

            for code, freq in freqstr_map.items():
                MIPITx._phy_freq_map[int(code)] = float(freq)
                MIPITx._phy_freq2code[float(freq)] = int(code)

    def build_generic_pin(self):
        """
        Dynamically generate generic pins for tx based on device info
        """
        assert self._device_port_map is not None
        self.gen_pin.clear()

        from device.block_definition import Port as DevicePort

        for device_port in self._device_port_map.values():
            # Ref clock is hardwired, skip it
            if device_port.get_name() == "REF_CLK":
                continue

            if device_port.get_type() == DevicePort.TYPE_CLOCK:
                self.gen_pin.add_clock_pin(device_port.get_name(), "", device_port.is_bus_port())
            else:
                self.gen_pin.add_pin(device_port.get_name(), "", device_port.is_bus_port())

    def get_pin_property_name(self, pin_type_name):
        """
        Based on device pin type, get its pin name

        :param pin_type_name: Device pin type name
        :return: Pin name if found, else empty string
        """
        assert self._device_port_map is not None
        device_port = self._device_port_map.get(pin_type_name, None)
        if device_port is not None:
            return device_port.get_description()

        return ""

    def get_pin_class(self, pin_type_name):
        """
        Based on device pin type, get its pin class

        :param pin_type_name: Device pin type name
        :return: Device pin class if found, else None
        """
        assert self._device_port_map is not None
        device_port = self._device_port_map.get(pin_type_name, None)
        if device_port is not None:
            return device_port.get_class()
        return None

    def get_pin_by_class(self, device_class):
        """
        Get a list of pin for target class

        :param device_class:
        :return: A list of pin, if any
        """
        pin_list = []
        for pin in self.gen_pin.get_all_pin():
            if self.get_pin_class(pin.type_name) == device_class:
                pin_list.append(pin)

        return pin_list

    def is_match_pin_name(self, text):
        """
        Override
        """
        return self.gen_pin.is_match_pin_name(text)

    def get_phy_freq_map(self):
        """
        Get a map of PHY Bandwidth code to its frequency value

        :return: A map
        """
        return MIPITx._phy_freq_map

    def get_phy_freq_str(self, freq_code=None):
        """
        Based on configured PHY Bandwidth in code, returns a formatted string value

        :param freq_code: Frequency code to convert. If None, use the tx code
        :return: A frequency value string
        """
        assert MIPITx._phy_freq_map is not None

        if freq_code is None:
            freq = MIPITx._phy_freq_map.get(self.phy_tx_freq_code, None)
        else:
            freq = MIPITx._phy_freq_map.get(freq_code, None)

        if freq is None:
            freq = MIPITx._phy_freq_map.get(self.get_default_phy_freq_code(), 0.0)

        return gen_util.get_freq_str(freq)

    def get_ref_clock_freq_str(self):
        return gen_util.get_freq_str(self.ref_clock_freq)

    def get_esc_clock_freq_str(self):
        return gen_util.get_freq_str(self.esc_clock_freq)

    def find_refclock_info(self, mipi_def, mipi_tx_dev_service, gpio_reg):
        """
        For the assigned MIPI device resource, find the GPIO resource name and its instance info.

        :param mipi_def: MIPI resource name
        :param mipi_tx_dev_service: MIPI TX device service object
        :param gpio_reg : GPIO registry object

        :return: A resource, instance, input pin name tuple, (resource name, instance name, input pin name)
        """
        if mipi_def == "" or mipi_tx_dev_service is None or gpio_reg is None:
            return "unknown", "unknown", "unknown"

        inst_name = "unknown"
        pin_name = "unknown"

        res_name = mipi_tx_dev_service.get_resource_on_ref_clk_pin(mipi_def)
        if res_name is not None:
            gpio_inst = gpio_reg.get_gpio_by_device_name(res_name)
            if gpio_inst is not None:
                inst_name = gpio_inst.name
                # It must be GPIO in input mode
                if gpio_inst.input is not None:
                    pin_name = gpio_inst.input.name

        if res_name is None:
            res_name = "unknown"

        return res_name, inst_name, pin_name

    def is_reset_pin_configured(self):
        """
        Check if any of the reset pin name is non-empty.

        :return True if any pin is configured, else False
        """

        from device.block_definition import Port as DevicePort

        if self._device_port_map is not None:
            # iterate through _device_port_map to find port of type reset
            for dev_port in self._device_port_map.values():
                if dev_port is not None and dev_port.get_type() == DevicePort.TYPE_RESET:
                    # Check if the generic pin is nonempty
                    reset_gen_pin = self.gen_pin.get_pin_by_type_name(dev_port.get_name())
                    if reset_gen_pin is not None and reset_gen_pin.name != "":
                        return True

        return False

    def is_pin_type_clock(self, pin_type_name):
        """
        Based device pin type, check if it is a clock pin

        :param pin_type_name: Device pin type name
        :return: True, if clock, else False

        .. note::
           For unit test, this will always return False
        """

        if self._device_port_map is not None:

            from device.block_definition import Port as DevicePort

            device_port = self._device_port_map.get(pin_type_name, None)
            if device_port is not None:
                dev_port_type = device_port.get_type()
                return dev_port_type == DevicePort.TYPE_CLOCK

        return False


@gen_util.freeze_it
class MIPIRx(MIPIBase):
    """
    Specific mipi instance property in rx mode
    """

    class TimerType(IntEnum):
        clk_settle = 21
        hs_settle = 22

    _device_port_map = {}  #: A map of device port type to its info, linked to device db

    def __init__(self, apply_default=True):
        self.timer_def_value = {
            MIPIRx.TimerType.clk_settle: 9,
            MIPIRx.TimerType.hs_settle: 8
        }
        super().__init__()

        # self.gen_pin = db_item.GenericPinGroup()  #: Other pins with generic properties
        self.phy_lane_map = PhyLaneMap()  #: Physical lane data, pin swap is applicable for mipi rx

        self.is_status_en = False
        self.dphy_calib_clock_freq = 0.0  #: DPHY calibration clock frequency, MHz
        self.t_clk_settle = 0  #: Settle timer parameter for clock lane, integer type code
        self.t_hs_settle = 0  #: Settle timer parameter to all data lane, integer type code

        if apply_default:
            self.set_default_setting()

    def __str__(self, *args, **kwargs):
        info = 'pin_count:{}'.format(self.gen_pin.get_pin_count())
        return info

    def get_default_calibclk_freq(self):
        """
        Get default calibration clock frequency value

        :return: A freq value, float
        """
        assert self._param_info is not None
        return self._param_info.get_default(MIPIParamInfo.Id.CAL_CLK_FREQ)

    def set_default_setting(self, device_db=None):
        self.clear_all_status_pin()
        assert self._param_info is not None
        self.is_status_en = False

        # TODO: TIMING Default setting for timing stuff and confirms its data type
        self.dphy_calib_clock_freq = self._param_info.get_default(MIPIParamInfo.Id.CAL_CLK_FREQ) # type: ignore
        self.t_clk_settle = self._param_info.get_default(MIPIParamInfo.Id.TCLK_SETTLE)
        self.t_hs_settle = self._param_info.get_default(MIPIParamInfo.Id.THS_SETTLE)

    def get_timer_default_value(self, timer_type):
        return self.timer_def_value.get(timer_type, 1)

    def create_chksum(self):
        pass

    def generate_pin_name_from_inst(self, inst_name):
        self.gen_pin.generate_pin_name(inst_name)

        pin = self.gen_pin.get_pin_by_type_name("CAL_CLK")
        if pin is not None:
            pin.name = ""

        pin = self.gen_pin.get_pin_by_type_name("PIXEL_CLK")
        if pin is not None:
            pin.name = ""

    def refresh_pin_name(self, old_substr, new_substr):
        """
        Override
        """
        self.gen_pin.refresh_pin_name(old_substr, new_substr)

    @staticmethod
    def build_port_info(device_db=None, is_mockup=False):
        """
        Build port info for MIPI RX block from definition in device db.
        This function need to be call only once since it is static shared
        between class.

        :param device_db:
        :param is_mockup: True, build a mockup data, else build from device db
        """

        if is_mockup:
            raise NotImplementedError
        else:
            if device_db is None:
                return

            from device.db_interface import DeviceDBService

            # Pull info from device db
            dbi = DeviceDBService(device_db)
            dev_service = dbi.get_block_service(DeviceDBService.BlockType.MIPI_RX)

            MIPIRx._device_port_map = dev_service.get_ports()

    def build_generic_pin(self):
        """
        Dynamically generate generic pins for rx based on device info
        """
        assert self._device_port_map is not None
        self.gen_pin.clear()

        from device.block_definition import Port as DevicePort

        for device_port in self._device_port_map.values():

            if device_port.get_type() == DevicePort.TYPE_CLOCK:
                self.gen_pin.add_clock_pin(device_port.get_name(), "", device_port.is_bus_port())
            else:
                self.gen_pin.add_pin(device_port.get_name(), "", device_port.is_bus_port())

    def get_pin_property_name(self, pin_type_name):
        """
        Based on device pin type, get its pin name

        :param pin_type_name: Device pin type name
        :return: Pin name if found, else empty string
        """
        assert self._device_port_map is not None
        device_port = self._device_port_map.get(pin_type_name, None)
        if device_port is not None:
            return device_port.get_description()

        return ""

    def get_pin_class(self, pin_type_name):
        """
        Based on device pin type, get its pin class

        :param pin_type_name: Device pin type name
        :return: Design pin class if found, else unknown, :class:`PinClassType`
        """
        assert self._device_port_map is not None
        device_port = self._device_port_map.get(pin_type_name, None)
        if device_port is not None:
            return device_port.get_class()

        return ""

    def get_pin_by_class(self, device_class_name):
        """
        Get a list of pin for target class

        :param device_class_name:
        :return: A list of pin, if any
        """
        pin_list = []
        for pin in self.gen_pin.get_all_pin():
            if self.get_pin_class(pin.type_name) == device_class_name:
                pin_list.append(pin)

        return pin_list

    def is_match_pin_name(self, text):
        """
        Override
        """
        return self.gen_pin.is_match_pin_name(text)

    def is_reset_pin_configured(self):
        """
        Check if any of the reset pin name is non-empty.

        :return True if any pin is configured, else False
        """

        from device.block_definition import Port as DevicePort

        if self._device_port_map is not None:
            # iterate through _device_port_map to find port of type reset
            for dev_port in self._device_port_map.values():
                if dev_port is not None and dev_port.get_type() == DevicePort.TYPE_RESET:
                    # Check if the generic pin is nonempty
                    reset_gen_pin = self.gen_pin.get_pin_by_type_name(dev_port.get_name())
                    if reset_gen_pin is not None and reset_gen_pin.name != "":
                        return True

        return False

    def is_status_pin_configured(self):
        """
        Check if any of the status pin name is non-empty.

        :return: True if any pin is configured, else False
        """
        return self.is_status_en

    def clear_all_status_pin(self):
        """
        Reset all status pin name
        """
        from device.block_definition import Port as DevicePort

        status_pin_list = self.get_pin_by_class(DevicePort.CLASS_MIPI_STATUS)
        for pin in status_pin_list:
            pin.name = ""

    def generate_status_pin_name_from_inst(self, inst_name):
        """
        Generate all status pin names

        :param inst_name: Parent mipi instance name
        """

        from device.block_definition import Port as DevicePort

        status_pin_list = self.get_pin_by_class(DevicePort.CLASS_MIPI_STATUS)
        for pin in status_pin_list:
            pin.generate_pin_name(inst_name)

    def is_pin_type_clock(self, pin_type_name):
        """
        Based device pin type, check if it is a clock pin.

        :param pin_type_name: Device pin type name
        :return: True, if clock, else False

        .. note::
           For unit test, this will always return False
        """

        if self._device_port_map is not None:
            from device.block_definition import Port as DevicePort
            device_port = self._device_port_map.get(pin_type_name, None)
            if device_port is not None:
                dev_port_type = device_port.get_type()
                return dev_port_type == DevicePort.TYPE_CLOCK

        return False

    def get_dphy_calib_clock_freq_str(self):
        return gen_util.get_freq_str(self.dphy_calib_clock_freq)

    @property
    def is_status_en(self):
        return self.param_group.get_param_value(MIPIParamInfo.Id.STATUS_EN.value)

    @is_status_en.setter
    def is_status_en(self, value: bool):
        self.param_group.set_param_value(MIPIParamInfo.Id.STATUS_EN.value, value)

    @property
    def dphy_calib_clock_freq(self) -> float:
        return self.param_group.get_param_value(MIPIParamInfo.Id.CAL_CLK_FREQ.value)

    @dphy_calib_clock_freq.setter
    def dphy_calib_clock_freq(self, value: float):
        return self.param_group.set_param_value(MIPIParamInfo.Id.CAL_CLK_FREQ.value, value)

    @property
    def t_clk_settle(self):
        return self.param_group.get_param_value(MIPIParamInfo.Id.TCLK_SETTLE.value)

    @t_clk_settle.setter
    def t_clk_settle(self, value):
        return self.param_group.set_param_value(MIPIParamInfo.Id.TCLK_SETTLE.value, value)

    @property
    def t_hs_settle(self):
        return self.param_group.get_param_value(MIPIParamInfo.Id.THS_SETTLE.value)

    @t_hs_settle.setter
    def t_hs_settle(self, value):
        return self.param_group.set_param_value(MIPIParamInfo.Id.THS_SETTLE.value, value)

    def build_param_info(self, param_info: MIPIParamInfo):
        """
        Build information about additional RX supported properties

        :param param_info: Property info
        """

        if param_info is not None:
            super().build_param_info(param_info)
            param_info.add_prop(MIPIParamInfo.Id.STATUS_EN, MIPIParamInfo.Id.STATUS_EN.value, db_item.GenericParam.DataType.dbool,
                                False)

            param_info.add_prop(MIPIParamInfo.Id.CAL_CLK_INVERTED_EN, MIPIParamInfo.Id.CAL_CLK_INVERTED_EN.value, db_item.GenericParam.DataType.dbool,
                                False)
            freq_range = self.get_cal_clk_freq()
            param_info.add_prop(MIPIParamInfo.Id.CAL_CLK_FREQ, MIPIParamInfo.Id.CAL_CLK_FREQ.value, db_item.GenericParam.DataType.dflo,
                                100.00, valid_setting=freq_range)
            param_info.add_prop(MIPIParamInfo.Id.TCLK_SETTLE, MIPIParamInfo.Id.TCLK_SETTLE.value, db_item.GenericParam.DataType.dint, self.get_timer_default_value(MIPIRx.TimerType.clk_settle))
            param_info.add_prop(MIPIParamInfo.Id.THS_SETTLE, MIPIParamInfo.Id.THS_SETTLE.value, db_item.GenericParam.DataType.dint, self.get_timer_default_value(MIPIRx.TimerType.hs_settle))

            lane_map_param = [
                (MIPIParamInfo.Id.RXD0_LANE, PhyLane.LogLaneIdType.data0_lane),
                (MIPIParamInfo.Id.RXD1_LANE, PhyLane.LogLaneIdType.data1_lane),
                (MIPIParamInfo.Id.RXD2_LANE, PhyLane.LogLaneIdType.clk_lane),
                (MIPIParamInfo.Id.RXD3_LANE, PhyLane.LogLaneIdType.data2_lane),
                (MIPIParamInfo.Id.RXD4_LANE, PhyLane.LogLaneIdType.data3_lane),
            ]
            lane_map_options = list(PhyLane.LogLaneIdType)
            for lane_info in lane_map_param:
                param_id, default_val = lane_info
                param_info.add_prop(param_id, param_id.value, db_item.GenericParam.DataType.denum, default_val,
                                lane_map_options)

            lane_swap = [
                MIPIParamInfo.Id.RXD0_PN_SWAP,
                MIPIParamInfo.Id.RXD1_PN_SWAP,
                MIPIParamInfo.Id.RXD2_PN_SWAP,
                MIPIParamInfo.Id.RXD3_PN_SWAP,
                MIPIParamInfo.Id.RXD4_PN_SWAP,
            ]
            for param_id in lane_swap:
                param_info.add_prop(param_id, param_id.value, db_item.GenericParam.DataType.dbool, False)

    def get_cal_clk_freq(self):
        from device.db_interface import DeviceDBService
        from common_device.mipi.mipi_rx_device_service import MIPIRxService

        freq_range: Tuple[float, float] = (0.0, 0.0)
        if self.device is not None:
            dbi = DeviceDBService(self.device)
            dev_service = dbi.get_block_service(DeviceDBService.BlockType.MIPI_RX)
            assert isinstance(dev_service, MIPIRxService)
            min_freq, max_freq = dev_service.get_min_max_calib_freq()
            freq_range = (min_freq, max_freq)

        return freq_range


@gen_util.freeze_it
class MIPI(db_item.PeriDesignItem):
    """
    A mipi block instance
    """

    class OpsType(Enum):
        """
        Operation type. Use as tx or rx.
        """
        op_tx = 1
        op_rx = 2
        unknown = 3

    opstype2str_map = \
        {
            OpsType.op_tx: "tx",
            OpsType.op_rx: "rx"
        }  #: Mapping of MIPI type in enum to string type

    str2opstype_map = \
        {
            "tx": OpsType.op_tx,
            "rx": OpsType.op_rx
        }  #: Mapping of MIPI type in string to enum type

    def __init__(self, name, mipi_def="", apply_default=True):
        super().__init__()

        self.name = name
        self.mipi_def = mipi_def
        self.ops_type = None
        self.tx_info = None  #: mipi use in tx mode
        self.rx_info = None  #: mipi use in rx mode

        if apply_default:
            self.set_default_setting()

    def __str__(self, *args, **kwargs):

        info = ""
        if self.tx_info is not None:
            info = "{}".format(self.tx_info)

        if self.rx_info is not None:
            info = "{}".format(self.rx_info)

        info = 'name:{} mipi_def:{} op_type:{} info:({})' \
            .format(self.name, self.mipi_def, self.ops_type, info)

        return info

    def get_device(self):
        return self.mipi_def

    def set_device(self, device_name):
        self.mipi_def = device_name

    def set_default_setting(self, device_db=None):
        if self.ops_type == MIPI.OpsType.op_tx:
            self.tx_info = MIPITx()
            self.tx_info.device = device_db
            mipi = self.tx_info
        elif self.ops_type == MIPI.OpsType.op_rx:
            self.rx_info = MIPIRx()
            self.rx_info.device = device_db
            mipi = self.rx_info
        else:
            # If not configured, it means right after creation. Should be changed
            # after user input is processed.
            self.tx_info = MIPITx()
            self.tx_info.device = device_db
            mipi = self.tx_info
            self.ops_type = MIPI.OpsType.op_tx

        if mipi is not None:
            mipi.set_default_setting(device_db)

    def create_chksum(self):
        pass

    def update_type(self, mipi_type, device_db=None, auto_pin=False):
        self.ops_type = mipi_type
        if self.ops_type == MIPI.OpsType.op_tx:
            self._set_tx_type(device_db, auto_pin)
        elif self.ops_type == MIPI.OpsType.op_rx:
            self._set_rx_type(device_db, auto_pin)

    def _set_tx_type(self, device_db=None, auto_pin=False):
        """
        Set to TX type.
        Clear existing tx/rx data and reset to defaults.
        """
        self.rx_info = None
        self.tx_info = MIPITx()
        self.ops_type = MIPI.OpsType.op_tx
        self.tx_info.set_default_setting(device_db)

        if auto_pin:
            self.generate_pin_name()

    def _set_rx_type(self, device_db=None, auto_pin=False):
        """
        Set to RX type.
        Clear existing tx/rx data and reset to defaults.
        """
        self.tx_info = None
        self.rx_info = MIPIRx()
        self.ops_type = MIPI.OpsType.op_rx
        self.rx_info.set_default_setting(device_db)

        # Auto pin name not supported for internal pins
        if auto_pin:
            self.generate_pin_name()

            # Ensure that status are not set as a default
            self.rx_info.clear_all_status_pin()

    def build_generic_pin(self):
        """
        Build simple pins design
        """
        if self.ops_type == MIPI.OpsType.op_tx:
            mipi_info = self.tx_info
        else:
            mipi_info = self.rx_info

        assert mipi_info is not None
        mipi_info.build_generic_pin()

    def generate_pin_name(self):

        if self.ops_type == MIPI.OpsType.op_rx:
            if self.rx_info is not None:
                self.rx_info.generate_pin_name_from_inst(self.name)

        elif self.ops_type == MIPI.OpsType.op_tx:
            if self.tx_info is not None:
                self.tx_info.generate_pin_name_from_inst(self.name)

    def refresh_pin_name(self, old_substr, new_substr):
        if self.ops_type == MIPI.OpsType.op_rx:
            if self.rx_info is not None:
                self.rx_info.refresh_pin_name(old_substr, new_substr)

        elif self.ops_type == MIPI.OpsType.op_tx:
            if self.tx_info is not None:
                self.tx_info.refresh_pin_name(old_substr, new_substr)

    def is_tx(self):
        return self.ops_type == self.OpsType.op_tx and \
                self.tx_info is not None and \
                self.rx_info is None

    def is_rx(self):
        return self.ops_type == self.OpsType.op_rx and \
                self.rx_info is not None and \
                self.tx_info is None


@gen_util.freeze_it
class MIPIRegistry(db_item.PeriDesignRegistry):
    """
    A registry of all mipi instances
    """

    def __init__(self):
        self.timing_calc: Optional[MIPITimingCalculator] = None  #: Timing calculator
        self._name2inst_map: Dict[str, MIPI]
        super().__init__()

    def get_timing_calc(self):
        return self.timing_calc

    def create_instance(self, name, apply_default=True, auto_pin=False):
        """
        Create and register a mipi instance

        :param name: Instance name
        :param apply_default: True to apply default setting as per spec
        :param auto_pin:
        :return: A mipi instance
        :raise: db_item.CreateException
        """
        with self._write_lock:
            mipi = MIPI(name)
            if apply_default:
                mipi.set_default_setting(self.device_db)

            if auto_pin:
                mipi.generate_pin_name()

            self._register_new_instance(mipi)

            return mipi

    def create_tx_instance(self, name, apply_default=True, auto_pin=False):
        """
        Create and register a tx mipi instance

        :param auto_pin:
        :param name: Instance name
        :param apply_default: True to apply default setting as per spec
        :return: A mipi instance
        :raise: db_item.CreateException
        """
        return self.create_instance(name, apply_default, auto_pin)

    def create_rx_instance(self, name, apply_default=True, auto_pin=False):
        """
        Create and register a rx mipi instance

        :param auto_pin:
        :param name: Instance name
        :param apply_default: True to apply default setting as per spec
        :return: A mipi instance
        :raise: db_item.CreateException
        """
        mipi = self.create_instance(name, apply_default, auto_pin)
        mipi.update_type(MIPI.OpsType.op_rx, self.device_db, auto_pin)
        return mipi

    def is_resource_tx(self, mipi_device):
        """
        Check if a resource name is TX or RX resource
        :param mipi_device: MIPI resource name
        :return: True, if TX else RX
        """

        lc_device = mipi_device.lower()
        if "tx" in lc_device:
            return True

        # This contains either RX or CLK in the name
        return False

    def is_device_valid(self, device_def):
        """
        Override
        """

        # Doing import at top level cause circular import problem
        import device.db_interface as devdb_int
        dbi = devdb_int.DeviceDBService(self.device_db)
        if self.is_resource_tx(device_def):
            svc = dbi.get_block_service(devdb_int.DeviceDBService.BlockType.MIPI_TX)
        else:
            svc = dbi.get_block_service(devdb_int.DeviceDBService.BlockType.MIPI_RX)

        dev_list = svc.get_usable_instance_names()
        if device_def not in dev_list:
            return False

        return True

    def apply_device_db(self, device_db):
        """
        Set device db and use it to extract device dependent data.
        This must be called before using the registry.

        :param device_db: Device db instance
        """

        if device_db is None:
            return

        self.device_db = device_db

        # Extract generic pin info from device
        MIPITx.build_port_info(self.device_db, False)
        MIPIRx.build_port_info(self.device_db, False)

        # Cache min-max MIPI Tx PHY freq limits for easy access
        from device.db_interface import DeviceDBService

        dbi = DeviceDBService(device_db)
        dev_service = dbi.get_block_service(DeviceDBService.BlockType.MIPI_TX)

        min_code, max_code = dev_service.get_min_max_phy_freq()
        MIPITx._min_freq_code = min_code
        MIPITx._max_freq_code = max_code

    def is_apply_port_with_change_device(self):
        is_apply = False
        if MIPITx._device_port_map is not None:
            is_apply = True

        if MIPIRx._device_port_map is not None:
            is_apply = True

        return is_apply

    def change_device(self, device_db):
        """
        Override. MIPI uses dynamic pins, need to re-generate pin info from device db
        before migrating pin design data.
        """
        self.apply_device_db(device_db)

        if self.is_apply_port_with_change_device():
            super().change_device(device_db)

    def get_type_inst_count(self, is_tx):
        count = 0

        # We check depending on the ops_type since the resource
        # can be None
        for ins in self._name2inst_map.values():
            if ins is not None:
                if is_tx and ins.ops_type == MIPI.OpsType.op_tx:
                    count += 1
                elif not is_tx and ins.ops_type == MIPI.OpsType.op_rx:
                    count += 1

        return count

    def get_all_inst(self) -> List[MIPI]:
        return super().get_all_inst()

    def delete_all_tx_inst(self):
        for inst in self.get_all_inst():
            if inst.ops_type == MIPI.OpsType.op_tx:
                self.delete_inst(inst.name)

    def delete_all_rx_inst(self):
        for inst in self.get_all_inst():
            if inst.ops_type == MIPI.OpsType.op_rx:
                self.delete_inst(inst.name)

    def vertical_device_migration(self, device_db, old2new_resource_map):
        """
        Override (using change_device as a reference) now that
        the device db has changed.
        """

        self.apply_device_db(device_db)

        is_apply = False
        if MIPITx._device_port_map is not None:
            is_apply = True

        if MIPIRx._device_port_map is not None:
            is_apply = True

        if is_apply:
            super().vertical_device_migration(device_db, old2new_resource_map)


if __name__ == "__main__":
    pass
