'''
Copyright (C) 2017-2021 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Jun 17, 2021

@author: maryam
'''
import os
import sys
import gc

import util.gen_util as pt_util
import util.excp as app_excp

import device.excp as dev_excp
import design.db as des_db

import common_device.writer as tbi
from common_device.hyper_ram.writer.interface_config import HyperRAMInterfaceConfig
from common_device.hsio.writer.gpio_timing import HSIOGPIOTiming
from tx60_device.gpio.gpio_design_service_comp import GPIODesignServiceComplex

sys.path.append(os.path.join(os.path.dirname(__file__), '..'))


class HyperRAMTiming(tbi.BlockTimingWithArcs):
    '''
    Builds the timing data for SPI Flash used for printing out
    timing report and sdc file.
    '''

    def __init__(self, bname, device, design, report, sdc,
                 max_model, min_model, ins2iostd_map):
        '''
        Constructor
        '''
        super().__init__(bname, device, design, report, sdc,
                         max_model, min_model)

        # We're building it a bit different since there's no
        # really a mode for this block. The mode is actually
        # to differentiate between input and outupt arcs
        self._mode2arcs = {}

        # Map of instance to io standard
        self._ins2iostd_map = ins2iostd_map

    def build_arc(self):
        """
        Build Timing Arcs
        """
        pass

    def write(self, index, ins_to_block_map, clk2delay_map):
        '''
        Write out the report and sdc constraint for this block.

        :param index: The index of this section. Used in printing to report
        :param ins_to_block_map: Map of device instance name to the block name

        :return True if there was any valid instance found

        '''
        pt_util.mark_unused(clk2delay_map)

        ins_exists = False

        try:
            # Iterate through the list of oscillator
            hram_reg = self._design.get_block_reg(
                des_db.PeriDesign.BlockType.hyper_ram)

            if hram_reg is not None:

                ins_name2obj_map = hram_reg.get_name2inst_map()

                # Build the info if the instance exists
                if ins_name2obj_map:

                    ins_exists = self.write_output(
                        index, ins_to_block_map,
                        ins_name2obj_map)

        except Exception as excp:
            raise excp

        return ins_exists

    def _get_used_config_file_name(self, hram_blk):
        '''
        :return the filename to use based on the instance configuration
        '''

        # Check the parameter dq_ext_clk
        used_cfg_file = hram_blk.get_config_file("used")

        if used_cfg_file == "":
            msg = "No used config files for {} defined".format(self._name)
            raise dev_excp.ConfigurationInvalidException(
                msg, app_excp.MsgLevel.error)

        return used_cfg_file

    def _load_used_config_design(self, used_cfg_file):
        '''
        :param used_cfg_file: The config file that contain the used configuration
        :return a gpio registry
        '''

        gpio_service = GPIODesignServiceComplex()
        local_gpio_reg = gpio_service.load_design(used_cfg_file)
        # Turn off the check since the file is internally controlled
        # if not gpio_service.check_design_file(used_cfg_file):
        #    raise ValueError(
        #        "Failure parsing SPIFlash config file: {}".format(used_cfg_file))

        return local_gpio_reg

    def write_output(self, index, ins_to_block_map,
                     ins_name2obj_map):
        '''
        :param index: The index of this section. Used in printing to report
        :param ins_to_block_map: Map of device instance name to the block name
        :param ins_name2obj_map: A map of the jtag design instance name to the design object

        :return True if there was any jtag instance
                that got printed
        '''

        ins_written = False
        write_successful = None

        rptfile = None
        sdcfile = None

        try:
            write_successful = False

            # Get the device block definition
            ref_blk = self._device.find_block(self._name)
            if ref_blk is None:
                raise ValueError("Unable to find block definition for {}".format(self._name))

            # Get the block definition
            hsio_block = self._device.find_block("hsio")

            if hsio_block is None:
                msg = "Unable to find block {}".format(self._name)

                raise dev_excp.BlockDoesNotExistException(
                    msg, app_excp.MsgLevel.error)

            # TODO: We're assuming that routing delay is not on device supporting HyperRAM
            # Hence, we'll pass empty container            
            gpio_writer = HSIOGPIOTiming(
                "hsio", self._device, self._design, self._report_file, self._sdc_file,
                self._max_model, self._min_model, self._ins2iostd_map, {}, {}, {})

            # Disable printing the report table header and the instance name column
            # along with other embedded instance reporting
            gpio_writer.print_header = False

            # Get the used config file
            used_cfg_file = self._get_used_config_file_name(ref_blk)

            # Read the contents of the file by calling the GPIODesignService
            local_gpio_reg = self._load_used_config_design(used_cfg_file)

            # call garbage collector since we create a registry
            gc.collect()

            hram_icfg = HyperRAMInterfaceConfig(
                self._device, self._name)

            if ins_name2obj_map:
                # Open the file
                rptfile = open(self._report_file, 'a')
                sdcfile = open(self._sdc_file, 'a')

                sdcfile.write("\n# HyperRAM Constraints\n")
                sdcfile.write("########################\n")

                rptfile.write(
                    "\n---------- {}. HyperRAM Timing Report (begin) ----------\n".format(index))

                # Close it so that the GPIO will use it
                sdcfile.close()
                sdcfile = None

                rptfile.close()
                rptfile = None

            for ins_name in ins_name2obj_map.keys():
                ins_obj = ins_name2obj_map[ins_name]

                # Work on instance that are part of device
                if ins_obj.block_def in ins_to_block_map:

                    if local_gpio_reg is not None:
                        local_gpio_reg = hram_icfg.apply_setting_to_embedded_items(ins_obj, local_gpio_reg)

                        all_gpio = local_gpio_reg.get_all_gpio()

                        for gpio_ins in all_gpio:
                            # Check that the resource is in the device
                            # okay if the are not in active list (not in
                            # resource map)
                            if gpio_ins is not None:
                                if gpio_ins.gpio_def == "":
                                    msg = 'Instance {} of {} has invalid resource for {}'.format(
                                        ins_obj.name, self._name, gpio_ins.name)
                                    raise dev_excp.ConfigurationInvalidException(
                                        msg, app_excp.MsgLevel.error)

                                if self._device.find_instance(gpio_ins.gpio_def) is not None:
                                    # Before writing it, we need to replace the pin name in the
                                    # file to the user pin name using the gen_pin.type_name and
                                    # do a string replace
                                    hram_icfg.replace_instance_pin_names(
                                        ins_obj, gpio_ins)


                        # Use the HSIO GPIO Timing writer to write out the instance.
                        gpio_writer.write(index, 0, hsio_block, all_gpio, {})
                        #self.clkout_str_names.update(gpio_writer.clkout_str_names)

                        ins_written = True

            if ins_written:
                rptfile = open(self._report_file, 'a')
                rptfile.write(
                    "\n---------- HyperRAM Timing Report (end) ----------\n")

                rptfile.close()

            write_successful = True

        except Exception as excp:
            if write_successful is not None and\
                    not write_successful:
                if rptfile is not None:
                    rptfile.close()
                if sdcfile is not None:
                    sdcfile.close()

            raise excp

        return ins_written
