'''
Copyright (C) 2017-2021 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Jun 17, 2021

@author: maryam
'''
import os
import sys

import xml.etree.ElementTree as et

# 3rd party library to display pretty table
from prettytable import PrettyTable

from typing import Dict, List
from enum import Enum, unique

import util.gen_util as pt_util
import util.excp as app_excp

import device.excp as dev_excp

import device.db_interface as device_dbi
from device.block_definition import TimingArc

import design.db as des_db

import common_device.writer as tbi
from common_device.hsio.hsio_device_service import HSIOService

sys.path.append(os.path.join(os.path.dirname(__file__), '..'))

@unique
class HSIORoutingArcLabel(Enum):
    hsio_routing_tco = "HSIO_ROUTING_TCO"
    hsio_routing_tsetup = "HSIO_ROUTING_TSETUP"
    hsio_routing_thold = "HSIO_ROUTING_THOLD"
    hsio_routing_to_core = "HSIO_ROUTING_TO_CORE"
    hsio_routing_from_core = "HSIO_ROUTING_FROM_CORE"
    hsio_routing_clkout = "HSIO_ROUTING_CLKOUT"


class HSIOTiming(tbi.BlockTimingWithArcs):
    '''
    Builds the timing data for SPI Flash used for printing out
    timing report and sdc file.
    '''

    # TODO: Use this if it varies per bank's io voltage
    class VariableType(Enum):
        '''
        The list of supported timing variables.
        '''
        io_standard = 0
        drive_strength = 1
        schmitt_trigger = 2
        io_voltage = 3

    var2str_map = \
        {
            VariableType.io_standard: "io_standard",
            VariableType.drive_strength: "drive_strength",
            VariableType.schmitt_trigger: "schmitt_trigger",
            VariableType.io_voltage: "io_voltage"
        }

    str2var_map = \
        {
            "io_standard": VariableType.io_standard,
            "drive_strength": VariableType.drive_strength,
            "schmitt_trigger": VariableType.schmitt_trigger,
            "io_voltage": VariableType.io_voltage
        }

    def __init__(self, bname, device, design, report, sdc,
                 max_model, min_model, ins2iostd_map):
        '''
        Constructor
        '''
        super().__init__(bname, device, design, report, sdc,
                         max_model, min_model)

        # We're building it a bit different since there's no
        # really a mode for this block. The mode is actually
        # to differentiate between input and outupt arcs
        self._mode2arcs = {}

        # Map of instance to io standard
        self._ins2iostd_map = ins2iostd_map

    def get_ins_bank_names(self):
        """
        This returns the map of instance to the bank name for all instances
        that are bonded out in the pad list.
        """
        device_db = device_dbi.DeviceDBService(self._device)
        ins2bank = device_db.get_instance_to_raw_io_bank_names()

        return ins2bank
    
    def build_arc(self, hsio_block, tx_blk_mode, rx_blk_mode):
        """
        Build Timing Arcs
        """
        # Nothing to be done
        pass

    def save_source_latency_and_clkout(self, ins_to_block_map):
        '''
        '''
        try:
            # We don't need to use the mipi lane writer
            hsio_block, hsio_svc, lvds_writer, gpio_writer, _ = \
                self.create_hsio_mode_writer()

            # We iterate through all 3 types of blocks that a HSIO can be associated to
            all_lvds, all_gpio, _ = hsio_svc.get_all_type_instances(
                self._design)
                                    
            # Save the alt delay and clkout info to the top (HSIO)
            if all_gpio:
                gpio_writer.save_source_latency_and_clkout_hsio(hsio_block, all_gpio)

                # Copy the lvds gpio container data to LVDS (this overwrites
                # what was already saved in the dict)
                self.ins2_alt_delay_map = gpio_writer.ins2_alt_delay_map
                self.ins2_clkout_delay_map = gpio_writer.ins2_clkout_delay_map


            if all_lvds:
                ins2_alt_lvds_delay_map = lvds_writer.save_lvds_rx_alternate_type_hsio(hsio_block, all_lvds)

                if ins2_alt_lvds_delay_map:
                    self.ins2_alt_delay_map.update(ins2_alt_lvds_delay_map)

        except Exception as excp:
            raise excp
    
    def create_hsio_mode_writer(self):
        from common_device.hsio.writer.gpio_timing import HSIOGPIOTiming
        from common_device.hsio.writer.lvds_timing import HSIOLVDSTiming
        from common_device.hsio.writer.mipi_dphy_timing import HSIOMDPhyTiming

        # Get the block definition
        hsio_block = self._device.find_block(self._name)
        hsio_svc = HSIOService(self._device, self._name)

        if hsio_block is None:
            msg = "Unable to find block {}".format(self._name)

            raise dev_excp.BlockDoesNotExistException(
                msg, app_excp.MsgLevel.error)

        ins2bank_names_map = self.get_ins_bank_names()

        # Populate HSIO Routing delay if the device supports it
        routing_label2bank_tbl_max_map, routing_label2bank_tbl_min_map = \
            self.populate_routing_delay_per_bank()
        
        lvds_writer = HSIOLVDSTiming(
            self._name, self._device, self._design, self._report_file, self._sdc_file,
            self._max_model, self._min_model, self._ins2iostd_map, ins2bank_names_map,
            routing_label2bank_tbl_max_map, routing_label2bank_tbl_min_map)

        gpio_writer = HSIOGPIOTiming(
            self._name, self._device, self._design, self._report_file, self._sdc_file,
            self._max_model, self._min_model, self._ins2iostd_map, ins2bank_names_map,
            routing_label2bank_tbl_max_map, routing_label2bank_tbl_min_map)

        mlane_writer = HSIOMDPhyTiming(
            self._name, self._device, self._design, self._report_file, self._sdc_file,
            self._max_model, self._min_model, self._ins2iostd_map, ins2bank_names_map,
            routing_label2bank_tbl_max_map, routing_label2bank_tbl_min_map)
                    
        return hsio_block, hsio_svc, lvds_writer, gpio_writer, mlane_writer
    
    def write(self, index, ins_to_block_map, clk2delay_map):
        '''
        Write out the report and sdc constraint for this block.

        :param index: The index of this section. Used in printing to report
        :param ins_to_block_map: Map of device instance name to the block name

        :return True if there was any valid instance found

        '''
        pt_util.mark_unused(clk2delay_map)

        ins_exists = False

        try:
            hsio_block, hsio_svc, lvds_writer, gpio_writer, mdphy_writer = \
                self.create_hsio_mode_writer()
 
            # We iterate through all 3 types of blocks that a HSIO can be associated to
            all_lvds, all_gpio, all_mipi_dphy = hsio_svc.get_all_type_instances(
                self._design)

            if all_lvds or all_gpio or all_mipi_dphy:
                ins_exists = True

                # Iterate through the instances of each type
                sub_index = 1
                if all_gpio:
                    if gpio_writer.write(index, sub_index, hsio_block, all_gpio, ins_to_block_map):
                        sub_index += 1
                    #self.clkout_str_names.update(gpio_writer.clkout_str_names)

                if all_lvds:
                    if lvds_writer.write(index, sub_index, hsio_block, all_lvds, ins_to_block_map):
                        sub_index += 1
                    #self.clkout_str_names.update(lvds_writer.clkout_str_names)

                if all_mipi_dphy:
                    mdphy_writer.write(
                        index, sub_index, hsio_block, all_mipi_dphy, ins_to_block_map)
                    #self.clkout_str_names.update(mdphy_writer.clkout_str_names)

        except Exception as excp:
            raise excp

        return ins_exists

    def build_delay_param_arc_table(self, hsio_block_max, hsio_block_min):
        label2arctbl_max_map = {}
        label2arctbl_min_map = {}

        for arc_label in HSIORoutingArcLabel:
            # Get the arc table for that label
            label_name = arc_label.value

            arc_table_max = self._load_arc_parameter(
                hsio_block_max, label_name, is_max=True, is_optional=True)
            
            if arc_table_max is not None:
                label2arctbl_max_map[label_name] = arc_table_max

            arc_table_min = self._load_arc_parameter(
                hsio_block_min, label_name, is_max=False, is_optional=True)

            if arc_table_min is not None:
                label2arctbl_min_map[label_name] = arc_table_min

        return label2arctbl_max_map, label2arctbl_min_map

    def get_xml_block(self, blk_name: str, is_max: bool):
        timing_block = None
        
        if not is_max:
            tree = et.ElementTree(file=self._min_model.get_filename())
        else:
            tree = et.ElementTree(file=self._max_model.get_filename())

        block_tag = ".//" + self.xml_ns + "block"
        for elem in tree.iterfind(block_tag):

            block_attrib = elem.attrib
            if block_attrib["type"] == blk_name:
                timing_block = elem
                break        

        return timing_block
           
    def populate_routing_delay_per_bank(self):
                    #label2arctbl_max_map: Dict[RoutingArcLabel, ArcDelayTable],
                    #label2arctbl_min_map: Dict[RoutingArcLabel, ArcDelayTable]):

        hsio_block_max = self.get_xml_block(self._name, True)
        assert hsio_block_max is not None
        hsio_block_min = self.get_xml_block(self._name, False)
        assert hsio_block_min is not None

        blk_label2arc_max_map, blk_label2arc_min_map = \
            self.build_delay_param_arc_table(hsio_block_max, hsio_block_min)

        # Get the delay and populate per bank
        label2bank_tbl_max_map = {}
        label2bank_tbl_min_map = {}
                            
        if blk_label2arc_max_map and blk_label2arc_min_map:

            label_names = blk_label2arc_max_map.keys()
            assert len(label_names) == len(blk_label2arc_min_map)

            for lbl_name in label_names:
                max_lbl_delay_map = {}
                min_lbl_delay_map = {}
            
                max_arc_tbl = blk_label2arc_max_map[lbl_name]
                min_arc_tbl = blk_label2arc_min_map[lbl_name]

                assert max_arc_tbl.get_pcount() == 1 and\
                        min_arc_tbl.get_pcount() == 1
                
                #var_str_max = max_arc_tbl.get_variable_names()
                #var_str_min = max_arc_tbl.get_variable_names()
                #assert sorted(var_str_max) == sorted(var_str_min)
                var_name2val_max: Dict[str, List[str]] = max_arc_tbl.get_variable_values()
                var_name2val_min: Dict[str, List[str]] = min_arc_tbl.get_variable_values()

                assert len(var_name2val_max.keys()) == 1 and \
                    len(var_name2val_min.keys()) == 1
                
                assert sorted(var_name2val_max.values()) == \
                        sorted(var_name2val_min.values())
                
                if "hsio_bank" in var_name2val_max:
                    assert "hsio_bank" in var_name2val_min

                    bank_values = var_name2val_max["hsio_bank"]
                    for bank_name in bank_values:
                        var_lbl = f"hsio_bank:{bank_name}"
                        
                        max_delay = max_arc_tbl.get_variable_delay(var_lbl)
                        min_delay = min_arc_tbl.get_variable_delay(var_lbl)

                        max_lbl_delay_map[bank_name] = max_delay
                        min_lbl_delay_map[bank_name] = min_delay

                    label2bank_tbl_max_map[lbl_name] = max_lbl_delay_map
                    label2bank_tbl_min_map[lbl_name] = min_lbl_delay_map

        return label2bank_tbl_max_map, label2bank_tbl_min_map

    @staticmethod
    def get_instance_bank_name_for_routing(dev_ins_name,
                                           lbl_name, ins2bankname_map,
                                           label2bank_tbl_map: Dict[str, Dict[str, float]]):
        '''
        Find the bank name associated to the device instance.
        '''
        bank_name = ""

        if ins2bankname_map and label2bank_tbl_map and\
            dev_ins_name in ins2bankname_map:
            
            bank_name_raw = ins2bankname_map[dev_ins_name]

            if lbl_name in label2bank_tbl_map:
            
                bank_map = label2bank_tbl_map[lbl_name]

                if bank_name_raw in bank_map:
                    bank_name = bank_name_raw

                #print(f'get_ins_bank_name_routing: {dev_ins_name} - {bank_name_raw} - {lbl_name} - {bank_name}')
            
        return bank_name
    
    @staticmethod    
    def get_routing_delay_value(lbl_name: str, bank_name: str,
                                label2bank_tbl_map: Dict[str, Dict[str, float]]):
        delay_val = 0

        if label2bank_tbl_map and lbl_name in label2bank_tbl_map:
            bank_map = label2bank_tbl_map[lbl_name]
            if bank_name in bank_map:
                delay_val = bank_map[bank_name]

        return delay_val
    
    @staticmethod    
    def identify_routing_delay_on_ins(dev_ins: str, arc_type: TimingArc.TimingArcType,
                                      is_to_core: bool, tscale: float, ins2bankname_map: Dict[str, str],
                                      label2bank_tbl_map: Dict[str, Dict[str, float]], is_clkout=False):
        '''
        Get the additional routing delay associated to an arc based on the
        instance device resource name.

        :param dev_ins: The device instance name
        :param arc_type: The TimingArc type that we're analyzing where we want to
                determine the additional routing delay associated to it
        :param label2bank_tbl_map: The routing delay table for the GPIO where it contains
                the map of the arc label mapped to a dictionary of param to delay value
                where param is the bank name for Ti375
        :param is_to_core: To indicate the direction of the connection. True it's
                an output port and input to core. Otherwise, False.                
        :return the routing delay associated to the arc
        '''

        routing_delay = 0

        if label2bank_tbl_map:
            label_name = ""
           
            match arc_type:
                case TimingArc.TimingArcType.setup:
                    label_name = HSIORoutingArcLabel.hsio_routing_tsetup.value
                case TimingArc.TimingArcType.hold:
                    label_name = HSIORoutingArcLabel.hsio_routing_thold.value
                case TimingArc.TimingArcType.clk_to_q:
                    label_name = HSIORoutingArcLabel.hsio_routing_tco.value
                case TimingArc.TimingArcType.delay:
                    # This depends on the port direction:
                    # input port / clkout.output interface - from_core
                    # output port / input interface - to_core
                    if is_to_core:
                        label_name = HSIORoutingArcLabel.hsio_routing_to_core.value
                    elif is_clkout:
                        label_name = HSIORoutingArcLabel.hsio_routing_clkout.value
                    else:
                        label_name = HSIORoutingArcLabel.hsio_routing_from_core.value
                case _:
                    raise NotImplementedError(f'Unexpected timing arc type: {arc_type}')

            if label_name != "":
                # Determine the label to use based on the arc type
                dly_bank_name = HSIOTiming.get_instance_bank_name_for_routing(
                    dev_ins, label_name, ins2bankname_map, label2bank_tbl_map)

                if dly_bank_name != "":
                    routing_delay = HSIOTiming.get_routing_delay_value(
                        label_name, dly_bank_name, label2bank_tbl_map)
                    
                    routing_delay = routing_delay * tscale / 1000

                    #print(f"identify_routing_delay_on_ins {dev_ins}: {is_to_core}: {label_name}: {dly_bank_name} - {routing_delay}")

        return routing_delay
