'''
Copyright (C) 2017-2021 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Jun 17, 2021

@author: maryam
'''
import os
import sys
import enum
import gc

import xml.etree.ElementTree as et

# 3rd party library to display pretty table
from prettytable import PrettyTable

from typing import Dict

import util.gen_util as pt_util
import util.excp as app_excp

import device.excp as dev_excp

import device.db_interface as device_dbi
from device.block_definition import TimingArc

import design.db as des_db

import common_device.writer as tbi
from common_device.mipi_dphy.mipi_dphy_design import MIPIDPhy
from common_device.hsio.hsio_device_service import HSIOService
from common_device.hsio.writer.timing import  HSIOTiming

sys.path.append(os.path.join(os.path.dirname(__file__), '..'))


class HSIOMDPhyTiming(tbi.BlockTimingWithArcs):
    '''
    Builds the timing data for SPI Flash used for printing out
    timing report and sdc file.
    '''

    class VariableType(enum.Enum):
        '''
        The list of supported timing variables.
        '''
        io_voltage = 0
        hsio_io_std_group = 1
        drive_strength = 2
        slew_rate = 3
        schmitt_trigger = 4
        serial_type = 5
        fifo_enable = 6
        pre_emphasis = 7
        bank = 8

    var2str_map = \
        {
            VariableType.io_voltage: "io_voltage",
            VariableType.hsio_io_std_group: "hsio_io_std_group",
            VariableType.drive_strength: "drive_strength",
            VariableType.slew_rate: "slew_rate",
            VariableType.schmitt_trigger: "schmitt_trigger",
            VariableType.serial_type: "serial_type",
            VariableType.fifo_enable: "fifo_enable",
            VariableType.pre_emphasis: "pre_emphasis",
            VariableType.bank: "hsio_bank"
        }

    str2var_map = \
        {
            "io_voltage": VariableType.io_voltage,
            "hsio_io_std_group": VariableType.hsio_io_std_group,
            "drive_strength": VariableType.drive_strength,
            "slew_rate": VariableType.slew_rate,
            "schmitt_trigger": VariableType.schmitt_trigger,
            "serial_type": VariableType.serial_type,
            "fifo_enable": VariableType.fifo_enable,
            "pre_emphasis": VariableType.pre_emphasis,
            "hsio_bank": VariableType.bank
        }

    class MIPIDPhyPathType(enum.Enum):
        tx_input = 0
        tx_comb_input = 1
        tx_comb_output = 2
        rx_output = 3
        rx_input = 4
        rx_comb_input = 5
        rx_comb_output = 6

    def __init__(self, bname, device, design, report, sdc,
                 max_model, min_model, ins2iostd_map,
                 ins2bank_names_map,
                 routing_label2bank_tbl_max_map, routing_label2bank_tbl_min_map):
        '''
        Constructor
        '''
        super().__init__(bname, device, design, report, sdc,
                         max_model, min_model)

        # We're building it a bit different since there's no
        # really a mode for this block. The mode is actually
        # to differentiate between input and outupt arcs
        self._mode2arcs = {}

        # Map of instance to io standard
        self._ins2iostd_map = ins2iostd_map

        # Keep the list of instance (in resource map) map
        # to bank name for use later. This is the raw bank name
        # and not the package specific bank name where it could be
        # a merge bank name
        self.ins2bankname_map: Dict[str, str] = ins2bank_names_map         

        # The device delay table for the HSIO routing delay
        self.routing_label2bank_tbl_max_map = routing_label2bank_tbl_max_map
        self.routing_label2bank_tbl_min_map = routing_label2bank_tbl_min_map

    def build_arc(self):
        """
        Build Timing Arcs
        """
        # Nothing to be done
        pass

    def build_blk_arc(self, hsio_block, tx_blk_mode, rx_blk_mode):
        """
        Build Timing Arcs
        """
        self._mode2arcs = {}

        tx_input_arcs = tx_blk_mode.get_all_input_arcs(True)
        # tx_comb_in_arcs = tx_blk_mode.get_all_input_arcs(False)
        # tx_comb_out_arcs = tx_blk_mode.get_all_output_arcs(False)
        # For MIPI RX, only the LP_P/N_IN is included as a combinational arc
        tx_core_to_pad_arcs = tx_blk_mode.get_timing_arc("HS_OE", "P", False)
        tx_core_to_pad_arcs += tx_blk_mode.get_timing_arc(
            "LP_N_OE", "N", False)
        tx_core_to_pad_arcs += tx_blk_mode.get_timing_arc(
            "LP_P_OE", "P", False)
        tx_core_to_pad_arcs += tx_blk_mode.get_timing_arc(
            "LP_N_OUT", "N", False)
        tx_core_to_pad_arcs += tx_blk_mode.get_timing_arc(
            "LP_P_OUT", "P", False)

        self._mode2arcs[self.MIPIDPhyPathType.tx_input] = tx_input_arcs
        self._mode2arcs[self.MIPIDPhyPathType.tx_comb_input] = tx_core_to_pad_arcs
        # self._mode2arcs[self.MIPIDPhyPathType.tx_comb_output] = tx_comb_out_arcs

        rx_input_arcs = rx_blk_mode.get_all_input_arcs(True)
        rx_output_arcs = rx_blk_mode.get_all_output_arcs(True)
        # For MIPI RX, only the LP_P/N_IN is included as a combinational arc
        # rx_comb_in_arcs = rx_blk_mode.get_all_input_arcs(False)
        # rx_comb_out_arcs = rx_blk_mode.get_all_output_arcs(False)
        # Just choose one of the pad since both has same delay
        rx_pad_to_core_arcs = rx_blk_mode.get_timing_arc("P", "LP_P_IN", False)
        rx_pad_to_core_arcs += rx_blk_mode.get_timing_arc(
            "N", "LP_N_IN", False)

        self._mode2arcs[self.MIPIDPhyPathType.rx_input] = rx_input_arcs
        self._mode2arcs[self.MIPIDPhyPathType.rx_output] = rx_output_arcs
        # self._mode2arcs[self.MIPIDPhyPathType.rx_comb_input] = rx_in_arcs
        self._mode2arcs[self.MIPIDPhyPathType.rx_comb_output] = rx_pad_to_core_arcs

    def write(self, index, sub_index, hsio_block, all_mipi_dphy, ins_to_block_map):
        '''
        Write out the report and sdc constraint for this block.

        :param index: The index of this section. Used in printing to report
        :param sub_index: The sub index since it all belongs to the same section. Therefore
                it will become <index>.<sub_index>.
        :param all_mipi_dphy: The list of MIPI DPHY design instance objects
        :param ins_to_block_map: Map of device instance name to the block name

        '''

        write_successful = None
        ins_exists = False

        rptfile = None
        sdcfile = None

        try:
            write_successful = False

            # Get the Tx mode
            tx_blk_mode = hsio_block.get_mode(HSIOService.MODE_MIPI_DPHY_TX)

            # Get the Rx mode
            rx_blk_mode = hsio_block.get_mode(HSIOService.MODE_MIPI_DPHY_RX)

            if tx_blk_mode is None or rx_blk_mode is None:
                raise ValueError("Unable to find hsio mipi dphy block mode")

            # Build the arc
            self.build_blk_arc(hsio_block, tx_blk_mode, rx_blk_mode)

            # Iterate through the list of mipi dphy
            mdphy_reg = self._design.get_block_reg(
                des_db.PeriDesign.BlockType.mipi_dphy)

            # Open the file
            rptfile = open(self._report_file, 'a')
            sdcfile = open(self._sdc_file, 'a')

            if mdphy_reg is not None:

                def get_name(ins_obj):
                    return ins_obj.name

                all_tx = mdphy_reg.get_all_tx_inst()
                all_rx = mdphy_reg.get_all_rx_inst()

                # Get the device service based on the mipi dphy type
                dbi = device_dbi.DeviceDBService(self._device)

                mipi_dphy_tx_svc = dbi.get_block_service(
                    device_dbi.DeviceDBService.BlockType.HSIO_MIPI_DPHY_TX)
                mipi_dphy_rx_svc = dbi.get_block_service(
                    device_dbi.DeviceDBService.BlockType.HSIO_MIPI_DPHY_RX)

                print_report = False
                type_index = 1

                # Sort the list based on instance name so that the
                # output is deterministic

                # Write out the Rx first so that we write the clk source (Rx in clock lane mode)
                for ins_obj in sorted(all_rx, key=get_name):
                    if ins_obj is not None and ins_obj in all_mipi_dphy:
                        if ins_obj.rx_info is not None:
                            rx_cfg = ins_obj.rx_info

                            if rx_cfg.mode == rx_cfg.ModeType.clock:
                                ins_exists = True

                                if not print_report:
                                    # Print the title/table header for the first iteration
                                    comb_table = PrettyTable(
                                        ["Instance Name", "Pin Name", "Parameter", "Max (ns)", "Min (ns)"])

                                    rptfile.write(
                                        "\n---------- {}.{}.{} MIPI RX Lane Timing Report (begin) ----------\n".format(
                                            index, sub_index, type_index))

                                    sdcfile.write(
                                        "\n# MIPI RX Lane Constraints\n")
                                    sdcfile.write(
                                        "############################\n")

                                    print_report = True
                                    type_index = type_index + 1

                                # Write out the create_clock constraint for this first
                                sdcfile.write(
                                    "# create_clock -period <USER_PERIOD> -name {} [get_ports {{{}}}]\n".format(
                                        rx_cfg.get_alt_clockout_pin_name(), rx_cfg.get_alt_clockout_pin_name()))

                for ins_obj in sorted(all_rx, key=get_name):
                    if ins_obj is not None and ins_obj in all_mipi_dphy:
                        ins_exists = True

                        if not print_report:
                            # Print the title/table header for the first iteration
                            comb_table = PrettyTable(
                                ["Instance Name", "Pin Name", "Parameter", "Max (ns)", "Min (ns)"])

                            rptfile.write(
                                "\n---------- {}.{}.{} MIPI RX Lane Timing Report (begin) ----------\n".format(
                                    index, sub_index, type_index))

                            sdcfile.write("\n# MIPI RX Lane Constraints\n")
                            sdcfile.write("############################\n")

                            print_report = True
                            type_index = type_index + 1

                        self._write_registered_input(
                            sdcfile, mipi_dphy_rx_svc, ins_obj, rx_blk_mode, False)
                        self._write_registered_output(
                            sdcfile, mipi_dphy_rx_svc, ins_obj, rx_blk_mode)
                        # Not applicable to RX since we only want LP_P/N_IN which is pad to core (input interface)
                        # self._write_combinational_input(comb_table, mipi_dphy_rx_svc, ins_obj, False)
                        self._write_combinational_output(
                            comb_table, mipi_dphy_rx_svc, ins_obj, False)

                if print_report:
                    rptfile.write("\n{}\n".format(comb_table.get_string()))
                    rptfile.write(
                        "\n---------- MIPI RX Lane Timing Report (end) ----------\n")

                print_report = False

                # Write out the Tx
                for ins_obj in sorted(all_tx, key=get_name):
                    if ins_obj is not None and ins_obj in all_mipi_dphy:
                        ins_exists = True

                        if not print_report:
                            # Print the title/table header for the first iteration
                            comb_table = PrettyTable(
                                ["Instance Name", "Pin Name", "Parameter", "Max (ns)", "Min (ns)"])

                            rptfile.write(
                                "\n---------- {}.{}.{} MIPI TX Lane Timing Report (begin) ----------\n".format(
                                    index, sub_index, type_index))

                            sdcfile.write("\n# MIPI TX Lane Constraints\n")
                            sdcfile.write("############################\n")

                            print_report = True
                            type_index = type_index + 1

                        self._write_registered_input(
                            sdcfile, mipi_dphy_tx_svc, ins_obj, tx_blk_mode, True)
                        self._write_combinational_input(
                            comb_table, mipi_dphy_tx_svc, ins_obj, True)
                        # Not applicable to RX since we only want the one that is core to pad (output interface)
                        # self._write_combinational_output(comb_table, mipi_dphy_tx_svc, ins_obj, True)

                if print_report:
                    rptfile.write("\n{}\n".format(comb_table.get_string()))
                    rptfile.write(
                        "\n---------- MIPI TX Lane Timing Report (end) ----------\n")

            rptfile.close()
            sdcfile.close()

            write_successful = True

        except Exception as excp:
            if write_successful is not None and\
                    not write_successful:
                if rptfile is not None:
                    rptfile.close()
                if sdcfile is not None:
                    sdcfile.close()

            raise excp

        return ins_exists

    def get_reg_pin_and_clk_name(self, mdphy_dev_svc, ins_obj, sink_name, source_name, blk_mode, is_tx):
        # The sink and source name are interface name in the mode and
        # not necessarily the block port name
        if is_tx:
            mipi_info = ins_obj.tx_info
        else:
            mipi_info = ins_obj.rx_info

        gen_pin = mdphy_dev_svc.get_user_gen_pin(
            ins_obj.name, mipi_info.gen_pin, sink_name)

        if gen_pin is None:
            msg = "Unable to find pin {} in instance {}".format(
                sink_name, ins_obj.name)

            raise ValueError(msg)

        clk_pin = mdphy_dev_svc.get_user_gen_pin(
            ins_obj.name, mipi_info.gen_pin, source_name)

        if clk_pin is None:
            # if it is FIFOCLK on a data lane, then we use the corresponding clock
            # lane in the same group CLKOUT pin name. Don't need to check for is_fifo
            # because the caller already filter it
            if not is_tx and source_name == "FIFOCLK" and \
                    mipi_info.mode == mipi_info.ModeType.data:
                clk_pin = mipi_info.get_data_clock_gen_pin(
                    ins_obj, self._design.mipi_dphy_reg, mdphy_dev_svc)

            if clk_pin is None:
                msg = "Unable to find pin {} in instance {}".format(
                    source_name, ins_obj.name)

                raise ValueError(msg)

        # We need to pass the clkpin type name that is matching the
        # device instance pin connection (port) and not the interface name
        clk_inf = blk_mode.get_interface_object(clk_pin.type_name)
        clk_port_name = clk_pin.type_name
        if clk_inf is not None:
            if not clk_inf.is_bus_port():
                # Clock interface is not a bus port
                if source_name == "FIFOCLK" and clk_pin.type_name == "CLKOUT":
                    # FIFOCLK is not a gen pin for MIPI RX
                    clk_port_name = "RX_SCLKN"
                else:
                    clk_port_name = clk_inf.get_port_name()
            else:
                msg = "Unexpected bus clkout interface on HSIO mode {} interface {}".format(
                    blk_mode.get_name(), clk_pin.type_name)
                raise ValueError(msg)

        clk_name = clk_pin.name

        # Non clock port we use the bracket if it was a bus
        # Find the port based on the sink interface name and check
        # if the interface itself is a bus interface type
        src_inf = blk_mode.get_interface_object(gen_pin.type_name)
        if src_inf is not None:
            if src_inf.is_bus_port() and gen_pin.name != "":
                pin_name = "{}[*]".format(gen_pin.name)
            else:
                pin_name = gen_pin.name

        else:
            pin_name = gen_pin.name

        return pin_name, clk_name, clk_port_name

    def _write_registered_input(self, sdcfile, mipi_dphy_svc, ins_obj,
                                blk_mode, is_tx):
        '''
        Writes out  the set_output_delay constraints based on
        block port with input direction which connects to core
        (driven by core).
        '''

        # Contains the map of arc object to the raw delay value
        # (without scaling). The arcs are related to the
        # path type only
        if is_tx:
            path_type = self.MIPIDPhyPathType.tx_input
        else:
            path_type = self.MIPIDPhyPathType.rx_input

        max_arc2delay_map, min_arc2delay_map = self._get_timing_delay(
            ins_obj, path_type)

        in_arcs = self._mode2arcs[path_type]

        is_fifo_enabled = False
        is_dyn_delay = False
        skip_ref_pin = False

        rx_config = None
        if not is_tx:
            rx_config = ins_obj.rx_info
            if rx_config is not None:
                if rx_config.is_fifo:
                    is_fifo_enabled = True
                else:
                    skip_ref_pin = True

                if rx_config.delay_mode == rx_config.DelayModeType.dynamic:
                    is_dyn_delay = True

        routing_delay_max = HSIOTiming.identify_routing_delay_on_ins(
            ins_obj.block_def, TimingArc.TimingArcType.setup,
            False, self._max_model.get_tscale(), self.ins2bankname_map,
            self.routing_label2bank_tbl_max_map)
        routing_delay_min = HSIOTiming.identify_routing_delay_on_ins(
            ins_obj.block_def, TimingArc.TimingArcType.hold,
            False, self._min_model.get_tscale(), self.ins2bankname_map,
            self.routing_label2bank_tbl_min_map) 
                        
        for arc in in_arcs:
            sink_name = arc.get_sink()
            source_name = arc.get_source()
            # PT-2541: requires clock_fall for DLY related
            is_clk_fall = False

            # If it is a FIFOCLK arc
            if source_name == "FIFOCLK":
                if sink_name.startswith("FIFO_") and not is_fifo_enabled:
                    # skip pins that are dependent on if fifo is enabled
                    continue
                elif rx_config is not None and rx_config.mode == rx_config.ModeType.clock:
                    # Skip the arc for clock lane
                    continue
                elif sink_name.find("DLY") != -1:
                    if not is_dyn_delay:
                        # Skip pin related to Dynamic delay if dynamic delay is not set
                        continue
                    elif sink_name != "DLY_RST":
                        is_clk_fall = True

            pin_name, clk_name, clk_type_name = self.get_reg_pin_and_clk_name(
                mipi_dphy_svc, ins_obj, sink_name, source_name, blk_mode, is_tx)

            # No constraint is written if the pin name or clock name is empty
            if pin_name is not None and pin_name != "" and\
                    clk_name is not None and clk_name != "":

                # If it was rx and fifo is disabled, we don't need the reference_pin
                ref_arg_str = ""
                if not skip_ref_pin:
                    # Get the clkout string
                    ref_arg_str = self.set_clkout_ref_pin(
                        ins_obj, clk_type_name, clk_name, ins_obj.block_def)

                clk_fall_str = " -clock_fall" if is_clk_fall else ""

                if arc.get_type() == TimingArc.TimingArcType.setup:
                    if arc in max_arc2delay_map:
                        # Get the calculated delay
                        delay = self._get_delay_by_arc(
                            max_arc2delay_map, self._max_model.get_tscale(), arc)

                        max_delay = "{0:.3f}".format(delay + routing_delay_max)

                        sdcfile.write(
                            "set_output_delay -clock {}{}{} -max {} [get_ports {{{}}}]\n".format(
                                clk_name, clk_fall_str, ref_arg_str, max_delay, pin_name))

                elif arc.get_type() == TimingArc.TimingArcType.hold:
                    if arc in min_arc2delay_map:
                        delay = self._get_delay_by_arc(
                            min_arc2delay_map, self._min_model.get_tscale(), arc)

                        min_delay = "{0:.3f}".format((-1 * delay) + routing_delay_min)

                        sdcfile.write(
                            "set_output_delay -clock {}{}{} -min {} [get_ports {{{}}}]\n".format(
                                clk_name, clk_fall_str, ref_arg_str, min_delay, pin_name))

    def _write_registered_output(self, sdcfile, mipi_dphy_svc, ins_obj,
                                 blk_mode):
        '''
        Writes out  the set_input_delay constraints based on
        block port with output direction which connects to core
        (driving core).
        '''
        # Contains the map of arc object to the raw delay value
        # (without scaling). The arcs are related to the
        # path type only
        max_arc2delay_map, min_arc2delay_map = self._get_timing_delay(
            ins_obj, self.MIPIDPhyPathType.rx_output)

        out_arcs = self._mode2arcs[self.MIPIDPhyPathType.rx_output]

        is_fifo_enabled = False
        skip_ref_pin = False

        rx_config = ins_obj.rx_info
        if rx_config is not None:
            if rx_config.is_fifo:
                is_fifo_enabled = True
            else:
                skip_ref_pin = True

        routing_delay_max = HSIOTiming.identify_routing_delay_on_ins(
            ins_obj.block_def, TimingArc.TimingArcType.clk_to_q,
            True, self._max_model.get_tscale(), self.ins2bankname_map,
            self.routing_label2bank_tbl_max_map)
        routing_delay_min = HSIOTiming.identify_routing_delay_on_ins(
            ins_obj.block_def, TimingArc.TimingArcType.clk_to_q,
            True, self._min_model.get_tscale(), self.ins2bankname_map,
            self.routing_label2bank_tbl_min_map) 
        
        for arc in out_arcs:
            sink_name = arc.get_sink()
            source_name = arc.get_source()

            # If it is a FIFOCLK arc
            if source_name == "FIFOCLK":
                if sink_name != "RST" and sink_name != "HS_IN" and not is_fifo_enabled:
                    # non-RST pin is dependent on if fifo is enabled
                    continue
                elif rx_config is not None and rx_config.mode == rx_config.ModeType.clock:
                    # Skip the arc for clock lane
                    continue

            pin_name, clk_name, clk_type_name = self.get_reg_pin_and_clk_name(
                mipi_dphy_svc, ins_obj, sink_name, source_name, blk_mode, False)

            # No constraint is written if the pin name or clock name is empty
            if pin_name is not None and pin_name != "" and\
                    clk_name is not None and clk_name != "":

                # Get the clkout string
                ref_arg_str = ""
                if not skip_ref_pin:
                    ref_arg_str = self.set_clkout_ref_pin(
                        ins_obj, clk_type_name, clk_name, ins_obj.block_def)

                if arc in max_arc2delay_map:
                    # Get the calculated delay
                    delay = self._get_delay_by_arc(
                        max_arc2delay_map, self._max_model.get_tscale(), arc)

                    max_delay = "{0:.3f}".format(delay + routing_delay_max)

                    sdcfile.write(
                        "set_input_delay -clock {}{} -max {} [get_ports {{{}}}]\n".format(
                            clk_name, ref_arg_str, max_delay, pin_name))

                if arc in min_arc2delay_map:
                    delay = self._get_delay_by_arc(
                        min_arc2delay_map, self._min_model.get_tscale(), arc)

                    min_delay = "{0:.3f}".format(delay + routing_delay_min)

                    sdcfile.write(
                        "set_input_delay -clock {}{} -min {} [get_ports {{{}}}]\n".format(
                            clk_name, ref_arg_str, min_delay, pin_name))

    def _write_combinational_output(self, comb_table, mipi_dphy_svc, ins_obj, is_tx):

        if is_tx:
            path_type = self.MIPIDPhyPathType.tx_comb_output
        else:
            path_type = self.MIPIDPhyPathType.rx_comb_output

        max_arc2delay_map, min_arc2delay_map = self._get_timing_delay(
            ins_obj, path_type)

        comb_arcs = self._mode2arcs[path_type]

        self.get_comb_param_delay(comb_table, mipi_dphy_svc, ins_obj,
                                  max_arc2delay_map, min_arc2delay_map, comb_arcs, False, is_tx)

    def _write_combinational_input(self, comb_table, mipi_dphy_svc, ins_obj, is_tx):

        if is_tx:
            path_type = self.MIPIDPhyPathType.tx_comb_input
        else:
            path_type = self.MIPIDPhyPathType.rx_comb_input

        max_arc2delay_map, min_arc2delay_map = self._get_timing_delay(
            ins_obj, path_type)

        comb_arcs = self._mode2arcs[path_type]

        self.get_comb_param_delay(comb_table, mipi_dphy_svc, ins_obj,
                                  max_arc2delay_map, min_arc2delay_map, comb_arcs, True, is_tx)

    def get_comb_param_delay(self, comb_table, mipi_dphy_svc, ins_obj,
                             max_arc2delay_map, min_arc2delay_map, comb_arcs, is_out_inf, is_tx):

        if is_tx:
            mipi_info = ins_obj.tx_info
        else:
            mipi_info = ins_obj.rx_info

        for arc in comb_arcs:
            row_list = []

            sink_name = arc.get_sink()

            gen_pin = mipi_dphy_svc.get_user_gen_pin(
                ins_obj.name, mipi_info.gen_pin, sink_name)

            if gen_pin is None:
                # Then search for the source since it could be that sink is a pad
                source_name = arc.get_source()

                gen_pin = mipi_dphy_svc.get_user_gen_pin(
                    ins_obj.name, mipi_info.gen_pin, source_name)

                if gen_pin is None:
                    msg = "Unable to find pin {} in instance {}".format(
                        source_name, ins_obj.name)

                    raise ValueError(msg)

            pin_name = gen_pin.name

            if pin_name is not None and pin_name != "":
                row_list.append(ins_obj.name)
                row_list.append(pin_name)

                # Parameter
                if is_out_inf:
                    if is_tx:
                        row_list.append("MIPI_TX_LANE_OUT")
                    else:
                        row_list.append("MIPI_RX_LANE_OUT")
                else:
                    if is_tx:
                        row_list.append("MIPI_TX_LANE_IN")
                    else:
                        row_list.append("MIPI_RX_LANE_IN")

                delay = self._get_delay_by_arc(
                    max_arc2delay_map, self._max_model.get_tscale(), arc)

                row_list.append("{0:.3f}".format(delay))

                delay = self._get_delay_by_arc(
                    min_arc2delay_map, self._min_model.get_tscale(), arc)

                row_list.append("{0:.3f}".format(delay))

                comb_table.add_row(row_list)

    def _get_calculated_static_delay(self, ins_obj, delay_param, hsio_elem_tag, is_max):
        static_delay = 0
        static_delay_step_param = "HSIO_STATIC_DELAY_DIFFERENTIAL"

        arc_table = self._load_arc_parameter(
            hsio_elem_tag, static_delay_step_param, is_max)
                
        if arc_table is not None:
            value = arc_table.get_variable_delay("")
            
            if value is not None:

                if delay_param == "HSIO_MIPI_DPHY_RX_LP_N_IN" or \
                        delay_param == "HSIO_MIPI_DPHY_RX_LP_P_IN":
                    if ins_obj.ops_type == ins_obj.OpsType.op_rx and ins_obj.rx_info is not None:
                        static_delay = ins_obj.rx_info.delay * value
                elif delay_param == "HSIO_MIPI_DPHY_TX_LP_N_OUT" or \
                        delay_param == "HSIO_MIPI_DPHY_TX_LP_P_OUT":
                    if ins_obj.ops_type == ins_obj.OpsType.op_tx and ins_obj.tx_info is not None:
                        static_delay = ins_obj.tx_info.delay * value

        return static_delay
    
    def _get_delay_value(self, ins_obj, delay_var_map, arc_table, is_max=True):
        '''
        :param ins_obj: The user jtag design object
        :param delay_var_map: The table of variables mapped to acceptable
                        values
        :param arc_table: The arc_parameter table with the name
                        representing the variables concatenated (as
                        in the xml file). An ArcDelayTable object

        :return the delay value of that parameter based on the required
                jtag configuration
        '''

        self.logger.debug("Finding MIPI Lane {} delay for {}".format(
            ins_obj.name, arc_table.get_parameter_name()))

        # Only if the arc parameter has variable associated, then
        # we figure out, else take that one value
        if arc_table.get_pcount() == 0:
            delay_val = arc_table.get_variable_delay("")
            return delay_val

        fifo_enable = "false"
        if ins_obj.ops_type == ins_obj.OpsType.op_rx:
            rx_cfg = ins_obj.rx_info

            if rx_cfg is not None and rx_cfg.is_fifo:
                fifo_enable = "true"

        # Get the arc delay parameter name
        delay_param = arc_table.get_parameter_name()

        hsio_elem_tag = self._get_block_timing_tag_section(is_max)
        additional_delay = self._get_calculated_static_delay(
            ins_obj, delay_param, hsio_elem_tag, is_max)

        # Now that we have all the required setting, we
        # get the delay based on it
        var_list = arc_table.get_variables()
        concat_name = ""
        read_vcount = 0

        for index in range(len(var_list)):
            vname = var_list[index]
            vfound = True

            if vname in self.str2var_map:
                vtype = self.str2var_map[vname]

                if vtype == self.VariableType.fifo_enable:
                    vsetting = vname + ":" + fifo_enable

                else:
                    vfound = False

                if vfound:
                    read_vcount += 1

                if concat_name == "":
                    concat_name = vsetting
                else:
                    concat_name = concat_name + "," + vsetting

            else:
                raise ValueError(
                    "Unexpected timing variable name {}".format(vname))

        # Verify that the variable count matches
        if read_vcount != arc_table.get_pcount():
            raise ValueError(
                "Mismatch variable count, {} and the pcount parameter, {}".format(
                    read_vcount, arc_table.get_pcount()))

        # Get the delay from table based on the concat variable name
        delay_val = 0
        tmp_delay = arc_table.get_variable_delay(concat_name)
        if tmp_delay is not None:
            delay_val = tmp_delay
        else:
            raise ValueError(
                "Unable to get the delay for variable {}".format(concat_name))

        delay_val += additional_delay

        return delay_val
