'''
Copyright (C) 2017-2021 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Jun 17, 2021

@author: maryam
'''
import os
import sys
import enum

# 3rd party library to display pretty table
from prettytable import PrettyTable
import xml.etree.ElementTree as et

from typing import Dict

import util.gen_util as pt_util
import util.excp as app_excp

import device.excp as dev_excp

import device.db_interface as device_dbi
from device.block_definition import TimingArc

import design.db as des_db

import common_device.writer as tbi
from common_device.hsio.hsio_device_service import HSIOService
from common_device.hsio.writer.timing import HSIOTiming
from tx60_device.lvds.lvds_design_adv import LVDSAdvanceParamId

sys.path.append(os.path.join(os.path.dirname(__file__), '..'))


class HSIOLVDSTiming(tbi.BlockTimingWithArcs):
    '''
    Builds the timing data for SPI Flash used for printing out
    timing report and sdc file.
    '''

    class VariableType(enum.Enum):
        '''
        The list of supported timing variables.
        '''
        io_voltage = 0
        hsio_io_std_group = 1
        drive_strength = 2
        slew_rate = 3
        schmitt_trigger = 4
        serial_type = 5
        fifo_enable = 6
        pre_emphasis = 7
        bank = 8

    var2str_map = \
        {
            VariableType.io_voltage: "io_voltage",
            VariableType.hsio_io_std_group: "hsio_io_std_group",
            VariableType.drive_strength: "drive_strength",
            VariableType.slew_rate: "slew_rate",
            VariableType.schmitt_trigger: "schmitt_trigger",
            VariableType.serial_type: "serial_type",
            VariableType.fifo_enable: "fifo_enable",
            VariableType.pre_emphasis: "pre_emphasis",
            VariableType.bank: "hsio_bank"
        }

    str2var_map = \
        {
            "io_voltage": VariableType.io_voltage,
            "hsio_io_std_group": VariableType.hsio_io_std_group,
            "drive_strength": VariableType.drive_strength,
            "slew_rate": VariableType.slew_rate,
            "schmitt_trigger": VariableType.schmitt_trigger,
            "serial_type": VariableType.serial_type,
            "fifo_enable": VariableType.fifo_enable,
            "pre_emphasis": VariableType.pre_emphasis,
            "hsio_bank": VariableType.bank
        }

    class LVDSPathType(enum.Enum):
        tx_output = 0
        tx_out_enable = 1
        tx_clkout = 2
        tx_bypass_out = 3
        tx_bypass_oe = 4
        tx_reset = 5
        rx_input = 6
        rx_bypass = 7
        rx_reset = 8
        rx_dyn_dpa_dly = 9
        rx_fifo = 10

    def __init__(self, bname, device, design, report, sdc,
                 max_model, min_model, ins2iostd_map,
                 ins2bank_names_map,
                 routing_label2bank_tbl_max_map, routing_label2bank_tbl_min_map):
        '''
        Constructor
        '''
        super().__init__(bname, device, design, report, sdc,
                         max_model, min_model)

        # We're building it a bit different since there's no
        # really a mode for this block. The mode is actually
        # to differentiate between input and outupt arcs
        self._mode2arcs = {}

        # Map of instance to io standard
        self._ins2iostd_map = ins2iostd_map

        # Map of alternate gpio instance to the delay value (GPIO_IN: max,min)
        self.ins2_alt_delay_map = {}

        # Keep the list of instance (in resource map) map
        # to bank name for use later. This is the raw bank name
        # and not the package specific bank name where it could be
        # a merge bank name
        self.ins2bankname_map: Dict[str, str] = ins2bank_names_map        

        # The device delay table for the HSIO routing delay
        self.routing_label2bank_tbl_max_map = routing_label2bank_tbl_max_map
        self.routing_label2bank_tbl_min_map = routing_label2bank_tbl_min_map

    def build_arc(self):
        """
        Build Timing Arcs
        """
        # Nothing to be done
        pass

    def build_prelim_arc(self, blk_mode):
        # Add the bypass arcs. Skip N-IN since same parameter
        bypass_arcs = blk_mode.get_timing_arc("P", "IN", False)
        self._mode2arcs[self.LVDSPathType.rx_bypass] = bypass_arcs

    def build_blk_rx_arc(self, blk_mode):

        # Add the out arcs
        # TODO: Restructure this when have the time. This is a temporary
        # hack since the assumption is that all member bus have
        # the same parameter. Also, ignore the N (just take P since
        # they are of the same delay)
        in_arcs = blk_mode.get_timing_arc("INSLOWCLK", "IN", True)
        in_arcs += blk_mode.get_timing_arc("FIFOCLK", "IN", True)
        self._mode2arcs[self.LVDSPathType.rx_input] = in_arcs

        # Add the bypass arcs. Skip N-IN since same parameter
        bypass_arcs = blk_mode.get_timing_arc("P", "IN", False)
        self._mode2arcs[self.LVDSPathType.rx_bypass] = bypass_arcs

        # Add both the reset arcs. But when using it we check the fifo flag
        rst_arcs = blk_mode.get_timing_arc("INSLOWCLK", "INRST", True)
        rst_arcs += blk_mode.get_timing_arc("FIFOCLK", "INRST", True)
        self._mode2arcs[self.LVDSPathType.rx_reset] = rst_arcs

        # Arcs for FIFOCLK related
        fifo_arcs = blk_mode.get_timing_arc("FIFOCLK", "FIFO_EMPTY", True)
        fifo_arcs += blk_mode.get_timing_arc("FIFOCLK", "FIFO_RD", True)
        self._mode2arcs[self.LVDSPathType.rx_fifo] = fifo_arcs

        # Ar related to DYNAMIC delay: dly_inc, dly_ena, dly_rst
        dyn_dpa_dly_arcs = blk_mode.get_timing_arc(
            "INSLOWCLK", "DLY_ENA", True)
        dyn_dpa_dly_arcs += blk_mode.get_timing_arc(
            "INSLOWCLK", "DLY_INC", True)
        dyn_dpa_dly_arcs += blk_mode.get_timing_arc(
            "INSLOWCLK", "DLY_RST", True)
        dyn_dpa_dly_arcs += blk_mode.get_timing_arc("INSLOWCLK", "LOCK", True)
        dyn_dpa_dly_arcs += blk_mode.get_timing_arc("INSLOWCLK", "DBG", True)
        self._mode2arcs[self.LVDSPathType.rx_dyn_dpa_dly] = dyn_dpa_dly_arcs

    def build_blk_tx_arc(self, blk_mode):
        # Add the out arcs
        # TODO: Restructure this when have the time. This is a temporary
        # hack since the assumption is that all member bus have
        # the same parameter. Also, ignore the N (just take P since
        # they are of the same delay)

        out_arcs = blk_mode.get_timing_arc("OUTSLOWCLK", "OUT", True)
        self._mode2arcs[self.LVDSPathType.tx_output] = out_arcs

        oe_arcs = blk_mode.get_timing_arc("OUTSLOWCLK", "OE", True)
        self._mode2arcs[self.LVDSPathType.tx_out_enable] = oe_arcs

        # Add the clkout arc
        clkout_arcs = blk_mode.get_timing_arc("OUTFASTCLK", "P", False)
        self._mode2arcs[self.LVDSPathType.tx_clkout] = clkout_arcs

        # Add the bypass arcs (Skip N since file has same delay)
        # TODO:Change if P and N has different parameter
        bypass_out_arcs = blk_mode.get_timing_arc("OUT", "P", False)
        self._mode2arcs[self.LVDSPathType.tx_bypass_out] = bypass_out_arcs

        bypass_oe_arcs = blk_mode.get_timing_arc("OE", "P", False)
        self._mode2arcs[self.LVDSPathType.tx_bypass_oe] = bypass_oe_arcs

        # Add the reset arcs
        rst_arcs = blk_mode.get_timing_arc("OUTSLOWCLK", "OUTRST", True)
        self._mode2arcs[self.LVDSPathType.tx_reset] = rst_arcs

    def save_lvds_rx_alternate_type_hsio(self, hsio_block, lvds_list):
        rx_blk_mode = hsio_block.get_mode(HSIOService.MODE_LVDS_RX)

        self.build_prelim_arc(rx_blk_mode)

        def get_name(lvds_ins):
            return lvds_ins.name
        
        # This is a map of design instance to the design object
        for lvds_obj in sorted(lvds_list, key=get_name):
            
            rx_config = lvds_obj.rx_info

            # We allow bidir
            if rx_config is None or lvds_obj.ops_type == lvds_obj.OpsType.op_tx:
                continue

            if not rx_config.is_serial:
                # Get the arcs for that mode
                arcs_list = self._mode2arcs.get(self.LVDSPathType.rx_bypass, [])
                pin_name = rx_config.input_bus_name

                max_delay = 0
                min_delay = 0

                is_clk = rx_config.is_alternate_clock_type()

                max_arc2delay_map, min_arc2delay_map = self._get_timing_delay(
                    lvds_obj, self.LVDSPathType.rx_bypass)

                for arc in arcs_list:

                    # Get the routing delay if necessary
                    routing_delay_max = HSIOTiming.identify_routing_delay_on_ins(
                        lvds_obj.block_def, TimingArc.TimingArcType.delay,
                        True, self._max_model.get_tscale(), self.ins2bankname_map,
                        self.routing_label2bank_tbl_max_map)
                    routing_delay_min = HSIOTiming.identify_routing_delay_on_ins(
                        lvds_obj.block_def, TimingArc.TimingArcType.delay,
                        True, self._min_model.get_tscale(), self.ins2bankname_map,
                        self.routing_label2bank_tbl_min_map)
                    
                    if arc in max_arc2delay_map:
                        # Get the calculated delay
                        max_delay = self._get_delay_by_arc(
                            max_arc2delay_map, self._max_model.get_tscale(), arc)
                        
                        max_delay = max_delay + routing_delay_max

                    if arc in min_arc2delay_map:
                        # Get the calculated delay
                        min_delay = self._get_delay_by_arc(
                            min_arc2delay_map, self._min_model.get_tscale(), arc)
                        
                        min_delay = min_delay + routing_delay_min

                if rx_config.is_alternate_connection():
                    assert lvds_obj.name not in self.ins2_alt_delay_map
                    self.ins2_alt_delay_map[lvds_obj.name] = \
                        (pin_name, max_delay, min_delay, is_clk)

        return self.ins2_alt_delay_map

    def write(self, index, sub_index, hsio_block, all_lvds, ins_to_block_map):
        '''
        Write out the report and sdc constraint for this block.

        :param index: The index of this section. Used in printing to report
        :param sub_index: The sub index since it all belongs to the same section. Therefore
                it will become <index>.<sub_index>.
        :param all_mipi_dphy: The list of MIPI DPHY design instance objects
        :param ins_to_block_map: Map of device instance name to the block name

        '''
        ins_exists = False
        write_successful = None

        rptfile = None
        sdcfile = None

        try:
            write_successful = False

            # Get the Tx mode
            tx_blk_mode = hsio_block.get_mode(HSIOService.MODE_LVDS_TX)

            # Get the Rx mode
            rx_blk_mode = hsio_block.get_mode(HSIOService.MODE_LVDS_RX)

            # Get the bidir mode
            bidir_blk_mode = hsio_block.get_mode(HSIOService.MODE_LVDS_BIDIR)

            if tx_blk_mode is None or rx_blk_mode is None or bidir_blk_mode is None:
                raise ValueError("Unable to find hsio lvds block mode")

            # Iterate through the list of mipi dphy
            lvds_reg = self._design.get_block_reg(
                des_db.PeriDesign.BlockType.adv_lvds)

            # Open the file
            rptfile = open(self._report_file, 'a')
            sdcfile = open(self._sdc_file, 'a')

            if lvds_reg is not None:

                all_tx = lvds_reg.get_all_tx_inst()
                all_rx = lvds_reg.get_all_rx_inst()
                all_bidir = lvds_reg.get_all_bidir_inst()

                # Get the device service based on the mipi dphy type
                dbi = device_dbi.DeviceDBService(self._device)

                lvds_tx_svc = dbi.get_block_service(
                    device_dbi.DeviceDBService.BlockType.HSIO_LVDS_TX)
                lvds_rx_svc = dbi.get_block_service(
                    device_dbi.DeviceDBService.BlockType.HSIO_LVDS_RX)
                lvds_bidir_svc = dbi.get_block_service(
                    device_dbi.DeviceDBService.BlockType.HSIO_LVDS_BIDIR)

                # Write out the create_clock constraints first
                self._write_clocks(sdcfile, all_lvds, all_rx, all_bidir)

                # Write out the RX
                type_index = 1
                if all_rx:
                    ins_exists = True
                    self._write_rx(index, sub_index, type_index, all_lvds,
                                   all_rx, rptfile, sdcfile, rx_blk_mode, lvds_rx_svc)
                    type_index += 1

                # Write out the TX
                if all_tx:
                    ins_exists = True
                    self._write_tx(index, sub_index, type_index, all_lvds,
                                   all_tx, rptfile, sdcfile, tx_blk_mode, lvds_tx_svc)
                    type_index += 1

                # Write out the BIDIR
                if all_bidir:
                    ins_exists = True
                    self._write_bidir(index, sub_index, type_index, all_lvds,
                                      all_bidir, rptfile, sdcfile, bidir_blk_mode, lvds_bidir_svc)

            rptfile.close()
            sdcfile.close()

            write_successful = True

        except Exception as excp:
            if write_successful is not None and\
                    not write_successful:
                if rptfile is not None:
                    rptfile.close()
                if sdcfile is not None:
                    sdcfile.close()

            raise excp

        return ins_exists

    def _write_clocks(self, sdcfile, all_lvds, all_rx, all_bidir):
        def get_name(ins_obj):
            return ins_obj.name

        # Combined bidir and rx to find clock only
        all_bidir_rx = all_rx + all_bidir

        print_title = False

        # Write out the Rx first so that we write the clk source (Rx in clock lane mode)
        for ins_obj in sorted(all_bidir_rx, key=get_name):
            if ins_obj is not None and ins_obj in all_lvds:
                if ins_obj.rx_info is not None:
                    rx_cfg = ins_obj.rx_info

                    if rx_cfg.conn_type == rx_cfg.ConnType.gclk_conn or \
                            rx_cfg.conn_type == rx_cfg.ConnType.rclk_conn:

                        ins_exists = True

                        if not print_title:
                            # Print the title/table header for the first iteration
                            sdcfile.write("\n# LVDS Clock Constraints\n")
                            sdcfile.write("##########################\n")

                            print_title = True

                        # Write out the create_clock constraint for this first
                        if rx_cfg.input_bus_name != "":
                            sdcfile.write(
                                "# create_clock -period <USER_PERIOD> -name {} [get_ports {{{}}}]\n".format(
                                    rx_cfg.input_bus_name, rx_cfg.input_bus_name))

    def _write_rx(self, index, sub_index, type_index, all_lvds, all_rx,
                  rptfile, sdcfile, rx_blk_mode, lvds_rx_svc):

        # Build the arc with clearing the map here
        self._mode2arcs = {}
        self.build_blk_rx_arc(rx_blk_mode)

        # Print the title/table header for the first iteration
        rptfile.write(
            "\n---------- {}.{}.{} LVDS Rx Timing Report (begin) ----------\n".format(
                index, sub_index, type_index))

        sdcfile.write("\n# LVDS Rx Constraints\n")
        sdcfile.write("#######################\n")

        # Write out the Rx arcs
        self._write_rx_report_and_constraints(rptfile, sdcfile,
                                              all_rx, lvds_rx_svc, rx_blk_mode)

        rptfile.write(
            "\n---------- LVDS Rx Timing Report (end) ----------\n")

    def _write_tx(self, index, sub_index, type_index, all_lvds, all_tx,
                  rptfile, sdcfile, tx_blk_mode, lvds_tx_svc):

        # Build the arc with clearing the map here
        self._mode2arcs = {}
        self.build_blk_tx_arc(tx_blk_mode)

        # Print the title/table header for the first iteration
        rptfile.write(
            "\n---------- {}.{}.{} LVDS Tx Timing Report (begin) ----------\n".format(
                index, sub_index, type_index))

        sdcfile.write("\n# LVDS Tx Constraints\n")
        sdcfile.write("#######################\n")

        # Write out the Tx arcs
        self._write_tx_report_and_constraints(rptfile, sdcfile,
                                              all_tx, lvds_tx_svc, tx_blk_mode)

        rptfile.write(
            "\n---------- LVDS Tx Timing Report (end) ----------\n")

    def _write_bidir(self, index, sub_index, type_index, all_lvds,
                     all_bidir, rptfile, sdcfile, bidir_blk_mode, lvds_bidir_svc):

        # Build the arc with clearing the map here
        self._mode2arcs = {}
        self.build_blk_tx_arc(bidir_blk_mode)
        self.build_blk_rx_arc(bidir_blk_mode)

        rptfile.write(
            "\n---------- {}.{}.{} Bidirectional LVDS Timing Report (begin) ----------\n".format(
                index, sub_index, type_index))

        sdcfile.write("\n# Bidirectional LVDS Constraints\n")
        sdcfile.write("##################################\n")

        # Write out the Rx and Tx arcs which makes up the bidir mode
        self._write_rx_report_and_constraints(rptfile, sdcfile,
                                              all_bidir, lvds_bidir_svc, bidir_blk_mode)
        self._write_tx_report_and_constraints(rptfile, sdcfile,
                                              all_bidir, lvds_bidir_svc, bidir_blk_mode)

        rptfile.write(
            "\n---------- Bidirectional LVDS Timing Report (end) ----------\n")

    def _write_rx_report_and_constraints(self, rptfile, sdcfile, all_rx, lvds_dev_svc, blk_mode):

        # Print out lvds out mode
        self._write_lvds_rx_input_mode(sdcfile, all_rx, lvds_dev_svc, blk_mode)

        # Print out the reset
        self._write_lvds_rx_reset(sdcfile, all_rx, lvds_dev_svc, blk_mode)

        # Print out bypass lvds mode and any clock conn type
        self._write_lvds_rx_bypass_mode_and_clock(rptfile, all_rx)

        # Print out the fifo constraints
        self._write_lvds_rx_fifo(sdcfile, all_rx, lvds_dev_svc, blk_mode)

        # Print out the delay constraints
        self._write_lvds_rx_delay(sdcfile, all_rx, lvds_dev_svc, blk_mode)

    def _write_tx_report_and_constraints(self, rptfile, sdcfile, all_tx, lvds_dev_svc, blk_mode):
        # Print out lvds out mode
        self._write_lvds_tx_out_mode(sdcfile, all_tx, lvds_dev_svc, blk_mode)

        # Print out the reset
        self._write_lvds_tx_reset(sdcfile, all_tx, lvds_dev_svc, blk_mode)

        # Print out clkout mode
        self._write_lvds_tx_clkout_mode(rptfile, all_tx)

        # Print out bypass lvds mode
        self._write_lvds_tx_bypass_mode(rptfile, all_tx)

    def _write_lvds_tx_out_mode(self, sdcfile, all_tx, lvds_dev_svc, blk_mode):
        def get_name(ins_obj):
            return ins_obj.name

        # This is a map of design instance to the design object
        for lvds_obj in sorted(all_tx, key=get_name):

            tx_config = lvds_obj.tx_info

            if tx_config is None:
                continue

            # clkout only works with serial_width > 1. So, we also want to write
            # out the constraint on OE as a register even in clkout mode
            if (tx_config.mode == tx_config.ModeType.out or
                tx_config.mode == tx_config.ModeType.clkout) \
                    and tx_config.is_serial:

                # We don't check the validity of the members
                clk_name = tx_config.slow_clock_name
                clk_str = "-clock " + clk_name

                if tx_config.mode == tx_config.ModeType.out:
                    if tx_config.oe_name != "":
                        mode_list = [self.LVDSPathType.tx_output,
                                     self.LVDSPathType.tx_out_enable]
                        pin_name_list = [
                            tx_config.output_bus_name, tx_config.oe_name]
                    else:
                        mode_list = [self.LVDSPathType.tx_output]
                        pin_name_list = [tx_config.output_bus_name]

                elif tx_config.mode == tx_config.ModeType.clkout and \
                        tx_config.oe_name != "":
                    # We're not checking for width > 1 since that should have been guaranteed by rule
                    mode_list = [self.LVDSPathType.tx_out_enable]
                    pin_name_list = [tx_config.oe_name]

                else:
                    # Skip
                    continue

                # Iterate through the two list
                arr_size = len(mode_list)

                for pindex in range(arr_size):

                    # Get the arcs for that mode
                    arcs_list = self._mode2arcs[mode_list[pindex]]
                    pin_name = pin_name_list[pindex]

                    # Update pin name since it is a bus for output only
                    if pin_name == tx_config.output_bus_name:
                        pin_name = "{}[*]".format(pin_name)

                    setup_arc = None
                    hold_arc = None
                    max_delay = 0
                    min_delay = 0

                    max_arc2delay_map, min_arc2delay_map = self._get_timing_delay(
                        lvds_obj, mode_list[pindex])

                    for arc in arcs_list:
                        param_name = arc.get_delay()
                        sink_name = arc.get_sink()
                        source_name = arc.get_source()

                        if not tx_config.is_serial_width_gt2():
                            # Get the pin associated to the TX_SCLKP
                            clk_type_name = "TX_SCLKP"

                        else:
                            _, _, clk_type_name = self.get_reg_pin_and_clk_name(
                                lvds_dev_svc, lvds_obj, sink_name, source_name, blk_mode, True)

                        ref_arg_str = self.set_clkout_ref_pin(
                            lvds_obj, clk_type_name, clk_name, lvds_obj.lvds_def)

                        if arc.get_type() == TimingArc.TimingArcType.setup:
                            if arc in max_arc2delay_map:
                                # Get the calculated delay
                                max_delay = self._get_delay_by_arc(
                                    max_arc2delay_map, self._max_model.get_tscale(), arc)

                            self.logger.debug("Reading tx out setup param {} delay {}".format(
                                param_name, max_delay))

                            setup_arc = arc

                        elif arc.get_type() == TimingArc.TimingArcType.hold:
                            if arc in min_arc2delay_map:
                                # Get the calculated delay
                                min_delay = self._get_delay_by_arc(
                                    min_arc2delay_map, self._min_model.get_tscale(), arc)

                            self.logger.debug("Reading tx out hold param {} delay {}".format(
                                param_name, min_delay))

                            hold_arc = arc

                    routing_delay_max = HSIOTiming.identify_routing_delay_on_ins(
                        lvds_obj.block_def, TimingArc.TimingArcType.setup,
                        False, self._max_model.get_tscale(), self.ins2bankname_map,
                        self.routing_label2bank_tbl_max_map)
                    routing_delay_min = HSIOTiming.identify_routing_delay_on_ins(
                        lvds_obj.block_def, TimingArc.TimingArcType.hold,
                        False, self._min_model.get_tscale(), self.ins2bankname_map,
                        self.routing_label2bank_tbl_min_map) 
                    
                    if setup_arc is not None:
                        max_out_delay = "{0:.3f}".format(max_delay + routing_delay_max)

                        sdcfile.write("set_output_delay {}{} -max {} [get_ports {{{}}}]\n".format(
                            clk_str, ref_arg_str, max_out_delay, pin_name))

                    if hold_arc is not None:
                        min_out_delay = "{0:.3f}".format((-1 * min_delay) + routing_delay_min)

                        sdcfile.write("set_output_delay {}{} -min {} [get_ports {{{}}}]\n".format(
                            clk_str, ref_arg_str, min_out_delay, pin_name))

    def _write_lvds_tx_reset(self, sdcfile, all_tx, lvds_dev_svc, blk_mode):
        def get_name(ins_obj):
            return ins_obj.name

        # This is a map of design instance to the design object
        for lvds_obj in sorted(all_tx, key=get_name):

            tx_config = lvds_obj.tx_info

            # Reset is applicable whenever the pin is specified
            # and not a bypass
            if tx_config is None or tx_config.reset_name == "":
                continue

            pin_name = tx_config.reset_name

            if tx_config.mode == tx_config.ModeType.out and tx_config.is_serial:

                # We don't check the validity of the members
                clk_name = tx_config.slow_clock_name
                clk_str = "-clock " + clk_name

                arcs_list = self._mode2arcs[self.LVDSPathType.tx_reset]

                setup_arc = None
                hold_arc = None
                max_delay = 0
                min_delay = 0

                max_arc2delay_map, min_arc2delay_map = self._get_timing_delay(
                    lvds_obj, self.LVDSPathType.tx_reset)

                for arc in arcs_list:

                    # param_name = arc.get_delay()
                    sink_name = arc.get_sink()
                    source_name = arc.get_source()

                    if not tx_config.is_serial_width_gt2():
                        # Get the pin associated to the TX_SCLKP
                        clk_type_name = "TX_SCLKP"
                    else:
                        _, _, clk_type_name = self.get_reg_pin_and_clk_name(
                            lvds_dev_svc, lvds_obj, sink_name, source_name, blk_mode, True)

                    ref_arg_str = self.set_clkout_ref_pin(
                        lvds_obj, clk_type_name, clk_name, lvds_obj.lvds_def)

                    if arc.get_type() == TimingArc.TimingArcType.setup:
                        if arc in max_arc2delay_map:
                            # Get the calculated delay
                            max_delay = self._get_delay_by_arc(
                                max_arc2delay_map, self._max_model.get_tscale(), arc)

                        setup_arc = arc

                    elif arc.get_type() == TimingArc.TimingArcType.hold:
                        if arc in min_arc2delay_map:
                            # Get the calculated delay
                            min_delay = self._get_delay_by_arc(
                                min_arc2delay_map, self._min_model.get_tscale(), arc)

                        hold_arc = arc

                routing_delay_max = HSIOTiming.identify_routing_delay_on_ins(
                    lvds_obj.block_def, TimingArc.TimingArcType.setup,
                    False, self._max_model.get_tscale(), self.ins2bankname_map,
                    self.routing_label2bank_tbl_max_map)
                routing_delay_min = HSIOTiming.identify_routing_delay_on_ins(
                    lvds_obj.block_def, TimingArc.TimingArcType.hold,
                    False, self._min_model.get_tscale(), self.ins2bankname_map,
                    self.routing_label2bank_tbl_min_map) 
                
                if setup_arc is not None:
                    max_out_delay = "{0:.3f}".format(max_delay + routing_delay_max)

                    sdcfile.write("set_output_delay {}{} -max {} [get_ports {{{}}}]\n".format(
                        clk_str, ref_arg_str, max_out_delay, pin_name))

                if hold_arc is not None:
                    min_out_delay = "{0:.3f}".format((-1 * min_delay) + routing_delay_min)

                    sdcfile.write("set_output_delay {}{} -min {} [get_ports {{{}}}]\n".format(
                        clk_str, ref_arg_str, min_out_delay, pin_name))

    def _write_lvds_tx_clkout_mode(self, rptfile, all_tx):

        comb_table = PrettyTable(
            ["Instance Name", "Clock Pin", "Parameter", "Max (ns)", "Min (ns)"])
        found = False

        def get_name(ins_obj):
            return ins_obj.name

        # This is a map of design instance to the design object
        for lvds_obj in sorted(all_tx, key=get_name):
            tx_config = lvds_obj.tx_info

            row_list = []

            if tx_config is not None and tx_config.mode == tx_config.ModeType.clkout:

                clk_name = tx_config.fast_clock_name

                # Get the arcs for that mode
                arcs_list = self._mode2arcs[self.LVDSPathType.tx_clkout]

                found = True
                max_delay = 0
                min_delay = 0

                max_arc2delay_map, min_arc2delay_map = self._get_timing_delay(
                    lvds_obj, self.LVDSPathType.tx_clkout)

                for arc in arcs_list:

                    if arc in max_arc2delay_map:
                        # Get the calculated delay
                        max_delay = self._get_delay_by_arc(
                            max_arc2delay_map, self._max_model.get_tscale(), arc)

                    if arc in min_arc2delay_map:
                        # Get the calculated delay
                        min_delay = self._get_delay_by_arc(
                            min_arc2delay_map, self._min_model.get_tscale(), arc)

                routing_delay_max = HSIOTiming.identify_routing_delay_on_ins(
                    lvds_obj.block_def, TimingArc.TimingArcType.delay,
                    False, self._max_model.get_tscale(), self.ins2bankname_map,
                    self.routing_label2bank_tbl_max_map, is_clkout=True)
                routing_delay_min = HSIOTiming.identify_routing_delay_on_ins(
                    lvds_obj.block_def, TimingArc.TimingArcType.delay,
                    False, self._min_model.get_tscale(), self.ins2bankname_map,
                    self.routing_label2bank_tbl_min_map, is_clkout=True) 
                
                row_list.append(lvds_obj.name)
                row_list.append(clk_name)
                row_list.append("GPIO_CLK_OUT")
                row_list.append("{0:.3f}".format(max_delay + routing_delay_max))
                row_list.append("{0:.3f}".format(min_delay + routing_delay_min))

                comb_table.add_row(row_list)

        if found:
            rptfile.write("\nLVDS Tx Clkout Configuration\n")
            rptfile.write("=============================\n")
            rptfile.write("\n{}\n".format(comb_table.get_string()))

    # Print out bypass lvds mode
    def _write_lvds_tx_bypass_mode(self, rptfile, all_tx):

        found = False

        table = PrettyTable(["Instance Name", "Pin Name",
                            "Parameter", "Max (ns)", "Min (ns)"])

        def get_name(ins_obj):
            return ins_obj.name

        # This is a map of design instance to the design object
        for lvds_obj in sorted(all_tx, key=get_name):
            tx_config = lvds_obj.tx_info

            if tx_config is not None and tx_config.mode == tx_config.ModeType.out and\
                    not tx_config.is_serial:

                if tx_config.oe_name != "":
                    mode_list = [self.LVDSPathType.tx_bypass_out,
                                 self.LVDSPathType.tx_bypass_oe]
                    pin_name_list = [
                        tx_config.output_bus_name, tx_config.oe_name]
                else:
                    mode_list = [self.LVDSPathType.tx_bypass_out]
                    pin_name_list = [tx_config.output_bus_name]

                # Iterate through the two list
                arr_size = len(mode_list)
                found = True

                routing_delay_max = HSIOTiming.identify_routing_delay_on_ins(
                    lvds_obj.block_def, TimingArc.TimingArcType.delay,
                    False, self._max_model.get_tscale(), self.ins2bankname_map,
                    self.routing_label2bank_tbl_max_map)
                routing_delay_min = HSIOTiming.identify_routing_delay_on_ins(
                    lvds_obj.block_def, TimingArc.TimingArcType.delay,
                    False, self._min_model.get_tscale(), self.ins2bankname_map,
                    self.routing_label2bank_tbl_min_map) 
                
                for pindex in range(arr_size):
                    row_list = []

                    # Get the arcs for that mode
                    arcs_list = self._mode2arcs[mode_list[pindex]]
                    pin_name = pin_name_list[pindex]

                    max_delay = 0
                    min_delay = 0

                    max_arc2delay_map, min_arc2delay_map = self._get_timing_delay(
                        lvds_obj, mode_list[pindex])

                    for arc in arcs_list:

                        if arc in max_arc2delay_map:
                            # Get the calculated delay
                            max_delay = self._get_delay_by_arc(
                                max_arc2delay_map, self._max_model.get_tscale(), arc)

                        if arc in min_arc2delay_map:
                            # Get the calculated delay
                            min_delay = self._get_delay_by_arc(
                                min_arc2delay_map, self._min_model.get_tscale(), arc)
                
                    row_list.append(lvds_obj.name)
                    row_list.append(pin_name)
                    row_list.append("GPIO_OUT")
                    row_list.append("{0:.3f}".format(max_delay + routing_delay_max))
                    row_list.append("{0:.3f}".format(min_delay + routing_delay_min))

                    table.add_row(row_list)

        if found:
            rptfile.write("\nLVDS Tx Bypass Configuration\n")
            rptfile.write("=============================\n")
            rptfile.write("\n{}\n".format(table.get_string()))

    def _write_lvds_rx_input_mode(self, sdcfile, all_rx, lvds_dev_svc, blk_mode):
        def get_name(ins_obj):
            return ins_obj.name

        # This is a map of design instance to the design object
        for lvds_obj in sorted(all_rx, key=get_name):
            rx_config = lvds_obj.rx_info

            if rx_config is None:
                continue

            pin_name = rx_config.input_bus_name

            if rx_config.conn_type == rx_config.ConnType.normal_conn and rx_config.is_serial:

                # We don't check the validity of the members
                clk_name = rx_config.slow_clock_name
                clk_str = "-clock " + clk_name

                arcs_list = self._mode2arcs[self.LVDSPathType.rx_input]

                # Update pin name since it is a bus
                pin_name = "{}[*]".format(pin_name)

                tco_arc = None
                max_delay = 0
                min_delay = 0

                max_arc2delay_map, min_arc2delay_map = self._get_timing_delay(
                    lvds_obj, self.LVDSPathType.rx_input)

                for arc in arcs_list:
                    param_name = arc.get_delay()
                    sink_name = arc.get_sink()
                    source_name = arc.get_source()

                    # Skip irrelevant arc depending on fifo feature used
                    if rx_config.is_fifo:
                        if source_name == "INSLOWCLK":
                            continue

                        # The name should be the fifo clk name
                        fifo_pin = rx_config.gen_pin.get_pin_by_type_name(
                            "FIFOCLK")
                        if fifo_pin is not None and fifo_pin.name != "":
                            clk_name = fifo_pin.name
                            clk_str = "-clock " + clk_name

                    elif source_name == "FIFOCLK":
                        continue

                    if not rx_config.is_serial_width_gt2():
                        # Get the pin associated to the RX_SCLKP
                        # Use the FASTCLK for < 2
                        clk_type_name = "RX_SCLKP"
                    else:
                        _, _, clk_type_name = self.get_reg_pin_and_clk_name(
                            lvds_dev_svc, lvds_obj, sink_name, source_name, blk_mode, False)

                    ref_arg_str = self.set_clkout_ref_pin(
                        lvds_obj, clk_type_name, clk_name, lvds_obj.lvds_def)

                    if arc.get_type() == TimingArc.TimingArcType.clk_to_q:
                        if arc in max_arc2delay_map:
                            # Get the calculated delay
                            max_delay = self._get_delay_by_arc(
                                max_arc2delay_map, self._max_model.get_tscale(), arc)

                        #self.logger.debug("Reading rx tco max param {} delay {}".format(
                        #    param_name, max_delay))

                        if arc in min_arc2delay_map:
                            # Get the calculated delay
                            min_delay = self._get_delay_by_arc(
                                min_arc2delay_map, self._min_model.get_tscale(), arc)

                        #self.logger.debug("Reading rx tco min param {} delay {}".format(
                        #    param_name, min_delay))

                        tco_arc = arc

                if tco_arc is not None:
                    routing_delay_max = HSIOTiming.identify_routing_delay_on_ins(
                        lvds_obj.block_def, TimingArc.TimingArcType.clk_to_q,
                        True, self._max_model.get_tscale(), self.ins2bankname_map,
                        self.routing_label2bank_tbl_max_map)
                    routing_delay_min = HSIOTiming.identify_routing_delay_on_ins(
                        lvds_obj.block_def, TimingArc.TimingArcType.clk_to_q,
                        True, self._min_model.get_tscale(), self.ins2bankname_map,
                        self.routing_label2bank_tbl_min_map) 

                    max_in_delay = "{0:.3f}".format(max_delay + routing_delay_max)

                    sdcfile.write("set_input_delay {}{} -max {} [get_ports {{{}}}]\n".format(
                        clk_str, ref_arg_str, max_in_delay, pin_name))

                    min_in_delay = "{0:.3f}".format(min_delay + routing_delay_min)

                    sdcfile.write("set_input_delay {}{} -min {} [get_ports {{{}}}]\n".format(
                        clk_str, ref_arg_str, min_in_delay, pin_name))

    def _write_pin_sdc(self, sdcfile, arc, max_arc2delay_map, min_arc2delay_map,
                       clk_str, ref_arg_str, pin_name, lvds_obj):
        
        if arc.get_type() == TimingArc.TimingArcType.setup:
            if arc in max_arc2delay_map:
                routing_delay_max = HSIOTiming.identify_routing_delay_on_ins(
                    lvds_obj.block_def, TimingArc.TimingArcType.setup,
                    False, self._max_model.get_tscale(), self.ins2bankname_map,
                    self.routing_label2bank_tbl_max_map)
                                
                # Get the calculated delay
                max_delay = self._get_delay_by_arc(
                    max_arc2delay_map, self._max_model.get_tscale(), arc)
           
                max_out_delay = "{0:.3f}".format(max_delay + routing_delay_max)

                sdcfile.write("set_output_delay {}{} -max {} [get_ports {{{}}}]\n".format(
                    clk_str, ref_arg_str, max_out_delay, pin_name))

        elif arc.get_type() == TimingArc.TimingArcType.hold:
            if arc in min_arc2delay_map:
                routing_delay_min = HSIOTiming.identify_routing_delay_on_ins(
                    lvds_obj.block_def, TimingArc.TimingArcType.hold,
                    False, self._min_model.get_tscale(), self.ins2bankname_map,
                    self.routing_label2bank_tbl_min_map) 
                                
                # Get the calculated delay
                min_delay = self._get_delay_by_arc(
                    min_arc2delay_map, self._min_model.get_tscale(), arc)

                min_out_delay = "{0:.3f}".format((-1 * min_delay) + routing_delay_min)

                sdcfile.write("set_output_delay {}{} -min {} [get_ports {{{}}}]\n".format(
                    clk_str, ref_arg_str, min_out_delay, pin_name))

        elif arc.get_type() == TimingArc.TimingArcType.clk_to_q:
            routing_delay_max = HSIOTiming.identify_routing_delay_on_ins(
                lvds_obj.block_def, TimingArc.TimingArcType.clk_to_q,
                True, self._max_model.get_tscale(), self.ins2bankname_map,
                self.routing_label2bank_tbl_max_map)
            routing_delay_min = HSIOTiming.identify_routing_delay_on_ins(
                lvds_obj.block_def, TimingArc.TimingArcType.clk_to_q,
                True, self._min_model.get_tscale(), self.ins2bankname_map,
                self.routing_label2bank_tbl_min_map) 
                        
            if arc in max_arc2delay_map:
                # Get the calculated delay
                max_delay = self._get_delay_by_arc(
                    max_arc2delay_map, self._max_model.get_tscale(), arc)

                max_in_delay = "{0:.3f}".format(max_delay + routing_delay_max)

                sdcfile.write("set_input_delay {}{} -max {} [get_ports {{{}}}]\n".format(
                    clk_str, ref_arg_str, max_in_delay, pin_name))

            if arc in min_arc2delay_map:
                # Get the calculated delay
                min_delay = self._get_delay_by_arc(
                    min_arc2delay_map, self._min_model.get_tscale(), arc)

                min_in_delay = "{0:.3f}".format(min_delay + routing_delay_min)

                sdcfile.write("set_input_delay {}{} -min {} [get_ports {{{}}}]\n".format(
                    clk_str, ref_arg_str, min_in_delay, pin_name))

    def _write_lvds_rx_fifo(self, sdcfile, all_rx, lvds_dev_svc, blk_mode):
        def get_name(ins_obj):
            return ins_obj.name

        # This is a map of design instance to the design object
        for lvds_obj in sorted(all_rx, key=get_name):
            rx_config = lvds_obj.rx_info

            # Reset is applicable whenever the pin is specified
            # and not a bypass
            if rx_config is None or rx_config.gen_pin is None:
                continue

            if rx_config.conn_type == rx_config.ConnType.normal_conn and \
                    rx_config.is_serial and rx_config.is_fifo:
                # Fifo is only applicable with serialization enabled
                # Get the FIFOCLK
                clk_name = ""
                gpin = rx_config.gen_pin.get_pin_by_type_name("FIFOCLK")
                if gpin is not None and gpin.name != "":
                    clk_name = gpin.name

                if clk_name == "":
                    continue

                clk_str = "-clock " + clk_name

                arcs_list = self._mode2arcs[self.LVDSPathType.rx_fifo]

                max_arc2delay_map, min_arc2delay_map = self._get_timing_delay(
                    lvds_obj, self.LVDSPathType.rx_fifo)

                for arc in arcs_list:
                    sink_name = arc.get_sink()
                    source_name = arc.get_source()

                    gpin = rx_config.gen_pin.get_pin_by_type_name(sink_name)
                    if gpin is not None and gpin.name != "":
                        pin_name = gpin.name

                        _, _, clk_type_name = self.get_reg_pin_and_clk_name(
                            lvds_dev_svc, lvds_obj, sink_name, source_name, blk_mode, False)

                        ref_arg_str = self.set_clkout_ref_pin(
                            lvds_obj, clk_type_name, clk_name, lvds_obj.lvds_def)

                        self._write_pin_sdc(sdcfile, arc, max_arc2delay_map, min_arc2delay_map,
                                            clk_str, ref_arg_str, pin_name, lvds_obj)

    def _write_lvds_rx_delay(self, sdcfile, all_rx, lvds_dev_svc, blk_mode):
        def get_name(ins_obj):
            return ins_obj.name

        # This is a map of design instance to the design object
        for lvds_obj in sorted(all_rx, key=get_name):
            rx_config = lvds_obj.rx_info
            is_dyn_ena = False

            # Reset is applicable whenever the pin is specified
            # and not a bypass
            if rx_config is None or rx_config.gen_pin is None:
                continue

            # Only dynamic delay need clock fall on the 2 delay pins
            if rx_config.delay_mode == rx_config.DelayModeType.dynamic:
                valid_pin_names = rx_config.get_delay_pin()
                is_dyn_ena = True
            elif rx_config.delay_mode == rx_config.DelayModeType.dpa:
                valid_pin_names = rx_config.get_delay_dpa_pin()
            else:
                continue

            # We don't check the validity of the members
            clk_name = rx_config.slow_clock_name
            clk_sdc_str = "-clock " + clk_name
            
            arcs_list = self._mode2arcs[self.LVDSPathType.rx_dyn_dpa_dly]

            max_arc2delay_map, min_arc2delay_map = self._get_timing_delay(
                lvds_obj, self.LVDSPathType.rx_dyn_dpa_dly)

            for arc in arcs_list:

                sink_name = arc.get_sink()
                source_name = arc.get_source()

                # Skip irrelevant arc
                if sink_name not in valid_pin_names:
                    continue

                if sink_name in ["DLY_ENA", "DLY_INC"] and is_dyn_ena:
                    clk_str = clk_sdc_str + " -clock_fall"
                else:
                    clk_str = clk_sdc_str

                gpin = rx_config.gen_pin.get_pin_by_type_name(sink_name)
                if gpin is not None and gpin.name != "":
                    pin_name = gpin.name

                    _, _, clk_type_name = self.get_reg_pin_and_clk_name(
                        lvds_dev_svc, lvds_obj, sink_name, source_name, blk_mode, False)

                    ref_arg_str = self.set_clkout_ref_pin(
                        lvds_obj, clk_type_name, clk_name, lvds_obj.lvds_def)

                    self._write_pin_sdc(sdcfile, arc, max_arc2delay_map, min_arc2delay_map,
                                        clk_str, ref_arg_str, pin_name, lvds_obj)

    def _write_lvds_rx_reset(self, sdcfile, all_rx, lvds_dev_svc, blk_mode):
        def get_name(ins_obj):
            return ins_obj.name

        # This is a map of design instance to the design object
        for lvds_obj in sorted(all_rx, key=get_name):
            rx_config = lvds_obj.rx_info

            # Reset is applicable whenever the pin is specified
            # and not a bypass
            if rx_config is None or rx_config.reset_name == "":
                continue

            pin_name = rx_config.reset_name

            if rx_config.conn_type == rx_config.ConnType.normal_conn and rx_config.is_serial:

                # We don't check the validity of the members
                clk_name = rx_config.slow_clock_name
                clk_str = "-clock " + clk_name

                arcs_list = self._mode2arcs[self.LVDSPathType.rx_reset]

                setup_arc = None
                hold_arc = None
                max_delay = 0
                min_delay = 0

                max_arc2delay_map, min_arc2delay_map = self._get_timing_delay(
                    lvds_obj, self.LVDSPathType.rx_reset)

                for arc in arcs_list:

                    sink_name = arc.get_sink()
                    source_name = arc.get_source()

                    # Skip irrelevant arc depending on fifo feature used
                    if rx_config.is_fifo:
                        if source_name == "INSLOWCLK":
                            continue

                        # The name should be the fifo clk name
                        fifo_pin = rx_config.gen_pin.get_pin_by_type_name(
                            "FIFOCLK")
                        if fifo_pin is not None and fifo_pin.name != "":
                            clk_name = fifo_pin.name
                            clk_str = "-clock " + clk_name

                    elif source_name == "FIFOCLK":
                        continue

                    if not rx_config.is_serial_width_gt2():
                        # Get the pin associated to the RX_SCLKP
                        # Use the FASTCLK for < 2
                        clk_type_name = "RX_SCLKP"
                    else:
                        _, _, clk_type_name = self.get_reg_pin_and_clk_name(
                            lvds_dev_svc, lvds_obj, sink_name, source_name, blk_mode, False)

                    ref_arg_str = self.set_clkout_ref_pin(
                        lvds_obj, clk_type_name, clk_name, lvds_obj.lvds_def)

                    if arc.get_type() == TimingArc.TimingArcType.setup:
                        if arc in max_arc2delay_map:
                            # Get the calculated delay
                            max_delay = self._get_delay_by_arc(
                                max_arc2delay_map, self._max_model.get_tscale(), arc)

                        setup_arc = arc

                    elif arc.get_type() == TimingArc.TimingArcType.hold:
                        if arc in min_arc2delay_map:
                            # Get the calculated delay
                            min_delay = self._get_delay_by_arc(
                                min_arc2delay_map, self._min_model.get_tscale(), arc)

                        hold_arc = arc

                routing_delay_max = HSIOTiming.identify_routing_delay_on_ins(
                    lvds_obj.block_def, TimingArc.TimingArcType.setup,
                    False, self._max_model.get_tscale(), self.ins2bankname_map,
                    self.routing_label2bank_tbl_max_map)
                routing_delay_min = HSIOTiming.identify_routing_delay_on_ins(
                    lvds_obj.block_def, TimingArc.TimingArcType.hold,
                    False, self._min_model.get_tscale(), self.ins2bankname_map,
                    self.routing_label2bank_tbl_min_map) 

                if setup_arc is not None:
                    max_out_delay = "{0:.3f}".format(max_delay + routing_delay_max)

                    sdcfile.write("set_output_delay {}{} -max {} [get_ports {{{}}}]\n".format(
                        clk_str, ref_arg_str, max_out_delay, pin_name))

                if hold_arc is not None:
                    min_out_delay = "{0:.3f}".format((-1 * min_delay) + routing_delay_min)

                    sdcfile.write("set_output_delay {}{} -min {} [get_ports {{{}}}]\n".format(
                        clk_str, ref_arg_str, min_out_delay, pin_name))

    def _write_lvds_rx_bypass_mode_and_clock(self, rptfile, all_rx):
        def get_name(ins_obj):
            return ins_obj.name

        found = False

        table = PrettyTable(["Instance Name", "Pin Name",
                            "Parameter", "Max (ns)", "Min (ns)"])

        # This is a map of design instance to the design object
        for lvds_obj in sorted(all_rx, key=get_name):
            rx_config = lvds_obj.rx_info

            if rx_config is None:
                continue

            if not rx_config.is_serial:

                mode_list = [self.LVDSPathType.rx_bypass]
                pin_name_list = [rx_config.input_bus_name]

                # Iterate through the two list
                arr_size = len(mode_list)
                found = True

                for pindex in range(arr_size):
                    row_list = []

                    # Get the arcs for that mode
                    arcs_list = self._mode2arcs[mode_list[pindex]]
                    pin_name = pin_name_list[pindex]

                    max_delay = 0
                    min_delay = 0

                    max_arc2delay_map, min_arc2delay_map = self._get_timing_delay(
                        lvds_obj, self.LVDSPathType.rx_bypass)

                    for arc in arcs_list:

                        if arc in max_arc2delay_map:
                            # Get the calculated delay
                            max_delay = self._get_delay_by_arc(
                                max_arc2delay_map, self._max_model.get_tscale(), arc)

                        if arc in min_arc2delay_map:
                            # Get the calculated delay
                            min_delay = self._get_delay_by_arc(
                                min_arc2delay_map, self._min_model.get_tscale(), arc)

                    row_list.append(lvds_obj.name)
                    row_list.append(pin_name)
                    # PT-1350: Added the report for LVDS GCLK/RCLK
                    if rx_config.conn_type == rx_config.ConnType.gclk_conn or \
                        rx_config.conn_type == rx_config.ConnType.rclk_conn:
                        row_list.append("GPIO_CLK_IN")
                    else:
                        row_list.append("GPIO_IN")

                    routing_delay_max = HSIOTiming.identify_routing_delay_on_ins(
                        lvds_obj.block_def, TimingArc.TimingArcType.delay,
                        True, self._max_model.get_tscale(), self.ins2bankname_map,
                        self.routing_label2bank_tbl_max_map)
                    routing_delay_min = HSIOTiming.identify_routing_delay_on_ins(
                        lvds_obj.block_def, TimingArc.TimingArcType.delay,
                        True, self._min_model.get_tscale(), self.ins2bankname_map,
                        self.routing_label2bank_tbl_min_map) 
                    
                    total_max_delay = max_delay + routing_delay_max
                    total_min_delay = min_delay + routing_delay_min

                    row_list.append("{0:.3f}".format(total_max_delay))
                    row_list.append("{0:.3f}".format(total_min_delay))

                    table.add_row(row_list)

                    if rx_config.is_alternate_connection():
                        is_clk = rx_config.is_alternate_clock_type()

                        assert lvds_obj.name not in self.ins2_alt_delay_map
                        self.ins2_alt_delay_map[lvds_obj.name] = \
                            (pin_name, total_max_delay, total_min_delay, is_clk)

        if found:
            rptfile.write("\nLVDS Rx Bypass Configuration\n")
            rptfile.write("=============================\n")
            rptfile.write("\n{}\n".format(table.get_string()))

    def get_reg_pin_and_clk_name(self, lvds_dev_svc, ins_obj, sink_name, source_name, blk_mode, is_tx):
        # The sink and source name are interface name in the mode and
        # not necessarily the block port name
        pin_name = ""
        clk_name = ""
        clk_port_name = ""

        if is_tx:
            lvds_info = ins_obj.tx_info

            # For TX, all pins are non-gen_pin
            lvds_pin_name_to_variable = {
                "OUT": lvds_info.output_bus_name,
                "OUTSLOWCLK": lvds_info.slow_clock_name,
                "OUTFASTCLK": lvds_info.fast_clock_name,
                "OUTRST": lvds_info.reset_name,
                "OE": lvds_info.oe_name
            }
        else:
            lvds_info = ins_obj.rx_info

            # For RX, some are non-gen_pin and some are gen_pin
            lvds_pin_name_to_variable = {
                "IN": lvds_info.input_bus_name,
                "INSLOWCLK": lvds_info.slow_clock_name,
                "INFASTCLK": lvds_info.fast_clock_name,
                "INRST": lvds_info.reset_name
            }

        self.logger.debug("lvds  get_reg_pin_and_clk_name src: {}-{}, sink: {}-{}".format(
            sink_name, lvds_pin_name_to_variable.get(sink_name, "NULL"),
            source_name, lvds_pin_name_to_variable.get(source_name, "NULL")))

        # Try to attempt based on the map first since not all pins are using gen_pin
        if sink_name in lvds_pin_name_to_variable:
            pin_name = lvds_pin_name_to_variable[sink_name]

            src_inf = blk_mode.get_interface_object(sink_name)
            if src_inf is not None:
                if src_inf.is_bus_port():
                    pin_name = "{}[*]".format(pin_name)

        if source_name in lvds_pin_name_to_variable:
            clk_name = lvds_pin_name_to_variable[source_name]
            clk_inf = blk_mode.get_interface_object(source_name)
            clk_port_name = source_name

            if clk_inf is not None:
                if clk_inf.is_bus_port():
                    clk_name = "{}[*]".format(clk_name)

                clk_port_name = clk_inf.get_port_name()

        # We search for the gen pin if it has not been found
        if pin_name == "":
            gen_pin = lvds_dev_svc.get_user_gen_pin(
                ins_obj.name, lvds_info.gen_pin, sink_name)

            if gen_pin is None:
                msg = "Unable to find pin {} in instance {}".format(
                    sink_name, ins_obj.name)

                raise ValueError(msg)

            # Non clock port we use the bracket if it was a bus
            # Find the port based on the sink interface name and check
            # if the interface itself is a bus interface type
            src_inf = blk_mode.get_interface_object(gen_pin.type_name)
            if src_inf is not None:
                if src_inf.is_bus_port() and gen_pin.name != "":
                    pin_name = "{}[*]".format(gen_pin.name)
                else:
                    pin_name = gen_pin.name

            else:
                pin_name = gen_pin.name

        if clk_name == "":
            clk_pin = lvds_dev_svc.get_user_gen_pin(
                ins_obj.name, lvds_info.gen_pin, source_name)

            if clk_pin is None:
                if clk_pin is None:
                    msg = "Unable to find pin {} in instance {}".format(
                        source_name, ins_obj.name)

                    raise ValueError(msg)

            # We need to pass the clkpin type name that is matching the
            # device instance pin connection (port) and not the interface name
            clk_inf = blk_mode.get_interface_object(clk_pin.type_name)
            clk_port_name = clk_pin.type_name
            if clk_inf is not None:
                if not clk_inf.is_bus_port():
                    # Clock interface is not a bus port
                    clk_port_name = clk_inf.get_port_name()
                else:
                    msg = "Unexpected bus clkout interface on HSIO mode {} interface {}".format(
                        blk_mode.get_name(), clk_pin.type_name)
                    raise ValueError(msg)

            clk_name = clk_pin.name

        self.logger.debug("return: pin: {}, clk: {}, clk_port: {}".format(
            pin_name, clk_name, clk_port_name))

        return pin_name, clk_name, clk_port_name

    def _get_calculated_static_delay(self, ins_obj, delay_param, hsio_elem_tag, is_max):
        static_delay = 0
        static_delay_step_param = "HSIO_STATIC_DELAY_DIFFERENTIAL"

        arc_table = self._load_arc_parameter(
            hsio_elem_tag, static_delay_step_param, is_max)
                
        if arc_table is not None:
            value = arc_table.get_variable_delay("")

            if value is not None:

                if delay_param == "HSIO_LVDS_RX_IN_PAD":
                    if (ins_obj.ops_type == ins_obj.OpsType.op_rx or ins_obj.ops_type == ins_obj.OpsType.op_bidir) and\
                            ins_obj.rx_info is not None:
                        static_delay = ins_obj.rx_info.delay * value

                elif delay_param == "HSIO_LVDS_TX_OUT_PAD" or \
                        delay_param == "HSIO_LVDS_TX_OE_PAD":
                    if (ins_obj.ops_type == ins_obj.OpsType.op_tx or ins_obj.ops_type == ins_obj.OpsType.op_bidir) and\
                            ins_obj.tx_info is not None:
                        static_delay = ins_obj.tx_info.delay * value

        return static_delay
    
    def _get_delay_value(self, ins_obj, delay_var_map, arc_table, is_max=True):
        '''
        :param gpio_obj: The user lvds design object
        :param delay_var_map: The table of variables mapped to acceptable
                        values
        :param arc_table: The arc_parameter table with the name
                        representing the variables concatenated (as
                        in the xml file). An ArcDelayTable object

        :return the delay value of that parameter based on the required
                lvds configuration
        '''

        # Get the arc delay parameter name
        delay_param = arc_table.get_parameter_name()
        # self.logger.debug("LVDS Timing delay param {}".format(delay_param))

        # Only if the arc parameter has variable associated, then
        # we figure out, else take that one value
        if arc_table.get_pcount() == 0:
            delay_val = arc_table.get_variable_delay("")

            hsio_elem_tag = self._get_block_timing_tag_section(is_max)
            
            # Exception is for some arc which has no variation but it has
            # variation due to specific instance parameter
            delay_val += self._get_calculated_static_delay(
                ins_obj, delay_param, hsio_elem_tag, is_max)

            return delay_val

        pre_emphasis = "medium low"
        serial_type = "sdio_ddio"

        if (ins_obj.ops_type == ins_obj.OpsType.op_tx or ins_obj.ops_type == ins_obj.OpsType.op_bidir) and \
                ins_obj.tx_info is not None:
            # Convert the pre-emphasis value to string
            pre_emphasis = ins_obj.tx_info.enumtype2str(
                LVDSAdvanceParamId.pre_emphasis, ins_obj.tx_info.pre_emphasis)

            # Tx
            if ins_obj.tx_info.is_serial_width_gt2():
                serial_type = "ser"

        elif ins_obj.rx_info is not None:
            # Rx
            if ins_obj.rx_info.is_serial_width_gt2():
                serial_type = "ser"

        hsio_elem_tag = self._get_block_timing_tag_section(is_max)
            
        additional_delay = self._get_calculated_static_delay(
            ins_obj, delay_param, hsio_elem_tag, is_max)

        # Now that we have all the required setting, we
        # get the delay based on it
        var_list = arc_table.get_variables()
        concat_name = ""
        read_vcount = 0

        for index in range(len(var_list)):
            vname = var_list[index]
            vfound = True

            if vname in self.str2var_map:
                vtype = self.str2var_map[vname]

                if vtype == self.VariableType.serial_type:
                    vsetting = vname + ":" + serial_type
                elif vtype == self.VariableType.pre_emphasis:
                    vsetting = vname + ":" + pre_emphasis

                else:
                    vfound = False

                if vfound:
                    read_vcount += 1

                if concat_name == "":
                    concat_name = vsetting
                else:
                    concat_name = concat_name + "," + vsetting

            else:
                raise ValueError(
                    "Unexpected timing variable name {}".format(vname))

        # Verify that the variable count matches
        if read_vcount != arc_table.get_pcount():
            raise ValueError(
                "Mismatch variable count, {} and the pcount parameter, {}".format(
                    read_vcount, arc_table.get_pcount()))

        # Get the delay from table based on the concat variable name
        delay_val = 0
        tmp_delay = arc_table.get_variable_delay(concat_name)
        if tmp_delay is not None:
            delay_val = tmp_delay
        else:
            raise ValueError(
                "Unable to get the delay for variable {}".format(concat_name))

        delay_val += additional_delay

        return delay_val
