'''
Copyright (C) 2017-2020 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Nov 11, 2020

@author: maryam
'''
import os
import sys

import util.excp as app_excp
from util.singleton_logger import Logger

import device.excp as dev_excp
from device.block_instance import SecondaryConn

from common_device.hsio.hsio_device_service import HSIOService
from common_device.gpio.gpio_design import GPIOOutput

import writer.used_core_port as ucp

sys.path.append(os.path.join(os.path.dirname(__file__), '..'))


class HSIOLVDSInterfaceConfig(object):
    '''
    Class for Interface config generation.
    '''

    # PT-1059: The following is a hardcode of the map of pins which isn't captured
    # in the block definition:
    # The ports listed here starts with the LSB [0] to the [WIDTH-1],
    # in which the index in the array matches with the bits:
    # index 0 - bus[0]
    # index 1 - bus[1] ...
    # index n - bus[n] with WIDTH = n + 1

    # This is a continuous pattern
    _rx_half_width_to_ports_map = {
        4: [6, 7, 8, 9],
        6: [4, 5, 6, 7, 8, 9],
        8: [2, 3, 4, 5, 6, 7, 8, 9],
        10: [0, 1, 2, 3, 4, 5, 6, 7, 8, 9]
    }

    _rx_full_width_to_ports_map = {
        3: [4, 6, 8],
        4: [2, 4, 6, 8],
        5: [0, 2, 4, 6, 8],
        6: [9, 0, 2, 4, 6, 8],
        7: [7, 9, 0, 2, 4, 6, 8],
        8: [5, 7, 9, 0, 2, 4, 6, 8],
        10: [1, 3, 5, 7, 9, 0, 2, 4, 6, 8]
    }

    _tx_half_width_to_ports_map = {
        4: [0, 5, 1, 6],
        6: [0, 5, 1, 6, 2, 7],
        8: [0, 5, 1, 6, 2, 7, 3, 8],
        10: [0, 5, 1, 6, 2, 7, 3, 8, 4, 9]
    }

    # This is a continuous pattern
    _tx_full_width_to_ports_map = {
        3: [0, 1, 2],
        4: [0, 1, 2, 3],
        5: [0, 1, 2, 3, 4],
        6: [0, 1, 2, 3, 4, 5],
        7: [0, 1, 2, 3, 4, 5, 6],
        8: [0, 1, 2, 3, 4, 5, 6, 7],
        10: [0, 1, 2, 3, 4, 5, 6, 7, 8, 9]
    }

    # Mapping of the p_in value for x3-x10 with LVDS Tx clkout mode
    # It indicates the serialization width mapped to the p_in index that
    # requires to be set to vcc. The rest if not mentioned is tied to gnd
    # x2 is special as it sets only p_in[9] and p_in[3]
    _tx_clkout_half_width_tied_vcc_map = {
        4: [0, 5],
        6: [0, 1, 5],
        8: [0, 1, 5, 6],
        10: [0, 1, 2, 5, 6]
    }

    _tx_clkout_full_width_tied_vcc_map = {
        3: [0],
        4: [0, 1],
        5: [0, 1],
        6: [0, 1, 2],
        7: [0, 1, 2],
        8: [0, 1, 2, 3],
        10: [0, 1, 2, 3, 4]
    }

    def __init__(self, device_db, name, lvds_reg):
        '''
        constructor
        '''
        self._device_db = device_db
        self._lvds_reg = lvds_reg
        self._name = name
        self.logger = Logger

    def generate_hsio_lvds_interface(self, hsio_block, lvds_obj):
        '''
        Capture the core interface required for the GPIO user
        instances.

        :param hsio_block: The HSIO Block Definition object
        :param gpio_obj: The design GPIO object

        :return inf_config: A list of ucp.UsedCorePort
        '''

        inf_config = []

        ins_name = lvds_obj.lvds_def
        ins_obj = self._device_db.find_instance(ins_name)

        if ins_obj is not None and ins_obj.get_ref_name() == self._name:
            # Get the pin connection based on lvds mode

            # Check the lvds type
            inf_pins = {}

            if hsio_block.has_mode():
                if lvds_obj.ops_type == lvds_obj.OpsType.op_rx:
                    # Get the pin connection based on lvds mode
                    inf_pins = ins_obj.get_inf_pins_by_mode(
                        HSIOService.MODE_LVDS_RX)

                    if lvds_obj.rx_info is not None:
                        inf_config = self._generate_lvds_rx_interface(
                            lvds_obj, inf_pins)

                elif lvds_obj.ops_type == lvds_obj.OpsType.op_tx:
                    inf_pins = ins_obj.get_inf_pins_by_mode(
                        HSIOService.MODE_LVDS_TX)

                    if lvds_obj.tx_info is not None:
                        inf_config = self._generate_lvds_tx_interface(
                            lvds_obj, inf_pins)

                elif lvds_obj.ops_type == lvds_obj.OpsType.op_bidir:
                    inf_pins = ins_obj.get_inf_pins_by_mode(
                        HSIOService.MODE_LVDS_BIDIR)

                    if lvds_obj.rx_info is not None and lvds_obj.tx_info is not None:
                        inf_config = self._generate_lvds_bidir_interface(
                            lvds_obj, inf_pins)
                    else:
                        msg = 'Instance {} of {} is bidir but its rx_info or tx_info is invalid'.format(
                            lvds_obj.name, self._name)
                        raise dev_excp.ConfigurationInvalidException(
                            msg, app_excp.MsgLevel.error)

                else:
                    msg = 'Instance {} of {} has invalid design configuration'.format(
                        lvds_obj.name, self._name)
                    raise dev_excp.ConfigurationInvalidException(
                        msg, app_excp.MsgLevel.error)

            else:
                msg = 'Instance {} of {} has invalid design configuration'.format(
                    lvds_obj.name, self._name)
                raise dev_excp.ConfigurationInvalidException(
                    msg, app_excp.MsgLevel.error)

        return inf_config

    def _get_sec_alt_conn(self, ins_obj, global_clk, lvds):
        #inf_name = ""
        alt_conn_coord = None
        is_optional = False

        # PT-952: The pin is not associated to the lvds mode but is placed
        # as part of the GPIO mode.
        alt_pin = ins_obj.get_connection_pin("DOUT_EVENP")

        if alt_pin is not None:
            rx_cfg = lvds.rx_info
            conn_type = rx_cfg.conn_type
            conn_type_str = rx_cfg.conn2str_map[conn_type]
            ins_name = lvds.lvds_def
            # Get the secondary conn
            secondary_list = alt_pin.get_secondary_conn()

            for secondary_pin in secondary_list:
                # Check that their connection type matches with the secondary connection
                # Skip if the type does not match
                if secondary_pin.get_function() != conn_type_str.upper():
                    continue

                if secondary_pin.get_type() == \
                        SecondaryConn.SecondaryConnType.external:

                    self.logger.debug(
                        'Check if ins {} has glob'
                        ' clk'.format(ins_name))

                    if global_clk.is_instance_has_global_clock(ins_name) and \
                            global_clk.is_instance_function_exists(ins_name, conn_type_str.upper()):

                        # Check if the type match
                        alt_conn_coord = secondary_pin.get_coord()
                        break

                elif lvds.rx_info.is_alternate_optional_type():
                    # We need to give the secondary conn that has the
                    # coordinate
                    alt_conn_coord = alt_pin.get_coord()
                    is_optional = True

                    self.logger.debug("Setting hsio lvds option to inf ins {}, pin {}".format(
                        ins_name, alt_pin.get_expanded_name()))
                    break

        return alt_conn_coord, is_optional

    def _generate_lvds_rx_interface(self, lvds_obj, inf_pins):
        '''
        Generate interface for an lvds instance
        :return:
        '''
        rx_config = lvds_obj.rx_info

        gen_pin = rx_config.gen_pin
        if gen_pin is None:
            msg = 'Generic pin for LVDS instance {} of {} invalid'.format(
                lvds_obj.name, self._name)
            raise dev_excp.ConfigurationInvalidException(
                msg, app_excp.MsgLevel.error)

        inf_config = []
        ins_name = lvds_obj.lvds_def
        global_clk = self._device_db.get_global_clocks()
        ins_obj = self._device_db.find_instance(ins_name)

        # The pin name is the actual raw port name not the interface name in
        # the mode
        for pin_name, pin_conn in sorted(inf_pins.items()):
            if pin_conn is None:
                msg = 'Unable to find {} connection' \
                      ' in LVDS {}'.format(pin_name, lvds_obj.name)
                raise dev_excp.InstancePinConnectionNotFoundException(
                    msg, app_excp.MsgLevel.warning)

            inf_name = ''
            clk_name = ''
            is_required = False
            is_optional = False

            # Assigned it at the top, so that later if there's
            # a pin that needs a different connection coordinate, it
            # can be overwritten (i.e. alternate)
            primary_conn = pin_conn.get_coord()

            # IN
            if pin_name.startswith("P_OUT") and pin_conn.get_index() is not None and \
                    rx_config.input_bus_name != "":

                if rx_config.conn_type is not None and \
                        rx_config.is_alternate_connection():
                    if pin_conn.get_index() > 0:
                        continue
                    else:
                        inf_name = rx_config.input_bus_name
                        alt_conn_coord, is_optional = self._get_sec_alt_conn(
                            ins_obj, global_clk, lvds_obj)
                        if alt_conn_coord is not None:
                            # Overwrite the pin_conn coordinate to write
                            primary_conn = alt_conn_coord
                        else:
                            continue

                # Prints it to MSB (Unlike Tx) depending on the serialization
                # type
                elif rx_config.conn_type == rx_config.ConnType.normal_conn:
                    # We don't currently support x1. We only have bypass. But with bypass,
                    # it is using a different port (DOUT_EVENP instead of P_OUT)
                    # TODO: If we support x1, then this has to be taken care of
                    if not rx_config.is_serial:
                        if pin_conn.get_index() != 0:
                            continue
                        else:
                            # This is a quick solution to assign to a different port other than
                            # P_OUT. We overwrite the pin_conn and also the primary_conn to
                            # get the coord from DOUT_EVENP
                            bypass_pin = ins_obj.get_connection_pin(
                                "DOUT_EVENP")

                            if bypass_pin is not None:
                                pin_conn = bypass_pin
                                primary_conn = bypass_pin.get_coord()
                                inf_name = rx_config.input_bus_name

                            else:
                                msg = 'Unable to find DOUT_EVENP connection' \
                                      ' in LVDS {}'.format(lvds_obj.name)
                                raise dev_excp.InstancePinConnectionNotFoundException(
                                    msg, app_excp.MsgLevel.warning)
                    else:
                        ser_int = int(rx_config.serial_width)

                        # PT-1059: The bits assigned depends on the width +
                        # half/full rate
                        if rx_config.is_half_rate:
                            rx_map = self._rx_half_width_to_ports_map
                        else:
                            rx_map = self._rx_full_width_to_ports_map

                        # Get the listing from the map corresponding to the
                        # width
                        inf_port_list = []
                        if ser_int in rx_map:
                            inf_port_list = rx_map[ser_int]

                            # Check if the current index is used in the list and then
                            # we need to map the index to the position of the
                            # bits in the list
                            if pin_conn.get_index() in inf_port_list:

                                index_bit = inf_port_list.index(
                                    pin_conn.get_index())

                                # Append the bits
                                inf_name = "{}[{}]".format(
                                    rx_config.input_bus_name, index_bit)

                            else:
                                # Not part of it
                                continue

                        elif ser_int == 2:
                            # PT-1059: We use a different pin with x2 (hardcode for now)
                            # TODO: Figure out how to map it cleanly from
                            # device file later

                            # We use index 0 and 1 as an iter ref
                            if pin_conn.get_index() != 0 and pin_conn.get_index() != 1:
                                continue
                            else:
                                # Unlike GPIO DDIO which has different pin name, in this
                                # case, the user pin name is a bus of 2 bits
                                # wide
                                if pin_conn.get_index() == 0:
                                    ddio_pin = ins_obj.get_connection_pin(
                                        "DOUT_EVENP")
                                else:
                                    ddio_pin = ins_obj.get_connection_pin(
                                        "DOUT_ODDP")

                                if ddio_pin is not None:
                                    # Save the index (0/1) before we change
                                    # pin_conn
                                    pin_index = pin_conn.get_index()
                                    inf_name = "{}[{}]".format(
                                        rx_config.input_bus_name, pin_index)
                                    pin_conn = ddio_pin
                                    primary_conn = ddio_pin.get_coord()

                                else:
                                    msg = 'Unable to find DOUT_EVENP connection' \
                                          ' in LVDS {}'.format(lvds_obj.name)
                                    raise dev_excp.InstancePinConnectionNotFoundException(
                                        msg, app_excp.MsgLevel.warning)

                        elif ser_int == 1:
                            # PT-1112: Support serialization x1 (with a different pin).
                            # same concept as x2 but it only has 1 bit
                            # (DOUT_EVENP)
                            if pin_conn.get_index() != 0:
                                continue
                            else:
                                x1_pin = ins_obj.get_connection_pin(
                                    "DOUT_EVENP")

                                if x1_pin is not None:
                                    pin_index = pin_conn.get_index()
                                    inf_name = rx_config.input_bus_name
                                    pin_conn = x1_pin
                                    primary_conn = x1_pin.get_coord()

                                else:
                                    msg = 'Unable to find DOUT_EVENP connection' \
                                          ' in LVDS {}'.format(lvds_obj.name)
                                    raise dev_excp.InstancePinConnectionNotFoundException(
                                        msg, app_excp.MsgLevel.warning)

                        else:
                            # Skip for unsupported width: i.e x9 (if it will be supported
                            # with is_serial is True)
                            continue

            # INRST
            elif pin_name == "FIFO_RESET" and rx_config.reset_name != "" and rx_config.is_serial:
                # RST is only relevant when serialization is enabled
                inf_name = rx_config.reset_name
                # RST is optional but when use it becomes required
                is_required = True

            # PT-1008 -> PT-978: DPA_LOCK applicable only to dpa delay mode
            elif pin_name == "DPA_LOCK" and rx_config.delay_mode == rx_config.DelayModeType.dpa:
                lock_pin = gen_pin.get_pin_by_type_name("LOCK")
                if lock_pin is not None and lock_pin.name != "":
                    inf_name = lock_pin.name
                else:
                    continue

            # For the clk signal we specify it to the clk name INFASTCLK
            elif pin_name == "RX_SCLKP" and rx_config.is_serial:
                if rx_config.is_serial_width_gt2():
                    if rx_config.fast_clock_name != "":
                        clk_name = rx_config.fast_clock_name
                    else:
                        continue
                else:
                    # For x2, user specify the clock name at the slow (parallel) clock, but the connection is
                    # made to the fast clock (RX_SCLKP)
                    if rx_config.slow_clock_name != "":
                        clk_name = rx_config.slow_clock_name
                    else:
                        continue

            # INSLOWCLK
            elif pin_name == "RX_PCLKP" \
                    and rx_config.slow_clock_name != "":

                # This name is used in 3 cases:
                # 1) serialization > 2
                # 2) delay mode is dynamic regardless of serialization width
                if rx_config.is_serial_width_gt2() or \
                        rx_config.is_delay_mode_non_static():
                    clk_name = rx_config.slow_clock_name
                else:
                    continue

            # The follow are pins based on usage
            elif pin_name == "LVDS_TERMEN" and rx_config.termination == rx_config.TerminationType.dynamic:
                term_pin = gen_pin.get_pin_by_type_name("TERM")
                if term_pin is not None and term_pin.name != "":
                    inf_name = term_pin.name

            # Based on PT-1111, FIFO is only valid when serialization is enabled. Rule
            # will trap, but we just add it here in case rule excluded
            elif pin_name == "FIFO_RD_ENABLE" and rx_config.is_fifo and rx_config.is_serial:
                fifo_pin = gen_pin.get_pin_by_type_name("FIFO_RD")
                if fifo_pin is not None and fifo_pin.name != "":
                    inf_name = fifo_pin.name

            elif pin_name == "FIFO_EMPTY" and rx_config.is_fifo and rx_config.is_serial:
                fifo_pin = gen_pin.get_pin_by_type_name("FIFO_EMPTY")
                if fifo_pin is not None and fifo_pin.name != "":
                    inf_name = fifo_pin.name

            elif pin_name == "RX_SCLKN" and rx_config.is_fifo and rx_config.is_serial:
                fifo_pin = gen_pin.get_pin_by_type_name("FIFOCLK")
                if fifo_pin is not None and fifo_pin.name != "":
                    clk_name = fifo_pin.name

            elif pin_name == "DLY_INC" and rx_config.delay_mode == rx_config.DelayModeType.dynamic:
                dly_pin = gen_pin.get_pin_by_type_name("DLY_INC")
                if dly_pin is not None and dly_pin.name != "":
                    inf_name = dly_pin.name

            # Rule would have trapped DPA usage on ES device. Not checking device type here.
            elif pin_name == "DPA_DLY_ENABLE" and rx_config.is_delay_mode_non_static():
                # Skip if it is DPA but with half rate
                if rx_config.delay_mode == rx_config.DelayModeType.dpa and rx_config.is_serial and\
                    rx_config.is_half_rate:
                    continue

                dly_pin = gen_pin.get_pin_by_type_name("DLY_ENA")
                if dly_pin is not None and dly_pin.name != "":
                    inf_name = dly_pin.name

            # Rule would have trapped DPA usage on ES device. Not checking device type here.
            elif pin_name == "DPA_DLY_RESET" and rx_config.is_delay_mode_non_static():
                # Skip if it is DPA but with half rate
                if rx_config.delay_mode == rx_config.DelayModeType.dpa and rx_config.is_serial and\
                    rx_config.is_half_rate:
                    continue

                dly_pin = gen_pin.get_pin_by_type_name("DLY_RST")
                if dly_pin is not None and dly_pin.name != "":
                    inf_name = dly_pin.name
                else:
                    # The pin is optional and can be empty
                    continue

            # We don't need to check for the device type since ES device does not have the DBG
            # ports
            elif pin_name.startswith("DBG") and rx_config.delay_mode == rx_config.DelayModeType.dpa:
                # Skip if it is DPA but with half rate
                if rx_config.is_serial and rx_config.is_half_rate:
                    continue

                # This is a bus pin
                dly_pin = gen_pin.get_pin_by_type_name("DBG")
                if dly_pin is not None and dly_pin.name != "":
                    pin_index = pin_conn.get_index()
                    inf_name = "{}[{}]".format(
                        dly_pin.name, pin_index)

                else:
                    # The pin is optional and can be empty
                    continue
                
            # PT-1001: This is actually independent of the delay mode
            elif pin_name == "LVDS_IE":
                dly_pin = gen_pin.get_pin_by_type_name("ENA")
                if dly_pin is not None and dly_pin.name != "":
                    inf_name = dly_pin.name
                else:
                    continue

            else:
                # Ignore others (i.e. connection to PAD)
                continue

            # TODO: if there's multiple connection, this has
            # to be changed

            x_loc = primary_conn.get_x()
            y_loc = primary_conn.get_y()
            z_loc = primary_conn.get_z()

            if inf_name != '' or clk_name != '':
                port_config = ucp.UsedCorePort(
                    pin_conn.get_type(), x_loc, y_loc,
                    z_loc, clk_name, False, inf_name, ins_name,
                    lvds_obj.name, self._name,
                    is_required, optional=is_optional)

                inf_config.append(port_config)

            else:
                msg = 'Unexpected undefined LVDS Rx interface pin {} on instance {}'.format(
                    pin_name, lvds_obj.name)

                raise dev_excp.InterfacePinDoesNotExistException(
                    msg, app_excp.MsgLevel.warning)

        return inf_config

    def _generate_lvds_tx_interface(self, lvds_obj, inf_pins):
        '''
        Generate interface for an lvds instance
        :return:
        '''
        tx_config = lvds_obj.tx_info

        inf_config = []
        ins_name = lvds_obj.lvds_def
        ins_obj = self._device_db.find_instance(ins_name)

        # PT-1233: Production device uses a different index than ES
        if self._device_db.is_sample_device():
            even_idx = 9
            odd_idx = 3
        else:
            even_idx = 4
            odd_idx = 3

        # The pin name is the actual raw port name not the interface name in
        # the mode
        for pin_name, pin_conn in sorted(inf_pins.items()):
            if pin_conn is None:
                msg = 'Unable to find {} connection' \
                      ' in LVDS {}'.format(pin_name, lvds_obj.name)
                raise dev_excp.InstancePinConnectionNotFoundException(
                    msg, app_excp.MsgLevel.warning)

            inf_name = ''
            clk_name = ''
            is_required = False
            tied_option = None

            if pin_name.startswith("P_IN") and pin_conn.get_index() is not None:

                if tx_config.mode == tx_config.ModeType.clkout:
                    if tx_config.is_serial:
                        # PT-1135: For clkout mode, the P_IN bits are all tied
                        # except for x2, only bits dependent on device (3,4/9)
                        # are tied
                        bit_int = pin_conn.get_index()
                        ser_int = int(tx_config.serial_width)

                        if not tx_config.is_half_rate:
                            bits_map = self._tx_clkout_full_width_tied_vcc_map
                        else:
                            bits_map = self._tx_clkout_half_width_tied_vcc_map

                        if ser_int in bits_map:
                            bits_list = bits_map[ser_int]
                            if bit_int in bits_list:
                                tied_option = GPIOOutput.TiedOptType.vcc
                            else:
                                tied_option = GPIOOutput.TiedOptType.gnd

                        elif ser_int == 2:
                            # Only writes the 2 bits for x2
                            if pin_conn.get_index() != even_idx and pin_conn.get_index() != odd_idx:
                                continue
                            else:
                                # P_IN[9] - even (1'b1), P_IN[3] - odd (1'b0)
                                if pin_conn.get_index() == odd_idx:
                                    tied_option = GPIOOutput.TiedOptType.gnd
                                else:
                                    tied_option = GPIOOutput.TiedOptType.vcc

                        else:
                            # We don't do anything for serial width that's not
                            # supported (x9) and unsupported width specifically
                            # in clkout mode for x1
                            continue

                        inf_name = GPIOOutput.tiedopt2str_map[tied_option]

                    else:
                        # PT-1661: LVDS Tx clkout serialization disabled doesn't connect to P_IN
                        continue
                else:

                    if tx_config.output_bus_name != "":
                        # With the many serialization support, we assign
                        # the connection based on the type

                        # We don't currently support x1. We only have bypass.
                        # TODO: If we support x1, then this has to be taken
                        # care of

                        if not tx_config.is_serial:
                            # Bypass
                            if pin_conn.get_index() != even_idx:
                                continue
                            else:
                                # bypass is connected to bit 9/4 depending on
                                # device
                                inf_name = tx_config.output_bus_name

                        else:
                            ser_int = int(tx_config.serial_width)

                            # PT-1059: The bits assigned depends on the width +
                            # half/full rate
                            if tx_config.is_half_rate:
                                tx_map = self._tx_half_width_to_ports_map
                            else:
                                tx_map = self._tx_full_width_to_ports_map

                            # Get the listing from the map corresponding to the
                            # width
                            inf_port_list = []
                            if ser_int in tx_map:
                                inf_port_list = tx_map[ser_int]

                                # Check if the current index is used in the list and then
                                # we need to map the index to the position of
                                # the bits in the list
                                if pin_conn.get_index() in inf_port_list:

                                    index_bit = inf_port_list.index(
                                        pin_conn.get_index())

                                    # Append the bits
                                    inf_name = "{}[{}]".format(
                                        tx_config.output_bus_name, index_bit)

                                else:
                                    # Not part of it
                                    continue

                            elif ser_int == 2:

                                if pin_conn.get_index() != even_idx and pin_conn.get_index() != odd_idx:
                                    continue
                                else:
                                    # P_IN[9] - even (0), P_IN[3] - odd (1)
                                    if pin_conn.get_index() == odd_idx:
                                        mapped_index = 1
                                    else:
                                        mapped_index = 0

                                    # Unlike GPIO DDIO which has different pin name, in this
                                    # case, the user pin name is a bus of 2
                                    # bits wide
                                    inf_name = "{}[{}]".format(
                                        tx_config.output_bus_name, mapped_index)

                            elif ser_int == 1:
                                if pin_conn.get_index() != even_idx:
                                    continue
                                else:
                                    inf_name = tx_config.output_bus_name

                            else:
                                # Unused ser width (i.e. 9)
                                continue

            # OUTRST
            elif pin_name == "SER_RESET" and tx_config.reset_name != "" and tx_config.is_serial:
                # PT-1111: Reset is only relevant with serialization enabled
                inf_name = tx_config.reset_name
                is_required = True

            elif pin_name == "TX_SCLKN" and tx_config.is_serial:
                if tx_config.is_serial_width_gt2():
                    if tx_config.fast_clock_name != "":
                        clk_name = tx_config.fast_clock_name
                    else:
                        continue

                elif not tx_config.is_serial_width_gt2() and tx_config.slow_clock_name != "":
                    # PT-1059: We hardcode to use TX_SCLKP with the slow (parallel) clock name
                    # as checked by the rule although the connection is to the pin that is not
                    # listed in the block mode port (TX_SCLKP)
                    # Update pin_conn and thereafter (below), the primary coord will be from
                    # this hardcode pin
                    ddio_clk_pin = ins_obj.get_connection_pin("TX_SCLKP")

                    if ddio_clk_pin is not None:
                        pin_conn = ddio_clk_pin
                        clk_name = tx_config.slow_clock_name
                    else:
                        continue
                else:
                    continue

            elif pin_name == "TX_PCLKN" and tx_config.slow_clock_name != "":
                if tx_config.is_serial_width_gt2() and tx_config.is_serial:
                    # For the clk signal we specify it to the clk name (skip x2)
                    clk_name = tx_config.slow_clock_name

                elif not tx_config.is_serial and tx_config.mode == tx_config.ModeType.clkout:
                    # PT-1661: LVDS Tx clkout mode with serialization disabled
                    clkout_pin = ins_obj.get_connection_pin("TX_SCLKP")
                    if clkout_pin is not None:
                        pin_conn = clkout_pin
                        clk_name = tx_config.slow_clock_name
                    else:
                        continue

                else:
                    continue

                """
                # For the clk signal we specify it to the clk name (skip x2)
                elif pin_name == "TX_PCLKN" and tx_config.slow_clock_name != "" and \
                         tx_config.is_serial_width_gt2() and tx_config.is_serial:
                    clk_name = tx_config.slow_clock_name
    
                elif pin_name == "TX_PCLKN" and tx_config.mode == tx_config.ModeType.clkout and\
                        not tx_config.is_serial:
                    # PT-1661: LVDS Tx clkout mode with serialization disabled
                    clkout_pin = ins_obj.get_connection_pin("TX_SCLKP")
                    if clkout_pin is not None:
                        pin_conn = clkout_pin
                        clk_name = tx_config.slow_clock_name
                    else:
                        continue
                """

            # OE is allowed in any of the modes.
            elif pin_name == "LVDS_OE" and tx_config.oe_name != "":
                inf_name = tx_config.oe_name
                is_required = True

            else:
                # Ignore others (i.e. connection to PAD)
                continue

            # TODO: if there's multiple connection, this has
            # to be changed
            primary_conn = pin_conn.get_coord()

            x_loc = primary_conn.get_x()
            y_loc = primary_conn.get_y()
            z_loc = primary_conn.get_z()

            if inf_name != '' or clk_name != '':
                port_config = ucp.UsedCorePort(
                    pin_conn.get_type(), x_loc, y_loc,
                    z_loc, clk_name, False, inf_name, ins_name,
                    lvds_obj.name, self._name,
                    is_required, tied_option)
                # self.logger.debug("Creating port config {}:{}".format(
                #    inf_name, clk_name))

                inf_config.append(port_config)
            else:
                msg = 'Unexpected undefined LVDS Tx interface pin {} on instance {}'.format(
                    pin_name, lvds_obj.name)
                raise dev_excp.InterfacePinDoesNotExistException(
                    msg, app_excp.MsgLevel.warning)

        return inf_config

    def _generate_lvds_bidir_interface(self, lvds_obj, inf_pins):
        inf_config = []

        # We go through the rx and tx
        if lvds_obj.rx_info is not None:
            rx_config = self._generate_lvds_rx_interface(lvds_obj, inf_pins)
            if rx_config:
                # Add the rx first
                inf_config = rx_config

        if lvds_obj.tx_info is not None:
            tx_config = self._generate_lvds_tx_interface(lvds_obj, inf_pins)
            if tx_config:
                inf_config = inf_config + tx_config

        return inf_config
