'''
Copyright (C) 2017-2021 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Jun 17, 2021

@author: maryam
'''
import os
import sys
from enum import Enum
import re

# 3rd party library to display pretty table
from prettytable import PrettyTable
import xml.etree.ElementTree as et

from typing import Dict

import util.gen_util as pt_util
import util.excp as app_excp

import device.excp as dev_excp

import device.db_interface as device_dbi
from device.block_definition import TimingArc

import design.db as des_db

import common_device.writer as tbi
from common_device.hsio.hsio_device_service import HSIOService
from common_device.hsio.writer.timing import  HSIOTiming
from tx60_device.gpio.gpio_design_comp import GPIOIOStandard

sys.path.append(os.path.join(os.path.dirname(__file__), '..'))


class HSIOGPIOTiming(tbi.BlockTimingWithArcs):
    '''
    Builds the timing data for SPI Flash used for printing out
    timing report and sdc file.
    '''

    class VariableType(Enum):
        '''
        The list of supported timing variables.
        '''
        io_voltage = 0
        hsio_io_std_group = 1
        drive_strength = 2
        slew_rate = 3
        schmitt_trigger = 4
        serial_type = 5
        fifo_enable = 6
        pre_emphasis = 7
        hsio_bank = 8

    var2str_map = \
        {
            VariableType.io_voltage: "io_voltage",
            VariableType.hsio_io_std_group: "hsio_io_std_group",
            VariableType.drive_strength: "drive_strength",
            VariableType.slew_rate: "slew_rate",
            VariableType.schmitt_trigger: "schmitt_trigger",
            VariableType.serial_type: "serial_type",
            VariableType.fifo_enable: "fifo_enable",
            VariableType.pre_emphasis: "pre_emphasis",
            VariableType.hsio_bank: "hsio_bank"
        }

    str2var_map = \
        {
            "io_voltage": VariableType.io_voltage,
            "hsio_io_std_group": VariableType.hsio_io_std_group,
            "drive_strength": VariableType.drive_strength,
            "slew_rate": VariableType.slew_rate,
            "schmitt_trigger": VariableType.schmitt_trigger,
            "serial_type": VariableType.serial_type,
            "fifo_enable": VariableType.fifo_enable,
            "pre_emphasis": VariableType.pre_emphasis,
            "hsio_bank": VariableType.hsio_bank
        }

    class HSIOGPIOPathType(Enum):
        '''
        Indication of the gpio physical path. It does not equate
        directly to the GPIO PadModeType
        '''
        lvttl1_input = 0
        lvttl1_output = 1
        lvttl1_oe = 2
        lvttl1_clkout = 3
        lvttl2_input = 4
        lvttl2_output = 5
        lvttl2_oe = 6
        lvttl2_clkout = 7
        lvttl1_dly_pullup = 8
        lvttl2_dly_pullup = 9

    class HSIOGPIOBlockMode(Enum):
        '''
        Indicates the mode type
        '''
        mode_gpiop = 0
        mode_gpion = 1
        mode_gpio4xp = 2
        mode_gpio4xn = 3

    def __init__(self, bname, device, design, report, sdc,
                 max_model, min_model, ins2iostd_map, ins2bank_names_map,
                 routing_label2bank_tbl_max_map, routing_label2bank_tbl_min_map):
        '''
        Constructor
        '''
        super().__init__(bname, device, design, report, sdc,
                         max_model, min_model)

        # We're building it a bit different since there's no
        # really a mode for this block. The mode is actually
        # to differentiate between input and output arcs.
        # The one with _serialize is associated to mode that has the serializer support
        self._mode2arcs = {}
        self._mode2arcs_serializer = {}

        self._gpio_p = []
        self._gpio_n = []

        # Map of HSIOGPIOBlockMode to the mode object
        self._blkmode2obj_map = {}

        # self.p_blk_mode = None
        # self.n_blk_mode = None

        # Map of instance to io standard
        self._ins2iostd_map = ins2iostd_map

        # Skip writing out the Report table. Instead we store the table
        # to the class
        self.print_header = True

        # Map of alternate gpio instance to the delay value (GPIO_IN: max,min)
        self.ins2_alt_delay_map = {}

        # Map of clkout gpio mode instance to the delay value (GPIO_CLK_OUT: max,min)
        self.ins2_clkout_delay_map = {}

        # Keep the list of instance (in resource map) map
        # to bank name for use later. This is the raw bank name
        # and not the package specific bank name where it could be
        # a merge bank name
        self.ins2bankname_map: Dict[str, str] = ins2bank_names_map

        # The device delay table for the HSIO routing delay
        self.routing_label2bank_tbl_max_map = routing_label2bank_tbl_max_map
        self.routing_label2bank_tbl_min_map = routing_label2bank_tbl_min_map

    def _build_prelim_in_clkout_arc(self, blk_mode, pos_pname, is_p):
        clkout_arcs = blk_mode.get_timing_arc("OUTCLK", pos_pname, False)
        if is_p:
            self._mode2arcs[self.HSIOGPIOPathType.lvttl1_clkout] = clkout_arcs
        else:
            self._mode2arcs[self.HSIOGPIOPathType.lvttl2_clkout] = clkout_arcs

        in_arcs = blk_mode.get_timing_arc(pos_pname, "IN", False)
        if is_p:
            self._mode2arcs[self.HSIOGPIOPathType.lvttl1_input] = in_arcs
        else:
            self._mode2arcs[self.HSIOGPIOPathType.lvttl2_input] = in_arcs

    def build_prelim_arc(self):
        self._mode2arcs = {}
        self._mode2arcs_serializer = {}

        # The LVTTL1 arcs
        self._build_prelim_in_clkout_arc(
            self._blkmode2obj_map[self.HSIOGPIOBlockMode.mode_gpio4xp],
            "P", True)

        self._build_prelim_in_clkout_arc(
            self._blkmode2obj_map[self.HSIOGPIOBlockMode.mode_gpio4xn],
            "N", False)
        
    def build_arc(self):
        """
        Build Timing Arcs
        """
        self._mode2arcs = {}
        self._mode2arcs_serializer = {}

        # At this point, we have already created the _blkmode2obj_map. Need to
        # create the arcs associated to the different modes (only taking 4xp) because
        # in terms of arcs created, they are based on interface name which is common across
        # all modes. The differentiation is the mapped port pointed by the interface.
        # Arc names are based on interface names.
        self.build_arc_by_mode(
            self._blkmode2obj_map[self.HSIOGPIOBlockMode.mode_gpio4xp], "P", True)
        self.build_arc_by_mode(
            self._blkmode2obj_map[self.HSIOGPIOBlockMode.mode_gpio4xn], "N", False)

    def get_block_mode(self, is_p, is_serial):

        if is_p:
            if is_serial:
                blk_mode = self._blkmode2obj_map[self.HSIOGPIOBlockMode.mode_gpio4xp]
            else:
                blk_mode = self._blkmode2obj_map[self.HSIOGPIOBlockMode.mode_gpiop]
        else:
            if is_serial:
                blk_mode = self._blkmode2obj_map[self.HSIOGPIOBlockMode.mode_gpio4xn]
            else:
                blk_mode = self._blkmode2obj_map[self.HSIOGPIOBlockMode.mode_gpion]

        return blk_mode

    def get_arcs_and_blk_mode(self, gpio_obj, path_type, is_p):
        is_serial = False

        if gpio_obj.mode == gpio_obj.PadModeType.input:
            if gpio_obj.input is not None and \
                    gpio_obj.input.is_serial:
                is_serial = True

        elif gpio_obj.mode == gpio_obj.PadModeType.output or \
                gpio_obj.mode == gpio_obj.PadModeType.clkout:

            if gpio_obj.output is not None and \
                    gpio_obj.output.is_serial:
                is_serial = True

        elif gpio_obj.mode == gpio_obj.PadModeType.inout:
            output_config = gpio_obj.output
            input_config = gpio_obj.input

            if (output_config is not None and output_config.is_serial) or \
                    (input_config is not None and input_config.is_serial):
                is_serial = True

        arcs_list = []
        if path_type in self._mode2arcs:
            arcs_list = self._mode2arcs[path_type]

        blk_mode = self.get_block_mode(is_p, is_serial)

        return arcs_list, blk_mode

    def build_arc_by_mode(self, blk_mode, pos_pname, is_p):
        '''
        A gpio mode is associated to a list of arcs.
        Given a mode, the corresponding list of arcs is returned
        which correspond to the block timing_arcs.

        GPIO:
        1) clkout: OUTCLK-PAD
        2) input:
            -nonreg: PAD-IN
            -reg: INCLK-IN, INCLK-PAD
        3) output:
            -nonreg: OUT-PAD
            -reg: OUTCLK-OUT, OUTCLK-PAD
        4) inout
            - input
            - output
            - oe:
                -nonreg: OE-PAD
                -reg: OUTCLK-OE, OUTCLK-PAD
        '''

        # The LVTTL1 arcs
        clkout_arcs = blk_mode.get_timing_arc("OUTCLK", pos_pname, False)
        if is_p:
            self._mode2arcs[self.HSIOGPIOPathType.lvttl1_clkout] = clkout_arcs
        else:
            self._mode2arcs[self.HSIOGPIOPathType.lvttl2_clkout] = clkout_arcs

        in_arcs = blk_mode.get_timing_arc(pos_pname, "IN", False)
        in_arcs += blk_mode.get_timing_arc("INCLK", pos_pname, True)
        in_arcs += blk_mode.get_timing_arc("INCLK", "IN", True)
        if is_p:
            self._mode2arcs[self.HSIOGPIOPathType.lvttl1_input] = in_arcs
        else:
            self._mode2arcs[self.HSIOGPIOPathType.lvttl2_input] = in_arcs

        out_arcs = blk_mode.get_timing_arc("OUT", pos_pname, False)
        out_arcs += blk_mode.get_timing_arc("OUTCLK", pos_pname, True, "OUT")
        out_arcs += blk_mode.get_timing_arc("OUTCLK", "OUT", True)
        if is_p:
            self._mode2arcs[self.HSIOGPIOPathType.lvttl1_output] = out_arcs
        else:
            self._mode2arcs[self.HSIOGPIOPathType.lvttl2_output] = out_arcs

        oe_arcs = blk_mode.get_timing_arc("OE", pos_pname, False)
        oe_arcs += blk_mode.get_timing_arc("OUTCLK", pos_pname, True, "OE")
        oe_arcs += blk_mode.get_timing_arc("OUTCLK", "OE", True)
        if is_p:
            self._mode2arcs[self.HSIOGPIOPathType.lvttl1_oe] = oe_arcs
        else:
            self._mode2arcs[self.HSIOGPIOPathType.lvttl2_oe] = oe_arcs

        dly_pullup = blk_mode.get_timing_arc("INCLK", "DLY_INC", True)
        dly_pullup += blk_mode.get_timing_arc("INCLK", "DLY_ENA", True)
        dly_pullup += blk_mode.get_timing_arc("INCLK", "DLY_RST", True)
        dly_pullup += blk_mode.get_timing_arc("PULL_UP_ENA", pos_pname, False)
        if is_p:
            self._mode2arcs[self.HSIOGPIOPathType.lvttl1_dly_pullup] = dly_pullup
        else:
            self._mode2arcs[self.HSIOGPIOPathType.lvttl2_dly_pullup] = dly_pullup

    def determine_gpio_pin_type(self, hsio_gpio_svc, all_hsio_gpio):
        for ins_obj in all_hsio_gpio:
            if ins_obj is not None and hsio_gpio_svc.is_resource_on_p(ins_obj):
                self._gpio_p.append(ins_obj)
            else:
                self._gpio_n.append(ins_obj)


    def save_source_latency_and_clkout_hsio(self, hsio_block, all_gpio):
        try:
            self.populate_mode_map(hsio_block)

            # Build the arc that is only required for this purpose
            # gpio clk and clkout
            self.build_prelim_arc()

            # Get the device service based on the mipi dphy type
            dbi = device_dbi.DeviceDBService(self._device)

            hsio_gpio_svc = dbi.get_block_service(
                device_dbi.DeviceDBService.BlockType.HSIO_GPIO)

            # classify the gpio instances into p/n instance types
            self.determine_gpio_pin_type(hsio_gpio_svc, all_gpio)

            clkout_gpio, input_gpio, inout_gpio = \
                self._get_gpio_prelim_data(all_gpio)

            if clkout_gpio or input_gpio or inout_gpio:
                # There are 2 reports to write
                self.get_prelim_output(clkout_gpio, 
                                  input_gpio, inout_gpio)
                
        except Exception as excp:
            raise excp

    def get_prelim_output(self, clkout_gpio, 
                        input_gpio, inout_gpio):
        
        # Read out clkout mode
        self.get_gpio_clkout_mode_data(clkout_gpio)

        # Readout input mode
        self.get_gpio_input_alt_mode_data(input_gpio)

        # Read out inout mode
        self.get_gpio_inout_mode_data(inout_gpio)

    def get_gpio_clkout_mode_data(self, gpio_map):

        for ins_name in sorted(gpio_map.keys(),
                                key=pt_util.natural_sort_key_for_list):

            
            gpio_obj = gpio_map[ins_name]

            output_config = gpio_obj.output

            if output_config is None or output_config.clock_name == "":
                continue

            # Check which type it is
            if gpio_obj in self._gpio_p:
                ptype = self.HSIOGPIOPathType.lvttl1_clkout
                arcs_list, blk_mode = self.get_arcs_and_blk_mode(
                    gpio_obj, ptype, True)

            else:
                ptype = self.HSIOGPIOPathType.lvttl2_clkout
                arcs_list, blk_mode = self.get_arcs_and_blk_mode(
                    gpio_obj, ptype, False)

            max_delay = 0
            min_delay = 0
            ref_pin_name = ""
            clkout_coord = None

            max_arc2delay_map, min_arc2delay_map = self._get_timing_delay(
                gpio_obj, ptype)

            dev_ins_obj = self._device.find_instance(gpio_obj.gpio_def)

            for arc in arcs_list:
                sink_name = arc.get_sink()
                source_name = arc.get_source()

                # Get the clkout pin so that we can write it to the sdc file
                # in the clkout category
                _, _, clk_type_name = self.get_reg_pin_and_clk_name(
                    gpio_obj, sink_name, source_name, blk_mode)

                # We're just saving the string to the object for writing later
                # We only want the ref pin name
                if dev_ins_obj is not None:
                    ref_pin_name, _, clkout_coord = self.get_clkout_info_with_coord(
                        gpio_obj.name, dev_ins_obj, clk_type_name, output_config.clock_name)

                # Get the routing delay if necessary
                routing_delay_max = HSIOTiming.identify_routing_delay_on_ins(
                    gpio_obj.gpio_def, TimingArc.TimingArcType.delay,
                    False, self._max_model.get_tscale(), self.ins2bankname_map,
                    self.routing_label2bank_tbl_max_map, is_clkout=True)
                routing_delay_min = HSIOTiming.identify_routing_delay_on_ins(
                    gpio_obj.gpio_def, TimingArc.TimingArcType.delay,
                    False, self._min_model.get_tscale(), self.ins2bankname_map,
                    self.routing_label2bank_tbl_min_map, is_clkout=True)
                
                if arc in max_arc2delay_map:
                    # Get the calculated delay
                    max_delay = self._get_delay_by_arc(
                        max_arc2delay_map, self._max_model.get_tscale(), arc)
                    
                    max_delay = max_delay + routing_delay_max

                if arc in min_arc2delay_map:
                    # Get the calculated delay
                    min_delay = self._get_delay_by_arc(
                        min_arc2delay_map, self._min_model.get_tscale(), arc)

                    min_delay = min_delay + routing_delay_min

            # print("max: {} vs {} clk max: {} result: {}".format(
            #    max_delay_str, str(max_delay_str), max_clk, calc_max))
            # print("min: {} clk min: {} result: {}".format(min_delay_str, min_clk, calc_min))

            assert gpio_obj.name not in self.ins2_clkout_delay_map
            # Save the delay value to be used in other places
            self.ins2_clkout_delay_map[gpio_obj.name] = \
                (output_config.clock_name, ref_pin_name, max_delay, min_delay, clkout_coord)

    
    def get_gpio_input_alt_mode_data(self, gpio_map):

        for ins_name in sorted(gpio_map.keys(),
                                key=pt_util.natural_sort_key_for_list):

            row_list = []

            gpio_obj = gpio_map[ins_name]

            input_config = gpio_obj.input

            if input_config is None:
                continue

            pin_name = input_config.name

            # Check which type it is
            arcs_list = []

            if gpio_obj in self._gpio_p:
                ptype = self.HSIOGPIOPathType.lvttl1_input
                arcs_list, blk_mode = self.get_arcs_and_blk_mode(
                    gpio_obj, ptype, True)

            else:
                ptype = self.HSIOGPIOPathType.lvttl2_input
                arcs_list, blk_mode = self.get_arcs_and_blk_mode(
                    gpio_obj, ptype, False)

            assert input_config.is_alternate_connection()
            
            is_clk = input_config.is_alternate_clock_type()

            # print("INPUT: Max: {} min: {}".format(
            #    max_delay_str, min_delay_str))

            max_delay = 0
            min_delay = 0

            max_arc2delay_map, min_arc2delay_map = self._get_timing_delay(
                gpio_obj, ptype)

            for arc in arcs_list:

                if arc.get_type() == TimingArc.TimingArcType.delay:

                    # Get the routing delay if necessary
                    routing_delay_max = HSIOTiming.identify_routing_delay_on_ins(
                        gpio_obj.gpio_def, TimingArc.TimingArcType.delay,
                        True, self._max_model.get_tscale(), self.ins2bankname_map,
                        self.routing_label2bank_tbl_max_map)
                    routing_delay_min = HSIOTiming.identify_routing_delay_on_ins(
                        gpio_obj.gpio_def, TimingArc.TimingArcType.delay,
                        True, self._min_model.get_tscale(), self.ins2bankname_map,
                        self.routing_label2bank_tbl_min_map)
                    
                    if arc in max_arc2delay_map:
                        # Get the calculated delay
                        max_delay = self._get_delay_by_arc(
                            max_arc2delay_map, self._max_model.get_tscale(), arc)
                        
                        max_delay = max_delay + routing_delay_max

                    if arc in min_arc2delay_map:
                        # Get the calculated delay
                        min_delay = self._get_delay_by_arc(
                            min_arc2delay_map, self._min_model.get_tscale(), arc)
                        
                        min_delay = min_delay + routing_delay_min

            # Save this info to the map:
            assert gpio_obj.name not in self.ins2_alt_delay_map
            self.ins2_alt_delay_map[gpio_obj.name] = \
                (pin_name, max_delay, min_delay, is_clk)

    def get_gpio_inout_mode_data(self, gpio_map):

        for ins_name in sorted(gpio_map.keys(),
                                key=pt_util.natural_sort_key_for_list):

            gpio_obj = gpio_map[ins_name]

            # We do input
            input_config = gpio_obj.input

            if input_config is not None and input_config.name != "":

                pin_name = input_config.name

                # Check which type it is
                arcs_list = []
                if gpio_obj in self._gpio_p:
                    ptype = self.HSIOGPIOPathType.lvttl1_input
                    arcs_list, blk_mode = self.get_arcs_and_blk_mode(
                        gpio_obj, ptype, True)

                else:
                    ptype = self.HSIOGPIOPathType.lvttl2_input
                    arcs_list, blk_mode = self.get_arcs_and_blk_mode(
                        gpio_obj, ptype, False)

                assert input_config.is_alternate_connection()

                is_clk = input_config.is_alternate_clock_type()

                max_delay = 0
                min_delay = 0

                max_arc2delay_map, min_arc2delay_map = self._get_timing_delay(
                    gpio_obj, ptype)

                for arc in arcs_list:

                    if arc.get_type() == TimingArc.TimingArcType.delay:

                        # Get the routing delay if necessary
                        routing_delay_max = HSIOTiming.identify_routing_delay_on_ins(
                            gpio_obj.gpio_def, TimingArc.TimingArcType.delay,
                            True, self._max_model.get_tscale(), self.ins2bankname_map,
                            self.routing_label2bank_tbl_max_map)
                        routing_delay_min = HSIOTiming.identify_routing_delay_on_ins(
                            gpio_obj.gpio_def, TimingArc.TimingArcType.delay,
                            True, self._min_model.get_tscale(), self.ins2bankname_map,
                            self.routing_label2bank_tbl_min_map)
                                                
                        if arc in max_arc2delay_map:
                            # Get the calculated delay
                            max_delay = self._get_delay_by_arc(
                                max_arc2delay_map, self._max_model.get_tscale(), arc)
                            
                            max_delay = max_delay + routing_delay_max

                        if arc in min_arc2delay_map:
                            # Get the calculated delay
                            min_delay = self._get_delay_by_arc(
                                min_arc2delay_map, self._min_model.get_tscale(), arc)
                            
                            min_delay = min_delay + routing_delay_min

                # Save this info to the map:
                assert gpio_obj.name not in self.ins2_alt_delay_map
                self.ins2_alt_delay_map[gpio_obj.name] = \
                    (pin_name, max_delay, min_delay, is_clk)


    def populate_mode_map(self, hsio_block):
        self._blkmode2obj_map[self.HSIOGPIOBlockMode.mode_gpiop] = hsio_block.get_mode(
            HSIOService.MODE_GPIO_P)
        self._blkmode2obj_map[self.HSIOGPIOBlockMode.mode_gpion] = hsio_block.get_mode(
            HSIOService.MODE_GPIO_N)
        self._blkmode2obj_map[self.HSIOGPIOBlockMode.mode_gpio4xp] = hsio_block.get_mode(
            HSIOService.MODE_GPIO_X4P)
        self._blkmode2obj_map[self.HSIOGPIOBlockMode.mode_gpio4xn] = hsio_block.get_mode(
            HSIOService.MODE_GPIO_X4N)

        for blk_mode in self._blkmode2obj_map.values():
            if blk_mode is None:
                raise ValueError("Unable to find hsio gpio block mode")

    def write(self, index, sub_index, hsio_block, all_hsio_gpio, ins_to_block_map):
        '''
        Write out the report and sdc constraint for this block.

        :param index: The index of this section. Used in printing to report
        :param sub_index: The sub index since it all belongs to the same section. Therefore
                it will become <index>.<sub_index>.
        :param all_hsio_gpio: The list of HSIO GPIO design instance objects
        :param ins_to_block_map: Map of device instance name to the block name

        '''
        ins_exists = False

        write_successful = None

        rptfile = None
        sdcfile = None

        # Clear it
        self._gpio_p = []
        self._gpio_n = []

        try:
            write_successful = False

            self.populate_mode_map(hsio_block)

            # Build the arc
            self.build_arc()

            # Get the device service based on the mipi dphy type
            dbi = device_dbi.DeviceDBService(self._device)

            hsio_gpio_svc = dbi.get_block_service(
                device_dbi.DeviceDBService.BlockType.HSIO_GPIO)

            # classify the gpio instances into p/n instance types
            self.determine_gpio_pin_type(hsio_gpio_svc, all_hsio_gpio)

            clkout_gpio, input_gpio, output_gpio, inout_gpio = \
                self._get_gpio_data(all_hsio_gpio)

            # Open the file
            rptfile = open(self._report_file, 'a')
            sdcfile = open(self._sdc_file, 'a')

            if clkout_gpio or input_gpio or output_gpio or inout_gpio:
                if self.print_header:
                    rptfile.write(
                        "\n---------- {}.{} HSIO GPIO Timing Report (begin) ----------\n".format(
                            index, sub_index))

                    sdcfile.write("\n# HSIO GPIO Constraints\n")
                    sdcfile.write("#########################\n")

                # There are 2 reports to write
                self.write_output(rptfile, sdcfile, clkout_gpio, input_gpio,
                                  output_gpio, inout_gpio)

                ins_exists = True

                if self.print_header:
                    rptfile.write(
                        "\n---------- HSIO GPIO Timing Report (end) ----------\n")

            rptfile.close()
            sdcfile.close()

            write_successful = True

        except Exception as excp:
            if write_successful is not None and\
                    not write_successful:
                if rptfile is not None:
                    rptfile.close()
                if sdcfile is not None:
                    sdcfile.close()

            raise excp

        return ins_exists

    def _get_gpio_prelim_data(self, all_hsio_gpio):
        input_gpio = {}
        inout_gpio = {}
        clkout_gpio = {}

        def get_name(ins_obj):
            return ins_obj.name

        for gpio_obj in sorted(all_hsio_gpio, key=get_name):

            if gpio_obj is None:
                continue

            if gpio_obj.mode == gpio_obj.PadModeType.input and \
                    gpio_obj.input is not None:
                if gpio_obj.input.is_alternate_connection():
                    input_gpio[gpio_obj.name] = gpio_obj

            elif gpio_obj.mode == gpio_obj.PadModeType.clkout and \
                    gpio_obj.output is not None:
                clkout_gpio[gpio_obj.name] = gpio_obj

            elif gpio_obj.mode == gpio_obj.PadModeType.inout:
                if gpio_obj.input is not None and \
                    gpio_obj.input.is_alternate_connection():
                    inout_gpio[gpio_obj.name] = gpio_obj

        return clkout_gpio, input_gpio, inout_gpio



    def _get_gpio_data(self, all_hsio_gpio):

        # We already have the list of design object arraged
        # based on their modes
        # Map of device instance name to  gpio_obj stored
        # according to their mode type
        input_gpio = {}
        output_gpio = {}
        inout_gpio = {}
        clkout_gpio = {}

        def get_name(ins_obj):
            return ins_obj.name

        for gpio_obj in sorted(all_hsio_gpio, key=get_name):

            if gpio_obj is None:
                continue

            if gpio_obj.mode == gpio_obj.PadModeType.input and \
                    gpio_obj.input is not None:
                input_gpio[gpio_obj.name] = gpio_obj

            elif gpio_obj.mode == gpio_obj.PadModeType.output and \
                    gpio_obj.output is not None:
                output_gpio[gpio_obj.name] = gpio_obj

            elif gpio_obj.mode == gpio_obj.PadModeType.clkout and \
                    gpio_obj.output is not None:
                clkout_gpio[gpio_obj.name] = gpio_obj

            elif gpio_obj.mode == gpio_obj.PadModeType.inout:
                inout_gpio[gpio_obj.name] = gpio_obj

        return clkout_gpio, input_gpio, output_gpio, inout_gpio

    def write_output(self, rptfile, sdcfile, clkout_gpio,
                     input_gpio, output_gpio, inout_gpio):
        '''
        Writes out gpio related summary and sdc

        '''

        # Print out clkout mode
        self._write_gpio_clkout_mode(rptfile, clkout_gpio)

        # Global table to be printed based on path (not mode)
        if self.print_header:
            global_reg_table = PrettyTable(["Instance Name", "Clock Pin",
                                            "Max Setup (ns)", "Min Setup (ns)",
                                            "Max Hold (ns)", "Min Hold (ns)",
                                            "Max Clock To Out (ns)", "Min Clock To Out (ns)"])

            global_comb_table = PrettyTable(
                ["Instance Name", "Pin Name", "Parameter", "Max (ns)", "Min (ns)"])

        else:
            global_reg_table = PrettyTable(["Clock Pin",
                                            "Max Setup (ns)", "Min Setup (ns)",
                                            "Max Hold (ns)", "Min Hold (ns)",
                                            "Max Clock To Out (ns)",
                                            "Min Clock To Out (ns)"])

            global_comb_table = PrettyTable(
                ["Pin Name", "Parameter", "Max (ns)", "Min (ns)"])

        # Print out input mode first so that the clock goes first
        in_comb, in_reg = self._write_gpio_input_mode(rptfile, sdcfile,
                                                      input_gpio, global_reg_table, global_comb_table)

        # Print out output mode
        out_comb, out_reg = self._write_gpio_output_mode(rptfile, sdcfile,
                                                         output_gpio, global_reg_table, global_comb_table)

        # Print out inout mode
        inout_comb, inout_reg = self._write_gpio_inout_mode(rptfile, sdcfile,
                                                            inout_gpio, global_reg_table, global_comb_table)

        # If they are input, then print out the dynamic delay and pullup
        self._write_input_dyn_delay_pullup(
            sdcfile, input_gpio, inout_gpio, global_comb_table)

        if in_comb or out_comb or inout_comb:
            if self.print_header:
                rptfile.write("\nNon-registered HSIO GPIO Configuration:\n")
                rptfile.write("========================================\n")

            else:
                rptfile.write("\nNon-registered Configuration:\n")
                rptfile.write("==============================\n")

            rptfile.write("\n{}\n".format(global_comb_table.get_string()))

        if in_reg or out_reg or inout_reg:
            # For embedded instances, we won't print the registered constraint
            # since they are not bonded out
            if self.print_header:
                rptfile.write("\nRegistered HSIO GPIO Configuration:\n")
                rptfile.write("====================================\n")

                rptfile.write("\n{}\n".format(global_reg_table.get_string()))

    def _write_input_dyn_delay_pullup(self, sdcfile, input_gpio_map,
                                      inout_gpio_map, global_comb_table):
        '''

        :param sdcfile:
        :param input_gpio_map: Map of input gpio
        :param inout_gpio_map: Map of inout gpio
        :param global_comb_table: To write out the pullup ena
        :return:
        '''

        # Combine the two map. There shouldn't be any duplicates in the map
        combined_map = input_gpio_map
        combined_map.update(inout_gpio_map)

        if combined_map:

            # We print the registered constraint after any
            # create clock

            for ins_name in sorted(combined_map.keys(),
                                   key=pt_util.natural_sort_key_for_list):

                row_list = []

                gpio_obj = combined_map[ins_name]

                input_config = gpio_obj.input

                if input_config is None:
                    continue
                elif not input_config.is_dyn_delay and \
                        input_config.pull_option != input_config.PullOptType.dynamic:
                    # Skif if dyn delay and dynamic pullup is disabled
                    continue

                # Check which type it is
                if gpio_obj in self._gpio_p:
                    ptype = self.HSIOGPIOPathType.lvttl1_dly_pullup
                    arcs_list, blk_mode = self.get_arcs_and_blk_mode(
                        gpio_obj, ptype, True)

                else:
                    ptype = self.HSIOGPIOPathType.lvttl2_dly_pullup
                    arcs_list, blk_mode = self.get_arcs_and_blk_mode(
                        gpio_obj, ptype, False)

                max_delay = 0
                min_delay = 0

                max_arc2delay_map, min_arc2delay_map = self._get_timing_delay(
                    gpio_obj, ptype)

                for arc in arcs_list:
                    sink_name = arc.get_sink()
                    source_name = arc.get_source()

                    if arc.get_type() == TimingArc.TimingArcType.delay and\
                            input_config.pull_option == input_config.PullOptType.dynamic:

                        pin_name = input_config.pullup_ena_name

                        if arc in max_arc2delay_map:
                            # Get the calculated delay
                            max_delay = self._get_delay_by_arc(
                                max_arc2delay_map, self._max_model.get_tscale(), arc)

                        if arc in min_arc2delay_map:
                            # Get the calculated delay
                            min_delay = self._get_delay_by_arc(
                                min_arc2delay_map, self._min_model.get_tscale(), arc)

                        sdcfile.write("# set_output_delay -clock <CLOCK> [-reference_pin <clkout_pad>] -max <MAX CALCULATION> [get_ports {{{}}}]\n".format(
                            pin_name))

                        sdcfile.write("# set_output_delay -clock <CLOCK> [-reference_pin <clkout_pad>] -min <MIN CALCULATION> [get_ports {{{}}}]\n".format(
                            pin_name))

                        # We print the info into Timing Report for comb path
                        if self.print_header:
                            row_list.append(gpio_obj.name)

                        # Get the routing delay if necessary
                        routing_delay_max = HSIOTiming.identify_routing_delay_on_ins(
                            gpio_obj.gpio_def, TimingArc.TimingArcType.delay,
                            False, self._max_model.get_tscale(), self.ins2bankname_map,
                            self.routing_label2bank_tbl_max_map)
                        routing_delay_min = HSIOTiming.identify_routing_delay_on_ins(
                            gpio_obj.gpio_def, TimingArc.TimingArcType.delay,
                            False, self._min_model.get_tscale(), self.ins2bankname_map,
                            self.routing_label2bank_tbl_min_map)
                        
                        row_list.append(pin_name)
                        row_list.append("GPIO_OUT")

                        row_list.append("{0:.3f}".format(max_delay + routing_delay_max))
                        row_list.append("{0:.3f}".format(min_delay + routing_delay_min))

                        global_comb_table.add_row(row_list)

                    elif input_config.is_dyn_delay:
                        # This is dyn delay arcs
                        # print("input tco max: {} min: {}".format(
                        #    tco_max_delay, tco_min_delay))
                        tsu_max_delay = 0
                        thold_min_delay = 0

                        clk_name = input_config.clock_name
                        clk_str = "-clock " + clk_name + " -clock_fall"

                        if sink_name == "DLY_INC":
                            pin_name = input_config.dyn_delay_ctrl_name
                        elif sink_name == "DLY_ENA":
                            pin_name = input_config.dyn_delay_ena_name
                        else:
                            pin_name = input_config.dyn_delay_rst_name
                            clk_str = "-clock " + clk_name
                            
                        if pin_name != "":
                            # Get the clkout pin so that we can write it to the sdc file
                            # in the clkout category. If serial is off, the connection
                            # is made to the DLY_CLK instead of INCLK
                            if not input_config.is_serial:
                                clk_type_name = "RX_PCLKP"
                            else:
                                _, _, clk_type_name = self.get_reg_pin_and_clk_name(
                                    gpio_obj, sink_name, source_name, blk_mode)

                            # We're just saving the string to the object for writing later
                            ref_arg_str = self.set_clkout_ref_pin(
                                gpio_obj, clk_type_name, clk_name, gpio_obj.gpio_def)

                            # Get the routing delay if necessary
                            routing_delay_max = HSIOTiming.identify_routing_delay_on_ins(
                                gpio_obj.gpio_def, TimingArc.TimingArcType.setup,
                                False, self._max_model.get_tscale(), self.ins2bankname_map,
                                self.routing_label2bank_tbl_max_map)
                            routing_delay_min = HSIOTiming.identify_routing_delay_on_ins(
                                gpio_obj.gpio_def, TimingArc.TimingArcType.hold,
                                False, self._min_model.get_tscale(), self.ins2bankname_map,
                                self.routing_label2bank_tbl_min_map)
                            
                            if arc.get_type() == TimingArc.TimingArcType.setup:
                                if arc in max_arc2delay_map:
                                    # Get the calculated delay
                                    tsu_max_delay = self._get_delay_by_arc(
                                        max_arc2delay_map, self._max_model.get_tscale(), arc)

                                    tsu_max_delay = "{0:.3f}".format(
                                        tsu_max_delay + routing_delay_max)

                                    sdcfile.write("set_output_delay {}{} -max {} [get_ports {{{}}}]\n".format(
                                        clk_str, ref_arg_str, tsu_max_delay, pin_name))

                            elif arc.get_type() == TimingArc.TimingArcType.hold:
                                if arc in min_arc2delay_map:
                                    # Get the calculated delay
                                    thold_min_delay = self._get_delay_by_arc(
                                        min_arc2delay_map, self._min_model.get_tscale(), arc)

                                    thold_min_delay = "{0:.3f}".format(
                                        (-1 * thold_min_delay) + routing_delay_min)

                                    sdcfile.write("set_output_delay {}{} -min {} [get_ports {{{}}}]\n".format(
                                        clk_str, ref_arg_str, thold_min_delay, pin_name))

    def _write_gpio_clkout_mode(self, rptfile, gpio_map):
        '''
        Generate the table summary of the Unused GPIO configuration.

        :param rptfile: file handle that has already been opened
        :param clk2delay_map: Map of delay type to the delay value
                        (in float) for core clock network delay
        :param gpio_map: Map of device instance name to GPIOs
                that are set as clkout

        '''

        if gpio_map:

            # Table header
            if self.print_header:
                rptfile.write("\nClkout GPIO Configuration:\n")
                rptfile.write("===========================\n")

                table = PrettyTable(
                    ["Instance Name", "Clock Pin", "Parameter", "Max (ns)", "Min (ns)", "Reference Pin Name"])
            else:
                rptfile.write("\nClkout Configuration:\n")
                rptfile.write("======================\n")
                table = PrettyTable(
                    ["Clock Pin", "Parameter", "Max (ns)", "Min (ns)", "Reference Pin Name"])

            parameter_name = "GPIO_CLK_OUT"

            for ins_name in sorted(gpio_map.keys(),
                                   key=pt_util.natural_sort_key_for_list):

                row_list = []
                gpio_obj = gpio_map[ins_name]

                output_config = gpio_obj.output

                if output_config is None or output_config.clock_name == "":
                    continue

                clk_name = output_config.clock_name
                if output_config.is_clk_inverted:
                    clk_name = "~" + clk_name

                # Check which type it is
                if gpio_obj in self._gpio_p:
                    ptype = self.HSIOGPIOPathType.lvttl1_clkout
                    arcs_list, blk_mode = self.get_arcs_and_blk_mode(
                        gpio_obj, ptype, True)

                else:
                    ptype = self.HSIOGPIOPathType.lvttl2_clkout
                    arcs_list, blk_mode = self.get_arcs_and_blk_mode(
                        gpio_obj, ptype, False)

                max_delay = 0
                min_delay = 0
                ref_pin_name = ""
                clkout_coord = None

                max_arc2delay_map, min_arc2delay_map = self._get_timing_delay(
                    gpio_obj, ptype)

                for arc in arcs_list:
                    sink_name = arc.get_sink()
                    source_name = arc.get_source()

                    # Get the clkout pin so that we can write it to the sdc file
                    # in the clkout category
                    _, _, clk_type_name = self.get_reg_pin_and_clk_name(
                        gpio_obj, sink_name, source_name, blk_mode)

                    # We're just saving the string to the object for writing later
                    # We only want the ref pin name
                    ref_pin_name = self.set_clkout_ref_pin(
                        gpio_obj, clk_type_name, output_config.clock_name, gpio_obj.gpio_def, True)

                    if arc in max_arc2delay_map:
                        # Get the calculated delay
                        max_delay = self._get_delay_by_arc(
                            max_arc2delay_map, self._max_model.get_tscale(), arc)

                    if arc in min_arc2delay_map:
                        # Get the calculated delay
                        min_delay = self._get_delay_by_arc(
                            min_arc2delay_map, self._min_model.get_tscale(), arc)

                # print("max: {} vs {} clk max: {} result: {}".format(
                #    max_delay_str, str(max_delay_str), max_clk, calc_max))
                # print("min: {} clk min: {} result: {}".format(min_delay_str, min_clk, calc_min))

                # We print the Instance name and Clock name
                if self.print_header:
                    row_list.append(gpio_obj.name)

                # Get the routing delay if necessary
                routing_delay_max = HSIOTiming.identify_routing_delay_on_ins(
                    gpio_obj.gpio_def, TimingArc.TimingArcType.delay,
                    False, self._max_model.get_tscale(), self.ins2bankname_map,
                    self.routing_label2bank_tbl_max_map, is_clkout=True)
                routing_delay_min = HSIOTiming.identify_routing_delay_on_ins(
                    gpio_obj.gpio_def, TimingArc.TimingArcType.delay,
                    False, self._min_model.get_tscale(), self.ins2bankname_map,
                    self.routing_label2bank_tbl_min_map, is_clkout=True)
                
                row_list.append(clk_name)
                row_list.append(parameter_name)
                row_list.append("{0:.3f}".format(max_delay + routing_delay_max))
                row_list.append("{0:.3f}".format(min_delay + routing_delay_min))

                row_list.append(ref_pin_name)

                table.add_row(row_list)

            rptfile.write("\n{}\n".format(table.get_string()))

    def _write_gpio_input_mode(self, rptfile, sdcfile,
                               gpio_map, global_reg_table, global_comb_table):
        '''
        Generate the table summary of the Unused GPIO configuration.

        :param sdcfile: sdc file handle that has already been opened
        :param clk2delay_map: Map of delay type to the delay value
                        (in float) for core clock network delay
        :param gpio_map: Map of device instance name to GPIOs
                that are set as input

        :return tuple in_comb, in_reg:
                in_comb: True if there are combinational path to print
                in_reg: True if there are registered path to print
        '''
        in_comb = False
        in_reg = False

        if gpio_map:

            # We print the registered constraint after any
            # create clock

            for ins_name in sorted(gpio_map.keys(),
                                   key=pt_util.natural_sort_key_for_list):

                row_list = []

                gpio_obj = gpio_map[ins_name]

                input_config = gpio_obj.input

                if input_config is None:
                    continue

                pin_name = input_config.name

                # Check which type it is
                arcs_list = []

                if gpio_obj in self._gpio_p:
                    ptype = self.HSIOGPIOPathType.lvttl1_input
                    arcs_list, blk_mode = self.get_arcs_and_blk_mode(
                        gpio_obj, ptype, True)

                else:
                    ptype = self.HSIOGPIOPathType.lvttl2_input
                    arcs_list, blk_mode = self.get_arcs_and_blk_mode(
                        gpio_obj, ptype, False)

                if not input_config.is_register:
                    # Non-registered input
                    # If it was a global clock type, then it would be
                    # a create_clock constraint

                    # Check connection type and secondary function
                    is_gclk = False
                    if input_config.conn_type == input_config.ConnType.gclk_conn or \
                            input_config.conn_type == input_config.ConnType.rclk_conn:
                        # Unlike the Trion GPIO, we ignore checking the sec connection
                        # and assume rule has already validate it. So, we only accept
                        # gclk and rclk
                        is_gclk = True

                    # print("INPUT: Max: {} min: {}".format(
                    #    max_delay_str, min_delay_str))

                    max_delay = 0
                    min_delay = 0

                    max_arc2delay_map, min_arc2delay_map = self._get_timing_delay(
                        gpio_obj, ptype)

                    for arc in arcs_list:

                        if arc.get_type() == TimingArc.TimingArcType.delay:

                            if arc in max_arc2delay_map:
                                # Get the calculated delay
                                max_delay = self._get_delay_by_arc(
                                    max_arc2delay_map, self._max_model.get_tscale(), arc)

                            if arc in min_arc2delay_map:
                                # Get the calculated delay
                                min_delay = self._get_delay_by_arc(
                                    min_arc2delay_map, self._min_model.get_tscale(), arc)

                    in_comb = True

                    routing_delay_max = HSIOTiming.identify_routing_delay_on_ins(
                        gpio_obj.gpio_def, TimingArc.TimingArcType.delay,
                        True, self._max_model.get_tscale(), self.ins2bankname_map,
                        self.routing_label2bank_tbl_max_map)
                    routing_delay_min = HSIOTiming.identify_routing_delay_on_ins(
                        gpio_obj.gpio_def, TimingArc.TimingArcType.delay,
                        True, self._min_model.get_tscale(), self.ins2bankname_map,
                        self.routing_label2bank_tbl_min_map)                    

                    if is_gclk:

                        sdcfile.write("create_clock -period <USER_PERIOD> -name {} [get_ports {{{}}}]\n".format(
                            pin_name, pin_name))

                        if self.print_header:
                            row_list.append(gpio_obj.name)

                        row_list.append(pin_name)
                        row_list.append("GPIO_CLK_IN")

                        row_list.append("{0:.3f}".format(max_delay + routing_delay_max))
                        row_list.append("{0:.3f}".format(min_delay + routing_delay_min))

                        global_comb_table.add_row(row_list)

                    else:

                        # PT-331: Add the non-registered constraint as a template and comment it out
                        # We just give a string representation that the user shall use
                        # to refer to the user guide (no scenario specific equations in the
                        # generated template
                        if not input_config.is_alternate_connection() or\
                            input_config.conn_type == input_config.ConnType.gctrl_conn  or\
                            input_config.conn_type == input_config.ConnType.rctrl_conn:
                            sdcfile.write("# set_input_delay -clock <CLOCK> [-reference_pin <clkout_pad>] -max <MAX CALCULATION> [get_ports {{{}}}]\n".format(
                                pin_name))

                            sdcfile.write("# set_input_delay -clock <CLOCK> [-reference_pin <clkout_pad>] -min <MIN CALCULATION> [get_ports {{{}}}]\n".format(
                                pin_name))

                        # We print the info into Timing Report for comb path
                        if self.print_header:
                            row_list.append(gpio_obj.name)

                        row_list.append(pin_name)
                        row_list.append("GPIO_IN")

                        row_list.append("{0:.3f}".format(max_delay + routing_delay_max))
                        row_list.append("{0:.3f}".format(min_delay + routing_delay_min))

                        global_comb_table.add_row(row_list)

                else:

                    clk_name = input_config.clock_name
                    clk_str = "-clock " + clk_name

                    if input_config.is_clk_inverted:
                        clk_name = "~" + clk_name
                        clk_str = "-clock_fall " + clk_str

                    # Check if it was configured in ddio mode
                    ddio_normal = False
                    
                    # Check if it was configured in ddio mode
                    if input_config.ddio_type is not None and\
                            input_config.ddio_type != input_config.DDIOType.none:

                        if input_config.ddio_type == input_config.DDIOType.normal:
                            ddio_normal = True
                            pin_name = input_config.name
                        else:
                            # pin_name is now a list for the SDC constraint
                            pin_name = "{} {}".format(
                                input_config.name_ddio_lo, input_config.name)

                    # Check if serializer is enabled
                    elif input_config.is_serial:
                        pin_name = "{}[*]".format(pin_name)

                    # print("input tco max: {} min: {}".format(
                    #    tco_max_delay, tco_min_delay))
                    tco_arc = None
                    tsu_max_delay = 0
                    tsu_min_delay = 0
                    thold_max_delay = 0
                    thold_min_delay = 0
                    tco_max_delay = 0
                    tco_min_delay = 0
                    ref_arg_str = ""

                    max_arc2delay_map, min_arc2delay_map = self._get_timing_delay(
                        gpio_obj, ptype)

                    for arc in arcs_list:

                        sink_name = arc.get_sink()
                        source_name = arc.get_source()

                        if arc.get_type() == TimingArc.TimingArcType.setup:
                            if arc in max_arc2delay_map:
                                # Get the calculated delay
                                tsu_max_delay = self._get_delay_by_arc(
                                    max_arc2delay_map, self._max_model.get_tscale(), arc)

                            if arc in min_arc2delay_map:
                                tsu_min_delay = self._get_delay_by_arc(
                                    min_arc2delay_map, self._min_model.get_tscale(), arc)

                        elif arc.get_type() == TimingArc.TimingArcType.hold:
                            if arc in max_arc2delay_map:
                                thold_max_delay = self._get_delay_by_arc(
                                    max_arc2delay_map, self._max_model.get_tscale(), arc)

                            if arc in min_arc2delay_map:
                                # Get the calculated delay
                                thold_min_delay = self._get_delay_by_arc(
                                    min_arc2delay_map, self._min_model.get_tscale(), arc)

                        elif arc.get_type() == TimingArc.TimingArcType.clk_to_q:
                            # Get the clkout pin so that we can write it to the sdc file
                            # in the clkout category
                            _, _, clk_type_name = self.get_reg_pin_and_clk_name(
                                gpio_obj, sink_name, source_name, blk_mode)

                            # We're just saving the string to the object for writing later
                            ref_arg_str = self.set_clkout_ref_pin(
                                gpio_obj, clk_type_name, input_config.clock_name, gpio_obj.gpio_def)

                            if arc in min_arc2delay_map:
                                # Get the calculated delay
                                tco_min_delay = self._get_delay_by_arc(
                                    min_arc2delay_map, self._min_model.get_tscale(), arc)

                            if arc in max_arc2delay_map:
                                # Get the calculated delay
                                tco_max_delay = self._get_delay_by_arc(
                                    max_arc2delay_map, self._max_model.get_tscale(), arc)

                            tco_arc = arc

                    routing_delay_max = HSIOTiming.identify_routing_delay_on_ins(
                        gpio_obj.gpio_def, TimingArc.TimingArcType.clk_to_q,
                        True, self._max_model.get_tscale(), self.ins2bankname_map,
                        self.routing_label2bank_tbl_max_map)
                    routing_delay_min = HSIOTiming.identify_routing_delay_on_ins(
                        gpio_obj.gpio_def, TimingArc.TimingArcType.clk_to_q,
                        True, self._min_model.get_tscale(), self.ins2bankname_map,
                        self.routing_label2bank_tbl_min_map) 
                    
                    if tco_arc is not None:
                        tco_max_delay = "{0:.3f}".format(tco_max_delay + routing_delay_max)
                        tco_min_delay = "{0:.3f}".format(tco_min_delay + routing_delay_min)

                    # If it is using DDIO feature, then we need
                    # to write out the DDIO pin
                    # if input_config.ddio_type is not None:
                    sdcfile.write("set_input_delay {}{} -max {} [get_ports {{{}}}]\n".format(
                        clk_str, ref_arg_str, tco_max_delay, pin_name))
                    sdcfile.write("set_input_delay {}{} -min {} [get_ports {{{}}}]\n".format(
                        clk_str, ref_arg_str, tco_min_delay, pin_name))

                    if ddio_normal:
                        ddio_clk_str = "-clock " + input_config.clock_name

                        if not input_config.is_clk_inverted:
                            ddio_clk_str = "-clock_fall " + ddio_clk_str

                        ddio_pin_name = input_config.name_ddio_lo

                        sdcfile.write("set_input_delay {}{} -max {} [get_ports {{{}}}]\n".format(
                            ddio_clk_str, ref_arg_str, tco_max_delay, ddio_pin_name))
                        sdcfile.write("set_input_delay {}{} -min {} [get_ports {{{}}}]\n".format(
                            ddio_clk_str, ref_arg_str, tco_min_delay, ddio_pin_name))

                    tmp_list = []
                    if self.print_header:
                        tmp_list.append(gpio_obj.name)

                    tmp_list.append(clk_name)
                    tmp_list.append("{0:.3f}".format(tsu_max_delay))
                    tmp_list.append("{0:.3f}".format(tsu_min_delay))
                    tmp_list.append("{0:.3f}".format(thold_max_delay))
                    tmp_list.append("{0:.3f}".format(thold_min_delay))
                    tmp_list.append("")
                    tmp_list.append("")

                    if self.print_header:
                        # Skip if it was embedded instance where print_header
                        # is set to False
                        global_reg_table.add_row(tmp_list)

                    in_reg = True

        return in_comb, in_reg

    def _write_gpio_output_mode(self, rptfile, sdcfile,
                                gpio_map, global_reg_table, global_comb_table):
        '''
        Generate the table summary of the Unused GPIO configuration.

        :param gpio_map: Map of device instance name to GPIOs
                that are set as output

        :return tuple out_comb, out_reg:
                out_comb: True if there are combinational path to print
                out_reg: True if there are registered path to print

        '''
        out_comb = False
        out_reg = False

        if gpio_map:

            for ins_name in sorted(gpio_map.keys(),
                                   key=pt_util.natural_sort_key_for_list):

                row_list = []
                out_tied = False

                gpio_obj = gpio_map[ins_name]

                output_config = gpio_obj.output

                if output_config is None:
                    continue

                # If the output is tied, then there's no constraint to
                # print in sdc file
                if output_config.tied_option == output_config.TiedOptType.vcc or\
                        output_config.tied_option == output_config.TiedOptType.gnd:
                    out_tied = True

                # Check which type it is
                arcs_list = []
                if gpio_obj in self._gpio_p:
                    ptype = self.HSIOGPIOPathType.lvttl1_output
                    arcs_list, blk_mode = self.get_arcs_and_blk_mode(
                        gpio_obj, ptype, True)

                else:
                    ptype = self.HSIOGPIOPathType.lvttl2_output
                    arcs_list, blk_mode = self.get_arcs_and_blk_mode(
                        gpio_obj, ptype, False)

                pin_name = output_config.name

                if output_config.register_option == output_config.RegOptType.none:

                    max_delay = 0
                    min_delay = 0

                    max_arc2delay_map, min_arc2delay_map = self._get_timing_delay(
                        gpio_obj, ptype)

                    for arc in arcs_list:

                        if arc.get_type() == TimingArc.TimingArcType.delay:
                            if arc in max_arc2delay_map:
                                # Get the calculated delay
                                max_delay = self._get_delay_by_arc(
                                    max_arc2delay_map, self._max_model.get_tscale(), arc)

                            if arc in min_arc2delay_map:
                                # Get the calculated delay
                                min_delay = self._get_delay_by_arc(
                                    min_arc2delay_map, self._min_model.get_tscale(), arc)

                    if self.print_header:
                        row_list.append(gpio_obj.name)

                    routing_delay_max = HSIOTiming.identify_routing_delay_on_ins(
                        gpio_obj.gpio_def, TimingArc.TimingArcType.delay,
                        False, self._max_model.get_tscale(), self.ins2bankname_map,
                        self.routing_label2bank_tbl_max_map)
                    routing_delay_min = HSIOTiming.identify_routing_delay_on_ins(
                        gpio_obj.gpio_def, TimingArc.TimingArcType.delay,
                        False, self._min_model.get_tscale(), self.ins2bankname_map,
                        self.routing_label2bank_tbl_min_map) 
                    
                    row_list.append(pin_name)
                    row_list.append("GPIO_OUT")

                    row_list.append("{0:.3f}".format(max_delay + routing_delay_max))
                    row_list.append("{0:.3f}".format(min_delay + routing_delay_min))

                    global_comb_table.add_row(row_list)
                    out_comb = True

                    # If it was tied, we don't print constraint but we just give
                    # the delay value in report
                    if not out_tied:
                        # PT-331: Add the non-registered constraint as a template and comment it out
                        # We just give a string representation that the user shall use
                        # to refer to the user guide (no scenario specific equations in the
                        # generated template
                        sdcfile.write("# set_output_delay -clock <CLOCK> [-reference_pin <clkout_pad>] -max <MAX CALCULATION> [get_ports {{{}}}]\n".format(
                            pin_name))
                        sdcfile.write("# set_output_delay -clock <CLOCK> [-reference_pin <clkout_pad>] -min <MIN CALCULATION> [get_ports {{{}}}]\n".format(
                            pin_name))

                else:

                    clk_name = output_config.clock_name
                    clk_str = "-clock " + clk_name
                    ddio_normal = False

                    if output_config.is_clk_inverted:
                        clk_name = "~" + clk_name
                        clk_str = "-clock_fall " + clk_str

                    # Check if it was configured in ddio mode
                    if output_config.ddio_type is not None and\
                            output_config.ddio_type != output_config.DDIOType.none:
                        
                        if output_config.ddio_type == output_config.DDIOType.normal:
                            ddio_normal = True
                            pin_name = output_config.name
                        else:                        
                            # pin_name is now a list for the SDC constraint
                            pin_name = "{} {}".format(
                                output_config.name_ddio_lo, output_config.name)

                    elif output_config.is_serial:
                        pin_name = "{}[*]".format(pin_name)

                    # sdc template
                    tsu_max_delay = 0
                    thold_min_delay = 0
                    tco_max_delay = 0
                    tco_min_delay = 0
                    ref_arg_str = ""

                    max_arc2delay_map, min_arc2delay_map = self._get_timing_delay(
                        gpio_obj, ptype)

                    for arc in arcs_list:
                        sink_name = arc.get_sink()
                        source_name = arc.get_source()

                        if arc.get_type() == TimingArc.TimingArcType.setup:
                            # Get the clkout pin so that we can write it to the sdc file
                            # in the clkout category. Just use setup instead of both setup and
                            # hold.
                            _, _, clk_type_name = self.get_reg_pin_and_clk_name(
                                gpio_obj, sink_name, source_name, blk_mode)

                            # We're just saving the string to the object for writing later
                            ref_arg_str = self.set_clkout_ref_pin(
                                gpio_obj, clk_type_name, output_config.clock_name, gpio_obj.gpio_def)

                            if arc in max_arc2delay_map:
                                # Get the calculated delay
                                tsu_max_delay = self._get_delay_by_arc(
                                    max_arc2delay_map, self._max_model.get_tscale(), arc)

                        elif arc.get_type() == TimingArc.TimingArcType.hold:

                            if arc in min_arc2delay_map:
                                # Get the calculated delay
                                thold_min_delay = self._get_delay_by_arc(
                                    min_arc2delay_map, self._min_model.get_tscale(), arc)

                        elif arc.get_type() == TimingArc.TimingArcType.clk_to_q:

                            if arc in min_arc2delay_map:
                                # Get the calculated delay
                                tco_min_delay = self._get_delay_by_arc(
                                    min_arc2delay_map, self._min_model.get_tscale(), arc)

                            if arc in max_arc2delay_map:
                                # Get the calculated delay
                                tco_max_delay = self._get_delay_by_arc(
                                    max_arc2delay_map, self._max_model.get_tscale(), arc)

                    # Get the routing delay if necessary
                    routing_delay_max = HSIOTiming.identify_routing_delay_on_ins(
                        gpio_obj.gpio_def, TimingArc.TimingArcType.setup,
                        False, self._max_model.get_tscale(), self.ins2bankname_map,
                        self.routing_label2bank_tbl_max_map)
                    routing_delay_min = HSIOTiming.identify_routing_delay_on_ins(
                        gpio_obj.gpio_def, TimingArc.TimingArcType.hold,
                        False, self._min_model.get_tscale(), self.ins2bankname_map,
                        self.routing_label2bank_tbl_min_map)
                    
                    # -max: (GPIO_OUT_PAD_TSU) - CORE CLK (max)
                    max_out_delay = "{0:.3f}".format(tsu_max_delay + routing_delay_max)

                    # -min: (GPIO_OUT_PAD_THOLD) + CORE CLK (min)
                    min_out_delay = "{0:.3f}".format((-1 * thold_min_delay) + routing_delay_min)

                    # print("MAX: {} MIN: {}".format(max_out_delay, min_out_delay))
                    if not out_tied:
                        sdcfile.write("set_output_delay {}{} -max {} [get_ports {{{}}}]\n".format(
                            clk_str, ref_arg_str, max_out_delay, pin_name))
                        sdcfile.write("set_output_delay {}{} -min {} [get_ports {{{}}}]\n".format(
                            clk_str, ref_arg_str, min_out_delay, pin_name))

                        if ddio_normal:
                            ddio_clk_str = "-clock " + output_config.clock_name

                            if not output_config.is_clk_inverted:
                                ddio_clk_str = "-clock_fall " + ddio_clk_str

                            ddio_pin_name = output_config.name_ddio_lo
                            
                            sdcfile.write("set_output_delay {}{} -max {} [get_ports {{{}}}]\n".format(
                                ddio_clk_str, ref_arg_str, max_out_delay, ddio_pin_name))
                            sdcfile.write("set_output_delay {}{} -min {} [get_ports {{{}}}]\n".format(
                                ddio_clk_str, ref_arg_str, min_out_delay, ddio_pin_name))

                    # Report file

                    # print("output Max: {}, min: {}".format(tco_max_delay, tco_min_delay))
                    tmp_list = []
                    if self.print_header:
                        tmp_list.append(gpio_obj.name)

                    tmp_list.append(clk_name)
                    tmp_list.append("")  # Setup Max
                    tmp_list.append("")  # Setup Min
                    tmp_list.append("")  # Hold Max
                    tmp_list.append("")  # Hold Min
                    tmp_list.append("{0:.3f}".format(tco_max_delay))
                    tmp_list.append("{0:.3f}".format(tco_min_delay))

                    if self.print_header:
                        global_reg_table.add_row(tmp_list)

                    out_reg = True

        return out_comb, out_reg

    def _write_gpio_inout_mode(self, rptfile, sdcfile,
                               gpio_map, global_reg_table, global_comb_table):
        '''
        Generate the table summary of the Unused GPIO configuration.

        :param gpio_map: Map of device instance name to GPIOs
                that are set as inout

        :return tuple inout_comb, inout_reg:
                inout_comb: True if there are combinational path to print
                inout_reg: True if there are registered path to print
        '''
        inout_comb = False
        inout_reg = False

        if gpio_map:

            for ins_name in sorted(gpio_map.keys(),
                                   key=pt_util.natural_sort_key_for_list):

                is_ins_reg = False

                row_list = []
                gpio_obj = gpio_map[ins_name]

                # Default settings for cases where the path does not apply
                in_clk_name = ""
                out_clk_name = ""

                in_tsu = 0
                in_min_tsu = 0
                in_thold = 0
                in_max_thold = 0
                out_tco_min = 0
                out_tco_max = 0
                oe_tco_min = 0
                oe_tco_max = 0

                # We do input
                input_config = gpio_obj.input

                if input_config is not None and input_config.name != "":

                    pin_name = input_config.name

                    # Check which type it is
                    arcs_list = []
                    if gpio_obj in self._gpio_p:
                        ptype = self.HSIOGPIOPathType.lvttl1_input
                        arcs_list, blk_mode = self.get_arcs_and_blk_mode(
                            gpio_obj, ptype, True)

                    else:
                        ptype = self.HSIOGPIOPathType.lvttl2_input
                        arcs_list, blk_mode = self.get_arcs_and_blk_mode(
                            gpio_obj, ptype, False)

                    if not input_config.is_register:

                        # Check connection type and secondary function
                        is_gclk = False

                        max_delay = 0
                        min_delay = 0

                        max_arc2delay_map, min_arc2delay_map = self._get_timing_delay(
                            gpio_obj, ptype)

                        for arc in arcs_list:

                            if arc.get_type() == TimingArc.TimingArcType.delay:
                                if arc in max_arc2delay_map:
                                    # Get the calculated delay
                                    max_delay = self._get_delay_by_arc(
                                        max_arc2delay_map, self._max_model.get_tscale(), arc)

                                if arc in min_arc2delay_map:
                                    # Get the calculated delay
                                    min_delay = self._get_delay_by_arc(
                                        min_arc2delay_map, self._min_model.get_tscale(), arc)

                        # Get the routing delay if necessary
                        routing_delay_max = HSIOTiming.identify_routing_delay_on_ins(
                            gpio_obj.gpio_def, TimingArc.TimingArcType.delay,
                            True, self._max_model.get_tscale(), self.ins2bankname_map,
                            self.routing_label2bank_tbl_max_map)
                        routing_delay_min = HSIOTiming.identify_routing_delay_on_ins(
                            gpio_obj.gpio_def, TimingArc.TimingArcType.delay,
                            True, self._min_model.get_tscale(), self.ins2bankname_map,
                            self.routing_label2bank_tbl_min_map) 
                        
                        if input_config.conn_type == input_config.ConnType.gclk_conn or \
                                input_config.conn_type == input_config.ConnType.rclk_conn:
                            # Unlike the Trion GPIO, we ignore checking the sec connection
                            # and assume rule has already validate it. So, we only accept
                            # gclk and rclk

                            sdcfile.write("create_clock -period <USER_PERIOD> -name {} [get_ports {{{}}}]\n".format(
                                pin_name, pin_name))

                            row_list = []
                            if self.print_header:
                                row_list.append(gpio_obj.name)
                        
                            row_list.append(pin_name)
                            row_list.append("GPIO_CLK_IN")

                            row_list.append("{0:.3f}".format(max_delay + routing_delay_max))
                            row_list.append("{0:.3f}".format(min_delay + routing_delay_min))

                            global_comb_table.add_row(row_list)

                        elif input_config.name != "":
                            # Normal combination path
                            # PT-331: Add the non-registered constraint as a template and comment it out
                            if not input_config.is_alternate_connection() or\
                                input_config.conn_type == input_config.ConnType.gctrl_conn or\
                                input_config.conn_type == input_config.ConnType.rctrl_conn:
                                sdcfile.write("# set_input_delay -clock <CLOCK> [-reference_pin <clkout_pad>] -max <MAX CALCULATION> [get_ports {{{}}}]\n".format(
                                    pin_name))
                                sdcfile.write("# set_input_delay -clock <CLOCK> [-reference_pin <clkout_pad>] -min <MIN CALCULATION> [get_ports {{{}}}]\n".format(
                                    pin_name))

                            # We print the info into Timing Report for comb path
                            comb_row_list = []
                            if self.print_header:
                                comb_row_list.append(gpio_obj.name)

                            comb_row_list.append(pin_name)
                            comb_row_list.append("GPIO_IN")

                            comb_row_list.append("{0:.3f}".format(max_delay + routing_delay_max))
                            comb_row_list.append("{0:.3f}".format(min_delay + routing_delay_min))

                            global_comb_table.add_row(comb_row_list)
                            inout_comb = True

                    else:

                        is_ins_reg = True
                        ddio_normal = False

                        in_clk_name = input_config.clock_name
                        clk_str = "-clock " + in_clk_name

                        if input_config.is_clk_inverted:
                            in_clk_name = "~" + in_clk_name
                            clk_str = "-clock_fall " + clk_str

                        # Check if it was configured in ddio mode
                        if input_config.ddio_type is not None and\
                                input_config.ddio_type != input_config.DDIOType.none:
                            
                            if input_config.ddio_type == input_config.DDIOType.normal:
                                ddio_normal = True
                                pin_name = input_config.name
                            else:                            
                                # pin_name is now a list for the SDC constraint
                                pin_name = "{} {}".format(
                                    input_config.name_ddio_lo, input_config.name)
                                
                        elif input_config.is_serial:
                            pin_name = "{}[*]".format(input_config.name)

                        # For sdc template

                        tco_arc = None
                        tsu_max_delay = 0
                        tsu_min_delay = 0
                        thold_min_delay = 0
                        thold_max_delay = 0
                        tco_max_delay = 0
                        tco_min_delay = 0
                        ref_arg_str = ""

                        max_arc2delay_map, min_arc2delay_map = self._get_timing_delay(
                            gpio_obj, ptype)

                        for arc in arcs_list:
                            source_name = arc.get_source()
                            sink_name = arc.get_sink()

                            if arc.get_type() == TimingArc.TimingArcType.setup:

                                if arc in max_arc2delay_map:
                                    # Get the calculated delay
                                    tsu_max_delay = self._get_delay_by_arc(
                                        max_arc2delay_map, self._max_model.get_tscale(), arc)

                                if arc in min_arc2delay_map:
                                    tsu_min_delay = self._get_delay_by_arc(
                                        min_arc2delay_map, self._min_model.get_tscale(), arc)

                            elif arc.get_type() == TimingArc.TimingArcType.hold:

                                if arc in max_arc2delay_map:
                                    thold_max_delay = self._get_delay_by_arc(
                                        max_arc2delay_map, self._max_model.get_tscale(), arc)

                                if arc in min_arc2delay_map:
                                    # Get the calculated delay
                                    thold_min_delay = self._get_delay_by_arc(
                                        min_arc2delay_map, self._min_model.get_tscale(), arc)

                            elif arc.get_type() == TimingArc.TimingArcType.clk_to_q:
                                # Get the clkout pin so that we can write it to the sdc file
                                # in the clkout category. Just use setup instead of both setup and
                                # hold.
                                _, _, clk_type_name = self.get_reg_pin_and_clk_name(
                                    gpio_obj, sink_name, source_name, blk_mode)

                                # We're just saving the string to the object for writing later
                                ref_arg_str = self.set_clkout_ref_pin(
                                    gpio_obj, clk_type_name, input_config.clock_name, gpio_obj.gpio_def)

                                if arc in min_arc2delay_map:
                                    # Get the calculated delay
                                    tco_min_delay = self._get_delay_by_arc(
                                        min_arc2delay_map, self._min_model.get_tscale(), arc)

                                if arc in max_arc2delay_map:
                                    # Get the calculated delay
                                    tco_max_delay = self._get_delay_by_arc(
                                        max_arc2delay_map, self._max_model.get_tscale(), arc)

                                tco_arc = arc

                        # Get the routing delay if necessary
                        routing_delay_max = HSIOTiming.identify_routing_delay_on_ins(
                            gpio_obj.gpio_def, TimingArc.TimingArcType.clk_to_q,
                            True, self._max_model.get_tscale(), self.ins2bankname_map,
                            self.routing_label2bank_tbl_max_map)
                        routing_delay_min = HSIOTiming.identify_routing_delay_on_ins(
                            gpio_obj.gpio_def, TimingArc.TimingArcType.clk_to_q,
                            True, self._min_model.get_tscale(), self.ins2bankname_map,
                            self.routing_label2bank_tbl_min_map) 
                        
                        if tco_arc is not None:
                            tco_max_delay = "{0:.3f}".format(tco_max_delay + routing_delay_max)
                            tco_min_delay = "{0:.3f}".format(tco_min_delay + routing_delay_min)

                        sdcfile.write("set_input_delay {}{} -max {} [get_ports {{{}}}]\n".format(
                            clk_str, ref_arg_str, tco_max_delay, pin_name))
                        sdcfile.write("set_input_delay {}{} -min {} [get_ports {{{}}}]\n".format(
                            clk_str, ref_arg_str, tco_min_delay, pin_name))

                        if ddio_normal:
                            ddio_clk_str = "-clock " + input_config.clock_name

                            if not input_config.is_clk_inverted:
                                ddio_clk_str = "-clock_fall " + ddio_clk_str

                            ddio_pin_name = input_config.name_ddio_lo
                            
                            sdcfile.write("set_input_delay {}{} -max {} [get_ports {{{}}}]\n".format(
                                ddio_clk_str, ref_arg_str, tco_max_delay, ddio_pin_name))
                            sdcfile.write("set_input_delay {}{} -min {} [get_ports {{{}}}]\n".format(
                                ddio_clk_str, ref_arg_str, tco_min_delay, ddio_pin_name))
                         
                        in_tsu = tsu_max_delay
                        in_min_tsu = tsu_min_delay
                        in_thold = thold_min_delay
                        in_max_thold = thold_max_delay

                        inout_reg = True

                # We do output
                output_config = gpio_obj.output

                if output_config is not None and output_config.name != "":

                    pin_name = output_config.name
                    out_tied = False

                    # Check which type it is
                    arcs_list = []
                    if gpio_obj in self._gpio_p:
                        ptype = self.HSIOGPIOPathType.lvttl1_output
                        arcs_list, blk_mode = self.get_arcs_and_blk_mode(
                            gpio_obj, ptype, True)

                    else:
                        ptype = self.HSIOGPIOPathType.lvttl2_output
                        arcs_list, blk_mode = self.get_arcs_and_blk_mode(
                            gpio_obj, ptype, False)

                    # If the output is tied, then there's no constraint to
                    # print in sdc file
                    if output_config.tied_option == output_config.TiedOptType.vcc or\
                            output_config.tied_option == output_config.TiedOptType.gnd:
                        out_tied = True

                    if output_config is not None and\
                            output_config.register_option == output_config.RegOptType.none:

                        inout_comb = True

                        max_delay = 0
                        min_delay = 0

                        max_arc2delay_map, min_arc2delay_map = self._get_timing_delay(
                            gpio_obj, ptype)

                        for arc in arcs_list:

                            if arc.get_type() == TimingArc.TimingArcType.delay:

                                if arc in max_arc2delay_map:
                                    # Get the calculated delay
                                    max_delay = self._get_delay_by_arc(
                                        max_arc2delay_map, self._max_model.get_tscale(), arc)

                                if arc in min_arc2delay_map:
                                    # Get the calculated delay
                                    min_delay = self._get_delay_by_arc(
                                        min_arc2delay_map, self._min_model.get_tscale(), arc)

                        if not out_tied:
                            # PT-331: Add the non-registered constraint as a template and comment it out
                            sdcfile.write("# set_output_delay -clock <CLOCK> [-reference_pin <clkout_pad>] -max <MAX CALCULATION> [get_ports {{{}}}]\n".format(
                                pin_name))
                            sdcfile.write("# set_output_delay -clock <CLOCK> [-reference_pin <clkout_pad>] -min <MIN CALCULATION> [get_ports {{{}}}]\n".format(
                                pin_name))

                        # Get the routing delay if necessary
                        routing_delay_max = HSIOTiming.identify_routing_delay_on_ins(
                            gpio_obj.gpio_def, TimingArc.TimingArcType.delay,
                            False, self._max_model.get_tscale(), self.ins2bankname_map,
                            self.routing_label2bank_tbl_max_map)
                        routing_delay_min = HSIOTiming.identify_routing_delay_on_ins(
                            gpio_obj.gpio_def, TimingArc.TimingArcType.delay,
                            False, self._min_model.get_tscale(), self.ins2bankname_map,
                            self.routing_label2bank_tbl_min_map) 
                        
                        comb_row_list = []
                        if self.print_header:
                            comb_row_list.append(gpio_obj.name)

                        comb_row_list.append(pin_name)
                        comb_row_list.append("GPIO_OUT")

                        comb_row_list.append("{0:.3f}".format(max_delay + routing_delay_max))
                        comb_row_list.append("{0:.3f}".format(min_delay + routing_delay_min))

                        global_comb_table.add_row(comb_row_list)

                    else:
                        is_ins_reg = True
                        ddio_normal = False

                        out_clk_name = output_config.clock_name
                        clk_str = "-clock " + out_clk_name

                        if output_config.is_clk_inverted:
                            out_clk_name = "~" + out_clk_name
                            clk_str = "-clock_fall " + clk_str

                        # Check if it was configured in ddio mode
                        if output_config.ddio_type is not None and\
                                output_config.ddio_type != output_config.DDIOType.none:
                            
                            if output_config.ddio_type == output_config.DDIOType.normal:
                                ddio_normal = True
                                pin_name = output_config.name
                            else:                            
                                # pin_name is now a list for the SDC constraint
                                pin_name = "{} {}".format(
                                    output_config.name_ddio_lo, output_config.name)

                        elif output_config.is_serial:
                            pin_name = "{}[*]".format(output_config.name)

                        # sdc template
                        tco_arc = None
                        tsu_max_delay = 0
                        thold_min_delay = 0
                        tco_max_delay = 0
                        tco_min_delay = 0
                        ref_arg_str = ""

                        max_arc2delay_map, min_arc2delay_map = self._get_timing_delay(
                            gpio_obj, ptype)

                        for arc in arcs_list:
                            source_name = arc.get_source()
                            sink_name = arc.get_sink()

                            if arc.get_type() == TimingArc.TimingArcType.setup:

                                # Get the clkout pin so that we can write it to the sdc file
                                # in the clkout category. Just use setup instead of both setup and
                                # hold.
                                _, _, clk_type_name = self.get_reg_pin_and_clk_name(
                                    gpio_obj, sink_name, source_name, blk_mode)

                                # We're just saving the string to the object for writing later
                                ref_arg_str = self.set_clkout_ref_pin(
                                    gpio_obj, clk_type_name, output_config.clock_name, gpio_obj.gpio_def)

                                if arc in max_arc2delay_map:
                                    # Get the calculated delay
                                    tsu_max_delay = self._get_delay_by_arc(
                                        max_arc2delay_map, self._max_model.get_tscale(), arc)

                            elif arc.get_type() == TimingArc.TimingArcType.hold:

                                if arc in min_arc2delay_map:
                                    # Get the calculated delay
                                    thold_min_delay = self._get_delay_by_arc(
                                        min_arc2delay_map, self._min_model.get_tscale(), arc)

                            elif arc.get_type() == TimingArc.TimingArcType.clk_to_q:

                                if arc in min_arc2delay_map:
                                    # Get the calculated delay
                                    tco_min_delay = self._get_delay_by_arc(
                                        min_arc2delay_map, self._min_model.get_tscale(), arc)

                                if arc in max_arc2delay_map:
                                    # Get the calculated delay
                                    tco_max_delay = self._get_delay_by_arc(
                                        max_arc2delay_map, self._max_model.get_tscale(), arc)

                                tco_arc = arc

                        # Get the routing delay if necessary
                        routing_delay_max = HSIOTiming.identify_routing_delay_on_ins(
                            gpio_obj.gpio_def, TimingArc.TimingArcType.setup,
                            False, self._max_model.get_tscale(), self.ins2bankname_map,
                            self.routing_label2bank_tbl_max_map)
                        routing_delay_min = HSIOTiming.identify_routing_delay_on_ins(
                            gpio_obj.gpio_def, TimingArc.TimingArcType.hold,
                            False, self._min_model.get_tscale(), self.ins2bankname_map,
                            self.routing_label2bank_tbl_min_map) 
                        
                        # -max: (GPIO_OUT_PAD_TSU) - CORE CLK (max)
                        max_out_delay = "{0:.3f}".format(tsu_max_delay + routing_delay_max)

                        # -min: (GPIO_OUT_PAD_THOLD) + CORE CLK (min)
                        min_out_delay = "{0:.3f}".format((-1 * thold_min_delay) + routing_delay_min)

                        if not out_tied:
                            sdcfile.write("set_output_delay {}{} -max {} [get_ports {{{}}}]\n".format(
                                clk_str, ref_arg_str, max_out_delay, pin_name))
                            sdcfile.write("set_output_delay {}{} -min {} [get_ports {{{}}}]\n".format(
                                clk_str, ref_arg_str, min_out_delay, pin_name))

                            if ddio_normal:
                                ddio_clk_str = "-clock " + output_config.clock_name

                                if not output_config.is_clk_inverted:
                                    ddio_clk_str = "-clock_fall " + ddio_clk_str

                                ddio_pin_name = output_config.name_ddio_lo
                                
                                sdcfile.write("set_output_delay {}{} -max {} [get_ports {{{}}}]\n".format(
                                    ddio_clk_str, ref_arg_str, max_out_delay, ddio_pin_name))
                                sdcfile.write("set_output_delay {}{} -min {} [get_ports {{{}}}]\n".format(
                                    ddio_clk_str, ref_arg_str, min_out_delay, ddio_pin_name))

                        # Report file
                        out_tco_max = tco_max_delay
                        out_tco_min = tco_min_delay

                        inout_reg = True

                # We do OE
                oe_config = gpio_obj.output_enable

                if oe_config is not None and oe_config.name != "":

                    pin_name = oe_config.name

                    # Check which type it is
                    arcs_list = []
                    if gpio_obj in self._gpio_p:
                        ptype = self.HSIOGPIOPathType.lvttl1_oe
                        arcs_list, blk_mode = self.get_arcs_and_blk_mode(
                            gpio_obj, ptype, True)

                    else:
                        ptype = self.HSIOGPIOPathType.lvttl2_oe
                        arcs_list, blk_mode = self.get_arcs_and_blk_mode(
                            gpio_obj, ptype, False)

                    if oe_config is not None and not oe_config.is_register:

                        max_delay = 0
                        min_delay = 0

                        max_arc2delay_map, min_arc2delay_map = self._get_timing_delay(
                            gpio_obj, ptype)

                        for arc in arcs_list:

                            if arc.get_type() == TimingArc.TimingArcType.delay:

                                if arc in max_arc2delay_map:
                                    # Get the calculated delay
                                    max_delay = self._get_delay_by_arc(
                                        max_arc2delay_map, self._max_model.get_tscale(), arc)

                                if arc in min_arc2delay_map:
                                    # Get the calculated delay
                                    min_delay = self._get_delay_by_arc(
                                        min_arc2delay_map, self._min_model.get_tscale(), arc)

                        # PT-331: Add the non-registered constraint as a template and comment it out
                        sdcfile.write("# set_output_delay -clock <CLOCK> [-reference_pin <clkout_pad>] -max <MAX CALCULATION> [get_ports {{{}}}]\n".format(
                            pin_name))
                        sdcfile.write("# set_output_delay -clock <CLOCK> [-reference_pin <clkout_pad>] -min <MIN CALCULATION> [get_ports {{{}}}]\n".format(
                            pin_name))

                        comb_row_list = []
                        if self.print_header:
                            comb_row_list.append(gpio_obj.name)

                        # Get the routing delay if necessary
                        routing_delay_max = HSIOTiming.identify_routing_delay_on_ins(
                            gpio_obj.gpio_def, TimingArc.TimingArcType.delay,
                            False, self._max_model.get_tscale(), self.ins2bankname_map,
                            self.routing_label2bank_tbl_max_map)
                        routing_delay_min = HSIOTiming.identify_routing_delay_on_ins(
                            gpio_obj.gpio_def, TimingArc.TimingArcType.delay,
                            False, self._min_model.get_tscale(), self.ins2bankname_map,
                            self.routing_label2bank_tbl_min_map) 
                        
                        comb_row_list.append(pin_name)
                        comb_row_list.append("GPIO_OUT")

                        comb_row_list.append("{0:.3f}".format(max_delay + routing_delay_max))
                        comb_row_list.append("{0:.3f}".format(min_delay + routing_delay_min))

                        global_comb_table.add_row(comb_row_list)

                        if gpio_obj.is_gpio_differential_stl_io_std() and oe_config.name_oen != "":
                            sdcfile.write(
                                "# set_output_delay -clock <CLOCK> [-reference_pin <clkout_pad>] -max <MAX CALCULATION> [get_ports {{{}}}]\n".format(
                                    oe_config.name_oen))
                            sdcfile.write(
                                "# set_output_delay -clock <CLOCK> [-reference_pin <clkout_pad>] -min <MIN CALCULATION> [get_ports {{{}}}]\n".format(
                                    oe_config.name_oen))

                            comb_row_list = []
                            if self.print_header:
                                comb_row_list.append(gpio_obj.name)

                            comb_row_list.append(oe_config.name_oen)
                            comb_row_list.append("GPIO_OUT")

                            comb_row_list.append("{0:.3f}".format(max_delay + routing_delay_max))
                            comb_row_list.append("{0:.3f}".format(min_delay + routing_delay_min))

                            global_comb_table.add_row(comb_row_list)

                        inout_comb = True
                    else:
                        is_ins_reg = True

                        out_clk_name = oe_config.clock_name
                        clk_str = "-clock " + out_clk_name

                        if oe_config.is_clk_inverted:
                            out_clk_name = "~" + out_clk_name
                            clk_str = "-clock_fall " + clk_str

                        # sdc template

                        tco_arc = None
                        tsu_max_delay = 0
                        thold_min_delay = 0
                        tco_max_delay = 0
                        tco_min_delay = 0
                        ref_arg_str = ""

                        max_arc2delay_map, min_arc2delay_map = self._get_timing_delay(
                            gpio_obj, ptype)

                        # If it was differential gpio, we add in the OEN which is associated
                        # to the TX_SCLKN pin
                        for arc in arcs_list:
                            sink_name = arc.get_sink()
                            source_name = arc.get_source()

                            if arc.get_type() == TimingArc.TimingArcType.setup:
                                # Get the clkout pin so that we can write it to the sdc file
                                # in the clkout category. Just use setup instead of both setup and
                                # hold.
                                _, _, clk_type_name = self.get_reg_pin_and_clk_name(
                                    gpio_obj, sink_name, source_name, blk_mode)

                                # We're just saving the string to the object for writing later
                                ref_arg_str = self.set_clkout_ref_pin(
                                    gpio_obj, clk_type_name, oe_config.clock_name, gpio_obj.gpio_def)

                                if arc in max_arc2delay_map:
                                    # Get the calculated delay
                                    tsu_max_delay = self._get_delay_by_arc(
                                        max_arc2delay_map, self._max_model.get_tscale(), arc)

                            elif arc.get_type() == TimingArc.TimingArcType.hold:

                                if arc in min_arc2delay_map:
                                    # Get the calculated delay
                                    thold_min_delay = self._get_delay_by_arc(
                                        min_arc2delay_map, self._min_model.get_tscale(), arc)

                            elif arc.get_type() == TimingArc.TimingArcType.clk_to_q:

                                if arc in min_arc2delay_map:
                                    # Get the calculated delay
                                    tco_min_delay = self._get_delay_by_arc(
                                        min_arc2delay_map, self._min_model.get_tscale(), arc)

                                if arc in max_arc2delay_map:
                                    # Get the calculated delay
                                    tco_max_delay = self._get_delay_by_arc(
                                        max_arc2delay_map, self._max_model.get_tscale(), arc)

                                tco_arc = arc

                        # Get the routing delay if necessary
                        routing_delay_max = HSIOTiming.identify_routing_delay_on_ins(
                            gpio_obj.gpio_def, TimingArc.TimingArcType.setup,
                            False, self._max_model.get_tscale(), self.ins2bankname_map,
                            self.routing_label2bank_tbl_max_map)
                        routing_delay_min = HSIOTiming.identify_routing_delay_on_ins(
                            gpio_obj.gpio_def, TimingArc.TimingArcType.hold,
                            False, self._min_model.get_tscale(), self.ins2bankname_map,
                            self.routing_label2bank_tbl_min_map)
                        
                        # -max: max(GPIO_OUT_PAD_TSU) - CORE CLK (max)
                        max_out_delay = "{0:.3f}".format(tsu_max_delay + routing_delay_max)

                        # -min: min(GPIO_OUT_PAD_THOLD) - CORE CLK (min)
                        min_out_delay = "{0:.3f}".format((-1 * thold_min_delay) + routing_delay_min)

                        sdcfile.write("set_output_delay {}{} -max {} [get_ports {{{}}}]\n".format(
                            clk_str, ref_arg_str, max_out_delay, pin_name))
                        sdcfile.write("set_output_delay {}{} -min {} [get_ports {{{}}}]\n".format(
                            clk_str, ref_arg_str, min_out_delay, pin_name))

                        if gpio_obj.is_gpio_differential_stl_io_std() and oe_config.name_oen != "":
                            # If differential, we also create the oen constraint
                            # We're just saving the string to the object for writing later
                            ref_arg_str = self.set_clkout_ref_pin(
                                gpio_obj, "TX_SCLKN", oe_config.clock_name, gpio_obj.gpio_def)

                            sdcfile.write("set_output_delay {}{} -max {} [get_ports {{{}}}]\n".format(
                                clk_str, ref_arg_str, max_out_delay, oe_config.name_oen))
                            sdcfile.write("set_output_delay {}{} -min {} [get_ports {{{}}}]\n".format(
                                clk_str, ref_arg_str, min_out_delay, oe_config.name_oen))

                        # Rerport
                        oe_tco_max = tco_max_delay
                        oe_tco_min = tco_min_delay

                        inout_reg = True

                if is_ins_reg:

                    tmp_list = []
                    # Do input first
                    if in_clk_name != "":
                        if self.print_header:
                            tmp_list.append(gpio_obj.name)

                        tmp_list.append(in_clk_name)
                        tmp_list.append("{0:.3f}".format(in_tsu))
                        tmp_list.append("{0:.3f}".format(in_min_tsu))
                        tmp_list.append("{0:.3f}".format(in_max_thold))
                        tmp_list.append("{0:.3f}".format(in_thold))
                        tmp_list.append("")
                        tmp_list.append("")

                        if self.print_header:
                            global_reg_table.add_row(tmp_list)

                        tmp_list = []

                    if out_clk_name != "":
                        if self.print_header:
                            tmp_list.append(gpio_obj.name)

                        tmp_list.append(out_clk_name)
                        tmp_list.append("")  # Setup
                        tmp_list.append("")
                        tmp_list.append("")  # Hold
                        tmp_list.append("")

                        if out_tco_max != 0 and oe_tco_max != 0:
                            # Get the max between the two
                            max_tco = max(out_tco_max, oe_tco_max)
                            min_tco = min(out_tco_min, oe_tco_min)

                            tmp_list.append("{0:.3f}".format(max_tco))
                            tmp_list.append("{0:.3f}".format(min_tco))
                        elif out_tco_max != 0:
                            tmp_list.append("{0:.3f}".format(out_tco_max))
                            tmp_list.append("{0:.3f}".format(out_tco_min))
                        elif oe_tco_max != 0:
                            tmp_list.append("{0:.3f}".format(oe_tco_max))
                            tmp_list.append("{0:.3f}".format(oe_tco_min))
                        else:
                            tmp_list.append("")
                            tmp_list.append("")

                        if self.print_header:
                            global_reg_table.add_row(tmp_list)

        return inout_comb, inout_reg

    def get_reg_pin_and_clk_name(self, ins_obj, sink_name, source_name, blk_mode):
        # The sink and source name are interface name in the mode and
        # not necessarily the block port name
        pin_name = ""
        clk_name = ""
        clk_port_name = ""
        pin_name_to_variable = {}
        use_bus = False

        # Get the pin name based on the gpio mode
        if ins_obj.mode == ins_obj.PadModeType.output:
            out_cfg = ins_obj.output
            if out_cfg.is_serial:
                use_bus = True

            pin_name_to_variable = {
                "OUT": out_cfg.name,
                "OUTCLK": out_cfg.clock_name,
                "OUTFASTCLK": out_cfg.fastclk_name
            }

        elif ins_obj.mode == ins_obj.PadModeType.input:
            in_cfg = ins_obj.input
            if in_cfg.is_serial:
                use_bus = True

            pin_name_to_variable = {
                "IN": in_cfg.name,
                "INCLK": in_cfg.clock_name,
                "INFASTCLK": in_cfg.fastclk_name,
                "DLY_RST": in_cfg.dyn_delay_rst_name,
                "DLY_INC:": in_cfg.dyn_delay_ctrl_name,
                "DLY_ENA": in_cfg.dyn_delay_ena_name,
                "PULL_UP_ENA": in_cfg.pullup_ena_name
            }

        elif ins_obj.mode == ins_obj.PadModeType.inout:
            oe_cfg = ins_obj.output_enable
            in_cfg = ins_obj.input
            out_cfg = ins_obj.output

            pin_name_to_variable = {
                "OUT": out_cfg.name,
                "OUTCLK": out_cfg.clock_name,
                "OUTFASTCLK": out_cfg.fastclk_name,
                "IN": in_cfg.name,
                "INCLK": in_cfg.clock_name,
                "INFASTCLK": in_cfg.fastclk_name,
                "DLY_RST": in_cfg.dyn_delay_rst_name,
                "DLY_INC:": in_cfg.dyn_delay_ctrl_name,
                "DLY_ENA": in_cfg.dyn_delay_ena_name,
                "PULL_UP_ENA": in_cfg.pullup_ena_name,
                "OE": oe_cfg.name,
                "OUTCLK": oe_cfg.clock_name
            }
        elif ins_obj.mode == ins_obj.PadModeType.clkout:
            out_cfg = ins_obj.output
            pin_name_to_variable = {
                "OUTCLK": out_cfg.clock_name
            }

        # Try to attempt based on the map first since not all pins are using gen_pin
        if sink_name in pin_name_to_variable:
            pin_name = pin_name_to_variable[sink_name]

            src_inf = blk_mode.get_interface_object(sink_name)
            if src_inf is not None:
                if src_inf.is_bus_port() and use_bus:
                    pin_name = "{}[*]".format(pin_name)

        if source_name in pin_name_to_variable:
            clk_name = pin_name_to_variable[source_name]
            clk_inf = blk_mode.get_interface_object(source_name)
            clk_port_name = source_name

            if clk_inf is not None:
                if clk_inf.is_bus_port():
                    clk_name = "{}[*]".format(clk_name)

                clk_port_name = clk_inf.get_port_name()

        self.logger.debug("return: pin: {}, clk: {}, clk_port: {}".format(
            pin_name, clk_name, clk_port_name))

        return pin_name, clk_name, clk_port_name

    def get_static_delay(self, hsio_elem_tag, is_diff_delay, static_pvalue, is_max):

        if is_diff_delay:
            # The delay is incremental of 22 (0-63)
            static_delay_step_param = "HSIO_STATIC_DELAY_DIFFERENTIAL"
        else:
            # The delay is incremental of 60 (0-15)
            static_delay_step_param = "HSIO_STATIC_DELAY_SINGLE_ENDED"

        arc_table = self._load_arc_parameter(
            hsio_elem_tag, static_delay_step_param, is_max)

        # Get the value in PS (top-level caller will add the delay
        # to the main delay and then scaled and covert to ns))
        static_delay_ps = 0
        if arc_table is not None:
            value = arc_table.get_variable_delay("")
            if value is not None:
                static_delay_ps = static_pvalue * value

        return static_delay_ps

    def _get_delay_value(self, gpio_obj, delay_var_map, arc_table, is_max=True):
        '''
        :param gpio_obj: The user gpio design object
        :param delay_var_map: The table of variables mapped to acceptable
                        values
        :param arc_table: The arc_parameter table with the name
                        representing the variables concatenated (as
                        in the xml file). An ArcDelayTable object

        :return the delay value of that parameter based on the required
                gpio configuration
        '''
        self.logger.debug("HSIOGPIOComplexTiming _get_delay_value for {} mode {}".format(
            gpio_obj.name, gpio_obj.mode))

        # Only if the arc parameter has variable associated, then
        # we figure out, else take that one value
        if arc_table.get_pcount() == 0:
            delay_val = arc_table.get_variable_delay("")
            return delay_val

        iovolt_name = ""
        iostd_group = ""
        drive_strength_pos = "1"
        slew_rate = "false"
        serial_type = "sdio_ddio"
        schmitt_trigger = "false"
        additional_delay = 0
                    
        if gpio_obj.gpio_def in self._ins2iostd_map:
            io_volt = self._ins2iostd_map[gpio_obj.gpio_def]

            # Get the value associated to the io_standard
            if self.VariableType.io_voltage in delay_var_map:
                io_volt_list = delay_var_map[self.VariableType.io_voltage]

                if io_volt in io_volt_list:
                    iovolt_name = io_volt
                else:
                    "I/O Voltage {} not defined in timing io_voltage list".format(
                        io_volt)

        # Get the io standard from the gpio
        io_std = gpio_obj.io_standard

        # Get the head "(#) V" without the voltage and add "xSTL" if it
        # was SSTL/HSTL (don't need to distinguish if Differential or not)
        io_std_group_list = delay_var_map[self.VariableType.hsio_io_std_group]
        self.logger.debug(
            "Reading inst {} io std {}".format(gpio_obj.name, io_std))

        match_obj = re.match(r'^([0-9.]+) V', io_std)
        if match_obj:
            iostd_group = match_obj.group(1)
            self.logger.debug("Check if {} in {}".format(
                iostd_group, io_std_group_list))

            if iostd_group != "" and iostd_group in io_std_group_list:
                if io_std.find("HSTL") != -1 or io_std.find("SSTL") != -1:
                    iostd_group = "{}xSTL".format(iostd_group)
                    if iostd_group not in io_std_group_list:
                        raise ValueError(
                            "I/O Standard {} not defined in timing hsio_io_std_group list".format(io_std))

        self.logger.debug(
            "Will search for hsio_io_std_group:{}".format(iostd_group))

        if iostd_group == "":
            raise ValueError(
                "I/O Standard {} not defined in timing hsio_io_std_group list".format(io_std))

        # Depending on the gpio mode, we check the setting mentioned
        # in the delay_var_map and the arc_table to find the
        # suitable delay to pick.
        if gpio_obj.mode == gpio_obj.PadModeType.output or\
                gpio_obj.mode == gpio_obj.PadModeType.clkout or \
                gpio_obj.mode == gpio_obj.PadModeType.inout:
            # Drive Strength
            if gpio_obj.output is not None:
                # Get the drive strength map
                drv_str_list = delay_var_map[self.VariableType.drive_strength]
                out_cfg = gpio_obj.output

                gpio_io_std = GPIOIOStandard()
                valid_drv_strength = gpio_io_std.get_drive_strength_by_iostd(
                    gpio_obj.io_standard)
                # Sort the drive strength as the value saved in the timing table is 1-4
                # This is integer. Whereas the drive strength in parameter is string of number 1-4
                sorted_valid_drv_strength = sorted(valid_drv_strength)

                gpio_drv = out_cfg.adv_drive_strength

                # Search for the position of the drive strength in the sorted valid drive strength
                if gpio_drv in sorted_valid_drv_strength:
                    drv_index = sorted_valid_drv_strength.index(gpio_drv)
                    # Add 1 to the index
                    drive_strength_pos = str(drv_index + 1)

                else:
                    raise ValueError(
                        "Drive Strength {} not defined in timing variable".format(
                            gpio_drv))

                # Get the slew rate
                if out_cfg.is_slew_rate_output:
                    slew_rate = "true"

        # Depending on the gpio mode, we check the setting mentioned
        # in the delay_var_map and the arc_table to find the
        # suitable delay to pick.
        if gpio_obj.mode == gpio_obj.PadModeType.input or \
                gpio_obj.mode == gpio_obj.PadModeType.inout:
            # Get the gpio setting that is relevant:
            # schmitt_trigger
            if gpio_obj.input is not None:
                if gpio_obj.input.is_schmitt_trigger_enable:
                    schmitt_trigger = "true"

        # Get the arc delay parameter name
        delay_param = arc_table.get_parameter_name()

        # Static delay value to attached depends on the delay parameter name
        # because both output and input has their own static delay. Static Delay
        # is an addition to arc connecting to the PAD either sequential or comb arc.
        if delay_param.find("_OUT_") != -1 or delay_param.find("_OUTCLK_") != -1:
            if gpio_obj.output is not None:
                # Check for serialization
                if gpio_obj.output.is_serial:
                    serial_type = "ser"

                # TODO: don't hardcode the name and value here
                if gpio_obj.mode != gpio_obj.PadModeType.clkout and \
                        (delay_param == "HSIO_GPIO_OUTCLK_OUT_PAD_TCO" or delay_param == "HSIO_GPIO_OUT_PAD"):
                    # clkout mode does not support static delay. checking it although design
                    # would have reset it to 0. ICD indicates that even clkout arc has
                    # static delay but in fact clkout arc does not support it in UI
                    static_delay = gpio_obj.output.delay
                    hsio_elem_tag = self._get_block_timing_tag_section(is_max)

                    # The delay is incremental of 60 (0-15)
                    additional_delay += self.get_static_delay(
                        hsio_elem_tag, False, static_delay, is_max)

        elif delay_param.find("_IN_") != -1 or delay_param.find("_INCLK_"):
            if gpio_obj.input is not None:

                if not gpio_obj.input.is_dyn_delay and\
                        (delay_param == "HSIO_GPIO_INCLK_PAD_TSETUP" or
                            delay_param == "HSIO_GPIO_INCLK_PAD_THOLD" or
                            delay_param == "HSIO_GPIO_IN_PAD"):
            
                    static_delay = gpio_obj.input.delay
                    hsio_elem_tag = self._get_block_timing_tag_section(is_max)

                    is_diff_delay = False
                    if gpio_obj.is_gpio_differential_stl_io_std():
                        is_diff_delay = True

                    additional_delay += self.get_static_delay(
                        hsio_elem_tag, is_diff_delay, static_delay, is_max)

                if gpio_obj.input.is_serial:
                    serial_type = "ser"

        # Now that we have all the required setting, we
        # get the delay based on it
        var_list = arc_table.get_variables()
        concat_name = ""
        read_vcount = 0

        for index in range(len(var_list)):
            vname = var_list[index]
            vfound = True

            if vname in self.str2var_map:
                vtype = self.str2var_map[vname]

                if vtype == self.VariableType.io_voltage:
                    vsetting = vname + ":" + iovolt_name
                elif vtype == self.VariableType.drive_strength:
                    vsetting = vname + ":" + drive_strength_pos
                elif vtype == self.VariableType.schmitt_trigger:
                    vsetting = vname + ":" + schmitt_trigger
                elif vtype == self.VariableType.hsio_io_std_group:
                    vsetting = vname + ":" + iostd_group
                elif vtype == self.VariableType.slew_rate:
                    vsetting = vname + ":" + slew_rate
                elif vtype == self.VariableType.serial_type:
                    vsetting = vname + ":" + serial_type

                else:
                    vfound = False

                if vfound:
                    read_vcount += 1

                if concat_name == "":
                    concat_name = vsetting
                else:
                    concat_name = concat_name + "," + vsetting

            else:
                raise ValueError(
                    "Unexpected timing variable name {}".format(vname))

        # Verify that the variable count matches
        if read_vcount != arc_table.get_pcount():
            raise ValueError(
                "Mismatch variable count, {} and the pcount parameter, {}".format(
                    read_vcount, arc_table.get_pcount()))

        # Get the delay from table based on the concat variable name
        delay_val = 0
        tmp_delay = arc_table.get_variable_delay(concat_name)
        if tmp_delay is not None:
            delay_val = tmp_delay
        else:
            raise ValueError(
                "Unable to get the delay for variable {}".format(concat_name))

        delay_val += additional_delay

        return delay_val
    
