'''
Copyright (C) 2017-2020 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Nov 17, 2020

@author: maryam
'''
from __future__ import annotations

import sys
import os
from typing import Dict, TYPE_CHECKING, List

import xml.etree.ElementTree as et

import util.excp as app_excp
from util.singleton_logger import Logger

import device.excp as dev_excp
import device.pcr_device as dev_pcr
import common_device.hsio.hsio_device_service as hsio_svc

if TYPE_CHECKING:
    from tx60_device.gpio.gpio_design_comp import GPIOComplex, GPIOInputComplex

sys.path.append(os.path.join(os.path.dirname(__file__), '..'))


class HSIOGPIOLogicalPeriphery(object):
    '''
    This class is meant for writing out the LPF for HSIO
    that has been set as GPIO.
    '''
    iostd_drvstr_pcdp_map = \
        {
            "1.8 V LVCMOS": {16: 8, 12: 5, 8: 3, 4: 2},
            "1.5 V LVCMOS": {16: 17, 12: 9, 8: 5, 4: 2},
            "1.2 V LVCMOS": {12: 25, 8: 10, 4: 4, 2: 2},
            "1.8 V SSTL": {12: 19, 10: 13, 8: 8, 4: 2},
            "1.5 V SSTL": {12: 14, 10: 9, 8: 7, 4: 3},
            "1.35 V SSTL": {12: 14, 10: 9, 8: 7, 4: 3},
            "1.2 V SSTL": {12: 31, 10: 26, 8: 14, 4: 5},
            "1.8 V HSTL": {12: 6, 10: 4, 8: 3, 4: 1},
            "1.5 V HSTL": {12: 7, 10: 6, 8: 4, 4: 2},
            "1.2 V HSTL": {12: 28, 10: 18, 8: 10, 4: 4},
            "1.8 V Differential SSTL": {12: 19, 10: 13, 8: 8, 4: 2},
            "1.5 V Differential SSTL": {12: 14, 10: 9, 8: 7, 4: 3},
            "1.35 V Differential SSTL": {12: 14, 10: 9, 8: 7, 4: 3},
            "1.2 V Differential SSTL": {12: 31, 10: 26, 8: 14, 4: 5},
            "1.8 V Differential HSTL": {12: 6, 10: 4, 8: 3, 4: 1},
            "1.5 V Differential HSTL": {12: 7, 10: 6, 8: 4, 4: 2},
            "1.2 V Differential HSTL": {12: 28, 10: 18, 8: 10, 4: 4}
        }

    # ICD Separated them, but at this point, the value is exactly the
    # same in both maps (pcdp and pcdn)
    iostd_drvstr_pcdn_map = \
        {
            "1.8 V LVCMOS": {16: 8, 12: 5, 8: 3, 4: 2},
            "1.5 V LVCMOS": {16: 17, 12: 9, 8: 5, 4: 2},
            "1.2 V LVCMOS": {12: 25, 8: 10, 4: 4, 2: 2},
            "1.8 V SSTL": {12: 19, 10: 13, 8: 8, 4: 2},
            "1.5 V SSTL": {12: 14, 10: 9, 8: 7, 4: 3},
            "1.35 V SSTL": {12: 14, 10: 9, 8: 7, 4: 3},
            "1.2 V SSTL": {12: 31, 10: 26, 8: 14, 4: 5},
            "1.8 V HSTL": {12: 6, 10: 4, 8: 3, 4: 1},
            "1.5 V HSTL": {12: 7, 10: 6, 8: 4, 4: 2},
            "1.2 V HSTL": {12: 28, 10: 18, 8: 10, 4: 4},
            "1.8 V Differential SSTL": {12: 19, 10: 13, 8: 8, 4: 2},
            "1.5 V Differential SSTL": {12: 14, 10: 9, 8: 7, 4: 3},
            "1.35 V Differential SSTL": {12: 14, 10: 9, 8: 7, 4: 3},
            "1.2 V Differential SSTL": {12: 31, 10: 26, 8: 14, 4: 5},
            "1.8 V Differential HSTL": {12: 6, 10: 4, 8: 3, 4: 1},
            "1.5 V Differential HSTL": {12: 7, 10: 6, 8: 4, 4: 2},
            "1.2 V Differential HSTL": {12: 28, 10: 18, 8: 10, 4: 4}
        }

    def __init__(self, device_db, name):
        '''
        constructor
        '''
        self._device_db = device_db
        self._name = name

        self.logger = Logger

    def _get_pcdp_value(self, io_std, drv_strength):
        # default
        int_val = 0

        #self.logger.debug("get_pcdp_value: {} - {}".format(
        #    io_std, drv_strength))

        # Get the io standard
        if io_std in self.iostd_drvstr_pcdp_map:
            drv_str_map = self.iostd_drvstr_pcdp_map[io_std]

            if drv_strength in drv_str_map:
                int_val = drv_str_map[drv_strength]

        return int_val

    def _get_pcdn_value(self, io_std, drv_strength):
        # default
        int_val = 0

        #self.logger.debug("get_pcdn_value: {} - {}".format(
        #    io_std, drv_strength))

        # Get the io standard
        if io_std in self.iostd_drvstr_pcdn_map:
            drv_str_map = self.iostd_drvstr_pcdn_map[io_std]

            if drv_strength in drv_str_map:
                int_val = drv_str_map[drv_strength]

        return int_val

    def generate_hsio_gpio_pcr_diff_stl_npin_inout(self, parent, pcr_map,
                               gpio_design):
        '''

        :param parent:
        :param pcr_map:
        :param gpio_p_design: The GPIO P design instance that we want to read from
                when writing out the unconfigured N mode
        :return: used_pcr_name_list: List of pcr settings that were configured
        '''
        used_pcr_name_list = []

        skip_in = False
        skip_out = False

        # A valid INOUT is when BOTH Out and OE are configured. By right
        # input will always be created
        input_config = gpio_design.input
        if input_config is None:
            skip_in = True
        elif input_config.name == "":
            skip_in = True

        output_config = gpio_design.output
        if output_config is None:
            skip_out = True

        output_enable = gpio_design.output_enable

        in_pcr_name_list = []
        out_pcr_name_list = []

        # They may have shared parameters (ie. pcr_oe_sel_n) but it's passed to output to handle if inout
        if not skip_in:
            in_pcr_name_list = self.generate_hsio_gpio_pcr_diff_stl_npin_in(parent, pcr_map, gpio_design)
        if not skip_out:
            out_pcr_name_list = self.generate_hsio_gpio_pcr_diff_stl_npin_out(parent, pcr_map, gpio_design)

        used_pcr_name_list = in_pcr_name_list + out_pcr_name_list

        return used_pcr_name_list

    def generate_hsio_gpio_pcr_diff_stl_npin_in(self, parent, pcr_map: Dict[str, dev_pcr.PCRMap],
                               gpio_design: GPIOComplex):
        '''

        :param parent:
        :param pcr_map:
        :param gpio_p_design: The GPIO P design instance that we want to read from
                when writing out the unconfigured N mode
        :return: used_pcr_name_list: List of pcr settings that were configured
        '''
        used_pcr_name_list = []

        # Set related PCR settings to default except if the gpio io standard
        # is Differential SSTL in inout mode. Then some of the pcr
        # settings of the N pin has to be set although N is disabled from being configured
        input_config: GPIOInputComplex = gpio_design.input
        if input_config is None:
            # Resort to throwing exception since mode is output/inout/clkout
            # but config not set properly
            msg = 'HSIO GPIO {} input differential STL with invalid configuration.'.format(
                gpio_design.name)
            raise dev_excp.ConfigurationInvalidException(
                msg, app_excp.MsgLevel.error)

        for pcr_name, pcr_obj in sorted(pcr_map.items()):
            is_mode = False
            if pcr_obj.get_type() == dev_pcr.PCRMap.PCRMapType.mode:
                is_mode = True

            value = None

            # Set all PCR settings with that number to default
            if hsio_svc.HSIOService.is_pcr_defn_match(
                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_ie_n):

                mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                    hsio_svc.HSIOService.PCRModeType.disable]

                if is_mode and pcr_obj.is_mode_exists(mode_value):
                    value = mode_value

            elif hsio_svc.HSIOService.is_pcr_defn_match(
                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_oe_sel_n):

                # If it is inout, the setting will come from TX
                if gpio_design.mode != gpio_design.PadModeType.inout:
                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.pcr]

                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

            elif hsio_svc.HSIOService.is_pcr_defn_match(
                pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_lvdsrx_mode):

                if input_config.is_dyn_delay:
                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.core]

                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

            # Others that were stated in PT-1175 but are not searched here since
            # they are engineering settings are: pcr_oe_ph, pcr_oe_shift. Exception for
            # pcr_oe_inv since OE don't have inversion in PT
            if value is not None:
                if pcr_name not in used_pcr_name_list:
                    used_pcr_name_list.append(pcr_name)

                    #self.logger.debug("Saving instance diff stl {}: {} pcr {} - {}".format(
                    #    gpio_design.name, mode_no_str, pcr_name, value))

                    et.SubElement(parent, "efxpt:parameter",
                              name=pcr_name, value=value)

        return used_pcr_name_list

    def generate_pcr_lvdsrx_mode(self, parent, configured_pcr: List[hsio_svc.HSIOService.PCRDefnType], pcr_map: Dict[str, dev_pcr.PCRMap],
                                 design_obj: GPIOComplex):
        '''
        This function is used to set PCR_LVDSRX_MODE that is different
        than the default state (DISABLE). The supported scenarios:

        1) Differential IO: Set to PCR when not dynamic delay isn't use (will use static)
        2) Single-ended IO: Set to CORE only if dynamic delay is enabled

            Set the PCR_LVDSRX_MODE that is for the entire HSIO. So, if the resource was set on
            single-ended GPIO, we still need to check whether the instance has the dynamic delay
            set.  The pcr setting should be set to core only once even if both P and N pin is used.
            Rule traps that dynamic can only be used on resource P. But, user can still use
            resource N without dynamic delay set.

        :param parent:
        :param configured_pcr: The list of PCR that we already defined.
                Note that the returned value is the updated configured_pcr where
                it only needs to be updated if it is relevant and hasn't been added
                to the list yet.

        :param pcr_map:
        :param design_obj: The gpio design instance.  Set this to None if we want to
                assign the default setting to PCR_LVDSRX_MODE instead.
        :return: the updated configured_pcr
        '''
        # PT-1617: set default pcr lvdsrx_mode if not relevant
        # PT-1668: set to CORE if this was delay set even when it's not differential (single-ended)
        is_pcr_mode = False
        is_gpio_in_dyn_delay = False

        if design_obj is not None:

            # Dyn delay is only applicable to the P pin (rule trap). So, we don't
            # check it by resource since rule traps it.
            if design_obj.input is not None and \
                    (design_obj.mode == design_obj.PadModeType.input or \
                    design_obj.mode == design_obj.PadModeType.inout):
                if design_obj.input.is_dyn_delay:
                    is_gpio_in_dyn_delay = True
        else:
            is_pcr_mode = True

        pcr_lvdsrx_mode = hsio_svc.HSIOService.PCRDefnType.pcr_lvdsrx_mode
        str_pcr_lvdsrx_mode = hsio_svc.HSIOService.pcrdefn2str_map[pcr_lvdsrx_mode]

        if str_pcr_lvdsrx_mode not in configured_pcr:
            pcr_obj = pcr_map.get(str_pcr_lvdsrx_mode, None)

            if pcr_obj is not None:

                value = None
                is_mode = False

                if pcr_obj.get_type() == dev_pcr.PCRMap.PCRMapType.mode:
                    is_mode = True

                if is_pcr_mode:
                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.pcr]

                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

                elif is_gpio_in_dyn_delay:
                    # Only add this if we see that the general gpio usage
                    # should be not the default
                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.core]

                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

                if value is not None:
                    configured_pcr.append(str_pcr_lvdsrx_mode)

                    et.SubElement(parent, "efxpt:parameter",
                                name=str_pcr_lvdsrx_mode, value=value)
        return configured_pcr

    def generate_hsio_gpio_pcr_diff_stl_npin_out(self, parent, pcr_map,
                               gpio_design):
        '''

        :param parent:
        :param pcr_map:
        :param gpio_design: The GPIO P design instance that we want to read from
                when writing out the unconfigured N mode
        :return: used_pcr_name_list: List of pcr settings that were configured
        '''
        used_pcr_name_list = []

        # Set related PCR settings to default except if the gpio io standard
        # is Differential SSTL in output/inout/clkout mode. Then some of the pcr
        # settings of the N pin has to be set although N is disabled from being configured
        output_config = gpio_design.output
        if output_config is None:
            # Resort to throwing exception since mode is output/inout/clkout
            # but config not set properly
            msg = 'HSIO GPIO {} output differential STL with invalid configuration.'.format(
                gpio_design.name)
            raise dev_excp.ConfigurationInvalidException(
                msg, app_excp.MsgLevel.error)

        for pcr_name, pcr_obj in sorted(pcr_map.items()):
            is_mode = False
            if pcr_obj.get_type() == dev_pcr.PCRMap.PCRMapType.mode:
                is_mode = True

            value = None
            # Set all PCR settings with that number to default
            if hsio_svc.HSIOService.is_pcr_defn_match(
                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_pcdp_n):

                drv_strength = output_config.adv_drive_strength

                # Convert it to the value based on the io standard assigned to
                # the io
                value = str(self._get_pcdp_value(
                    gpio_design.io_standard, drv_strength))

            elif hsio_svc.HSIOService.is_pcr_defn_match(
                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_pcdn_n):

                drv_strength = output_config.adv_drive_strength

                # Convert it to the value based on the io standard assigned to
                # the io
                value = str(self._get_pcdn_value(
                    gpio_design.io_standard, drv_strength))

            elif hsio_svc.HSIOService.is_pcr_defn_match(
                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_oe_ctrl_n):

                if gpio_design.mode != gpio_design.PadModeType.inout:
                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.one]
                else:
                    # default is zero
                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.zero]

                if is_mode and pcr_obj.is_mode_exists(mode_value):
                    value = mode_value

            elif hsio_svc.HSIOService.is_pcr_defn_match(
                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_oe_sel_n):

                if gpio_design.mode != gpio_design.PadModeType.inout:
                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.pcr]

                else:
                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.core]

                if is_mode and pcr_obj.is_mode_exists(mode_value):
                    value = mode_value

            elif hsio_svc.HSIOService.is_pcr_defn_match(
                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_out_byp_n):
                # PT-1175: Use same setting as _p as long as output is registered
                if output_config.register_option != output_config.RegOptType.none and\
                        output_config.register_option is not None:
                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.register]
                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

            elif hsio_svc.HSIOService.is_pcr_defn_match(
                pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_out_ddren_n):
                # PT-1175: Use same setting as _p as long as output is registered
                if output_config.register_option != output_config.RegOptType.none and\
                        output_config.register_option is not None:
                    value = self._get_out_ddren_pcr(output_config, is_mode, pcr_obj)

            elif hsio_svc.HSIOService.is_pcr_defn_match(
                pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_out_mode_n):
                # PT-1175: Use same setting as _p as long as output is registered
                if output_config.register_option != output_config.RegOptType.none and\
                        output_config.register_option is not None:
                    value = self._get_out_mode_pcr(output_config, True, pcr_obj, is_mode)

            # Others that were stated in PT-1175 but are not searched here since
            # they are engineering settings are: pcr_oe_ph, pcr_oe_shift. Exception for
            # pcr_oe_inv since OE don't have inversion in PT
            if value is not None:
                if pcr_name not in used_pcr_name_list:
                    used_pcr_name_list.append(pcr_name)

                    #self.logger.debug("Saving instance diff stl {}: {} pcr {} - {}".format(
                    #    gpio_design.name, mode_no_str, pcr_name, value))

                    et.SubElement(parent, "efxpt:parameter",
                              name=pcr_name, value=value)

        return used_pcr_name_list

    def generate_hsio_gpio_pcr(self, design, parent, pcr_map,
                               mode_type, gpio_design,
                               used_pcr_name_list: List[hsio_svc.HSIOService.PCRDefnType]):
        '''
        Generate the PCR settings for the specific LVDS GPIO mode.
        If only one of the lvttl was configured, then the non-configured
        one would have gpio_design assigned to None. In that case, the
        unconfigured lvttl should be set to default.

        :param parent: The XML parent where the info should go to
        :param pcr_map: The block PCR mapping
        :param mode_type: The block mode type (expecting either lvttl1
                    or lvttl2)
        :param gpio_design: The GPIO Design object or None if not configured
        :return: used_pcr_name_list: List of pcr settings that were configured combined
		with the passed argument
        '''

        # The pcr names are _P (lvttl1) and _N (lvttl2). But there are
        # some setting which does not have those postfix (i.e. seriali
        char_to_find = ""
        if mode_type == hsio_svc.HSIOService.HSIOBlockModeType.lvttl1:
            char_to_find = "_P"
        else:
            char_to_find = "_N"

        if gpio_design is None:
            used_pcr_name_list = self._generate_unassigned_gpio_design(
                design, parent, pcr_map, char_to_find, used_pcr_name_list)
        else:
            #self.logger.debug("Write HSIO GPIO LPF instance: {}".format(gpio_design.name))

            if gpio_design.mode == gpio_design.PadModeType.input:
                used_pcr_name_list = self._generate_input_mode(
                    parent, pcr_map, char_to_find, gpio_design, used_pcr_name_list)
            elif gpio_design.mode == gpio_design.PadModeType.output:
                used_pcr_name_list = self._generate_clkout_output_mode(
                    parent, pcr_map, char_to_find, gpio_design, True, used_pcr_name_list)
            elif gpio_design.mode == gpio_design.PadModeType.clkout:
                used_pcr_name_list = self._generate_clkout_output_mode(
                    parent, pcr_map, char_to_find, gpio_design, False, used_pcr_name_list)
            elif gpio_design.mode == gpio_design.PadModeType.inout:
                used_pcr_name_list = self._generate_inout_mode(
                    parent, pcr_map, char_to_find, gpio_design, used_pcr_name_list)
            elif gpio_design.mode == gpio_design.PadModeType.none:
                used_pcr_name_list = self._generate_unconfig_mode(
                    parent, pcr_map, char_to_find, gpio_design, used_pcr_name_list)

        return used_pcr_name_list

    def _generate_unassigned_gpio_design(self, design, parent, pcr_map,
                                         mode_no_str, used_pcr_name_list: List[hsio_svc.HSIOService.PCRDefnType]):
        '''
        Set the LVTTL pcr settings for a particular lvttl mode number
        to default since it is not configured at all.
        :param parent: The XML parent where the info should go to
        :param pcr_map: The block PCR mapping
        :param mode_no_str: The string that contains the lvttl mode number (1/2)
        :return: used_pcr_name_list: List of pcr settings that were configured combined
		with the passed argument
        '''

        # Get the global gpio setting
        unused_config = None

        if design is not None:
            gpio_reg = design.get_block_reg(design.BlockType.gpio)

            if gpio_reg is not None:
                unused_config = gpio_reg.global_unused_config

        for pcr_name, pcr_obj in sorted(pcr_map.items()):
            is_mode = False
            if pcr_obj.get_type() == dev_pcr.PCRMap.PCRMapType.mode:
                is_mode = True

            value = None
            # Set all PCR settings with that number to default
            if pcr_name.endswith(mode_no_str):

                # PT-1253: Set the WKPU/WKPD based on the GPIO Global Unused State
                if unused_config is not None and (hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpu_p) or \
                                                  hsio_svc.HSIOService.is_pcr_defn_match(
                                                      pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpu_n) or \
                                                  hsio_svc.HSIOService.is_pcr_defn_match(
                                                      pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpd_p) or \
                                                  hsio_svc.HSIOService.is_pcr_defn_match(
                                                      pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpd_n)):

                    mode_enable = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.enable]
                    mode_disable = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.disable]

                    if is_mode and pcr_obj.is_mode_exists(mode_enable) and \
                            pcr_obj.is_mode_exists(mode_disable):

                        if unused_config.state == \
                                unused_config.StateType.input_weak_pullup:

                            if hsio_svc.HSIOService.is_pcr_defn_match(
                                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpu_p) or \
                                    hsio_svc.HSIOService.is_pcr_defn_match(
                                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpu_n):
                                value = mode_enable
                            else:
                                value = mode_disable

                        elif unused_config.state == \
                                unused_config.StateType.input_weak_pulldown:

                            if hsio_svc.HSIOService.is_pcr_defn_match(
                                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpu_p) or \
                                    hsio_svc.HSIOService.is_pcr_defn_match(
                                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpu_n):
                                value = mode_disable
                            else:
                                value = mode_enable
                        else:
                            value = mode_disable

                    else:
                        # Get default
                        value = pcr_obj.get_default()

                else:
                    # Get default
                    value = pcr_obj.get_default()


            if value is not None:
                if pcr_name not in used_pcr_name_list:
                    used_pcr_name_list.append(pcr_name)

                    et.SubElement(parent, "efxpt:parameter",
                              name=pcr_name, value=value)

        return used_pcr_name_list

    def _is_input_conn_type_internal(self, gpio_design: GPIOComplex):
        '''
        PT-1070: special setting for specific alt connection type
        :param gpio_design: The gpio design to check for
        :return:
        '''
        input_config = gpio_design.input
        if input_config is not None:
            # Make it an explicit check rather than check for inverse of not acceptable
            if input_config.conn_type in (
                input_config.ConnType.gclk_conn,
                input_config.ConnType.gctrl_conn,
                input_config.ConnType.pll_clkin_conn,
                input_config.ConnType.pll_extfb_conn,
                input_config.ConnType.rclk_conn,
                input_config.ConnType.rctrl_conn,
                input_config.ConnType.pcie_perstn_conn,
            ):
                return True

        return False

    def _generate_input_mode(self, parent, pcr_map,
                             mode_no_str, gpio_design: GPIOComplex,
                             used_pcr_name_list: List[hsio_svc.HSIOService.PCRDefnType]):
        '''
        Generate the PCR setting for an LVTTL that was set
        as input mode
        :param parent: The XML parent where the info should go to
        :param pcr_map: The block PCR mapping
        :param mode_no_str: The string that contains the lvttl mode number (1/2)
        :param gpio_design: The GPIO design object
        :return: used_pcr_name_list: List of pcr settings that were configured combined
		with the passed argument
        '''

        input_config = gpio_design.input
        if input_config is None:
            # Resort to throwing exception since mode is input
            # but config not set properly
            msg = 'HSIO GPIO {} set to input with invalid configuration.'.format(
                gpio_design.name)
            raise dev_excp.ConfigurationInvalidException(
                msg, app_excp.MsgLevel.error)

        for pcr_name, pcr_obj in sorted(pcr_map.items()):
            is_mode = False
            if pcr_obj.get_type() == dev_pcr.PCRMap.PCRMapType.mode:
                is_mode = True

            value = None
            # Set all PCR settings if relevant to the mode
            # Meaning if it was on lvttl1, only pcr_<name>_p gets set
            # while pcr_<name>_n is not touched
            if pcr_name.endswith(mode_no_str):

                if hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_in_byp_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_in_byp_n):

                    if input_config.conn_type == input_config.ConnType.normal_conn:
                        # Check whether it is registered or not
                        if input_config.is_register:
                            mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                                hsio_svc.HSIOService.PCRModeType.register]
                            if is_mode and pcr_obj.is_mode_exists(mode_value):
                                value = mode_value
                        else:
                            mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                                hsio_svc.HSIOService.PCRModeType.bypass]
                            if is_mode and pcr_obj.is_mode_exists(mode_value):
                                value = mode_value

                    elif input_config.is_alternate_connection():

                        # This is a global connection. including vref
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.bypass]
                        if is_mode and pcr_obj.is_mode_exists(mode_value):
                            value = mode_value

                    else:
                        msg = 'Unexpected Input mode connection type'
                        raise dev_excp.ConfigurationInvalidException(
                            msg, app_excp.MsgLevel.error)

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_in_mode_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_in_mode_n):

                    value = self._get_in_mode_pcr(input_config, pcr_obj, is_mode)

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_ie_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_ie_n):

                    if gpio_design.is_gpio_differential_stl_io_std():
                        # PT-1226: Set up unique for the Differential STL input
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.disable]

                    else:
                        # Check the io standard assignment
                        io_std = gpio_design.io_standard
                        if io_std.find("STL") != -1:
                            # The Differential STL is in the if cond
                            mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                                hsio_svc.HSIOService.PCRModeType.xstl]
                        else:
                            mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                                hsio_svc.HSIOService.PCRModeType.ttl]

                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_st_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_st_n):

                    if input_config.is_schmitt_trigger_enable:
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.enable]
                    else:
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.disable]

                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_rx_dly_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_rx_dly_n):

                    # Get the delay value right from object
                    # If it was Differential io std, we skip this because the range is diff
                    if not gpio_design.is_gpio_differential_stl_io_std():
                        value = str(input_config.delay)
                    else:
                        # Set to default
                        value = pcr_obj.get_default()

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpullup_sel_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpullup_sel_n):

                    if input_config.pull_option == input_config.PullOptType.dynamic:
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.core]
                    else:
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.pcr]

                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpu_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpu_n) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpd_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpd_n):

                    mode_enable = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.enable]
                    mode_disable = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.disable]

                    if is_mode and pcr_obj.is_mode_exists(mode_enable) and \
                            pcr_obj.is_mode_exists(mode_disable):

                        if input_config.pull_option == \
                                input_config.PullOptType.weak_pullup:

                            if hsio_svc.HSIOService.is_pcr_defn_match(
                                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpu_p) or \
                                hsio_svc.HSIOService.is_pcr_defn_match(
                                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpu_n):
                                value = mode_enable
                            else:
                                value = mode_disable

                        elif input_config.pull_option == \
                                input_config.PullOptType.weak_pulldown:

                            if hsio_svc.HSIOService.is_pcr_defn_match(
                                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpu_p) or \
                                    hsio_svc.HSIOService.is_pcr_defn_match(
                                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpu_n):
                                value = mode_disable
                            else:
                                value = mode_enable
                        else:
                            value = mode_disable

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkbh_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkbh_n):

                    if input_config.is_bus_hold:
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.enable]
                    else:
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.disable]

                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_oe_sel_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_oe_sel_n):

                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.pcr]
                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_oe_ctrl_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_oe_ctrl_n):

                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.zero]
                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_gpio_ser_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_gpio_ser_n):

                    # Not applicable to deser
                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.disable]
                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

                # Not sure if we should explicitly set out_byp
                # The rest set to default
                else:
                    value = pcr_obj.get_default()

            # Settings that are shared across P and N and does not end in _P/N
            elif hsio_svc.HSIOService.is_pcr_defn_match(
                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_vref_pin):

                #vref_enable = False
                # If this is a vref resource
                # We don't check for valid resource here. Rule traps it.
                if input_config.conn_type == input_config.ConnType.vref_conn:
                    # Strict resource check since non vref resource has the
                    # bit always disable
                    if self._device_db is not None:
                        sec_func_list = self._device_db.get_secondary_connection(
                            gpio_design.gpio_def)
                        if "VREF" in sec_func_list:
                            #vref_enable = True
                            mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                                hsio_svc.HSIOService.PCRModeType.enable]

                            # Only write if enabled since default is disabled
                            if is_mode and pcr_obj.is_mode_exists(mode_value):
                                value = mode_value

                #if not vref_enable:
                #    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                #        hsio_svc.HSIOService.PCRModeType.disable]


            elif hsio_svc.HSIOService.is_pcr_defn_match(
                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_des_gpio):

                # PT-1070: To also set it for specific conn type. Since default is LVDS,
                # we don't write it out so in case other gpio of the same base resource
                # needs it.
                if input_config.is_serial or self._is_input_conn_type_internal(gpio_design):
                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.gpio]

                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

                #else:
                #    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                #        hsio_svc.HSIOService.PCRModeType.lvds]

                #if is_mode and pcr_obj.is_mode_exists(mode_value):
                #    value = mode_value

            elif hsio_svc.HSIOService.is_pcr_defn_match(
                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_dly_cas):

                # Fixed to 1001 when dynamic delay is enabled
                if gpio_design.is_gpio_differential_stl_io_std():
                    # PT-1226
                    value = "1"
                elif input_config.is_dyn_delay:
                    value = "9"

            elif hsio_svc.HSIOService.is_pcr_defn_match(
                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_dpa_rst_sel):

                # PT-1230: Set the PCR_DPA_RST_SEL if dynamic delay is used
                if input_config.is_dyn_delay:
                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.core]

                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

            elif hsio_svc.HSIOService.is_pcr_defn_match(
                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_dpa_rst):

                # PT-1230: Set the PCR_DPA_RST_SEL if dynamic delay is used
                if input_config.is_dyn_delay:
                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.no_reset]

                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

            elif hsio_svc.HSIOService.is_pcr_defn_match(
                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_lvdsie):

                if gpio_design.is_gpio_differential_stl_io_std():
                    # PT-1226: Set up unique for the Differential STL input
                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.enable]

                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

            elif hsio_svc.HSIOService.is_pcr_defn_match(
                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_lvdsrx_dly):

                if gpio_design.is_gpio_differential_stl_io_std():
                    # PT-1226: Set differential input delay to pcr_lvdsrx_dly
                    value = str(input_config.delay)

            if value is not None:
                if pcr_name not in used_pcr_name_list:
                    used_pcr_name_list.append(pcr_name)

                    #self.logger.debug("Saving instance input {}: {} pcr {} - {}".format(
                    #    gpio_design.name, mode_no_str, pcr_name, value))

                    et.SubElement(parent, "efxpt:parameter",
                              name=pcr_name, value=value)

        return used_pcr_name_list

    def _generate_clkout_output_mode(self, parent, pcr_map,
                                     mode_no_str, gpio_design, is_output,
                                     used_pcr_name_list: List[hsio_svc.HSIOService.PCRDefnType]):
        '''
        Generate the PCR setting for an LVTTL that was set
        as output mode
        :param parent: The XML parent where the info should go to
        :param pcr_map: The block PCR mapping
        :param mode_no_str: The string that contains the lvttl mode number (1/2)
        :param gpio_design: The GPIO design object
        :param is_output: If true, it indicates output mode, False indicates
                clkout mode
        :return: used_pcr_name_list: List of pcr settings that were configured combined
		with the passed argument
        '''

        output_config = gpio_design.output
        if output_config is None:
            # Resort to throwing exception since mode is input
            # but config not set properly
            msg = 'HSIO GPIO {} set to output with invalid configuration.'.format(
                gpio_design.name)
            raise dev_excp.ConfigurationInvalidException(
                msg, app_excp.MsgLevel.error)

        for pcr_name, pcr_obj in sorted(pcr_map.items()):
            is_mode = False
            if pcr_obj.get_type() == dev_pcr.PCRMap.PCRMapType.mode:
                is_mode = True

            value = None
            # Set all PCR settings with that number to default
            if pcr_name.endswith(mode_no_str):

                if hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_pcdp_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_pcdp_n):

                    drv_strength = output_config.adv_drive_strength

                    # Convert it to the value based on the io standard assigned to
                    # the io
                    value = str(self._get_pcdp_value(
                        gpio_design.io_standard, drv_strength))

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_pcdn_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_pcdn_n):

                    drv_strength = output_config.adv_drive_strength

                    # Convert it to the value based on the io standard assigned to
                    # the io
                    value = str(self._get_pcdn_value(
                        gpio_design.io_standard, drv_strength))

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_slew_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_slew_n):

                    # fast is mapped to enable and slow to disable
                    if output_config.is_slew_rate_output:
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.fast]
                    else:
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.slow]

                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_tx_dly_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_tx_dly_n):

                    value = str(output_config.delay)

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_clk_out_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_clk_out_n):

                    if not is_output:
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.enable]
                        if is_mode and pcr_obj.is_mode_exists(mode_value):
                            value = mode_value

                    else:
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.disable]
                        if is_mode and pcr_obj.is_mode_exists(mode_value):
                            value = mode_value

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_out_byp_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_out_byp_n):

                    if is_output:
                        if output_config.register_option != output_config.RegOptType.none:
                            mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                                hsio_svc.HSIOService.PCRModeType.register]
                            if is_mode and pcr_obj.is_mode_exists(mode_value):
                                value = mode_value

                        else:
                            mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                                hsio_svc.HSIOService.PCRModeType.bypass]
                            if is_mode and pcr_obj.is_mode_exists(mode_value):
                                value = mode_value

                    else:
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.register]
                        if is_mode and pcr_obj.is_mode_exists(mode_value):
                            value = mode_value

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_out_ddren_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_out_ddren_n):

                    if is_output:
                        value = self._get_out_ddren_pcr(output_config, is_mode, pcr_obj)

                    else:
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.disable]
                        if is_mode and pcr_obj.is_mode_exists(mode_value):
                            value = mode_value

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_out_resync_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_out_resync_n):

                    if is_output:
                        value = self._get_out_resync_pcr(output_config, is_mode, pcr_obj)

                    else:
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.disable]
                        if is_mode and pcr_obj.is_mode_exists(mode_value):
                            value = mode_value

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_out_inv_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_out_inv_n):

                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.disable]
                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_out_mode_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_out_mode_n):

                    value = self._get_out_mode_pcr(output_config, is_output, pcr_obj, is_mode)

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_oe_sel_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_oe_sel_n):

                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.pcr]
                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_oe_ctrl_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_oe_ctrl_n):

                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.one]
                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_gpio_ser_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_gpio_ser_n):

                    if output_config.is_serial:
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.enable]
                    else:
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.disable]

                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

                # Not sure if we should explicitly set in_byp, in_mode
                # The rest set to default
                else:
                    value = pcr_obj.get_default()

            elif hsio_svc.HSIOService.is_pcr_defn_match(
                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_ser_gpio):

                # Default is LVDS so ignore
                if output_config.is_serial:
                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.gpio]

                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

            elif hsio_svc.HSIOService.is_pcr_defn_match(
                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_ser_mode):

                # Default is half
                if output_config.is_serial:
                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.full]

                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

            elif hsio_svc.HSIOService.is_pcr_defn_match(
                pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_split_en):

                if gpio_design.is_gpio_differential_stl_io_std():
                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.enable]

                    # PT-1175: Set this when output path is used
                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

            elif hsio_svc.HSIOService.is_pcr_defn_match(
                pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_lvdstx_dinsel):

                if gpio_design.is_gpio_differential_stl_io_std():
                    # PT-1175: Set to GPIO if it is not with serialization
                    if not output_config.is_serial:
                        # PT-1175: Set this when output path is used as long as it is
                        # not in serial mode
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.gpio]

                    else:
                        # Default: SER
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.ser]

                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

            if value is not None:
                if pcr_name not in used_pcr_name_list:
                    used_pcr_name_list.append(pcr_name)

                    #self.logger.debug("Saving instance clkout/output {}: {} pcr {} - {}".format(
                    #    gpio_design.name, mode_no_str, pcr_name, value))

                    et.SubElement(parent, "efxpt:parameter",
                              name=pcr_name, value=value)

        return used_pcr_name_list

    def _get_in_mode_pcr(self, input_config, pcr_obj, is_mode):
        '''
        :param input_config: The GPIOInput object
        :return: the pcr setting that matches with the current input config
        '''
        value = None

        if input_config is not None and input_config.conn_type == input_config.ConnType.normal_conn:
            if input_config.is_register:
                if input_config.ddio_type is not None and \
                        input_config.ddio_type != input_config.DDIOType.none:

                    if input_config.ddio_type == input_config.DDIOType.resync:
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.same_edge]
                        if is_mode and pcr_obj.is_mode_exists(mode_value):
                            value = mode_value

                    elif input_config.ddio_type == input_config.DDIOType.pipeline:
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.same_edge_pipelined]
                        if is_mode and pcr_obj.is_mode_exists(mode_value):
                            value = mode_value

                    else:
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.opp_edge]
                        if is_mode and pcr_obj.is_mode_exists(mode_value):
                            value = mode_value

                else:
                    # Use default
                    value = pcr_obj.get_default()
            else:
                # Use default
                value = pcr_obj.get_default()
        else:
            # Use default
            value = pcr_obj.get_default()

        return value

    def _get_out_mode_pcr(self, output_config, is_output, pcr_obj, is_mode):
        value = None

        if is_output:
            if output_config.ddio_type is not None and \
                    output_config.register_option == output_config.RegOptType.inv_register and \
                    output_config.ddio_type == output_config.DDIOType.none:

                mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                    hsio_svc.HSIOService.PCRModeType.negedge]
                if is_mode and pcr_obj.is_mode_exists(mode_value):
                    value = mode_value
            else:
                # Default
                value = pcr_obj.get_default()

        else:
            value = pcr_obj.get_default()

        return value

    def _get_out_resync_pcr(self, output_config, is_mode, pcr_obj):
        value = None

        if output_config is not None:
            if output_config.register_option == \
                    output_config.RegOptType.register:

                if output_config.ddio_type is not None and \
                        output_config.ddio_type == output_config.DDIOType.resync:

                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.enable]
                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

                else:
                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.disable]
                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

            else:
                mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                    hsio_svc.HSIOService.PCRModeType.disable]
                if is_mode and pcr_obj.is_mode_exists(mode_value):
                    value = mode_value

        else:
            mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                hsio_svc.HSIOService.PCRModeType.disable]
            if is_mode and pcr_obj.is_mode_exists(mode_value):
                value = mode_value

        return value

    def _get_out_ddren_pcr(self, output_config, is_mode, pcr_obj):
        value = None

        if output_config is not None:
            if output_config.register_option == \
                    output_config.RegOptType.register:

                if output_config.ddio_type is not None and \
                        (output_config.ddio_type == output_config.DDIOType.normal or \
                        output_config.ddio_type == output_config.DDIOType.resync):

                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.enable]
                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

                else:
                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.disable]
                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

            else:
                mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                    hsio_svc.HSIOService.PCRModeType.disable]
                if is_mode and pcr_obj.is_mode_exists(mode_value):
                    value = mode_value

        else:
            mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                hsio_svc.HSIOService.PCRModeType.disable]
            if is_mode and pcr_obj.is_mode_exists(mode_value):
                value = mode_value

        return value

    def _generate_inout_mode(self, parent, pcr_map,
                             mode_no_str, gpio_design,
                             used_pcr_name_list: List[hsio_svc.HSIOService.PCRDefnType]):
        '''
        Generate the PCR setting for an LVTTL that was set
        as inout mode
        :param parent: The XML parent where the info should go to
        :param pcr_map: The block PCR mapping
        :param mode_no_str: The string that contains the lvttl mode number (1/2)
        :param gpio_design: The GPIO design object
        :return: used_pcr_name_list: List of pcr settings that were configured combined
		with the passed argument
        '''

        skip_in = False
        skip_out = False

        # A valid INOUT is when BOTH Out and OE are configured. By right
        # input will always be created
        input_config = gpio_design.input
        if input_config is None:
            skip_in = True
        elif input_config.name == "":
            skip_in = True

        output_config = gpio_design.output
        if output_config is None:
            skip_out = True

        output_enable = gpio_design.output_enable

        for pcr_name, pcr_obj in sorted(pcr_map.items()):
            is_mode = False
            if pcr_obj.get_type() == dev_pcr.PCRMap.PCRMapType.mode:
                is_mode = True

            value = None
            # Set all PCR settings with that number to default
            if pcr_name.endswith(mode_no_str):

                # INPUT
                if hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_in_byp_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_in_byp_n):

                    if skip_in or input_config.conn_type == input_config.ConnType.normal_conn:
                        # Check whether it is registered or not
                        if not skip_in and input_config.is_register:
                            mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                                hsio_svc.HSIOService.PCRModeType.register]
                            if is_mode and pcr_obj.is_mode_exists(mode_value):
                                value = mode_value
                        else:
                            mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                                hsio_svc.HSIOService.PCRModeType.bypass]
                            if is_mode and pcr_obj.is_mode_exists(mode_value):
                                value = mode_value

                    elif input_config.is_alternate_connection():

                        # This is a global connection. including vref
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.bypass]
                        if is_mode and pcr_obj.is_mode_exists(mode_value):
                            value = mode_value

                    else:
                        msg = 'Unexpected Input mode connection type'
                        raise dev_excp.ConfigurationInvalidException(
                            msg, app_excp.MsgLevel.error)

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_in_mode_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_in_mode_n):

                    value = self._get_in_mode_pcr(input_config, pcr_obj, is_mode)

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_ie_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_ie_n):

                    # It should only be enabled if input is specified
                    if not skip_in and gpio_design.is_gpio_differential_stl_io_std():
                        # PT-1226: Set up unique for the Differential STL input
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.disable]

                    else:
                        # Check the io standard assignment
                        io_std = gpio_design.io_standard
                        if io_std.find("STL") != -1:
                            mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                                hsio_svc.HSIOService.PCRModeType.xstl]
                        else:
                            mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                                hsio_svc.HSIOService.PCRModeType.ttl]

                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_st_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_st_n):

                    if not skip_in and input_config.is_schmitt_trigger_enable:
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.enable]
                    else:
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.disable]

                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_rx_dly_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_rx_dly_n):

                    if not skip_in:
                        if not gpio_design.is_gpio_differential_stl_io_std():
                            # Get the delay value right from object
                            value = str(input_config.delay)
                        else:
                            value = pcr_obj.get_default()
                    else:
                        value = pcr_obj.get_default()

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpullup_sel_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpullup_sel_n):

                    if not skip_in and input_config.pull_option == input_config.PullOptType.dynamic:
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.core]
                    else:
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.pcr]

                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpu_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpu_n) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpd_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpd_n):

                    mode_enable = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.enable]
                    mode_disable = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.disable]

                    if is_mode and pcr_obj.is_mode_exists(mode_enable) and \
                            pcr_obj.is_mode_exists(mode_disable):

                        if not skip_in and input_config.pull_option == \
                                input_config.PullOptType.weak_pullup:

                            if hsio_svc.HSIOService.is_pcr_defn_match(
                                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpu_p) or \
                                hsio_svc.HSIOService.is_pcr_defn_match(
                                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpu_n):
                                value = mode_enable
                            else:
                                value = mode_disable

                        elif not skip_in and input_config.pull_option == \
                                input_config.PullOptType.weak_pulldown:

                            if hsio_svc.HSIOService.is_pcr_defn_match(
                                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpu_p) or \
                                    hsio_svc.HSIOService.is_pcr_defn_match(
                                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpu_n):
                                value = mode_disable
                            else:
                                value = mode_enable
                        else:
                            value = mode_disable

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkbh_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkbh_n):

                    if not skip_in and input_config.is_bus_hold:
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.enable]
                    else:
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.disable]

                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_pcdp_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_pcdp_n):

                    if not skip_out:
                        drv_strength = output_config.adv_drive_strength

                        # Convert it to the value based on the io standard assigned to
                        # the io
                        value = str(self._get_pcdp_value(
                            gpio_design.io_standard, drv_strength))

                    else:
                        value = pcr_obj.get_default()

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_pcdn_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_pcdn_n):

                    if not skip_out:
                        drv_strength = output_config.adv_drive_strength

                        # Convert it to the value based on the io standard assigned to
                        # the io
                        value = str(self._get_pcdn_value(
                            gpio_design.io_standard, drv_strength))

                    else:
                        value = pcr_obj.get_default()

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_slew_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_slew_n):

                    # fast - enable, slow - disable
                    if not skip_out and output_config.is_slew_rate_output:
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.fast]
                    else:
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.slow]

                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_tx_dly_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_tx_dly_n):

                    if not skip_out:
                        value = str(output_config.delay)
                    else:
                        value = pcr_obj.get_default()

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_clk_out_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_clk_out_n):

                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.disable]
                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_out_byp_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_out_byp_n):

                    if not skip_out:
                        if output_config.register_option != output_config.RegOptType.none:
                            mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                                hsio_svc.HSIOService.PCRModeType.register]
                            if is_mode and pcr_obj.is_mode_exists(mode_value):
                                value = mode_value

                        else:
                            mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                                hsio_svc.HSIOService.PCRModeType.bypass]
                            if is_mode and pcr_obj.is_mode_exists(mode_value):
                                value = mode_value

                    else:
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.register]
                        if is_mode and pcr_obj.is_mode_exists(mode_value):
                            value = mode_value

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_out_ddren_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_out_ddren_n):

                    value = self._get_out_ddren_pcr(output_config, is_mode, pcr_obj)

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_out_resync_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_out_resync_n):

                    value = self._get_out_resync_pcr(output_config, is_mode, pcr_obj)

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_out_inv_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_out_inv_n):

                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.disable]
                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_out_mode_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_out_mode_n):

                    value = self._get_out_mode_pcr(output_config, not skip_out, pcr_obj, is_mode)

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_oe_sel_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_oe_sel_n):

                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.core]
                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_oe_ctrl_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_oe_ctrl_n):

                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.zero]
                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_oe_byp_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_oe_byp_n):

                    if output_enable.is_register:
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.register]
                        if is_mode and pcr_obj.is_mode_exists(mode_value):
                            value = mode_value

                    else:
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.bypass]
                        if is_mode and pcr_obj.is_mode_exists(mode_value):
                            value = mode_value

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_gpio_ser_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_gpio_ser_n):

                    # Only applicable to serializer (not deser)
                    if not skip_out and output_config.is_serial:
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.enable]
                    else:
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.disable]

                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

                else:
                    value = pcr_obj.get_default()

            # Setting that does not ends with _P/_N
            elif hsio_svc.HSIOService.is_pcr_defn_match(
                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_vref_pin):

                #vref_enable = False
                # If this is a vref resource
                # We don't check for valid resource here. Rule traps it.
                if not skip_in and input_config.conn_type == input_config.ConnType.vref_conn:
                    # Strict resource check since non vref resource has the
                    # bit always disable
                    if self._device_db is not None:
                        sec_func_list = self._device_db.get_secondary_connection(
                            gpio_design.gpio_def)
                        if "VREF" in sec_func_list:
                            #vref_enable = True
                            mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                                hsio_svc.HSIOService.PCRModeType.enable]

                            # Only write if enabled as default is disabled
                            if is_mode and pcr_obj.is_mode_exists(mode_value):
                                value = mode_value


            elif hsio_svc.HSIOService.is_pcr_defn_match(
                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_des_gpio):

                # PT-1070: Set it also to GPIO if connection type is not normal. Since default is LVDS,
                # we don't write it out so in case other gpio of the same base resource
                # needs it.
                if not skip_in and (input_config.is_serial or self._is_input_conn_type_internal(gpio_design)):
                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.gpio]

                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

                #else:
                #    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                #        hsio_svc.HSIOService.PCRModeType.lvds]

            elif hsio_svc.HSIOService.is_pcr_defn_match(
                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_ser_gpio):

                # Default is LVDS, so ignore it (top-level or the other pin will manage)
                if not skip_out and output_config.is_serial:
                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.gpio]

                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

            elif hsio_svc.HSIOService.is_pcr_defn_match(
                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_ser_mode):

                # Default is half
                if not skip_out and output_config.is_serial:
                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.full]

                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

            elif hsio_svc.HSIOService.is_pcr_defn_match(
                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_dly_cas):

                # Fixed to 1001 when dynamic delay is enabled
                if not skip_in:
                    if gpio_design.is_gpio_differential_stl_io_std():
                        # PT-1226
                        value = "1"
                    elif input_config.is_dyn_delay:
                        value = "9"

            elif hsio_svc.HSIOService.is_pcr_defn_match(
                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_dpa_rst_sel):

                # PT-1230 & PT-2736
                if not skip_in and input_config.is_dyn_delay:
                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.core]

                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

            elif hsio_svc.HSIOService.is_pcr_defn_match(
                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_dpa_rst):

                # PT-1230 & PT-2736
                if not skip_in and input_config.is_dyn_delay:
                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.no_reset]

                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

            elif hsio_svc.HSIOService.is_pcr_defn_match(
                pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_split_en):

                if gpio_design.is_gpio_differential_stl_io_std():
                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.enable]

                    # Only write if enabled since default is disabled
                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

            elif hsio_svc.HSIOService.is_pcr_defn_match(
                pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_lvdstx_dinsel):

                if gpio_design.is_gpio_differential_stl_io_std():
                    # PT-1175: Set to GPIO if it is not with serialization
                    if not skip_out and not output_config.is_serial:
                        # PT-1175: Set this when output path is used as long as it is
                        # not in serial mode
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.gpio]

                    else:
                        # Default: SER
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.ser]

                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

            elif hsio_svc.HSIOService.is_pcr_defn_match(
                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_lvdsie):

                if not skip_in and gpio_design.is_gpio_differential_stl_io_std():
                    # PT-1226: Set up unique for the Differential STL input
                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.enable]

                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

            elif hsio_svc.HSIOService.is_pcr_defn_match(
                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_lvdsrx_dly):

                if not skip_in and gpio_design.is_gpio_differential_stl_io_std():
                    # PT-1226: Set differential input delay to pcr_lvdsrx_dly
                    value = str(input_config.delay)

            if value is not None:
                if pcr_name not in used_pcr_name_list:
                    used_pcr_name_list.append(pcr_name)

                    #self.logger.debug("Saving instance inout {}: {} pcr {} - {}".format(
                    #    gpio_design.name, mode_no_str, pcr_name, value))

                    et.SubElement(parent, "efxpt:parameter",
                              name=pcr_name, value=value)

        return used_pcr_name_list

    def _generate_unconfig_mode(self, parent, pcr_map,
                                mode_no_str, gpio_design,
                                used_pcr_name_list: List[hsio_svc.HSIOService.PCRDefnType]):
        '''
        Generate the PCR setting for an LVTTL that was set
        as none mode
        :param parent: The XML parent where the info should go to
        :param pcr_map: The block PCR mapping
        :param mode_no_str: The string that contains the lvttl mode number (1/2)
        :param gpio_design: The GPIO design object
        :return: used_pcr_name_list: List of pcr settings that were configured combined
		with the passed argument
        '''

        unused_config = gpio_design.unused_config

        if unused_config is None:
            # Resort to throwing exception since mode is input
            # but config not set properly
            msg = 'HSIO GPIO {} set to none with invalid configuration.'.format(
                gpio_design.name)
            raise dev_excp.ConfigurationInvalidException(
                msg, app_excp.MsgLevel.error)

        for pcr_name, pcr_obj in sorted(pcr_map.items()):
            is_mode = False
            if pcr_obj.get_type() == dev_pcr.PCRMap.PCRMapType.mode:
                is_mode = True

            value = None
            # Customize those that are different than default only
            if pcr_name.endswith(mode_no_str):

                if hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_in_byp_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_in_byp_n):

                    # This is a global connection. including vref
                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.bypass]
                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_oe_sel_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_oe_sel_n):

                    # This is a global connection. including vref
                    mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.pcr]
                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_ie_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_ie_n):

                    # Check the io standard assignment
                    io_std = gpio_design.io_standard
                    if io_std.find("STL") != -1:
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.xstl]
                    else:
                        mode_value = hsio_svc.HSIOService.pcrmode2str_map[
                            hsio_svc.HSIOService.PCRModeType.ttl]

                    if is_mode and pcr_obj.is_mode_exists(mode_value):
                        value = mode_value

                elif hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpu_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpu_n) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpd_p) or \
                    hsio_svc.HSIOService.is_pcr_defn_match(
                        pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpd_n):

                    mode_enable = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.enable]
                    mode_disable = hsio_svc.HSIOService.pcrmode2str_map[
                        hsio_svc.HSIOService.PCRModeType.disable]

                    if is_mode and pcr_obj.is_mode_exists(mode_enable) and \
                            pcr_obj.is_mode_exists(mode_disable):

                        if unused_config.state == \
                                unused_config.StateType.input_weak_pullup:

                            if hsio_svc.HSIOService.is_pcr_defn_match(
                                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpu_p) or \
                                hsio_svc.HSIOService.is_pcr_defn_match(
                                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpu_n):
                                value = mode_enable
                            else:
                                value = mode_disable

                        elif unused_config.state == \
                                unused_config.StateType.input_weak_pulldown:

                            if hsio_svc.HSIOService.is_pcr_defn_match(
                                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpu_p) or \
                                    hsio_svc.HSIOService.is_pcr_defn_match(
                                    pcr_name, hsio_svc.HSIOService.PCRDefnType.pcr_wkpu_n):
                                value = mode_disable
                            else:
                                value = mode_enable
                        else:
                            value = mode_disable

                else:
                    value = pcr_obj.get_default()

            if value is not None:
                if pcr_name not in used_pcr_name_list:
                    used_pcr_name_list.append(pcr_name)

                    #self.logger.debug("Saving instance unconfig {}: {} pcr {} - {}".format(
                    #    gpio_design.name, mode_no_str, pcr_name, value))

                    et.SubElement(parent, "efxpt:parameter",
                              name=pcr_name, value=value)

        return used_pcr_name_list
