'''
Copyright (C) 2017-2018 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc

Created on Sep 1, 2018

@author: maryam
'''
import os
import sys
import enum
import decimal

import xml.etree.ElementTree as et

# 3rd party library to display pretty table
from prettytable import PrettyTable

from typing import Dict

import util.gen_util as pt_util

import device.db_interface as device_dbi
import device.block_definition as dev_blk

import design.db as des_db

import common_device.gpio.gpio_design as gd
import common_device.writer as tbi

sys.path.append(os.path.join(os.path.dirname(__file__), '..'))


class GPIOTiming(tbi.BlockTimingWithArcs):
    '''
    Builds the timing data for GPIO used for printing out
    timing report and sdc file.
    '''

    class VariableType(enum.Enum):
        '''
        The list of supported timing variables.
        '''
        io_standard = 0
        schmitt_trigger = 1
        slew_rate = 2
        drive_strength = 3
        input_alternate = 4

    var2str_map = \
        {
            VariableType.io_standard: "io_standard",
            VariableType.schmitt_trigger: "schmitt_trigger",
            VariableType.slew_rate: "slew_rate",
            VariableType.drive_strength: "drive_strength",
            VariableType.input_alternate: "input_alternate"
        }

    str2var_map = \
        {
            "io_standard": VariableType.io_standard,
            "schmitt_trigger": VariableType.schmitt_trigger,
            "slew_rate": VariableType.slew_rate,
            "drive_strength": VariableType.drive_strength,
            "input_alternate": VariableType.input_alternate
        }

    class GPIOPathType(enum.Enum):
        '''
        Indication of the gpio physical path. It does not equate
        directly to the GPIO PadModeType
        '''
        input = 0
        output = 1
        oe = 3
        clkout = 4

    mode2path_map = \
        {
            gd.GPIO.PadModeType.input: GPIOPathType.input,
            gd.GPIO.PadModeType.output: GPIOPathType.output,
            gd.GPIO.PadModeType.inout: GPIOPathType.oe,
            gd.GPIO.PadModeType.clkout: GPIOPathType.clkout
        }

    def __init__(self, bname, device, design, report, sdc,
                 max_model, min_model, ins2iostd_map):
        '''
        Constructor
        '''
        super().__init__(bname, device, design, report, sdc,
                         max_model, min_model)

        # Map of instance to io standard
        self._ins2iostd_map = ins2iostd_map

        # Map of alternate gpio instance to the tuple of:
        # clock name, max delay, min delay (GPIO_CLK_IN/GPIO_IN), is_clk
        self.ins2_alt_delay_map = {}

        # Map of clkout gpio mode instance to the tuple of:
        # clock_name, clkout ref pin, max delay value, min delay value (GPIO_CLK_OUT), clkout_coord
        self.ins2_clkout_delay_map = {}

        # Map of max and min routing delay table baaed on io bank name. Only needed
        # for derived class, Titanium Ti375 and later
        self.routing_label2bank_tbl_max_map = {}
        self.routing_label2bank_tbl_min_map = {}

    def save_source_latency_and_clkout(self, ins_to_block_map):
        '''
        '''
        try:
            # Build the arc that is only required for this purpose
            # gpio clk and clkout
            self.build_prelim_arc()

            clkout_gpio, input_gpio, inout_gpio, devinsname2obj_map = \
                self._get_gpio_prelim_data(ins_to_block_map)

            if clkout_gpio or input_gpio or inout_gpio:
                # There are 2 reports to write
                self.get_prelim_output(clkout_gpio, 
                                  input_gpio, inout_gpio,
                                  devinsname2obj_map)
                
        except Exception as excp:
            raise excp
        
    def write(self, index, ins_to_block_map, clk2delay_map):
        '''
        Write out the report and sdc constraint for this block.

        :param index: The index of this section. Used in printing to report
        :param ins_to_block_map: Map of device instance name to the block name

        :return True if there was any valid instance found

        '''

        ins_exists = False

        try:
            # Build the arc
            self.build_arc()

            clk2ins_map, clkout_gpio, input_gpio, output_gpio, inout_gpio, devinsname2obj_map = \
                self._get_gpio_data(ins_to_block_map)

            if clkout_gpio or input_gpio or output_gpio or inout_gpio:
                # There are 2 reports to write
                self.write_output(index, clk2ins_map, clk2delay_map,
                                  clkout_gpio, input_gpio, output_gpio, inout_gpio,
                                  devinsname2obj_map)
                ins_exists = True

        except Exception as excp:
            raise excp

        return ins_exists

    def build_prelim_arc(self):
        self._mode2arcs = {}

        # Get the gpio block
        gpio_block = self._device.find_block("gpio")
        if gpio_block is None:
            msg = "Unable to find gpio block definition to build timing arcs"
            raise ValueError(msg)

        # Add the clkout arcs
        clkout_arcs = gpio_block.get_timing_arc("OUTCLK", "PAD", False)
        self._mode2arcs[GPIOTiming.GPIOPathType.clkout] = clkout_arcs

        # Add the input arcs
        self._get_in_arcs(gpio_block)


    def build_arc(self):
        '''
        A gpio mode is associated to a list of arcs.
        Given a mode, the corresponding list of arcs is returned
        which correspond to the block timing_arcs.

        GPIO:
        1) clkout: OUTCLK-PAD
        2) input:
            -nonreg: PAD-IN
            -reg: INCLK-IN, INCLK-PAD
        3) output:
            -nonreg: OUT-PAD
            -reg: OUTCLK-OUT, OUTCLK-PAD
        4) inout
            - input
            - output
            - oe:
                -nonreg: OE-PAD
                -reg: OUTCLK-OE, OUTCLK-PAD
        '''
        self._mode2arcs = {}

        # Get the gpio block
        gpio_block = self._device.find_block("gpio")
        if gpio_block is None:
            msg = "Unable to find gpio block definition to build timing arcs"
            raise ValueError(msg)

        # Add the clkout arcs
        clkout_arcs = gpio_block.get_timing_arc("OUTCLK", "PAD", False)
        self._mode2arcs[GPIOTiming.GPIOPathType.clkout] = clkout_arcs

        # Add the input arcs
        self._get_in_arcs(gpio_block)

        # Add the output arcs
        self._get_out_arcs(gpio_block)

        # For inout, only add OE arc
        oe_arcs = gpio_block.get_timing_arc("OE", "PAD", False)
        oe_arcs += gpio_block.get_timing_arc("OUTCLK", "PAD", True, "OE")
        oe_arcs += gpio_block.get_timing_arc("OUTCLK", "OE", True)
        self._mode2arcs[GPIOTiming.GPIOPathType.oe] = oe_arcs

    def _get_in_arcs(self, gpio_block):
        # TODO: Restructure this when have the time. This is a temporary
        # hack since the assumption is that IN[0] and IN[1] will
        # point to the same parameter name
        input_arcs = []
        in_list = ["IN", "IN[0]"]

        for in_name in in_list:
            tmp_arcs = gpio_block.get_timing_arc("PAD", in_name, False)
            if tmp_arcs:
                input_arcs += tmp_arcs

        input_arcs += gpio_block.get_timing_arc("INCLK", "PAD", True)

        for in_name in in_list:
            tmp_arcs = gpio_block.get_timing_arc("INCLK", in_name, True)
            if tmp_arcs:
                input_arcs += tmp_arcs

        self._mode2arcs[GPIOTiming.GPIOPathType.input] = input_arcs

    def _get_out_arcs(self, gpio_block):
        # TODO: restructure this up when have the time. This is a temporary
        # hack since the assumption is that OUT[0] and OUT[1] will
        # point to the same parameter name
        output_arcs = []
        out_list = ["OUT", "OUT[0]"]

        for out_name in out_list:
            tmp_arcs = gpio_block.get_timing_arc(out_name, "PAD", False)
            if tmp_arcs:
                output_arcs += tmp_arcs

            tmp_arcs = gpio_block.get_timing_arc(
                "OUTCLK", "PAD", True, out_name)
            if tmp_arcs:
                output_arcs += tmp_arcs

            tmp_arcs = gpio_block.get_timing_arc("OUTCLK", out_name, True)
            if tmp_arcs:
                output_arcs += tmp_arcs

        self._mode2arcs[GPIOTiming.GPIOPathType.output] = output_arcs

    def _get_gpio_prelim_data(self, ins_to_blk_map):
        gpio_reg = self._design.get_block_reg(
            des_db.PeriDesign.BlockType.gpio)

        # Map of device instance name to  gpio_obj stored
        # according to their mode type
        input_gpio = {}
        inout_gpio = {}
        clkout_gpio = {}

        # Map of device instance name to the instance object
        devinsname2obj_map = {}

        db_global = self._device.get_global_clocks()
        ins2func_map = db_global.get_instance_to_function_map()

        # Sort the gpio_obj name to the instance name
        for ins_name in sorted(ins_to_blk_map.keys(),
                               key=pt_util.natural_sort_key_for_list):

            # Skip if not a GPIO
            blk_name = ins_to_blk_map[ins_name]
            if blk_name not in device_dbi.DeviceDBService.block_str2type_map or\
                device_dbi.DeviceDBService.block_str2type_map[blk_name] != \
                    device_dbi.DeviceDBService.BlockType.GPIO:
                continue

            # Go through all instances and find out those
            # that are registered and get their clocks
            gpio_obj = gpio_reg.get_gpio_by_device_name(ins_name)

            # Skip instances that are not configured in design or
            # not a gpio instance
            if gpio_obj is None:
                continue

            if self._device is not None:
                ins_obj = self._device.find_instance(ins_name)
                devinsname2obj_map[ins_name] = ins_obj
                # self.logger.debug("Saving instance {} to devinsname2obj".format(ins_obj.get_name()))

            if gpio_obj.mode == gpio_obj.PadModeType.input and \
                    gpio_obj.input is not None:
                # Skip normal conn type instances
                if gpio_obj.input.is_alternate_connection():
                    input_gpio[gpio_obj.name] = gpio_obj

            elif gpio_obj.mode == gpio_obj.PadModeType.clkout and \
                    gpio_obj.output is not None:
                clkout_gpio[gpio_obj.name] = gpio_obj

            elif gpio_obj.mode == gpio_obj.PadModeType.inout:

                if gpio_obj.input is not None and\
                    gpio_obj.input.is_alternate_connection():
                    inout_gpio[gpio_obj.name] = gpio_obj

        return clkout_gpio, input_gpio, inout_gpio, devinsname2obj_map
    

    def _get_gpio_data(self, ins_to_blk_map):
        '''
        '''
        gpio_reg = self._design.get_block_reg(
            des_db.PeriDesign.BlockType.gpio)

        # Map of device instance name to  gpio_obj stored
        # according to their mode type
        input_gpio = {}
        output_gpio = {}
        inout_gpio = {}
        clkout_gpio = {}

        # Map of clk name to gpio obj names associated with the clock
        clk2ins_map = {}
        # Map of device instance name to the instance object
        devinsname2obj_map = {}

        db_global = self._device.get_global_clocks()
        ins2func_map = db_global.get_instance_to_function_map()

        # Sort the gpio_obj name to the instance name
        for ins_name in sorted(ins_to_blk_map.keys(),
                               key=pt_util.natural_sort_key_for_list):

            # Skip if not a GPIO
            blk_name = ins_to_blk_map[ins_name]
            if blk_name not in device_dbi.DeviceDBService.block_str2type_map or\
                device_dbi.DeviceDBService.block_str2type_map[blk_name] != \
                    device_dbi.DeviceDBService.BlockType.GPIO:
                continue

            # Go through all instances and find out those
            # that are registered and get their clocks
            gpio_obj = gpio_reg.get_gpio_by_device_name(ins_name)

            # Skip instances that are not configured in design or
            # not a gpio instance
            if gpio_obj is None:
                continue

            if self._device is not None:
                ins_obj = self._device.find_instance(ins_name)
                devinsname2obj_map[ins_name] = ins_obj
                # self.logger.debug("Saving instance {} to devinsname2obj".format(ins_obj.get_name()))

            if gpio_obj.mode == gpio_obj.PadModeType.input and \
                    gpio_obj.input is not None:
                input_gpio[gpio_obj.name] = gpio_obj

                # Get the clock name
                if gpio_obj.input.clock_name != "" and\
                        gpio_obj.input.is_register:
                    self._add_clk_name(
                        clk2ins_map, gpio_obj.input.clock_name, gpio_obj)
                elif gpio_obj.input.is_alternate_connection():
                    if gpio_obj.gpio_def in ins2func_map:
                        # TODO: When we allow GPIO instance to have multiple function
                        # then this needs to change
                        func_set = ins2func_map[gpio_obj.gpio_def]
                        if len(func_set) > 1:
                            raise ValueError(
                                'Found GPIO {} with having multiple'
                                ' global functions.'.format(gpio_obj.gpio_def))

                        for func_name in func_set:
                            if func_name == "GCLK":
                                self._add_clk_name(
                                    clk2ins_map, gpio_obj.input.name, gpio_obj)

            elif gpio_obj.mode == gpio_obj.PadModeType.output and \
                    gpio_obj.output is not None:
                output_gpio[gpio_obj.name] = gpio_obj

                if gpio_obj.output.clock_name != "" and \
                    (gpio_obj.output.register_option == gpio_obj.output.RegOptType.register or
                     gpio_obj.output.register_option == gpio_obj.output.RegOptType.inv_register):
                    self._add_clk_name(
                        clk2ins_map, gpio_obj.output.clock_name, gpio_obj)

            elif gpio_obj.mode == gpio_obj.PadModeType.clkout and \
                    gpio_obj.output is not None:
                clkout_gpio[gpio_obj.name] = gpio_obj

                if gpio_obj.output.clock_name != "":
                    self._add_clk_name(
                        clk2ins_map, gpio_obj.output.clock_name, gpio_obj)

            elif gpio_obj.mode == gpio_obj.PadModeType.inout:
                inout_gpio[gpio_obj.name] = gpio_obj

                if gpio_obj.input is not None and\
                        gpio_obj.input.clock_name != "" and\
                        gpio_obj.input.is_register:
                    self._add_clk_name(
                        clk2ins_map, gpio_obj.input.clock_name, gpio_obj)

                if gpio_obj.output is not None and\
                        gpio_obj.output.clock_name != "" and\
                        (gpio_obj.output.register_option == gpio_obj.output.RegOptType.register or
                         gpio_obj.output.register_option == gpio_obj.output.RegOptType.inv_register):
                    self._add_clk_name(
                        clk2ins_map, gpio_obj.output.clock_name, gpio_obj)

                if gpio_obj.output_enable is not None and\
                        gpio_obj.output_enable.clock_name != "" and\
                        gpio_obj.output_enable.is_register:
                    self._add_clk_name(
                        clk2ins_map, gpio_obj.output_enable.clock_name, gpio_obj)

        return clk2ins_map, clkout_gpio, input_gpio, \
            output_gpio, inout_gpio, devinsname2obj_map

    def get_prelim_output(self, clkout_gpio, 
                                  input_gpio, inout_gpio,
                                  devinsname2obj_map):
        # Read out clkout mode
        self.get_gpio_clkout_mode_data(
            clkout_gpio, devinsname2obj_map)

        # Readout input mode
        self.get_gpio_input_alt_mode_data(
            input_gpio, devinsname2obj_map)

        # Read out inout mode
        self.get_gpio_inout_mode_data(
            inout_gpio, devinsname2obj_map)
        
    def get_gpio_clkout_mode_data(self, gpio_map, devinsname2obj_map):
        for ins_name in sorted(gpio_map.keys(),
                                key=pt_util.natural_sort_key_for_list):

            gpio_obj = gpio_map[ins_name]

            output_config = gpio_obj.output

            if output_config is None or output_config.clock_name == "":
                continue

            clk_name = output_config.clock_name
            if output_config.is_clk_inverted:
                clk_name = "~" + clk_name

            max_delay, min_delay = self._get_timing_delay(
                gpio_obj, self.GPIOPathType.clkout)

            max_delay_str = self._get_delay(
                max_delay, self._max_model.get_tscale(),
                dev_blk.TimingArc.TimingArcType.delay)
            min_delay_str = self._get_delay(
                min_delay, self._min_model.get_tscale(),
                dev_blk.TimingArc.TimingArcType.delay)

            # Get the routing delay if necessary
            routing_delay_max = self.identify_routing_delay_on_ins(
                gpio_obj.gpio_def, dev_blk.TimingArc.TimingArcType.delay,
                False, self._max_model.get_tscale(), 
                self.routing_label2bank_tbl_max_map, is_clkout=True)
            routing_delay_min = self.identify_routing_delay_on_ins(
                gpio_obj.gpio_def, dev_blk.TimingArc.TimingArcType.delay,
                False, self._min_model.get_tscale(), 
                self.routing_label2bank_tbl_min_map, is_clkout=True)
    
            if gpio_obj.gpio_def in devinsname2obj_map:
                ins_obj = devinsname2obj_map[gpio_obj.gpio_def]

                max_delay_add = self.get_instance_specific_delay(
                    ins_obj, self._max_model.get_tscale())
                min_delay_add = self.get_instance_specific_delay(
                    ins_obj, self._min_model.get_tscale())

                out_delay_add_max, out_delay_add_min = self.get_clkout_additional_delay(
                    gpio_obj, devinsname2obj_map)

                # Add the clock network delay (min, max) and additional
                # instance delay to the value
                calc_max = max_delay_str + max_delay_add + out_delay_add_max + routing_delay_max
                calc_min = min_delay_str + min_delay_add + out_delay_add_min + routing_delay_min

            else:
                # Add the clock network delay (min, max) to the value
                calc_max = max_delay_str + routing_delay_max
                calc_min = min_delay_str + routing_delay_min

            # print("max: {} vs {} clk max: {} result: {}".format(
            #    max_delay_str, str(max_delay_str), max_clk, calc_max))
            # print("min: {} clk min: {} result: {}".format(min_delay_str, min_clk, calc_min))

            ref_pin_name, clockout_coord = self._save_output_clkout_names(
                gpio_obj, devinsname2obj_map)

            assert gpio_obj.name not in self.ins2_clkout_delay_map
            # Save the delay value to be used in other places
            # We don't take in the '~' inversion symbol
            self.ins2_clkout_delay_map[gpio_obj.name] = \
                (output_config.clock_name, ref_pin_name, calc_max, calc_min, clockout_coord)

    def get_gpio_input_alt_mode_data(self, gpio_map, devinsname2obj_map):
        for ins_name in sorted(gpio_map.keys(),
                                key=pt_util.natural_sort_key_for_list):

            gpio_obj = gpio_map[ins_name]

            input_config = gpio_obj.input

            if input_config is None:
                continue

            pin_name = input_config.name

            max_delay_add = 0
            min_delay_add = 0
            
            # Check if this instance has additional delay
            if gpio_obj.gpio_def in devinsname2obj_map:
                ins_obj = devinsname2obj_map[gpio_obj.gpio_def]

                max_delay_add = self.get_instance_specific_delay(
                    ins_obj, self._max_model.get_tscale())
                min_delay_add = self.get_instance_specific_delay(
                    ins_obj, self._min_model.get_tscale())

            assert input_config.is_alternate_connection()
            is_clk = input_config.is_alternate_clock_type()

            max_delay, min_delay = self._get_timing_delay(
                gpio_obj, self.GPIOPathType.input)

            # Get the relevent timing delay for the specified
            # arc type
            max_delay_str = self._get_delay(
                max_delay, self._max_model.get_tscale(),
                dev_blk.TimingArc.TimingArcType.delay)
            min_delay_str = self._get_delay(
                min_delay, self._min_model.get_tscale(),
                dev_blk.TimingArc.TimingArcType.delay)

            # Get the routing delay if necessary
            routing_delay_max = self.identify_routing_delay_on_ins(
                gpio_obj.gpio_def, dev_blk.TimingArc.TimingArcType.delay,
                True, self._max_model.get_tscale(), 
                self.routing_label2bank_tbl_max_map)
            routing_delay_min = self.identify_routing_delay_on_ins(
                gpio_obj.gpio_def, dev_blk.TimingArc.TimingArcType.delay,
                True, self._min_model.get_tscale(), 
                self.routing_label2bank_tbl_min_map)
            
            total_max_delay = max_delay_str + max_delay_add + routing_delay_max
            total_min_delay = min_delay_str + min_delay_add + routing_delay_min

            # Save this info to the map:
            assert gpio_obj.name not in self.ins2_alt_delay_map
            self.ins2_alt_delay_map[gpio_obj.name] = \
                (pin_name, total_max_delay, total_min_delay, is_clk)

    def get_gpio_inout_mode_data(self, gpio_map, devinsname2obj_map):
        for ins_name in sorted(gpio_map.keys(),
                                key=pt_util.natural_sort_key_for_list):

            gpio_obj = gpio_map[ins_name]

            in_max_map, in_min_map = self._get_timing_delay(
                gpio_obj, self.GPIOPathType.input)

            # Get the GPIO_PAD_TO_IN arc
            clkin_max_delay = self._get_delay(
                in_max_map, self._max_model.get_tscale(),
                dev_blk.TimingArc.TimingArcType.delay)
            clkin_min_delay = self._get_delay(
                in_min_map, self._min_model.get_tscale(),
                dev_blk.TimingArc.TimingArcType.delay)

            max_delay_add = 0
            min_delay_add = 0

            # Check if this instance has additional delay
            if gpio_obj.gpio_def in devinsname2obj_map:
                ins_obj = devinsname2obj_map[gpio_obj.gpio_def]

                max_delay_add = self.get_instance_specific_delay(
                    ins_obj, self._max_model.get_tscale())
                min_delay_add = self.get_instance_specific_delay(
                    ins_obj, self._min_model.get_tscale())

            # We do input
            input_config = gpio_obj.input

            if input_config is not None and input_config.name != "":
                assert input_config.is_alternate_connection()

                pin_name = input_config.name

                is_clk = input_config.is_alternate_clock_type()

                # Get the routing delay if necessary
                routing_delay_max = self.identify_routing_delay_on_ins(
                    gpio_obj.gpio_def, dev_blk.TimingArc.TimingArcType.delay,
                    True, self._max_model.get_tscale(), 
                    self.routing_label2bank_tbl_max_map)
                routing_delay_min = self.identify_routing_delay_on_ins(
                    gpio_obj.gpio_def, dev_blk.TimingArc.TimingArcType.delay,
                    True, self._min_model.get_tscale(), 
                    self.routing_label2bank_tbl_min_map)
                
                total_max_delay = clkin_max_delay + max_delay_add + routing_delay_max
                total_min_delay = clkin_min_delay + min_delay_add + routing_delay_min

                # Save this info to the map:
                assert gpio_obj.name not in self.ins2_alt_delay_map
                self.ins2_alt_delay_map[gpio_obj.name] = \
                    (pin_name, total_max_delay, total_min_delay, is_clk)

    def write_output(self, index, clk2ins_map, clk2delay_map, clkout_gpio,
                     input_gpio, output_gpio, inout_gpio, devinsname2obj_map):
        '''
        Writes out gpio related summary and sdc

        '''
        write_successful = None

        try:

            # Open the file
            rptfile = open(self._report_file, 'a')
            sdcfile = open(self._sdc_file, 'a')

            write_successful = False

            rptfile.write(
                "\n---------- {}. GPIO Timing Report (begin) ----------\n".format(index))

            sdcfile.write("\n# GPIO Constraints\n")
            sdcfile.write("####################\n")

            # Print the clock network delay
            # self._write_clock_network_delay(
            #    rptfile, clk2ins_map, clk2delay_map,
            #    clkout_gpio, input_gpio,
            #    output_gpio, inout_gpio)

            # Print out clkout mode
            self._write_gpio_clkout_mode(
                rptfile, clk2delay_map, clkout_gpio,
                devinsname2obj_map)

            # Global table to be printed based on path (not mode)
            global_reg_table = PrettyTable(["Instance Name", "Clock Pin",
                                            "Max Setup (ns)", "Min Setup (ns)",
                                            "Max Hold (ns)", "Min Hold (ns)",
                                            "Max Clock To Out (ns)", "Min Clock To Out (ns)"])

            global_comb_table = PrettyTable(
                ["Instance Name", "Pin Name", "Parameter", "Max (ns)", "Min (ns)"])

            # Print out input mode
            in_comb, in_reg = self._write_gpio_input_mode(
                sdcfile, clk2delay_map, input_gpio,
                global_reg_table, global_comb_table,
                devinsname2obj_map)

            # Print out output mode
            out_comb, out_reg = self._write_gpio_output_mode(
                sdcfile, clk2delay_map, output_gpio,
                global_reg_table, global_comb_table,
                devinsname2obj_map)

            # Print out inout mode
            inout_comb, inout_reg = self._write_gpio_inout_mode(
                sdcfile, clk2delay_map, inout_gpio,
                global_reg_table, global_comb_table,
                devinsname2obj_map)

            if in_comb or out_comb or inout_comb:
                rptfile.write("\nNon-registered GPIO Configuration:\n")
                rptfile.write("===================================\n")

                rptfile.write("\n{}\n".format(global_comb_table.get_string()))

            if in_reg or out_reg or inout_reg:
                rptfile.write("\nRegistered GPIO Configuration:\n")
                rptfile.write("===============================\n")

                rptfile.write("\n{}\n".format(global_reg_table.get_string()))

            rptfile.write(
                "\n---------- GPIO Timing Report (end) ----------\n")

            rptfile.close()
            sdcfile.close()

            write_successful = True

        except Exception as excp:
            if write_successful is not None and\
                    not write_successful:
                rptfile.close()
                sdcfile.close()
            raise excp

    def _write_clock_network_delay(self, rptfile, clk2ins_map,
                                   clk2delay_map, clkout_map, input_map,
                                   output_map, inout_map):
        '''
        Prints the or each clock that goes through the clock network.
        :param rptfile: file handle that has already been opened
        :param *_map: Map of device instance name to GPIOs

        :return the map of clock name to the pair of max and min delay
        '''
        # clock2delay_map = {}

        # Iterate through all modes and print out the delay of
        # the clock that goes through the outclk/inclk pin
        # Scenario: Table D and E
        if clk2ins_map and (clkout_map or input_map or output_map or inout_map):
            rptfile.write("\nClock Network Delay:\n")
            rptfile.write("=====================\n")

            # Table header
            table = PrettyTable(["Clock Pin", "Max (ns)", "Min (ns)"])

            # Set to zero as default
            max_delay_str = "0"
            min_delay_str = "0"

            if self.DelayType.max in clk2delay_map:
                max_delay_str = "{}".format(round(
                    decimal.Decimal(clk2delay_map[self.DelayType.max]), 3))

            if self.DelayType.min in clk2delay_map:
                min_delay_str = "{}".format(round(
                    decimal.Decimal(clk2delay_map[self.DelayType.min]), 3))

            # We print the Instance name and Clock name by finding
            # all the clock pin names used in the GPIO. The Parameter
            # name is fixed and all clocks technically have the same
            # delay in core.

            for clk_name in sorted(clk2ins_map.keys()):
                row_list = []

                row_list.append(clk_name)
                # row_list.append("CORE_CLK_DELAY")
                row_list.append(max_delay_str)
                row_list.append(min_delay_str)

                table.add_row(row_list)

            rptfile.write("\n{}\n".format(table.get_string()))

    def get_instance_specific_delay(self, ins_obj, tscale: float):
        '''
        Return the additional delay routing associated to the device instance
        if there's any.

        :param ins_obj: Device instance object
        :param tscale: tscale value
        :return: the additional routing delay that has been scaled in ns
        '''
        from device.block_instance import InstanceTiming

        delay = 0

        if ins_obj is not None:
            ins_timing = ins_obj.get_instance_timing()

            if ins_timing is not None:
                # Get the routing type delay
                ret_val = ins_timing.get_delay_of_type(
                    InstanceTiming.DelayType.routing)

                if ret_val is not None:
                    delay = (ret_val * tscale) / 1000
                    # self.logger.debug("get_instance_specific_delay: {} -> {} : {} ({})".format(
                    #    ins_obj.get_name(), ret_val, delay, tscale))

        return delay

    def _write_gpio_clkout_mode(self, rptfile,
                                clk2delay_map, gpio_map,
                                devinsname2obj_map):
        '''
        Generate the table summary of the Unused GPIO configuration.

        :param rptfile: file handle that has already been opened
        :param clk2delay_map: Map of delay type to the delay value
                        (in float) for core clock network delay
        :param gpio_map: Map of device instance name to GPIOs
                that are set as clkout
        :param devinsname2obj_map: The map of device instance name
                to the instance object
        '''

        if gpio_map:
            rptfile.write("\nClkout GPIO Configuration:\n")
            rptfile.write("===========================\n")

            # Table header
            table = PrettyTable(
                ["Instance Name", "Clock Pin", "Parameter", "Max (ns)", "Min (ns)", "Reference Pin Name"])
            parameter_name = "GPIO_CLK_OUT"

            # Get the clock network delay
            max_clk = clk2delay_map[self.DelayType.max]
            min_clk = clk2delay_map[self.DelayType.min]

            for ins_name in sorted(gpio_map.keys(),
                                   key=pt_util.natural_sort_key_for_list):

                gpio_obj = gpio_map[ins_name]

                row_list = self._create_clkout_delay_report(gpio_obj, devinsname2obj_map,
                                                            max_clk, min_clk, parameter_name)

                if row_list:
                    table.add_row(row_list)

            rptfile.write("\n{}\n".format(table.get_string()))

    def _create_clkout_delay_report(self, gpio_obj, devinsname2obj_map,
                                    max_clk, min_clk, parameter_name):
        row_list = []

        output_config = gpio_obj.output

        if output_config is None or output_config.clock_name == "":
            return row_list

        clk_name = output_config.clock_name
        if output_config.is_clk_inverted:
            clk_name = "~" + clk_name

        max_delay, min_delay = self._get_timing_delay(
            gpio_obj, self.GPIOPathType.clkout)

        max_delay_str = self._get_delay(
            max_delay, self._max_model.get_tscale(),
            dev_blk.TimingArc.TimingArcType.delay)
        min_delay_str = self._get_delay(
            min_delay, self._min_model.get_tscale(),
            dev_blk.TimingArc.TimingArcType.delay)

        # Get the routing delay if necessary
        routing_delay_max = self.identify_routing_delay_on_ins(
            gpio_obj.gpio_def, dev_blk.TimingArc.TimingArcType.delay,
            False, self._max_model.get_tscale(),
            self.routing_label2bank_tbl_max_map, is_clkout=True)
        routing_delay_min = self.identify_routing_delay_on_ins(
            gpio_obj.gpio_def, dev_blk.TimingArc.TimingArcType.delay,
            False, self._min_model.get_tscale(),
            self.routing_label2bank_tbl_min_map, is_clkout=True)
        
        if gpio_obj.gpio_def in devinsname2obj_map:
            ins_obj = devinsname2obj_map[gpio_obj.gpio_def]

            max_delay_add = self.get_instance_specific_delay(
                ins_obj, self._max_model.get_tscale())
            min_delay_add = self.get_instance_specific_delay(
                ins_obj, self._min_model.get_tscale())

            out_delay_add_max, out_delay_add_min = self.get_clkout_additional_delay(
                gpio_obj, devinsname2obj_map)

            # Add the clock network delay (min, max) and additional
            # instance delay to the value
            calc_max = max_delay_str + max_clk + max_delay_add + out_delay_add_max + routing_delay_max
            calc_min = min_delay_str + min_clk + min_delay_add + out_delay_add_min + routing_delay_min

        else:
            # Add the clock network delay (min, max) to the value
            calc_max = max_delay_str + max_clk + routing_delay_max
            calc_min = min_delay_str + min_clk + routing_delay_min

        # print("max: {} vs {} clk max: {} result: {}".format(
        #    max_delay_str, str(max_delay_str), max_clk, calc_max))
        # print("min: {} clk min: {} result: {}".format(min_delay_str, min_clk, calc_min))

        # We print the Instance name and Clock name
        row_list.append(gpio_obj.name)
        row_list.append(clk_name)
        row_list.append(parameter_name)
        row_list.append("{0:.3f}".format(calc_max))
        row_list.append("{0:.3f}".format(calc_min))

        ref_pin_name, _ = self._save_output_clkout_names(
            gpio_obj, devinsname2obj_map)

        row_list.append(ref_pin_name)

        return row_list

    def get_outclk_clkout_info(self, gpio_obj, devinsname2obj_map):

        output_config = gpio_obj.output
        clkout_str = ""
        ref_pin_name = ""

        if output_config is not None and output_config.clock_name != "":

            if gpio_obj.gpio_def in devinsname2obj_map:
                ins_obj = devinsname2obj_map[gpio_obj.gpio_def]

                ref_pin_name, clkout_str, clkout_coord = self.get_clkout_info_with_coord(
                    gpio_obj.name, ins_obj, "OUTCLK", output_config.clock_name)

        return ref_pin_name, clkout_str, clkout_coord

    def get_outclk_oe_clkout_info(self, gpio_obj, devinsname2obj_map):
        clkout_str = ""
        ref_pin_name = ""

        oe_config = gpio_obj.output_enable

        if oe_config is not None and oe_config.clock_name != "":

            if gpio_obj.gpio_def in devinsname2obj_map:
                ins_obj = devinsname2obj_map[gpio_obj.gpio_def]

                ref_pin_name, clkout_str = self.get_clkout_info(
                    gpio_obj.name, ins_obj, "OUTCLK", oe_config.clock_name)

        return ref_pin_name, clkout_str

    def get_inclk_clkout_info(self, gpio_obj, devinsname2obj_map):
        clkout_str = ""
        ref_pin_name = ""

        input_config = gpio_obj.input

        if input_config is not None and input_config.clock_name != "":

            if gpio_obj.gpio_def in devinsname2obj_map:
                ins_obj = devinsname2obj_map[gpio_obj.gpio_def]

                ref_pin_name, clkout_str = self.get_clkout_info(
                    gpio_obj.name, ins_obj, "INCLK", input_config.clock_name)

        return ref_pin_name, clkout_str

    def _save_output_clkout_names(self, gpio_obj, devinsname2obj_map):

        # Format:
        # <clkout/block instance name> -clock <clkout pin name in PT> -reference_pin [get_ports {<clkout pad name in core>}]

        # Example string:
        # rx_fclk_output -clock rx_fclk -reference_pin [get_ports {rx_fclk~CLKOUT~1~15}]
        int_ref_pin_name, _, clockout_coord = self.get_outclk_clkout_info(
            gpio_obj, devinsname2obj_map)

        #if clkout_str != "":
        #    self.clkout_str_names.add(clkout_str)

        return int_ref_pin_name, clockout_coord

    def _save_input_clkout_names(self, gpio_obj, devinsname2obj_map):

        # Save the clkout pins info
        int_ref_pin_name, clkout_str = self.get_inclk_clkout_info(
            gpio_obj, devinsname2obj_map)

        #if clkout_str != "":
        #    self.clkout_str_names.add(clkout_str)

        return int_ref_pin_name

    def _save_oe_clkout_names(self, gpio_obj, devinsname2obj_map):

        # Save the clkout pins info. OE and OUT clock name is the same
        int_ref_pin_name, clkout_str = self.get_outclk_oe_clkout_info(
            gpio_obj, devinsname2obj_map)

        #if clkout_str != "":
        #    self.clkout_str_names.add(clkout_str)

        return int_ref_pin_name

    def _is_alternate_with_constraint(self, gpio_obj):
        is_optional = False

        if gpio_obj.input is not None:
            input_cfg = gpio_obj.input

            if input_cfg.is_alternate_optional_type() or\
                    input_cfg.conn_type == input_cfg.ConnType.gctrl_conn:
                # There isn't a gctrl connection on LVDS GPIO. Rules should have trapped it
                is_optional = True

        return is_optional

    def identify_routing_delay_on_ins(self, dev_ins: str, arc_type: dev_blk.TimingArc.TimingArcType,
                                      is_to_core: bool, tscale: float, label2bank_tbl_map: Dict[str, Dict[str, float]],
                                      is_clkout: bool = False):
        # Not used at the base
        return 0

    def _write_gpio_input_mode(self, sdcfile,
                               clk2delay_map, gpio_map,
                               global_reg_table, global_comb_table,
                               devinsname2obj_map):
        '''
        Generate the table summary of the Unused GPIO configuration.

        :param sdcfile: sdc file handle that has already been opened
        :param clk2delay_map: Map of delay type to the delay value
                        (in float) for core clock network delay
        :param gpio_map: Map of device instance name to GPIOs
                that are set as input
        :param devinsname2obj_map: The map of device instance name
                to the instance object

        :return tuple in_comb, in_reg:
                in_comb: True if there are combinational path to print
                in_reg: True if there are registered path to print
        '''
        in_comb = False
        in_reg = False

        if gpio_map:

            # Get the clock network delay
            max_clk = clk2delay_map[self.DelayType.max]
            min_clk = clk2delay_map[self.DelayType.min]

            # We print the registered constraint after any
            # create clock

            for ins_name in sorted(gpio_map.keys(),
                                   key=pt_util.natural_sort_key_for_list):

                row_list = []
                # param_name = ""

                gpio_obj = gpio_map[ins_name]

                input_config = gpio_obj.input

                if input_config is None:
                    continue

                pin_name = input_config.name

                max_delay_add = 0
                min_delay_add = 0

                # Check if this instance has additional delay
                if gpio_obj.gpio_def in devinsname2obj_map:
                    ins_obj = devinsname2obj_map[gpio_obj.gpio_def]

                    max_delay_add = self.get_instance_specific_delay(
                        ins_obj, self._max_model.get_tscale())
                    min_delay_add = self.get_instance_specific_delay(
                        ins_obj, self._min_model.get_tscale())

                if not input_config.is_register:
                    # Non-registered input
                    # If it was a global clock type, then it would be
                    # a create_clock constraint

                    max_delay, min_delay = self._get_timing_delay(
                        gpio_obj, self.GPIOPathType.input)

                    # Get the relevent timing delay for the specified
                    # arc type
                    max_delay_str = self._get_delay(
                        max_delay, self._max_model.get_tscale(),
                        dev_blk.TimingArc.TimingArcType.delay)
                    min_delay_str = self._get_delay(
                        min_delay, self._min_model.get_tscale(),
                        dev_blk.TimingArc.TimingArcType.delay)

                    # print("INPUT: Max: {} min: {}".format(
                    #    max_delay_str, min_delay_str))

                    # Get the routing delay if necessary
                    routing_delay_max = self.identify_routing_delay_on_ins(
                        gpio_obj.gpio_def, dev_blk.TimingArc.TimingArcType.delay,
                        True, self._max_model.get_tscale(), 
                        self.routing_label2bank_tbl_max_map)
                    routing_delay_min = self.identify_routing_delay_on_ins(
                        gpio_obj.gpio_def, dev_blk.TimingArc.TimingArcType.delay,
                        True, self._min_model.get_tscale(), 
                        self.routing_label2bank_tbl_min_map)
                    
                    in_comb = True

                    # Check connection type and secondary function                    
                    device_db = device_dbi.DeviceDBService(self._device)
                        
                    if input_config.is_alternate_connection() and\
                        device_db.is_instance_has_clock_conn(gpio_obj.gpio_def):
                        
                        self.logger.debug("Found gclk max {} min {}".format(
                            max_delay, min_delay))

                        sdcfile.write("create_clock -period <USER_PERIOD> -name {} [get_ports {{{}}}]\n".format(
                            pin_name, pin_name))

                        row_list.append(gpio_obj.name)
                        row_list.append(pin_name)
                        row_list.append("GPIO_CLK_IN")

                        total_max_delay = max_delay_str + max_delay_add + routing_delay_max
                        total_min_delay = min_delay_str + min_delay_add + routing_delay_min
                        row_list.append("{0:.3f}".format(total_max_delay))
                        row_list.append("{0:.3f}".format(total_min_delay))

                        # comb_table.add_row(row_list)
                        global_comb_table.add_row(row_list)

                    else:

                        # PT-331: Add the non-registered constraint as a template and comment it out
                        # We just give a string representation that the user shall use
                        # to refer to the user guide (no scenario specific equations in the
                        # generated template
                        # PT-2098: Non-clock doesn't need set_input__delay
                        if not input_config.is_alternate_connection() or\
                            input_config.conn_type == input_config.ConnType.gctrl_conn:
                            sdcfile.write(
                                "# set_input_delay -clock <CLOCK> [-reference_pin <clkout_pad>] -max <MAX CALCULATION> [get_ports {{{}}}]\n".format(
                                    pin_name))

                            sdcfile.write(
                                "# set_input_delay -clock <CLOCK> [-reference_pin <clkout_pad>] -min <MIN CALCULATION> [get_ports {{{}}}]\n".format(
                                    pin_name))

                        # We print the info into Timing Report for comb path

                        row_list.append(gpio_obj.name)
                        row_list.append(pin_name)
                        row_list.append("GPIO_IN")

                        total_max_delay = max_delay_str + max_delay_add + routing_delay_max
                        total_min_delay = min_delay_str + min_delay_add + routing_delay_min
                        row_list.append("{0:.3f}".format(total_max_delay))
                        row_list.append("{0:.3f}".format(total_min_delay))

                        # comb_table.add_row(row_list)
                        global_comb_table.add_row(row_list)

                else:

                    self._write_gpio_input_register(devinsname2obj_map, sdcfile, global_reg_table,
                                                    gpio_obj, max_clk, min_clk,
                                                    max_delay_add, min_delay_add, "")
                    in_reg = True

        return in_comb, in_reg

    def _write_gpio_input_register(self, devinsname2obj_map, sdcfile, global_reg_table,
                                   gpio_obj, max_clk, min_clk,
                                   max_delay_add, min_delay_add, ref_pin_name):

        int_ref_pin_name = self._save_input_clkout_names(
            gpio_obj, devinsname2obj_map)

        # TODO: Cleanup. For now overwrite
        if ref_pin_name == "":
            ref_pin_name = int_ref_pin_name

        input_config = gpio_obj.input
        pin_name = input_config.name

        clk_name = input_config.clock_name
        clk_str = "-clock " + clk_name

        if input_config.is_clk_inverted:
            clk_name = "~" + clk_name
            clk_str = "-clock_fall " + clk_str

        # Check if it was configured in ddio mode
        ddio_normal = False

        if input_config.ddio_type is not None and \
                input_config.ddio_type != input_config.DDIOType.none:
            # pin_name is now a list for the SDC constraint. However,
            # if it was in normal mode, then [1] has clock_fall. Whereas,
            # if it was clock inverted, bit [0] would have clock_fall.
            # In resync mode, both pins have the same behavior
            if input_config.ddio_type == input_config.DDIOType.normal:
                ddio_normal = True
                pin_name = input_config.name
            else:
                pin_name = "{} {}".format(
                    input_config.name_ddio_lo, input_config.name)

        max_delay, min_delay = self._get_timing_delay(
            gpio_obj, self.GPIOPathType.input)

        # Get the relevent timing delay for the specified
        # arc type
        tco_max_delay = self._get_delay(
            max_delay, self._max_model.get_tscale(),
            dev_blk.TimingArc.TimingArcType.clk_to_q)
        tco_min_delay = self._get_delay(
            min_delay, self._min_model.get_tscale(),
            dev_blk.TimingArc.TimingArcType.clk_to_q)

        # print("input tco max: {} min: {}".format(
        #    tco_max_delay, tco_min_delay))

        routing_delay_max = self.identify_routing_delay_on_ins(
            gpio_obj.gpio_def, dev_blk.TimingArc.TimingArcType.clk_to_q,
            True,self._max_model.get_tscale(), 
            self.routing_label2bank_tbl_max_map)
        routing_delay_min = self.identify_routing_delay_on_ins(
            gpio_obj.gpio_def, dev_blk.TimingArc.TimingArcType.clk_to_q,
            True,self._min_model.get_tscale(), 
            self.routing_label2bank_tbl_min_map)

        tco_max_delay = "{0:.3f}".format(
            tco_max_delay + max_clk + (2 * max_delay_add) + routing_delay_max)
        tco_min_delay = "{0:.3f}".format(
            tco_min_delay + min_clk + (2 * min_delay_add) + routing_delay_min)

        # If it is using DDIO feature, then we need
        # to write out the DDIO pin
        # if input_config.ddio_type is not None:
        if ref_pin_name != "":
            ref_const_str = "-reference_pin [get_ports {{{}}}]".format(
                ref_pin_name)

            sdcfile.write("set_input_delay {} {} -max {} [get_ports {{{}}}]\n".format(
                clk_str, ref_const_str, tco_max_delay, pin_name))
            sdcfile.write("set_input_delay {} {} -min {} [get_ports {{{}}}]\n".format(
                clk_str, ref_const_str, tco_min_delay, pin_name))

        else:
            sdcfile.write("set_input_delay {} -max {} [get_ports {{{}}}]\n".format(
                clk_str, tco_max_delay, pin_name))
            sdcfile.write("set_input_delay {} -min {} [get_ports {{{}}}]\n".format(
                clk_str, tco_min_delay, pin_name))

        if ddio_normal:
            # Prints out the bit [1] which is opposite
            # (falling edge) of bit [0] setting for normal
            # ddio mode. But the delay value is still the same
            ddio_clk_str = "-clock " + input_config.clock_name

            if not input_config.is_clk_inverted:
                ddio_clk_str = "-clock_fall " + ddio_clk_str

            ddio_pin_name = input_config.name_ddio_lo

            # For normal mode, we need to write out the other pin constraint
            # on the opposite clock edge constraint
            if ref_pin_name != "":
                ref_const_str = "-reference_pin [get_ports {{{}}}]".format(
                    ref_pin_name)

                sdcfile.write("set_input_delay {} {} -max {} [get_ports {{{}}}]\n".format(
                    ddio_clk_str, ref_const_str, tco_max_delay, ddio_pin_name))
                sdcfile.write("set_input_delay {} {} -min {} [get_ports {{{}}}]\n".format(
                    ddio_clk_str, ref_const_str, tco_min_delay, ddio_pin_name))
            else:
                sdcfile.write("set_input_delay {} -max {} [get_ports {{{}}}]\n".format(
                    ddio_clk_str, tco_max_delay, ddio_pin_name))
                sdcfile.write("set_input_delay {} -min {} [get_ports {{{}}}]\n".format(
                    ddio_clk_str, tco_min_delay, ddio_pin_name))

        # For timing report
        tsu_in_delay = self._get_delay(
            max_delay, self._max_model.get_tscale(),
            dev_blk.TimingArc.TimingArcType.setup)
        tsu_in_min_delay = self._get_delay(
            min_delay, self._min_model.get_tscale(),
            dev_blk.TimingArc.TimingArcType.setup)

        thold_in_max_delay = self._get_delay(
            max_delay, self._max_model.get_tscale(),
            dev_blk.TimingArc.TimingArcType.hold)
        thold_in_delay = self._get_delay(
            min_delay, self._min_model.get_tscale(),
            dev_blk.TimingArc.TimingArcType.hold)

               
        tmp_list = []
        tmp_list.append(gpio_obj.name)
        tmp_list.append(clk_name)
        tmp_list.append("{0:.3f}".format(
            tsu_in_delay - max_delay_add - max_clk))
        tmp_list.append("{0:.3f}".format(
            tsu_in_min_delay - min_delay_add - min_clk))
        tmp_list.append("{0:.3f}".format(
            thold_in_max_delay + max_delay_add + max_clk))
        tmp_list.append("{0:.3f}".format(
            thold_in_delay + min_delay_add + min_clk))
        tmp_list.append("")
        tmp_list.append("")

        global_reg_table.add_row(tmp_list)

    def print_map(self, delay_map):

        for key, value in delay_map.items():
            print("{}: {}".format(key, value))

    def _write_gpio_output_mode(self, sdcfile,
                                clk2delay_map, gpio_map,
                                global_reg_table, global_comb_table,
                                devinsname2obj_map):
        '''
        Generate the table summary of the Unused GPIO configuration.

        :param gpio_map: Map of device instance name to GPIOs
                that are set as output
        :param devinsname2obj_map: The map of device instance name
                to the instance object

        :return tuple out_comb, out_reg:
                out_comb: True if there are combinational path to print
                out_reg: True if there are registered path to print

        '''
        out_comb = False
        out_reg = False

        if gpio_map:

            # Get the clock network delay
            max_clk = clk2delay_map[self.DelayType.max]
            min_clk = clk2delay_map[self.DelayType.min]

            for ins_name in sorted(gpio_map.keys(),
                                   key=pt_util.natural_sort_key_for_list):

                row_list = []
                out_tied = False

                gpio_obj = gpio_map[ins_name]

                output_config = gpio_obj.output

                if output_config is None:
                    continue

                # If the output is tied, then there's no constraint to
                # print in sdc file
                if output_config.tied_option == output_config.TiedOptType.vcc or\
                        output_config.tied_option == output_config.TiedOptType.gnd:
                    out_tied = True

                max_delay, min_delay = self._get_timing_delay(
                    gpio_obj, self.GPIOPathType.output)

                # Non-registered output
                max_delay_str = self._get_delay(
                    max_delay, self._max_model.get_tscale(),
                    dev_blk.TimingArc.TimingArcType.delay)

                min_delay_str = self._get_delay(
                    min_delay, self._min_model.get_tscale(),
                    dev_blk.TimingArc.TimingArcType.delay)

                pin_name = output_config.name

                # Check if this instance has additional delay
                max_delay_add, min_delay_add = self.get_output_additional_delay(
                    gpio_obj, devinsname2obj_map)
               
                if output_config.register_option == output_config.RegOptType.none:

                    # Get the routing delay if necessary
                    routing_delay_max = self.identify_routing_delay_on_ins(
                        gpio_obj.gpio_def, dev_blk.TimingArc.TimingArcType.delay,
                        False, self._max_model.get_tscale(),
                        self.routing_label2bank_tbl_max_map)
                    routing_delay_min = self.identify_routing_delay_on_ins(
                        gpio_obj.gpio_def, dev_blk.TimingArc.TimingArcType.delay,
                        False, self._min_model.get_tscale(),
                        self.routing_label2bank_tbl_min_map)
                    
                    row_list.append(gpio_obj.name)
                    row_list.append(pin_name)
                    row_list.append("GPIO_OUT")

                    row_list.append("{0:.3f}".format(
                        max_delay_str + max_delay_add + routing_delay_max))
                    row_list.append("{0:.3f}".format(
                        min_delay_str + min_delay_add + routing_delay_min))

                    # comb_table.add_row(row_list)
                    global_comb_table.add_row(row_list)
                    out_comb = True

                    # If it was tied, we don't print constraint but we just give
                    # the delay value in report
                    if not out_tied:
                        # PT-331: Add the non-registered constraint as a template and comment it out
                        # We just give a string representation that the user shall use
                        # to refer to the user guide (no scenario specific equations in the
                        # generated template
                        sdcfile.write("# set_output_delay -clock <CLOCK> [-reference_pin <clkout_pad>] -max <MAX CALCULATION> [get_ports {{{}}}]\n".format(
                            pin_name))
                        sdcfile.write("# set_output_delay -clock <CLOCK> [-reference_pin <clkout_pad>] -min <MIN CALCULATION> [get_ports {{{}}}]\n".format(
                            pin_name))

                else:

                    self._write_gpio_output_register(devinsname2obj_map, sdcfile, global_reg_table,
                                                     gpio_obj, max_clk, min_clk,
                                                     max_delay_add, min_delay_add, max_delay,
                                                     min_delay, out_tied, "")
                    out_reg = True

        return out_comb, out_reg

    def get_output_additional_delay(self, gpio_obj, devinsname2obj_map):
        max_delay_add = 0
        min_delay_add = 0

        # Check if this instance has additional delay
        if gpio_obj.gpio_def in devinsname2obj_map:
            ins_obj = devinsname2obj_map[gpio_obj.gpio_def]

            max_delay_add = self.get_instance_specific_delay(
                ins_obj, self._max_model.get_tscale())
            min_delay_add = self.get_instance_specific_delay(
                ins_obj, self._min_model.get_tscale())

        return max_delay_add, min_delay_add

    def get_clkout_additional_delay(self, gpio_obj, devinsname2obj_map):
        return 0.0, 0.0

    def _write_gpio_output_register(self, devinsname2obj_map, sdcfile, global_reg_table,
                                    gpio_obj, max_clk, min_clk,
                                    max_delay_add, min_delay_add, max_delay,
                                    min_delay, out_tied, ref_pin_name):

        int_ref_pin_name, _ = self._save_output_clkout_names(
            gpio_obj, devinsname2obj_map)

        # TODO: cleanup. for now overwrite
        if ref_pin_name == "":
            ref_pin_name = int_ref_pin_name

        output_config = gpio_obj.output
        pin_name = output_config.name

        ddio_normal = False

        clk_name = output_config.clock_name
        clk_str = "-clock " + clk_name

        if output_config.is_clk_inverted:
            clk_name = "~" + clk_name
            clk_str = "-clock_fall " + clk_str

        # Check if it was configured in ddio mode
        if output_config.ddio_type is not None and \
                output_config.ddio_type != output_config.DDIOType.none:

            if output_config.ddio_type == output_config.DDIOType.normal:
                ddio_normal = True
                pin_name = output_config.name
            else:
                # pin_name is now a list for the SDC constraint
                pin_name = "{} {}".format(
                    output_config.name_ddio_lo, output_config.name)

        # sdc template
        setup_max_delay = self._get_delay(
            max_delay, self._max_model.get_tscale(),
            dev_blk.TimingArc.TimingArcType.setup)

        hold_min_delay = self._get_delay(
            min_delay, self._min_model.get_tscale(),
            dev_blk.TimingArc.TimingArcType.hold)

        # Get the routing delay if necessary
        routing_delay_tsu = self.identify_routing_delay_on_ins(
            gpio_obj.gpio_def, dev_blk.TimingArc.TimingArcType.setup,
            False, self._max_model.get_tscale(),
            self.routing_label2bank_tbl_max_map)
        routing_delay_thold = self.identify_routing_delay_on_ins(
            gpio_obj.gpio_def, dev_blk.TimingArc.TimingArcType.hold,
            False, self._min_model.get_tscale(),
            self.routing_label2bank_tbl_min_map)

        # -max: (GPIO_OUT_PAD_TSU) - CORE CLK (max)
        max_out_delay = "{0:.3f}".format((setup_max_delay + routing_delay_tsu) - max_clk)

        # -min: (GPIO_OUT_PAD_THOLD) + CORE CLK (min)
        min_out_delay = "{0:.3f}".format((-1 * hold_min_delay) + routing_delay_thold - min_clk)

        # print("MAX: {} MIN: {}".format(max_out_delay, min_out_delay))

        # PT-692: Only print constraint if not_tied ?
        if not out_tied:
            if ref_pin_name != "":
                ref_const_str = "-reference_pin [get_ports {{{}}}]".format(
                    ref_pin_name)

                sdcfile.write("set_output_delay {} {} -max {} [get_ports {{{}}}]\n".format(
                    clk_str, ref_const_str, max_out_delay, pin_name))
                sdcfile.write("set_output_delay {} {} -min {} [get_ports {{{}}}]\n".format(
                    clk_str, ref_const_str, min_out_delay, pin_name))

            else:
                sdcfile.write("set_output_delay {} -max {} [get_ports {{{}}}]\n".format(
                    clk_str, max_out_delay, pin_name))
                sdcfile.write("set_output_delay {} -min {} [get_ports {{{}}}]\n".format(
                    clk_str, min_out_delay, pin_name))

            if ddio_normal:
                ddio_clk_str = "-clock " + output_config.clock_name

                if not output_config.is_clk_inverted:
                    ddio_clk_str = "-clock_fall " + ddio_clk_str

                ddio_pin_name = output_config.name_ddio_lo

                if ref_pin_name != "":
                    ref_const_str = "-reference_pin [get_ports {{{}}}]".format(
                        ref_pin_name)

                    sdcfile.write("set_output_delay {} {} -max {} [get_ports {{{}}}]\n".format(
                        ddio_clk_str, ref_const_str, max_out_delay, ddio_pin_name))
                    sdcfile.write("set_output_delay {} {} -min {} [get_ports {{{}}}]\n".format(
                        ddio_clk_str, ref_const_str, min_out_delay, ddio_pin_name))

                else:
                    sdcfile.write("set_output_delay {} -max {} [get_ports {{{}}}]\n".format(
                        ddio_clk_str, max_out_delay, ddio_pin_name))
                    sdcfile.write("set_output_delay {} -min {} [get_ports {{{}}}]\n".format(
                        ddio_clk_str, min_out_delay, ddio_pin_name))

        # Report file
        tco_max_delay = self._get_delay(
            max_delay, self._max_model.get_tscale(),
            dev_blk.TimingArc.TimingArcType.clk_to_q)

        tco_min_delay = self._get_delay(
            min_delay, self._min_model.get_tscale(),
            dev_blk.TimingArc.TimingArcType.clk_to_q)
   
        # print("output Max: {}, min: {}".format(tco_max_delay, tco_min_delay))
        tmp_list = []
        tmp_list.append(gpio_obj.name)
        tmp_list.append(clk_name)
        tmp_list.append("")  # Setup Max
        tmp_list.append("")  # Setup Min
        tmp_list.append("")  # Hold Max
        tmp_list.append("")  # Hold Min
        tmp_list.append("{0:.3f}".format(
            tco_max_delay + max_clk + max_delay_add))
        tmp_list.append("{0:.3f}".format(
            tco_min_delay + min_clk + min_delay_add))

        global_reg_table.add_row(tmp_list)

    def _write_gpio_inout_mode(self, sdcfile,
                               clk2delay_map, gpio_map,
                               global_reg_table, global_comb_table,
                               devinsname2obj_map):
        '''
        Generate the table summary of the Unused GPIO configuration.

        :param gpio_map: Map of device instance name to GPIOs
                that are set as inout
        :param devinsname2obj_map: The map of device instance name
                to the instance object

        :return tuple inout_comb, inout_reg:
                inout_comb: True if there are combinational path to print
                inout_reg: True if there are registered path to print
        '''
        inout_comb = False
        inout_reg = False

        if gpio_map:

            # Get the clock network delay
            max_clk = clk2delay_map[self.DelayType.max]
            min_clk = clk2delay_map[self.DelayType.min]

            for ins_name in sorted(gpio_map.keys(),
                                   key=pt_util.natural_sort_key_for_list):

                is_ins_reg = False

                row_list = []
                gpio_obj = gpio_map[ins_name]

                # Default settings for cases where the path does not apply
                in_clk_name = ""
                out_clk_name = ""

                in_tsu = 0
                in_min_tsu = 0
                in_max_thold = 0
                in_thold = 0
                out_tco_min = 0
                out_tco_max = 0
                oe_tco_min = 0
                oe_tco_max = 0

                in_max_map, in_min_map = self._get_timing_delay(
                    gpio_obj, self.GPIOPathType.input)

                out_max_map, out_min_map = self._get_timing_delay(
                    gpio_obj, self.GPIOPathType.output)

                oe_max_map, oe_min_map = self._get_timing_delay(
                    gpio_obj, self.GPIOPathType.oe)

                # Get the GPIO_PAD_TO_IN arc
                clkin_max_delay = self._get_delay(
                    in_max_map, self._max_model.get_tscale(),
                    dev_blk.TimingArc.TimingArcType.delay)
                clkin_min_delay = self._get_delay(
                    in_min_map, self._min_model.get_tscale(),
                    dev_blk.TimingArc.TimingArcType.delay)

                # Get the GPIO_OUT_TO_PAD arc
                clkout_max_delay = self._get_delay(
                    out_max_map, self._max_model.get_tscale(),
                    dev_blk.TimingArc.TimingArcType.delay)
                clkout_min_delay = self._get_delay(
                    out_min_map, self._min_model.get_tscale(),
                    dev_blk.TimingArc.TimingArcType.delay)

                max_delay_add = 0
                min_delay_add = 0

                # Check if this instance has additional delay
                if gpio_obj.gpio_def in devinsname2obj_map:
                    ins_obj = devinsname2obj_map[gpio_obj.gpio_def]

                    max_delay_add = self.get_instance_specific_delay(
                        ins_obj, self._max_model.get_tscale())
                    min_delay_add = self.get_instance_specific_delay(
                        ins_obj, self._min_model.get_tscale())

                # We do input
                input_config = gpio_obj.input

                if input_config is not None and input_config.name != "":

                    pin_name = input_config.name

                    if not input_config.is_register:

                        # Check connection type and secondary function
                        device_db = device_dbi.DeviceDBService(
                                self._device)
                            
                        # Get the routing delay if necessary
                        routing_delay_max = self.identify_routing_delay_on_ins(
                            gpio_obj.gpio_def, dev_blk.TimingArc.TimingArcType.delay,
                            True, self._max_model.get_tscale(),
                            self.routing_label2bank_tbl_max_map)
                        routing_delay_min = self.identify_routing_delay_on_ins(
                            gpio_obj.gpio_def, dev_blk.TimingArc.TimingArcType.delay,
                            True, self._min_model.get_tscale(),
                            self.routing_label2bank_tbl_min_map)    
                                                
                        if input_config.is_alternate_connection() and\
                            device_db.is_instance_has_clock_conn(gpio_obj.gpio_def):

                            sdcfile.write("create_clock -period <USER_PERIOD> -name {} [get_ports {{{}}}]\n".format(
                                pin_name, pin_name))

                            row_list = []
                            row_list.append(gpio_obj.name)
                            row_list.append(pin_name)
                            row_list.append("GPIO_CLK_IN")  
                            
                            total_max_delay = clkin_max_delay + max_delay_add + routing_delay_max
                            total_min_delay = clkin_min_delay + min_delay_add + routing_delay_min

                            row_list.append("{0:.3f}".format(total_max_delay))
                            row_list.append("{0:.3f}".format(total_min_delay))

                            # comb_table.add_row(row_list)
                            global_comb_table.add_row(row_list)

                        elif input_config.name != "":
                            # Normal combination path
                            # PT-331: Add the non-registered constraint as a template and comment it out
                            if not input_config.is_alternate_connection() or\
                                input_config.conn_type == input_config.ConnType.gctrl_conn:
                                sdcfile.write("# set_input_delay -clock <CLOCK> [-reference_pin <clkout_pad>] -max <MAX CALCULATION> [get_ports {{{}}}]\n".format(
                                    pin_name))
                                sdcfile.write("# set_input_delay -clock <CLOCK> [-reference_pin <clkout_pad>] -min <MIN CALCULATION> [get_ports {{{}}}]\n".format(
                                    pin_name))

                            # We print the info into Timing Report for comb path
                            comb_row_list = []
                            comb_row_list.append(gpio_obj.name)
                            comb_row_list.append(pin_name)
                            comb_row_list.append("GPIO_IN")

                            total_max_delay = clkin_max_delay + max_delay_add + routing_delay_max
                            total_min_delay = clkin_min_delay + min_delay_add + routing_delay_min

                            comb_row_list.append("{0:.3f}".format(total_max_delay))
                            comb_row_list.append("{0:.3f}".format(total_min_delay))

                            global_comb_table.add_row(comb_row_list)
                            inout_comb = True

                    else:
                        in_clk_name, in_tsu, in_thold, in_min_tsu, in_max_thold = self._write_gpio_inout_input_register(
                            devinsname2obj_map, sdcfile, gpio_obj, max_clk,
                            min_clk, max_delay_add, min_delay_add,
                            in_max_map, in_min_map, "")

                        is_ins_reg = True
                        inout_reg = True

                # We do output
                output_config = gpio_obj.output

                # For Trion, this is the instance specific delay
                # For titanium, this is the additional delay for HVIO
                out_max_extra_delay, out_min_extra_delay = self.get_output_additional_delay(
                    gpio_obj, devinsname2obj_map)
                if output_config is not None and output_config.name != "":

                    pin_name = output_config.name
                    out_tied = False

                    # If the output is tied, then there's no constraint to
                    # print in sdc file
                    if output_config.tied_option == output_config.TiedOptType.vcc or\
                            output_config.tied_option == output_config.TiedOptType.gnd:
                        out_tied = True

                    if output_config is not None and\
                            output_config.register_option == output_config.RegOptType.none:

                        inout_comb = True

                        # Get the routing delay if necessary
                        routing_delay_max = self.identify_routing_delay_on_ins(
                            gpio_obj.gpio_def, dev_blk.TimingArc.TimingArcType.delay,
                            False, self._max_model.get_tscale(),
                            self.routing_label2bank_tbl_max_map)
                        routing_delay_min = self.identify_routing_delay_on_ins(
                            gpio_obj.gpio_def, dev_blk.TimingArc.TimingArcType.delay,
                            False, self._min_model.get_tscale(),
                            self.routing_label2bank_tbl_min_map) 
                        
                        if not out_tied:

                            # PT-331: Add the non-registered constraint as a template and comment it out
                            sdcfile.write("# set_output_delay -clock <CLOCK> [-reference_pin <clkout_pad>] -max <MAX CALCULATION> [get_ports {{{}}}]\n".format(
                                pin_name))
                            sdcfile.write("# set_output_delay -clock <CLOCK> [-reference_pin <clkout_pad>] -min <MIN CALCULATION> [get_ports {{{}}}]\n".format(
                                pin_name))

                        comb_row_list = []
                        comb_row_list.append(gpio_obj.name)
                        comb_row_list.append(pin_name)
                        comb_row_list.append("GPIO_OUT")

                        comb_row_list.append("{0:.3f}".format(
                            clkout_max_delay + out_max_extra_delay + routing_delay_max))
                        comb_row_list.append("{0:.3f}".format(
                            clkout_min_delay + out_min_extra_delay + routing_delay_min))

                        # comb_table.add_row(comb_row_list)

                        global_comb_table.add_row(comb_row_list)

                    else:
                        out_clk_name, out_tco_max, out_tco_min = self._write_gpio_inout_output_register(
                            devinsname2obj_map, sdcfile, gpio_obj, max_clk,
                            min_clk, out_max_extra_delay, out_min_extra_delay,
                            out_max_map, out_min_map, out_tied, "")

                        is_ins_reg = True
                        inout_reg = True

                # We do OE
                oe_config = gpio_obj.output_enable

                if oe_config is not None and oe_config.name != "":

                    if oe_config is not None and not oe_config.is_register:

                        pin_name = oe_config.name

                        # Non-registered output
                        oe_max_delay = self._get_delay(
                            oe_max_map, self._max_model.get_tscale(),
                            dev_blk.TimingArc.TimingArcType.delay)

                        oe_min_delay = self._get_delay(
                            oe_min_map, self._min_model.get_tscale(),
                            dev_blk.TimingArc.TimingArcType.delay)
                        
                        # Get the routing delay if necessary
                        routing_delay_max = self.identify_routing_delay_on_ins(
                            gpio_obj.gpio_def, dev_blk.TimingArc.TimingArcType.delay,
                            False, self._max_model.get_tscale(),
                            self.routing_label2bank_tbl_max_map)
                        routing_delay_min = self.identify_routing_delay_on_ins(
                            gpio_obj.gpio_def, dev_blk.TimingArc.TimingArcType.delay,
                            False, self._min_model.get_tscale(),
                            self.routing_label2bank_tbl_min_map)                         

                        # PT-331: Add the non-registered constraint as a template and comment it out
                        sdcfile.write("# set_output_delay -clock <CLOCK> [-reference_pin <clkout_pad>] -max <MAX CALCULATION> [get_ports {{{}}}]\n".format(
                            pin_name))
                        sdcfile.write("# set_output_delay -clock <CLOCK> [-reference_pin <clkout_pad>] -min <MIN CALCULATION> [get_ports {{{}}}]\n".format(
                            pin_name))

                        comb_row_list = []
                        comb_row_list.append(gpio_obj.name)
                        comb_row_list.append(pin_name)
                        comb_row_list.append("GPIO_OUT")

                        comb_row_list.append("{0:.3f}".format(
                            oe_max_delay + max_delay_add + routing_delay_max))
                        comb_row_list.append("{0:.3f}".format(
                            oe_min_delay + min_delay_add + routing_delay_min))

                        # comb_table.add_row(comb_row_list)
                        global_comb_table.add_row(comb_row_list)
                        inout_comb = True
                    else:

                        out_clk_name, oe_tco_max, oe_tco_min = self._write_gpio_inout_oe_register(
                            devinsname2obj_map, sdcfile, gpio_obj, max_clk,
                            min_clk, max_delay_add, min_delay_add,
                            oe_max_map, oe_min_map, "")

                        is_ins_reg = True
                        inout_reg = True

                if is_ins_reg:

                    tmp_list = []
                    # Do input first
                    if in_clk_name != "":
                        tmp_list.append(gpio_obj.name)
                        tmp_list.append(in_clk_name)
                        tmp_list.append("{0:.3f}".format(in_tsu))
                        tmp_list.append("{0:.3f}".format(in_min_tsu))
                        tmp_list.append("{0:.3f}".format(in_max_thold))
                        tmp_list.append("{0:.3f}".format(in_thold))
                        tmp_list.append("")
                        tmp_list.append("")

                        global_reg_table.add_row(tmp_list)
                        tmp_list = []

                    if out_clk_name != "":
                        tmp_list.append(gpio_obj.name)
                        tmp_list.append(out_clk_name)
                        tmp_list.append("")  # Setup
                        tmp_list.append("")
                        tmp_list.append("")  # Hold
                        tmp_list.append("")

                        if out_tco_max != 0 and oe_tco_max != 0:
                            # Get the max between the two
                            max_tco = max(out_tco_max, oe_tco_max)
                            min_tco = min(out_tco_min, oe_tco_min)

                            tmp_list.append("{0:.3f}".format(max_tco))
                            tmp_list.append("{0:.3f}".format(min_tco))
                        elif out_tco_max != 0:
                            tmp_list.append("{0:.3f}".format(out_tco_max))
                            tmp_list.append("{0:.3f}".format(out_tco_min))
                        elif oe_tco_max != 0:
                            tmp_list.append("{0:.3f}".format(oe_tco_max))
                            tmp_list.append("{0:.3f}".format(oe_tco_min))
                        else:
                            tmp_list.append("")
                            tmp_list.append("")

                        global_reg_table.add_row(tmp_list)

        return inout_comb, inout_reg

    def _write_gpio_inout_input_register(self, devinsname2obj_map, sdcfile, gpio_obj, max_clk,
                                         min_clk, max_delay_add, min_delay_add,
                                         in_max_map, in_min_map, ref_pin_name):
        int_ref_pin_name = self._save_input_clkout_names(
            gpio_obj, devinsname2obj_map)

        # TODO: cleanup
        if ref_pin_name == "":
            ref_pin_name = int_ref_pin_name

        input_config = gpio_obj.input
        pin_name = input_config.name

        ddio_normal = False

        in_clk_name = input_config.clock_name
        clk_str = "-clock " + in_clk_name

        if input_config.is_clk_inverted:
            in_clk_name = "~" + in_clk_name
            clk_str = "-clock_fall " + clk_str

        # Check if it was configured in ddio mode
        if input_config.ddio_type is not None and \
                input_config.ddio_type != input_config.DDIOType.none:
            if input_config.ddio_type == input_config.DDIOType.normal:
                ddio_normal = True
                pin_name = input_config.name
            else:
                # pin_name is now a list for the SDC constraint
                pin_name = "{} {}".format(
                    input_config.name_ddio_lo, input_config.name)

        # For sdc template
        tco_max_delay = self._get_delay(
            in_max_map, self._max_model.get_tscale(),
            dev_blk.TimingArc.TimingArcType.clk_to_q)
        tco_min_delay = self._get_delay(
            in_min_map, self._min_model.get_tscale(),
            dev_blk.TimingArc.TimingArcType.clk_to_q)

        # Get the routing delay if necessary
        routing_delay_max = self.identify_routing_delay_on_ins(
            gpio_obj.gpio_def, dev_blk.TimingArc.TimingArcType.clk_to_q,
            True, self._max_model.get_tscale(),
            self.routing_label2bank_tbl_max_map)
        routing_delay_min = self.identify_routing_delay_on_ins(
            gpio_obj.gpio_def, dev_blk.TimingArc.TimingArcType.clk_to_q,
            True, self._min_model.get_tscale(),
            self.routing_label2bank_tbl_min_map) 

        tco_max_delay = "{0:.3f}".format(
            tco_max_delay + max_clk + (2 * max_delay_add) + routing_delay_max)
        tco_min_delay = "{0:.3f}".format(
            tco_min_delay + min_clk + (2 * min_delay_add) + routing_delay_min)

        if ref_pin_name != "":
            ref_const_str = "-reference_pin [get_ports {{{}}}]".format(
                ref_pin_name)

            sdcfile.write("set_input_delay {} {} -max {} [get_ports {{{}}}]\n".format(
                clk_str, ref_const_str, tco_max_delay, pin_name))
            sdcfile.write("set_input_delay {} {} -min {} [get_ports {{{}}}]\n".format(
                clk_str, ref_const_str, tco_min_delay, pin_name))

        else:
            sdcfile.write("set_input_delay {} -max {} [get_ports {{{}}}]\n".format(
                clk_str, tco_max_delay, pin_name))
            sdcfile.write("set_input_delay {} -min {} [get_ports {{{}}}]\n".format(
                clk_str, tco_min_delay, pin_name))

        if ddio_normal:
            ddio_clk_str = "-clock " + input_config.clock_name

            if not input_config.is_clk_inverted:
                ddio_clk_str = "-clock_fall " + clk_str

            ddio_pin_name = input_config.name_ddio_lo

            if ref_pin_name != "":
                ref_const_str = "-reference_pin [get_ports {{{}}}]".format(
                    ref_pin_name)

                sdcfile.write("set_input_delay {} {} -max {} [get_ports {{{}}}]\n".format(
                    ddio_clk_str, ref_const_str, tco_max_delay, ddio_pin_name))
                sdcfile.write("set_input_delay {} {} -min {} [get_ports {{{}}}]\n".format(
                    ddio_clk_str, ref_const_str, tco_min_delay, ddio_pin_name))

            else:
                sdcfile.write("set_input_delay {} -max {} [get_ports {{{}}}]\n".format(
                    ddio_clk_str, tco_max_delay, ddio_pin_name))
                sdcfile.write("set_input_delay {} -min {} [get_ports {{{}}}]\n".format(
                    ddio_clk_str, tco_min_delay, ddio_pin_name))

        # For timing report
        in_tsu = self._get_delay(
            in_max_map, self._max_model.get_tscale(),
            dev_blk.TimingArc.TimingArcType.setup)
        in_min_tsu = self._get_delay(
            in_min_map, self._min_model.get_tscale(),
            dev_blk.TimingArc.TimingArcType.setup)

        in_thold = self._get_delay(
            in_min_map, self._min_model.get_tscale(),
            dev_blk.TimingArc.TimingArcType.hold)
        in_max_thold = self._get_delay(
            in_max_map, self._max_model.get_tscale(),
            dev_blk.TimingArc.TimingArcType.hold)
                
        # TODO: Verify routing delay + or -
        in_tsu = in_tsu - max_delay_add - max_clk
        in_thold = in_thold + min_delay_add + min_clk

        in_min_tsu = in_min_tsu - min_delay_add - min_clk
        in_max_thold = in_max_thold + max_delay_add + max_clk

        return in_clk_name, in_tsu, in_thold, in_min_tsu, in_max_thold

    def _write_gpio_inout_output_register(self, devinsname2obj_map, sdcfile, gpio_obj, max_clk,
                                          min_clk, max_delay_add, min_delay_add,
                                          out_max_map, out_min_map, out_tied, ref_pin_name):

        int_ref_pin_name, _ = self._save_output_clkout_names(
            gpio_obj, devinsname2obj_map)

        # TODO: cleanup
        if ref_pin_name == "":
            ref_pin_name = int_ref_pin_name

        output_config = gpio_obj.output
        pin_name = output_config.name

        ddio_normal = False

        out_clk_name = output_config.clock_name
        clk_str = "-clock " + out_clk_name

        if output_config.is_clk_inverted:
            out_clk_name = "~" + out_clk_name
            clk_str = "-clock_fall " + clk_str

        # Check if it was configured in ddio mode
        if output_config.ddio_type is not None and \
                output_config.ddio_type != output_config.DDIOType.none:
            # pin_name is now a list for the SDC constraint
            if output_config.ddio_type == output_config.DDIOType.normal:
                ddio_normal = True
                pin_name = output_config.name
            else:
                pin_name = "{} {}".format(
                    output_config.name_ddio_lo, output_config.name)

        # sdc template
        setup_max_delay = self._get_delay(
            out_max_map, self._max_model.get_tscale(),
            dev_blk.TimingArc.TimingArcType.setup)

        hold_min_delay = self._get_delay(
            out_min_map, self._min_model.get_tscale(),
            dev_blk.TimingArc.TimingArcType.hold)

        # Get the routing delay if necessary
        routing_delay_tsu_max = self.identify_routing_delay_on_ins(
            gpio_obj.gpio_def, dev_blk.TimingArc.TimingArcType.setup,
            False, self._max_model.get_tscale(),
            self.routing_label2bank_tbl_max_map)
        routing_delay_thold_min = self.identify_routing_delay_on_ins(
            gpio_obj.gpio_def, dev_blk.TimingArc.TimingArcType.hold,
            False, self._min_model.get_tscale(),
            self.routing_label2bank_tbl_min_map) 
        
        # TODO: This equation is different than the one used
        # in output mode. Changed based on PT-331
        # -max: max(GPIO_OUT_PAD_TSU) - CORE CLK (max)
        max_out_delay = "{0:.3f}".format(setup_max_delay + routing_delay_tsu_max - max_clk)

        # -min: min(GPIO_OUT_PAD_THOLD) - CORE CLK (min)
        min_out_delay = "{0:.3f}".format((-1 * hold_min_delay) + routing_delay_thold_min - min_clk )

        if not out_tied:
            if ref_pin_name != "":
                ref_const_str = "-reference_pin [get_ports {{{}}}]".format(
                    ref_pin_name)

                sdcfile.write("set_output_delay {} {} -max {} [get_ports {{{}}}]\n".format(
                    clk_str, ref_const_str, max_out_delay, pin_name))
                sdcfile.write("set_output_delay {} {} -min {} [get_ports {{{}}}]\n".format(
                    clk_str, ref_const_str, min_out_delay, pin_name))

            else:
                sdcfile.write("set_output_delay {} -max {} [get_ports {{{}}}]\n".format(
                    clk_str, max_out_delay, pin_name))
                sdcfile.write("set_output_delay {} -min {} [get_ports {{{}}}]\n".format(
                    clk_str, min_out_delay, pin_name))

        if ddio_normal:
            ddio_clk_str = "-clock " + output_config.clock_name

            if not output_config.is_clk_inverted:
                ddio_clk_str = "-clock_fall " + ddio_clk_str

            ddio_pin_name = output_config.name_ddio_lo

            if not out_tied:
                if ref_pin_name != "":
                    ref_const_str = "-reference_pin [get_ports {{{}}}]".format(
                        ref_pin_name)

                    sdcfile.write("set_output_delay {} {} -max {} [get_ports {{{}}}]\n".format(
                        ddio_clk_str, ref_const_str, max_out_delay, ddio_pin_name))
                    sdcfile.write("set_output_delay {} {} -min {} [get_ports {{{}}}]\n".format(
                        ddio_clk_str, ref_const_str, min_out_delay, ddio_pin_name))

                else:
                    sdcfile.write("set_output_delay {} -max {} [get_ports {{{}}}]\n".format(
                        ddio_clk_str, max_out_delay, ddio_pin_name))
                    sdcfile.write("set_output_delay {} -min {} [get_ports {{{}}}]\n".format(
                        ddio_clk_str, min_out_delay, ddio_pin_name))

        # Report file
        tco_max_delay = self._get_delay(
            out_max_map, self._max_model.get_tscale(),
            dev_blk.TimingArc.TimingArcType.clk_to_q)

        tco_min_delay = self._get_delay(
            out_min_map, self._min_model.get_tscale(),
            dev_blk.TimingArc.TimingArcType.clk_to_q)
       
        out_tco_max = tco_max_delay + max_clk + max_delay_add
        out_tco_min = tco_min_delay + min_clk + min_delay_add

        return out_clk_name, out_tco_max, out_tco_min

    def _write_gpio_inout_oe_register(self, devinsname2obj_map, sdcfile, gpio_obj, max_clk,
                                      min_clk, max_delay_add, min_delay_add,
                                      oe_max_map, oe_min_map, ref_pin_name):

        int_ref_pin_name = self._save_oe_clkout_names(
            gpio_obj, devinsname2obj_map)

        # TODO: cleanup
        if ref_pin_name == "":
            ref_pin_name = int_ref_pin_name

        oe_config = gpio_obj.output_enable
        pin_name = oe_config.name

        out_clk_name = oe_config.clock_name
        clk_str = "-clock " + out_clk_name

        if oe_config.is_clk_inverted:
            out_clk_name = "~" + out_clk_name
            clk_str = "-clock_fall " + clk_str

        # sdc template
        setup_max_delay = self._get_delay(
            oe_max_map, self._max_model.get_tscale(),
            dev_blk.TimingArc.TimingArcType.setup)

        hold_min_delay = self._get_delay(
            oe_min_map, self._min_model.get_tscale(),
            dev_blk.TimingArc.TimingArcType.hold)

        # Get the routing delay if necessary
        routing_delay_tsu_max = self.identify_routing_delay_on_ins(
            gpio_obj.gpio_def, dev_blk.TimingArc.TimingArcType.setup,
            False, self._max_model.get_tscale(),
            self.routing_label2bank_tbl_max_map)
        routing_delay_thold_min = self.identify_routing_delay_on_ins(
            gpio_obj.gpio_def, dev_blk.TimingArc.TimingArcType.hold,
            False, self._min_model.get_tscale(),
            self.routing_label2bank_tbl_min_map) 
        
        # -max: max(GPIO_OUT_PAD_TSU) - CORE CLK (max)
        max_out_delay = "{0:.3f}".format(setup_max_delay + routing_delay_tsu_max - max_clk)

        # -min: min(GPIO_OUT_PAD_THOLD) - CORE CLK (min)
        min_out_delay = "{0:.3f}".format((-1 * hold_min_delay) + routing_delay_thold_min - min_clk)

        if ref_pin_name != "":
            ref_const_str = "-reference_pin [get_ports {{{}}}]".format(
                ref_pin_name)

            sdcfile.write("set_output_delay {} {} -max {} [get_ports {{{}}}]\n".format(
                clk_str, ref_const_str, max_out_delay, pin_name))
            sdcfile.write("set_output_delay {} {} -min {} [get_ports {{{}}}]\n".format(
                clk_str, ref_const_str, min_out_delay, pin_name))

        else:
            sdcfile.write("set_output_delay {} -max {} [get_ports {{{}}}]\n".format(
                clk_str, max_out_delay, pin_name))
            sdcfile.write("set_output_delay {} -min {} [get_ports {{{}}}]\n".format(
                clk_str, min_out_delay, pin_name))

        # Rerport
        tco_max_delay = self._get_delay(
            oe_max_map, self._max_model.get_tscale(),
            dev_blk.TimingArc.TimingArcType.clk_to_q)

        tco_min_delay = self._get_delay(
            oe_min_map, self._min_model.get_tscale(),
            dev_blk.TimingArc.TimingArcType.clk_to_q)
        
        oe_tco_max = tco_max_delay + max_clk + max_delay_add
        oe_tco_min = tco_min_delay + min_clk + min_delay_add

        return out_clk_name, oe_tco_max, oe_tco_min

    def _get_timing_delay(self, gpio_obj, path_type):
        '''
        Based on the object, it gets the required delay.

        :param gpio_obj: The gpio object

        :return the delay value represented in tuple max,min
        '''
        # Get the max model
        # max_model = timing_comb.get_model(tmodel.Model.AnalysisType.max)
        max_delay = self._parse_timing_table(
            self._max_model, gpio_obj, path_type, True)

        # Get the min model
        # min_model = timing_comb.get_model(tmodel.Model.AnalysisType.min)
        min_delay = self._parse_timing_table(
            self._min_model, gpio_obj, path_type, False)

        return max_delay, min_delay

    def _parse_timing_table(self, table_model,
                            gpio_obj, path_type, is_max):
        '''
        :param table_model: a Model object
        :param gpio_obj: GPIO design object
        :param path_type: GPIOPathType
        :param is_max: Boolean that indicates if it is trying
                to read max or min model
        :return a map of arc to the delay value
        '''

        arc2delay_map = {}

        try:
            tree = et.ElementTree(file=table_model.get_filename())
            # print("REading file: {}".format(table_model.get_filename()))

            block_tag = ".//" + self.xml_ns + "block"

            gpio_block = None
            for elem in tree.iterfind(block_tag):
                # print("{}: {}".format(elem.tag, elem.attrib))

                block_attrib = elem.attrib
                if block_attrib["type"] == self._name:
                    gpio_block = elem
                    break

            if gpio_block is not None:
                # Get the delay variable
                delay_var_map = self._get_delay_variable(
                    self._name, gpio_block)

                # TODO: Change this when we implement INOUT mode
                if gpio_obj.mode in self.mode2path_map:
                    # path_type = self.mode2path_map[gpio_obj.mode]

                    # Get the timing arc associated to this mode
                    if path_type in self._mode2arcs:
                        arcs_list = self._mode2arcs[path_type]

                        # For inout, a special case is required where
                        # we append the output and input mode arcs
                        # if they are relevant (not NOne)

                        if arcs_list:
                            for arc in arcs_list:

                                # We want to skip sequential arcs if
                                # it is not a registered path
                                # TODO: Not needed when we also need the
                                # non-registered delay when current gpio is registered
                                # if not self._is_arc_relevant(gpio_obj, arc, path_type):
                                #    continue

                                param_name = arc.get_delay()

                                # Get the block parameter table. If it has not
                                # been created, then do so (cache)
                                arc_table = self._load_arc_parameter(
                                    gpio_block, param_name, is_max)

                                delay_val = self._get_delay_value(
                                    gpio_obj, delay_var_map, arc_table)

                                # We shouldn't have multiple arc of the same
                                # time at once. Even with inout mode, there will be
                                # separated input, output and oe  mode
                                # If yes, this has to change
                                # print("arc: {} delay: {}".format(arc, delay_val))
                                arc2delay_map[arc] = delay_val
                    else:
                        self.logger.error("Unable to get the arcs")

        except Exception as excp:
            # self.logger.error("Error with reading the timing common_models: {}".format(excp))
            raise excp

        return arc2delay_map

    def _is_arc_relevant(self, gpio_obj, arc, path_type):
        '''
        Check if the arc is applicable to the gpio.

        :param gpio_obj: GPIO object
        :param arc: TimingArc object

        :return True if the arc is applicable to the object
        '''

        is_relevant = False
        is_comb_arc = False

        # Check arc type
        if arc.get_type() == dev_blk.TimingArc.TimingArcType.delay:
            is_comb_arc = True

        # Check mode. Only input, output and inout has sequential arc
        is_reg_mode = False
        if path_type == self.GPIOPathType.input:
            if gpio_obj.input is not None and gpio_obj.input.is_register:
                is_reg_mode = True
        elif path_type == self.GPIOPathType.output:
            if gpio_obj.output is not None and \
                    gpio_obj.output.register_option != gpio_obj.output.RegOptType.none:
                is_reg_mode = True
        elif path_type == self.GPIOPathType.oe:
            # For inout, only the OE path is checked here.
            # The rest will be checked above
            if gpio_obj.output_enable is not None and \
                    gpio_obj.output_enable.is_register:
                is_reg_mode = True

        if not is_comb_arc and is_reg_mode:
            is_relevant = True
        elif is_comb_arc and not is_reg_mode:
            is_relevant = True

        return is_relevant

    def _get_delay_value(self, gpio_obj, delay_var_map, arc_table, is_max=True):
        '''
        :param gpio_obj: The user gpio design object
        :param delay_var_map: The table of variables mapped to acceptable
                        values
        :param arc_table: The arc_parameter table with the name
                        representing the variables concatenated (as
                        in the xml file). An ArcDelayTable object

        :return the delay value of that parameter based on the required
                gpio configuration
        '''

        delay_val = 0

        # Only if the arc parameter has variable associated, then
        # we figure out, else take that one value
        if arc_table.get_pcount() == 0:
            delay_val = arc_table.get_variable_delay("")
            return delay_val

        iostd_name = ""
        schmitt_trigger = "false"
        slew_rate = "false"
        drive_strength = "1"
        input_alt = "other"

        # Get the io std
        if gpio_obj.gpio_def in self._ins2iostd_map:
            io_std = self._ins2iostd_map[gpio_obj.gpio_def]

            # Get the value associated to the io_standard
            if self.VariableType.io_standard in delay_var_map:
                io_std_list = delay_var_map[self.VariableType.io_standard]

                # self.logger.debug("Finding delay for gpio {} with io std {} and bank io std {} in {}".format(
                #    gpio_obj.name, gpio_obj.io_standard, io_std, ",".join(io_std_list)))

                # str_list = io_std.split(" ")
                iostd_name = io_std
                if iostd_name not in io_std_list:
                    # Then check if it could be based on the actual gpio instance
                    # io standard and not just pure voltage
                    gpio_io_std = gpio_obj.io_standard

                    if gpio_io_std in io_std_list:
                        iostd_name = gpio_io_std

                    else:
                        raise ValueError(
                            "I/O Standard {} not defined in timing variable list".format(iostd_name))

            else:
                raise ValueError(
                    "I/O Standard not defined in timing variable list")

        # Depending on the gpio mode, we check the setting mentioned
        # in the delay_var_map and the arc_table to find the
        # suitable delay to pick.
        if gpio_obj.mode == gpio_obj.PadModeType.input or\
            gpio_obj.mode == gpio_obj.PadModeType.inout:
            # Get the gpio setting that is relevant:
            # schmitt_trigger
            if gpio_obj.input is not None:
                if gpio_obj.input.is_schmitt_trigger_enable:
                    schmitt_trigger = "true"

                # Also check if this input mode was set as alternate
                # connection and get its secondary function
                gpio_reg = self._design.get_block_reg(
                    des_db.PeriDesign.BlockType.gpio)
                assert gpio_reg is not None

                if gpio_reg.is_input_dedicated_clock(gpio_obj) and gpio_reg.is_resource_dedicated_clock(gpio_obj):
                    input_alt = "gclk"
                elif gpio_reg.is_input_dedicated_control(gpio_obj) and gpio_reg.is_resource_input_dedicated_control(gpio_obj):
                    input_alt = "gctrl"

        if gpio_obj.mode == gpio_obj.PadModeType.output or\
                gpio_obj.mode == gpio_obj.PadModeType.clkout or \
                gpio_obj.mode == gpio_obj.PadModeType.inout:
            # Slew Rate is not applicable for clkout
            if gpio_obj.mode != gpio_obj.PadModeType.clkout:
                if gpio_obj.output is not None and \
                        gpio_obj.output.is_slew_rate_output:
                    slew_rate = "true"

            # Drive Strength
            if gpio_obj.output is not None:
                # Get the drive strength map
                drv_str_list = delay_var_map[self.VariableType.drive_strength]

                gpio_drv = gpio_obj.output.drv2str_map[gpio_obj.output.drive_strength]
                if gpio_drv not in drv_str_list:
                    raise ValueError(
                        "Drive Strength {} not defined in timing variable".format(
                            gpio_drv))

                else:
                    drive_strength = gpio_drv

        # Now that we have all the required setting, we
        # get the delay based on it
        var_list = arc_table.get_variables()
        concat_name = ""
        read_vcount = 0

        for index in range(len(var_list)):
            vname = var_list[index]
            vfound = True

            if vname in self.str2var_map:
                vtype = self.str2var_map[vname]

                if vtype == self.VariableType.io_standard:
                    vsetting = vname + ":" + iostd_name
                elif vtype == self.VariableType.schmitt_trigger:
                    vsetting = vname + ":" + schmitt_trigger
                elif vtype == self.VariableType.drive_strength:
                    vsetting = vname + ":" + drive_strength
                elif vtype == self.VariableType.slew_rate:
                    vsetting = vname + ":" + slew_rate
                elif vtype == self.VariableType.input_alternate:
                    vsetting = vname + ":" + input_alt
                else:
                    vfound = False

                if vfound:
                    read_vcount += 1

                if concat_name == "":
                    concat_name = vsetting
                else:
                    concat_name = concat_name + "," + vsetting

            else:
                raise ValueError(
                    "Unexpected timing variable name {}".format(vname))

        # Verify that the variable count matches
        if read_vcount != arc_table.get_pcount():
            raise ValueError(
                "Mismatch variable count, {} and the pcount parameter, {}".format(
                    read_vcount, arc_table.get_pcount()))

        # Get the delay from table based on the concat variable name
        tmp_delay = arc_table.get_variable_delay(concat_name)
        if tmp_delay is not None:
            delay_val = tmp_delay
        else:
            raise ValueError(
                "Unable to get the delay for variable {}".format(concat_name))

        return delay_val
