"""
Copyright (C) 2017-2018 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Sep 13, 2018

@author: maryam
"""
from __future__ import annotations
import abc
from typing import Mapping, TYPE_CHECKING

import util.gen_util

import common_device.rule as base_rule

if TYPE_CHECKING:
    from common_device.ctrl.ctrl_design import Control


@util.gen_util.freeze_it
class ControlChecker(base_rule.Checker):
    """
    Provide checkers to validate user configuration of configuration block design.

    These checks mostly focused on logical check but also include redundant schema checks.
    This is because the checker is meant to check in memory database which may not be written
    to file yet.
    """

    def __init__(self, design):
        """
        Constructor
        """
        super().__init__(design)
        self.reg = None
        self.block_tag = "configuration"

        if self.design is not None:
            if self.design.device_setting is not None:
                self.reg = self.design.device_setting.ctrl_reg

    def _build_rules(self):
        """
        Build control bank rule database
        """
        self._add_rule(ConfigurationRuleClock())
        self._add_rule(ConfigurationRuleBusName())
        self._add_rule(ConfigurationRulePinEmptyName())
        self._add_rule(ConfigurationRulePinInvalidName())
        self._add_rule(ConfigurationRuleUserStatusPin())


class ConfigurationRuleClock(base_rule.Rule):
    """
    Check that if the remote update feature is enabled, the
    clock pin name is defined.

    TODO: This rule can be removed since all clock pins have their own empty/invalid checks
    """

    def __init__(self):
        super().__init__()
        self.name = "configuration_rule_clock"

    def check(self, checker, design_block: Control):
        util.gen_util.mark_unused(checker)

        control = design_block

        if control.is_remote_update_enable:
            if self.is_name_empty(control.clock_name):
                self.error("Internal Reconfiguration Interface is enabled but clock pin name is empty.")
                return
            if self.is_pin_name_pattern_invalid(control.clock_name):
                self.error("Internal Reconfiguration Interface is enabled but clock pin name is invalid.")
                return


class ConfigurationRuleBusName(base_rule.Rule):
    """
    Check that if the remote update feature is enabled, the
    image selector bus name is defined and valid.

    PT-1341 Split from RulePin which allows name[x] format
    """

    def __init__(self):
        super().__init__()
        self.name = "configuration_rule_bus_name"

    def check(self, checker, design_block: Control):
        util.gen_util.mark_unused(checker)

        control = design_block

        if control.is_remote_update_enable:
            if self.check_name_empty_or_invalid(control.cbsel_bus_name,
                "Internal Reconfiguration Interface is enabled but bus name is empty.",
                "Internal Reconfiguration Interface is enabled but bus name is invalid."
            ):
                return


class ConfigurationRulePin(base_rule.Rule, metaclass=abc.ABCMeta):
    """
    Helper class for checking empty/invalid pin names
    Implmented class needs to inherit either RuleEmptyName or RuleInvalidName
    """

    @abc.abstractmethod
    def validate(self, pins : Mapping[str, base_rule.PinData]):
        pass

    def check(self, checker, design_block: Control):
        util.gen_util.mark_unused(checker)

        control = design_block
        pins: Mapping[str, base_rule.PinData] = {
            "Configuration Control Name" : base_rule.PinData(control.config_ctrl_name),
            "Image Selector Capture Pin Name" : base_rule.PinData(control.ena_capture_name),
            "Error Status Pin Name" : base_rule.PinData(control.error_status_name, allow_empty=True), # PT-416
        }
        user_mode_pins: Mapping[str, base_rule.PinData] = {
            "User Mode Signal Status Pin Name" :base_rule.PinData(control.um_signal_status_name),
        }

        if control.is_remote_update_enable:
            self.validate(pins)
        if control.is_user_mode_enable:
            self.validate(user_mode_pins)


class ConfigurationRulePinEmptyName(base_rule.RuleEmptyName, ConfigurationRulePin):
    """
    Check that if the remote update feature is enabled, the
    bus and pin names are filled
    """
    def __init__(self):
        super().__init__()
        self.name = "configuration_rule_empty_pins"
        # PT-1341: ignore for now
        self.severity_mode = self.SeverityType.info

class ConfigurationRulePinInvalidName(base_rule.RuleInvalidName, ConfigurationRulePin):
    """
    Check that if the remote update feature is enabled, the
    bus and pin names are valid
    """
    def __init__(self):
        super().__init__()
        self.name = "configuration_rule_invalid_pins"


class ConfigurationRuleUserStatusPin(base_rule.Rule):

    def __init__(self):
        super().__init__()
        self.name = "configuration_rule_in_user_pin"

    def is_serdes_quad_instance_exists(self, checker):
        is_quad_ins_found = False
        usage_str_list = []
        blk_names = ""

        if checker.design.quad_pcie_reg is not None and checker.design.quad_pcie_reg.get_inst_count() > 0:
            usage_str_list.append("PCIe")
        if checker.design.lane_10g_reg is not None and checker.design.lane_10g_reg.get_inst_count() > 0:
            usage_str_list.append("Ethernet XGMII")
        if checker.design.lane_1g_reg is not None and checker.design.lane_1g_reg.get_inst_count() > 0:
            usage_str_list.append("Ethernet SGMII")
        if checker.design.raw_serdes_reg is not None and checker.design.raw_serdes_reg.get_inst_count() > 0:
            usage_str_list.append("PMA Direct")

        if usage_str_list:
            blk_names = ",".join(usage_str_list)            
            is_quad_ins_found = True

        return is_quad_ins_found, blk_names
    
    def get_serdes_design_supported_type(self, checker):
        quad_types = []

        if checker.design.quad_pcie_reg is not None:
            quad_types.append("PCIe")
        if checker.design.lane_10g_reg is not None:
            quad_types.append("Ethernet XGMII")
        if checker.design.lane_1g_reg is not None:
            quad_types.append("Ethernet SGMII")
        if checker.design.raw_serdes_reg is not None:
            quad_types.append("PMA Direct")

        return ",".join(quad_types)
    
    def check(self, checker: ControlChecker, design_block: Control):
        # Only used for SERDES
        quad_type_str = self.get_serdes_design_supported_type(checker)
                
        if quad_type_str == "":
            return

        is_quad_used, blk_name_msg = self.is_serdes_quad_instance_exists(checker)

        if not design_block.is_user_mode_enable:
            # It is compulsory with PCIE but optional with 10G
            if blk_name_msg.find("PCIe") != -1:
                return self.error("User Status Pin Name should be configured for PCIe")
        else:
            if design_block.um_signal_status_name == "":
                return self.error("User Status Pin Name should not be empty when enable user status control.")

            if not is_quad_used:                
                quad_type_str = self.get_serdes_design_supported_type(checker)
                return self.error(f"User status control should be configured with instance of {quad_type_str}.")


if __name__ == "__main__":
    pass
